"use strict";
var _a, _b, _c, _d, _e, _f;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Env = exports.extractContainerPorts = exports.EnvFrom = exports.Cpu = exports.MountPropagation = exports.Container = exports.ImagePullPolicy = exports.EnvValue = exports.ResourceFieldPaths = exports.EnvFieldPaths = exports.ContainerSecurityContext = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const k8s = require("./imports/k8s");
const utils_1 = require("./utils");
/**
 * Container security attributes and settings.
 */
class ContainerSecurityContext {
    constructor(props = {}) {
        this.ensureNonRoot = props.ensureNonRoot ?? false;
        this.privileged = props.privileged ?? false;
        this.readOnlyRootFilesystem = props.readOnlyRootFilesystem ?? false;
        this.user = props.user;
        this.group = props.group;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            runAsGroup: this.group,
            runAsUser: this.user,
            runAsNonRoot: this.ensureNonRoot,
            privileged: this.privileged,
            readOnlyRootFilesystem: this.readOnlyRootFilesystem,
        };
    }
}
exports.ContainerSecurityContext = ContainerSecurityContext;
_a = JSII_RTTI_SYMBOL_1;
ContainerSecurityContext[_a] = { fqn: "cdk8s-plus-23.ContainerSecurityContext", version: "2.0.0-rc.98" };
var EnvFieldPaths;
(function (EnvFieldPaths) {
    /**
     * The name of the pod.
     */
    EnvFieldPaths["POD_NAME"] = "metadata.name";
    /**
     * The namespace of the pod.
     */
    EnvFieldPaths["POD_NAMESPACE"] = "metadata.namespace";
    /**
     * The uid of the pod.
     */
    EnvFieldPaths["POD_UID"] = "metadata.uid";
    /**
     * The labels of the pod.
     */
    EnvFieldPaths["POD_LABEL"] = "metadata.labels";
    /**
     * The annotations of the pod.
     */
    EnvFieldPaths["POD_ANNOTATION"] = "metadata.annotations";
    /**
     * The ipAddress of the pod.
     */
    EnvFieldPaths["POD_IP"] = "status.podIP";
    /**
     * The service account name of the pod.
     */
    EnvFieldPaths["SERVICE_ACCOUNT_NAME"] = "spec.serviceAccountName";
    /**
     * The name of the node.
     */
    EnvFieldPaths["NODE_NAME"] = "spec.nodeName";
    /**
     * The ipAddress of the node.
     */
    EnvFieldPaths["NODE_IP"] = "status.hostIP";
    /**
     * The ipAddresess of the pod.
     */
    EnvFieldPaths["POD_IPS"] = "status.podIPs";
})(EnvFieldPaths = exports.EnvFieldPaths || (exports.EnvFieldPaths = {}));
var ResourceFieldPaths;
(function (ResourceFieldPaths) {
    /**
     * CPU limit of the container.
     */
    ResourceFieldPaths["CPU_LIMIT"] = "limits.cpu";
    /**
     * Memory limit of the container.
     */
    ResourceFieldPaths["MEMORY_LIMIT"] = "limits.memory";
    /**
     * CPU request of the container.
     */
    ResourceFieldPaths["CPU_REQUEST"] = "requests.cpu";
    /**
     * Memory request of the container.
     */
    ResourceFieldPaths["MEMORY_REQUEST"] = "requests.memory";
    /**
     * Ephemeral storage limit of the container.
     */
    ResourceFieldPaths["STORAGE_LIMIT"] = "limits.ephemeral-storage";
    /**
     * Ephemeral storage request of the container.
     */
    ResourceFieldPaths["STORAGE_REQUEST"] = "requests.ephemeral-storage";
})(ResourceFieldPaths = exports.ResourceFieldPaths || (exports.ResourceFieldPaths = {}));
/**
 * Utility class for creating reading env values from various sources.
 */
class EnvValue {
    constructor(value, valueFrom) {
        this.value = value;
        this.valueFrom = valueFrom;
    }
    /**
     * Create a value by reading a specific key inside a config map.
     *
     * @param configMap - The config map.
     * @param key - The key to extract the value from.
     * @param options - Additional options.
     */
    static fromConfigMap(configMap, key, options = {}) {
        const source = {
            configMapKeyRef: {
                name: configMap.name,
                key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Defines an environment value from a secret JSON value.
     *
     * @param secretValue The secret value (secrent + key)
     * @param options Additional options
     */
    static fromSecretValue(secretValue, options = {}) {
        const source = {
            secretKeyRef: {
                name: secretValue.secret.name,
                key: secretValue.key,
                optional: options.optional,
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from the given argument.
     *
     * @param value - The value.
     */
    static fromValue(value) {
        return new EnvValue(value);
    }
    /**
     *
     * Create a value from a field reference.
     *
     * @param fieldPath: The field reference.
     * @param options: Additional options.
     */
    static fromFieldRef(fieldPath, options = {}) {
        let needsKey = false;
        if (fieldPath === EnvFieldPaths.POD_LABEL || fieldPath === EnvFieldPaths.POD_ANNOTATION) {
            needsKey = true;
        }
        if (needsKey && options.key === undefined) {
            throw new Error(`${fieldPath} requires a key`);
        }
        const source = {
            fieldRef: {
                fieldPath: (`${fieldPath}` + (needsKey ? `['${options.key}']` : '')),
                ...(options.apiVersion ? { apiVersion: options.apiVersion } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a resource.
     *
     * @param resource: Resource to select the value from.
     * @param options: Additional options.
     */
    static fromResource(resource, options = {}) {
        const source = {
            resourceFieldRef: {
                resource: `${resource}`,
                ...(options.divisor ? { divisor: k8s.IntOrString.fromString(options.divisor) } : {}),
                ...(options.container ? { containerName: options.container.name } : {}),
            },
        };
        return new EnvValue(undefined, source);
    }
    /**
     * Create a value from a key in the current process environment.
     *
     * @param key - The key to read.
     * @param options - Additional options.
     */
    static fromProcess(key, options = {}) {
        const value = process.env[key];
        if (options.required && !value) {
            throw new Error(`Missing ${key} env variable`);
        }
        return EnvValue.fromValue(value);
    }
}
exports.EnvValue = EnvValue;
_b = JSII_RTTI_SYMBOL_1;
EnvValue[_b] = { fqn: "cdk8s-plus-23.EnvValue", version: "2.0.0-rc.98" };
var ImagePullPolicy;
(function (ImagePullPolicy) {
    /**
     * Every time the kubelet launches a container, the kubelet queries the container image registry
     * to resolve the name to an image digest. If the kubelet has a container image with that exact
     * digest cached locally, the kubelet uses its cached image; otherwise, the kubelet downloads
     * (pulls) the image with the resolved digest, and uses that image to launch the container.
     *
     * Default is Always if ImagePullPolicy is omitted and either the image tag is :latest or
     * the image tag is omitted.
     */
    ImagePullPolicy["ALWAYS"] = "Always";
    /**
     * The image is pulled only if it is not already present locally.
     *
     * Default is IfNotPresent if ImagePullPolicy is omitted and the image tag is present but
     * not :latest
     */
    ImagePullPolicy["IF_NOT_PRESENT"] = "IfNotPresent";
    /**
     * The image is assumed to exist locally. No attempt is made to pull the image.
     */
    ImagePullPolicy["NEVER"] = "Never";
})(ImagePullPolicy = exports.ImagePullPolicy || (exports.ImagePullPolicy = {}));
/**
 * A single application container that you want to run within a pod.
 */
class Container {
    constructor(props) {
        if (props instanceof Container) {
            throw new Error('Attempted to construct a container from a Container object.');
        }
        this.name = props.name ?? 'main';
        this.image = props.image;
        this.port = props.port;
        this._command = props.command;
        this._args = props.args;
        this._readiness = props.readiness;
        this._liveness = props.liveness;
        this._startup = props.startup;
        this._lifecycle = props.lifecycle;
        this.resources = props.resources;
        this.workingDir = props.workingDir;
        this.mounts = props.volumeMounts ?? [];
        this.imagePullPolicy = props.imagePullPolicy ?? ImagePullPolicy.ALWAYS;
        this.securityContext = new ContainerSecurityContext(props.securityContext);
        this.env = new Env(props.envFrom ?? [], props.envVariables ?? {});
    }
    /**
     * Entrypoint array (the command to execute when the container starts).
     * @returns a copy of the entrypoint array, cannot be modified
     */
    get command() {
        return this._command ? [...this._command] : undefined;
    }
    /**
     * Arguments to the entrypoint.
     *
     * @returns a copy of the arguments array, cannot be modified.
     */
    get args() {
        return this._args ? [...this._args] : undefined;
    }
    /**
     * Mount a volume to a specific path so that it is accessible by the container.
     * Every pod that is configured to use this container will autmoatically have access to the volume.
     *
     * @param path - The desired path in the container.
     * @param storage - The storage to mount.
     */
    mount(path, storage, options = {}) {
        this.mounts.push({ path, volume: storage.asVolume(), ...options });
    }
    /**
     * @internal
     */
    _toKube() {
        const volumeMounts = [];
        for (const mount of this.mounts) {
            volumeMounts.push({
                name: mount.volume.name,
                mountPath: mount.path,
                readOnly: mount.readOnly,
                mountPropagation: mount.propagation,
                subPath: mount.subPath,
                subPathExpr: mount.subPathExpr,
            });
        }
        const ports = new Array();
        if (this.port) {
            ports.push({
                containerPort: this.port,
            });
        }
        // Resource requests and limits
        const cpuLimit = this.resources?.cpu?.limit?.amount;
        const cpuRequest = this.resources?.cpu?.request?.amount;
        const memoryLimit = this.resources?.memory?.limit;
        const memoryRequest = this.resources?.memory?.request;
        const ephemeralStorageLimit = this.resources?.ephemeralStorage?.limit;
        const ephemeralStorageRequest = this.resources?.ephemeralStorage?.request;
        const limits = {};
        const requests = {};
        if (cpuLimit) {
            limits.cpu = k8s.Quantity.fromString(cpuLimit);
        }
        if (memoryLimit) {
            limits.memory = k8s.Quantity.fromString(memoryLimit.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageLimit) {
            limits['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageLimit.toGibibytes().toString() + 'Gi');
        }
        if (cpuRequest) {
            requests.cpu = k8s.Quantity.fromString(cpuRequest);
        }
        if (memoryRequest) {
            requests.memory = k8s.Quantity.fromString(memoryRequest.toMebibytes().toString() + 'Mi');
        }
        if (ephemeralStorageRequest) {
            requests['ephemeral-storage'] = k8s.Quantity.fromString(ephemeralStorageRequest.toGibibytes().toString() + 'Gi');
        }
        let resourceRequirements = undefined;
        if (Object.keys(limits).length > 0 || Object.keys(requests).length > 0) {
            resourceRequirements = {
                limits: utils_1.undefinedIfEmpty(limits),
                requests: utils_1.undefinedIfEmpty(requests),
            };
        }
        const env = this.env._toKube();
        return {
            name: this.name,
            image: this.image,
            imagePullPolicy: this._imagePullPolicyToKube(this.imagePullPolicy),
            ports: utils_1.undefinedIfEmpty(ports),
            volumeMounts: utils_1.undefinedIfEmpty(volumeMounts),
            command: this.command,
            args: this.args,
            workingDir: this.workingDir,
            env: env.variables,
            envFrom: env.from,
            readinessProbe: this._readiness?._toKube(this),
            livenessProbe: this._liveness?._toKube(this),
            startupProbe: this._startup?._toKube(this),
            lifecycle: this._lifecycle ? {
                postStart: this._lifecycle.postStart?._toKube(this),
                preStop: this._lifecycle.preStop?._toKube(this),
            } : undefined,
            resources: resourceRequirements,
            securityContext: this.securityContext._toKube(),
        };
    }
    _imagePullPolicyToKube(imagePullPolicy) {
        switch (imagePullPolicy) {
            case ImagePullPolicy.ALWAYS:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.ALWAYS;
            case ImagePullPolicy.IF_NOT_PRESENT:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.IF_NOT_PRESENT;
            case ImagePullPolicy.NEVER:
                return k8s.IoK8SApiCoreV1ContainerImagePullPolicy.NEVER;
            default:
                throw new Error(`Unsupported image pull policy: ${imagePullPolicy}`);
        }
    }
}
exports.Container = Container;
_c = JSII_RTTI_SYMBOL_1;
Container[_c] = { fqn: "cdk8s-plus-23.Container", version: "2.0.0-rc.98" };
var MountPropagation;
(function (MountPropagation) {
    /**
     * This volume mount will not receive any subsequent mounts that are mounted
     * to this volume or any of its subdirectories by the host. In similar
     * fashion, no mounts created by the Container will be visible on the host.
     *
     * This is the default mode.
     *
     * This mode is equal to `private` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["NONE"] = "None";
    /**
     * This volume mount will receive all subsequent mounts that are mounted to
     * this volume or any of its subdirectories.
     *
     * In other words, if the host mounts anything inside the volume mount, the
     * Container will see it mounted there.
     *
     * Similarly, if any Pod with Bidirectional mount propagation to the same
     * volume mounts anything there, the Container with HostToContainer mount
     * propagation will see it.
     *
     * This mode is equal to `rslave` mount propagation as described in the Linux
     * kernel documentation
     */
    MountPropagation["HOST_TO_CONTAINER"] = "HostToContainer";
    /**
     * This volume mount behaves the same the HostToContainer mount. In addition,
     * all volume mounts created by the Container will be propagated back to the
     * host and to all Containers of all Pods that use the same volume
     *
     * A typical use case for this mode is a Pod with a FlexVolume or CSI driver
     * or a Pod that needs to mount something on the host using a hostPath volume.
     *
     * This mode is equal to `rshared` mount propagation as described in the Linux
     * kernel documentation
     *
     * Caution: Bidirectional mount propagation can be dangerous. It can damage
     * the host operating system and therefore it is allowed only in privileged
     * Containers. Familiarity with Linux kernel behavior is strongly recommended.
     * In addition, any volume mounts created by Containers in Pods must be
     * destroyed (unmounted) by the Containers on termination.
     *
     */
    MountPropagation["BIDIRECTIONAL"] = "Bidirectional";
})(MountPropagation = exports.MountPropagation || (exports.MountPropagation = {}));
/**
 * Represents the amount of CPU.
 * The amount can be passed as millis or units.
 */
class Cpu {
    constructor(amount) {
        this.amount = amount;
    }
    static millis(amount) {
        return new Cpu(amount + 'm');
    }
    static units(amount) {
        return new Cpu(amount.toString());
    }
}
exports.Cpu = Cpu;
_d = JSII_RTTI_SYMBOL_1;
Cpu[_d] = { fqn: "cdk8s-plus-23.Cpu", version: "2.0.0-rc.98" };
/**
 * A collection of env variables defined in other resources.
 */
class EnvFrom {
    constructor(configMap, prefix, sec) {
        this.configMap = configMap;
        this.prefix = prefix;
        this.sec = sec;
    }
    ;
    /**
     * @internal
     */
    _toKube() {
        return {
            configMapRef: this.configMap ? {
                name: this.configMap.name,
            } : undefined,
            secretRef: this.sec ? {
                name: this.sec.name,
            } : undefined,
            prefix: this.prefix,
        };
    }
}
exports.EnvFrom = EnvFrom;
_e = JSII_RTTI_SYMBOL_1;
EnvFrom[_e] = { fqn: "cdk8s-plus-23.EnvFrom", version: "2.0.0-rc.98" };
function extractContainerPorts(selector) {
    if (!selector) {
        return [];
    }
    const ports = [];
    // we don't use instanceof intentionally since it can create
    // cyclic import problems.
    const containers = selector.containers;
    for (const con of containers ?? []) {
        if (con.port) {
            ports.push(con.port);
        }
    }
    return ports;
}
exports.extractContainerPorts = extractContainerPorts;
/**
 * Container environment variables.
 */
class Env {
    constructor(sources, variables) {
        this._sources = sources;
        this._variables = variables;
    }
    /**
     * Selects a ConfigMap to populate the environment variables with.
     * The contents of the target ConfigMap's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromConfigMap(configMap, prefix) {
        return new EnvFrom(configMap, prefix, undefined);
    }
    /**
     * Selects a Secret to populate the environment variables with.
     * The contents of the target Secret's Data field will represent
     * the key-value pairs as environment variables.
     */
    static fromSecret(secr) {
        return new EnvFrom(undefined, undefined, secr);
    }
    /**
     * Add a single variable by name and value.
     * The variable value can come from various dynamic sources such a secrets of config maps.
     * Use `EnvValue.fromXXX` to select sources.
     */
    addVariable(name, value) {
        this._variables[name] = value;
    }
    /**
     * The environment variables for this container.
     * Returns a copy. To add environment variables use `container.env.addVariable()`.
     */
    get variables() {
        return { ...this._variables };
    }
    /**
     * Add a collection of variables by copying from another source.
     * Use `Env.fromXXX` functions to select sources.
     */
    copyFrom(from) {
        this._sources.push(from);
    }
    /**
     * The list of sources used to populate the container environment,
     * in addition to the `variables`.
     *
     * Returns a copy. To add a source use `container.env.copyFrom()`.
     */
    get sources() {
        return [...this._sources];
    }
    renderEnv(env) {
        const result = new Array();
        for (const [name, v] of Object.entries(env)) {
            result.push({
                name,
                value: v.value,
                valueFrom: v.valueFrom,
            });
        }
        return result;
    }
    /**
     * @internal
     */
    _toKube() {
        return {
            from: utils_1.undefinedIfEmpty(this._sources.map(s => s._toKube())),
            variables: utils_1.undefinedIfEmpty(this.renderEnv(this._variables)),
        };
    }
}
exports.Env = Env;
_f = JSII_RTTI_SYMBOL_1;
Env[_f] = { fqn: "cdk8s-plus-23.Env", version: "2.0.0-rc.98" };
//# sourceMappingURL=data:application/json;base64,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