"use strict";
var _a, _b, _c, _d;
Object.defineProperty(exports, "__esModule", { value: true });
exports.GCEPersistentDiskPersistentVolume = exports.AzureDiskPersistentVolume = exports.AwsElasticBlockStorePersistentVolume = exports.PersistentVolumeReclaimPolicy = exports.PersistentVolume = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk8s_1 = require("cdk8s");
const constructs_1 = require("constructs");
const base = require("./base");
const k8s = require("./imports/k8s");
const pvc = require("./pvc");
const volume = require("./volume");
class ImportedPersistentVolume extends constructs_1.Construct {
    constructor(scope, id, name) {
        super(scope, id);
        this._name = name;
    }
    get name() {
        return this._name;
    }
    get apiVersion() {
        return k8s.KubePersistentVolume.GVK.apiVersion;
    }
    get apiGroup() {
        return '';
    }
    get kind() {
        return k8s.KubePersistentVolume.GVK.kind;
    }
}
/**
 * A PersistentVolume (PV) is a piece of storage in the cluster that has been
 * provisioned by an administrator or dynamically provisioned using Storage Classes.
 * It is a resource in the cluster just like a node is a cluster resource.
 * PVs are volume plugins like Volumes, but have a lifecycle independent of any
 * individual Pod that uses the PV. This API object captures the details of the
 * implementation of the storage, be that NFS, iSCSI, or a
 * cloud-provider-specific storage system.
 */
class PersistentVolume extends base.Resource {
    constructor(scope, id, props = {}) {
        super(scope, id);
        this.resourceType = 'persistentvolumes';
        this.storage = props.storage;
        this.mode = props.volumeMode ?? pvc.PersistentVolumeMode.FILE_SYSTEM;
        this.storageClassName = props.storageClassName;
        this._accessModes = props.accessModes;
        this.mountOptions = props.mountOptions;
        this.reclaimPolicy = props.reclaimPolicy ?? PersistentVolumeReclaimPolicy.RETAIN;
        if (props.claim) {
            this.bind(props.claim);
        }
        this.apiObject = new k8s.KubePersistentVolume(this, 'Resource', {
            metadata: props.metadata,
            spec: cdk8s_1.Lazy.any({ produce: () => this._toKube() }),
        });
    }
    /**
     * Imports a pv from the cluster as a reference.
     */
    static fromPersistentVolumeName(scope, id, volumeName) {
        return new ImportedPersistentVolume(scope, id, volumeName);
    }
    /**
     * Access modes requirement of this claim.
     */
    get accessModes() {
        return this._accessModes ? [...this._accessModes] : undefined;
    }
    /**
     * PVC this volume is bound to. Undefined means this volume is not yet
     * claimed by any PVC.
     */
    get claim() {
        return this._claim;
    }
    /**
     * Reserve a `PersistentVolume` by creating a `PersistentVolumeClaim`
     * that is wired to claim this volume.
     *
     * Note that this method will throw in case the volume is already claimed.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#reserving-a-persistentvolume
     */
    reserve() {
        const claim = new pvc.PersistentVolumeClaim(this, `${this.name}PVC`, {
            metadata: { name: `pvc-${this.name}`, namespace: this.metadata.namespace },
            // the storage classes must match, otherwise the claim
            // will use the default class (or no class at all), which may be different than the class
            // of this volume. note that other requirements are not needed since
            // when they are not defined, any volume satisifies them.
            storageClassName: this.storageClassName,
        });
        this.bind(claim);
        claim.bind(this);
        return claim;
    }
    /**
     * Bind a volume to a specific claim.
     * Note that you must also bind the claim to the volume.
     *
     * @see https://kubernetes.io/docs/concepts/storage/persistent-volumes/#binding
     *
     * @param claim The PVC to bind to.
     */
    bind(claim) {
        if (this._claim && this._claim.name !== claim.name) {
            throw new Error(`Cannot bind volume '${this.name}' to claim '${claim.name}' since it is already bound to claim '${this._claim.name}'`);
        }
        this._claim = claim;
    }
    asVolume() {
        const claim = this.reserve();
        return volume.Volume.fromPersistentVolumeClaim(this, 'Volume', claim);
    }
    /**
     * @internal
     */
    _toKube() {
        const storage = this.storage ? k8s.Quantity.fromString(this.storage.toGibibytes() + 'Gi') : undefined;
        return {
            claimRef: this._claim ? { name: this._claim?.name } : undefined,
            accessModes: this.accessModes?.map(a => a.toString()),
            capacity: storage ? { storage } : undefined,
            mountOptions: this.mountOptions?.map(o => o),
            storageClassName: this.storageClassName,
            persistentVolumeReclaimPolicy: this._reclaimPolicyToKube(this.reclaimPolicy),
            volumeMode: this.mode,
        };
    }
    _reclaimPolicyToKube(reclaimPolicy) {
        switch (reclaimPolicy) {
            case PersistentVolumeReclaimPolicy.DELETE:
                return k8s.IoK8SApiCoreV1PersistentVolumeSpecPersistentVolumeReclaimPolicy.DELETE;
            case PersistentVolumeReclaimPolicy.RETAIN:
                return k8s.IoK8SApiCoreV1PersistentVolumeSpecPersistentVolumeReclaimPolicy.RETAIN;
            default:
                throw new Error(`Unsupported reclaim policy: ${reclaimPolicy}`);
        }
    }
}
exports.PersistentVolume = PersistentVolume;
_a = JSII_RTTI_SYMBOL_1;
PersistentVolume[_a] = { fqn: "cdk8s-plus-23.PersistentVolume", version: "2.0.0-rc.98" };
/**
 * Reclaim Policies.
 */
var PersistentVolumeReclaimPolicy;
(function (PersistentVolumeReclaimPolicy) {
    /**
     * The Retain reclaim policy allows for manual reclamation of the resource.
     * When the PersistentVolumeClaim is deleted, the PersistentVolume still exists and the
     * volume is considered "released". But it is not yet available for another claim
     * because the previous claimant's data remains on the volume.
     * An administrator can manually reclaim the volume with the following steps:
     *
     * 1. Delete the PersistentVolume. The associated storage asset in external
     *    infrastructure (such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume) still exists after the PV is deleted.
     * 2. Manually clean up the data on the associated storage asset accordingly.
     * 3. Manually delete the associated storage asset.
     *
     * If you want to reuse the same storage asset, create a new PersistentVolume
     * with the same storage asset definition.
     */
    PersistentVolumeReclaimPolicy["RETAIN"] = "Retain";
    /**
     * For volume plugins that support the Delete reclaim policy, deletion removes both the
     * PersistentVolume object from Kubernetes, as well as the associated storage asset in
     * the external infrastructure, such as an AWS EBS, GCE PD, Azure Disk, or Cinder volume.
     * Volumes that were dynamically provisioned inherit the reclaim policy of their StorageClass, which defaults to Delete.
     * The administrator should configure the StorageClass according to users' expectations; otherwise,
     * the PV must be edited or patched after it is created
     */
    PersistentVolumeReclaimPolicy["DELETE"] = "Delete";
})(PersistentVolumeReclaimPolicy = exports.PersistentVolumeReclaimPolicy || (exports.PersistentVolumeReclaimPolicy = {}));
/**
 * Represents an AWS Disk resource that is attached to a kubelet's host machine and
 * then exposed to the pod.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#awselasticblockstore
 */
class AwsElasticBlockStorePersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.volumeId = props.volumeId;
        this.readOnly = props.readOnly ?? false;
        this.fsType = props.fsType ?? 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            awsElasticBlockStore: {
                volumeId: this.volumeId,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AwsElasticBlockStorePersistentVolume = AwsElasticBlockStorePersistentVolume;
_b = JSII_RTTI_SYMBOL_1;
AwsElasticBlockStorePersistentVolume[_b] = { fqn: "cdk8s-plus-23.AwsElasticBlockStorePersistentVolume", version: "2.0.0-rc.98" };
/**
 * AzureDisk represents an Azure Data Disk mount on the host and bind mount to the pod.
 */
class AzureDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.diskName = props.diskName;
        this.diskUri = props.diskUri;
        this.cachingMode = props.cachingMode ?? volume.AzureDiskPersistentVolumeCachingMode.NONE;
        this.fsType = props.fsType ?? 'ext4';
        this.azureKind = props.kind ?? volume.AzureDiskPersistentVolumeKind.SHARED;
        this.readOnly = props.readOnly ?? false;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            azureDisk: {
                diskName: this.diskName,
                diskUri: this.diskUri,
                cachingMode: this.cachingMode,
                fsType: this.fsType,
                kind: this.azureKind,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.AzureDiskPersistentVolume = AzureDiskPersistentVolume;
_c = JSII_RTTI_SYMBOL_1;
AzureDiskPersistentVolume[_c] = { fqn: "cdk8s-plus-23.AzureDiskPersistentVolume", version: "2.0.0-rc.98" };
/**
 * GCEPersistentDisk represents a GCE Disk resource that is attached to a kubelet's host machine
 * and then exposed to the pod. Provisioned by an admin.
 *
 * @see https://kubernetes.io/docs/concepts/storage/volumes#gcepersistentdisk
 */
class GCEPersistentDiskPersistentVolume extends PersistentVolume {
    constructor(scope, id, props) {
        super(scope, id, props);
        this.pdName = props.pdName;
        this.readOnly = props.readOnly ?? false;
        this.fsType = props.fsType ?? 'ext4';
        this.partition = props.partition;
    }
    /**
     * @internal
     *
     * @see https://github.com/kubernetes/examples/blob/master/staging/volumes/azure_disk/README.md
     */
    _toKube() {
        const spec = super._toKube();
        return {
            ...spec,
            gcePersistentDisk: {
                pdName: this.pdName,
                fsType: this.fsType,
                partition: this.partition,
                readOnly: this.readOnly,
            },
        };
    }
}
exports.GCEPersistentDiskPersistentVolume = GCEPersistentDiskPersistentVolume;
_d = JSII_RTTI_SYMBOL_1;
GCEPersistentDiskPersistentVolume[_d] = { fqn: "cdk8s-plus-23.GCEPersistentDiskPersistentVolume", version: "2.0.0-rc.98" };
//# sourceMappingURL=data:application/json;base64,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