import asyncio
import json

from archytas.react import ReActAgent

from .config import config

OUTPUT_CHAR_LIMIT = 1000

class Summarizer(ReActAgent):

    def __init__(
        self,
        notebook: dict = {},
        **kwargs,
    ):
        super().__init__(
            model=config.LLM_SERVICE_MODEL,
            api_key=config.default_llm_service_token,
            tools=[],
            verbose=False,
            spinner=None,
            rich_print=False,
            allow_ask_user=False,
            **kwargs
        )

        context = f"""
You are working to provide summaries of this provided LLM-Powered Beaker notebook:
```
{notebook}
```

Beaker notebooks are an extension of Jupyter Notebooks that supports a new subtype of
cell called a 'query' cell. This means that some markdown cells with the `beaker_cell_type`
of `query` contain the the user query, the agents thoughts and possibly the final response
of the agent. If a code cell has a parent_id, that means the code cell was produced by the LLM Agent
in the parent query cell.
"""
        self.add_context(context)


async def summarize(notebook: dict, summary_prompts: dict[str, str] | None = None):
    summary_prompts = summary_prompts or {
        "title": "a sentence of no more than 10 words that explains the central theme of the notebook",
        "summary": "a ~400 character summary in passive voice",
    }
    for cell in notebook["cells"]:
        if "outputs" in cell and len(cell["outputs"]) > 0:
            for output in cell["outputs"]:
                if "text" in output:
                    output["text"] = output["text"][:OUTPUT_CHAR_LIMIT]
                if "traceback" in output:
                    del output["traceback"]
                if "data" in output:
                    for data_type in output["data"]:
                        output["data"][data_type] = output["data"][data_type][:OUTPUT_CHAR_LIMIT]

    agent = Summarizer(notebook)
    history_query = """
An event is one or more code cells that have been executed.
If the cell is a code-cell, summarize the code and the result, along with what order it was executed in based on the execute count.
If the cell is a markdown cell with metadata where beaker_cell_type = "query", then this cell contains an interaction between the
user and the LLM agent. The Agent may have provided a "response" message or may have responded with code that is added in
subsequent cell. For these query cells, make sure to summarize the query and the response, if provided, and note if and which code cell
was generated, if a code cell was generated by the query.

Generate a list of strings from the provided notebook where each string is a summary of an event following these steps

Step 1: Generate the initial list from the notebook

Step 2: Reorder list conservatively based on execution order (first to last executed) if needed. Also, remove
events that were accidentally added even though they were not executed

Step 3: Return a JSON Array containing strings as a final answer. Only respond with a JSON array. Do not include backticks '`' or extra text.
"""
    history = await agent.react_async(history_query)

    summaries = {
        "history": history
    }
    for name, summary_type in summary_prompts.items():
        query = f"""
Produce a {summary_type} by following these steps:

Step 1: Generate a rough draft of {summary_type} by referencing the list of events previously generated.

Step 2: Rewrite the draft by reviewing details from the original notebook.

Step 3: Make the draft more concise by removing information the audience will already know. The information the audience
will already know is that this was a process done in a notebook and there was an interaction between a user and an agent.
Therefore, do NOT include phrases like "the notebook" or "the process".
Additionally, don't refer to a subject like "the team". The draft should strictly focus on the content of the notebook
and not infer an author. Remove fluff and self-referential words like "this summary".

Step 4: Repeat Step 3

Step 5: Rewrite the draft as needed to ensure that the grammatical mood, tense, etc is respected from the original
request.

Step 6: Return the final draft as the final answer.
"""
        response = await agent.react_async(query)
        summaries[name] = response
    return summaries
