import sqlite3
import re as РегВыр

Ошибка = sqlite3.Error

Разбор_объявленных_типов = sqlite3.PARSE_DECLTYPES
Разбор_названий_столбцов = sqlite3.PARSE_COLNAMES
Успешно = sqlite3.SQLITE_OK
Отказ = sqlite3.SQLITE_DENY
Игнорировать = sqlite3.SQLITE_IGNORE
Стиль_параметров = sqlite3.paramstyle
Потокобезопасность = sqlite3.threadsafety

Словарь_перевода = {
    'НАЧАТЬ': 'BEGIN',
    'ЗАВЕРШИТЬ': 'END',
    'ТРАНЗАКЦИЮ': 'TRANSACTION',
    'УТВЕРДИТЬ': 'COMMIT',
    'ОТЛОЖЕННО': 'DEFERRED',
    'НЕМЕДЛЕННО': 'IMMEDIATE',
    'ИСКЛЮЧИТЕЛЬНО': 'EXCLUSIVE',
    'ОТКАТИТЬ': 'ROLLBACK',
    'ТОЧКА_СОХРАНЕНИЯ': 'SAVEPOINT',
    'НАСТРОИТЬ': 'PRAGMA',
    'ВЫБРАТЬ': 'SELECT',
    'ИЗ': 'FROM',
    'ГДЕ': 'WHERE',
    'УПОРЯДОЧИВ ПО': 'ORDER BY',
    'ПО ВОЗРАСТАНИЮ': 'ASC',
    'ПО УБЫВАНИЮ': 'DESC',
    'СОБРАВ ПО': 'GROUP BY',
    'ИМЕЮЩИМИ': 'HAVING',
    'РАЗЛИЧАЮЩИЕСЯ': 'DISTINCT',
    'ОБЪЕДИНИВ С': 'UNION',
    'СОДЕРЖАЩИЕСЯ В': 'INTERSECT',
    'ВСЕ': 'ALL',
    'НЕКОТОРЫЕ': 'ANY',
    'ИСКЛЮЧАЯ': 'EXCEPT',
    'НЕ БОЛЕЕ': 'LIMIT',
    'СО СМЕЩЕНИЕМ': 'OFFSET',
    'ПРИСОЕДИНИВ': 'JOIN',
    'ВНУТРЕННЕ': 'INNER',
    'ВНЕШНЕ': 'OUTER',
    'ПО ЛЕВОЙ': 'LEFT',
    'ПО ПРАВОЙ': 'RIGHT',
    'ПОЛНОСТЬЮ': 'FULL',
    'ПО': 'ON',
    'ПЕРЕКРЁСТНО': 'CROSS',
    'ИСПОЛЬЗУЯ': 'USING',
    'ДОБАВИТЬ ИЛИ ПРЕРВАТЬ В': 'INSERT OR ABORT INTO',
    'ДОБАВИТЬ ИЛИ СБОЙ В': 'INSERT OR FAIL INTO',
    'ДОБАВИТЬ ИЛИ ПРОПУСТИТЬ В': 'INSERT OR IGNORE INTO',
    'ДОБАВИТЬ ИЛИ ЗАМЕНИТЬ В': 'INSERT OR REPLACE INTO',
    'ДОБАВИТЬ ИЛИ ОТКАТИТЬ В': 'INSERT OR ROLLBACK INTO',
    'ДОБАВИТЬ В': 'INSERT INTO',
    'ЗНАЧЕНИЯ': 'VALUES',
    'ИЗМЕНИТЬ': 'UPDATE',
    'УСТАНОВИВ': 'SET',
    'УДАЛИТЬ ИЗ': 'DELETE FROM',
    'ПРИ КОНФЛИКТЕ': 'ON CONFLICT',
    'ПРИ УДАЛЕНИИ': 'ON DELETE',
    'ПРИ ИЗМЕНЕНИИ': 'ON UPDATE',
    'КАСКАДНО': 'CASCADE',
    'УСТАНОВИТЬ ПУСТЫМ': 'SET NULL',
    'УСТАНОВИТЬ ПО УМОЛЧАНИЮ': 'SET DEFAULT',
    'ОГРАНИЧИТЬ': 'RESTRICT',
    'БЕЗ ДЕЙСТВИЯ': 'NO ACTION',
    'ВЫПОЛНЯТЬ': 'DO',
    'НИЧЕГО': 'NOTHING',
    'ПРОПУСТИТЬ': 'IGNORE',
    'ПРЕРВАТЬ': 'ABORT',
    'ЗАМЕНИТЬ': 'REPLACE',
    'ОТКАТИТЬ': 'ROLLBACK',
    'СБОЙ': 'FAIL',
    'ИЗМЕНИТЬ ТАБЛИЦУ': 'ALTER TABLE',
    'ИЗМЕНИТЬ СТОЛБЕЦ': 'ALTER COLUMN',
    'СОЗДАТЬ': 'CREATE',
    'УДАЛИТЬ': 'DROP',
    'БАЗУ ДАННЫХ': 'DATABASE',
    'ТАБЛИЦУ': 'TABLE',
    'ВРЕМЕННУЮ': 'TEMPORARY',
    'ДОБАВИВ': 'ADD',
    'УДАЛИВ': 'DROP',
    'ИНДЕКС': 'INDEX',
    'СХЕМУ': 'SCHEMA',
    'УНИКАЛЬНЫЙ': 'UNIQUE',
    'ПРЕДСТАВЛЕНИЕ': 'VIEW',
    'СТОЛБЕЦ': 'COLUMN',
    'ТРИГГЕР': 'TRIGGER',
    'СТРОГО': 'STRICT',
    'ПУСТОЕ': 'IS NULL',
    'НЕ ПУСТОЕ': 'IS NOT NULL',
    'НЕ ПУСТО': 'NOT NULL',
    'ПУСТО': 'NULL',
    'И': 'AND',
    'ИЛИ': 'OR',
    'НЕ': 'NOT',
    'ПОДОБНО': 'LIKE',
    'МЕЖДУ': 'BETWEEN',
    'В': 'IN',
    'ЕСЛИ': 'IF',
    'СУЩЕСТВУЕТ': 'EXISTS',
    'БЕЗ': 'WITHOUT',
    'ОПРЕДЕЛИВ': 'WITH',
    'ВЫБОР': 'CASE',
    'ПРИ': 'WHEN',
    'ЭТО': 'THEN',
    'КОНЕЦ': 'END',
    'РЕКУРСИВНО': 'RECURSIVE',
    'ЦЕЛОЕ': 'INTEGER',
    'ВЕЩЕСТВЕННОЕ': 'REAL',
    'ТЕКСТ': 'TEXT',
    'ДАННЫЕ': 'BLOB',
    'ДАТА': 'DATE',
    'ВРЕМЯ': 'TIME',
    'ВРЕМЕННАЯ_МЕТКА': 'TIMESTAMP',
    'ЛОГИЧЕСКОЕ': 'BOOLEAN',
    'КЛЮЧ': 'KEY',
    'ПЕРВИЧНЫЙ': 'PRIMARY',
    'ВНЕШНИЙ': 'FOREIGN',
    'ПРОВЕРЯТЬ': 'CHECK',
    'КАК': 'AS',
    'ДА': 'TRUE',
    'НЕТ': 'FALSE',
    'ПО УМОЛЧАНИЮ': 'DEFAULT',
    'СЧЁТЧИК': 'AUTOINCREMENT',
    'ССЫЛАЕТСЯ НА': 'REFERENCES',
    'БЕЗ ИСКЛЮЧЕНИЙ': 'EXCLUDE NO OTHERS',
    'ИСКЛЮЧАЯ ТЕКУЩУЮ СТРОКУ': 'EXCLUDE CURRENT ROW',
    'ИСКЛЮЧАЯ ГРУППУ': 'EXCLUDE GROUP',
    'ИСКЛЮЧАЯ РАВНЫЕ': 'EXCLUDE TIES',
    'РАЗДЕЛИВ ПО': 'PARTITION BY',
    'ОТОБРАВ': 'FILTER',
    'ПРИВЕСТИ': 'CAST',
    'НАД': 'OVER',
    'ГРУППЫ': 'GROUPS',
    'ДИАПАЗОН': 'RANGE',
    'СТРОКИ': 'ROWS',
    'ПЕРВОЕ_НЕПУСТОЕ': 'COALESCE',
    'ТЕКУЩЕЙ СТРОКОЙ': 'CURRENT ROW',
    'НЕОГРАНИЧЕННО': 'UNBOUNDED',
    'ВПЕРЁД': 'FOLLOWING',
    'НАЗАД': 'PRECEDING',
    'НОМЕР_СТРОКИ': 'ROW_NUMBER',
    'РАНГ': 'RANK',
    'СЖАТЫЙ_РАНГ': 'DENSE_RANK',
    'РАЗДЕЛИТЬ_НА_ЧАСТИ': 'NTILE',
    'ПРЕДЫДУЩЕЕ': 'LAG',
    'СЛЕДУЮЩЕЕ': 'LEAD',
    'SIN': 'SIN',
    'COS': 'COS',
    'TG': 'TAN',
    'ARCSIN': 'ASIN',
    'ARCCOS': 'ACOS',
    'ARCTG': 'ATAN',
    'LN': 'LOG',
    'EXP': 'EXP',
    'СТЕПЕНЬ': 'POWER',
    'КВАДРАТНЫЙ_КОРЕНЬ': 'SQRT',
    'КОЛИЧЕСТВО': 'COUNT',
    'СУММА': 'SUM',
    'СРЕДНЕЕ': 'AVG',
    'НАИМЕНЬШЕЕ': 'MIN',
    'НАИБОЛЬШЕЕ': 'MAX',
    'СЦЕПИТЬ': 'CONCAT',
    'ДЛИНА': 'LENGTH',
    'В_ЗАГЛАВНЫЕ': 'UPPER',
    'В_СТРОЧНЫЕ': 'LOWER',
    'ПОДСТРОКА': 'SUBSTR',
    'ЗАМЕНИТЬ': 'REPLACE',
    'ОКРУГЛИТЬ': 'ROUND',
    'ОКРУГЛИТЬ_ВНИЗ': 'FLOOR',
    'ОКРУГЛИТЬ_ВВЕРХ': 'CEIL',
    'МОДУЛЬ': 'ABS',
    'ТЕКУЩАЯ_ДАТА': 'CURRENT_DATE',
    'ТЕКУЩЕЕ_ВРЕМЯ': 'CURRENT_TIME',
    'ТЕКУЩАЯ_ВРЕМЕННАЯ_МЕТКА': 'CURRENT_TIMESTAMP',
    'ИЗВЛЕЧЬ': 'EXTRACT',
    'ГОД': 'YEAR',
    'МЕСЯЦ': 'MONTH',
    'ДЕНЬ': 'DAY',
    'СЛУЧАЙНОЕ': 'RANDOM',
    'ТИП_ДАННЫХ': 'TYPEOF',
    'ПОСЛЕДНИЙ_ИДЕНТИФИКАТОР': 'LAST_INSERT_ROWID',
    'ЗНАК': 'SIGN',
    'ЕСЛИ_ПУСТО': 'IFNULL',
    'ЯВЛЯЕТСЯ_ПУСТЫМ': 'NULLIF',
    'СОВПАДЕНИЕ': 'LIKE',
    'ГЛОБАЛЬНОЕ_СОВПАДЕНИЕ': 'GLOB',
    'ОБРЕЗАТЬ': 'TRIM',
    'ОБРЕЗАТЬ_СЛЕВА': 'LTRIM',
    'ОБРЕЗАТЬ_СПРАВА': 'RTRIM',
    'ОБЩЕЕ': 'TOTAL',
    'ГРУППОВОЕ_СЦЕПЛЕНИЕ': 'GROUP_CONCAT',
    'СТАНДАРТНОЕ_ОТКЛОНЕНИЕ': 'STDDEV',
    'ДИСПЕРСИЯ': 'VARIANCE',
    'ПЕРВОЕ_ЗНАЧЕНИЕ': 'FIRST_VALUE',
    'ПОСЛЕДНЕЕ_ЗНАЧЕНИЕ': 'LAST_VALUE',
    'МАКСИМАЛЬНАЯ_ДЛИНА': 'MAXLEN',
    'МИНИМАЛЬНАЯ_ДЛИНА': 'MINLEN',
    'СЖАТЬ': 'COMPRESS',
    'РАСЖАТЬ': 'UNCOMPRESS',
    'В_ШЕСТН_ТЕКСТ': 'HEX',
    'ЗАКОДОВАТЬ_64': 'BASE64',
    'РАСКОДОВАТЬ_64': 'UNBASE64',
    'ВКЛЮЧЕНО': 'ON',
    'ОТКЛЮЧЕНО': 'OFF',
    'Синхронизация': 'synchronous',
    'Журнал': 'journal_mode',
    'Размер_кэша': 'cache_size',
    'Режим_блокования': 'locking_mode',
    'Самоочищение': 'auto_vacuum',
    'Ключ_шифрования': 'key',
    'Чтение_без_утверждения': 'read_uncommitted',
    'Режим_чтения': 'query_only',
    'Кодование': 'encoding',
    'Размер_страницы': 'page_size',
    'Максимум_страниц': 'max_page_count',
    'Временное_хранилище': 'temp_store',
    'Предел_анализа': 'analysis_limit',
    'Проверка_целостности': 'integrity_check',
    'Быстрая_проверка': 'quick_check',
    'Внешние_ключи': 'foreign_keys',
    'Пропускать_проверки': 'ignore_check_constraints',
    'Рекурсивные_триггеры': 'recursive_triggers',
    'Список_сопоставлений': 'collation_list',
    'Информация_о_таблице': 'table_info',
    'Информация_об_индексе': 'index_info',
    'Список_индексов': 'index_list',
    'Список_таблиц': 'table_list'
}

def Получить_токены(запрос):
    токены = []
    текущий_токен = ''
    в_строке = False
    в_комментарии = False
    строка_символ = None
    i = 0

    while i < len(запрос):
        символ = запрос[i]

        if символ == '\\' and в_строке and i + 1 < len(запрос):
            текущий_токен += символ + запрос[i + 1]
            i += 2
            continue

        if символ in ('"', "'") and not в_комментарии:
            if в_строке and символ == строка_символ:
                в_строке = False
                текущий_токен += символ
                токены.append(текущий_токен)
                текущий_токен = ''
                i += 1
                continue
            elif not в_строке:
                в_строке = True
                строка_символ = символ
                if текущий_токен:
                    токены.append(текущий_токен)
                    текущий_токен = ''
                текущий_токен = символ
                i += 1
                continue

        if символ == '-' and i + 1 < len(запрос) and запрос[i + 1] == '-' and not в_строке:
            if текущий_токен:
                токены.append(текущий_токен)
                текущий_токен = ''
            в_комментарии = True
            текущий_токен = '--'
            i += 2
            while i < len(запрос) and запрос[i] != '\n':
                текущий_токен += запрос[i]
                i += 1
            токены.append(текущий_токен)
            текущий_токен = ''
            в_комментарии = False
            continue

        if символ == '/' and i + 1 < len(запрос) and запрос[i + 1] == '*' and not в_строке:
            if текущий_токен:
                токены.append(текущий_токен)
                текущий_токен = ''
            в_комментарии = True
            текущий_токен = '/*'
            i += 2
            while i < len(запрос) - 1 and not (запрос[i] == '*' and запрос[i + 1] == '/'):
                текущий_токен += запрос[i]
                i += 1
            if i < len(запрос) - 1:
                текущий_токен += '*/'
                i += 2
            токены.append(текущий_токен)
            текущий_токен = ''
            в_комментарии = False
            continue

        if в_строке or в_комментарии:
            текущий_токен += символ
            i += 1
            continue

        if символ.isspace() or символ in (',', ';', '(', ')', '=', '<', '>', '!'):
            if текущий_токен:
                токены.append(текущий_токен)
                текущий_токен = ''
            if not символ.isspace():
                токены.append(символ)
            i += 1
            continue
        
        текущий_токен += символ
        i += 1

    if текущий_токен:
        токены.append(текущий_токен)

    return токены

def Перевести_токены(токены):
    переведённые_токены = []
    i = 0

    while i < len(токены):
        токен = токены[i]

        if (токен.startswith('"') or токен.startswith("'") or 
            токен.startswith('--') or токен.startswith('/*') or
            токен in (',', ';', '(', ')', '=', '<', '>', '!')):
            переведённые_токены.append(токен)
            i += 1
            continue

        найдено_ключевое_слово = False
        for длина in range(5, 0, -1):
            if i + длина <= len(токены):
                фраза = ' '.join(токены[i:i + длина])
                if фраза in Словарь_перевода:
                    переведённые_токены.append(Словарь_перевода[фраза])
                    i += длина
                    найдено_ключевое_слово = True
                    break
        if not найдено_ключевое_слово:
            переведённые_токены.append(токен)
            i += 1

    return переведённые_токены

def Собрать_запрос(токены):
    запрос = ''
    for i, токен in enumerate(токены):
        if токен in (',', ';', '(', ')', '=', '<', '>', '!'):
            запрос += токен
        else:
            if i > 0 and токены[i - 1] not in (' ', ',', ';', '(', '=', '<', '>', '!'):
                запрос += ' '
            запрос += токен
    return запрос.strip()

def Перевести_код_запроса(текст):
    токены = Получить_токены(текст)
    переведённые_токены = Перевести_токены(токены)
    запрос = Собрать_запрос(переведённые_токены)
    return запрос

Переводы_ошибок = {
    r"no such table: (.+)": r'Таблица "\1" не существует.',
    r"database is locked": r'База данных заблокована.',
    r"near \"(.+)\": syntax error": r'Синтаксическая ошибка около "\1".',
    r"table (.+) already exists": r'Таблица "\1" уже существует.',
    r"no such column: (.+)": r'Столбец "\1" не существует.',
    r"(.+) has no column named (.+)": r'Таблица "\1" не имеет столбца "\2".',
    r"cannot open database file": r'Не удалось открыть файл базы данных.',
    r"database or disk is full": r'База данных или диск переполнены.',
    r"attempt to write a readonly database": r'Попытка записи в базу данных, доступную только для чтения.',
    r"no such index: (.+)": r'Индекс "\1" не существует.',
    r"too many SQL variables": r'Слишком много переменных в запросе.',
    r"maximum recursion depth exceeded": r'Превышена максимальная глубина рекурсии в запросе.',
    r"no such database: (.+)": r'База данных "\1" не существует.',
    r"incomplete input": r'Неполный запрос.',
    r"malformed database schema \((.+)\)": r'Некорректная схема базы данных: \1.',
    r"file is not a database": r'Файл не является базой данных.',
    r"database disk image is malformed": r'Образ диска базы данных повреждён.',
    r"incorrect number of bindings supplied. The statement has (\d+) parameters, and (\d+) were supplied": r'Неверное количество переданных параметров. Ожидаемых запросом параметров - \1, передано - \2.',
    r"You did not supply a value for binding (\d+)": r'Не указано значение для параметра \1.',
    r"Cannot operate on a closed database": r'Невозможно выполнить операцию на закрытой базе данных.',
    r"Cannot operate on a closed cursor": r'Невозможно выполнить операцию на закрытом указателе.',
    r"column (.+) is not unique": r'Столбец "\1" должен быть уникальным.',
    r"number of bound variables does not match number of parameters": r'Количество привязанных переменных не соответствует количеству параметров.',
    r"only one statement is allowed": r'Разрешено выполнять только один запрос за раз.',
    r"NOT NULL constraint failed: (.+)": r'Нарушение ограничения НЕ ПУСТО для столбца "\1".',
    r"UNIQUE constraint failed: (.+)": r'Нарушение ограничения УНИКАЛЬНЫЙ для столбца "\1".',
    r"FOREIGN KEY constraint failed": r'Нарушение ограничения ВНЕШНИЙ КЛЮЧ.',
    r"CHECK constraint failed: (.+)": r'Нарушение ограничения ПРОВЕРЯТЬ: \1.',
    r"PRIMARY KEY must be unique": r'Первичный ключ должен быть уникальным.',
    r"Error binding parameter (\d+) - probably unsupported type": r'Ошибка привязки параметра \1 - вероятно, неподдерживаемый тип.',
    r"Cursor needed to be reset because of commit/rollback and can no longer be fetched from": r'Указатель должен быть сброшен из-за утверждения или отката и больше не может быть использован.',
    r"Warning: You can only execute one statement at a time": r'Можно выполнять только один запрос за раз.',
    r"(.+) not supported by this database": r'Операция "\1" не поддерживается этой базой данных.',
    r"too many connections": r'Слишком много активных подключений к базе данных.',
    r"no such function: (.+)": r'Функция "\1" не существует.',
    r"no such module: (.+)": r'Модуль "\1" не существует.',
    r"parameters are of unsupported type": r'Параметры имеют неподдерживаемый тип.',
    r"interrupted": r'Операция была прервана.',
    r"out of memory": r'Недостаточно памяти для выполнения операции.',
    r"no such pragma: (.+)": r'Настройка "\1" не существует.',
    r"invalid value for pragma (.+)": r'Недопустимое значение для настройки "\1".',
    r"unsupported journal mode: (.+)": r'Неподдерживаемый режим журнала: \1.',
    r"invalid encoding: (.+)": r'Недопустимая кодировка: \1.',
    r"invalid synchronous value: (.+)": r'Недопустимое значение синхронизации: \1.'
}

def Перевести_ошибку(исключение):
    сообщение = str(исключение)
    for шаблон, перевод in Переводы_ошибок.items():
        совпадение = РегВыр.match(шаблон, сообщение)
        if совпадение:
            return РегВыр.sub(шаблон, перевод, сообщение)
    return f"Неизвестная ошибка: {сообщение}"

class Указатель():
    def __init__(экземпляр, курсор):
        экземпляр._курсор = курсор

    def Выполнить_запрос(экземпляр, запрос, параметры=()):
        try:
            запрос = Перевести_код_запроса(запрос)
            экземпляр._курсор.execute(запрос, параметры)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Выполнить_запросы(экземпляр, запрос, последовательность_параметров):
        try:
            запрос = Перевести_код_запроса(запрос)
            экземпляр._курсор.executemany(запрос, последовательность_параметров)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Выполнить_сценарий(экземпляр, сценарий):
        try:
            скрипт = Перевести_код_запроса(сценарий)
            экземпляр._курсор.executescript(скрипт)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Извлечь_запись(экземпляр):
        try:
            return экземпляр._курсор.fetchone()
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Извлечь_записи(экземпляр, количество):
        try:
            return экземпляр._курсор.fetchmany(количество)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Извлечь_все_записи(экземпляр):
        try:
            return экземпляр._курсор.fetchall()
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Закрыть(экземпляр):
        try:
            экземпляр._курсор.close()
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    @property
    def Описание(экземпляр):
        return экземпляр._курсор.description

    @property
    def Количество_строк(экземпляр):
        return экземпляр._курсор.rowcount

    @property
    def Последний_идентификатор(экземпляр):
        return экземпляр._курсор.lastrowid

class Соединение():
    def __init__(экземпляр,
        путь, *,
        таймаут = 5.0,
        обнаружение_типов = 0,
        уровень_изоляции = None,
        проверять_тот_же_ли_поток = True,
        кэш_инструкций = 128,
        является_ссылкой = False
    ):
        if уровень_изоляции == 'ОТЛОЖЕННО': уровень_изоляции = 'DEFERRED'
        elif уровень_изоляции == 'НЕМЕДЛЕННО': уровень_изоляции = 'IMMEDIATE'
        elif уровень_изоляции == 'ИСКЛЮЧИТЕЛЬНО': уровень_изоляции = 'EXCLUSIVE'
        else: уровень_изоляции = None

        try:
            экземпляр._соединение = sqlite3.connect(
                путь,
                timeout=таймаут,
                detect_types=обнаружение_типов,
                isolation_level=уровень_изоляции,
                check_same_thread=проверять_тот_же_ли_поток,
                cached_statements=кэш_инструкций,
                uri=является_ссылкой
            )
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def __enter__(экземпляр):
        return экземпляр

    def __exit__(экземпляр, exc_type, exc_value, traceback):
        экземпляр.Закрыть()

    def Создать_указатель(экземпляр):
        try:
            курсор = экземпляр._соединение.cursor()
            return Указатель(курсор)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Выполнить_запрос(экземпляр, запрос, параметры=()):
        try:
            запрос = Перевести_код_запроса(запрос)
            указатель = экземпляр._соединение.execute(запрос, параметры)
            return Указатель(указатель)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Выполнить_запросы(экземпляр, запрос, параметры):
        try:
            запрос = Перевести_код_запроса(запрос)
            указатель = экземпляр._соединение.executemany(запрос, параметры)
            return Указатель(указатель)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Выполнить_сценарий(экземпляр, сценарий):
        try:
            скрипт = Перевести_код_запроса(сценарий)
            экземпляр._соединение.executescript(скрипт)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Создать_функцию(экземпляр, имя, количество_параметров, функция):
        try:
            экземпляр._соединение.create_function(имя, количество_параметров, функция)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Создать_агрегатную_функцию(экземпляр, имя, число_аргументов, класс):
        try:
            экземпляр._соединение.create_aggregate(имя, число_аргументов, класс)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Создать_сопоставление(экземпляр, имя, функция):
        try:
            экземпляр._соединение.create_collation(имя, функция)
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Прервать(экземпляр):
        try:
            экземпляр._соединение.interrupt()
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Закрыть(экземпляр):
        try:
            экземпляр._соединение.close()
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Утвердить(экземпляр):
        try:
            экземпляр._соединение.commit()
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    def Откатить(экземпляр):
        try:
            экземпляр._соединение.rollback()
        except Ошибка as ош:
            raise Ошибка(Перевести_ошибку(ош)) from ош

    @property
    def В_транзакции(экземпляр):
        return экземпляр._соединение.in_transaction

    @property
    def Изоляция(экземпляр):
        return экземпляр._соединение.isolation_level

    @Изоляция.setter
    def Изоляция(экземпляр, значение):
        if значение not in (None, 'DEFERRED', 'IMMEDIATE', 'EXCLUSIVE'):
            raise ValueError(f"Недопустимый уровень изоляции: {значение}")
        экземпляр._соединение.isolation_level = значение

class Обёртка:
    def __init__(экземпляр, путь_к_файлу, **аргументы_инициализации):
        экземпляр._соединение = None

        try:
            экземпляр._соединение = Соединение(путь_к_файлу, **аргументы_инициализации)._соединение

        except Ошибка as ош:
            экземпляр._Обновить_ошибку(ош)
        except Exception as ош: 
            экземпляр._Обновить_ошибку(ош)

    def __del__(экземпляр):
        if экземпляр._соединение:
            try:
                экземпляр._соединение.close()
            except Ошибка as ош:
                print(f"Ошибка при закрытии соединения: {Перевести_ошибку(ош)}")

    def _Обновить_ошибку(экземпляр, ош=None):
        if isinstance(ош, Ошибка):
            raise Ошибка(Перевести_ошибку(ош)) from ош
        elif ош:
            raise Exception(f"Неизвестная ошибка: {ош}") from ош
        else:
            raise Exception("Неизвестная ошибка или проблема соединения.")

    def Выполнить_запрос(экземпляр, запрос, параметры=()):
    # Выполняет запрос к БД.
        if not экземпляр._соединение:
            экземпляр._Обновить_ошибку(Ошибка("Соединение с БД не установлено."))

        try:
            переведённый_запрос = Перевести_код_запроса(запрос)
            курсор = экземпляр._соединение.cursor()
            курсор.execute(переведённый_запрос, параметры)
            экземпляр._соединение.commit()

        except Ошибка as ош:
            экземпляр._Обновить_ошибку(ош)

    def Запросить_таблицу(экземпляр, запрос, параметры=()):
    # Выполняет запрос ВЫБРАТЬ и возвращает все строки как список словарей.
        if not экземпляр._соединение:
            экземпляр._Обновить_ошибку(Ошибка("Соединение с БД не установлено."))

        try:
            экземпляр._соединение.row_factory = sqlite3.Row 

            переведённый_запрос = Перевести_код_запроса(запрос)
            курсор = экземпляр._соединение.cursor()
            курсор.execute(переведённый_запрос, параметры)

            результат = [dict(запись) for запись in курсор.fetchall()]

            return результат
        except Ошибка as ош:
            экземпляр._Обновить_ошибку(ош)

    def Запросить_столбец(экземпляр, запрос, параметры=()):
    # Выполняет запрос ВЫБРАТЬ и возвращает значения первого столбца всех строк как список.
        if not экземпляр._соединение:
            экземпляр._Обновить_ошибку(Ошибка("Соединение с БД не установлено."))

        try:
            экземпляр._соединение.row_factory = None

            переведённый_запрос = Перевести_код_запроса(запрос)
            курсор = экземпляр._соединение.cursor()
            курсор.execute(переведённый_запрос, параметры)
            результат = [строка[0] for строка in курсор.fetchall()]

            return результат

        except Ошибка as ош:
            экземпляр._Обновить_ошибку(ош)

    def Запросить_строку(экземпляр, запрос, параметры=()):
    # Выполняет запрос ВЫБРАТЬ и возвращает первую строку как словарь.
        if not экземпляр._соединение:
            экземпляр._Обновить_ошибку(Ошибка("Соединение с БД не установлено."))

        try:
            экземпляр._соединение.row_factory = sqlite3.Row

            переведённый_запрос = Перевести_код_запроса(запрос)
            курсор = экземпляр._соединение.cursor()
            курсор.execute(переведённый_запрос, параметры)
            строка = курсор.fetchone()

            return dict(строка) if строка else None

        except Ошибка as ош:
            экземпляр._Обновить_ошибку(ош)

    def Запросить_таблицу_нум_стр(экземпляр, запрос, параметры=()):
    # Выполняет запрос ВЫБРАТЬ и возвращает все строки как список кортежей.
        if not экземпляр._соединение:
            экземпляр._Обновить_ошибку(Ошибка("Соединение с БД не установлено."))

        try:
            экземпляр._соединение.row_factory = None

            переведённый_запрос = Перевести_код_запроса(запрос)
            курсор = экземпляр._соединение.cursor()
            курсор.execute(переведённый_запрос, параметры)
            результат = курсор.fetchall()

            return результат

        except Ошибка as ош:
            экземпляр._Обновить_ошибку(ош)

    def Запросить_строку_нум(экземпляр, запрос, параметры=()):
    # Выполняет запрос ВЫБРАТЬ и возвращает первую строку как кортеж.
        if not экземпляр._соединение:
            экземпляр._Обновить_ошибку(Ошибка("Соединение с БД не установлено."))

        try:
            экземпляр._соединение.row_factory = None

            переведённый_запрос = Перевести_код_запроса(запрос)
            курсор = экземпляр._соединение.cursor()
            курсор.execute(переведённый_запрос, параметры)
            строка = курсор.fetchone()

            return строка

        except Ошибка as ош:
            экземпляр._Обновить_ошибку(ош)

    def Запросить_ячейку(экземпляр, запрос, параметры=()):
    # Выполняет запрос ВЫБРАТЬ и возвращает значение первой ячейки первой строки.
        if not экземпляр._соединение:
            экземпляр._Обновить_ошибку(Ошибка("Соединение с БД не установлено."))

        try:
            экземпляр._соединение.row_factory = None

            переведённый_запрос = Перевести_код_запроса(запрос)
            курсор = экземпляр._соединение.cursor()
            курсор.execute(переведённый_запрос, параметры)
            строка = курсор.fetchone()

            return строка[0] if строка else None

        except Ошибка as ош:
            экземпляр._Обновить_ошибку(ош)

    def Добавленный_номер(экземпляр):
    # Возвращает номер последней вставленной строки.
        if not экземпляр._соединение:
            экземпляр._Обновить_ошибку(Ошибка("Соединение с БД не установлено."))

        return экземпляр._соединение.lastrowid
