"use strict";
var _a, _b;
Object.defineProperty(exports, "__esModule", { value: true });
exports.CompositeDependable = exports.Grant = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cdk = require("@aws-cdk/core");
const policy_statement_1 = require("./policy-statement");
/**
 * Result of a grant() operation
 *
 * This class is not instantiable by consumers on purpose, so that they will be
 * required to call the Grant factory functions.
 */
class Grant {
    constructor(props) {
        this.options = props.options;
        this.principalStatement = props.principalStatement;
        this.resourceStatement = props.resourceStatement;
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return props.policyDependable ? cdk.DependableTrait.get(props.policyDependable).dependencyRoots : [];
            },
        });
    }
    /**
     * Grant the given permissions to the principal
     *
     * The permissions will be added to the principal policy primarily, falling
     * back to the resource policy if necessary. The permissions must be granted
     * somewhere.
     *
     * - Trying to grant permissions to a principal that does not admit adding to
     *   the principal policy while not providing a resource with a resource policy
     *   is an error.
     * - Trying to grant permissions to an absent principal (possible in the
     *   case of imported resources) leads to a warning being added to the
     *   resource construct.
     */
    static addToPrincipalOrResource(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_GrantWithResourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPrincipalOrResource);
            }
            throw error;
        }
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const resourceAndPrincipalAccountComparison = options.grantee.grantPrincipal.principalAccount
            ? cdk.Token.compareStrings(options.resource.env.account, options.grantee.grantPrincipal.principalAccount)
            : undefined;
        // if both accounts are tokens, we assume here they are the same
        const equalOrBothUnresolved = resourceAndPrincipalAccountComparison === cdk.TokenComparison.SAME
            || resourceAndPrincipalAccountComparison == cdk.TokenComparison.BOTH_UNRESOLVED;
        const sameAccount = resourceAndPrincipalAccountComparison
            ? equalOrBothUnresolved
            // if the principal doesn't have an account (for example, a service principal),
            // we should modify the resource's trust policy
            : false;
        // If we added to the principal AND we're in the same account, then we're done.
        // If not, it's a different account and we must also add a trust policy on the resource.
        if (result.success && sameAccount) {
            return result;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        return new Grant({
            resourceStatement: statement,
            options,
            policyDependable: resourceResult.statementAdded ? resourceResult.policyDependable ?? options.resource : undefined,
        });
    }
    /**
     * Try to grant the given permissions to the given principal
     *
     * Absence of a principal leads to a warning, but failing to add
     * the permissions to a present principal is not an error.
     */
    static addToPrincipal(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_GrantOnPrincipalOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPrincipal);
            }
            throw error;
        }
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: options.resourceArns,
        });
        const addedToPrincipal = options.grantee.grantPrincipal.addToPrincipalPolicy(statement);
        if (!addedToPrincipal.statementAdded) {
            return new Grant({ principalStatement: undefined, options });
        }
        if (!addedToPrincipal.policyDependable) {
            throw new Error('Contract violation: when Principal returns statementAdded=true, it should return a dependable');
        }
        return new Grant({ principalStatement: statement, options, policyDependable: addedToPrincipal.policyDependable });
    }
    /**
     * Add a grant both on the principal and on the resource
     *
     * As long as any principal is given, granting on the principal may fail (in
     * case of a non-identity principal), but granting on the resource will
     * never fail.
     *
     * Statement will be the resource statement.
     */
    static addToPrincipalAndResource(options) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_GrantOnPrincipalAndResourceOptions(options);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.addToPrincipalAndResource);
            }
            throw error;
        }
        const result = Grant.addToPrincipal({
            ...options,
            scope: options.resource,
        });
        const statement = new policy_statement_1.PolicyStatement({
            actions: options.actions,
            resources: (options.resourceSelfArns || options.resourceArns),
            principals: [options.resourcePolicyPrincipal || options.grantee.grantPrincipal],
        });
        const resourceResult = options.resource.addToResourcePolicy(statement);
        const resourceDependable = resourceResult.statementAdded ? resourceResult.policyDependable ?? options.resource : undefined;
        return new Grant({
            principalStatement: statement,
            resourceStatement: result.resourceStatement,
            options,
            policyDependable: resourceDependable ? new CompositeDependable(result, resourceDependable) : result,
        });
    }
    /**
     * Returns a "no-op" `Grant` object which represents a "dropped grant".
     *
     * This can be used for e.g. imported resources where you may not be able to modify
     * the resource's policy or some underlying policy which you don't know about.
     *
     * @param grantee The intended grantee
     * @param _intent The user's intent (will be ignored at the moment)
     */
    static drop(grantee, _intent) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_iam_IGrantable(grantee);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.drop);
            }
            throw error;
        }
        return new Grant({
            options: { grantee, actions: [], resourceArns: [] },
        });
    }
    /**
     * Whether the grant operation was successful
     */
    get success() {
        return this.principalStatement !== undefined || this.resourceStatement !== undefined;
    }
    /**
     * Throw an error if this grant wasn't successful
     */
    assertSuccess() {
        if (!this.success) {
            // eslint-disable-next-line max-len
            throw new Error(`${describeGrant(this.options)} could not be added on either identity or resource policy.`);
        }
    }
    /**
     * Make sure this grant is applied before the given constructs are deployed
     *
     * The same as construct.node.addDependency(grant), but slightly nicer to read.
     */
    applyBefore(...constructs) {
        for (const construct of constructs) {
            construct.node.addDependency(this);
        }
    }
}
exports.Grant = Grant;
_a = JSII_RTTI_SYMBOL_1;
Grant[_a] = { fqn: "@aws-cdk/aws-iam.Grant", version: "1.201.0" };
function describeGrant(options) {
    return `Permissions for '${options.grantee}' to call '${options.actions}' on '${options.resourceArns}'`;
}
/**
 * Composite dependable
 *
 * Not as simple as eagerly getting the dependency roots from the
 * inner dependables, as they may be mutable so we need to defer
 * the query.
 */
class CompositeDependable {
    constructor(...dependables) {
        cdk.DependableTrait.implement(this, {
            get dependencyRoots() {
                return Array.prototype.concat.apply([], dependables.map(d => cdk.DependableTrait.get(d).dependencyRoots));
            },
        });
    }
}
exports.CompositeDependable = CompositeDependable;
_b = JSII_RTTI_SYMBOL_1;
CompositeDependable[_b] = { fqn: "@aws-cdk/aws-iam.CompositeDependable", version: "1.201.0" };
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZ3JhbnQuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJncmFudC50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiOzs7Ozs7QUFBQSxxQ0FBcUM7QUFDckMseURBQXFEO0FBMkZyRDs7Ozs7R0FLRztBQUNILE1BQWEsS0FBSztJQWtKaEIsWUFBb0IsS0FBaUI7UUFDbkMsSUFBSSxDQUFDLE9BQU8sR0FBRyxLQUFLLENBQUMsT0FBTyxDQUFDO1FBQzdCLElBQUksQ0FBQyxrQkFBa0IsR0FBRyxLQUFLLENBQUMsa0JBQWtCLENBQUM7UUFDbkQsSUFBSSxDQUFDLGlCQUFpQixHQUFHLEtBQUssQ0FBQyxpQkFBaUIsQ0FBQztRQUVqRCxHQUFHLENBQUMsZUFBZSxDQUFDLFNBQVMsQ0FBQyxJQUFJLEVBQUU7WUFDbEMsSUFBSSxlQUFlO2dCQUNqQixPQUFPLEtBQUssQ0FBQyxnQkFBZ0IsQ0FBQyxDQUFDLENBQUMsR0FBRyxDQUFDLGVBQWUsQ0FBQyxHQUFHLENBQUMsS0FBSyxDQUFDLGdCQUFnQixDQUFDLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7WUFDdkcsQ0FBQztTQUNGLENBQUMsQ0FBQztLQUNKO0lBM0pEOzs7Ozs7Ozs7Ozs7O09BYUc7SUFDSSxNQUFNLENBQUMsd0JBQXdCLENBQUMsT0FBaUM7Ozs7Ozs7Ozs7UUFDdEUsTUFBTSxNQUFNLEdBQUcsS0FBSyxDQUFDLGNBQWMsQ0FBQztZQUNsQyxHQUFHLE9BQU87WUFDVixLQUFLLEVBQUUsT0FBTyxDQUFDLFFBQVE7U0FDeEIsQ0FBQyxDQUFDO1FBRUgsTUFBTSxxQ0FBcUMsR0FBRyxPQUFPLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0I7WUFDM0YsQ0FBQyxDQUFDLEdBQUcsQ0FBQyxLQUFLLENBQUMsY0FBYyxDQUFDLE9BQU8sQ0FBQyxRQUFRLENBQUMsR0FBRyxDQUFDLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTyxDQUFDLGNBQWMsQ0FBQyxnQkFBZ0IsQ0FBQztZQUN6RyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBQ2QsZ0VBQWdFO1FBQ2hFLE1BQU0scUJBQXFCLEdBQUcscUNBQXFDLEtBQUssR0FBRyxDQUFDLGVBQWUsQ0FBQyxJQUFJO2VBQzNGLHFDQUFxQyxJQUFJLEdBQUcsQ0FBQyxlQUFlLENBQUMsZUFBZSxDQUFDO1FBQ2xGLE1BQU0sV0FBVyxHQUFZLHFDQUFxQztZQUNoRSxDQUFDLENBQUMscUJBQXFCO1lBQ3ZCLCtFQUErRTtZQUMvRSwrQ0FBK0M7WUFDL0MsQ0FBQyxDQUFDLEtBQUssQ0FBQztRQUNWLCtFQUErRTtRQUMvRSx3RkFBd0Y7UUFDeEYsSUFBSSxNQUFNLENBQUMsT0FBTyxJQUFJLFdBQVcsRUFBRTtZQUNqQyxPQUFPLE1BQU0sQ0FBQztTQUNmO1FBRUQsTUFBTSxTQUFTLEdBQUcsSUFBSSxrQ0FBZSxDQUFDO1lBQ3BDLE9BQU8sRUFBRSxPQUFPLENBQUMsT0FBTztZQUN4QixTQUFTLEVBQUUsQ0FBQyxPQUFPLENBQUMsZ0JBQWdCLElBQUksT0FBTyxDQUFDLFlBQVksQ0FBQztZQUM3RCxVQUFVLEVBQUUsQ0FBQyxPQUFPLENBQUMsT0FBUSxDQUFDLGNBQWMsQ0FBQztTQUM5QyxDQUFDLENBQUM7UUFFSCxNQUFNLGNBQWMsR0FBRyxPQUFPLENBQUMsUUFBUSxDQUFDLG1CQUFtQixDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBRXZFLE9BQU8sSUFBSSxLQUFLLENBQUM7WUFDZixpQkFBaUIsRUFBRSxTQUFTO1lBQzVCLE9BQU87WUFDUCxnQkFBZ0IsRUFBRSxjQUFjLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLElBQUksT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsU0FBUztTQUNsSCxDQUFDLENBQUM7S0FDSjtJQUVEOzs7OztPQUtHO0lBQ0ksTUFBTSxDQUFDLGNBQWMsQ0FBQyxPQUFnQzs7Ozs7Ozs7OztRQUMzRCxNQUFNLFNBQVMsR0FBRyxJQUFJLGtDQUFlLENBQUM7WUFDcEMsT0FBTyxFQUFFLE9BQU8sQ0FBQyxPQUFPO1lBQ3hCLFNBQVMsRUFBRSxPQUFPLENBQUMsWUFBWTtTQUNoQyxDQUFDLENBQUM7UUFFSCxNQUFNLGdCQUFnQixHQUFHLE9BQU8sQ0FBQyxPQUFPLENBQUMsY0FBYyxDQUFDLG9CQUFvQixDQUFDLFNBQVMsQ0FBQyxDQUFDO1FBQ3hGLElBQUksQ0FBQyxnQkFBZ0IsQ0FBQyxjQUFjLEVBQUU7WUFDcEMsT0FBTyxJQUFJLEtBQUssQ0FBQyxFQUFFLGtCQUFrQixFQUFFLFNBQVMsRUFBRSxPQUFPLEVBQUUsQ0FBQyxDQUFDO1NBQzlEO1FBRUQsSUFBSSxDQUFDLGdCQUFnQixDQUFDLGdCQUFnQixFQUFFO1lBQ3RDLE1BQU0sSUFBSSxLQUFLLENBQUMsK0ZBQStGLENBQUMsQ0FBQztTQUNsSDtRQUVELE9BQU8sSUFBSSxLQUFLLENBQUMsRUFBRSxrQkFBa0IsRUFBRSxTQUFTLEVBQUUsT0FBTyxFQUFFLGdCQUFnQixFQUFFLGdCQUFnQixDQUFDLGdCQUFnQixFQUFFLENBQUMsQ0FBQztLQUNuSDtJQUVEOzs7Ozs7OztPQVFHO0lBQ0ksTUFBTSxDQUFDLHlCQUF5QixDQUFDLE9BQTJDOzs7Ozs7Ozs7O1FBQ2pGLE1BQU0sTUFBTSxHQUFHLEtBQUssQ0FBQyxjQUFjLENBQUM7WUFDbEMsR0FBRyxPQUFPO1lBQ1YsS0FBSyxFQUFFLE9BQU8sQ0FBQyxRQUFRO1NBQ3hCLENBQUMsQ0FBQztRQUVILE1BQU0sU0FBUyxHQUFHLElBQUksa0NBQWUsQ0FBQztZQUNwQyxPQUFPLEVBQUUsT0FBTyxDQUFDLE9BQU87WUFDeEIsU0FBUyxFQUFFLENBQUMsT0FBTyxDQUFDLGdCQUFnQixJQUFJLE9BQU8sQ0FBQyxZQUFZLENBQUM7WUFDN0QsVUFBVSxFQUFFLENBQUMsT0FBTyxDQUFDLHVCQUF1QixJQUFJLE9BQU8sQ0FBQyxPQUFRLENBQUMsY0FBYyxDQUFDO1NBQ2pGLENBQUMsQ0FBQztRQUVILE1BQU0sY0FBYyxHQUFHLE9BQU8sQ0FBQyxRQUFRLENBQUMsbUJBQW1CLENBQUMsU0FBUyxDQUFDLENBQUM7UUFDdkUsTUFBTSxrQkFBa0IsR0FBRyxjQUFjLENBQUMsY0FBYyxDQUFDLENBQUMsQ0FBQyxjQUFjLENBQUMsZ0JBQWdCLElBQUksT0FBTyxDQUFDLFFBQVEsQ0FBQyxDQUFDLENBQUMsU0FBUyxDQUFDO1FBRTNILE9BQU8sSUFBSSxLQUFLLENBQUM7WUFDZixrQkFBa0IsRUFBRSxTQUFTO1lBQzdCLGlCQUFpQixFQUFFLE1BQU0sQ0FBQyxpQkFBaUI7WUFDM0MsT0FBTztZQUNQLGdCQUFnQixFQUFFLGtCQUFrQixDQUFDLENBQUMsQ0FBQyxJQUFJLG1CQUFtQixDQUFDLE1BQU0sRUFBRSxrQkFBa0IsQ0FBQyxDQUFDLENBQUMsQ0FBQyxNQUFNO1NBQ3BHLENBQUMsQ0FBQztLQUNKO0lBRUQ7Ozs7Ozs7O09BUUc7SUFDSSxNQUFNLENBQUMsSUFBSSxDQUFDLE9BQW1CLEVBQUUsT0FBZTs7Ozs7Ozs7OztRQUNyRCxPQUFPLElBQUksS0FBSyxDQUFDO1lBQ2YsT0FBTyxFQUFFLEVBQUUsT0FBTyxFQUFFLE9BQU8sRUFBRSxFQUFFLEVBQUUsWUFBWSxFQUFFLEVBQUUsRUFBRTtTQUNwRCxDQUFDLENBQUM7S0FDSjtJQW9DRDs7T0FFRztJQUNILElBQVcsT0FBTztRQUNoQixPQUFPLElBQUksQ0FBQyxrQkFBa0IsS0FBSyxTQUFTLElBQUksSUFBSSxDQUFDLGlCQUFpQixLQUFLLFNBQVMsQ0FBQztLQUN0RjtJQUVEOztPQUVHO0lBQ0ksYUFBYTtRQUNsQixJQUFJLENBQUMsSUFBSSxDQUFDLE9BQU8sRUFBRTtZQUNqQixtQ0FBbUM7WUFDbkMsTUFBTSxJQUFJLEtBQUssQ0FBQyxHQUFHLGFBQWEsQ0FBQyxJQUFJLENBQUMsT0FBTyxDQUFDLDREQUE0RCxDQUFDLENBQUM7U0FDN0c7S0FDRjtJQUVEOzs7O09BSUc7SUFDSSxXQUFXLENBQUMsR0FBRyxVQUE0QjtRQUNoRCxLQUFLLE1BQU0sU0FBUyxJQUFJLFVBQVUsRUFBRTtZQUNsQyxTQUFTLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxJQUFJLENBQUMsQ0FBQztTQUNwQztLQUNGOztBQXhMSCxzQkF5TEM7OztBQUVELFNBQVMsYUFBYSxDQUFDLE9BQTJCO0lBQ2hELE9BQU8sb0JBQW9CLE9BQU8sQ0FBQyxPQUFPLGNBQWMsT0FBTyxDQUFDLE9BQU8sU0FBUyxPQUFPLENBQUMsWUFBWSxHQUFHLENBQUM7QUFDMUcsQ0FBQztBQTJDRDs7Ozs7O0dBTUc7QUFDSCxNQUFhLG1CQUFtQjtJQUM5QixZQUFZLEdBQUcsV0FBOEI7UUFDM0MsR0FBRyxDQUFDLGVBQWUsQ0FBQyxTQUFTLENBQUMsSUFBSSxFQUFFO1lBQ2xDLElBQUksZUFBZTtnQkFDakIsT0FBTyxLQUFLLENBQUMsU0FBUyxDQUFDLE1BQU0sQ0FBQyxLQUFLLENBQUMsRUFBRSxFQUFFLFdBQVcsQ0FBQyxHQUFHLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQyxHQUFHLENBQUMsZUFBZSxDQUFDLEdBQUcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDO1lBQzVHLENBQUM7U0FDRixDQUFDLENBQUM7S0FDSjs7QUFQSCxrREFRQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCAqIGFzIGNkayBmcm9tICdAYXdzLWNkay9jb3JlJztcbmltcG9ydCB7IFBvbGljeVN0YXRlbWVudCB9IGZyb20gJy4vcG9saWN5LXN0YXRlbWVudCc7XG5pbXBvcnQgeyBJR3JhbnRhYmxlLCBJUHJpbmNpcGFsIH0gZnJvbSAnLi9wcmluY2lwYWxzJztcblxuLyoqXG4gKiBCYXNpYyBvcHRpb25zIGZvciBhIGdyYW50IG9wZXJhdGlvblxuICpcbiAqL1xuZXhwb3J0IGludGVyZmFjZSBDb21tb25HcmFudE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHByaW5jaXBhbCB0byBncmFudCB0b1xuICAgKlxuICAgKiBAZGVmYXVsdCBpZiBwcmluY2lwYWwgaXMgdW5kZWZpbmVkLCBubyB3b3JrIGlzIGRvbmUuXG4gICAqL1xuICByZWFkb25seSBncmFudGVlOiBJR3JhbnRhYmxlO1xuXG4gIC8qKlxuICAgKiBUaGUgYWN0aW9ucyB0byBncmFudFxuICAgKi9cbiAgcmVhZG9ubHkgYWN0aW9uczogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFRoZSByZXNvdXJjZSBBUk5zIHRvIGdyYW50IHRvXG4gICAqL1xuICByZWFkb25seSByZXNvdXJjZUFybnM6IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGEgZ3JhbnQgb3BlcmF0aW9uXG4gKlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEdyYW50V2l0aFJlc291cmNlT3B0aW9ucyBleHRlbmRzIENvbW1vbkdyYW50T3B0aW9ucyB7XG4gIC8qKlxuICAgKiBUaGUgcmVzb3VyY2Ugd2l0aCBhIHJlc291cmNlIHBvbGljeVxuICAgKlxuICAgKiBUaGUgc3RhdGVtZW50IHdpbGwgYmUgYWRkZWQgdG8gdGhlIHJlc291cmNlIHBvbGljeSBpZiBpdCBjb3VsZG4ndCBiZVxuICAgKiBhZGRlZCB0byB0aGUgcHJpbmNpcGFsIHBvbGljeS5cbiAgICovXG4gIHJlYWRvbmx5IHJlc291cmNlOiBJUmVzb3VyY2VXaXRoUG9saWN5O1xuXG4gIC8qKlxuICAgKiBXaGVuIHJlZmVycmluZyB0byB0aGUgcmVzb3VyY2UgaW4gYSByZXNvdXJjZSBwb2xpY3ksIHVzZSB0aGlzIGFzIEFSTi5cbiAgICpcbiAgICogKERlcGVuZGluZyBvbiB0aGUgcmVzb3VyY2UgdHlwZSwgdGhpcyBuZWVkcyB0byBiZSAnKicgaW4gYSByZXNvdXJjZSBwb2xpY3kpLlxuICAgKlxuICAgKiBAZGVmYXVsdCBTYW1lIGFzIHJlZ3VsYXIgcmVzb3VyY2UgQVJOc1xuICAgKi9cbiAgcmVhZG9ubHkgcmVzb3VyY2VTZWxmQXJucz86IHN0cmluZ1tdO1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGEgZ3JhbnQgb3BlcmF0aW9uIHRoYXQgb25seSBhcHBsaWVzIHRvIHByaW5jaXBhbHNcbiAqXG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgR3JhbnRPblByaW5jaXBhbE9wdGlvbnMgZXh0ZW5kcyBDb21tb25HcmFudE9wdGlvbnMge1xuICAvKipcbiAgICogQ29uc3RydWN0IHRvIHJlcG9ydCB3YXJuaW5ncyBvbiBpbiBjYXNlIGdyYW50IGNvdWxkIG5vdCBiZSByZWdpc3RlcmVkXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIGNvbnN0cnVjdCBpbiB3aGljaCB0aGlzIGNvbnN0cnVjdCBpcyBkZWZpbmVkXG4gICAqL1xuICByZWFkb25seSBzY29wZT86IGNkay5JQ29uc3RydWN0O1xufVxuXG4vKipcbiAqIE9wdGlvbnMgZm9yIGEgZ3JhbnQgb3BlcmF0aW9uIHRvIGJvdGggaWRlbnRpdHkgYW5kIHJlc291cmNlXG4gKlxuICovXG5leHBvcnQgaW50ZXJmYWNlIEdyYW50T25QcmluY2lwYWxBbmRSZXNvdXJjZU9wdGlvbnMgZXh0ZW5kcyBDb21tb25HcmFudE9wdGlvbnMge1xuICAvKipcbiAgICogVGhlIHJlc291cmNlIHdpdGggYSByZXNvdXJjZSBwb2xpY3lcbiAgICpcbiAgICogVGhlIHN0YXRlbWVudCB3aWxsIGFsd2F5cyBiZSBhZGRlZCB0byB0aGUgcmVzb3VyY2UgcG9saWN5LlxuICAgKi9cbiAgcmVhZG9ubHkgcmVzb3VyY2U6IElSZXNvdXJjZVdpdGhQb2xpY3k7XG5cbiAgLyoqXG4gICAqIFdoZW4gcmVmZXJyaW5nIHRvIHRoZSByZXNvdXJjZSBpbiBhIHJlc291cmNlIHBvbGljeSwgdXNlIHRoaXMgYXMgQVJOLlxuICAgKlxuICAgKiAoRGVwZW5kaW5nIG9uIHRoZSByZXNvdXJjZSB0eXBlLCB0aGlzIG5lZWRzIHRvIGJlICcqJyBpbiBhIHJlc291cmNlIHBvbGljeSkuXG4gICAqXG4gICAqIEBkZWZhdWx0IFNhbWUgYXMgcmVndWxhciByZXNvdXJjZSBBUk5zXG4gICAqL1xuICByZWFkb25seSByZXNvdXJjZVNlbGZBcm5zPzogc3RyaW5nW107XG5cbiAgLyoqXG4gICAqIFRoZSBwcmluY2lwYWwgdG8gdXNlIGluIHRoZSBzdGF0ZW1lbnQgZm9yIHRoZSByZXNvdXJjZSBwb2xpY3kuXG4gICAqXG4gICAqIEBkZWZhdWx0IC0gdGhlIHByaW5jaXBhbCBvZiB0aGUgZ3JhbnRlZSB3aWxsIGJlIHVzZWRcbiAgICovXG4gIHJlYWRvbmx5IHJlc291cmNlUG9saWN5UHJpbmNpcGFsPzogSVByaW5jaXBhbDtcbn1cblxuLyoqXG4gKiBSZXN1bHQgb2YgYSBncmFudCgpIG9wZXJhdGlvblxuICpcbiAqIFRoaXMgY2xhc3MgaXMgbm90IGluc3RhbnRpYWJsZSBieSBjb25zdW1lcnMgb24gcHVycG9zZSwgc28gdGhhdCB0aGV5IHdpbGwgYmVcbiAqIHJlcXVpcmVkIHRvIGNhbGwgdGhlIEdyYW50IGZhY3RvcnkgZnVuY3Rpb25zLlxuICovXG5leHBvcnQgY2xhc3MgR3JhbnQgaW1wbGVtZW50cyBjZGsuSURlcGVuZGFibGUge1xuICAvKipcbiAgICogR3JhbnQgdGhlIGdpdmVuIHBlcm1pc3Npb25zIHRvIHRoZSBwcmluY2lwYWxcbiAgICpcbiAgICogVGhlIHBlcm1pc3Npb25zIHdpbGwgYmUgYWRkZWQgdG8gdGhlIHByaW5jaXBhbCBwb2xpY3kgcHJpbWFyaWx5LCBmYWxsaW5nXG4gICAqIGJhY2sgdG8gdGhlIHJlc291cmNlIHBvbGljeSBpZiBuZWNlc3NhcnkuIFRoZSBwZXJtaXNzaW9ucyBtdXN0IGJlIGdyYW50ZWRcbiAgICogc29tZXdoZXJlLlxuICAgKlxuICAgKiAtIFRyeWluZyB0byBncmFudCBwZXJtaXNzaW9ucyB0byBhIHByaW5jaXBhbCB0aGF0IGRvZXMgbm90IGFkbWl0IGFkZGluZyB0b1xuICAgKiAgIHRoZSBwcmluY2lwYWwgcG9saWN5IHdoaWxlIG5vdCBwcm92aWRpbmcgYSByZXNvdXJjZSB3aXRoIGEgcmVzb3VyY2UgcG9saWN5XG4gICAqICAgaXMgYW4gZXJyb3IuXG4gICAqIC0gVHJ5aW5nIHRvIGdyYW50IHBlcm1pc3Npb25zIHRvIGFuIGFic2VudCBwcmluY2lwYWwgKHBvc3NpYmxlIGluIHRoZVxuICAgKiAgIGNhc2Ugb2YgaW1wb3J0ZWQgcmVzb3VyY2VzKSBsZWFkcyB0byBhIHdhcm5pbmcgYmVpbmcgYWRkZWQgdG8gdGhlXG4gICAqICAgcmVzb3VyY2UgY29uc3RydWN0LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhZGRUb1ByaW5jaXBhbE9yUmVzb3VyY2Uob3B0aW9uczogR3JhbnRXaXRoUmVzb3VyY2VPcHRpb25zKTogR3JhbnQge1xuICAgIGNvbnN0IHJlc3VsdCA9IEdyYW50LmFkZFRvUHJpbmNpcGFsKHtcbiAgICAgIC4uLm9wdGlvbnMsXG4gICAgICBzY29wZTogb3B0aW9ucy5yZXNvdXJjZSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHJlc291cmNlQW5kUHJpbmNpcGFsQWNjb3VudENvbXBhcmlzb24gPSBvcHRpb25zLmdyYW50ZWUuZ3JhbnRQcmluY2lwYWwucHJpbmNpcGFsQWNjb3VudFxuICAgICAgPyBjZGsuVG9rZW4uY29tcGFyZVN0cmluZ3Mob3B0aW9ucy5yZXNvdXJjZS5lbnYuYWNjb3VudCwgb3B0aW9ucy5ncmFudGVlLmdyYW50UHJpbmNpcGFsLnByaW5jaXBhbEFjY291bnQpXG4gICAgICA6IHVuZGVmaW5lZDtcbiAgICAvLyBpZiBib3RoIGFjY291bnRzIGFyZSB0b2tlbnMsIHdlIGFzc3VtZSBoZXJlIHRoZXkgYXJlIHRoZSBzYW1lXG4gICAgY29uc3QgZXF1YWxPckJvdGhVbnJlc29sdmVkID0gcmVzb3VyY2VBbmRQcmluY2lwYWxBY2NvdW50Q29tcGFyaXNvbiA9PT0gY2RrLlRva2VuQ29tcGFyaXNvbi5TQU1FXG4gICAgICB8fCByZXNvdXJjZUFuZFByaW5jaXBhbEFjY291bnRDb21wYXJpc29uID09IGNkay5Ub2tlbkNvbXBhcmlzb24uQk9USF9VTlJFU09MVkVEO1xuICAgIGNvbnN0IHNhbWVBY2NvdW50OiBib29sZWFuID0gcmVzb3VyY2VBbmRQcmluY2lwYWxBY2NvdW50Q29tcGFyaXNvblxuICAgICAgPyBlcXVhbE9yQm90aFVucmVzb2x2ZWRcbiAgICAgIC8vIGlmIHRoZSBwcmluY2lwYWwgZG9lc24ndCBoYXZlIGFuIGFjY291bnQgKGZvciBleGFtcGxlLCBhIHNlcnZpY2UgcHJpbmNpcGFsKSxcbiAgICAgIC8vIHdlIHNob3VsZCBtb2RpZnkgdGhlIHJlc291cmNlJ3MgdHJ1c3QgcG9saWN5XG4gICAgICA6IGZhbHNlO1xuICAgIC8vIElmIHdlIGFkZGVkIHRvIHRoZSBwcmluY2lwYWwgQU5EIHdlJ3JlIGluIHRoZSBzYW1lIGFjY291bnQsIHRoZW4gd2UncmUgZG9uZS5cbiAgICAvLyBJZiBub3QsIGl0J3MgYSBkaWZmZXJlbnQgYWNjb3VudCBhbmQgd2UgbXVzdCBhbHNvIGFkZCBhIHRydXN0IHBvbGljeSBvbiB0aGUgcmVzb3VyY2UuXG4gICAgaWYgKHJlc3VsdC5zdWNjZXNzICYmIHNhbWVBY2NvdW50KSB7XG4gICAgICByZXR1cm4gcmVzdWx0O1xuICAgIH1cblxuICAgIGNvbnN0IHN0YXRlbWVudCA9IG5ldyBQb2xpY3lTdGF0ZW1lbnQoe1xuICAgICAgYWN0aW9uczogb3B0aW9ucy5hY3Rpb25zLFxuICAgICAgcmVzb3VyY2VzOiAob3B0aW9ucy5yZXNvdXJjZVNlbGZBcm5zIHx8IG9wdGlvbnMucmVzb3VyY2VBcm5zKSxcbiAgICAgIHByaW5jaXBhbHM6IFtvcHRpb25zLmdyYW50ZWUhLmdyYW50UHJpbmNpcGFsXSxcbiAgICB9KTtcblxuICAgIGNvbnN0IHJlc291cmNlUmVzdWx0ID0gb3B0aW9ucy5yZXNvdXJjZS5hZGRUb1Jlc291cmNlUG9saWN5KHN0YXRlbWVudCk7XG5cbiAgICByZXR1cm4gbmV3IEdyYW50KHtcbiAgICAgIHJlc291cmNlU3RhdGVtZW50OiBzdGF0ZW1lbnQsXG4gICAgICBvcHRpb25zLFxuICAgICAgcG9saWN5RGVwZW5kYWJsZTogcmVzb3VyY2VSZXN1bHQuc3RhdGVtZW50QWRkZWQgPyByZXNvdXJjZVJlc3VsdC5wb2xpY3lEZXBlbmRhYmxlID8/IG9wdGlvbnMucmVzb3VyY2UgOiB1bmRlZmluZWQsXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogVHJ5IHRvIGdyYW50IHRoZSBnaXZlbiBwZXJtaXNzaW9ucyB0byB0aGUgZ2l2ZW4gcHJpbmNpcGFsXG4gICAqXG4gICAqIEFic2VuY2Ugb2YgYSBwcmluY2lwYWwgbGVhZHMgdG8gYSB3YXJuaW5nLCBidXQgZmFpbGluZyB0byBhZGRcbiAgICogdGhlIHBlcm1pc3Npb25zIHRvIGEgcHJlc2VudCBwcmluY2lwYWwgaXMgbm90IGFuIGVycm9yLlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhZGRUb1ByaW5jaXBhbChvcHRpb25zOiBHcmFudE9uUHJpbmNpcGFsT3B0aW9ucyk6IEdyYW50IHtcbiAgICBjb25zdCBzdGF0ZW1lbnQgPSBuZXcgUG9saWN5U3RhdGVtZW50KHtcbiAgICAgIGFjdGlvbnM6IG9wdGlvbnMuYWN0aW9ucyxcbiAgICAgIHJlc291cmNlczogb3B0aW9ucy5yZXNvdXJjZUFybnMsXG4gICAgfSk7XG5cbiAgICBjb25zdCBhZGRlZFRvUHJpbmNpcGFsID0gb3B0aW9ucy5ncmFudGVlLmdyYW50UHJpbmNpcGFsLmFkZFRvUHJpbmNpcGFsUG9saWN5KHN0YXRlbWVudCk7XG4gICAgaWYgKCFhZGRlZFRvUHJpbmNpcGFsLnN0YXRlbWVudEFkZGVkKSB7XG4gICAgICByZXR1cm4gbmV3IEdyYW50KHsgcHJpbmNpcGFsU3RhdGVtZW50OiB1bmRlZmluZWQsIG9wdGlvbnMgfSk7XG4gICAgfVxuXG4gICAgaWYgKCFhZGRlZFRvUHJpbmNpcGFsLnBvbGljeURlcGVuZGFibGUpIHtcbiAgICAgIHRocm93IG5ldyBFcnJvcignQ29udHJhY3QgdmlvbGF0aW9uOiB3aGVuIFByaW5jaXBhbCByZXR1cm5zIHN0YXRlbWVudEFkZGVkPXRydWUsIGl0IHNob3VsZCByZXR1cm4gYSBkZXBlbmRhYmxlJyk7XG4gICAgfVxuXG4gICAgcmV0dXJuIG5ldyBHcmFudCh7IHByaW5jaXBhbFN0YXRlbWVudDogc3RhdGVtZW50LCBvcHRpb25zLCBwb2xpY3lEZXBlbmRhYmxlOiBhZGRlZFRvUHJpbmNpcGFsLnBvbGljeURlcGVuZGFibGUgfSk7XG4gIH1cblxuICAvKipcbiAgICogQWRkIGEgZ3JhbnQgYm90aCBvbiB0aGUgcHJpbmNpcGFsIGFuZCBvbiB0aGUgcmVzb3VyY2VcbiAgICpcbiAgICogQXMgbG9uZyBhcyBhbnkgcHJpbmNpcGFsIGlzIGdpdmVuLCBncmFudGluZyBvbiB0aGUgcHJpbmNpcGFsIG1heSBmYWlsIChpblxuICAgKiBjYXNlIG9mIGEgbm9uLWlkZW50aXR5IHByaW5jaXBhbCksIGJ1dCBncmFudGluZyBvbiB0aGUgcmVzb3VyY2Ugd2lsbFxuICAgKiBuZXZlciBmYWlsLlxuICAgKlxuICAgKiBTdGF0ZW1lbnQgd2lsbCBiZSB0aGUgcmVzb3VyY2Ugc3RhdGVtZW50LlxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBhZGRUb1ByaW5jaXBhbEFuZFJlc291cmNlKG9wdGlvbnM6IEdyYW50T25QcmluY2lwYWxBbmRSZXNvdXJjZU9wdGlvbnMpOiBHcmFudCB7XG4gICAgY29uc3QgcmVzdWx0ID0gR3JhbnQuYWRkVG9QcmluY2lwYWwoe1xuICAgICAgLi4ub3B0aW9ucyxcbiAgICAgIHNjb3BlOiBvcHRpb25zLnJlc291cmNlLFxuICAgIH0pO1xuXG4gICAgY29uc3Qgc3RhdGVtZW50ID0gbmV3IFBvbGljeVN0YXRlbWVudCh7XG4gICAgICBhY3Rpb25zOiBvcHRpb25zLmFjdGlvbnMsXG4gICAgICByZXNvdXJjZXM6IChvcHRpb25zLnJlc291cmNlU2VsZkFybnMgfHwgb3B0aW9ucy5yZXNvdXJjZUFybnMpLFxuICAgICAgcHJpbmNpcGFsczogW29wdGlvbnMucmVzb3VyY2VQb2xpY3lQcmluY2lwYWwgfHwgb3B0aW9ucy5ncmFudGVlIS5ncmFudFByaW5jaXBhbF0sXG4gICAgfSk7XG5cbiAgICBjb25zdCByZXNvdXJjZVJlc3VsdCA9IG9wdGlvbnMucmVzb3VyY2UuYWRkVG9SZXNvdXJjZVBvbGljeShzdGF0ZW1lbnQpO1xuICAgIGNvbnN0IHJlc291cmNlRGVwZW5kYWJsZSA9IHJlc291cmNlUmVzdWx0LnN0YXRlbWVudEFkZGVkID8gcmVzb3VyY2VSZXN1bHQucG9saWN5RGVwZW5kYWJsZSA/PyBvcHRpb25zLnJlc291cmNlIDogdW5kZWZpbmVkO1xuXG4gICAgcmV0dXJuIG5ldyBHcmFudCh7XG4gICAgICBwcmluY2lwYWxTdGF0ZW1lbnQ6IHN0YXRlbWVudCxcbiAgICAgIHJlc291cmNlU3RhdGVtZW50OiByZXN1bHQucmVzb3VyY2VTdGF0ZW1lbnQsXG4gICAgICBvcHRpb25zLFxuICAgICAgcG9saWN5RGVwZW5kYWJsZTogcmVzb3VyY2VEZXBlbmRhYmxlID8gbmV3IENvbXBvc2l0ZURlcGVuZGFibGUocmVzdWx0LCByZXNvdXJjZURlcGVuZGFibGUpIDogcmVzdWx0LFxuICAgIH0pO1xuICB9XG5cbiAgLyoqXG4gICAqIFJldHVybnMgYSBcIm5vLW9wXCIgYEdyYW50YCBvYmplY3Qgd2hpY2ggcmVwcmVzZW50cyBhIFwiZHJvcHBlZCBncmFudFwiLlxuICAgKlxuICAgKiBUaGlzIGNhbiBiZSB1c2VkIGZvciBlLmcuIGltcG9ydGVkIHJlc291cmNlcyB3aGVyZSB5b3UgbWF5IG5vdCBiZSBhYmxlIHRvIG1vZGlmeVxuICAgKiB0aGUgcmVzb3VyY2UncyBwb2xpY3kgb3Igc29tZSB1bmRlcmx5aW5nIHBvbGljeSB3aGljaCB5b3UgZG9uJ3Qga25vdyBhYm91dC5cbiAgICpcbiAgICogQHBhcmFtIGdyYW50ZWUgVGhlIGludGVuZGVkIGdyYW50ZWVcbiAgICogQHBhcmFtIF9pbnRlbnQgVGhlIHVzZXIncyBpbnRlbnQgKHdpbGwgYmUgaWdub3JlZCBhdCB0aGUgbW9tZW50KVxuICAgKi9cbiAgcHVibGljIHN0YXRpYyBkcm9wKGdyYW50ZWU6IElHcmFudGFibGUsIF9pbnRlbnQ6IHN0cmluZyk6IEdyYW50IHtcbiAgICByZXR1cm4gbmV3IEdyYW50KHtcbiAgICAgIG9wdGlvbnM6IHsgZ3JhbnRlZSwgYWN0aW9uczogW10sIHJlc291cmNlQXJuczogW10gfSxcbiAgICB9KTtcbiAgfVxuXG4gIC8qKlxuICAgKiBUaGUgc3RhdGVtZW50IHRoYXQgd2FzIGFkZGVkIHRvIHRoZSBwcmluY2lwYWwncyBwb2xpY3lcbiAgICpcbiAgICogQ2FuIGJlIGFjY2Vzc2VkIHRvIChlLmcuKSBhZGQgYWRkaXRpb25hbCBjb25kaXRpb25zIHRvIHRoZSBzdGF0ZW1lbnQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcHJpbmNpcGFsU3RhdGVtZW50PzogUG9saWN5U3RhdGVtZW50O1xuXG4gIC8qKlxuICAgKiBUaGUgc3RhdGVtZW50IHRoYXQgd2FzIGFkZGVkIHRvIHRoZSByZXNvdXJjZSBwb2xpY3lcbiAgICpcbiAgICogQ2FuIGJlIGFjY2Vzc2VkIHRvIChlLmcuKSBhZGQgYWRkaXRpb25hbCBjb25kaXRpb25zIHRvIHRoZSBzdGF0ZW1lbnQuXG4gICAqL1xuICBwdWJsaWMgcmVhZG9ubHkgcmVzb3VyY2VTdGF0ZW1lbnQ/OiBQb2xpY3lTdGF0ZW1lbnQ7XG5cbiAgLyoqXG4gICAqIFRoZSBvcHRpb25zIG9yaWdpbmFsbHkgdXNlZCB0byBzZXQgdGhpcyByZXN1bHRcbiAgICpcbiAgICogUHJpdmF0ZSBtZW1iZXIgZG91YmxlcyBhcyBhIHdheSB0byBtYWtlIGl0IGltcG9zc2libGUgZm9yIGFuIG9iamVjdCBsaXRlcmFsIHRvXG4gICAqIGJlIHN0cnVjdHVyYWxseSB0aGUgc2FtZSBhcyB0aGlzIGNsYXNzLlxuICAgKi9cbiAgcHJpdmF0ZSByZWFkb25seSBvcHRpb25zOiBDb21tb25HcmFudE9wdGlvbnM7XG5cbiAgcHJpdmF0ZSBjb25zdHJ1Y3Rvcihwcm9wczogR3JhbnRQcm9wcykge1xuICAgIHRoaXMub3B0aW9ucyA9IHByb3BzLm9wdGlvbnM7XG4gICAgdGhpcy5wcmluY2lwYWxTdGF0ZW1lbnQgPSBwcm9wcy5wcmluY2lwYWxTdGF0ZW1lbnQ7XG4gICAgdGhpcy5yZXNvdXJjZVN0YXRlbWVudCA9IHByb3BzLnJlc291cmNlU3RhdGVtZW50O1xuXG4gICAgY2RrLkRlcGVuZGFibGVUcmFpdC5pbXBsZW1lbnQodGhpcywge1xuICAgICAgZ2V0IGRlcGVuZGVuY3lSb290cygpIHtcbiAgICAgICAgcmV0dXJuIHByb3BzLnBvbGljeURlcGVuZGFibGUgPyBjZGsuRGVwZW5kYWJsZVRyYWl0LmdldChwcm9wcy5wb2xpY3lEZXBlbmRhYmxlKS5kZXBlbmRlbmN5Um9vdHMgOiBbXTtcbiAgICAgIH0sXG4gICAgfSk7XG4gIH1cblxuICAvKipcbiAgICogV2hldGhlciB0aGUgZ3JhbnQgb3BlcmF0aW9uIHdhcyBzdWNjZXNzZnVsXG4gICAqL1xuICBwdWJsaWMgZ2V0IHN1Y2Nlc3MoKTogYm9vbGVhbiB7XG4gICAgcmV0dXJuIHRoaXMucHJpbmNpcGFsU3RhdGVtZW50ICE9PSB1bmRlZmluZWQgfHwgdGhpcy5yZXNvdXJjZVN0YXRlbWVudCAhPT0gdW5kZWZpbmVkO1xuICB9XG5cbiAgLyoqXG4gICAqIFRocm93IGFuIGVycm9yIGlmIHRoaXMgZ3JhbnQgd2Fzbid0IHN1Y2Nlc3NmdWxcbiAgICovXG4gIHB1YmxpYyBhc3NlcnRTdWNjZXNzKCk6IHZvaWQge1xuICAgIGlmICghdGhpcy5zdWNjZXNzKSB7XG4gICAgICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgbWF4LWxlblxuICAgICAgdGhyb3cgbmV3IEVycm9yKGAke2Rlc2NyaWJlR3JhbnQodGhpcy5vcHRpb25zKX0gY291bGQgbm90IGJlIGFkZGVkIG9uIGVpdGhlciBpZGVudGl0eSBvciByZXNvdXJjZSBwb2xpY3kuYCk7XG4gICAgfVxuICB9XG5cbiAgLyoqXG4gICAqIE1ha2Ugc3VyZSB0aGlzIGdyYW50IGlzIGFwcGxpZWQgYmVmb3JlIHRoZSBnaXZlbiBjb25zdHJ1Y3RzIGFyZSBkZXBsb3llZFxuICAgKlxuICAgKiBUaGUgc2FtZSBhcyBjb25zdHJ1Y3Qubm9kZS5hZGREZXBlbmRlbmN5KGdyYW50KSwgYnV0IHNsaWdodGx5IG5pY2VyIHRvIHJlYWQuXG4gICAqL1xuICBwdWJsaWMgYXBwbHlCZWZvcmUoLi4uY29uc3RydWN0czogY2RrLklDb25zdHJ1Y3RbXSkge1xuICAgIGZvciAoY29uc3QgY29uc3RydWN0IG9mIGNvbnN0cnVjdHMpIHtcbiAgICAgIGNvbnN0cnVjdC5ub2RlLmFkZERlcGVuZGVuY3kodGhpcyk7XG4gICAgfVxuICB9XG59XG5cbmZ1bmN0aW9uIGRlc2NyaWJlR3JhbnQob3B0aW9uczogQ29tbW9uR3JhbnRPcHRpb25zKSB7XG4gIHJldHVybiBgUGVybWlzc2lvbnMgZm9yICcke29wdGlvbnMuZ3JhbnRlZX0nIHRvIGNhbGwgJyR7b3B0aW9ucy5hY3Rpb25zfScgb24gJyR7b3B0aW9ucy5yZXNvdXJjZUFybnN9J2A7XG59XG5cbmludGVyZmFjZSBHcmFudFByb3BzIHtcbiAgcmVhZG9ubHkgb3B0aW9uczogQ29tbW9uR3JhbnRPcHRpb25zO1xuICByZWFkb25seSBwcmluY2lwYWxTdGF0ZW1lbnQ/OiBQb2xpY3lTdGF0ZW1lbnQ7XG4gIHJlYWRvbmx5IHJlc291cmNlU3RhdGVtZW50PzogUG9saWN5U3RhdGVtZW50O1xuXG4gIC8qKlxuICAgKiBDb25zdHJ1Y3RzIHdob3NlIGRlcGxveW1lbnQgYXBwbGllcyB0aGUgZ3JhbnRcbiAgICpcbiAgICogVXNlZCB0byBhZGQgZGVwZW5kZW5jaWVzIG9uIGdyYW50c1xuICAgKi9cbiAgcmVhZG9ubHkgcG9saWN5RGVwZW5kYWJsZT86IGNkay5JRGVwZW5kYWJsZTtcbn1cblxuLyoqXG4gKiBBIHJlc291cmNlIHdpdGggYSByZXNvdXJjZSBwb2xpY3kgdGhhdCBjYW4gYmUgYWRkZWQgdG9cbiAqL1xuZXhwb3J0IGludGVyZmFjZSBJUmVzb3VyY2VXaXRoUG9saWN5IGV4dGVuZHMgY2RrLklSZXNvdXJjZSB7XG4gIC8qKlxuICAgKiBBZGQgYSBzdGF0ZW1lbnQgdG8gdGhlIHJlc291cmNlJ3MgcmVzb3VyY2UgcG9saWN5XG4gICAqL1xuICBhZGRUb1Jlc291cmNlUG9saWN5KHN0YXRlbWVudDogUG9saWN5U3RhdGVtZW50KTogQWRkVG9SZXNvdXJjZVBvbGljeVJlc3VsdDtcbn1cblxuLyoqXG4gKiBSZXN1bHQgb2YgY2FsbGluZyBhZGRUb1Jlc291cmNlUG9saWN5XG4gKi9cbmV4cG9ydCBpbnRlcmZhY2UgQWRkVG9SZXNvdXJjZVBvbGljeVJlc3VsdCB7XG4gIC8qKlxuICAgKiBXaGV0aGVyIHRoZSBzdGF0ZW1lbnQgd2FzIGFkZGVkXG4gICAqL1xuICByZWFkb25seSBzdGF0ZW1lbnRBZGRlZDogYm9vbGVhbjtcblxuICAvKipcbiAgICogRGVwZW5kYWJsZSB3aGljaCBhbGxvd3MgZGVwZW5kaW5nIG9uIHRoZSBwb2xpY3kgY2hhbmdlIGJlaW5nIGFwcGxpZWRcbiAgICpcbiAgICogQGRlZmF1bHQgLSBJZiBgc3RhdGVtZW50QWRkZWRgIGlzIHRydWUsIHRoZSByZXNvdXJjZSBvYmplY3QgaXRzZWxmLlxuICAgKiBPdGhlcndpc2UsIG5vIGRlcGVuZGFibGUuXG4gICAqL1xuICByZWFkb25seSBwb2xpY3lEZXBlbmRhYmxlPzogY2RrLklEZXBlbmRhYmxlO1xufVxuXG4vKipcbiAqIENvbXBvc2l0ZSBkZXBlbmRhYmxlXG4gKlxuICogTm90IGFzIHNpbXBsZSBhcyBlYWdlcmx5IGdldHRpbmcgdGhlIGRlcGVuZGVuY3kgcm9vdHMgZnJvbSB0aGVcbiAqIGlubmVyIGRlcGVuZGFibGVzLCBhcyB0aGV5IG1heSBiZSBtdXRhYmxlIHNvIHdlIG5lZWQgdG8gZGVmZXJcbiAqIHRoZSBxdWVyeS5cbiAqL1xuZXhwb3J0IGNsYXNzIENvbXBvc2l0ZURlcGVuZGFibGUgaW1wbGVtZW50cyBjZGsuSURlcGVuZGFibGUge1xuICBjb25zdHJ1Y3RvciguLi5kZXBlbmRhYmxlczogY2RrLklEZXBlbmRhYmxlW10pIHtcbiAgICBjZGsuRGVwZW5kYWJsZVRyYWl0LmltcGxlbWVudCh0aGlzLCB7XG4gICAgICBnZXQgZGVwZW5kZW5jeVJvb3RzKCk6IGNkay5JQ29uc3RydWN0W10ge1xuICAgICAgICByZXR1cm4gQXJyYXkucHJvdG90eXBlLmNvbmNhdC5hcHBseShbXSwgZGVwZW5kYWJsZXMubWFwKGQgPT4gY2RrLkRlcGVuZGFibGVUcmFpdC5nZXQoZCkuZGVwZW5kZW5jeVJvb3RzKSk7XG4gICAgICB9LFxuICAgIH0pO1xuICB9XG59XG4iXX0=