# -*- coding: utf-8 -*-
from setuptools import setup

packages = \
['eth_hentai']

package_data = \
{'': ['*'], 'eth_hentai': ['abi/*']}

install_requires = \
['psutil>=5.9.0,<6.0.0', 'web3[tester]>=5.26.0,<6.0.0']

extras_require = \
{':extra == "docs"': ['sphinx-autodoc-typehints[docs]>=1.16.0,<2.0.0'],
 'docs': ['Sphinx>=4.4.0,<5.0.0',
          'sphinx-rtd-theme>=1.0.0,<2.0.0',
          'sphinx-sitemap>=2.2.0,<3.0.0']}

setup_kwargs = {
    'name': 'eth-hentai',
    'version': '0.4',
    'description': 'Common Ethereum smart contracts, and related utilities, for developing automated test suites, backend integration and trading bots for EVM based blockchains.',
    'long_description': '[![PyPI version](https://badge.fury.io/py/eth-hentai.svg)](https://badge.fury.io/py/eth-hentai)\n\n[![Automated test suite](https://github.com/tradingstrategy-ai/eth-hentai/actions/workflows/tests.yml/badge.svg)](https://github.com/tradingstrategy-ai/eth-hentai/actions/workflows/tests.yml)\n\n[![Documentation Status](https://readthedocs.org/projects/smart-contracts-for-testing/badge/?version=latest)](https://smart-contracts-for-testing.readthedocs.io/en/latest/?badge=latest)\n\n# ETH-Hentai \n\nThis project contains common Ethereum smart contracts, and related utilities, \nfor developing automated test suites, backend integration and trading bots for EVM based blockchains.  \n\n[![ETH-Hentai](https://raw.githubusercontent.com/tradingstrategy-ai/eth-hentai/master/docs/source/_static/hentai_teacher_mikisugi_by_ilmaris_d6tjrn8-fullview.jpg)](https://www.deviantart.com/ilmaris)\n\nFeatures include \n\n* [Made for 99% developers](https://future.a16z.com/software-development-building-for-99-developers/)\n* [High-quality API documentation](https://smart-contracts-for-testing.readthedocs.io/)\n* [Fully type hinted](https://smart-contracts-for-testing.readthedocs.io/) for good developer experience\n* [ERC-20 token issuance and manipulation](https://eth-hentai.readthedocs.io/en/latest/_autosummary/eth_hentai.token.html#module-eth_hentai.token)\n* [Uniswap v2 tools](https://github.com/sushiswap/sushiswap): deployment, trading, price estimation for Sushiswap, PancakeSwape, QuickSwap, Trader Joe, others\n* [Parallel transaction execution](https://smart-contracts-for-testing.readthedocs.io/en/latest/_autosummary/eth_hentai.txmonitor.html)\n* [Mainnet forking with ganache-cli](https://smart-contracts-for-testing.readthedocs.io/en/latest/_autosummary/eth_hentai.ganache.fork_network.html#eth_hentai.ganache.fork_network)\n* As opposite to slower and messier [mainnet forking workflows](https://www.quicknode.com/guides/web3-sdks/how-to-fork-ethereum-blockchain-with-ganache), \nthis project aims to explicit clean deployments and very fast test execution.\n* (More integrations to come)\n\nTable of contents\n\n* [Precompiled ABI file distribution](#precompiled-abi-file-distribution)\n* [Python usage](#python-usage)\n   * [Prerequisites](#prerequisites)\n   * [ERC-20 token example](#erc-20-token-example)\n   * [Uniswap swap example](#uniswap-swap-example)\n   * [How to use hhe library in your Python project](#how-to-use-hhe-library-in-your-python-project)\n* [Development](#development)\n   * [Requires](#requires)\n   * [Make](#make)\n* [Version history](#version-history)\n* [Discord](#discord)\n* [Notes](#notes)\n* [License](#license)\n\n# Precompiled ABI file distribution\n\nThe project provides a precompiled smart contract bundle, including ABI files, full source\nand debug maps, to make deploying test instances trivial.\n\nThis package primarly supports Python, Web3.p3 and Brownie developers.\nFor other programming languages and frameworks,\nyou can [find precompiled Solidity smart contracts in abi folder](https://github.com/tradingstrategy-ai/eth-hentai/tree/master/eth_hentai/abi).\n\nThese files are good to go with any framework:\n* Web3.js\n* Ethers.js\n* Hardhat\n* Truffle\n* Web3j\n\nEach JSON file has `abi` and `bytecode` keys you need to deploy a contract.\n\nJust download and embed in your project. \nThe compiled source code files are mixture of MIT and GPL v2 license.\n\n# Python usage\n\nThe Python support is available as `eth-hentai` Python package.\n\nThe package depends only on [web3.py](github.com/ethereum/web3.py) and not others, like [Brownie](https://eth-brownie.readthedocs.io/).\nIt grabs popular ABI files with their bytecode and compilation artifacts so that the contracts\nare easily deployable on any Ethereum tester interface. No Ganache is needed and everything\ncan be executed on faster [eth-tester enginer](https://github.com/ethereum/eth-tester).\n\nUnlike Brownie, which is a framework, `eth_hentai` is a library. It is designed\nto be included in any other Python application and you can only use bits of its that you need.\nThere are no expectations on configuration files or folder structure.\n\n[Read the full API documentation](High-quality API documentation](https://eth-hentai.readthedocs.io/)).\nFor code examples please see below.\n\n## Prerequisites\n\n* [Proficient in Python programming](https://wiki.python.org/moin/BeginnersGuide)\n* [Understanding of Web3.py library](https://web3py.readthedocs.io/en/stable/) \n* [pytest basics](https://docs.pytest.org/)\n\n## ERC-20 token example\n\nTo use the package to deploy a simple ERC-20 token in [pytest](https://docs.pytest.org/) testing: \n\n```python\nimport pytest\nfrom web3 import Web3, EthereumTesterProvider\n\nfrom eth_hentai.token import create_token\n\n\n@pytest.fixture\ndef tester_provider():\n    return EthereumTesterProvider()\n\n\n@pytest.fixture\ndef eth_tester(tester_provider):\n    return tester_provider.ethereum_tester\n\n\n@pytest.fixture\ndef web3(tester_provider):\n    return Web3(tester_provider)\n\n\n@pytest.fixture()\ndef deployer(web3) -> str:\n    """Deploy account."""\n    return web3.eth.accounts[0]\n\n\n@pytest.fixture()\ndef user_1(web3) -> str:\n    """User account."""\n    return web3.eth.accounts[1]\n\n\n@pytest.fixture()\ndef user_2(web3) -> str:\n    """User account."""\n    return web3.eth.accounts[2]\n\n\ndef test_deploy_token(web3: Web3, deployer: str):\n    """Deploy mock ERC-20."""\n    token = create_token(web3, deployer, "Hentai books token", "HENTAI", 100_000 * 10**18)\n    assert token.functions.name().call() == "Hentai books token"\n    assert token.functions.symbol().call() == "HENTAI"\n    assert token.functions.totalSupply().call() == 100_000 * 10**18\n    assert token.functions.decimals().call() == 18\n\n\ndef test_tranfer_tokens_between_users(web3: Web3, deployer: str, user_1: str, user_2: str):\n    """Transfer tokens between users."""\n    token = create_token(web3, deployer, "Telos EVM rocks", "TELOS", 100_000 * 10**18)\n\n    # Move 10 tokens from deployer to user1\n    token.functions.transfer(user_1, 10 * 10**18).transact({"from": deployer})\n    assert token.functions.balanceOf(user_1).call() == 10 * 10**18\n\n    # Move 10 tokens from deployer to user1\n    token.functions.transfer(user_2, 6 * 10**18).transact({"from": user_1})\n    assert token.functions.balanceOf(user_1).call() == 4 * 10**18\n    assert token.functions.balanceOf(user_2).call() == 6 * 10**18\n```\n\n[See full example](https://github.com/tradingstrategy-ai/eth-hentai/blob/master/tests/test_token.py).\n\n[For more information how to user Web3.py in testing, see Web3.py documentation](https://web3py.readthedocs.io/en/stable/examples.html#contract-unit-tests-in-python).\n\n## Uniswap v2 trade example\n\n```python\nimport pytest\nfrom web3 import Web3\nfrom web3.contract import Contract\n\nfrom eth_hentai.uniswap_v2 import UniswapV2Deployment, deploy_trading_pair, FOREVER_DEADLINE\n\n\ndef test_swap(web3: Web3, deployer: str, user_1: str, uniswap_v2: UniswapV2Deployment, weth: Contract, usdc: Contract):\n    """User buys WETH on Uniswap v2 using mock USDC."""\n\n    # Create the trading pair and add initial liquidity\n    deploy_trading_pair(\n        web3,\n        deployer,\n        uniswap_v2,\n        weth,\n        usdc,\n        10 * 10**18,  # 10 ETH liquidity\n        17_000 * 10**18,  # 17000 USDC liquidity\n    )\n\n    router = uniswap_v2.router\n\n    # Give user_1 500 dollars to buy ETH and approve it on the router\n    usdc_amount_to_pay = 500 * 10**18\n    usdc.functions.transfer(user_1, usdc_amount_to_pay).transact({"from": deployer})\n    usdc.functions.approve(router.address, usdc_amount_to_pay).transact({"from": user_1})\n\n    # Perform a swap USDC->WETH\n    path = [usdc.address, weth.address]  # Path tell how the swap is routed\n    # https://docs.uniswap.org/protocol/V2/reference/smart-contracts/router-02#swapexacttokensfortokens\n    router.functions.swapExactTokensForTokens(\n        usdc_amount_to_pay,\n        0,\n        path,\n        user_1,\n        FOREVER_DEADLINE,\n    ).transact({\n        "from": user_1\n    })\n\n    # Check the user_1 received ~0.284 ethers\n    assert weth.functions.balanceOf(user_1).call() / 1e18 == pytest.approx(0.28488156127668085)\n```\n\n[See the full example](https://github.com/tradingstrategy-ai/eth-hentai/blob/master/tests/test_uniswap_v2_pair.py).\n\n## Uniswap v2 price estimation example\n\n```python\n# Create the trading pair and add initial liquidity\ndeploy_trading_pair(\n    web3,\n    deployer,\n    uniswap_v2,\n    weth,\n    usdc,\n    1_000 * 10**18,  # 1000 ETH liquidity\n    1_700_000 * 10**18,  # 1.7M USDC liquidity\n)\n\n# Estimate the price of buying 1 ETH\nusdc_per_eth = estimate_buy_price_decimals(\n    uniswap_v2,\n    weth.address,\n    usdc.address,\n    Decimal(1.0),\n)\nassert usdc_per_eth == pytest.approx(Decimal(1706.82216820632059904))\n```\n\n[See full example](https://github.com/tradingstrategy-ai/eth-hentai/blob/34a9e1b948a4ef5a620ee520b568c0fd27c75457/tests/test_uniswap_v2_pair.py#L243).\n\n## How to use the library in your Python project\n\nAdd `eth-hentai` as a development dependency:\n\nUsing [Poetry](https://python-poetry.org/):\n\n```shell\npoetry add -D eth-hentai\n```\n\n# Development\n\nThis step will extract compiled smart contract from Sushiswap repository. \n\n## Build requirements\n\nYou need to be able to compile Solidity code in `sushiswap` that is included as a submodule.\n\n* Node v14 \n* npx \n* yarn\n* GNU Make\n* Unix shell\n\n## Make\n\nTo build the ABI distribution:\n\n```shell\ngit submodule update --recursive --init\nmake all\n```\n\n[See SushiSwap continuous integration files for more information](https://github.com/sushiswap/sushiswap/blob/canary/.github/workflows/sushiswap.yml).\n\n# Version history\n\n[See change log](https://github.com/tradingstrategy-ai/eth-hentai/blob/master/CHANGELOG.md).\n\n# Discord\n\n[Join Discord for any questions](https://tradingstrategy.ai/community).\n\n# Notes\n\nCurrently there is no [Brownie](https://eth-brownie.readthedocs.io/) support.\nTo support Brownie, one would need to figure out how to import an existing Hardhat\nbased project (Sushiswap) to Brownie project format.\n\nCover art by [Ilmaris](https://www.deviantart.com/ilmaris).\n\n# History\n\n[Originally created for Trading Strategy](https://tradingstrategy.ai).\n\n# License \n\nMIT',
    'author': 'Mikko Ohtamaa',
    'author_email': 'mikko@capitalgram.com',
    'maintainer': None,
    'maintainer_email': None,
    'url': 'https://tradingstrategy.ai',
    'packages': packages,
    'package_data': package_data,
    'install_requires': install_requires,
    'extras_require': extras_require,
    'python_requires': '>=3.8,<4.0',
}


setup(**setup_kwargs)
