# -*- coding:utf-8 -*-
# !/usr/bin/env python
"""
Date: 2022/5/16 18:36
Desc: 新股和风险警示股
新浪-行情中心-沪深股市-次新股
http://vip.stock.finance.sina.com.cn/mkt/#new_stock
东方财富网-行情中心-沪深个股-风险警示板
http://quote.eastmoney.com/center/gridlist.html#st_board
"""
import math

import pandas as pd
import requests


def stock_zh_a_st_em() -> pd.DataFrame:
    """
    东方财富网-行情中心-沪深个股-风险警示板
    http://quote.eastmoney.com/center/gridlist.html#st_board
    :return: 风险警示板
    :rtype: pandas.DataFrame
    """
    url = 'http://40.push2.eastmoney.com/api/qt/clist/get'
    params = {
        'pn': '1',
        'pz': '2000',
        'po': '1',
        'np': '1',
        'ut': 'bd1d9ddb04089700cf9c27f6f7426281',
        'fltt': '2',
        'invt': '2',
        'fid': 'f3',
        'fs': 'm:0 f:4,m:1 f:4',
        'fields': 'f2,f3,f4,f5,f6,f7,f8,f9,f10,f12,f13,f14,f15,f16,f17,f18,f20,f21,f23,f24,f25,f22,f11,f62,f128,f136,f115,f152',
        '_': '1631107510188',
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df = pd.DataFrame(data_json['data']['diff'])
    temp_df.reset_index(inplace=True)
    temp_df['index'] = range(1, len(temp_df)+1)
    temp_df.columns = [
        '序号',
        '最新价',
        '涨跌幅',
        '涨跌额',
        '成交量',
        '成交额',
        '振幅',
        '换手率',
        '市盈率-动态',
        '量比',
        '_',
        '代码',
        '_',
        '名称',
        '最高',
        '最低',
        '今开',
        '昨收',
        '_',
        '_',
        '_',
        '市净率',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
    ]
    temp_df = temp_df[[
        '序号',
        '代码',
        '名称',
        '最新价',
        '涨跌幅',
        '涨跌额',
        '成交量',
        '成交额',
        '振幅',
        '最高',
        '最低',
        '今开',
        '昨收',
        '量比',
        '换手率',
        '市盈率-动态',
        '市净率',
    ]]
    temp_df['最新价'] = pd.to_numeric(temp_df['最新价'], errors="coerce")
    temp_df['涨跌幅'] = pd.to_numeric(temp_df['涨跌幅'], errors="coerce")
    temp_df['涨跌额'] = pd.to_numeric(temp_df['涨跌额'], errors="coerce")
    temp_df['成交量'] = pd.to_numeric(temp_df['成交量'], errors="coerce")
    temp_df['成交额'] = pd.to_numeric(temp_df['成交额'], errors="coerce")
    temp_df['振幅'] = pd.to_numeric(temp_df['振幅'], errors="coerce")
    temp_df['最高'] = pd.to_numeric(temp_df['最高'], errors="coerce")
    temp_df['最低'] = pd.to_numeric(temp_df['最低'], errors="coerce")
    temp_df['今开'] = pd.to_numeric(temp_df['今开'], errors="coerce")
    temp_df['量比'] = pd.to_numeric(temp_df['量比'], errors="coerce")
    temp_df['换手率'] = pd.to_numeric(temp_df['换手率'], errors="coerce")
    return temp_df


def stock_zh_a_new_em() -> pd.DataFrame:
    """
    东方财富网-行情中心-沪深个股-新股
    http://quote.eastmoney.com/center/gridlist.html#newshares
    :return: 新股
    :rtype: pandas.DataFrame
    """
    url = 'http://40.push2.eastmoney.com/api/qt/clist/get'
    params = {
        'pn': '1',
        'pz': '2000',
        'po': '1',
        'np': '1',
        'ut': 'bd1d9ddb04089700cf9c27f6f7426281',
        'fltt': '2',
        'invt': '2',
        'fid': 'f26',
        'fs': 'm:0 f:8,m:1 f:8',
        'fields': 'f2,f3,f4,f5,f6,f7,f8,f9,f10,f12,f13,f14,f15,f16,f17,f18,f20,f21,f23,f24,f25,f22,f11,f62,f128,f136,f115,f152',
        '_': '1631107510188',
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df = pd.DataFrame(data_json['data']['diff'])
    temp_df.reset_index(inplace=True)
    temp_df['index'] = range(1, len(temp_df)+1)
    temp_df.columns = [
        '序号',
        '最新价',
        '涨跌幅',
        '涨跌额',
        '成交量',
        '成交额',
        '振幅',
        '换手率',
        '市盈率-动态',
        '量比',
        '_',
        '代码',
        '_',
        '名称',
        '最高',
        '最低',
        '今开',
        '昨收',
        '_',
        '_',
        '_',
        '市净率',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
    ]
    temp_df = temp_df[[
        '序号',
        '代码',
        '名称',
        '最新价',
        '涨跌幅',
        '涨跌额',
        '成交量',
        '成交额',
        '振幅',
        '最高',
        '最低',
        '今开',
        '昨收',
        '量比',
        '换手率',
        '市盈率-动态',
        '市净率',
    ]]
    temp_df['最新价'] = pd.to_numeric(temp_df['最新价'], errors="coerce")
    temp_df['涨跌幅'] = pd.to_numeric(temp_df['涨跌幅'], errors="coerce")
    temp_df['涨跌额'] = pd.to_numeric(temp_df['涨跌额'], errors="coerce")
    temp_df['成交量'] = pd.to_numeric(temp_df['成交量'], errors="coerce")
    temp_df['成交额'] = pd.to_numeric(temp_df['成交额'], errors="coerce")
    temp_df['振幅'] = pd.to_numeric(temp_df['振幅'], errors="coerce")
    temp_df['最高'] = pd.to_numeric(temp_df['最高'], errors="coerce")
    temp_df['最低'] = pd.to_numeric(temp_df['最低'], errors="coerce")
    temp_df['今开'] = pd.to_numeric(temp_df['今开'], errors="coerce")
    temp_df['量比'] = pd.to_numeric(temp_df['量比'], errors="coerce")
    temp_df['换手率'] = pd.to_numeric(temp_df['换手率'], errors="coerce")
    return temp_df


def stock_zh_a_stop_em() -> pd.DataFrame:
    """
    东方财富网-行情中心-沪深个股-两网及退市
    http://quote.eastmoney.com/center/gridlist.html#staq_net_board
    :return: 两网及退市
    :rtype: pandas.DataFrame
    """
    url = 'http://40.push2.eastmoney.com/api/qt/clist/get'
    params = {
        'pn': '1',
        'pz': '2000',
        'po': '1',
        'np': '1',
        'ut': 'bd1d9ddb04089700cf9c27f6f7426281',
        'fltt': '2',
        'invt': '2',
        'fid': 'f3',
        'fs': 'm:0 s:3',
        'fields': 'f2,f3,f4,f5,f6,f7,f8,f9,f10,f12,f13,f14,f15,f16,f17,f18,f20,f21,f23,f24,f25,f22,f11,f62,f128,f136,f115,f152',
        '_': '1631107510188',
    }
    r = requests.get(url, params=params)
    data_json = r.json()
    temp_df = pd.DataFrame(data_json['data']['diff'])
    temp_df.reset_index(inplace=True)
    temp_df['index'] = range(1, len(temp_df)+1)
    temp_df.columns = [
        '序号',
        '最新价',
        '涨跌幅',
        '涨跌额',
        '成交量',
        '成交额',
        '振幅',
        '换手率',
        '市盈率-动态',
        '量比',
        '_',
        '代码',
        '_',
        '名称',
        '最高',
        '最低',
        '今开',
        '昨收',
        '_',
        '_',
        '_',
        '市净率',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
        '_',
    ]
    temp_df = temp_df[[
        '序号',
        '代码',
        '名称',
        '最新价',
        '涨跌幅',
        '涨跌额',
        '成交量',
        '成交额',
        '振幅',
        '最高',
        '最低',
        '今开',
        '昨收',
        '量比',
        '换手率',
        '市盈率-动态',
        '市净率',
    ]]
    temp_df['最新价'] = pd.to_numeric(temp_df['最新价'], errors="coerce")
    temp_df['涨跌幅'] = pd.to_numeric(temp_df['涨跌幅'], errors="coerce")
    temp_df['涨跌额'] = pd.to_numeric(temp_df['涨跌额'], errors="coerce")
    temp_df['成交量'] = pd.to_numeric(temp_df['成交量'], errors="coerce")
    temp_df['成交额'] = pd.to_numeric(temp_df['成交额'], errors="coerce")
    temp_df['振幅'] = pd.to_numeric(temp_df['振幅'], errors="coerce")
    temp_df['最高'] = pd.to_numeric(temp_df['最高'], errors="coerce")
    temp_df['最低'] = pd.to_numeric(temp_df['最低'], errors="coerce")
    temp_df['今开'] = pd.to_numeric(temp_df['今开'], errors="coerce")
    temp_df['量比'] = pd.to_numeric(temp_df['量比'], errors="coerce")
    temp_df['换手率'] = pd.to_numeric(temp_df['换手率'], errors="coerce")
    return temp_df


def stock_zh_a_new() -> pd.DataFrame:
    """
    新浪财经-行情中心-沪深股市-次新股
    http://vip.stock.finance.sina.com.cn/mkt/#new_stock
    :return: 次新股行情数据
    :rtype: pandas.DataFrame
    """
    url = "http://vip.stock.finance.sina.com.cn/quotes_service/api/json_v2.php/Market_Center.getHQNodeStockCount"
    params = {"node": "new_stock"}
    r = requests.get(url, params=params)
    total_page = math.ceil(int(r.json()) / 80)
    url = "http://vip.stock.finance.sina.com.cn/quotes_service/api/json_v2.php/Market_Center.getHQNodeData"
    big_df = pd.DataFrame()
    for page in range(1, total_page + 1):
        params = {
            "page": str(page),
            "num": "80",
            "sort": "symbol",
            "asc": "1",
            "node": "new_stock",
            "symbol": "",
            "_s_r_a": "page",
        }
        r = requests.get(url, params=params)
        r.encoding = "gb2312"
        data_json = r.json()
        temp_df = pd.DataFrame(data_json)
        big_df = pd.concat([big_df, temp_df], ignore_index=True)
    big_df = big_df[
        [
            "symbol",
            "code",
            "name",
            "open",
            "high",
            "low",
            "volume",
            "amount",
            "mktcap",
            "turnoverratio",
        ]
    ]
    big_df['open'] = pd.to_numeric(big_df['open'])
    big_df['high'] = pd.to_numeric(big_df['high'])
    big_df['low'] = pd.to_numeric(big_df['low'])
    return big_df


if __name__ == "__main__":
    stock_zh_a_st_em_df = stock_zh_a_st_em()
    print(stock_zh_a_st_em_df)

    stock_zh_a_new_em_df = stock_zh_a_new_em()
    print(stock_zh_a_new_em_df)

    stock_zh_a_stop_em_df = stock_zh_a_stop_em()
    print(stock_zh_a_stop_em_df)

    stock_zh_a_new_df = stock_zh_a_new()
    print(stock_zh_a_new_df)
