import os
from dotenv import load_dotenv
from evoagent.llms.llms import GeminiLLM
from evoagent.embeddings import GeminiEmbeddings
from evoagent.vector_stores import FaissVectorStore
from evoagent.tools.file_reader import DocumentReader
from typing import List
import textwrap

class RAGApplication:
    def __init__(self, chunk_size: int = 1000, chunk_overlap: int = 200):
        """
        Initialize RAG application with EvoAgent components.

        Args:
            chunk_size (int): Size of text chunks for processing
            chunk_overlap (int): Overlap between chunks
        """
        load_dotenv()

        # Initialize components
        self.llm = GeminiLLM("gemini-1.5-flash")
        self.embeddings = GeminiEmbeddings()
        self.reader = DocumentReader()

        # Get embedding dimension from a sample
        sample_embedding = self.embeddings.embed("Sample text").values
        self.vector_store = FaissVectorStore(embedding_dim=len(sample_embedding))

        self.chunk_size = chunk_size
        self.chunk_overlap = chunk_overlap

    def chunk_text(self, text: str) -> List[str]:
        """
        Split text into overlapping chunks.

        Args:
            text (str): Input text to chunk

        Returns:
            List[str]: List of text chunks
        """
        chunks = []
        start = 0

        while start < len(text):
            # Get chunk with overlap
            end = start + self.chunk_size
            chunk = text[start:end]

            # If not the first chunk, include overlap from previous chunk
            if start > 0:
                chunk = text[start - self.chunk_overlap:end]

            # Clean and add chunk
            chunk = chunk.strip()
            if chunk:
                chunks.append(chunk)

            start += self.chunk_size - self.chunk_overlap

        return chunks

    def index_document(self, file_path: str) -> None:
        """
        Read and index a document.

        Args:
            file_path (str): Path to the document
        """
        # Read document
        text = self.reader.read(file_path)

        # Split into chunks
        chunks = self.chunk_text(text)

        # Generate embeddings for chunks
        embeddings_list = [self.embeddings.embed(chunk).values for chunk in chunks]

        # Store in vector store
        self.vector_store.add_batch(chunks, embeddings_list)
        print(f"Indexed {len(chunks)} chunks from document")

    def generate_prompt(self, query: str, context: List[str]) -> str:
        """
        Generate prompt with retrieved context.

        Args:
            query (str): User query
            context (List[str]): Retrieved context passages

        Returns:
            str: Generated prompt
        """
        return f"""Use the following passages to answer the question. If you cannot answer the question based on the passages, please say "I cannot answer this question based on the provided information."

Relevant passages:
{'-' * 80}
{' '.join(context)}
{'-' * 80}

Question: {query}

Answer:"""

    def query(self, question: str, num_chunks: int = 3) -> str:
        """
        Query the RAG system.

        Args:
            question (str): User question
            num_chunks (int): Number of relevant chunks to retrieve

        Returns:
            str: Generated answer
        """
        # Generate embedding for question
        query_embedding = self.embeddings.embed(question).values

        # Retrieve relevant chunks
        relevant_chunks = self.vector_store.query(query_embedding, k=num_chunks)

        # Generate prompt with context
        prompt = self.generate_prompt(question, relevant_chunks)

        # Get response from LLM
        response = self.llm.generate(prompt)

        return response
