"""
mfriv module.  Contains the ModflowRiv class. Note that the user can access
the ModflowRiv class as `flopy.modflow.ModflowRiv`.

Additional information for this MODFLOW package can be found at the `Online
MODFLOW Guide
<http://water.usgs.gov/ogw/modflow/MODFLOW-2005-Guide/index.html?riv.htm>`_.

"""

from flopy.mbase import Package

class ModflowRiv(Package):
    """
    MODFLOW River Package Class.

    Parameters
    ----------
    model : model object
        The model object (of type :class:`flopy.modflow.mf.Modflow`) to which
        this package will be added.
    irivcb : int
        is a flag and a unit number. (the default is 0).
    layer_row_column_data : list of records
        In its most general form, this is a triple list of river records  The
        innermost list is the layer, row, column, and flux rate for a single
        river.  Then for a stress period, there can be a list of rivers.  Then
        for a simulation, there can be a separate list for each stress period.
        This gives the form of
            lrcd = [
                     [  #stress period 1
                       [l1, r1, c1, stg1, cond1, rbot1],
                       [l2, r2, c2, stg2, cond2, rbot2],
                       [l3, r3, c3, stg3, cond3, rbot3],
                       ],
                     [  #stress period 2
                       [l1, r1, c1, stg1, cond1, rbot1],
                       [l2, r2, c2, stg2, cond2, rbot2],
                       [l3, r3, c3, stg3, cond3, rbot3],
                       ], ...
                     [  #stress period kper
                       [l1, r1, c1, stg1, cond1, rbot1],
                       [l2, r2, c2, stg2, cond2, rbot2],
                       [l3, r3, c3, stg3, cond3, rbot3],
                       ],
                    ]
        Note that if there are not records in layer_row_column_Q, then the last
        group of rivers will apply until the end of the simulation.
    options : list of strings
        Package options. (default is None).
    naux : int
        number of auxiliary variables
    extension : string
        Filename extension (default is 'riv')
    unitnumber : int
        File unit number (default is 18).

    Attributes
    ----------
    mxactr : int
        Maximum number of river cells for a stress period.  This is calculated
        automatically by FloPy based on the information in
        layer_row_column_data.

    Methods
    -------

    See Also
    --------

    Notes
    -----
    Parameters are not supported in FloPy.

    Examples
    --------

    >>> import flopy
    >>> m = flopy.modflow.Modflow()
    >>> lrcd = [[[2, 3, 4, 15.6, 1050., -4]]]  #this river boundary will be
    >>>                                        #applied to all stress periods
    >>> riv = flopy.modflow.ModflowRiv(m, layer_row_column_data=lrcd)

    """
    def __init__(self, model, irivcb=0, layer_row_column_data=None,
                 extension ='riv', unitnumber=18, options=None, naux=0):
        """
        Package constructor.

        """
        # Call parent init to set self.parent, extension, name and unit number
        Package.__init__(self, model, extension, 'RIV', unitnumber)
        self.heading = '# RIV for MODFLOW, generated by Flopy.'
        self.url = 'riv.htm'
        self.irivcb = irivcb
        self.mxactr = 0
        self.mxactr, self.layer_row_column_data = \
            self.assign_layer_row_column_data(layer_row_column_data, 6+naux)
        self.np = 0
        if options is None:
            options = []
        self.options = options
        self.parent.add_package(self)

    def __repr__( self ):
        return 'River class'

    def ncells( self):
        # Returns the  maximum number of cells that have river (developed for MT3DMS SSM package)
        #print 'Function must be implemented properly for riv package'
        return self.mxactr

    def write_file(self):
        """
        Write the file.

        """
        f_riv = open(self.fn_path, 'w')
        f_riv.write('{0}\n'.format(self.heading))
        line = '{0:10d}{1:10d}'.format(self.mxactr, self.irivcb)
        for opt in self.options:
            line += ' ' + str(opt)
        line += '\n'
        f_riv.write(line)
        self.write_layer_row_column_data(f_riv, self.layer_row_column_data)
        f_riv.close()

    @staticmethod
    def load(f, model, nper=None, ext_unit_dict=None):
        """
        Load an existing package.

        Parameters
        ----------
        f : filename or file handle
            File to load.
        model : model object
            The model object (of type :class:`flopy.modflow.mf.Modflow`) to
            which this package will be added.
        nper : int
            The number of stress periods.  If nper is None, then nper will be
            obtained from the model object. (default is None).
        ext_unit_dict : dictionary, optional
            If the arrays in the file are specified using EXTERNAL,
            or older style array control records, then `f` should be a file
            handle.  In this case ext_unit_dict is required, which can be
            constructed using the function
            :class:`flopy.utils.mfreadnam.parsenamefile`.

        Returns
        -------
        rch : ModflowRiv object
            ModflowRiv object.

        Examples
        --------

        >>> import flopy
        >>> m = flopy.modflow.Modflow()
        >>> riv = flopy.modflow.mfriv.load('test.riv', m)

        """
        if type(f) is not file:
            filename = f
            f = open(filename, 'r')
        #dataset 0 -- header
        while True:
            line = f.readline()
            if line[0] != '#':
                break
        #--check for parameters
        if "parameter" in line.lower():
            raw = line.strip().split()
            assert int(raw[1]) == 0,"Parameters are not supported"
            line = f.readline()
        #dataset 2a
        t = line.strip().split()
        mxactr = int(t[0])
        irivcb = None
        irivcb = 0
        try:
            if int(t[1]) != 0:
                irivcb = 53
        except:
            pass

        options = []
        naux = 0
        if len(t) > 2:
            for toption in t[3:-1]:
                if toption.lower() is 'noprint':
                    options.append(toption)
                elif 'aux' in toption.lower():
                    naux += 1
                    options.append(toption)
        if nper is None:
            nrow, ncol, nlay, nper = model.get_nrow_ncol_nlay_nper()
        #read data for every stress period
        layer_row_column_data = []
        current = []
        nitems = 6 + naux
        for iper in xrange(nper):
            print "   loading rivers for kper {0:5d}".format(iper+1)
            line = f.readline()
            t = line.strip().split()
            itmp = int(t[0])
            if itmp == 0:
                current = []
            elif itmp > 0:
                for ibnd in xrange(itmp):
                    line = f.readline()
                    t = line.strip().split()
                    bnd = []
                    for jdx in xrange(nitems):
                        if jdx < 3:
                            bnd.append(int(t[jdx]))
                        else:
                            bnd.append(float(t[jdx]))
                    current.append(bnd)
            layer_row_column_data.append(current)
        riv = ModflowRiv(model, irivcb=irivcb,
                         layer_row_column_data=layer_row_column_data,
                         options=options, naux=naux)
        return riv
