"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.FullRegionS3CodeCommitBackup = void 0;
const path = require("path");
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const aws_s3_assets_1 = require("@aws-cdk/aws-s3-assets");
const core_1 = require("@aws-cdk/core");
const S3_BUCKET_ENV = 'SCRIPTS_BUCKET';
const S3_KEY_ENV = 'SCRIPTS_BUCKET_KEY';
class FullRegionS3CodeCommitBackup extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { backupBucket, schedule, repositoryNames = [], computeType } = props;
        const asset = new aws_s3_assets_1.Asset(this, 'ScriptsDirectory', {
            path: path.join(__dirname, '..', 'scripts'),
        });
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_2_0;
        this.backupProject = new aws_codebuild_1.Project(this, 'FullRegionBackupProject', {
            environment: {
                buildImage,
                computeType: computeType || buildImage.defaultComputeType,
            },
            environmentVariables: {
                BACKUP_BUCKET: {
                    value: backupBucket.bucketName,
                },
                REPOSITORY_NAMES: {
                    value: repositoryNames.join(' '),
                },
                [S3_BUCKET_ENV]: { value: asset.s3BucketName },
                [S3_KEY_ENV]: { value: asset.s3ObjectKey },
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                env: {
                    'git-credential-helper': 'yes',
                },
                phases: {
                    pre_build: {
                        commands: [
                            `echo "Downloading scripts from s3://\${${S3_BUCKET_ENV}}/\${${S3_KEY_ENV}}"`,
                            `aws s3 cp s3://\${${S3_BUCKET_ENV}}/\${${S3_KEY_ENV}} ./`,
                            `unzip ./$(basename \${${S3_KEY_ENV}})`,
                        ],
                    },
                    build: {
                        commands: ['chmod +x backup_codecommit.sh', './backup_codecommit.sh'],
                    },
                },
            }),
        });
        asset.grantRead(this.backupProject);
        backupBucket.grantPut(this.backupProject);
        this.backupProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: ['*'],
            actions: ['codecommit:BatchGet*', 'codecommit:Get*', 'codecommit:Describe*', 'codecommit:List*', 'codecommit:GitPull'],
        }));
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule,
            targets: [new aws_events_targets_1.CodeBuildProject(this.backupProject)],
        });
    }
    /**
     * Defines an event rule which triggers when a backup fails.
     */
    onBackupFailed(id, options) {
        return this.backupProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup starts.
     */
    onBackupStarted(id, options) {
        return this.backupProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup complets successfully.
     */
    onBackupSucceeded(id, options) {
        return this.backupProject.onBuildSucceeded(id, options);
    }
}
exports.FullRegionS3CodeCommitBackup = FullRegionS3CodeCommitBackup;
//# sourceMappingURL=data:application/json;base64,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