"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.S3CodeCommitBackup = void 0;
const aws_codebuild_1 = require("@aws-cdk/aws-codebuild");
const aws_events_1 = require("@aws-cdk/aws-events");
const aws_events_targets_1 = require("@aws-cdk/aws-events-targets");
const aws_iam_1 = require("@aws-cdk/aws-iam");
const core_1 = require("@aws-cdk/core");
class S3CodeCommitBackup extends core_1.Construct {
    constructor(scope, id, props) {
        super(scope, id);
        const { backupBucket, schedule, computeType, repository } = props;
        const { repositoryName, repositoryCloneUrlHttp, repositoryArn } = repository;
        const buildImage = aws_codebuild_1.LinuxBuildImage.STANDARD_2_0;
        this.backupProject = new aws_codebuild_1.Project(this, 'BackupProject', {
            environment: {
                buildImage,
                computeType: computeType || buildImage.defaultComputeType,
            },
            buildSpec: aws_codebuild_1.BuildSpec.fromObject({
                version: '0.2',
                env: {
                    'git-credential-helper': 'yes',
                },
                phases: {
                    pre_build: {
                        commands: [`echo "[===== Clone repository: ${repositoryName} =====]"`, `git clone "${repositoryCloneUrlHttp}"`],
                    },
                    build: {
                        commands: [
                            "dt=$(date -u '+%Y_%m_%d_%H_%M')",
                            `zipfile="${repositoryName}_backup_\${dt}_UTC.tar.gz"`,
                            `echo "Compressing repository: ${repositoryName} into file: \${zipfile} and uploading to S3 bucket: ${backupBucket.bucketName}/${repositoryName}"`,
                            `tar -zcvf "\${zipfile}" "${repositoryName}/"`,
                            `aws s3 cp "\${zipfile}" "s3://${backupBucket.bucketName}/${repositoryName}/\${zipfile}"`,
                        ],
                    },
                },
            }),
        });
        backupBucket.grantPut(this.backupProject);
        this.backupProject.addToRolePolicy(new aws_iam_1.PolicyStatement({
            resources: [repositoryArn],
            actions: ['codecommit:BatchGet*', 'codecommit:Get*', 'codecommit:Describe*', 'codecommit:List*', 'codecommit:GitPull'],
        }));
        new aws_events_1.Rule(this, 'ScheduleRule', {
            schedule,
            targets: [new aws_events_targets_1.CodeBuildProject(this.backupProject)],
        });
    }
    /**
     * Defines an event rule which triggers when a backup fails.
     */
    onBackupFailed(id, options) {
        return this.backupProject.onBuildFailed(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup starts.
     */
    onBackupStarted(id, options) {
        return this.backupProject.onBuildStarted(id, options);
    }
    /**
     * Defines an event rule which triggers when a backup complets successfully.
     */
    onBackupSucceeded(id, options) {
        return this.backupProject.onBuildSucceeded(id, options);
    }
}
exports.S3CodeCommitBackup = S3CodeCommitBackup;
//# sourceMappingURL=data:application/json;base64,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