import json
import pytest  # type: ignore

import csaf.cvss as cvss
from csaf.vulnerability import Vulnerability, Score

DATA = {
    'baseScore': 10.0,
    'baseSeverity': 'CRITICAL',
    'vectorString': 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H',
    'version': '3.1',
}
JSON = json.dumps(DATA)


def test_vuln_cvss31_minimal():
    vector_string = 'CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H'
    expected_value = (
        f'{{"version": "3.1", "vectorString": "{vector_string}", "attackVector": null,'
        ' "attackComplexity": null, "privilegesRequired": null, "userInteraction": null, "scope": null,'
        ' "confidentialityImpact": null, "integrityImpact": null, "availabilityImpact": null, "baseScore": 10.0,'
        ' "baseSeverity": "CRITICAL", "exploitCodeMaturity": null, "remediationLevel": null, "reportConfidence": null,'
        ' "temporalScore": null, "temporalSeverity": null, "confidentialityRequirement": null,'
        ' "integrityRequirement": null, "availabilityRequirement": null, "modifiedAttackVector": null,'
        ' "modifiedAttackComplexity": null, "modifiedPrivilegesRequired": null, "modifiedUserInteraction": null,'
        ' "modifiedScope": null, "modifiedConfidentialityImpact": null, "modifiedIntegrityImpact": null,'
        ' "modifiedAvailabilityImpact": null, "environmentalScore": null, "environmentalSeverity": null}'
    )
    c31 = cvss.CVSS31.parse_raw(JSON)
    assert c31.json() == expected_value
    assert c31.vector_string == vector_string

    json_lines = c31.json(indent=2).split('\n')
    json_rep_of_vs = [line for line in json_lines if 'vectorString' in line]
    assert len(json_rep_of_vs) == 1
    assert vector_string in json_rep_of_vs[0]
    assert '"vectorString":' in json_rep_of_vs[0]
    vul = Vulnerability(cve='CVE-2000-0001', scores=[Score(cvss_v3=c31, products=['sample'])])
    v_json = vul.json(exclude_unset=True, indent=2)
    expect_v = """\
{
  "cve": "CVE-2000-0001",
  "scores": [
    {
      "cvss_v3": {
        "version": "3.1",
        "vectorString": "CVSS:3.1/AV:N/AC:L/PR:N/UI:N/S:C/C:H/I:H/A:H",
        "baseScore": 10.0,
        "baseSeverity": "CRITICAL"
      },
      "products": [
        "sample"
      ]
    }
  ]
}"""
    assert v_json == expect_v
