#!/usr/bin/env python3

"""
Shims additional custom method http_range to SQLAalchemy's Query class
"""

import re

import cherrypy
import sqlalchemy.orm.query


def http_range(self, *, total=None, required=False, response=True, limit=None):
    """Check items range in request headers and return obtained numbers as range() object.

    total -> total number of rows that system holds for given resource, may be given as callback function
    required -> if set to true, and there's no range header, exception is thrown, otherwise
    [:] slice is returned
    response -> whether response headers should be generated by this function automatically
    """

    if 'Range' in cherrypy.request.headers:

        if callable(total):
            total = total()

        if total is None:
            total = self.count()

        rrange = re.match('items=([0-9]+)-([0-9]*)',
                          cherrypy.request.headers['Range'])
        if not rrange:
            raise cherrypy.HTTPError(400, 'Invalid range format')
        first, last = [int(a) if len(a) else None for a in rrange.groups()]

        if (total < first) or ((last is not None) and (first > last)):
            cherrypy.response.headers['Content-Range'] = 'items *'
            raise cherrypy.HTTPError(416, 'Requested range not satisfiable: (%d, %d)/%d' %
                                     (first, last, total))
        if limit and total > limit and ((last if last else total)-first) > limit:
            raise cherrypy.HTTPError(413, 'Maximum request range size is %d' % limit)

        if response:
            cherrypy.response.headers['Content-Range'] = 'items %d-%d/%d' % (
                first, min(total-1, last) if last is not None else (total-1), total)
            cherrypy.response.status = "206 Partial Content"
        # headers range is right inclusive unlike Python one:
        return self[slice(first, last+1 if last is not None else None)]
    else:
        if required:
            raise cherrypy.HTTPError(416, 'No range given')
        if limit and (total() if callable(total) else total) > limit:
            raise cherrypy.HTTPError(413, 'Maximum request range size is %d' % limit)
        return self[:]

sqlalchemy.orm.query.Query.http_range = http_range
