#!/usr/bin/env python3

"""Collection of helper functions.

Collection of helper functions that didn't find its place
somewhere else and are too specialized to go to core.env:

addLanugage: adds new language to the database
urlify: replaces national characters with latin transliteration
range_from_header: pareses cherrypy request Content-Range and sets 206 partial content
mail: simple wrapper around the smtplib
make_htaccess: produces .htaccess from config/htaccess.d
deprecated: function decorator to mark functions, uses warnings.warn_explicit()
"""

import functools
import json
import re
import sys
import uuid
import warnings

import cherrypy

from babel import Locale

from sqlalchemy.ext.compiler import compiles
from sqlalchemy.sql.expression import FromClause
from sqlalchemy.sql import column


def add_language(language, territory):
    """Add language definition to database and create initial landing page for it.

    @arg language two letter (mostly) ISO code of language
    @arg territory two letter code of country
    """

    from .db import db
    from .models import Language, SiteNode, ArticleNode, HeadNode

    locale = Locale(language)

    lang = db.query(Language).filter_by(iso_code=language).first()

    if not lang:

        lang = Language(name=locale.english_name,
                        native_name=locale.display_name,
                        iso_code=language,
                        long_code=language+'_'+territory+'.UTF-8')
        db.add(lang)
        db.flush()

    page = db.query(SiteNode).filter_by(placement='lang').filter_by(lang=language).first()

    if not page:
        page = ArticleNode(lang=language,
                           title=locale.display_name,
                           linkName=locale.display_name,
                           visible=True,
                           urlPart=language)
        page.parent = HeadNode.get()
        page.placement = 'lang'
        db.add(page)
        db.flush()

    return page, lang


def urlify(name):
    """Replace exotic characters with transliterated ones or “-” symbol."""
    import icu
    return re.sub(r'[^a-zA-Z0-9_]+', '-',
                  icu.Transliterator  # pylint: disable=no-member
                  .createInstance('Any-Latin; Latin-ASCII;',
                                  icu.UTransDirection.FORWARD)  # pylint: disable=no-member
                  .transliterate(name))


def parse_range_header(required=False):
    """
    Check items range in request headers and return obtained numbers as range() object.
    required -> if set to true, and there's no range header, exception is thrown, otherwise
    [:] slice is returned
    """

    if 'Range' in cherrypy.request.headers:
        rrange = re.match('items=([0-9]+)-([0-9]*)',
                          cherrypy.request.headers['Range'])
        if not rrange:
            raise cherrypy.HTTPError(400, 'Invalid range format')

        first, last = [int(a) if len(a) else None for a in rrange.groups()]

        # headers range is right inclusive unlike Python one:
        return slice(first, last+1 if last is not None else None)

    elif required:
        raise cherrypy.HTTPError(416, 'No range given')

    return slice(None)


def range_from_header(total, *, required=False, response=True, limit=None):
    """Check items range in request headers and return obtained numbers as range() object.

    total -> total number of rows that system holds for given resource
    required -> if set to true, and there's no range header, exception is thrown, otherwise
    [:] slice is returned
    response -> whether response headers should be generated by this function automatically
    """

    rrange = parse_range_header(required)

    if 'Range' in cherrypy.request.headers:
        first = rrange.start
        last = rrange.stop

        if last is not None:
            last = last - 1  # we need decrease back last value

        if (total < first) or ((last is not None) and (first > last)):
            cherrypy.response.headers['Content-Range'] = 'items *'
            raise cherrypy.HTTPError(416, 'Requested range not satisfiable: (%d, %d)/%d' %
                                     (first, last, total))
        if limit and total > limit and ((last if last else total)-first) > limit:
            raise cherrypy.HTTPError(413, 'Maximum request range size is %d' % limit)

        if response:
            cherrypy.response.headers['Content-Range'] = 'items %d-%d/%d' % (
                first, min(total-1, last) if last is not None else (total-1), total)
            cherrypy.response.status = "206 Partial Content"
    else:
        if limit and (total() if callable(total) else total) > limit:
            raise cherrypy.HTTPError(413, 'Maximum request range size is %d' % limit)
    return rrange


def mail(mailFrom, rcptTo, subject, msgPlain=None, msgHTML=None):
    """Send mail using smtplib."""

    import smtplib
    from email.mime.multipart import MIMEMultipart
    from email.mime.text import MIMEText

    msg = MIMEMultipart('alternative')
    msg['Subject'] = subject
    msg['From'] = mailFrom
    msg['To'] = rcptTo

    if msgPlain:
        msg.attach(MIMEText(msgPlain, 'plain', 'utf-8'))

    if msgHTML:
        msg.attach(MIMEText(msgHTML, 'html'))

    sconn = smtplib.SMTP('localhost')
    sconn.sendmail(mailFrom, rcptTo, msg.as_string())
    sconn.quit()


def make_htaccess():
    """Create http/.htaccess from fragments in config."""

    from .core import ROOTPATH

    srcdir = ROOTPATH / 'config/htaccess.d'

    if not srcdir.is_dir():
        raise Exception("config/htaccess.d not found. Exiting.")

    fmask = re.compile(r"^\d{2}-.*[^~]$")
    files = sorted((f for f in srcdir.iterdir() if fmask.match(f.name)))
    nfile = ROOTPATH / ('http/.htaccess.'+uuid.uuid4().hex)
    # output
    with nfile.open('w') as dst:
        for fname in files:
            try:
                with fname.open('r') as fh:
                    dst.write(re.sub('{%BASEPATH%}', str(ROOTPATH), fh.read())+'\n')
            except IOError:
                print("Warning: problem with parsing file %s" % fname.name, file=sys.stderr)
    nfile.rename(ROOTPATH / 'http/.htaccess')


def deprecated(func):
    """Decorator which can be used to mark functions
    as deprecated. It will result in a warning being emitted
    when the function is used."""

    @functools.wraps(func)
    def new_func(*args, **kwargs):
        """Wrapper function"""
        warnings.warn_explicit(
            "Call to deprecated function {}.".format(func.__name__),
            category=DeprecationWarning,
            filename=func.func_code.co_filename,
            lineno=func.func_code.co_firstlineno + 1
        )
        return func(*args, **kwargs)
    return new_func


def all_subclasses(bclass):
    """Iterate over all, possibly nested subclasses."""
    yield bclass
    for subclass in bclass.__subclasses__():
        for sclass in all_subclasses(subclass):
            yield sclass


class values(FromClause):
    """ simple extension for SQLA providing values syntax in queries"""
    def __init__(self, *args):
        self.list = args

    def _populate_column_collection(self):
        self._columns.update([("column%d" % i, column("column%d" % i))
                              for i in range(1, len(self.list[0]) + 1)])


@compiles(values)
def compile_values(element, compiler, asfrom=False, **kwargs):
    """produces query representation for values object"""
    def __compiler_process(obj):
        if hasattr(obj, "_compiler_dispatch"):
            return compiler.process(obj)
        return repr(obj)

    v = "VALUES %s" % ", ".join("(%s)" % ", ".join(__compiler_process(e) for e in t) for t in element.list)
    if asfrom:
        v = "(%s)" % v
    return v


def structural(func):
    """ decorator that add __html__ method to regular function output
        so Chameleon treats it as structural one"""

    class _exstr(str):
        """str structural wrapper"""
        def __html__(self):
            return self

    @functools.wraps(func)
    def new_func(*args, **kwargs):
        """ decorator that add __html__ method to regular function output
            so Chameleon treats it as structural one"""
        response = _exstr(func(*args, **kwargs))
        return response
    return new_func


def parse_decorated_json(buffer):
    """like json.loads, but ignores line comments and jsonp wrapper
    """
    # ignore // line comments
    lineComment = re.compile(r'^\s*\/\/.*$', re.MULTILINE)
    buffer = lineComment.sub('', buffer)

    # and var dummy = { ... };
    dummyvar = re.compile(r'^\s*var\s+[a-z]+\s*=\s*(.*)\s*;\s*$', re.S)
    matched = dummyvar.match(buffer)
    if matched:
        buffer = matched.group(1)

    return json.loads(buffer)


def decamelize(text):
    """produce underscore separated string from camelized one"""
    return re.sub('([a-z0-9])([A-Z])', r'\1_\2',
                  re.sub('(.)([A-Z][a-z]+)', r'\1_\2', text))\
             .lower()


class memoize(object):
    '''This decorator allows you to create a property which can be computed once and
    accessed many times. Sort of like memoization.
    From the Python Cookbook (Denis Otkidach)
    '''
    def __init__(self, method, name=None):
        # record the unbound-method and the name
        self.method = method
        self.name = name or method.__name__
        self.__doc__ = method.__doc__

    def __get__(self, inst, cls):
        # self: <__main__.cache object at 0xb781340c>
        # inst: <__main__.Foo object at 0xb781348c>
        # cls: <class '__main__.Foo'>
        if inst is None:
            # instance attribute accessed on class, return self
            # You get here if you write `Foo.bar`
            return self
        # compute, cache and return the instance's attribute value
        result = self.method(inst)
        # setattr redefines the instance's attribute so this doesn't get called again
        setattr(inst, self.name, result)
        return result


def is_sequence(arg):
    """ simple check if arg is iterable, but not a string,
    more performant than using duck typing and exception handling
    """
    return (not hasattr(arg, "strip") and
            (hasattr(arg, "__getitem__") or
             hasattr(arg, "__iter__")))


def escape_pcre(text):

    """Escape potential regex symbols"""

    chars = '\\.^$*+?()[{| '
    for ch in chars:
        text = text.replace(ch, '\\'+ch)
    return text


def escape_ldap(text, escape='\\*()\0'):
    """Escape all characters potentially harmful for LDAP query."""
    for char in escape:
        text = text.replace(char, ''.join('\\%02x' % c for c in char.encode()))
    return text


def is_ldap_safe(text, chars='()*\0', strict=True):
    """Checks whether input may be passed as ldap filter literal without escaping"""
    import re
    if not type(text) == ((bytes != str) and str or unicode): #requires str for Py3 or unicode for Py2
          raise ValueError('unicode input expected')
    return (all(c not in text for c in chars) and
            not re.search('\\\\([^0-9a-fA-F]|(.[^0-9a-fA-F]))', text))


class Bunch(dict):
    """Basic class behaving both as dictionary and attributes
    collection"""

    def __init__(self, **kw):
        dict.__init__(self, kw)
        self.__dict__ = self

    def __getstate__(self):
        return self

    def __setstate__(self, state):
        self.update(state)
        self.__dict__ = self
