"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.IdentitySource = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const aws_verifiedpermissions_1 = require("aws-cdk-lib/aws-verifiedpermissions");
const core_1 = require("aws-cdk-lib/core");
class IdentitySourceBase extends core_1.Resource {
}
class IdentitySource extends IdentitySourceBase {
    /**
     * Create an Identity Source from its ARN
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param identitySourceArn The Identity Source ARN.
     */
    static fromIdentitySourceArn(scope, id, identitySourceArn) {
        return IdentitySource.fromIdentitySourceAttributes(scope, id, {
            identitySourceArn,
        });
    }
    /**
     * Creates Identity Source from its attributes
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param attrs An `IdentitySourceAttributes` object.
     */
    static fromIdentitySourceAttributes(scope, id, attrs) {
        class Import extends IdentitySourceBase {
            constructor(identitySourceArn, identitySourceId) {
                super(scope, id);
                this.identitySourceArn = identitySourceArn;
                this.identitySourceId = identitySourceId;
            }
        }
        let identitySourceArn;
        let identitySourceId;
        const stack = core_1.Stack.of(scope);
        if (!attrs.identitySourceId) {
            if (!attrs.identitySourceArn) {
                throw new Error('One of identitySourceId or identitySourceArn is required!');
            }
            identitySourceArn = attrs.identitySourceArn;
            const maybeId = stack.splitArn(attrs.identitySourceArn, core_1.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
            if (!maybeId) {
                throw new Error(`ARN for IdentitySource must be in the form: ${core_1.ArnFormat.SLASH_RESOURCE_NAME}`);
            }
            identitySourceId = maybeId;
        }
        else {
            if (attrs.identitySourceArn) {
                throw new Error('Only one of identitySourceArn or identitySourceId can be provided');
            }
            identitySourceId = attrs.identitySourceId;
            identitySourceArn = stack.formatArn({
                resource: 'identity-source',
                resourceName: attrs.identitySourceId,
                service: 'verifiedpermissions',
            });
        }
        return new Import(identitySourceArn, identitySourceId);
    }
    /**
     * Create an Identity Source from its identifier
     *
     * @param scope The parent creating construct (usually `this`).
     * @param id The construct's name.
     * @param identitySourceId The Identity Source identifier.
     */
    static fromIdentitySourceId(scope, id, identitySourceId) {
        return IdentitySource.fromIdentitySourceAttributes(scope, id, {
            identitySourceId,
        });
    }
    constructor(scope, id, props) {
        super(scope, id);
        this.clientIds =
            props.configuration.cognitoUserPoolConfiguration.clientIds ?? [];
        this.userPoolArn =
            props.configuration.cognitoUserPoolConfiguration.userPool.userPoolArn;
        this.identitySource = new aws_verifiedpermissions_1.CfnIdentitySource(this, id, {
            configuration: {
                cognitoUserPoolConfiguration: {
                    clientIds: core_1.Lazy.list({ produce: () => this.clientIds }),
                    userPoolArn: this.userPoolArn,
                },
            },
            policyStoreId: props.policyStore?.policyStoreId,
            principalEntityType: props.principalEntityType,
        });
        this.discoveryUrl = this.identitySource.attrDetailsDiscoveryUrl;
        this.identitySourceId = this.identitySource.attrIdentitySourceId;
        this.identitySourceArn = this.stack.formatArn({
            resource: 'identity-source',
            resourceName: this.identitySourceId,
            service: 'verifiedpermissions',
        });
        this.openIdIssuer = this.identitySource.attrDetailsOpenIdIssuer;
        this.policyStore = props.policyStore;
    }
    /**
     * Add a User Pool Client
     *
     * @param userPoolClient The User Pool Client Construct.
     */
    addUserPoolClient(userPoolClient) {
        this.clientIds.push(userPoolClient.userPoolClientId);
    }
}
exports.IdentitySource = IdentitySource;
_a = JSII_RTTI_SYMBOL_1;
IdentitySource[_a] = { fqn: "@cdklabs/cdk-verified-permissions.IdentitySource", version: "0.0.1-alpha.3" };
//# sourceMappingURL=data:application/json;base64,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