/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.models.earth.atmosphere;

import org.hipparchus.Field;
import org.hipparchus.RealFieldElement;
import org.hipparchus.exception.Localizable;
import org.hipparchus.geometry.euclidean.threed.FieldVector3D;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.MathArrays;
import org.orekit.bodies.BodyShape;
import org.orekit.bodies.FieldGeodeticPoint;
import org.orekit.bodies.GeodeticPoint;
import org.orekit.errors.OrekitException;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.models.earth.atmosphere.Atmosphere;
import org.orekit.models.earth.atmosphere.JB2008InputParameters;
import org.orekit.time.AbsoluteDate;
import org.orekit.time.FieldAbsoluteDate;
import org.orekit.utils.PVCoordinatesProvider;

public class JB2008
implements Atmosphere {
    private static final long serialVersionUID = -4201270765122160831L;
    private static final double ALT_MIN = 90000.0;
    private static final double EARTH_RADIUS = 6356.766;
    private static final double LOG10 = FastMath.log(10.0);
    private static final double AVOGAD = 6.02257E26;
    private static final double RSTAR = 8.31432;
    private static final double[] ALPHA = new double[]{0.0, 0.0, 0.0, 0.0, -0.38};
    private static final double[] AMW = new double[]{28.0134, 31.9988, 15.9994, 39.948, 4.0026, 1.00797};
    private static final double[] FRAC = new double[]{0.7811, 0.20955, 0.00934, 1.289E-5};
    private static final double R1 = 0.01;
    private static final double R2 = 0.025;
    private static final double R3 = 0.075;
    private static final double[] WT = new double[]{0.311111111111111, 1.422222222222222, 0.533333333333333, 1.422222222222222, 0.311111111111111};
    private static final double[] CHT = new double[]{0.22, -0.002, 0.00115, -2.11E-6};
    private static final double[] FZM = new double[]{0.2689, -0.01176, 0.02782, -0.02782, 3.47E-4};
    private static final double[] GTM = new double[]{-0.3633, 0.08506, 0.2401, -0.1897, -0.2554, -0.0179, 5.65E-4, -6.407E-4, -0.003418, -0.001252};
    private static final double[] CMB = new double[]{28.15204, -0.085586, 1.284E-4, -1.0056E-5, -1.021E-5, 1.5044E-6, 9.9826E-8};
    private static final double[] BDTC = new double[]{-4.57512297, -5.12114909, -69.3003609, 203.716701, 703.316291, -1943.49234, 1106.51308, -174.378996, 1885.94601, -7093.71517, 9224.54523, -3845.08073, -6.45841789, 40.9703319, -482.00656, 1818.70931, -2373.89204, 996.703815, 36.1416936};
    private static final double[] CDTC = new double[]{-15.5986211, -5.12114909, -69.3003609, 203.716701, 703.316291, -1943.49234, 1106.51308, -220.835117, 1432.56989, -3184.81844, 3289.81513, -1353.32119, 19.9956489, -12.7093998, 21.2825156, -2.75555432, 11.0234982, 148.881951, -751.640284, 637.876542, 12.7093998, -21.2825156, 2.75555432};
    private PVCoordinatesProvider sun;
    private JB2008InputParameters inputParams;
    private BodyShape earth;

    public JB2008(JB2008InputParameters parameters, PVCoordinatesProvider sun, BodyShape earth) {
        this.earth = earth;
        this.sun = sun;
        this.inputParams = parameters;
    }

    @Override
    public Frame getFrame() {
        return this.earth.getBodyFrame();
    }

    public double getDensity(double dateMJD, double sunRA, double sunDecli, double satLon, double satLat, double satAlt, double f10, double f10B, double s10, double s10B, double xm10, double xm10B, double y10, double y10B, double dstdtc) {
        double[] tc;
        double solarTime;
        if (satAlt < 90000.0) {
            throw new OrekitException((Localizable)OrekitMessages.ALTITUDE_BELOW_ALLOWED_THRESHOLD, satAlt, 90000.0);
        }
        double altKm = satAlt / 1000.0;
        double fn = FastMath.min(1.0, FastMath.pow(f10B / 240.0, 0.25));
        double fsb = f10B * fn + s10B * (1.0 - fn);
        double tsubc = 392.4 + 3.227 * fsb + 0.298 * (f10 - f10B) + 2.259 * (s10 - s10B) + 0.312 * (xm10 - xm10B) + 0.178 * (y10 - y10B);
        double eta = 0.5 * FastMath.abs(satLat - sunDecli);
        double theta = 0.5 * FastMath.abs(satLat + sunDecli);
        double h = satLon - sunRA;
        double tau = h - 0.64577182 + 0.10471976 * FastMath.sin(h + 0.75049158);
        for (solarTime = FastMath.toDegrees(h + Math.PI) / 15.0; solarTime >= 24.0; solarTime -= 24.0) {
        }
        while (solarTime < 0.0) {
            solarTime += 24.0;
        }
        double cosEta = FastMath.pow(FastMath.cos(eta), 2.5);
        double sinTeta = FastMath.pow(FastMath.sin(theta), 2.5);
        double cosTau = FastMath.abs(FastMath.cos(0.5 * tau));
        double df = sinTeta + (cosEta - sinTeta) * cosTau * cosTau * cosTau;
        double tsubl = tsubc * (1.0 + 0.31 * df);
        double dtclst = JB2008.dTc(f10, solarTime, satLat, altKm);
        double[] temp = new double[2];
        temp[0] = tsubl + dstdtc;
        double tinf = temp[0] + dtclst;
        double tsubx = 444.3807 + 0.02385 * tinf - 392.8292 * FastMath.exp(-0.0021357 * tinf);
        double gsubx = 0.054285714 * (tsubx - 183.0);
        tc = new double[]{tsubx, gsubx, (tinf - tsubx) * 2.0 / Math.PI, gsubx / tc[2]};
        double z1 = 90.0;
        double z2 = FastMath.min(altKm, 105.0);
        double al = FastMath.log(z2 / 90.0);
        int n = 1 + (int)FastMath.floor(al / 0.01);
        double zr = FastMath.exp(al / (double)n);
        double mb1 = JB2008.mBar(90.0);
        double tloc1 = JB2008.localTemp(90.0, tc);
        double zend = 90.0;
        double sub2 = 0.0;
        double ain = mb1 * JB2008.gravity(90.0) / tloc1;
        double mb2 = 0.0;
        double tloc2 = 0.0;
        double z = 0.0;
        double gravl = 0.0;
        for (int i = 0; i < n; ++i) {
            z = zend;
            zend = zr * z;
            double dz = 0.25 * (zend - z);
            double sum1 = WT[0] * ain;
            for (int j = 1; j < 5; ++j) {
                mb2 = JB2008.mBar(z += dz);
                tloc2 = JB2008.localTemp(z, tc);
                gravl = JB2008.gravity(z);
                ain = mb2 * gravl / tloc2;
                sum1 += WT[j] * ain;
            }
            sub2 += dz * sum1;
        }
        double rho = 3.46E-6 * mb2 * tloc1 / FastMath.exp(sub2 / 8.31432) / (mb1 * tloc2);
        double anm = 6.02257E26 * rho;
        double an = anm / mb2;
        double fact2 = anm / 28.96;
        double[] aln = new double[6];
        aln[0] = FastMath.log(FRAC[0] * fact2);
        aln[3] = FastMath.log(FRAC[2] * fact2);
        aln[4] = FastMath.log(FRAC[3] * fact2);
        aln[1] = FastMath.log(fact2 * (1.0 + FRAC[1]) - an);
        aln[2] = FastMath.log(2.0 * (an - fact2));
        if (altKm <= 105.0) {
            temp[1] = tloc2;
            aln[5] = aln[4] - 25.0;
        } else {
            double hSign;
            double altr;
            al = FastMath.log(FastMath.min(altKm, 500.0) / z);
            n = 1 + (int)FastMath.floor(al / 0.025);
            zr = FastMath.exp(al / (double)n);
            sub2 = 0.0;
            ain = gravl / tloc2;
            double tloc3 = 0.0;
            for (int i = 0; i < n; ++i) {
                z = zend;
                zend = zr * z;
                double dz = 0.25 * (zend - z);
                double sum1 = WT[0] * ain;
                for (int j = 1; j < 5; ++j) {
                    tloc3 = JB2008.localTemp(z += dz, tc);
                    gravl = JB2008.gravity(z);
                    ain = gravl / tloc3;
                    sum1 += WT[j] * ain;
                }
                sub2 += dz * sum1;
            }
            al = FastMath.log(FastMath.max(altKm, 500.0) / z);
            double r = altKm > 500.0 ? 0.075 : 0.025;
            n = 1 + (int)FastMath.floor(al / r);
            zr = FastMath.exp(al / (double)n);
            double sum3 = 0.0;
            double tloc4 = 0.0;
            for (int i = 0; i < n; ++i) {
                z = zend;
                zend = zr * z;
                double dz = 0.25 * (zend - z);
                double sum1 = WT[0] * ain;
                for (int j = 1; j < 5; ++j) {
                    tloc4 = JB2008.localTemp(z += dz, tc);
                    gravl = JB2008.gravity(z);
                    ain = gravl / tloc4;
                    sum1 += WT[j] * ain;
                }
                sum3 += dz * sum1;
            }
            if (altKm <= 500.0) {
                temp[1] = tloc3;
                altr = FastMath.log(tloc3 / tloc2);
                fact2 = sub2 / 8.31432;
                hSign = 1.0;
            } else {
                temp[1] = tloc4;
                altr = FastMath.log(tloc4 / tloc2);
                fact2 = (sub2 + sum3) / 8.31432;
                hSign = -1.0;
            }
            for (int i = 0; i < 5; ++i) {
                aln[i] = aln[i] - (1.0 + ALPHA[i]) * altr - fact2 * AMW[i];
            }
            double al10t5 = FastMath.log10(tinf);
            double alnh5 = (5.5 * al10t5 - 39.4) * al10t5 + 73.13;
            aln[5] = LOG10 * (alnh5 + 6.0) + hSign * (FastMath.log(tloc4 / tloc3) + sum3 * AMW[5] / 8.31432);
        }
        double capPhi = (dateMJD - 36204.0) / 365.2422 % 1.0;
        int signum = satLat >= 0.0 ? 1 : -1;
        double sinLat = FastMath.sin(satLat);
        double hm90 = altKm - 90.0;
        double dlrsl = 0.02 * hm90 * FastMath.exp(-0.045 * hm90) * (double)signum * FastMath.sin(Math.PI * 2 * capPhi + 1.72) * sinLat * sinLat;
        double dlrsa = 0.0;
        if (z < 2000.0) {
            dlrsa = JB2008.semian08(JB2008.dayOfYear(dateMJD), altKm, f10B, s10B, xm10B);
        }
        double dlr = LOG10 * (dlrsl + dlrsa);
        int i = 0;
        while (i < 6) {
            int n2 = i++;
            aln[n2] = aln[n2] + dlr;
        }
        double sumnm = 0.0;
        for (int i2 = 0; i2 < 6; ++i2) {
            sumnm += FastMath.exp(aln[i2]) * AMW[i2];
        }
        rho = sumnm / 6.02257E26;
        double fex = 1.0;
        if (altKm >= 1000.0 && altKm < 1500.0) {
            double zeta = (altKm - 1000.0) * 0.002;
            double f15c = CHT[0] + CHT[1] * f10B + (CHT[2] + CHT[3] * f10B) * 1500.0;
            double f15cZeta = (CHT[2] + CHT[3] * f10B) * 500.0;
            double fex2 = 3.0 * f15c - f15cZeta - 3.0;
            double fex3 = f15cZeta - 2.0 * f15c + 2.0;
            fex += zeta * zeta * (fex2 + fex3 * zeta);
        } else if (altKm >= 1500.0) {
            fex = CHT[0] + CHT[1] * f10B + CHT[2] * altKm + CHT[3] * f10B * altKm;
        }
        return rho *= fex;
    }

    public <T extends RealFieldElement<T>> T getDensity(T dateMJD, T sunRA, T sunDecli, T satLon, T satLat, T satAlt, double f10, double f10B, double s10, double s10B, double xm10, double xm10B, double y10, double y10B, double dstdtc) {
        if (satAlt.getReal() < 90000.0) {
            throw new OrekitException((Localizable)OrekitMessages.ALTITUDE_BELOW_ALLOWED_THRESHOLD, satAlt.getReal(), 90000.0);
        }
        Field field = satAlt.getField();
        RealFieldElement altKm = (RealFieldElement)satAlt.divide(1000.0);
        double fn = FastMath.min(1.0, FastMath.pow(f10B / 240.0, 0.25));
        double fsb = f10B * fn + s10B * (1.0 - fn);
        double tsubc = 392.4 + 3.227 * fsb + 0.298 * (f10 - f10B) + 2.259 * (s10 - s10B) + 0.312 * (xm10 - xm10B) + 0.178 * (y10 - y10B);
        RealFieldElement eta = (RealFieldElement)((RealFieldElement)satLat.subtract(sunDecli).abs()).multiply(0.5);
        RealFieldElement theta = (RealFieldElement)((RealFieldElement)satLat.add(sunDecli).abs()).multiply(0.5);
        T h = satLon.subtract(sunRA);
        RealFieldElement tau = (RealFieldElement)((RealFieldElement)h.subtract(0.64577182)).add(((RealFieldElement)((RealFieldElement)h.add(0.75049158)).sin()).multiply(0.10471976));
        RealFieldElement solarTime = (RealFieldElement)((RealFieldElement)h.add(Math.PI)).multiply(3.819718634205488);
        while (solarTime.getReal() >= 24.0) {
            solarTime = (RealFieldElement)solarTime.subtract(24.0);
        }
        while (solarTime.getReal() < 0.0) {
            solarTime = (RealFieldElement)solarTime.add(24.0);
        }
        RealFieldElement cos = (RealFieldElement)eta.cos();
        RealFieldElement cosEta = (RealFieldElement)cos.multiply(cos).multiply(cos.sqrt());
        RealFieldElement sin = (RealFieldElement)theta.sin();
        RealFieldElement sinTeta = (RealFieldElement)sin.multiply(sin).multiply(sin.sqrt());
        RealFieldElement cosTau = (RealFieldElement)((RealFieldElement)((RealFieldElement)tau.multiply(0.5)).cos()).abs();
        RealFieldElement df = sinTeta.add(cosEta.subtract(sinTeta).multiply(cosTau).multiply(cosTau).multiply(cosTau));
        RealFieldElement tsubl = (RealFieldElement)((RealFieldElement)((RealFieldElement)df.multiply(0.31)).add(1.0)).multiply(tsubc);
        RealFieldElement dtclst = JB2008.dTc(f10, solarTime, satLat, altKm);
        RealFieldElement[] temp = (RealFieldElement[])MathArrays.buildArray(field, 2);
        temp[0] = (RealFieldElement)tsubl.add(dstdtc);
        RealFieldElement tinf = temp[0].add(dtclst);
        RealFieldElement tsubx = (RealFieldElement)((RealFieldElement)((RealFieldElement)tinf.multiply(0.02385)).add(444.3807)).subtract(((RealFieldElement)((RealFieldElement)tinf.multiply(-0.0021357)).exp()).multiply(392.8292));
        RealFieldElement gsubx = (RealFieldElement)((RealFieldElement)tsubx.subtract(183.0)).multiply(0.054285714);
        RealFieldElement[] tc = (RealFieldElement[])MathArrays.buildArray(field, 4);
        tc[0] = tsubx;
        tc[1] = gsubx;
        tc[2] = (RealFieldElement)tinf.subtract(tsubx).multiply(0.6366197723675814);
        tc[3] = gsubx.divide(tc[2]);
        RealFieldElement z1 = (RealFieldElement)((RealFieldElement)field.getZero()).add(90.0);
        RealFieldElement z2 = this.min(105.0, altKm);
        RealFieldElement al = (RealFieldElement)z2.divide(z1).log();
        int n = 1 + (int)FastMath.floor(al.getReal() / 0.01);
        RealFieldElement zr = (RealFieldElement)((RealFieldElement)al.divide(n)).exp();
        RealFieldElement mb1 = JB2008.mBar(z1);
        RealFieldElement tloc1 = JB2008.localTemp((RealFieldElement)z1, (RealFieldElement[])tc);
        RealFieldElement zend = z1;
        RealFieldElement sub2 = (RealFieldElement)field.getZero();
        RealFieldElement ain = mb1.multiply(JB2008.gravity(z1)).divide(tloc1);
        RealFieldElement mb2 = (RealFieldElement)field.getZero();
        RealFieldElement tloc2 = (RealFieldElement)field.getZero();
        RealFieldElement z = (RealFieldElement)field.getZero();
        RealFieldElement gravl = (RealFieldElement)field.getZero();
        for (int i = 0; i < n; ++i) {
            z = zend;
            zend = zr.multiply(z);
            RealFieldElement dz = (RealFieldElement)zend.subtract(z).multiply(0.25);
            RealFieldElement sum1 = (RealFieldElement)ain.multiply(WT[0]);
            for (int j = 1; j < 5; ++j) {
                z = z.add(dz);
                mb2 = JB2008.mBar(z);
                tloc2 = JB2008.localTemp((RealFieldElement)z, (RealFieldElement[])tc);
                gravl = JB2008.gravity(z);
                ain = mb2.multiply(gravl).divide(tloc2);
                sum1 = (RealFieldElement)sum1.add(ain.multiply(WT[j]));
            }
            sub2 = sub2.add(dz.multiply(sum1));
        }
        RealFieldElement rho = ((RealFieldElement)mb2.multiply(3.46E-6)).multiply(tloc1).divide(((RealFieldElement)((RealFieldElement)sub2.divide(8.31432)).exp()).multiply(mb1.multiply(tloc2)));
        RealFieldElement anm = (RealFieldElement)rho.multiply(6.02257E26);
        RealFieldElement an = anm.divide(mb2);
        RealFieldElement fact2 = (RealFieldElement)anm.divide(28.96);
        RealFieldElement[] aln = (RealFieldElement[])MathArrays.buildArray(field, 6);
        aln[0] = (RealFieldElement)((RealFieldElement)fact2.multiply(FRAC[0])).log();
        aln[3] = (RealFieldElement)((RealFieldElement)fact2.multiply(FRAC[2])).log();
        aln[4] = (RealFieldElement)((RealFieldElement)fact2.multiply(FRAC[3])).log();
        aln[1] = (RealFieldElement)((RealFieldElement)fact2.multiply(1.0 + FRAC[1])).subtract(an).log();
        aln[2] = (RealFieldElement)((RealFieldElement)an.subtract(fact2).multiply(2)).log();
        if (altKm.getReal() <= 105.0) {
            temp[1] = tloc2;
            aln[5] = (RealFieldElement)aln[4].subtract(25.0);
        } else {
            double hSign;
            RealFieldElement altr;
            al = (RealFieldElement)this.min(500.0, altKm).divide(z).log();
            n = 1 + (int)FastMath.floor(al.getReal() / 0.025);
            zr = (RealFieldElement)((RealFieldElement)al.divide(n)).exp();
            sub2 = (RealFieldElement)field.getZero();
            ain = gravl.divide(tloc2);
            RealFieldElement tloc3 = (RealFieldElement)field.getZero();
            for (int i = 0; i < n; ++i) {
                z = zend;
                zend = zr.multiply(z);
                RealFieldElement dz = (RealFieldElement)zend.subtract(z).multiply(0.25);
                RealFieldElement sum1 = (RealFieldElement)ain.multiply(WT[0]);
                for (int j = 1; j < 5; ++j) {
                    z = z.add(dz);
                    tloc3 = JB2008.localTemp((RealFieldElement)z, (RealFieldElement[])tc);
                    gravl = JB2008.gravity(z);
                    ain = gravl.divide(tloc3);
                    sum1 = (RealFieldElement)sum1.add(ain.multiply(WT[j]));
                }
                sub2 = sub2.add(dz.multiply(sum1));
            }
            al = (RealFieldElement)this.max(500.0, altKm).divide(z).log();
            double r = altKm.getReal() > 500.0 ? 0.075 : 0.025;
            n = 1 + (int)FastMath.floor(al.getReal() / r);
            zr = (RealFieldElement)((RealFieldElement)al.divide(n)).exp();
            RealFieldElement sum3 = (RealFieldElement)field.getZero();
            RealFieldElement tloc4 = (RealFieldElement)field.getZero();
            for (int i = 0; i < n; ++i) {
                z = zend;
                zend = zr.multiply(z);
                RealFieldElement dz = (RealFieldElement)zend.subtract(z).multiply(0.25);
                RealFieldElement sum1 = (RealFieldElement)ain.multiply(WT[0]);
                for (int j = 1; j < 5; ++j) {
                    z = z.add(dz);
                    tloc4 = JB2008.localTemp((RealFieldElement)z, (RealFieldElement[])tc);
                    gravl = JB2008.gravity(z);
                    ain = gravl.divide(tloc4);
                    sum1 = (RealFieldElement)sum1.add(ain.multiply(WT[j]));
                }
                sum3 = sum3.add(dz.multiply(sum1));
            }
            if (altKm.getReal() <= 500.0) {
                temp[1] = tloc3;
                altr = (RealFieldElement)tloc3.divide(tloc2).log();
                fact2 = (RealFieldElement)sub2.divide(8.31432);
                hSign = 1.0;
            } else {
                temp[1] = tloc4;
                altr = (RealFieldElement)tloc4.divide(tloc2).log();
                fact2 = (RealFieldElement)sub2.add(sum3).divide(8.31432);
                hSign = -1.0;
            }
            for (int i = 0; i < 5; ++i) {
                aln[i] = (RealFieldElement)((RealFieldElement)aln[i].subtract(altr.multiply(1.0 + ALPHA[i]))).subtract(fact2.multiply(AMW[i]));
            }
            RealFieldElement al10t5 = (RealFieldElement)tinf.log10();
            RealFieldElement alnh5 = (RealFieldElement)((RealFieldElement)((RealFieldElement)al10t5.multiply(5.5)).subtract(39.4)).multiply(al10t5).add(73.13);
            aln[5] = (RealFieldElement)((RealFieldElement)((RealFieldElement)alnh5.add(6.0)).multiply(LOG10)).add(((RealFieldElement)((RealFieldElement)tloc4.divide(tloc3).log()).add(sum3.multiply(AMW[5] / 8.31432))).multiply(hSign));
        }
        RealFieldElement capPhi = (RealFieldElement)((RealFieldElement)dateMJD.subtract(36204.0)).divide(365.2422);
        capPhi = (RealFieldElement)capPhi.subtract(FastMath.floor(capPhi.getReal()));
        int signum = satLat.getReal() >= 0.0 ? 1 : -1;
        RealFieldElement sinLat = (RealFieldElement)satLat.sin();
        RealFieldElement hm90 = (RealFieldElement)altKm.subtract(90.0);
        RealFieldElement dlrsl = ((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)hm90.multiply(0.02)).multiply(((RealFieldElement)hm90.multiply(-0.045)).exp())).multiply(((RealFieldElement)((RealFieldElement)capPhi.multiply(Math.PI * 2)).add(1.72)).sin())).multiply(signum)).multiply(sinLat).multiply(sinLat);
        RealFieldElement dlrsa = (RealFieldElement)field.getZero();
        if (z.getReal() < 2000.0) {
            dlrsa = JB2008.semian08(JB2008.dayOfYear(dateMJD), altKm, f10B, s10B, xm10B);
        }
        RealFieldElement dlr = (RealFieldElement)dlrsl.add(dlrsa).multiply(LOG10);
        for (int i = 0; i < 6; ++i) {
            aln[i] = aln[i].add(dlr);
        }
        RealFieldElement sumnm = (RealFieldElement)field.getZero();
        for (int i = 0; i < 6; ++i) {
            sumnm = (RealFieldElement)sumnm.add(((RealFieldElement)aln[i].exp()).multiply(AMW[i]));
        }
        rho = (RealFieldElement)sumnm.divide(6.02257E26);
        RealFieldElement fex = (RealFieldElement)field.getOne();
        if (altKm.getReal() >= 1000.0 && altKm.getReal() < 1500.0) {
            RealFieldElement zeta = (RealFieldElement)((RealFieldElement)altKm.subtract(1000.0)).multiply(0.002);
            double f15c = CHT[0] + CHT[1] * f10B + (CHT[2] + CHT[3] * f10B) * 1500.0;
            double f15cZeta = (CHT[2] + CHT[3] * f10B) * 500.0;
            double fex2 = 3.0 * f15c - f15cZeta - 3.0;
            double fex3 = f15cZeta - 2.0 * f15c + 2.0;
            fex = (RealFieldElement)fex.add(zeta.multiply(zeta).multiply(((RealFieldElement)zeta.multiply(fex3)).add(fex2)));
        } else if (altKm.getReal() >= 1500.0) {
            fex = (RealFieldElement)((RealFieldElement)((RealFieldElement)altKm.multiply(CHT[3] * f10B)).add(altKm.multiply(CHT[2]))).add(CHT[0] + CHT[1] * f10B);
        }
        rho = rho.multiply(fex);
        return (T)rho;
    }

    private static double dTc(double f10, double solarTime, double satLat, double satAlt) {
        double dTc = 0.0;
        double st = solarTime / 24.0;
        double cs = FastMath.cos(satLat);
        double fs = (f10 - 100.0) / 100.0;
        if (satAlt >= 120.0 && satAlt <= 200.0) {
            double dtc200 = JB2008.poly2CDTC(fs, st, cs);
            double dtc200dz = JB2008.poly1CDTC(fs, st, cs);
            double cc = 3.0 * dtc200 - dtc200dz;
            double dd = dtc200 - cc;
            double zp = (satAlt - 120.0) / 80.0;
            dTc = zp * zp * (cc + dd * zp);
        } else if (satAlt > 200.0 && satAlt <= 240.0) {
            double h = (satAlt - 200.0) / 50.0;
            dTc = JB2008.poly1CDTC(fs, st, cs) * h + JB2008.poly2CDTC(fs, st, cs);
        } else if (satAlt > 240.0 && satAlt <= 300.0) {
            double h = 0.8;
            double bb = JB2008.poly1CDTC(fs, st, cs);
            double aa = bb * 0.8 + JB2008.poly2CDTC(fs, st, cs);
            double p2BDT = JB2008.poly2BDTC(st);
            double dtc300 = JB2008.poly1BDTC(fs, st, cs, 3.0 * p2BDT);
            double dtc300dz = cs * p2BDT;
            double cc = 3.0 * dtc300 - dtc300dz - 3.0 * aa - 2.0 * bb;
            double dd = dtc300 - aa - bb - cc;
            double zp = (satAlt - 240.0) / 60.0;
            dTc = aa + zp * (bb + zp * (cc + zp * dd));
        } else if (satAlt > 300.0 && satAlt <= 600.0) {
            double h = satAlt / 100.0;
            dTc = JB2008.poly1BDTC(fs, st, cs, h * JB2008.poly2BDTC(st));
        } else if (satAlt > 600.0 && satAlt <= 800.0) {
            double poly2 = JB2008.poly2BDTC(st);
            double aa = JB2008.poly1BDTC(fs, st, cs, 6.0 * poly2);
            double bb = cs * poly2;
            double cc = -(3.0 * aa + 4.0 * bb) / 4.0;
            double dd = (aa + bb) / 4.0;
            double zp = (satAlt - 600.0) / 100.0;
            dTc = aa + zp * (bb + zp * (cc + zp * dd));
        }
        return dTc;
    }

    private static <T extends RealFieldElement<T>> T dTc(double f10, T solarTime, T satLat, T satAlt) {
        RealFieldElement dTc = (RealFieldElement)solarTime.getField().getZero();
        RealFieldElement st = (RealFieldElement)solarTime.divide(24.0);
        RealFieldElement cs = (RealFieldElement)satLat.cos();
        double fs = (f10 - 100.0) / 100.0;
        if (satAlt.getReal() >= 120.0 && satAlt.getReal() <= 200.0) {
            RealFieldElement dtc200 = JB2008.poly2CDTC(fs, st, cs);
            RealFieldElement dtc200dz = JB2008.poly1CDTC(fs, st, cs);
            RealFieldElement cc = ((RealFieldElement)dtc200.multiply(3)).subtract(dtc200dz);
            RealFieldElement dd = dtc200.subtract(cc);
            RealFieldElement zp = (RealFieldElement)((RealFieldElement)satAlt.subtract(120.0)).divide(80.0);
            dTc = zp.multiply(zp).multiply(cc.add(dd.multiply(zp)));
        } else if (satAlt.getReal() > 200.0 && satAlt.getReal() <= 240.0) {
            RealFieldElement h = (RealFieldElement)((RealFieldElement)satAlt.subtract(200.0)).divide(50.0);
            dTc = JB2008.poly1CDTC(fs, st, cs).multiply(h).add(JB2008.poly2CDTC(fs, st, cs));
        } else if (satAlt.getReal() > 240.0 && satAlt.getReal() <= 300.0) {
            RealFieldElement h = (RealFieldElement)((RealFieldElement)solarTime.getField().getZero()).add(0.8);
            RealFieldElement bb = JB2008.poly1CDTC(fs, st, cs);
            RealFieldElement aa = bb.multiply(h).add(JB2008.poly2CDTC(fs, st, cs));
            RealFieldElement p2BDT = JB2008.poly2BDTC(st);
            RealFieldElement dtc300 = JB2008.poly1BDTC(fs, st, cs, (RealFieldElement)p2BDT.multiply(3));
            RealFieldElement dtc300dz = cs.multiply(p2BDT);
            RealFieldElement cc = (RealFieldElement)((RealFieldElement)((RealFieldElement)dtc300.multiply(3)).subtract(dtc300dz).subtract(aa.multiply(3))).subtract(bb.multiply(2));
            RealFieldElement dd = dtc300.subtract(aa).subtract(bb).subtract(cc);
            RealFieldElement zp = (RealFieldElement)((RealFieldElement)satAlt.subtract(240.0)).divide(60.0);
            dTc = aa.add(zp.multiply(bb.add(zp.multiply(cc.add(zp.multiply(dd))))));
        } else if (satAlt.getReal() > 300.0 && satAlt.getReal() <= 600.0) {
            RealFieldElement h = (RealFieldElement)satAlt.divide(100.0);
            dTc = JB2008.poly1BDTC(fs, st, cs, h.multiply(JB2008.poly2BDTC(st)));
        } else if (satAlt.getReal() > 600.0 && satAlt.getReal() <= 800.0) {
            RealFieldElement poly2 = JB2008.poly2BDTC(st);
            RealFieldElement aa = JB2008.poly1BDTC(fs, st, cs, (RealFieldElement)poly2.multiply(6));
            RealFieldElement bb = cs.multiply(poly2);
            RealFieldElement cc = (RealFieldElement)((RealFieldElement)((RealFieldElement)aa.multiply(3)).add(bb.multiply(4))).divide(-4.0);
            RealFieldElement dd = (RealFieldElement)aa.add(bb).divide(4.0);
            RealFieldElement zp = (RealFieldElement)((RealFieldElement)satAlt.subtract(600.0)).divide(100.0);
            dTc = aa.add(zp.multiply(bb.add(zp.multiply(cc.add(zp.multiply(dd))))));
        }
        return (T)dTc;
    }

    private static double poly1CDTC(double fs, double st, double cs) {
        return CDTC[0] + fs * (CDTC[1] + st * (CDTC[2] + st * (CDTC[3] + st * (CDTC[4] + st * (CDTC[5] + st * CDTC[6]))))) + cs * st * (CDTC[7] + st * (CDTC[8] + st * (CDTC[9] + st * (CDTC[10] + st * CDTC[11])))) + cs * (CDTC[12] + fs * (CDTC[13] + st * (CDTC[14] + st * CDTC[15])));
    }

    private static <T extends RealFieldElement<T>> T poly1CDTC(double fs, T st, T cs) {
        return (T)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)st.multiply(CDTC[6])).add(CDTC[5])).multiply(st).add(CDTC[4])).multiply(st).add(CDTC[3])).multiply(st).add(CDTC[2])).multiply(st).add(CDTC[1])).multiply(fs)).add(((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)st.multiply(CDTC[11])).add(CDTC[10])).multiply(st).add(CDTC[9])).multiply(st).add(CDTC[8])).multiply(st).add(CDTC[7])).multiply(st).multiply(cs)).add(((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)st.multiply(CDTC[15])).add(CDTC[14])).multiply(st).add(CDTC[13])).multiply(fs)).add(CDTC[12])).multiply(cs)).add(CDTC[0]));
    }

    private static double poly2CDTC(double fs, double st, double cs) {
        return CDTC[16] + st * cs * (CDTC[17] + st * (CDTC[18] + st * CDTC[19])) + fs * cs * (CDTC[20] + st * (CDTC[21] + st * CDTC[22]));
    }

    private static <T extends RealFieldElement<T>> T poly2CDTC(double fs, T st, T cs) {
        return (T)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)st.multiply(CDTC[19])).add(CDTC[18])).multiply(st).add(CDTC[17])).multiply(cs).multiply(st).add(((RealFieldElement)((RealFieldElement)((RealFieldElement)st.multiply(CDTC[22])).add(CDTC[21])).multiply(st).add(CDTC[20])).multiply(cs).multiply(fs))).add(CDTC[16]));
    }

    private static double poly1BDTC(double fs, double st, double cs, double hp) {
        return BDTC[0] + fs * (BDTC[1] + st * (BDTC[2] + st * (BDTC[3] + st * (BDTC[4] + st * (BDTC[5] + st * BDTC[6]))))) + cs * (st * (BDTC[7] + st * (BDTC[8] + st * (BDTC[9] + st * (BDTC[10] + st * BDTC[11])))) + hp + BDTC[18]);
    }

    private static <T extends RealFieldElement<T>> T poly1BDTC(double fs, T st, T cs, T hp) {
        return (T)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)st.multiply(BDTC[6])).add(BDTC[5])).multiply(st).add(BDTC[4])).multiply(st).add(BDTC[3])).multiply(st).add(BDTC[2])).multiply(st).add(BDTC[1])).multiply(fs)).add(((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)st.multiply(BDTC[11])).add(BDTC[10])).multiply(st).add(BDTC[9])).multiply(st).add(BDTC[8])).multiply(st).add(BDTC[7])).multiply(st).add(hp).add(BDTC[18])).multiply(cs)).add(BDTC[0]));
    }

    private static double poly2BDTC(double st) {
        return BDTC[12] + st * (BDTC[13] + st * (BDTC[14] + st * (BDTC[15] + st * (BDTC[16] + st * BDTC[17]))));
    }

    private static <T extends RealFieldElement<T>> T poly2BDTC(T st) {
        return (T)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)st.multiply(BDTC[17])).add(BDTC[16])).multiply(st).add(BDTC[15])).multiply(st).add(BDTC[14])).multiply(st).add(BDTC[13])).multiply(st).add(BDTC[12]));
    }

    private static double mBar(double z) {
        double dz = z - 100.0;
        double amb = CMB[6];
        for (int i = 5; i >= 0; --i) {
            amb = dz * amb + CMB[i];
        }
        return amb;
    }

    private static <T extends RealFieldElement<T>> T mBar(T z) {
        RealFieldElement dz = (RealFieldElement)z.subtract(100.0);
        RealFieldElement amb = (RealFieldElement)((RealFieldElement)z.getField().getZero()).add(CMB[6]);
        for (int i = 5; i >= 0; --i) {
            amb = (RealFieldElement)dz.multiply(amb).add(CMB[i]);
        }
        return (T)amb;
    }

    private static double localTemp(double z, double[] tc) {
        double dz = z - 125.0;
        if (dz <= 0.0) {
            return ((-9.8204695E-6 * dz - 7.3039742E-4) * dz * dz + 1.0) * dz * tc[1] + tc[0];
        }
        return tc[0] + tc[2] * FastMath.atan(tc[3] * dz * (1.0 + 4.5E-6 * FastMath.pow(dz, 2.5)));
    }

    private static <T extends RealFieldElement<T>> T localTemp(T z, T[] tc) {
        RealFieldElement dz = (RealFieldElement)z.subtract(125.0);
        if (dz.getReal() <= 0.0) {
            return (T)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)dz.multiply(-9.8204695E-6)).subtract(7.3039742E-4)).multiply(dz).multiply(dz).add(1.0)).multiply(dz).multiply(tc[1])).add(tc[0]));
        }
        return (T)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)dz.multiply(dz).multiply(dz.sqrt())).multiply(4.5E-6)).add(1.0)).multiply(dz).multiply(tc[3])).atan()).multiply(tc[2])).add(tc[0]));
    }

    private static double gravity(double z) {
        double tmp = 1.0 + z / 6356.766;
        return 9.80665 / (tmp * tmp);
    }

    private static <T extends RealFieldElement<T>> T gravity(T z) {
        RealFieldElement tmp = (RealFieldElement)((RealFieldElement)z.divide(6356.766)).add(1.0);
        return (T)((RealFieldElement)((RealFieldElement)tmp.multiply(tmp).reciprocal()).multiply(9.80665));
    }

    private static double semian08(double doy, double alt, double f10B, double s10B, double xm10B) {
        double htz = alt / 1000.0;
        double fsmb = f10B - 0.7 * s10B - 0.04 * xm10B;
        double fzz = FZM[0] + fsmb * (FZM[1] + htz * (FZM[2] + FZM[3] * htz + FZM[4] * fsmb));
        fsmb = f10B - 0.75 * s10B - 0.37 * xm10B;
        double tau = Math.PI * 2 * (doy - 1.0) / 365.0;
        double taux2 = tau + tau;
        double sin1P = FastMath.sin(tau);
        double cos1P = FastMath.cos(tau);
        double sin2P = FastMath.sin(taux2);
        double cos2P = FastMath.cos(taux2);
        double gtz = GTM[0] + GTM[1] * sin1P + GTM[2] * cos1P + GTM[3] * sin2P + GTM[4] * cos2P + fsmb * (GTM[5] + GTM[6] * sin1P + GTM[7] * cos1P + GTM[8] * sin2P + GTM[9] * cos2P);
        return FastMath.max(1.0E-6, fzz) * gtz;
    }

    private static <T extends RealFieldElement<T>> T semian08(T doy, T alt, double f10B, double s10B, double xm10B) {
        RealFieldElement htz = (RealFieldElement)alt.divide(1000.0);
        double fsmb = f10B - 0.7 * s10B - 0.04 * xm10B;
        RealFieldElement fzz = (RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)htz.multiply(FZM[3])).add(FZM[2] + FZM[4] * fsmb)).multiply(htz).add(FZM[1])).multiply(fsmb)).add(FZM[0]);
        fsmb = f10B - 0.75 * s10B - 0.37 * xm10B;
        RealFieldElement tau = (RealFieldElement)((RealFieldElement)((RealFieldElement)doy.subtract(1.0)).divide(365.0)).multiply(Math.PI * 2);
        RealFieldElement taux2 = tau.add(tau);
        RealFieldElement sin1P = (RealFieldElement)tau.sin();
        RealFieldElement cos1P = (RealFieldElement)tau.cos();
        RealFieldElement sin2P = (RealFieldElement)taux2.sin();
        RealFieldElement cos2P = (RealFieldElement)taux2.cos();
        RealFieldElement gtz = (RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)cos2P.multiply(GTM[9])).add(sin2P.multiply(GTM[8]))).add(cos1P.multiply(GTM[7]))).add(sin1P.multiply(GTM[6]))).add(GTM[5])).multiply(fsmb)).add(((RealFieldElement)((RealFieldElement)((RealFieldElement)((RealFieldElement)cos2P.multiply(GTM[4])).add(sin2P.multiply(GTM[3]))).add(cos1P.multiply(GTM[2]))).add(sin1P.multiply(GTM[1]))).add(GTM[0]));
        return (T)(fzz.getReal() > 1.0E-6 ? gtz.multiply(fzz) : (RealFieldElement)gtz.multiply(1.0E-6));
    }

    private static double dayOfYear(double dateMJD) {
        double d1950 = dateMJD - 33281.0;
        int iyday = (int)d1950;
        double frac = d1950 - (double)iyday;
        int itemp = (iyday += 364) / 1461;
        if ((itemp = (iyday -= itemp * 1461) / 365) >= 3) {
            itemp = 3;
        }
        iyday = iyday - 365 * itemp + 1;
        return (double)iyday + frac;
    }

    private static <T extends RealFieldElement<T>> T dayOfYear(T dateMJD) {
        RealFieldElement d1950 = (RealFieldElement)dateMJD.subtract(33281.0);
        int iyday = (int)d1950.getReal();
        RealFieldElement frac = (RealFieldElement)d1950.subtract(iyday);
        int itemp = (iyday += 364) / 1461;
        if ((itemp = (iyday -= itemp * 1461) / 365) >= 3) {
            itemp = 3;
        }
        iyday = iyday - 365 * itemp + 1;
        return (T)((RealFieldElement)frac.add(iyday));
    }

    private <T extends RealFieldElement<T>> T min(double d, T f) {
        return (T)(f.getReal() > d ? (RealFieldElement)((RealFieldElement)f.getField().getZero()).add(d) : f);
    }

    private <T extends RealFieldElement<T>> T max(double d, T f) {
        return (T)(f.getReal() <= d ? (RealFieldElement)((RealFieldElement)f.getField().getZero()).add(d) : f);
    }

    @Override
    public double getDensity(AbsoluteDate date, Vector3D position, Frame frame) {
        if (date.compareTo(this.inputParams.getMaxDate()) > 0 || date.compareTo(this.inputParams.getMinDate()) < 0) {
            throw new OrekitException((Localizable)OrekitMessages.NO_SOLAR_ACTIVITY_AT_DATE, date, this.inputParams.getMinDate(), this.inputParams.getMaxDate());
        }
        double dateMJD = date.durationFrom(AbsoluteDate.MODIFIED_JULIAN_EPOCH) / 86400.0;
        GeodeticPoint inBody = this.earth.transform(position, frame, date);
        Frame ecef = this.earth.getBodyFrame();
        Vector3D sunPos = this.sun.getPVCoordinates(date, ecef).getPosition();
        GeodeticPoint sunInBody = this.earth.transform(sunPos, ecef, date);
        return this.getDensity(dateMJD, sunInBody.getLongitude(), sunInBody.getLatitude(), inBody.getLongitude(), inBody.getLatitude(), inBody.getAltitude(), this.inputParams.getF10(date), this.inputParams.getF10B(date), this.inputParams.getS10(date), this.inputParams.getS10B(date), this.inputParams.getXM10(date), this.inputParams.getXM10B(date), this.inputParams.getY10(date), this.inputParams.getY10B(date), this.inputParams.getDSTDTC(date));
    }

    @Override
    public <T extends RealFieldElement<T>> T getDensity(FieldAbsoluteDate<T> date, FieldVector3D<T> position, Frame frame) {
        AbsoluteDate dateD = date.toAbsoluteDate();
        if (dateD.compareTo(this.inputParams.getMaxDate()) > 0 || dateD.compareTo(this.inputParams.getMinDate()) < 0) {
            throw new OrekitException((Localizable)OrekitMessages.NO_SOLAR_ACTIVITY_AT_DATE, dateD, this.inputParams.getMinDate(), this.inputParams.getMaxDate());
        }
        RealFieldElement dateMJD = (RealFieldElement)date.durationFrom(AbsoluteDate.MODIFIED_JULIAN_EPOCH).divide(86400.0);
        FieldGeodeticPoint<T> inBody = this.earth.transform(position, frame, date);
        Frame ecef = this.earth.getBodyFrame();
        FieldVector3D<T> sunPos = new FieldVector3D<T>(date.getField(), this.sun.getPVCoordinates(dateD, ecef).getPosition());
        FieldGeodeticPoint<T> sunInBody = this.earth.transform(sunPos, ecef, date);
        return (T)this.getDensity((T)dateMJD, sunInBody.getLongitude(), sunInBody.getLatitude(), inBody.getLongitude(), inBody.getLatitude(), inBody.getAltitude(), this.inputParams.getF10(dateD), this.inputParams.getF10B(dateD), this.inputParams.getS10(dateD), this.inputParams.getS10B(dateD), this.inputParams.getXM10(dateD), this.inputParams.getXM10B(dateD), this.inputParams.getY10(dateD), this.inputParams.getY10B(dateD), this.inputParams.getDSTDTC(dateD));
    }
}

