/*
 * Decompiled with CFR 0.152.
 */
package org.orekit.orbits;

import java.io.Serializable;
import java.util.List;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.hipparchus.analysis.differentiation.DSFactory;
import org.hipparchus.analysis.differentiation.DerivativeStructure;
import org.hipparchus.analysis.interpolation.HermiteInterpolator;
import org.hipparchus.geometry.Vector;
import org.hipparchus.geometry.euclidean.threed.Vector3D;
import org.hipparchus.util.FastMath;
import org.hipparchus.util.MathUtils;
import org.orekit.errors.OrekitIllegalArgumentException;
import org.orekit.errors.OrekitInternalError;
import org.orekit.errors.OrekitMessages;
import org.orekit.frames.Frame;
import org.orekit.orbits.FieldEquinoctialOrbit;
import org.orekit.orbits.Orbit;
import org.orekit.orbits.OrbitType;
import org.orekit.orbits.PositionAngle;
import org.orekit.time.AbsoluteDate;
import org.orekit.utils.PVCoordinates;
import org.orekit.utils.TimeStampedPVCoordinates;

public class EquinoctialOrbit
extends Orbit {
    private static final long serialVersionUID = 20170414L;
    private static final DSFactory FACTORY = new DSFactory(1, 1);
    private final double a;
    private final double ex;
    private final double ey;
    private final double hx;
    private final double hy;
    private final double lv;
    private final double aDot;
    private final double exDot;
    private final double eyDot;
    private final double hxDot;
    private final double hyDot;
    private final double lvDot;
    private transient PVCoordinates partialPV;

    public EquinoctialOrbit(double a, double ex, double ey, double hx, double hy, double l, PositionAngle type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(a, ex, ey, hx, hy, l, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, Double.NaN, type, frame, date, mu);
    }

    public EquinoctialOrbit(double a, double ex, double ey, double hx, double hy, double l, double aDot, double exDot, double eyDot, double hxDot, double hyDot, double lDot, PositionAngle type, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        super(frame, date, mu);
        if (ex * ex + ey * ey >= 1.0) {
            throw new OrekitIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        this.a = a;
        this.aDot = aDot;
        this.ex = ex;
        this.exDot = exDot;
        this.ey = ey;
        this.eyDot = eyDot;
        this.hx = hx;
        this.hxDot = hxDot;
        this.hy = hy;
        this.hyDot = hyDot;
        if (this.hasDerivatives()) {
            DerivativeStructure lvDS;
            DerivativeStructure exDS = FACTORY.build(ex, exDot);
            DerivativeStructure eyDS = FACTORY.build(ey, eyDot);
            DerivativeStructure lDS = FACTORY.build(l, lDot);
            switch (type) {
                case MEAN: {
                    lvDS = FieldEquinoctialOrbit.eccentricToTrue(FieldEquinoctialOrbit.meanToEccentric(lDS, exDS, eyDS), exDS, eyDS);
                    break;
                }
                case ECCENTRIC: {
                    lvDS = FieldEquinoctialOrbit.eccentricToTrue(lDS, exDS, eyDS);
                    break;
                }
                case TRUE: {
                    lvDS = lDS;
                    break;
                }
                default: {
                    throw new OrekitInternalError(null);
                }
            }
            this.lv = lvDS.getValue();
            this.lvDot = lvDS.getPartialDerivative(1);
        } else {
            switch (type) {
                case MEAN: {
                    this.lv = EquinoctialOrbit.eccentricToTrue(EquinoctialOrbit.meanToEccentric(l, ex, ey), ex, ey);
                    break;
                }
                case ECCENTRIC: {
                    this.lv = EquinoctialOrbit.eccentricToTrue(l, ex, ey);
                    break;
                }
                case TRUE: {
                    this.lv = l;
                    break;
                }
                default: {
                    throw new OrekitInternalError(null);
                }
            }
            this.lvDot = Double.NaN;
        }
        this.partialPV = null;
    }

    public EquinoctialOrbit(TimeStampedPVCoordinates pvCoordinates, Frame frame, double mu) throws IllegalArgumentException {
        super(pvCoordinates, frame, mu);
        Vector3D pvP = pvCoordinates.getPosition();
        Vector3D pvV = pvCoordinates.getVelocity();
        Vector3D pvA = pvCoordinates.getAcceleration();
        double r2 = pvP.getNormSq();
        double r = FastMath.sqrt(r2);
        double V2 = pvV.getNormSq();
        double rV2OnMu = r * V2 / mu;
        if (rV2OnMu > 2.0) {
            throw new OrekitIllegalArgumentException(OrekitMessages.HYPERBOLIC_ORBIT_NOT_HANDLED_AS, this.getClass().getName());
        }
        Vector3D w = pvCoordinates.getMomentum().normalize();
        double d = 1.0 / (1.0 + w.getZ());
        this.hx = -d * w.getY();
        this.hy = d * w.getX();
        double cLv = (pvP.getX() - d * pvP.getZ() * w.getX()) / r;
        double sLv = (pvP.getY() - d * pvP.getZ() * w.getY()) / r;
        this.lv = FastMath.atan2(sLv, cLv);
        this.a = r / (2.0 - rV2OnMu);
        double eSE = Vector3D.dotProduct(pvP, pvV) / FastMath.sqrt(mu * this.a);
        double eCE = rV2OnMu - 1.0;
        double e2 = eCE * eCE + eSE * eSE;
        double f = eCE - e2;
        double g = FastMath.sqrt(1.0 - e2) * eSE;
        this.ex = this.a * (f * cLv + g * sLv) / r;
        this.ey = this.a * (f * sLv - g * cLv) / r;
        this.partialPV = pvCoordinates;
        if (EquinoctialOrbit.hasNonKeplerianAcceleration(pvCoordinates, mu)) {
            double[][] jacobian = new double[6][6];
            this.getJacobianWrtCartesian(PositionAngle.MEAN, jacobian);
            Vector3D keplerianAcceleration = new Vector3D(-mu / (r * r2), pvP);
            Vector nonKeplerianAcceleration = pvA.subtract((Vector)keplerianAcceleration);
            double aX = ((Vector3D)nonKeplerianAcceleration).getX();
            double aY = ((Vector3D)nonKeplerianAcceleration).getY();
            double aZ = ((Vector3D)nonKeplerianAcceleration).getZ();
            this.aDot = jacobian[0][3] * aX + jacobian[0][4] * aY + jacobian[0][5] * aZ;
            this.exDot = jacobian[1][3] * aX + jacobian[1][4] * aY + jacobian[1][5] * aZ;
            this.eyDot = jacobian[2][3] * aX + jacobian[2][4] * aY + jacobian[2][5] * aZ;
            this.hxDot = jacobian[3][3] * aX + jacobian[3][4] * aY + jacobian[3][5] * aZ;
            this.hyDot = jacobian[4][3] * aX + jacobian[4][4] * aY + jacobian[4][5] * aZ;
            double lMDot = this.getKeplerianMeanMotion() + jacobian[5][3] * aX + jacobian[5][4] * aY + jacobian[5][5] * aZ;
            DerivativeStructure exDS = FACTORY.build(this.ex, this.exDot);
            DerivativeStructure eyDS = FACTORY.build(this.ey, this.eyDot);
            DerivativeStructure lMDS = FACTORY.build(this.getLM(), lMDot);
            DerivativeStructure lvDS = FieldEquinoctialOrbit.eccentricToTrue(FieldEquinoctialOrbit.meanToEccentric(lMDS, exDS, eyDS), exDS, eyDS);
            this.lvDot = lvDS.getPartialDerivative(1);
        } else {
            this.aDot = Double.NaN;
            this.exDot = Double.NaN;
            this.eyDot = Double.NaN;
            this.hxDot = Double.NaN;
            this.hyDot = Double.NaN;
            this.lvDot = Double.NaN;
        }
    }

    public EquinoctialOrbit(PVCoordinates pvCoordinates, Frame frame, AbsoluteDate date, double mu) throws IllegalArgumentException {
        this(new TimeStampedPVCoordinates(date, pvCoordinates), frame, mu);
    }

    public EquinoctialOrbit(Orbit op) {
        super(op.getFrame(), op.getDate(), op.getMu());
        this.a = op.getA();
        this.aDot = op.getADot();
        this.ex = op.getEquinoctialEx();
        this.exDot = op.getEquinoctialExDot();
        this.ey = op.getEquinoctialEy();
        this.eyDot = op.getEquinoctialEyDot();
        this.hx = op.getHx();
        this.hxDot = op.getHxDot();
        this.hy = op.getHy();
        this.hyDot = op.getHyDot();
        this.lv = op.getLv();
        this.lvDot = op.getLvDot();
        this.partialPV = null;
    }

    @Override
    public OrbitType getType() {
        return OrbitType.EQUINOCTIAL;
    }

    @Override
    public double getA() {
        return this.a;
    }

    @Override
    public double getADot() {
        return this.aDot;
    }

    @Override
    public double getEquinoctialEx() {
        return this.ex;
    }

    @Override
    public double getEquinoctialExDot() {
        return this.exDot;
    }

    @Override
    public double getEquinoctialEy() {
        return this.ey;
    }

    @Override
    public double getEquinoctialEyDot() {
        return this.eyDot;
    }

    @Override
    public double getHx() {
        return this.hx;
    }

    @Override
    public double getHxDot() {
        return this.hxDot;
    }

    @Override
    public double getHy() {
        return this.hy;
    }

    @Override
    public double getHyDot() {
        return this.hyDot;
    }

    @Override
    public double getLv() {
        return this.lv;
    }

    @Override
    public double getLvDot() {
        return this.lvDot;
    }

    @Override
    public double getLE() {
        return EquinoctialOrbit.trueToEccentric(this.lv, this.ex, this.ey);
    }

    @Override
    public double getLEDot() {
        DerivativeStructure lVDS = FACTORY.build(this.lv, this.lvDot);
        DerivativeStructure exDS = FACTORY.build(this.ex, this.exDot);
        DerivativeStructure eyDS = FACTORY.build(this.ey, this.eyDot);
        DerivativeStructure lEDS = FieldEquinoctialOrbit.trueToEccentric(lVDS, exDS, eyDS);
        return lEDS.getPartialDerivative(1);
    }

    @Override
    public double getLM() {
        return EquinoctialOrbit.eccentricToMean(EquinoctialOrbit.trueToEccentric(this.lv, this.ex, this.ey), this.ex, this.ey);
    }

    @Override
    public double getLMDot() {
        DerivativeStructure lVDS = FACTORY.build(this.lv, this.lvDot);
        DerivativeStructure exDS = FACTORY.build(this.ex, this.exDot);
        DerivativeStructure eyDS = FACTORY.build(this.ey, this.eyDot);
        DerivativeStructure lMDS = FieldEquinoctialOrbit.eccentricToMean(FieldEquinoctialOrbit.trueToEccentric(lVDS, exDS, eyDS), exDS, eyDS);
        return lMDS.getPartialDerivative(1);
    }

    public double getL(PositionAngle type) {
        return type == PositionAngle.MEAN ? this.getLM() : (type == PositionAngle.ECCENTRIC ? this.getLE() : this.getLv());
    }

    public double getLDot(PositionAngle type) {
        return type == PositionAngle.MEAN ? this.getLMDot() : (type == PositionAngle.ECCENTRIC ? this.getLEDot() : this.getLvDot());
    }

    public static double eccentricToTrue(double lE, double ex, double ey) {
        double epsilon = FastMath.sqrt(1.0 - ex * ex - ey * ey);
        double cosLE = FastMath.cos(lE);
        double sinLE = FastMath.sin(lE);
        double num = ex * sinLE - ey * cosLE;
        double den = epsilon + 1.0 - ex * cosLE - ey * sinLE;
        return lE + 2.0 * FastMath.atan(num / den);
    }

    public static double trueToEccentric(double lv, double ex, double ey) {
        double epsilon = FastMath.sqrt(1.0 - ex * ex - ey * ey);
        double cosLv = FastMath.cos(lv);
        double sinLv = FastMath.sin(lv);
        double num = ey * cosLv - ex * sinLv;
        double den = epsilon + 1.0 + ex * cosLv + ey * sinLv;
        return lv + 2.0 * FastMath.atan(num / den);
    }

    public static double meanToEccentric(double lM, double ex, double ey) {
        double lE = lM;
        double shift = 0.0;
        double lEmlM = 0.0;
        double cosLE = FastMath.cos(lE);
        double sinLE = FastMath.sin(lE);
        int iter = 0;
        do {
            double f2 = ex * sinLE - ey * cosLE;
            double f1 = 1.0 - ex * cosLE - ey * sinLE;
            double f0 = lEmlM - f2;
            double f12 = 2.0 * f1;
            shift = f0 * f12 / (f1 * f12 - f0 * f2);
            lE = lM + (lEmlM -= shift);
            cosLE = FastMath.cos(lE);
            sinLE = FastMath.sin(lE);
        } while (++iter < 50 && FastMath.abs(shift) > 1.0E-12);
        return lE;
    }

    public static double eccentricToMean(double lE, double ex, double ey) {
        return lE - ex * FastMath.sin(lE) + ey * FastMath.cos(lE);
    }

    @Override
    public double getE() {
        return FastMath.sqrt(this.ex * this.ex + this.ey * this.ey);
    }

    @Override
    public double getEDot() {
        return (this.ex * this.exDot + this.ey * this.eyDot) / FastMath.sqrt(this.ex * this.ex + this.ey * this.ey);
    }

    @Override
    public double getI() {
        return 2.0 * FastMath.atan(FastMath.sqrt(this.hx * this.hx + this.hy * this.hy));
    }

    @Override
    public double getIDot() {
        double h2 = this.hx * this.hx + this.hy * this.hy;
        double h = FastMath.sqrt(h2);
        return 2.0 * (this.hx * this.hxDot + this.hy * this.hyDot) / (h * (1.0 + h2));
    }

    private void computePVWithoutA() {
        if (this.partialPV != null) {
            return;
        }
        double lE = this.getLE();
        double hx2 = this.hx * this.hx;
        double hy2 = this.hy * this.hy;
        double factH = 1.0 / (1.0 + hx2 + hy2);
        double ux = (1.0 + hx2 - hy2) * factH;
        double uy = 2.0 * this.hx * this.hy * factH;
        double uz = -2.0 * this.hy * factH;
        double vx = uy;
        double vy = (1.0 - hx2 + hy2) * factH;
        double vz = 2.0 * this.hx * factH;
        double exey = this.ex * this.ey;
        double ex2 = this.ex * this.ex;
        double ey2 = this.ey * this.ey;
        double e2 = ex2 + ey2;
        double eta = 1.0 + FastMath.sqrt(1.0 - e2);
        double beta = 1.0 / eta;
        double cLe = FastMath.cos(lE);
        double sLe = FastMath.sin(lE);
        double exCeyS = this.ex * cLe + this.ey * sLe;
        double x = this.a * ((1.0 - beta * ey2) * cLe + beta * exey * sLe - this.ex);
        double y = this.a * ((1.0 - beta * ex2) * sLe + beta * exey * cLe - this.ey);
        double factor = FastMath.sqrt(this.getMu() / this.a) / (1.0 - exCeyS);
        double xdot = factor * (-sLe + beta * this.ey * exCeyS);
        double ydot = factor * (cLe - beta * this.ex * exCeyS);
        Vector3D position = new Vector3D(x * ux + y * vx, x * uy + y * vy, x * uz + y * vz);
        Vector3D velocity = new Vector3D(xdot * ux + ydot * vx, xdot * uy + ydot * vy, xdot * uz + ydot * vz);
        this.partialPV = new PVCoordinates(position, velocity);
    }

    private Vector3D nonKeplerianAcceleration() {
        double[][] dCdP = new double[6][6];
        this.getJacobianWrtParameters(PositionAngle.MEAN, dCdP);
        double nonKeplerianMeanMotion = this.getLMDot() - this.getKeplerianMeanMotion();
        double nonKeplerianAx = dCdP[3][0] * this.aDot + dCdP[3][1] * this.exDot + dCdP[3][2] * this.eyDot + dCdP[3][3] * this.hxDot + dCdP[3][4] * this.hyDot + dCdP[3][5] * nonKeplerianMeanMotion;
        double nonKeplerianAy = dCdP[4][0] * this.aDot + dCdP[4][1] * this.exDot + dCdP[4][2] * this.eyDot + dCdP[4][3] * this.hxDot + dCdP[4][4] * this.hyDot + dCdP[4][5] * nonKeplerianMeanMotion;
        double nonKeplerianAz = dCdP[5][0] * this.aDot + dCdP[5][1] * this.exDot + dCdP[5][2] * this.eyDot + dCdP[5][3] * this.hxDot + dCdP[5][4] * this.hyDot + dCdP[5][5] * nonKeplerianMeanMotion;
        return new Vector3D(nonKeplerianAx, nonKeplerianAy, nonKeplerianAz);
    }

    @Override
    protected TimeStampedPVCoordinates initPVCoordinates() {
        this.computePVWithoutA();
        double r2 = this.partialPV.getPosition().getNormSq();
        Vector3D keplerianAcceleration = new Vector3D(-this.getMu() / (r2 * FastMath.sqrt(r2)), this.partialPV.getPosition());
        Vector3D acceleration = this.hasDerivatives() ? keplerianAcceleration.add((Vector)this.nonKeplerianAcceleration()) : keplerianAcceleration;
        return new TimeStampedPVCoordinates(this.getDate(), this.partialPV.getPosition(), this.partialPV.getVelocity(), acceleration);
    }

    @Override
    public EquinoctialOrbit shiftedBy(double dt) {
        EquinoctialOrbit keplerianShifted = new EquinoctialOrbit(this.a, this.ex, this.ey, this.hx, this.hy, this.getLM() + this.getKeplerianMeanMotion() * dt, PositionAngle.MEAN, this.getFrame(), this.getDate().shiftedBy(dt), this.getMu());
        if (this.hasDerivatives()) {
            Vector3D nonKeplerianAcceleration = this.nonKeplerianAcceleration();
            keplerianShifted.computePVWithoutA();
            Vector3D fixedP = new Vector3D(1.0, keplerianShifted.partialPV.getPosition(), 0.5 * dt * dt, nonKeplerianAcceleration);
            double fixedR2 = fixedP.getNormSq();
            double fixedR = FastMath.sqrt(fixedR2);
            Vector3D fixedV = new Vector3D(1.0, keplerianShifted.partialPV.getVelocity(), dt, nonKeplerianAcceleration);
            Vector3D fixedA = new Vector3D(-this.getMu() / (fixedR2 * fixedR), keplerianShifted.partialPV.getPosition(), 1.0, nonKeplerianAcceleration);
            return new EquinoctialOrbit(new TimeStampedPVCoordinates(keplerianShifted.getDate(), fixedP, fixedV, fixedA), keplerianShifted.getFrame(), keplerianShifted.getMu());
        }
        return keplerianShifted;
    }

    @Override
    public EquinoctialOrbit interpolate(AbsoluteDate date, Stream<Orbit> sample) {
        List list = sample.collect(Collectors.toList());
        boolean useDerivatives = true;
        for (Orbit orbit : list) {
            useDerivatives = useDerivatives && orbit.hasDerivatives();
        }
        HermiteInterpolator interpolator = new HermiteInterpolator();
        AbsoluteDate previousDate = null;
        double previousLm = Double.NaN;
        for (Orbit orbit : list) {
            double continuousLm;
            EquinoctialOrbit equi = (EquinoctialOrbit)OrbitType.EQUINOCTIAL.convertType(orbit);
            if (previousDate == null) {
                continuousLm = equi.getLM();
            } else {
                double dt = equi.getDate().durationFrom(previousDate);
                double keplerLm = previousLm + equi.getKeplerianMeanMotion() * dt;
                continuousLm = MathUtils.normalizeAngle(equi.getLM(), keplerLm);
            }
            previousDate = equi.getDate();
            previousLm = continuousLm;
            if (useDerivatives) {
                interpolator.addSamplePoint(equi.getDate().durationFrom(date), {equi.getA(), equi.getEquinoctialEx(), equi.getEquinoctialEy(), equi.getHx(), equi.getHy(), continuousLm}, {equi.getADot(), equi.getEquinoctialExDot(), equi.getEquinoctialEyDot(), equi.getHxDot(), equi.getHyDot(), equi.getLMDot()});
                continue;
            }
            interpolator.addSamplePoint(equi.getDate().durationFrom(date), new double[][]{{equi.getA(), equi.getEquinoctialEx(), equi.getEquinoctialEy(), equi.getHx(), equi.getHy(), continuousLm}});
        }
        double[][] interpolated = interpolator.derivatives(0.0, 1);
        return new EquinoctialOrbit(interpolated[0][0], interpolated[0][1], interpolated[0][2], interpolated[0][3], interpolated[0][4], interpolated[0][5], interpolated[1][0], interpolated[1][1], interpolated[1][2], interpolated[1][3], interpolated[1][4], interpolated[1][5], PositionAngle.MEAN, this.getFrame(), date, this.getMu());
    }

    @Override
    protected double[][] computeJacobianMeanWrtCartesian() {
        double[][] jacobian = new double[6][6];
        this.computePVWithoutA();
        Vector3D position = this.partialPV.getPosition();
        Vector3D velocity = this.partialPV.getVelocity();
        double r2 = position.getNormSq();
        double r = FastMath.sqrt(r2);
        double r3 = r * r2;
        double mu = this.getMu();
        double sqrtMuA = FastMath.sqrt(this.a * mu);
        double a2 = this.a * this.a;
        double e2 = this.ex * this.ex + this.ey * this.ey;
        double oMe2 = 1.0 - e2;
        double epsilon = FastMath.sqrt(oMe2);
        double beta = 1.0 / (1.0 + epsilon);
        double ratio = epsilon * beta;
        double hx2 = this.hx * this.hx;
        double hy2 = this.hy * this.hy;
        double hxhy = this.hx * this.hy;
        Vector3D f = new Vector3D(1.0 - hy2 + hx2, 2.0 * hxhy, -2.0 * this.hy).normalize();
        Vector3D g = new Vector3D(2.0 * hxhy, 1.0 + hy2 - hx2, 2.0 * this.hx).normalize();
        Vector3D w = Vector3D.crossProduct(position, velocity).normalize();
        double x = Vector3D.dotProduct(position, f);
        double y = Vector3D.dotProduct(position, g);
        double xDot = Vector3D.dotProduct(velocity, f);
        double yDot = Vector3D.dotProduct(velocity, g);
        double c1 = this.a / (sqrtMuA * epsilon);
        double c2 = this.a * sqrtMuA * beta / r3;
        double c3 = sqrtMuA / (r3 * epsilon);
        Vector3D drDotSdEx = new Vector3D(c1 * xDot * yDot - c2 * this.ey * x - c3 * x * y, f, -c1 * xDot * xDot - c2 * this.ey * y + c3 * x * x, g);
        Vector3D drDotSdEy = new Vector3D(c1 * yDot * yDot + c2 * this.ex * x - c3 * y * y, f, -c1 * xDot * yDot + c2 * this.ex * y + c3 * x * y, g);
        Vector3D vectorAR = new Vector3D(2.0 * a2 / r3, position);
        Vector3D vectorARDot = new Vector3D(2.0 * a2 / mu, velocity);
        EquinoctialOrbit.fillHalfRow(1.0, vectorAR, jacobian[0], 0);
        EquinoctialOrbit.fillHalfRow(1.0, vectorARDot, jacobian[0], 3);
        double d1 = -this.a * ratio / r3;
        double d2 = (this.hy * xDot - this.hx * yDot) / (sqrtMuA * epsilon);
        double d3 = (this.hx * y - this.hy * x) / sqrtMuA;
        Vector3D vectorExRDot = new Vector3D((2.0 * x * yDot - xDot * y) / mu, g, -y * yDot / mu, f, -this.ey * d3 / epsilon, w);
        EquinoctialOrbit.fillHalfRow(this.ex * d1, position, -this.ey * d2, w, epsilon / sqrtMuA, drDotSdEy, jacobian[1], 0);
        EquinoctialOrbit.fillHalfRow(1.0, vectorExRDot, jacobian[1], 3);
        Vector3D vectorEyRDot = new Vector3D((2.0 * xDot * y - x * yDot) / mu, f, -x * xDot / mu, g, this.ex * d3 / epsilon, w);
        EquinoctialOrbit.fillHalfRow(this.ey * d1, position, this.ex * d2, w, -epsilon / sqrtMuA, drDotSdEx, jacobian[2], 0);
        EquinoctialOrbit.fillHalfRow(1.0, vectorEyRDot, jacobian[2], 3);
        double h = (1.0 + hx2 + hy2) / (2.0 * sqrtMuA * epsilon);
        EquinoctialOrbit.fillHalfRow(-h * xDot, w, jacobian[3], 0);
        EquinoctialOrbit.fillHalfRow(h * x, w, jacobian[3], 3);
        EquinoctialOrbit.fillHalfRow(-h * yDot, w, jacobian[4], 0);
        EquinoctialOrbit.fillHalfRow(h * y, w, jacobian[4], 3);
        double l = -ratio / sqrtMuA;
        EquinoctialOrbit.fillHalfRow(-1.0 / sqrtMuA, velocity, d2, w, l * this.ex, drDotSdEx, l * this.ey, drDotSdEy, jacobian[5], 0);
        EquinoctialOrbit.fillHalfRow(-2.0 / sqrtMuA, position, this.ex * beta, vectorEyRDot, -this.ey * beta, vectorExRDot, d3, w, jacobian[5], 3);
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianEccentricWrtCartesian() {
        double[][] jacobian = this.computeJacobianMeanWrtCartesian();
        double le = this.getLE();
        double cosLe = FastMath.cos(le);
        double sinLe = FastMath.sin(le);
        double aOr = 1.0 / (1.0 - this.ex * cosLe - this.ey * sinLe);
        double[] rowEx = jacobian[1];
        double[] rowEy = jacobian[2];
        double[] rowL = jacobian[5];
        for (int j = 0; j < 6; ++j) {
            rowL[j] = aOr * (rowL[j] + sinLe * rowEx[j] - cosLe * rowEy[j]);
        }
        return jacobian;
    }

    @Override
    protected double[][] computeJacobianTrueWrtCartesian() {
        double[][] jacobian = this.computeJacobianEccentricWrtCartesian();
        double le = this.getLE();
        double cosLe = FastMath.cos(le);
        double sinLe = FastMath.sin(le);
        double eSinE = this.ex * sinLe - this.ey * cosLe;
        double ecosE = this.ex * cosLe + this.ey * sinLe;
        double e2 = this.ex * this.ex + this.ey * this.ey;
        double epsilon = FastMath.sqrt(1.0 - e2);
        double onePeps = 1.0 + epsilon;
        double d = onePeps - ecosE;
        double cT = (d * d + eSinE * eSinE) / 2.0;
        double cE = ecosE * onePeps - e2;
        double cX = this.ex * eSinE / epsilon - this.ey + sinLe * onePeps;
        double cY = this.ey * eSinE / epsilon + this.ex - cosLe * onePeps;
        double factorLe = (cT + cE) / cT;
        double factorEx = cX / cT;
        double factorEy = cY / cT;
        double[] rowEx = jacobian[1];
        double[] rowEy = jacobian[2];
        double[] rowL = jacobian[5];
        for (int j = 0; j < 6; ++j) {
            rowL[j] = factorLe * rowL[j] + factorEx * rowEx[j] + factorEy * rowEy[j];
        }
        return jacobian;
    }

    @Override
    public void addKeplerContribution(PositionAngle type, double gm, double[] pDot) {
        double n = FastMath.sqrt(gm / this.a) / this.a;
        switch (type) {
            case MEAN: {
                pDot[5] = pDot[5] + n;
                break;
            }
            case ECCENTRIC: {
                double oMe2 = 1.0 - this.ex * this.ex - this.ey * this.ey;
                double ksi = 1.0 + this.ex * FastMath.cos(this.lv) + this.ey * FastMath.sin(this.lv);
                pDot[5] = pDot[5] + n * ksi / oMe2;
                break;
            }
            case TRUE: {
                double oMe2 = 1.0 - this.ex * this.ex - this.ey * this.ey;
                double ksi = 1.0 + this.ex * FastMath.cos(this.lv) + this.ey * FastMath.sin(this.lv);
                pDot[5] = pDot[5] + n * ksi * ksi / (oMe2 * FastMath.sqrt(oMe2));
                break;
            }
            default: {
                throw new OrekitInternalError(null);
            }
        }
    }

    public String toString() {
        return new StringBuffer().append("equinoctial parameters: ").append('{').append("a: ").append(this.a).append("; ex: ").append(this.ex).append("; ey: ").append(this.ey).append("; hx: ").append(this.hx).append("; hy: ").append(this.hy).append("; lv: ").append(FastMath.toDegrees(this.lv)).append(";}").toString();
    }

    private Object writeReplace() {
        return new DTO(this);
    }

    private static class DTO
    implements Serializable {
        private static final long serialVersionUID = 20170414L;
        private double[] d;
        private final Frame frame;

        private DTO(EquinoctialOrbit orbit) {
            TimeStampedPVCoordinates pv = orbit.getPVCoordinates();
            double epoch = FastMath.floor(pv.getDate().durationFrom(AbsoluteDate.J2000_EPOCH));
            double offset = pv.getDate().durationFrom(AbsoluteDate.J2000_EPOCH.shiftedBy(epoch));
            this.d = orbit.hasDerivatives() ? new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.hx, orbit.hy, orbit.lv, orbit.aDot, orbit.exDot, orbit.eyDot, orbit.hxDot, orbit.hyDot, orbit.lvDot} : new double[]{epoch, offset, orbit.getMu(), orbit.a, orbit.ex, orbit.ey, orbit.hx, orbit.hy, orbit.lv};
            this.frame = orbit.getFrame();
        }

        private Object readResolve() {
            if (this.d.length >= 15) {
                return new EquinoctialOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], this.d[9], this.d[10], this.d[11], this.d[12], this.d[13], this.d[14], PositionAngle.TRUE, this.frame, AbsoluteDate.J2000_EPOCH.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
            }
            return new EquinoctialOrbit(this.d[3], this.d[4], this.d[5], this.d[6], this.d[7], this.d[8], PositionAngle.TRUE, this.frame, AbsoluteDate.J2000_EPOCH.shiftedBy(this.d[0]).shiftedBy(this.d[1]), this.d[2]);
        }
    }
}

