# -*- coding: utf-8 -*-
from __future__ import print_function, unicode_literals
from eight import *
from future.utils import python_2_unicode_compatible

from . import config
from .project import writable_project
from .serialization import SerializedDict, PickledDict, CompoundJSONDict
import datetime
import pprint


@python_2_unicode_compatible
class Mapping(PickledDict):
    """A dictionary that maps object ids, like ``("Ecoinvent 2.2", 42)``, to integers. Needed because parameter arrays have integer ``row`` and ``column`` fields.

    File data is saved in ``mapping.pickle``.

    This dictionary does not support setting items directly; instead, use the ``add`` method to add multiple keys."""
    filename = "mapping.pickle"

    @writable_project
    def add(self, keys):
        """Add a set of keys. These keys can already be in the mapping; only new keys will be added.

        Args:
            * *keys* (list): The keys to add.

        """
        index = max(self.data.values()) if self.data else 0
        for i, key in enumerate(keys):
            if key not in self.data:
                self.data[key] = index + i + 1
        self.flush()

    def delete(self, keys):
        """Delete a set of keys.

        Args:
            *keys* (list): The keys to delete.

        """
        for key in keys:
            del self.data[key]
        self.flush()

    def __setitem__(self, key, value):
        raise NotImplemented

    def __str__(self):
        return u"Mapping from databases and methods to parameter indices."

    def __len__(self):
        return len(self.data)


class GeoMapping(Mapping):
    """A dictionary that maps location codes to integers. Needed because parameter arrays have integer ``geo`` fields.

    File data is stored in ``geomapping.pickle``.

    This dictionary does not support setting items directly; instead, use the ``add`` method to add multiple keys."""
    filename = "geomapping.pickle"

    def __init__(self, *args, **kwargs):
        super(GeoMapping, self).__init__(*args, **kwargs)
        # At a minimum, "GLO" should always be present
        if "GLO" not in self:
            self.add(["GLO"])

    def __unicode__(self):
        return u"Mapping from locations to parameter indices."


@python_2_unicode_compatible
class Databases(SerializedDict):
    """A dictionary for database metadata. This class includes methods to manage database versions. File data is saved in ``databases.json``."""
    filename = "databases.json"

    @writable_project
    def increment_version(self, database, number=None):
        """Increment the ``database`` version. Returns the new version."""
        self.data[database]["version"] += 1
        if number is not None:
            self.data[database]["number"] = number
        self.flush()
        return self.data[database]["version"]

    def version(self, database):
        """Return the ``database`` version"""
        return self.data[database].get("version")

    @writable_project
    def set_modified(self, database):
        self[database]['modified'] = datetime.datetime.now().isoformat()
        self.flush()

    @writable_project
    def set_dirty(self, database):
        self.set_modified(database)
        if self[database].get('dirty'):
            pass
        else:
            self[database]['dirty'] = True
            self.flush()

    @writable_project
    def clean(self):
        from . import Database
        for x in self:
            if self[x].get('dirty'):
                Database(x).process()
                del self[x]['dirty']
        self.flush()

    def __str__(self):
        if len(self) > 20:
            return ("Brightway2 databases metadata with {} objects, including:"
                    "{}\nUse `list(databases)` to get full list.").format(
                len(self),
                "".join(["\n\t{}".format(x) for x in sorted(self.data)[:10]])
            )
        else:
            return ("Brightway2 databases metadata with {} objects:{}").format(
                len(self),
                "".join(["\n\t{}".format(x) for x in sorted(self.data)])
            )

    @writable_project
    def __delitem__(self, name):
        from . import Database
        try:
            Database(name).delete()
        except:
            pass

        super(Databases, self).__delitem__(name)


@python_2_unicode_compatible
class CalculationSetups(PickledDict):
    """A dictionary for calculation setups: a set of functional units and LCIA methods."""
    filename = "setups.pickle"

    def __str__(self):
        return "Brightway2 calculation setups metadata with {} objects, including:\n{}".format(
            len(self.data),
            "".join(["\t{}\n".format(x) for x in sorted(self.data.keys())[:20]])
        )


@python_2_unicode_compatible
class Methods(CompoundJSONDict):
    """A dictionary for method metadata. File data is saved in ``methods.json``."""
    filename = "methods.json"

    def __str__(self):
        return u"Brightway2 methods metadata with %i objects" % len(
            self.data)


class WeightingMeta(Methods):
    """A dictionary for weighting metadata. File data is saved in ``methods.json``."""
    filename = "weightings.json"


class NormalizationMeta(Methods):
    """A dictionary for normalization metadata. File data is saved in ``methods.json``."""
    filename = "normalizations.json"


class Preferences(PickledDict):
    """A dictionary of project-specific preferences."""
    filename = "preferences.pickle"

    def __init__(self, *args, **kwargs):
        super(Preferences, self).__init__(*args, **kwargs)

        # Default preferences
        if 'use_cache' not in self:
            self['use_cache'] = True


databases = Databases()
geomapping = GeoMapping()
mapping = Mapping()
methods = Methods()
normalizations = NormalizationMeta()
preferences = Preferences()
weightings = WeightingMeta()
calculation_setups = CalculationSetups()
