# Copyright CNRS/Inria/UCA
# Contributor(s): Eric Debreuve (since 2019)
#
# eric.debreuve@cnrs.fr
#
# This software is governed by the CeCILL  license under French law and
# abiding by the rules of distribution of free software.  You can  use,
# modify and/ or redistribute the software under the terms of the CeCILL
# license as circulated by CEA, CNRS and INRIA at the following URL
# "http://www.cecill.info".
#
# As a counterpart to the access to the source code and  rights to copy,
# modify and redistribute granted by the license, users are provided only
# with a limited warranty  and the software's author,  the holder of the
# economic rights,  and the successive licensors  have only  limited
# liability.
#
# In this respect, the user's attention is drawn to the risks associated
# with loading,  using,  modifying and/or developing or reproducing the
# software by the user in light of its specific status of free software,
# that may mean  that it is complicated to manipulate,  and  that  also
# therefore means  that it is reserved for developers  and  experienced
# professionals having in-depth computer knowledge. Users are therefore
# encouraged to load and test the software's suitability as regards their
# requirements in conditions enabling the security of their systems and/or
# data to be ensured and,  more generally, to use and operate it in the
# same conditions as regards security.
#
# The fact that you are presently reading this means that you have had
# knowledge of the CeCILL license and that you accept its terms.

import datetime as dttm
import secrets as scrt
from configparser import ConfigParser as ini_config_t
from pathlib import Path as path_t

import dominate.tags as html
import flask as flsk
from flask_bootstrap import Bootstrap as BootstrapFlask
from flask_session import Session as flask_session_t

from daccuracy.flask.session.form import input_form_t
from daccuracy.flask.session.processing import ProcessSession
from daccuracy.flask.session.session import INI_SECTION, session_t
from daccuracy.flask.templates.constants import (
    ABOUT,
    MAX_IMAGE_SIZE,
    NAME,
    NAME_MEANING,
)
from daccuracy.flask.templates.session import SessionAsHTML, SessionOutputsAsHTML


HTML_TEMPLATE = "main.html"
BASE_FOLDER = "static"

RUNTIME_FOLDER = path_t(BASE_FOLDER) / "runtime"
INPUT_FOLDER = RUNTIME_FOLDER / "input"
OUTPUT_FOLDER = RUNTIME_FOLDER / "output"
SESSION_FOLDER = RUNTIME_FOLDER / "session"


GLOBAL_APP = flsk.Flask(__name__)


def HomePage(
    session_id: str,
    /,
    *,
    session: session_t = None,
    form: input_form_t = None,
) -> str:
    """"""
    has_no_data = session.form_file_fields is None
    outputs = session.SavableOutputs(session_id)

    return flsk.render_template(
        HTML_TEMPLATE,
        name=NAME,
        name_meaning=NAME_MEANING,
        about=ABOUT,
        session=SessionAsHTML(session, session_id),
        form=form,
        max_file_size=MAX_IMAGE_SIZE,
        outputs=SessionOutputsAsHTML(session),
        data_management=_DataManagement(session_id, has_no_data, outputs),
    )


def _DataManagement(
    session_id: str,
    has_no_data: bool,
    outputs: tuple[str, str] | None,
    /,
) -> html.html_tag | None:
    """"""
    if has_no_data:
        return None

    if outputs is None:
        output_url = None
    else:
        name, outputs = outputs
        path = OUTPUT_FOLDER / name
        with open(path, "w") as accessor:
            accessor.write(outputs)
        output_url = _URLOfPath(path)

    output = html.div()
    with output:
        if output_url is not None:
            html.a(
                html.button(
                    "Download Result",
                    type="button",
                    _class="btn btn-primary",
                    style="margin-top: 8pt",
                ),
                href=output_url,
                download="",
            )
            html.span(style="margin-right:48pt")
        html.a(
            html.button(
                html.b("Clear All Data"),
                type="button",
                _class="btn btn-primary",
                style="margin-top: 8pt",
            ),
            href=flsk.url_for(".DeleteSession", session_id=session_id),
        )

    return output


@GLOBAL_APP.route("/")
def LaunchSession() -> flsk.Response:
    """"""
    session_id = scrt.token_urlsafe()
    flsk.session[session_id] = session_t()

    return flsk.redirect(f"/{session_id}")


@GLOBAL_APP.route("/<session_id>", methods=("GET", "POST"))
def UpdateSessionPage(*, session_id: str = None) -> str:
    """"""
    session = flsk.session[session_id]
    form = input_form_t()  # Do not pass flask.request.form

    if flsk.request.method == "GET":
        form.Update(session.AsDictionary())

        if session.is_complete:
            return HomePage(
                session_id,
                session=session,
                form=form,
            )
    elif form.validate_on_submit():
        submission = form.Submission(INPUT_FOLDER)
        session.Update(submission, form.file_fields)

        if session.is_complete:
            ProcessSession(session)

            return HomePage(
                session_id,
                session=session,
                form=form,
            )

    return HomePage(session_id, session=session, form=form)


@GLOBAL_APP.route("/load/<session_id>", methods=("POST",))
def LoadSession(*, session_id: str = None) -> flsk.Response:
    """"""
    session = flsk.session[session_id]

    session_file = tuple(flsk.request.files.values())[0]
    new_as_str = session_file.read().decode("utf-8")
    new_as_ini = ini_config_t()
    new_as_ini.read_string(new_as_str)

    # Form file fields are not file_t's then, so the loaded session cannot be used as is
    for key, value in new_as_ini.items(INI_SECTION):
        session[key] = value

    return flsk.redirect(f"/{session_id}")


@GLOBAL_APP.route("/save/<session_id>")
def SaveSession(*, session_id: str = None) -> flsk.Response:
    """"""
    session = flsk.session[session_id]

    name, path = _SessionNameAndPath(session_id)
    with open(path, "w") as accessor:
        session.AsINI().write(accessor)
    # path.write_text(session_as_str)

    return flsk.send_file(
        path,
        mimetype="text/plain",
        as_attachment=True,
        download_name=name,
    )


@GLOBAL_APP.route("/delete/<session_id>")
def DeleteSession(*, session_id: str = None) -> flsk.Response:
    """"""
    session = flsk.session[session_id]

    session.DeleteInputFiles()
    session_t.DeleteOutputFiles(session_id, OUTPUT_FOLDER)
    _, path = _SessionNameAndPath(session_id)
    if path.is_file():
        path.unlink()

    flsk.session.pop(session_id, None)

    return flsk.redirect("/")


def _SessionNameAndPath(session_id: str, /) -> tuple[str, path_t]:
    """"""
    name = f"session-{session_id}.ini"
    path = OUTPUT_FOLDER / name

    return name, path


def _URLOfPath(path: path_t, /) -> str:
    """"""
    parts = path.parts

    return flsk.url_for(str(parts[0]), filename=str(path_t(*parts[1:])))


if __name__ == "__main__":
    #
    GLOBAL_APP.config.from_mapping(
        PREFERRED_URL_SCHEME="https",
        SESSION_TYPE="filesystem",
        PERMANENT_SESSION_LIFETIME=dttm.timedelta(hours=1),
        SESSION_FILE_DIR=SESSION_FOLDER,
        SECRET_KEY=scrt.token_bytes(),
        MAX_CONTENT_LENGTH=MAX_IMAGE_SIZE * 1024 * 1024,
    )
    flask_session_t(GLOBAL_APP)

    BootstrapFlask(GLOBAL_APP)
    GLOBAL_APP.run()
