"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const kendra = require("aws-cdk-lib/aws-kendra");
const iam = require("aws-cdk-lib/aws-iam");
const cdk = require("aws-cdk-lib");
const assertions_1 = require("aws-cdk-lib/assertions");
const defaults = require("@aws-solutions-constructs/core");
test('Launch with minimal code and check  structure', () => {
    const stack = new cdk.Stack();
    const testFunctionName = 'test-function-name24334';
    const testBucketName = 'test-bucket-name12344';
    const lambdaProps = {
        functionName: testFunctionName,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ]
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
        Environment: {
            Variables: {
                KENDRA_INDEX_ID: {
                    "Fn::GetAtt": ["samplekendraindexsample8A81A6C2", "Id"]
                }
            }
        },
    });
    template.hasResourceProperties("AWS::Kendra::Index", {
        RoleArn: {
            "Fn::GetAtt": [
                "samplekendraindexrolesample4F9E7B66",
                "Arn",
            ],
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["sampledatasourcerolesample05A05F8BD", "Arn"]
        },
    });
    // Confirm policy for Kendra index
    template.hasResourceProperties("AWS::IAM::Role", {
        Description: "Allow Kendra index to write CloudWatch Logs",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "cloudwatch:PutMetricData",
                            Condition: {
                                StringEquals: {
                                    "cloudwatch:namespace": "AWS/Kendra"
                                }
                            },
                            Effect: "Allow",
                            Resource: "*"
                        },
                        {
                            Action: "logs:CreateLogGroup",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "logs:DescribeLogGroups",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                                "logs:DescribeLogStream"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*:log-stream:*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "AllowLogging"
            }
        ],
    });
    // Confirm policy for Kendra index
    template.hasResourceProperties("AWS::IAM::Role", {
        Description: "Policy for Kendra S3 Data Source",
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "kendra.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "s3:GetObject",
                            Effect: "Allow",
                            Resource: `arn:aws:s3:::test-bucket-name12344/*`
                        },
                        {
                            Action: "s3:ListBucket",
                            Effect: "Allow",
                            Resource: `arn:aws:s3:::test-bucket-name12344`
                        },
                        {
                            Action: [
                                "kendra:BatchPutDocument",
                                "kendra:BatchDeleteDocument"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::GetAtt": [
                                    "samplekendraindexsample8A81A6C2",
                                    "Arn"
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "s3CrawlPolicy"
            }
        ]
    });
    // Confirm that Lambda function has QUERY access
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ],
        },
        Roles: [
            {
                Ref: "sampletestfunctionname24334ServiceRole99395A01"
            }
        ]
    });
});
test('Check pattern properties on minimal launch', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    const newConstruct = new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'your-bucket-name',
                    }
                }
            }
        ],
    });
    expect(newConstruct.lambdaFunction).toBeDefined();
    expect(newConstruct.kendraDataSources).toBeDefined();
    expect(newConstruct.kendraDataSources.length).toEqual(1);
    expect(newConstruct.kendraIndex).toBeDefined();
});
test('Launch with VPC', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name12539';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        deployVpc: true,
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    // Check the VPC
    template.hasResourceProperties("AWS::EC2::VPC", {
        CidrBlock: "10.0.0.0/16",
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
        InstanceTenancy: "default",
    });
    // Is the Lambda function associated with the VPC
    template.hasResourceProperties("AWS::Lambda::Function", {
        VpcConfig: {
            SecurityGroupIds: [
                {
                    "Fn::GetAtt": [
                        "sampleReplaceDefaultSecurityGroupsecuritygroupE5725669",
                        "GroupId"
                    ]
                }
            ],
            SubnetIds: [
                {
                    Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
                },
                {
                    Ref: "VpcisolatedSubnet2Subnet39217055"
                }
            ]
        }
    });
    // Check that the Lambda function Policy has proper network access
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "ec2:CreateNetworkInterface",
                        "ec2:DescribeNetworkInterfaces",
                        "ec2:DeleteNetworkInterface",
                        "ec2:AssignPrivateIpAddresses",
                        "ec2:UnassignPrivateIpAddresses"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {},
                {},
                {}
            ],
        },
        Roles: [
            {
                Ref: "sampleLambdaFunctionServiceRole7A3C4AF5"
            }
        ]
    });
    // Check for the Kendra endpoint in the VPC
    template.hasResourceProperties("AWS::EC2::VPCEndpoint", {
        ServiceName: {
            "Fn::Join": [
                "",
                [
                    "com.amazonaws.",
                    {
                        Ref: "AWS::Region"
                    },
                    ".kendra"
                ]
            ]
        },
        VpcId: {
            Ref: "Vpc8378EB38"
        },
        PrivateDnsEnabled: true,
        SecurityGroupIds: [
            {
                "Fn::GetAtt": [
                    "DefaultKENDRAsecuritygroup34536A79",
                    "GroupId"
                ]
            }
        ],
        SubnetIds: [
            {
                Ref: "VpcisolatedSubnet1SubnetE62B1B9B"
            },
            {
                Ref: "VpcisolatedSubnet2Subnet39217055"
            }
        ],
        VpcEndpointType: "Interface"
    });
});
test('Launch with existing lambda', () => {
    const stack = new cdk.Stack();
    const testTimeout = 17;
    const testFunctionName = 'test-name';
    const existingFunction = new lambda.Function(stack, 'existing-function', {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        timeout: cdk.Duration.seconds(testTimeout),
        functionName: testFunctionName
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        existingLambdaObj: existingFunction,
        kendraDataSourcesProps: [],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Lambda::Function", 1);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Timeout: testTimeout,
        FunctionName: testFunctionName,
    });
});
test('Confirm error with data source with no bucket name', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'sample', {
            lambdaFunctionProps: lambdaProps,
            kendraDataSourcesProps: [{
                    type: 'S3',
                    dataSourceConfiguration: {
                        s3Configuration: {}
                    }
                }
            ],
        });
    };
    expect(app).toThrowError(/Error - an S3 Kendra DataSource requires the DataSourceConfiguration.S3Configuration.bucketName prop/);
});
test('Launch with existing vpc', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        existingVpc: defaults.getTestVpc(stack),
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::EC2::VPC", 1);
    template.resourceCountIs("AWS::EC2::VPCEndpoint", 1);
    template.hasResourceProperties("AWS::EC2::VPC", {
        EnableDnsHostnames: true,
        EnableDnsSupport: true,
    });
});
test('Launch with Read/Write permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["ReaD", "SubmitFeedBack", "wrITE"], // this also checks case sensitivity
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: [
                        "kendra:Associate*",
                        "kendra:BatchPut*",
                        "kendra:Clear",
                        "kendra:Create*",
                        "kendra:Delete*",
                        "kendra:Disassociate*",
                        "kendra:Put*",
                        "kendra:Update*",
                        "kendra:Start*",
                        "kendra:Submit*",
                        "kendra:Stop*",
                        "kendra:TagResource",
                        "kendra:UntagResource"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with Write permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Associate*",
                        "kendra:BatchPut*",
                        "kendra:Clear",
                        "kendra:Create*",
                        "kendra:Delete*",
                        "kendra:Disassociate*",
                        "kendra:Put*",
                        "kendra:Update*",
                        "kendra:Start*",
                        "kendra:Submit*",
                        "kendra:Stop*",
                        "kendra:TagResource",
                        "kendra:UntagResource"
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with Read permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["READ"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with SubmitFeedback permissions on kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["SUBMITFEEDBACK"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ]
        }
    });
});
test('Launch with existing kendra index', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    const existingRole = new iam.Role(stack, 'existing-role', {
        assumedBy: new iam.ServicePrincipal('kendra.amazonaws.com'),
    });
    const existingIndex = new kendra.CfnIndex(stack, 'existing-index', {
        edition: 'ENTERPRISE',
        name: 'existingIndex',
        roleArn: existingRole.roleArn
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        existingKendraIndexObj: existingIndex,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    // Make sure we didn't create an index anyway
    template.resourceCountIs("AWS::Kendra::Index", 1);
    template.hasResourceProperties("AWS::Kendra::Index", {
        Edition: 'ENTERPRISE'
    });
});
test('Launch with S3 data source with overridden defaults', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name223423';
    const testInclusionPattern = 'this-folder';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        inclusionPatterns: [testInclusionPattern],
                        bucketName: testBucketName,
                    }
                }
            }
        ],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 1);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                InclusionPatterns: [testInclusionPattern],
                BucketName: testBucketName
            },
        }
    });
});
test('Launch with S3 data source and unsupported data source', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22';
    const nonImplementedSourceType = "WEBCRAWLER";
    const nonImplementedSourceName = "test-other-source";
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    // Create a role
    const fakeRole = new iam.Role(stack, 'fakeRole', {
        assumedBy: new iam.ServicePrincipal('kendra.amazonaws.com'),
        roleName: 'externalFakeRole'
    });
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            },
            {
                name: nonImplementedSourceName,
                roleArn: fakeRole.roleArn,
                type: nonImplementedSourceType,
            }],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 2);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["sampledatasourcerolesample05A05F8BD", "Arn"]
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Name: nonImplementedSourceName,
        Type: nonImplementedSourceType,
    });
});
test('Launch with multiple S3 data sources', () => {
    const stack = new cdk.Stack();
    const testBucketName = 'test-bucket-name22342';
    const secondBucketName = 'second-bucket-name22342342';
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'two-sources', {
        lambdaFunctionProps: lambdaProps,
        indexPermissions: ["WRITE"],
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            },
            {
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: secondBucketName,
                    }
                }
            }],
    });
    const template = assertions_1.Template.fromStack(stack);
    template.resourceCountIs("AWS::Kendra::DataSource", 2);
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["twosourcesdatasourceroletwosources0B8E24996", "Arn"]
        },
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: secondBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["twosourcesdatasourceroletwosources164176C5E", "Arn"]
        },
    });
    // Confirm we have Roles that appear correct
    template.resourceCountIs("AWS::IAM::Role", 4);
    template.hasResourceProperties("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "kendra.amazonaws.com"
                    }
                }
            ]
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "s3:GetObject",
                            Effect: "Allow",
                            Resource: assertions_1.Match.stringLikeRegexp(`.*test-bucket-name.*`)
                        },
                        assertions_1.Match.anyValue(),
                        assertions_1.Match.anyValue()
                    ],
                }
            }
        ]
    });
    template.hasResourceProperties("AWS::IAM::Role", {
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "kendra.amazonaws.com"
                    }
                }
            ]
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "s3:GetObject",
                            Effect: "Allow",
                            Resource: assertions_1.Match.stringLikeRegexp(`.*second-bucket-name.*`)
                        },
                        assertions_1.Match.anyValue(),
                        assertions_1.Match.anyValue()
                    ],
                }
            }
        ]
    });
});
test('Test with custom environment variable name', () => {
    const stack = new cdk.Stack();
    const lambdaProps = {
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'your-bucket-name',
                    }
                }
            }
        ],
        indexIdEnvironmentVariableName: "MY_VAR_NAME",
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        Environment: {
            Variables: {
                MY_VAR_NAME: {
                    "Fn::GetAtt": ["samplekendraindexsample8A81A6C2", "Id"]
                }
            }
        }
    });
});
test('Confirm CheckVpcProps is being called', () => {
    const stack = new cdk.Stack();
    const testFunctionName = 'test-function-name24334';
    const testBucketName = 'test-bucket-name12344';
    const vpc = defaults.buildVpc(stack, {
        defaultVpcProps: defaults.DefaultIsolatedVpcProps(),
        constructVpcProps: {
            enableDnsHostnames: true,
            enableDnsSupport: true,
        },
    });
    const lambdaProps = {
        functionName: testFunctionName,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    const props = {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }
        ],
        deployVpc: true,
        existingVpc: vpc
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'sample', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide an existingVpc or some combination of deployVpc and vpcProps, but not both.\n');
});
test('Confirm call to CheckLambdaProps', () => {
    // Initial Setup
    const stack = new cdk.Stack();
    const lambdaFunction = new lambda.Function(stack, 'a-function', {
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler',
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    });
    const props = {
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: 'testBucketName',
                    }
                }
            }
        ],
        lambdaFunctionProps: {
            runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
            handler: 'index.handler',
            code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        },
        existingLambdaObj: lambdaFunction,
    };
    const app = () => {
        new lib_1.LambdaToKendra(stack, 'test-construct', props);
    };
    // Assertion
    expect(app).toThrowError('Error - Either provide lambdaFunctionProps or existingLambdaObj, but not both.\n');
});
test('Launch with non default index type', () => {
    const stack = new cdk.Stack();
    const testFunctionName = 'test-function-name24334';
    const testBucketName = 'test-bucket-name12344';
    const indexType = 'GEN_AI_ENTERPRISE_EDITION';
    const lambdaProps = {
        functionName: testFunctionName,
        code: lambda.Code.fromAsset(`${__dirname}/lambda`),
        runtime: defaults.COMMERCIAL_REGION_LAMBDA_NODE_RUNTIME,
        handler: 'index.handler'
    };
    new lib_1.LambdaToKendra(stack, 'sample', {
        lambdaFunctionProps: lambdaProps,
        kendraDataSourcesProps: [{
                type: 'S3',
                dataSourceConfiguration: {
                    s3Configuration: {
                        bucketName: testBucketName,
                    }
                }
            }],
        kendraIndexProps: {
            edition: indexType,
            name: `enterprise-index`
        },
    });
    const template = assertions_1.Template.fromStack(stack);
    template.hasResourceProperties("AWS::Lambda::Function", {
        FunctionName: testFunctionName,
        Environment: {
            Variables: {
                KENDRA_INDEX_ID: {
                    "Fn::GetAtt": ["samplekendraindexsample8A81A6C2", "Id"]
                }
            }
        },
    });
    template.hasResourceProperties("AWS::Kendra::Index", {
        RoleArn: {
            "Fn::GetAtt": [
                "samplekendraindexrolesample4F9E7B66",
                "Arn",
            ],
        },
        Edition: indexType
    });
    template.hasResourceProperties("AWS::Kendra::DataSource", {
        Type: 'S3',
        DataSourceConfiguration: {
            S3Configuration: {
                BucketName: testBucketName
            },
        },
        RoleArn: {
            "Fn::GetAtt": ["sampledatasourcerolesample05A05F8BD", "Arn"]
        },
    });
    // Confirm policy for Kendra index
    template.hasResourceProperties("AWS::IAM::Role", {
        Description: "Allow Kendra index to write CloudWatch Logs",
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "cloudwatch:PutMetricData",
                            Condition: {
                                StringEquals: {
                                    "cloudwatch:namespace": "AWS/Kendra"
                                }
                            },
                            Effect: "Allow",
                            Resource: "*"
                        },
                        {
                            Action: "logs:CreateLogGroup",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:aws:logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: "logs:DescribeLogGroups",
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*"
                                    ]
                                ]
                            }
                        },
                        {
                            Action: [
                                "logs:CreateLogStream",
                                "logs:PutLogEvents",
                                "logs:DescribeLogStream"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::Join": [
                                    "",
                                    [
                                        "arn:",
                                        {
                                            Ref: "AWS::Partition"
                                        },
                                        ":logs:",
                                        {
                                            Ref: "AWS::Region"
                                        },
                                        ":",
                                        {
                                            Ref: "AWS::AccountId"
                                        },
                                        ":log-group:/aws/kendra/*:log-stream:*"
                                    ]
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "AllowLogging"
            }
        ],
    });
    // Confirm policy for Kendra index
    template.hasResourceProperties("AWS::IAM::Role", {
        Description: "Policy for Kendra S3 Data Source",
        AssumeRolePolicyDocument: {
            Statement: [
                {
                    Action: "sts:AssumeRole",
                    Effect: "Allow",
                    Principal: {
                        Service: "kendra.amazonaws.com"
                    }
                }
            ],
            Version: "2012-10-17"
        },
        Policies: [
            {
                PolicyDocument: {
                    Statement: [
                        {
                            Action: "s3:GetObject",
                            Effect: "Allow",
                            Resource: `arn:aws:s3:::test-bucket-name12344/*`
                        },
                        {
                            Action: "s3:ListBucket",
                            Effect: "Allow",
                            Resource: `arn:aws:s3:::test-bucket-name12344`
                        },
                        {
                            Action: [
                                "kendra:BatchPutDocument",
                                "kendra:BatchDeleteDocument"
                            ],
                            Effect: "Allow",
                            Resource: {
                                "Fn::GetAtt": [
                                    "samplekendraindexsample8A81A6C2",
                                    "Arn"
                                ]
                            }
                        }
                    ],
                    Version: "2012-10-17"
                },
                PolicyName: "s3CrawlPolicy"
            }
        ]
    });
    // Confirm that Lambda function has QUERY access
    template.hasResourceProperties("AWS::IAM::Policy", {
        PolicyDocument: {
            Statement: [
                {
                    Action: [
                        "xray:PutTraceSegments",
                        "xray:PutTelemetryRecords"
                    ],
                    Effect: "Allow",
                    Resource: "*"
                },
                {
                    Action: [
                        "kendra:Query",
                        "kendra:Describe*",
                        "kendra:Get*",
                        "kendra:BatchGet*",
                        "kendra:List*",
                        "kendra:Retrieve",
                    ],
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                },
                {
                    Action: "kendra:SubmitFeedback",
                    Effect: "Allow",
                    Resource: {
                        "Fn::GetAtt": [
                            "samplekendraindexsample8A81A6C2",
                            "Arn"
                        ]
                    }
                }
            ],
        },
        Roles: [
            {
                Ref: "sampletestfunctionname24334ServiceRole99395A01"
            }
        ]
    });
});
//# sourceMappingURL=data:application/json;base64,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