"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const construct_1 = require("./construct");
const runtime_info_1 = require("./private/runtime-info");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps_and_stacks.html
 */
class App extends construct_1.Construct {
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        super(undefined, '');
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        if (props.runtimeInfo === false) {
            this.node.setContext(cxapi.DISABLE_VERSION_REPORTING, true);
        }
        // both are reverse logic
        this.runtimeInfo = this.node.tryGetContext(cxapi.DISABLE_VERSION_REPORTING) ? false : true;
        this.outdir = props.outdir || process.env[cxapi.OUTDIR_ENV];
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
    }
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Synthesizes a cloud assembly for this app. Emits it to the directory
     * specified by `outdir`.
     *
     * @returns a `CloudAssembly` which can be used to inspect synthesized
     * artifacts such as CloudFormation templates and assets.
     */
    synth() {
        // we already have a cloud assembly, no-op for you
        if (this._assembly) {
            return this._assembly;
        }
        const assembly = construct_1.ConstructNode.synth(this.node, {
            outdir: this.outdir,
            runtimeInfo: this.runtimeInfo ? runtime_info_1.collectRuntimeInformation() : undefined
        });
        this._assembly = assembly;
        return assembly;
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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