"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const stack_1 = require("./stack");
const token_1 = require("./token");
/**
 * Base class for the model side of context providers
 *
 * Instances of this class communicate with context provider plugins in the 'cdk
 * toolkit' via context variables (input), outputting specialized queries for
 * more context variables (output).
 *
 * ContextProvider needs access to a Construct to hook into the context mechanism.
 *
 * @experimental
 */
class ContextProvider {
    /**
     * @returns the context key or undefined if a key cannot be rendered (due to tokens used in any of the props)
     */
    static getKey(scope, options) {
        const stack = stack_1.Stack.of(scope);
        const props = {
            account: stack.account,
            region: stack.region,
            ...options.props || {},
        };
        if (Object.values(props).find(x => token_1.Token.isUnresolved(x))) {
            throw new Error(`Cannot determine scope for context provider ${options.provider}.\n` +
                'This usually happens when one or more of the provider props have unresolved tokens');
        }
        const propStrings = propsToArray(props);
        return {
            key: `${options.provider}:${propStrings.join(':')}`,
            props
        };
    }
    static getValue(scope, options) {
        const stack = stack_1.Stack.of(scope);
        if (token_1.Token.isUnresolved(stack.account) || token_1.Token.isUnresolved(stack.region)) {
            throw new Error(`Cannot retrieve value from context provider ${options.provider} since account/region ` +
                'are not specified at the stack level. Either configure "env" with explicit account and region when ' +
                'you define your stack, or use the environment variables "CDK_DEFAULT_ACCOUNT" and "CDK_DEFAULT_REGION" ' +
                'to inherit environment information from the CLI (not recommended for production stacks)');
        }
        const { key, props } = this.getKey(scope, options);
        const value = scope.node.tryGetContext(key);
        const providerError = extractProviderError(value);
        // if context is missing or an error occurred during context retrieval,
        // report and return a dummy value.
        if (value === undefined || providerError !== undefined) {
            stack.reportMissingContext({ key, props, provider: options.provider, });
            if (providerError !== undefined) {
                scope.node.addError(providerError);
            }
            return { value: options.dummyValue };
        }
        return { value };
    }
    constructor() { }
}
exports.ContextProvider = ContextProvider;
/**
 * If the context value represents an error, return the error message
 */
function extractProviderError(value) {
    if (typeof value === 'object' && value !== null) {
        return value[cxapi.PROVIDER_ERROR_KEY];
    }
    return undefined;
}
/**
 * Quote colons in all strings so that we can undo the quoting at a later point
 *
 * We'll use $ as a quoting character, for no particularly good reason other
 * than that \ is going to lead to quoting hell when the keys are stored in JSON.
 */
function colonQuote(xs) {
    return xs.replace('$', '$$').replace(':', '$:');
}
function propsToArray(props, keyPrefix = '') {
    const ret = [];
    for (const key of Object.keys(props)) {
        // skip undefined values
        if (props[key] === undefined) {
            continue;
        }
        switch (typeof props[key]) {
            case 'object': {
                ret.push(...propsToArray(props[key], `${keyPrefix}${key}.`));
                break;
            }
            case 'string': {
                ret.push(`${keyPrefix}${key}=${colonQuote(props[key])}`);
                break;
            }
            default: {
                ret.push(`${keyPrefix}${key}=${JSON.stringify(props[key])}`);
                break;
            }
        }
    }
    ret.sort();
    return ret;
}
//# sourceMappingURL=data:application/json;base64,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