"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
// ----------------------------------------------------
// CROSS REFERENCES
// ----------------------------------------------------
const cfn_element_1 = require("../cfn-element");
const cfn_output_1 = require("../cfn-output");
const cfn_parameter_1 = require("../cfn-parameter");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
const token_1 = require("../token");
const cfn_reference_1 = require("./cfn-reference");
const intrinsic_1 = require("./intrinsic");
const resolve_1 = require("./resolve");
const uniqueid_1 = require("./uniqueid");
/**
 * This is called from the App level to resolve all references defined. Each
 * reference is resolved based on it's consumption context.
 */
function resolveReferences(scope) {
    const edges = findAllReferences(scope);
    for (const { source, value } of edges) {
        const consumer = stack_1.Stack.of(source);
        // resolve the value in the context of the consumer
        if (!value.hasValueForStack(consumer)) {
            const resolved = resolveValue(consumer, value);
            value.assignValueForStack(consumer, resolved);
        }
    }
}
exports.resolveReferences = resolveReferences;
/**
 * Resolves the value for `reference` in the context of `consumer`.
 */
function resolveValue(consumer, reference) {
    const producer = stack_1.Stack.of(reference.target);
    // produce and consumer stacks are the same, we can just return the value itself.
    if (producer === consumer) {
        return reference;
    }
    // unsupported: stacks from different apps
    if (producer.node.root !== consumer.node.root) {
        throw new Error('Cannot reference across apps. Consuming and producing stacks must be defined within the same CDK app.');
    }
    // unsupported: stacks are not in the same environment
    if (producer.environment !== consumer.environment) {
        throw new Error(`Stack "${consumer.node.path}" cannot consume a cross reference from stack "${producer.node.path}". ` +
            'Cross stack references are only supported for stacks deployed to the same environment or between nested stacks and their parent stack');
    }
    // ----------------------------------------------------------------------
    // consumer is nested in the producer (directly or indirectly)
    // ----------------------------------------------------------------------
    // if the consumer is nested within the producer (directly or indirectly),
    // wire through a CloudFormation parameter and then resolve the reference with
    // the parent stack as the consumer.
    if (consumer.nestedStackParent && isNested(consumer, producer)) {
        const parameterValue = resolveValue(consumer.nestedStackParent, reference);
        return createNestedStackParameter(consumer, reference, parameterValue);
    }
    // ----------------------------------------------------------------------
    // producer is a nested stack
    // ----------------------------------------------------------------------
    // if the producer is nested, always publish the value through a
    // cloudformation output and resolve recursively with the Fn::GetAtt
    // of the output in the parent stack.
    // one might ask, if the consumer is not a parent of the producer,
    // why not just use export/import? the reason is that we cannot
    // generate an "export name" from a nested stack because the export
    // name must contain the stack name to ensure uniqueness, and we
    // don't know the stack name of a nested stack before we deploy it.
    // therefore, we can only export from a top-level stack.
    if (producer.nested) {
        const outputValue = createNestedStackOutput(producer, reference);
        return resolveValue(consumer, outputValue);
    }
    // ----------------------------------------------------------------------
    // export/import
    // ----------------------------------------------------------------------
    // export the value through a cloudformation "export name" and use an
    // Fn::ImportValue in the consumption site.
    // add a dependency between the producer and the consumer. dependency logic
    // will take care of applying the dependency at the right level (e.g. the
    // top-level stacks).
    consumer.addDependency(producer, `${consumer.node.path} -> ${reference.target.node.path}.${reference.displayName}`);
    return createImportValue(reference);
}
/**
 * Finds all the CloudFormation references in a construct tree.
 */
function findAllReferences(root) {
    const result = new Array();
    for (const consumer of root.node.findAll()) {
        // include only CfnElements (i.e. resources)
        if (!cfn_element_1.CfnElement.isCfnElement(consumer)) {
            continue;
        }
        try {
            const tokens = resolve_1.findTokens(consumer, () => consumer._toCloudFormation());
            // iterate over all the tokens (e.g. intrinsic functions, lazies, etc) that
            // were found in the cloudformation representation of this resource.
            for (const token of tokens) {
                // include only CfnReferences (i.e. "Ref" and "Fn::GetAtt")
                if (!cfn_reference_1.CfnReference.isCfnReference(token)) {
                    continue;
                }
                result.push({
                    source: consumer,
                    value: token
                });
            }
        }
        catch (e) {
            // Note: it might be that the properties of the CFN object aren't valid.
            // This will usually be preventatively caught in a construct's validate()
            // and turned into a nicely descriptive error, but we're running prepare()
            // before validate(). Swallow errors that occur because the CFN layer
            // doesn't validate completely.
            //
            // This does make the assumption that the error will not be rectified,
            // but the error will be thrown later on anyway. If the error doesn't
            // get thrown down the line, we may miss references.
            if (e.type === 'CfnSynthesisError') {
                continue;
            }
            throw e;
        }
    }
    return result;
}
// ------------------------------------------------------------------------------------------------
// export/import
// ------------------------------------------------------------------------------------------------
/**
 * Imports a value from another stack by creating an "Output" with an "ExportName"
 * and returning an "Fn::ImportValue" token.
 */
function createImportValue(reference) {
    const exportingStack = stack_1.Stack.of(reference.target);
    // Ensure a singleton "Exports" scoping Construct
    // This mostly exists to trigger LogicalID munging, which would be
    // disabled if we parented constructs directly under Stack.
    // Also it nicely prevents likely construct name clashes
    const exportsScope = getCreateExportsScope(exportingStack);
    // Ensure a singleton CfnOutput for this value
    const resolved = exportingStack.resolve(reference);
    const id = 'Output' + JSON.stringify(resolved);
    const exportName = generateExportName(exportsScope, id);
    if (token_1.Token.isUnresolved(exportName)) {
        throw new Error(`unresolved token in generated export name: ${JSON.stringify(exportingStack.resolve(exportName))}`);
    }
    const output = exportsScope.node.tryFindChild(id);
    if (!output) {
        new cfn_output_1.CfnOutput(exportsScope, id, { value: token_1.Token.asString(reference), exportName });
    }
    // We want to return an actual FnImportValue Token here, but Fn.importValue() returns a 'string',
    // so construct one in-place.
    return new intrinsic_1.Intrinsic({ 'Fn::ImportValue': exportName });
}
function getCreateExportsScope(stack) {
    const exportsName = 'Exports';
    let stackExports = stack.node.tryFindChild(exportsName);
    if (stackExports === undefined) {
        stackExports = new construct_compat_1.Construct(stack, exportsName);
    }
    return stackExports;
}
function generateExportName(stackExports, id) {
    const stack = stack_1.Stack.of(stackExports);
    const components = [...stackExports.node.scopes.slice(2).map(c => c.node.id), id];
    const prefix = stack.stackName ? stack.stackName + ':' : '';
    const exportName = prefix + uniqueid_1.makeUniqueId(components);
    return exportName;
}
// ------------------------------------------------------------------------------------------------
// nested stacks
// ------------------------------------------------------------------------------------------------
/**
 * Adds a CloudFormation parameter to a nested stack and assigns it with the
 * value of the reference.
 */
function createNestedStackParameter(nested, reference, value) {
    // we call "this.resolve" to ensure that tokens do not creep in (for example, if the reference display name includes tokens)
    const paramId = nested.resolve(`reference-to-${reference.target.node.uniqueId}.${reference.displayName}`);
    let param = nested.node.tryFindChild(paramId);
    if (!param) {
        param = new cfn_parameter_1.CfnParameter(nested, paramId, { type: 'String' });
        // Ugly little hack until we move NestedStack to this module.
        if (!('setParameter' in nested)) {
            throw new Error('assertion failed: nested stack should have a "setParameter" method');
        }
        nested.setParameter(param.logicalId, token_1.Token.asString(value));
    }
    return param.value;
}
/**
 * Adds a CloudFormation output to a nested stack and returns an "Fn::GetAtt"
 * intrinsic that can be used to reference this output in the parent stack.
 */
function createNestedStackOutput(producer, reference) {
    const outputId = `${reference.target.node.uniqueId}${reference.displayName}`;
    let output = producer.node.tryFindChild(outputId);
    if (!output) {
        output = new cfn_output_1.CfnOutput(producer, outputId, { value: token_1.Token.asString(reference) });
    }
    if (!producer.nestedStackResource) {
        throw new Error('assertion failed');
    }
    return producer.nestedStackResource.getAtt(`Outputs.${output.logicalId}`);
}
/**
 * @returns true if this stack is a direct or indirect parent of the nested
 * stack `nested`.
 *
 * If `child` is not a nested stack, always returns `false` because it can't
 * have a parent, dah.
 */
function isNested(nested, parent) {
    // if the parent is a direct parent
    if (nested.nestedStackParent === parent) {
        return true;
    }
    // we reached a top-level (non-nested) stack without finding the parent
    if (!nested.nestedStackParent) {
        return false;
    }
    // recurse with the child's direct parent
    return isNested(nested.nestedStackParent, parent);
}
//# sourceMappingURL=data:application/json;base64,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