"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const fs = require("fs");
const path = require("path");
const cloud_assembly_schema_1 = require("@aws-cdk/cloud-assembly-schema");
const construct_compat_1 = require("../construct-compat");
const stack_1 = require("../stack");
const tree_1 = require("../tree");
const FILE_PATH = 'tree.json';
/**
 * Construct that is automatically attached to the top-level `App`.
 * This generates, as part of synthesis, a file containing the construct tree and the metadata for each node in the tree.
 * The output is in a tree format so as to preserve the construct hierarchy.
 *
 * @experimental
 */
class TreeMetadata extends construct_compat_1.Construct {
    constructor(scope) {
        super(scope, 'Tree');
    }
    synthesize(session) {
        const lookup = {};
        const visit = (construct) => {
            const children = construct.node.children.map((c) => {
                try {
                    return visit(c);
                }
                catch (e) {
                    this.node.addWarning(`Failed to render tree metadata for node [${c.node.id}]. Reason: ${e}`);
                    return undefined;
                }
            });
            const childrenMap = children
                .filter((child) => child !== undefined)
                .reduce((map, child) => Object.assign(map, { [child.id]: child }), {});
            const node = {
                id: construct.node.id || 'App',
                path: construct.node.path,
                children: Object.keys(childrenMap).length === 0 ? undefined : childrenMap,
                attributes: this.synthAttributes(construct)
            };
            lookup[node.path] = node;
            return node;
        };
        const tree = {
            version: 'tree-0.1',
            tree: visit(this.node.root),
        };
        const builder = session.assembly;
        fs.writeFileSync(path.join(builder.outdir, FILE_PATH), JSON.stringify(tree, undefined, 2), { encoding: 'utf-8' });
        builder.addArtifact('Tree', {
            type: cloud_assembly_schema_1.ArtifactType.CDK_TREE,
            properties: {
                file: FILE_PATH
            }
        });
    }
    synthAttributes(construct) {
        // check if a construct implements IInspectable
        function canInspect(inspectable) {
            return inspectable.inspect !== undefined;
        }
        const inspector = new tree_1.TreeInspector();
        // get attributes from the inspector
        if (canInspect(construct)) {
            construct.inspect(inspector);
            return stack_1.Stack.of(construct).resolve(inspector.attributes);
        }
        return undefined;
    }
}
exports.TreeMetadata = TreeMetadata;
//# sourceMappingURL=data:application/json;base64,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