"use strict";
const lib_1 = require("../lib");
const intrinsic_1 = require("../lib/private/intrinsic");
const util_1 = require("./util");
module.exports = {
    'create from components with defaults'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'sqs',
            resource: 'myqueuename'
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.resolve(arn), stack.resolve(`arn:${pseudo.partition}:sqs:${pseudo.region}:${pseudo.accountId}:myqueuename`));
        test.done();
    },
    'create from components with specific values for the various components'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'dynamodb',
            resource: 'table',
            account: '123456789012',
            region: 'us-east-1',
            partition: 'aws-cn',
            resourceName: 'mytable/stream/label'
        });
        test.deepEqual(stack.resolve(arn), 'arn:aws-cn:dynamodb:us-east-1:123456789012:table/mytable/stream/label');
        test.done();
    },
    'allow empty string in components'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 's3',
            resource: 'my-bucket',
            account: '',
            region: '',
            partition: 'aws-cn',
        });
        test.deepEqual(stack.resolve(arn), 'arn:aws-cn:s3:::my-bucket');
        test.done();
    },
    'resourcePathSep can be set to ":" instead of the default "/"'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'codedeploy',
            resource: 'application',
            sep: ':',
            resourceName: 'WordPress_App'
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.resolve(arn), stack.resolve(`arn:${pseudo.partition}:codedeploy:${pseudo.region}:${pseudo.accountId}:application:WordPress_App`));
        test.done();
    },
    'resourcePathSep can be set to "" instead of the default "/"'(test) {
        const stack = new lib_1.Stack();
        const arn = stack.formatArn({
            service: 'ssm',
            resource: 'parameter',
            sep: '',
            resourceName: '/parameter-name'
        });
        const pseudo = new lib_1.ScopedAws(stack);
        test.deepEqual(stack.resolve(arn), stack.resolve(`arn:${pseudo.partition}:ssm:${pseudo.region}:${pseudo.accountId}:parameter/parameter-name`));
        test.done();
    },
    'fails if resourcePathSep is neither ":" nor "/"'(test) {
        const stack = new lib_1.Stack();
        test.throws(() => stack.formatArn({
            service: 'foo',
            resource: 'bar',
            sep: 'x'
        }));
        test.done();
    },
    'Arn.parse(s)': {
        'fails': {
            'if doesn\'t start with "arn:"'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('barn:foo:x:a:1:2'), /ARNs must start with "arn:": barn:foo/);
                test.done();
            },
            'if the ARN doesnt have enough components'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:is:too:short'), /ARNs must have at least 6 components: arn:is:too:short/);
                test.done();
            },
            'if "service" is not specified'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:aws::4:5:6'), /The `service` component \(3rd component\) is required/);
                test.done();
            },
            'if "resource" is not specified'(test) {
                const stack = new lib_1.Stack();
                test.throws(() => stack.parseArn('arn:aws:service:::'), /The `resource` component \(6th component\) is required/);
                test.done();
            }
        },
        'various successful parses'(test) {
            const stack = new lib_1.Stack();
            const tests = {
                'arn:aws:a4b:region:accountid:resourcetype/resource': {
                    partition: 'aws',
                    service: 'a4b',
                    region: 'region',
                    account: 'accountid',
                    resource: 'resourcetype',
                    resourceName: 'resource',
                    sep: '/'
                },
                'arn:aws:apigateway:us-east-1::a123456789012bc3de45678901f23a45:/test/mydemoresource/*': {
                    partition: 'aws',
                    service: 'apigateway',
                    region: 'us-east-1',
                    account: '',
                    resource: 'a123456789012bc3de45678901f23a45',
                    sep: ':',
                    resourceName: '/test/mydemoresource/*'
                },
                'arn:aws-cn:cloud9::123456789012:environment:81e900317347585a0601e04c8d52eaEX': {
                    partition: 'aws-cn',
                    service: 'cloud9',
                    region: '',
                    account: '123456789012',
                    resource: 'environment',
                    resourceName: '81e900317347585a0601e04c8d52eaEX',
                    sep: ':'
                },
                'arn::cognito-sync:::identitypool/us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla': {
                    service: 'cognito-sync',
                    region: '',
                    account: '',
                    resource: 'identitypool',
                    resourceName: 'us-east-1:1a1a1a1a-ffff-1111-9999-12345678:bla',
                    sep: '/'
                },
                'arn:aws:s3:::my_corporate_bucket': {
                    partition: 'aws',
                    service: 's3',
                    region: '',
                    account: '',
                    resource: 'my_corporate_bucket'
                }
            };
            Object.keys(tests).forEach(arn => {
                const expected = tests[arn];
                test.deepEqual(stack.parseArn(arn), expected, arn);
            });
            test.done();
        },
        'a Token with : separator'(test) {
            const stack = new lib_1.Stack();
            const theToken = { Ref: 'SomeParameter' };
            const parsed = stack.parseArn(new intrinsic_1.Intrinsic(theToken).toString(), ':');
            test.deepEqual(stack.resolve(parsed.partition), { 'Fn::Select': [1, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.service), { 'Fn::Select': [2, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.region), { 'Fn::Select': [3, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.account), { 'Fn::Select': [4, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.resource), { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] });
            test.deepEqual(stack.resolve(parsed.resourceName), { 'Fn::Select': [6, { 'Fn::Split': [':', theToken] }] });
            test.equal(parsed.sep, ':');
            test.done();
        },
        'a Token with / separator'(test) {
            const stack = new lib_1.Stack();
            const theToken = { Ref: 'SomeParameter' };
            const parsed = stack.parseArn(new intrinsic_1.Intrinsic(theToken).toString());
            test.equal(parsed.sep, '/');
            // tslint:disable-next-line:max-line-length
            test.deepEqual(stack.resolve(parsed.resource), { 'Fn::Select': [0, { 'Fn::Split': ['/', { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] }] }] });
            // tslint:disable-next-line:max-line-length
            test.deepEqual(stack.resolve(parsed.resourceName), { 'Fn::Select': [1, { 'Fn::Split': ['/', { 'Fn::Select': [5, { 'Fn::Split': [':', theToken] }] }] }] });
            test.done();
        },
        'returns empty string ARN components'(test) {
            const stack = new lib_1.Stack();
            const arn = 'arn:aws:iam::123456789012:role/abc123';
            const expected = {
                partition: 'aws',
                service: 'iam',
                region: '',
                account: '123456789012',
                resource: 'role',
                resourceName: 'abc123',
                sep: '/'
            };
            test.deepEqual(stack.parseArn(arn), expected, arn);
            test.done();
        }
    },
    'can use a fully specified ARN from a different stack without incurring an import'(test) {
        // GIVEN
        const stack1 = new lib_1.Stack(undefined, 'Stack1', { env: { account: '12345678', region: 'us-turbo-5' } });
        const stack2 = new lib_1.Stack(undefined, 'Stack2', { env: { account: '87654321', region: 'us-turbo-1' } });
        // WHEN
        const arn = stack1.formatArn({
            // No partition specified here
            service: 'bla',
            resource: 'thing',
            resourceName: 'thong',
        });
        new lib_1.CfnOutput(stack2, 'SomeValue', { value: arn });
        // THEN
        test.deepEqual(util_1.toCloudFormation(stack2), {
            Outputs: {
                SomeValue: {
                    Value: {
                        // Look ma, no Fn::ImportValue!
                        'Fn::Join': ['', ['arn:', { Ref: 'AWS::Partition' }, ':bla:us-turbo-5:12345678:thing/thong']]
                    }
                }
            }
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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