from typing import Tuple, Any, Optional, MutableMapping

from typing_extensions import Protocol
from diplomat.predictors.fpe import fpe_math
from diplomat.predictors.fpe.arr_utils import find_peaks
from diplomat.predictors.fpe.sparse_storage import SparseTrackingData, ForwardBackwardFrame, ForwardBackwardData, \
    video_to_sparse_tracking_data_point, sparse_tracking_data_to_video_point
from diplomat.wx_gui import labeler_lib

import numpy as np


class EditableFramePassEngine(Protocol):
    @property
    def changed_frames(self) -> MutableMapping[Tuple[int, int], ForwardBackwardFrame]:
        raise NotImplementedError

    @property
    def frame_data(self) -> ForwardBackwardData:
        raise NotImplementedError

    def get_maximum_with_defaults(self, frame: ForwardBackwardFrame) -> Tuple[float, float, float]:
        pass


class Point(labeler_lib.PoseLabeler):
    """
    The manual labeling mode, sets probability map to exact location of the
    user click always.
    """
    def __init__(self, frame_engine: EditableFramePassEngine):
        super().__init__()
        self._frame_engine = frame_engine

    def predict_location(
        self,
        frame_idx: int,
        bp_idx: int,
        x: float,
        y: float,
        probability: float
    ) -> Tuple[Any, Tuple[float, float, float]]:
        frame = self._frame_engine.frame_data.frames[frame_idx][bp_idx]

        if(x is None):
            #should we be returning this prob value or the probability value?
            x, y, prob = sparse_tracking_data_to_video_point(
                *self._frame_engine.get_maximum_with_defaults(frame), frame.src_data.downscaling
            )
            return ((frame_idx, bp_idx, x, y, 0), (x, y, 0))

        return ((frame_idx, bp_idx, x, y, probability), (x, y, probability))

    def pose_change(self, new_state: Any) -> Any:
        frm, bp, x, y, p = new_state
        changed_frames = self._frame_engine.changed_frames
        frames = self._frame_engine.frame_data.frames

        old_frame_data = frames[frm][bp]
        x, y, prob = video_to_sparse_tracking_data_point(x, y, p, old_frame_data.src_data.downscaling)
        is_orig = False

        idx = (frm, bp)
        if (idx not in changed_frames):
            changed_frames[idx] = old_frame_data
            is_orig = True

        new_data = SparseTrackingData(old_frame_data.src_data.downscaling)
        if (prob > 0):
            new_data.pack(*[np.array([item]) for item in [x, y, prob]])

        new_frame = ForwardBackwardFrame()
        new_frame.orig_data = new_data
        new_frame.src_data = new_data
        new_frame.disable_occluded = True
        new_frame.ignore_clustering = True

        frames[frm][bp] = new_frame

        return (frm, bp, is_orig, old_frame_data)

    def undo(self, data: Any) -> Any:
        frames = self._frame_engine.frame_data.frames
        changed_frames = self._frame_engine.changed_frames
        frm, bp, is_orig, frame_data = data

        idx = (frm, bp)
        new_is_orig = False
        new_old_frame_data = frames[frm][bp]

        if (idx not in changed_frames):
            changed_frames[idx] = new_old_frame_data
            new_is_orig = True
        elif (is_orig):
            del changed_frames[idx]

        frames[frm][bp] = frame_data

        return (frm, bp, new_is_orig, new_old_frame_data)

    def redo(self, data: Any) -> Any:
        return self.undo(data)


class Approximate(labeler_lib.PoseLabeler):
    """
    Approximate labeling mode, adds a Gaussian centered around the user
    predicted location to generate a new frame. This makes results 'snap' to
    already existing probs when the user input is close enough the predictions.
    """

    # We set a hard limit on the number of allowed probabilities generated by this method in a frame for
    # performance reasons...
    CELL_LIMIT = 75

    def __init__(self, frame_engine: EditableFramePassEngine):
        super().__init__()
        self._frame_engine = frame_engine
        self._settings = labeler_lib.SettingCollection(
            user_input_strength = labeler_lib.Slider(500, 1000, 667),
            user_input_spread = labeler_lib.FloatSpin(0.5, None, 20, 1, 4)
        )
        self._cached_gaussian_std = None
        self._cached_gaussian = None

    def _make_gaussian(self, new_std: float, down_scaling: float):
        self._cached_gaussian_std = new_std
        meta = self._frame_engine.frame_data.metadata

        std = self._cached_gaussian_std / down_scaling
        two_std = min(
            np.ceil(self._cached_gaussian_std * 2),
            int(max(meta.width / down_scaling, meta.height / down_scaling))
        )

        eval_vals = np.arange(-two_std, two_std + 1)
        x, y = np.meshgrid(eval_vals, eval_vals)
        g = fpe_math.gaussian_formula(0, x, 0, y, std, 1, 0)

        # Filter to improve memory usage, and performance....
        good_loc = g > meta.threshold
        g = g[good_loc]
        x = x[good_loc]
        y = y[good_loc]

        self._cached_gaussian = (
            g.reshape(-1),
            np.asarray([x.reshape(-1), y.reshape(-1)], dtype=int)
        )

    @staticmethod
    def _absorb_frame_data(p1, c1, off1, p2, c2, off2):
        comb_c = np.concatenate([c1.T, c2.T])
        comb_p = np.concatenate([p1, p2])
        comb_off = np.concatenate([off1.T, off2.T])
        from_dlc = np.repeat([True, False], [len(p1), len(p2)])

        sort_idx = np.lexsort([comb_c[:, 1], comb_c[:, 0]])
        comb_c = comb_c[sort_idx]
        comb_p = comb_p[sort_idx]
        comb_off = comb_off[sort_idx]
        from_dlc = from_dlc[sort_idx]

        match_idx, = np.nonzero(np.all(comb_c[1:] == comb_c[:-1], axis=1))
        match_idx_after = match_idx + 1

        comb_p[match_idx_after] = comb_p[match_idx] + comb_p[match_idx_after]
        comb_off[match_idx_after] = comb_off[match_idx]

        return (
            np.delete(comb_p, from_dlc, axis=0),
            np.delete(comb_c, from_dlc, axis=0).T,
            np.delete(comb_off, from_dlc, axis=0).T
        )

    @staticmethod
    def _filter_cell_count(
        x: np.ndarray,
        y: np.ndarray,
        probs: np.ndarray,
        max_cell_count: int
    ) -> Tuple[np.ndarray, np.ndarray, np.ndarray]:
        # Remove duplicate positions...
        ordered = np.argsort(-probs)
        __, idx = np.unique(np.stack([x, y])[:, ordered].astype(np.int64), True, axis=-1)
        x = x[ordered][idx]
        y = y[ordered][idx]
        probs = probs[ordered][idx]
        # Avoid filtering if below max count...
        if(probs.shape[0] <= max_cell_count):
            return (x, y, probs)
        # Filter to max amount...
        top_k = np.argpartition(probs, -max_cell_count)[-max_cell_count:]
        return (x[top_k], y[top_k], probs[top_k])

    def predict_location(
        self,
        frame_idx: int,
        bp_idx: int,
        x: float,
        y: float,
        probability: float,
    ) -> Tuple[Any, Tuple[float, float, float]]:
        info = self._settings.get_values()
        user_amp = info.user_input_strength / 1000

        meta = self._frame_engine.frame_data.metadata
        modified_frames = self._frame_engine.changed_frames
        if((frame_idx, bp_idx) in modified_frames):
            frame = modified_frames[(frame_idx, bp_idx)]
        else:
            frame = self._frame_engine.frame_data.frames[frame_idx][bp_idx]

        if((self._cached_gaussian is None) or (info.user_input_spread != self._cached_gaussian_std)):
            self._make_gaussian(info.user_input_spread, frame.src_data.downscaling)

        if(x is None):
            x, y, prob = sparse_tracking_data_to_video_point(
                *self._frame_engine.get_maximum_with_defaults(frame), frame.src_data.downscaling
            )
            return ((frame_idx, bp_idx, None, (x, y)), (x, y, 0))

        d_scale = frame.orig_data.downscaling

        xvid, yvid = x, y
        x, y, prob = video_to_sparse_tracking_data_point(x, y, probability, d_scale)
        gp, gc = self._cached_gaussian
        gc = gc + np.array([[x], [y]], dtype=int)

        good_locs = ((0 <= gc[0]) & (gc[0] < int(meta.width / d_scale))) & ((0 <= gc[1]) & (gc[1] < (meta.height / d_scale)))
        gc = gc[:, good_locs]
        gp = gp[good_locs]

        fx, fy, fp = [a if(a is not None) else np.array([]) for a in frame.orig_data.unpack()]
        f_coords = np.asarray([fx, fy])
        f_off = f_coords - f_coords.astype(int)

        final_p, (final_x, final_y), (fox, foy) = self._absorb_frame_data(
            fp * ((1 - user_amp) / np.max(fp)) if(len(fp) > 0) else fp,
            f_coords.astype(int),
            f_off,
            gp * user_amp,
            gc,
            np.asarray([
                (xvid / d_scale) - (gc[0]),
                (yvid / d_scale) - (gc[1])
            ])
        )

        final_x = final_x + fox
        final_y = final_y + foy

        final_x, final_y, final_p = self._filter_cell_count(
            final_x,
            final_y,
            final_p,
            self.CELL_LIMIT
        )

        final_p /= np.max(final_p)

        sp = SparseTrackingData(frame.src_data.downscaling)
        sp.pack(final_x, final_y, final_p)
        temp_f = ForwardBackwardFrame(src_data=sp, frame_probs=final_p)

        x, y, prob = sparse_tracking_data_to_video_point(
            *self._frame_engine.get_maximum_with_defaults(temp_f), temp_f.src_data.downscaling
        )

        return ((frame_idx, bp_idx, temp_f, (x, y)), (x, y, prob))

    def pose_change(self, new_state: Any) -> Any:
        """
        Handles the change in pose by updating the frame data with the new state.

        This method is responsible for updating the frame data when there is a change in the pose. It checks if the
        frame and body part index combination (frm, bp) has already been modified. If not, it marks the original frame
        data for this combination as changed. Depending on whether a suggested frame is provided or not, it either
        creates new tracking data from the suggested frame or from the provided coordinates. This new data is then
        used to update the frame data for the given frame and body part index. The method returns a tuple containing
        the frame index, body part index, a flag indicating if the original data was modified, and the old frame data.

        Parameters:
        - new_state (Any): A tuple containing the frame index, body part index, an optional suggested frame, and
          coordinates. The suggested frame is None if not provided.

        Returns:
        - Tuple[Any, Any, bool, Any]: A tuple containing the frame index, body part index, a boolean indicating if the
          original data was modified (is_orig), and the old frame data before the change.
        """
        frm, bp, suggested_frame, coord = new_state
        changed_frames = self._frame_engine.changed_frames
        frames = self._frame_engine.frame_data.frames

        old_frame_data = frames[frm][bp] 
        is_orig = False

        idx = (frm, bp)

        #which body parts are in the same group as bp? 
        num_outputs = self._frame_engine.frame_data.metadata.num_outputs #the number of individuals


        group = bp // num_outputs
        other_body_part_indices = (group * num_outputs + i for i in range(num_outputs) if i != bp % num_outputs)
        #print(f"Other body part indices: {other_body_part_indices}")

        body_part_is_orig = {bp: False for bp in other_body_part_indices}
        old_body_part_data = {} #keep track of these for the undo function 

        #changed frames is a dictionary used by the forward-backward process 
        for other_bp in other_body_part_indices:
            old_bp_frame_data = frames[frm][other_bp]
            other_bp_idx = (frm, other_bp)
            if other_bp_idx not in changed_frames:
                changed_frames[other_bp_idx] = old_bp_frame_data
                body_part_is_orig[other_bp] = True

            old_body_part_data[other_bp] = old_bp_frame_data
        
        new_body_part_data = {}

        if (idx not in changed_frames):
            changed_frames[idx] = old_frame_data
            is_orig = True

        if(suggested_frame is None):
            new_data = SparseTrackingData(old_frame_data.src_data.downscaling)
            x, y, prob = video_to_sparse_tracking_data_point(
                *coord, 0, old_frame_data.src_data.downscaling
            )
            new_data.pack(*[np.array([item]) for item in [x, y, prob]])
        else:
            x, y, prob = suggested_frame.src_data.unpack()
            max_prob_idx = np.argmax(prob)
            new_data = SparseTrackingData(old_frame_data.src_data.downscaling)
            new_data.pack(*[np.array([item]) for item in [x[max_prob_idx], y[max_prob_idx], 1]])

            #print(f"Point location: {(x[max_prob_idx], y[max_prob_idx])}")
            for other_bp in other_body_part_indices:
                bp_x, bp_y, bp_prob = frames[frm][other_bp].orig_data.unpack()

                any_matches = (
                    (x[max_prob_idx].astype(int) == bp_x.astype(int))
                    & (y[max_prob_idx].astype(int) == bp_y.astype(int))
                )

                if(np.any(any_matches)):
                    bp_prob = bp_prob.copy()
                    bp_prob[any_matches] = 0

                    new_bp_data = SparseTrackingData(old_frame_data.src_data.downscaling).pack(bp_x, bp_y, bp_prob)
                    new_bp_frame = ForwardBackwardFrame()
                    new_bp_frame.orig_data = new_bp_data
                    new_bp_frame.src_data = new_bp_data
                    new_bp_frame.disable_occluded = True
                    new_bp_frame.ignore_clustering = True
                    new_body_part_data[other_bp] = new_bp_frame

        new_frame = ForwardBackwardFrame()
        new_frame.orig_data = new_data
        new_frame.src_data = new_data
        new_frame.disable_occluded = True
        new_frame.ignore_clustering = True

        frames[frm][bp] = new_frame
        for idx, other_bp in enumerate(other_body_part_indices):
            if other_bp in new_body_part_data:
                frames[frm][other_bp] = new_body_part_data[other_bp]

        return (frm, bp, is_orig, old_frame_data, old_body_part_data, body_part_is_orig)

    def undo(self, data: Any) -> Any:
        frames = self._frame_engine.frame_data.frames
        changed_frames = self._frame_engine.changed_frames
        frm, bp, is_orig, frame_data, old_bp_data, bp_is_orig = data

        idx = (frm, bp)
        new_is_orig = False
        new_old_frame_data = frames[frm][bp]

        if (idx not in changed_frames):
            changed_frames[idx] = new_old_frame_data
            new_is_orig = True
        elif (is_orig):
            del changed_frames[idx]

        frames[frm][bp] = frame_data

        new_body_part_is_orig = {}
        new_other_old_frame_data = {}

        for other_bp, other_frame in old_bp_data.items():
            old_frame_data = frames[frm][other_bp]
            other_is_orig = bp_is_orig[other_bp]
            other_idx = (frm, other_bp)
            new_other_is_orig = False 
            if (other_idx not in changed_frames):
                changed_frames[other_idx] = old_frame_data
            elif (other_is_orig):
                del changed_frames[other_idx]
            new_other_old_frame_data[other_bp] = frames[frm][other_bp]
            
            frames[frm][other_bp] = other_frame
            
            new_body_part_is_orig[other_bp] = new_other_is_orig

        return (frm, bp, new_is_orig, new_old_frame_data, new_other_old_frame_data, new_body_part_is_orig)

    def redo(self, data: Any) -> Any:
        return self.undo(data)

    def get_settings(self) -> Optional[labeler_lib.SettingCollection]:
        return self._settings

    @classmethod
    def supports_multi_label(cls) -> bool:
        return True


class ApproximateSourceOnly(Approximate):
    @staticmethod
    def _absorb_frame_data(p1, c1, off1, p2, c2, off2):
        comb_c = np.concatenate([c1.T, c2.T])
        comb_p = np.concatenate([p1, p2])
        comb_off = np.concatenate([off1.T, off2.T])
        from_dlc = np.repeat([True, False], [len(p1), len(p2)])

        sort_idx = np.lexsort([comb_c[:, 1], comb_c[:, 0]])
        comb_c = comb_c[sort_idx]
        comb_p = comb_p[sort_idx]
        comb_off = comb_off[sort_idx]
        from_dlc = from_dlc[sort_idx]

        match_idx, = np.nonzero(np.all(comb_c[1:] == comb_c[:-1], axis=1))
        match_idx_after = match_idx + 1

        comb_p[match_idx] = comb_p[match_idx] + comb_p[match_idx_after]

        return (
            np.delete(comb_p, ~from_dlc, axis=0),
            np.delete(comb_c, ~from_dlc, axis=0).T,
            np.delete(comb_off, ~from_dlc, axis=0).T
        )

    @staticmethod
    def _filter_cell_count(
        x: np.ndarray,
        y: np.ndarray,
        probs: np.ndarray,
        max_cell_count: int
    ) -> Tuple[np.ndarray, np.ndarray, np.ndarray]:
        return (x, y, probs)


class NearestPeakInSource(labeler_lib.PoseLabeler):
    """
    Nearest labeling mode, similar to approximate but always selects the nearest peak location in the source probability
    map, assigning it a fixed value, and all the other locations a lower value...
    """

    def __init__(self, frame_engine: EditableFramePassEngine):
        super().__init__()
        self._frame_engine = frame_engine
        self._settings = labeler_lib.SettingCollection(
            minimum_peak_value=labeler_lib.FloatSpin(0, 1, 0.05, 0.001, 4),
            selected_peak_value=labeler_lib.FloatSpin(0.5, 1, 0.95, 0.001, 4),
            unselected_peak_value=labeler_lib.FloatSpin(0, 0.5, 0.05, 0.001, 4)
        )

    def predict_location(
        self,
        frame_idx: int,
        bp_idx: int,
        x: float,
        y: float,
        probability: float
    ) -> Tuple[Any, Tuple[float, float, float]]:
        meta = self._frame_engine.frame_data.metadata
        config = self._settings.get_values()

        modified_frames = self._frame_engine.changed_frames
        if((frame_idx, bp_idx) in modified_frames):
            frame = modified_frames[(frame_idx, bp_idx)]
        else:
            frame = self._frame_engine.frame_data.frames[frame_idx][bp_idx]

        if(x is None):
            x, y, prob = sparse_tracking_data_to_video_point(
                *self._frame_engine.get_maximum_with_defaults(frame), frame.src_data.downscaling
            )
            return ((frame_idx, bp_idx, None, (x, y)), (x, y, 0))

        xs, ys, probs = frame.orig_data.unpack()

        peak_locs = find_peaks(xs.astype(int), ys.astype(int), probs)
        peak_locs = peak_locs[probs[peak_locs] >= config.minimum_peak_value]
        #print(peak_locs)
        if(len(peak_locs) <= 1):
            # No peaks, or only one peak, perform basically a no-op, return prior frame state...
            x, y, prob = sparse_tracking_data_to_video_point(
                *self._frame_engine.get_maximum_with_defaults(frame), frame.src_data.downscaling
            )
            return ((frame_idx, bp_idx, frame, (x, y)), (x, y, prob))

        def to_exact(_x, _y, _x_off, _y_off):
            return _x + 0.5 + (_x_off / meta.down_scaling), _y + 0.5 + (_y_off / meta.down_scaling)

        # Compute nearest location...
        xp, yp, pp = video_to_sparse_tracking_data_point(x, y, probability, frame.src_data.downscaling)

        #xp_ex, yp_ex = to_exact(xp, yp)
        x_ex, y_ex = xs[peak_locs], ys[peak_locs]

        dists = (xp - x_ex) ** 2 + (yp - y_ex) ** 2
        nearest_idx = np.argmin(dists)

        # Compute belonging of every cell...
        owner_peak = np.argmin(
            ((np.expand_dims(xs, 1) - np.expand_dims(xs[peak_locs], 0)) ** 2)
            + ((np.expand_dims(ys, 1) - np.expand_dims(ys[peak_locs], 0)) ** 2),
            axis=-1
        )

        # Compute how much we need to scale each peak and it's neighbors by to get the configured weighting...
        multipliers = config.unselected_peak_value / probs[peak_locs]
        multipliers[nearest_idx] = config.selected_peak_value / probs[peak_locs[nearest_idx]]

        # Apply scaling to all peaks...
        probs = probs * multipliers[owner_peak]

        temp_f = ForwardBackwardFrame(
            src_data=SparseTrackingData(frame.orig_data.downscaling).pack(xs, ys, probs), frame_probs=probs
        )

        x, y, prob = sparse_tracking_data_to_video_point(
            *self._frame_engine.get_maximum_with_defaults(temp_f), frame.src_data.downscaling
        )

        return ((frame_idx, bp_idx, temp_f, (x, y)), (x, y, prob))

    def pose_change(self, new_state: Any) -> Any:
        frm, bp, suggested_frame, coord = new_state
        changed_frames = self._frame_engine.changed_frames
        frames = self._frame_engine.frame_data.frames

        old_frame_data = frames[frm][bp]
        is_orig = False

        idx = (frm, bp)
        num_outputs = self._frame_engine.frame_data.metadata.num_outputs #the number of individuals

        group = bp // num_outputs
        other_body_part_indices = (group * num_outputs + i for i in range(num_outputs) if i != bp % num_outputs)

        body_part_is_orig = {bp: False for bp in other_body_part_indices}
        old_body_part_data = {} #keep track of these for the undo function 

        #changed frames is a dictionary used by the forward-backward process 
        for other_bp in other_body_part_indices:
            old_bp_frame_data = frames[frm][other_bp]
            other_bp_idx = (frm, other_bp)
            if other_bp_idx not in changed_frames:
                changed_frames[other_bp_idx] = old_bp_frame_data
                body_part_is_orig[other_bp] = True

            old_body_part_data[other_bp] = old_bp_frame_data
        
        new_body_part_data = {}

        if(idx not in changed_frames):
            changed_frames[idx] = old_frame_data
            is_orig = True

        if(suggested_frame is None):
            new_data = SparseTrackingData(old_frame_data.src_data.downscaling)
            x, y, prob = video_to_sparse_tracking_data_point(
                *coord, 0, old_frame_data.src_data.downscaling
            )
            new_data.pack(*[np.array([item]) for item in [x, y, prob]])
        else:
            x, y, prob = suggested_frame.src_data.unpack()
            max_prob_idx = np.argmax(prob)
            new_data = SparseTrackingData(old_frame_data.src_data.downscaling)
            new_data.pack(*[np.array([item]) for item in [x[max_prob_idx], y[max_prob_idx], 1]])

            for other_bp in other_body_part_indices:
                bp_x, bp_y, bp_prob = frames[frm][other_bp].orig_data.unpack()

                any_matches = (
                    (x[max_prob_idx].astype(int) == bp_x.astype(int))
                    & (y[max_prob_idx].astype(int) == bp_y.astype(int))
                )

                if(np.any(any_matches)):
                    bp_prob = bp_prob.copy()
                    bp_prob[any_matches] = 0

                    new_bp_data = SparseTrackingData(old_frame_data.src_data.downscaling).pack(bp_x, bp_y, bp_prob)
                    new_bp_frame = ForwardBackwardFrame()
                    new_bp_frame.orig_data = new_bp_data
                    new_bp_frame.src_data = new_bp_data
                    new_bp_frame.disable_occluded = True
                    new_bp_frame.ignore_clustering = True
                    new_body_part_data[other_bp] = new_bp_frame

        new_frame = ForwardBackwardFrame()
        new_frame.orig_data = new_data
        new_frame.src_data = new_data
        new_frame.disable_occluded = True
        new_frame.ignore_clustering = True

        frames[frm][bp] = new_frame
        for idx, other_bp in enumerate(other_body_part_indices):
            if other_bp in new_body_part_data:
                frames[frm][other_bp] = new_body_part_data[other_bp]

        return (frm, bp, is_orig, old_frame_data, old_body_part_data, body_part_is_orig)

    def undo(self, data: Any) -> Any:
        frames = self._frame_engine.frame_data.frames
        changed_frames = self._frame_engine.changed_frames
        frm, bp, is_orig, frame_data, old_bp_data, bp_is_orig = data

        idx = (frm, bp)
        new_is_orig = False
        new_old_frame_data = frames[frm][bp]

        if (idx not in changed_frames):
            changed_frames[idx] = new_old_frame_data
            new_is_orig = True
        elif (is_orig):
            del changed_frames[idx]

        frames[frm][bp] = frame_data

        new_body_part_is_orig = {}
        new_other_old_frame_data = {}

        for other_bp, other_frame in old_bp_data.items():
            old_frame_data = frames[frm][other_bp]
            other_is_orig = bp_is_orig[other_bp]
            other_idx = (frm, other_bp)
            new_other_is_orig = False 
            if (other_idx not in changed_frames):
                changed_frames[other_idx] = old_frame_data
            elif (other_is_orig):
                del changed_frames[other_idx]
            new_other_old_frame_data[other_bp] = frames[frm][other_bp]
            
            frames[frm][other_bp] = other_frame
            
            new_body_part_is_orig[other_bp] = new_other_is_orig

        return (frm, bp, new_is_orig, new_old_frame_data, new_other_old_frame_data, new_body_part_is_orig)

    def redo(self, data: Any) -> Any:
        return self.undo(data)

    def get_settings(self) -> Optional[labeler_lib.SettingCollection]:
        return self._settings

    @classmethod
    def supports_multi_label(cls) -> bool:
        return True
