import os
import unittest

import httpretty

from RigorEvaluation.TFIDFEvaluation import TFIDFEvaluation


class test_tfidf_evaluation(unittest.TestCase):
    def setUp(self):
        # Mock JSON response
        # Adapted from request made to https://youtube.googleapis.com/youtube/v3/videos?part=contentDetails&id=T9xsTO6ujqM&key=[YOUR_API_KEY]
        # on 6/22/2023.
        mock_video_duration_response = """
        {
            "kind": "youtube#videoListResponse",
            "etag": "jRM3UkS0H97J0JE_RooCw1CNA0c",
            "items": [
                {
                    "kind": "youtube#video",
                    "etag": "2_4jVYm_8tYAVsj-Vg_P8HJ6qkc",
                    "id": "T9xsTO6ujqM",
                    "contentDetails": {
                        "duration": "PT24M20S",
                        "dimension": "2d",
                        "definition": "hd",
                        "caption": "false",
                        "licensedContent": true,
                        "contentRating": {},
                        "projection": "rectangular"
                    }
                }
            ],
            "pageInfo": {
                "totalResults": 1,
                "resultsPerPage": 1
            }
        }
        """
        videoID = "example_videoID"
        api_key = "example_api_key"
        os.environ["api_key"] = api_key
        expected_score = 2.9120317139990197
        httpretty.enable()
        httpretty.register_uri(
            httpretty.GET,
            f"https://youtube.googleapis.com/youtube/v3/videos?part=contentDetails&id={videoID}&key={api_key}&alt=json",
            body=mock_video_duration_response,
        )
        with open("tests/sample_data/example_youtube_video.html", "r") as f:
            mock_youtube_html_response = f.read()
        with open("tests/sample_data/example_xml_youtube_transcript.xml", "r") as f:
            mock_youtube_xml_response = f.read()
        httpretty.register_uri(
            httpretty.GET,
            f"https://www.youtube.com/watch?v={videoID}",
            body=mock_youtube_html_response,
        )
        # This is the request that is made to get the final transcript
        httpretty.register_uri(
            httpretty.GET,
            "https://www.youtube.com/api/timedtext?v=T9xsTO6ujqM&caps=asr&opi=112496729&xoaf=5&hl=en&ip=0.0.0.0&ipbits=0&expire=1687325993&sparams=ip,ipbits,expire,v,caps,opi,xoaf&signature=82EC42B531F868F24D3EF4C6765BFA55A735C121.851DCD4FDCD046D8793E98EFEDC7F451B8156CFE&key=yt8&kind=asr&lang=en",
            body=mock_youtube_xml_response,
        )
        self.evaluation = TFIDFEvaluation(videoID)

    def test_lower_case_conversion(self):
        self.evaluation.lower_case_conversion()
        expected_sentences = [
            "this is me pitching a ball that's impossible to hit and i engineered it out of necessity because i'm facing off against the world's greatest wiffle ball players now wiffle ball is the popular backyard version of baseball here in america what makes it really interesting is the balls have these holes on one side which means they can curve like crazy but how does that work and how do baseballs or really balls from any sport actually curve through the air for that matter so today we're going to test answer to that question and since i grew up playing countless hours of all myself i'm going to revive a ble against the pros and i don't want to give too much away but i might have had to rely on my engineering skills to level the playing field [music] now the whole idea for this journey of discovery started with an impromptu visit to my childhood home in brea california last time i lived here it was over two decades ago so i'm just gonna go ambush them i want a tour hey what's up i'm mark how are you good how are you good i used to live here oh word got out so after meeting the whole family they were kind enough to let me take a walk down memory lane oh my gosh if these walls could talk you know i used to keep my micro machines right in this little cabinet here up there on the roof is where me and scott glacier used to throw water balloons on my sister and her friends sunbathing down here that's an option for you kids i also took the chance to confess that all the holes in their metal shed were aaron's shots from a homemade crossbow i fashioned in eighth grade there was a target it wasn't very good at it apparently but what made me most happy to hear is how all the neighborhood kids still played out front all the time because while that may look like just a street to some for us it was a soccer hockey basketball and baseball stadium where long forgotten neighborhood legends were born where each game always felt so big and the only thing that could stop us was when the street lights came on signaling it was time to go home for dinner and to my absolute delight they let me in the game [applause] hummers that mark the long ball legend rober had returned i'm not gonna deny it it's just like old times this is great even at the mound i was still slinging lasers still got it and as invigorating as this was it did make me question if i missed out on my true calling as a professional wiffle ball player because you should know they're actually professional wiffle ball players and while we share a very similar origin story our history split paths when they kept the dream alive and turned those adolescent visions of grandeur into the world's first pro with football league in fact this is kyle as an 11 year old and now 14 years later as the league founder and commissioner he told me every year starting in spring 18s compete in a four-month season with players traveling from all over the country there was a draft playoff a world series and of course for 14 years they've tracked every possible stat that is possibly trackable as you'd expect there's heated team rivalries and league legends like jimmy norp aka the norpedo who's considered by many to be the best pitching hitting dual threat in the league and as the rain back-to-back world series champ it's easy to see why none of them get paid to play so their truest compensation is the sheer thrill of victory in this game that has united them since childhood he also told me they'd be playing in oklahoma right around the time i'd be traveling through i was finally gonna get my chance to play under the lights in front of an actual crowd which meant it was time for the long ball legend to refer to himself from the third person and make his professional debut and i was playing it cool but right out of the gate i spotted the unmistakable flowing hair who's the best back-to-back champ in the hoster here of jimmy captain clutch north [music] this was definitely the big leagues so how many per team we like to do 3v3 so we divided up into teams and it was game on all right here we go jimmy [music] how connected early to get one man on so it was over to the red baron to make it too oh yeah baron nice and that made it my turn but i wasn't here to waste time just getting on basis i'm gonna go ahead and do it calling the shot over the green monster with the street lights of my youth finally being replaced with the stadium lights i'd always dreamed of this was my uh-oh what that feels impossible to hit now just like in baseball in wiffle ball you get four balls and three strikes and it's a strike if it hits anywhere on the rectangular pipes or more embarrassingly the metal plate in the middle i i'm sorry team and with the crowd visibly unimpressed i swapped with kyle to pinch run his second base [music] everywhere i go i'm getting out so we had two outs but now that i'd cleared the cobwebs out i could just sense things were about to change ah well i think we all know i was always more of a pitcher anyways it was time to give these boys a little taste of old school suburban california streetball because jimmy norm's reign on mount olympus had come to an end [laughter] you take a base on that [music] and with that absolute dinger to center field captain clutch had struck again [music] that's witchcraft and look i'm not proud of my performance but in my defense unlike the west coast ball i grew up playing in the pros as you can see here these pitches are moving 10 feet left to right it's like the ball had propellers and it makes it nearly impossible to figure out if it's a pitch you should swing at or a pitch you should duck from or perhaps both how did he do that and the rest of the game unfortunately was just more of the same although i am proud to say in the third inning i finally got on base okay [music] and so despite all my best efforts they narrowly edged us out for the win this was not how it was supposed to happen it was going to be a long plane ride home but truthfully a plan was already forming in my head to science the credit out of this and use my engineering abilities to level the playing field and step one was to fly jimmy and kyle out to crunch labs to study exactly how they can curve a ball as much as they do and after making a bunch of observations and gathering a bunch of data myself i then marched them straight over to the home of the san francisco giants because they're regarded as the team with the most advanced pitcher development program in all of major league baseball [laughter] wow this feels pretty big and right out of the gate i was incredibly relieved to see their elite players and coaches struggling against the norpedo just as much as i did [music] but then it was into the clubhouse for the real purpose of the trip to meet with brian bannister whose deep understanding of the science behind his spinning baseball is sort of their secret sauce for developing all these great pictures and he's got the hardware to show for it when you use these concepts you win world series and so after that i was feeling pretty good about my handle on how exactly they were making the ball curve but his final confirmation on my suspicion she also met up with the stanford baseball team to run one last test with a normal baseball versus an identical one that was just missing the scene and as a further point of indication i'm happy to report their nationally ranked batters also fell victim to kyle and the norpedo and so after my extensive data collection research and interviews here's the answer as to how any sort of ball curves through the air and it starts with a riddle let's say you're an astronaut on the space station and you go out for a spacewalk to fix a solar panel with your big wrench and as you're heading out you get distracted by the view and miss grabbing the handle and now you're very slowly drifting away what do you do to save yourself and you might say well you just sort of swim your way back but in zero gravity as demonstrated here by an actual astronaut you could flail your arms all you want but your center of mass won't actually move alright so did you think of the answer the trick is you throw the wrench as fast as possible in the opposite direction and that will give you just a little push to slowly move you back to the space station and this makes intuitive sense right like if you're on a skateboard and you throw something heavy to the right you're gonna move back to the left nate you had one job well this is exactly how rockets move around in space where there's no air they create a chemical reaction from their fuel to make a lot of tiny wrenches and they just throw them out of the back really fast and the rocket naturally has to go the other way just like you as an astronaut so that's one thing you need to know and the other is the kiwanda effect which states that fluids like to curve and flow around a smooth surface you actually already know this if you've ever noticed how water will flow around a curved spoon well air is also a fluid and so it curves just like water as you could see here because the strings are following with the airflow as it turns the round corner on the frisbee okay so now let's put those two concepts together from the ball's perspective air is rushing past on the top side here and curving around really nicely because the rotation of the ball matches great with the airflow but on the bottom side they're opposite so there's sort of a head-on collision with the air rushing by creating turbulence that means more air curves around the top which means more little air wrenches are thrown diagonally down and as a result the back spun ball moves diagonally up just like the astronaut causing it to curve up the fancy term for this is the magnus effect and big light balls like beach balls are great for demonstrating this effect because they have lots of surface areas so they throw off lots of wrenches but they're also really light so the air wrenches have more of an effect and now that you know this you know exactly what causes the curve on tennis balls soccer balls ping-pong balls record-breaking basketballs golf balls and baseballs with the baseball the seams help grab even more air wrenches as the ball spins but they also provide the pitchers a better grip so they can spin the ball even faster our normal versus smooth ball tests confirmed this when the normal baseball the seams was curving so much more by the way this same principle is how frisbees seem to defy gravity because air curves over the smooth edge throwing wrenches down all around turning it into a freaking jet pack now having said that you should know that the wiffle ball is very similar yet slightly different than all of those as long as you scuff up the side with the holes which is something all serious players will do then by comparison this is now much smoother that means the air has a much easier time curving around like water on a smooth spoon throwing down the air wrenches causing the ball to lift towards the smooth side and sure enough when we check the high speed footage we see that in every case the ball is curving towards the smooth side and away from the holes so now whenever you see a ball curving just think of tiny little astronaut wrenches flying off the back of a ball as it spins through the air and now that i knew their secrets to ball curvature it was time to use that knowledge to engineer some sweet revenge with an 18 second build montage [music] [music] [music] and so it was back to oklahoma with the duffel bag full of surprises addressed specifically for jimmy noor but before we get to that you might have noticed crunch slabs has a new addition my backyard that's because to fight summer brain drain and make this your least boring summer ever we created camp crunch slabs now camp crunch slabs is a 12-week virtual summer camp featuring weekly videos with mega experiments that i do and then weekly super challenges that you do and the best weekly challenge submission of the whole summer it's a platinum ticket to come out here with me for the biggest mega experiment of them all the final video of the summer on top of that usually the crunchlab's build boxes where we build a really fun toy together where i teach you all the juicy physics of how they work comes every month but to coincide with the camp crunch lab's weekly challenges and videos we can ship them to you every week so if you want a 12-week summer camp where you can learn to think like an engineer with zero percent chance of bug bites and poison ivy head to camp crunchlabs.com to reserve one of the limited spots and i'll see you at summer camp oh look who it is number one i took the opportunity of admiring my new gift to avoid making eye contact with the norpedo and his tactics of intimidation got a bag of stuff here should be a little bit better than last time play ball the rematch was officially on all right i'm at the mound let's go edward must have spread because an even larger crowd had gathered to watch playing with the pros had been my childhood dream yet my last outing was closer to a nightmare so i decided to give them a small taste of what they could expect for round two and wound up to deliver the hot steamy appetizer dinner is served now all i had to do was retrieve and reload the brass slug because if you preload a brass cylinder against the spring in a 3d printed ball like this and then take apart a one dollar kitchen timer and harvest the geared mechanism then when you wind up the wiffle ball hemispheres you get a one and a half second delay until this screw rotates and releases the spring-loaded plug just like an astronaut wrench and that alters to trajectory in midair making it impossible to predict where to swing and as you can see unpredictability is a pitcher's best friend unfortunately after just two strikes they found my achilles heel and rendered the ball useless which is exactly why you always have a backup plan oh just missed the strikeout this this one used the same kitchen timer geared mechanism only this time the two wiffle ball hemispheres were pre-loaded against each other with the spring so after the one and a half second delay your hitting options went from one good one to two bad ones which was just enough to secure my very first strikeout unfortunately the crafting rapido deduced correctly that by simply stepping forward they could make contact before the ball had time to split which meant it was time for the backup backup plan the wiffle copter and you might think now i'm just being a bit cheeky but i did a thorough reading of the rules and it's a strike if it hits anywhere on the pipe or metal plates above the two legs but it makes no mention of front or back let's go let's go exploding loopholes unfortunately my smugness was short-lived because you don't become a professional wiffle ball player without a tremendous amount of hand-eye coordination so with two runners on it was time to level things up with this 3d printed triple 10 000 rpm fly wheeled monstrosity of a personal pitching machine aka the demogorgon now the demogorgon is calibrated to launch his cruise missiles at just under the 74 mile per hour speed limit on pitches in the league the only problem is that kind of speed requires a lot of flowing electrons and battery power soon became a concern and that's just not a position you want to find yourself in with captain clutch at the plate sadly giving them the first lead of the game and i was concerned things would start to get away from me but thankfully jimmy can't take every at-bat and after a pop-up and a routine catch by the red baron i'd survived the first inning with only one allowed run to show for it for the first time i was feeling good about our chances which meant it was time to show the world i was more than just a pitcher oh how is that possible so it was another strikeout but then baron got on base followed by kyle with an absolute bond to center field straight over the green monster which meant suddenly we were up one to two for the first time ever we were in the lead and at this rate oh yeah i struck out again but we were still winning and with the final play of the inning baron capitalized on a bad throw to second for a potential inside the park home run but there's a major league wiffle ball rule where the strike zone doubles as the catcher so if you don't beat the throw home then you're out and heading into the second inning i was fresh out of batteries so we had kyle take them out and after two strikes they got on base or at least they would have if first base didn't go rogue and now the first base had a taste for freedom going free range with a top speed of 50 miles per hour i decided to bring him back after seeing the torpedo was up next because there was plenty more where that came from but my dreams were crushed when he just northed it out of the park oh come on come on and now the game was tied and jimmy and clearly demonstrated he's a sore winner on the next play they hit one deep and were threatening to score again but i had a short circuited first base that needed some avenge [music] and with them ahead three to two headed into the bottom of the second this was the perfect time for my lucky bat and just as i'd hoped the increased surface area put me on first base and with a little teamwork i scored a run and we would have kept it going if someone wouldn't have struck out baron oh my gosh okay fine me too so we're headed into the final inning tied at three and we kept kyle on the mound even though he said his arm wasn't at full strength and of course captain clutch took full advantage on the first pitch so i volunteered to take over pitching duties the problem was i had run out of gimmick wiffle balls and gimmick personal pitching machines but thanks to the power of pneumatics i hadn't run out of gimmick strike zones which gave my floater a bit more room to work with [music] so after getting one out with the generous strike zone some good old-fashioned hustle got us the second but with jimmy at bat and knowing what was at stake one of their players disconnected the airline which honestly felt a little like cheating but i took the high road and let it slide this time i was really out of tricks it was just me versus captain clutchnorp himself mono imano and on my first pitch i got him on the delayed riser my second pitch got him with my modified slider if i was really gonna do this i would need to rely on everything i had learned on this journey of discovery [music] and that was strike three i had struck out the world's greatest wiffle ball player inerting that subtle thought of mutual respect from the torpedo sort of felt like a michael passing the torch to kobe type of moment and down by one headed into our final at-bat it lasted about as long as it took to immediately strike me out again that wasn't cool now the game wasn't over yet but instead of the two runs we needed to win we quickly got two outs but baron kept our hopes alive with the base hit and by some absolute miracle i finally got a real hit fair and square oh yeah which put me and baron on second and third foreign so it all came down to kyle i saw contact i knew it wasn't enough to clear the wall so i started my mad dash with my sight set on home plate it was a risky move but i knew it would mean victory for our team with jimmy taking the cut off at second the gauntlet was thrown his arm versus my legs no more tricks okay well maybe just one more with a blast of air that 100 psi there either triumphant returned to that most beautiful of white rubber pentagons [applause] in the game it almost felt like at any moment the street lights would come on signaling it would be time to head home for dinner and i was reminded of a truth it feels like jimmy and kyle and all of their buddies have never lost sight of finding something you're truly passionate about that brings you joy is one of life's most lovely gifts mark what'd you learn today and if you're lucky enough to have found it you should protect and cherish it like your most prized possession thank you have you ever seen a firewood shopping robot well now you have and while you can shop more than just firewood jenga skills are severely lacking welcome to your least boring summer ever aka camp crunch slabs what the heck is camp crunchlabs you ask well it's a 12-week virtual summer camp featuring weekly videos with mega experiments that i do and then weekly super challenges that you do and the best weekly challenge of the whole summer will get a platinum ticket come out here with me to film the final video where the mega experiment is so big i'm only like 20 sure that this roof is going to survive on top of that usually the crunch labs build boxes where we build a really fun toy together and where i teach you all the juicy physics of how they work comes every month but to coincide with the camp crunchlab's weekly challenges and videos we can ship them to you every week for 12 weeks straight so don't spend summer super bored for getting everything you learned in school last year instead come to camp crunchlabs where you'll have a blast growing your brain in delightful ways as you challenge yourself and learn to think like an engineer now if you're already a crunch slab subscriber don't worry you get all this for free but if you haven't somehow subscribed yet then this is the perfect chance to visit camp crunchlabs.com or use the link in the video description to reserve one of the limited spots and i'll see you at summer camp foreign"
        ]
        self.assertEqual(self.evaluation.total_sentences, expected_sentences)

    def test_evaluate_term_freq(self):
        self.evaluation.evaluate_term_freq()
        expected_tf_score = {
            "pitching": 5,
            "ball": 33,
            "impossible": 4,
            "hit": 5,
            "engineered": 1,
            "necessity": 1,
            "facing": 1,
            "world's": 3,
            "greatest": 2,
            "wiffle": 13,
            "players": 6,
            "popular": 1,
            "backyard": 2,
            "version": 1,
            "baseball": 9,
            "america": 1,
            "interesting": 1,
            "balls": 10,
            "holes": 4,
            "means": 4,
            "curve": 7,
            "crazy": 1,
            "work": 4,
            "baseballs": 2,
            "sport": 1,
            "matter": 1,
            "today": 2,
            "test": 2,
            "answer": 3,
            "grew": 2,
            "playing": 7,
            "countless": 1,
            "hours": 1,
            "revive": 1,
            "ble": 1,
            "pros": 3,
            "away": 4,
            "rely": 2,
            "engineering": 2,
            "skills": 2,
            "level": 3,
            "field": 4,
            "[music]": 14,
            "journey": 2,
            "discovery": 2,
            "started": 2,
            "impromptu": 1,
            "visit": 2,
            "childhood": 3,
            "home": 8,
            "brea": 1,
            "california": 2,
            "lived": 1,
            "decades": 1,
            "ago": 1,
            "ambush": 1,
            "tour": 1,
            "hey": 1,
            "what's": 1,
            "mark": 3,
            "good": 7,
            "live": 1,
            "oh": 10,
            "word": 1,
            "got": 12,
            "meeting": 1,
            "let": 3,
            "walk": 1,
            "lane": 1,
            "gosh": 2,
            "walls": 1,
            "know": 8,
            "micro": 1,
            "machines": 2,
            "little": 8,
            "cabinet": 1,
            "roof": 2,
            "scott": 1,
            "glacier": 1,
            "throw": 7,
            "balloons": 1,
            "sister": 1,
            "friends": 1,
            "sunbathing": 1,
            "option": 1,
            "kids": 2,
            "took": 5,
            "chance": 4,
            "confess": 1,
            "metal": 3,
            "shed": 1,
            "aaron's": 1,
            "shots": 1,
            "homemade": 1,
            "crossbow": 1,
            "fashioned": 1,
            "eighth": 1,
            "grade": 1,
            "target": 1,
            "apparently": 1,
            "happy": 2,
            "hear": 1,
            "neighborhood": 2,
            "played": 1,
            "street": 4,
            "soccer": 2,
            "hockey": 1,
            "basketball": 1,
            "stadium": 2,
            "long": 6,
            "forgotten": 1,
            "legends": 2,
            "born": 1,
            "game": 9,
            "felt": 4,
            "big": 6,
            "stop": 1,
            "lights": 5,
            "came": 3,
            "signaling": 2,
            "dinner": 3,
            "absolute": 4,
            "delight": 1,
            "[applause]": 2,
            "hummers": 1,
            "legend": 2,
            "rober": 1,
            "returned": 2,
            "deny": 1,
            "old": 3,
            "times": 1,
            "great": 4,
            "mound": 3,
            "slinging": 1,
            "lasers": 1,
            "invigorating": 1,
            "missed": 2,
            "true": 1,
            "calling": 2,
            "professional": 4,
            "player": 3,
            "they're": 4,
            "share": 1,
            "similar": 2,
            "origin": 1,
            "story": 1,
            "split": 2,
            "paths": 1,
            "kept": 4,
            "alive": 2,
            "turned": 1,
            "adolescent": 1,
            "visions": 1,
            "grandeur": 1,
            "pro": 1,
            "football": 1,
            "league": 7,
            "fact": 1,
            "kyle": 9,
            "years": 2,
            "later": 1,
            "founder": 1,
            "commissioner": 1,
            "told": 2,
            "starting": 1,
            "spring": 3,
            "18s": 1,
            "compete": 1,
            "four-month": 1,
            "season": 1,
            "traveling": 2,
            "draft": 1,
            "playoff": 1,
            "world": 4,
            "series": 3,
            "course": 2,
            "they've": 1,
            "tracked": 1,
            "possible": 3,
            "stat": 1,
            "possibly": 1,
            "trackable": 1,
            "expect": 2,
            "heated": 1,
            "team": 6,
            "rivalries": 1,
            "jimmy": 11,
            "norp": 1,
            "aka": 3,
            "norpedo": 4,
            "who's": 2,
            "considered": 1,
            "best": 6,
            "hitting": 2,
            "dual": 1,
            "threat": 1,
            "rain": 1,
            "back-to-back": 2,
            "champ": 2,
            "easy": 1,
            "paid": 1,
            "play": 5,
            "truest": 1,
            "compensation": 1,
            "sheer": 1,
            "thrill": 1,
            "victory": 2,
            "united": 1,
            "they'd": 1,
            "oklahoma": 2,
            "i'd": 5,
            "finally": 4,
            "actual": 2,
            "crowd": 3,
            "meant": 4,
            "refer": 1,
            "person": 1,
            "debut": 1,
            "cool": 2,
            "gate": 2,
            "spotted": 1,
            "unmistakable": 1,
            "flowing": 2,
            "hair": 1,
            "hoster": 1,
            "captain": 5,
            "clutch": 4,
            "north": 1,
            "definitely": 1,
            "leagues": 1,
            "3v3": 1,
            "divided": 1,
            "teams": 1,
            "connected": 1,
            "early": 1,
            "man": 1,
            "red": 2,
            "baron": 8,
            "yeah": 3,
            "nice": 1,
            "turn": 1,
            "waste": 1,
            "getting": 4,
            "basis": 1,
            "ahead": 2,
            "shot": 1,
            "green": 2,
            "monster": 2,
            "youth": 1,
            "replaced": 1,
            "dreamed": 1,
            "uh-oh": 1,
            "feels": 3,
            "strikes": 3,
            "strike": 7,
            "hits": 2,
            "rectangular": 1,
            "pipes": 1,
            "embarrassingly": 1,
            "plate": 3,
            "middle": 1,
            "sorry": 1,
            "visibly": 1,
            "unimpressed": 1,
            "swapped": 1,
            "pinch": 1,
            "run": 7,
            "second": 11,
            "base": 10,
            "outs": 2,
            "cleared": 1,
            "cobwebs": 1,
            "sense": 2,
            "things": 3,
            "change": 1,
            "ah": 1,
            "think": 6,
            "pitcher": 3,
            "anyways": 1,
            "boys": 1,
            "school": 2,
            "suburban": 1,
            "streetball": 1,
            "norm's": 1,
            "reign": 1,
            "mount": 1,
            "olympus": 1,
            "come": 7,
            "end": 1,
            "[laughter]": 2,
            "dinger": 1,
            "center": 3,
            "struck": 4,
            "witchcraft": 1,
            "proud": 2,
            "performance": 1,
            "defense": 1,
            "unlike": 1,
            "west": 1,
            "coast": 1,
            "pitches": 2,
            "moving": 1,
            "feet": 1,
            "propellers": 1,
            "nearly": 1,
            "figure": 1,
            "pitch": 5,
            "swing": 2,
            "duck": 1,
            "rest": 1,
            "unfortunately": 4,
            "inning": 5,
            "despite": 1,
            "efforts": 1,
            "narrowly": 1,
            "edged": 1,
            "win": 3,
            "supposed": 1,
            "happen": 1,
            "plane": 1,
            "ride": 1,
            "truthfully": 1,
            "plan": 3,
            "forming": 1,
            "head": 3,
            "credit": 1,
            "abilities": 1,
            "step": 1,
            "fly": 2,
            "crunch": 8,
            "labs": 2,
            "study": 1,
            "exactly": 5,
            "making": 4,
            "bunch": 2,
            "observations": 1,
            "gathering": 1,
            "data": 2,
            "marched": 1,
            "straight": 3,
            "san": 1,
            "francisco": 1,
            "giants": 1,
            "regarded": 1,
            "advanced": 1,
            "development": 1,
            "program": 1,
            "major": 2,
            "wow": 1,
            "pretty": 2,
            "incredibly": 1,
            "relieved": 1,
            "elite": 1,
            "coaches": 1,
            "struggling": 1,
            "clubhouse": 1,
            "real": 2,
            "purpose": 1,
            "trip": 1,
            "meet": 1,
            "brian": 1,
            "bannister": 1,
            "deep": 2,
            "understanding": 1,
            "spinning": 1,
            "sort": 5,
            "secret": 1,
            "sauce": 1,
            "developing": 1,
            "pictures": 1,
            "he's": 2,
            "hardware": 1,
            "concepts": 2,
            "handle": 2,
            "final": 6,
            "confirmation": 1,
            "suspicion": 1,
            "met": 1,
            "stanford": 1,
            "normal": 3,
            "versus": 4,
            "identical": 1,
            "missing": 1,
            "scene": 1,
            "point": 1,
            "indication": 1,
            "report": 1,
            "nationally": 1,
            "ranked": 1,
            "batters": 1,
            "fell": 1,
            "victim": 1,
            "extensive": 1,
            "collection": 1,
            "research": 1,
            "interviews": 1,
            "here's": 1,
            "curves": 4,
            "starts": 1,
            "riddle": 1,
            "astronaut": 6,
            "space": 3,
            "station": 2,
            "spacewalk": 1,
            "fix": 1,
            "solar": 1,
            "panel": 1,
            "wrench": 3,
            "heading": 2,
            "distracted": 1,
            "view": 1,
            "miss": 1,
            "grabbing": 1,
            "slowly": 2,
            "drifting": 1,
            "save": 1,
            "swim": 1,
            "way": 3,
            "zero": 2,
            "gravity": 2,
            "demonstrated": 2,
            "flail": 1,
            "arms": 1,
            "mass": 1,
            "alright": 1,
            "trick": 1,
            "fast": 2,
            "opposite": 2,
            "direction": 1,
            "push": 1,
            "intuitive": 1,
            "skateboard": 1,
            "heavy": 1,
            "nate": 1,
            "job": 1,
            "rockets": 1,
            "create": 1,
            "chemical": 1,
            "reaction": 1,
            "fuel": 1,
            "lot": 2,
            "tiny": 2,
            "wrenches": 8,
            "rocket": 1,
            "naturally": 1,
            "kiwanda": 1,
            "effect": 4,
            "states": 1,
            "fluids": 1,
            "flow": 2,
            "smooth": 6,
            "surface": 3,
            "noticed": 2,
            "curved": 1,
            "spoon": 2,
            "fluid": 1,
            "strings": 1,
            "following": 1,
            "airflow": 2,
            "turns": 1,
            "round": 2,
            "corner": 1,
            "frisbee": 1,
            "ball's": 1,
            "perspective": 1,
            "rushing": 2,
            "past": 1,
            "curving": 5,
            "nicely": 1,
            "rotation": 1,
            "matches": 1,
            "head-on": 1,
            "collision": 1,
            "creating": 1,
            "turbulence": 1,
            "thrown": 2,
            "diagonally": 2,
            "result": 1,
            "spun": 1,
            "moves": 1,
            "causing": 2,
            "fancy": 1,
            "term": 1,
            "magnus": 1,
            "light": 2,
            "beach": 1,
            "demonstrating": 1,
            "lots": 2,
            "areas": 1,
            "causes": 1,
            "tennis": 1,
            "ping-pong": 1,
            "record-breaking": 1,
            "basketballs": 1,
            "golf": 1,
            "seams": 2,
            "help": 1,
            "grab": 1,
            "spins": 2,
            "provide": 1,
            "pitchers": 1,
            "better": 2,
            "grip": 1,
            "spin": 1,
            "faster": 1,
            "tests": 1,
            "confirmed": 1,
            "principle": 1,
            "frisbees": 1,
            "defy": 1,
            "edge": 1,
            "throwing": 2,
            "turning": 1,
            "freaking": 1,
            "jet": 1,
            "pack": 1,
            "said": 2,
            "slightly": 1,
            "different": 1,
            "scuff": 1,
            "comparison": 1,
            "smoother": 1,
            "easier": 1,
            "lift": 1,
            "sure": 2,
            "check": 1,
            "high": 2,
            "footage": 1,
            "case": 1,
            "flying": 1,
            "knew": 3,
            "secrets": 1,
            "curvature": 1,
            "knowledge": 1,
            "sweet": 1,
            "revenge": 1,
            "build": 5,
            "montage": 1,
            "duffel": 1,
            "bag": 2,
            "surprises": 1,
            "addressed": 1,
            "specifically": 1,
            "noor": 1,
            "slabs": 4,
            "new": 2,
            "addition": 1,
            "fight": 1,
            "summer": 12,
            "brain": 2,
            "drain": 1,
            "boring": 2,
            "created": 1,
            "camp": 14,
            "12-week": 3,
            "virtual": 2,
            "featuring": 2,
            "weekly": 8,
            "videos": 4,
            "mega": 4,
            "experiments": 2,
            "super": 3,
            "challenges": 4,
            "challenge": 3,
            "submission": 1,
            "platinum": 2,
            "ticket": 2,
            "biggest": 1,
            "experiment": 2,
            "video": 3,
            "usually": 2,
            "crunchlab's": 2,
            "boxes": 2,
            "fun": 2,
            "toy": 2,
            "teach": 2,
            "juicy": 2,
            "physics": 2,
            "comes": 2,
            "coincide": 2,
            "lab's": 1,
            "ship": 2,
            "learn": 3,
            "percent": 1,
            "bug": 1,
            "bites": 1,
            "poison": 1,
            "ivy": 1,
            "crunchlabscom": 2,
            "reserve": 2,
            "limited": 2,
            "spots": 2,
            "i'll": 2,
            "number": 1,
            "opportunity": 1,
            "admiring": 1,
            "gift": 1,
            "avoid": 1,
            "eye": 1,
            "contact": 3,
            "tactics": 1,
            "intimidation": 1,
            "stuff": 1,
            "bit": 3,
            "rematch": 1,
            "officially": 1,
            "edward": 1,
            "spread": 1,
            "larger": 1,
            "gathered": 1,
            "watch": 1,
            "outing": 1,
            "closer": 1,
            "nightmare": 1,
            "decided": 2,
            "small": 1,
            "wound": 1,
            "deliver": 1,
            "hot": 1,
            "steamy": 1,
            "appetizer": 1,
            "served": 1,
            "retrieve": 1,
            "reload": 1,
            "brass": 2,
            "slug": 1,
            "preload": 1,
            "cylinder": 1,
            "3d": 2,
            "printed": 2,
            "apart": 1,
            "dollar": 1,
            "kitchen": 2,
            "timer": 2,
            "harvest": 1,
            "geared": 2,
            "mechanism": 2,
            "wind": 1,
            "hemispheres": 2,
            "half": 2,
            "delay": 2,
            "screw": 1,
            "rotates": 1,
            "releases": 1,
            "spring-loaded": 1,
            "plug": 1,
            "alters": 1,
            "trajectory": 1,
            "midair": 1,
            "predict": 1,
            "unpredictability": 1,
            "pitcher's": 1,
            "found": 2,
            "achilles": 1,
            "heel": 1,
            "rendered": 1,
            "useless": 1,
            "backup": 3,
            "strikeout": 3,
            "pre-loaded": 1,
            "options": 1,
            "went": 1,
            "bad": 2,
            "ones": 1,
            "secure": 1,
            "crafting": 1,
            "rapido": 1,
            "deduced": 1,
            "correctly": 1,
            "simply": 1,
            "stepping": 1,
            "forward": 1,
            "copter": 1,
            "cheeky": 1,
            "thorough": 1,
            "reading": 1,
            "rules": 1,
            "pipe": 1,
            "plates": 1,
            "legs": 2,
            "mention": 1,
            "exploding": 1,
            "loopholes": 1,
            "smugness": 1,
            "short-lived": 1,
            "tremendous": 1,
            "hand-eye": 1,
            "coordination": 1,
            "runners": 1,
            "triple": 1,
            "000": 1,
            "rpm": 1,
            "wheeled": 1,
            "monstrosity": 1,
            "personal": 2,
            "machine": 1,
            "demogorgon": 2,
            "calibrated": 1,
            "launch": 1,
            "cruise": 1,
            "missiles": 1,
            "mile": 1,
            "limit": 1,
            "problem": 2,
            "requires": 1,
            "electrons": 1,
            "battery": 1,
            "power": 2,
            "soon": 1,
            "concern": 1,
            "position": 1,
            "find": 1,
            "sadly": 1,
            "giving": 1,
            "lead": 2,
            "concerned": 1,
            "start": 1,
            "thankfully": 1,
            "can't": 1,
            "at-bat": 2,
            "pop-up": 1,
            "routine": 1,
            "catch": 1,
            "survived": 1,
            "allowed": 1,
            "chances": 1,
            "followed": 1,
            "bond": 1,
            "suddenly": 1,
            "rate": 1,
            "winning": 1,
            "capitalized": 1,
            "potential": 1,
            "park": 2,
            "rule": 1,
            "zone": 2,
            "doubles": 1,
            "catcher": 1,
            "beat": 1,
            "fresh": 1,
            "batteries": 1,
            "rogue": 1,
            "freedom": 1,
            "free": 2,
            "range": 1,
            "miles": 1,
            "bring": 1,
            "seeing": 1,
            "torpedo": 2,
            "plenty": 1,
            "dreams": 1,
            "crushed": 1,
            "northed": 1,
            "tied": 2,
            "clearly": 1,
            "sore": 1,
            "winner": 1,
            "threatening": 1,
            "score": 1,
            "short": 1,
            "circuited": 1,
            "needed": 2,
            "avenge": 1,
            "headed": 3,
            "perfect": 2,
            "lucky": 2,
            "bat": 2,
            "hoped": 1,
            "increased": 1,
            "area": 1,
            "teamwork": 1,
            "scored": 1,
            "fine": 1,
            "arm": 2,
            "strength": 1,
            "advantage": 1,
            "volunteered": 1,
            "duties": 1,
            "gimmick": 3,
            "thanks": 1,
            "pneumatics": 1,
            "zones": 1,
            "gave": 1,
            "floater": 1,
            "room": 1,
            "generous": 1,
            "old-fashioned": 1,
            "hustle": 1,
            "knowing": 1,
            "stake": 1,
            "disconnected": 1,
            "airline": 1,
            "honestly": 1,
            "cheating": 1,
            "road": 1,
            "slide": 1,
            "tricks": 2,
            "clutchnorp": 1,
            "mono": 1,
            "imano": 1,
            "delayed": 1,
            "riser": 1,
            "modified": 1,
            "slider": 1,
            "learned": 2,
            "inerting": 1,
            "subtle": 1,
            "mutual": 1,
            "respect": 1,
            "michael": 1,
            "passing": 1,
            "torch": 1,
            "kobe": 1,
            "type": 1,
            "lasted": 1,
            "immediately": 1,
            "instead": 2,
            "runs": 1,
            "quickly": 1,
            "hopes": 1,
            "miracle": 1,
            "fair": 1,
            "square": 1,
            "foreign": 2,
            "saw": 1,
            "clear": 1,
            "wall": 1,
            "mad": 1,
            "dash": 1,
            "sight": 2,
            "set": 1,
            "risky": 1,
            "mean": 1,
            "taking": 1,
            "cut": 1,
            "gauntlet": 1,
            "blast": 2,
            "psi": 1,
            "triumphant": 1,
            "beautiful": 1,
            "white": 1,
            "rubber": 1,
            "pentagons": 1,
            "reminded": 1,
            "truth": 1,
            "buddies": 1,
            "lost": 1,
            "finding": 1,
            "truly": 1,
            "passionate": 1,
            "brings": 1,
            "joy": 1,
            "life's": 1,
            "lovely": 1,
            "gifts": 1,
            "what'd": 1,
            "protect": 1,
            "cherish": 1,
            "prized": 1,
            "possession": 1,
            "thank": 1,
            "seen": 1,
            "firewood": 2,
            "shopping": 1,
            "robot": 1,
            "shop": 1,
            "jenga": 1,
            "severely": 1,
            "lacking": 1,
            "welcome": 1,
            "heck": 1,
            "crunchlabs": 2,
            "ask": 1,
            "film": 1,
            "survive": 1,
            "weeks": 1,
            "spend": 1,
            "bored": 1,
            "growing": 1,
            "delightful": 1,
            "ways": 1,
            "slab": 1,
            "subscriber": 1,
            "worry": 1,
            "subscribed": 1,
            "link": 1,
            "description": 1,
        }
        self.assertEqual(self.evaluation.tf_score, expected_tf_score)

    def test_evaluate_inverse_doc_freq(self):
        self.evaluation.evaluate_inverse_doc_freq()
        expected_idf_score = {
            "pitching": 1,
            "ball": 1,
            "impossible": 1,
            "hit": 1,
            "engineered": 1,
            "necessity": 1,
            "facing": 1,
            "world's": 1,
            "greatest": 1,
            "wiffle": 1,
            "players": 1,
            "popular": 1,
            "backyard": 1,
            "version": 1,
            "baseball": 1,
            "america": 1,
            "interesting": 1,
            "balls": 1,
            "holes": 1,
            "means": 1,
            "curve": 1,
            "crazy": 1,
            "work": 1,
            "baseballs": 1,
            "sport": 1,
            "matter": 1,
            "today": 1,
            "test": 1,
            "answer": 1,
            "grew": 1,
            "playing": 1,
            "countless": 1,
            "hours": 1,
            "revive": 1,
            "ble": 1,
            "pros": 1,
            "away": 1,
            "rely": 1,
            "engineering": 1,
            "skills": 1,
            "level": 1,
            "field": 1,
            "[music]": 1,
            "journey": 1,
            "discovery": 1,
            "started": 1,
            "impromptu": 1,
            "visit": 1,
            "childhood": 1,
            "home": 1,
            "brea": 1,
            "california": 1,
            "lived": 1,
            "decades": 1,
            "ago": 1,
            "ambush": 1,
            "tour": 1,
            "hey": 1,
            "what's": 1,
            "mark": 1,
            "good": 1,
            "live": 1,
            "oh": 1,
            "word": 1,
            "got": 1,
            "meeting": 1,
            "let": 1,
            "walk": 1,
            "lane": 1,
            "gosh": 1,
            "walls": 1,
            "know": 1,
            "micro": 1,
            "machines": 1,
            "little": 1,
            "cabinet": 1,
            "roof": 1,
            "scott": 1,
            "glacier": 1,
            "throw": 1,
            "balloons": 1,
            "sister": 1,
            "friends": 1,
            "sunbathing": 1,
            "option": 1,
            "kids": 1,
            "took": 1,
            "chance": 1,
            "confess": 1,
            "metal": 1,
            "shed": 1,
            "aaron's": 1,
            "shots": 1,
            "homemade": 1,
            "crossbow": 1,
            "fashioned": 1,
            "eighth": 1,
            "grade": 1,
            "target": 1,
            "apparently": 1,
            "happy": 1,
            "hear": 1,
            "neighborhood": 1,
            "played": 1,
            "street": 1,
            "soccer": 1,
            "hockey": 1,
            "basketball": 1,
            "stadium": 1,
            "long": 1,
            "forgotten": 1,
            "legends": 1,
            "born": 1,
            "game": 1,
            "felt": 1,
            "big": 1,
            "stop": 1,
            "lights": 1,
            "came": 1,
            "signaling": 1,
            "dinner": 1,
            "absolute": 1,
            "delight": 1,
            "[applause]": 1,
            "hummers": 1,
            "legend": 1,
            "rober": 1,
            "returned": 1,
            "deny": 1,
            "old": 1,
            "times": 1,
            "great": 1,
            "mound": 1,
            "slinging": 1,
            "lasers": 1,
            "invigorating": 1,
            "missed": 1,
            "true": 1,
            "calling": 1,
            "professional": 1,
            "player": 1,
            "they're": 1,
            "share": 1,
            "similar": 1,
            "origin": 1,
            "story": 1,
            "split": 1,
            "paths": 1,
            "kept": 1,
            "alive": 1,
            "turned": 1,
            "adolescent": 1,
            "visions": 1,
            "grandeur": 1,
            "pro": 1,
            "football": 1,
            "league": 1,
            "fact": 1,
            "kyle": 1,
            "years": 1,
            "later": 1,
            "founder": 1,
            "commissioner": 1,
            "told": 1,
            "starting": 1,
            "spring": 1,
            "18s": 1,
            "compete": 1,
            "four-month": 1,
            "season": 1,
            "traveling": 1,
            "draft": 1,
            "playoff": 1,
            "world": 1,
            "series": 1,
            "course": 1,
            "they've": 1,
            "tracked": 1,
            "possible": 1,
            "stat": 1,
            "possibly": 1,
            "trackable": 1,
            "expect": 1,
            "heated": 1,
            "team": 1,
            "rivalries": 1,
            "jimmy": 1,
            "norp": 1,
            "aka": 1,
            "norpedo": 1,
            "who's": 1,
            "considered": 1,
            "best": 1,
            "hitting": 1,
            "dual": 1,
            "threat": 1,
            "rain": 1,
            "back-to-back": 1,
            "champ": 1,
            "easy": 1,
            "paid": 1,
            "play": 1,
            "truest": 1,
            "compensation": 1,
            "sheer": 1,
            "thrill": 1,
            "victory": 1,
            "united": 1,
            "they'd": 1,
            "oklahoma": 1,
            "i'd": 1,
            "finally": 1,
            "actual": 1,
            "crowd": 1,
            "meant": 1,
            "refer": 1,
            "person": 1,
            "debut": 1,
            "cool": 1,
            "gate": 1,
            "spotted": 1,
            "unmistakable": 1,
            "flowing": 1,
            "hair": 1,
            "hoster": 1,
            "captain": 1,
            "clutch": 1,
            "north": 1,
            "definitely": 1,
            "leagues": 1,
            "3v3": 1,
            "divided": 1,
            "teams": 1,
            "connected": 1,
            "early": 1,
            "man": 1,
            "red": 1,
            "baron": 1,
            "yeah": 1,
            "nice": 1,
            "turn": 1,
            "waste": 1,
            "getting": 1,
            "basis": 1,
            "ahead": 1,
            "shot": 1,
            "green": 1,
            "monster": 1,
            "youth": 1,
            "replaced": 1,
            "dreamed": 1,
            "uh-oh": 1,
            "feels": 1,
            "strikes": 1,
            "strike": 1,
            "hits": 1,
            "rectangular": 1,
            "pipes": 1,
            "embarrassingly": 1,
            "plate": 1,
            "middle": 1,
            "sorry": 1,
            "visibly": 1,
            "unimpressed": 1,
            "swapped": 1,
            "pinch": 1,
            "run": 1,
            "second": 1,
            "base": 1,
            "outs": 1,
            "cleared": 1,
            "cobwebs": 1,
            "sense": 1,
            "things": 1,
            "change": 1,
            "ah": 1,
            "think": 1,
            "pitcher": 1,
            "anyways": 1,
            "boys": 1,
            "school": 1,
            "suburban": 1,
            "streetball": 1,
            "norm's": 1,
            "reign": 1,
            "mount": 1,
            "olympus": 1,
            "come": 1,
            "end": 1,
            "[laughter]": 1,
            "dinger": 1,
            "center": 1,
            "struck": 1,
            "witchcraft": 1,
            "proud": 1,
            "performance": 1,
            "defense": 1,
            "unlike": 1,
            "west": 1,
            "coast": 1,
            "pitches": 1,
            "moving": 1,
            "feet": 1,
            "propellers": 1,
            "nearly": 1,
            "figure": 1,
            "pitch": 1,
            "swing": 1,
            "duck": 1,
            "rest": 1,
            "unfortunately": 1,
            "inning": 1,
            "despite": 1,
            "efforts": 1,
            "narrowly": 1,
            "edged": 1,
            "win": 1,
            "supposed": 1,
            "happen": 1,
            "plane": 1,
            "ride": 1,
            "truthfully": 1,
            "plan": 1,
            "forming": 1,
            "head": 1,
            "credit": 1,
            "abilities": 1,
            "step": 1,
            "fly": 1,
            "crunch": 1,
            "labs": 1,
            "study": 1,
            "exactly": 1,
            "making": 1,
            "bunch": 1,
            "observations": 1,
            "gathering": 1,
            "data": 1,
            "marched": 1,
            "straight": 1,
            "san": 1,
            "francisco": 1,
            "giants": 1,
            "regarded": 1,
            "advanced": 1,
            "development": 1,
            "program": 1,
            "major": 1,
            "wow": 1,
            "pretty": 1,
            "incredibly": 1,
            "relieved": 1,
            "elite": 1,
            "coaches": 1,
            "struggling": 1,
            "clubhouse": 1,
            "real": 1,
            "purpose": 1,
            "trip": 1,
            "meet": 1,
            "brian": 1,
            "bannister": 1,
            "deep": 1,
            "understanding": 1,
            "spinning": 1,
            "sort": 1,
            "secret": 1,
            "sauce": 1,
            "developing": 1,
            "pictures": 1,
            "he's": 1,
            "hardware": 1,
            "concepts": 1,
            "handle": 1,
            "final": 1,
            "confirmation": 1,
            "suspicion": 1,
            "met": 1,
            "stanford": 1,
            "normal": 1,
            "versus": 1,
            "identical": 1,
            "missing": 1,
            "scene": 1,
            "point": 1,
            "indication": 1,
            "report": 1,
            "nationally": 1,
            "ranked": 1,
            "batters": 1,
            "fell": 1,
            "victim": 1,
            "extensive": 1,
            "collection": 1,
            "research": 1,
            "interviews": 1,
            "here's": 1,
            "curves": 1,
            "starts": 1,
            "riddle": 1,
            "astronaut": 1,
            "space": 1,
            "station": 1,
            "spacewalk": 1,
            "fix": 1,
            "solar": 1,
            "panel": 1,
            "wrench": 1,
            "heading": 1,
            "distracted": 1,
            "view": 1,
            "miss": 1,
            "grabbing": 1,
            "slowly": 1,
            "drifting": 1,
            "save": 1,
            "swim": 1,
            "way": 1,
            "zero": 1,
            "gravity": 1,
            "demonstrated": 1,
            "flail": 1,
            "arms": 1,
            "mass": 1,
            "alright": 1,
            "trick": 1,
            "fast": 1,
            "opposite": 1,
            "direction": 1,
            "push": 1,
            "intuitive": 1,
            "skateboard": 1,
            "heavy": 1,
            "nate": 1,
            "job": 1,
            "rockets": 1,
            "create": 1,
            "chemical": 1,
            "reaction": 1,
            "fuel": 1,
            "lot": 1,
            "tiny": 1,
            "wrenches": 1,
            "rocket": 1,
            "naturally": 1,
            "kiwanda": 1,
            "effect": 1,
            "states": 1,
            "fluids": 1,
            "flow": 1,
            "smooth": 1,
            "surface": 1,
            "noticed": 1,
            "curved": 1,
            "spoon": 1,
            "fluid": 1,
            "strings": 1,
            "following": 1,
            "airflow": 1,
            "turns": 1,
            "round": 1,
            "corner": 1,
            "frisbee": 1,
            "ball's": 1,
            "perspective": 1,
            "rushing": 1,
            "past": 1,
            "curving": 1,
            "nicely": 1,
            "rotation": 1,
            "matches": 1,
            "head-on": 1,
            "collision": 1,
            "creating": 1,
            "turbulence": 1,
            "thrown": 1,
            "diagonally": 1,
            "result": 1,
            "spun": 1,
            "moves": 1,
            "causing": 1,
            "fancy": 1,
            "term": 1,
            "magnus": 1,
            "light": 1,
            "beach": 1,
            "demonstrating": 1,
            "lots": 1,
            "areas": 1,
            "causes": 1,
            "tennis": 1,
            "ping-pong": 1,
            "record-breaking": 1,
            "basketballs": 1,
            "golf": 1,
            "seams": 1,
            "help": 1,
            "grab": 1,
            "spins": 1,
            "provide": 1,
            "pitchers": 1,
            "better": 1,
            "grip": 1,
            "spin": 1,
            "faster": 1,
            "tests": 1,
            "confirmed": 1,
            "principle": 1,
            "frisbees": 1,
            "defy": 1,
            "edge": 1,
            "throwing": 1,
            "turning": 1,
            "freaking": 1,
            "jet": 1,
            "pack": 1,
            "said": 1,
            "slightly": 1,
            "different": 1,
            "scuff": 1,
            "comparison": 1,
            "smoother": 1,
            "easier": 1,
            "lift": 1,
            "sure": 1,
            "check": 1,
            "high": 1,
            "footage": 1,
            "case": 1,
            "flying": 1,
            "knew": 1,
            "secrets": 1,
            "curvature": 1,
            "knowledge": 1,
            "sweet": 1,
            "revenge": 1,
            "build": 1,
            "montage": 1,
            "duffel": 1,
            "bag": 1,
            "surprises": 1,
            "addressed": 1,
            "specifically": 1,
            "noor": 1,
            "slabs": 1,
            "new": 1,
            "addition": 1,
            "fight": 1,
            "summer": 1,
            "brain": 1,
            "drain": 1,
            "boring": 1,
            "created": 1,
            "camp": 1,
            "12-week": 1,
            "virtual": 1,
            "featuring": 1,
            "weekly": 1,
            "videos": 1,
            "mega": 1,
            "experiments": 1,
            "super": 1,
            "challenges": 1,
            "challenge": 1,
            "submission": 1,
            "platinum": 1,
            "ticket": 1,
            "biggest": 1,
            "experiment": 1,
            "video": 1,
            "usually": 1,
            "crunchlab's": 1,
            "boxes": 1,
            "fun": 1,
            "toy": 1,
            "teach": 1,
            "juicy": 1,
            "physics": 1,
            "comes": 1,
            "coincide": 1,
            "lab's": 1,
            "ship": 1,
            "learn": 1,
            "percent": 1,
            "bug": 1,
            "bites": 1,
            "poison": 1,
            "ivy": 1,
            "crunchlabscom": 1,
            "reserve": 1,
            "limited": 1,
            "spots": 1,
            "i'll": 1,
            "number": 1,
            "opportunity": 1,
            "admiring": 1,
            "gift": 1,
            "avoid": 1,
            "eye": 1,
            "contact": 1,
            "tactics": 1,
            "intimidation": 1,
            "stuff": 1,
            "bit": 1,
            "rematch": 1,
            "officially": 1,
            "edward": 1,
            "spread": 1,
            "larger": 1,
            "gathered": 1,
            "watch": 1,
            "outing": 1,
            "closer": 1,
            "nightmare": 1,
            "decided": 1,
            "small": 1,
            "wound": 1,
            "deliver": 1,
            "hot": 1,
            "steamy": 1,
            "appetizer": 1,
            "served": 1,
            "retrieve": 1,
            "reload": 1,
            "brass": 1,
            "slug": 1,
            "preload": 1,
            "cylinder": 1,
            "3d": 1,
            "printed": 1,
            "apart": 1,
            "dollar": 1,
            "kitchen": 1,
            "timer": 1,
            "harvest": 1,
            "geared": 1,
            "mechanism": 1,
            "wind": 1,
            "hemispheres": 1,
            "half": 1,
            "delay": 1,
            "screw": 1,
            "rotates": 1,
            "releases": 1,
            "spring-loaded": 1,
            "plug": 1,
            "alters": 1,
            "trajectory": 1,
            "midair": 1,
            "predict": 1,
            "unpredictability": 1,
            "pitcher's": 1,
            "found": 1,
            "achilles": 1,
            "heel": 1,
            "rendered": 1,
            "useless": 1,
            "backup": 1,
            "strikeout": 1,
            "pre-loaded": 1,
            "options": 1,
            "went": 1,
            "bad": 1,
            "ones": 1,
            "secure": 1,
            "crafting": 1,
            "rapido": 1,
            "deduced": 1,
            "correctly": 1,
            "simply": 1,
            "stepping": 1,
            "forward": 1,
            "copter": 1,
            "cheeky": 1,
            "thorough": 1,
            "reading": 1,
            "rules": 1,
            "pipe": 1,
            "plates": 1,
            "legs": 1,
            "mention": 1,
            "exploding": 1,
            "loopholes": 1,
            "smugness": 1,
            "short-lived": 1,
            "tremendous": 1,
            "hand-eye": 1,
            "coordination": 1,
            "runners": 1,
            "triple": 1,
            "000": 1,
            "rpm": 1,
            "wheeled": 1,
            "monstrosity": 1,
            "personal": 1,
            "machine": 1,
            "demogorgon": 1,
            "calibrated": 1,
            "launch": 1,
            "cruise": 1,
            "missiles": 1,
            "mile": 1,
            "limit": 1,
            "problem": 1,
            "requires": 1,
            "electrons": 1,
            "battery": 1,
            "power": 1,
            "soon": 1,
            "concern": 1,
            "position": 1,
            "find": 1,
            "sadly": 1,
            "giving": 1,
            "lead": 1,
            "concerned": 1,
            "start": 1,
            "thankfully": 1,
            "can't": 1,
            "at-bat": 1,
            "pop-up": 1,
            "routine": 1,
            "catch": 1,
            "survived": 1,
            "allowed": 1,
            "chances": 1,
            "followed": 1,
            "bond": 1,
            "suddenly": 1,
            "rate": 1,
            "winning": 1,
            "capitalized": 1,
            "potential": 1,
            "park": 1,
            "rule": 1,
            "zone": 1,
            "doubles": 1,
            "catcher": 1,
            "beat": 1,
            "fresh": 1,
            "batteries": 1,
            "rogue": 1,
            "freedom": 1,
            "free": 1,
            "range": 1,
            "miles": 1,
            "bring": 1,
            "seeing": 1,
            "torpedo": 1,
            "plenty": 1,
            "dreams": 1,
            "crushed": 1,
            "northed": 1,
            "tied": 1,
            "clearly": 1,
            "sore": 1,
            "winner": 1,
            "threatening": 1,
            "score": 1,
            "short": 1,
            "circuited": 1,
            "needed": 1,
            "avenge": 1,
            "headed": 1,
            "perfect": 1,
            "lucky": 1,
            "bat": 1,
            "hoped": 1,
            "increased": 1,
            "area": 1,
            "teamwork": 1,
            "scored": 1,
            "fine": 1,
            "arm": 1,
            "strength": 1,
            "advantage": 1,
            "volunteered": 1,
            "duties": 1,
            "gimmick": 1,
            "thanks": 1,
            "pneumatics": 1,
            "zones": 1,
            "gave": 1,
            "floater": 1,
            "room": 1,
            "generous": 1,
            "old-fashioned": 1,
            "hustle": 1,
            "knowing": 1,
            "stake": 1,
            "disconnected": 1,
            "airline": 1,
            "honestly": 1,
            "cheating": 1,
            "road": 1,
            "slide": 1,
            "tricks": 1,
            "clutchnorp": 1,
            "mono": 1,
            "imano": 1,
            "delayed": 1,
            "riser": 1,
            "modified": 1,
            "slider": 1,
            "learned": 1,
            "inerting": 1,
            "subtle": 1,
            "mutual": 1,
            "respect": 1,
            "michael": 1,
            "passing": 1,
            "torch": 1,
            "kobe": 1,
            "type": 1,
            "lasted": 1,
            "immediately": 1,
            "instead": 1,
            "runs": 1,
            "quickly": 1,
            "hopes": 1,
            "miracle": 1,
            "fair": 1,
            "square": 1,
            "foreign": 1,
            "saw": 1,
            "clear": 1,
            "wall": 1,
            "mad": 1,
            "dash": 1,
            "sight": 1,
            "set": 1,
            "risky": 1,
            "mean": 1,
            "taking": 1,
            "cut": 1,
            "gauntlet": 1,
            "blast": 1,
            "psi": 1,
            "triumphant": 1,
            "beautiful": 1,
            "white": 1,
            "rubber": 1,
            "pentagons": 1,
            "reminded": 1,
            "truth": 1,
            "buddies": 1,
            "lost": 1,
            "finding": 1,
            "truly": 1,
            "passionate": 1,
            "brings": 1,
            "joy": 1,
            "life's": 1,
            "lovely": 1,
            "gifts": 1,
            "what'd": 1,
            "protect": 1,
            "cherish": 1,
            "prized": 1,
            "possession": 1,
            "thank": 1,
            "seen": 1,
            "firewood": 1,
            "shopping": 1,
            "robot": 1,
            "shop": 1,
            "jenga": 1,
            "severely": 1,
            "lacking": 1,
            "welcome": 1,
            "heck": 1,
            "crunchlabs": 1,
            "ask": 1,
            "film": 1,
            "survive": 1,
            "weeks": 1,
            "spend": 1,
            "bored": 1,
            "growing": 1,
            "delightful": 1,
            "ways": 1,
            "slab": 1,
            "subscriber": 1,
            "worry": 1,
            "subscribed": 1,
            "link": 1,
            "description": 1,
        }
        self.assertEqual(self.evaluation.idf_score, expected_idf_score)

    def test_initialize_json_scores(self):
        self.evaluation.evaluate_term_freq()
        self.evaluation.initialize_json_scores()
        expected_idf_score = {}
        print(self.evaluation.idf_score)
        self.assertEqual(self.evaluation.idf_score, expected_idf_score)

    def test_strip_non_ascii(self):
        self.evaluation.strip_non_ascii()
        expected_tf_idf_score = {}
        self.assertEqual(self.evaluation.tf_idf_score, expected_tf_idf_score)

    def test_filter_words(self):
        self.evaluation.strip_non_ascii()
        self.evaluation.filter_words()
        expected_tf_idf_score_n = {}
        self.assertEqual(self.evaluation.tf_idf_score_n, expected_tf_idf_score_n)

    def test_fallback_evaluation(self):
        self.evaluation.strip_non_ascii()
        self.evaluation.filter_words()
        tf_idf_score = self.evaluation.fallback_evaluation()
        expected_tf_idf_score = {
            "ball": 3.541858830019222,
            "just": 3.0209972373693366,
            "like": 2.5001356447194514,
            "time": 2.0834463705995425,
            "air": 1.562584777949657,
            "music": 1.45841245941968,
            "camp": 1.45841245941968,
            "wiffle": 1.3542401408897025,
            "got": 1.2500678223597257,
            "summer": 1.2500678223597257,
            "oh": 1.1458955038297485,
            "jimmy": 1.1458955038297485,
            "second": 1.1458955038297485,
            "balls": 1.0417231852997713,
            "base": 1.0417231852997713,
            "right": 0.9375508667697942,
            "really": 0.9375508667697942,
            "game": 0.9375508667697942,
            "baseball": 0.9375508667697942,
            "kyle": 0.9375508667697942,
            "crunch": 0.833378548239817,
            "baron": 0.833378548239817,
            "weekly": 0.833378548239817,
            "home": 0.833378548239817,
            "wrenches": 0.833378548239817,
            "little": 0.833378548239817,
            "know": 0.833378548239817,
            "let": 0.833378548239817,
            "strike": 0.72920622970984,
            "curve": 0.72920622970984,
            "run": 0.72920622970984,
            "playing": 0.72920622970984,
            "good": 0.72920622970984,
            "come": 0.72920622970984,
            "league": 0.72920622970984,
            "world": 0.72920622970984,
            "throw": 0.72920622970984,
            "wasn": 0.6250339111798628,
            "big": 0.6250339111798628,
            "make": 0.6250339111798628,
            "going": 0.6250339111798628,
            "players": 0.6250339111798628,
            "think": 0.6250339111798628,
            "smooth": 0.6250339111798628,
            "astronaut": 0.6250339111798628,
            "gonna": 0.6250339111798628,
            "team": 0.6250339111798628,
            "best": 0.6250339111798628,
            "final": 0.6250339111798628,
            "long": 0.6250339111798628,
            "captain": 0.5208615926498856,
            "sort": 0.5208615926498856,
            "inning": 0.5208615926498856,
            "took": 0.5208615926498856,
            "hit": 0.5208615926498856,
            "build": 0.5208615926498856,
            "curving": 0.5208615926498856,
            "did": 0.5208615926498856,
            "play": 0.5208615926498856,
            "week": 0.5208615926498856,
            "lights": 0.5208615926498856,
            "want": 0.5208615926498856,
            "exactly": 0.5208615926498856,
            "pitch": 0.5208615926498856,
            "pitching": 0.5208615926498856,
            "don": 0.5208615926498856,
            "effect": 0.4166892741199085,
            "field": 0.4166892741199085,
            "making": 0.4166892741199085,
            "professional": 0.4166892741199085,
            "impossible": 0.4166892741199085,
            "means": 0.4166892741199085,
            "head": 0.4166892741199085,
            "meant": 0.4166892741199085,
            "mega": 0.4166892741199085,
            "bat": 0.4166892741199085,
            "getting": 0.4166892741199085,
            "finally": 0.4166892741199085,
            "curves": 0.4166892741199085,
            "norpedo": 0.4166892741199085,
            "struck": 0.4166892741199085,
            "street": 0.4166892741199085,
            "great": 0.4166892741199085,
            "felt": 0.4166892741199085,
            "kept": 0.4166892741199085,
            "okay": 0.4166892741199085,
            "holes": 0.4166892741199085,
            "spring": 0.4166892741199085,
            "challenges": 0.4166892741199085,
            "chance": 0.4166892741199085,
            "speed": 0.4166892741199085,
            "crunchlabs": 0.4166892741199085,
            "slabs": 0.4166892741199085,
            "clutch": 0.4166892741199085,
            "pitcher": 0.4166892741199085,
            "old": 0.4166892741199085,
            "makes": 0.4166892741199085,
            "water": 0.4166892741199085,
            "12": 0.4166892741199085,
            "use": 0.4166892741199085,
            "absolute": 0.4166892741199085,
            "videos": 0.4166892741199085,
            "away": 0.4166892741199085,
            "actually": 0.4166892741199085,
            "used": 0.4166892741199085,
            "versus": 0.4166892741199085,
            "work": 0.4166892741199085,
            "unfortunately": 0.4166892741199085,
            "backup": 0.3125169555899314,
            "ll": 0.3125169555899314,
            "strikes": 0.3125169555899314,
            "straight": 0.3125169555899314,
            "strikeout": 0.3125169555899314,
            "win": 0.3125169555899314,
            "level": 0.3125169555899314,
            "player": 0.3125169555899314,
            "possible": 0.3125169555899314,
            "look": 0.3125169555899314,
            "gimmick": 0.3125169555899314,
            "challenge": 0.3125169555899314,
            "video": 0.3125169555899314,
            "plate": 0.3125169555899314,
            "plan": 0.3125169555899314,
            "super": 0.3125169555899314,
            "came": 0.3125169555899314,
            "mark": 0.3125169555899314,
            "engineer": 0.3125169555899314,
            "way": 0.3125169555899314,
            "metal": 0.3125169555899314,
            "center": 0.3125169555899314,
            "crowd": 0.3125169555899314,
            "month": 0.3125169555899314,
            "mound": 0.3125169555899314,
            "space": 0.3125169555899314,
            "normal": 0.3125169555899314,
            "feels": 0.3125169555899314,
            "answer": 0.3125169555899314,
            "series": 0.3125169555899314,
            "learn": 0.3125169555899314,
            "contact": 0.3125169555899314,
            "aka": 0.3125169555899314,
            "headed": 0.3125169555899314,
            "things": 0.3125169555899314,
            "year": 0.3125169555899314,
            "yeah": 0.3125169555899314,
            "wrench": 0.3125169555899314,
            "say": 0.3125169555899314,
            "taste": 0.3125169555899314,
            "knew": 0.3125169555899314,
            "dinner": 0.3125169555899314,
            "childhood": 0.3125169555899314,
            "bit": 0.3125169555899314,
            "pros": 0.3125169555899314,
            "surface": 0.3125169555899314,
            "eye": 0.20834463705995426,
            "deep": 0.20834463705995426,
            "personal": 0.20834463705995426,
            "reserve": 0.20834463705995426,
            "expect": 0.20834463705995426,
            "experiment": 0.20834463705995426,
            "perfect": 0.20834463705995426,
            "decided": 0.20834463705995426,
            "tied": 0.20834463705995426,
            "experiments": 0.20834463705995426,
            "proud": 0.20834463705995426,
            "virtual": 0.20834463705995426,
            "outs": 0.20834463705995426,
            "data": 0.20834463705995426,
            "fashioned": 0.20834463705995426,
            "fast": 0.20834463705995426,
            "featuring": 0.20834463705995426,
            "opposite": 0.20834463705995426,
            "alive": 0.20834463705995426,
            "feeling": 0.20834463705995426,
            "crunchlab": 0.20834463705995426,
            "park": 0.20834463705995426,
            "delay": 0.20834463705995426,
            "dream": 0.20834463705995426,
            "physics": 0.20834463705995426,
            "discovery": 0.20834463705995426,
            "question": 0.20834463705995426,
            "oklahoma": 0.20834463705995426,
            "problem": 0.20834463705995426,
            "airflow": 0.20834463705995426,
            "victory": 0.20834463705995426,
            "platinum": 0.20834463705995426,
            "diagonally": 0.20834463705995426,
            "ve": 0.20834463705995426,
            "real": 0.20834463705995426,
            "usually": 0.20834463705995426,
            "pitches": 0.20834463705995426,
            "red": 0.20834463705995426,
            "demonstrated": 0.20834463705995426,
            "printed": 0.20834463705995426,
            "ahead": 0.20834463705995426,
            "rely": 0.20834463705995426,
            "power": 0.20834463705995426,
            "engineering": 0.20834463705995426,
            "demogorgon": 0.20834463705995426,
            "timer": 0.20834463705995426,
            "visit": 0.20834463705995426,
            "noticed": 0.20834463705995426,
            "left": 0.20834463705995426,
            "learned": 0.20834463705995426,
            "lead": 0.20834463705995426,
            "gosh": 0.20834463705995426,
            "laughter": 0.20834463705995426,
            "gravity": 0.20834463705995426,
            "labs": 0.20834463705995426,
            "14": 0.20834463705995426,
            "greatest": 0.20834463705995426,
            "green": 0.20834463705995426,
            "grew": 0.20834463705995426,
            "kitchen": 0.20834463705995426,
            "kind": 0.20834463705995426,
            "kids": 0.20834463705995426,
            "juicy": 0.20834463705995426,
            "journey": 0.20834463705995426,
            "half": 0.20834463705995426,
            "handle": 0.20834463705995426,
            "happy": 0.20834463705995426,
            "instead": 0.20834463705995426,
            "years": 0.20834463705995426,
            "heading": 0.20834463705995426,
            "10": 0.20834463705995426,
            "hemispheres": 0.20834463705995426,
            "hour": 0.20834463705995426,
            "high": 0.20834463705995426,
            "zero": 0.20834463705995426,
            "hits": 0.20834463705995426,
            "hitting": 0.20834463705995426,
            "legend": 0.20834463705995426,
            "legends": 0.20834463705995426,
            "legs": 0.20834463705995426,
            "fun": 0.20834463705995426,
            "firewood": 0.20834463705995426,
            "new": 0.20834463705995426,
            "neighborhood": 0.20834463705995426,
            "needed": 0.20834463705995426,
            "need": 0.20834463705995426,
            "flow": 0.20834463705995426,
            "flowing": 0.20834463705995426,
            "fly": 0.20834463705995426,
            "monster": 0.20834463705995426,
            "moment": 0.20834463705995426,
            "foreign": 0.20834463705995426,
            "missed": 0.20834463705995426,
            "free": 0.20834463705995426,
            "mechanism": 0.20834463705995426,
            "3d": 0.20834463705995426,
            "actual": 0.20834463705995426,
            "gate": 0.20834463705995426,
            "geared": 0.20834463705995426,
            "zone": 0.20834463705995426,
            "major": 0.20834463705995426,
            "machines": 0.20834463705995426,
            "lucky": 0.20834463705995426,
            "lots": 0.20834463705995426,
            "lot": 0.20834463705995426,
            "loaded": 0.20834463705995426,
            "lived": 0.20834463705995426,
            "limited": 0.20834463705995426,
            "light": 0.20834463705995426,
            "roof": 0.20834463705995426,
            "pretty": 0.20834463705995426,
            "applause": 0.20834463705995426,
            "bunch": 0.20834463705995426,
            "station": 0.20834463705995426,
            "calling": 0.20834463705995426,
            "started": 0.20834463705995426,
            "stadium": 0.20834463705995426,
            "causing": 0.20834463705995426,
            "spots": 0.20834463705995426,
            "spoon": 0.20834463705995426,
            "split": 0.20834463705995426,
            "spins": 0.20834463705995426,
            "arm": 0.20834463705995426,
            "champ": 0.20834463705995426,
            "soccer": 0.20834463705995426,
            "slowly": 0.20834463705995426,
            "skills": 0.20834463705995426,
            "similar": 0.20834463705995426,
            "signaling": 0.20834463705995426,
            "sight": 0.20834463705995426,
            "coincide": 0.20834463705995426,
            "short": 0.20834463705995426,
            "ship": 0.20834463705995426,
            "com": 0.20834463705995426,
            "comes": 0.20834463705995426,
            "sense": 0.20834463705995426,
            "california": 0.20834463705995426,
            "tricks": 0.20834463705995426,
            "seams": 0.20834463705995426,
            "brass": 0.20834463705995426,
            "ticket": 0.20834463705995426,
            "thrown": 0.20834463705995426,
            "throwing": 0.20834463705995426,
            "tiny": 0.20834463705995426,
            "today": 0.20834463705995426,
            "told": 0.20834463705995426,
            "baseballs": 0.20834463705995426,
            "torpedo": 0.20834463705995426,
            "thing": 0.20834463705995426,
            "toy": 0.20834463705995426,
            "test": 0.20834463705995426,
            "bag": 0.20834463705995426,
            "teach": 0.20834463705995426,
            "bad": 0.20834463705995426,
            "better": 0.20834463705995426,
            "swing": 0.20834463705995426,
            "backyard": 0.20834463705995426,
            "blast": 0.20834463705995426,
            "sure": 0.20834463705995426,
            "boring": 0.20834463705995426,
            "traveling": 0.20834463705995426,
            "boxes": 0.20834463705995426,
            "brain": 0.20834463705995426,
            "concepts": 0.20834463705995426,
            "returned": 0.20834463705995426,
            "said": 0.20834463705995426,
            "round": 0.20834463705995426,
            "school": 0.20834463705995426,
            "science": 0.20834463705995426,
            "rushing": 0.20834463705995426,
            "course": 0.20834463705995426,
            "cool": 0.20834463705995426,
            "bug": 0.10417231852997713,
            "gifts": 0.10417231852997713,
            "brian": 0.10417231852997713,
            "aaron": 0.10417231852997713,
            "buddies": 0.10417231852997713,
            "brings": 0.10417231852997713,
            "bring": 0.10417231852997713,
            "giving": 0.10417231852997713,
            "74": 0.10417231852997713,
            "glacier": 0.10417231852997713,
            "breaking": 0.10417231852997713,
            "delayed": 0.10417231852997713,
            "50": 0.10417231852997713,
            "3v3": 0.10417231852997713,
            "brea": 0.10417231852997713,
            "golf": 0.10417231852997713,
            "defy": 0.10417231852997713,
            "definitely": 0.10417231852997713,
            "20": 0.10417231852997713,
            "boys": 0.10417231852997713,
            "defense": 0.10417231852997713,
            "born": 0.10417231852997713,
            "crafting": 0.10417231852997713,
            "america": 0.10417231852997713,
            "bored": 0.10417231852997713,
            "avoid": 0.10417231852997713,
            "cabinet": 0.10417231852997713,
            "gift": 0.10417231852997713,
            "18": 0.10417231852997713,
            "fresh": 0.10417231852997713,
            "friend": 0.10417231852997713,
            "arms": 0.10417231852997713,
            "friends": 0.10417231852997713,
            "frisbee": 0.10417231852997713,
            "frisbees": 0.10417231852997713,
            "causes": 0.10417231852997713,
            "fuel": 0.10417231852997713,
            "catcher": 0.10417231852997713,
            "country": 0.10417231852997713,
            "catch": 0.10417231852997713,
            "case": 0.10417231852997713,
            "ask": 0.10417231852997713,
            "deliver": 0.10417231852997713,
            "gathered": 0.10417231852997713,
            "gathering": 0.10417231852997713,
            "gauntlet": 0.10417231852997713,
            "capitalized": 0.10417231852997713,
            "gave": 0.10417231852997713,
            "avenge": 0.10417231852997713,
            "achilles": 0.10417231852997713,
            "delightful": 0.10417231852997713,
            "delight": 0.10417231852997713,
            "generous": 0.10417231852997713,
            "abilities": 0.10417231852997713,
            "calibrated": 0.10417231852997713,
            "giants": 0.10417231852997713,
            "18s": 0.10417231852997713,
            "considered": 0.10417231852997713,
            "deduced": 0.10417231852997713,
            "alters": 0.10417231852997713,
            "happen": 0.10417231852997713,
            "basketball": 0.10417231852997713,
            "hardware": 0.10417231852997713,
            "harvest": 0.10417231852997713,
            "basis": 0.10417231852997713,
            "haven": 0.10417231852997713,
            "allowed": 0.10417231852997713,
            "having": 0.10417231852997713,
            "alright": 0.10417231852997713,
            "curved": 0.10417231852997713,
            "balloons": 0.10417231852997713,
            "hear": 0.10417231852997713,
            "heated": 0.10417231852997713,
            "hand": 0.10417231852997713,
            "heavy": 0.10417231852997713,
            "heck": 0.10417231852997713,
            "heel": 0.10417231852997713,
            "help": 0.10417231852997713,
            "curvature": 0.10417231852997713,
            "hey": 0.10417231852997713,
            "bannister": 0.10417231852997713,
            "crushed": 0.10417231852997713,
            "history": 0.10417231852997713,
            "credit": 0.10417231852997713,
            "ambush": 0.10417231852997713,
            "hockey": 0.10417231852997713,
            "basketballs": 0.10417231852997713,
            "creating": 0.10417231852997713,
            "freedom": 0.10417231852997713,
            "beautiful": 0.10417231852997713,
            "ble": 0.10417231852997713,
            "crazy": 0.10417231852997713,
            "grab": 0.10417231852997713,
            "grabbing": 0.10417231852997713,
            "grade": 0.10417231852997713,
            "grandeur": 0.10417231852997713,
            "bites": 0.10417231852997713,
            "biggest": 0.10417231852997713,
            "decades": 0.10417231852997713,
            "debut": 0.10417231852997713,
            "create": 0.10417231852997713,
            "created": 0.10417231852997713,
            "11": 0.10417231852997713,
            "hair": 0.10417231852997713,
            "beat": 0.10417231852997713,
            "grip": 0.10417231852997713,
            "dash": 0.10417231852997713,
            "cylinder": 0.10417231852997713,
            "beach": 0.10417231852997713,
            "100": 0.10417231852997713,
            "battery": 0.10417231852997713,
            "batters": 0.10417231852997713,
            "growing": 0.10417231852997713,
            "batteries": 0.10417231852997713,
            "hadn": 0.10417231852997713,
            "cut": 0.10417231852997713,
            "bond": 0.10417231852997713,
            "freaking": 0.10417231852997713,
            "divided": 0.10417231852997713,
            "different": 0.10417231852997713,
            "efforts": 0.10417231852997713,
            "commissioner": 0.10417231852997713,
            "direction": 0.10417231852997713,
            "eighth": 0.10417231852997713,
            "anyways": 0.10417231852997713,
            "electrons": 0.10417231852997713,
            "collision": 0.10417231852997713,
            "elite": 0.10417231852997713,
            "collection": 0.10417231852997713,
            "dinger": 0.10417231852997713,
            "embarrassingly": 0.10417231852997713,
            "end": 0.10417231852997713,
            "engineered": 0.10417231852997713,
            "compensation": 0.10417231852997713,
            "cobwebs": 0.10417231852997713,
            "didn": 0.10417231852997713,
            "coordination": 0.10417231852997713,
            "copter": 0.10417231852997713,
            "coast": 0.10417231852997713,
            "coaches": 0.10417231852997713,
            "clutchnorp": 0.10417231852997713,
            "development": 0.10417231852997713,
            "exploding": 0.10417231852997713,
            "extensive": 0.10417231852997713,
            "apart": 0.10417231852997713,
            "facing": 0.10417231852997713,
            "comparison": 0.10417231852997713,
            "disconnected": 0.10417231852997713,
            "apparently": 0.10417231852997713,
            "dreamed": 0.10417231852997713,
            "connected": 0.10417231852997713,
            "does": 0.10417231852997713,
            "dollar": 0.10417231852997713,
            "airline": 0.10417231852997713,
            "confirmed": 0.10417231852997713,
            "distracted": 0.10417231852997713,
            "doubles": 0.10417231852997713,
            "confirmation": 0.10417231852997713,
            "draft": 0.10417231852997713,
            "confess": 0.10417231852997713,
            "drain": 0.10417231852997713,
            "concerned": 0.10417231852997713,
            "concern": 0.10417231852997713,
            "compete": 0.10417231852997713,
            "dreams": 0.10417231852997713,
            "drifting": 0.10417231852997713,
            "dual": 0.10417231852997713,
            "duck": 0.10417231852997713,
            "duffel": 0.10417231852997713,
            "duties": 0.10417231852997713,
            "early": 0.10417231852997713,
            "easier": 0.10417231852997713,
            "easy": 0.10417231852997713,
            "edge": 0.10417231852997713,
            "edged": 0.10417231852997713,
            "edward": 0.10417231852997713,
            "fact": 0.10417231852997713,
            "fair": 0.10417231852997713,
            "francisco": 0.10417231852997713,
            "followed": 0.10417231852997713,
            "crossbow": 0.10417231852997713,
            "change": 0.10417231852997713,
            "fix": 0.10417231852997713,
            "flail": 0.10417231852997713,
            "floater": 0.10417231852997713,
            "chances": 0.10417231852997713,
            "area": 0.10417231852997713,
            "fluid": 0.10417231852997713,
            "fluids": 0.10417231852997713,
            "addition": 0.10417231852997713,
            "areas": 0.10417231852997713,
            "flying": 0.10417231852997713,
            "correctly": 0.10417231852997713,
            "check": 0.10417231852997713,
            "despite": 0.10417231852997713,
            "following": 0.10417231852997713,
            "footage": 0.10417231852997713,
            "description": 0.10417231852997713,
            "football": 0.10417231852997713,
            "countless": 0.10417231852997713,
            "deny": 0.10417231852997713,
            "forgotten": 0.10417231852997713,
            "forming": 0.10417231852997713,
            "demonstrating": 0.10417231852997713,
            "forward": 0.10417231852997713,
            "founder": 0.10417231852997713,
            "cheating": 0.10417231852997713,
            "cheeky": 0.10417231852997713,
            "family": 0.10417231852997713,
            "ago": 0.10417231852997713,
            "fancy": 0.10417231852997713,
            "clubhouse": 0.10417231852997713,
            "closer": 0.10417231852997713,
            "clearly": 0.10417231852997713,
            "faster": 0.10417231852997713,
            "cleared": 0.10417231852997713,
            "developing": 0.10417231852997713,
            "clear": 0.10417231852997713,
            "circuited": 0.10417231852997713,
            "feet": 0.10417231852997713,
            "ah": 0.10417231852997713,
            "appetizer": 0.10417231852997713,
            "cherish": 0.10417231852997713,
            "fine": 0.10417231852997713,
            "fell": 0.10417231852997713,
            "advantage": 0.10417231852997713,
            "advanced": 0.10417231852997713,
            "corner": 0.10417231852997713,
            "fight": 0.10417231852997713,
            "figure": 0.10417231852997713,
            "adolescent": 0.10417231852997713,
            "film": 0.10417231852997713,
            "chemical": 0.10417231852997713,
            "admiring": 0.10417231852997713,
            "addressed": 0.10417231852997713,
            "finding": 0.10417231852997713,
            "cruise": 0.10417231852997713,
            "zones": 0.10417231852997713,
            "homemade": 0.10417231852997713,
            "sport": 0.10417231852997713,
            "spin": 0.10417231852997713,
            "spend": 0.10417231852997713,
            "specifically": 0.10417231852997713,
            "spacewalk": 0.10417231852997713,
            "sorry": 0.10417231852997713,
            "sore": 0.10417231852997713,
            "soon": 0.10417231852997713,
            "solar": 0.10417231852997713,
            "smugness": 0.10417231852997713,
            "smoother": 0.10417231852997713,
            "small": 0.10417231852997713,
            "slug": 0.10417231852997713,
            "slinging": 0.10417231852997713,
            "slightly": 0.10417231852997713,
            "slider": 0.10417231852997713,
            "slide": 0.10417231852997713,
            "slab": 0.10417231852997713,
            "skateboard": 0.10417231852997713,
            "sister": 0.10417231852997713,
            "spinning": 0.10417231852997713,
            "spotted": 0.10417231852997713,
            "shots": 0.10417231852997713,
            "spread": 0.10417231852997713,
            "study": 0.10417231852997713,
            "struggling": 0.10417231852997713,
            "strings": 0.10417231852997713,
            "strength": 0.10417231852997713,
            "streetball": 0.10417231852997713,
            "story": 0.10417231852997713,
            "stop": 0.10417231852997713,
            "stepping": 0.10417231852997713,
            "step": 0.10417231852997713,
            "steamy": 0.10417231852997713,
            "states": 0.10417231852997713,
            "stat": 0.10417231852997713,
            "starts": 0.10417231852997713,
            "starting": 0.10417231852997713,
            "start": 0.10417231852997713,
            "stanford": 0.10417231852997713,
            "stake": 0.10417231852997713,
            "square": 0.10417231852997713,
            "spun": 0.10417231852997713,
            "simply": 0.10417231852997713,
            "shot": 0.10417231852997713,
            "submission": 0.10417231852997713,
            "sauce": 0.10417231852997713,
            "sadly": 0.10417231852997713,
            "runs": 0.10417231852997713,
            "runners": 0.10417231852997713,
            "rules": 0.10417231852997713,
            "rule": 0.10417231852997713,
            "rubber": 0.10417231852997713,
            "rpm": 0.10417231852997713,
            "routine": 0.10417231852997713,
            "rotation": 0.10417231852997713,
            "rotates": 0.10417231852997713,
            "room": 0.10417231852997713,
            "rogue": 0.10417231852997713,
            "rockets": 0.10417231852997713,
            "rocket": 0.10417231852997713,
            "robot": 0.10417231852997713,
            "rober": 0.10417231852997713,
            "road": 0.10417231852997713,
            "rivalries": 0.10417231852997713,
            "risky": 0.10417231852997713,
            "san": 0.10417231852997713,
            "save": 0.10417231852997713,
            "shopping": 0.10417231852997713,
            "saw": 0.10417231852997713,
            "shop": 0.10417231852997713,
            "sheer": 0.10417231852997713,
            "shed": 0.10417231852997713,
            "share": 0.10417231852997713,
            "severely": 0.10417231852997713,
            "set": 0.10417231852997713,
            "served": 0.10417231852997713,
            "seen": 0.10417231852997713,
            "seeing": 0.10417231852997713,
            "secure": 0.10417231852997713,
            "secrets": 0.10417231852997713,
            "secret": 0.10417231852997713,
            "season": 0.10417231852997713,
            "scuff": 0.10417231852997713,
            "screw": 0.10417231852997713,
            "scott": 0.10417231852997713,
            "scored": 0.10417231852997713,
            "score": 0.10417231852997713,
            "scene": 0.10417231852997713,
            "stuff": 0.10417231852997713,
            "subscribed": 0.10417231852997713,
            "ride": 0.10417231852997713,
            "wall": 0.10417231852997713,
            "volunteered": 0.10417231852997713,
            "visions": 0.10417231852997713,
            "visibly": 0.10417231852997713,
            "view": 0.10417231852997713,
            "victim": 0.10417231852997713,
            "version": 0.10417231852997713,
            "useless": 0.10417231852997713,
            "unpredictability": 0.10417231852997713,
            "unmistakable": 0.10417231852997713,
            "unlike": 0.10417231852997713,
            "united": 0.10417231852997713,
            "unimpressed": 0.10417231852997713,
            "understanding": 0.10417231852997713,
            "uh": 0.10417231852997713,
            "type": 0.10417231852997713,
            "turns": 0.10417231852997713,
            "turning": 0.10417231852997713,
            "turned": 0.10417231852997713,
            "turn": 0.10417231852997713,
            "walk": 0.10417231852997713,
            "walls": 0.10417231852997713,
            "truthfully": 0.10417231852997713,
            "waste": 0.10417231852997713,
            "youth": 0.10417231852997713,
            "wow": 0.10417231852997713,
            "wound": 0.10417231852997713,
            "wouldn": 0.10417231852997713,
            "worry": 0.10417231852997713,
            "word": 0.10417231852997713,
            "won": 0.10417231852997713,
            "witchcraft": 0.10417231852997713,
            "winning": 0.10417231852997713,
            "winner": 0.10417231852997713,
            "wind": 0.10417231852997713,
            "white": 0.10417231852997713,
            "wheeled": 0.10417231852997713,
            "west": 0.10417231852997713,
            "went": 0.10417231852997713,
            "welcome": 0.10417231852997713,
            "weeks": 0.10417231852997713,
            "ways": 0.10417231852997713,
            "watch": 0.10417231852997713,
            "turbulence": 0.10417231852997713,
            "truth": 0.10417231852997713,
            "subscriber": 0.10417231852997713,
            "tests": 0.10417231852997713,
            "tennis": 0.10417231852997713,
            "teamwork": 0.10417231852997713,
            "teams": 0.10417231852997713,
            "target": 0.10417231852997713,
            "talk": 0.10417231852997713,
            "taking": 0.10417231852997713,
            "tactics": 0.10417231852997713,
            "swim": 0.10417231852997713,
            "sweet": 0.10417231852997713,
            "swapped": 0.10417231852997713,
            "suspicion": 0.10417231852997713,
            "survived": 0.10417231852997713,
            "survive": 0.10417231852997713,
            "surprises": 0.10417231852997713,
            "supposed": 0.10417231852997713,
            "sunbathing": 0.10417231852997713,
            "suddenly": 0.10417231852997713,
            "suburban": 0.10417231852997713,
            "subtle": 0.10417231852997713,
            "term": 0.10417231852997713,
            "thank": 0.10417231852997713,
            "truly": 0.10417231852997713,
            "thankfully": 0.10417231852997713,
            "truest": 0.10417231852997713,
            "true": 0.10417231852997713,
            "triumphant": 0.10417231852997713,
            "triple": 0.10417231852997713,
            "trip": 0.10417231852997713,
            "trick": 0.10417231852997713,
            "tremendous": 0.10417231852997713,
            "trajectory": 0.10417231852997713,
            "tracked": 0.10417231852997713,
            "trackable": 0.10417231852997713,
            "tour": 0.10417231852997713,
            "torch": 0.10417231852997713,
            "times": 0.10417231852997713,
            "thrill": 0.10417231852997713,
            "threatening": 0.10417231852997713,
            "threat": 0.10417231852997713,
            "thought": 0.10417231852997713,
            "thorough": 0.10417231852997713,
            "thanks": 0.10417231852997713,
            "riser": 0.10417231852997713,
            "riddle": 0.10417231852997713,
            "honestly": 0.10417231852997713,
            "middle": 0.10417231852997713,
            "micro": 0.10417231852997713,
            "michael": 0.10417231852997713,
            "met": 0.10417231852997713,
            "mention": 0.10417231852997713,
            "memory": 0.10417231852997713,
            "meeting": 0.10417231852997713,
            "meet": 0.10417231852997713,
            "mean": 0.10417231852997713,
            "maybe": 0.10417231852997713,
            "matter": 0.10417231852997713,
            "matches": 0.10417231852997713,
            "mass": 0.10417231852997713,
            "marched": 0.10417231852997713,
            "man": 0.10417231852997713,
            "magnus": 0.10417231852997713,
            "mad": 0.10417231852997713,
            "machine": 0.10417231852997713,
            "lovely": 0.10417231852997713,
            "lost": 0.10417231852997713,
            "midair": 0.10417231852997713,
            "mile": 0.10417231852997713,
            "live": 0.10417231852997713,
            "miles": 0.10417231852997713,
            "nice": 0.10417231852997713,
            "necessity": 0.10417231852997713,
            "nearly": 0.10417231852997713,
            "naturally": 0.10417231852997713,
            "nationally": 0.10417231852997713,
            "nate": 0.10417231852997713,
            "narrowly": 0.10417231852997713,
            "mutual": 0.10417231852997713,
            "moving": 0.10417231852997713,
            "moves": 0.10417231852997713,
            "mount": 0.10417231852997713,
            "montage": 0.10417231852997713,
            "monstrosity": 0.10417231852997713,
            "mono": 0.10417231852997713,
            "modified": 0.10417231852997713,
            "missing": 0.10417231852997713,
            "missiles": 0.10417231852997713,
            "miss": 0.10417231852997713,
            "miracle": 0.10417231852997713,
            "loopholes": 0.10417231852997713,
            "link": 0.10417231852997713,
            "nightmare": 0.10417231852997713,
            "intuitive": 0.10417231852997713,
            "interviews": 0.10417231852997713,
            "interesting": 0.10417231852997713,
            "inside": 0.10417231852997713,
            "inerting": 0.10417231852997713,
            "indication": 0.10417231852997713,
            "incredibly": 0.10417231852997713,
            "increased": 0.10417231852997713,
            "impromptu": 0.10417231852997713,
            "immediately": 0.10417231852997713,
            "imano": 0.10417231852997713,
            "identical": 0.10417231852997713,
            "idea": 0.10417231852997713,
            "hustle": 0.10417231852997713,
            "hummers": 0.10417231852997713,
            "hours": 0.10417231852997713,
            "hot": 0.10417231852997713,
            "hoster": 0.10417231852997713,
            "hopes": 0.10417231852997713,
            "hoped": 0.10417231852997713,
            "intimidation": 0.10417231852997713,
            "invigorating": 0.10417231852997713,
            "limit": 0.10417231852997713,
            "ivy": 0.10417231852997713,
            "lift": 0.10417231852997713,
            "life": 0.10417231852997713,
            "leagues": 0.10417231852997713,
            "launch": 0.10417231852997713,
            "later": 0.10417231852997713,
            "lasted": 0.10417231852997713,
            "lasers": 0.10417231852997713,
            "larger": 0.10417231852997713,
            "lane": 0.10417231852997713,
            "lacking": 0.10417231852997713,
            "lab": 0.10417231852997713,
            "kobe": 0.10417231852997713,
            "knowledge": 0.10417231852997713,
            "knowing": 0.10417231852997713,
            "kiwanda": 0.10417231852997713,
            "joy": 0.10417231852997713,
            "job": 0.10417231852997713,
            "jet": 0.10417231852997713,
            "jenga": 0.10417231852997713,
            "nicely": 0.10417231852997713,
            "noor": 0.10417231852997713,
            "revive": 0.10417231852997713,
            "reaction": 0.10417231852997713,
            "rapido": 0.10417231852997713,
            "ranked": 0.10417231852997713,
            "range": 0.10417231852997713,
            "rain": 0.10417231852997713,
            "quickly": 0.10417231852997713,
            "push": 0.10417231852997713,
            "purpose": 0.10417231852997713,
            "psi": 0.10417231852997713,
            "provide": 0.10417231852997713,
            "protect": 0.10417231852997713,
            "propellers": 0.10417231852997713,
            "program": 0.10417231852997713,
            "pro": 0.10417231852997713,
            "prized": 0.10417231852997713,
            "principle": 0.10417231852997713,
            "preload": 0.10417231852997713,
            "predict": 0.10417231852997713,
            "pre": 0.10417231852997713,
            "potential": 0.10417231852997713,
            "rate": 0.10417231852997713,
            "reading": 0.10417231852997713,
            "possession": 0.10417231852997713,
            "record": 0.10417231852997713,
            "revenge": 0.10417231852997713,
            "retrieve": 0.10417231852997713,
            "result": 0.10417231852997713,
            "rest": 0.10417231852997713,
            "respect": 0.10417231852997713,
            "research": 0.10417231852997713,
            "requires": 0.10417231852997713,
            "report": 0.10417231852997713,
            "replaced": 0.10417231852997713,
            "rendered": 0.10417231852997713,
            "reminded": 0.10417231852997713,
            "rematch": 0.10417231852997713,
            "reload": 0.10417231852997713,
            "relieved": 0.10417231852997713,
            "releases": 0.10417231852997713,
            "reign": 0.10417231852997713,
            "regarded": 0.10417231852997713,
            "refer": 0.10417231852997713,
            "rectangular": 0.10417231852997713,
            "possibly": 0.10417231852997713,
            "position": 0.10417231852997713,
            "norm": 0.10417231852997713,
            "pentagons": 0.10417231852997713,
            "past": 0.10417231852997713,
            "passionate": 0.10417231852997713,
            "passing": 0.10417231852997713,
            "panel": 0.10417231852997713,
            "paid": 0.10417231852997713,
            "pack": 0.10417231852997713,
            "outing": 0.10417231852997713,
            "origin": 0.10417231852997713,
            "options": 0.10417231852997713,
            "option": 0.10417231852997713,
            "opportunity": 0.10417231852997713,
            "ones": 0.10417231852997713,
            "olympus": 0.10417231852997713,
            "officially": 0.10417231852997713,
            "observations": 0.10417231852997713,
            "number": 0.10417231852997713,
            "northed": 0.10417231852997713,
            "north": 0.10417231852997713,
            "norp": 0.10417231852997713,
            "paths": 0.10417231852997713,
            "percent": 0.10417231852997713,
            "popular": 0.10417231852997713,
            "performance": 0.10417231852997713,
            "pop": 0.10417231852997713,
            "pong": 0.10417231852997713,
            "poison": 0.10417231852997713,
            "point": 0.10417231852997713,
            "pneumatics": 0.10417231852997713,
            "plug": 0.10417231852997713,
            "plenty": 0.10417231852997713,
            "playoff": 0.10417231852997713,
            "played": 0.10417231852997713,
            "plates": 0.10417231852997713,
            "plane": 0.10417231852997713,
            "pitchers": 0.10417231852997713,
            "pipes": 0.10417231852997713,
            "pipe": 0.10417231852997713,
            "ping": 0.10417231852997713,
            "pinch": 0.10417231852997713,
            "pictures": 0.10417231852997713,
            "perspective": 0.10417231852997713,
            "person": 0.10417231852997713,
            "000": 0.10417231852997713,
        }
        self.assertEqual(tf_idf_score, expected_tf_idf_score)


if __name__ == "__main__":
    unittest.main()
