import { ArnComponents } from './arn';
import { CfnResource } from './cfn-resource';
import { IConstruct, Construct as CoreConstruct } from './construct-compat';
import { RemovalPolicy } from './removal-policy';
import { Stack } from './stack';
import { Construct } from 'constructs';
/**
 * Represents the environment a given resource lives in.
 *
 * Used as the return value for the {@link IResource.env} property.
 *
 * @stability stable
 */
export interface ResourceEnvironment {
    /**
     * The AWS account ID that this resource belongs to.
     *
     * Since this can be a Token
     * (for example, when the account is CloudFormation's AWS::AccountId intrinsic),
     * make sure to use Token.compareStrings()
     * instead of just comparing the values for equality.
     *
     * @stability stable
     */
    readonly account: string;
    /**
     * The AWS region that this resource belongs to.
     *
     * Since this can be a Token
     * (for example, when the region is CloudFormation's AWS::Region intrinsic),
     * make sure to use Token.compareStrings()
     * instead of just comparing the values for equality.
     *
     * @stability stable
     */
    readonly region: string;
}
/**
 * Interface for the Resource construct.
 *
 * @stability stable
 */
export interface IResource extends IConstruct {
    /**
     * The stack in which this resource is defined.
     *
     * @stability stable
     */
    readonly stack: Stack;
    /**
     * The environment this resource belongs to.
     *
     * For resources that are created and managed by the CDK
     * (generally, those created by creating new class instances like Role, Bucket, etc.),
     * this is always the same as the environment of the stack they belong to;
     * however, for imported resources
     * (those obtained from static methods like fromRoleArn, fromBucketName, etc.),
     * that might be different than the stack they were imported into.
     *
     * @stability stable
     */
    readonly env: ResourceEnvironment;
    /**
     * Apply the given removal policy to this resource.
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be deleted (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * @stability stable
     */
    applyRemovalPolicy(policy: RemovalPolicy): void;
}
/**
 * Construction properties for {@link Resource}.
 *
 * @stability stable
 */
export interface ResourceProps {
    /**
     * The value passed in by users to the physical name prop of the resource.
     *
     * - `undefined` implies that a physical name will be allocated by
     *    CloudFormation during deployment.
     * - a concrete value implies a specific physical name
     * - `PhysicalName.GENERATE_IF_NEEDED` is a marker that indicates that a physical will only be generated
     *    by the CDK if it is needed for cross-environment references. Otherwise, it will be allocated by CloudFormation.
     *
     * @default - The physical name will be allocated by CloudFormation at deployment time
     * @stability stable
     */
    readonly physicalName?: string;
    /**
     * The AWS account ID this resource belongs to.
     *
     * @default - the resource is in the same account as the stack it belongs to
     * @stability stable
     */
    readonly account?: string;
    /**
     * The AWS region this resource belongs to.
     *
     * @default - the resource is in the same region as the stack it belongs to
     * @stability stable
     */
    readonly region?: string;
    /**
     * ARN to deduce region and account from.
     *
     * The ARN is parsed and the account and region are taken from the ARN.
     * This should be used for imported resources.
     *
     * Cannot be supplied together with either `account` or `region`.
     *
     * @default - take environment from `account`, `region` parameters, or use Stack environment.
     * @stability stable
     */
    readonly environmentFromArn?: string;
}
/**
 * A construct which represents an AWS resource.
 *
 * @stability stable
 */
export declare abstract class Resource extends CoreConstruct implements IResource {
    /**
     * Check whether the given construct is a Resource.
     *
     * @stability stable
     */
    static isResource(construct: IConstruct): construct is CfnResource;
    /**
     * The stack in which this resource is defined.
     *
     * @stability stable
     */
    readonly stack: Stack;
    /**
     * The environment this resource belongs to.
     *
     * For resources that are created and managed by the CDK
     * (generally, those created by creating new class instances like Role, Bucket, etc.),
     * this is always the same as the environment of the stack they belong to;
     * however, for imported resources
     * (those obtained from static methods like fromRoleArn, fromBucketName, etc.),
     * that might be different than the stack they were imported into.
     *
     * @stability stable
     */
    readonly env: ResourceEnvironment;
    /**
     * Returns a string-encoded token that resolves to the physical name that should be passed to the CloudFormation resource.
     *
     * This value will resolve to one of the following:
     * - a concrete value (e.g. `"my-awesome-bucket"`)
     * - `undefined`, when a name should be generated by CloudFormation
     * - a concrete name generated automatically during synthesis, in
     *    cross-environment scenarios.
     *
     * @stability stable
     */
    protected readonly physicalName: string;
    private _physicalName;
    private readonly _allowCrossEnvironment;
    /**
     * @stability stable
     */
    constructor(scope: Construct, id: string, props?: ResourceProps);
    /**
     * Called when this resource is referenced across environments
     * (account/region) to order to request that a physical name will be generated
     * for this resource during synthesis, so the resource can be referenced
     * through it's absolute name/arn.
     *
     * @internal
     */
    _enableCrossEnvironment(): void;
    /**
     * Apply the given removal policy to this resource.
     *
     * The Removal Policy controls what happens to this resource when it stops
     * being managed by CloudFormation, either because you've removed it from the
     * CDK application or because you've made a change that requires the resource
     * to be replaced.
     *
     * The resource can be deleted (`RemovalPolicy.DESTROY`), or left in your AWS
     * account for data recovery and cleanup later (`RemovalPolicy.RETAIN`).
     *
     * @stability stable
     */
    applyRemovalPolicy(policy: RemovalPolicy): void;
    /**
     * @stability stable
     */
    protected generatePhysicalName(): string;
    /**
     * Returns an environment-sensitive token that should be used for the resource's "name" attribute (e.g. `bucket.bucketName`).
     *
     * Normally, this token will resolve to `nameAttr`, but if the resource is
     * referenced across environments, it will be resolved to `this.physicalName`,
     * which will be a concrete name.
     *
     * @param nameAttr The CFN attribute which resolves to the resource's name.
     * @stability stable
     */
    protected getResourceNameAttribute(nameAttr: string): string;
    /**
     * Returns an environment-sensitive token that should be used for the resource's "ARN" attribute (e.g. `bucket.bucketArn`).
     *
     * Normally, this token will resolve to `arnAttr`, but if the resource is
     * referenced across environments, `arnComponents` will be used to synthesize
     * a concrete ARN with the resource's physical name. Make sure to reference
     * `this.physicalName` in `arnComponents`.
     *
     * @param arnAttr The CFN attribute which resolves to the ARN of the resource.
     * @param arnComponents The format of the ARN of this resource.
     * @stability stable
     */
    protected getResourceArnAttribute(arnAttr: string, arnComponents: ArnComponents): string;
}
