def payoff_call(ST: float, K: float) -> float:
    """
    Calculate the payoff of a European call option at expiration.
    
    A call option gives the holder the right to buy the underlying asset
    at the strike price. The payoff is max(ST - K, 0).
    
    Args:
        ST: Stock price at expiration
        K: Strike price
        
    Returns:
        Call option payoff (always non-negative)
        
    Example:
        >>> payoff_call(110.0, 100.0)
        10.0
        >>> payoff_call(90.0, 100.0)
        0.0
    """
    return max(ST - K, 0.0)

def payoff_put(ST: float, K: float) -> float:
    """
    Calculate the payoff of a European put option at expiration.
    
    A put option gives the holder the right to sell the underlying asset
    at the strike price. The payoff is max(K - ST, 0).
    
    Args:
        ST: Stock price at expiration
        K: Strike price
        
    Returns:
        Put option payoff (always non-negative)
        
    Example:
        >>> payoff_put(90.0, 100.0)
        10.0
        >>> payoff_put(110.0, 100.0)
        0.0
    """
    return max(K - ST, 0.0)

def payoff_asian_call(average_price: float, K: float) -> float:
    """
    Calculate the payoff of an Asian call option at expiration.
    
    An Asian call option's payoff is based on the average price of the
    underlying asset over a specified period. The payoff is max(average - K, 0).
    
    Args:
        average_price: Average price of the underlying asset over the option's life
        K: Strike price
        
    Returns:
        Asian call option payoff (always non-negative)
        
    Example:
        >>> payoff_asian_call(105.0, 100.0)
        5.0
        >>> payoff_asian_call(95.0, 100.0)
        0.0
    """
    return max(average_price - K, 0.0)
