"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.Application = void 0;
const jsiiDeprecationWarnings = require("../.warnings.jsii.js");
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
const cloudwatch = require("aws-cdk-lib/aws-cloudwatch");
const ec2 = require("aws-cdk-lib/aws-ec2");
const iam = require("aws-cdk-lib/aws-iam");
const aws_kinesisanalytics_1 = require("aws-cdk-lib/aws-kinesisanalytics");
const logs = require("aws-cdk-lib/aws-logs");
const core = require("aws-cdk-lib/core");
const environment_properties_1 = require("./private/environment-properties");
const flink_application_configuration_1 = require("./private/flink-application-configuration");
const validation_1 = require("./private/validation");
const metadata_resource_1 = require("aws-cdk-lib/core/lib/metadata-resource");
/**
 * Implements the functionality shared between CDK created and imported
 * IApplications.
 */
class ApplicationBase extends core.Resource {
    /** Implement the convenience `IApplication.addToPrincipalPolicy` method. */
    addToRolePolicy(policyStatement) {
        if (this.role) {
            this.role.addToPrincipalPolicy(policyStatement);
            return true;
        }
        return false;
    }
    get connections() {
        if (!this._connections) {
            throw new Error('This Application isn\'t associated with a VPC. Provide a "vpc" prop when creating the Application or "securityGroups" when importing it.');
        }
        return this._connections;
    }
    /**
     * Return a CloudWatch metric associated with this Flink application.
     *
     * @param metricName The name of the metric
     * @param props Customization properties
     */
    metric(metricName, props) {
        return new cloudwatch.Metric({
            namespace: 'AWS/KinesisAnalytics',
            metricName,
            dimensionsMap: { Application: this.applicationName },
            ...props,
        }).attachTo(this);
    }
    /**
     * The number of Kinesis Processing Units that are used to run your stream
     * processing application. The average number of KPUs used each hour
     * determines the billing for your application.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricKpus(props) {
        return this.metric('KPUs', { statistic: 'Average', ...props });
    }
    /**
     * The time elapsed during an outage for failing/recovering jobs.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricDowntime(props) {
        return this.metric('downtime', { statistic: 'Average', ...props });
    }
    /**
     * The time that the job has been running without interruption.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricUptime(props) {
        return this.metric('uptime', { statistic: 'Average', ...props });
    }
    /**
     * The total number of times this job has fully restarted since it was
     * submitted. This metric does not measure fine-grained restarts.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricFullRestarts(props) {
        return this.metric('fullRestarts', { statistic: 'Sum', ...props });
    }
    /**
     * The number of times checkpointing has failed.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricNumberOfFailedCheckpoints(props) {
        return this.metric('numberOfFailedCheckpoints', { statistic: 'Sum', ...props });
    }
    /**
     * The time it took to complete the last checkpoint.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - maximum over 5 minutes
     */
    metricLastCheckpointDuration(props) {
        return this.metric('lastCheckpointDuration', { statistic: 'Maximum', ...props });
    }
    /**
     * The total size of the last checkpoint.
     *
     * Units: Bytes
     *
     * Reporting Level: Application
     *
     * @default - maximum over 5 minutes
     */
    metricLastCheckpointSize(props) {
        return this.metric('lastCheckpointSize', { statistic: 'Maximum', ...props });
    }
    /**
     * The overall percentage of CPU utilization across task managers. For
     * example, if there are five task managers, Kinesis Data Analytics publishes
     * five samples of this metric per reporting interval.
     *
     * Units: Percentage
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricCpuUtilization(props) {
        return this.metric('cpuUtilization', { statistic: 'Average', ...props });
    }
    /**
     * Overall heap memory utilization across task managers. For example, if there
     * are five task managers, Kinesis Data Analytics publishes five samples of
     * this metric per reporting interval.
     *
     * Units: Percentage
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricHeapMemoryUtilization(props) {
        return this.metric('heapMemoryUtilization', { statistic: 'Average', ...props });
    }
    /**
     * The total time spent performing old garbage collection operations.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricOldGenerationGCTime(props) {
        return this.metric('oldGenerationGCTime', { statistic: 'Sum', ...props });
    }
    /**
     * The total number of old garbage collection operations that have occurred
     * across all task managers.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - sum over 5 minutes
     */
    metricOldGenerationGCCount(props) {
        return this.metric('oldGenerationGCCount', { statistic: 'Sum', ...props });
    }
    /**
     * The total number of live threads used by the application.
     *
     * Units: Count
     *
     * Reporting Level: Application
     *
     * @default - average over 5 minutes
     */
    metricThreadsCount(props) {
        return this.metric('threadsCount', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator, or task has
     * received.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsIn(props) {
        return this.metric('numRecordsIn', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has received
     * per second.
     *
     * Units: Count/Second
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsInPerSecond(props) {
        return this.metric('numRecordsInPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has emitted.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsOut(props) {
        return this.metric('numRecordsOut', { statistic: 'Average', ...props });
    }
    /**
     * The total number of records this application, operator or task has emitted
     * per second.
     *
     * Units: Count/Second
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricNumRecordsOutPerSecond(props) {
        return this.metric('numRecordsOutPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The number of records this operator or task has dropped due to arriving
     * late.
     *
     * Units: Count
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - sum over 5 minutes
     */
    metricNumLateRecordsDropped(props) {
        return this.metric('numLateRecordsDropped', { statistic: 'Sum', ...props });
    }
    /**
     * The last watermark this application/operator/task/thread has received.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - maximum over 5 minutes
     */
    metricCurrentInputWatermark(props) {
        return this.metric('currentInputWatermark', { statistic: 'Maximum', ...props });
    }
    /**
     * The last watermark this application/operator/task/thread has received.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - maximum over 5 minutes
     */
    metricCurrentOutputWatermark(props) {
        return this.metric('currentOutputWatermark', { statistic: 'Maximum', ...props });
    }
    /**
     * The amount of managed memory currently used.
     *
     * Units: Bytes
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryUsed(props) {
        return this.metric('managedMemoryUsed', { statistic: 'Average', ...props });
    }
    /**
     * The total amount of managed memory.
     *
     * Units: Bytes
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryTotal(props) {
        return this.metric('managedMemoryTotal', { statistic: 'Average', ...props });
    }
    /**
     * Derived from managedMemoryUsed/managedMemoryTotal.
     *
     * Units: Percentage
     *
     * Reporting Level: Application, Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricManagedMemoryUtilization(props) {
        return this.metric('managedMemoryUtilization', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is idle (has no data to
     * process) per second. Idle time excludes back pressured time, so if the task
     * is back pressured it is not idle.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricIdleTimeMsPerSecond(props) {
        return this.metric('idleTimeMsPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is back pressured per
     * second.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricBackPressuredTimeMsPerSecond(props) {
        return this.metric('backPressuredTimeMsPerSecond', { statistic: 'Average', ...props });
    }
    /**
     * The time (in milliseconds) this task or operator is busy (neither idle nor
     * back pressured) per second. Can be NaN, if the value could not be
     * calculated.
     *
     * Units: Milliseconds
     *
     * Reporting Level: Operator, Task, Parallelism
     *
     * @default - average over 5 minutes
     */
    metricBusyTimePerMsPerSecond(props) {
        return this.metric('busyTimePerMsPerSecond', { statistic: 'Average', ...props });
    }
}
/**
 * An imported Flink application.
 */
class Import extends ApplicationBase {
    constructor(scope, id, attrs) {
        super(scope, id);
        // Enhanced CDK Analytics Telemetry
        (0, metadata_resource_1.addConstructMetadata)(this, attrs);
        // Imported applications have no associated role or grantPrincipal
        this.grantPrincipal = new iam.UnknownPrincipal({ resource: this });
        this.role = undefined;
        this.applicationArn = attrs.applicationArn;
        const applicationName = core.Stack.of(scope).splitArn(attrs.applicationArn, core.ArnFormat.SLASH_RESOURCE_NAME).resourceName;
        if (!applicationName) {
            throw new Error(`applicationArn for fromApplicationArn (${attrs.applicationArn}) must include resource name`);
        }
        this.applicationName = applicationName;
        const securityGroups = attrs.securityGroups ?? [];
        if (securityGroups.length > 0) {
            this._connections = new ec2.Connections({ securityGroups });
        }
    }
}
/**
 * The L2 construct for Flink Kinesis Data Applications.
 *
 * @resource AWS::KinesisAnalyticsV2::Application
 *
 */
class Application extends ApplicationBase {
    /**
     * Import an existing Flink application defined outside of CDK code by
     * applicationName.
     */
    static fromApplicationName(scope, id, applicationName) {
        const applicationArn = core.Stack.of(scope).formatArn(applicationArnComponents(applicationName));
        return new Import(scope, id, { applicationArn });
    }
    /**
     * Import an existing application defined outside of CDK code by
     * applicationArn.
     */
    static fromApplicationArn(scope, id, applicationArn) {
        return new Import(scope, id, { applicationArn });
    }
    /**
     * Import an existing application defined outside of CDK code.
     */
    static fromApplicationAttributes(scope, id, attrs) {
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_kinesisanalytics_flink_alpha_ApplicationAttributes(attrs);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, this.fromApplicationAttributes);
            }
            throw error;
        }
        return new Import(scope, id, {
            applicationArn: attrs.applicationArn,
            securityGroups: attrs.securityGroups,
        });
    }
    constructor(scope, id, props) {
        super(scope, id, { physicalName: props.applicationName });
        try {
            jsiiDeprecationWarnings._aws_cdk_aws_kinesisanalytics_flink_alpha_ApplicationProps(props);
        }
        catch (error) {
            if (process.env.JSII_DEBUG !== "1" && error.name === "DeprecationError") {
                Error.captureStackTrace(error, Application);
            }
            throw error;
        }
        // Enhanced CDK Analytics Telemetry
        (0, metadata_resource_1.addConstructMetadata)(this, props);
        (0, validation_1.validateFlinkApplicationProps)(props);
        this.role = props.role ?? new iam.Role(this, 'Role', {
            assumedBy: new iam.ServicePrincipal('kinesisanalytics.amazonaws.com'),
        });
        this.grantPrincipal = this.role;
        // Permit metric publishing to CloudWatch
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['cloudwatch:PutMetricData'],
            resources: ['*'],
        }));
        const code = props.code.bind(this);
        code.bucket.grantRead(this);
        let vpcConfigurations;
        if (props.vpc) {
            const securityGroups = props.securityGroups ?? [
                new ec2.SecurityGroup(this, 'SecurityGroup', {
                    vpc: props.vpc,
                }),
            ];
            this._connections = new ec2.Connections({ securityGroups });
            const subnetSelection = props.vpcSubnets ?? {
                subnetType: ec2.SubnetType.PRIVATE_WITH_EGRESS,
            };
            vpcConfigurations = [{
                    securityGroupIds: securityGroups.map(sg => sg.securityGroupId),
                    subnetIds: props.vpc.selectSubnets(subnetSelection).subnetIds,
                }];
        }
        const resource = new aws_kinesisanalytics_1.CfnApplicationV2(this, 'Resource', {
            applicationName: props.applicationName,
            runtimeEnvironment: props.runtime.value,
            serviceExecutionRole: this.role.roleArn,
            applicationConfiguration: {
                ...code.applicationCodeConfigurationProperty,
                environmentProperties: (0, environment_properties_1.environmentProperties)(props.propertyGroups),
                flinkApplicationConfiguration: (0, flink_application_configuration_1.flinkApplicationConfiguration)({
                    checkpointingEnabled: props.checkpointingEnabled,
                    checkpointInterval: props.checkpointInterval,
                    minPauseBetweenCheckpoints: props.minPauseBetweenCheckpoints,
                    logLevel: props.logLevel,
                    metricsLevel: props.metricsLevel,
                    autoScalingEnabled: props.autoScalingEnabled,
                    parallelism: props.parallelism,
                    parallelismPerKpu: props.parallelismPerKpu,
                }),
                applicationSnapshotConfiguration: {
                    snapshotsEnabled: props.snapshotsEnabled ?? true,
                },
                vpcConfigurations,
            },
        });
        resource.node.addDependency(this.role);
        const logGroup = props.logGroup ?? new logs.LogGroup(this, 'LogGroup');
        const logStream = new logs.LogStream(this, 'LogStream', { logGroup });
        /* Permit logging */
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogGroups'],
            resources: [
                core.Stack.of(this).formatArn({
                    service: 'logs',
                    resource: 'log-group',
                    arnFormat: core.ArnFormat.COLON_RESOURCE_NAME,
                    resourceName: '*',
                }),
            ],
        }));
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:DescribeLogStreams'],
            resources: [logGroup.logGroupArn],
        }));
        const logStreamArn = `arn:${core.Aws.PARTITION}:logs:${core.Aws.REGION}:${core.Aws.ACCOUNT_ID}:log-group:${logGroup.logGroupName}:log-stream:${logStream.logStreamName}`;
        this.role.addToPrincipalPolicy(new iam.PolicyStatement({
            actions: ['logs:PutLogEvents'],
            resources: [logStreamArn],
        }));
        new aws_kinesisanalytics_1.CfnApplicationCloudWatchLoggingOptionV2(this, 'LoggingOption', {
            applicationName: resource.ref,
            cloudWatchLoggingOption: {
                logStreamArn,
            },
        });
        // Permissions required for VPC usage per:
        // https://docs.aws.amazon.com/kinesisanalytics/latest/java/vpc-permissions.html
        if (props.vpc) {
            this.role.addToPrincipalPolicy(new iam.PolicyStatement({
                actions: [
                    'ec2:DescribeVpcs',
                    'ec2:DescribeSubnets',
                    'ec2:DescribeSecurityGroups',
                    'ec2:DescribeDhcpOptions',
                    'ec2:CreateNetworkInterface',
                    'ec2:CreateNetworkInterfacePermission',
                    'ec2:DescribeNetworkInterfaces',
                    'ec2:DeleteNetworkInterface',
                ],
                resources: ['*'],
            }));
        }
        this.applicationName = this.getResourceNameAttribute(resource.ref);
        this.applicationArn = this.getResourceArnAttribute(core.Stack.of(this).formatArn(applicationArnComponents(resource.ref)), applicationArnComponents(this.physicalName));
        resource.applyRemovalPolicy(props.removalPolicy, {
            default: core.RemovalPolicy.DESTROY,
        });
    }
}
exports.Application = Application;
_a = JSII_RTTI_SYMBOL_1;
Application[_a] = { fqn: "@aws-cdk/aws-kinesisanalytics-flink-alpha.Application", version: "2.191.0-alpha.0" };
function applicationArnComponents(resourceName) {
    return {
        service: 'kinesisanalytics',
        resource: 'application',
        resourceName,
    };
}
//# sourceMappingURL=data:application/json;base64,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