import pytest

from math import factorial

from sweetpea.combinatorics import extract_components, compute_jth_inversion_sequence, construct_permutation, compute_jth_combination


@pytest.mark.parametrize('sizes, n, expected', [
    [[4, 2, 3], 0,  [0, 0, 0]],
    [[4, 2, 3], 1,  [1, 0, 0]],
    [[4, 2, 3], 2,  [2, 0, 0]],
    [[4, 2, 3], 3,  [3, 0, 0]],
    [[4, 2, 3], 4,  [0, 1, 0]],
    [[4, 2, 3], 5,  [1, 1, 0]],
    [[4, 2, 3], 6,  [2, 1, 0]],
    [[4, 2, 3], 7,  [3, 1, 0]],
    [[4, 2, 3], 8,  [0, 0, 1]],
    [[4, 2, 3], 9,  [1, 0, 1]],
    [[4, 2, 3], 10, [2, 0, 1]],
    [[4, 2, 3], 11, [3, 0, 1]],
    [[4, 2, 3], 12, [0, 1, 1]],
    [[4, 2, 3], 13, [1, 1, 1]],
    [[4, 2, 3], 14, [2, 1, 1]],
    [[4, 2, 3], 15, [3, 1, 1]],
    [[4, 2, 3], 16, [0, 0, 2]],
    [[4, 2, 3], 17, [1, 0, 2]],
    [[4, 2, 3], 18, [2, 0, 2]],
    [[4, 2, 3], 19, [3, 0, 2]],
    [[4, 2, 3], 20, [0, 1, 2]],
    [[4, 2, 3], 21, [1, 1, 2]],
    [[4, 2, 3], 22, [2, 1, 2]],
    [[4, 2, 3], 23, [3, 1, 2]]
])
def test_extract_components(sizes, n, expected):
    assert extract_components(sizes, n) == expected


@pytest.mark.parametrize('n, j, sequence', [
    # 4! = 24 Sequences
    [4, 0,  [0, 0, 0, 0]],
    [4, 1,  [1, 0, 0, 0]],
    [4, 2,  [2, 0, 0, 0]],
    [4, 3,  [3, 0, 0, 0]],
    [4, 4,  [0, 1, 0, 0]],
    [4, 5,  [1, 1, 0, 0]],
    [4, 6,  [2, 1, 0, 0]],
    [4, 7,  [3, 1, 0, 0]],
    [4, 8,  [0, 2, 0, 0]],
    [4, 9,  [1, 2, 0, 0]],
    [4, 10, [2, 2, 0, 0]],
    [4, 11, [3, 2, 0, 0]],
    [4, 12, [0, 0, 1, 0]],
    [4, 13, [1, 0, 1, 0]],
    [4, 14, [2, 0, 1, 0]],
    [4, 15, [3, 0, 1, 0]],
    [4, 16, [0, 1, 1, 0]],
    [4, 17, [1, 1, 1, 0]],
    [4, 18, [2, 1, 1, 0]],
    [4, 19, [3, 1, 1, 0]],
    [4, 20, [0, 2, 1, 0]],
    [4, 21, [1, 2, 1, 0]],
    [4, 22, [2, 2, 1, 0]],
    [4, 23, [3, 2, 1, 0]],

    # First is alway 0s, Last is always counting down to zero
    [10, 0,                 [0, 0, 0, 0, 0, 0, 0, 0, 0, 0]],
    [10, factorial(10) - 1, [9, 8, 7, 6, 5, 4, 3, 2, 1, 0]],
])
def test_compute_jth_inversion_sequence(n, j, sequence):
    assert compute_jth_inversion_sequence(n, j) == sequence


@pytest.mark.parametrize('inversion_sequence, expected_permutation', [
    [[0, 0, 0, 0], [0, 1, 2, 3]],
    [[1, 0, 0, 0], [1, 0, 2, 3]],
    [[2, 0, 0, 0], [1, 2, 0, 3]],
    [[3, 0, 0, 0], [1, 2, 3, 0]],
    [[0, 1, 0, 0], [0, 2, 1, 3]],
    [[1, 1, 0, 0], [2, 0, 1, 3]],
    [[2, 1, 0, 0], [2, 1, 0, 3]],
    [[3, 1, 0, 0], [2, 1, 3, 0]],
    [[0, 2, 0, 0], [0, 2, 3, 1]],
    [[1, 2, 0, 0], [2, 0, 3, 1]],
    [[2, 2, 0, 0], [2, 3, 0, 1]],
    [[3, 2, 0, 0], [2, 3, 1, 0]],
    [[0, 0, 1, 0], [0, 1, 3, 2]],
    [[1, 0, 1, 0], [1, 0, 3, 2]],
    [[2, 0, 1, 0], [1, 3, 0, 2]],
    [[3, 0, 1, 0], [1, 3, 2, 0]],
    [[0, 1, 1, 0], [0, 3, 1, 2]],
    [[1, 1, 1, 0], [3, 0, 1, 2]],
    [[2, 1, 1, 0], [3, 1, 0, 2]],
    [[3, 1, 1, 0], [3, 1, 2, 0]],
    [[0, 2, 1, 0], [0, 3, 2, 1]],
    [[1, 2, 1, 0], [3, 0, 2, 1]],
    [[2, 2, 1, 0], [3, 2, 0, 1]],
    [[3, 2, 1, 0], [3, 2, 1, 0]]
])
def test_construct_permutation(inversion_sequence, expected_permutation):
    assert construct_permutation(inversion_sequence) == expected_permutation


@pytest.mark.parametrize('l, n, j, expected_combination', [
    [4, 2, 0,  [0, 0, 0, 0]],
    [4, 2, 1,  [0, 0, 0, 1]],
    [4, 2, 2,  [0, 0, 1, 0]],
    [4, 2, 3,  [0, 0, 1, 1]],
    [4, 2, 4,  [0, 1, 0, 0]],
    [4, 2, 5,  [0, 1, 0, 1]],
    [4, 2, 6,  [0, 1, 1, 0]],
    [4, 2, 7,  [0, 1, 1, 1]],
    [4, 2, 8,  [1, 0, 0, 0]],
    [4, 2, 9,  [1, 0, 0, 1]],
    [4, 2, 10, [1, 0, 1, 0]],
    [4, 2, 11, [1, 0, 1, 1]],
    [4, 2, 12, [1, 1, 0, 0]],
    [4, 2, 13, [1, 1, 0, 1]],
    [4, 2, 14, [1, 1, 1, 0]],
    [4, 2, 15, [1, 1, 1, 1]],

    [2, 3, 0, [0, 0]],
    [2, 3, 1, [0, 1]],
    [2, 3, 2, [0, 2]],
    [2, 3, 3, [1, 0]],
    [2, 3, 4, [1, 1]],
    [2, 3, 5, [1, 2]],
    [2, 3, 6, [2, 0]],
    [2, 3, 7, [2, 1]],
    [2, 3, 8, [2, 2]]
])
def test_compute_jth_combination(l, n, j, expected_combination):
    assert compute_jth_combination(l, n, j) == expected_combination
