"""CMake configuration generator for variant-specific configuration."""

from pathlib import Path
from typing import Any, Optional

from yanga.domain.execution_context import ExecutionContext

from .cmake_backend import CMakeComment, CMakeElement, CMakeVariable
from .generator import CMakeGenerator


class ConfigCMakeGenerator(CMakeGenerator):
    """Generates CMake configuration file with variant-specific configuration variables."""

    def __init__(self, execution_context: ExecutionContext, output_dir: Path, config: Optional[dict[str, Any]] = None) -> None:
        super().__init__(execution_context, output_dir, config)

    def generate(self) -> list[CMakeElement]:
        """Generate CMake elements for variant configuration."""
        elements: list[CMakeElement] = []
        elements.append(CMakeComment(f"Generated by {self.__class__.__name__}"))

        if self.execution_context.variant:
            variant = self.execution_context.variant

            # Add base variant configuration
            if variant.config:
                elements.append(CMakeComment("Variant-specific configuration variables"))
                for key, value in variant.config.items():
                    elements.append(CMakeVariable(key, str(value)))

            # Add platform-specific configuration if available
            if self.execution_context.platform and variant.platforms and self.execution_context.platform.name in variant.platforms:
                platform_config = variant.platforms[self.execution_context.platform.name]
                if platform_config.config:
                    elements.append(CMakeComment(f"Platform-specific configuration variables for '{self.execution_context.platform.name}'"))
                    for key, value in platform_config.config.items():
                        elements.append(CMakeVariable(key, str(value)))

        return elements
