"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const os = require("os");
const path = require("path");
const aws_lambda_1 = require("aws-cdk-lib/aws-lambda");
const cdk = require("aws-cdk-lib/core");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    static bundle(options) {
        const bundling = new Bundling(options);
        return aws_lambda_1.Code.fromAsset(options.solutionDir, {
            assetHash: options.assetHash,
            assetHashType: options.assetHash
                ? cdk.AssetHashType.CUSTOM
                : cdk.AssetHashType.SOURCE,
            exclude: ['**/bin/', '**/obj/'],
            bundling: {
                image: bundling.image,
                command: bundling.command,
                environment: bundling.environment,
                local: bundling.local,
            },
        });
    }
    constructor(props) {
        this.props = props;
        Bundling.runsLocally = Bundling.runsLocally ?? (0, util_1.getDotNetLambdaTools)();
        const { solutionDir, projectDir } = props;
        this.relativeProjectPath = path.relative(path.resolve(solutionDir), path.resolve(projectDir));
        this.relativeProjectPath =
            this.relativeProjectPath === '' ? '.' : this.relativeProjectPath;
        this.msbuildParameters = props.msbuildParameters ?? [];
        if (props.runtime.family === aws_lambda_1.RuntimeFamily.OTHER) {
            this.msbuildParameters.push('--self-contained');
        }
        const environment = {
            ...props.environment,
        };
        // Docker Bundling
        const shouldBuildImage = props.forcedDockerBundling || !Bundling.runsLocally;
        this.image = shouldBuildImage
            ? props.dockerImage ??
                cdk.DockerImage.fromRegistry(Bundling.defaultBuildImage)
            : cdk.DockerImage.fromRegistry('dummy'); // Do not build if we don't need to
        const bundlingCommand = this.createBundlingCommand(cdk.AssetStaging.BUNDLING_INPUT_DIR, cdk.AssetStaging.BUNDLING_OUTPUT_DIR, props.architecture);
        this.command = ['bash', '-c', bundlingCommand];
        this.environment = environment;
        // Local bundling
        if (!props.forcedDockerBundling) {
            // only if Docker is not forced
            const osPlatform = os.platform();
            const createLocalCommand = (outputDir) => this.createBundlingCommand(solutionDir, outputDir, props.architecture, osPlatform);
            this.local = {
                tryBundle(outputDir) {
                    if (Bundling.runsLocally == false) {
                        process.stderr.write('dotnet build cannot run locally. Switching to Docker bundling.\n');
                        return false;
                    }
                    const localCommand = createLocalCommand(outputDir);
                    (0, util_1.exec)(osPlatform === 'win32' ? 'cmd' : 'bash', [osPlatform === 'win32' ? '/c' : '-c', localCommand], {
                        env: { ...process.env, ...(environment ?? {}) },
                        stdio: [
                            // show output
                            'ignore', // ignore stdio
                            process.stderr, // redirect stdout to stderr
                            'inherit', // inherit stderr
                        ],
                        cwd: props.solutionDir,
                        windowsVerbatimArguments: osPlatform === 'win32',
                    });
                    return true;
                },
            };
        }
    }
    createBundlingCommand(inputDir, outputDir, architecture, osPlatform = 'linux') {
        const pathJoin = osPathJoin(osPlatform);
        const projectLocation = this.relativeProjectPath.replace(/\\/g, '/');
        const packageFile = pathJoin(outputDir, 'package.zip');
        const dotnetPackageCommand = [
            'dotnet',
            'lambda',
            'package',
            '--project-location',
            projectLocation,
            '-farch',
            architecture.name,
            '--output-package',
            packageFile,
            this.msbuildParameters.length > 0
                ? `--msbuild-parameters "${this.msbuildParameters.join(' ')}"`
                : '',
        ]
            .filter((c) => !!c)
            .join(' ');
        const unzipCommand = osPlatform === 'win32'
            ? [
                'powershell',
                '-command',
                'Expand-Archive',
                packageFile,
                outputDir,
            ].join(' ')
            : ['unzip', '-od', outputDir, packageFile].filter((c) => !!c).join(' ');
        const deleteCommand = osPlatform === 'win32'
            ? ['powershell', '-command', 'Remove-Item', packageFile]
                .filter((c) => !!c)
                .join(' ')
            : ['rm', packageFile].filter((c) => !!c).join(' ');
        return chain([
            ...(this.props.commandHooks?.beforeBundling(inputDir, outputDir) ?? []),
            dotnetPackageCommand,
            unzipCommand,
            deleteCommand,
            ...(this.props.commandHooks?.afterBundling(inputDir, outputDir) ?? []),
        ]);
    }
}
exports.Bundling = Bundling;
Bundling.defaultBuildImage = 'public.ecr.aws/sam/build-dotnet8';
/**
 * Platform specific path join
 */
function osPathJoin(platform) {
    return function (...paths) {
        const joined = path.join(...paths);
        // If we are on win32 but need posix style paths
        if (os.platform() === 'win32' && platform !== 'win32') {
            return joined.replace(/\\/g, '/');
        }
        return joined;
    };
}
function chain(commands) {
    return commands.filter((c) => !!c).join(' && ');
}
//# sourceMappingURL=data:application/json;base64,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