"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.Bundling = void 0;
const fs = require("fs");
const path = require("path");
const lambda = require("@aws-cdk/aws-lambda");
const cdk = require("@aws-cdk/core");
const bundlers_1 = require("./bundlers");
const package_json_manager_1 = require("./package-json-manager");
const util_1 = require("./util");
/**
 * Bundling
 */
class Bundling {
    /**
     * Parcel bundled Lambda asset code
     */
    static parcel(options) {
        var _a, _b, _c, _d, _e, _f, _g, _h;
        // Find project root
        const projectRoot = (_d = (_c = (_b = (_a = options.projectRoot) !== null && _a !== void 0 ? _a : util_1.findUp(`.git${path.sep}`)) !== null && _b !== void 0 ? _b : util_1.findUp(bundlers_1.LockFile.YARN)) !== null && _c !== void 0 ? _c : util_1.findUp(bundlers_1.LockFile.NPM)) !== null && _d !== void 0 ? _d : util_1.findUp('package.json');
        if (!projectRoot) {
            throw new Error('Cannot find project root. Please specify it with `projectRoot`.');
        }
        const relativeEntryPath = path.relative(projectRoot, path.resolve(options.entry));
        const packageJsonManager = new package_json_manager_1.PackageJsonManager(path.dirname(options.entry));
        // Collect external and install modules
        let includeNodeModules;
        let dependencies;
        const externalModules = (_e = options.externalModules) !== null && _e !== void 0 ? _e : ['aws-sdk'];
        if (externalModules || options.nodeModules) {
            const modules = [...externalModules, ...(_f = options.nodeModules) !== null && _f !== void 0 ? _f : []];
            includeNodeModules = {};
            for (const mod of modules) {
                includeNodeModules[mod] = false;
            }
            if (options.nodeModules) {
                dependencies = packageJsonManager.getVersions(options.nodeModules);
            }
        }
        let installer = bundlers_1.Installer.NPM;
        let lockFile;
        if (dependencies) {
            // Use npm unless we have a yarn.lock.
            if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.YARN))) {
                installer = bundlers_1.Installer.YARN;
                lockFile = bundlers_1.LockFile.YARN;
            }
            else if (fs.existsSync(path.join(projectRoot, bundlers_1.LockFile.NPM))) {
                lockFile = bundlers_1.LockFile.NPM;
            }
        }
        // Configure target in package.json for Parcel
        packageJsonManager.update({
            targets: {
                'cdk-lambda': {
                    context: 'node',
                    includeNodeModules: includeNodeModules !== null && includeNodeModules !== void 0 ? includeNodeModules : true,
                    sourceMap: (_g = options.sourceMaps) !== null && _g !== void 0 ? _g : false,
                    minify: (_h = options.minify) !== null && _h !== void 0 ? _h : false,
                    engines: {
                        node: `>= ${runtimeVersion(options.runtime)}`,
                    },
                },
            },
        });
        // Local
        let localBundler;
        if (!options.forceDockerBundling) {
            localBundler = new bundlers_1.LocalBundler({
                projectRoot,
                relativeEntryPath,
                cacheDir: options.cacheDir,
                environment: options.parcelEnvironment,
                dependencies,
                installer,
                lockFile,
            });
        }
        // Docker
        const dockerBundler = new bundlers_1.DockerBundler({
            runtime: options.runtime,
            relativeEntryPath,
            cacheDir: options.cacheDir,
            environment: options.parcelEnvironment,
            buildImage: !bundlers_1.LocalBundler.runsLocally || options.forceDockerBundling,
            buildArgs: options.buildArgs,
            parcelVersion: options.parcelVersion,
            dependencies,
            installer,
            lockFile,
        });
        return lambda.Code.fromAsset(projectRoot, {
            assetHashType: cdk.AssetHashType.BUNDLE,
            bundling: {
                local: localBundler,
                ...dockerBundler.bundlingOptions,
            },
        });
    }
}
exports.Bundling = Bundling;
function runtimeVersion(runtime) {
    const match = runtime.name.match(/nodejs(\d+)/);
    if (!match) {
        throw new Error('Cannot extract version from runtime.');
    }
    return match[1];
}
//# sourceMappingURL=data:application/json;base64,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