"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const child_process = require("child_process");
const fs = require("fs");
const path = require("path");
const aws_lambda_1 = require("@aws-cdk/aws-lambda");
const core_1 = require("@aws-cdk/core");
const package_json_1 = require("delay/package.json");
const bundlers_1 = require("../lib/bundlers");
const bundling_1 = require("../lib/bundling");
const util = require("../lib/util");
jest.mock('@aws-cdk/aws-lambda');
const writeFileSyncMock = jest.spyOn(fs, 'writeFileSync').mockReturnValue();
const existsSyncOriginal = fs.existsSync;
const existsSyncMock = jest.spyOn(fs, 'existsSync');
const originalFindUp = util.findUp;
const findUpMock = jest.spyOn(util, 'findUp').mockImplementation((name, directory) => {
    if (name === 'package.json') {
        return path.join(__dirname, '..');
    }
    return originalFindUp(name, directory);
});
const fromAssetMock = jest.spyOn(core_1.BundlingDockerImage, 'fromAsset');
beforeEach(() => {
    jest.clearAllMocks();
});
test('Parcel bundling', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        cacheDir: 'cache-dir',
        projectRoot: '/project',
        parcelEnvironment: {
            KEY: 'value',
        },
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            local: {
                props: expect.objectContaining({
                    projectRoot: '/project',
                }),
            },
            environment: {
                KEY: 'value',
            },
            workingDirectory: '/asset-input/folder',
            command: [
                'bash', '-c',
                [
                    '$(node -p "require.resolve(\'parcel\')") build /asset-input/folder/entry.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist --cache-dir /asset-input/cache-dir',
                    'mv /asset-output/entry.js /asset-output/index.js',
                ].join(' && '),
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        targets: {
            'cdk-lambda': {
                context: 'node',
                includeNodeModules: {
                    'aws-sdk': false,
                },
                sourceMap: false,
                minify: false,
                engines: {
                    node: '>= 12',
                },
            },
        },
    }));
    // Searches for the package.json starting in the directory of the entry file
    expect(findUpMock).toHaveBeenCalledWith('package.json', '/project/folder');
});
test('Parcel bundling with handler named index.ts', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/index.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                '$(node -p "require.resolve(\'parcel\')") build /asset-input/folder/index.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist',
            ],
        }),
    });
});
test('Parcel bundling with tsx handler', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/handler.tsx',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                [
                    '$(node -p "require.resolve(\'parcel\')") build /asset-input/folder/handler.tsx --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist',
                    'mv /asset-output/handler.js /asset-output/index.js',
                ].join(' && '),
            ],
        }),
    });
});
test('Parcel with Windows paths', () => {
    bundling_1.Bundling.parcel({
        entry: 'C:\\my-project\\lib\\entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: 'C:\\my-project',
    });
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('C:\\my-project', expect.objectContaining({
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringContaining('/lib/entry.ts'),
            ]),
        }),
    }));
});
test('Parcel bundling with externals and dependencies', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        externalModules: ['abc'],
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: [
                'bash', '-c',
                [
                    '$(node -p "require.resolve(\'parcel\')") build /asset-input/folder/entry.ts --target cdk-lambda --dist-dir /asset-output --no-autoinstall --no-scope-hoist',
                    'mv /asset-output/entry.js /asset-output/index.js',
                    `echo \'{\"dependencies\":{\"delay\":\"${package_json_1.version}\"}}\' > /asset-output/package.json`,
                    'cd /asset-output',
                    'npm install',
                ].join(' && '),
            ],
        }),
    });
    // Correctly updates package.json
    const call = writeFileSyncMock.mock.calls[0];
    expect(call[0]).toMatch('package.json');
    expect(JSON.parse(call[1])).toEqual(expect.objectContaining({
        targets: expect.objectContaining({
            'cdk-lambda': expect.objectContaining({
                includeNodeModules: {
                    delay: false,
                    abc: false,
                },
            }),
        }),
    }));
});
test('Detects yarn.lock', () => {
    existsSyncMock.mockImplementation((p) => {
        if (/yarn.lock/.test(p.toString())) {
            return true;
        }
        return existsSyncOriginal(p);
    });
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        nodeModules: ['delay'],
    });
    // Correctly bundles with parcel
    expect(aws_lambda_1.Code.fromAsset).toHaveBeenCalledWith('/project', {
        assetHashType: core_1.AssetHashType.BUNDLE,
        bundling: expect.objectContaining({
            command: expect.arrayContaining([
                expect.stringMatching(/yarn\.lock.+yarn install/),
            ]),
        }),
    });
});
test('with Docker build args', () => {
    bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
        projectRoot: '/project',
        buildArgs: {
            HELLO: 'WORLD',
        },
        forceDockerBundling: true,
    });
    expect(fromAssetMock).toHaveBeenCalledWith(expect.stringMatching(/parcel$/), expect.objectContaining({
        buildArgs: expect.objectContaining({
            HELLO: 'WORLD',
        }),
    }));
});
test('Local bundling', () => {
    const spawnSyncMock = jest.spyOn(child_process, 'spawnSync').mockReturnValue({
        status: 0,
        stderr: Buffer.from('stderr'),
        stdout: Buffer.from('stdout'),
        pid: 123,
        output: ['stdout', 'stderr'],
        signal: null,
    });
    const bundler = new bundlers_1.LocalBundler({
        installer: bundlers_1.Installer.NPM,
        projectRoot: '/project',
        relativeEntryPath: 'folder/entry.ts',
        dependencies: {
            dep: 'version',
        },
        environment: {
            KEY: 'value',
        },
        lockFile: bundlers_1.LockFile.NPM,
    });
    bundler.tryBundle('/outdir');
    expect(spawnSyncMock).toHaveBeenCalledWith('bash', [
        '-c',
        [
            '$(node -p \"require.resolve(\'parcel\')\") build /project/folder/entry.ts --target cdk-lambda --dist-dir /outdir --no-autoinstall --no-scope-hoist',
            'mv /outdir/entry.js /outdir/index.js',
            'echo \'{\"dependencies\":{\"dep\":\"version\"}}\' > /outdir/package.json',
            'cp /project/package-lock.json /outdir/package-lock.json',
            'cd /outdir',
            'npm install',
        ].join(' && '),
    ], expect.objectContaining({
        env: expect.objectContaining({ KEY: 'value' }),
        cwd: '/project/folder',
    }));
    // Docker image is not built
    expect(fromAssetMock).not.toHaveBeenCalled();
});
test('LocalBundler.runsLocally checks parcel version and caches results', () => {
    bundlers_1.LocalBundler._runsLocally = undefined;
    const spawnSyncMock = jest.spyOn(child_process, 'spawnSync').mockReturnValue({
        status: 0,
        stderr: Buffer.from('stderr'),
        stdout: Buffer.from('2.0.0-beta.1'),
        pid: 123,
        output: ['stdout', 'stderr'],
        signal: null,
    });
    expect(bundlers_1.LocalBundler.runsLocally).toBe(true);
    expect(bundlers_1.LocalBundler.runsLocally).toBe(true);
    expect(spawnSyncMock).toHaveBeenCalledTimes(1);
    expect(spawnSyncMock).toHaveBeenCalledWith(expect.stringContaining('parcel'), ['--version']);
});
test('LocalBundler.runsLocally with incorrect parcel version', () => {
    bundlers_1.LocalBundler._runsLocally = undefined;
    jest.spyOn(child_process, 'spawnSync').mockReturnValue({
        status: 0,
        stderr: Buffer.from('stderr'),
        stdout: Buffer.from('3.5.1'),
        pid: 123,
        output: ['stdout', 'stderr'],
        signal: null,
    });
    expect(bundlers_1.LocalBundler.runsLocally).toBe(false);
});
test('Project root detection', () => {
    findUpMock.mockImplementation(() => undefined);
    expect(() => bundling_1.Bundling.parcel({
        entry: '/project/folder/entry.ts',
        runtime: aws_lambda_1.Runtime.NODEJS_12_X,
    })).toThrow(/Cannot find project root/);
    expect(findUpMock).toHaveBeenNthCalledWith(1, `.git${path.sep}`);
    expect(findUpMock).toHaveBeenNthCalledWith(2, bundlers_1.LockFile.YARN);
    expect(findUpMock).toHaveBeenNthCalledWith(3, bundlers_1.LockFile.NPM);
    expect(findUpMock).toHaveBeenNthCalledWith(4, 'package.json');
});
//# sourceMappingURL=data:application/json;base64,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