"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const tree_metadata_1 = require("./private/tree-metadata");
const stage_1 = require("./stage");
const APP_SYMBOL = Symbol.for('@aws-cdk/core.App');
/**
 * A construct which represents an entire CDK app. This construct is normally
 * the root of the construct tree.
 *
 * You would normally define an `App` instance in your program's entrypoint,
 * then define constructs where the app is used as the parent scope.
 *
 * After all the child constructs are defined within the app, you should call
 * `app.synth()` which will emit a "cloud assembly" from this app into the
 * directory specified by `outdir`. Cloud assemblies includes artifacts such as
 * CloudFormation templates and assets that are needed to deploy this app into
 * the AWS cloud.
 *
 * @see https://docs.aws.amazon.com/cdk/latest/guide/apps.html
 */
class App extends stage_1.Stage {
    /**
     * Checks if an object is an instance of the `App` class.
     * @returns `true` if `obj` is an `App`.
     * @param obj The object to evaluate
     */
    static isApp(obj) {
        return APP_SYMBOL in obj;
    }
    /**
     * Initializes a CDK application.
     * @param props initialization properties
     */
    constructor(props = {}) {
        var _a;
        super(undefined, '', {
            outdir: (_a = props.outdir) !== null && _a !== void 0 ? _a : process.env[cxapi.OUTDIR_ENV],
        });
        Object.defineProperty(this, APP_SYMBOL, { value: true });
        this.loadContext(props.context);
        if (props.stackTraces === false) {
            this.node.setContext(cxapi.DISABLE_METADATA_STACK_TRACE, true);
        }
        if (props.runtimeInfo === false) {
            this.node.setContext(cxapi.DISABLE_VERSION_REPORTING, true);
        }
        const autoSynth = props.autoSynth !== undefined ? props.autoSynth : cxapi.OUTDIR_ENV in process.env;
        if (autoSynth) {
            // synth() guarantuees it will only execute once, so a default of 'true'
            // doesn't bite manual calling of the function.
            process.once('beforeExit', () => this.synth());
        }
        if (props.treeMetadata === undefined || props.treeMetadata) {
            new tree_metadata_1.TreeMetadata(this);
        }
    }
    loadContext(defaults = {}) {
        // prime with defaults passed through constructor
        for (const [k, v] of Object.entries(defaults)) {
            this.node.setContext(k, v);
        }
        // read from environment
        const contextJson = process.env[cxapi.CONTEXT_ENV];
        const contextFromEnvironment = contextJson
            ? JSON.parse(contextJson)
            : {};
        for (const [k, v] of Object.entries(contextFromEnvironment)) {
            this.node.setContext(k, v);
        }
    }
}
exports.App = App;
//# sourceMappingURL=data:application/json;base64,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