"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const os = require("os");
const path = require("path");
const assets_1 = require("./assets");
const bundling_1 = require("./bundling");
const construct_compat_1 = require("./construct-compat");
const fs_1 = require("./fs");
/**
 * Stages a file or directory from a location on the file system into a staging
 * directory.
 *
 * This is controlled by the context key 'aws:cdk:asset-staging' and enabled
 * by the CLI by default in order to ensure that when the CDK app exists, all
 * assets are available for deployment. Otherwise, if an app references assets
 * in temporary locations, those will not be available when it exists (see
 * https://github.com/aws/aws-cdk/issues/1716).
 *
 * The `stagedPath` property is a stringified token that represents the location
 * of the file or directory after staging. It will be resolved only during the
 * "prepare" stage and may be either the original path or the staged path
 * depending on the context setting.
 *
 * The file/directory are staged based on their content hash (fingerprint). This
 * means that only if content was changed, copy will happen.
 */
class AssetStaging extends construct_compat_1.Construct {
    constructor(scope, id, props) {
        var _a, _b;
        super(scope, id);
        this.sourcePath = props.sourcePath;
        this.fingerprintOptions = props;
        if (props.bundling) {
            this.bundleDir = this.bundle(props.bundling);
        }
        this.assetHash = this.calculateHash(props);
        const stagingDisabled = this.node.tryGetContext(cxapi.DISABLE_ASSET_STAGING_CONTEXT);
        if (stagingDisabled) {
            this.stagedPath = (_a = this.bundleDir) !== null && _a !== void 0 ? _a : this.sourcePath;
        }
        else {
            this.relativePath = `asset.${this.assetHash}${path.extname((_b = this.bundleDir) !== null && _b !== void 0 ? _b : this.sourcePath)}`;
            this.stagedPath = this.relativePath;
        }
        this.sourceHash = this.assetHash;
    }
    synthesize(session) {
        // Staging is disabled
        if (!this.relativePath) {
            return;
        }
        const targetPath = path.join(session.assembly.outdir, this.relativePath);
        // Already staged
        if (fs.existsSync(targetPath)) {
            return;
        }
        // Asset has been bundled
        if (this.bundleDir) {
            // Try to rename bundling directory to staging directory
            try {
                fs.renameSync(this.bundleDir, targetPath);
                return;
            }
            catch (err) {
                // /tmp and cdk.out could be mounted across different mount points
                // in this case we will fallback to copying. This can happen in Windows
                // Subsystem for Linux (WSL).
                if (err.code === 'EXDEV') {
                    fs.mkdirSync(targetPath);
                    fs_1.FileSystem.copyDirectory(this.bundleDir, targetPath, this.fingerprintOptions);
                    return;
                }
                throw err;
            }
        }
        // Copy file/directory to staging directory
        const stat = fs.statSync(this.sourcePath);
        if (stat.isFile()) {
            fs.copyFileSync(this.sourcePath, targetPath);
        }
        else if (stat.isDirectory()) {
            fs.mkdirSync(targetPath);
            fs_1.FileSystem.copyDirectory(this.sourcePath, targetPath, this.fingerprintOptions);
        }
        else {
            throw new Error(`Unknown file type: ${this.sourcePath}`);
        }
    }
    bundle(options) {
        var _a, _b;
        // Create temporary directory for bundling
        const bundleDir = fs.mkdtempSync(path.resolve(path.join(os.tmpdir(), 'cdk-asset-bundle-')));
        // Always mount input and output dir
        const volumes = [
            {
                hostPath: this.sourcePath,
                containerPath: bundling_1.BUNDLING_INPUT_DIR,
            },
            {
                hostPath: bundleDir,
                containerPath: bundling_1.BUNDLING_OUTPUT_DIR,
            },
            ...(_a = options.volumes) !== null && _a !== void 0 ? _a : [],
        ];
        try {
            options.image._run({
                command: options.command,
                volumes,
                environment: options.environment,
                workingDirectory: (_b = options.workingDirectory) !== null && _b !== void 0 ? _b : bundling_1.BUNDLING_INPUT_DIR,
            });
        }
        catch (err) {
            throw new Error(`Failed to run bundling Docker image for asset ${this.node.path}: ${err}`);
        }
        if (fs_1.FileSystem.isEmpty(bundleDir)) {
            throw new Error(`Bundling did not produce any output. Check that your container writes content to ${bundling_1.BUNDLING_OUTPUT_DIR}.`);
        }
        return bundleDir;
    }
    calculateHash(props) {
        let hashType;
        if (props.assetHash) {
            if (props.assetHashType && props.assetHashType !== assets_1.AssetHashType.CUSTOM) {
                throw new Error(`Cannot specify \`${props.assetHashType}\` for \`assetHashType\` when \`assetHash\` is specified. Use \`CUSTOM\` or leave \`undefined\`.`);
            }
            hashType = assets_1.AssetHashType.CUSTOM;
        }
        else if (props.assetHashType) {
            hashType = props.assetHashType;
        }
        else {
            hashType = assets_1.AssetHashType.SOURCE;
        }
        switch (hashType) {
            case assets_1.AssetHashType.SOURCE:
                return fs_1.FileSystem.fingerprint(this.sourcePath, this.fingerprintOptions);
            case assets_1.AssetHashType.BUNDLE:
                if (!this.bundleDir) {
                    throw new Error('Cannot use `AssetHashType.BUNDLE` when `bundling` is not specified.');
                }
                return fs_1.FileSystem.fingerprint(this.bundleDir, this.fingerprintOptions);
            case assets_1.AssetHashType.CUSTOM:
                if (!props.assetHash) {
                    throw new Error('`assetHash` must be specified when `assetHashType` is set to `AssetHashType.CUSTOM`.');
                }
                return props.assetHash;
            default:
                throw new Error('Unknown asset hash type.');
        }
    }
}
exports.AssetStaging = AssetStaging;
//# sourceMappingURL=data:application/json;base64,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