"use strict";
/**
 * Constructs compatibility layer.
 *
 * This file includes types that shadow types in the "constructs" module in
 * order to allow backwards-compatiblity in the AWS CDK v1.0 release line.
 *
 * There are pretty ugly hacks here, which mostly involve downcasting types to
 * adhere to legacy AWS CDK APIs.
 *
 * This file, in its entirety, is expected to be removed in v2.0.
 */
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const constructs = require("constructs");
const token_1 = require("./token");
const ORIGINAL_CONSTRUCT_NODE_SYMBOL = Symbol.for('@aws-cdk/core.ConstructNode');
const CONSTRUCT_SYMBOL = Symbol.for('@aws-cdk/core.Construct');
/**
 * Represents the building block of the construct graph.
 *
 * All constructs besides the root construct must be created within the scope of
 * another construct.
 */
class Construct extends constructs.Construct {
    constructor(scope, id) {
        super(scope, id, {
            nodeFactory: {
                createNode: (h, s, i) => new ConstructNode(h, s, i)._actualNode,
            },
        });
        if (token_1.Token.isUnresolved(id)) {
            throw new Error(`Cannot use tokens in construct ID: ${id}`);
        }
        Object.defineProperty(this, CONSTRUCT_SYMBOL, { value: true });
        this.node = ConstructNode._unwrap(constructs.Node.of(this));
        const disableTrace = this.node.tryGetContext(cxapi.DISABLE_METADATA_STACK_TRACE);
        if (disableTrace) {
            this.node.setContext(constructs.ConstructMetadata.DISABLE_STACK_TRACE_IN_METADATA, true);
        }
    }
    /**
     * Return whether the given object is a Construct
     */
    static isConstruct(x) {
        return typeof x === 'object' && x !== null && CONSTRUCT_SYMBOL in x;
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if the construct is valid.
     */
    onValidate() {
        return this.validate();
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    onPrepare() {
        this.prepare();
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    onSynthesize(session) {
        this.synthesize({
            outdir: session.outdir,
            assembly: session.assembly,
        });
    }
    /**
     * Validate the current construct.
     *
     * This method can be implemented by derived constructs in order to perform
     * validation logic. It is called on all constructs before synthesis.
     *
     * @returns An array of validation error messages, or an empty array if the construct is valid.
     */
    validate() {
        return [];
    }
    /**
     * Perform final modifications before synthesis
     *
     * This method can be implemented by derived constructs in order to perform
     * final changes before synthesis. prepare() will be called after child
     * constructs have been prepared.
     *
     * This is an advanced framework feature. Only use this if you
     * understand the implications.
     */
    prepare() {
        return;
    }
    /**
     * Allows this construct to emit artifacts into the cloud assembly during synthesis.
     *
     * This method is usually implemented by framework-level constructs such as `Stack` and `Asset`
     * as they participate in synthesizing the cloud assembly.
     *
     * @param session The synthesis session.
     */
    synthesize(session) {
        ignore(session);
    }
}
exports.Construct = Construct;
/**
 * In what order to return constructs
 */
var ConstructOrder;
(function (ConstructOrder) {
    /**
     * Depth-first, pre-order
     */
    ConstructOrder[ConstructOrder["PREORDER"] = 0] = "PREORDER";
    /**
     * Depth-first, post-order (leaf nodes first)
     */
    ConstructOrder[ConstructOrder["POSTORDER"] = 1] = "POSTORDER";
})(ConstructOrder = exports.ConstructOrder || (exports.ConstructOrder = {}));
/**
 * Represents the construct node in the scope tree.
 */
class ConstructNode {
    constructor(host, scope, id) {
        this._actualNode = new constructs.Node(host, scope, id);
        // store a back reference on _actualNode so we can our ConstructNode from it
        Object.defineProperty(this._actualNode, ORIGINAL_CONSTRUCT_NODE_SYMBOL, {
            value: this,
            configurable: false,
            enumerable: false,
        });
    }
    /**
     * Returns the wrapping `@aws-cdk/core.ConstructNode` instance from a `constructs.ConstructNode`.
     *
     * @internal
     */
    static _unwrap(c) {
        const x = c[ORIGINAL_CONSTRUCT_NODE_SYMBOL];
        if (!x) {
            throw new Error('invalid ConstructNode type');
        }
        return x;
    }
    /**
     * Synthesizes a CloudAssembly from a construct tree.
     * @param node The root of the construct tree.
     * @param options Synthesis options.
     * @deprecated Use `app.synth()` or `stage.synth()` instead
     */
    static synth(node, options = {}) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const a = require('./private/synthesis');
        return a.synthesize(node.root, options);
    }
    /**
     * Invokes "prepare" on all constructs (depth-first, post-order) in the tree under `node`.
     * @param node The root node
     * @deprecated Use `app.synth()` instead
     */
    static prepare(node) {
        // eslint-disable-next-line @typescript-eslint/no-require-imports
        const p = require('./private/prepare-app');
        p.prepareApp(node.root); // resolve cross refs and nested stack assets.
        return node._actualNode.prepare();
    }
    /**
     * Invokes "validate" on all constructs in the tree (depth-first, pre-order) and returns
     * the list of all errors. An empty list indicates that there are no errors.
     *
     * @param node The root node
     */
    static validate(node) {
        return node._actualNode.validate().map(e => ({ source: e.source, message: e.message }));
    }
    /**
     * Returns the scope in which this construct is defined.
     *
     * The value is `undefined` at the root of the construct scope tree.
     */
    get scope() {
        return this._actualNode.scope;
    }
    /**
     * The id of this construct within the current scope.
     *
     * This is a a scope-unique id. To obtain an app-unique id for this construct, use `uniqueId`.
     */
    get id() { return this._actualNode.id; }
    /**
     * The full, absolute path of this construct in the tree.
     *
     * Components are separated by '/'.
     */
    get path() { return this._actualNode.path; }
    /**
     * A tree-global unique alphanumeric identifier for this construct.
     * Includes all components of the tree.
     */
    get uniqueId() { return this._actualNode.uniqueId; }
    /**
     * Return a direct child by id, or undefined
     *
     * @param id Identifier of direct child
     * @returns the child if found, or undefined
     */
    tryFindChild(id) { return this._actualNode.tryFindChild(id); }
    /**
     * Return a direct child by id
     *
     * Throws an error if the child is not found.
     *
     * @param id Identifier of direct child
     * @returns Child with the given id.
     */
    findChild(id) { return this._actualNode.findChild(id); }
    /**
     * Returns the child construct that has the id `Default` or `Resource"`.
     * This is usually the construct that provides the bulk of the underlying functionality.
     * Useful for modifications of the underlying construct that are not available at the higher levels.
     *
     * @throws if there is more than one child
     * @returns a construct or undefined if there is no default child
     */
    get defaultChild() { return this._actualNode.defaultChild; }
    /**
     * Override the defaultChild property.
     *
     * This should only be used in the cases where the correct
     * default child is not named 'Resource' or 'Default' as it
     * should be.
     *
     * If you set this to undefined, the default behavior of finding
     * the child named 'Resource' or 'Default' will be used.
     */
    set defaultChild(value) { this._actualNode.defaultChild = value; }
    /**
     * All direct children of this construct.
     */
    get children() { return this._actualNode.children; }
    /**
     * Return this construct and all of its children in the given order
     */
    findAll(order = ConstructOrder.PREORDER) { return this._actualNode.findAll(order); }
    /**
     * This can be used to set contextual values.
     * Context must be set before any children are added, since children may consult context info during construction.
     * If the key already exists, it will be overridden.
     * @param key The context key
     * @param value The context value
     */
    setContext(key, value) {
        if (token_1.Token.isUnresolved(key)) {
            throw new Error('Invalid context key: context keys can\'t include tokens');
        }
        this._actualNode.setContext(key, value);
    }
    /**
     * Retrieves a value from tree context.
     *
     * Context is usually initialized at the root, but can be overridden at any point in the tree.
     *
     * @param key The context key
     * @returns The context value or `undefined` if there is no context value for thie key.
     */
    tryGetContext(key) {
        if (token_1.Token.isUnresolved(key)) {
            throw new Error('Invalid context key: context keys can\'t include tokens');
        }
        return this._actualNode.tryGetContext(key);
    }
    /**
     * An immutable array of metadata objects associated with this construct.
     * This can be used, for example, to implement support for deprecation notices, source mapping, etc.
     */
    get metadata() { return this._actualNode.metadata; }
    /**
     * Adds a metadata entry to this construct.
     * Entries are arbitrary values and will also include a stack trace to allow tracing back to
     * the code location for when the entry was added. It can be used, for example, to include source
     * mapping in CloudFormation templates to improve diagnostics.
     *
     * @param type a string denoting the type of metadata
     * @param data the value of the metadata (can be a Token). If null/undefined, metadata will not be added.
     * @param fromFunction a function under which to restrict the metadata entry's stack trace (defaults to this.addMetadata)
     */
    addMetadata(type, data, fromFunction) { this._actualNode.addMetadata(type, data, fromFunction); }
    /**
     * Adds a { "info": <message> } metadata entry to this construct.
     * The toolkit will display the info message when apps are synthesized.
     * @param message The info message.
     */
    addInfo(message) {
        this._actualNode.addMetadata(cxschema.ArtifactMetadataEntryType.INFO, message);
    }
    /**
     * Adds a { "warning": <message> } metadata entry to this construct.
     * The toolkit will display the warning when an app is synthesized, or fail
     * if run in --strict mode.
     * @param message The warning message.
     */
    addWarning(message) {
        this._actualNode.addMetadata(cxschema.ArtifactMetadataEntryType.WARN, message);
    }
    /**
     * Adds an { "error": <message> } metadata entry to this construct.
     * The toolkit will fail synthesis when errors are reported.
     * @param message The error message.
     */
    addError(message) {
        this._actualNode.addMetadata(cxschema.ArtifactMetadataEntryType.ERROR, message);
    }
    /**
     * Applies the aspect to this Constructs node
     */
    applyAspect(aspect) { this._actualNode.applyAspect(aspect); }
    /**
     * All parent scopes of this construct.
     *
     * @returns a list of parent scopes. The last element in the list will always
     * be the current construct and the first element will be the root of the
     * tree.
     */
    get scopes() { return this._actualNode.scopes; }
    /**
     * @returns The root of the construct tree.
     */
    get root() { return this._actualNode.root; }
    /**
     * Returns true if this construct or the scopes in which it is defined are
     * locked.
     */
    get locked() { return this._actualNode.locked; }
    /**
     * Add an ordering dependency on another Construct.
     *
     * All constructs in the dependency's scope will be deployed before any
     * construct in this construct's scope.
     */
    addDependency(...dependencies) { this._actualNode.addDependency(...dependencies); }
    /**
     * Return all dependencies registered on this node or any of its children
     */
    get dependencies() { return this._actualNode.dependencies; }
    /**
     * Remove the child with the given name, if present.
     *
     * @returns Whether a child with the given name was deleted.
     * @experimental
     */
    tryRemoveChild(childName) { return this._actualNode.tryRemoveChild(childName); }
}
exports.ConstructNode = ConstructNode;
/**
 * Separator used to delimit construct path components.
 */
ConstructNode.PATH_SEP = '/';
function ignore(_x) {
    return;
}
//# sourceMappingURL=data:application/json;base64,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