"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const stack_1 = require("./stack");
const stage_1 = require("./stage");
const util_1 = require("./util");
/**
 * Adds a dependency between two resources or stacks, across stack and nested
 * stack boundaries.
 *
 * The algorithm consists of:
 * - Try to find the deepest common stack between the two elements
 * - If there isn't a common stack, it means the elements belong to two
 *   disjoined stack-trees and therefore we apply the dependency at the
 *   assembly/app level between the two topl-level stacks.
 * - If we did find a common stack, we apply the dependency as a CloudFormation
 *   "DependsOn" between the resources that "represent" our source and target
 *   either directly or through the AWS::CloudFormation::Stack resources that
 *   "lead" to them.
 *
 * @param source The source resource/stack (the depedent)
 * @param target The target resource/stack (the dependency)
 * @param reason Optional resource to associate with the dependency for
 * diagnostics
 */
function addDependency(source, target, reason) {
    if (source === target) {
        return;
    }
    const sourceStack = stack_1.Stack.of(source);
    const targetStack = stack_1.Stack.of(target);
    const sourceStage = stage_1.Stage.of(sourceStack);
    const targetStage = stage_1.Stage.of(targetStack);
    if (sourceStage !== targetStage) {
        throw new Error(`You cannot add a dependency from '${source.node.path}' (in ${describeStage(sourceStage)}) to '${target.node.path}' (in ${describeStage(targetStage)}): dependency cannot cross stage boundaries`);
    }
    // find the deepest common stack between the two elements
    const sourcePath = util_1.pathToTopLevelStack(sourceStack);
    const targetPath = util_1.pathToTopLevelStack(targetStack);
    const commonStack = util_1.findLastCommonElement(sourcePath, targetPath);
    // if there is no common stack, then define a assembly-level dependency
    // between the two top-level stacks
    if (!commonStack) {
        const topLevelSource = sourcePath[0]; // first path element is the top-level stack
        const topLevelTarget = targetPath[0];
        topLevelSource._addAssemblyDependency(topLevelTarget, reason);
        return;
    }
    // assertion: at this point if source and target are stacks, both are nested stacks.
    // since we have a common stack, it is impossible that both are top-level
    // stacks, so let's examine the two cases where one of them is top-level and
    // the other is nested.
    // case 1 - source is top-level and target is nested: this implies that
    // `target` is a direct or indirect nested stack of `source`, and an explicit
    // dependency is not required because nested stacks will always be deployed
    // before their parents.
    if (commonStack === source) {
        return;
    }
    // case 2 - source is nested and target is top-level: this implies that
    // `source` is a direct or indirect nested stack of `target`, and this is not
    // possible (nested stacks cannot depend on their parents).
    if (commonStack === target) {
        throw new Error(`Nested stack '${sourceStack.node.path}' cannot depend on a parent stack '${targetStack.node.path}': ${reason}`);
    }
    // we have a common stack from which we can reach both `source` and `target`
    // now we need to find two resources which are defined directly in this stack
    // and which can "lead us" to the source/target.
    const sourceResource = resourceInCommonStackFor(source);
    const targetResource = resourceInCommonStackFor(target);
    sourceResource._addResourceDependency(targetResource);
    function resourceInCommonStackFor(element) {
        const resource = stack_1.Stack.isStack(element) ? element.nestedStackResource : element;
        if (!resource) {
            throw new Error('assertion failure'); // see "assertion" above
        }
        const resourceStack = stack_1.Stack.of(resource);
        // we reached a resource defined in the common stack
        if (commonStack === resourceStack) {
            return resource;
        }
        return resourceInCommonStackFor(resourceStack);
    }
}
exports.addDependency = addDependency;
/**
 * Return a string representation of the given assembler, for use in error messages
 */
function describeStage(assembly) {
    if (!assembly) {
        return 'an unrooted construct tree';
    }
    if (!assembly.parentStage) {
        return 'the App';
    }
    return `Stage '${assembly.node.path}'`;
}
//# sourceMappingURL=data:application/json;base64,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