"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const constructs_1 = require("constructs");
const cfn_resource_1 = require("../cfn-resource");
const stack_1 = require("../stack");
const stage_1 = require("../stage");
const refs_1 = require("./refs");
/**
 * Prepares the app for synthesis. This function is called by the root `prepare`
 * (normally this the App, but if a Stack is a root, it is called by the stack),
 * which means it's the last 'prepare' that executes.
 *
 * It takes care of reifying cross-references between stacks (or nested stacks),
 * and of creating assets for nested stack templates.
 *
 * @param root The root of the construct tree.
 */
function prepareApp(root) {
    if (root.node.scope && !stage_1.Stage.isStage(root)) {
        throw new Error('prepareApp can only be called on a stage or a root construct');
    }
    // apply dependencies between resources in depending subtrees
    for (const dependency of root.node.dependencies) {
        const targetCfnResources = findCfnResources(dependency.target);
        const sourceCfnResources = findCfnResources(dependency.source);
        for (const target of targetCfnResources) {
            for (const source of sourceCfnResources) {
                source.addDependsOn(target);
            }
        }
    }
    // depth-first (children first) queue of nested stacks. We will pop a stack
    // from the head of this queue to prepare its template asset.
    const queue = findAllNestedStacks(root);
    while (true) {
        refs_1.resolveReferences(root);
        const nested = queue.shift();
        if (!nested) {
            break;
        }
        defineNestedStackAsset(nested);
    }
}
exports.prepareApp = prepareApp;
/**
 * Prepares the assets for nested stacks in this app.
 * @returns `true` if assets were added to the parent of a nested stack, which
 * implies that another round of reference resolution is in order. If this
 * function returns `false`, we know we are done.
 */
function defineNestedStackAsset(nestedStack) {
    // this is needed temporarily until we move NestedStack to '@aws-cdk/core'.
    const nested = nestedStack;
    nested._prepareTemplateAsset();
}
function findAllNestedStacks(root) {
    const result = new Array();
    const includeStack = (stack) => {
        if (!stack_1.Stack.isStack(stack)) {
            return false;
        }
        if (!stack.nested) {
            return false;
        }
        // test: if we are not within a stage, then include it.
        if (!stage_1.Stage.of(stack)) {
            return true;
        }
        return stage_1.Stage.of(stack) === root;
    };
    // create a list of all nested stacks in depth-first post order this means
    // that we first prepare the leaves and then work our way up.
    for (const stack of root.node.findAll(constructs_1.ConstructOrder.POSTORDER /* <== important */)) {
        if (includeStack(stack)) {
            result.push(stack);
        }
    }
    return result;
}
/**
 * Find all resources in a set of constructs
 */
function findCfnResources(root) {
    return root.node.findAll().filter(cfn_resource_1.CfnResource.isCfnResource);
}
//# sourceMappingURL=data:application/json;base64,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