"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const stage_1 = require("../stage");
const prepare_app_1 = require("./prepare-app");
function synthesize(root, options = {}) {
    // we start by calling "synth" on all nested assemblies (which will take care of all their children)
    synthNestedAssemblies(root, options);
    invokeAspects(root);
    // This is mostly here for legacy purposes as the framework itself does not use prepare anymore.
    prepareTree(root);
    // resolve references
    prepare_app_1.prepareApp(root);
    // give all children an opportunity to validate now that we've finished prepare
    if (!options.skipValidation) {
        validateTree(root);
    }
    // in unit tests, we support creating free-standing stacks, so we create the
    // assembly builder here.
    const builder = stage_1.Stage.isStage(root)
        ? root._assemblyBuilder
        : new cxapi.CloudAssemblyBuilder(options.outdir);
    // next, we invoke "onSynthesize" on all of our children. this will allow
    // stacks to add themselves to the synthesized cloud assembly.
    synthesizeTree(root, builder);
    return builder.buildAssembly({
        runtimeInfo: options.runtimeInfo,
    });
}
exports.synthesize = synthesize;
/**
 * Find Assemblies inside the construct and call 'synth' on them
 *
 * (They will in turn recurse again)
 */
function synthNestedAssemblies(root, options) {
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            child.synth(options);
        }
        else {
            synthNestedAssemblies(child, options);
        }
    }
}
/**
 * Invoke aspects on the given construct tree.
 *
 * Aspects are not propagated across Assembly boundaries. The same Aspect will not be invoked
 * twice for the same construct.
 */
function invokeAspects(root) {
    recurse(root, []);
    function recurse(construct, inheritedAspects) {
        // hackery to be able to access some private members with strong types (yack!)
        const node = construct.node._actualNode;
        const allAspectsHere = [...inheritedAspects !== null && inheritedAspects !== void 0 ? inheritedAspects : [], ...node._aspects];
        for (const aspect of allAspectsHere) {
            if (node.invokedAspects.includes(aspect)) {
                continue;
            }
            aspect.visit(construct);
            node.invokedAspects.push(aspect);
        }
        for (const child of construct.node.children) {
            if (!stage_1.Stage.isStage(child)) {
                recurse(child, allAspectsHere);
            }
        }
    }
}
/**
 * Prepare all constructs in the given construct tree in post-order.
 *
 * Stop at Assembly boundaries.
 */
function prepareTree(root) {
    visit(root, 'post', construct => construct.onPrepare());
}
/**
 * Synthesize children in post-order into the given builder
 *
 * Stop at Assembly boundaries.
 */
function synthesizeTree(root, builder) {
    visit(root, 'post', construct => construct.onSynthesize({
        outdir: builder.outdir,
        assembly: builder,
    }));
}
/**
 * Validate all constructs in the given construct tree
 */
function validateTree(root) {
    const errors = new Array();
    visit(root, 'pre', construct => {
        for (const message of construct.onValidate()) {
            errors.push({ message, source: construct });
        }
    });
    if (errors.length > 0) {
        const errorList = errors.map(e => `[${e.source.node.path}] ${e.message}`).join('\n  ');
        throw new Error(`Validation failed with the following errors:\n  ${errorList}`);
    }
}
/**
 * Visit the given construct tree in either pre or post order, stopping at Assemblies
 */
function visit(root, order, cb) {
    if (order === 'pre') {
        cb(root);
    }
    for (const child of root.node.children) {
        if (stage_1.Stage.isStage(child)) {
            continue;
        }
        visit(child, order, cb);
    }
    if (order === 'post') {
        cb(root);
    }
}
//# sourceMappingURL=data:application/json;base64,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