"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxschema = require("@aws-cdk/cloud-assembly-schema");
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const path = require("path");
const construct_compat_1 = require("./construct-compat");
const context_provider_1 = require("./context-provider");
const cloudformation_lang_1 = require("./private/cloudformation-lang");
const logical_id_1 = require("./private/logical-id");
const resolve_1 = require("./private/resolve");
const uniqueid_1 = require("./private/uniqueid");
const STACK_SYMBOL = Symbol.for('@aws-cdk/core.Stack');
const MY_STACK_CACHE = Symbol.for('@aws-cdk/core.Stack.myStack');
const VALID_STACK_NAME_REGEX = /^[A-Za-z][A-Za-z0-9-]*$/;
/**
 * A root construct which represents a single CloudFormation stack.
 */
class Stack extends construct_compat_1.Construct {
    /**
     * Creates a new stack.
     *
     * @param scope Parent of this stack, usually a Program instance.
     * @param id The construct ID of this stack. If `stackName` is not explicitly
     * defined, this id (and any parent IDs) will be used to determine the
     * physical ID of the stack.
     * @param props Stack properties.
     */
    constructor(scope, id, props = {}) {
        var _a;
        // For unit test convenience parents are optional, so bypass the type check when calling the parent.
        super(scope, id);
        /**
         * Options for CloudFormation template (like version, transform, description).
         */
        this.templateOptions = {};
        /**
         * Other stacks this stack depends on
         */
        this._stackDependencies = {};
        /**
         * Lists all missing contextual information.
         * This is returned when the stack is synthesized under the 'missing' attribute
         * and allows tooling to obtain the context and re-synthesize.
         */
        this._missingContext = new Array();
        Object.defineProperty(this, STACK_SYMBOL, { value: true });
        this._logicalIds = new logical_id_1.LogicalIDs();
        const { account, region, environment } = this.parseEnvironment(props.env);
        this.account = account;
        this.region = region;
        this.environment = environment;
        this.terminationProtection = props.terminationProtection;
        if (props.description !== undefined) {
            // Max length 1024 bytes
            // Typically 2 bytes per character, may be more for more exotic characters
            if (props.description.length > 512) {
                throw new Error(`Stack description must be <= 1024 bytes. Received description: '${props.description}'`);
            }
            this.templateOptions.description = props.description;
        }
        this._stackName = props.stackName !== undefined ? props.stackName : this.generateStackName();
        this.tags = new tag_manager_1.TagManager(cfn_resource_1.TagType.KEY_VALUE, 'aws:cdk:stack', props.tags);
        if (!VALID_STACK_NAME_REGEX.test(this.stackName)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${this.stackName}'`);
        }
        // the preferred behavior is to generate a unique id for this stack and use
        // it as the artifact ID in the assembly. this allows multiple stacks to use
        // the same name. however, this behavior is breaking for 1.x so it's only
        // applied under a feature flag which is applied automatically for new
        // projects created using `cdk init`.
        //
        // Also use the new behavior if we are using the new CI/CD-ready synthesizer; that way
        // people only have to flip one flag.
        // tslint:disable-next-line: max-line-length
        this.artifactId = this.node.tryGetContext(cxapi.ENABLE_STACK_NAME_DUPLICATES_CONTEXT) || this.node.tryGetContext(cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT)
            ? this.generateStackArtifactId()
            : this.stackName;
        this.templateFile = `${this.artifactId}.template.json`;
        this.synthesizer = (_a = props.synthesizer) !== null && _a !== void 0 ? _a : (this.node.tryGetContext(cxapi.NEW_STYLE_STACK_SYNTHESIS_CONTEXT)
            ? new stack_synthesizers_1.DefaultStackSynthesizer()
            : new stack_synthesizers_1.LegacyStackSynthesizer());
        this.synthesizer.bind(this);
    }
    /**
     * Return whether the given object is a Stack.
     *
     * We do attribute detection since we can't reliably use 'instanceof'.
     */
    static isStack(x) {
        return x !== null && typeof (x) === 'object' && STACK_SYMBOL in x;
    }
    /**
     * Looks up the first stack scope in which `construct` is defined. Fails if there is no stack up the tree.
     * @param construct The construct to start the search from.
     */
    static of(construct) {
        // we want this to be as cheap as possible. cache this result by mutating
        // the object. anecdotally, at the time of this writing, @aws-cdk/core unit
        // tests hit this cache 1,112 times, @aws-cdk/aws-cloudformation unit tests
        // hit this 2,435 times).
        const cache = construct[MY_STACK_CACHE];
        if (cache) {
            return cache;
        }
        else {
            const value = _lookup(construct);
            Object.defineProperty(construct, MY_STACK_CACHE, {
                enumerable: false,
                writable: false,
                configurable: false,
                value,
            });
            return value;
        }
        function _lookup(c) {
            if (Stack.isStack(c)) {
                return c;
            }
            if (!c.node.scope) {
                throw new Error(`No stack could be identified for the construct at path ${construct.node.path}`);
            }
            return _lookup(c.node.scope);
        }
    }
    /**
     * Resolve a tokenized value in the context of the current stack.
     */
    resolve(obj) {
        return resolve_1.resolve(obj, {
            scope: this,
            prefix: [],
            resolver: cloudformation_lang_1.CLOUDFORMATION_TOKEN_RESOLVER,
            preparing: false,
        });
    }
    /**
     * Convert an object, potentially containing tokens, to a JSON string
     */
    toJsonString(obj, space) {
        return cloudformation_lang_1.CloudFormationLang.toJSON(obj, space).toString();
    }
    /**
     * Indicate that a context key was expected
     *
     * Contains instructions which will be emitted into the cloud assembly on how
     * the key should be supplied.
     *
     * @param report The set of parameters needed to obtain the context
     */
    reportMissingContext(report) {
        if (!Object.values(cxschema.ContextProvider).includes(report.provider)) {
            throw new Error(`Unknown context provider requested in: ${JSON.stringify(report)}`);
        }
        this._missingContext.push(report);
    }
    /**
     * Rename a generated logical identities
     *
     * To modify the naming scheme strategy, extend the `Stack` class and
     * override the `allocateLogicalId` method.
     */
    renameLogicalId(oldId, newId) {
        this._logicalIds.addRename(oldId, newId);
    }
    /**
     * Allocates a stack-unique CloudFormation-compatible logical identity for a
     * specific resource.
     *
     * This method is called when a `CfnElement` is created and used to render the
     * initial logical identity of resources. Logical ID renames are applied at
     * this stage.
     *
     * This method uses the protected method `allocateLogicalId` to render the
     * logical ID for an element. To modify the naming scheme, extend the `Stack`
     * class and override this method.
     *
     * @param element The CloudFormation element for which a logical identity is
     * needed.
     */
    getLogicalId(element) {
        const logicalId = this.allocateLogicalId(element);
        return this._logicalIds.applyRename(logicalId);
    }
    /**
     * Add a dependency between this stack and another stack.
     *
     * This can be used to define dependencies between any two stacks within an
     * app, and also supports nested stacks.
     */
    addDependency(target, reason) {
        deps_1.addDependency(this, target, reason);
    }
    /**
     * Return the stacks this stack depends on
     */
    get dependencies() {
        return Object.values(this._stackDependencies).map(x => x.stack);
    }
    /**
     * The concrete CloudFormation physical stack name.
     *
     * This is either the name defined explicitly in the `stackName` prop or
     * allocated based on the stack's location in the construct tree. Stacks that
     * are directly defined under the app use their construct `id` as their stack
     * name. Stacks that are defined deeper within the tree will use a hashed naming
     * scheme based on the construct path to ensure uniqueness.
     *
     * If you wish to obtain the deploy-time AWS::StackName intrinsic,
     * you can use `Aws.stackName` directly.
     */
    get stackName() {
        return this._stackName;
    }
    /**
     * The partition in which this stack is defined
     */
    get partition() {
        // Always return a non-scoped partition intrinsic. These will usually
        // be used to construct an ARN, but there are no cross-partition
        // calls anyway.
        return cfn_pseudo_1.Aws.PARTITION;
    }
    /**
     * The Amazon domain suffix for the region in which this stack is defined
     */
    get urlSuffix() {
        // Since URL Suffix always follows partition, it is unscoped like partition is.
        return cfn_pseudo_1.Aws.URL_SUFFIX;
    }
    /**
     * The ID of the stack
     *
     * @example After resolving, looks like arn:aws:cloudformation:us-west-2:123456789012:stack/teststack/51af3dc0-da77-11e4-872e-1234567db123
     */
    get stackId() {
        return new cfn_pseudo_1.ScopedAws(this).stackId;
    }
    /**
     * Returns the list of notification Amazon Resource Names (ARNs) for the current stack.
     */
    get notificationArns() {
        return new cfn_pseudo_1.ScopedAws(this).notificationArns;
    }
    /**
     * Indicates if this is a nested stack, in which case `parentStack` will include a reference to it's parent.
     */
    get nested() {
        return this.nestedStackResource !== undefined;
    }
    /**
     * Creates an ARN from components.
     *
     * If `partition`, `region` or `account` are not specified, the stack's
     * partition, region and account will be used.
     *
     * If any component is the empty string, an empty string will be inserted
     * into the generated ARN at the location that component corresponds to.
     *
     * The ARN will be formatted as follows:
     *
     *   arn:{partition}:{service}:{region}:{account}:{resource}{sep}}{resource-name}
     *
     * The required ARN pieces that are omitted will be taken from the stack that
     * the 'scope' is attached to. If all ARN pieces are supplied, the supplied scope
     * can be 'undefined'.
     */
    formatArn(components) {
        return arn_1.Arn.format(components, this);
    }
    /**
     * Given an ARN, parses it and returns components.
     *
     * If the ARN is a concrete string, it will be parsed and validated. The
     * separator (`sep`) will be set to '/' if the 6th component includes a '/',
     * in which case, `resource` will be set to the value before the '/' and
     * `resourceName` will be the rest. In case there is no '/', `resource` will
     * be set to the 6th components and `resourceName` will be set to the rest
     * of the string.
     *
     * If the ARN includes tokens (or is a token), the ARN cannot be validated,
     * since we don't have the actual value yet at the time of this function
     * call. You will have to know the separator and the type of ARN. The
     * resulting `ArnComponents` object will contain tokens for the
     * subexpressions of the ARN, not string literals. In this case this
     * function cannot properly parse the complete final resourceName (path) out
     * of ARNs that use '/' to both separate the 'resource' from the
     * 'resourceName' AND to subdivide the resourceName further. For example, in
     * S3 ARNs:
     *
     *    arn:aws:s3:::my_corporate_bucket/path/to/exampleobject.png
     *
     * After parsing the resourceName will not contain
     * 'path/to/exampleobject.png' but simply 'path'. This is a limitation
     * because there is no slicing functionality in CloudFormation templates.
     *
     * @param arn The ARN string to parse
     * @param sepIfToken The separator used to separate resource from resourceName
     * @param hasName Whether there is a name component in the ARN at all. For
     * example, SNS Topics ARNs have the 'resource' component contain the topic
     * name, and no 'resourceName' component.
     *
     * @returns an ArnComponents object which allows access to the various
     * components of the ARN.
     *
     * @returns an ArnComponents object which allows access to the various
     *      components of the ARN.
     */
    parseArn(arn, sepIfToken = '/', hasName = true) {
        return arn_1.Arn.parse(arn, sepIfToken, hasName);
    }
    /**
     * Returnst the list of AZs that are availability in the AWS environment
     * (account/region) associated with this stack.
     *
     * If the stack is environment-agnostic (either account and/or region are
     * tokens), this property will return an array with 2 tokens that will resolve
     * at deploy-time to the first two availability zones returned from CloudFormation's
     * `Fn::GetAZs` intrinsic function.
     *
     * If they are not available in the context, returns a set of dummy values and
     * reports them as missing, and let the CLI resolve them by calling EC2
     * `DescribeAvailabilityZones` on the target environment.
     */
    get availabilityZones() {
        // if account/region are tokens, we can't obtain AZs through the context
        // provider, so we fallback to use Fn::GetAZs. the current lowest common
        // denominator is 2 AZs across all AWS regions.
        const agnostic = token_1.Token.isUnresolved(this.account) || token_1.Token.isUnresolved(this.region);
        if (agnostic) {
            return this.node.tryGetContext(cxapi.AVAILABILITY_ZONE_FALLBACK_CONTEXT_KEY) || [
                cfn_fn_1.Fn.select(0, cfn_fn_1.Fn.getAzs()),
                cfn_fn_1.Fn.select(1, cfn_fn_1.Fn.getAzs()),
            ];
        }
        const value = context_provider_1.ContextProvider.getValue(this, {
            provider: cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER,
            dummyValue: ['dummy1a', 'dummy1b', 'dummy1c'],
        }).value;
        if (!Array.isArray(value)) {
            throw new Error(`Provider ${cxschema.ContextProvider.AVAILABILITY_ZONE_PROVIDER} expects a list`);
        }
        return value;
    }
    /**
     * Register a file asset on this Stack
     *
     * @deprecated Use `stack.synthesizer.addFileAsset()` if you are calling,
     * and a different IDeploymentEnvironment class if you are implementing.
     */
    addFileAsset(asset) {
        return this.synthesizer.addFileAsset(asset);
    }
    /**
     * Register a docker image asset on this Stack
     *
     * @deprecated Use `stack.synthesizer.addDockerImageAsset()` if you are calling,
     * and a different `IDeploymentEnvironment` class if you are implementing.
     */
    addDockerImageAsset(asset) {
        return this.synthesizer.addDockerImageAsset(asset);
    }
    /**
     * If this is a nested stack, returns it's parent stack.
     */
    get nestedStackParent() {
        return this.nestedStackResource && Stack.of(this.nestedStackResource);
    }
    /**
     * Returns the parent of a nested stack.
     *
     * @deprecated use `nestedStackParent`
     */
    get parentStack() {
        return this.nestedStackParent;
    }
    /**
     * Add a Transform to this stack. A Transform is a macro that AWS
     * CloudFormation uses to process your template.
     *
     * Duplicate values are removed when stack is synthesized.
     *
     * @example addTransform('AWS::Serverless-2016-10-31')
     *
     * @see https://docs.aws.amazon.com/AWSCloudFormation/latest/UserGuide/transform-section-structure.html
     *
     * @param transform The transform to add
     */
    addTransform(transform) {
        if (!this.templateOptions.transforms) {
            this.templateOptions.transforms = [];
        }
        this.templateOptions.transforms.push(transform);
    }
    /**
     * Called implicitly by the `addDependency` helper function in order to
     * realize a dependency between two top-level stacks at the assembly level.
     *
     * Use `stack.addDependency` to define the dependency between any two stacks,
     * and take into account nested stack relationships.
     *
     * @internal
     */
    _addAssemblyDependency(target, reason) {
        // defensive: we should never get here for nested stacks
        if (this.nested || target.nested) {
            throw new Error('Cannot add assembly-level dependencies for nested stacks');
        }
        reason = reason || 'dependency added using stack.addDependency()';
        const cycle = target.stackDependencyReasons(this);
        if (cycle !== undefined) {
            // tslint:disable-next-line:max-line-length
            throw new Error(`'${target.node.path}' depends on '${this.node.path}' (${cycle.join(', ')}). Adding this dependency (${reason}) would create a cyclic reference.`);
        }
        let dep = this._stackDependencies[target.node.uniqueId];
        if (!dep) {
            dep = this._stackDependencies[target.node.uniqueId] = {
                stack: target,
                reasons: [],
            };
        }
        dep.reasons.push(reason);
        if (process.env.CDK_DEBUG_DEPS) {
            // tslint:disable-next-line:no-console
            console.error(`[CDK_DEBUG_DEPS] stack "${this.node.path}" depends on "${target.node.path}" because: ${reason}`);
        }
    }
    /**
     * Returns the naming scheme used to allocate logical IDs. By default, uses
     * the `HashedAddressingScheme` but this method can be overridden to customize
     * this behavior.
     *
     * In order to make sure logical IDs are unique and stable, we hash the resource
     * construct tree path (i.e. toplevel/secondlevel/.../myresource) and add it as
     * a suffix to the path components joined without a separator (CloudFormation
     * IDs only allow alphanumeric characters).
     *
     * The result will be:
     *
     *   <path.join('')><md5(path.join('/')>
     *     "human"      "hash"
     *
     * If the "human" part of the ID exceeds 240 characters, we simply trim it so
     * the total ID doesn't exceed CloudFormation's 255 character limit.
     *
     * We only take 8 characters from the md5 hash (0.000005 chance of collision).
     *
     * Special cases:
     *
     * - If the path only contains a single component (i.e. it's a top-level
     *   resource), we won't add the hash to it. The hash is not needed for
     *   disamiguation and also, it allows for a more straightforward migration an
     *   existing CloudFormation template to a CDK stack without logical ID changes
     *   (or renames).
     * - For aesthetic reasons, if the last components of the path are the same
     *   (i.e. `L1/L2/Pipeline/Pipeline`), they will be de-duplicated to make the
     *   resulting human portion of the ID more pleasing: `L1L2Pipeline<HASH>`
     *   instead of `L1L2PipelinePipeline<HASH>`
     * - If a component is named "Default" it will be omitted from the path. This
     *   allows refactoring higher level abstractions around constructs without affecting
     *   the IDs of already deployed resources.
     * - If a component is named "Resource" it will be omitted from the user-visible
     *   path, but included in the hash. This reduces visual noise in the human readable
     *   part of the identifier.
     *
     * @param cfnElement The element for which the logical ID is allocated.
     */
    allocateLogicalId(cfnElement) {
        const scopes = cfnElement.node.scopes;
        const stackIndex = scopes.indexOf(cfnElement.stack);
        const pathComponents = scopes.slice(stackIndex + 1).map(x => x.node.id);
        return uniqueid_1.makeUniqueId(pathComponents);
    }
    /**
     * Validate stack name
     *
     * CloudFormation stack names can include dashes in addition to the regular identifier
     * character classes, and we don't allow one of the magic markers.
     *
     * @internal
     */
    _validateId(name) {
        if (name && !VALID_STACK_NAME_REGEX.test(name)) {
            throw new Error(`Stack name must match the regular expression: ${VALID_STACK_NAME_REGEX.toString()}, got '${name}'`);
        }
    }
    synthesize(session) {
        // In principle, stack synthesis is delegated to the
        // StackSynthesis object.
        //
        // However, some parts of synthesis currently use some private
        // methods on Stack, and I don't really see the value in refactoring
        // this right now, so some parts still happen here.
        const builder = session.assembly;
        // write the CloudFormation template as a JSON file
        const outPath = path.join(builder.outdir, this.templateFile);
        const text = JSON.stringify(this._toCloudFormation(), undefined, 2);
        fs.writeFileSync(outPath, text);
        for (const ctx of this._missingContext) {
            builder.addMissing(ctx);
        }
        // Delegate adding artifacts to the Synthesizer
        this.synthesizer.synthesizeStackArtifacts(session);
    }
    /**
     * Returns the CloudFormation template for this stack by traversing
     * the tree and invoking _toCloudFormation() on all Entity objects.
     *
     * @internal
     */
    _toCloudFormation() {
        let transform;
        if (this.templateOptions.transform) {
            // tslint:disable-next-line: max-line-length
            this.node.addWarning('This stack is using the deprecated `templateOptions.transform` property. Consider switching to `addTransform()`.');
            this.addTransform(this.templateOptions.transform);
        }
        if (this.templateOptions.transforms) {
            if (this.templateOptions.transforms.length === 1) { // Extract single value
                transform = this.templateOptions.transforms[0];
            }
            else { // Remove duplicate values
                transform = Array.from(new Set(this.templateOptions.transforms));
            }
        }
        const template = {
            Description: this.templateOptions.description,
            Transform: transform,
            AWSTemplateFormatVersion: this.templateOptions.templateFormatVersion,
            Metadata: this.templateOptions.metadata,
        };
        const elements = cfnElements(this);
        const fragments = elements.map(e => this.resolve(e._toCloudFormation()));
        // merge in all CloudFormation fragments collected from the tree
        for (const fragment of fragments) {
            merge(template, fragment);
        }
        // resolve all tokens and remove all empties
        const ret = this.resolve(template) || {};
        this._logicalIds.assertAllRenamesApplied();
        return ret;
    }
    /**
     * Deprecated.
     *
     * @see https://github.com/aws/aws-cdk/pull/7187
     * @returns reference itself without any change
     * @deprecated cross reference handling has been moved to `App.prepare()`.
     */
    prepareCrossReference(_sourceStack, reference) {
        return reference;
    }
    /**
     * Determine the various stack environment attributes.
     *
     */
    parseEnvironment(env = {}) {
        var _a, _b, _c, _d;
        // if an environment property is explicitly specified when the stack is
        // created, it will be used. if not, use tokens for account and region.
        //
        // (They do not need to be anchored to any construct like resource attributes
        // are, because we'll never Export/Fn::ImportValue them -- the only situation
        // in which Export/Fn::ImportValue would work is if the value are the same
        // between producer and consumer anyway, so we can just assume that they are).
        const containingAssembly = stage_1.Stage.of(this);
        const account = (_b = (_a = env.account) !== null && _a !== void 0 ? _a : containingAssembly === null || containingAssembly === void 0 ? void 0 : containingAssembly.account) !== null && _b !== void 0 ? _b : cfn_pseudo_1.Aws.ACCOUNT_ID;
        const region = (_d = (_c = env.region) !== null && _c !== void 0 ? _c : containingAssembly === null || containingAssembly === void 0 ? void 0 : containingAssembly.region) !== null && _d !== void 0 ? _d : cfn_pseudo_1.Aws.REGION;
        // this is the "aws://" env specification that will be written to the cloud assembly
        // manifest. it will use "unknown-account" and "unknown-region" to indicate
        // environment-agnosticness.
        const envAccount = !token_1.Token.isUnresolved(account) ? account : cxapi.UNKNOWN_ACCOUNT;
        const envRegion = !token_1.Token.isUnresolved(region) ? region : cxapi.UNKNOWN_REGION;
        return {
            account, region,
            environment: cxapi.EnvironmentUtils.format(envAccount, envRegion),
        };
    }
    /**
     * Check whether this stack has a (transitive) dependency on another stack
     *
     * Returns the list of reasons on the dependency path, or undefined
     * if there is no dependency.
     */
    stackDependencyReasons(other) {
        if (this === other) {
            return [];
        }
        for (const dep of Object.values(this._stackDependencies)) {
            const ret = dep.stack.stackDependencyReasons(other);
            if (ret !== undefined) {
                return [...dep.reasons, ...ret];
            }
        }
        return undefined;
    }
    /**
     * Calculate the stack name based on the construct path
     *
     * The stack name is the name under which we'll deploy the stack,
     * and incorporates containing Stage names by default.
     *
     * Generally this looks a lot like how logical IDs are calculated.
     * The stack name is calculated based on the construct root path,
     * as follows:
     *
     * - Path is calculated with respect to containing App or Stage (if any)
     * - If the path is one component long just use that component, otherwise
     *   combine them with a hash.
     *
     * Since the hash is quite ugly and we'd like to avoid it if possible -- but
     * we can't anymore in the general case since it has been written into legacy
     * stacks. The introduction of Stages makes it possible to make this nicer however.
     * When a Stack is nested inside a Stage, we use the path components below the
     * Stage, and prefix the path components of the Stage before it.
     */
    generateStackName() {
        const assembly = stage_1.Stage.of(this);
        const prefix = (assembly && assembly.stageName) ? `${assembly.stageName}-` : '';
        return `${prefix}${this.generateStackId(assembly)}`;
    }
    /**
     * The artifact ID for this stack
     *
     * Stack artifact ID is unique within the App's Cloud Assembly.
     */
    generateStackArtifactId() {
        return this.generateStackId(this.node.root);
    }
    /**
     * Generate an ID with respect to the given container construct.
     */
    generateStackId(container) {
        const rootPath = rootPathTo(this, container);
        const ids = rootPath.map(c => c.node.id);
        // In unit tests our Stack (which is the only component) may not have an
        // id, so in that case just pretend it's "Stack".
        if (ids.length === 1 && !ids[0]) {
            ids[0] = 'Stack';
        }
        return makeStackName(ids);
    }
}
exports.Stack = Stack;
function merge(template, fragment) {
    for (const section of Object.keys(fragment)) {
        const src = fragment[section];
        // create top-level section if it doesn't exist
        const dest = template[section];
        if (!dest) {
            template[section] = src;
        }
        else {
            template[section] = mergeSection(section, dest, src);
        }
    }
}
function mergeSection(section, val1, val2) {
    switch (section) {
        case 'Description':
            return `${val1}\n${val2}`;
        case 'AWSTemplateFormatVersion':
            if (val1 != null && val2 != null && val1 !== val2) {
                throw new Error(`Conflicting CloudFormation template versions provided: '${val1}' and '${val2}`);
            }
            return val1 !== null && val1 !== void 0 ? val1 : val2;
        case 'Resources':
        case 'Conditions':
        case 'Parameters':
        case 'Outputs':
        case 'Mappings':
        case 'Metadata':
        case 'Transform':
            return mergeObjectsWithoutDuplicates(section, val1, val2);
        default:
            throw new Error(`CDK doesn't know how to merge two instances of the CFN template section '${section}' - ` +
                'please remove one of them from your code');
    }
}
function mergeObjectsWithoutDuplicates(section, dest, src) {
    if (typeof dest !== 'object') {
        throw new Error(`Expecting ${JSON.stringify(dest)} to be an object`);
    }
    if (typeof src !== 'object') {
        throw new Error(`Expecting ${JSON.stringify(src)} to be an object`);
    }
    // add all entities from source section to destination section
    for (const id of Object.keys(src)) {
        if (id in dest) {
            throw new Error(`section '${section}' already contains '${id}'`);
        }
        dest[id] = src[id];
    }
    return dest;
}
/**
 * Collect all CfnElements from a Stack.
 *
 * @param node Root node to collect all CfnElements from
 * @param into Array to append CfnElements to
 * @returns The same array as is being collected into
 */
function cfnElements(node, into = []) {
    if (cfn_element_1.CfnElement.isCfnElement(node)) {
        into.push(node);
    }
    for (const child of node.node.children) {
        // Don't recurse into a substack
        if (Stack.isStack(child)) {
            continue;
        }
        cfnElements(child, into);
    }
    return into;
}
/**
 * Return the construct root path of the given construct relative to the given ancestor
 *
 * If no ancestor is given or the ancestor is not found, return the entire root path.
 */
function rootPathTo(construct, ancestor) {
    const scopes = construct.node.scopes;
    for (let i = scopes.length - 2; i >= 0; i--) {
        if (scopes[i] === ancestor) {
            return scopes.slice(i + 1);
        }
    }
    return scopes;
}
exports.rootPathTo = rootPathTo;
/**
 * makeUniqueId, specialized for Stack names
 *
 * Stack names may contain '-', so we allow that character if the stack name
 * has only one component. Otherwise we fall back to the regular "makeUniqueId"
 * behavior.
 */
function makeStackName(components) {
    if (components.length === 1) {
        return components[0];
    }
    return uniqueid_1.makeUniqueId(components);
}
// These imports have to be at the end to prevent circular imports
const arn_1 = require("./arn");
const cfn_element_1 = require("./cfn-element");
const cfn_fn_1 = require("./cfn-fn");
const cfn_pseudo_1 = require("./cfn-pseudo");
const cfn_resource_1 = require("./cfn-resource");
const deps_1 = require("./deps");
const stack_synthesizers_1 = require("./stack-synthesizers");
const stage_1 = require("./stage");
const tag_manager_1 = require("./tag-manager");
const token_1 = require("./token");
//# sourceMappingURL=data:application/json;base64,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