"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const cxapi = require("@aws-cdk/cx-api");
const construct_compat_1 = require("./construct-compat");
const runtime_info_1 = require("./private/runtime-info");
const synthesis_1 = require("./private/synthesis");
/**
 * An abstract application modeling unit consisting of Stacks that should be
 * deployed together.
 *
 * Derive a subclass of `Stage` and use it to model a single instance of your
 * application.
 *
 * You can then instantiate your subclass multiple times to model multiple
 * copies of your application which should be be deployed to different
 * environments.
 */
class Stage extends construct_compat_1.Construct {
    constructor(scope, id, props = {}) {
        var _a, _b, _c, _d, _e, _f, _g;
        super(scope, id);
        if (id !== '' && !/^[a-z][a-z0-9\-\_\.]+$/i.test(id)) {
            throw new Error(`invalid stage name "${id}". Stage name must start with a letter and contain only alphanumeric characters, hypens ('-'), underscores ('_') and periods ('.')`);
        }
        this.parentStage = Stage.of(this);
        this.region = (_b = (_a = props.env) === null || _a === void 0 ? void 0 : _a.region) !== null && _b !== void 0 ? _b : (_c = this.parentStage) === null || _c === void 0 ? void 0 : _c.region;
        this.account = (_e = (_d = props.env) === null || _d === void 0 ? void 0 : _d.account) !== null && _e !== void 0 ? _e : (_f = this.parentStage) === null || _f === void 0 ? void 0 : _f.account;
        this._assemblyBuilder = this.createBuilder(props.outdir);
        this.stageName = [(_g = this.parentStage) === null || _g === void 0 ? void 0 : _g.stageName, id].filter(x => x).join('-');
    }
    /**
     * Return the stage this construct is contained with, if available. If called
     * on a nested stage, returns its parent.
     *
     * @experimental
     */
    static of(construct) {
        return construct.node.scopes.reverse().slice(1).find(Stage.isStage);
    }
    /**
     * Test whether the given construct is a stage.
     *
     * @experimental
     */
    static isStage(x) {
        return x !== null && x instanceof Stage;
    }
    /**
     * Artifact ID of the assembly if it is a nested stage. The root stage (app)
     * will return an empty string.
     *
     * Derived from the construct path.
     *
     * @experimental
     */
    get artifactId() {
        if (!this.node.path) {
            return '';
        }
        return `assembly-${this.node.path.replace(/\//g, '-').replace(/^-+|-+$/g, '')}`;
    }
    /**
     * Synthesize this stage into a cloud assembly.
     *
     * Once an assembly has been synthesized, it cannot be modified. Subsequent
     * calls will return the same assembly.
     */
    synth(options = {}) {
        if (!this.assembly) {
            const runtimeInfo = this.node.tryGetContext(cxapi.DISABLE_VERSION_REPORTING) ? undefined : runtime_info_1.collectRuntimeInformation();
            this.assembly = synthesis_1.synthesize(this, {
                skipValidation: options.skipValidation,
                runtimeInfo,
            });
        }
        return this.assembly;
    }
    createBuilder(outdir) {
        // cannot specify "outdir" if we are a nested stage
        if (this.parentStage && outdir) {
            throw new Error('"outdir" cannot be specified for nested stages');
        }
        // Need to determine fixed output directory already, because we must know where
        // to write sub-assemblies (which must happen before we actually get to this app's
        // synthesize() phase).
        return this.parentStage
            ? this.parentStage._assemblyBuilder.createNestedAssembly(this.artifactId, this.node.path)
            : new cxapi.CloudAssemblyBuilder(outdir);
    }
}
exports.Stage = Stage;
//# sourceMappingURL=data:application/json;base64,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