"""GeoTessera: Access to geospatial embeddings from the Tessera foundation model.

GeoTessera provides a Python interface to access and work with representation maps
generated by the Tessera foundation model. Tessera processes Sentinel-1 and Sentinel-2
satellite imagery to generate 128-dimensional embeddings at 10m spatial resolution,
compressing a full year of temporal-spectral features into dense geospatial representations.

Key Features:
    - Access pre-computed embeddings for any global location
    - Automatic data fetching and caching from remote servers
    - Efficient dequantization of compressed embeddings
    - Visualization tools for RGB composite generation
    - Integration with standard geospatial libraries (GeoPandas, Shapely)

Basic Usage:
    >>> from geotessera import GeoTessera
    >>> gt = GeoTessera()
    >>>
    >>> # Fetch embeddings for a specific location
    >>> embedding = gt.fetch_embedding(lat=52.2053, lon=0.1218)  # Cambridge, UK
    >>> print(embedding.shape)  # (height, width, 128)
    >>>
    >>> # List available embeddings for a region
    >>> tiles = gt.get_available_embeddings(lat=52.2, lon=0.1, buffer=0.2)
    >>>
    >>> # Visualize embeddings as RGB composite
    >>> gt.visualize_embedding(embedding)

The embeddings represent learned features from a full year of satellite observations,
enabling various downstream applications including land classification, change detection,
and environmental monitoring.

Note:
    Embeddings are organized in a 0.1-degree grid system. Each tile contains
    quantized embeddings and corresponding scale factors for efficient storage
    and transmission. Files are cached locally after first download.

See Also:
    GeoTessera: Main class for accessing Tessera embeddings
    https://github.com/ucam-eo/tessera: Tessera foundation model documentation
"""

from .core import GeoTessera
from . import io
from . import spatial
from . import parallel
from . import export

try:
    import importlib.metadata

    __version__ = importlib.metadata.version("geotessera")
except importlib.metadata.PackageNotFoundError:
    # Fallback for development installs
    __version__ = "unknown"
__all__ = ["GeoTessera", "io", "spatial", "parallel", "export"]
