"use strict";
const cxapi = require("@aws-cdk/cx-api");
const fs = require("fs");
const path = require("path");
const index_1 = require("../../lib/index");
class AbstractCfnResource extends index_1.CfnResource {
    constructor(scope, id) {
        super(scope, id, {
            type: 'CDK::UnitTest::MyCfnResource'
        });
    }
    inspect(inspector) {
        inspector.addAttribute('aws:cdk:cloudformation:type', 'CDK::UnitTest::MyCfnResource');
        inspector.addAttribute('aws:cdk:cloudformation:props', this.cfnProperties);
    }
}
function readJson(outdir, file) {
    return JSON.parse(fs.readFileSync(path.join(outdir, file), 'utf-8'));
}
module.exports = {
    'tree metadata is generated as expected'(test) {
        const app = new index_1.App();
        const stack = new index_1.Stack(app, 'mystack');
        new index_1.Construct(stack, 'myconstruct');
        const assembly = app.synth();
        const treeArtifact = assembly.tree();
        test.ok(treeArtifact);
        test.deepEqual(readJson(assembly.directory, treeArtifact.file), {
            version: 'tree-0.1',
            tree: {
                id: 'App',
                path: '',
                children: {
                    Tree: {
                        id: 'Tree',
                        path: 'Tree'
                    },
                    mystack: {
                        id: 'mystack',
                        path: 'mystack',
                        children: {
                            myconstruct: {
                                id: 'myconstruct',
                                path: 'mystack/myconstruct'
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    'tree metadata for a Cfn resource'(test) {
        class MyCfnResource extends AbstractCfnResource {
            get cfnProperties() {
                return {
                    mystringpropkey: 'mystringpropval',
                    mylistpropkey: ['listitem1'],
                    mystructpropkey: {
                        myboolpropkey: true,
                        mynumpropkey: 50
                    }
                };
            }
        }
        const app = new index_1.App();
        const stack = new index_1.Stack(app, 'mystack');
        new MyCfnResource(stack, 'mycfnresource');
        const assembly = app.synth();
        const treeArtifact = assembly.tree();
        test.ok(treeArtifact);
        test.deepEqual(readJson(assembly.directory, treeArtifact.file), {
            version: 'tree-0.1',
            tree: {
                id: 'App',
                path: '',
                children: {
                    Tree: {
                        id: 'Tree',
                        path: 'Tree'
                    },
                    mystack: {
                        id: 'mystack',
                        path: 'mystack',
                        children: {
                            mycfnresource: {
                                id: 'mycfnresource',
                                path: 'mystack/mycfnresource',
                                attributes: {
                                    'aws:cdk:cloudformation:type': 'CDK::UnitTest::MyCfnResource',
                                    'aws:cdk:cloudformation:props': {
                                        mystringpropkey: 'mystringpropval',
                                        mylistpropkey: ['listitem1'],
                                        mystructpropkey: {
                                            myboolpropkey: true,
                                            mynumpropkey: 50
                                        }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    'token resolution & cfn parameter'(test) {
        const app = new index_1.App();
        const stack = new index_1.Stack(app, 'mystack');
        const cfnparam = new index_1.CfnParameter(stack, 'mycfnparam');
        class MyCfnResource extends AbstractCfnResource {
            get cfnProperties() {
                return {
                    lazykey: index_1.Lazy.stringValue({ produce: () => 'LazyResolved!' }),
                    cfnparamkey: cfnparam
                };
            }
        }
        new MyCfnResource(stack, 'mycfnresource');
        const assembly = app.synth();
        const treeArtifact = assembly.tree();
        test.ok(treeArtifact);
        test.deepEqual(readJson(assembly.directory, treeArtifact.file), {
            version: 'tree-0.1',
            tree: {
                id: 'App',
                path: '',
                children: {
                    Tree: {
                        id: 'Tree',
                        path: 'Tree'
                    },
                    mystack: {
                        id: 'mystack',
                        path: 'mystack',
                        children: {
                            mycfnparam: {
                                id: 'mycfnparam',
                                path: 'mystack/mycfnparam'
                            },
                            mycfnresource: {
                                id: 'mycfnresource',
                                path: 'mystack/mycfnresource',
                                attributes: {
                                    'aws:cdk:cloudformation:type': 'CDK::UnitTest::MyCfnResource',
                                    'aws:cdk:cloudformation:props': {
                                        lazykey: 'LazyResolved!',
                                        cfnparamkey: { Ref: 'mycfnparam' }
                                    }
                                }
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    'cross-stack tokens'(test) {
        class MyFirstResource extends AbstractCfnResource {
            constructor(scope, id) {
                super(scope, id);
                this.lazykey = index_1.Lazy.stringValue({ produce: () => 'LazyResolved!' });
            }
            get cfnProperties() {
                return {
                    lazykey: this.lazykey
                };
            }
        }
        class MySecondResource extends AbstractCfnResource {
            constructor(scope, id, myprop) {
                super(scope, id);
                this.myprop = myprop;
            }
            get cfnProperties() {
                return {
                    myprop: this.myprop
                };
            }
        }
        const app = new index_1.App();
        const firststack = new index_1.Stack(app, 'myfirststack');
        const firstres = new MyFirstResource(firststack, 'myfirstresource');
        const secondstack = new index_1.Stack(app, 'mysecondstack');
        new MySecondResource(secondstack, 'mysecondresource', firstres.lazykey);
        const assembly = app.synth();
        const treeArtifact = assembly.tree();
        test.ok(treeArtifact);
        test.deepEqual(readJson(assembly.directory, treeArtifact.file), {
            version: 'tree-0.1',
            tree: {
                id: 'App',
                path: '',
                children: {
                    Tree: {
                        id: 'Tree',
                        path: 'Tree'
                    },
                    myfirststack: {
                        id: 'myfirststack',
                        path: 'myfirststack',
                        children: {
                            myfirstresource: {
                                id: 'myfirstresource',
                                path: 'myfirststack/myfirstresource',
                                attributes: {
                                    'aws:cdk:cloudformation:type': 'CDK::UnitTest::MyCfnResource',
                                    'aws:cdk:cloudformation:props': {
                                        lazykey: 'LazyResolved!'
                                    }
                                }
                            }
                        }
                    },
                    mysecondstack: {
                        id: 'mysecondstack',
                        path: 'mysecondstack',
                        children: {
                            mysecondresource: {
                                id: 'mysecondresource',
                                path: 'mysecondstack/mysecondresource',
                                attributes: {
                                    'aws:cdk:cloudformation:type': 'CDK::UnitTest::MyCfnResource',
                                    'aws:cdk:cloudformation:props': {
                                        myprop: 'LazyResolved!'
                                    }
                                }
                            }
                        }
                    }
                }
            }
        });
        test.done();
    },
    'failing nodes'(test) {
        class MyCfnResource extends index_1.CfnResource {
            inspect(_) {
                throw new Error('Forcing an inspect error');
            }
        }
        const app = new index_1.App();
        const stack = new index_1.Stack(app, 'mystack');
        new MyCfnResource(stack, 'mycfnresource', {
            type: 'CDK::UnitTest::MyCfnResource'
        });
        const assembly = app.synth();
        const treeArtifact = assembly.tree();
        test.ok(treeArtifact);
        const treenode = app.node.findChild('Tree');
        const warn = treenode.node.metadata.find((md) => {
            return md.type === cxapi.WARNING_METADATA_KEY
                && /Forcing an inspect error/.test(md.data)
                && /mycfnresource/.test(md.data);
        });
        test.ok(warn);
        // assert that the rest of the construct tree is rendered
        test.deepEqual(readJson(assembly.directory, treeArtifact.file), {
            version: 'tree-0.1',
            tree: {
                id: 'App',
                path: '',
                children: {
                    Tree: {
                        id: 'Tree',
                        path: 'Tree'
                    },
                    mystack: {
                        id: 'mystack',
                        path: 'mystack'
                    }
                }
            }
        });
        test.done();
    },
};
//# sourceMappingURL=data:application/json;base64,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