from tuna_util import *
import re
import ast




def generate_custom_basis(filepath):

    """
    
    Parses the TUNA/ORCA style basis set file and turns it into a machine-readable dictionary.

    Args:
        filepath (str): Filepath to basis set file
    
    Returns:
        basis_dict (dict): Dictionary of basis data by atomic number
    
    """


    try:

        # Opens file path
        with open(filepath, "r") as basis_file:

            text = basis_file.read()
    
    except FileNotFoundError:

        error(f"Basis path \"{filepath}\" not found!")

    except TypeError:

        error("A custom basis was requested but no filepath given via BASIS [filepath.tuna] keyword!")



    lines = text.strip().splitlines()
    basis = {}
    current_Z = None

    # Element map to link the names of elements (in the basis file) with their atomic numbers, via atomic_properties
    element_map = {

        props["name"].upper(): props["charge"] for symbol, props in atomic_properties.items() if symbol != "X"
    
    }

    current_block = []
    orb_type = None
    nlines_expected = 0

    for line in lines:

        line = line.strip()

        # Start with dollar signs to avoid begin read
        if not line or line.startswith("$"): continue

        if line in element_map:

            if current_Z and current_block:

                basis[current_Z] = current_block
                current_block = []

            current_Z = element_map[line]

            continue
        
        # Looks for lines beginning with S, P, D, etc.
        if re.match(r"^[A-Z]\s+\d+", line):

            parts = line.split()
            orb_type = parts[0]
            nlines_expected = int(parts[1])
            data_lines = []

            continue

        if orb_type:

            nums = line.split()

            # Replaces any "D" with scientific notation E
            exp = float(nums[1].replace("D", "E"))
            coeffs = [float(x.replace("D", "E")) for x in nums[2:]]
            data_lines.append((exp, *coeffs))
            nlines_expected -= 1

            if nlines_expected == 0:
                
                # Handles case of "L" orbital type, meaning add to both S and P shells
                if orb_type == "L":

                    s_list = [(exp, c[0]) for exp, *c in data_lines]
                    p_list = [(exp, c[1]) for exp, *c in data_lines]
                    current_block.append(("S", s_list))
                    current_block.append(("P", p_list))

                else:

                    current_block.append((orb_type, [(exp, c) for exp, c in data_lines]))

                orb_type = None

    # Builds basis data for current atomic number
    if current_Z and current_block:

        basis[current_Z] = current_block



    parts = []
    parts.append("{")

    # Build string in correct format
    for Z, blocks in basis.items():

        block_parts = []

        for orb_type, entries in blocks:

            pairs = ", ".join(f"({exp:.10E}, {coef:.10E})" for exp, coef in entries)
            block_parts.append(f"(\"{orb_type}\", [{pairs}])")

        blocks_str = ", ".join(block_parts)
        parts.append(f"{Z}: [{blocks_str}],")

    parts.append("}")

    # Join into one string (no spaces/newlines if you want it compact)
    dict_str = "".join(parts)

    # Convert safely to dictionary
    basis_dict = ast.literal_eval(dict_str)


    return basis_dict










def generate_basis(basis_set, atomic_number, calculation):

    """
    
    Returns the correct atomic basis, given the requested basis set and atomic number.

    Args:
        basis_set (str): Requested basis set
        atomic_number (int): Atomic number

    Returns:
        basis_dict (dict): Atomic number as the key of the dictionary, the requested basis as the value 

    """

    # Converts user basis set from human-readable to more computer-readable form, replacing *, + and - to STAR, PLUS and _ 
    bas = basis_set.upper().replace("-", "_").replace("*", "STAR").replace("+", "PLUS").replace("[", "BRA").replace(",", "COMMA").replace("]", "KET")

    # If a custom basis set is requested, read the provided file
    if bas == "CUSTOM": 
        
        basis_dict = generate_custom_basis(calculation.custom_basis_file)

    else:

        # If the basis set starts with a number, prepend an underscore
        if bas[0].isdigit(): bas = "_" + bas

        # Find the correct basis set in this file
        d = globals().get(bas)

        # Find the correct atomic number within that basis set
        atomic_basis = d.get(atomic_number, None)

        # If no atomic basis can be found, print an error that says the chosen basis set is not parameterised for the chosen atom
        if atomic_basis == None:

            broken_atom = next((atom for atom, props in atomic_properties.items() if props.get("charge") == atomic_number), None)

            error(f"The chosen basis set, {basis_set}, is not parameterised for {broken_atom.lower().capitalize()}!")

        basis_dict = {atomic_number : atomic_basis}


    return basis_dict






STO_3G = {
    
     1 : [("S", [(0.3425250914E+01, 0.1543289673E+00), (0.6239137298E+00, 0.5353281423E+00), (0.1688554040E+00, 0.4446345422E+00)])],
     2 : [("S", [(0.6362421394E+01, 0.1543289673E+00), (0.1158922999E+01, 0.5353281423E+00), (0.3136497915E+00, 0.4446345422E+00)])],
     3 : [("S", [(0.1611957475E+02, 0.1543289673E+00), (0.2936200663E+01, 0.5353281423E+00), (0.7946504870E+00, 0.4446345422E+00)]), ("S", [(0.6362897469E+00, -0.9996722919E-01), (0.1478600533E+00, 0.3995128261E+00), (0.4808867840E-01, 0.7001154689E+00)]), ("P", [(0.6362897469E+00, 0.1559162750E+00), (0.1478600533E+00, 0.6076837186E+00), (0.4808867840E-01, 0.3919573931E+00)])],
     4 : [("S", [(0.3016787069E+02, 0.1543289673E+00), (0.5495115306E+01, 0.5353281423E+00), (0.1487192653E+01, 0.4446345422E+00)]), ("S", [(0.1314833110E+01, -0.9996722919E-01), (0.3055389383E+00, 0.3995128261E+00), (0.9937074560E-01, 0.7001154689E+00)]), ("P", [(0.1314833110E+01, 0.1559162750E+00), (0.3055389383E+00, 0.6076837186E+00), (0.9937074560E-01, 0.3919573931E+00)])],
     5 : [("S", [(0.4879111318E+02, 0.1543289673E+00), (0.8887362172E+01, 0.5353281423E+00), (0.2405267040E+01, 0.4446345422E+00)]), ("S", [(0.2236956142E+01, -0.9996722919E-01), (0.5198204999E+00, 0.3995128261E+00), (0.1690617600E+00, 0.7001154689E+00)]), ("P", [(0.2236956142E+01, 0.1559162750E+00), (0.5198204999E+00, 0.6076837186E+00), (0.1690617600E+00, 0.3919573931E+00)])],
     6 : [("S", [(0.7161683735E+02, 0.1543289673E+00), (0.1304509632E+02, 0.5353281423E+00), (0.3530512160E+01, 0.4446345422E+00)]), ("S", [(0.2941249355E+01, -0.9996722919E-01), (0.6834830964E+00, 0.3995128261E+00), (0.2222899159E+00, 0.7001154689E+00)]), ("P", [(0.2941249355E+01, 0.1559162750E+00), (0.6834830964E+00, 0.6076837186E+00), (0.2222899159E+00, 0.3919573931E+00)])],
     7 : [("S", [(0.9910616896E+02, 0.1543289673E+00), (0.1805231239E+02, 0.5353281423E+00), (0.4885660238E+01, 0.4446345422E+00)]), ("S", [(0.3780455879E+01, -0.9996722919E-01), (0.8784966449E+00, 0.3995128261E+00), (0.2857143744E+00, 0.7001154689E+00)]), ("P", [(0.3780455879E+01, 0.1559162750E+00), (0.8784966449E+00, 0.6076837186E+00), (0.2857143744E+00, 0.3919573931E+00)])],
     8 : [("S", [(0.1307093214E+03, 0.1543289673E+00), (0.2380886605E+02, 0.5353281423E+00), (0.6443608313E+01, 0.4446345422E+00)]), ("S", [(0.5033151319E+01, -0.9996722919E-01), (0.1169596125E+01, 0.3995128261E+00), (0.3803889600E+00, 0.7001154689E+00)]), ("P", [(0.5033151319E+01, 0.1559162750E+00), (0.1169596125E+01, 0.6076837186E+00), (0.3803889600E+00, 0.3919573931E+00)])],
     9 : [("S", [(0.1666791340E+03, 0.1543289673E+00), (0.3036081233E+02, 0.5353281423E+00), (0.8216820672E+01, 0.4446345422E+00)]), ("S", [(0.6464803249E+01, -0.9996722919E-01), (0.1502281245E+01, 0.3995128261E+00), (0.4885884864E+00, 0.7001154689E+00)]), ("P", [(0.6464803249E+01, 0.1559162750E+00), (0.1502281245E+01, 0.6076837186E+00), (0.4885884864E+00, 0.3919573931E+00)])],
    10 : [("S", [(0.2070156070E+03, 0.1543289673E+00), (0.3770815124E+02, 0.5353281423E+00), (0.1020529731E+02, 0.4446345422E+00)]), ("S", [(0.8246315120E+01, -0.9996722919E-01), (0.1916266291E+01, 0.3995128261E+00), (0.6232292721E+00, 0.7001154689E+00)]), ("P", [(0.8246315120E+01, 0.1559162750E+00), (0.1916266291E+01, 0.6076837186E+00), (0.6232292721E+00, 0.3919573931E+00)])],
    11 : [("S", [(0.2507724300E+03, 0.1543289673E+00), (0.4567851117E+02, 0.5353281423E+00), (0.1236238776E+02, 0.4446345422E+00)]), ("S", [(0.1204019274E+02, -0.9996722919E-01), (0.2797881859E+01, 0.3995128261E+00), (0.9099580170E+00, 0.7001154689E+00)]), ("P", [(0.1204019274E+02, 0.1559162750E+00), (0.2797881859E+01, 0.6076837186E+00), (0.9099580170E+00, 0.3919573931E+00)]), ("S", [(0.1478740622E+01, -0.2196203690E+00), (0.4125648801E+00, 0.2255954336E+00), (0.1614750979E+00, 0.9003984260E+00)]), ("P", [(0.1478740622E+01, 0.1058760429E-01), (0.4125648801E+00, 0.5951670053E+00), (0.1614750979E+00, 0.4620010120E+00)])],
    12 : [("S", [(2.9923741370E+02, 1.5432896730E-01), (5.4506468450E+01, 5.3532814230E-01), (1.4751577520E+01, 4.4463454220E-01)]), ("S", [(1.5121823520E+01, -9.9967229190E-02), (3.5139865790E+00, 3.9951282610E-01), (1.1428574980E+00, 7.0011546890E-01)]), ("P", [(1.5121823520E+01, 1.5591627500E-01), (3.5139865790E+00, 6.0768371860E-01), (1.1428574980E+00, 3.9195739310E-01)]), ("S", [(1.3954482930E+00, -2.1962036900E-01), (3.8932653180E-01, 2.2559543360E-01), (1.5237976590E-01, 9.0039842600E-01)]), ("P", [(1.3954482930E+00, 1.0587604290E-02), (3.8932653180E-01, 5.9516700530E-01), (1.5237976590E-01, 4.6200101200E-01)])],
    13 : [("S", [(3.5142147670E+02, 1.5432896730E-01), (6.4011860670E+01, 5.3532814230E-01), (1.7324107610E+01, 4.4463454220E-01)]), ("S", [(1.8899396210E+01, -9.9967229190E-02), (4.3918132330E+00, 3.9951282610E-01), (1.4283539700E+00, 7.0011546890E-01)]), ("P", [(1.8899396210E+01, 1.5591627500E-01), (4.3918132330E+00, 6.0768371860E-01), (1.4283539700E+00, 3.9195739310E-01)]), ("S", [(1.3954482930E+00, -2.1962036900E-01), (3.8932653180E-01, 2.2559543360E-01), (1.5237976590E-01, 9.0039842600E-01)]), ("P", [(1.3954482930E+00, 1.0587604290E-02), (3.8932653180E-01, 5.9516700530E-01), (1.5237976590E-01, 4.6200101200E-01)])],
    14 : [("S", [(4.0779755140E+02, 1.5432896730E-01), (7.4280833050E+01, 5.3532814230E-01), (2.0103292290E+01, 4.4463454220E-01)]), ("S", [(2.3193656060E+01, -9.9967229190E-02), (5.3897068710E+00, 3.9951282610E-01), (1.7528999520E+00, 7.0011546890E-01)]), ("P", [(2.3193656060E+01, 1.5591627500E-01), (5.3897068710E+00, 6.0768371860E-01), (1.7528999520E+00, 3.9195739310E-01)]), ("S", [(1.4787406220E+00, -2.1962036900E-01), (4.1256488010E-01, 2.2559543360E-01), (1.6147509790E-01, 9.0039842600E-01)]), ("P", [(1.4787406220E+00, 1.0587604290E-02), (4.1256488010E-01, 5.9516700530E-01), (1.6147509790E-01, 4.6200101200E-01)])],
    15 : [("S", [(4.6836563780E+02, 1.5432896730E-01), (8.5313385590E+01, 5.3532814230E-01), (2.3089131560E+01, 4.4463454220E-01)]), ("S", [(2.8032639580E+01, -9.9967229190E-02), (6.5141825770E+00, 3.9951282610E-01), (2.1186143520E+00, 7.0011546890E-01)]), ("P", [(2.8032639580E+01, 1.5591627500E-01), (6.5141825770E+00, 6.0768371860E-01), (2.1186143520E+00, 3.9195739310E-01)]), ("S", [(1.7431032310E+00, -2.1962036900E-01), (4.8632137710E-01, 2.2559543360E-01), (1.9034289090E-01, 9.0039842600E-01)]), ("P", [(1.7431032310E+00, 1.0587604290E-02), (4.8632137710E-01, 5.9516700530E-01), (1.9034289090E-01, 4.6200101200E-01)])],
    16 : [("S", [(5.3312573590E+02, 1.5432896730E-01), (9.7109518300E+01, 5.3532814230E-01), (2.6281625420E+01, 4.4463454220E-01)]), ("S", [(3.3329751730E+01, -9.9967229190E-02), (7.7451175210E+00, 3.9951282610E-01), (2.5189525990E+00, 7.0011546890E-01)]), ("P", [(3.3329751730E+01, 1.5591627500E-01), (7.7451175210E+00, 6.0768371860E-01), (2.5189525990E+00, 3.9195739310E-01)]), ("S", [(2.0291942740E+00, -2.1962036900E-01), (5.6614005180E-01, 2.2559543360E-01), (2.2158337920E-01, 9.0039842600E-01)]), ("P", [(2.0291942740E+00, 1.0587604290E-02), (5.6614005180E-01, 5.9516700530E-01), (2.2158337920E-01, 4.6200101200E-01)])],
    17 : [("S", [(6.0134561360E+02, 1.5432896730E-01), (1.0953585420E+02, 5.3532814230E-01), (2.9644676860E+01, 4.4463454220E-01)]), ("S", [(3.8960418890E+01, -9.9967229190E-02), (9.0535634770E+00, 3.9951282610E-01), (2.9444998340E+00, 7.0011546890E-01)]), ("P", [(3.8960418890E+01, 1.5591627500E-01), (9.0535634770E+00, 6.0768371860E-01), (2.9444998340E+00, 3.9195739310E-01)]), ("S", [(2.1293864950E+00, -2.1962036900E-01), (5.9409342740E-01, 2.2559543360E-01), (2.3252414100E-01, 9.0039842600E-01)]), ("P", [(2.1293864950E+00, 1.0587604290E-02), (5.9409342740E-01, 5.9516700530E-01), (2.3252414100E-01, 4.6200101200E-01)])],
    18 : [("S", [(6.7444651840E+02, 1.5432896730E-01), (1.2285127530E+02, 5.3532814230E-01), (3.3248349450E+01, 4.4463454220E-01)]), ("S", [(4.5164243920E+01, -9.9967229190E-02), (1.0495199000E+01, 3.9951282610E-01), (3.4133644480E+00, 7.0011546890E-01)]), ("P", [(4.5164243920E+01, 1.5591627500E-01), (1.0495199000E+01, 6.0768371860E-01), (3.4133644480E+00, 3.9195739310E-01)]), ("S", [(2.6213665180E+00, -2.1962036900E-01), (7.3135460500E-01, 2.2559543360E-01), (2.8624723560E-01, 9.0039842600E-01)]), ("P", [(2.6213665180E+00, 1.0587604290E-02), (7.3135460500E-01, 5.9516700530E-01), (2.8624723560E-01, 4.6200101200E-01)])],

    
    }







STO_6G = {

    1 : [("S", [(3.5523221220E+01, 9.1635962810E-03), (6.5131437250E+00, 4.9361492940E-02), (1.8221429040E+00, 1.6853830490E-01), (6.2595526590E-01, 3.7056279970E-01), (2.4307674710E-01, 4.1649152980E-01), (1.0011242800E-01, 1.3033408410E-01)])],
    2 : [("S", [(6.5984568240E+01, 9.1635962810E-03), (1.2098198360E+01, 4.9361492940E-02), (3.3846399240E+00, 1.6853830490E-01), (1.1627151630E+00, 3.7056279970E-01), (4.5151632240E-01, 4.1649152980E-01), (1.8595935590E-01, 1.3033408410E-01)])],
    3 : [("S", [(1.6717584620E+02, 9.1635962810E-03), (3.0651508400E+01, 4.9361492940E-02), (8.5751874770E+00, 1.6853830490E-01), (2.9458083370E+00, 3.7056279970E-01), (1.1439435810E+00, 4.1649152980E-01), (4.7113913910E-01, 1.3033408410E-01)]), ("S", [(6.5975639810E+00, -1.3252788090E-02), (1.3058300920E+00, -4.6991710140E-02), (4.0585101930E-01, -3.3785371510E-02), (1.5614551580E-01, 2.5024178610E-01), (6.7814103940E-02, 5.9511725260E-01), (3.1084165500E-02, 2.4070617630E-01)]), ("P", [(6.5975639810E+00, 3.7596966230E-03), (1.3058300920E+00, 3.7679369840E-02), (4.0585101930E-01, 1.7389674350E-01), (1.5614551580E-01, 4.1803643470E-01), (6.7814103940E-02, 4.2585954770E-01), (3.1084165500E-02, 1.0170829550E-01)])],
    4 : [("S", [(3.1287049370E+02, 9.1635962810E-03), (5.7364462530E+01, 4.9361492940E-02), (1.6048509400E+01, 1.6853830490E-01), (5.5130961190E+00, 3.7056279970E-01), (2.1408965530E+00, 4.1649152980E-01), (8.8173942830E-01, 1.3033408410E-01)]), ("S", [(1.3633247440E+01, -1.3252788090E-02), (2.6983754640E+00, -4.6991710140E-02), (8.3865308290E-01, -3.3785371510E-02), (3.2266006980E-01, 2.5024178610E-01), (1.4013148820E-01, 5.9511725260E-01), (6.4232513870E-02, 2.4070617630E-01)]), ("P", [(1.3633247440E+01, 3.7596966230E-03), (2.6983754640E+00, 3.7679369840E-02), (8.3865308290E-01, 1.7389674350E-01), (3.2266006980E-01, 4.1803643470E-01), (1.4013148820E-01, 4.2585954770E-01), (6.4232513870E-02, 1.0170829550E-01)])],
    5 : [("S", [(5.0601183690E+02, 9.1635962810E-03), (9.2776716390E+01, 4.9361492940E-02), (2.5955581900E+01, 1.6853830490E-01), (8.9164429080E+00, 3.7056279970E-01), (3.4625157030E+00, 4.1649152980E-01), (1.4260551790E+00, 1.3033408410E-01)]), ("S", [(2.3194560870E+01, -1.3252788090E-02), (4.5908089180E+00, -4.6991710140E-02), (1.4268199900E+00, -3.3785371510E-02), (5.4894907910E-01, 2.5024178610E-01), (2.3840895910E-01, 5.9511725260E-01), (1.0928026940E-01, 2.4070617630E-01)]), ("P", [(2.3194560870E+01, 3.7596966230E-03), (4.5908089180E+00, 3.7679369840E-02), (1.4268199900E+00, 1.7389674350E-01), (5.4894907910E-01, 4.1803643470E-01), (2.3840895910E-01, 4.2585954770E-01), (1.0928026940E-01, 1.0170829550E-01)])],
    6 : [("S", [(7.4273704910E+02, 9.1635962810E-03), (1.3618002490E+02, 4.9361492940E-02), (3.8098263520E+01, 1.6853830490E-01), (1.3087781770E+01, 3.7056279970E-01), (5.0823686480E+00, 4.1649152980E-01), (2.0932000760E+00, 1.3033408410E-01)]), ("S", [(3.0497239500E+01, -1.3252788090E-02), (6.0361996010E+00, -4.6991710140E-02), (1.8760463370E+00, -3.3785371510E-02), (7.2178264700E-01, 2.5024178610E-01), (3.1347069540E-01, 5.9511725260E-01), (1.4368655500E-01, 2.4070617630E-01)]), ("P", [(3.0497239500E+01, 3.7596966230E-03), (6.0361996010E+00, 3.7679369840E-02), (1.8760463370E+00, 1.7389674350E-01), (7.2178264700E-01, 4.1803643470E-01), (3.1347069540E-01, 4.2585954770E-01), (1.4368655500E-01, 1.0170829550E-01)])],
    7 : [("S", [(1.0278284580E+03, 9.1635962810E-03), (1.8845122260E+02, 4.9361492940E-02), (5.2721860970E+01, 1.6853830490E-01), (1.8111382170E+01, 3.7056279970E-01), (7.0331796910E+00, 4.1649152980E-01), (2.8966517940E+00, 1.3033408410E-01)]), ("S", [(3.9198807870E+01, -1.3252788090E-02), (7.7584670710E+00, -4.6991710140E-02), (2.4113257830E+00, -3.3785371510E-02), (9.2772394370E-01, 2.5024178610E-01), (4.0291114100E-01, 5.9511725260E-01), (1.8468365520E-01, 2.4070617630E-01)]), ("P", [(3.9198807870E+01, 3.7596966230E-03), (7.7584670710E+00, 3.7679369840E-02), (2.4113257830E+00, 1.7389674350E-01), (9.2772394370E-01, 4.1803643470E-01), (4.0291114100E-01, 4.2585954770E-01), (1.8468365520E-01, 1.0170829550E-01)])],
    8 : [("S", [(1.3555842340E+03, 9.1635962810E-03), (2.4854488550E+02, 4.9361492940E-02), (6.9533902290E+01, 1.6853830490E-01), (2.3886772110E+01, 3.7056279970E-01), (9.2759326090E+00, 4.1649152980E-01), (3.8203412980E+00, 1.3033408410E-01)]), ("S", [(5.2187761960E+01, -1.3252788090E-02), (1.0329320060E+01, -4.6991710140E-02), (3.2103449770E+00, -3.3785371510E-02), (1.2351354280E+00, 2.5024178610E-01), (5.3642015810E-01, 5.9511725260E-01), (2.4588060600E-01, 2.4070617630E-01)]), ("P", [(5.2187761960E+01, 3.7596966230E-03), (1.0329320060E+01, 3.7679369840E-02), (3.2103449770E+00, 1.7389674350E-01), (1.2351354280E+00, 4.1803643470E-01), (5.3642015810E-01, 4.2585954770E-01), (2.4588060600E-01, 1.0170829550E-01)])],
    9 : [("S", [(1.7286265740E+03, 9.1635962810E-03), (3.1694179000E+02, 4.9361492940E-02), (8.8668891390E+01, 1.6853830490E-01), (3.0460157310E+01, 3.7056279970E-01), (1.1828570440E+01, 4.1649152980E-01), (4.8716585220E+00, 1.3033408410E-01)]), ("S", [(6.7032280910E+01, -1.3252788090E-02), (1.3267437770E+01, -4.6991710140E-02), (4.1235097710E+00, -3.3785371510E-02), (1.5864628390E+00, 2.5024178610E-01), (6.8900189190E-01, 5.9511725260E-01), (3.1581997840E-01, 2.4070617630E-01)]), ("P", [(6.7032280910E+01, 3.7596966230E-03), (1.3267437770E+01, 3.7679369840E-02), (4.1235097710E+00, 1.7389674350E-01), (1.5864628390E+00, 4.1803643470E-01), (6.8900189190E-01, 4.2585954770E-01), (3.1581997840E-01, 1.0170829550E-01)])],
    10 : [("S", [(2.1469554750E+03, 9.1635962810E-03), (3.9364193620E+02, 4.9361492940E-02), (1.1012682830E+02, 1.6853830490E-01), (3.7831537770E+01, 3.7056279970E-01), (1.4691093180E+01, 4.1649152980E-01), (6.0506034660E+00, 1.3033408410E-01)]), ("S", [(8.5504429190E+01, -1.3252788090E-02), (1.6923557990E+01, -4.6991710140E-02), (5.2598292100E+00, -3.3785371510E-02), (2.0236458850E+00, 2.5024178610E-01), (8.7887078700E-01, 5.9511725260E-01), (4.0285078490E-01, 2.4070617630E-01)]), ("P", [(8.5504429190E+01, 3.7596966230E-03), (1.6923557990E+01, 3.7679369840E-02), (5.2598292100E+00, 1.7389674350E-01), (2.0236458850E+00, 4.1803643470E-01), (8.7887078700E-01, 4.2585954770E-01), (4.0285078490E-01, 1.0170829550E-01)])],
    11 : [("S", [(2.6007567710E+03, 9.1635962810E-03), (4.7684590710E+02, 4.9361492940E-02), (1.3340430100E+02, 1.6853830490E-01), (4.5827977880E+01, 3.7056279970E-01), (1.7796344870E+01, 4.1649152980E-01), (7.3295175960E+00, 1.3033408410E-01)]), ("S", [(1.2484240440E+02, -1.3252788090E-02), (2.4709569920E+01, -4.6991710140E-02), (7.6797159130E+00, -3.3785371510E-02), (2.9546635230E+00, 2.5024178610E-01), (1.2832123820E+00, 5.9511725260E-01), (5.8819012170E-01, 2.4070617630E-01)]), ("P", [(1.2484240440E+02, 3.7596966230E-03), (2.4709569920E+01, 3.7679369840E-02), (7.6797159130E+00, 1.7389674350E-01), (2.9546635230E+00, 4.1803643470E-01), (1.2832123820E+00, 4.2585954770E-01), (5.8819012170E-01, 1.0170829550E-01)]), ("S", [(9.4330060470E+00, -7.9431263620E-03), (2.5262437560E+00, -7.1002641720E-02), (9.4736825060E-01, -1.7850269250E-01), (4.2405944350E-01, 1.5106350580E-01), (2.0984540790E-01, 7.3549147670E-01), (1.0814709430E-01, 2.7605931230E-01)]), ("P", [(9.4330060470E+00, -7.1393589070E-03), (2.5262437560E+00, -1.8292770700E-02), (9.4736825060E-01, 7.6216214290E-02), (4.2405944350E-01, 4.1450985970E-01), (2.0984540790E-01, 4.8896214710E-01), (1.0814709430E-01, 1.0588165210E-01)])],
    12 : [("S", [(3.1033863240E+03, 9.1635962810E-03), (5.6900248540E+02, 4.9361492940E-02), (1.5918639060E+02, 1.6853830490E-01), (5.4684821510E+01, 3.7056279970E-01), (2.1235716430E+01, 4.1649152980E-01), (8.7460407380E+00, 1.3033408410E-01)]), ("S", [(1.5679523150E+02, -1.3252788090E-02), (3.1033868280E+01, -4.6991710140E-02), (9.6453031310E+00, -3.3785371510E-02), (3.7108957750E+00, 2.5024178610E-01), (1.6116445640E+00, 5.9511725260E-01), (7.3873462080E-01, 2.4070617630E-01)]), ("P", [(1.5679523150E+02, 3.7596966230E-03), (3.1033868280E+01, 3.7679369840E-02), (9.6453031310E+00, 1.7389674350E-01), (3.7108957750E+00, 4.1803643470E-01), (1.6116445640E+00, 4.2585954770E-01), (7.3873462080E-01, 1.0170829550E-01)]), ("S", [(8.9016775440E+00, -7.9431263620E-03), (2.3839492090E+00, -7.1002641720E-02), (8.9400628390E-01, -1.7850269250E-01), (4.0017364620E-01, 1.5106350580E-01), (1.9802554410E-01, 7.3549147670E-01), (1.0205554360E-01, 2.7605931230E-01)]), ("P", [(8.9016775440E+00, -7.1393589070E-03), (2.3839492090E+00, -1.8292770700E-02), (8.9400628390E-01, 7.6216214290E-02), (4.0017364620E-01, 4.1450985970E-01), (1.9802554410E-01, 4.8896214710E-01), (1.0205554360E-01, 1.0588165210E-01)])],
    13 : [("S", [(3.6445863880E+03, 9.1635962810E-03), (6.6823092480E+02, 4.9361492940E-02), (1.8694693210E+02, 1.6853830490E-01), (6.4221316750E+01, 3.7056279970E-01), (2.4939016600E+01, 4.1649152980E-01), (1.0271264260E+01, 1.3033408410E-01)]), ("S", [(1.9596414410E+02, -1.3252788090E-02), (3.8786418310E+01, -4.6991710140E-02), (1.2054789900E+01, -3.3785371510E-02), (4.6379121840E+00, 2.5024178610E-01), (2.0142484220E+00, 5.9511725260E-01), (9.2327742590E-01, 2.4070617630E-01)]), ("P", [(1.9596414410E+02, 3.7596966230E-03), (3.8786418310E+01, 3.7679369840E-02), (1.2054789900E+01, 1.7389674350E-01), (4.6379121840E+00, 4.1803643470E-01), (2.0142484220E+00, 4.2585954770E-01), (9.2327742590E-01, 1.0170829550E-01)]), ("S", [(8.9016775440E+00, -7.9431263620E-03), (2.3839492090E+00, -7.1002641720E-02), (8.9400628390E-01, -1.7850269250E-01), (4.0017364620E-01, 1.5106350580E-01), (1.9802554410E-01, 7.3549147670E-01), (1.0205554360E-01, 2.7605931230E-01)]), ("P", [(8.9016775440E+00, -7.1393589070E-03), (2.3839492090E+00, -1.8292770700E-02), (8.9400628390E-01, 7.6216214290E-02), (4.0017364620E-01, 4.1450985970E-01), (1.9802554410E-01, 4.8896214710E-01), (1.0205554360E-01, 1.0588165210E-01)])],
    14 : [("S", [(4.2292617370E+03, 9.1635962810E-03), (7.7543051000E+02, 4.9361492940E-02), (2.1693751290E+02, 1.6853830490E-01), (7.4523890690E+01, 3.7056279970E-01), (2.8939807550E+01, 4.1649152980E-01), (1.1919010910E+01, 1.3033408410E-01)]), ("S", [(2.4049048490E+02, -1.3252788090E-02), (4.7599343180E+01, -4.6991710140E-02), (1.4793840380E+01, -3.3785371510E-02), (5.6917236320E+00, 2.5024178610E-01), (2.4719194520E+00, 5.9511725260E-01), (1.1330615450E+00, 2.4070617630E-01)]), ("P", [(2.4049048490E+02, 3.7596966230E-03), (4.7599343180E+01, 3.7679369840E-02), (1.4793840380E+01, 1.7389674350E-01), (5.6917236320E+00, 4.1803643470E-01), (2.4719194520E+00, 4.2585954770E-01), (1.1330615450E+00, 1.0170829550E-01)]), ("S", [(9.4330060470E+00, -7.9431263620E-03), (2.5262437560E+00, -7.1002641720E-02), (9.4736825060E-01, -1.7850269250E-01), (4.2405944350E-01, 1.5106350580E-01), (2.0984540790E-01, 7.3549147670E-01), (1.0814709430E-01, 2.7605931230E-01)]), ("P", [(9.4330060470E+00, -7.1393589070E-03), (2.5262437560E+00, -1.8292770700E-02), (9.4736825060E-01, 7.6216214290E-02), (4.2405944350E-01, 4.1450985970E-01), (2.0984540790E-01, 4.8896214710E-01), (1.0814709430E-01, 1.0588165210E-01)])],
    15 : [("S", [(4.8574123710E+03, 9.1635962810E-03), (8.9060124100E+02, 4.9361492940E-02), (2.4915813310E+02, 1.6853830490E-01), (8.5592543350E+01, 3.7056279970E-01), (3.3238089270E+01, 4.1649152980E-01), (1.3689280690E+01, 1.3033408410E-01)]), ("S", [(2.9066495900E+02, -1.3252788090E-02), (5.7530181030E+01, -4.6991710140E-02), (1.7880337380E+01, -3.3785371510E-02), (6.8792102800E+00, 2.5024178610E-01), (2.9876457120E+00, 5.9511725260E-01), (1.3694566230E+00, 2.4070617630E-01)]), ("P", [(2.9066495900E+02, 3.7596966230E-03), (5.7530181030E+01, 3.7679369840E-02), (1.7880337380E+01, 1.7389674350E-01), (6.8792102800E+00, 4.1803643470E-01), (2.9876457120E+00, 4.2585954770E-01), (1.3694566230E+00, 1.0170829550E-01)]), ("S", [(1.1119396520E+01, -7.9431263620E-03), (2.9778742720E+00, -7.1002641720E-02), (1.1167344930E+00, -1.7850269250E-01), (4.9987088680E-01, 1.5106350580E-01), (2.4736062770E-01, 7.3549147670E-01), (1.2748114620E-01, 2.7605931230E-01)]), ("P", [(1.1119396520E+01, -7.1393589070E-03), (2.9778742720E+00, -1.8292770700E-02), (1.1167344930E+00, 7.6216214290E-02), (4.9987088680E-01, 4.1450985970E-01), (2.4736062770E-01, 4.8896214710E-01), (1.2748114620E-01, 1.0588165210E-01)])],
    16 : [("S", [(5.5290382890E+03, 9.1635962810E-03), (1.0137431180E+03, 4.9361492940E-02), (2.8360879270E+02, 1.6853830490E-01), (9.7427274710E+01, 3.7056279970E-01), (3.7833861780E+01, 4.1649152980E-01), (1.5582073600E+01, 1.3033408410E-01)]), ("S", [(3.4558967910E+02, -1.3252788090E-02), (6.8401216550E+01, -4.6991710140E-02), (2.1259047120E+01, -3.3785371510E-02), (8.1791216990E+00, 2.5024178610E-01), (3.5521981280E+00, 5.9511725260E-01), (1.6282323010E+00, 2.4070617630E-01)]), ("P", [(3.4558967910E+02, 3.7596966230E-03), (6.8401216550E+01, 3.7679369840E-02), (2.1259047120E+01, 1.7389674350E-01), (8.1791216990E+00, 4.1803643470E-01), (3.5521981280E+00, 4.2585954770E-01), (1.6282323010E+00, 1.0170829550E-01)]), ("S", [(1.2944394420E+01, -7.9431263620E-03), (3.4666251050E+00, -7.1002641720E-02), (1.3000212480E+00, -1.7850269250E-01), (5.8191340770E-01, 1.5106350580E-01), (2.8795929030E-01, 7.3549147670E-01), (1.4840429830E-01, 2.7605931230E-01)]), ("P", [(1.2944394420E+01, -7.1393589070E-03), (3.4666251050E+00, -1.8292770700E-02), (1.3000212480E+00, 7.6216214290E-02), (5.8191340770E-01, 4.1450985970E-01), (2.8795929030E-01, 4.8896214710E-01), (1.4840429830E-01, 1.0588165210E-01)])],
    17 : [("S", [(6.2365455250E+03, 9.1635962810E-03), (1.1434637950E+03, 4.9361492940E-02), (3.1989996350E+02, 1.6853830490E-01), (1.0989427140E+02, 3.7056279970E-01), (4.2675161410E+01, 4.1649152980E-01), (1.7575988140E+01, 1.3033408410E-01)]), ("S", [(4.0397296600E+02, -1.3252788090E-02), (7.9956792690E+01, -4.6991710140E-02), (2.4850511570E+01, -3.3785371510E-02), (9.5608875260E+00, 2.5024178610E-01), (4.1522999680E+00, 5.9511725260E-01), (1.9033028810E+00, 2.4070617630E-01)]), ("P", [(4.0397296600E+02, 3.7596966230E-03), (7.9956792690E+01, 3.7679369840E-02), (2.4850511570E+01, 1.7389674350E-01), (9.5608875260E+00, 4.1803643470E-01), (4.1522999680E+00, 4.2585954770E-01), (1.9033028810E+00, 1.0170829550E-01)]), ("S", [(1.3583528710E+01, -7.9431263620E-03), (3.6377910080E+00, -7.1002641720E-02), (1.3642102810E+00, -1.7850269250E-01), (6.1064559860E-01, 1.5106350580E-01), (3.0217738730E-01, 7.3549147670E-01), (1.5573181570E-01, 2.7605931230E-01)]), ("P", [(1.3583528710E+01, -7.1393589070E-03), (3.6377910080E+00, -1.8292770700E-02), (1.3642102810E+00, 7.6216214290E-02), (6.1064559860E-01, 4.1450985970E-01), (3.0217738730E-01, 4.8896214710E-01), (1.5573181570E-01, 1.0588165210E-01)])],
    18 : [("S", [(6.9946738140E+03, 9.1635962810E-03), (1.2824657870E+03, 4.9361492940E-02), (3.5878771170E+02, 1.6853830490E-01), (1.2325326240E+02, 3.7056279970E-01), (4.7862848560E+01, 4.1649152980E-01), (1.9712564190E+01, 1.3033408410E-01)]), ("S", [(4.6829921480E+02, -1.3252788090E-02), (9.2688636090E+01, -4.6991710140E-02), (2.8807559010E+01, -3.3785371510E-02), (1.1083306310E+01, 2.5024178610E-01), (4.8134874810E+00, 5.9511725260E-01), (2.2063734950E+00, 2.4070617630E-01)]), ("P", [(4.6829921480E+02, 3.7596966230E-03), (9.2688636090E+01, 3.7679369840E-02), (2.8807559010E+01, 1.7389674350E-01), (1.1083306310E+01, 4.1803643470E-01), (4.8134874810E+00, 4.2585954770E-01), (2.2063734950E+00, 1.0170829550E-01)]), ("S", [(1.6721909070E+01, -7.9431263620E-03), (4.4782774610E+00, -7.1002641720E-02), (1.6794016310E+00, -1.7850269250E-01), (7.5173104080E-01, 1.5106350580E-01), (3.7199338280E-01, 7.3549147670E-01), (1.9171257470E-01, 2.7605931230E-01)]), ("P", [(1.6721909070E+01, -7.1393589070E-03), (4.4782774610E+00, -1.8292770700E-02), (1.6794016310E+00, 7.6216214290E-02), (7.5173104080E-01, 4.1450985970E-01), (3.7199338280E-01, 4.8896214710E-01), (1.9171257470E-01, 1.0588165210E-01)])],

    }







_3_21G = {


    1 : [("S", [(5.4471780000E+00, 1.5628497870E-01), (8.2454724000E-01, 9.0469087670E-01)]), ("S", [(1.8319158000E-01, 1.0000000000E+00)])],
    2 : [("S", [(1.3626700000E+01, 1.7522987180E-01), (1.9993500000E+00, 8.9348234650E-01)]), ("S", [(3.8299300000E-01, 1.0000000000E+00)])],
    3 : [("S", [(3.6838200000E+01, 6.9668663810E-02), (5.4817200000E+00, 3.8134634930E-01), (1.1132700000E+00, 6.8170262440E-01)]), ("S", [(5.4020500000E-01, -2.6312640580E-01), (1.0225500000E-01, 1.1433874180E+00)]), ("P", [(5.4020500000E-01, 1.6154597080E-01), (1.0225500000E-01, 9.1566283470E-01)]), ("S", [(2.8564500000E-02, 1.0000000000E+00)]), ("P", [(2.8564500000E-02, 1.0000000000E+00)])],
    4 : [("S", [(7.1887600000E+01, 6.4426309750E-02), (1.0728900000E+01, 3.6609605540E-01), (2.2220500000E+00, 6.9593410530E-01)]), ("S", [(1.2954800000E+00, -4.2106406590E-01), (2.6888100000E-01, 1.2240701920E+00)]), ("P", [(1.2954800000E+00, 2.0513192370E-01), (2.6888100000E-01, 8.8252767190E-01)]), ("S", [(7.7350100000E-02, 1.0000000000E+00)]), ("P", [(7.7350100000E-02, 1.0000000000E+00)])],
    5 : [("S", [(1.1643400000E+02, 6.2960465890E-02), (1.7431400000E+01, 3.6330380320E-01), (3.6801600000E+00, 6.9725462230E-01)]), ("S", [(2.2818700000E+00, -3.6866347730E-01), (4.6524800000E-01, 1.1994448060E+00)]), ("P", [(2.2818700000E+00, 2.3115190230E-01), (4.6524800000E-01, 8.6676363370E-01)]), ("S", [(1.2432800000E-01, 1.0000000000E+00)]), ("P", [(1.2432800000E-01, 1.0000000000E+00)])],
    6 : [("S", [(1.7225600000E+02, 6.1766907380E-02), (2.5910900000E+01, 3.5879404290E-01), (5.5333500000E+00, 7.0071308370E-01)]), ("S", [(3.6649800000E+00, -3.9589516210E-01), (7.7054500000E-01, 1.2158343560E+00)]), ("P", [(3.6649800000E+00, 2.3645994660E-01), (7.7054500000E-01, 8.6061880570E-01)]), ("S", [(1.9585700000E-01, 1.0000000000E+00)]), ("P", [(1.9585700000E-01, 1.0000000000E+00)])],
    7 : [("S", [(2.4276600000E+02, 5.9865700510E-02), (3.6485100000E+01, 3.5295500300E-01), (7.8144900000E+00, 7.0651300600E-01)]), ("S", [(5.4252200000E+00, -4.1330007740E-01), (1.1491500000E+00, 1.2244172670E+00)]), ("P", [(5.4252200000E+00, 2.3797201620E-01), (1.1491500000E+00, 8.5895305860E-01)]), ("S", [(2.8320500000E-01, 1.0000000000E+00)]), ("P", [(2.8320500000E-01, 1.0000000000E+00)])],
    8 : [("S", [(3.2203700000E+02, 5.9239393390E-02), (4.8430800000E+01, 3.5149996080E-01), (1.0420600000E+01, 7.0765792100E-01)]), ("S", [(7.4029400000E+00, -4.0445358320E-01), (1.5762000000E+00, 1.2215617610E+00)]), ("P", [(7.4029400000E+00, 2.4458610700E-01), (1.5762000000E+00, 8.5395537350E-01)]), ("S", [(3.7368400000E-01, 1.0000000000E+00)]), ("P", [(3.7368400000E-01, 1.0000000000E+00)])],
    9 : [("S", [(4.1380100000E+02, 5.8548302930E-02), (6.2244600000E+01, 3.4930801750E-01), (1.3434000000E+01, 7.0963203550E-01)]), ("S", [(9.7775900000E+00, -4.0732627770E-01), (2.0861700000E+00, 1.2231378310E+00)]), ("P", [(9.7775900000E+00, 2.4668000320E-01), (2.0861700000E+00, 8.5232101100E-01)]), ("S", [(4.8238300000E-01, 1.0000000000E+00)]), ("P", [(4.8238300000E-01, 1.0000000000E+00)])],
    10 : [("S", [(5.1572400000E+02, 5.8143030440E-02), (7.7653800000E+01, 3.4795118220E-01), (1.6813600000E+01, 7.1071437210E-01)]), ("S", [(1.2483000000E+01, -4.0992232080E-01), (2.6645100000E+00, 1.2243109580E+00)]), ("P", [(1.2483000000E+01, 2.4745998360E-01), (2.6645100000E+00, 8.5174294350E-01)]), ("S", [(6.0625000000E-01, 1.0000000000E+00)]), ("P", [(6.0625000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(5.4761300000E+02, 6.7491118010E-02), (8.2067800000E+01, 3.9350510500E-01), (1.7691700000E+01, 6.6560517760E-01)]), ("S", [(1.7540700000E+01, -1.1193702900E-01), (3.7939800000E+00, 2.5465406590E-01), (9.0644100000E-01, 8.4441721870E-01)]), ("P", [(1.7540700000E+01, 1.2823300930E-01), (3.7939800000E+00, 4.7153303410E-01), (9.0644100000E-01, 6.0427304370E-01)]), ("S", [(5.0182400000E-01, -2.1966049750E-01), (6.0945800000E-02, 1.0891224670E+00)]), ("P", [(5.0182400000E-01, 9.0664879580E-03), (6.0945800000E-02, 9.9720177540E-01)]), ("S", [(2.4434900000E-02, 1.0000000000E+00)]), ("P", [(2.4434900000E-02, 1.0000000000E+00)])],
    12 : [("S", [(6.5284100000E+02, 6.7598142950E-02), (9.8380500000E+01, 3.9177766940E-01), (2.1299600000E+01, 6.6666043740E-01)]), ("S", [(2.3372700000E+01, -1.1024595240E-01), (5.1995300000E+00, 1.8411892060E-01), (1.3150800000E+00, 8.9639861330E-01)]), ("P", [(2.3372700000E+01, 1.2101389910E-01), (5.1995300000E+00, 4.6280961400E-01), (1.3150800000E+00, 6.0690649380E-01)]), ("S", [(6.1134900000E-01, -3.6110259330E-01), (1.4184100000E-01, 1.2150553610E+00)]), ("P", [(6.1134900000E-01, 2.4263309200E-02), (1.4184100000E-01, 9.8667337390E-01)]), ("S", [(4.6401100000E-02, 1.0000000000E+00)]), ("P", [(4.6401100000E-02, 1.0000000000E+00)])],
    13 : [("S", [(7.7573700000E+02, 6.6834696470E-02), (1.1695200000E+02, 3.8906097950E-01), (2.5332600000E+01, 6.6946796470E-01)]), ("S", [(2.9479600000E+01, -1.0790202330E-01), (6.6331400000E+00, 1.4624503160E-01), (1.7267500000E+00, 9.2373019940E-01)]), ("P", [(2.9479600000E+01, 1.1757398560E-01), (6.6331400000E+00, 4.6117394330E-01), (1.7267500000E+00, 6.0553492560E-01)]), ("S", [(9.4616000000E-01, -3.2032690910E-01), (2.0250600000E-01, 1.1841196640E+00)]), ("P", [(9.4616000000E-01, 5.1938280870E-02), (2.0250600000E-01, 9.7265964170E-01)]), ("S", [(6.3908800000E-02, 1.0000000000E+00)]), ("P", [(6.3908800000E-02, 1.0000000000E+00)])],
    14 : [("S", [(9.1065500000E+02, 6.6082239590E-02), (1.3733600000E+02, 3.8622864690E-01), (2.9760100000E+01, 6.7237938540E-01)]), ("S", [(3.6671600000E+01, -1.0451103590E-01), (8.3172900000E+00, 1.0741003690E-01), (2.2164500000E+00, 9.5144632690E-01)]), ("P", [(3.6671600000E+01, 1.1335501470E-01), (8.3172900000E+00, 4.5757805930E-01), (2.2164500000E+00, 6.0742707870E-01)]), ("S", [(1.0791300000E+00, -3.7610787950E-01), (3.0242200000E-01, 1.2516495990E+00)]), ("P", [(1.0791300000E+00, 6.7102991120E-02), (3.0242200000E-01, 9.5688287340E-01)]), ("S", [(9.3339200000E-02, 1.0000000000E+00)]), ("P", [(9.3339200000E-02, 1.0000000000E+00)])],
    15 : [("S", [(1.0549000000E+03, 6.5540713550E-02), (1.5919500000E+02, 3.8403607940E-01), (3.4530400000E+01, 6.7454113940E-01)]), ("S", [(4.4286600000E+01, -1.0213005350E-01), (1.0101900000E+01, 8.1592242710E-02), (2.7399700000E+00, 9.6978850760E-01)]), ("P", [(4.4286600000E+01, 1.1085100250E-01), (1.0101900000E+01, 4.5649501040E-01), (2.7399700000E+00, 6.0693601390E-01)]), ("S", [(1.2186500000E+00, -3.7149602190E-01), (3.9554600000E-01, 1.2709934960E+00)]), ("P", [(1.2186500000E+00, 9.1582310220E-02), (3.9554600000E-01, 9.3492410430E-01)]), ("S", [(1.2281100000E-01, 1.0000000000E+00)]), ("P", [(1.2281100000E-01, 1.0000000000E+00)])],
    16 : [("S", [(1.2106200000E+03, 6.5007081870E-02), (1.8274700000E+02, 3.8203989350E-01), (3.9667300000E+01, 6.7654481130E-01)]), ("S", [(5.2223600000E+01, -1.0030993990E-01), (1.1962900000E+01, 6.5087660990E-02), (3.2891100000E+00, 9.8145441170E-01)]), ("P", [(5.2223600000E+01, 1.0964597020E-01), (1.1962900000E+01, 4.5764887570E-01), (3.2891100000E+00, 6.0426083590E-01)]), ("S", [(1.2238400000E+00, -2.8608885370E-01), (4.5730300000E-01, 1.2280593720E+00)]), ("P", [(1.2238400000E+00, 1.6477699470E-01), (4.5730300000E-01, 8.7085497220E-01)]), ("S", [(1.4226900000E-01, 1.0000000000E+00)]), ("P", [(1.4226900000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.3764000000E+03, 6.4582702280E-02), (2.0785700000E+02, 3.8036301340E-01), (4.5155400000E+01, 6.7819002390E-01)]), ("S", [(6.0801400000E+01, -9.8763952750E-02), (1.3976500000E+01, 5.1133827310E-02), (3.8871000000E+00, 9.9133752950E-01)]), ("P", [(6.0801400000E+01, 1.0859805850E-01), (1.3976500000E+01, 4.5868224710E-01), (3.8871000000E+00, 6.0196232430E-01)]), ("S", [(1.3529900000E+00, -2.2240148410E-01), (5.2695500000E-01, 1.1825225740E+00)]), ("P", [(1.3529900000E+00, 2.1921579720E-01), (5.2695500000E-01, 8.2232023930E-01)]), ("S", [(1.6671400000E-01, 1.0000000000E+00)]), ("P", [(1.6671400000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.5537100000E+03, 6.4170712580E-02), (2.3467800000E+02, 3.7879707420E-01), (5.1012100000E+01, 6.7975213320E-01)]), ("S", [(7.0045300000E+01, -9.7466139890E-02), (1.6147300000E+01, 3.9056915990E-02), (4.5349200000E+00, 9.9991640930E-01)]), ("P", [(7.0045300000E+01, 1.0761904080E-01), (1.6147300000E+01, 4.5957617440E-01), (4.5349200000E+00, 6.0004122770E-01)]), ("S", [(1.5420900000E+00, -1.7686556850E-01), (6.0726700000E-01, 1.1468972020E+00)]), ("P", [(1.5420900000E+00, 2.5568701300E-01), (6.0726700000E-01, 7.8984204010E-01)]), ("S", [(1.9537300000E-01, 1.0000000000E+00)]), ("P", [(1.9537300000E-01, 1.0000000000E+00)])],

    }







_4_31G = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]), ("S", [(1.6127775880E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]), ("S", [(2.9796400000E-01, 1.0000000000E+00)])],
    5 : [("S", [(3.3075285200E+02, 1.7994179600E-02), (4.9843865000E+01, 1.2469370000E-01), (1.1117053500E+01, 4.3433537500E-01), (2.9227243100E+00, 5.6097937400E-01)]), ("S", [(5.6812646210E+00, -1.3038707790E-01), (1.4544045930E+00, -2.5143438980E-01), (4.2837857570E-01, 1.2051291990E+00)]), ("P", [(5.6812646210E+00, 6.3742922520E-02), (1.4544045930E+00, 2.7613305310E-01), (4.2837857570E-01, 7.7738659620E-01)]), ("S", [(1.4421917330E-01, 1.0000000000E+00)]), ("P", [(1.4421917330E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.8696692800E+02, 1.7725822900E-02), (7.3371094200E+01, 1.2347786700E-01), (1.6413457900E+01, 4.3387540000E-01), (4.3449835600E+00, 5.6150419700E-01)]), ("S", [(8.6735253100E+00, -1.2138374870E-01), (2.0966192600E+00, -2.2733849750E-01), (6.0465132900E-01, 1.1851739170E+00)]), ("P", [(8.6735253100E+00, 6.3545384110E-02), (2.0966192600E+00, 2.9826775710E-01), (6.0465132900E-01, 7.6210322810E-01)]), ("S", [(1.8355782980E-01, 1.0000000000E+00)]), ("P", [(1.8355782980E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.7127950300E+02, 1.7598251110E-02), (1.0120166200E+02, 1.2284624110E-01), (2.2699965900E+01, 4.3378214140E-01), (6.0406090000E+00, 5.6141821750E-01)]), ("S", [(1.2393599720E+01, -1.1748929910E-01), (2.9223828310E+00, -2.1399401620E-01), (8.3252807680E-01, 1.1745021110E+00)]), ("P", [(1.2393599720E+01, 6.4020344330E-02), (2.9223828310E+00, 3.1120255510E-01), (8.3252807680E-01, 7.5274823930E-01)]), ("S", [(2.2596417300E-01, 1.0000000000E+00)]), ("P", [(2.2596417300E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.8327286000E+02, 1.7550627990E-02), (1.3312928000E+02, 1.2282922300E-01), (2.9906407900E+01, 4.3488358380E-01), (7.9786771600E+00, 5.6001080380E-01)]), ("S", [(1.6194446640E+01, -1.1340100290E-01), (3.7800860220E+00, -1.7728646590E-01), (1.0709835750E+00, 1.1504079290E+00)]), ("P", [(1.6194446640E+01, 6.8545274710E-02), (3.7800860220E+00, 3.3122543500E-01), (1.0709835750E+00, 7.3460787810E-01)]), ("S", [(2.8387984070E-01, 1.0000000000E+00)]), ("P", [(2.8387984070E-01, 1.0000000000E+00)])],
    9 : [("S", [(1.1261626900E+03, 1.7475760900E-02), (1.6974315700E+02, 1.2252308900E-01), (3.8181511200E+01, 4.3499850200E-01), (1.0212035900E+01, 5.5981216700E-01)]), ("S", [(2.1495366700E+01, -1.1105707950E-01), (4.9897775700E+00, -1.6832210180E-01), (1.4035738600E+00, 1.1436255550E+00)]), ("P", [(2.1495366700E+01, 6.9888750800E-02), (4.9897775700E+00, 3.3938751000E-01), (1.4035738600E+00, 7.2795898100E-01)]), ("S", [(3.7303183500E-01, 1.0000000000E+00)]), ("P", [(3.7303183500E-01, 1.0000000000E+00)])],
    10 : [("S", [(1.3979320800E+03, 1.7423805400E-02), (2.1076978100E+02, 1.2227274500E-01), (4.7467256900E+01, 4.3501423200E-01), (1.2722626300E+01, 5.5971464200E-01)]), ("S", [(2.7213033200E+01, -1.0960943870E-01), (6.2941343500E+00, -1.6412488950E-01), (1.7600512500E+00, 1.1401515860E+00)]), ("P", [(2.7213033200E+01, 7.0440306680E-02), (6.2941343500E+00, 3.4399304690E-01), (1.7600512500E+00, 7.2451495980E-01)]), ("S", [(4.6186699200E-01, 1.0000000000E+00)]), ("P", [(4.6186699200E-01, 1.0000000000E+00)])],
    15 : [("S", [(3.0186717800E+03, 1.8521313720E-02), (4.5512712100E+02, 1.2990486420E-01), (1.0231473000E+02, 4.5510028860E-01), (2.7617847300E+01, 5.3313186170E-01)]), ("S", [(1.1442940100E+02, -2.4750296130E-02), (2.6582295900E+01, -1.3509246010E-01), (7.8718889000E+00, 2.2773608020E-01), (2.4878572500E+00, 8.7559311690E-01)]), ("P", [(1.1442940100E+02, 2.7414002550E-02), (2.6582295900E+01, 1.6907914230E-01), (7.8718889000E+00, 4.6910208990E-01), (2.4878572500E+00, 5.1815306000E-01)]), ("S", [(5.0750619000E+01, -4.5119223380E-02), (1.6728624200E+00, -8.5047299710E-01), (6.2109741200E-01, 1.5962858630E+00)]), ("P", [(5.0750619000E+01, 3.7790712150E-03), (1.6728624200E+00, -4.6343840930E-02), (6.2109741200E-01, 1.0339443000E+00)]), ("S", [(1.6701600700E-01, 1.0000000000E+00)]), ("P", [(1.6701600700E-01, 1.0000000000E+00)])],
    16 : [("S", [(3.4421244100E+03, 1.8492123620E-02), (5.1891310000E+02, 1.2982202210E-01), (1.1669090300E+02, 4.5504178740E-01), (3.1571647200E+01, 5.3300835650E-01)]), ("S", [(1.2744057600E+02, -2.7264610620E-02), (2.9747667300E+01, -1.4248341510E-01), (8.8346642800E+00, 2.5970435220E-01), (2.8173898200E+00, 8.5254729550E-01)]), ("P", [(1.2744057600E+02, 2.9151999540E-02), (2.9747667300E+01, 1.7795967630E-01), (8.8346642800E+00, 4.8362371270E-01), (2.8173898200E+00, 4.9425530270E-01)]), ("S", [(3.7291853700E+00, -2.7753152500E-01), (1.4067701700E+00, -4.5764345830E-01), (5.4810996900E-01, 1.4316842800E+00)]), ("P", [(3.7291853700E+00, -3.3750926340E-02), (1.4067701700E+00, 1.4571104520E-01), (5.4810996900E-01, 8.9828874420E-01)]), ("S", [(1.7038090500E-01, 1.0000000000E+00)]), ("P", [(1.7038090500E-01, 1.0000000000E+00)])],
    17 : [("S", [(3.9103026900E+03, 1.8379431120E-02), (5.8955180700E+02, 1.2914012320E-01), (1.3259392400E+02, 4.5404489060E-01), (3.5903542500E+01, 5.3443943670E-01)]), ("S", [(1.4776535300E+02, -2.6743323030E-02), (3.4506075300E+01, -1.4469118220E-01), (1.0286471500E+01, 2.5170356930E-01), (3.3111473800E+00, 8.5982038190E-01)]), ("P", [(1.4776535300E+02, 2.8864468810E-02), (3.4506075300E+01, 1.7796467010E-01), (1.0286471500E+01, 4.8699980720E-01), (3.3111473800E+00, 4.8901845020E-01)]), ("S", [(4.2802849100E+00, -2.7039627540E-01), (1.6410166700E+00, -3.4162971950E-01), (6.1447850300E-01, 1.3500244820E+00)]), ("P", [(4.2802849100E+00, -3.6702885140E-02), (1.6410166700E+00, 1.9184924220E-01), (6.1447850300E-01, 8.6433768180E-01)]), ("S", [(1.9565941100E-01, 1.0000000000E+00)]), ("P", [(1.9565941100E-01, 1.0000000000E+00)])],

    }








_6_31G = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]), ("S", [(1.6127775880E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]), ("S", [(2.9796400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]), ("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]), ("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]), ("S", [(3.5961971750E-02, 1.0000000000E+00)]), ("P", [(3.5961971750E-02, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]), ("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]), ("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]), ("S", [(8.2309900700E-02, 1.0000000000E+00)]), ("P", [(8.2309900700E-02, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]), ("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]), ("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]), ("S", [(1.2675124690E-01, 1.0000000000E+00)]), ("P", [(1.2675124690E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]), ("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]), ("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]), ("S", [(1.6871447820E-01, 1.0000000000E+00)]), ("P", [(1.6871447820E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]), ("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]), ("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]), ("S", [(2.1203149750E-01, 1.0000000000E+00)]), ("P", [(2.1203149750E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]), ("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]), ("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]), ("S", [(2.7000582260E-01, 1.0000000000E+00)]), ("P", [(2.7000582260E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]), ("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]), ("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]), ("S", [(3.5815139300E-01, 1.0000000000E+00)]), ("P", [(3.5815139300E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]), ("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]), ("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]), ("S", [(4.4581870000E-01, 1.0000000000E+00)]), ("P", [(4.4581870000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]), ("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]), ("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]), ("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]), ("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]), ("S", [(2.5954400000E-02, 1.0000000000E+00)]), ("P", [(2.5954400000E-02, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]), ("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]), ("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]), ("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]), ("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]), ("S", [(4.2106100000E-02, 1.0000000000E+00)]), ("P", [(4.2106100000E-02, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]), ("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]), ("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]), ("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]), ("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]), ("S", [(5.5657700000E-02, 1.0000000000E+00)]), ("P", [(5.5657700000E-02, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]), ("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]), ("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]), ("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]), ("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]), ("S", [(7.7836900000E-02, 1.0000000000E+00)]), ("P", [(7.7836900000E-02, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]), ("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]), ("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]), ("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]), ("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]), ("S", [(9.9831700000E-02, 1.0000000000E+00)]), ("P", [(9.9831700000E-02, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]), ("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]), ("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]), ("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]), ("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]), ("S", [(1.1716700000E-01, 1.0000000000E+00)]), ("P", [(1.1716700000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]), ("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]), ("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]), ("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]), ("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]), ("S", [(1.4265700000E-01, 1.0000000000E+00)]), ("P", [(1.4265700000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]), ("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]), ("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]), ("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]), ("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]), ("S", [(1.7388800000E-01, 1.0000000000E+00)]), ("P", [(1.7388800000E-01, 1.0000000000E+00)])],

    }








_6_31PLUSG = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]), ("S", [(1.6127775880E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]), ("S", [(2.9796400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]), ("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]), ("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]), ("S", [(3.5961971750E-02, 1.0000000000E+00)]), ("P", [(3.5961971750E-02, 1.0000000000E+00)]), ("S", [(7.4000000000E-03, 1.0000000000E+00)]), ("P", [(7.4000000000E-03, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]), ("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]), ("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]), ("S", [(8.2309900700E-02, 1.0000000000E+00)]), ("P", [(8.2309900700E-02, 1.0000000000E+00)]), ("S", [(2.0700000000E-02, 1.0000000000E+00)]), ("P", [(2.0700000000E-02, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]), ("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]), ("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]), ("S", [(1.2675124690E-01, 1.0000000000E+00)]), ("P", [(1.2675124690E-01, 1.0000000000E+00)]), ("S", [(3.1500000000E-02, 1.0000000000E+00)]), ("P", [(3.1500000000E-02, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]), ("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]), ("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]), ("S", [(1.6871447820E-01, 1.0000000000E+00)]), ("P", [(1.6871447820E-01, 1.0000000000E+00)]), ("S", [(4.3800000000E-02, 1.0000000000E+00)]), ("P", [(4.3800000000E-02, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]), ("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]), ("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]), ("S", [(2.1203149750E-01, 1.0000000000E+00)]), ("P", [(2.1203149750E-01, 1.0000000000E+00)]), ("S", [(6.3900000000E-02, 1.0000000000E+00)]), ("P", [(6.3900000000E-02, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]), ("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]), ("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]), ("S", [(2.7000582260E-01, 1.0000000000E+00)]), ("P", [(2.7000582260E-01, 1.0000000000E+00)]), ("S", [(8.4500000000E-02, 1.0000000000E+00)]), ("P", [(8.4500000000E-02, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]), ("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]), ("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]), ("S", [(3.5815139300E-01, 1.0000000000E+00)]), ("P", [(3.5815139300E-01, 1.0000000000E+00)]), ("S", [(1.0760000000E-01, 1.0000000000E+00)]), ("P", [(1.0760000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]), ("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]), ("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]), ("S", [(4.4581870000E-01, 1.0000000000E+00)]), ("P", [(4.4581870000E-01, 1.0000000000E+00)]), ("S", [(1.3000000000E-01, 1.0000000000E+00)]), ("P", [(1.3000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]), ("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]), ("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]), ("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]), ("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]), ("S", [(2.5954400000E-02, 1.0000000000E+00)]), ("P", [(2.5954400000E-02, 1.0000000000E+00)]), ("S", [(7.6000000000E-03, 1.0000000000E+00)]), ("P", [(7.6000000000E-03, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]), ("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]), ("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]), ("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]), ("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]), ("S", [(4.2106100000E-02, 1.0000000000E+00)]), ("P", [(4.2106100000E-02, 1.0000000000E+00)]), ("S", [(1.4600000000E-02, 1.0000000000E+00)]), ("P", [(1.4600000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]), ("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]), ("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]), ("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]), ("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]), ("S", [(5.5657700000E-02, 1.0000000000E+00)]), ("P", [(5.5657700000E-02, 1.0000000000E+00)]), ("S", [(3.1800000000E-02, 1.0000000000E+00)]), ("P", [(3.1800000000E-02, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]), ("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]), ("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]), ("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]), ("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]), ("S", [(7.7836900000E-02, 1.0000000000E+00)]), ("P", [(7.7836900000E-02, 1.0000000000E+00)]), ("S", [(3.3100000000E-02, 1.0000000000E+00)]), ("P", [(3.3100000000E-02, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]), ("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]), ("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]), ("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]), ("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]), ("S", [(9.9831700000E-02, 1.0000000000E+00)]), ("P", [(9.9831700000E-02, 1.0000000000E+00)]), ("S", [(3.4800000000E-02, 1.0000000000E+00)]), ("P", [(3.4800000000E-02, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]), ("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]), ("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]), ("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]), ("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]), ("S", [(1.1716700000E-01, 1.0000000000E+00)]), ("P", [(1.1716700000E-01, 1.0000000000E+00)]), ("S", [(4.0500000000E-02, 1.0000000000E+00)]), ("P", [(4.0500000000E-02, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]), ("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]), ("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]), ("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]), ("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]), ("S", [(1.4265700000E-01, 1.0000000000E+00)]), ("P", [(1.4265700000E-01, 1.0000000000E+00)]), ("S", [(4.8300000000E-02, 1.0000000000E+00)]), ("P", [(4.8300000000E-02, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]), ("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]), ("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]), ("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]), ("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]), ("S", [(1.7388800000E-01, 1.0000000000E+00)]), ("P", [(1.7388800000E-01, 1.0000000000E+00)]), ("S", [(6.0000000000E-02, 1.0000000000E+00)]), ("P", [(6.0000000000E-02, 1.0000000000E+00)])],

    }






_6_31PLUSPLUSG = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]), ("S", [(1.6127775880E-01, 1.0000000000E+00)]), ("S", [(3.6000000000E-02, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]), ("S", [(2.9796400000E-01, 1.0000000000E+00)]), ("S", [(8.6000000000E-02, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]), ("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]), ("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]), ("S", [(3.5961971750E-02, 1.0000000000E+00)]), ("P", [(3.5961971750E-02, 1.0000000000E+00)]), ("S", [(7.4000000000E-03, 1.0000000000E+00)]), ("P", [(7.4000000000E-03, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]), ("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]), ("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]), ("S", [(8.2309900700E-02, 1.0000000000E+00)]), ("P", [(8.2309900700E-02, 1.0000000000E+00)]), ("S", [(2.0700000000E-02, 1.0000000000E+00)]), ("P", [(2.0700000000E-02, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]), ("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]), ("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]), ("S", [(1.2675124690E-01, 1.0000000000E+00)]), ("P", [(1.2675124690E-01, 1.0000000000E+00)]), ("S", [(3.1500000000E-02, 1.0000000000E+00)]), ("P", [(3.1500000000E-02, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]), ("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]), ("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]), ("S", [(1.6871447820E-01, 1.0000000000E+00)]), ("P", [(1.6871447820E-01, 1.0000000000E+00)]), ("S", [(4.3800000000E-02, 1.0000000000E+00)]), ("P", [(4.3800000000E-02, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]), ("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]), ("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]), ("S", [(2.1203149750E-01, 1.0000000000E+00)]), ("P", [(2.1203149750E-01, 1.0000000000E+00)]), ("S", [(6.3900000000E-02, 1.0000000000E+00)]), ("P", [(6.3900000000E-02, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]), ("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]), ("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]), ("S", [(2.7000582260E-01, 1.0000000000E+00)]), ("P", [(2.7000582260E-01, 1.0000000000E+00)]), ("S", [(8.4500000000E-02, 1.0000000000E+00)]), ("P", [(8.4500000000E-02, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]), ("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]), ("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]), ("S", [(3.5815139300E-01, 1.0000000000E+00)]), ("P", [(3.5815139300E-01, 1.0000000000E+00)]), ("S", [(1.0760000000E-01, 1.0000000000E+00)]), ("P", [(1.0760000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]), ("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]), ("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]), ("S", [(4.4581870000E-01, 1.0000000000E+00)]), ("P", [(4.4581870000E-01, 1.0000000000E+00)]), ("S", [(1.3000000000E-01, 1.0000000000E+00)]), ("P", [(1.3000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]), ("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]), ("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]), ("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]), ("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]), ("S", [(2.5954400000E-02, 1.0000000000E+00)]), ("P", [(2.5954400000E-02, 1.0000000000E+00)]), ("S", [(7.6000000000E-03, 1.0000000000E+00)]), ("P", [(7.6000000000E-03, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]), ("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]), ("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]), ("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]), ("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]), ("S", [(4.2106100000E-02, 1.0000000000E+00)]), ("P", [(4.2106100000E-02, 1.0000000000E+00)]), ("S", [(1.4600000000E-02, 1.0000000000E+00)]), ("P", [(1.4600000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]), ("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]), ("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]), ("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]), ("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]), ("S", [(5.5657700000E-02, 1.0000000000E+00)]), ("P", [(5.5657700000E-02, 1.0000000000E+00)]), ("S", [(3.1800000000E-02, 1.0000000000E+00)]), ("P", [(3.1800000000E-02, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]), ("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]), ("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]), ("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]), ("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]), ("S", [(7.7836900000E-02, 1.0000000000E+00)]), ("P", [(7.7836900000E-02, 1.0000000000E+00)]), ("S", [(3.3100000000E-02, 1.0000000000E+00)]), ("P", [(3.3100000000E-02, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]), ("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]), ("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]), ("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]), ("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]), ("S", [(9.9831700000E-02, 1.0000000000E+00)]), ("P", [(9.9831700000E-02, 1.0000000000E+00)]), ("S", [(3.4800000000E-02, 1.0000000000E+00)]), ("P", [(3.4800000000E-02, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]), ("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]), ("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]), ("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]), ("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]), ("S", [(1.1716700000E-01, 1.0000000000E+00)]), ("P", [(1.1716700000E-01, 1.0000000000E+00)]), ("S", [(4.0500000000E-02, 1.0000000000E+00)]), ("P", [(4.0500000000E-02, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]), ("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]), ("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]), ("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]), ("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]), ("S", [(1.4265700000E-01, 1.0000000000E+00)]), ("P", [(1.4265700000E-01, 1.0000000000E+00)]), ("S", [(4.8300000000E-02, 1.0000000000E+00)]), ("P", [(4.8300000000E-02, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]), ("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]), ("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]), ("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]), ("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]), ("S", [(1.7388800000E-01, 1.0000000000E+00)]), ("P", [(1.7388800000E-01, 1.0000000000E+00)]), ("S", [(6.0000000000E-02, 1.0000000000E+00)]), ("P", [(6.0000000000E-02, 1.0000000000E+00)])],

    }








_6_311G = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]), ("S", [(3.2584000000E-01, 1.0000000000E+00)]), ("S", [(1.0274100000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]), ("S", [(8.7404700000E-01, 1.0000000000E+00)]), ("S", [(2.4456400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]), ("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]), ("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]), ("S", [(6.3507000000E-02, 1.0000000000E+00)]), ("P", [(6.3507000000E-02, 1.0000000000E+00)]), ("S", [(2.4368300000E-02, 1.0000000000E+00)]), ("P", [(2.4368300000E-02, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]), ("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]), ("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]), ("S", [(1.6361300000E-01, 1.0000000000E+00)]), ("P", [(1.6361300000E-01, 1.0000000000E+00)]), ("S", [(5.6728500000E-02, 1.0000000000E+00)]), ("P", [(5.6728500000E-02, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]), ("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]), ("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]), ("S", [(3.1545400000E-01, 1.0000000000E+00)]), ("P", [(3.1545400000E-01, 1.0000000000E+00)]), ("S", [(9.8856300000E-02, 1.0000000000E+00)]), ("P", [(9.8856300000E-02, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]), ("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]), ("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]), ("S", [(4.8345600000E-01, 1.0000000000E+00)]), ("P", [(4.8345600000E-01, 1.0000000000E+00)]), ("S", [(1.4558500000E-01, 1.0000000000E+00)]), ("P", [(1.4558500000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]), ("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]), ("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]), ("S", [(6.8400900000E-01, 1.0000000000E+00)]), ("P", [(6.8400900000E-01, 1.0000000000E+00)]), ("S", [(2.0087800000E-01, 1.0000000000E+00)]), ("P", [(2.0087800000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]), ("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]), ("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]), ("S", [(9.0566100000E-01, 1.0000000000E+00)]), ("P", [(9.0566100000E-01, 1.0000000000E+00)]), ("S", [(2.5561100000E-01, 1.0000000000E+00)]), ("P", [(2.5561100000E-01, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]), ("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]), ("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]), ("S", [(1.1654500000E+00, 1.0000000000E+00)]), ("P", [(1.1654500000E+00, 1.0000000000E+00)]), ("S", [(3.2189200000E-01, 1.0000000000E+00)]), ("P", [(3.2189200000E-01, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]), ("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]), ("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]), ("S", [(1.4575600000E+00, 1.0000000000E+00)]), ("P", [(1.4575600000E+00, 1.0000000000E+00)]), ("S", [(3.9705700000E-01, 1.0000000000E+00)]), ("P", [(3.9705700000E-01, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]), ("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]), ("S", [(1.8277700000E+00, 1.0000000000E+00)]), ("S", [(6.1994800000E-01, 1.0000000000E+00)]), ("S", [(5.7240000000E-02, 1.0000000000E+00)]), ("S", [(2.4048000000E-02, 1.0000000000E+00)]), ("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]), ("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]), ("P", [(1.8872000000E-01, 1.0000000000E+00)]), ("P", [(4.6501000000E-02, 1.0000000000E+00)]), ("P", [(1.6285000000E-02, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]), ("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]), ("S", [(2.5081700000E+00, 1.0000000000E+00)]), ("S", [(8.7153100000E-01, 1.0000000000E+00)]), ("S", [(1.0818800000E-01, 1.0000000000E+00)]), ("S", [(4.0130000000E-02, 1.0000000000E+00)]), ("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]), ("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]), ("P", [(2.3641700000E-01, 1.0000000000E+00)]), ("P", [(9.3358000000E-02, 1.0000000000E+00)]), ("P", [(3.4809000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]), ("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]), ("S", [(3.2067110000E+00, 1.0000000000E+00)]), ("S", [(1.1525550000E+00, 1.0000000000E+00)]), ("S", [(1.7667800000E-01, 1.0000000000E+00)]), ("S", [(6.5237000000E-02, 1.0000000000E+00)]), ("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]), ("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]), ("P", [(3.1681900000E-01, 1.0000000000E+00)]), ("P", [(1.1425700000E-01, 1.0000000000E+00)]), ("P", [(4.1397000000E-02, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]), ("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]), ("S", [(3.9268660000E+00, 1.0000000000E+00)]), ("S", [(1.4523430000E+00, 1.0000000000E+00)]), ("S", [(2.5623400000E-01, 1.0000000000E+00)]), ("S", [(9.4279000000E-02, 1.0000000000E+00)]), ("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]), ("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]), ("P", [(5.0497700000E-01, 1.0000000000E+00)]), ("P", [(1.8631700000E-01, 1.0000000000E+00)]), ("P", [(6.5432000000E-02, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]), ("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]), ("S", [(4.7941700000E+00, 1.0000000000E+00)]), ("S", [(1.8079300000E+00, 1.0000000000E+00)]), ("S", [(3.5681600000E-01, 1.0000000000E+00)]), ("S", [(1.1478300000E-01, 1.0000000000E+00)]), ("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]), ("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]), ("P", [(6.9700500000E-01, 1.0000000000E+00)]), ("P", [(2.5353200000E-01, 1.0000000000E+00)]), ("P", [(6.8493000000E-02, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]), ("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]), ("S", [(5.5600900000E+00, 1.0000000000E+00)]), ("S", [(2.1318300000E+00, 1.0000000000E+00)]), ("S", [(4.2040300000E-01, 1.0000000000E+00)]), ("S", [(1.3604500000E-01, 1.0000000000E+00)]), ("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]), ("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]), ("P", [(8.0799400000E-01, 1.0000000000E+00)]), ("P", [(2.7746000000E-01, 1.0000000000E+00)]), ("P", [(7.7141000000E-02, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]), ("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]), ("S", [(6.5834600000E+00, 1.0000000000E+00)]), ("S", [(2.5646800000E+00, 1.0000000000E+00)]), ("S", [(5.5976300000E-01, 1.0000000000E+00)]), ("S", [(1.8327300000E-01, 1.0000000000E+00)]), ("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]), ("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]), ("P", [(1.0238700000E+00, 1.0000000000E+00)]), ("P", [(3.8136800000E-01, 1.0000000000E+00)]), ("P", [(1.0943700000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]), ("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]), ("S", [(7.3778600000E+00, 1.0000000000E+00)]), ("S", [(2.9236880000E+00, 1.0000000000E+00)]), ("S", [(6.5040500000E-01, 1.0000000000E+00)]), ("S", [(2.3282500000E-01, 1.0000000000E+00)]), ("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]), ("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]), ("P", [(1.1064630000E+00, 1.0000000000E+00)]), ("P", [(4.1560100000E-01, 1.0000000000E+00)]), ("P", [(1.4544900000E-01, 1.0000000000E+00)])],


    }








_6_311PLUSG = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]), ("S", [(3.2584000000E-01, 1.0000000000E+00)]), ("S", [(1.0274100000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]), ("S", [(8.7404700000E-01, 1.0000000000E+00)]), ("S", [(2.4456400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]), ("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]), ("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]), ("S", [(6.3507000000E-02, 1.0000000000E+00)]), ("P", [(6.3507000000E-02, 1.0000000000E+00)]), ("S", [(2.4368300000E-02, 1.0000000000E+00)]), ("P", [(2.4368300000E-02, 1.0000000000E+00)]), ("S", [(7.4000000000E-03, 1.0000000000E+00)]), ("P", [(7.4000000000E-03, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]), ("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]), ("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]), ("S", [(1.6361300000E-01, 1.0000000000E+00)]), ("P", [(1.6361300000E-01, 1.0000000000E+00)]), ("S", [(5.6728500000E-02, 1.0000000000E+00)]), ("P", [(5.6728500000E-02, 1.0000000000E+00)]), ("S", [(2.0700000000E-02, 1.0000000000E+00)]), ("P", [(2.0700000000E-02, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]), ("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]), ("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]), ("S", [(3.1545400000E-01, 1.0000000000E+00)]), ("P", [(3.1545400000E-01, 1.0000000000E+00)]), ("S", [(9.8856300000E-02, 1.0000000000E+00)]), ("P", [(9.8856300000E-02, 1.0000000000E+00)]), ("S", [(3.1500000000E-02, 1.0000000000E+00)]), ("P", [(3.1500000000E-02, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]), ("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]), ("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]), ("S", [(4.8345600000E-01, 1.0000000000E+00)]), ("P", [(4.8345600000E-01, 1.0000000000E+00)]), ("S", [(1.4558500000E-01, 1.0000000000E+00)]), ("P", [(1.4558500000E-01, 1.0000000000E+00)]), ("S", [(4.3800000000E-02, 1.0000000000E+00)]), ("P", [(4.3800000000E-02, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]), ("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]), ("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]), ("S", [(6.8400900000E-01, 1.0000000000E+00)]), ("P", [(6.8400900000E-01, 1.0000000000E+00)]), ("S", [(2.0087800000E-01, 1.0000000000E+00)]), ("P", [(2.0087800000E-01, 1.0000000000E+00)]), ("S", [(6.3900000000E-02, 1.0000000000E+00)]), ("P", [(6.3900000000E-02, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]), ("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]), ("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]), ("S", [(9.0566100000E-01, 1.0000000000E+00)]), ("P", [(9.0566100000E-01, 1.0000000000E+00)]), ("S", [(2.5561100000E-01, 1.0000000000E+00)]), ("P", [(2.5561100000E-01, 1.0000000000E+00)]), ("S", [(8.4500000000E-02, 1.0000000000E+00)]), ("P", [(8.4500000000E-02, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]), ("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]), ("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]), ("S", [(1.1654500000E+00, 1.0000000000E+00)]), ("P", [(1.1654500000E+00, 1.0000000000E+00)]), ("S", [(3.2189200000E-01, 1.0000000000E+00)]), ("P", [(3.2189200000E-01, 1.0000000000E+00)]), ("S", [(1.0760000000E-01, 1.0000000000E+00)]), ("P", [(1.0760000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]), ("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]), ("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]), ("S", [(1.4575600000E+00, 1.0000000000E+00)]), ("P", [(1.4575600000E+00, 1.0000000000E+00)]), ("S", [(3.9705700000E-01, 1.0000000000E+00)]), ("P", [(3.9705700000E-01, 1.0000000000E+00)]), ("S", [(1.3000000000E-01, 1.0000000000E+00)]), ("P", [(1.3000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]), ("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]), ("S", [(1.8277700000E+00, 1.0000000000E+00)]), ("S", [(6.1994800000E-01, 1.0000000000E+00)]), ("S", [(5.7240000000E-02, 1.0000000000E+00)]), ("S", [(2.4048000000E-02, 1.0000000000E+00)]), ("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]), ("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]), ("P", [(1.8872000000E-01, 1.0000000000E+00)]), ("P", [(4.6501000000E-02, 1.0000000000E+00)]), ("P", [(1.6285000000E-02, 1.0000000000E+00)]), ("S", [(7.6000000000E-03, 1.0000000000E+00)]), ("P", [(7.6000000000E-03, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]), ("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]), ("S", [(2.5081700000E+00, 1.0000000000E+00)]), ("S", [(8.7153100000E-01, 1.0000000000E+00)]), ("S", [(1.0818800000E-01, 1.0000000000E+00)]), ("S", [(4.0130000000E-02, 1.0000000000E+00)]), ("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]), ("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]), ("P", [(2.3641700000E-01, 1.0000000000E+00)]), ("P", [(9.3358000000E-02, 1.0000000000E+00)]), ("P", [(3.4809000000E-02, 1.0000000000E+00)]), ("S", [(1.4600000000E-02, 1.0000000000E+00)]), ("P", [(1.4600000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]), ("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]), ("S", [(3.2067110000E+00, 1.0000000000E+00)]), ("S", [(1.1525550000E+00, 1.0000000000E+00)]), ("S", [(1.7667800000E-01, 1.0000000000E+00)]), ("S", [(6.5237000000E-02, 1.0000000000E+00)]), ("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]), ("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]), ("P", [(3.1681900000E-01, 1.0000000000E+00)]), ("P", [(1.1425700000E-01, 1.0000000000E+00)]), ("S", [(3.1800000000E-02, 1.0000000000E+00)]), ("P", [(3.1800000000E-02, 1.0000000000E+00)]), ("P", [(4.1397000000E-02, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]), ("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]), ("S", [(3.9268660000E+00, 1.0000000000E+00)]), ("S", [(1.4523430000E+00, 1.0000000000E+00)]), ("S", [(2.5623400000E-01, 1.0000000000E+00)]), ("S", [(9.4279000000E-02, 1.0000000000E+00)]), ("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]), ("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]), ("P", [(5.0497700000E-01, 1.0000000000E+00)]), ("P", [(1.8631700000E-01, 1.0000000000E+00)]), ("P", [(6.5432000000E-02, 1.0000000000E+00)]), ("S", [(3.3100000000E-02, 1.0000000000E+00)]), ("P", [(3.3100000000E-02, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]), ("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]), ("S", [(4.7941700000E+00, 1.0000000000E+00)]), ("S", [(1.8079300000E+00, 1.0000000000E+00)]), ("S", [(3.5681600000E-01, 1.0000000000E+00)]), ("S", [(1.1478300000E-01, 1.0000000000E+00)]), ("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]), ("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]), ("P", [(6.9700500000E-01, 1.0000000000E+00)]), ("P", [(2.5353200000E-01, 1.0000000000E+00)]), ("P", [(6.8493000000E-02, 1.0000000000E+00)]), ("S", [(3.4800000000E-02, 1.0000000000E+00)]), ("P", [(3.4800000000E-02, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]), ("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]), ("S", [(5.5600900000E+00, 1.0000000000E+00)]), ("S", [(2.1318300000E+00, 1.0000000000E+00)]), ("S", [(4.2040300000E-01, 1.0000000000E+00)]), ("S", [(1.3604500000E-01, 1.0000000000E+00)]), ("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]), ("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]), ("P", [(8.0799400000E-01, 1.0000000000E+00)]), ("P", [(2.7746000000E-01, 1.0000000000E+00)]), ("P", [(7.7141000000E-02, 1.0000000000E+00)]), ("S", [(4.0500000000E-02, 1.0000000000E+00)]), ("P", [(4.0500000000E-02, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]), ("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]), ("S", [(6.5834600000E+00, 1.0000000000E+00)]), ("S", [(2.5646800000E+00, 1.0000000000E+00)]), ("S", [(5.5976300000E-01, 1.0000000000E+00)]), ("S", [(1.8327300000E-01, 1.0000000000E+00)]), ("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]), ("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]), ("P", [(1.0238700000E+00, 1.0000000000E+00)]), ("P", [(3.8136800000E-01, 1.0000000000E+00)]), ("P", [(1.0943700000E-01, 1.0000000000E+00)]), ("S", [(4.8300000000E-02, 1.0000000000E+00)]), ("P", [(4.8300000000E-02, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]), ("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]), ("S", [(7.3778600000E+00, 1.0000000000E+00)]), ("S", [(2.9236880000E+00, 1.0000000000E+00)]), ("S", [(6.5040500000E-01, 1.0000000000E+00)]), ("S", [(2.3282500000E-01, 1.0000000000E+00)]), ("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]), ("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]), ("P", [(1.1064630000E+00, 1.0000000000E+00)]), ("P", [(4.1560100000E-01, 1.0000000000E+00)]), ("P", [(1.4544900000E-01, 1.0000000000E+00)]), ("S", [(6.0000000000E-02, 1.0000000000E+00)]), ("P", [(6.0000000000E-02, 1.0000000000E+00)])],

    }









_6_311PLUSPLUSG = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]), ("S", [(3.2584000000E-01, 1.0000000000E+00)]), ("S", [(1.0274100000E-01, 1.0000000000E+00)]), ("S", [(3.6000000000E-02, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]), ("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]), ("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]), ("S", [(6.3507000000E-02, 1.0000000000E+00)]), ("P", [(6.3507000000E-02, 1.0000000000E+00)]), ("S", [(2.4368300000E-02, 1.0000000000E+00)]), ("P", [(2.4368300000E-02, 1.0000000000E+00)]), ("S", [(7.4000000000E-03, 1.0000000000E+00)]), ("P", [(7.4000000000E-03, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]), ("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]), ("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]), ("S", [(1.6361300000E-01, 1.0000000000E+00)]), ("P", [(1.6361300000E-01, 1.0000000000E+00)]), ("S", [(5.6728500000E-02, 1.0000000000E+00)]), ("P", [(5.6728500000E-02, 1.0000000000E+00)]), ("S", [(2.0700000000E-02, 1.0000000000E+00)]), ("P", [(2.0700000000E-02, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]), ("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]), ("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]), ("S", [(3.1545400000E-01, 1.0000000000E+00)]), ("P", [(3.1545400000E-01, 1.0000000000E+00)]), ("S", [(9.8856300000E-02, 1.0000000000E+00)]), ("P", [(9.8856300000E-02, 1.0000000000E+00)]), ("S", [(3.1500000000E-02, 1.0000000000E+00)]), ("P", [(3.1500000000E-02, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]), ("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]), ("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]), ("S", [(4.8345600000E-01, 1.0000000000E+00)]), ("P", [(4.8345600000E-01, 1.0000000000E+00)]), ("S", [(1.4558500000E-01, 1.0000000000E+00)]), ("P", [(1.4558500000E-01, 1.0000000000E+00)]), ("S", [(4.3800000000E-02, 1.0000000000E+00)]), ("P", [(4.3800000000E-02, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]), ("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]), ("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]), ("S", [(6.8400900000E-01, 1.0000000000E+00)]), ("P", [(6.8400900000E-01, 1.0000000000E+00)]), ("S", [(2.0087800000E-01, 1.0000000000E+00)]), ("P", [(2.0087800000E-01, 1.0000000000E+00)]), ("S", [(6.3900000000E-02, 1.0000000000E+00)]), ("P", [(6.3900000000E-02, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]), ("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]), ("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]), ("S", [(9.0566100000E-01, 1.0000000000E+00)]), ("P", [(9.0566100000E-01, 1.0000000000E+00)]), ("S", [(2.5561100000E-01, 1.0000000000E+00)]), ("P", [(2.5561100000E-01, 1.0000000000E+00)]), ("S", [(8.4500000000E-02, 1.0000000000E+00)]), ("P", [(8.4500000000E-02, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]), ("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]), ("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]), ("S", [(1.1654500000E+00, 1.0000000000E+00)]), ("P", [(1.1654500000E+00, 1.0000000000E+00)]), ("S", [(3.2189200000E-01, 1.0000000000E+00)]), ("P", [(3.2189200000E-01, 1.0000000000E+00)]), ("S", [(1.0760000000E-01, 1.0000000000E+00)]), ("P", [(1.0760000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]), ("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]), ("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]), ("S", [(1.4575600000E+00, 1.0000000000E+00)]), ("P", [(1.4575600000E+00, 1.0000000000E+00)]), ("S", [(3.9705700000E-01, 1.0000000000E+00)]), ("P", [(3.9705700000E-01, 1.0000000000E+00)]), ("S", [(1.3000000000E-01, 1.0000000000E+00)]), ("P", [(1.3000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]), ("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]), ("S", [(1.8277700000E+00, 1.0000000000E+00)]), ("S", [(6.1994800000E-01, 1.0000000000E+00)]), ("S", [(5.7240000000E-02, 1.0000000000E+00)]), ("S", [(2.4048000000E-02, 1.0000000000E+00)]), ("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]), ("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]), ("P", [(1.8872000000E-01, 1.0000000000E+00)]), ("P", [(4.6501000000E-02, 1.0000000000E+00)]), ("P", [(1.6285000000E-02, 1.0000000000E+00)]), ("S", [(7.6000000000E-03, 1.0000000000E+00)]), ("P", [(7.6000000000E-03, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]), ("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]), ("S", [(2.5081700000E+00, 1.0000000000E+00)]), ("S", [(8.7153100000E-01, 1.0000000000E+00)]), ("S", [(1.0818800000E-01, 1.0000000000E+00)]), ("S", [(4.0130000000E-02, 1.0000000000E+00)]), ("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]), ("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]), ("P", [(2.3641700000E-01, 1.0000000000E+00)]), ("P", [(9.3358000000E-02, 1.0000000000E+00)]), ("P", [(3.4809000000E-02, 1.0000000000E+00)]), ("S", [(1.4600000000E-02, 1.0000000000E+00)]), ("P", [(1.4600000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]), ("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]), ("S", [(3.2067110000E+00, 1.0000000000E+00)]), ("S", [(1.1525550000E+00, 1.0000000000E+00)]), ("S", [(1.7667800000E-01, 1.0000000000E+00)]), ("S", [(6.5237000000E-02, 1.0000000000E+00)]), ("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]), ("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]), ("P", [(3.1681900000E-01, 1.0000000000E+00)]), ("P", [(1.1425700000E-01, 1.0000000000E+00)]), ("S", [(3.1800000000E-02, 1.0000000000E+00)]), ("P", [(3.1800000000E-02, 1.0000000000E+00)]), ("P", [(4.1397000000E-02, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]), ("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]), ("S", [(3.9268660000E+00, 1.0000000000E+00)]), ("S", [(1.4523430000E+00, 1.0000000000E+00)]), ("S", [(2.5623400000E-01, 1.0000000000E+00)]), ("S", [(9.4279000000E-02, 1.0000000000E+00)]), ("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]), ("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]), ("P", [(5.0497700000E-01, 1.0000000000E+00)]), ("P", [(1.8631700000E-01, 1.0000000000E+00)]), ("P", [(6.5432000000E-02, 1.0000000000E+00)]), ("S", [(3.3100000000E-02, 1.0000000000E+00)]), ("P", [(3.3100000000E-02, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]), ("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]), ("S", [(4.7941700000E+00, 1.0000000000E+00)]), ("S", [(1.8079300000E+00, 1.0000000000E+00)]), ("S", [(3.5681600000E-01, 1.0000000000E+00)]), ("S", [(1.1478300000E-01, 1.0000000000E+00)]), ("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]), ("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]), ("P", [(6.9700500000E-01, 1.0000000000E+00)]), ("P", [(2.5353200000E-01, 1.0000000000E+00)]), ("P", [(6.8493000000E-02, 1.0000000000E+00)]), ("S", [(3.4800000000E-02, 1.0000000000E+00)]), ("P", [(3.4800000000E-02, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]), ("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]), ("S", [(5.5600900000E+00, 1.0000000000E+00)]), ("S", [(2.1318300000E+00, 1.0000000000E+00)]), ("S", [(4.2040300000E-01, 1.0000000000E+00)]), ("S", [(1.3604500000E-01, 1.0000000000E+00)]), ("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]), ("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]), ("P", [(8.0799400000E-01, 1.0000000000E+00)]), ("P", [(2.7746000000E-01, 1.0000000000E+00)]), ("P", [(7.7141000000E-02, 1.0000000000E+00)]), ("S", [(4.0500000000E-02, 1.0000000000E+00)]), ("P", [(4.0500000000E-02, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]), ("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]), ("S", [(6.5834600000E+00, 1.0000000000E+00)]), ("S", [(2.5646800000E+00, 1.0000000000E+00)]), ("S", [(5.5976300000E-01, 1.0000000000E+00)]), ("S", [(1.8327300000E-01, 1.0000000000E+00)]), ("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]), ("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]), ("P", [(1.0238700000E+00, 1.0000000000E+00)]), ("P", [(3.8136800000E-01, 1.0000000000E+00)]), ("P", [(1.0943700000E-01, 1.0000000000E+00)]), ("S", [(4.8300000000E-02, 1.0000000000E+00)]), ("P", [(4.8300000000E-02, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]), ("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]), ("S", [(7.3778600000E+00, 1.0000000000E+00)]), ("S", [(2.9236880000E+00, 1.0000000000E+00)]), ("S", [(6.5040500000E-01, 1.0000000000E+00)]), ("S", [(2.3282500000E-01, 1.0000000000E+00)]), ("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]), ("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]), ("P", [(1.1064630000E+00, 1.0000000000E+00)]), ("P", [(4.1560100000E-01, 1.0000000000E+00)]), ("P", [(1.4544900000E-01, 1.0000000000E+00)]), ("S", [(6.0000000000E-02, 1.0000000000E+00)]), ("P", [(6.0000000000E-02, 1.0000000000E+00)])],
    
    }







_6_31GSTAR = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]), ("S", [(1.6127775880E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]), ("S", [(2.9796400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]), ("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]), ("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]), ("S", [(3.5961971750E-02, 1.0000000000E+00)]), ("P", [(3.5961971750E-02, 1.0000000000E+00)]), ("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]), ("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]), ("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]), ("S", [(8.2309900700E-02, 1.0000000000E+00)]), ("P", [(8.2309900700E-02, 1.0000000000E+00)]), ("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]), ("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]), ("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]), ("S", [(1.2675124690E-01, 1.0000000000E+00)]), ("P", [(1.2675124690E-01, 1.0000000000E+00)]), ("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]), ("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]), ("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]), ("S", [(1.6871447820E-01, 1.0000000000E+00)]), ("P", [(1.6871447820E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]), ("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]), ("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]), ("S", [(2.1203149750E-01, 1.0000000000E+00)]), ("P", [(2.1203149750E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]), ("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]), ("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]), ("S", [(2.7000582260E-01, 1.0000000000E+00)]), ("P", [(2.7000582260E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]), ("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]), ("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]), ("S", [(3.5815139300E-01, 1.0000000000E+00)]), ("P", [(3.5815139300E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]), ("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]), ("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]), ("S", [(4.4581870000E-01, 1.0000000000E+00)]), ("P", [(4.4581870000E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]), ("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]), ("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]), ("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]), ("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]), ("S", [(2.5954400000E-02, 1.0000000000E+00)]), ("P", [(2.5954400000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]), ("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]), ("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]), ("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]), ("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]), ("S", [(4.2106100000E-02, 1.0000000000E+00)]), ("P", [(4.2106100000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]), ("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]), ("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]), ("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]), ("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]), ("S", [(5.5657700000E-02, 1.0000000000E+00)]), ("P", [(5.5657700000E-02, 1.0000000000E+00)]), ("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]), ("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]), ("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]), ("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]), ("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]), ("S", [(7.7836900000E-02, 1.0000000000E+00)]), ("P", [(7.7836900000E-02, 1.0000000000E+00)]), ("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]), ("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]), ("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]), ("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]), ("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]), ("S", [(9.9831700000E-02, 1.0000000000E+00)]), ("P", [(9.9831700000E-02, 1.0000000000E+00)]), ("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]), ("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]), ("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]), ("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]), ("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]), ("S", [(1.1716700000E-01, 1.0000000000E+00)]), ("P", [(1.1716700000E-01, 1.0000000000E+00)]), ("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]), ("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]), ("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]), ("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]), ("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]), ("S", [(1.4265700000E-01, 1.0000000000E+00)]), ("P", [(1.4265700000E-01, 1.0000000000E+00)]), ("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]), ("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]), ("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]), ("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]), ("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]), ("S", [(1.7388800000E-01, 1.0000000000E+00)]), ("P", [(1.7388800000E-01, 1.0000000000E+00)]), ("D", [(8.5000000000E-01, 1.0000000000E+00)])],

    }



_6_31GBRADKET = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]), ("S", [(1.6127775880E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]), ("S", [(2.9796400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]), ("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]), ("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]), ("S", [(3.5961971750E-02, 1.0000000000E+00)]), ("P", [(3.5961971750E-02, 1.0000000000E+00)]), ("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]), ("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]), ("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]), ("S", [(8.2309900700E-02, 1.0000000000E+00)]), ("P", [(8.2309900700E-02, 1.0000000000E+00)]), ("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]), ("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]), ("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]), ("S", [(1.2675124690E-01, 1.0000000000E+00)]), ("P", [(1.2675124690E-01, 1.0000000000E+00)]), ("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]), ("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]), ("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]), ("S", [(1.6871447820E-01, 1.0000000000E+00)]), ("P", [(1.6871447820E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]), ("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]), ("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]), ("S", [(2.1203149750E-01, 1.0000000000E+00)]), ("P", [(2.1203149750E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]), ("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]), ("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]), ("S", [(2.7000582260E-01, 1.0000000000E+00)]), ("P", [(2.7000582260E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]), ("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]), ("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]), ("S", [(3.5815139300E-01, 1.0000000000E+00)]), ("P", [(3.5815139300E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]), ("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]), ("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]), ("S", [(4.4581870000E-01, 1.0000000000E+00)]), ("P", [(4.4581870000E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]), ("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]), ("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]), ("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]), ("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]), ("S", [(2.5954400000E-02, 1.0000000000E+00)]), ("P", [(2.5954400000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]), ("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]), ("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]), ("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]), ("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]), ("S", [(4.2106100000E-02, 1.0000000000E+00)]), ("P", [(4.2106100000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]), ("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]), ("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]), ("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]), ("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]), ("S", [(5.5657700000E-02, 1.0000000000E+00)]), ("P", [(5.5657700000E-02, 1.0000000000E+00)]), ("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]), ("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]), ("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]), ("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]), ("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]), ("S", [(7.7836900000E-02, 1.0000000000E+00)]), ("P", [(7.7836900000E-02, 1.0000000000E+00)]), ("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]), ("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]), ("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]), ("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]), ("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]), ("S", [(9.9831700000E-02, 1.0000000000E+00)]), ("P", [(9.9831700000E-02, 1.0000000000E+00)]), ("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]), ("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]), ("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]), ("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]), ("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]), ("S", [(1.1716700000E-01, 1.0000000000E+00)]), ("P", [(1.1716700000E-01, 1.0000000000E+00)]), ("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]), ("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]), ("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]), ("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]), ("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]), ("S", [(1.4265700000E-01, 1.0000000000E+00)]), ("P", [(1.4265700000E-01, 1.0000000000E+00)]), ("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]), ("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]), ("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]), ("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]), ("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]), ("S", [(1.7388800000E-01, 1.0000000000E+00)]), ("P", [(1.7388800000E-01, 1.0000000000E+00)]), ("D", [(8.5000000000E-01, 1.0000000000E+00)])],

    }


_6_31GSTARSTAR = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]), ("S", [(1.6127775880E-01, 1.0000000000E+00)]), ("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]), ("S", [(2.9796400000E-01, 1.0000000000E+00)]), ("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]), ("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]), ("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]), ("S", [(3.5961971750E-02, 1.0000000000E+00)]), ("P", [(3.5961971750E-02, 1.0000000000E+00)]), ("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]), ("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]), ("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]), ("S", [(8.2309900700E-02, 1.0000000000E+00)]), ("P", [(8.2309900700E-02, 1.0000000000E+00)]), ("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]), ("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]), ("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]), ("S", [(1.2675124690E-01, 1.0000000000E+00)]), ("P", [(1.2675124690E-01, 1.0000000000E+00)]), ("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]), ("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]), ("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]), ("S", [(1.6871447820E-01, 1.0000000000E+00)]), ("P", [(1.6871447820E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]), ("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]), ("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]), ("S", [(2.1203149750E-01, 1.0000000000E+00)]), ("P", [(2.1203149750E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]), ("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]), ("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]), ("S", [(2.7000582260E-01, 1.0000000000E+00)]), ("P", [(2.7000582260E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]), ("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]), ("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]), ("S", [(3.5815139300E-01, 1.0000000000E+00)]), ("P", [(3.5815139300E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]), ("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]), ("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]), ("S", [(4.4581870000E-01, 1.0000000000E+00)]), ("P", [(4.4581870000E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]), ("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]), ("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]), ("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]), ("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]), ("S", [(2.5954400000E-02, 1.0000000000E+00)]), ("P", [(2.5954400000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]), ("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]), ("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]), ("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]), ("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]), ("S", [(4.2106100000E-02, 1.0000000000E+00)]), ("P", [(4.2106100000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]), ("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]), ("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]), ("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]), ("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]), ("S", [(5.5657700000E-02, 1.0000000000E+00)]), ("P", [(5.5657700000E-02, 1.0000000000E+00)]), ("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]), ("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]), ("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]), ("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]), ("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]), ("S", [(7.7836900000E-02, 1.0000000000E+00)]), ("P", [(7.7836900000E-02, 1.0000000000E+00)]), ("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]), ("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]), ("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]), ("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]), ("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]), ("S", [(9.9831700000E-02, 1.0000000000E+00)]), ("P", [(9.9831700000E-02, 1.0000000000E+00)]), ("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]), ("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]), ("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]), ("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]), ("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]), ("S", [(1.1716700000E-01, 1.0000000000E+00)]), ("P", [(1.1716700000E-01, 1.0000000000E+00)]), ("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]), ("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]), ("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]), ("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]), ("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]), ("S", [(1.4265700000E-01, 1.0000000000E+00)]), ("P", [(1.4265700000E-01, 1.0000000000E+00)]), ("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]), ("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]), ("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]), ("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]), ("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]), ("S", [(1.7388800000E-01, 1.0000000000E+00)]), ("P", [(1.7388800000E-01, 1.0000000000E+00)]), ("D", [(8.5000000000E-01, 1.0000000000E+00)])],

    }


_6_31GBRADCOMMAPKET = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]), ("S", [(1.6127775880E-01, 1.0000000000E+00)]), ("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]), ("S", [(2.9796400000E-01, 1.0000000000E+00)]), ("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]), ("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]), ("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]), ("S", [(3.5961971750E-02, 1.0000000000E+00)]), ("P", [(3.5961971750E-02, 1.0000000000E+00)]), ("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]), ("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]), ("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]), ("S", [(8.2309900700E-02, 1.0000000000E+00)]), ("P", [(8.2309900700E-02, 1.0000000000E+00)]), ("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]), ("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]), ("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]), ("S", [(1.2675124690E-01, 1.0000000000E+00)]), ("P", [(1.2675124690E-01, 1.0000000000E+00)]), ("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]), ("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]), ("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]), ("S", [(1.6871447820E-01, 1.0000000000E+00)]), ("P", [(1.6871447820E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]), ("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]), ("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]), ("S", [(2.1203149750E-01, 1.0000000000E+00)]), ("P", [(2.1203149750E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]), ("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]), ("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]), ("S", [(2.7000582260E-01, 1.0000000000E+00)]), ("P", [(2.7000582260E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]), ("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]), ("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]), ("S", [(3.5815139300E-01, 1.0000000000E+00)]), ("P", [(3.5815139300E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]), ("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]), ("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]), ("S", [(4.4581870000E-01, 1.0000000000E+00)]), ("P", [(4.4581870000E-01, 1.0000000000E+00)]), ("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]), ("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]), ("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]), ("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]), ("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]), ("S", [(2.5954400000E-02, 1.0000000000E+00)]), ("P", [(2.5954400000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]), ("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]), ("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]), ("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]), ("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]), ("S", [(4.2106100000E-02, 1.0000000000E+00)]), ("P", [(4.2106100000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]), ("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]), ("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]), ("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]), ("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]), ("S", [(5.5657700000E-02, 1.0000000000E+00)]), ("P", [(5.5657700000E-02, 1.0000000000E+00)]), ("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]), ("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]), ("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]), ("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]), ("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]), ("S", [(7.7836900000E-02, 1.0000000000E+00)]), ("P", [(7.7836900000E-02, 1.0000000000E+00)]), ("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]), ("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]), ("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]), ("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]), ("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]), ("S", [(9.9831700000E-02, 1.0000000000E+00)]), ("P", [(9.9831700000E-02, 1.0000000000E+00)]), ("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]), ("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]), ("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]), ("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]), ("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]), ("S", [(1.1716700000E-01, 1.0000000000E+00)]), ("P", [(1.1716700000E-01, 1.0000000000E+00)]), ("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]), ("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]), ("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]), ("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]), ("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]), ("S", [(1.4265700000E-01, 1.0000000000E+00)]), ("P", [(1.4265700000E-01, 1.0000000000E+00)]), ("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]), ("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]), ("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]), ("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]), ("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]), ("S", [(1.7388800000E-01, 1.0000000000E+00)]), ("P", [(1.7388800000E-01, 1.0000000000E+00)]), ("D", [(8.5000000000E-01, 1.0000000000E+00)])],

    }





_6_31PLUSGSTAR = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]),("S", [(1.6127775880E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]),("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]),("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]),("S", [(3.5961971750E-02, 1.0000000000E+00)]),("P", [(3.5961971750E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]),("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]),("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]),("S", [(8.2309900700E-02, 1.0000000000E+00)]),("P", [(8.2309900700E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]),("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]),("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]),("S", [(1.2675124690E-01, 1.0000000000E+00)]),("P", [(1.2675124690E-01, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]),("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]),("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]),("S", [(1.6871447820E-01, 1.0000000000E+00)]),("P", [(1.6871447820E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]),("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]),("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]),("S", [(2.1203149750E-01, 1.0000000000E+00)]),("P", [(2.1203149750E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]),("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]),("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]),("S", [(2.7000582260E-01, 1.0000000000E+00)]),("P", [(2.7000582260E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]),("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]),("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]),("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]),("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]),("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]),("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]),("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]),("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]),("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]),("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]),("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]),("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]),("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]),("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]),("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]),("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]),("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]),("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]),("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]),("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]),("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]),("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]),("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]),("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]),("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]),("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]),("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]),("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]),("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]),("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]),("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]),("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]),("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]),("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]),("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}



_6_31PLUSGBRADKET = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]),("S", [(1.6127775880E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]),("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]),("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]),("S", [(3.5961971750E-02, 1.0000000000E+00)]),("P", [(3.5961971750E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]),("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]),("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]),("S", [(8.2309900700E-02, 1.0000000000E+00)]),("P", [(8.2309900700E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]),("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]),("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]),("S", [(1.2675124690E-01, 1.0000000000E+00)]),("P", [(1.2675124690E-01, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]),("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]),("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]),("S", [(1.6871447820E-01, 1.0000000000E+00)]),("P", [(1.6871447820E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]),("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]),("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]),("S", [(2.1203149750E-01, 1.0000000000E+00)]),("P", [(2.1203149750E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]),("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]),("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]),("S", [(2.7000582260E-01, 1.0000000000E+00)]),("P", [(2.7000582260E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]),("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]),("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]),("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]),("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]),("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]),("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]),("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]),("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]),("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]),("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]),("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]),("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]),("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]),("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]),("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]),("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]),("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]),("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]),("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]),("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]),("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]),("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]),("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]),("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]),("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]),("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]),("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]),("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]),("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]),("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]),("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]),("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]),("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]),("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]),("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}






_6_31PLUSGSTARSTAR = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]),("S", [(1.6127775880E-01, 1.0000000000E+00)]),("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)]),("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]),("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]),("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]),("S", [(3.5961971750E-02, 1.0000000000E+00)]),("P", [(3.5961971750E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]),("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]),("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]),("S", [(8.2309900700E-02, 1.0000000000E+00)]),("P", [(8.2309900700E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]),("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]),("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]),("S", [(1.2675124690E-01, 1.0000000000E+00)]),("P", [(1.2675124690E-01, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]),("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]),("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]),("S", [(1.6871447820E-01, 1.0000000000E+00)]),("P", [(1.6871447820E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]),("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]),("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]),("S", [(2.1203149750E-01, 1.0000000000E+00)]),("P", [(2.1203149750E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]),("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]),("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]),("S", [(2.7000582260E-01, 1.0000000000E+00)]),("P", [(2.7000582260E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]),("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]),("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]),("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]),("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]),("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]),("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]),("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]),("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]),("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]),("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]),("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]),("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]),("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]),("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]),("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]),("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]),("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]),("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]),("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]),("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]),("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]),("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]),("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]),("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]),("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]),("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]),("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]),("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]),("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]),("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]),("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]),("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]),("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]),("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]),("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}


_6_31PLUSGBRADCOMMAPKET = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]),("S", [(1.6127775880E-01, 1.0000000000E+00)]),("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)]),("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]),("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]),("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]),("S", [(3.5961971750E-02, 1.0000000000E+00)]),("P", [(3.5961971750E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]),("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]),("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]),("S", [(8.2309900700E-02, 1.0000000000E+00)]),("P", [(8.2309900700E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]),("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]),("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]),("S", [(1.2675124690E-01, 1.0000000000E+00)]),("P", [(1.2675124690E-01, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]),("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]),("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]),("S", [(1.6871447820E-01, 1.0000000000E+00)]),("P", [(1.6871447820E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]),("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]),("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]),("S", [(2.1203149750E-01, 1.0000000000E+00)]),("P", [(2.1203149750E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]),("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]),("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]),("S", [(2.7000582260E-01, 1.0000000000E+00)]),("P", [(2.7000582260E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]),("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]),("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]),("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]),("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]),("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]),("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]),("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]),("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]),("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]),("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]),("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]),("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]),("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]),("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]),("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]),("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]),("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]),("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]),("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]),("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]),("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]),("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]),("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]),("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]),("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]),("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]),("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]),("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]),("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]),("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]),("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]),("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]),("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]),("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]),("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}




_6_31PLUSPLUSGSTAR = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]),("S", [(1.6127775880E-01, 1.0000000000E+00)]),("S", [(3.6000000000E-02, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)]),("S", [(8.6000000000E-02, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]),("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]),("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]),("S", [(3.5961971750E-02, 1.0000000000E+00)]),("P", [(3.5961971750E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]),("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]),("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]),("S", [(8.2309900700E-02, 1.0000000000E+00)]),("P", [(8.2309900700E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]),("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]),("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]),("S", [(1.2675124690E-01, 1.0000000000E+00)]),("P", [(1.2675124690E-01, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]),("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]),("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]),("S", [(1.6871447820E-01, 1.0000000000E+00)]),("P", [(1.6871447820E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]),("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]),("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]),("S", [(2.1203149750E-01, 1.0000000000E+00)]),("P", [(2.1203149750E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]),("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]),("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]),("S", [(2.7000582260E-01, 1.0000000000E+00)]),("P", [(2.7000582260E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]),("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]),("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]),("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]),("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]),("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]),("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]),("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]),("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]),("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]),("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]),("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]),("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]),("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]),("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]),("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]),("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]),("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]),("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]),("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]),("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]),("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]),("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]),("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]),("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]),("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]),("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]),("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]),("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]),("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]),("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]),("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]),("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]),("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]),("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]),("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}



_6_31PLUSPLUSGBRADKET = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]),("S", [(1.6127775880E-01, 1.0000000000E+00)]),("S", [(3.6000000000E-02, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)]),("S", [(8.6000000000E-02, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]),("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]),("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]),("S", [(3.5961971750E-02, 1.0000000000E+00)]),("P", [(3.5961971750E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]),("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]),("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]),("S", [(8.2309900700E-02, 1.0000000000E+00)]),("P", [(8.2309900700E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]),("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]),("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]),("S", [(1.2675124690E-01, 1.0000000000E+00)]),("P", [(1.2675124690E-01, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]),("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]),("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]),("S", [(1.6871447820E-01, 1.0000000000E+00)]),("P", [(1.6871447820E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]),("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]),("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]),("S", [(2.1203149750E-01, 1.0000000000E+00)]),("P", [(2.1203149750E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]),("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]),("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]),("S", [(2.7000582260E-01, 1.0000000000E+00)]),("P", [(2.7000582260E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]),("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]),("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]),("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]),("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]),("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]),("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]),("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]),("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]),("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]),("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]),("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]),("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]),("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]),("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]),("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]),("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]),("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]),("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]),("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]),("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]),("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]),("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]),("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]),("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]),("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]),("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]),("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]),("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]),("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]),("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]),("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]),("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]),("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]),("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]),("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}



_6_31PLUSPLUSGSTARSTAR = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]),("S", [(1.6127775880E-01, 1.0000000000E+00)]),("S", [(3.6000000000E-02, 1.0000000000E+00)]),("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)]),("S", [(8.6000000000E-02, 1.0000000000E+00)]),("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]),("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]),("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]),("S", [(3.5961971750E-02, 1.0000000000E+00)]),("P", [(3.5961971750E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]),("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]),("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]),("S", [(8.2309900700E-02, 1.0000000000E+00)]),("P", [(8.2309900700E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]),("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]),("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]),("S", [(1.2675124690E-01, 1.0000000000E+00)]),("P", [(1.2675124690E-01, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]),("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]),("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]),("S", [(1.6871447820E-01, 1.0000000000E+00)]),("P", [(1.6871447820E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]),("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]),("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]),("S", [(2.1203149750E-01, 1.0000000000E+00)]),("P", [(2.1203149750E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]),("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]),("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]),("S", [(2.7000582260E-01, 1.0000000000E+00)]),("P", [(2.7000582260E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]),("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]),("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]),("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]),("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]),("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]),("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]),("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]),("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]),("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]),("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]),("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]),("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]),("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]),("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]),("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]),("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]),("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]),("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]),("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]),("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]),("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]),("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]),("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]),("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]),("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]),("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]),("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]),("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]),("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]),("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]),("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]),("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]),("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]),("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]),("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}




_6_31PLUSPLUSGBRADCOMMAPKET = {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]),("S", [(1.6127775880E-01, 1.0000000000E+00)]),("S", [(3.6000000000E-02, 1.0000000000E+00)]),("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)]),("S", [(8.6000000000E-02, 1.0000000000E+00)]),("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]),("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]),("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]),("S", [(3.5961971750E-02, 1.0000000000E+00)]),("P", [(3.5961971750E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]),("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]),("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]),("S", [(8.2309900700E-02, 1.0000000000E+00)]),("P", [(8.2309900700E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]),("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]),("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]),("S", [(1.2675124690E-01, 1.0000000000E+00)]),("P", [(1.2675124690E-01, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]),("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]),("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]),("S", [(1.6871447820E-01, 1.0000000000E+00)]),("P", [(1.6871447820E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]),("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]),("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]),("S", [(2.1203149750E-01, 1.0000000000E+00)]),("P", [(2.1203149750E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]),("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]),("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]),("S", [(2.7000582260E-01, 1.0000000000E+00)]),("P", [(2.7000582260E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]),("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]),("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]),("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]),("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]),("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]),("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]),("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]),("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]),("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]),("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]),("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]),("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]),("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]),("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]),("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]),("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]),("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]),("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]),("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]),("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]),("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]),("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]),("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]),("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]),("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]),("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]),("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]),("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]),("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]),("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]),("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]),("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]),("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]),("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]),("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}






_6_311GSTAR = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]), ("S", [(3.2584000000E-01, 1.0000000000E+00)]), ("S", [(1.0274100000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]), ("S", [(8.7404700000E-01, 1.0000000000E+00)]), ("S", [(2.4456400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]), ("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]), ("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]), ("S", [(6.3507000000E-02, 1.0000000000E+00)]), ("P", [(6.3507000000E-02, 1.0000000000E+00)]), ("S", [(2.4368300000E-02, 1.0000000000E+00)]), ("P", [(2.4368300000E-02, 1.0000000000E+00)]), ("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]), ("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]), ("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]), ("S", [(1.6361300000E-01, 1.0000000000E+00)]), ("P", [(1.6361300000E-01, 1.0000000000E+00)]), ("S", [(5.6728500000E-02, 1.0000000000E+00)]), ("P", [(5.6728500000E-02, 1.0000000000E+00)]), ("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]), ("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]), ("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]), ("S", [(3.1545400000E-01, 1.0000000000E+00)]), ("P", [(3.1545400000E-01, 1.0000000000E+00)]), ("S", [(9.8856300000E-02, 1.0000000000E+00)]), ("P", [(9.8856300000E-02, 1.0000000000E+00)]), ("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]), ("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]), ("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]), ("S", [(4.8345600000E-01, 1.0000000000E+00)]), ("P", [(4.8345600000E-01, 1.0000000000E+00)]), ("S", [(1.4558500000E-01, 1.0000000000E+00)]), ("P", [(1.4558500000E-01, 1.0000000000E+00)]), ("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]), ("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]), ("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]), ("S", [(6.8400900000E-01, 1.0000000000E+00)]), ("P", [(6.8400900000E-01, 1.0000000000E+00)]), ("S", [(2.0087800000E-01, 1.0000000000E+00)]), ("P", [(2.0087800000E-01, 1.0000000000E+00)]), ("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]), ("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]), ("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]), ("S", [(9.0566100000E-01, 1.0000000000E+00)]), ("P", [(9.0566100000E-01, 1.0000000000E+00)]), ("S", [(2.5561100000E-01, 1.0000000000E+00)]), ("P", [(2.5561100000E-01, 1.0000000000E+00)]), ("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]), ("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]), ("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]), ("S", [(1.1654500000E+00, 1.0000000000E+00)]), ("P", [(1.1654500000E+00, 1.0000000000E+00)]), ("S", [(3.2189200000E-01, 1.0000000000E+00)]), ("P", [(3.2189200000E-01, 1.0000000000E+00)]), ("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]), ("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]), ("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]), ("S", [(1.4575600000E+00, 1.0000000000E+00)]), ("P", [(1.4575600000E+00, 1.0000000000E+00)]), ("S", [(3.9705700000E-01, 1.0000000000E+00)]), ("P", [(3.9705700000E-01, 1.0000000000E+00)]), ("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]), ("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]), ("S", [(1.8277700000E+00, 1.0000000000E+00)]), ("S", [(6.1994800000E-01, 1.0000000000E+00)]), ("S", [(5.7240000000E-02, 1.0000000000E+00)]), ("S", [(2.4048000000E-02, 1.0000000000E+00)]), ("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]), ("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]), ("P", [(1.8872000000E-01, 1.0000000000E+00)]), ("P", [(4.6501000000E-02, 1.0000000000E+00)]), ("P", [(1.6285000000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]), ("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]), ("S", [(2.5081700000E+00, 1.0000000000E+00)]), ("S", [(8.7153100000E-01, 1.0000000000E+00)]), ("S", [(1.0818800000E-01, 1.0000000000E+00)]), ("S", [(4.0130000000E-02, 1.0000000000E+00)]), ("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]), ("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]), ("P", [(2.3641700000E-01, 1.0000000000E+00)]), ("P", [(9.3358000000E-02, 1.0000000000E+00)]), ("P", [(3.4809000000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]), ("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]), ("S", [(3.2067110000E+00, 1.0000000000E+00)]), ("S", [(1.1525550000E+00, 1.0000000000E+00)]), ("S", [(1.7667800000E-01, 1.0000000000E+00)]), ("S", [(6.5237000000E-02, 1.0000000000E+00)]), ("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]), ("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]), ("P", [(3.1681900000E-01, 1.0000000000E+00)]), ("P", [(1.1425700000E-01, 1.0000000000E+00)]), ("P", [(4.1397000000E-02, 1.0000000000E+00)]), ("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]), ("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]), ("S", [(3.9268660000E+00, 1.0000000000E+00)]), ("S", [(1.4523430000E+00, 1.0000000000E+00)]), ("S", [(2.5623400000E-01, 1.0000000000E+00)]), ("S", [(9.4279000000E-02, 1.0000000000E+00)]), ("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]), ("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]), ("P", [(5.0497700000E-01, 1.0000000000E+00)]), ("P", [(1.8631700000E-01, 1.0000000000E+00)]), ("P", [(6.5432000000E-02, 1.0000000000E+00)]), ("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]), ("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]), ("S", [(4.7941700000E+00, 1.0000000000E+00)]), ("S", [(1.8079300000E+00, 1.0000000000E+00)]), ("S", [(3.5681600000E-01, 1.0000000000E+00)]), ("S", [(1.1478300000E-01, 1.0000000000E+00)]), ("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]), ("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]), ("P", [(6.9700500000E-01, 1.0000000000E+00)]), ("P", [(2.5353200000E-01, 1.0000000000E+00)]), ("P", [(6.8493000000E-02, 1.0000000000E+00)]), ("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]), ("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]), ("S", [(5.5600900000E+00, 1.0000000000E+00)]), ("S", [(2.1318300000E+00, 1.0000000000E+00)]), ("S", [(4.2040300000E-01, 1.0000000000E+00)]), ("S", [(1.3604500000E-01, 1.0000000000E+00)]), ("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]), ("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]), ("P", [(8.0799400000E-01, 1.0000000000E+00)]), ("P", [(2.7746000000E-01, 1.0000000000E+00)]), ("P", [(7.7141000000E-02, 1.0000000000E+00)]), ("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]), ("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]), ("S", [(6.5834600000E+00, 1.0000000000E+00)]), ("S", [(2.5646800000E+00, 1.0000000000E+00)]), ("S", [(5.5976300000E-01, 1.0000000000E+00)]), ("S", [(1.8327300000E-01, 1.0000000000E+00)]), ("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]), ("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]), ("P", [(1.0238700000E+00, 1.0000000000E+00)]), ("P", [(3.8136800000E-01, 1.0000000000E+00)]), ("P", [(1.0943700000E-01, 1.0000000000E+00)]), ("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]), ("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]), ("S", [(7.3778600000E+00, 1.0000000000E+00)]), ("S", [(2.9236880000E+00, 1.0000000000E+00)]), ("S", [(6.5040500000E-01, 1.0000000000E+00)]), ("S", [(2.3282500000E-01, 1.0000000000E+00)]), ("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]), ("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]), ("P", [(1.1064630000E+00, 1.0000000000E+00)]), ("P", [(4.1560100000E-01, 1.0000000000E+00)]), ("P", [(1.4544900000E-01, 1.0000000000E+00)]), ("D", [(8.5000000000E-01, 1.0000000000E+00)])],


    }


_6_311GBRADKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]), ("S", [(3.2584000000E-01, 1.0000000000E+00)]), ("S", [(1.0274100000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]), ("S", [(8.7404700000E-01, 1.0000000000E+00)]), ("S", [(2.4456400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]), ("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]), ("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]), ("S", [(6.3507000000E-02, 1.0000000000E+00)]), ("P", [(6.3507000000E-02, 1.0000000000E+00)]), ("S", [(2.4368300000E-02, 1.0000000000E+00)]), ("P", [(2.4368300000E-02, 1.0000000000E+00)]), ("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]), ("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]), ("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]), ("S", [(1.6361300000E-01, 1.0000000000E+00)]), ("P", [(1.6361300000E-01, 1.0000000000E+00)]), ("S", [(5.6728500000E-02, 1.0000000000E+00)]), ("P", [(5.6728500000E-02, 1.0000000000E+00)]), ("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]), ("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]), ("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]), ("S", [(3.1545400000E-01, 1.0000000000E+00)]), ("P", [(3.1545400000E-01, 1.0000000000E+00)]), ("S", [(9.8856300000E-02, 1.0000000000E+00)]), ("P", [(9.8856300000E-02, 1.0000000000E+00)]), ("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]), ("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]), ("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]), ("S", [(4.8345600000E-01, 1.0000000000E+00)]), ("P", [(4.8345600000E-01, 1.0000000000E+00)]), ("S", [(1.4558500000E-01, 1.0000000000E+00)]), ("P", [(1.4558500000E-01, 1.0000000000E+00)]), ("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]), ("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]), ("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]), ("S", [(6.8400900000E-01, 1.0000000000E+00)]), ("P", [(6.8400900000E-01, 1.0000000000E+00)]), ("S", [(2.0087800000E-01, 1.0000000000E+00)]), ("P", [(2.0087800000E-01, 1.0000000000E+00)]), ("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]), ("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]), ("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]), ("S", [(9.0566100000E-01, 1.0000000000E+00)]), ("P", [(9.0566100000E-01, 1.0000000000E+00)]), ("S", [(2.5561100000E-01, 1.0000000000E+00)]), ("P", [(2.5561100000E-01, 1.0000000000E+00)]), ("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]), ("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]), ("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]), ("S", [(1.1654500000E+00, 1.0000000000E+00)]), ("P", [(1.1654500000E+00, 1.0000000000E+00)]), ("S", [(3.2189200000E-01, 1.0000000000E+00)]), ("P", [(3.2189200000E-01, 1.0000000000E+00)]), ("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]), ("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]), ("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]), ("S", [(1.4575600000E+00, 1.0000000000E+00)]), ("P", [(1.4575600000E+00, 1.0000000000E+00)]), ("S", [(3.9705700000E-01, 1.0000000000E+00)]), ("P", [(3.9705700000E-01, 1.0000000000E+00)]), ("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]), ("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]), ("S", [(1.8277700000E+00, 1.0000000000E+00)]), ("S", [(6.1994800000E-01, 1.0000000000E+00)]), ("S", [(5.7240000000E-02, 1.0000000000E+00)]), ("S", [(2.4048000000E-02, 1.0000000000E+00)]), ("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]), ("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]), ("P", [(1.8872000000E-01, 1.0000000000E+00)]), ("P", [(4.6501000000E-02, 1.0000000000E+00)]), ("P", [(1.6285000000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]), ("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]), ("S", [(2.5081700000E+00, 1.0000000000E+00)]), ("S", [(8.7153100000E-01, 1.0000000000E+00)]), ("S", [(1.0818800000E-01, 1.0000000000E+00)]), ("S", [(4.0130000000E-02, 1.0000000000E+00)]), ("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]), ("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]), ("P", [(2.3641700000E-01, 1.0000000000E+00)]), ("P", [(9.3358000000E-02, 1.0000000000E+00)]), ("P", [(3.4809000000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]), ("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]), ("S", [(3.2067110000E+00, 1.0000000000E+00)]), ("S", [(1.1525550000E+00, 1.0000000000E+00)]), ("S", [(1.7667800000E-01, 1.0000000000E+00)]), ("S", [(6.5237000000E-02, 1.0000000000E+00)]), ("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]), ("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]), ("P", [(3.1681900000E-01, 1.0000000000E+00)]), ("P", [(1.1425700000E-01, 1.0000000000E+00)]), ("P", [(4.1397000000E-02, 1.0000000000E+00)]), ("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]), ("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]), ("S", [(3.9268660000E+00, 1.0000000000E+00)]), ("S", [(1.4523430000E+00, 1.0000000000E+00)]), ("S", [(2.5623400000E-01, 1.0000000000E+00)]), ("S", [(9.4279000000E-02, 1.0000000000E+00)]), ("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]), ("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]), ("P", [(5.0497700000E-01, 1.0000000000E+00)]), ("P", [(1.8631700000E-01, 1.0000000000E+00)]), ("P", [(6.5432000000E-02, 1.0000000000E+00)]), ("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]), ("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]), ("S", [(4.7941700000E+00, 1.0000000000E+00)]), ("S", [(1.8079300000E+00, 1.0000000000E+00)]), ("S", [(3.5681600000E-01, 1.0000000000E+00)]), ("S", [(1.1478300000E-01, 1.0000000000E+00)]), ("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]), ("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]), ("P", [(6.9700500000E-01, 1.0000000000E+00)]), ("P", [(2.5353200000E-01, 1.0000000000E+00)]), ("P", [(6.8493000000E-02, 1.0000000000E+00)]), ("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]), ("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]), ("S", [(5.5600900000E+00, 1.0000000000E+00)]), ("S", [(2.1318300000E+00, 1.0000000000E+00)]), ("S", [(4.2040300000E-01, 1.0000000000E+00)]), ("S", [(1.3604500000E-01, 1.0000000000E+00)]), ("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]), ("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]), ("P", [(8.0799400000E-01, 1.0000000000E+00)]), ("P", [(2.7746000000E-01, 1.0000000000E+00)]), ("P", [(7.7141000000E-02, 1.0000000000E+00)]), ("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]), ("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]), ("S", [(6.5834600000E+00, 1.0000000000E+00)]), ("S", [(2.5646800000E+00, 1.0000000000E+00)]), ("S", [(5.5976300000E-01, 1.0000000000E+00)]), ("S", [(1.8327300000E-01, 1.0000000000E+00)]), ("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]), ("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]), ("P", [(1.0238700000E+00, 1.0000000000E+00)]), ("P", [(3.8136800000E-01, 1.0000000000E+00)]), ("P", [(1.0943700000E-01, 1.0000000000E+00)]), ("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]), ("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]), ("S", [(7.3778600000E+00, 1.0000000000E+00)]), ("S", [(2.9236880000E+00, 1.0000000000E+00)]), ("S", [(6.5040500000E-01, 1.0000000000E+00)]), ("S", [(2.3282500000E-01, 1.0000000000E+00)]), ("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]), ("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]), ("P", [(1.1064630000E+00, 1.0000000000E+00)]), ("P", [(4.1560100000E-01, 1.0000000000E+00)]), ("P", [(1.4544900000E-01, 1.0000000000E+00)]), ("D", [(8.5000000000E-01, 1.0000000000E+00)])],


    }





_6_311GSTARSTAR = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]), ("S", [(3.2584000000E-01, 1.0000000000E+00)]), ("S", [(1.0274100000E-01, 1.0000000000E+00)]), ("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]), ("S", [(8.7404700000E-01, 1.0000000000E+00)]), ("S", [(2.4456400000E-01, 1.0000000000E+00)]), ("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]), ("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]), ("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]), ("S", [(6.3507000000E-02, 1.0000000000E+00)]), ("P", [(6.3507000000E-02, 1.0000000000E+00)]), ("S", [(2.4368300000E-02, 1.0000000000E+00)]), ("P", [(2.4368300000E-02, 1.0000000000E+00)]), ("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]), ("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]), ("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]), ("S", [(1.6361300000E-01, 1.0000000000E+00)]), ("P", [(1.6361300000E-01, 1.0000000000E+00)]), ("S", [(5.6728500000E-02, 1.0000000000E+00)]), ("P", [(5.6728500000E-02, 1.0000000000E+00)]), ("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]), ("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]), ("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]), ("S", [(3.1545400000E-01, 1.0000000000E+00)]), ("P", [(3.1545400000E-01, 1.0000000000E+00)]), ("S", [(9.8856300000E-02, 1.0000000000E+00)]), ("P", [(9.8856300000E-02, 1.0000000000E+00)]), ("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]), ("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]), ("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]), ("S", [(4.8345600000E-01, 1.0000000000E+00)]), ("P", [(4.8345600000E-01, 1.0000000000E+00)]), ("S", [(1.4558500000E-01, 1.0000000000E+00)]), ("P", [(1.4558500000E-01, 1.0000000000E+00)]), ("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]), ("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]), ("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]), ("S", [(6.8400900000E-01, 1.0000000000E+00)]), ("P", [(6.8400900000E-01, 1.0000000000E+00)]), ("S", [(2.0087800000E-01, 1.0000000000E+00)]), ("P", [(2.0087800000E-01, 1.0000000000E+00)]), ("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]), ("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]), ("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]), ("S", [(9.0566100000E-01, 1.0000000000E+00)]), ("P", [(9.0566100000E-01, 1.0000000000E+00)]), ("S", [(2.5561100000E-01, 1.0000000000E+00)]), ("P", [(2.5561100000E-01, 1.0000000000E+00)]), ("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]), ("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]), ("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]), ("S", [(1.1654500000E+00, 1.0000000000E+00)]), ("P", [(1.1654500000E+00, 1.0000000000E+00)]), ("S", [(3.2189200000E-01, 1.0000000000E+00)]), ("P", [(3.2189200000E-01, 1.0000000000E+00)]), ("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]), ("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]), ("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]), ("S", [(1.4575600000E+00, 1.0000000000E+00)]), ("P", [(1.4575600000E+00, 1.0000000000E+00)]), ("S", [(3.9705700000E-01, 1.0000000000E+00)]), ("P", [(3.9705700000E-01, 1.0000000000E+00)]), ("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]), ("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]), ("S", [(1.8277700000E+00, 1.0000000000E+00)]), ("S", [(6.1994800000E-01, 1.0000000000E+00)]), ("S", [(5.7240000000E-02, 1.0000000000E+00)]), ("S", [(2.4048000000E-02, 1.0000000000E+00)]), ("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]), ("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]), ("P", [(1.8872000000E-01, 1.0000000000E+00)]), ("P", [(4.6501000000E-02, 1.0000000000E+00)]), ("P", [(1.6285000000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]), ("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]), ("S", [(2.5081700000E+00, 1.0000000000E+00)]), ("S", [(8.7153100000E-01, 1.0000000000E+00)]), ("S", [(1.0818800000E-01, 1.0000000000E+00)]), ("S", [(4.0130000000E-02, 1.0000000000E+00)]), ("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]), ("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]), ("P", [(2.3641700000E-01, 1.0000000000E+00)]), ("P", [(9.3358000000E-02, 1.0000000000E+00)]), ("P", [(3.4809000000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]), ("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]), ("S", [(3.2067110000E+00, 1.0000000000E+00)]), ("S", [(1.1525550000E+00, 1.0000000000E+00)]), ("S", [(1.7667800000E-01, 1.0000000000E+00)]), ("S", [(6.5237000000E-02, 1.0000000000E+00)]), ("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]), ("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]), ("P", [(3.1681900000E-01, 1.0000000000E+00)]), ("P", [(1.1425700000E-01, 1.0000000000E+00)]), ("P", [(4.1397000000E-02, 1.0000000000E+00)]), ("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]), ("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]), ("S", [(3.9268660000E+00, 1.0000000000E+00)]), ("S", [(1.4523430000E+00, 1.0000000000E+00)]), ("S", [(2.5623400000E-01, 1.0000000000E+00)]), ("S", [(9.4279000000E-02, 1.0000000000E+00)]), ("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]), ("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]), ("P", [(5.0497700000E-01, 1.0000000000E+00)]), ("P", [(1.8631700000E-01, 1.0000000000E+00)]), ("P", [(6.5432000000E-02, 1.0000000000E+00)]), ("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]), ("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]), ("S", [(4.7941700000E+00, 1.0000000000E+00)]), ("S", [(1.8079300000E+00, 1.0000000000E+00)]), ("S", [(3.5681600000E-01, 1.0000000000E+00)]), ("S", [(1.1478300000E-01, 1.0000000000E+00)]), ("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]), ("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]), ("P", [(6.9700500000E-01, 1.0000000000E+00)]), ("P", [(2.5353200000E-01, 1.0000000000E+00)]), ("P", [(6.8493000000E-02, 1.0000000000E+00)]), ("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]), ("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]), ("S", [(5.5600900000E+00, 1.0000000000E+00)]), ("S", [(2.1318300000E+00, 1.0000000000E+00)]), ("S", [(4.2040300000E-01, 1.0000000000E+00)]), ("S", [(1.3604500000E-01, 1.0000000000E+00)]), ("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]), ("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]), ("P", [(8.0799400000E-01, 1.0000000000E+00)]), ("P", [(2.7746000000E-01, 1.0000000000E+00)]), ("P", [(7.7141000000E-02, 1.0000000000E+00)]), ("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]), ("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]), ("S", [(6.5834600000E+00, 1.0000000000E+00)]), ("S", [(2.5646800000E+00, 1.0000000000E+00)]), ("S", [(5.5976300000E-01, 1.0000000000E+00)]), ("S", [(1.8327300000E-01, 1.0000000000E+00)]), ("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]), ("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]), ("P", [(1.0238700000E+00, 1.0000000000E+00)]), ("P", [(3.8136800000E-01, 1.0000000000E+00)]), ("P", [(1.0943700000E-01, 1.0000000000E+00)]), ("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]), ("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]), ("S", [(7.3778600000E+00, 1.0000000000E+00)]), ("S", [(2.9236880000E+00, 1.0000000000E+00)]), ("S", [(6.5040500000E-01, 1.0000000000E+00)]), ("S", [(2.3282500000E-01, 1.0000000000E+00)]), ("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]), ("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]), ("P", [(1.1064630000E+00, 1.0000000000E+00)]), ("P", [(4.1560100000E-01, 1.0000000000E+00)]), ("P", [(1.4544900000E-01, 1.0000000000E+00)]), ("D", [(8.5000000000E-01, 1.0000000000E+00)])],

}



_6_311GBRADCOMMAPKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]), ("S", [(3.2584000000E-01, 1.0000000000E+00)]), ("S", [(1.0274100000E-01, 1.0000000000E+00)]), ("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]), ("S", [(8.7404700000E-01, 1.0000000000E+00)]), ("S", [(2.4456400000E-01, 1.0000000000E+00)]), ("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]), ("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]), ("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]), ("S", [(6.3507000000E-02, 1.0000000000E+00)]), ("P", [(6.3507000000E-02, 1.0000000000E+00)]), ("S", [(2.4368300000E-02, 1.0000000000E+00)]), ("P", [(2.4368300000E-02, 1.0000000000E+00)]), ("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]), ("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]), ("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]), ("S", [(1.6361300000E-01, 1.0000000000E+00)]), ("P", [(1.6361300000E-01, 1.0000000000E+00)]), ("S", [(5.6728500000E-02, 1.0000000000E+00)]), ("P", [(5.6728500000E-02, 1.0000000000E+00)]), ("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]), ("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]), ("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]), ("S", [(3.1545400000E-01, 1.0000000000E+00)]), ("P", [(3.1545400000E-01, 1.0000000000E+00)]), ("S", [(9.8856300000E-02, 1.0000000000E+00)]), ("P", [(9.8856300000E-02, 1.0000000000E+00)]), ("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]), ("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]), ("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]), ("S", [(4.8345600000E-01, 1.0000000000E+00)]), ("P", [(4.8345600000E-01, 1.0000000000E+00)]), ("S", [(1.4558500000E-01, 1.0000000000E+00)]), ("P", [(1.4558500000E-01, 1.0000000000E+00)]), ("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]), ("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]), ("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]), ("S", [(6.8400900000E-01, 1.0000000000E+00)]), ("P", [(6.8400900000E-01, 1.0000000000E+00)]), ("S", [(2.0087800000E-01, 1.0000000000E+00)]), ("P", [(2.0087800000E-01, 1.0000000000E+00)]), ("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]), ("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]), ("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]), ("S", [(9.0566100000E-01, 1.0000000000E+00)]), ("P", [(9.0566100000E-01, 1.0000000000E+00)]), ("S", [(2.5561100000E-01, 1.0000000000E+00)]), ("P", [(2.5561100000E-01, 1.0000000000E+00)]), ("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]), ("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]), ("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]), ("S", [(1.1654500000E+00, 1.0000000000E+00)]), ("P", [(1.1654500000E+00, 1.0000000000E+00)]), ("S", [(3.2189200000E-01, 1.0000000000E+00)]), ("P", [(3.2189200000E-01, 1.0000000000E+00)]), ("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]), ("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]), ("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]), ("S", [(1.4575600000E+00, 1.0000000000E+00)]), ("P", [(1.4575600000E+00, 1.0000000000E+00)]), ("S", [(3.9705700000E-01, 1.0000000000E+00)]), ("P", [(3.9705700000E-01, 1.0000000000E+00)]), ("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]), ("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]), ("S", [(1.8277700000E+00, 1.0000000000E+00)]), ("S", [(6.1994800000E-01, 1.0000000000E+00)]), ("S", [(5.7240000000E-02, 1.0000000000E+00)]), ("S", [(2.4048000000E-02, 1.0000000000E+00)]), ("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]), ("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]), ("P", [(1.8872000000E-01, 1.0000000000E+00)]), ("P", [(4.6501000000E-02, 1.0000000000E+00)]), ("P", [(1.6285000000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]), ("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]), ("S", [(2.5081700000E+00, 1.0000000000E+00)]), ("S", [(8.7153100000E-01, 1.0000000000E+00)]), ("S", [(1.0818800000E-01, 1.0000000000E+00)]), ("S", [(4.0130000000E-02, 1.0000000000E+00)]), ("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]), ("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]), ("P", [(2.3641700000E-01, 1.0000000000E+00)]), ("P", [(9.3358000000E-02, 1.0000000000E+00)]), ("P", [(3.4809000000E-02, 1.0000000000E+00)]), ("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]), ("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]), ("S", [(3.2067110000E+00, 1.0000000000E+00)]), ("S", [(1.1525550000E+00, 1.0000000000E+00)]), ("S", [(1.7667800000E-01, 1.0000000000E+00)]), ("S", [(6.5237000000E-02, 1.0000000000E+00)]), ("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]), ("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]), ("P", [(3.1681900000E-01, 1.0000000000E+00)]), ("P", [(1.1425700000E-01, 1.0000000000E+00)]), ("P", [(4.1397000000E-02, 1.0000000000E+00)]), ("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]), ("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]), ("S", [(3.9268660000E+00, 1.0000000000E+00)]), ("S", [(1.4523430000E+00, 1.0000000000E+00)]), ("S", [(2.5623400000E-01, 1.0000000000E+00)]), ("S", [(9.4279000000E-02, 1.0000000000E+00)]), ("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]), ("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]), ("P", [(5.0497700000E-01, 1.0000000000E+00)]), ("P", [(1.8631700000E-01, 1.0000000000E+00)]), ("P", [(6.5432000000E-02, 1.0000000000E+00)]), ("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]), ("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]), ("S", [(4.7941700000E+00, 1.0000000000E+00)]), ("S", [(1.8079300000E+00, 1.0000000000E+00)]), ("S", [(3.5681600000E-01, 1.0000000000E+00)]), ("S", [(1.1478300000E-01, 1.0000000000E+00)]), ("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]), ("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]), ("P", [(6.9700500000E-01, 1.0000000000E+00)]), ("P", [(2.5353200000E-01, 1.0000000000E+00)]), ("P", [(6.8493000000E-02, 1.0000000000E+00)]), ("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]), ("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]), ("S", [(5.5600900000E+00, 1.0000000000E+00)]), ("S", [(2.1318300000E+00, 1.0000000000E+00)]), ("S", [(4.2040300000E-01, 1.0000000000E+00)]), ("S", [(1.3604500000E-01, 1.0000000000E+00)]), ("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]), ("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]), ("P", [(8.0799400000E-01, 1.0000000000E+00)]), ("P", [(2.7746000000E-01, 1.0000000000E+00)]), ("P", [(7.7141000000E-02, 1.0000000000E+00)]), ("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]), ("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]), ("S", [(6.5834600000E+00, 1.0000000000E+00)]), ("S", [(2.5646800000E+00, 1.0000000000E+00)]), ("S", [(5.5976300000E-01, 1.0000000000E+00)]), ("S", [(1.8327300000E-01, 1.0000000000E+00)]), ("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]), ("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]), ("P", [(1.0238700000E+00, 1.0000000000E+00)]), ("P", [(3.8136800000E-01, 1.0000000000E+00)]), ("P", [(1.0943700000E-01, 1.0000000000E+00)]), ("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]), ("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]), ("S", [(7.3778600000E+00, 1.0000000000E+00)]), ("S", [(2.9236880000E+00, 1.0000000000E+00)]), ("S", [(6.5040500000E-01, 1.0000000000E+00)]), ("S", [(2.3282500000E-01, 1.0000000000E+00)]), ("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]), ("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]), ("P", [(1.1064630000E+00, 1.0000000000E+00)]), ("P", [(4.1560100000E-01, 1.0000000000E+00)]), ("P", [(1.4544900000E-01, 1.0000000000E+00)]), ("D", [(8.5000000000E-01, 1.0000000000E+00)])],

}


_6_311PLUSGSTAR = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]),("S", [(8.7404700000E-01, 1.0000000000E+00)]),("S", [(2.4456400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],

    }

_6_311PLUSGBRADKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]),("S", [(8.7404700000E-01, 1.0000000000E+00)]),("S", [(2.4456400000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],

    }

_6_311PLUSGSTARSTAR = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]),("S", [(8.7404700000E-01, 1.0000000000E+00)]),("S", [(2.4456400000E-01, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}



_6_311PLUSGBRADCOMMAPKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]),("S", [(8.7404700000E-01, 1.0000000000E+00)]),("S", [(2.4456400000E-01, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}




_6_311PLUSPLUSGSTAR = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("S", [(3.6000000000E-02, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}


_6_311PLUSPLUSGBRADKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("S", [(3.6000000000E-02, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}



_6_311PLUSPLUSGSTARSTAR = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("S", [(3.6000000000E-02, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}


_6_311PLUSPLUSGBRADCOMMAPKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("S", [(3.6000000000E-02, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}








CC_PVDZ = {

    1 : [("S", [(1.3010000000E+01,  0.0334987264), (1.9620000000E+00,  0.2348008012), (4.4460000000E-01, 0.8136829579)]), ("S", [(1.2200000000E-01, 1.0000000000E+00)]), ("P", [(7.2700000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8360000000E+01, 0.0401760754), (5.7700000000E+00, 0.2613680748), (1.2400000000E+00, 0.7930712395)]), ("S", [(2.9760000000E-01, 1.0000000000E+00)]), ("P", [(1.2750000000E+00, 1.0000000000E+00)])],
    3: [("S", [(1.4690000000E+03, 7.6560120000E-04), (2.2050000000E+02, 5.8889327000E-03), (5.0260000000E+01, 2.9655553700E-02), (1.4240000000E+01, 1.0912316260E-01), (4.5810000000E+00, 2.8264178440E-01), (1.5800000000E+00, 4.5288711110E-01), (5.6400000000E-01, 2.7463095690E-01), (7.3450000000E-02, 9.7459238000E-03)]), ("S", [(1.4690000000E+03, -2.2499200000E-04), (2.2050000000E+02, -1.7305632000E-03), (5.0260000000E+01, -8.7915609000E-03), (1.4240000000E+01, -3.3152565500E-02), (4.5810000000E+00, -9.1687974000E-02), (1.5800000000E+00, -1.8001044330E-01), (5.6400000000E-01, -2.5570336380E-01), (7.3450000000E-02, 1.0782777234E+00)]), ("S", [(2.8050000000E-02, 1.0000000000E+00)]), ("P", [(1.5340000000E+00, 3.8003981000E-02), (2.7490000000E-01, 2.3203211860E-01), (7.3620000000E-02, 8.3463140850E-01)]), ("P", [(2.4030000000E-02, 1.0000000000E+00)]), ("D", [(1.1440000000E-01, 1.0000000000E+00)])],
    4: [("S", [(2.9400000000E+03, 6.7955730000E-04), (4.4120000000E+02, 5.2325911000E-03), (1.0050000000E+02, 2.6588678200E-02), (2.8430000000E+01, 9.9927899600E-02), (9.1690000000E+00, 2.6952641070E-01), (3.1960000000E+00, 4.5117507140E-01), (1.1590000000E+00, 2.9488189230E-01), (1.8110000000E-01, 1.2578805200E-02)]), ("S", [(2.9400000000E+03, -2.5151360000E-04), (4.4120000000E+02, -1.9753019000E-03), (1.0050000000E+02, -9.8785546000E-03), (2.8430000000E+01, -3.9493770000E-02), (9.1690000000E+00, -1.0894833110E-01), (3.1960000000E+00, -2.4685753340E-01), (1.1590000000E+00, -2.7285136190E-01), (1.8110000000E-01, 1.0853261799E+00)]), ("S", [(5.8900000000E-02, 1.0000000000E+00)]), ("P", [(3.6190000000E+00, 4.5560679000E-02), (7.1100000000E-01, 2.6506765130E-01), (1.9510000000E-01, 8.0359641080E-01)]), ("P", [(6.0180000000E-02, 1.0000000000E+00)]), ("D", [(2.3540000000E-01, 1.0000000000E+00)])],
    5: [("S", [(4.5700000000E+03, 6.9555440000E-04), (6.8590000000E+02, 5.3495730000E-03), (1.5650000000E+02, 2.7116628700E-02), (4.4470000000E+01, 1.0131509610E-01), (1.4480000000E+01, 2.7188082930E-01), (5.1310000000E+00, 4.4811593060E-01), (1.8980000000E+00, 2.8993726210E-01), (3.3290000000E-01, 1.4312831000E-02)]), ("S", [(4.5700000000E+03, -2.8128020000E-04), (6.8590000000E+02, -2.2198878000E-03), (1.5650000000E+02, -1.1016471500E-02), (4.4470000000E+01, -4.4349189900E-02), (1.4480000000E+01, -1.2091204800E-01), (5.1310000000E+00, -2.8073789960E-01), (1.8980000000E+00, -2.6606680870E-01), (3.3290000000E-01, 1.0917841305E+00)]), ("S", [(1.0430000000E-01, 1.0000000000E+00)]), ("P", [(6.0010000000E+00, 5.4165863300E-02), (1.2410000000E+00, 3.0237989020E-01), (3.3640000000E-01, 7.7129221650E-01)]), ("P", [(9.5380000000E-02, 1.0000000000E+00)]), ("D", [(3.4300000000E-01, 1.0000000000E+00)])],
    6: [("S", [(6.6650000000E+03, 6.9158400000E-04), (1.0000000000E+03, 5.3257962000E-03), (2.2800000000E+02, 2.7060721000E-02), (6.4710000000E+01, 1.0165684610E-01), (2.1060000000E+01, 2.7457482360E-01), (7.4950000000E+00, 4.4829431890E-01), (2.7970000000E+00, 2.8490261070E-01), (5.2150000000E-01, 1.5194859200E-02)]), ("S", [(6.6650000000E+03, -2.9326970000E-04), (1.0000000000E+03, -2.3180355000E-03), (2.2800000000E+02, -1.1499786000E-02), (6.4710000000E+01, -4.6826727000E-02), (2.1060000000E+01, -1.2846616880E-01), (7.4950000000E+00, -3.0126627250E-01), (2.7970000000E+00, -2.5563070230E-01), (5.2150000000E-01, 1.0937933610E+00)]), ("S", [(1.5960000000E-01, 1.0000000000E+00)]), ("P", [(9.4390000000E+00, 5.6979251600E-02), (2.0020000000E+00, 3.1320721150E-01), (5.4560000000E-01, 7.6037674170E-01)]), ("P", [(1.5170000000E-01, 1.0000000000E+00)]), ("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    7: [("S", [(9.0460000000E+03, 6.9961740000E-04), (1.3570000000E+03, 5.3860546000E-03), (3.0930000000E+02, 2.7391021200E-02), (8.7730000000E+01, 1.0315059200E-01), (2.8560000000E+01, 2.7857066330E-01), (1.0210000000E+01, 4.4829484950E-01), (3.8380000000E+00, 2.7808592840E-01), (7.4660000000E-01, 1.5431561200E-02)]), ("S", [(9.0460000000E+03, -3.0499010000E-04), (1.3570000000E+03, -2.4080264000E-03), (3.0930000000E+02, -1.1944448700E-02), (8.7730000000E+01, -4.8925992900E-02), (2.8560000000E+01, -1.3447272470E-01), (1.0210000000E+01, -3.1511257770E-01), (3.8380000000E+00, -2.4285783250E-01), (7.4660000000E-01, 1.0943822069E+00)]), ("S", [(2.2480000000E-01, 1.0000000000E+00)]), ("P", [(1.3550000000E+01, 5.8905676800E-02), (2.9170000000E+00, 3.2046110670E-01), (7.9730000000E-01, 7.5304206180E-01)]), ("P", [(2.1850000000E-01, 1.0000000000E+00)]), ("D", [(8.1700000000E-01, 1.0000000000E+00)])],
    8: [("S", [(1.1720000000E+04, 7.0964590000E-04), (1.7590000000E+03, 5.4672723000E-03), (4.0080000000E+02, 2.7823118600E-02), (1.1370000000E+02, 1.0474773970E-01), (3.7030000000E+01, 2.8292084640E-01), (1.3270000000E+01, 4.4849523870E-01), (5.0250000000E+00, 2.7081688520E-01), (1.0130000000E+00, 1.5450291600E-02)]), ("S", [(1.1720000000E+04, -3.1444340000E-04), (1.7590000000E+03, -2.4821377000E-03), (4.0080000000E+02, -1.2316355400E-02), (1.1370000000E+02, -5.0538917300E-02), (3.7030000000E+01, -1.3938490330E-01), (1.3270000000E+01, -3.2507749480E-01), (5.0250000000E+00, -2.2984830760E-01), (1.0130000000E+00, 1.0953793468E+00)]), ("S", [(3.0230000000E-01, 1.0000000000E+00)]), ("P", [(1.7700000000E+01, 6.2679166300E-02), (3.8540000000E+00, 3.3353656590E-01), (1.0460000000E+00, 7.4123964160E-01)]), ("P", [(2.7530000000E-01, 1.0000000000E+00)]), ("D", [(1.1850000000E+00, 1.0000000000E+00)])],
    9: [("S", [(1.4710000000E+04, 7.2067370000E-04), (2.2070000000E+03, 5.5504870000E-03), (5.0280000000E+02, 2.8254207700E-02), (1.4260000000E+02, 1.0639582840E-01), (4.6470000000E+01, 2.8668420120E-01), (1.6700000000E+01, 4.4843796580E-01), (6.3560000000E+00, 2.6464118140E-01), (1.3160000000E+00, 1.5326061000E-02)]), ("S", [(1.4710000000E+04, -3.2119390000E-04), (2.2070000000E+03, -2.5461916000E-03), (5.0280000000E+02, -1.2643359500E-02), (1.4260000000E+02, -5.1957491700E-02), (4.6470000000E+01, -1.4344713810E-01), (1.6700000000E+01, -3.3243762330E-01), (6.3560000000E+00, -2.1865906750E-01), (1.3160000000E+00, 1.0955904139E+00)]), ("S", [(3.8970000000E-01, 1.0000000000E+00)]), ("P", [(2.2670000000E+01, 6.4834021500E-02), (4.9770000000E+00, 3.4053535980E-01), (1.3470000000E+00, 7.3464640680E-01)]), ("P", [(3.4710000000E-01, 1.0000000000E+00)]), ("D", [(1.6400000000E+00, 1.0000000000E+00)])],
    10: [("S", [(1.7880000000E+04, 7.3769820000E-04), (2.6830000000E+03, 5.6746782000E-03), (6.1150000000E+02, 2.8871187500E-02), (1.7350000000E+02, 1.0849560940E-01), (5.6640000000E+01, 2.9078802510E-01), (2.0420000000E+01, 4.4814064480E-01), (7.8100000000E+00, 2.5792047270E-01), (1.6530000000E+00, 1.5056839500E-02)]), ("S", [(1.7880000000E+04, -3.3214910000E-04), (2.6830000000E+03, -2.6205019000E-03), (6.1150000000E+02, -1.3009816800E-02), (1.7350000000E+02, -5.3420003100E-02), (5.6640000000E+01, -1.4716522420E-01), (2.0420000000E+01, -3.3838075720E-01), (7.8100000000E+00, -2.0670101920E-01), (1.6530000000E+00, 1.0950299235E+00)]), ("S", [(4.8690000000E-01, 1.0000000000E+00)]), ("P", [(2.8390000000E+01, 6.6171986600E-02), (6.2700000000E+00, 3.4485329750E-01), (1.6950000000E+00, 7.3045763820E-01)]), ("P", [(4.3170000000E-01, 1.0000000000E+00)]), ("D", [(2.2020000000E+00, 1.0000000000E+00)])],
    11: [("S", [(3.1700000000E+04, 4.5888010000E-04), (4.7550000000E+03, 3.5507160000E-03), (1.0820000000E+03, 1.8261882100E-02), (3.0640000000E+02, 7.1665322100E-02), (9.9530000000E+01, 2.1234695440E-01), (3.5420000000E+01, 4.1620487070E-01), (1.3300000000E+01, 3.7302167660E-01), (4.3920000000E+00, 6.2505680900E-02), (1.6760000000E+00, -6.2453481000E-03), (5.8890000000E-01, 2.4337509000E-03), (5.6400000000E-02, -4.4238300000E-04)]), ("S", [(3.1700000000E+04, -1.1204890000E-04), (4.7550000000E+03, -8.6763650000E-04), (1.0820000000E+03, -4.5087506000E-03), (3.0640000000E+02, -1.8125311400E-02), (9.9530000000E+01, -5.8021355900E-02), (3.5420000000E+01, -1.3751424670E-01), (1.3300000000E+01, -1.9371254210E-01), (4.3920000000E+00, 8.5714413300E-02), (1.6760000000E+00, 6.0380974990E-01), (5.8890000000E-01, 4.4127375040E-01), (5.6400000000E-02, 1.3041541000E-02)]), ("S", [(3.1700000000E+04, 2.7785300000E-05), (4.7550000000E+03, 2.1340790000E-04), (1.0820000000E+03, 1.1230788000E-03), (3.0640000000E+02, 4.4473356000E-03), (9.9530000000E+01, 1.4623045500E-02), (3.5420000000E+01, 3.3936776800E-02), (1.3300000000E+01, 5.2151390500E-02), (4.3920000000E+00, -3.1248379000E-02), (1.6760000000E+00, -1.6752683040E-01), (5.8890000000E-01, -3.2485623340E-01), (5.6400000000E-02, 1.0711005354E+00)]), ("S", [(2.3070000000E-02, 1.0000000000E+00)]), ("P", [(1.3810000000E+02, 5.7958549000E-03), (3.2240000000E+01, 4.1571618500E-02), (9.9850000000E+00, 1.6285740260E-01), (3.4840000000E+00, 3.5936658220E-01), (1.2310000000E+00, 4.4994490720E-01), (4.1770000000E-01, 2.2748521290E-01), (6.5130000000E-02, 8.0816960000E-03)]), ("P", [(1.3810000000E+02, -1.3458687000E-03), (3.2240000000E+01, -9.4265034000E-03), (9.9850000000E+00, -3.8866569600E-02), (3.4840000000E+00, -8.1758726800E-02), (1.2310000000E+00, -1.2080257590E-01), (4.1770000000E-01, -3.8964235300E-02), (6.5130000000E-02, 1.0058484076E+00)]), ("P", [(2.0530000000E-02, 1.0000000000E+00)]), ("D", [(9.2800000000E-02, 1.0000000000E+00)])],
    12: [("S", [(4.7390000000E+04, 3.4602290000E-04), (7.1080000000E+03, 2.6807692000E-03), (1.6180000000E+03, 1.3836695800E-02), (4.5840000000E+02, 5.5176683200E-02), (1.4930000000E+02, 1.6965994830E-01), (5.3590000000E+01, 3.6470288890E-01), (2.0700000000E+01, 4.0685587600E-01), (8.3840000000E+00, 1.3508895880E-01), (2.5420000000E+00, 4.9088385000E-03), (8.7870000000E-01, 2.8645990000E-04), (1.0770000000E-01, 2.6459000000E-05)]), ("S", [(4.7390000000E+04, -8.7626800000E-05), (7.1080000000E+03, -6.7351730000E-04), (1.6180000000E+03, -3.5496648000E-03), (4.5840000000E+02, -1.4189954800E-02), (1.4930000000E+02, -4.7589463300E-02), (5.3590000000E+01, -1.1468634630E-01), (2.0700000000E+01, -2.0031679520E-01), (8.3840000000E+00, -3.4061321800E-02), (2.5420000000E+00, 5.6943290220E-01), (8.7870000000E-01, 5.4133828100E-01), (1.0770000000E-01, 2.1773755700E-02)]), ("S", [(4.7390000000E+04, 3.0646400000E-05), (7.1080000000E+03, 2.3462500000E-04), (1.6180000000E+03, 1.2445000000E-03), (4.5840000000E+02, 4.9418772000E-03), (1.4930000000E+02, 1.6824275800E-02), (5.3590000000E+01, 4.0336932300E-02), (2.0700000000E+01, 7.4289946300E-02), (8.3840000000E+00, 9.8580272000E-03), (2.5420000000E+00, -2.4211734790E-01), (8.7870000000E-01, -4.6282910270E-01), (1.0770000000E-01, 1.0945903944E+00)]), ("S", [(3.9990000000E-02, 1.0000000000E+00)]), ("P", [(1.7990000000E+02, 5.3815616000E-03), (4.2140000000E+01, 3.9241446800E-02), (1.3130000000E+01, 1.5744358280E-01), (4.6280000000E+00, 3.5853177270E-01), (1.6700000000E+00, 4.5722188430E-01), (5.8570000000E-01, 2.1591605640E-01), (1.3110000000E-01, 6.6494201000E-03)]), ("P", [(1.7990000000E+02, -1.7325893000E-03), (4.2140000000E+01, -1.2324491900E-02), (1.3130000000E+01, -5.2324738900E-02), (4.6280000000E+00, -1.1417508970E-01), (1.6700000000E+00, -1.7485117060E-01), (5.8570000000E-01, -2.4607847400E-02), (1.3110000000E-01, 1.0045720016E+00)]), ("P", [(4.1120000000E-02, 1.0000000000E+00)]), ("D", [(1.9320000000E-01, 1.0000000000E+00)])],
    13: [("S", [(6.4150000000E+04, 2.9024820000E-04), (9.6170000000E+03, 2.2506263000E-03), (2.1890000000E+03, 1.1645828900E-02), (6.2050000000E+02, 4.6737414800E-02), (2.0270000000E+02, 1.4629810730E-01), (7.3150000000E+01, 3.3028098460E-01), (2.8550000000E+01, 4.1585846240E-01), (1.1770000000E+01, 1.8925184520E-01), (3.3000000000E+00, 1.1588829300E-02), (1.1730000000E+00, -1.2838422000E-03), (1.7520000000E-01, 4.2588040000E-04)]), ("S", [(6.4150000000E+04, -7.5619700000E-05), (9.6170000000E+03, -5.8037070000E-04), (2.1890000000E+03, -3.0736082000E-03), (6.2050000000E+02, -1.2281145200E-02), (2.0270000000E+02, -4.1875620800E-02), (7.3150000000E+01, -1.0311864520E-01), (2.8550000000E+01, -1.9582876240E-01), (1.1770000000E+01, -8.2797575500E-02), (3.3000000000E+00, 5.3971918430E-01), (1.1730000000E+00, 5.7738301230E-01), (1.7520000000E-01, 2.8744356000E-02)]), ("S", [(6.4150000000E+04, 3.1679100000E-05), (9.6170000000E+03, 2.4283960000E-04), (2.1890000000E+03, 1.2891119000E-03), (6.2050000000E+02, 5.1447443000E-03), (2.0270000000E+02, 1.7675244600E-02), (7.3150000000E+01, 4.3760996200E-02), (2.8550000000E+01, 8.5946524200E-02), (1.1770000000E+01, 3.6843737100E-02), (3.3000000000E+00, -2.8731532220E-01), (1.1730000000E+00, -5.6398759370E-01), (1.7520000000E-01, 1.1221108339E+00)]), ("S", [(6.4730000000E-02, 1.0000000000E+00)]), ("P", [(2.5880000000E+02, 4.0674635000E-03), (6.0890000000E+01, 3.0673909800E-02), (1.9140000000E+01, 1.2911705030E-01), (6.8810000000E+00, 3.2075163070E-01), (2.5740000000E+00, 4.5370273230E-01), (9.5720000000E-01, 2.7499795240E-01), (2.0990000000E-01, 1.9075979700E-02)]), ("P", [(2.5880000000E+02, -1.6231043000E-03), (6.0890000000E+01, -1.1842527400E-02), (1.9140000000E+01, -5.3239906300E-02), (6.8810000000E+00, -1.2631065840E-01), (2.5740000000E+00, -2.1345259720E-01), (9.5720000000E-01, -5.6427951900E-02), (2.0990000000E-01, 1.0068174848E+00)]), ("P", [(5.9860000000E-02, 1.0000000000E+00)]), ("D", [(1.8900000000E-01, 1.0000000000E+00)])],
    14: [("S", [(7.8860000000E+04, 2.7044040000E-04), (1.1820000000E+04, 2.0971495000E-03), (2.6920000000E+03, 1.0850494100E-02), (7.6340000000E+02, 4.3674973700E-02), (2.4960000000E+02, 1.3765165660E-01), (9.0280000000E+01, 3.1664090970E-01), (3.5290000000E+01, 4.1857691480E-01), (1.4510000000E+01, 2.1020994840E-01), (4.0530000000E+00, 1.4495058500E-02), (1.4820000000E+00, -2.0358801000E-03), (2.5170000000E-01, 6.2417990000E-04)]), ("S", [(7.8860000000E+04, -7.2118600000E-05), (1.1820000000E+04, -5.5358800000E-04), (2.6920000000E+03, -2.9299628000E-03), (7.6340000000E+02, -1.1736305800E-02), (2.4960000000E+02, -4.0179796800E-02), (9.0280000000E+01, -1.0033206620E-01), (3.5290000000E+01, -1.9598704200E-01), (1.4510000000E+01, -1.0210018590E-01), (4.0530000000E+00, 5.2573887010E-01), (1.4820000000E+00, 5.9161803230E-01), (2.5170000000E-01, 3.3173635100E-02)]), ("S", [(7.8860000000E+04, 3.3894400000E-05), (1.1820000000E+04, 2.6043580000E-04), (2.6920000000E+03, 1.3772598000E-03), (7.6340000000E+02, 5.5347651000E-03), (2.4960000000E+02, 1.8983384300E-02), (9.0280000000E+01, 4.8075603300E-02), (3.5290000000E+01, 9.5943020500E-02), (1.4510000000E+01, 5.3274945300E-02), (4.0530000000E+00, -3.2592564890E-01), (1.4820000000E+00, -6.3513049530E-01), (2.5170000000E-01, 1.1407571659E+00)]), ("S", [(9.2430000000E-02, 1.0000000000E+00)]), ("P", [(3.1590000000E+02, 3.9256209000E-03), (7.4420000000E+01, 2.9873953400E-02), (2.3480000000E+01, 1.2718157510E-01), (8.4880000000E+00, 3.2086624100E-01), (3.2170000000E+00, 4.5532007640E-01), (1.2290000000E+00, 2.6849876860E-01), (2.9640000000E-01, 1.8829095600E-02)]), ("P", [(3.1590000000E+02, -1.6889270000E-03), (7.4420000000E+01, -1.2403302700E-02), (2.3480000000E+01, -5.6721485100E-02), (8.4880000000E+00, -1.3667230290E-01), (3.2170000000E+00, -2.3513279610E-01), (1.2290000000E+00, -3.9272625700E-02), (2.9640000000E-01, 1.0040817589E+00)]), ("P", [(8.7680000000E-02, 1.0000000000E+00)]), ("D", [(2.7500000000E-01, 1.0000000000E+00)])],
    15: [("S", [(9.4840000000E+04, 2.5550570000E-04), (1.4220000000E+04, 1.9819044000E-03), (3.2360000000E+03, 1.0275867100E-02), (9.1710000000E+02, 4.1481763500E-02), (2.9950000000E+02, 1.3198229320E-01), (1.0810000000E+02, 3.0865800830E-01), (4.2180000000E+01, 4.2064156010E-01), (1.7280000000E+01, 2.2287511770E-01), (4.8580000000E+00, 1.6403287900E-02), (1.8180000000E+00, -2.5425171000E-03), (3.3720000000E-01, 7.4804030000E-04)]), ("S", [(9.4840000000E+04, -6.9489400000E-05), (1.4220000000E+04, -5.3369550000E-04), (3.2360000000E+03, -2.8287659000E-03), (9.1710000000E+02, -1.1364856900E-02), (2.9950000000E+02, -3.9177612900E-02), (1.0810000000E+02, -9.9344062400E-02), (4.2180000000E+01, -1.9740210910E-01), (1.7280000000E+01, -1.1452299570E-01), (4.8580000000E+00, 5.1707341940E-01), (1.8180000000E+00, 6.0008115430E-01), (3.3720000000E-01, 3.6753047600E-02)]), ("S", [(9.4840000000E+04, 3.5328800000E-05), (1.4220000000E+04, 2.7203140000E-04), (3.2360000000E+03, 1.4373873000E-03), (9.1710000000E+02, 5.8120253000E-03), (2.9950000000E+02, 1.9992660400E-02), (1.0810000000E+02, 5.1729068700E-02), (4.2180000000E+01, 1.0420906330E-01), (1.7280000000E+01, 6.6184575200E-02), (4.8580000000E+00, -3.5733092640E-01), (1.8180000000E+00, -6.8754241860E-01), (3.3720000000E-01, 1.1534508600E+00)]), ("S", [(1.2320000000E-01, 1.0000000000E+00)]), ("P", [(3.7050000000E+02, 3.9492416000E-03), (8.7330000000E+01, 3.0243009000E-02), (2.7590000000E+01, 1.2952748460E-01), (1.0000000000E+01, 3.2752695250E-01), (3.8250000000E+00, 4.5689846910E-01), (1.4940000000E+00, 2.5303420180E-01), (3.9210000000E-01, 1.6876345300E-02)]), ("P", [(3.7050000000E+02, -1.7832784000E-03), (8.7330000000E+01, -1.3213005500E-02), (2.7590000000E+01, -6.0776217300E-02), (1.0000000000E+01, -1.4784924680E-01), (3.8250000000E+00, -2.5084713830E-01), (1.4940000000E+00, -1.6917820200E-02), (3.9210000000E-01, 9.9918596800E-01)]), ("P", [(1.1860000000E-01, 1.0000000000E+00)]), ("D", [(3.7300000000E-01, 1.0000000000E+00)])],
    16: [("S", [(1.1080000000E+05, 2.4763110000E-04), (1.6610000000E+04, 1.9202298000E-03), (3.7810000000E+03, 9.9617633000E-03), (1.0710000000E+03, 4.0296865900E-02), (3.4980000000E+02, 1.2860197650E-01), (1.2630000000E+02, 3.0347522490E-01), (4.9260000000E+01, 4.2142536890E-01), (2.0160000000E+01, 2.3077736880E-01), (5.7200000000E+00, 1.7896818400E-02), (2.1820000000E+00, -2.9751132000E-03), (4.3270000000E-01, 8.4950860000E-04)]), ("S", [(1.1080000000E+05, -6.8497700000E-05), (1.6610000000E+04, -5.2609750000E-04), (3.7810000000E+03, -2.7883175000E-03), (1.0710000000E+03, -1.1231295000E-02), (3.4980000000E+02, -3.8766716300E-02), (1.2630000000E+02, -9.9203906700E-02), (4.9260000000E+01, -1.9914060770E-01), (2.0160000000E+01, -1.2298981360E-01), (5.7200000000E+00, 5.1165397530E-01), (2.1820000000E+00, 6.0529811630E-01), (4.3270000000E-01, 3.9556239900E-02)]), ("S", [(1.1080000000E+05, 3.6677100000E-05), (1.6610000000E+04, 2.8277040000E-04), (3.7810000000E+03, 1.4913933000E-03), (1.0710000000E+03, 6.0608745000E-03), (3.4980000000E+02, 2.0812550700E-02), (1.2630000000E+02, 5.4604688400E-02), (4.9260000000E+01, 1.1051394950E-01), (2.0160000000E+01, 7.6135021200E-02), (5.7200000000E+00, -3.8224110930E-01), (2.1820000000E+00, -7.2384167260E-01), (4.3270000000E-01, 1.1659169996E+00)]), ("S", [(1.5700000000E-01, 1.0000000000E+00)]), ("P", [(3.9970000000E+02, 4.4753833000E-03), (9.4190000000E+01, 3.4170596200E-02), (2.9750000000E+01, 1.4424913980E-01), (1.0770000000E+01, 3.5392588940E-01), (4.1190000000E+00, 4.5908226220E-01), (1.6250000000E+00, 2.0638176920E-01), (4.7260000000E-01, 1.0214039100E-02)]), ("P", [(3.9970000000E+02, -2.0164240000E-03), (9.4190000000E+01, -1.5015317200E-02), (2.9750000000E+01, -6.7800870700E-02), (1.0770000000E+01, -1.6211475660E-01), (4.1190000000E+00, -2.5670200660E-01), (1.6250000000E+00, 5.2366557200E-02), (4.7260000000E-01, 9.7407270520E-01)]), ("P", [(1.4070000000E-01, 1.0000000000E+00)]), ("D", [(4.7900000000E-01, 1.0000000000E+00)])],
    17: [("S", [(1.2790000000E+05, 2.4114870000E-04), (1.9170000000E+04, 1.8709168000E-03), (4.3630000000E+03, 9.7080975000E-03), (1.2360000000E+03, 3.9314601600E-02), (4.0360000000E+02, 1.2592976280E-01), (1.4570000000E+02, 2.9933568220E-01), (5.6810000000E+01, 4.2187850520E-01), (2.3230000000E+01, 2.3719678610E-01), (6.6440000000E+00, 1.9152759700E-02), (2.5750000000E+00, -3.3478605000E-03), (5.3710000000E-01, 9.2986650000E-04)]), ("S", [(1.2790000000E+05, -6.7686200000E-05), (1.9170000000E+04, -5.2025270000E-04), (4.3630000000E+03, -2.7567405000E-03), (1.2360000000E+03, -1.1119859200E-02), (4.0360000000E+02, -3.8474810500E-02), (1.4570000000E+02, -9.9182958800E-02), (5.6810000000E+01, -2.0078096800E-01), (2.3230000000E+01, -1.2991762480E-01), (6.6440000000E+00, 5.0789732800E-01), (2.5750000000E+00, 6.0887203410E-01), (5.3710000000E-01, 4.2027000200E-02)]), ("S", [(1.2790000000E+05, 3.7727400000E-05), (1.9170000000E+04, 2.9134550000E-04), (4.3630000000E+03, 1.5343024000E-03), (1.2360000000E+03, 6.2554501000E-03), (4.0360000000E+02, 2.1485486900E-02), (1.4570000000E+02, 5.6985552500E-02), (5.6810000000E+01, 1.1586478290E-01), (2.3230000000E+01, 8.4709741000E-02), (6.6440000000E+00, -4.0364106840E-01), (2.5750000000E+00, -7.5234173440E-01), (5.3710000000E-01, 1.1750870665E+00)]), ("S", [(1.9380000000E-01, 1.0000000000E+00)]), ("P", [(4.1760000000E+02, 5.2607254000E-03), (9.8330000000E+01, 3.9840057100E-02), (3.1040000000E+01, 1.6468334450E-01), (1.1190000000E+01, 3.8738867540E-01), (4.2490000000E+00, 4.5715068250E-01), (1.6240000000E+00, 1.5166210330E-01), (5.3220000000E-01, 1.8164626000E-03)]), ("P", [(4.1760000000E+02, -2.2919790000E-03), (9.8330000000E+01, -1.7208759800E-02), (3.1040000000E+01, -7.5043672900E-02), (1.1190000000E+01, -1.7725041750E-01), (4.2490000000E+00, -2.4469114420E-01), (1.6240000000E+00, 1.4281709330E-01), (5.3220000000E-01, 9.2503549400E-01)]), ("P", [(1.6200000000E-01, 1.0000000000E+00)]), ("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    18: [("S", [(1.4570000000E+05, 2.3669520000E-04), (2.1840000000E+04, 1.8351932000E-03), (4.9720000000E+03, 9.5284088000E-03), (1.4080000000E+03, 3.8627524800E-02), (4.5970000000E+02, 1.2407851000E-01), (1.6590000000E+02, 2.9646505050E-01), (6.4690000000E+01, 4.2205953010E-01), (2.6440000000E+01, 2.4170614940E-01), (7.6280000000E+00, 2.0050497600E-02), (2.9960000000E+00, -3.6099276000E-03), (6.5040000000E-01, 9.7558740000E-04)]), ("S", [(1.4570000000E+05, -6.7285900000E-05), (2.1840000000E+04, -5.1694660000E-04), (4.9720000000E+03, -2.7399000000E-03), (1.4080000000E+03, -1.1066972700E-02), (4.5970000000E+02, -3.8365080100E-02), (1.6590000000E+02, -9.9456799400E-02), (6.4690000000E+01, -2.0247095750E-01), (2.6440000000E+01, -1.3519598210E-01), (7.6280000000E+00, 5.0565398890E-01), (2.9960000000E+00, 6.1103583130E-01), (6.5040000000E-01, 4.4162213000E-02)]), ("S", [(1.4570000000E+05, 3.8726000000E-05), (2.1840000000E+04, 2.9913440000E-04), (4.9720000000E+03, 1.5741299000E-03), (1.4080000000E+03, 6.4356266000E-03), (4.5970000000E+02, 2.2109798600E-02), (1.6590000000E+02, 5.9134639600E-02), (6.4690000000E+01, 1.2057730550E-01), (2.6440000000E+01, 9.1888629200E-02), (7.6280000000E+00, -4.2279589160E-01), (2.9960000000E+00, -7.7468939300E-01), (6.5040000000E-01, 1.1819475553E+00)]), ("S", [(2.3370000000E-01, 1.0000000000E+00)]), ("P", [(4.5370000000E+02, 5.7065768000E-03), (1.0680000000E+02, 4.3053746700E-02), (3.3730000000E+01, 1.7662278010E-01), (1.2130000000E+01, 4.0693622080E-01), (4.5940000000E+00, 4.5263044260E-01), (1.6780000000E+00, 1.2282309980E-01), (5.9090000000E-01, -4.4607626000E-03)]), ("P", [(4.5370000000E+02, -2.4037936000E-03), (1.0680000000E+02, -1.8211405100E-02), (3.3730000000E+01, -7.7922831100E-02), (1.2130000000E+01, -1.8514095640E-01), (4.5940000000E+00, -2.2685927950E-01), (1.6780000000E+00, 2.1310279630E-01), (5.9090000000E-01, 8.7455711820E-01)]), ("P", [(1.8520000000E-01, 1.0000000000E+00)]), ("D", [(7.3800000000E-01, 1.0000000000E+00)])]
    
    }





CC_PVTZ = {

    1 : [('S', [(33.87, 0.0254948632), (5.095, 0.1903627659), (1.159, 0.8521620222)]), ('S', [(0.3258, 1.0)]), ('S', [(0.1027, 1.0)]), ('P', [(1.407, 1.0)]), ('P', [(0.388, 1.0)]), ('D', [(1.057, 1.0)])],
    2 : [('S', [(234.0, 0.0072914569), (35.16, 0.0550537407), (7.989, 0.2564777708), (2.212, 0.7667726492)]), ('S', [(0.6669, 1.0)]), ('S', [(0.2089, 1.0)]), ('P', [(3.044, 1.0)]), ('P', [(0.758, 1.0)]), ('D', [(1.965, 1.0)])],
    3 : [('S', [(5988.0, 0.0001331958), (898.9, 0.001026509), (205.9, 0.0052797616), (59.24, 0.0209598123), (19.87, 0.0664376677), (7.406, 0.1660190588), (2.93, 0.3155018081), (1.189, 0.394102356), (0.4798, 0.1911510044)]), ('S', [(5988.0, -7.76563e-05), (898.9, -0.0005953647), (205.9, -0.003032292), (59.24, -0.0122992723), (19.87, -0.0388983901), (7.406, -0.103900377), (2.93, -0.2068466914), (1.189, -0.3669702002), (0.4798, -0.4148656226)]), ('S', [(0.07509, 1.0)]), ('S', [(0.02832, 1.0)]), ('P', [(3.266, 0.0355446886), (0.6511, 0.1957964548), (0.1696, 0.8639954123)]), ('P', [(0.05578, 1.0)]), ('P', [(0.0205, 1.0)]), ('D', [(0.19, 1.0)]), ('D', [(0.0725, 1.0)]), ('F', [(0.127, 1.0)])],
    4 : [('S', [(6863.0, 0.0002379364), (1030.0, 0.0018409828), (234.7, 0.0095295563), (66.56, 0.038268448), (21.69, 0.1209493469), (7.734, 0.2844772194), (2.916, 0.4309109713), (1.13, 0.2684628867), (0.1101, -0.0073346934)]), ('S', [(6863.0, -7.64231e-05), (1030.0, -0.0005918348), (234.7, -0.0030853608), (66.56, -0.0124622984), (21.69, -0.0411014136), (7.734, -0.1033275959), (2.916, -0.2035982675), (1.13, -0.2415467836), (0.1101, 1.0262752471)]), ('S', [(0.2577, 1.0)]), ('S', [(0.04409, 1.0)]), ('P', [(7.436, 0.036548778), (1.577, 0.2139751205), (0.4352, 0.844884103)]), ('P', [(0.1438, 1.0)]), ('P', [(0.04994, 1.0)]), ('D', [(0.3493, 1.0)]), ('D', [(0.1724, 1.0)]), ('F', [(0.3423, 1.0)])],
    5 : [('S', [(5473.0, 0.00056594), (820.9, 0.0043755832), (186.8, 0.0223816535), (52.83, 0.086105481), (17.08, 0.2432593791), (5.999, 0.4436480363), (2.208, 0.348695536), (0.2415, -0.0097331488)]), ('S', [(5473.0, -0.0001965094), (820.9, -0.0015229479), (186.8, -0.0078673943), (52.83, -0.0310256767), (17.08, -0.0941122135), (5.999, -0.2088000142), (2.208, -0.2909462086), (0.2415, 1.0456774192)]), ('S', [(0.5879, 1.0)]), ('S', [(0.0861, 1.0)]), ('P', [(12.05, 0.0389268857), (2.613, 0.2370866334), (0.7475, 0.8227970898)]), ('P', [(0.2385, 1.0)]), ('P', [(0.07698, 1.0)]), ('D', [(0.661, 1.0)]), ('D', [(0.199, 1.0)]), ('F', [(0.49, 1.0)])],
    6 : [('S', [(8236.0, 0.0005424302), (1235.0, 0.0041964279), (280.8, 0.0215409141), (79.27, 0.0836149496), (25.59, 0.2398716189), (8.997, 0.4437518201), (3.319, 0.3535796965), (0.3643, -0.0091763661)]), ('S', [(8236.0, -0.0001963922), (1235.0, -0.0015259503), (280.8, -0.007890449), (79.27, -0.0315148705), (25.59, -0.0969100083), (8.997, -0.2205415263), (3.319, -0.2960691129), (0.3643, 1.0405034329)]), ('S', [(0.9059, 1.0)]), ('S', [(0.1285, 1.0)]), ('P', [(18.71, 0.0394263872), (4.133, 0.2440889849), (1.2, 0.8154920089)]), ('P', [(0.3827, 1.0)]), ('P', [(0.1209, 1.0)]), ('D', [(1.097, 1.0)]), ('D', [(0.318, 1.0)]), ('F', [(0.761, 1.0)])],
    7 : [('S', [(11420.0, 0.0005349809), (1712.0, 0.0041376631), (389.3, 0.0212509149), (110.0, 0.0825762988), (35.57, 0.2384132727), (12.54, 0.4434316661), (4.644, 0.3554319088), (0.5118, -0.0087029018)]), ('S', [(11420.0, -0.000198777), (1712.0, -0.0015470035), (389.3, -0.0079925633), (110.0, -0.0320255651), (35.57, -0.0991102048), (12.54, -0.2282927747), (4.644, -0.2981827627), (0.5118, 1.0370005182)]), ('S', [(1.293, 1.0)]), ('S', [(0.1787, 1.0)]), ('P', [(26.63, 0.0398072123), (5.948, 0.2490026605), (1.742, 0.8104797267)]), ('P', [(0.555, 1.0)]), ('P', [(0.1725, 1.0)]), ('D', [(1.654, 1.0)]), ('D', [(0.469, 1.0)]), ('F', [(1.093, 1.0)])],
    8 : [('S', [(15330.0, 0.0005201983), (2299.0, 0.0040233448), (522.4, 0.0207290833), (147.3, 0.0810823271), (47.55, 0.2362263521), (16.76, 0.4435182094), (6.207, 0.3586705887), (0.6882, -0.0083497972)]), ('S', [(15330.0, -0.000197236), (2299.0, -0.0015350107), (522.4, -0.0079511839), (147.3, -0.0321134529), (47.55, -0.100269643), (16.76, -0.2340471118), (6.207, -0.3014109278), (0.6882, 1.0349196495)]), ('S', [(1.752, 1.0)]), ('S', [(0.2384, 1.0)]), ('P', [(34.46, 0.0411634896), (7.749, 0.2577628359), (2.28, 0.8024192744)]), ('P', [(0.7156, 1.0)]), ('P', [(0.214, 1.0)]), ('D', [(2.314, 1.0)]), ('D', [(0.645, 1.0)]), ('F', [(1.428, 1.0)])],
    9 : [('S', [(19500.0, 0.0005193716), (2923.0, 0.0040187274), (664.5, 0.0206929119), (187.5, 0.0809379686), (60.62, 0.236062075), (21.42, 0.4434347595), (7.95, 0.3585036736), (0.8815, -0.0080845772)]), ('S', [(19500.0, -0.0001995912), (2923.0, -0.0015557874), (664.5, -0.0080467647), (187.5, -0.0325589465), (60.62, -0.1017658994), (21.42, -0.2388440797), (7.95, -0.3015737026), (0.8815, 1.0321472646)]), ('S', [(2.257, 1.0)]), ('S', [(0.3041, 1.0)]), ('P', [(43.88, 0.0419046207), (9.926, 0.2626978417), (2.93, 0.7977593735)]), ('P', [(0.9132, 1.0)]), ('P', [(0.2672, 1.0)]), ('D', [(3.107, 1.0)]), ('D', [(0.855, 1.0)]), ('F', [(1.917, 1.0)])],
    10 : [('S', [(24350.0, 0.000514506), (3650.0, 0.0039776845), (829.6, 0.0204951706), (234.0, 0.0803715702), (75.61, 0.23539775), (26.73, 0.4435020864), (9.927, 0.3593772874), (1.102, -0.0078354543)]), ('S', [(24350.0, -0.0002006849), (3650.0, -0.0015561583), (829.6, -0.008056308), (234.0, -0.0327099371), (75.61, -0.1026707322), (26.73, -0.2423661267), (9.927, -0.3022348526), (1.102, 1.0303570659)]), ('S', [(2.836, 1.0)]), ('S', [(0.3782, 1.0)]), ('P', [(54.7, 0.042367017), (12.43, 0.2659357231), (3.679, 0.7946279757)]), ('P', [(1.143, 1.0)]), ('P', [(0.33, 1.0)]), ('D', [(4.014, 1.0)]), ('D', [(1.096, 1.0)]), ('F', [(2.544, 1.0)])],
    11 : [('S', [(423000.0, 1.80624e-05), (63340.0, 0.0001404348), (14410.0, 0.0007384633), (4077.0, 0.0031119264), (1328.0, 0.0112084833), (478.6, 0.0352840067), (186.2, 0.0959929829), (76.92, 0.2137423099), (33.32, 0.3486999254), (15.0, 0.3245771004), (6.869, 0.1126368521), (2.683, 0.0070682117), (1.109, 0.0005980305), (0.06015, -5.3089e-06)]), ('S', [(423000.0, -5.5313e-06), (63340.0, -4.31113e-05), (14410.0, -0.0002260912), (4077.0, -0.0009588986), (1328.0, -0.0034550979), (478.6, -0.0111218682), (186.2, -0.0311229693), (76.92, -0.0760685644), (33.32, -0.1461708439), (15.0, -0.2039018375), (6.869, -0.0550925475), (2.683, 0.4241761251), (1.109, 0.7046404747), (0.06015, 0.0051058495)]), ('S', [(423000.0, 1.097e-06), (63340.0, 8.5337e-06), (14410.0, 4.48799e-05), (4077.0, 0.0001896702), (1328.0, 0.0006868365), (478.6, 0.0022001973), (186.2, 0.0062144346), (76.92, 0.0151230677), (33.32, 0.0297587029), (15.0, 0.0416083116), (6.869, 0.0126329093), (2.683, -0.1016872742), (1.109, -0.1914670301), (0.06015, 1.0364246459)]), ('S', [(0.454, 1.0)]), ('S', [(0.02382, 1.0)]), ('P', [(243.3, 0.0022442788), (57.39, 0.0174024818), (18.1, 0.0774248764), (6.575, 0.2191370293), (2.521, 0.3785825168), (0.9607, 0.3949651356), (0.3512, 0.160449648), (0.09827, 0.002333483)]), ('P', [(243.3, -0.0011901332), (57.39, -0.0093260753), (18.1, -0.0414969333), (6.575, -0.1205041925), (2.521, -0.2056662964), (0.9607, -0.2409029645), (0.3512, -0.102815489), (0.09827, 0.9776654272)]), ('P', [(0.03734, 1.0)]), ('P', [(0.015, 1.0)]), ('D', [(0.1473, 1.0)]), ('D', [(0.0623, 1.0)]), ('F', [(0.1284, 1.0)])],
    12 : [('S', [(164900.0, 7.29888e-05), (24710.0, 0.0005666203), (5628.0, 0.0029625245), (1596.0, 0.0122955131), (521.0, 0.0427300129), (188.0, 0.1230061283), (73.01, 0.2748166474), (29.9, 0.4017955537), (12.54, 0.2646822135), (4.306, 0.033259342), (1.826, -0.0044131035), (0.7417, 0.0020601249), (0.07612, 0.0007081554)]), ('S', [(164900.0, -1.8621e-05), (24710.0, -0.000145028), (5628.0, -0.0007566823), (1596.0, -0.0031775481), (521.0, -0.0111657404), (188.0, -0.0339636353), (73.01, -0.0834740692), (29.9, -0.1610103792), (12.54, -0.154515955), (4.306, 0.1928811646), (1.826, 0.5861394701), (0.7417, 0.3759903734), (0.07612, -0.0120614778)]), ('S', [(164900.0, 6.6222e-06), (24710.0, 5.15375e-05), (5628.0, 0.000269236), (1596.0, 0.0011293878), (521.0, 0.0039811339), (188.0, 0.012117785), (73.01, 0.0301010242), (29.9, 0.0588734315), (12.54, 0.0589867909), (4.306, -0.0818760687), (1.826, -0.2817074475), (0.7417, -0.4058341331), (0.07612, 1.0203185594)]), ('S', [(0.1457, 1.0)]), ('S', [(0.0331, 1.0)]), ('P', [(316.9, 0.0020771864), (74.86, 0.0163015473), (23.72, 0.0739361332), (8.669, 0.2144897235), (3.363, 0.3824976821), (1.31, 0.398536093), (0.4911, 0.1530154876), (0.2364, -0.0043793349)]), ('P', [(316.9, -0.0017376238), (74.86, -0.0136360419), (23.72, -0.0627748868), (8.669, -0.1845656504), (3.363, -0.3372558658), (1.31, -0.3712297695), (0.4911, -0.1980613024), (0.2364, 0.9329964976)]), ('P', [(0.08733, 1.0)]), ('P', [(0.03237, 1.0)]), ('D', [(0.2998, 1.0)]), ('D', [(0.1285, 1.0)]), ('F', [(0.2417, 1.0)])],
    13 : [('S', [(205500.0, 6.78764e-05), (30780.0, 0.0005270928), (7006.0, 0.0027617355), (1985.0, 0.0114715766), (649.1, 0.0398145539), (235.0, 0.1150277328), (91.62, 0.2608591804), (37.67, 0.3963437316), (15.91, 0.2845666521), (5.85, 0.0444535592), (2.542, -0.0048978577), (1.057, 0.0026122514), (0.1455, 0.000721991)]), ('S', [(205500.0, -1.80366e-05), (30780.0, -0.0001402976), (7006.0, -0.0007351678), (1985.0, -0.0030795628), (649.1, -0.0108411457), (235.0, -0.0328612067), (91.62, -0.0821318996), (37.67, -0.1603398228), (15.91, -0.1721837443), (5.85, 0.1297483091), (2.542, 0.5741919236), (1.057, 0.4464867983), (0.1455, -0.0117153789)]), ('S', [(205500.0, 6.7989e-06), (30780.0, 5.28409e-05), (7006.0, 0.0002772793), (1985.0, 0.0011600742), (649.1, 0.0040980467), (235.0, 0.0124287603), (91.62, 0.0314230149), (37.67, 0.0622227541), (15.91, 0.0700218805), (5.85, -0.059151944), (2.542, -0.2923286986), (1.057, -0.4610354678), (0.1455, 1.0896626855)]), ('S', [(0.2931, 1.0)]), ('S', [(0.0565, 1.0)]), ('P', [(444.4, 0.0018468658), (105.1, 0.0148269109), (33.47, 0.0694722924), (12.33, 0.213145283), (4.869, 0.4089457793), (1.961, 0.4634057775), (0.1888, 0.0110789031)]), ('P', [(444.4, -0.0005663664), (105.1, -0.0047927158), (33.47, -0.0214919407), (12.33, -0.0720578736), (4.869, -0.126800828), (1.961, -0.1923034005), (0.1888, 0.9957918001)]), ('P', [(0.7834, 1.0)]), ('P', [(0.05557, 1.0)]), ('D', [(0.109, 1.0)]), ('D', [(0.333, 1.0)]), ('F', [(0.244, 1.0)])],
    14 : [('S', [(254900.0, 6.25027e-05), (38190.0, 0.0004854954), (8690.0, 0.0025448582), (2462.0, 0.0105853445), (804.8, 0.0368743263), (291.3, 0.1074662534), (113.6, 0.2479065957), (46.75, 0.3908806374), (19.82, 0.3019901808), (7.708, 0.0559169677), (3.34, -0.0040235828), (1.402, 0.002579994), (0.207, 0.0006078579)]), ('S', [(254900.0, -1.71868e-05), (38190.0, -0.0001335831), (8690.0, -0.0007012599), (2462.0, -0.0029350566), (804.8, -0.0103873711), (291.3, -0.0315878312), (113.6, -0.0802940449), (46.75, -0.1593327335), (19.82, -0.1863282663), (7.708, 0.0824595139), (3.34, 0.5655312053), (1.402, 0.4959845477), (0.207, -0.0110531686)]), ('S', [(254900.0, 6.9095e-06), (38190.0, 5.36641e-05), (8690.0, 0.0002820727), (2462.0, 0.0011793739), (804.8, 0.0041874944), (291.3, 0.012746544), (113.6, 0.0327686718), (46.75, 0.0660252548), (19.82, 0.0809443169), (7.708, -0.0404262851), (3.34, -0.3085511943), (1.402, -0.5160350549), (0.207, 1.1041707515)]), ('S', [(0.4387, 1.0)]), ('S', [(0.07944, 1.0)]), ('P', [(481.5, 0.002084375), (113.9, 0.0166658826), (36.23, 0.0774935539), (13.34, 0.2312376018), (5.252, 0.4236736703), (2.12, 0.4273280693), (0.2528, 0.0042940002)]), ('P', [(481.5, -0.0007329469), (113.9, -0.006075562), (36.23, -0.0276487487), (13.34, -0.0884879337), (5.252, -0.1546506694), (2.12, -0.2028292381), (0.2528, 0.9982905753)]), ('P', [(0.8561, 1.0)]), ('P', [(0.07889, 1.0)]), ('D', [(0.159, 1.0)]), ('D', [(0.481, 1.0)]), ('F', [(0.336, 1.0)])],
    15 : [('S', [(312400.0, 5.76896e-05), (46800.0, 0.0004482466), (10650.0, 0.0023491311), (3018.0, 0.0097815718), (986.8, 0.0341429365), (357.4, 0.100192956), (139.6, 0.2343461686), (57.63, 0.3823918499), (24.6, 0.3180529418), (10.12, 0.0707709991), (4.283, -0.0018177896), (1.805, 0.0021615617), (0.2782, 0.0004322494)]), ('S', [(312400.0, -1.63802e-05), (46800.0, -0.0001272335), (10650.0, -0.0006686252), (3018.0, -0.0027953037), (986.8, -0.0099282274), (357.4, -0.0302445665), (139.6, -0.0778847053), (57.63, -0.1567245715), (24.6, -0.1980427537), (10.12, 0.0379712515), (4.283, 0.5527515441), (1.805, 0.5426114525), (0.2782, -0.0096759429)]), ('S', [(312400.0, 6.8657e-06), (46800.0, 5.3282e-05), (10650.0, 0.000280421), (3018.0, 0.0011707898), (986.8, 0.0041735911), (357.4, 0.0127204967), (139.6, 0.0331527664), (57.63, 0.0676712468), (24.6, 0.0898310189), (10.12, -0.0203052324), (4.283, -0.313280409), (1.805, -0.5636879055), (0.2782, 1.1174940737)]), ('S', [(0.6158, 1.0)]), ('S', [(0.1055, 1.0)]), ('P', [(504.9, 0.0024336604), (119.4, 0.0193055595), (37.96, 0.0884731069), (13.95, 0.2547019811), (5.457, 0.4401992427), (2.177, 0.3836320063), (0.2877, -0.0039463371)]), ('P', [(504.9, -0.0009542989), (119.4, -0.0076640257), (37.96, -0.0354659253), (13.95, -0.1061647164), (5.457, -0.187210586), (2.177, -0.1814958198), (0.2877, 0.9916726446)]), ('P', [(0.801, 1.0)]), ('P', [(0.09714, 1.0)]), ('D', [(0.216, 1.0)]), ('D', [(0.652, 1.0)]), ('F', [(0.452, 1.0)])],
    16 : [('S', [(374100.0, 5.42093e-05), (56050.0, 0.0004208183), (12760.0, 0.0022067874), (3615.0, 0.0091917777), (1183.0, 0.0321094974), (428.8, 0.0946600377), (167.8, 0.2236104825), (69.47, 0.3743603245), (29.84, 0.3290792768), (12.72, 0.0846964074), (5.244, 0.0004408125), (2.219, 0.0016481261), (0.349, 0.0003012797)]), ('S', [(374100.0, -1.5779e-05), (56050.0, -0.0001223657), (12760.0, -0.0006440453), (3615.0, -0.0026892484), (1183.0, -0.009569415), (428.8, -0.0291750327), (167.8, -0.0758238087), (69.47, -0.1542118652), (29.84, -0.2055084062), (12.72, 0.0086267511), (5.244, 0.5440217687), (2.219, 0.5709563754), (0.349, -0.0096680253)]), ('S', [(374100.0, 6.8239e-06), (56050.0, 5.28789e-05), (12760.0, 0.0002786737), (3615.0, 0.0011624074), (1183.0, 0.0041499919), (428.8, 0.0126650737), (167.8, 0.0332945444), (69.47, 0.0687490087), (29.84, 0.0961017734), (12.72, -0.0056701232), (5.244, -0.3207648188), (2.219, -0.5989476414), (0.349, 1.120107736)]), ('S', [(0.7767, 1.0)]), ('S', [(0.1322, 1.0)]), ('P', [(574.4, 0.002496923), (135.8, 0.0198707511), (43.19, 0.0912549164), (15.87, 0.2624622005), (6.208, 0.4472908166), (2.483, 0.3658365683), (0.3229, -0.0051839928)]), ('P', [(574.4, -0.0010640239), (135.8, -0.0084744487), (43.19, -0.0399195571), (15.87, -0.1175716038), (6.208, -0.212591327), (2.483, -0.1663549097), (0.3229, 0.9770148029)]), ('P', [(0.8688, 1.0)]), ('P', [(0.1098, 1.0)]), ('D', [(0.269, 1.0)]), ('D', [(0.819, 1.0)]), ('F', [(0.557, 1.0)])],
    17 : [('S', [(456100.0, 4.92942e-05), (68330.0, 0.0003830071), (15550.0, 0.002008425), (4405.0, 0.0083851001), (1439.0, 0.0294686133), (520.4, 0.0878274731), (203.1, 0.2114608968), (83.96, 0.3653430893), (36.2, 0.3408644903), (15.83, 0.1021271547), (6.334, 0.0031165716), (2.694, 0.0010574495), (0.4313, 0.0001561271)]), ('S', [(456100.0, -1.47034e-05), (68330.0, -0.0001140508), (15550.0, -0.0006007529), (4405.0, -0.0025104062), (1439.0, -0.0089928112), (520.4, -0.027602721), (203.1, -0.0729687441), (83.96, -0.1508295563), (36.2, -0.2119006748), (15.83, -0.0208012833), (6.334, 0.5312863055), (2.694, 0.5993208817), (0.4313, -0.0088780471)]), ('S', [(456100.0, 6.4955e-06), (68330.0, 5.03429e-05), (15550.0, 0.000265538), (4405.0, 0.0011083303), (1439.0, 0.0039838068), (520.4, 0.0122375442), (203.1, 0.0327244651), (83.96, 0.0686349823), (36.2, 0.1011327147), (15.83, 0.0093581476), (6.334, -0.3204597407), (2.694, -0.6298715001), (0.4313, 1.1261538336)]), ('S', [(0.9768, 1.0)]), ('S', [(0.1625, 1.0)]), ('P', [(663.3, 0.0024738409), (156.8, 0.0197690804), (49.98, 0.091062898), (18.42, 0.2634052893), (7.24, 0.449530829), (2.922, 0.3604399212), (0.3818, -0.0047164691)]), ('P', [(663.3, -0.0011177497), (156.8, -0.0089030736), (49.98, -0.0423241575), (18.42, -0.1248048286), (7.24, -0.2297219068), (2.922, -0.1624390804), (0.3818, 0.9678160112)]), ('P', [(1.022, 1.0)]), ('P', [(0.1301, 1.0)]), ('D', [(1.046, 1.0)]), ('D', [(0.344, 1.0)]), ('F', [(0.706, 1.0)])],
    18 : [('S', [(545000.0, 4.55817e-05), (81640.0, 0.0003540993), (18580.0, 0.0018579246), (5261.0, 0.00776832), (1717.0, 0.0274225293), (619.9, 0.082381585), (241.6, 0.2012250782), (99.79, 0.3567722736), (43.15, 0.3495544502), (19.14, 0.1182631074), (7.488, 0.005601763), (3.205, 0.0004834612), (0.5204, 2.92018e-05)]), ('S', [(545000.0, -1.38905e-05), (81640.0, -0.0001076794), (18580.0, -0.0005678217), (5261.0, -0.0023738198), (1717.0, -0.008543815), (619.9, -0.0263551299), (241.6, -0.0705294555), (99.79, -0.1479021546), (43.15, -0.2161887709), (19.14, -0.0442642836), (7.488, 0.519676504), (3.205, 0.6210470895), (0.5204, -0.0078008624)]), ('S', [(545000.0, 6.2421e-06), (81640.0, 4.83489e-05), (18580.0, 0.0002553088), (5261.0, 0.0010660528), (1717.0, 0.003849992), (619.9, 0.0118844979), (241.6, 0.0321689111), (99.79, 0.0684279475), (43.15, 0.104842346), (19.14, 0.0217861552), (7.488, -0.3198010291), (3.205, -0.6551182396), (0.5204, 1.1307330489)]), ('S', [(1.196, 1.0)]), ('S', [(0.1954, 1.0)]), ('P', [(761.8, 0.0024360729), (180.2, 0.0195521334), (57.5, 0.0905454602), (21.24, 0.2635511917), (8.388, 0.4509874399), (3.416, 0.3572950154), (0.4523, -0.0053854178)]), ('P', [(761.8, -0.0011415941), (180.2, -0.0091147568), (57.5, -0.0437148745), (21.24, -0.1295559654), (8.388, -0.2414777404), (3.416, -0.1595960611), (0.4523, 0.9623486717)]), ('P', [(1.206, 1.0)]), ('P', [(0.1545, 1.0)]), ('D', [(0.41, 1.0)]), ('D', [(1.254, 1.0)]), ('F', [(0.89, 1.0)])],

    }






CC_PVQZ = {

    1 : [('S', [(82.64, 0.0229507578), (12.41, 0.1755401181), (2.824, 0.864703551)]), ('S', [(0.7977, 1.0)]), ('S', [(0.2581, 1.0)]), ('S', [(0.08989, 1.0)]), ('P', [(2.292, 1.0)]), ('P', [(0.838, 1.0)]), ('P', [(0.292, 1.0)]), ('D', [(2.062, 1.0)]), ('D', [(0.662, 1.0)]), ('F', [(1.397, 1.0)])],
    2 : [('S', [(528.5, 0.0058740153), (79.31, 0.0450799432), (18.05, 0.2248060657), (5.085, 0.7985036466)]), ('S', [(1.609, 1.0)]), ('S', [(0.5363, 1.0)]), ('S', [(0.1833, 1.0)]), ('P', [(5.994, 1.0)]), ('P', [(1.745, 1.0)]), ('P', [(0.56, 1.0)]), ('D', [(4.299, 1.0)]), ('D', [(1.223, 1.0)]), ('F', [(2.68, 1.0)])],
    3 : [('S', [(6601.0, 0.0001173021), (989.7, 0.000913352), (225.7, 0.0047402069), (64.29, 0.0192465633), (21.18, 0.0632097763), (7.724, 0.1636293744), (3.003, 0.315639828), (1.212, 0.3949530736), (0.493, 0.1974264077)]), ('S', [(6601.0, -6.58485e-05), (989.7, -0.0005194713), (225.7, -0.0027107624), (64.29, -0.0110479113), (21.18, -0.0370324525), (7.724, -0.0991165927), (3.003, -0.2098334923), (1.212, -0.3434912701), (0.493, -0.4429810042)]), ('S', [(0.09515, 1.0)]), ('S', [(0.04791, 1.0)]), ('S', [(0.0222, 1.0)]), ('P', [(6.25, 0.0364764639), (1.37, 0.207963216), (0.3672, 0.8516629861)]), ('P', [(0.1192, 1.0)]), ('P', [(0.04474, 1.0)]), ('P', [(0.01795, 1.0)]), ('D', [(0.3352, 1.0)]), ('D', [(0.1482, 1.0)]), ('D', [(0.0656, 1.0)]), ('F', [(0.228, 1.0)]), ('F', [(0.1135, 1.0)]), ('G', [(0.2319, 1.0)])],
    4 : [('S', [(14630.0, 9.24237e-05), (2191.0, 0.000716284), (498.2, 0.0037522032), (140.9, 0.0155392447), (45.86, 0.0531175346), (16.47, 0.1463650582), (6.319, 0.3040751313), (2.535, 0.4068011119), (1.035, 0.2234113017)]), ('S', [(14630.0, -5.62045e-05), (2191.0, -0.0004297992), (498.2, -0.0022448742), (140.9, -0.0094456636), (45.86, -0.0324432261), (16.47, -0.0945855758), (6.319, -0.210799969), (2.535, -0.3875829857), (1.035, -0.4007116977)]), ('S', [(0.2528, 1.0)]), ('S', [(0.1052, 1.0)]), ('S', [(0.04261, 1.0)]), ('P', [(14.03, 0.0334330386), (3.168, 0.2090156248), (0.9024, 0.8463721749)]), ('P', [(0.3036, 1.0)]), ('P', [(0.113, 1.0)]), ('P', [(0.04286, 1.0)]), ('D', [(1.1288, 1.0)]), ('D', [(0.4569, 1.0)]), ('D', [(0.185, 1.0)]), ('F', [(0.4768, 1.0)]), ('F', [(0.2521, 1.0)]), ('G', [(0.4131, 1.0)])],
    5 : [('S', [(23870.0, 8.85552e-05), (3575.0, 0.0006913346), (812.8, 0.0036227143), (229.7, 0.015043321), (74.69, 0.0517595302), (26.81, 0.1442061668), (10.32, 0.3028337554), (4.178, 0.4060720554), (1.727, 0.2267617873)]), ('S', [(23870.0, -5.49084e-05), (3575.0, -0.0004240151), (812.8, -0.0022115895), (229.7, -0.0093435843), (74.69, -0.0322770049), (26.81, -0.0956779376), (10.32, -0.2166198535), (4.178, -0.4029760112), (1.727, -0.3754272322)]), ('S', [(0.4704, 1.0)]), ('S', [(0.1896, 1.0)]), ('S', [(0.07394, 1.0)]), ('P', [(22.26, 0.0324282283), (5.058, 0.2113467611), (1.487, 0.8421410391)]), ('P', [(0.5071, 1.0)]), ('P', [(0.1812, 1.0)]), ('P', [(0.06463, 1.0)]), ('D', [(1.11, 1.0)]), ('D', [(0.402, 1.0)]), ('D', [(0.145, 1.0)]), ('F', [(0.882, 1.0)]), ('F', [(0.311, 1.0)]), ('G', [(0.673, 1.0)])],
    6 : [('S', [(33980.0, 9.16345e-05), (5089.0, 0.000708909), (1157.0, 0.0037187514), (326.6, 0.0154671057), (106.1, 0.0532980755), (38.11, 0.1480683352), (14.75, 0.307762174), (6.035, 0.4021296445), (2.53, 0.2185645031)]), ('S', [(33980.0, -5.57104e-05), (5089.0, -0.0004427508), (1157.0, -0.0023017175), (326.6, -0.0097463809), (106.1, -0.033754614), (38.11, -0.1001613634), (14.75, -0.2262808226), (6.035, -0.4148750525), (2.53, -0.3460463685)]), ('S', [(0.7355, 1.0)]), ('S', [(0.2905, 1.0)]), ('S', [(0.1111, 1.0)]), ('P', [(34.51, 0.0316878552), (7.915, 0.2128943073), (2.368, 0.839586752)]), ('P', [(0.8132, 1.0)]), ('P', [(0.289, 1.0)]), ('P', [(0.1007, 1.0)]), ('D', [(1.848, 1.0)]), ('D', [(0.649, 1.0)]), ('D', [(0.228, 1.0)]), ('F', [(1.419, 1.0)]), ('F', [(0.485, 1.0)]), ('G', [(1.011, 1.0)])],
    7 : [('S', [(45840.0, 9.27106e-05), (6868.0, 0.0007225383), (1563.0, 0.0037779583), (442.4, 0.0156519735), (144.3, 0.0535565145), (52.18, 0.1479208236), (20.34, 0.3070163017), (8.381, 0.4007558217), (3.529, 0.2193221195)]), ('S', [(45840.0, -5.73387e-05), (6868.0, -0.0004558425), (1563.0, -0.0023623535), (442.4, -0.0099711961), (144.3, -0.034305731), (52.18, -0.1014550566), (20.34, -0.2295754654), (8.381, -0.4206422748), (3.529, -0.3335964279)]), ('S', [(1.054, 1.0)]), ('S', [(0.4118, 1.0)]), ('S', [(0.1552, 1.0)]), ('P', [(49.33, 0.0311260974), (11.37, 0.2135566435), (3.435, 0.8383625589)]), ('P', [(1.182, 1.0)]), ('P', [(0.4173, 1.0)]), ('P', [(0.1428, 1.0)]), ('D', [(2.837, 1.0)]), ('D', [(0.968, 1.0)]), ('D', [(0.335, 1.0)]), ('F', [(2.027, 1.0)]), ('F', [(0.685, 1.0)]), ('G', [(1.427, 1.0)])],
    8 : [('S', [(61420.0, 9.07389e-05), (9199.0, 0.0007037308), (2091.0, 0.0036940825), (590.9, 0.0153429443), (192.3, 0.0528534084), (69.32, 0.1471190548), (26.97, 0.3077642591), (11.1, 0.4017798693), (4.682, 0.2187614704)]), ('S', [(61420.0, -5.61064e-05), (9199.0, -0.0004460458), (2091.0, -0.0023256097), (590.9, -0.0098410601), (192.3, -0.0341014613), (69.32, -0.1017236829), (26.97, -0.2328190588), (11.1, -0.4266610114), (4.682, -0.3235402795)]), ('S', [(1.428, 1.0)]), ('S', [(0.5547, 1.0)]), ('S', [(0.2067, 1.0)]), ('P', [(63.42, 0.0313275025), (14.66, 0.2166542485), (4.459, 0.8352428422)]), ('P', [(1.531, 1.0)]), ('P', [(0.5302, 1.0)]), ('P', [(0.175, 1.0)]), ('D', [(3.775, 1.0)]), ('D', [(1.3, 1.0)]), ('D', [(0.444, 1.0)]), ('F', [(2.666, 1.0)]), ('F', [(0.859, 1.0)]), ('G', [(1.846, 1.0)])],
    9 : [('S', [(74530.0, 9.57754e-05), (11170.0, 0.0007440233), (2543.0, 0.0038894876), (721.0, 0.0160559822), (235.9, 0.0547320869), (85.6, 0.1507332702), (33.55, 0.3107678397), (13.93, 0.3980756452), (5.915, 0.2127533576)]), ('S', [(74530.0, -6.08735e-05), (11170.0, -0.0004759203), (2543.0, -0.0024653776), (721.0, -0.0103706344), (235.9, -0.0355888741), (85.6, -0.1053139587), (33.55, -0.2386214363), (13.93, -0.4312750631), (5.915, -0.3068966243)]), ('S', [(1.843, 1.0)]), ('S', [(0.7124, 1.0)]), ('S', [(0.2637, 1.0)]), ('P', [(80.39, 0.0313841889), (18.63, 0.2185767586), (5.694, 0.8332559023)]), ('P', [(1.953, 1.0)]), ('P', [(0.6702, 1.0)]), ('P', [(0.2166, 1.0)]), ('D', [(5.014, 1.0)]), ('D', [(1.725, 1.0)]), ('D', [(0.586, 1.0)]), ('F', [(3.562, 1.0)]), ('F', [(1.148, 1.0)]), ('G', [(2.376, 1.0)])],
    10 : [('S', [(99920.0, 8.67723e-05), (14960.0, 0.0006750074), (3399.0, 0.0035495907), (958.9, 0.0147987059), (311.2, 0.0514196255), (111.7, 0.1450347838), (43.32, 0.3072968877), (17.8, 0.403697839), (7.503, 0.2206073663)]), ('S', [(99920.0, -5.47971e-05), (14960.0, -0.0004328971), (3399.0, -0.0022576404), (958.9, -0.0095894919), (311.2, -0.033516644), (111.7, -0.1014212057), (43.32, -0.2359371177), (17.8, -0.4339409456), (7.503, -0.3131325272)]), ('S', [(2.337, 1.0)]), ('S', [(0.9001, 1.0)]), ('S', [(0.3301, 1.0)]), ('P', [(99.68, 0.0314763218), (23.15, 0.2204157481), (7.108, 0.8313421044)]), ('P', [(2.441, 1.0)]), ('P', [(0.8339, 1.0)]), ('P', [(0.2662, 1.0)]), ('D', [(6.471, 1.0)]), ('D', [(2.213, 1.0)]), ('D', [(0.747, 1.0)]), ('F', [(4.657, 1.0)]), ('F', [(1.524, 1.0)]), ('G', [(2.983, 1.0)])],
    11 : [('S', [(1224000.0, 4.7889e-06), (183200.0, 3.72396e-05), (41700.0, 0.0001958315), (11810.0, 0.0008267001), (3853.0, 0.0030025176), (1391.0, 0.0097031245), (542.5, 0.0282337712), (224.9, 0.0732059846), (97.93, 0.1628974107), (44.31, 0.2887087279), (20.65, 0.3468298744), (9.729, 0.2068655215), (4.228, 0.0328009827), (1.969, -0.0006477376), (0.889, 0.0014587837), (0.3964, -0.0001783464)]), ('S', [(1224000.0, -1.1706e-06), (183200.0, -9.0989e-06), (41700.0, -4.78911e-05), (11810.0, -0.0002021359), (3853.0, -0.0007364706), (1391.0, -0.0023895157), (542.5, -0.0070557601), (224.9, -0.0188017753), (97.93, -0.0446537159), (44.31, -0.0898513997), (20.65, -0.143063078), (9.729, -0.124422041), (4.228, 0.1000508744), (1.969, 0.4174391254), (0.889, 0.4755321032), (0.3964, 0.1634085814)]), ('S', [(1224000.0, 5.72e-07), (183200.0, 4.4416e-06), (41700.0, 2.34061e-05), (11810.0, 9.86416e-05), (3853.0, 0.000360138), (1391.0, 0.0011660651), (542.5, 0.0034557014), (224.9, 0.0091922788), (97.93, 0.0220059682), (44.31, 0.0443799047), (20.65, 0.0724535762), (9.729, 0.0637379066), (4.228, -0.0545344743), (1.969, -0.2515990708), (0.889, -0.3690770488), (0.3964, -0.4524161884)]), ('S', [(0.06993, 1.0)]), ('S', [(0.03289, 1.0)]), ('S', [(0.01612, 1.0)]), ('P', [(413.4, 0.0009090712), (97.98, 0.0074248781), (31.37, 0.0357808471), (11.62, 0.118634212), (4.671, 0.2616549015), (1.918, 0.378759641), (0.7775, 0.3349544687), (0.3013, 0.1269662334), (0.2275, -0.014725877)]), ('P', [(413.4, -0.0008935521), (97.98, -0.0073262785), (31.37, -0.0354064188), (11.62, -0.1190509605), (4.671, -0.2647516899), (1.918, -0.389186312), (0.7775, -0.3726676964), (0.3013, -0.4292937484), (0.2275, 0.5132988854)]), ('P', [(0.07527, 1.0)]), ('P', [(0.03126, 1.0)]), ('P', [(0.01342, 1.0)]), ('D', [(0.2116, 1.0)]), ('D', [(0.1039, 1.0)]), ('D', [(0.051, 1.0)]), ('F', [(0.2006, 1.0)]), ('F', [(0.1009, 1.0)]), ('G', [(0.1776, 1.0)])],
    12 : [('S', [(327600.0, 3.09603e-05), (49050.0, 0.0002409501), (11150.0, 0.0012665795), (3152.0, 0.0053335035), (1025.0, 0.0190766906), (368.8, 0.0588048461), (143.2, 0.1514515433), (58.96, 0.3007111222), (25.4, 0.3811428176), (11.15, 0.2135805356), (4.004, 0.023120625), (1.701, -0.0023075326), (0.706, 0.0012889791)]), ('S', [(327600.0, -7.8729e-06), (49050.0, -6.11132e-05), (11150.0, -0.000322886), (3152.0, -0.0013566465), (1025.0, -0.0049314963), (368.8, -0.0154368489), (143.2, -0.0425635465), (58.96, -0.0945549092), (25.4, -0.1642843584), (11.15, -0.1254100098), (4.004, 0.2368620063), (1.701, 0.5806000727), (0.706, 0.3369947929)]), ('S', [(327600.0, 4.0419e-06), (49050.0, 3.13725e-05), (11150.0, 0.0001657841), (3152.0, 0.0006966049), (1025.0, 0.0025342965), (368.8, 0.007943832), (143.2, 0.0219957595), (58.96, 0.0492753544), (25.4, 0.0874899549), (11.15, 0.0689177876), (4.004, -0.1433853702), (1.701, -0.4202186538), (0.706, -0.553503724)]), ('S', [(0.141, 1.0)]), ('S', [(0.06808, 1.0)]), ('S', [(0.03063, 1.0)]), ('P', [(539.6, 0.0008351283), (127.9, 0.0069017309), (41.02, 0.0338343686), (15.25, 0.1145600313), (6.166, 0.2598747559), (2.561, 0.3856303288), (1.06, 0.3358392091), (0.4176, 0.1107948044), (0.269, -0.0121483643)]), ('P', [(539.6, -0.0008340077), (127.9, -0.0069168917), (41.02, -0.0340669766), (15.25, -0.1171813824), (6.166, -0.2698704184), (2.561, -0.4089868431), (1.06, -0.3964422494), (0.4176, -0.1546522467), (0.269, 0.6615243031)]), ('P', [(0.1223, 1.0)]), ('P', [(0.05476, 1.0)]), ('P', [(0.02388, 1.0)]), ('D', [(0.3651, 1.0)]), ('D', [(0.1918, 1.0)]), ('D', [(0.1008, 1.0)]), ('F', [(0.3622, 1.0)]), ('F', [(0.1791, 1.0)]), ('G', [(0.3063, 1.0)])],
    13 : [('S', [(419600.0, 2.7821e-05), (62830.0, 0.0002163231), (14290.0, 0.0011375036), (4038.0, 0.0047961966), (1312.0, 0.0172383487), (470.5, 0.0538048793), (181.8, 0.1413214804), (74.46, 0.2892587493), (31.9, 0.3848126934), (13.96, 0.2328445535), (5.18, 0.0293320619), (2.265, -0.0030056439), (0.9664, 0.0016666767)]), ('S', [(419600.0, -7.3121e-06), (62830.0, -5.6752e-05), (14290.0, -0.0002995828), (4038.0, -0.0012619985), (1312.0, -0.0045978944), (470.5, -0.0145880534), (181.8, -0.0407620477), (74.46, -0.093212279), (31.9, -0.1662047784), (13.96, -0.1427516283), (5.18, 0.197377646), (2.265, 0.5783726249), (0.9664, 0.3778943622)]), ('S', [(419600.0, 3.7322e-06), (62830.0, 2.89465e-05), (14290.0, 0.0001529707), (4038.0, 0.000643664), (1312.0, 0.0023506237), (470.5, 0.0074548952), (181.8, 0.0209710262), (74.46, 0.0482394089), (31.9, 0.08839132), (13.96, 0.077965613), (5.18, -0.1179855644), (2.265, -0.4284290591), (0.9664, -0.5673930849)]), ('S', [(0.2447, 1.0)]), ('S', [(0.1184, 1.0)]), ('S', [(0.05021, 1.0)]), ('P', [(891.3, 0.0004954384), (211.3, 0.0041895777), (68.28, 0.0214129963), (25.7, 0.0769780985), (10.63, 0.1957321831), (4.602, 0.3369329491), (2.015, 0.3778350382), (0.8706, 0.2055693179)]), ('P', [(891.3, -0.0004738862), (211.3, -0.0039770142), (68.28, -0.0206376568), (25.7, -0.0743067624), (10.63, -0.1956238167), (4.602, -0.334765429), (2.015, -0.4210460583), (0.8706, -0.1538867188)]), ('P', [(0.2972, 1.0)]), ('P', [(0.11, 1.0)]), ('P', [(0.03989, 1.0)]), ('D', [(0.0804, 1.0)]), ('D', [(0.199, 1.0)]), ('D', [(0.494, 1.0)]), ('F', [(0.154, 1.0)]), ('F', [(0.401, 1.0)]), ('G', [(0.357, 1.0)])],
    14 : [('S', [(513000.0, 2.60908e-05), (76820.0, 0.0002028959), (17470.0, 0.0010671024), (4935.0, 0.004505769), (1602.0, 0.0162351756), (574.1, 0.0508890294), (221.5, 0.1351489699), (90.54, 0.2812794498), (38.74, 0.3853188078), (16.95, 0.24564004), (6.452, 0.034312969), (2.874, -0.0033486906), (1.25, 0.0018761663)]), ('S', [(513000.0, -7.0534e-06), (76820.0, -5.47767e-05), (17470.0, -0.0002890034), (4935.0, -0.0012201307), (1602.0, -0.0044499858), (574.1, -0.0141880811), (221.5, -0.0399441743), (90.54, -0.0928050689), (38.74, -0.1681028157), (16.95, -0.1548014897), (6.452, 0.1710617111), (2.874, 0.5778565375), (1.25, 0.4040501083)]), ('S', [(513000.0, 3.5795e-06), (76820.0, 2.77697e-05), (17470.0, 0.0001467416), (4935.0, 0.0006184531), (1602.0, 0.002262679), (574.1, 0.0072050615), (221.5, 0.0204487983), (90.54, 0.0477449529), (38.74, 0.0891464926), (16.95, 0.0842332974), (6.452, -0.1010105666), (2.874, -0.4353530818), (1.25, -0.5758018801)]), ('S', [(0.3599, 1.0)]), ('S', [(0.1699, 1.0)]), ('S', [(0.07066, 1.0)]), ('P', [(1122.0, 0.0004515661), (266.0, 0.0038455409), (85.92, 0.0199618194), (32.33, 0.0732570207), (13.37, 0.1912890558), (5.8, 0.3382359786), (2.559, 0.3822627208), (1.124, 0.2027297816)]), ('P', [(1122.0, -0.0004424636), (266.0, -0.0037234317), (85.92, -0.019722374), (32.33, -0.0722252324), (13.37, -0.1969733626), (5.8, -0.3451056626), (2.559, -0.4454731568), (1.124, -0.1043272209)]), ('P', [(0.3988, 1.0)]), ('P', [(0.1533, 1.0)]), ('P', [(0.05728, 1.0)]), ('D', [(0.12, 1.0)]), ('D', [(0.302, 1.0)]), ('D', [(0.76, 1.0)]), ('F', [(0.212, 1.0)]), ('F', [(0.541, 1.0)]), ('G', [(0.461, 1.0)])],
    15 : [('S', [(615200.0, 2.47435e-05), (92120.0, 0.0001924534), (20950.0, 0.0010119588), (5920.0, 0.0042723514), (1922.0, 0.0154151671), (688.0, 0.0485946592), (265.0, 0.1300521295), (108.2, 0.274497388), (46.22, 0.3853786777), (20.23, 0.2559185124), (7.859, 0.0391213325), (3.547, -0.0036798773), (1.564, 0.002081984)]), ('S', [(615200.0, -6.8546e-06), (92120.0, -5.32608e-05), (20950.0, -0.0002807896), (5920.0, -0.0011859993), (1922.0, -0.0043264549), (688.0, -0.0138797419), (265.0, -0.0392706329), (108.2, -0.0924517011), (46.22, -0.1698681198), (20.23, -0.1646298464), (7.859, 0.1496747136), (3.547, 0.5778538508), (1.564, 0.4246427673)]), ('S', [(615200.0, 3.4618e-06), (92120.0, 2.6868e-05), (20950.0, 0.000141897), (5920.0, 0.0005981584), (1922.0, 0.0021896713), (688.0, 0.0070133906), (265.0, 0.0200164412), (108.2, 0.0473454037), (46.22, 0.0898128575), (20.23, 0.0894030355), (7.859, -0.0874221355), (3.547, -0.4403066143), (1.564, -0.5834145092)]), ('S', [(0.4888, 1.0)]), ('S', [(0.2266, 1.0)]), ('S', [(0.09331, 1.0)]), ('P', [(1367.0, 0.0004242182), (324.0, 0.0036373146), (104.6, 0.019065661), (39.37, 0.0710928078), (16.26, 0.1895885458), (7.056, 0.3412859832), (3.13, 0.3848489184), (1.394, 0.1967465948)]), ('P', [(1367.0, -0.0004215654), (324.0, -0.0035727257), (104.6, -0.0191123697), (39.37, -0.0712150219), (16.26, -0.1986864281), (7.056, -0.3565563512), (3.13, -0.4585637472), (1.394, -0.0673910093)]), ('P', [(0.5179, 1.0)]), ('P', [(0.2032, 1.0)]), ('P', [(0.07698, 1.0)]), ('D', [(0.165, 1.0)]), ('D', [(0.413, 1.0)]), ('D', [(1.036, 1.0)]), ('F', [(0.28, 1.0)]), ('F', [(0.703, 1.0)]), ('G', [(0.597, 1.0)])],
    16 : [('S', [(727800.0, 2.36008e-05), (109000.0, 0.0001834687), (24800.0, 0.0009642079), (7014.0, 0.0040650743), (2278.0, 0.0146962309), (814.7, 0.0465047171), (313.4, 0.1254988708), (127.7, 0.2684134747), (54.48, 0.3847810097), (23.85, 0.2653526973), (9.428, 0.043729419), (4.29, -0.0037877945), (1.909, 0.0021806714)]), ('S', [(727800.0, -6.6773e-06), (109000.0, -5.1871e-05), (24800.0, -0.000273195), (7014.0, -0.0011528572), (2278.0, -0.004209898), (814.7, -0.0135612067), (313.4, -0.0385992809), (127.7, -0.0919977991), (54.48, -0.1710820745), (23.85, -0.1733918638), (9.428, 0.1308716747), (4.29, 0.5783298591), (1.909, 0.4420615093)]), ('S', [(727800.0, 3.3566e-06), (109000.0, 2.60411e-05), (24800.0, 0.0001374218), (7014.0, 0.0005786175), (2278.0, 0.0021209954), (814.7, 0.0068191511), (313.4, 0.0195889841), (127.7, 0.0468986212), (54.48, 0.0901610021), (23.85, 0.0939283803), (9.428, -0.0754074609), (4.29, -0.4445449499), (1.909, -0.5903897467)]), ('S', [(0.627, 1.0)]), ('S', [(0.2873, 1.0)]), ('S', [(0.1172, 1.0)]), ('P', [(1546.0, 0.0004437995), (366.4, 0.0037981025), (118.4, 0.0199536407), (44.53, 0.074646393), (18.38, 0.198497281), (7.965, 0.3539377116), (3.541, 0.380932867), (1.591, 0.1719447351)]), ('P', [(1546.0, -0.0004570726), (366.4, -0.0038735838), (118.4, -0.0207491381), (44.53, -0.077845384), (18.38, -0.2165871696), (7.965, -0.3880663576), (3.541, -0.4775723248), (1.591, 0.037305865)]), ('P', [(0.6205, 1.0)]), ('P', [(0.242, 1.0)]), ('P', [(0.09014, 1.0)]), ('D', [(0.203, 1.0)]), ('D', [(0.504, 1.0)]), ('D', [(1.25, 1.0)]), ('F', [(0.335, 1.0)]), ('F', [(0.869, 1.0)]), ('G', [(0.683, 1.0)])],
    17 : [('S', [(834900.0, 2.31669e-05), (125000.0, 0.0001801395), (28430.0, 0.0009477058), (8033.0, 0.0040010681), (2608.0, 0.0144617365), (933.9, 0.0456549268), (360.0, 0.1232380847), (147.0, 0.2643477315), (62.88, 0.3829581885), (27.6, 0.2709122033), (11.08, 0.0471366075), (5.075, -0.0037173609), (2.278, 0.0021914037)]), ('S', [(834900.0, -6.6749e-06), (125000.0, -5.18758e-05), (28430.0, -0.0002734197), (8033.0, -0.0011558791), (2608.0, -0.0042176811), (933.9, -0.0135610999), (360.0, -0.0385647855), (147.0, -0.0921868542), (62.88, -0.1722747828), (27.6, -0.1795615096), (11.08, 0.118064084), (5.075, 0.5790933945), (2.278, 0.4537312818)]), ('S', [(834900.0, 3.3423e-06), (125000.0, 2.59393e-05), (28430.0, 0.0001370025), (8033.0, 0.0005777965), (2608.0, 0.0021168705), (933.9, 0.0067918111), (360.0, 0.019503096), (147.0, 0.0468246982), (62.88, 0.0905746476), (27.6, 0.0970890498), (11.08, -0.0671684171), (5.075, -0.4492777313), (2.278, -0.5936554706)]), ('S', [(0.7775, 1.0)]), ('S', [(0.3527, 1.0)]), ('S', [(0.1431, 1.0)]), ('P', [(1703.0, 0.000476096), (403.6, 0.0040817557), (130.3, 0.0214280825), (49.05, 0.0798059104), (20.26, 0.2098336098), (8.787, 0.3665286283), (3.919, 0.3733380492), (1.765, 0.1469268139)]), ('P', [(1703.0, -0.0005072075), (403.6, -0.0043243092), (130.3, -0.0230707725), (49.05, -0.0867020912), (20.26, -0.2378081399), (8.787, -0.4228337354), (3.919, -0.4842248803), (1.765, 0.1516960364)]), ('P', [(0.7207, 1.0)]), ('P', [(0.2839, 1.0)]), ('P', [(0.106, 1.0)]), ('D', [(0.254, 1.0)]), ('D', [(0.628, 1.0)]), ('D', [(1.551, 1.0)]), ('F', [(0.423, 1.0)]), ('F', [(1.089, 1.0)]), ('G', [(0.827, 1.0)])],
    18 : [('S', [(950600.0, 2.27525e-05), (142300.0, 0.0001769294), (32360.0, 0.0009311997), (9145.0, 0.0039282529), (2970.0, 0.0142051448), (1064.0, 0.0448074408), (410.8, 0.1209903091), (168.0, 0.2605559769), (71.99, 0.3813303052), (31.67, 0.2760336093), (12.89, 0.0505134366), (5.929, -0.003598342), (2.678, 0.0021877867)]), ('S', [(950600.0, -6.6613e-06), (142300.0, -5.17837e-05), (32360.0, -0.0002729697), (9145.0, -0.001153455), (2970.0, -0.0042086598), (1064.0, -0.01352623), (410.8, -0.0384352708), (168.0, -0.0922299089), (71.99, -0.1732362713), (31.67, -0.1851321296), (12.89, 0.1061277556), (5.929, 0.5799797882), (2.678, 0.4642425706)]), ('S', [(950600.0, 3.3252e-06), (142300.0, 2.58121e-05), (32360.0, 0.0001363608), (9145.0, 0.0005747641), (2970.0, 0.0021060288), (1064.0, 0.0067531452), (410.8, 0.0193838709), (168.0, 0.0467145545), (71.99, 0.0909063375), (31.67, 0.0999715296), (12.89, -0.0595742715), (5.929, -0.4532069443), (2.678, -0.5971135364)]), ('S', [(0.9416, 1.0)]), ('S', [(0.4239, 1.0)]), ('S', [(0.1714, 1.0)]), ('P', [(1890.0, 0.0004971033), (447.8, 0.0042633089), (144.6, 0.0223885587), (54.46, 0.0833142726), (22.51, 0.2177017773), (9.774, 0.3755277947), (4.368, 0.3674438201), (1.959, 0.1295972834)]), ('P', [(1890.0, -0.0005437457), (447.8, -0.0046545912), (144.6, -0.0247688704), (54.46, -0.0935119785), (22.51, -0.2542220477), (9.774, -0.4520439357), (4.368, -0.484180076), (1.959, 0.2541503904)]), ('P', [(0.826, 1.0)]), ('P', [(0.3297, 1.0)]), ('P', [(0.1242, 1.0)]), ('D', [(0.311, 1.0)]), ('D', [(0.763, 1.0)]), ('D', [(1.873, 1.0)]), ('F', [(0.543, 1.0)]), ('F', [(1.325, 1.0)]), ('G', [(1.007, 1.0)])],

}





CC_PV5Z = {

    1 : [('S', [(402.0, 0.0050881043), (60.24, 0.0394829597), (13.73, 0.2042718854), (3.905, 0.8184370745)]), ('S', [(1.283, 1.0)]), ('S', [(0.4655, 1.0)]), ('S', [(0.1811, 1.0)]), ('S', [(0.07279, 1.0)]), ('P', [(4.516, 1.0)]), ('P', [(1.712, 1.0)]), ('P', [(0.649, 1.0)]), ('P', [(0.246, 1.0)]), ('D', [(2.95, 1.0)]), ('D', [(1.206, 1.0)]), ('D', [(0.493, 1.0)]), ('F', [(2.506, 1.0)]), ('F', [(0.875, 1.0)]), ('G', [(2.358, 1.0)])],
    2 : [('S', [(1145.0, 0.0052613382), (171.7, 0.040610496), (39.07, 0.2088560729), (11.04, 0.8143496277)]), ('S', [(3.566, 1.0)]), ('S', [(1.24, 1.0)]), ('S', [(0.4473, 1.0)]), ('S', [(0.164, 1.0)]), ('P', [(10.153, 1.0)]), ('P', [(3.627, 1.0)]), ('P', [(1.296, 1.0)]), ('P', [(0.463, 1.0)]), ('D', [(7.666, 1.0)]), ('D', [(2.647, 1.0)]), ('D', [(0.914, 1.0)]), ('F', [(5.411, 1.0)]), ('F', [(1.707, 1.0)]), ('G', [(3.43, 1.0)])],
    3 : [('S', [(29493.0, 2.01e-05), (4417.101, 0.0001574499), (1005.223, 0.0008252159), (284.7009, 0.0034694802), (92.86543, 0.0124340721), (33.51179, 0.0387147985), (13.0418, 0.1029241907), (5.357536, 0.2228596662), (2.279338, 0.3671998696), (0.99399, 0.3863384024)]), ('S', [(29493.0, -1.56374e-05), (4417.101, -0.0001146741), (1005.223, -0.0005994328), (284.7009, -0.0025384675), (92.86543, -0.0091009532), (33.51179, -0.028772773), (13.0418, -0.077811586), (5.357536, -0.178297368), (2.279338, -0.3239803809), (0.99399, -0.4998852303)]), ('S', [(0.433471, 1.0)]), ('S', [(0.095566, 1.0)]), ('S', [(0.044657, 1.0)]), ('S', [(0.020633, 1.0)]), ('P', [(19.6635, 0.0086526274), (4.62311, 0.0619303793), (1.41378, 0.2430907594), (0.473721, 0.7884145887)]), ('P', [(0.176151, 1.0)]), ('P', [(0.072675, 1.0)]), ('P', [(0.032141, 1.0)]), ('P', [(0.014556, 1.0)]), ('D', [(0.531, 1.0)]), ('D', [(0.2481, 1.0)]), ('D', [(0.1159, 1.0)]), ('D', [(0.0542, 1.0)]), ('F', [(0.292, 1.0)]), ('F', [(0.1641, 1.0)]), ('F', [(0.0922, 1.0)]), ('G', [(0.2971, 1.0)]), ('G', [(0.15, 1.0)]), ('H', [(0.2949, 1.0)])],
    4 : [('S', [(54620.0, 2.0171e-05), (8180.0, 0.0001546442), (1862.0, 0.0008102009), (527.3, 0.0034055334), (172.0, 0.0122236125), (62.1, 0.0381399571), (24.21, 0.1021917764), (9.993, 0.2233016886), (4.305, 0.3690784655), (1.921, 0.3815553359)]), ('S', [(54620.0, -1.28386e-05), (8180.0, -0.0001069885), (1862.0, -0.0005606198), (527.3, -0.0023879834), (172.0, -0.0085077259), (62.1, -0.0272606711), (24.21, -0.0736808437), (9.993, -0.1748534537), (4.305, -0.3177002262), (1.921, -0.5102666967)]), ('S', [(0.8663, 1.0)]), ('S', [(0.2475, 1.0)]), ('S', [(0.1009, 1.0)]), ('S', [(0.04129, 1.0)]), ('P', [(43.75, 0.0073747779), (10.33, 0.0560156907), (3.226, 0.2391501837), (1.127, 0.7900915312)]), ('P', [(0.4334, 1.0)]), ('P', [(0.1808, 1.0)]), ('P', [(0.07827, 1.0)]), ('P', [(0.03372, 1.0)]), ('D', [(1.7175, 1.0)]), ('D', [(0.7646, 1.0)]), ('D', [(0.3404, 1.0)]), ('D', [(0.1515, 1.0)]), ('F', [(0.6127, 1.0)]), ('F', [(0.3568, 1.0)]), ('F', [(0.2078, 1.0)]), ('G', [(0.598, 1.0)]), ('G', [(0.3183, 1.0)]), ('H', [(0.5142, 1.0)])],
    5 : [('S', [(68260.0, 2.55912e-05), (10230.0, 0.0001972658), (2328.0, 0.0010343124), (660.4, 0.0043249186), (216.2, 0.0153536742), (78.6, 0.0468116989), (30.98, 0.1205528404), (12.96, 0.2493279311), (5.659, 0.377428053), (2.556, 0.3215399963)]), ('S', [(68260.0, -1.71486e-05), (10230.0, -0.0001268994), (2328.0, -0.0006722239), (660.4, -0.0028260841), (216.2, -0.0100250531), (78.6, -0.0313407238), (30.98, -0.0826732487), (12.96, -0.1877939736), (5.659, -0.3324867352), (2.556, -0.471534188)]), ('S', [(1.175, 1.0)]), ('S', [(0.4249, 1.0)]), ('S', [(0.1712, 1.0)]), ('S', [(0.06913, 1.0)]), ('P', [(66.44, 0.0071601204), (15.71, 0.0547603955), (4.936, 0.2399323868), (1.77, 0.7873740005)]), ('P', [(0.7008, 1.0)]), ('P', [(0.2901, 1.0)]), ('P', [(0.1211, 1.0)]), ('P', [(0.04973, 1.0)]), ('D', [(2.01, 1.0)]), ('D', [(0.796, 1.0)]), ('D', [(0.316, 1.0)]), ('D', [(0.125, 1.0)]), ('F', [(1.215, 1.0)]), ('F', [(0.525, 1.0)]), ('F', [(0.227, 1.0)]), ('G', [(1.124, 1.0)]), ('G', [(0.461, 1.0)]), ('H', [(0.834, 1.0)])],
    6 : [('S', [(96770.0, 2.64387e-05), (14500.0, 0.0002009342), (3300.0, 0.0010575486), (935.8, 0.0044237258), (306.2, 0.0157141148), (111.3, 0.0479080097), (43.9, 0.1232086435), (18.4, 0.2540749965), (8.054, 0.3794473845), (3.637, 0.3108568966)]), ('S', [(96770.0, -1.59189e-05), (14500.0, -0.0001305352), (3300.0, -0.0006781461), (935.8, -0.0028558545), (306.2, -0.0101467206), (111.3, -0.0317136755), (43.9, -0.0839723111), (18.4, -0.1910302423), (8.054, -0.3401077587), (3.637, -0.4589934486)]), ('S', [(1.656, 1.0)]), ('S', [(0.6333, 1.0)]), ('S', [(0.2545, 1.0)]), ('S', [(0.1019, 1.0)]), ('P', [(101.8, 0.0067482599), (24.04, 0.052834861), (7.571, 0.2398548186), (2.732, 0.7877211236)]), ('P', [(1.085, 1.0)]), ('P', [(0.4496, 1.0)]), ('P', [(0.1876, 1.0)]), ('P', [(0.07606, 1.0)]), ('D', [(3.134, 1.0)]), ('D', [(1.233, 1.0)]), ('D', [(0.485, 1.0)]), ('D', [(0.191, 1.0)]), ('F', [(2.006, 1.0)]), ('F', [(0.838, 1.0)]), ('F', [(0.35, 1.0)]), ('G', [(1.753, 1.0)]), ('G', [(0.678, 1.0)]), ('H', [(1.259, 1.0)])],
    7 : [('S', [(129200.0, 2.62722e-05), (19350.0, 0.0002070247), (4404.0, 0.0010845152), (1248.0, 0.0045450855), (408.0, 0.0161626394), (148.2, 0.0492519129), (58.5, 0.126228322), (24.59, 0.2581976387), (10.81, 0.3797684303), (4.882, 0.3019019201)]), ('S', [(129200.0, -1.82355e-05), (19350.0, -0.0001306877), (4404.0, -0.0006899097), (1248.0, -0.0029116012), (408.0, -0.0103820771), (148.2, -0.0324196768), (58.5, -0.085946943), (24.59, -0.1945727674), (10.81, -0.3462677996), (4.882, -0.4467545133)]), ('S', [(2.195, 1.0)]), ('S', [(0.8715, 1.0)]), ('S', [(0.3504, 1.0)]), ('S', [(0.1397, 1.0)]), ('P', [(147.0, 0.0064982238), (34.76, 0.0515924001), (11.0, 0.2390646995), (3.995, 0.7883030251)]), ('P', [(1.587, 1.0)]), ('P', [(0.6533, 1.0)]), ('P', [(0.2686, 1.0)]), ('P', [(0.1067, 1.0)]), ('D', [(4.647, 1.0)]), ('D', [(1.813, 1.0)]), ('D', [(0.707, 1.0)]), ('D', [(0.276, 1.0)]), ('F', [(2.942, 1.0)]), ('F', [(1.204, 1.0)]), ('F', [(0.493, 1.0)]), ('G', [(2.511, 1.0)]), ('G', [(0.942, 1.0)]), ('H', [(1.768, 1.0)])],
    8 : [('S', [(164200.0, 2.71683e-05), (24590.0, 0.0002142119), (5592.0, 0.0011243512), (1582.0, 0.0047252008), (516.1, 0.0168318298), (187.2, 0.0512906856), (73.93, 0.1304675794), (31.22, 0.2629957727), (13.81, 0.3787057204), (6.256, 0.2915904475)]), ('S', [(164200.0, -1.75236e-05), (24590.0, -0.0001343478), (5592.0, -0.0007126273), (1582.0, -0.0030111424), (516.1, -0.0107711865), (187.2, -0.0336278312), (73.93, -0.0888885742), (31.22, -0.1990303816), (13.81, -0.3515472284), (6.256, -0.4330087073)]), ('S', [(2.776, 1.0)]), ('S', [(1.138, 1.0)]), ('S', [(0.46, 1.0)]), ('S', [(0.1829, 1.0)]), ('P', [(195.5, 0.0062744515), (46.16, 0.0504963483), (14.58, 0.2389349411), (5.296, 0.7889610158)]), ('P', [(2.094, 1.0)]), ('P', [(0.8471, 1.0)]), ('P', [(0.3368, 1.0)]), ('P', [(0.1285, 1.0)]), ('D', [(5.879, 1.0)]), ('D', [(2.307, 1.0)]), ('D', [(0.905, 1.0)]), ('D', [(0.355, 1.0)]), ('F', [(4.016, 1.0)]), ('F', [(1.554, 1.0)]), ('F', [(0.601, 1.0)]), ('G', [(3.35, 1.0)]), ('G', [(1.189, 1.0)]), ('H', [(2.319, 1.0)])],
    9 : [('S', [(211400.0, 2.71685e-05), (31660.0, 0.0002100331), (7202.0, 0.0011034575), (2040.0, 0.0046311776), (666.4, 0.0164745365), (242.0, 0.0502741914), (95.53, 0.1287701438), (40.23, 0.2628224632), (17.72, 0.3809070424), (8.005, 0.2923389058)]), ('S', [(211400.0, -1.71937e-05), (31660.0, -0.0001346842), (7202.0, -0.0006992115), (2040.0, -0.002954455), (666.4, -0.0105540813), (242.0, -0.0329918919), (95.53, -0.0878685295), (40.23, -0.1993669678), (17.72, -0.3553140498), (8.005, -0.4304563575)]), ('S', [(3.538, 1.0)]), ('S', [(1.458, 1.0)]), ('S', [(0.5887, 1.0)]), ('S', [(0.2324, 1.0)]), ('P', [(241.9, 0.0063608428), (57.17, 0.0511279717), (18.13, 0.2415342769), (6.624, 0.7857672218)]), ('P', [(2.622, 1.0)]), ('P', [(1.057, 1.0)]), ('P', [(0.4176, 1.0)]), ('P', [(0.1574, 1.0)]), ('D', [(7.76, 1.0)]), ('D', [(3.032, 1.0)]), ('D', [(1.185, 1.0)]), ('D', [(0.463, 1.0)]), ('F', [(5.398, 1.0)]), ('F', [(2.078, 1.0)]), ('F', [(0.8, 1.0)]), ('G', [(4.338, 1.0)]), ('G', [(1.513, 1.0)]), ('H', [(2.995, 1.0)])],
    10 : [('S', [(262700.0, 2.7124e-05), (39350.0, 0.0002086463), (8955.0, 0.0010953929), (2538.0, 0.0045902179), (829.9, 0.0163255273), (301.5, 0.0498226425), (119.0, 0.1282579912), (50.0, 0.2633981795), (21.98, 0.3821502466), (9.891, 0.2917052188)]), ('S', [(262700.0, -1.69245e-05), (39350.0, -0.0001325753), (8955.0, -0.0006967254), (2538.0, -0.0029279391), (829.9, -0.0104678054), (301.5, -0.0327009613), (119.0, -0.0876858522), (50.0, -0.2001943093), (21.98, -0.3589856419), (9.891, -0.4265849292)]), ('S', [(4.327, 1.0)]), ('S', [(1.804, 1.0)]), ('S', [(0.7288, 1.0)]), ('S', [(0.2867, 1.0)]), ('P', [(299.1, 0.0063548825), (70.73, 0.0512737391), (22.48, 0.2430099733), (8.246, 0.7839892459)]), ('P', [(3.269, 1.0)]), ('P', [(1.315, 1.0)]), ('P', [(0.5158, 1.0)]), ('P', [(0.1918, 1.0)]), ('D', [(9.837, 1.0)]), ('D', [(3.844, 1.0)]), ('D', [(1.502, 1.0)]), ('D', [(0.587, 1.0)]), ('F', [(7.09, 1.0)]), ('F', [(2.738, 1.0)]), ('F', [(1.057, 1.0)]), ('G', [(5.46, 1.0)]), ('G', [(1.88, 1.0)]), ('H', [(3.776, 1.0)])],
    11 : [('S', [(2185572.0, 2e-06), (327228.4, 1.79998e-05), (74466.84, 9.49988e-05), (21093.15, 0.0004009948), (6881.898, 0.0014589811), (2484.696, 0.0047459386), (969.2232, 0.0140308184), (402.0643, 0.0377325116), (175.3545, 0.0907008261), (79.65199, 0.1864635866), (37.38672, 0.3018330934), (18.00194, 0.3238258088), (8.724371, 0.1686978166), (3.857715, 0.0230827012), (1.815686, -0.0001469981), (0.838254, 0.0010989858)]), ('S', [(2185572.0, -1.1321e-06), (327228.4, -4.5284e-06), (74466.84, -2.60384e-05), (21093.15, -0.0001109461), (6881.898, -0.0004041608), (2484.696, -0.0013189002), (969.2232, -0.0039216053), (402.0643, -0.0107493195), (175.3545, -0.0267029172), (79.65199, -0.059315413), (37.38672, -0.1109778086), (18.00194, -0.1626526574), (8.724371, -0.1158118891), (3.857715, 0.1562630672), (1.815686, 0.4856790485), (0.838254, 0.5069705127)]), ('S', [(327228.4, 5.2139e-06), (74466.84, 1.56416e-05), (21093.15, 7.82082e-05), (6881.898, 0.0002815496), (2484.696, 0.0009124293), (969.2232, 0.0027112185), (402.0643, 0.0074610648), (175.3545, 0.0185301357), (79.65199, 0.0414712154), (37.38672, 0.0779736016), (18.00194, 0.1171663507), (8.724371, 0.0844909537), (3.857715, -0.1227869152), (1.815686, -0.4177362057), (0.838254, -0.588756741)]), ('S', [(0.381935, 1.0)]), ('S', [(0.071679, 1.0)]), ('S', [(0.033916, 1.0)]), ('S', [(0.016525, 1.0)]), ('P', [(1119.578, 0.0001618725), (265.3239, 0.0014068918), (85.99553, 0.0075800292), (32.53759, 0.0295916905), (13.51565, 0.088407361), (5.966856, 0.1953651098), (2.700045, 0.3063796629), (1.218512, 0.3409734132), (0.542187, 0.2295092142), (0.227413, 0.0548777725)]), ('P', [(1119.578, -0.000144368), (265.3239, -0.00126322), (85.99553, -0.006803342), (32.53759, -0.0267802638), (13.51565, -0.0804761364), (5.966856, -0.1807216657), (2.700045, -0.283141738), (1.218512, -0.3292402447), (0.542187, -0.2673063731), (0.227413, -0.0915473573)]), ('P', [(0.13304, 1.0)]), ('P', [(0.057577, 1.0)]), ('P', [(0.025971, 1.0)]), ('P', [(0.011901, 1.0)]), ('D', [(0.4898, 1.0)]), ('D', [(0.2283, 1.0)]), ('D', [(0.1064, 1.0)]), ('D', [(0.0496, 1.0)]), ('F', [(0.2493, 1.0)]), ('F', [(0.1357, 1.0)]), ('F', [(0.0739, 1.0)]), ('G', [(0.2318, 1.0)]), ('G', [(0.1222, 1.0)]), ('H', [(0.224, 1.0)])],
    12 : [('S', [(2968000.0, 1.9718e-06), (444300.0, 1.53359e-05), (101100.0, 8.06837e-05), (28640.0, 0.0003407645), (9343.0, 0.0012419164), (3373.0, 0.0040411631), (1316.0, 0.0119794719), (545.8, 0.0324246827), (238.1, 0.0789316972), (108.2, 0.1665718286), (50.8, 0.2828656145), (24.48, 0.3318836812), (11.93, 0.2032831297), (5.543, 0.038288971), (2.675, 0.0003089101), (1.263, 0.0013890436)]), ('S', [(2968000.0, -6.028e-07), (444300.0, -4.6896e-06), (101100.0, -2.46669e-05), (28640.0, -0.0001042631), (9343.0, -0.0003801616), (3373.0, -0.00124121), (1316.0, -0.0037004581), (545.8, -0.0101770946), (238.1, -0.0255560254), (108.2, -0.0577040394), (50.8, -0.1118209519), (24.48, -0.1724870179), (11.93, -0.1526667654), (5.543, 0.086551098), (2.675, 0.4634877714), (1.263, 0.5930744698)]), ('S', [(2968000.0, 4.007e-07), (444300.0, 3.1144e-06), (101100.0, 1.64073e-05), (28640.0, 6.92035e-05), (9343.0, 0.0002530283), (3373.0, 0.0008235461), (1316.0, 0.00246611), (545.8, 0.0067572501), (238.1, 0.0171016391), (108.2, 0.03852201), (50.8, 0.076029915), (24.48, 0.1180648987), (11.93, 0.1106254294), (5.543, -0.0718083031), (2.675, -0.3697476734), (1.263, -0.6839345107)]), ('S', [(0.5883, 1.0)]), ('S', [(0.1496, 1.0)]), ('S', [(0.067, 1.0)]), ('S', [(0.02952, 1.0)]), ('P', [(1441.0, 0.0001523099), (341.4, 0.0013267941), (110.7, 0.0072147704), (41.97, 0.0285023292), (17.49, 0.0864794676), (7.753, 0.1948257946), (3.534, 0.3118521878), (1.614, 0.348253981), (0.7299, 0.2187465446), (0.3029, 0.0435301486)]), ('P', [(1441.0, -0.0001554914), (341.4, -0.0013599491), (110.7, -0.0073884403), (41.97, -0.0294438991), (17.49, -0.0900592364), (7.753, -0.2071479203), (3.534, -0.3330165459), (1.614, -0.3959121522), (0.7299, -0.2854384222), (0.3029, 0.3375155403)]), ('P', [(0.1583, 1.0)]), ('P', [(0.0819, 1.0)]), ('P', [(0.04123, 1.0)]), ('P', [(0.01988, 1.0)]), ('D', [(1.0837, 1.0)]), ('D', [(0.4968, 1.0)]), ('D', [(0.2278, 1.0)]), ('D', [(0.1044, 1.0)]), ('F', [(0.4917, 1.0)]), ('F', [(0.2764, 1.0)]), ('F', [(0.1554, 1.0)]), ('G', [(0.4427, 1.0)]), ('G', [(0.2311, 1.0)]), ('H', [(0.3731, 1.0)])],
    13 : [('S', [(3269000.0, 2.1395e-06), (489400.0, 1.66257e-05), (111400.0, 8.75131e-05), (31560.0, 0.0003689743), (10320.0, 0.0013389732), (3731.0, 0.0043561751), (1456.0, 0.0128949525), (604.1, 0.0348186218), (263.5, 0.0843494189), (119.8, 0.1758995321), (56.32, 0.2920785996), (27.19, 0.3282060658), (13.26, 0.1869190642), (6.052, 0.0310416821), (2.981, -0.0005089004), (1.476, 0.0014882968)]), ('S', [(3269000.0, -6.551e-07), (489400.0, -5.0938e-06), (111400.0, -2.67961e-05), (31560.0, -0.0001131305), (10320.0, -0.0004104917), (3731.0, -0.0013413303), (1456.0, -0.0039914228), (604.1, -0.0109759296), (263.5, -0.0274571156), (119.8, -0.0616823902), (56.32, -0.1177710336), (27.19, -0.177404948), (13.26, -0.1403603534), (6.052, 0.1273636971), (2.981, 0.4844335738), (1.476, 0.5387355599)]), ('S', [(3269000.0, 3.998e-07), (489400.0, 3.106e-06), (111400.0, 1.63625e-05), (31560.0, 6.89557e-05), (10320.0, 0.0002508074), (3731.0, 0.0008173978), (1456.0, 0.0024419239), (604.1, 0.0066959382), (263.5, 0.016876122), (119.8, 0.0378908912), (56.32, 0.0737425383), (27.19, 0.1123894662), (13.26, 0.0944373308), (6.052, -0.0962277802), (2.981, -0.3709375196), (1.476, -0.6574685843)]), ('S', [(0.7334, 1.0)]), ('S', [(0.2447, 1.0)]), ('S', [(0.1088, 1.0)]), ('S', [(0.04672, 1.0)]), ('P', [(1461.0, 0.0002302962), (346.2, 0.0019981866), (112.2, 0.0107461164), (42.51, 0.0417582973), (17.72, 0.1224247396), (7.852, 0.2586476255), (3.571, 0.3811297701), (1.637, 0.3658788388)]), ('P', [(1461.0, -0.0002054084), (346.2, -0.0018144947), (112.2, -0.009623759), (42.51, -0.0383719943), (17.72, -0.1120005102), (7.852, -0.2477715904), (3.571, -0.3552513682), (1.637, -0.415661077)]), ('P', [(0.7382, 1.0)]), ('P', [(0.2577, 1.0)]), ('P', [(0.09773, 1.0)]), ('P', [(0.0369, 1.0)]), ('D', [(1.317, 1.0)]), ('D', [(0.526, 1.0)]), ('D', [(0.21, 1.0)]), ('D', [(0.084, 1.0)]), ('F', [(0.13, 1.0)]), ('F', [(0.258, 1.0)]), ('F', [(0.513, 1.0)]), ('G', [(0.252, 1.0)]), ('G', [(0.543, 1.0)]), ('H', [(0.446, 1.0)])],
    14 : [('S', [(3948000.0, 2.037e-06), (591100.0, 1.58385e-05), (134500.0, 8.33541e-05), (38120.0, 0.0003513403), (12460.0, 0.0012765248), (4504.0, 0.0041516654), (1758.0, 0.0123022751), (729.1, 0.0333082373), (318.0, 0.0809797284), (144.6, 0.1702799664), (67.97, 0.2868620969), (32.82, 0.3303205362), (16.03, 0.1965904161), (7.396, 0.0354514111), (3.661, -0.0005351725), (1.823, 0.0016145549)]), ('S', [(3948000.0, -6.387e-07), (591100.0, -4.9682e-06), (134500.0, -2.61336e-05), (38120.0, -0.0001102813), (12460.0, -0.000400719), (4504.0, -0.0013085022), (1758.0, -0.0038983496), (729.1, -0.0107403432), (318.0, -0.0269556575), (144.6, -0.060926101), (67.97, -0.1177112409), (32.82, -0.1799639765), (16.03, -0.1502278062), (7.396, 0.1115696066), (3.661, 0.4878103331), (1.823, 0.5513120609)]), ('S', [(3948000.0, 3.743e-07), (591100.0, 2.909e-06), (134500.0, 1.53231e-05), (38120.0, 6.45491e-05), (12460.0, 0.0002350919), (4504.0, 0.0007657467), (1758.0, 0.0022901016), (729.1, 0.0062929559), (318.0, 0.0159115497), (144.6, 0.0359643288), (67.97, 0.0708418672), (32.82, 0.1097908949), (16.03, 0.0974085442), (7.396, -0.0818994815), (3.661, -0.3663801491), (1.823, -0.6740741538)]), ('S', [(0.9147, 1.0)]), ('S', [(0.3393, 1.0)]), ('S', [(0.15, 1.0)]), ('S', [(0.06438, 1.0)]), ('P', [(1780.0, 0.0002198446), (421.8, 0.0019114223), (136.7, 0.010359717), (51.81, 0.0406802081), (21.6, 0.1210234909), (9.563, 0.2599738383), (4.35, 0.3864550392), (2.006, 0.359300883)]), ('P', [(1780.0, -0.0001949178), (421.8, -0.0017205029), (136.7, -0.0092286076), (51.81, -0.0370911628), (21.6, -0.1105534617), (9.563, -0.2481580056), (4.35, -0.3646226314), (2.006, -0.4056489575)]), ('P', [(0.9205, 1.0)]), ('P', [(0.35, 1.0)]), ('P', [(0.1381, 1.0)]), ('P', [(0.05338, 1.0)]), ('D', [(0.126, 1.0)]), ('D', [(0.321, 1.0)]), ('D', [(0.817, 1.0)]), ('D', [(2.082, 1.0)]), ('F', [(0.169, 1.0)]), ('F', [(0.341, 1.0)]), ('F', [(0.688, 1.0)]), ('G', [(0.32, 1.0)]), ('G', [(0.705, 1.0)]), ('H', [(0.583, 1.0)])],
    15 : [('S', [(4666000.0, 1.9675e-06), (698600.0, 1.52953e-05), (159000.0, 8.04772e-05), (45040.0, 0.0003397142), (14720.0, 0.0012328271), (5323.0, 0.0040131802), (2076.0, 0.0119115992), (861.1, 0.032248932), (375.7, 0.0786590119), (170.8, 0.1664468101), (80.29, 0.283019973), (38.77, 0.3319196856), (18.93, 0.2033383299), (8.796, 0.0383157241), (4.358, -0.0003846941), (2.174, 0.0015873333)]), ('S', [(4666000.0, -6.242e-07), (698600.0, -4.8541e-06), (159000.0, -2.55294e-05), (45040.0, -0.0001078823), (14720.0, -0.0003915674), (5323.0, -0.0012795767), (2076.0, -0.0038185621), (861.1, -0.0105157277), (375.7, -0.0264692941), (170.8, -0.0601357105), (80.29, -0.1170651179), (38.77, -0.1812016964), (18.93, -0.1563633636), (8.796, 0.1026345338), (4.358, 0.4937765215), (2.174, 0.5549089436)]), ('S', [(4666000.0, 3.533e-07), (698600.0, 2.7454e-06), (159000.0, 1.44558e-05), (45040.0, 6.10002e-05), (14720.0, 0.0002218318), (5323.0, 0.0007234592), (2076.0, 0.002166033), (861.1, 0.0059538443), (375.7, 0.0150874497), (170.8, 0.0343235317), (80.29, 0.0680628812), (38.77, 0.1071419495), (18.93, 0.0980176833), (8.796, -0.0726696644), (4.358, -0.3680179283), (2.174, -0.6797671011)]), ('S', [(1.095, 1.0)]), ('S', [(0.44, 1.0)]), ('S', [(0.1945, 1.0)]), ('S', [(0.08376, 1.0)]), ('P', [(2010.0, 0.0002307582), (476.3, 0.002004283), (154.4, 0.0108736329), (58.51, 0.0427344398), (24.4, 0.1267137016), (10.8, 0.2691272781), (4.913, 0.3917647834), (2.269, 0.3379128229)]), ('P', [(2010.0, -0.0002050518), (476.3, -0.0017975814), (154.4, -0.0097160712), (58.51, -0.038882968), (24.4, -0.1166557998), (10.8, -0.2572854461), (4.913, -0.3789047254), (2.269, -0.3747467088)]), ('P', [(1.043, 1.0)]), ('P', [(0.4313, 1.0)]), ('P', [(0.1767, 1.0)]), ('P', [(0.07009, 1.0)]), ('D', [(0.166, 1.0)]), ('D', [(0.418, 1.0)]), ('D', [(1.054, 1.0)]), ('D', [(2.656, 1.0)]), ('F', [(0.219, 1.0)]), ('F', [(0.45, 1.0)]), ('F', [(0.923, 1.0)]), ('G', [(0.412, 1.0)]), ('G', [(0.903, 1.0)]), ('H', [(0.745, 1.0)])],
    16 : [('S', [(5481000.0, 1.8932e-06), (820600.0, 1.47199e-05), (186700.0, 7.75018e-05), (52880.0, 0.0003271962), (17250.0, 0.0011935487), (6226.0, 0.0038836005), (2429.0, 0.0115326215), (1007.0, 0.0312721467), (439.5, 0.076432215), (199.8, 0.1626861966), (93.92, 0.2793043019), (45.34, 0.333116736), (22.15, 0.2098181976), (10.34, 0.0415938709), (5.119, -0.0004505138), (2.553, 0.0016884067)]), ('S', [(5481000.0, -6.06e-07), (820600.0, -4.7132e-06), (186700.0, -2.48083e-05), (52880.0, -0.0001048284), (17250.0, -0.0003825343), (6226.0, -0.0012491004), (2429.0, -0.003730242), (1007.0, -0.0102820798), (439.5, -0.0259331288), (199.8, -0.0591714705), (93.92, -0.1161521304), (45.34, -0.1817118812), (22.15, -0.1619558785), (10.34, 0.0938788512), (5.119, 0.4993562326), (2.553, 0.5582348456)]), ('S', [(5481000.0, 3.335e-07), (820600.0, 2.5926e-06), (186700.0, 1.36583e-05), (52880.0, 5.76484e-05), (17250.0, 0.000210694), (6226.0, 0.0006869303), (2429.0, 0.0020570299), (1007.0, 0.0056634548), (439.5, 0.0143705524), (199.8, 0.0328725993), (93.92, 0.0656748971), (45.34, 0.1047814928), (22.15, 0.0987641133), (10.34, -0.0643719004), (5.119, -0.3710959026), (2.553, -0.6832746516)]), ('S', [(1.282, 1.0)]), ('S', [(0.545, 1.0)]), ('S', [(0.2411, 1.0)]), ('S', [(0.1035, 1.0)]), ('P', [(2200.0, 0.0002502034), (521.4, 0.0021737697), (169.0, 0.0117606044), (64.05, 0.0461253326), (26.72, 0.135195193), (11.83, 0.2816388588), (5.378, 0.3962776168), (2.482, 0.3106272)]), ('P', [(2200.0, -0.0002253877), (521.4, -0.0019644659), (169.0, -0.0106632532), (64.05, -0.0423682051), (26.72, -0.1269464712), (11.83, -0.2725158123), (5.378, -0.3991826608), (2.482, -0.3258322635)]), ('P', [(1.116, 1.0)]), ('P', [(0.4848, 1.0)]), ('P', [(0.2006, 1.0)]), ('P', [(0.07951, 1.0)]), ('D', [(0.205, 1.0)]), ('D', [(0.512, 1.0)]), ('D', [(1.281, 1.0)]), ('D', [(3.203, 1.0)]), ('F', [(0.255, 1.0)]), ('F', [(0.529, 1.0)]), ('F', [(1.096, 1.0)]), ('G', [(0.463, 1.0)]), ('G', [(1.071, 1.0)]), ('H', [(0.872, 1.0)])],
    17 : [('S', [(6410000.0, 1.8133e-06), (959600.0, 1.41106e-05), (218300.0, 7.42341e-05), (61810.0, 0.0003141033), (20140.0, 0.0011463189), (7264.0, 0.0037385502), (2832.0, 0.0110936214), (1175.0, 0.0301125438), (512.6, 0.0739079806), (233.0, 0.1582440414), (109.5, 0.2747287664), (52.86, 0.3340365349), (25.84, 0.2175698083), (12.17, 0.0457237667), (6.03, -0.0001347271), (3.012, 0.0016391854)]), ('S', [(6410000.0, -5.905e-07), (959600.0, -4.5963e-06), (218300.0, -2.41757e-05), (61810.0, -0.0001023715), (20140.0, -0.0003737874), (7264.0, -0.0012230674), (2832.0, -0.0036500564), (1175.0, -0.0100650697), (512.6, -0.0254835818), (233.0, -0.0583882623), (109.5, -0.1156433426), (52.86, -0.1831481779), (25.84, -0.1696450267), (12.17, 0.0804205999), (6.03, 0.5000359888), (3.012, 0.5701947515)]), ('S', [(6410000.0, 3.202e-07), (959600.0, 2.4914e-06), (218300.0, 1.31144e-05), (61810.0, 5.54804e-05), (20140.0, 0.0002028408), (7264.0, 0.0006628917), (2832.0, 0.001983047), (1175.0, 0.0054643385), (512.6, 0.013912481), (233.0, 0.0319816739), (109.5, 0.0644301789), (52.86, 0.1042432454), (25.84, 0.1019828003), (12.17, -0.0543054424), (6.03, -0.3715200463), (3.012, -0.6919676946)]), ('S', [(1.511, 1.0)]), ('S', [(0.6604, 1.0)]), ('S', [(0.2926, 1.0)]), ('S', [(0.1254, 1.0)]), ('P', [(2548.0, 0.0002454045), (603.7, 0.002136032), (195.6, 0.0116134597), (74.15, 0.0457920748), (30.94, 0.1353451209), (13.69, 0.2841951849), (6.229, 0.3994843566), (2.878, 0.3038846443)]), ('P', [(2548.0, -0.0002232937), (603.7, -0.0019442446), (195.6, -0.0106407227), (74.15, -0.0423984276), (30.94, -0.1287408136), (13.69, -0.2778876935), (6.229, -0.4126405444), (2.878, -0.3025497677)]), ('P', [(1.282, 1.0)]), ('P', [(0.5641, 1.0)]), ('P', [(0.2348, 1.0)]), ('P', [(0.09312, 1.0)]), ('D', [(0.25, 1.0)]), ('D', [(0.618, 1.0)]), ('D', [(1.529, 1.0)]), ('D', [(3.781, 1.0)]), ('F', [(0.32, 1.0)]), ('F', [(0.656, 1.0)]), ('F', [(1.345, 1.0)]), ('G', [(0.556, 1.0)]), ('G', [(1.302, 1.0)]), ('H', [(1.053, 1.0)])],
    18 : [('S', [(7401000.0, 1.7498e-06), (1108000.0, 1.36086e-05), (252100.0, 7.16227e-05), (71380.0, 0.0003029992), (23260.0, 0.0011059677), (8390.0, 0.0036063438), (3271.0, 0.0107121224), (1357.0, 0.029103815), (592.0, 0.0716528349), (269.1, 0.1541248512), (126.5, 0.2703896465), (61.03, 0.3348207046), (29.86, 0.2243235237), (14.17, 0.0499977636), (7.022, 6.45834e-05), (3.511, 0.0016862388)]), ('S', [(7401000.0, -5.824e-07), (1108000.0, -4.5081e-06), (252100.0, -2.36936e-05), (71380.0, -0.0001002843), (23260.0, -0.000366284), (8390.0, -0.0011979713), (3271.0, -0.0035792831), (1357.0, -0.0098727066), (592.0, -0.0250682495), (269.1, -0.0576023831), (126.5, -0.1150618124), (61.03, -0.1844096053), (29.86, -0.1763658956), (14.17, 0.0678455207), (7.022, 0.5001784465), (3.511, 0.5813771403)]), ('S', [(7401000.0, 3.198e-07), (1108000.0, 2.4182e-06), (252100.0, 1.27103e-05), (71380.0, 5.37391e-05), (23260.0, 0.0001965305), (8390.0, 0.0006420917), (3271.0, 0.0019225378), (1357.0, 0.0053008684), (592.0, 0.0135304094), (269.1, 0.0312108025), (126.5, 0.0633851122), (61.03, 0.1039158065), (29.86, 0.1048705203), (14.17, -0.0452485261), (7.022, -0.3718417944), (3.511, -0.6997587959)]), ('S', [(1.758, 1.0)]), ('S', [(0.7841, 1.0)]), ('S', [(0.348, 1.0)]), ('S', [(0.1491, 1.0)]), ('P', [(2927.0, 0.0002407085), (693.5, 0.0020993276), (224.7, 0.0114486809), (85.17, 0.0454872466), (35.53, 0.135258042), (15.73, 0.2861129003), (7.165, 0.4022113837), (3.322, 0.2982085026)]), ('P', [(2927.0, -0.0002206457), (693.5, -0.0019216333), (224.7, -0.0105712777), (85.17, -0.0423874965), (35.53, -0.1299353644), (15.73, -0.2824073871), (7.165, -0.423527221), (3.322, -0.2831482549)]), ('P', [(1.478, 1.0)]), ('P', [(0.6552, 1.0)]), ('P', [(0.2751, 1.0)]), ('P', [(0.1097, 1.0)]), ('D', [(0.309, 1.0)]), ('D', [(0.77, 1.0)]), ('D', [(1.917, 1.0)]), ('D', [(4.776, 1.0)]), ('F', [(0.408, 1.0)]), ('F', [(0.825, 1.0)]), ('F', [(1.668, 1.0)]), ('G', [(0.665, 1.0)]), ('G', [(1.562, 1.0)]), ('H', [(1.264, 1.0)])],

    }





CC_PV6Z = {
    1 : [('S', [(1776.776, 0.0011228867), (254.0177, 0.0094934962), (54.69804, 0.0534391966), (15.01834, 0.2261851001), (4.915078, 0.7793854178)]), ('S', [(1.794924, 1.0)]), ('S', [(0.710716, 1.0)]), ('S', [(0.304802, 1.0)]), ('S', [(0.138046, 1.0)]), ('S', [(0.062157, 1.0)]), ('P', [(8.649, 1.0)]), ('P', [(3.43, 1.0)]), ('P', [(1.36, 1.0)]), ('P', [(0.539, 1.0)]), ('P', [(0.214, 1.0)]), ('D', [(4.453, 1.0)]), ('D', [(1.958, 1.0)]), ('D', [(0.861, 1.0)]), ('D', [(0.378, 1.0)]), ('F', [(4.1, 1.0)]), ('F', [(1.78, 1.0)]), ('F', [(0.773, 1.0)]), ('G', [(3.199, 1.0)]), ('G', [(1.326, 1.0)]), ('H', [(2.653, 1.0)])],
    2 : [('S', [(4785.0, 0.001339635), (717.0, 0.0104938072), (163.2, 0.0544784886), (46.26, 0.2259517643), (15.1, 0.7783279154)]), ('S', [(5.437, 1.0)]), ('S', [(2.088, 1.0)]), ('S', [(0.8297, 1.0)]), ('S', [(0.3366, 1.0)]), ('S', [(0.1369, 1.0)]), ('P', [(0.387, 1.0)]), ('P', [(0.984, 1.0)]), ('P', [(2.498, 1.0)]), ('P', [(6.342, 1.0)]), ('P', [(16.104, 1.0)]), ('D', [(0.747, 1.0)]), ('D', [(1.91, 1.0)]), ('D', [(4.886, 1.0)]), ('D', [(12.498, 1.0)]), ('F', [(1.292, 1.0)]), ('F', [(3.462, 1.0)]), ('F', [(9.276, 1.0)]), ('G', [(2.236, 1.0)]), ('G', [(6.586, 1.0)]), ('H', [(4.159, 1.0)])],
    4 : [('S', [(126512.3, 6.9118e-06), (18942.5, 5.52945e-05), (4310.41, 0.0002914481), (1220.8, 0.0012314547), (398.24, 0.0044661834), (143.755, 0.0143523804), (56.0546, 0.0411414169), (23.2162, 0.1031876131), (10.0637, 0.216714147), (4.5065, 0.3562970652), (2.06881, 0.3900209544)]), ('S', [(126512.3, -4.5614e-06), (18942.5, -4.10524e-05), (4310.41, -0.0002098232), (1220.8, -0.0008894678), (398.24, -0.0032340136), (143.755, -0.010427299), (56.0546, -0.0303559383), (23.2162, -0.0779219374), (10.0637, -0.173852176), (4.5065, -0.3203269904), (2.06881, -0.4961359954)]), ('S', [(0.953986, 1.0)]), ('S', [(0.345883, 1.0)]), ('S', [(0.167457, 1.0)]), ('S', [(0.07587, 1.0)]), ('S', [(0.034438, 1.0)]), ('P', [(74.0856, 0.0063289699), (17.5667, 0.0512450314), (5.61489, 0.2379251125), (2.03369, 0.7895512599)]), ('P', [(0.791862, 1.0)]), ('P', [(0.33348, 1.0)]), ('P', [(0.148577, 1.0)]), ('P', [(0.067788, 1.0)]), ('P', [(0.030603, 1.0)]), ('D', [(2.0855, 1.0)]), ('D', [(1.0449, 1.0)]), ('D', [(0.5235, 1.0)]), ('D', [(0.2623, 1.0)]), ('D', [(0.1314, 1.0)]), ('F', [(1.6751, 1.0)]), ('F', [(0.8242, 1.0)]), ('F', [(0.4055, 1.0)]), ('F', [(0.1995, 1.0)]), ('G', [(0.7784, 1.0)]), ('G', [(0.4496, 1.0)]), ('G', [(0.2597, 1.0)]), ('H', [(0.7282, 1.0)]), ('H', [(0.3464, 1.0)]), ('I', [(0.6007, 1.0)])],
    5 : [('S', [(210400.0, 6.7928e-06), (31500.0, 5.28047e-05), (7169.0, 0.0002777488), (2030.0, 0.0011717926), (662.5, 0.0042469298), (239.2, 0.0136745415), (93.26, 0.0393902921), (38.64, 0.0996972515), (16.78, 0.2127603494), (7.541, 0.3563470664), (3.482, 0.397087629)]), ('S', [(210400.0, -4.8734e-06), (31500.0, -3.77892e-05), (7169.0, -0.000199023), (2030.0, -0.0008386308), (662.5, -0.0030527465), (239.2, -0.0098542213), (93.26, -0.0288541684), (38.64, -0.0748182051), (16.78, -0.1702838005), (7.541, -0.3213671855), (3.482, -0.5006831377)]), ('S', [(1.618, 1.0)]), ('S', [(0.627, 1.0)]), ('S', [(0.2934, 1.0)]), ('S', [(0.131, 1.0)]), ('S', [(0.05815, 1.0)]), ('P', [(192.5, 0.0016145712), (45.64, 0.0137329516), (14.75, 0.0699918424), (5.503, 0.2533872664), (2.222, 0.7500601367)]), ('P', [(0.959, 1.0)]), ('P', [(0.4314, 1.0)]), ('P', [(0.1969, 1.0)]), ('P', [(0.09033, 1.0)]), ('P', [(0.04066, 1.0)]), ('D', [(2.886, 1.0)]), ('D', [(1.267, 1.0)]), ('D', [(0.556, 1.0)]), ('D', [(0.244, 1.0)]), ('D', [(0.107, 1.0)]), ('F', [(1.651, 1.0)]), ('F', [(0.8002, 1.0)]), ('F', [(0.3878, 1.0)]), ('F', [(0.188, 1.0)]), ('G', [(1.6469, 1.0)]), ('G', [(0.7889, 1.0)]), ('G', [(0.3779, 1.0)]), ('H', [(1.312, 1.0)]), ('H', [(0.5806, 1.0)]), ('I', [(0.9847, 1.0)])],
    6 : [('S', [(312100.0, 6.6319e-06), (46740.0, 5.15817e-05), (10640.0, 0.0002712427), (3013.0, 0.0011450559), (982.8, 0.0041541772), (354.8, 0.013381552), (138.4, 0.0385968765), (57.35, 0.0983134532), (24.92, 0.2113284711), (11.23, 0.356640751), (5.201, 0.399338141)]), ('S', [(312100.0, -4.7233e-06), (46740.0, -3.66543e-05), (10640.0, -0.0001931083), (3013.0, -0.0008141622), (982.8, -0.0029672791), (354.8, -0.0095819909), (138.4, -0.0281114907), (57.35, -0.0734155953), (24.92, -0.1688283258), (11.23, -0.3224223287), (5.201, -0.501901709)]), ('S', [(2.426, 1.0)]), ('S', [(0.9673, 1.0)]), ('S', [(0.4456, 1.0)]), ('S', [(0.1971, 1.0)]), ('S', [(0.08635, 1.0)]), ('P', [(295.2, 0.0015305902), (69.98, 0.0131059939), (22.64, 0.0680699606), (8.485, 0.2526322378), (3.459, 0.7508954637)]), ('P', [(1.504, 1.0)]), ('P', [(0.6783, 1.0)]), ('P', [(0.3087, 1.0)]), ('P', [(0.14, 1.0)]), ('P', [(0.06178, 1.0)]), ('D', [(4.542, 1.0)]), ('D', [(1.979, 1.0)]), ('D', [(0.8621, 1.0)]), ('D', [(0.3756, 1.0)]), ('D', [(0.1636, 1.0)]), ('F', [(2.631, 1.0)]), ('F', [(1.255, 1.0)]), ('F', [(0.5988, 1.0)]), ('F', [(0.2857, 1.0)]), ('G', [(2.652, 1.0)]), ('G', [(1.204, 1.0)]), ('G', [(0.547, 1.0)]), ('H', [(2.03, 1.0)]), ('H', [(0.8511, 1.0)]), ('I', [(1.491, 1.0)])],
    7 : [('S', [(432300.0, 6.5484e-06), (64700.0, 5.09695e-05), (14720.0, 0.0002681786), (4170.0, 0.0011304667), (1361.0, 0.0041026188), (491.2, 0.0132280842), (191.6, 0.0382040981), (79.41, 0.0975780719), (34.53, 0.2108431354), (15.58, 0.3572944642), (7.232, 0.3996489526)]), ('S', [(432300.0, -4.6276e-06), (64700.0, -3.60429e-05), (14720.0, -0.0001900343), (4170.0, -0.0008000178), (1361.0, -0.00291707), (491.2, -0.0094292536), (191.6, -0.0277105095), (79.41, -0.0726188799), (34.53, -0.1682406781), (15.58, -0.3238088026), (7.232, -0.5015020982)]), ('S', [(3.382, 1.0)]), ('S', [(1.369, 1.0)]), ('S', [(0.6248, 1.0)]), ('S', [(0.2747, 1.0)]), ('S', [(0.1192, 1.0)]), ('P', [(415.9, 0.0014827249), (98.61, 0.0127515741), (31.92, 0.0669620986), (12.0, 0.2523829374), (4.919, 0.7511976314)]), ('P', [(2.148, 1.0)]), ('P', [(0.9696, 1.0)]), ('P', [(0.4399, 1.0)]), ('P', [(0.1978, 1.0)]), ('P', [(0.08603, 1.0)]), ('D', [(6.717, 1.0)]), ('D', [(2.896, 1.0)]), ('D', [(1.249, 1.0)]), ('D', [(0.538, 1.0)]), ('D', [(0.232, 1.0)]), ('F', [(3.829, 1.0)]), ('F', [(1.795, 1.0)]), ('F', [(0.841, 1.0)]), ('F', [(0.394, 1.0)]), ('G', [(3.856, 1.0)]), ('G', [(1.702, 1.0)]), ('G', [(0.751, 1.0)]), ('H', [(2.875, 1.0)]), ('H', [(1.17, 1.0)]), ('I', [(2.099, 1.0)])],
    8 : [('S', [(570800.0, 6.5065e-06), (85480.0, 5.05396e-05), (19460.0, 0.0002657342), (5512.0, 0.0011211905), (1798.0, 0.0040717701), (648.9, 0.0131276017), (253.1, 0.0379693385), (104.9, 0.0971397951), (45.65, 0.2105332122), (20.62, 0.3578228026), (9.587, 0.3996429721)]), ('S', [(570800.0, -4.59e-06), (85480.0, -3.55906e-05), (19460.0, -0.0001875339), (5512.0, -0.0007903526), (1798.0, -0.0028837525), (648.9, -0.0093220519), (253.1, -0.027442059), (104.9, -0.0720882399), (45.65, -0.1678001404), (20.62, -0.3249261488), (9.587, -0.5010455375)]), ('S', [(4.493, 1.0)]), ('S', [(1.837, 1.0)]), ('S', [(0.8349, 1.0)]), ('S', [(0.3658, 1.0)]), ('S', [(0.157, 1.0)]), ('P', [(525.6, 0.0014797642), (124.6, 0.0127282456), (40.34, 0.0670229118), (15.18, 0.2539198677), (6.245, 0.7493711065)]), ('P', [(2.732, 1.0)]), ('P', [(1.227, 1.0)]), ('P', [(0.5492, 1.0)]), ('P', [(0.2418, 1.0)]), ('P', [(0.1025, 1.0)]), ('D', [(8.253, 1.0)]), ('D', [(3.597, 1.0)]), ('D', [(1.568, 1.0)]), ('D', [(0.684, 1.0)]), ('D', [(0.298, 1.0)]), ('F', [(5.43, 1.0)]), ('F', [(2.416, 1.0)]), ('F', [(1.075, 1.0)]), ('F', [(0.478, 1.0)]), ('G', [(5.211, 1.0)]), ('G', [(2.19, 1.0)]), ('G', [(0.92, 1.0)]), ('H', [(3.872, 1.0)]), ('H', [(1.505, 1.0)]), ('I', [(2.773, 1.0)])],
    9 : [('S', [(723500.0, 6.5168e-06), (108400.0, 5.06109e-05), (24680.0, 0.0002660648), (6990.0, 0.0011228991), (2282.0, 0.0040673343), (824.6, 0.0131101515), (321.8, 0.0378923212), (133.5, 0.0970438676), (58.11, 0.2108361065), (26.28, 0.3581658182), (12.24, 0.3988257998)]), ('S', [(723500.0, -4.5905e-06), (108400.0, -3.55497e-05), (24680.0, -0.0001871784), (6990.0, -0.0007889962), (2282.0, -0.0028714453), (824.6, -0.009281239), (321.8, -0.0273081946), (133.5, -0.0718591881), (58.11, -0.1679179854), (26.28, -0.3258900705), (12.24, -0.4999215669)]), ('S', [(5.747, 1.0)]), ('S', [(2.365, 1.0)]), ('S', [(1.071, 1.0)]), ('S', [(0.4681, 1.0)]), ('S', [(0.1994, 1.0)]), ('P', [(660.0, 0.0014707889), (156.4, 0.012672887), (50.64, 0.0669957173), (19.08, 0.2551343605), (7.872, 0.747961356)]), ('P', [(3.449, 1.0)]), ('P', [(1.545, 1.0)]), ('P', [(0.6864, 1.0)]), ('P', [(0.2986, 1.0)]), ('P', [(0.1245, 1.0)]), ('D', [(10.573, 1.0)]), ('D', [(4.613, 1.0)]), ('D', [(2.013, 1.0)]), ('D', [(0.878, 1.0)]), ('D', [(0.383, 1.0)]), ('F', [(7.563, 1.0)]), ('F', [(3.33, 1.0)]), ('F', [(1.466, 1.0)]), ('F', [(0.645, 1.0)]), ('G', [(6.735, 1.0)]), ('G', [(2.783, 1.0)]), ('G', [(1.15, 1.0)]), ('H', [(5.088, 1.0)]), ('H', [(1.937, 1.0)]), ('I', [(3.581, 1.0)])],
    10 : [('S', [(902400.0, 6.4569e-06), (135100.0, 5.0179e-05), (30750.0, 0.0002638325), (8710.0, 0.0011134543), (2842.0, 0.0040396234), (1026.0, 0.0130374676), (400.1, 0.0377405416), (165.9, 0.0967943419), (72.21, 0.21082414), (32.66, 0.3586501519), (15.22, 0.3985795167)]), ('S', [(902400.0, -4.5125e-06), (135100.0, -3.51554e-05), (30750.0, -0.0001851517), (8710.0, -0.0007804845), (2842.0, -0.0028452422), (1026.0, -0.0092079116), (400.1, -0.0271417038), (165.9, -0.0715447701), (72.21, -0.1678190251), (32.66, -0.3267206864), (15.22, -0.4994236511)]), ('S', [(7.149, 1.0)]), ('S', [(2.957, 1.0)]), ('S', [(1.335, 1.0)]), ('S', [(0.5816, 1.0)]), ('S', [(0.2463, 1.0)]), ('P', [(815.6, 0.0014608751), (193.3, 0.0126013201), (62.6, 0.0668956161), (23.61, 0.2559896696), (9.762, 0.7470043852)]), ('P', [(4.281, 1.0)]), ('P', [(1.915, 1.0)]), ('P', [(0.8476, 1.0)]), ('P', [(0.366, 1.0)]), ('P', [(0.151, 1.0)]), ('D', [(13.317, 1.0)]), ('D', [(5.803, 1.0)]), ('D', [(2.529, 1.0)]), ('D', [(1.102, 1.0)]), ('D', [(0.48, 1.0)]), ('F', [(10.356, 1.0)]), ('F', [(4.538, 1.0)]), ('F', [(1.989, 1.0)]), ('F', [(0.871, 1.0)]), ('G', [(8.345, 1.0)]), ('G', [(3.417, 1.0)]), ('G', [(1.399, 1.0)]), ('H', [(6.519, 1.0)]), ('H', [(2.447, 1.0)]), ('I', [(4.489, 1.0)])],
    13 : [('S', [(3652000.0, 1.9002e-06), (546800.0, 1.45016e-05), (124500.0, 7.62086e-05), (35440.0, 0.0003158356), (11840.0, 0.0010975235), (4434.0, 0.0033700794), (1812.0, 0.0093232495), (791.5, 0.0238018794), (361.0, 0.0568254969), (169.5, 0.1224817879), (81.68, 0.2239222072), (40.28, 0.313481289), (20.25, 0.2750059578), (10.23, 0.1105764477), (4.802, 0.0119228422), (2.339, 0.0006528735)]), ('S', [(3652000.0, -8.318e-07), (546800.0, -6.3214e-06), (124500.0, -3.29378e-05), (35440.0, -0.0001365754), (11840.0, -0.0004754355), (4434.0, -0.001461407), (1812.0, -0.0040726428), (791.5, -0.010496845), (361.0, -0.0257603555), (169.5, -0.0581550165), (81.68, -0.1177325136), (40.28, -0.1986631885), (20.25, -0.2476024409), (10.23, -0.0982253499), (4.802, 0.3604742998), (2.339, 0.7927646526)]), ('S', [(3652000.0, 5.475e-07), (546800.0, 4.9275e-06), (124500.0, 2.51849e-05), (35440.0, 0.0001040247), (11840.0, 0.0003608014), (4434.0, 0.0011119691), (1812.0, 0.0030917229), (791.5, 0.0080044252), (361.0, 0.0195971543), (169.5, 0.0446298717), (81.68, 0.0904883295), (40.28, 0.156292698), (20.25, 0.1979118765), (10.23, 0.084207429), (4.802, -0.3351461586), (2.339, -0.8281623579)]), ('S', [(1.163, 1.0)]), ('S', [(0.5882, 1.0)]), ('S', [(0.2311, 1.0)]), ('S', [(0.1027, 1.0)]), ('S', [(0.04521, 1.0)]), ('P', [(2884.0, 7.35571e-05), (683.2, 0.0006492163), (222.0, 0.0036537586), (84.82, 0.0152649424), (35.81, 0.0499684431), (16.22, 0.1290708002), (7.702, 0.2511041001), (3.741, 0.3593402559), (1.831, 0.3651591066)]), ('P', [(2884.0, -6.69432e-05), (683.2, -0.0005447506), (222.0, -0.0033463249), (84.82, -0.0128606319), (35.81, -0.0465623658), (16.22, -0.109708244), (7.702, -0.248694111), (3.741, -0.2905294478), (1.831, -0.4615911649)]), ('P', [(0.8878, 1.0)]), ('P', [(0.3989, 1.0)]), ('P', [(0.1718, 1.0)]), ('P', [(0.07298, 1.0)]), ('P', [(0.03069, 1.0)]), ('D', [(2.2143, 1.0)]), ('D', [(0.9449, 1.0)]), ('D', [(0.4032, 1.0)]), ('D', [(0.1721, 1.0)]), ('D', [(0.07343, 1.0)]), ('F', [(0.8756, 1.0)]), ('F', [(0.4472, 1.0)]), ('F', [(0.2284, 1.0)]), ('F', [(0.1167, 1.0)]), ('G', [(0.6952, 1.0)]), ('G', [(0.3771, 1.0)]), ('G', [(0.2046, 1.0)]), ('H', [(0.656, 1.0)]), ('H', [(0.33, 1.0)]), ('I', [(0.5302, 1.0)])],
    14 : [('S', [(4465000.0, 1.7002e-06), (668500.0, 1.36019e-05), (152200.0, 7.14097e-05), (43300.0, 0.0002973405), (14410.0, 0.0010384415), (5394.0, 0.0031751325), (2212.0, 0.0087335897), (968.1, 0.0223860493), (441.2, 0.0537346195), (207.1, 0.1166648916), (99.8, 0.2160074237), (49.24, 0.3096081736), (24.74, 0.2839836832), (12.47, 0.1222486522), (5.795, 0.0141971339), (2.83, 0.0003121425)]), ('S', [(4465000.0, -8.22e-07), (668500.0, -5.9183e-06), (152200.0, -3.12356e-05), (43300.0, -0.0001300389), (14410.0, -0.0004552182), (5394.0, -0.0013927802), (2212.0, -0.0038597373), (968.1, -0.0099797833), (441.2, -0.0246121957), (207.1, -0.0558507834), (99.8, -0.1141882516), (49.24, -0.1956353174), (24.74, -0.2525893761), (12.47, -0.1158486719), (5.795, 0.3504127887), (2.83, 0.8081742127)]), ('S', [(4465000.0, 4.688e-07), (668500.0, 4.2191e-06), (152200.0, 2.29707e-05), (43300.0, 9.51642e-05), (14410.0, 0.0003323714), (5394.0, 0.0010182098), (2212.0, 0.0028188285), (968.1, 0.0073088899), (441.2, 0.018021657), (207.1, 0.0411582712), (99.8, 0.0845639171), (49.24, 0.1477735558), (24.74, 0.1954404941), (12.47, 0.0942139394), (5.795, -0.3129091825), (2.83, -0.8527553882)]), ('S', [(1.407, 1.0)]), ('S', [(0.6995, 1.0)]), ('S', [(0.3083, 1.0)]), ('S', [(0.1385, 1.0)]), ('S', [(0.06145, 1.0)]), ('P', [(3572.0, 6.84432e-05), (846.0, 0.0006051339), (274.8, 0.0034230742), (105.0, 0.0144353455), (44.35, 0.0478809904), (20.08, 0.1259846249), (9.53, 0.2500416424), (4.634, 0.3631580312), (2.28, 0.3649828024)]), ('P', [(3572.0, -6.24774e-05), (846.0, -0.0005496058), (274.8, -0.0031248457), (105.0, -0.0131929795), (44.35, -0.0443145294), (20.08, -0.1182911759), (9.53, -0.2408601005), (4.634, -0.3542951237), (2.28, -0.3925620369)]), ('P', [(1.116, 1.0)]), ('P', [(0.4991, 1.0)]), ('P', [(0.2254, 1.0)]), ('P', [(0.1001, 1.0)]), ('P', [(0.04332, 1.0)]), ('D', [(3.2386, 1.0)]), ('D', [(1.3767, 1.0)]), ('D', [(0.5853, 1.0)]), ('D', [(0.2488, 1.0)]), ('D', [(0.1058, 1.0)]), ('F', [(1.351, 1.0)]), ('F', [(0.66, 1.0)]), ('F', [(0.3225, 1.0)]), ('F', [(0.1575, 1.0)]), ('G', [(0.8528, 1.0)]), ('G', [(0.4631, 1.0)]), ('G', [(0.2515, 1.0)]), ('H', [(0.8557, 1.0)]), ('H', [(0.4231, 1.0)]), ('I', [(0.6946, 1.0)])],
    15 : [('S', [(5384000.0, 1.6003e-06), (806200.0, 1.28022e-05), (183600.0, 6.72117e-05), (52250.0, 0.0002797488), (17390.0, 0.0009768704), (6523.0, 0.0029689179), (2687.0, 0.0081254173), (1178.0, 0.0209236497), (536.2, 0.0505678206), (251.5, 0.1104982743), (121.3, 0.2069931059), (59.88, 0.3047901647), (30.05, 0.2930031087), (15.12, 0.1355846501), (7.01, 0.0173238218), (3.441, -3.52061e-05)]), ('S', [(5384000.0, -6.664e-07), (806200.0, -5.8313e-06), (183600.0, -3.04894e-05), (52250.0, -0.000126456), (17390.0, -0.0004426792), (6523.0, -0.0013461979), (2687.0, -0.0037108744), (1178.0, -0.0096354778), (536.2, -0.0238980108), (251.5, -0.0544911901), (121.3, -0.1122469099), (59.88, -0.1960100727), (30.05, -0.262042077), (15.12, -0.1397086797), (7.01, 0.3327474538), (3.441, 0.8307190348)]), ('S', [(5384000.0, 4.328e-07), (806200.0, 4.3276e-06), (183600.0, 2.1638e-05), (52250.0, 9.04467e-05), (17390.0, 0.0003159142), (6523.0, 0.0009611583), (2687.0, 0.002649352), (1178.0, 0.0068886615), (536.2, 0.0171087035), (251.5, 0.0391958692), (121.3, 0.0813193547), (59.88, 0.1444684474), (30.05, 0.1988459433), (15.12, 0.1104574666), (7.01, -0.2897305753), (3.441, -0.8812925395)]), ('S', [(1.712, 1.0)]), ('S', [(0.8337, 1.0)]), ('S', [(0.3912, 1.0)]), ('S', [(0.1777, 1.0)]), ('S', [(0.07939, 1.0)]), ('P', [(4552.0, 6.00201e-05), (1078.0, 0.0005314086), (350.1, 0.0030250123), (133.8, 0.0129127442), (56.52, 0.0436564168), (25.58, 0.1178655962), (12.14, 0.242751234), (5.902, 0.3663337433), (2.91, 0.3776244447)]), ('P', [(4552.0, -5.44446e-05), (1078.0, -0.0004803422), (350.1, -0.0027468196), (133.8, -0.0117380875), (56.52, -0.0401977028), (25.58, -0.1102034038), (12.14, -0.2335016202), (5.902, -0.3582316427), (2.91, -0.4038233159)]), ('P', [(1.435, 1.0)]), ('P', [(0.657, 1.0)]), ('P', [(0.3005, 1.0)]), ('P', [(0.134, 1.0)]), ('P', [(0.05783, 1.0)]), ('D', [(4.3008, 1.0)]), ('D', [(1.8346, 1.0)]), ('D', [(0.7826, 1.0)]), ('D', [(0.3339, 1.0)]), ('D', [(0.1424, 1.0)]), ('F', [(1.816, 1.0)]), ('F', [(0.8806, 1.0)]), ('F', [(0.427, 1.0)]), ('F', [(0.207, 1.0)]), ('G', [(1.0616, 1.0)]), ('G', [(0.5791, 1.0)]), ('G', [(0.3159, 1.0)]), ('H', [(1.085, 1.0)]), ('H', [(0.5277, 1.0)]), ('I', [(0.889, 1.0)])],
    16 : [('S', [(6297000.0, 1.6003e-06), (943100.0, 1.24024e-05), (214900.0, 6.49127e-05), (61250.0, 0.0002693525), (20450.0, 0.0009348823), (7719.0, 0.0028088478), (3198.0, 0.0076754971), (1402.0, 0.0198936801), (637.2, 0.0482683144), (298.9, 0.1057776312), (144.3, 0.2002620597), (71.21, 0.3007866663), (35.73, 0.2987462684), (17.97, 0.1463755495), (8.341, 0.0201198242), (4.112, -0.0002488485)]), ('S', [(6297000.0, -6.739e-07), (943100.0, -5.7283e-06), (214900.0, -3.0158e-05), (61250.0, -0.0001253493), (20450.0, -0.0004358582), (7719.0, -0.0013102704), (3198.0, -0.003604802), (1402.0, -0.0094190532), (637.2, -0.0234315501), (298.9, -0.0535243013), (144.3, -0.1110793225), (71.21, -0.1968387689), (35.73, -0.2692094321), (17.97, -0.1592680248), (8.341, 0.3164529606), (4.112, 0.8502908486)]), ('S', [(6297000.0, 4.059e-07), (943100.0, 4.0589e-06), (214900.0, 2.11061e-05), (61250.0, 8.76715e-05), (20450.0, 0.0003048207), (7719.0, 0.0009164917), (3198.0, 0.0025233963), (1402.0, 0.0065960612), (637.2, 0.0164526086), (298.9, 0.037707666), (144.3, 0.0789696798), (71.21, 0.1420765042), (35.73, 0.2008720023), (17.97, 0.1231634118), (8.341, -0.2693710286), (4.112, -0.9057519203)]), ('S', [(2.045, 1.0)]), ('S', [(0.977, 1.0)]), ('S', [(0.4766, 1.0)]), ('S', [(0.2185, 1.0)]), ('S', [(0.09759, 1.0)]), ('P', [(5266.0, 5.94418e-05), (1247.0, 0.0005267928), (405.0, 0.0030016391), (154.8, 0.0128622681), (65.38, 0.0437236874), (29.59, 0.118586909), (14.04, 0.2451365154), (6.824, 0.3696755578), (3.369, 0.3707189144)]), ('P', [(5266.0, -5.31344e-05), (1247.0, -0.0004710183), (405.0, -0.0027002655), (154.8, -0.0115749063), (65.38, -0.0399426753), (29.59, -0.1100305246), (14.04, -0.2348871406), (6.824, -0.3610585331), (3.369, -0.3994631077)]), ('P', [(1.666, 1.0)]), ('P', [(0.7681, 1.0)]), ('P', [(0.3504, 1.0)]), ('P', [(0.1556, 1.0)]), ('P', [(0.06681, 1.0)]), ('D', [(5.0755, 1.0)]), ('D', [(2.1833, 1.0)]), ('D', [(0.9392, 1.0)]), ('D', [(0.404, 1.0)]), ('D', [(0.1738, 1.0)]), ('F', [(2.212, 1.0)]), ('F', [(1.063, 1.0)]), ('F', [(0.51, 1.0)]), ('F', [(0.245, 1.0)]), ('G', [(1.3473, 1.0)]), ('G', [(0.7009, 1.0)]), ('G', [(0.3647, 1.0)]), ('H', [(1.2861, 1.0)]), ('H', [(0.6115, 1.0)]), ('I', [(1.0409, 1.0)])],
    17 : [('S', [(7733000.0, 1.4351e-06), (1158000.0, 1.11513e-05), (263700.0, 5.86009e-05), (75010.0, 0.0002445779), (24890.0, 0.0008584974), (9318.0, 0.0026108298), (3840.0, 0.0071395896), (1684.0, 0.018460924), (766.3, 0.0449054044), (359.5, 0.0994065603), (173.4, 0.190828764), (85.61, 0.2936369579), (42.93, 0.3065521229), (21.55, 0.1622487603), (10.05, 0.0249444128), (4.978, -0.0005132678)]), ('S', [(7733000.0, -7.065e-07), (1158000.0, -5.4882e-06), (263700.0, -2.88579e-05), (75010.0, -0.0001203955), (24890.0, -0.000423323), (9318.0, -0.0012884738), (3840.0, -0.0035451834), (1684.0, -0.0092411724), (766.3, -0.0230079471), (359.5, -0.0530107064), (173.4, -0.1109923631), (85.61, -0.1999955605), (42.93, -0.2832405354), (21.55, -0.1918716575), (10.05, 0.2863109316), (4.978, 0.8807417416)]), ('S', [(7733000.0, 4.939e-07), (1158000.0, 3.8358e-06), (263700.0, 2.01763e-05), (75010.0, 8.41449e-05), (24890.0, 0.0002960377), (9318.0, 0.0009006072), (3840.0, 0.0024812673), (1684.0, 0.0064669085), (766.3, 0.016153411), (359.5, 0.0373140826), (173.4, 0.0788994735), (85.61, 0.1441574663), (42.93, 0.2113226121), (21.55, 0.1484015818), (10.05, -0.2425034873), (4.978, -0.9401464485)]), ('S', [(2.478, 1.0)]), ('S', [(1.18, 1.0)]), ('S', [(0.5828, 1.0)]), ('S', [(0.2668, 1.0)]), ('S', [(0.1183, 1.0)]), ('P', [(6091.0, 5.81164e-05), (1442.0, 0.0005161727), (468.3, 0.0029442363), (179.0, 0.0126720521), (75.61, 0.0432981483), (34.22, 0.1183069377), (16.23, 0.2461172954), (7.89, 0.3725191968), (3.898, 0.3668955051)]), ('P', [(6091.0, -5.18656e-05), (1442.0, -0.0004593079), (468.3, -0.0026352031), (179.0, -0.0113571487), (75.61, -0.0393862768), (34.22, -0.1095323746), (16.23, -0.2354923335), (7.89, -0.3656876803), (3.898, -0.3943022468)]), ('P', [(1.933, 1.0)]), ('P', [(0.9057, 1.0)]), ('P', [(0.414, 1.0)]), ('P', [(0.1836, 1.0)]), ('P', [(0.07859, 1.0)]), ('D', [(6.2428, 1.0)]), ('D', [(2.6906, 1.0)]), ('D', [(1.1596, 1.0)]), ('D', [(0.4998, 1.0)]), ('D', [(0.2154, 1.0)]), ('F', [(2.5327, 1.0)]), ('F', [(1.2406, 1.0)]), ('F', [(0.6077, 1.0)]), ('F', [(0.2977, 1.0)]), ('G', [(1.5388, 1.0)]), ('G', [(0.805, 1.0)]), ('G', [(0.4212, 1.0)]), ('H', [(1.5613, 1.0)]), ('H', [(0.7397, 1.0)]), ('I', [(1.2572, 1.0)])],
    18 : [('S', [(9149000.0, 1.3003e-06), (1370000.0, 1.04028e-05), (311900.0, 5.49147e-05), (88650.0, 0.0002296613), (29330.0, 0.0008105162), (10930.0, 0.0024859632), (4480.0, 0.0068387245), (1962.0, 0.017624602), (894.1, 0.0428866416), (419.6, 0.095510781), (202.3, 0.1851133858), (99.84, 0.2891191331), (50.07, 0.3102487702), (25.14, 0.1722289484), (11.81, 0.0285303115), (5.882, -0.0005758536)]), ('S', [(9149000.0, -7.239e-07), (1370000.0, -5.4293e-06), (311900.0, -2.82325e-05), (88650.0, -0.0001179974), (29330.0, -0.0004169723), (10930.0, -0.0012804162), (4480.0, -0.0035422736), (1962.0, -0.0092037942), (894.1, -0.0228987272), (419.6, -0.0530381852), (202.3, -0.1117920034), (99.84, -0.2036739427), (50.07, -0.2937138926), (25.14, -0.2142991346), (11.81, 0.2644950674), (5.882, 0.9008193667)]), ('S', [(9149000.0, 4.041e-07), (1370000.0, 3.6367e-06), (311900.0, 1.97997e-05), (88650.0, 8.24314e-05), (29330.0, 0.0002909343), (10930.0, 0.0008930068), (4480.0, 0.0024749622), (1962.0, 0.0064280324), (894.1, 0.0160506853), (419.6, 0.0372573734), (202.3, 0.0793470852), (99.84, 0.1465056384), (50.07, 0.2188977768), (25.14, 0.165669321), (11.81, -0.2229462088), (5.882, -0.964746316)]), ('S', [(2.939, 1.0)]), ('S', [(1.405, 1.0)]), ('S', [(0.6963, 1.0)]), ('S', [(0.3188, 1.0)]), ('S', [(0.141, 1.0)]), ('P', [(7050.0, 5.6242e-05), (1669.0, 0.000499008), (542.1, 0.0028546751), (207.1, 0.012341316), (87.52, 0.0424044728), (39.61, 0.1169150773), (18.78, 0.2457339703), (9.13, 0.3748889711), (4.516, 0.3661008741)]), ('P', [(7050.0, -4.98567e-05), (1669.0, -0.0004426565), (542.1, -0.002545186), (207.1, -0.0110283082), (87.52, -0.0384470156), (39.61, -0.1080950836), (18.78, -0.2349622936), (9.13, -0.3699262362), (4.516, -0.3915747396)]), ('P', [(2.245, 1.0)]), ('P', [(1.065, 1.0)]), ('P', [(0.4885, 1.0)]), ('P', [(0.2166, 1.0)]), ('P', [(0.09255, 1.0)]), ('D', [(7.6327, 1.0)]), ('D', [(3.2876, 1.0)]), ('D', [(1.416, 1.0)]), ('D', [(0.6099, 1.0)]), ('D', [(0.2627, 1.0)]), ('F', [(3.0582, 1.0)]), ('F', [(1.5292, 1.0)]), ('F', [(0.7647, 1.0)]), ('F', [(0.3824, 1.0)]), ('G', [(1.845, 1.0)]), ('G', [(0.9657, 1.0)]), ('G', [(0.5055, 1.0)]), ('H', [(1.8743, 1.0)]), ('H', [(0.8871, 1.0)]), ('I', [(1.5066, 1.0)])],
}



STO_2G = {

    1 : [("S", [(1.3097563770E+00, 4.3012849830E-01), (2.3313597490E-01, 6.7891353050E-01)])],
    2 : [("S", [(2.4328792850E+00, 4.3012849830E-01), (4.3305128630E-01, 6.7891353050E-01)])],
    3 : [("S", [(6.1638450310E+00, 4.3012849830E-01), (1.0971613080E+00, 6.7891353050E-01)]),("S", [(2.4591632200E-01, 4.9471769200E-02), (6.2370872960E-02, 9.6378240810E-01)]),("P", [(2.4591632200E-01, 5.1154070760E-01), (6.2370872960E-02, 6.1281989610E-01)])],
    4 : [("S", [(1.1535669070E+01, 4.3012849830E-01), (2.0533432790E+00, 6.7891353050E-01)]),("S", [(5.0816302470E-01, 4.9471769200E-02), (1.2888356170E-01, 9.6378240810E-01)]),("P", [(5.0816302470E-01, 5.1154070760E-01), (1.2888356170E-01, 6.1281989610E-01)])],
    5 : [("S", [(1.8656873100E+01, 4.3012849830E-01), (3.3209140060E+00, 6.7891353050E-01)]),("S", [(8.6454956950E-01, 4.9471769200E-02), (2.1927260030E-01, 9.6378240810E-01)]),("P", [(8.6454956950E-01, 5.1154070760E-01), (2.1927260030E-01, 6.1281989610E-01)])],
    6 : [("S", [(2.7385033030E+01, 4.3012849830E-01), (4.8745220750E+00, 6.7891353050E-01)]),("S", [(1.1367481980E+00, 4.9471769200E-02), (2.8830936030E-01, 9.6378240810E-01)]),("P", [(1.1367481980E+00, 5.1154070760E-01), (2.8830936030E-01, 6.1281989610E-01)])],
    7 : [("S", [(3.7896475340E+01, 4.3012849830E-01), (6.7455535070E+00, 6.7891353050E-01)]),("S", [(1.4610887720E+00, 4.9471769200E-02), (3.7057069440E-01, 9.6378240810E-01)]),("P", [(1.4610887720E+00, 5.1154070760E-01), (3.7057069440E-01, 6.1281989610E-01)])],
    8 : [("S", [(4.9980971170E+01, 4.3012849830E-01), (8.8965876730E+00, 6.7891353050E-01)]),("S", [(1.9452365310E+00, 4.9471769200E-02), (4.9336335060E-01, 9.6378240810E-01)]),("P", [(1.9452365310E+00, 5.1154070760E-01), (4.9336335060E-01, 6.1281989610E-01)])],
    9 : [("S", [(6.3735201950E+01, 4.3012849830E-01), (1.1344833820E+01, 6.7891353050E-01)]),("S", [(2.4985482560E+00, 4.9471769200E-02), (6.3369781470E-01, 9.6378240810E-01)]),("P", [(2.4985482560E+00, 5.1154070760E-01), (6.3369781470E-01, 6.1281989610E-01)])],
    10 : [("S", [(7.9159167670E+01, 4.3012849830E-01), (1.4090291940E+01, 6.7891353050E-01)]),("S", [(3.1870755330E+00, 4.9471769200E-02), (8.0832651360E-01, 9.6378240810E-01)]),("P", [(3.1870755330E+00, 5.1154070760E-01), (8.0832651360E-01, 6.1281989610E-01)])],
    11 : [("S", [(9.5891015770E+01, 4.3012849830E-01), (1.7068552470E+01, 6.7891353050E-01)]),("S", [(4.6533516030E+00, 4.9471769200E-02), (1.1802128440E+00, 9.6378240810E-01)]),("P", [(4.6533516030E+00, 5.1154070760E-01), (1.1802128440E+00, 6.1281989610E-01)]),("S", [(5.9406079590E-01, -2.9839860450E-01), (2.0381633030E-01, 1.2279828870E+00)]),("P", [(5.9406079590E-01, 3.4804719120E-01), (2.0381633030E-01, 7.2225232210E-01)])],
    12 : [("S", [(1.1442318260E+02, 4.3012849830E-01), (2.0367268820E+01, 6.7891353050E-01)]),("S", [(5.8443550900E+00, 4.9471769200E-02), (1.4822827780E+00, 9.6378240810E-01)]),("P", [(5.8443550900E+00, 5.1154070760E-01), (1.4822827780E+00, 6.1281989610E-01)]),("S", [(5.6059941230E-01, -2.9839860450E-01), (1.9233606360E-01, 1.2279828870E+00)]),("P", [(5.6059941230E-01, 3.4804719120E-01), (1.9233606360E-01, 7.2225232210E-01)])],
    13 : [("S", [(1.3437746070E+02, 4.3012849830E-01), (2.3919120270E+01, 6.7891353050E-01)]),("S", [(7.3043295540E+00, 4.9471769200E-02), (1.8525708540E+00, 9.6378240810E-01)]),("P", [(7.3043295540E+00, 5.1154070760E-01), (1.8525708540E+00, 6.1281989610E-01)]),("S", [(5.6059941230E-01, -2.9839860450E-01), (1.9233606360E-01, 1.2279828870E+00)]),("P", [(5.6059941230E-01, 3.4804719120E-01), (1.9233606360E-01, 7.2225232210E-01)])],
    14 : [("S", [(1.5593469120E+02, 4.3012849830E-01), (2.7756296420E+01, 6.7891353050E-01)]),("S", [(8.9639957560E+00, 4.9471769200E-02), (2.2735060280E+00, 9.6378240810E-01)]),("P", [(8.9639957560E+00, 5.1154070760E-01), (2.2735060280E+00, 6.1281989610E-01)]),("S", [(5.9406079590E-01, -2.9839860450E-01), (2.0381633030E-01, 1.2279828870E+00)]),("P", [(5.9406079590E-01, 3.4804719120E-01), (2.0381633030E-01, 7.2225232210E-01)])],
    15 : [("S", [(1.7909487400E+02, 4.3012849830E-01), (3.1878797290E+01, 6.7891353050E-01)]),("S", [(1.0834189380E+01, 4.9471769200E-02), (2.7478365170E+00, 9.6378240810E-01)]),("P", [(1.0834189380E+01, 5.1154070760E-01), (2.7478365170E+00, 6.1281989610E-01)]),("S", [(7.0026431780E-01, -2.9839860450E-01), (2.4025369880E-01, 1.2279828870E+00)]),("P", [(7.0026431780E-01, 3.4804719120E-01), (2.4025369880E-01, 7.2225232210E-01)])],
    16 : [("S", [(2.0385800920E+02, 4.3012849830E-01), (3.6286622870E+01, 6.7891353050E-01)]),("S", [(1.2881442770E+01, 4.9471769200E-02), (3.2670740350E+00, 9.6378240810E-01)]),("P", [(1.2881442770E+01, 5.1154070760E-01), (3.2670740350E+00, 6.1281989610E-01)]),("S", [(8.1519689630E-01, -2.9839860450E-01), (2.7968591940E-01, 1.2279828870E+00)]),("P", [(8.1519689630E-01, 3.4804719120E-01), (2.7968591940E-01, 7.2225232210E-01)])],
    17 : [("S", [(2.2994410390E+02, 4.3012849830E-01), (4.0929934590E+01, 6.7891353050E-01)]),("S", [(1.5057610090E+01, 4.9471769200E-02), (3.8190075330E+00, 9.6378240810E-01)]),("P", [(1.5057610090E+01, 5.1154070760E-01), (3.8190075330E+00, 6.1281989610E-01)]),("S", [(8.5544754610E-01, -2.9839860450E-01), (2.9349551570E-01, 1.2279828870E+00)]),("P", [(8.5544754610E-01, 3.4804719120E-01), (2.9349551570E-01, 7.2225232210E-01)])],
    18 : [("S", [(2.5789661860E+02, 4.3012849830E-01), (4.5905468100E+01, 6.7891353050E-01)]),("S", [(1.7455294230E+01, 4.9471769200E-02), (4.4271235440E+00, 9.6378240810E-01)]),("P", [(1.7455294230E+01, 5.1154070760E-01), (4.4271235440E+00, 6.1281989610E-01)]),("S", [(1.0530927850E+00, -2.9839860450E-01), (3.6130562470E-01, 1.2279828870E+00)]),("P", [(1.0530927850E+00, 3.4804719120E-01), (3.6130562470E-01, 7.2225232210E-01)])],
}


STO_4G = {

    1 : [("S", [(8.0214201550E+00, 5.6752420800E-02), (1.4678210610E+00, 2.6014135500E-01), (4.0777676350E-01, 5.3284611430E-01), (1.3533744200E-01, 2.9162544050E-01)])],
    2 : [("S", [(1.4899829670E+01, 5.6752420800E-02), (2.7264852580E+00, 2.6014135500E-01), (7.5744745990E-01, 5.3284611430E-01), (2.5139000270E-01, 2.9162544050E-01)])],
    3 : [("S", [(3.7749608730E+01, 5.6752420800E-02), (6.9077133070E+00, 2.6014135500E-01), (1.9190383970E+00, 5.3284611430E-01), (6.3691159220E-01, 2.9162544050E-01)]),("S", [(1.4870423520E+00, -6.2207145650E-02), (3.2191276200E-01, 2.9768045960E-05), (1.0466603000E-01, 5.5885492210E-01), (4.0198682960E-02, 4.9776732180E-01)]),("P", [(1.4870423520E+00, 4.3684348840E-02), (3.2191276200E-01, 2.8637939840E-01), (1.0466603000E-01, 5.8357531410E-01), (4.0198682960E-02, 2.4631343780E-01)])],
    4 : [("S", [(7.0648595420E+01, 5.6752420800E-02), (1.2927822540E+01, 2.6014135500E-01), (3.5914906620E+00, 5.3284611430E-01), (1.1919834640E+00, 2.9162544050E-01)]),("S", [(3.0728336100E+00, -6.2207145650E-02), (6.6520254330E-01, 2.9768045960E-05), (2.1628253860E-01, 5.5885492210E-01), (8.3066809720E-02, 4.9776732180E-01)]),("P", [(3.0728336100E+00, 4.3684348840E-02), (6.6520254330E-01, 2.8637939840E-01), (2.1628253860E-01, 5.8357531410E-01), (8.3066809720E-02, 2.4631343780E-01)])],
    5 : [("S", [(1.1426141570E+02, 5.6752420800E-02), (2.0908431330E+01, 2.6014135500E-01), (5.8085911720E+00, 5.3284611430E-01), (1.9278191920E+00, 2.9162544050E-01)]),("S", [(5.2278832690E+00, -6.2207145650E-02), (1.1317245540E+00, 2.9768045960E-05), (3.6796651180E-01, 5.5885492210E-01), (1.4132349480E-01, 4.9776732180E-01)]),("P", [(5.2278832690E+00, 4.3684348840E-02), (1.1317245540E+00, 2.8637939840E-01), (3.6796651180E-01, 5.8357531410E-01), (1.4132349480E-01, 2.4631343780E-01)])],
    6 : [("S", [(1.6771581320E+02, 5.6752420800E-02), (3.0689927490E+01, 2.6014135500E-01), (8.5259979140E+00, 5.3284611430E-01), (2.8297020610E+00, 2.9162544050E-01)]),("S", [(6.8738532720E+00, -6.2207145650E-02), (1.4880417420E+00, 2.9768045960E-05), (4.8381872370E-01, 5.5885492210E-01), (1.8581841200E-01, 4.9776732180E-01)]),("P", [(6.8738532720E+00, 4.3684348840E-02), (1.4880417420E+00, 2.8637939840E-01), (4.8381872370E-01, 5.8357531410E-01), (1.8581841200E-01, 2.4631343780E-01)])],
    7 : [("S", [(2.3209167480E+02, 5.6752420800E-02), (4.2469917020E+01, 2.6014135500E-01), (1.1798608000E+01, 5.3284611430E-01), (3.9158519280E+00, 2.9162544050E-01)]),("S", [(8.8351227240E+00, -6.2207145650E-02), (1.9126144960E+00, 2.9768045960E-05), (6.2186340490E-01, 5.5885492210E-01), (2.3883670620E-01, 4.9776732180E-01)]),("P", [(8.8351227240E+00, 4.3684348840E-02), (1.9126144960E+00, 2.8637939840E-01), (6.2186340490E-01, 5.8357531410E-01), (2.3883670620E-01, 2.4631343780E-01)])],
    8 : [("S", [(3.0610148310E+02, 5.6752420800E-02), (5.6012800120E+01, 2.6014135500E-01), (1.5560969220E+01, 5.3284611430E-01), (5.1645457940E+00, 2.9162544050E-01)]),("S", [(1.1762737350E+01, -6.2207145650E-02), (2.5463802460E+00, 2.9768045960E-05), (8.2792465150E-01, 5.5885492210E-01), (3.1797786330E-01, 4.9776732180E-01)]),("P", [(1.1762737350E+01, 4.3684348840E-02), (2.5463802460E+00, 2.8637939840E-01), (8.2792465150E-01, 5.8357531410E-01), (3.1797786330E-01, 2.4631343780E-01)])],
    9 : [("S", [(3.9033735010E+02, 5.6752420800E-02), (7.1426925960E+01, 2.6014135500E-01), (1.9843182160E+01, 5.3284611430E-01), (6.5857737740E+00, 2.9162544050E-01)]),("S", [(1.5108582650E+01, -6.2207145650E-02), (3.2706839610E+00, 2.9768045960E-05), (1.0634232190E+00, 5.5885492210E-01), (4.0842490000E-01, 4.9776732180E-01)]),("P", [(1.5108582650E+01, 4.3684348840E-02), (3.2706839610E+00, 2.8637939840E-01), (1.0634232190E+00, 5.8357531410E-01), (4.0842490000E-01, 2.4631343780E-01)])],
    10 : [("S", [(4.8479927580E+02, 5.6752420800E-02), (8.8712294540E+01, 2.6014135500E-01), (2.4645246830E+01, 5.3284611430E-01), (8.1795358680E+00, 2.9162544050E-01)]),("S", [(1.9272068880E+01, -6.2207145650E-02), (4.1719893950E+00, 2.9768045960E-05), (1.3564717490E+00, 5.5885492210E-01), (5.2097493120E-01, 4.9776732180E-01)]),("P", [(1.9272068880E+01, 4.3684348840E-02), (4.1719893950E+00, 2.8637939840E-01), (1.3564717490E+00, 5.8357531410E-01), (5.2097493120E-01, 2.4631343780E-01)])],
    11 : [("S", [(5.8727114460E+02, 5.6752420800E-02), (1.0746338400E+02, 2.6014135500E-01), (2.9854504810E+01, 5.3284611430E-01), (9.9084417640E+00, 2.9162544050E-01)]),("S", [(2.8138558910E+01, -6.2207145650E-02), (6.0913942390E+00, 2.9768045960E-05), (1.9805429530E+00, 5.5885492210E-01), (7.6065957840E-01, 4.9776732180E-01)]),("P", [(2.8138558910E+01, 4.3684348840E-02), (6.0913942390E+00, 2.8637939840E-01), (1.9805429530E+00, 5.8357531410E-01), (7.6065957840E-01, 2.4631343780E-01)]),("S", [(2.8227284340E+00, -8.5290196440E-02), (7.7626110020E-01, -2.1320740340E-01), (3.0552959200E-01, 5.9208439280E-01), (1.3597471930E-01, 6.1155847460E-01)]),("P", [(2.8227284340E+00, -2.5049451810E-02), (7.7626110020E-01, 1.6866044610E-01), (3.0552959200E-01, 6.4095531510E-01), (1.3597471930E-01, 2.7795089570E-01)])],
    12 : [("S", [(7.0076881420E+02, 5.6752420800E-02), (1.2823205920E+02, 2.6014135500E-01), (3.5624270210E+01, 5.3284611430E-01), (1.1823375030E+01, 2.9162544050E-01)]),("S", [(3.5340490900E+01, -6.2207145650E-02), (7.6504579840E+00, 2.9768045960E-05), (2.4874536200E+00, 5.5885492210E-01), (9.5534682480E-01, 4.9776732180E-01)]),("P", [(3.5340490900E+01, 4.3684348840E-02), (7.6504579840E+00, 2.8637939840E-01), (2.4874536200E+00, 5.8357531410E-01), (9.5534682480E-01, 2.4631343780E-01)]),("S", [(2.6637339350E+00, -8.5290196440E-02), (7.3253700550E-01, -2.1320740340E-01), (2.8832017000E-01, 5.9208439280E-01), (1.2831573510E-01, 6.1155847460E-01)]),("P", [(2.6637339350E+00, -2.5049451810E-02), (7.3253700550E-01, 1.6866044610E-01), (2.8832017000E-01, 6.4095531510E-01), (1.2831573510E-01, 2.7795089570E-01)])],
    13 : [("S", [(8.2297600590E+02, 5.6752420800E-02), (1.5059446970E+02, 2.6014135500E-01), (4.1836792690E+01, 5.3284611430E-01), (1.3885255390E+01, 2.9162544050E-01)]),("S", [(4.4168875460E+01, -6.2207145650E-02), (9.5616138130E+00, 2.9768045960E-05), (3.1088427570E+00, 5.5885492210E-01), (1.1940013810E+00, 4.9776732180E-01)]),("P", [(4.4168875460E+01, 4.3684348840E-02), (9.5616138130E+00, 2.8637939840E-01), (3.1088427570E+00, 5.8357531410E-01), (1.1940013810E+00, 2.4631343780E-01)]),("S", [(2.6637339350E+00, -8.5290196440E-02), (7.3253700550E-01, -2.1320740340E-01), (2.8832017000E-01, 5.9208439280E-01), (1.2831573510E-01, 6.1155847460E-01)]),("P", [(2.6637339350E+00, -2.5049451810E-02), (7.3253700550E-01, 1.6866044610E-01), (2.8832017000E-01, 6.4095531510E-01), (1.2831573510E-01, 2.7795089570E-01)])],
    14 : [("S", [(9.5500025560E+02, 5.6752420800E-02), (1.7475328080E+02, 2.6014135500E-01), (4.8548374950E+01, 5.3284611430E-01), (1.6112769210E+01, 2.9162544050E-01)]),("S", [(5.4204784880E+01, -6.2207145650E-02), (1.1734172860E+01, 2.9768045960E-05), (3.8152239810E+00, 5.5885492210E-01), (1.4652985230E+00, 4.9776732180E-01)]),("P", [(5.4204784880E+01, 4.3684348840E-02), (1.1734172860E+01, 2.8637939840E-01), (3.8152239810E+00, 5.8357531410E-01), (1.4652985230E+00, 2.4631343780E-01)]),("S", [(2.8227284340E+00, -8.5290196440E-02), (7.7626110020E-01, -2.1320740340E-01), (3.0552959200E-01, 5.9208439280E-01), (1.3597471930E-01, 6.1155847460E-01)]),("P", [(2.8227284340E+00, -2.5049451810E-02), (7.7626110020E-01, 1.6866044610E-01), (3.0552959200E-01, 6.4095531510E-01), (1.3597471930E-01, 2.7795089570E-01)])],
    15 : [("S", [(1.0968415630E+03, 5.6752420800E-02), (2.0070849250E+02, 2.6014135500E-01), (5.5759016990E+01, 5.3284611430E-01), (1.8505916480E+01, 2.9162544050E-01)]),("S", [(6.5513741970E+01, -6.2207145650E-02), (1.4182319420E+01, 2.9768045960E-05), (4.6112091390E+00, 5.5885492210E-01), (1.7710095070E+00, 4.9776732180E-01)]),("P", [(6.5513741970E+01, 4.3684348840E-02), (1.4182319420E+01, 2.8637939840E-01), (4.6112091390E+00, 5.8357531410E-01), (1.7710095070E+00, 2.4631343780E-01)]),("S", [(3.3273631500E+00, -8.5290196440E-02), (9.1503757440E-01, -2.1320740340E-01), (3.6015080060E-01, 5.9208439280E-01), (1.6028366910E-01, 6.1155847460E-01)]),("P", [(3.3273631500E+00, -2.5049451810E-02), (9.1503757440E-01, 1.6866044610E-01), (3.6015080060E-01, 6.4095531510E-01), (1.6028366910E-01, 2.7795089570E-01)])],
    16 : [("S", [(1.2484999290E+03, 5.6752420800E-02), (2.2846010500E+02, 2.6014135500E-01), (6.3468718810E+01, 5.3284611430E-01), (2.1064697210E+01, 2.9162544050E-01)]),("S", [(7.7893369550E+01, -6.2207145650E-02), (1.6862243160E+01, 2.9768045960E-05), (5.4825538390E+00, 5.5885492210E-01), (2.1056635430E+00, 4.9776732180E-01)]),("P", [(7.7893369550E+01, 4.3684348840E-02), (1.6862243160E+01, 2.8637939840E-01), (5.4825538390E+00, 5.8357531410E-01), (2.1056635430E+00, 2.4631343780E-01)]),("S", [(3.8734746920E+00, -8.5290196440E-02), (1.0652203340E+00, -2.1320740340E-01), (4.1926142370E-01, 5.9208439280E-01), (1.8659061480E-01, 6.1155847460E-01)]),("P", [(3.8734746920E+00, -2.5049451810E-02), (1.0652203340E+00, 1.6866044610E-01), (4.1926142370E-01, 6.4095531510E-01), (1.8659061480E-01, 2.7795089570E-01)])],
    17 : [("S", [(1.4082605760E+03, 5.6752420800E-02), (2.5769433510E+02, 2.6014135500E-01), (7.1590308050E+01, 5.3284611430E-01), (2.3760179660E+01, 2.9162544050E-01)]),("S", [(9.1052532610E+01, -6.2207145650E-02), (1.9710919610E+01, 2.9768045960E-05), (6.4087664340E+00, 5.5885492210E-01), (2.4613904820E+00, 4.9776732180E-01)]),("P", [(9.1052532610E+01, 4.3684348840E-02), (1.9710919610E+01, 2.8637939840E-01), (6.4087664340E+00, 5.8357531410E-01), (2.4613904820E+00, 2.4631343780E-01)]),("S", [(4.0647289460E+00, -8.5290196440E-02), (1.1178159840E+00, -2.1320740340E-01), (4.3996261240E-01, 5.9208439280E-01), (1.9580359570E-01, 6.1155847460E-01)]),("P", [(4.0647289460E+00, -2.5049451810E-02), (1.1178159840E+00, 1.6866044610E-01), (4.3996261240E-01, 6.4095531510E-01), (1.9580359570E-01, 2.7795089570E-01)])],
    18 : [("S", [(1.5794518510E+03, 5.6752420800E-02), (2.8902022920E+02, 2.6014135500E-01), (8.0292984470E+01, 5.3284611430E-01), (2.6648519730E+01, 2.9162544050E-01)]),("S", [(1.0555119550E+02, -6.2207145650E-02), (2.2849569040E+01, 2.9768045960E-05), (7.4292602270E+00, 5.5885492210E-01), (2.8533276410E+00, 4.9776732180E-01)]),("P", [(1.0555119550E+02, 4.3684348840E-02), (2.2849569040E+01, 2.8637939840E-01), (7.4292602270E+00, 5.8357531410E-01), (2.8533276410E+00, 2.4631343780E-01)]),("S", [(5.0038564560E+00, -8.5290196440E-02), (1.3760796360E+00, -2.1320740340E-01), (5.4161293120E-01, 5.9208439280E-01), (2.4104266230E-01, 6.1155847460E-01)]),("P", [(5.0038564560E+00, -2.5049451810E-02), (1.3760796360E+00, 1.6866044610E-01), (5.4161293120E-01, 6.4095531510E-01), (2.4104266230E-01, 2.7795089570E-01)])],
}





STO_5G = {

    1 : [("S", [(1.7383547390E+01, 2.2140553120E-02), (3.1854892460E+00, 1.1354115200E-01), (8.8972990790E-01, 3.3181614840E-01), (3.0378741030E-01, 4.8257007130E-01), (1.1447849840E-01, 1.9357219660E-01)])],
    2 : [("S", [(3.2290029720E+01, 2.2140553120E-02), (5.9170628490E+00, 1.1354115200E-01), (1.6526779330E+00, 3.3181614840E-01), (5.6428669530E-01, 4.8257007130E-01), (2.1264440630E-01, 1.9357219660E-01)])],
    3 : [("S", [(8.1808719610E+01, 2.2140553120E-02), (1.4991232270E+01, 1.1354115200E-01), (4.1871582900E+00, 3.3181614840E-01), (1.4296540580E+00, 4.8257007130E-01), (5.3874730880E-01, 1.9357219660E-01)]),("S", [(3.2232283190E+00, -2.9408550880E-02), (6.6080222530E-01, -6.5327468830E-02), (2.1059829260E-01, 1.2899731810E-01), (8.1868808000E-02, 6.1228999380E-01), (3.4876713270E-02, 3.4612056550E-01)]),("P", [(3.2232283190E+00, 1.2556091280E-02), (6.6080222530E-01, 1.0755769620E-01), (2.1059829260E-01, 3.7359753670E-01), (8.1868808000E-02, 5.1023956370E-01), (3.4876713270E-02, 1.5682818010E-01)])],
    4 : [("S", [(1.5310545800E+02, 2.2140553120E-02), (2.8056171680E+01, 1.1354115200E-01), (7.8362892200E+00, 3.3181614840E-01), (2.6756052460E+00, 4.8257007130E-01), (1.0082684810E+00, 1.9357219660E-01)]),("S", [(6.6604991430E+00, -2.9408550880E-02), (1.3654858480E+00, -6.5327468830E-02), (4.3518162810E-01, 1.2899731810E-01), (1.6917421650E-01, 6.1228999380E-01), (7.2069458270E-02, 3.4612056550E-01)]),("P", [(6.6604991430E+00, 1.2556091280E-02), (1.3654858480E+00, 1.0755769620E-01), (4.3518162810E-01, 3.7359753670E-01), (1.6917421650E-01, 5.1023956370E-01), (7.2069458270E-02, 1.5682818010E-01)])],
    5 : [("S", [(2.4762058300E+02, 2.2140553120E-02), (4.5375819250E+01, 1.1354115200E-01), (1.2673790540E+01, 3.3181614840E-01), (4.3273109890E+00, 4.8257007130E-01), (1.6306931990E+00, 1.9357219660E-01)]),("S", [(1.1331662060E+01, -2.9408550880E-02), (2.3231328230E+00, -6.5327468830E-02), (7.4038462250E-01, 1.2899731810E-01), (2.8782002810E-01, 6.1228999380E-01), (1.2261344510E-01, 3.4612056550E-01)]),("P", [(1.1331662060E+01, 1.2556091280E-02), (2.3231328230E+00, 1.0755769620E-01), (7.4038462250E-01, 3.7359753670E-01), (2.8782002810E-01, 5.1023956370E-01), (1.2261344510E-01, 1.5682818010E-01)])],
    6 : [("S", [(3.6346379210E+02, 2.2140553120E-02), (6.6603782020E+01, 1.1354115200E-01), (1.8602912220E+01, 3.3181614840E-01), (6.3517371720E+00, 4.8257007130E-01), (2.3935729680E+00, 1.9357219660E-01)]),("S", [(1.4899372900E+01, -2.9408550880E-02), (3.0545582860E+00, -6.5327468830E-02), (9.7349060760E-01, 1.2899731810E-01), (3.7843856500E-01, 6.1228999380E-01), (1.6121760710E-01, 3.4612056550E-01)]),("P", [(1.4899372900E+01, 1.2556091280E-02), (3.0545582860E+00, 1.0755769620E-01), (9.7349060760E-01, 3.7359753670E-01), (3.7843856500E-01, 5.1023956370E-01), (1.6121760710E-01, 1.5682818010E-01)])],
    7 : [("S", [(5.0297535210E+02, 2.2140553120E-02), (9.2168907740E+01, 1.1354115200E-01), (2.5743434510E+01, 3.3181614840E-01), (8.7897812960E+00, 4.8257007130E-01), (3.3123195020E+00, 1.9357219660E-01)]),("S", [(1.9150508880E+01, -2.9408550880E-02), (3.9260944710E+00, -6.5327468830E-02), (1.2512500120E+00, 1.2899731810E-01), (4.8641584750E-01, 6.1228999380E-01), (2.0721672220E-01, 3.4612056550E-01)]),("P", [(1.9150508880E+01, 1.2556091280E-02), (3.9260944710E+00, 1.0755769620E-01), (1.2512500120E+00, 3.7359753670E-01), (4.8641584750E-01, 5.1023956370E-01), (2.0721672220E-01, 1.5682818010E-01)])],
    8 : [("S", [(6.6336503200E+02, 2.2140553120E-02), (1.2155989390E+02, 1.1354115200E-01), (3.3952546950E+01, 3.3181614840E-01), (1.1592682480E+01, 4.8257007130E-01), (4.3685578690E+00, 1.9357219660E-01)]),("S", [(2.5496239630E+01, -2.9408550880E-02), (5.2270488520E+00, -6.5327468830E-02), (1.6658654010E+00, 1.2899731810E-01), (6.4759506330E-01, 6.1228999380E-01), (2.7588025140E-01, 3.4612056550E-01)]),("P", [(2.5496239630E+01, 1.2556091280E-02), (5.2270488520E+00, 1.0755769620E-01), (1.6658654010E+00, 3.7359753670E-01), (6.4759506330E-01, 5.1023956370E-01), (2.7588025140E-01, 1.5682818010E-01)])],
    9 : [("S", [(8.4591602140E+02, 2.2140553120E-02), (1.5501188160E+02, 1.1354115200E-01), (4.3295926140E+01, 3.3181614840E-01), (1.4782865190E+01, 4.8257007130E-01), (5.5707384520E+00, 1.9357219660E-01)]),("S", [(3.2748503350E+01, -2.9408550880E-02), (6.7138538590E+00, -6.5327468830E-02), (2.1397115590E+00, 1.2899731810E-01), (8.3179988130E-01, 6.1228999380E-01), (3.5435285630E-01, 3.4612056550E-01)]),("P", [(3.2748503350E+01, 1.2556091280E-02), (6.7138538590E+00, 1.0755769620E-01), (2.1397115590E+00, 3.7359753670E-01), (8.3179988130E-01, 5.1023956370E-01), (3.5435285630E-01, 1.5682818010E-01)])],
    10 : [("S", [(1.0506283200E+03, 2.2140553120E-02), (1.9252487090E+02, 1.1354115200E-01), (5.3773572090E+01, 3.3181614840E-01), (1.8360329430E+01, 4.8257007130E-01), (6.9188612520E+00, 1.9357219660E-01)]),("S", [(4.1773039010E+01, -2.9408550880E-02), (8.5639968400E+00, -6.5327468830E-02), (2.7293538720E+00, 1.2899731810E-01), (1.0610197520E+00, 6.1228999380E-01), (4.5200220390E-01, 3.4612056550E-01)]),("P", [(4.1773039010E+01, 1.2556091280E-02), (8.5639968400E+00, 1.0755769620E-01), (2.7293538720E+00, 3.7359753670E-01), (1.0610197520E+00, 5.1023956370E-01), (4.5200220390E-01, 1.5682818010E-01)])],
    11 : [("S", [(1.2726992940E+03, 2.2140553120E-02), (2.3321879160E+02, 1.1354115200E-01), (6.5139674930E+01, 3.3181614840E-01), (2.2241146420E+01, 4.8257007130E-01), (8.3812987550E+00, 1.9357219660E-01)]),("S", [(6.0991537860E+01, -2.9408550880E-02), (1.2504030110E+01, -6.5327468830E-02), (3.9850461920E+00, 1.2899731810E-01), (1.5491625190E+00, 6.1228999380E-01), (6.5995460680E-01, 3.4612056550E-01)]),("P", [(6.0991537860E+01, 1.2556091280E-02), (1.2504030110E+01, 1.0755769620E-01), (3.9850461920E+00, 3.7359753670E-01), (1.5491625190E+00, 5.1023956370E-01), (6.5995460680E-01, 1.5682818010E-01)]),("S", [(5.2341600810E+00, -2.6622033910E-02), (1.4253627100E+00, -1.6034840720E-01), (5.4669580030E-01, -4.7794733070E-02), (2.4706474330E-01, 7.2751584410E-01), (1.1988673860E-01, 4.1238464080E-01)]),("P", [(5.2341600810E+00, -1.5668834480E-02), (1.4253627100E+00, 7.2149205060E-03), (5.4669580030E-01, 3.1708547620E-01), (2.4706474330E-01, 5.8188213820E-01), (1.1988673860E-01, 1.7017998240E-01)])],
    12 : [("S", [(1.5186647320E+03, 2.2140553120E-02), (2.7829130990E+02, 1.1354115200E-01), (7.7728751330E+01, 3.3181614840E-01), (2.6539532800E+01, 4.8257007130E-01), (1.0001092080E+01, 1.9357219660E-01)]),("S", [(7.6602035510E+01, -2.9408550880E-02), (1.5704377890E+01, -6.5327468830E-02), (5.0050000480E+00, 1.2899731810E-01), (1.9456633900E+00, 6.1228999380E-01), (8.2886688870E-01, 3.4612056550E-01)]),("P", [(7.6602035510E+01, 1.2556091280E-02), (1.5704377890E+01, 1.0755769620E-01), (5.0050000480E+00, 3.7359753670E-01), (1.9456633900E+00, 5.1023956370E-01), (8.2886688870E-01, 1.5682818010E-01)]),("S", [(4.9393380030E+00, -2.6622033910E-02), (1.3450769740E+00, -1.6034840720E-01), (5.1590232260E-01, -4.7794733070E-02), (2.3314844350E-01, 7.2751584410E-01), (1.1313393450E-01, 4.1238464080E-01)]),("P", [(4.9393380030E+00, -1.5668834480E-02), (1.3450769740E+00, 7.2149205060E-03), (5.1590232260E-01, 3.1708547620E-01), (2.3314844350E-01, 5.8188213820E-01), (1.1313393450E-01, 1.7017998240E-01)])],
    13 : [("S", [(1.7835049310E+03, 2.2140553120E-02), (3.2682257830E+02, 1.1354115200E-01), (9.1283881380E+01, 3.3181614840E-01), (3.1167766400E+01, 4.8257007130E-01), (1.1745184210E+01, 1.9357219660E-01)]),("S", [(9.5737939140E+01, -2.9408550880E-02), (1.9627478100E+01, -6.5327468830E-02), (6.2552957860E+00, 1.2899731810E-01), (2.4317082700E+00, 6.1228999380E-01), (1.0359255760E+00, 3.4612056550E-01)]),("P", [(9.5737939140E+01, 1.2556091280E-02), (1.9627478100E+01, 1.0755769620E-01), (6.2552957860E+00, 3.7359753670E-01), (2.4317082700E+00, 5.1023956370E-01), (1.0359255760E+00, 1.5682818010E-01)]),("S", [(4.9393380030E+00, -2.6622033910E-02), (1.3450769740E+00, -1.6034840720E-01), (5.1590232260E-01, -4.7794733070E-02), (2.3314844350E-01, 7.2751584410E-01), (1.1313393450E-01, 4.1238464080E-01)]),("P", [(4.9393380030E+00, -1.5668834480E-02), (1.3450769740E+00, 7.2149205060E-03), (5.1590232260E-01, 3.1708547620E-01), (2.3314844350E-01, 5.8188213820E-01), (1.1313393450E-01, 1.7017998240E-01)])],
    14 : [("S", [(2.0696200770E+03, 2.2140553120E-02), (3.7925242480E+02, 1.1354115200E-01), (1.0592791210E+02, 3.3181614840E-01), (3.6167791850E+01, 4.8257007130E-01), (1.3629381470E+01, 1.9357219660E-01)]),("S", [(1.1749120490E+02, -2.9408550880E-02), (2.4087170360E+01, -6.5327468830E-02), (7.6766039190E+00, 1.2899731810E-01), (2.9842331800E+00, 6.1228999380E-01), (1.2713052440E+00, 3.4612056550E-01)]),("P", [(1.1749120490E+02, 1.2556091280E-02), (2.4087170360E+01, 1.0755769620E-01), (7.6766039190E+00, 3.7359753670E-01), (2.9842331800E+00, 5.1023956370E-01), (1.2713052440E+00, 1.5682818010E-01)]),("S", [(5.2341600810E+00, -2.6622033910E-02), (1.4253627100E+00, -1.6034840720E-01), (5.4669580030E-01, -4.7794733070E-02), (2.4706474330E-01, 7.2751584410E-01), (1.1988673860E-01, 4.1238464080E-01)]),("P", [(5.2341600810E+00, -1.5668834480E-02), (1.4253627100E+00, 7.2149205060E-03), (5.4669580030E-01, 3.1708547620E-01), (2.4706474330E-01, 5.8188213820E-01), (1.1988673860E-01, 1.7017998240E-01)])],
    15 : [("S", [(2.3770101710E+03, 2.2140553120E-02), (4.3558084940E+02, 1.1354115200E-01), (1.2166084360E+02, 3.3181614840E-01), (4.1539609150E+01, 4.8257007130E-01), (1.5653683850E+01, 1.9357219660E-01)]),("S", [(1.4200385620E+02, -2.9408550880E-02), (2.9112571290E+01, -6.5327468830E-02), (9.2782039350E+00, 1.2899731810E-01), (3.6068454640E+00, 6.1228999380E-01), (1.5365426480E+00, 3.4612056550E-01)]),("P", [(1.4200385620E+02, 1.2556091280E-02), (2.9112571290E+01, 1.0755769620E-01), (9.2782039350E+00, 3.7359753670E-01), (3.6068454640E+00, 5.1023956370E-01), (1.5365426480E+00, 1.5682818010E-01)]),("S", [(6.1698997210E+00, -2.6622033910E-02), (1.6801826560E+00, -1.6034840720E-01), (6.4443162100E-01, -4.7794733070E-02), (2.9123386890E-01, 7.2751584410E-01), (1.4131955140E-01, 4.1238464080E-01)]),("P", [(6.1698997210E+00, -1.5668834480E-02), (1.6801826560E+00, 7.2149205060E-03), (6.4443162100E-01, 3.1708547620E-01), (2.9123386890E-01, 5.8188213820E-01), (1.4131955140E-01, 1.7017998240E-01)])],
    16 : [("S", [(2.7056752120E+03, 2.2140553120E-02), (4.9580785210E+02, 1.1354115200E-01), (1.3848267580E+02, 3.3181614840E-01), (4.7283218300E+01, 4.8257007130E-01), (1.7818091350E+01, 1.9357219660E-01)]),("S", [(1.6883723200E+02, -2.9408550880E-02), (3.4613749810E+01, -6.5327468830E-02), (1.1031434720E+01, 1.2899731810E-01), (4.2884032910E+00, 6.1228999380E-01), (1.8268912860E+00, 3.4612056550E-01)]),("P", [(1.6883723200E+02, 1.2556091280E-02), (3.4613749810E+01, 1.0755769620E-01), (1.1031434720E+01, 3.7359753670E-01), (4.2884032910E+00, 5.1023956370E-01), (1.8268912860E+00, 1.5682818010E-01)]),("S", [(7.1825494670E+00, -2.6622033910E-02), (1.9559467070E+00, -1.6034840720E-01), (7.5020052280E-01, -4.7794733070E-02), (3.3903333350E-01, 7.2751584410E-01), (1.6451396530E-01, 4.1238464080E-01)]),("P", [(7.1825494670E+00, -1.5668834480E-02), (1.9559467070E+00, 7.2149205060E-03), (7.5020052280E-01, 3.1708547620E-01), (3.3903333350E-01, 5.8188213820E-01), (1.6451396530E-01, 1.7017998240E-01)])],
    17 : [("S", [(3.0518990390E+03, 2.2140553120E-02), (5.5925245580E+02, 1.1354115200E-01), (1.5620320700E+02, 3.3181614840E-01), (5.3333677230E+01, 4.8257007130E-01), (2.0098131370E+01, 1.9357219660E-01)]),("S", [(1.9736028450E+02, -2.9408550880E-02), (4.0461333260E+01, -6.5327468830E-02), (1.2895065080E+01, 1.2899731810E-01), (5.0128782820E+00, 6.1228999380E-01), (2.1355229510E+00, 3.4612056550E-01)]),("P", [(1.9736028450E+02, 1.2556091280E-02), (4.0461333260E+01, 1.0755769620E-01), (1.2895065080E+01, 3.7359753670E-01), (5.0128782820E+00, 5.1023956370E-01), (2.1355229510E+00, 1.5682818010E-01)]),("S", [(7.5371905170E+00, -2.6622033910E-02), (2.0525223030E+00, -1.6034840720E-01), (7.8724195250E-01, -4.7794733070E-02), (3.5577323040E-01, 7.2751584410E-01), (1.7263690350E-01, 4.1238464080E-01)]),("P", [(7.5371905170E+00, -1.5668834480E-02), (2.0525223030E+00, 7.2149205060E-03), (7.8724195250E-01, 3.1708547620E-01), (3.5577323040E-01, 5.8188213820E-01), (1.7263690350E-01, 1.7017998240E-01)])],
    18 : [("S", [(3.4228946460E+03, 2.2140553120E-02), (6.2723642320E+02, 1.1354115200E-01), (1.7519161480E+02, 3.3181614840E-01), (5.9817037170E+01, 4.8257007130E-01), (2.2541304740E+01, 1.9357219660E-01)]),("S", [(2.2878676060E+02, -2.9408550880E-02), (4.6904154950E+01, -6.5327468830E-02), (1.4948398430E+01, 1.2899731810E-01), (5.8110991600E+00, 6.1228999380E-01), (2.4755709060E+00, 3.4612056550E-01)]),("P", [(2.2878676060E+02, 1.2556091280E-02), (4.6904154950E+01, 1.0755769620E-01), (1.4948398430E+01, 3.7359753670E-01), (5.8110991600E+00, 5.1023956370E-01), (2.4755709060E+00, 1.5682818010E-01)]),("S", [(9.2786062580E+00, -2.6622033910E-02), (2.5267433860E+00, -1.6034840720E-01), (9.6912876100E-01, -4.7794733070E-02), (4.3797217470E-01, 7.2751584410E-01), (2.1252346610E-01, 4.1238464080E-01)]),("P", [(9.2786062580E+00, -1.5668834480E-02), (2.5267433860E+00, 7.2149205060E-03), (9.6912876100E-01, 3.1708547620E-01), (4.3797217470E-01, 5.8188213820E-01), (2.1252346610E-01, 1.7017998240E-01)])],
}



DEF2_SVP = {

    1 : [("S", [(1.3010701000E+01, 1.9682158000E-02), (1.9622572000E+00, 1.3796524000E-01), (4.4453796000E-01, 4.7831935000E-01)]),("S", [(1.2194962000E-01, 1.0000000000E+00)]),("P", [(8.0000000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8354936737E+01, 2.3814288905E-02), (5.7689081479E+00, 1.5490906777E-01), (1.2399407035E+00, 4.6998096633E-01)]),("S", [(2.9757815953E-01, 1.0000000000E+00)]),("P", [(1.0000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(2.6627785516E+02, 6.4920150325E-03), (4.0069783447E+01, 4.7747863215E-02), (9.0559944389E+00, 2.0268796111E-01), (2.4503009051E+00, 4.8606574817E-01), (7.2209571855E-01, 4.3626977955E-01)]),("S", [(5.2810884721E-02, 1.0000000000E+00)]),("S", [(2.0960948798E-02, 1.0000000000E+00)]),("P", [(1.4500000000E+00, 2.5860000000E-01), (3.0000000000E-01, 1.0000000000E+00)]),("P", [(8.2000000000E-02, 1.0000000000E+00)])],
    4 : [("S", [(5.1518616125E+02, 5.5615307983E-03), (7.7511037595E+01, 4.1190068062E-02), (1.7552481693E+01, 1.7913378108E-01), (4.8028940596E+00, 4.4736716455E-01), (1.4516214316E+00, 4.2009581920E-01)]),("S", [(1.3281633620E-01, 1.0000000000E+00)]),("S", [(4.5837372213E-02, 1.0000000000E+00)]),("P", [(3.6316917145E+00, -2.9033998305E-02), (7.1695694366E-01, -1.6877854032E-01), (1.9541932860E-01, -5.1403419628E-01)]),("P", [(6.0515465890E-02, 1.0000000000E+00)])],
    5 : [("S", [(8.3931830086E+02, -5.5929201074E-03), (1.2626464843E+02, -4.1565520771E-02), (2.8620600763E+01, -1.8299816983E-01), (7.8793722710E+00, -4.6540391866E-01), (2.4088857172E+00, -4.4173884791E-01)]),("S", [(2.5105109036E-01, 1.0000000000E+00)]),("S", [(8.3648866069E-02, 1.0000000000E+00)]),("P", [(6.0332223619E+00, -3.5603672456E-02), (1.2499157866E+00, -1.9895775769E-01), (3.3871676350E-01, -5.0850202618E-01)]),("P", [(9.6415632351E-02, 1.0000000000E+00)]),("D", [(5.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(1.2384016938E+03, 5.4568832082E-03), (1.8629004992E+02, 4.0638409211E-02), (4.2251176346E+01, 1.8025593888E-01), (1.1676557932E+01, 4.6315121755E-01), (3.5930506482E+00, 4.4087173314E-01)]),("S", [(4.0245147363E-01, 1.0000000000E+00)]),("S", [(1.3090182668E-01, 1.0000000000E+00)]),("P", [(9.4680970621E+00, 3.8387871728E-02), (2.0103545142E+00, 2.1117025112E-01), (5.4771004707E-01, 5.1328172114E-01)]),("P", [(1.5268613795E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(1.7128415853E+03, -5.3934125305E-03), (2.5764812677E+02, -4.0221581118E-02), (5.8458245853E+01, -1.7931144990E-01), (1.6198367905E+01, -4.6376317823E-01), (5.0052600809E+00, -4.4171422662E-01)]),("S", [(5.8731856571E-01, 1.0000000000E+00)]),("S", [(1.8764592253E-01, 1.0000000000E+00)]),("P", [(1.3571470233E+01, -4.0072398852E-02), (2.9257372874E+00, -2.1807045028E-01), (7.9927750754E-01, -5.1294466049E-01)]),("P", [(2.1954348034E-01, 1.0000000000E+00)]),("D", [(1.0000000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(2.2661767785E+03, -5.3431809926E-03), (3.4087010191E+02, -3.9890039230E-02), (7.7363135167E+01, -1.7853911985E-01), (2.1479644940E+01, -4.6427684959E-01), (6.6589433124E+00, -4.4309745172E-01)]),("S", [(8.0975975668E-01, 1.0000000000E+00)]),("S", [(2.5530772234E-01, 1.0000000000E+00)]),("P", [(1.7721504317E+01, 4.3394573193E-02), (3.8635505440E+00, 2.3094120765E-01), (1.0480920883E+00, 5.1375311064E-01)]),("P", [(2.7641544411E-01, 1.0000000000E+00)]),("D", [(1.2000000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(2.8948325990E+03, -5.3408255515E-03), (4.3541939120E+02, -3.9904258866E-02), (9.8843328866E+01, -1.7912768038E-01), (2.7485198001E+01, -4.6758090825E-01), (8.5405498171E+00, -4.4653131020E-01)]),("S", [(1.0654578038E+00, 1.0000000000E+00)]),("S", [(3.3247346748E-01, 1.0000000000E+00)]),("P", [(2.2696633924E+01, -4.5212874436E-02), (4.9872339257E+00, -2.3754317067E-01), (1.3491613954E+00, -5.1287353587E-01)]),("P", [(3.4829881977E-01, 1.0000000000E+00)]),("D", [(1.4000000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(3.5989736625E+03, -5.3259297003E-03), (5.4132073112E+02, -3.9817417969E-02), (1.2290450062E+02, -1.7914358188E-01), (3.4216617022E+01, -4.6893582977E-01), (1.0650584124E+01, -4.4782537577E-01)]),("S", [(1.3545953960E+00, 1.0000000000E+00)]),("S", [(4.1919362639E-01, 1.0000000000E+00)]),("P", [(2.8424053785E+01, -4.6031944795E-02), (6.2822510953E+00, -2.3993183041E-01), (1.6978715079E+00, -5.0871724964E-01)]),("P", [(4.3300700172E-01, 1.0000000000E+00)]),("D", [(1.8880000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(4.0982003908E+03, -5.8535911879E-03), (6.1649374031E+02, -4.3647161872E-02), (1.3996644001E+02, -1.9431465884E-01), (3.9073441051E+01, -4.8685065731E-01), (1.1929847205E+01, -4.1881705137E-01)]),("S", [(2.0659966030E+01, 8.5949689854E-02), (1.9838860978E+00, -5.6359144041E-01), (6.4836323942E-01, -5.1954009048E-01)]),("S", [(5.2443967404E-02, 1.0000000000E+00)]),("S", [(2.8048160742E-02, 1.0000000000E+00)]),("P", [(7.5401862017E+01, 1.5435362532E-02), (1.7274818978E+01, 9.9738293184E-02), (5.1842347425E+00, 3.1209593966E-01), (1.6601211973E+00, 4.9295674807E-01), (5.1232528958E-01, 3.2420398318E-01)]),("P", [(5.2000000000E-02, 1.0000000000E+00)]),("D", [(1.3100000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.9538339196E+03, -5.7778967498E-03), (7.4518044154E+02, -4.3124761082E-02), (1.6921604972E+02, -1.9268216987E-01), (4.7300672019E+01, -4.8641439116E-01), (1.4461336973E+01, -4.2550894077E-01)]),("S", [(2.4768174789E+01, 8.7956969984E-02), (2.4940945349E+00, -5.5165058128E-01), (8.7807584533E-01, -5.3443294833E-01)]),("S", [(8.7212782497E-02, 1.0000000000E+00)]),("S", [(3.3599293780E-02, 1.0000000000E+00)]),("P", [(9.8053010494E+01, -1.4480564601E-02), (2.2586932277E+01, -9.5495750787E-02), (6.8391509842E+00, -3.0787672651E-01), (2.2332843818E+00, -4.9936292886E-01), (7.1606599387E-01, -3.1503476213E-01)]),("P", [(1.8914796195E-01, 1.0000000000E+00)]),("P", [(5.3768755187E-02, 1.0000000000E+00)]),("D", [(1.0100000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.8875727030E+03, 1.3483347987E-03), (8.8561225996E+02, 1.0071576809E-02), (2.0113604899E+02, 4.5132454056E-02), (5.6284974674E+01, 1.1461268043E-01), (1.7229551243E+01, 1.0159608943E-01)]),("S", [(2.9340249922E+01, 6.9347454208E-02), (3.0439630420E+00, -4.2528117679E-01), (1.1285539518E+00, -4.1449832210E-01)]),("S", [(1.4234175160E-01, 1.0000000000E+00)]),("S", [(5.4400192313E-02, 1.0000000000E+00)]),("P", [(1.4511918809E+02, 6.3963373134E-03), (3.3717894833E+01, 4.4189359965E-02), (1.0369863083E+01, 1.5581575993E-01), (3.5135616036E+00, 2.8635286951E-01), (1.1980050273E+00, 2.2921423248E-01)]),("P", [(2.6583005913E-01, 1.0000000000E+00)]),("P", [(7.1003361994E-02, 1.0000000000E+00)]),("D", [(3.0000000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9037118686E+03, 1.3373962995E-03), (1.0384346419E+03, 9.9966546241E-03), (2.3587581480E+02, 4.4910165101E-02), (6.6069385169E+01, 1.1463638540E-01), (2.0247945761E+01, 1.0280063858E-01)]),("S", [(3.4353481730E+01, 7.0837285010E-02), (3.6370788192E+00, -4.3028836252E-01), (1.4002048599E+00, -4.1382774969E-01)]),("S", [(2.0484414805E-01, 1.0000000000E+00)]),("S", [(7.7994095468E-02, 1.0000000000E+00)]),("P", [(1.7983907373E+02, 6.1916656462E-03), (4.1907258846E+01, 4.3399431982E-02), (1.2955294367E+01, 1.5632019351E-01), (4.4383267393E+00, 2.9419996982E-01), (1.5462247904E+00, 2.3536823814E-01)]),("P", [(3.5607612302E-01, 1.0000000000E+00)]),("P", [(1.0008513762E-01, 1.0000000000E+00)]),("D", [(3.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(8.0024795106E+03, 5.7503489302E-03), (1.2036813590E+03, 4.3007628959E-02), (2.7344227031E+02, 1.9363986252E-01), (7.6655541517E+01, 4.9651693399E-01), (2.3516927435E+01, 4.4983262479E-01)]),("S", [(3.9791683439E+01, 9.5188129789E-02), (4.2770343323E+00, -5.7649840368E-01), (1.6940256888E+00, -5.4239583865E-01)]),("S", [(2.7567674644E-01, 1.0000000000E+00)]),("S", [(1.0495590099E-01, 1.0000000000E+00)]),("P", [(2.1950755823E+02, 9.2100565257E-03), (5.1274155030E+01, 6.5409765745E-02), (1.5921595892E+01, 2.4033730279E-01), (5.5069913481E+00, 4.6318321788E-01), (1.9537719426E+00, 3.7392563382E-01)]),("P", [(4.7803397932E-01, 1.0000000000E+00)]),("P", [(1.3657952621E-01, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.1849303010E+03, -2.2294387756E-03), (1.3815105503E+03, -1.6683029937E-02), (3.1387147580E+02, -7.5262436116E-02), (8.8053870623E+01, -1.9376827038E-01), (2.7039914905E+01, -1.7718020803E-01)]),("S", [(4.5648731303E+01, -1.0736062573E-01), (4.9664522326E+00, 6.5066293018E-01), (2.0116242047E+00, 5.9712155354E-01)]),("S", [(3.5661077013E-01, 1.0000000000E+00)]),("S", [(1.3507221477E-01, 1.0000000000E+00)]),("P", [(2.6198233439E+02, -9.2729929822E-03), (6.1306894736E+01, -6.6547669241E-02), (1.9103729887E+01, -2.4828595903E-01), (6.6567720378E+00, -4.8703847402E-01), (2.3959635161E+00, -3.9337850312E-01)]),("P", [(6.1776161679E-01, 1.0000000000E+00)]),("P", [(1.6993376871E-01, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0449827566E+04, 1.9708362484E-03), (1.5717365221E+03, 1.4754727977E-02), (3.5712065523E+02, 6.6679112875E-02), (1.0025185935E+02, 1.7228924084E-01), (3.0812727554E+01, 1.5883786100E-01)]),("S", [(5.1923789434E+01, -1.0009298909E-01), (5.7045760975E+00, 6.0841752753E-01), (2.3508376809E+00, 5.4352153355E-01)]),("S", [(4.4605124672E-01, 1.0000000000E+00)]),("S", [(1.6848856190E-01, 1.0000000000E+00)]),("P", [(3.0766790569E+02, -8.7801484118E-03), (7.2102015515E+01, -6.3563355471E-02), (2.2532680262E+01, -2.4016428276E-01), (7.8991765444E+00, -4.7798866557E-01), (2.8767268321E+00, -3.8515850005E-01)]),("P", [(7.7459363955E-01, 1.0000000000E+00)]),("P", [(2.1037699698E-01, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1797119765E+04, 2.0214479973E-03), (1.7743522753E+03, 1.5139852688E-02), (4.0318875733E+02, 6.8525400873E-02), (1.1324933999E+02, 1.7762929207E-01), (3.4835298193E+01, 1.6496495001E-01)]),("S", [(5.8614775042E+01, -1.0343394000E-01), (6.4922045078E+00, 6.3133365899E-01), (2.7117014403E+00, 5.4887572301E-01)]),("S", [(5.4412974538E-01, 1.0000000000E+00)]),("S", [(2.0517411136E-01, 1.0000000000E+00)]),("P", [(3.5628707256E+02, -8.7321783271E-03), (8.3593132866E+01, -6.3680317444E-02), (2.6186704029E+01, -2.4311906886E-01), (9.2257275925E+00, -4.8956069653E-01), (3.3922754954E+00, -3.9229190040E-01)]),("P", [(9.4740534010E-01, 1.0000000000E+00)]),("P", [(2.5659135062E-01, 1.0000000000E+00)]),("D", [(6.9600000000E-01, 1.0000000000E+00)])],
}



DEF2_SVPD = {

    1 : [("S", [(1.3010701000E+01, 1.9682158000E-02), (1.9622572000E+00, 1.3796524000E-01), (4.4453796000E-01, 4.7831935000E-01)]),("S", [(1.2194962000E-01, 1.0000000000E+00)]),("P", [(8.0000000000E-01, 1.0000000000E+00)]),("P", [(1.1704099050E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8354936737E+01, 2.3814288905E-02), (5.7689081479E+00, 1.5490906777E-01), (1.2399407035E+00, 4.6998096633E-01)]),("S", [(2.9757815953E-01, 1.0000000000E+00)]),("P", [(1.0000000000E+00, 1.0000000000E+00)]),("P", [(2.2039119441E-01, 1.0000000000E+00)])],
    3 : [("S", [(2.6627785516E+02, 6.4920150325E-03), (4.0069783447E+01, 4.7747863215E-02), (9.0559944389E+00, 2.0268796111E-01), (2.4503009051E+00, 4.8606574817E-01), (7.2209571855E-01, 4.3626977955E-01)]),("S", [(5.2810884721E-02, 1.0000000000E+00)]),("S", [(2.0960948798E-02, 1.0000000000E+00)]),("P", [(1.4500000000E+00, 2.5860000000E-01), (3.0000000000E-01, 1.0000000000E+00)]),("P", [(8.2000000000E-02, 1.0000000000E+00)]),("P", [(2.6347662457E-02, 1.0000000000E+00)])],
    4 : [("S", [(5.1518616125E+02, 5.5615307983E-03), (7.7511037595E+01, 4.1190068062E-02), (1.7552481693E+01, 1.7913378108E-01), (4.8028940596E+00, 4.4736716455E-01), (1.4516214316E+00, 4.2009581920E-01)]),("S", [(1.3281633620E-01, 1.0000000000E+00)]),("S", [(4.5837372213E-02, 1.0000000000E+00)]),("P", [(3.6316917145E+00, -2.9033998305E-02), (7.1695694366E-01, -1.6877854032E-01), (1.9541932860E-01, -5.1403419628E-01)]),("P", [(6.0515465890E-02, 1.0000000000E+00)]),("P", [(3.0181880660E-02, 1.0000000000E+00)])],
    5 : [("S", [(8.3931830086E+02, -5.5929201074E-03), (1.2626464843E+02, -4.1565520771E-02), (2.8620600763E+01, -1.8299816983E-01), (7.8793722710E+00, -4.6540391866E-01), (2.4088857172E+00, -4.4173884791E-01)]),("S", [(2.5105109036E-01, 1.0000000000E+00)]),("S", [(8.3648866069E-02, 1.0000000000E+00)]),("S", [(4.2351565332E-02, 1.0000000000E+00)]),("P", [(6.0332223619E+00, -3.5603672456E-02), (1.2499157866E+00, -1.9895775769E-01), (3.3871676350E-01, -5.0850202618E-01)]),("P", [(9.6415632351E-02, 1.0000000000E+00)]),("D", [(5.0000000000E-01, 1.0000000000E+00)]),("D", [(7.4392443101E-02, 1.0000000000E+00)])],
    6 : [("S", [(1.2384016938E+03, 5.4568832082E-03), (1.8629004992E+02, 4.0638409211E-02), (4.2251176346E+01, 1.8025593888E-01), (1.1676557932E+01, 4.6315121755E-01), (3.5930506482E+00, 4.4087173314E-01)]),("S", [(4.0245147363E-01, 1.0000000000E+00)]),("S", [(1.3090182668E-01, 1.0000000000E+00)]),("S", [(6.7053540256E-02, 1.0000000000E+00)]),("P", [(9.4680970621E+00, 3.8387871728E-02), (2.0103545142E+00, 2.1117025112E-01), (5.4771004707E-01, 5.1328172114E-01)]),("P", [(1.5268613795E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)]),("D", [(1.1713185140E-01, 1.0000000000E+00)])],
    7 : [("S", [(1.7128415853E+03, -5.3934125305E-03), (2.5764812677E+02, -4.0221581118E-02), (5.8458245853E+01, -1.7931144990E-01), (1.6198367905E+01, -4.6376317823E-01), (5.0052600809E+00, -4.4171422662E-01)]),("S", [(5.8731856571E-01, 1.0000000000E+00)]),("S", [(1.8764592253E-01, 1.0000000000E+00)]),("S", [(9.6171241529E-02, 1.0000000000E+00)]),("P", [(1.3571470233E+01, -4.0072398852E-02), (2.9257372874E+00, -2.1807045028E-01), (7.9927750754E-01, -5.1294466049E-01)]),("P", [(2.1954348034E-01, 1.0000000000E+00)]),("D", [(1.0000000000E+00, 1.0000000000E+00)]),("D", [(1.6697708112E-01, 1.0000000000E+00)])],
    8 : [("S", [(2.2661767785E+03, -5.3431809926E-03), (3.4087010191E+02, -3.9890039230E-02), (7.7363135167E+01, -1.7853911985E-01), (2.1479644940E+01, -4.6427684959E-01), (6.6589433124E+00, -4.4309745172E-01)]),("S", [(8.0975975668E-01, 1.0000000000E+00)]),("S", [(2.5530772234E-01, 1.0000000000E+00)]),("S", [(7.6572453250E-02, 1.0000000000E+00)]),("P", [(1.7721504317E+01, 4.3394573193E-02), (3.8635505440E+00, 2.3094120765E-01), (1.0480920883E+00, 5.1375311064E-01)]),("P", [(2.7641544411E-01, 1.0000000000E+00)]),("P", [(6.9002276350E-02, 1.0000000000E+00)]),("D", [(1.2000000000E+00, 1.0000000000E+00)]),("D", [(1.7992024323E-01, 1.0000000000E+00)])],
    9 : [("S", [(2.8948325990E+03, -5.3408255515E-03), (4.3541939120E+02, -3.9904258866E-02), (9.8843328866E+01, -1.7912768038E-01), (2.7485198001E+01, -4.6758090825E-01), (8.5405498171E+00, -4.4653131020E-01)]),("S", [(1.0654578038E+00, 1.0000000000E+00)]),("S", [(3.3247346748E-01, 1.0000000000E+00)]),("S", [(9.8097752264E-02, 1.0000000000E+00)]),("P", [(2.2696633924E+01, -4.5212874436E-02), (4.9872339257E+00, -2.3754317067E-01), (1.3491613954E+00, -5.1287353587E-01)]),("P", [(3.4829881977E-01, 1.0000000000E+00)]),("P", [(8.3372148304E-02, 1.0000000000E+00)]),("D", [(1.4000000000E+00, 1.0000000000E+00)]),("D", [(2.2301361948E-01, 1.0000000000E+00)])],
    10 : [("S", [(3.5989736625E+03, -5.3259297003E-03), (5.4132073112E+02, -3.9817417969E-02), (1.2290450062E+02, -1.7914358188E-01), (3.4216617022E+01, -4.6893582977E-01), (1.0650584124E+01, -4.4782537577E-01)]),("S", [(1.3545953960E+00, 1.0000000000E+00)]),("S", [(4.1919362639E-01, 1.0000000000E+00)]),("S", [(1.2055043545E-01, 1.0000000000E+00)]),("P", [(2.8424053785E+01, -4.6031944795E-02), (6.2822510953E+00, -2.3993183041E-01), (1.6978715079E+00, -5.0871724964E-01)]),("P", [(4.3300700172E-01, 1.0000000000E+00)]),("P", [(1.0141942051E-01, 1.0000000000E+00)]),("D", [(1.8880000000E+00, 1.0000000000E+00)]),("D", [(2.8026803041E-01, 1.0000000000E+00)])],
    11 : [("S", [(4.0982003908E+03, -5.8535911879E-03), (6.1649374031E+02, -4.3647161872E-02), (1.3996644001E+02, -1.9431465884E-01), (3.9073441051E+01, -4.8685065731E-01), (1.1929847205E+01, -4.1881705137E-01)]),("S", [(2.0659966030E+01, 8.5949689854E-02), (1.9838860978E+00, -5.6359144041E-01), (6.4836323942E-01, -5.1954009048E-01)]),("S", [(5.2443967404E-02, 1.0000000000E+00)]),("S", [(2.8048160742E-02, 1.0000000000E+00)]),("P", [(7.5401862017E+01, 1.5435362532E-02), (1.7274818978E+01, 9.9738293184E-02), (5.1842347425E+00, 3.1209593966E-01), (1.6601211973E+00, 4.9295674807E-01), (5.1232528958E-01, 3.2420398318E-01)]),("P", [(5.2000000000E-02, 1.0000000000E+00)]),("P", [(1.9853008514E-02, 1.0000000000E+00)]),("D", [(1.3100000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.9538339196E+03, -5.7778967498E-03), (7.4518044154E+02, -4.3124761082E-02), (1.6921604972E+02, -1.9268216987E-01), (4.7300672019E+01, -4.8641439116E-01), (1.4461336973E+01, -4.2550894077E-01)]),("S", [(2.4768174789E+01, 8.7956969984E-02), (2.4940945349E+00, -5.5165058128E-01), (8.7807584533E-01, -5.3443294833E-01)]),("S", [(8.7212782497E-02, 1.0000000000E+00)]),("S", [(3.3599293780E-02, 1.0000000000E+00)]),("P", [(9.8053010494E+01, -1.4480564601E-02), (2.2586932277E+01, -9.5495750787E-02), (6.8391509842E+00, -3.0787672651E-01), (2.2332843818E+00, -4.9936292886E-01), (7.1606599387E-01, -3.1503476213E-01)]),("P", [(1.8914796195E-01, 1.0000000000E+00)]),("P", [(5.3768755187E-02, 1.0000000000E+00)]),("P", [(2.3963578287E-02, 1.0000000000E+00)]),("D", [(1.0100000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.8875727030E+03, 1.3483347987E-03), (8.8561225996E+02, 1.0071576809E-02), (2.0113604899E+02, 4.5132454056E-02), (5.6284974674E+01, 1.1461268043E-01), (1.7229551243E+01, 1.0159608943E-01)]),("S", [(2.9340249922E+01, 6.9347454208E-02), (3.0439630420E+00, -4.2528117679E-01), (1.1285539518E+00, -4.1449832210E-01)]),("S", [(1.4234175160E-01, 1.0000000000E+00)]),("S", [(5.4400192313E-02, 1.0000000000E+00)]),("S", [(3.2535359511E-02, 1.0000000000E+00)]),("P", [(1.4511918809E+02, 6.3963373134E-03), (3.3717894833E+01, 4.4189359965E-02), (1.0369863083E+01, 1.5581575993E-01), (3.5135616036E+00, 2.8635286951E-01), (1.1980050273E+00, 2.2921423248E-01)]),("P", [(2.6583005913E-01, 1.0000000000E+00)]),("P", [(7.1003361994E-02, 1.0000000000E+00)]),("D", [(3.0000000000E-01, 1.0000000000E+00)]),("D", [(5.3071685831E-02, 1.0000000000E+00)])],
    14 : [("S", [(6.9037118686E+03, 1.3373962995E-03), (1.0384346419E+03, 9.9966546241E-03), (2.3587581480E+02, 4.4910165101E-02), (6.6069385169E+01, 1.1463638540E-01), (2.0247945761E+01, 1.0280063858E-01)]),("S", [(3.4353481730E+01, 7.0837285010E-02), (3.6370788192E+00, -4.3028836252E-01), (1.4002048599E+00, -4.1382774969E-01)]),("S", [(2.0484414805E-01, 1.0000000000E+00)]),("S", [(7.7994095468E-02, 1.0000000000E+00)]),("S", [(4.6834778282E-02, 1.0000000000E+00)]),("P", [(1.7983907373E+02, 6.1916656462E-03), (4.1907258846E+01, 4.3399431982E-02), (1.2955294367E+01, 1.5632019351E-01), (4.4383267393E+00, 2.9419996982E-01), (1.5462247904E+00, 2.3536823814E-01)]),("P", [(3.5607612302E-01, 1.0000000000E+00)]),("P", [(1.0008513762E-01, 1.0000000000E+00)]),("D", [(3.5000000000E-01, 1.0000000000E+00)]),("D", [(7.4117707708E-02, 1.0000000000E+00)])],
    15 : [("S", [(8.0024795106E+03, 5.7503489302E-03), (1.2036813590E+03, 4.3007628959E-02), (2.7344227031E+02, 1.9363986252E-01), (7.6655541517E+01, 4.9651693399E-01), (2.3516927435E+01, 4.4983262479E-01)]),("S", [(3.9791683439E+01, 9.5188129789E-02), (4.2770343323E+00, -5.7649840368E-01), (1.6940256888E+00, -5.4239583865E-01)]),("S", [(2.7567674644E-01, 1.0000000000E+00)]),("S", [(1.0495590099E-01, 1.0000000000E+00)]),("S", [(6.4309734755E-02, 1.0000000000E+00)]),("P", [(2.1950755823E+02, 9.2100565257E-03), (5.1274155030E+01, 6.5409765745E-02), (1.5921595892E+01, 2.4033730279E-01), (5.5069913481E+00, 4.6318321788E-01), (1.9537719426E+00, 3.7392563382E-01)]),("P", [(4.7803397932E-01, 1.0000000000E+00)]),("P", [(1.3657952621E-01, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)]),("D", [(9.9587311089E-02, 1.0000000000E+00)])],
    16 : [("S", [(9.1849303010E+03, -2.2294387756E-03), (1.3815105503E+03, -1.6683029937E-02), (3.1387147580E+02, -7.5262436116E-02), (8.8053870623E+01, -1.9376827038E-01), (2.7039914905E+01, -1.7718020803E-01)]),("S", [(4.5648731303E+01, -1.0736062573E-01), (4.9664522326E+00, 6.5066293018E-01), (2.0116242047E+00, 5.9712155354E-01)]),("S", [(3.5661077013E-01, 1.0000000000E+00)]),("S", [(1.3507221477E-01, 1.0000000000E+00)]),("S", [(4.4999823506E-02, 1.0000000000E+00)]),("P", [(2.6198233439E+02, -9.2729929822E-03), (6.1306894736E+01, -6.6547669241E-02), (1.9103729887E+01, -2.4828595903E-01), (6.6567720378E+00, -4.8703847402E-01), (2.3959635161E+00, -3.9337850312E-01)]),("P", [(6.1776161679E-01, 1.0000000000E+00)]),("P", [(1.6993376871E-01, 1.0000000000E+00)]),("P", [(4.1706793676E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)]),("D", [(1.0172853717E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0449827566E+04, 1.9708362484E-03), (1.5717365221E+03, 1.4754727977E-02), (3.5712065523E+02, 6.6679112875E-02), (1.0025185935E+02, 1.7228924084E-01), (3.0812727554E+01, 1.5883786100E-01)]),("S", [(5.1923789434E+01, -1.0009298909E-01), (5.7045760975E+00, 6.0841752753E-01), (2.3508376809E+00, 5.4352153355E-01)]),("S", [(4.4605124672E-01, 1.0000000000E+00)]),("S", [(1.6848856190E-01, 1.0000000000E+00)]),("S", [(5.5344229413E-02, 1.0000000000E+00)]),("P", [(3.0766790569E+02, -8.7801484118E-03), (7.2102015515E+01, -6.3563355471E-02), (2.2532680262E+01, -2.4016428276E-01), (7.8991765444E+00, -4.7798866557E-01), (2.8767268321E+00, -3.8515850005E-01)]),("P", [(7.7459363955E-01, 1.0000000000E+00)]),("P", [(2.1037699698E-01, 1.0000000000E+00)]),("P", [(5.0046271086E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("D", [(1.2284803390E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1797119765E+04, 2.0214479973E-03), (1.7743522753E+03, 1.5139852688E-02), (4.0318875733E+02, 6.8525400873E-02), (1.1324933999E+02, 1.7762929207E-01), (3.4835298193E+01, 1.6496495001E-01)]),("S", [(5.8614775042E+01, -1.0343394000E-01), (6.4922045078E+00, 6.3133365899E-01), (2.7117014403E+00, 5.4887572301E-01)]),("S", [(5.4412974538E-01, 1.0000000000E+00)]),("S", [(2.0517411136E-01, 1.0000000000E+00)]),("S", [(6.6442487045E-02, 1.0000000000E+00)]),("P", [(3.5628707256E+02, -8.7321783271E-03), (8.3593132866E+01, -6.3680317444E-02), (2.6186704029E+01, -2.4311906886E-01), (9.2257275925E+00, -4.8956069653E-01), (3.3922754954E+00, -3.9229190040E-01)]),("P", [(9.4740534010E-01, 1.0000000000E+00)]),("P", [(2.5659135062E-01, 1.0000000000E+00)]),("P", [(5.9491455142E-02, 1.0000000000E+00)]),("D", [(6.9600000000E-01, 1.0000000000E+00)]),("D", [(1.4309493859E-01, 1.0000000000E+00)])],
}





DEF2_TZVP = {

    1 : [("S", [(3.4061341000E+01, 6.0251978000E-03), (5.1235746000E+00, 4.5021094000E-02), (1.1646626000E+00, 2.0189726000E-01)]),("S", [(3.2723041000E-01, 1.0000000000E+00)]),("S", [(1.0307241000E-01, 1.0000000000E+00)]),("P", [(8.0000000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8078321616E+01, 7.5803064967E-03), (1.4764404247E+01, 5.4848620937E-02), (3.3185831473E+00, 2.2074382186E-01)]),("S", [(8.7413869551E-01, 1.0000000000E+00)]),("S", [(2.4459897208E-01, 1.0000000000E+00)]),("P", [(1.0000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.2692628010E+03, 2.0540968826E-04), (9.4031612431E+02, 1.5916554089E-03), (2.1422107528E+02, 8.2869829707E-03), (6.0759840184E+01, 3.3856374249E-02), (1.9915152032E+01, 1.1103225876E-01), (7.3171509797E+00, 2.7449383329E-01)]),("S", [(2.9724674216E+00, 2.3792456411E-01), (1.2639852314E+00, 3.0765411924E-01)]),("S", [(5.1427489953E-01, 1.0000000000E+00)]),("S", [(7.7030885901E-02, 1.0000000000E+00)]),("S", [(2.8938896433E-02, 1.0000000000E+00)]),("P", [(3.3270000000E+00, 1.0000000000E+00)]),("P", [(4.0000000000E-01, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)])],
    4 : [("S", [(4.7002365626E+03, 2.3584389316E-04), (7.0482845622E+02, 1.8243791019E-03), (1.6043110478E+02, 9.3966148224E-03), (4.5425347336E+01, 3.6908924159E-02), (1.4798334125E+01, 1.0897561281E-01), (5.3512452537E+00, 2.1694284551E-01)]),("S", [(2.1542044819E+00, 4.4695408857E-01), (9.3363744400E-01, 2.0866985771E-01)]),("S", [(1.8791432989E-01, 1.0000000000E+00)]),("S", [(7.4648267947E-02, 1.0000000000E+00)]),("S", [(3.2650484598E-02, 1.0000000000E+00)]),("P", [(3.6316917145E+00, -2.9033998305E-02)]),("P", [(7.1695694366E-01, -1.6877854032E-01), (1.9541932860E-01, -5.1403419628E-01)]),("P", [(6.0515465890E-02, 1.0000000000E+00)]),("D", [(1.8000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(8.5648660687E+03, 2.2837198155E-04), (1.2841516263E+03, 1.7682576447E-03), (2.9227871604E+02, 9.1407080516E-03), (8.2775469176E+01, 3.6342638989E-02), (2.7017939269E+01, 1.1063458441E-01), (9.8149619660E+00, 2.3367344321E-01)]),("S", [(3.9318559059E+00, 4.1818777978E-01), (1.6595599712E+00, 2.2325473798E-01)]),("S", [(3.5762965239E-01, 1.0000000000E+00)]),("S", [(1.4246277496E-01, 1.0000000000E+00)]),("S", [(6.0560594768E-02, 1.0000000000E+00)]),("P", [(2.2453875803E+01, 5.0265575179E-03), (5.1045058330E+00, 3.2801738965E-02), (1.4986081344E+00, 1.3151230768E-01), (5.0927831315E-01, 3.3197167769E-01)]),("P", [(1.8147077798E-01, 4.7314319570E-01)]),("P", [(6.4621893904E-02, 2.5802783943E-01)]),("D", [(6.6100000000E-01, 1.0000000000E+00)]),("D", [(1.9900000000E-01, 1.0000000000E+00)]),("F", [(4.9000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(1.3575349682E+04, 2.2245814352E-04), (2.0352333680E+03, 1.7232738252E-03), (4.6322562359E+02, 8.9255715314E-03), (1.3120019598E+02, 3.5727984502E-02), (4.2853015891E+01, 1.1076259931E-01), (1.5584185766E+01, 2.4295627626E-01)]),("S", [(6.2067138508E+00, 4.1440263448E-01), (2.5764896527E+00, 2.3744968655E-01)]),("S", [(5.7696339419E-01, 1.0000000000E+00)]),("S", [(2.2972831358E-01, 1.0000000000E+00)]),("S", [(9.5164440028E-02, 1.0000000000E+00)]),("P", [(3.4697232244E+01, 5.3333657805E-03), (7.9582622826E+00, 3.5864109092E-02), (2.3780826883E+00, 1.4215873329E-01), (8.1433208183E-01, 3.4270471845E-01)]),("P", [(2.8887547253E-01, 4.6445822433E-01)]),("P", [(1.0056823671E-01, 2.4955789874E-01)]),("D", [(1.0970000000E+00, 1.0000000000E+00)]),("D", [(3.1800000000E-01, 1.0000000000E+00)]),("F", [(7.6100000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(1.9730800647E+04, 2.1887984991E-04), (2.9578958745E+03, 1.6960708803E-03), (6.7322133595E+02, 8.7954603538E-03), (1.9068249494E+02, 3.5359382605E-02), (6.2295441898E+01, 1.1095789217E-01), (2.2654161182E+01, 2.4982972552E-01)]),("S", [(8.9791477428E+00, 4.0623896148E-01), (3.6863002370E+00, 2.4338217176E-01)]),("S", [(8.4660076805E-01, 1.0000000000E+00)]),("S", [(3.3647133771E-01, 1.0000000000E+00)]),("S", [(1.3647653675E-01, 1.0000000000E+00)]),("P", [(4.9200380510E+01, 5.5552416751E-03), (1.1346790537E+01, 3.8052379723E-02), (3.4273972411E+00, 1.4953671029E-01), (1.1785525134E+00, 3.4949305230E-01)]),("P", [(4.1642204972E-01, 4.5843153697E-01)]),("P", [(1.4260826011E-01, 2.4428771672E-01)]),("D", [(1.6540000000E+00, 1.0000000000E+00)]),("D", [(4.6900000000E-01, 1.0000000000E+00)]),("F", [(1.0930000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(2.7032382631E+04, 2.1726302465E-04), (4.0523871392E+03, 1.6838662199E-03), (9.2232722710E+02, 8.7395616265E-03), (2.6124070989E+02, 3.5239968808E-02), (8.5354641351E+01, 1.1153519115E-01), (3.1035035245E+01, 2.5588953961E-01)]),("S", [(1.2260860728E+01, 3.9768730901E-01), (4.9987076005E+00, 2.4627849430E-01)]),("S", [(1.1703108158E+00, 1.0000000000E+00)]),("S", [(4.6474740994E-01, 1.0000000000E+00)]),("S", [(1.8504536357E-01, 1.0000000000E+00)]),("P", [(6.3274954801E+01, 6.0685103418E-03), (1.4627049379E+01, 4.1912575824E-02), (4.4501223456E+00, 1.6153841088E-01), (1.5275799647E+00, 3.5706951311E-01)]),("P", [(5.2935117943E-01, 4.4794207502E-01)]),("P", [(1.7478421270E-01, 2.4446069663E-01)]),("D", [(2.3140000000E+00, 1.0000000000E+00)]),("D", [(6.4500000000E-01, 1.0000000000E+00)]),("F", [(1.4280000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(3.5479100441E+04, 2.1545014888E-04), (5.3184728983E+03, 1.6700686527E-03), (1.2104810975E+03, 8.6733211476E-03), (3.4285518140E+02, 3.5049933175E-02), (1.1201943181E+02, 1.1165320133E-01), (4.0714740248E+01, 2.5988506647E-01)]),("S", [(1.6039678111E+01, 3.9422966880E-01), (6.5038186740E+00, 2.4998238551E-01)]),("S", [(1.5440477509E+00, 1.0000000000E+00)]),("S", [(6.1223452862E-01, 1.0000000000E+00)]),("S", [(2.4027979698E-01, 1.0000000000E+00)]),("P", [(8.0233900483E+01, 6.3685999134E-03), (1.8594010743E+01, 4.4303143530E-02), (5.6867902653E+00, 1.6867248708E-01), (1.9511006294E+00, 3.6166346255E-01)]),("P", [(6.6970211298E-01, 4.4202901491E-01)]),("P", [(2.1651300410E-01, 2.4319875730E-01)]),("D", [(3.1070000000E+00, 1.0000000000E+00)]),("D", [(8.5500000000E-01, 1.0000000000E+00)]),("F", [(1.9170000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(4.5069464022E+04, 2.1687155182E-04), (6.7559768656E+03, 1.6812736757E-03), (1.5376502864E+03, 8.7356062782E-03), (4.3551697667E+02, 3.5361266922E-02), (1.4228655638E+02, 1.1321521454E-01), (5.1692153804E+01, 2.6654653104E-01)]),("S", [(2.0315870490E+01, 3.9631959951E-01), (8.2021942646E+00, 2.5582811251E-01)]),("S", [(1.9681276278E+00, 1.0000000000E+00)]),("S", [(7.7904756001E-01, 1.0000000000E+00)]),("S", [(3.0229502043E-01, 1.0000000000E+00)]),("P", [(9.9782996032E+01, 6.5569234163E-03), (2.3176124101E+01, 4.5888009138E-02), (7.1163945872E+00, 1.7331287812E-01), (2.4418711435E+00, 3.6475267512E-01)]),("P", [(8.3389605766E-01, 4.3831075171E-01)]),("P", [(2.6607311301E-01, 2.4160029835E-01)]),("D", [(4.0140000000E+00, 1.0000000000E+00)]),("D", [(1.0960000000E+00, 1.0000000000E+00)]),("F", [(2.5440000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(2.6041109927E+04, 6.1806342811E-04), (3.9061268548E+03, 4.7748604414E-03), (8.8897454993E+02, 2.4471684829E-02), (2.5145497961E+02, 9.4755394977E-02), (8.1650143512E+01, 2.6867496920E-01), (2.8904158401E+01, 4.7925475440E-01), (1.0625782932E+01, 3.3248591469E-01)]),("S", [(5.3769410179E+01, 1.9527731872E-02), (1.6308243025E+01, 9.2648010794E-02), (2.3730384125E+00, -3.9938670172E-01)]),("S", [(9.5730772603E-01, 1.6428595391E+00), (4.0806460959E-01, 5.5692596966E-01)]),("S", [(4.9967582329E-02, 1.0000000000E+00)]),("S", [(1.9268616250E-02, 1.0000000000E+00)]),("P", [(1.3807979989E+02, 5.7951891929E-03), (3.2232700393E+01, 4.1620846251E-02), (9.9816075360E+00, 1.6281916885E-01), (3.4822033928E+00, 3.6011784647E-01), (1.2299134620E+00, 4.4858979889E-01)]),("P", [(4.1743959423E-01, 2.3040759559E-01)]),("P", [(9.1000000000E-02, 1.0000000000E+00)]),("P", [(3.0000000000E-02, 1.0000000000E+00)]),("D", [(2.6090000000E+00, 1.0000000000E+00)]),("D", [(4.3000000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(3.1438349555E+04, 6.0912311326E-04), (4.7155153354E+03, 4.7066196465E-03), (1.0731629247E+03, 2.4135820657E-02), (3.0357238768E+02, 9.3628959834E-02), (9.8626251042E+01, 2.6646742093E-01), (3.4943808417E+01, 4.7890929917E-01), (1.2859785199E+01, 3.3698490286E-01)]),("S", [(6.4876913004E+01, 1.9180889307E-02), (1.9725520777E+01, 9.0913704392E-02), (2.8951804339E+00, -3.9563756125E-01)]),("S", [(1.1960454710E+00, 1.6827603373E+00), (5.4329451156E-01, 5.2141091954E-01)]),("S", [(1.0099104092E-01, 1.0000000000E+00)]),("S", [(3.6865728085E-02, 1.0000000000E+00)]),("P", [(1.7987189612E+02, 5.3799549018E-03), (4.2120069376E+01, 3.9318014098E-02), (1.3120503032E+01, 1.5740129476E-01), (4.6257503609E+00, 3.5919094128E-01), (1.6695211016E+00, 4.5533379310E-01)]),("P", [(5.8551012105E-01, 2.1986432910E-01)]),("P", [(1.8914796195E-01, 1.0000000000E+00)]),("P", [(5.3768755187E-02, 1.0000000000E+00)]),("D", [(3.4440000000E+00, 1.0000000000E+00)]),("D", [(2.9000000000E-01, 1.0000000000E+00)]),("D", [(7.0000000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(3.7792550772E+04, 5.7047888709E-04), (5.6680682165E+03, 4.4093016538E-03), (1.2898582841E+03, 2.2630967411E-02), (3.6486596028E+02, 8.8025644295E-02), (1.1857631515E+02, 2.5223701612E-01), (4.2024867605E+01, 4.5960547169E-01), (1.5499501629E+01, 3.3277886014E-01)]),("S", [(7.5208026598E+01, 1.9250560190E-02), (2.3031408972E+01, 8.7906743952E-02), (3.6348797649E+00, -3.4246704535E-01)]),("S", [(1.6065049957E+00, 1.5106266058E+00), (7.6103394581E-01, 5.8071016470E-01)]),("S", [(1.6556708849E-01, 1.0000000000E+00)]),("S", [(6.0041577113E-02, 1.0000000000E+00)]),("P", [(4.5252303192E+02, 2.3110812466E-03), (1.0708195049E+02, 1.8568641823E-02), (3.4131021255E+01, 8.7216237035E-02), (1.2587037428E+01, 2.6902101523E-01), (4.9811919704E+00, 5.2128324272E-01)]),("P", [(2.0070350900E+00, 6.0271687494E-01)]),("P", [(8.0083714374E-01, 1.0000000000E+00)]),("P", [(2.0178927472E-01, 1.0000000000E+00)]),("P", [(5.7895550392E-02, 1.0000000000E+00)]),("D", [(1.5700000000E+00, 2.0000000000E-01), (3.3300000000E-01, 1.0000000000E+00)]),("D", [(1.1100000000E-01, 1.0000000000E+00)]),("F", [(2.4400000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(4.4773358078E+04, 5.5914765868E-04), (6.7171992104E+03, 4.3206040189E-03), (1.5288960325E+03, 2.2187096460E-02), (4.3254746585E+02, 8.6489249116E-02), (1.4061505226E+02, 2.4939889716E-01), (4.9857636724E+01, 4.6017197366E-01), (1.8434974885E+01, 3.4250236575E-01)]),("S", [(8.6533886111E+01, 2.1300063007E-02), (2.6624606846E+01, 9.4676139318E-02), (4.4953057159E+00, -3.2616264859E-01)]),("S", [(2.1035045710E+00, 1.3980803850E+00), (1.0106094922E+00, 6.3865786699E-01)]),("S", [(2.3701751489E-01, 1.0000000000E+00)]),("S", [(8.5703405362E-02, 1.0000000000E+00)]),("P", [(3.9447503628E+02, 2.6285693959E-03), (9.3137683104E+01, 2.0556257749E-02), (2.9519608742E+01, 9.2070262801E-02), (1.0781663791E+01, 2.5565889739E-01), (4.1626574778E+00, 4.2111707185E-01)]),("P", [(1.6247972989E+00, 3.4401746318E-01)]),("P", [(5.4306660493E-01, 1.0000000000E+00)]),("P", [(2.0582073956E-01, 1.0000000000E+00)]),("P", [(7.0053487306E-02, 1.0000000000E+00)]),("D", [(2.3030000000E+00, 2.0000000000E-01), (4.7600000000E-01, 1.0000000000E+00)]),("D", [(1.6000000000E-01, 1.0000000000E+00)]),("F", [(3.3600000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(5.2426999233E+04, 5.5207164100E-04), (7.8632660552E+03, 4.2678595308E-03), (1.7895227333E+03, 2.1931529186E-02), (5.0627300165E+02, 8.5667168373E-02), (1.6460698546E+02, 2.4840686605E-01), (5.8391918722E+01, 4.6336753971E-01), (2.1643663201E+01, 3.5350558156E-01)]),("S", [(9.9013837620E+01, 2.1895679958E-02), (3.0550439817E+01, 9.5650470295E-02), (5.4537087661E+00, -2.9454270186E-01)]),("S", [(2.6503362563E+00, 1.3294381200E+00), (1.2726688867E+00, 6.6109396473E-01)]),("S", [(3.1645005203E-01, 1.0000000000E+00)]),("S", [(1.1417466938E-01, 1.0000000000E+00)]),("P", [(4.7227219248E+02, 2.5710623052E-03), (1.1158882756E+02, 2.0250297999E-02), (3.5445936418E+01, 9.1580716787E-02), (1.2990776875E+01, 2.5749454014E-01), (5.0486221658E+00, 4.2862899758E-01)]),("P", [(1.9934049566E+00, 3.4359817849E-01)]),("P", [(6.6527284430E-01, 1.0000000000E+00)]),("P", [(2.5516832128E-01, 1.0000000000E+00)]),("P", [(9.0357762251E-02, 1.0000000000E+00)]),("D", [(3.1200000000E+00, 2.0000000000E-01), (6.4800000000E-01, 1.0000000000E+00)]),("D", [(2.1800000000E-01, 1.0000000000E+00)]),("F", [(4.5200000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(6.0700928104E+04, 5.4695944225E-04), (9.1026106854E+03, 4.2297224557E-03), (2.0714166009E+03, 2.1747824159E-02), (5.8602476821E+02, 8.5100053589E-02), (1.9055395021E+02, 2.4799128459E-01), (6.7630384260E+01, 4.6703640406E-01), (2.5127306905E+01, 3.6434587550E-01)]),("S", [(1.1257463010E+02, 2.1670040240E-02), (3.4795554217E+01, 9.3602301760E-02), (6.5115556215E+00, -2.6068001422E-01)]),("S", [(3.2399032261E+00, 1.2842089435E+00), (1.5477160881E+00, 6.6036416584E-01)]),("S", [(4.0541030112E-01, 1.0000000000E+00)]),("S", [(1.4550651059E-01, 1.0000000000E+00)]),("P", [(5.6436716027E+02, 2.4796796317E-03), (1.3342624379E+02, 1.9677930250E-02), (4.2468271189E+01, 8.9980008258E-02), (1.5616527580E+01, 2.5705880575E-01), (6.1093988469E+00, 4.3515167292E-01)]),("P", [(2.4404160198E+00, 3.4883240595E-01)]),("P", [(8.3882201296E-01, 1.0000000000E+00)]),("P", [(3.1288746900E-01, 1.0000000000E+00)]),("P", [(1.0770109004E-01, 1.0000000000E+00)]),("D", [(3.7560000000E+00, 2.0000000000E-01), (8.1200000000E-01, 1.0000000000E+00)]),("D", [(2.7300000000E-01, 1.0000000000E+00)]),("F", [(5.5700000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(6.9507990945E+04, 5.4314897497E-04), (1.0426156880E+04, 4.1990463961E-03), (2.3732334061E+03, 2.1592141679E-02), (6.7156420071E+02, 8.4598850094E-02), (2.1841999790E+02, 2.4757249724E-01), (7.7572249714E+01, 4.7016930228E-01), (2.8888815277E+01, 3.7436370716E-01)]),("S", [(1.2710527185E+02, 2.5182166603E-02), (3.9339582961E+01, 1.0786112456E-01), (7.6740679989E+00, -2.7408821574E-01)]),("S", [(3.8745627630E+00, 1.3213875014E+00), (1.8385832573E+00, 6.8636955368E-01)]),("S", [(5.0229057542E-01, 1.0000000000E+00)]),("S", [(1.7962723420E-01, 1.0000000000E+00)]),("P", [(6.6650423284E+02, 2.3632663836E-03), (1.5764241690E+02, 1.8879300374E-02), (5.0262520978E+01, 8.7206341273E-02), (1.8536078105E+01, 2.5285612970E-01), (7.2940532777E+00, 4.3507154820E-01)]),("P", [(2.9433248995E+00, 3.5026513165E-01)]),("P", [(1.0404970818E+00, 1.0000000000E+00)]),("P", [(3.8456415080E-01, 1.0000000000E+00)]),("P", [(1.3069642732E-01, 1.0000000000E+00)]),("D", [(4.6100000000E+00, 2.0000000000E-01), (1.0110000000E+00, 1.0000000000E+00)]),("D", [(3.3900000000E-01, 1.0000000000E+00)]),("F", [(7.0600000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(7.9111422998E+04, 5.1029325002E-04), (1.1864746009E+04, 3.9463036981E-03), (2.7001642973E+03, 2.0307073910E-02), (7.6395943485E+02, 7.9691825214E-02), (2.4845150561E+02, 2.3420623836E-01), (8.8283581000E+01, 4.4833849481E-01), (3.2948607069E+01, 3.6408167400E-01)]),("S", [(1.4255358000E+02, 2.6387407001E-02), (4.4163688009E+01, 1.1226433999E-01), (8.9524995000E+00, -2.6178922001E-01)]),("S", [(4.5546920941E+00, 1.3002484998E+00), (2.1444079001E+00, 6.7197237009E-01)]),("S", [(6.0708777004E-01, 1.0000000000E+00)]),("S", [(2.1651431999E-01, 1.0000000000E+00)]),("P", [(7.7677541998E+02, 2.2028005003E-03), (1.8380107018E+02, 1.7694180008E-02), (5.8694003175E+01, 8.2431293717E-02), (2.1701591695E+01, 2.4207278863E-01), (8.5821489635E+00, 4.2263558251E-01)]),("P", [(3.4922679161E+00, 3.4151806086E-01)]),("P", [(1.2637426998E+00, 1.0000000000E+00)]),("P", [(4.6607870005E-01, 1.0000000000E+00)]),("P", [(1.5766003000E-01, 1.0000000000E+00)]),("D", [(5.5510000000E+00, 2.0000000000E-01), (1.2350000000E+00, 1.0000000000E+00)]),("D", [(4.1200000000E-01, 1.0000000000E+00)]),("F", [(8.9000000000E-01, 1.0000000000E+00)])],
}



DEF2_TZVPD = {

    1 : [("S", [(3.4061341000E+01, 6.0251978000E-03), (5.1235746000E+00, 4.5021094000E-02), (1.1646626000E+00, 2.0189726000E-01)]),("S", [(3.2723041000E-01, 1.0000000000E+00)]),("S", [(1.0307241000E-01, 1.0000000000E+00)]),("P", [(8.0000000000E-01, 1.0000000000E+00)]),("P", [(9.5774129632E-02, 1.0000000000E+00)])],
    2 : [("S", [(9.8078321616E+01, 7.5803064967E-03), (1.4764404247E+01, 5.4848620937E-02), (3.3185831473E+00, 2.2074382186E-01)]),("S", [(8.7413869551E-01, 1.0000000000E+00)]),("S", [(2.4459897208E-01, 1.0000000000E+00)]),("P", [(1.0000000000E+00, 1.0000000000E+00)]),("P", [(1.9404104917E-01, 1.0000000000E+00)])],
    3 : [("S", [(6.2692628010E+03, 2.0540968826E-04), (9.4031612431E+02, 1.5916554089E-03), (2.1422107528E+02, 8.2869829707E-03), (6.0759840184E+01, 3.3856374249E-02), (1.9915152032E+01, 1.1103225876E-01), (7.3171509797E+00, 2.7449383329E-01)]),("S", [(2.9724674216E+00, 2.3792456411E-01), (1.2639852314E+00, 3.0765411924E-01)]),("S", [(5.1427489953E-01, 1.0000000000E+00)]),("S", [(7.7030885901E-02, 1.0000000000E+00)]),("S", [(2.8938896433E-02, 1.0000000000E+00)]),("P", [(3.3270000000E+00, 1.0000000000E+00)]),("P", [(4.0000000000E-01, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(2.0097234766E-02, 1.0000000000E+00)])],
    4 : [("S", [(4.7002365626E+03, 2.3584389316E-04), (7.0482845622E+02, 1.8243791019E-03), (1.6043110478E+02, 9.3966148224E-03), (4.5425347336E+01, 3.6908924159E-02), (1.4798334125E+01, 1.0897561281E-01), (5.3512452537E+00, 2.1694284551E-01)]),("S", [(2.1542044819E+00, 4.4695408857E-01), (9.3363744400E-01, 2.0866985771E-01)]),("S", [(1.8791432989E-01, 1.0000000000E+00)]),("S", [(7.4648267947E-02, 1.0000000000E+00)]),("S", [(3.2650484598E-02, 1.0000000000E+00)]),("P", [(3.6316917145E+00, -2.9033998305E-02)]),("P", [(7.1695694366E-01, -1.6877854032E-01), (1.9541932860E-01, -5.1403419628E-01)]),("P", [(6.0515465890E-02, 1.0000000000E+00)]),("P", [(2.7726324608E-02, 1.0000000000E+00)]),("D", [(1.8000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(8.5648660687E+03, 2.2837198155E-04), (1.2841516263E+03, 1.7682576447E-03), (2.9227871604E+02, 9.1407080516E-03), (8.2775469176E+01, 3.6342638989E-02), (2.7017939269E+01, 1.1063458441E-01), (9.8149619660E+00, 2.3367344321E-01)]),("S", [(3.9318559059E+00, 4.1818777978E-01), (1.6595599712E+00, 2.2325473798E-01)]),("S", [(3.5762965239E-01, 1.0000000000E+00)]),("S", [(1.4246277496E-01, 1.0000000000E+00)]),("S", [(6.0560594768E-02, 1.0000000000E+00)]),("S", [(3.1334737069E-02, 1.0000000000E+00)]),("P", [(2.2453875803E+01, 5.0265575179E-03), (5.1045058330E+00, 3.2801738965E-02), (1.4986081344E+00, 1.3151230768E-01), (5.0927831315E-01, 3.3197167769E-01)]),("P", [(1.8147077798E-01, 4.7314319570E-01)]),("P", [(6.4621893904E-02, 2.5802783943E-01)]),("D", [(6.6100000000E-01, 1.0000000000E+00)]),("D", [(1.9900000000E-01, 1.0000000000E+00)]),("D", [(5.9549085820E-02, 1.0000000000E+00)]),("F", [(4.9000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(1.3575349682E+04, 2.2245814352E-04), (2.0352333680E+03, 1.7232738252E-03), (4.6322562359E+02, 8.9255715314E-03), (1.3120019598E+02, 3.5727984502E-02), (4.2853015891E+01, 1.1076259931E-01), (1.5584185766E+01, 2.4295627626E-01)]),("S", [(6.2067138508E+00, 4.1440263448E-01), (2.5764896527E+00, 2.3744968655E-01)]),("S", [(5.7696339419E-01, 1.0000000000E+00)]),("S", [(2.2972831358E-01, 1.0000000000E+00)]),("S", [(9.5164440028E-02, 1.0000000000E+00)]),("S", [(4.8475401370E-02, 1.0000000000E+00)]),("P", [(3.4697232244E+01, 5.3333657805E-03), (7.9582622826E+00, 3.5864109092E-02), (2.3780826883E+00, 1.4215873329E-01), (8.1433208183E-01, 3.4270471845E-01)]),("P", [(2.8887547253E-01, 4.6445822433E-01)]),("P", [(1.0056823671E-01, 2.4955789874E-01)]),("D", [(1.0970000000E+00, 1.0000000000E+00)]),("D", [(3.1800000000E-01, 1.0000000000E+00)]),("D", [(9.0985336424E-02, 1.0000000000E+00)]),("F", [(7.6100000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(1.9730800647E+04, 2.1887984991E-04), (2.9578958745E+03, 1.6960708803E-03), (6.7322133595E+02, 8.7954603538E-03), (1.9068249494E+02, 3.5359382605E-02), (6.2295441898E+01, 1.1095789217E-01), (2.2654161182E+01, 2.4982972552E-01)]),("S", [(8.9791477428E+00, 4.0623896148E-01), (3.6863002370E+00, 2.4338217176E-01)]),("S", [(8.4660076805E-01, 1.0000000000E+00)]),("S", [(3.3647133771E-01, 1.0000000000E+00)]),("S", [(1.3647653675E-01, 1.0000000000E+00)]),("S", [(6.8441605847E-02, 1.0000000000E+00)]),("P", [(4.9200380510E+01, 5.5552416751E-03), (1.1346790537E+01, 3.8052379723E-02), (3.4273972411E+00, 1.4953671029E-01), (1.1785525134E+00, 3.4949305230E-01)]),("P", [(4.1642204972E-01, 4.5843153697E-01)]),("P", [(1.4260826011E-01, 2.4428771672E-01)]),("D", [(1.6540000000E+00, 1.0000000000E+00)]),("D", [(4.6900000000E-01, 1.0000000000E+00)]),("D", [(1.2829642058E-01, 1.0000000000E+00)]),("F", [(1.0930000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(2.7032382631E+04, 2.1726302465E-04), (4.0523871392E+03, 1.6838662199E-03), (9.2232722710E+02, 8.7395616265E-03), (2.6124070989E+02, 3.5239968808E-02), (8.5354641351E+01, 1.1153519115E-01), (3.1035035245E+01, 2.5588953961E-01)]),("S", [(1.2260860728E+01, 3.9768730901E-01), (4.9987076005E+00, 2.4627849430E-01)]),("S", [(1.1703108158E+00, 1.0000000000E+00)]),("S", [(4.6474740994E-01, 1.0000000000E+00)]),("S", [(1.8504536357E-01, 1.0000000000E+00)]),("S", [(7.0288026270E-02, 1.0000000000E+00)]),("P", [(6.3274954801E+01, 6.0685103418E-03), (1.4627049379E+01, 4.1912575824E-02), (4.4501223456E+00, 1.6153841088E-01), (1.5275799647E+00, 3.5706951311E-01)]),("P", [(5.2935117943E-01, 4.4794207502E-01)]),("P", [(1.7478421270E-01, 2.4446069663E-01)]),("P", [(5.1112745706E-02, 1.0000000000E+00)]),("D", [(2.3140000000E+00, 1.0000000000E+00)]),("D", [(6.4500000000E-01, 1.0000000000E+00)]),("D", [(1.4696477366E-01, 1.0000000000E+00)]),("F", [(1.4280000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(3.5479100441E+04, 2.1545014888E-04), (5.3184728983E+03, 1.6700686527E-03), (1.2104810975E+03, 8.6733211476E-03), (3.4285518140E+02, 3.5049933175E-02), (1.1201943181E+02, 1.1165320133E-01), (4.0714740248E+01, 2.5988506647E-01)]),("S", [(1.6039678111E+01, 3.9422966880E-01), (6.5038186740E+00, 2.4998238551E-01)]),("S", [(1.5440477509E+00, 1.0000000000E+00)]),("S", [(6.1223452862E-01, 1.0000000000E+00)]),("S", [(2.4027979698E-01, 1.0000000000E+00)]),("S", [(9.0918446478E-02, 1.0000000000E+00)]),("P", [(8.0233900483E+01, 6.3685999134E-03), (1.8594010743E+01, 4.4303143530E-02), (5.6867902653E+00, 1.6867248708E-01), (1.9511006294E+00, 3.6166346255E-01)]),("P", [(6.6970211298E-01, 4.4202901491E-01)]),("P", [(2.1651300410E-01, 2.4319875730E-01)]),("P", [(5.9613282472E-02, 1.0000000000E+00)]),("D", [(3.1070000000E+00, 1.0000000000E+00)]),("D", [(8.5500000000E-01, 1.0000000000E+00)]),("D", [(1.8608388111E-01, 1.0000000000E+00)]),("F", [(1.9170000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(4.5069464022E+04, 2.1687155182E-04), (6.7559768656E+03, 1.6812736757E-03), (1.5376502864E+03, 8.7356062782E-03), (4.3551697667E+02, 3.5361266922E-02), (1.4228655638E+02, 1.1321521454E-01), (5.1692153804E+01, 2.6654653104E-01)]),("S", [(2.0315870490E+01, 3.9631959951E-01), (8.2021942646E+00, 2.5582811251E-01)]),("S", [(1.9681276278E+00, 1.0000000000E+00)]),("S", [(7.7904756001E-01, 1.0000000000E+00)]),("S", [(3.0229502043E-01, 1.0000000000E+00)]),("S", [(1.1246533684E-01, 1.0000000000E+00)]),("P", [(9.9782996032E+01, 6.5569234163E-03), (2.3176124101E+01, 4.5888009138E-02), (7.1163945872E+00, 1.7331287812E-01), (2.4418711435E+00, 3.6475267512E-01)]),("P", [(8.3389605766E-01, 4.3831075171E-01)]),("P", [(2.6607311301E-01, 2.4160029835E-01)]),("P", [(7.1067552401E-02, 1.0000000000E+00)]),("D", [(4.0140000000E+00, 1.0000000000E+00)]),("D", [(1.0960000000E+00, 1.0000000000E+00)]),("D", [(2.3044808390E-01, 1.0000000000E+00)]),("F", [(2.5440000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(2.6041109927E+04, 6.1806342811E-04), (3.9061268548E+03, 4.7748604414E-03), (8.8897454993E+02, 2.4471684829E-02), (2.5145497961E+02, 9.4755394977E-02), (8.1650143512E+01, 2.6867496920E-01), (2.8904158401E+01, 4.7925475440E-01), (1.0625782932E+01, 3.3248591469E-01)]),("S", [(5.3769410179E+01, 1.9527731872E-02), (1.6308243025E+01, 9.2648010794E-02), (2.3730384125E+00, -3.9938670172E-01)]),("S", [(9.5730772603E-01, 1.6428595391E+00), (4.0806460959E-01, 5.5692596966E-01)]),("S", [(4.9967582329E-02, 1.0000000000E+00)]),("S", [(1.9268616250E-02, 1.0000000000E+00)]),("P", [(1.3807979989E+02, 5.7951891929E-03), (3.2232700393E+01, 4.1620846251E-02), (9.9816075360E+00, 1.6281916885E-01), (3.4822033928E+00, 3.6011784647E-01), (1.2299134620E+00, 4.4858979889E-01)]),("P", [(4.1743959423E-01, 2.3040759559E-01)]),("P", [(9.1000000000E-02, 1.0000000000E+00)]),("P", [(3.0000000000E-02, 1.0000000000E+00)]),("P", [(1.1082810218E-02, 1.0000000000E+00)]),("D", [(2.6090000000E+00, 1.0000000000E+00)]),("D", [(4.3000000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(3.1438349555E+04, 6.0912311326E-04), (4.7155153354E+03, 4.7066196465E-03), (1.0731629247E+03, 2.4135820657E-02), (3.0357238768E+02, 9.3628959834E-02), (9.8626251042E+01, 2.6646742093E-01), (3.4943808417E+01, 4.7890929917E-01), (1.2859785199E+01, 3.3698490286E-01)]),("S", [(6.4876913004E+01, 1.9180889307E-02), (1.9725520777E+01, 9.0913704392E-02), (2.8951804339E+00, -3.9563756125E-01)]),("S", [(1.1960454710E+00, 1.6827603373E+00), (5.4329451156E-01, 5.2141091954E-01)]),("S", [(1.0099104092E-01, 1.0000000000E+00)]),("S", [(3.6865728085E-02, 1.0000000000E+00)]),("P", [(1.7987189612E+02, 5.3799549018E-03), (4.2120069376E+01, 3.9318014098E-02), (1.3120503032E+01, 1.5740129476E-01), (4.6257503609E+00, 3.5919094128E-01), (1.6695211016E+00, 4.5533379310E-01)]),("P", [(5.8551012105E-01, 2.1986432910E-01)]),("P", [(1.8914796195E-01, 1.0000000000E+00)]),("P", [(5.3768755187E-02, 1.0000000000E+00)]),("P", [(2.4576992970E-02, 1.0000000000E+00)]),("D", [(3.4440000000E+00, 1.0000000000E+00)]),("D", [(2.9000000000E-01, 1.0000000000E+00)]),("D", [(7.0000000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(3.7792550772E+04, 5.7047888709E-04), (5.6680682165E+03, 4.4093016538E-03), (1.2898582841E+03, 2.2630967411E-02), (3.6486596028E+02, 8.8025644295E-02), (1.1857631515E+02, 2.5223701612E-01), (4.2024867605E+01, 4.5960547169E-01), (1.5499501629E+01, 3.3277886014E-01)]),("S", [(7.5208026598E+01, 1.9250560190E-02), (2.3031408972E+01, 8.7906743952E-02), (3.6348797649E+00, -3.4246704535E-01)]),("S", [(1.6065049957E+00, 1.5106266058E+00), (7.6103394581E-01, 5.8071016470E-01)]),("S", [(1.6556708849E-01, 1.0000000000E+00)]),("S", [(6.0041577113E-02, 1.0000000000E+00)]),("S", [(2.8536651434E-02, 1.0000000000E+00)]),("P", [(4.5252303192E+02, 2.3110812466E-03), (1.0708195049E+02, 1.8568641823E-02), (3.4131021255E+01, 8.7216237035E-02), (1.2587037428E+01, 2.6902101523E-01), (4.9811919704E+00, 5.2128324272E-01)]),("P", [(2.0070350900E+00, 6.0271687494E-01)]),("P", [(8.0083714374E-01, 1.0000000000E+00)]),("P", [(2.0178927472E-01, 1.0000000000E+00)]),("P", [(5.7895550392E-02, 1.0000000000E+00)]),("D", [(1.5700000000E+00, 2.0000000000E-01), (3.3300000000E-01, 1.0000000000E+00)]),("D", [(1.1100000000E-01, 1.0000000000E+00)]),("D", [(4.1068707243E-02, 1.0000000000E+00)]),("F", [(2.4400000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(4.4773358078E+04, 5.5914765868E-04), (6.7171992104E+03, 4.3206040189E-03), (1.5288960325E+03, 2.2187096460E-02), (4.3254746585E+02, 8.6489249116E-02), (1.4061505226E+02, 2.4939889716E-01), (4.9857636724E+01, 4.6017197366E-01), (1.8434974885E+01, 3.4250236575E-01)]),("S", [(8.6533886111E+01, 2.1300063007E-02), (2.6624606846E+01, 9.4676139318E-02), (4.4953057159E+00, -3.2616264859E-01)]),("S", [(2.1035045710E+00, 1.3980803850E+00), (1.0106094922E+00, 6.3865786699E-01)]),("S", [(2.3701751489E-01, 1.0000000000E+00)]),("S", [(8.5703405362E-02, 1.0000000000E+00)]),("S", [(3.7000771280E-02, 1.0000000000E+00)]),("P", [(3.9447503628E+02, 2.6285693959E-03), (9.3137683104E+01, 2.0556257749E-02), (2.9519608742E+01, 9.2070262801E-02), (1.0781663791E+01, 2.5565889739E-01), (4.1626574778E+00, 4.2111707185E-01)]),("P", [(1.6247972989E+00, 3.4401746318E-01)]),("P", [(5.4306660493E-01, 1.0000000000E+00)]),("P", [(2.0582073956E-01, 1.0000000000E+00)]),("P", [(7.0053487306E-02, 1.0000000000E+00)]),("D", [(2.3030000000E+00, 2.0000000000E-01), (4.7600000000E-01, 1.0000000000E+00)]),("D", [(1.6000000000E-01, 1.0000000000E+00)]),("D", [(5.5036090333E-02, 1.0000000000E+00)]),("F", [(3.3600000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(5.2426999233E+04, 5.5207164100E-04), (7.8632660552E+03, 4.2678595308E-03), (1.7895227333E+03, 2.1931529186E-02), (5.0627300165E+02, 8.5667168373E-02), (1.6460698546E+02, 2.4840686605E-01), (5.8391918722E+01, 4.6336753971E-01), (2.1643663201E+01, 3.5350558156E-01)]),("S", [(9.9013837620E+01, 2.1895679958E-02), (3.0550439817E+01, 9.5650470295E-02), (5.4537087661E+00, -2.9454270186E-01)]),("S", [(2.6503362563E+00, 1.3294381200E+00), (1.2726688867E+00, 6.6109396473E-01)]),("S", [(3.1645005203E-01, 1.0000000000E+00)]),("S", [(1.1417466938E-01, 1.0000000000E+00)]),("S", [(4.8831315225E-02, 1.0000000000E+00)]),("P", [(4.7227219248E+02, 2.5710623052E-03), (1.1158882756E+02, 2.0250297999E-02), (3.5445936418E+01, 9.1580716787E-02), (1.2990776875E+01, 2.5749454014E-01), (5.0486221658E+00, 4.2862899758E-01)]),("P", [(1.9934049566E+00, 3.4359817849E-01)]),("P", [(6.6527284430E-01, 1.0000000000E+00)]),("P", [(2.5516832128E-01, 1.0000000000E+00)]),("P", [(9.0357762251E-02, 1.0000000000E+00)]),("D", [(3.1200000000E+00, 2.0000000000E-01), (6.4800000000E-01, 1.0000000000E+00)]),("D", [(2.1800000000E-01, 1.0000000000E+00)]),("D", [(7.3569545334E-02, 1.0000000000E+00)]),("F", [(4.5200000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(6.0700928104E+04, 5.4695944225E-04), (9.1026106854E+03, 4.2297224557E-03), (2.0714166009E+03, 2.1747824159E-02), (5.8602476821E+02, 8.5100053589E-02), (1.9055395021E+02, 2.4799128459E-01), (6.7630384260E+01, 4.6703640406E-01), (2.5127306905E+01, 3.6434587550E-01)]),("S", [(1.1257463010E+02, 2.1670040240E-02), (3.4795554217E+01, 9.3602301760E-02), (6.5115556215E+00, -2.6068001422E-01)]),("S", [(3.2399032261E+00, 1.2842089435E+00), (1.5477160881E+00, 6.6036416584E-01)]),("S", [(4.0541030112E-01, 1.0000000000E+00)]),("S", [(1.4550651059E-01, 1.0000000000E+00)]),("S", [(5.0178839540E-02, 1.0000000000E+00)]),("P", [(5.6436716027E+02, 2.4796796317E-03), (1.3342624379E+02, 1.9677930250E-02), (4.2468271189E+01, 8.9980008258E-02), (1.5616527580E+01, 2.5705880575E-01), (6.1093988469E+00, 4.3515167292E-01)]),("P", [(2.4404160198E+00, 3.4883240595E-01)]),("P", [(8.3882201296E-01, 1.0000000000E+00)]),("P", [(3.1288746900E-01, 1.0000000000E+00)]),("P", [(1.0770109004E-01, 1.0000000000E+00)]),("P", [(3.0359833649E-02, 1.0000000000E+00)]),("D", [(3.7560000000E+00, 2.0000000000E-01), (8.1200000000E-01, 1.0000000000E+00)]),("D", [(2.7300000000E-01, 1.0000000000E+00)]),("D", [(8.1547024970E-02, 1.0000000000E+00)]),("F", [(5.5700000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(6.9507990945E+04, 5.4314897497E-04), (1.0426156880E+04, 4.1990463961E-03), (2.3732334061E+03, 2.1592141679E-02), (6.7156420071E+02, 8.4598850094E-02), (2.1841999790E+02, 2.4757249724E-01), (7.7572249714E+01, 4.7016930228E-01), (2.8888815277E+01, 3.7436370716E-01)]),("S", [(1.2710527185E+02, 2.5182166603E-02), (3.9339582961E+01, 1.0786112456E-01), (7.6740679989E+00, -2.7408821574E-01)]),("S", [(3.8745627630E+00, 1.3213875014E+00), (1.8385832573E+00, 6.8636955368E-01)]),("S", [(5.0229057542E-01, 1.0000000000E+00)]),("S", [(1.7962723420E-01, 1.0000000000E+00)]),("S", [(6.1133095854E-02, 1.0000000000E+00)]),("P", [(6.6650423284E+02, 2.3632663836E-03), (1.5764241690E+02, 1.8879300374E-02), (5.0262520978E+01, 8.7206341273E-02), (1.8536078105E+01, 2.5285612970E-01), (7.2940532777E+00, 4.3507154820E-01)]),("P", [(2.9433248995E+00, 3.5026513165E-01)]),("P", [(1.0404970818E+00, 1.0000000000E+00)]),("P", [(3.8456415080E-01, 1.0000000000E+00)]),("P", [(1.3069642732E-01, 1.0000000000E+00)]),("P", [(3.5381580722E-02, 1.0000000000E+00)]),("D", [(4.6100000000E+00, 2.0000000000E-01), (1.0110000000E+00, 1.0000000000E+00)]),("D", [(3.3900000000E-01, 1.0000000000E+00)]),("D", [(9.9601358790E-02, 1.0000000000E+00)]),("F", [(7.0600000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(7.9111422998E+04, 5.1029325002E-04), (1.1864746009E+04, 3.9463036981E-03), (2.7001642973E+03, 2.0307073910E-02), (7.6395943485E+02, 7.9691825214E-02), (2.4845150561E+02, 2.3420623836E-01), (8.8283581000E+01, 4.4833849481E-01), (3.2948607069E+01, 3.6408167400E-01)]),("S", [(1.4255358000E+02, 2.6387407001E-02), (4.4163688009E+01, 1.1226433999E-01), (8.9524995000E+00, -2.6178922001E-01)]),("S", [(4.5546920941E+00, 1.3002484998E+00), (2.1444079001E+00, 6.7197237009E-01)]),("S", [(6.0708777004E-01, 1.0000000000E+00)]),("S", [(2.1651431999E-01, 1.0000000000E+00)]),("S", [(7.3732186754E-02, 1.0000000000E+00)]),("P", [(7.7677541998E+02, 2.2028005003E-03), (1.8380107018E+02, 1.7694180008E-02), (5.8694003175E+01, 8.2431293717E-02), (2.1701591695E+01, 2.4207278863E-01), (8.5821489635E+00, 4.2263558251E-01)]),("P", [(3.4922679161E+00, 3.4151806086E-01)]),("P", [(1.2637426998E+00, 1.0000000000E+00)]),("P", [(4.6607870005E-01, 1.0000000000E+00)]),("P", [(1.5766003000E-01, 1.0000000000E+00)]),("P", [(4.1779128551E-02, 1.0000000000E+00)]),("D", [(5.5510000000E+00, 2.0000000000E-01), (1.2350000000E+00, 1.0000000000E+00)]),("D", [(4.1200000000E-01, 1.0000000000E+00)]),("D", [(1.2039311859E-01, 1.0000000000E+00)]),("F", [(8.9000000000E-01, 1.0000000000E+00)])],
}



DEF2_TZVPP = {

    1 : [("S", [(3.4061341000E+01, 6.0251978000E-03), (5.1235746000E+00, 4.5021094000E-02), (1.1646626000E+00, 2.0189726000E-01)]),("S", [(3.2723041000E-01, 1.0000000000E+00)]),("S", [(1.0307241000E-01, 1.0000000000E+00)]),("P", [(1.4070000000E+00, 1.0000000000E+00)]),("P", [(3.8800000000E-01, 1.0000000000E+00)]),("D", [(1.0570000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(9.8078321616E+01, 7.5803064967E-03), (1.4764404247E+01, 5.4848620937E-02), (3.3185831473E+00, 2.2074382186E-01)]),("S", [(8.7413869551E-01, 1.0000000000E+00)]),("S", [(2.4459897208E-01, 1.0000000000E+00)]),("P", [(3.0440000000E+00, 1.0000000000E+00)]),("P", [(7.5800000000E-01, 1.0000000000E+00)]),("D", [(1.9650000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.2692628010E+03, 2.0540968826E-04), (9.4031612431E+02, 1.5916554089E-03), (2.1422107528E+02, 8.2869829707E-03), (6.0759840184E+01, 3.3856374249E-02), (1.9915152032E+01, 1.1103225876E-01), (7.3171509797E+00, 2.7449383329E-01)]),("S", [(2.9724674216E+00, 2.3792456411E-01), (1.2639852314E+00, 3.0765411924E-01)]),("S", [(5.1427489953E-01, 1.0000000000E+00)]),("S", [(7.7030885901E-02, 1.0000000000E+00)]),("S", [(2.8938896433E-02, 1.0000000000E+00)]),("P", [(3.3270000000E+00, 1.0000000000E+00)]),("P", [(4.0000000000E-01, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(1.3190000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(4.7002365626E+03, 2.3584389316E-04), (7.0482845622E+02, 1.8243791019E-03), (1.6043110478E+02, 9.3966148224E-03), (4.5425347336E+01, 3.6908924159E-02), (1.4798334125E+01, 1.0897561281E-01), (5.3512452537E+00, 2.1694284551E-01)]),("S", [(2.1542044819E+00, 4.4695408857E-01), (9.3363744400E-01, 2.0866985771E-01)]),("S", [(1.8791432989E-01, 1.0000000000E+00)]),("S", [(7.4648267947E-02, 1.0000000000E+00)]),("S", [(3.2650484598E-02, 1.0000000000E+00)]),("P", [(3.6316917145E+00, -2.9033998305E-02)]),("P", [(7.1695694366E-01, -1.6877854032E-01), (1.9541932860E-01, -5.1403419628E-01)]),("P", [(6.0515465890E-02, 1.0000000000E+00)]),("D", [(1.8000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(8.5648660687E+03, 2.2837198155E-04), (1.2841516263E+03, 1.7682576447E-03), (2.9227871604E+02, 9.1407080516E-03), (8.2775469176E+01, 3.6342638989E-02), (2.7017939269E+01, 1.1063458441E-01), (9.8149619660E+00, 2.3367344321E-01)]),("S", [(3.9318559059E+00, 4.1818777978E-01), (1.6595599712E+00, 2.2325473798E-01)]),("S", [(3.5762965239E-01, 1.0000000000E+00)]),("S", [(1.4246277496E-01, 1.0000000000E+00)]),("S", [(6.0560594768E-02, 1.0000000000E+00)]),("P", [(2.2453875803E+01, 5.0265575179E-03), (5.1045058330E+00, 3.2801738965E-02), (1.4986081344E+00, 1.3151230768E-01), (5.0927831315E-01, 3.3197167769E-01)]),("P", [(1.8147077798E-01, 4.7314319570E-01)]),("P", [(6.4621893904E-02, 2.5802783943E-01)]),("D", [(6.6100000000E-01, 1.0000000000E+00)]),("D", [(1.9900000000E-01, 1.0000000000E+00)]),("F", [(4.9000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(1.3575349682E+04, 2.2245814352E-04), (2.0352333680E+03, 1.7232738252E-03), (4.6322562359E+02, 8.9255715314E-03), (1.3120019598E+02, 3.5727984502E-02), (4.2853015891E+01, 1.1076259931E-01), (1.5584185766E+01, 2.4295627626E-01)]),("S", [(6.2067138508E+00, 4.1440263448E-01), (2.5764896527E+00, 2.3744968655E-01)]),("S", [(5.7696339419E-01, 1.0000000000E+00)]),("S", [(2.2972831358E-01, 1.0000000000E+00)]),("S", [(9.5164440028E-02, 1.0000000000E+00)]),("P", [(3.4697232244E+01, 5.3333657805E-03), (7.9582622826E+00, 3.5864109092E-02), (2.3780826883E+00, 1.4215873329E-01), (8.1433208183E-01, 3.4270471845E-01)]),("P", [(2.8887547253E-01, 4.6445822433E-01)]),("P", [(1.0056823671E-01, 2.4955789874E-01)]),("D", [(1.0970000000E+00, 1.0000000000E+00)]),("D", [(3.1800000000E-01, 1.0000000000E+00)]),("F", [(7.6100000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(1.9730800647E+04, 2.1887984991E-04), (2.9578958745E+03, 1.6960708803E-03), (6.7322133595E+02, 8.7954603538E-03), (1.9068249494E+02, 3.5359382605E-02), (6.2295441898E+01, 1.1095789217E-01), (2.2654161182E+01, 2.4982972552E-01)]),("S", [(8.9791477428E+00, 4.0623896148E-01), (3.6863002370E+00, 2.4338217176E-01)]),("S", [(8.4660076805E-01, 1.0000000000E+00)]),("S", [(3.3647133771E-01, 1.0000000000E+00)]),("S", [(1.3647653675E-01, 1.0000000000E+00)]),("P", [(4.9200380510E+01, 5.5552416751E-03), (1.1346790537E+01, 3.8052379723E-02), (3.4273972411E+00, 1.4953671029E-01), (1.1785525134E+00, 3.4949305230E-01)]),("P", [(4.1642204972E-01, 4.5843153697E-01)]),("P", [(1.4260826011E-01, 2.4428771672E-01)]),("D", [(1.6540000000E+00, 1.0000000000E+00)]),("D", [(4.6900000000E-01, 1.0000000000E+00)]),("F", [(1.0930000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(2.7032382631E+04, 2.1726302465E-04), (4.0523871392E+03, 1.6838662199E-03), (9.2232722710E+02, 8.7395616265E-03), (2.6124070989E+02, 3.5239968808E-02), (8.5354641351E+01, 1.1153519115E-01), (3.1035035245E+01, 2.5588953961E-01)]),("S", [(1.2260860728E+01, 3.9768730901E-01), (4.9987076005E+00, 2.4627849430E-01)]),("S", [(1.1703108158E+00, 1.0000000000E+00)]),("S", [(4.6474740994E-01, 1.0000000000E+00)]),("S", [(1.8504536357E-01, 1.0000000000E+00)]),("P", [(6.3274954801E+01, 6.0685103418E-03), (1.4627049379E+01, 4.1912575824E-02), (4.4501223456E+00, 1.6153841088E-01), (1.5275799647E+00, 3.5706951311E-01)]),("P", [(5.2935117943E-01, 4.4794207502E-01)]),("P", [(1.7478421270E-01, 2.4446069663E-01)]),("D", [(2.3140000000E+00, 1.0000000000E+00)]),("D", [(6.4500000000E-01, 1.0000000000E+00)]),("F", [(1.4280000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(3.5479100441E+04, 2.1545014888E-04), (5.3184728983E+03, 1.6700686527E-03), (1.2104810975E+03, 8.6733211476E-03), (3.4285518140E+02, 3.5049933175E-02), (1.1201943181E+02, 1.1165320133E-01), (4.0714740248E+01, 2.5988506647E-01)]),("S", [(1.6039678111E+01, 3.9422966880E-01), (6.5038186740E+00, 2.4998238551E-01)]),("S", [(1.5440477509E+00, 1.0000000000E+00)]),("S", [(6.1223452862E-01, 1.0000000000E+00)]),("S", [(2.4027979698E-01, 1.0000000000E+00)]),("P", [(8.0233900483E+01, 6.3685999134E-03), (1.8594010743E+01, 4.4303143530E-02), (5.6867902653E+00, 1.6867248708E-01), (1.9511006294E+00, 3.6166346255E-01)]),("P", [(6.6970211298E-01, 4.4202901491E-01)]),("P", [(2.1651300410E-01, 2.4319875730E-01)]),("D", [(3.1070000000E+00, 1.0000000000E+00)]),("D", [(8.5500000000E-01, 1.0000000000E+00)]),("F", [(1.9170000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(4.5069464022E+04, 2.1687155182E-04), (6.7559768656E+03, 1.6812736757E-03), (1.5376502864E+03, 8.7356062782E-03), (4.3551697667E+02, 3.5361266922E-02), (1.4228655638E+02, 1.1321521454E-01), (5.1692153804E+01, 2.6654653104E-01)]),("S", [(2.0315870490E+01, 3.9631959951E-01), (8.2021942646E+00, 2.5582811251E-01)]),("S", [(1.9681276278E+00, 1.0000000000E+00)]),("S", [(7.7904756001E-01, 1.0000000000E+00)]),("S", [(3.0229502043E-01, 1.0000000000E+00)]),("P", [(9.9782996032E+01, 6.5569234163E-03), (2.3176124101E+01, 4.5888009138E-02), (7.1163945872E+00, 1.7331287812E-01), (2.4418711435E+00, 3.6475267512E-01)]),("P", [(8.3389605766E-01, 4.3831075171E-01)]),("P", [(2.6607311301E-01, 2.4160029835E-01)]),("D", [(4.0140000000E+00, 1.0000000000E+00)]),("D", [(1.0960000000E+00, 1.0000000000E+00)]),("F", [(2.5440000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(2.6041109927E+04, 6.1806342811E-04), (3.9061268548E+03, 4.7748604414E-03), (8.8897454993E+02, 2.4471684829E-02), (2.5145497961E+02, 9.4755394977E-02), (8.1650143512E+01, 2.6867496920E-01), (2.8904158401E+01, 4.7925475440E-01), (1.0625782932E+01, 3.3248591469E-01)]),("S", [(5.3769410179E+01, 1.9527731872E-02), (1.6308243025E+01, 9.2648010794E-02), (2.3730384125E+00, -3.9938670172E-01)]),("S", [(9.5730772603E-01, 1.6428595391E+00), (4.0806460959E-01, 5.5692596966E-01)]),("S", [(4.9967582329E-02, 1.0000000000E+00)]),("S", [(1.9268616250E-02, 1.0000000000E+00)]),("P", [(1.3807979989E+02, 5.7951891929E-03), (3.2232700393E+01, 4.1620846251E-02), (9.9816075360E+00, 1.6281916885E-01), (3.4822033928E+00, 3.6011784647E-01), (1.2299134620E+00, 4.4858979889E-01)]),("P", [(4.1743959423E-01, 2.3040759559E-01)]),("P", [(9.1000000000E-02, 1.0000000000E+00)]),("P", [(3.0000000000E-02, 1.0000000000E+00)]),("D", [(2.6090000000E+00, 1.0000000000E+00)]),("D", [(4.3000000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(3.1438349555E+04, 6.0912311326E-04), (4.7155153354E+03, 4.7066196465E-03), (1.0731629247E+03, 2.4135820657E-02), (3.0357238768E+02, 9.3628959834E-02), (9.8626251042E+01, 2.6646742093E-01), (3.4943808417E+01, 4.7890929917E-01), (1.2859785199E+01, 3.3698490286E-01)]),("S", [(6.4876913004E+01, 1.9180889307E-02), (1.9725520777E+01, 9.0913704392E-02), (2.8951804339E+00, -3.9563756125E-01)]),("S", [(1.1960454710E+00, 1.6827603373E+00), (5.4329451156E-01, 5.2141091954E-01)]),("S", [(1.0099104092E-01, 1.0000000000E+00)]),("S", [(3.6865728085E-02, 1.0000000000E+00)]),("P", [(1.7987189612E+02, 5.3799549018E-03), (4.2120069376E+01, 3.9318014098E-02), (1.3120503032E+01, 1.5740129476E-01), (4.6257503609E+00, 3.5919094128E-01), (1.6695211016E+00, 4.5533379310E-01)]),("P", [(5.8551012105E-01, 2.1986432910E-01)]),("P", [(1.8914796195E-01, 1.0000000000E+00)]),("P", [(5.3768755187E-02, 1.0000000000E+00)]),("D", [(3.4440000000E+00, 1.0000000000E+00)]),("D", [(2.9000000000E-01, 1.0000000000E+00)]),("D", [(7.0000000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(3.7792550772E+04, 5.7047888709E-04), (5.6680682165E+03, 4.4093016538E-03), (1.2898582841E+03, 2.2630967411E-02), (3.6486596028E+02, 8.8025644295E-02), (1.1857631515E+02, 2.5223701612E-01), (4.2024867605E+01, 4.5960547169E-01), (1.5499501629E+01, 3.3277886014E-01)]),("S", [(7.5208026598E+01, 1.9250560190E-02), (2.3031408972E+01, 8.7906743952E-02), (3.6348797649E+00, -3.4246704535E-01)]),("S", [(1.6065049957E+00, 1.5106266058E+00), (7.6103394581E-01, 5.8071016470E-01)]),("S", [(1.6556708849E-01, 1.0000000000E+00)]),("S", [(6.0041577113E-02, 1.0000000000E+00)]),("P", [(4.5252303192E+02, 2.3110812466E-03), (1.0708195049E+02, 1.8568641823E-02), (3.4131021255E+01, 8.7216237035E-02), (1.2587037428E+01, 2.6902101523E-01), (4.9811919704E+00, 5.2128324272E-01)]),("P", [(2.0070350900E+00, 6.0271687494E-01)]),("P", [(8.0083714374E-01, 1.0000000000E+00)]),("P", [(2.0178927472E-01, 1.0000000000E+00)]),("P", [(5.7895550392E-02, 1.0000000000E+00)]),("D", [(1.5700000000E+00, 1.0000000000E+00)]),("D", [(3.3300000000E-01, 1.0000000000E+00)]),("D", [(1.1100000000E-01, 1.0000000000E+00)]),("F", [(2.4400000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(4.4773358078E+04, 5.5914765868E-04), (6.7171992104E+03, 4.3206040189E-03), (1.5288960325E+03, 2.2187096460E-02), (4.3254746585E+02, 8.6489249116E-02), (1.4061505226E+02, 2.4939889716E-01), (4.9857636724E+01, 4.6017197366E-01), (1.8434974885E+01, 3.4250236575E-01)]),("S", [(8.6533886111E+01, 2.1300063007E-02), (2.6624606846E+01, 9.4676139318E-02), (4.4953057159E+00, -3.2616264859E-01)]),("S", [(2.1035045710E+00, 1.3980803850E+00), (1.0106094922E+00, 6.3865786699E-01)]),("S", [(2.3701751489E-01, 1.0000000000E+00)]),("S", [(8.5703405362E-02, 1.0000000000E+00)]),("P", [(3.9447503628E+02, 2.6285693959E-03), (9.3137683104E+01, 2.0556257749E-02), (2.9519608742E+01, 9.2070262801E-02), (1.0781663791E+01, 2.5565889739E-01), (4.1626574778E+00, 4.2111707185E-01)]),("P", [(1.6247972989E+00, 3.4401746318E-01)]),("P", [(5.4306660493E-01, 1.0000000000E+00)]),("P", [(2.0582073956E-01, 1.0000000000E+00)]),("P", [(7.0053487306E-02, 1.0000000000E+00)]),("D", [(2.3030000000E+00, 1.0000000000E+00)]),("D", [(4.7600000000E-01, 1.0000000000E+00)]),("D", [(1.6000000000E-01, 1.0000000000E+00)]),("F", [(3.3600000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(5.2426999233E+04, 5.5207164100E-04), (7.8632660552E+03, 4.2678595308E-03), (1.7895227333E+03, 2.1931529186E-02), (5.0627300165E+02, 8.5667168373E-02), (1.6460698546E+02, 2.4840686605E-01), (5.8391918722E+01, 4.6336753971E-01), (2.1643663201E+01, 3.5350558156E-01)]),("S", [(9.9013837620E+01, 2.1895679958E-02), (3.0550439817E+01, 9.5650470295E-02), (5.4537087661E+00, -2.9454270186E-01)]),("S", [(2.6503362563E+00, 1.3294381200E+00), (1.2726688867E+00, 6.6109396473E-01)]),("S", [(3.1645005203E-01, 1.0000000000E+00)]),("S", [(1.1417466938E-01, 1.0000000000E+00)]),("P", [(4.7227219248E+02, 2.5710623052E-03), (1.1158882756E+02, 2.0250297999E-02), (3.5445936418E+01, 9.1580716787E-02), (1.2990776875E+01, 2.5749454014E-01), (5.0486221658E+00, 4.2862899758E-01)]),("P", [(1.9934049566E+00, 3.4359817849E-01)]),("P", [(6.6527284430E-01, 1.0000000000E+00)]),("P", [(2.5516832128E-01, 1.0000000000E+00)]),("P", [(9.0357762251E-02, 1.0000000000E+00)]),("D", [(3.1200000000E+00, 1.0000000000E+00)]),("D", [(6.4800000000E-01, 1.0000000000E+00)]),("D", [(2.1800000000E-01, 1.0000000000E+00)]),("F", [(4.5200000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(6.0700928104E+04, 5.4695944225E-04), (9.1026106854E+03, 4.2297224557E-03), (2.0714166009E+03, 2.1747824159E-02), (5.8602476821E+02, 8.5100053589E-02), (1.9055395021E+02, 2.4799128459E-01), (6.7630384260E+01, 4.6703640406E-01), (2.5127306905E+01, 3.6434587550E-01)]),("S", [(1.1257463010E+02, 2.1670040240E-02), (3.4795554217E+01, 9.3602301760E-02), (6.5115556215E+00, -2.6068001422E-01)]),("S", [(3.2399032261E+00, 1.2842089435E+00), (1.5477160881E+00, 6.6036416584E-01)]),("S", [(4.0541030112E-01, 1.0000000000E+00)]),("S", [(1.4550651059E-01, 1.0000000000E+00)]),("P", [(5.6436716027E+02, 2.4796796317E-03), (1.3342624379E+02, 1.9677930250E-02), (4.2468271189E+01, 8.9980008258E-02), (1.5616527580E+01, 2.5705880575E-01), (6.1093988469E+00, 4.3515167292E-01)]),("P", [(2.4404160198E+00, 3.4883240595E-01)]),("P", [(8.3882201296E-01, 1.0000000000E+00)]),("P", [(3.1288746900E-01, 1.0000000000E+00)]),("P", [(1.0770109004E-01, 1.0000000000E+00)]),("D", [(3.7560000000E+00, 1.0000000000E+00)]),("D", [(8.1200000000E-01, 1.0000000000E+00)]),("D", [(2.7300000000E-01, 1.0000000000E+00)]),("F", [(5.5700000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(6.9507990945E+04, 5.4314897497E-04), (1.0426156880E+04, 4.1990463961E-03), (2.3732334061E+03, 2.1592141679E-02), (6.7156420071E+02, 8.4598850094E-02), (2.1841999790E+02, 2.4757249724E-01), (7.7572249714E+01, 4.7016930228E-01), (2.8888815277E+01, 3.7436370716E-01)]),("S", [(1.2710527185E+02, 2.5182166603E-02), (3.9339582961E+01, 1.0786112456E-01), (7.6740679989E+00, -2.7408821574E-01)]),("S", [(3.8745627630E+00, 1.3213875014E+00), (1.8385832573E+00, 6.8636955368E-01)]),("S", [(5.0229057542E-01, 1.0000000000E+00)]),("S", [(1.7962723420E-01, 1.0000000000E+00)]),("P", [(6.6650423284E+02, 2.3632663836E-03), (1.5764241690E+02, 1.8879300374E-02), (5.0262520978E+01, 8.7206341273E-02), (1.8536078105E+01, 2.5285612970E-01), (7.2940532777E+00, 4.3507154820E-01)]),("P", [(2.9433248995E+00, 3.5026513165E-01)]),("P", [(1.0404970818E+00, 1.0000000000E+00)]),("P", [(3.8456415080E-01, 1.0000000000E+00)]),("P", [(1.3069642732E-01, 1.0000000000E+00)]),("D", [(4.6100000000E+00, 1.0000000000E+00)]),("D", [(1.0110000000E+00, 1.0000000000E+00)]),("D", [(3.3900000000E-01, 1.0000000000E+00)]),("F", [(7.0600000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(7.9111422998E+04, 5.1029325002E-04), (1.1864746009E+04, 3.9463036981E-03), (2.7001642973E+03, 2.0307073910E-02), (7.6395943485E+02, 7.9691825214E-02), (2.4845150561E+02, 2.3420623836E-01), (8.8283581000E+01, 4.4833849481E-01), (3.2948607069E+01, 3.6408167400E-01)]),("S", [(1.4255358000E+02, 2.6387407001E-02), (4.4163688009E+01, 1.1226433999E-01), (8.9524995000E+00, -2.6178922001E-01)]),("S", [(4.5546920941E+00, 1.3002484998E+00), (2.1444079001E+00, 6.7197237009E-01)]),("S", [(6.0708777004E-01, 1.0000000000E+00)]),("S", [(2.1651431999E-01, 1.0000000000E+00)]),("P", [(7.7677541998E+02, 2.2028005003E-03), (1.8380107018E+02, 1.7694180008E-02), (5.8694003175E+01, 8.2431293717E-02), (2.1701591695E+01, 2.4207278863E-01), (8.5821489635E+00, 4.2263558251E-01)]),("P", [(3.4922679161E+00, 3.4151806086E-01)]),("P", [(1.2637426998E+00, 1.0000000000E+00)]),("P", [(4.6607870005E-01, 1.0000000000E+00)]),("P", [(1.5766003000E-01, 1.0000000000E+00)]),("D", [(5.5510000000E+00, 1.0000000000E+00)]),("D", [(1.2350000000E+00, 1.0000000000E+00)]),("D", [(4.1200000000E-01, 1.0000000000E+00)]),("F", [(8.9000000000E-01, 1.0000000000E+00)])],
}





DEF2_TZVPPD = {

    1 : [("S", [(3.4061341000E+01, 6.0251978000E-03), (5.1235746000E+00, 4.5021094000E-02), (1.1646626000E+00, 2.0189726000E-01)]),("S", [(3.2723041000E-01, 1.0000000000E+00)]),("S", [(1.0307241000E-01, 1.0000000000E+00)]),("P", [(1.4070000000E+00, 1.0000000000E+00)]),("P", [(3.8800000000E-01, 1.0000000000E+00)]),("P", [(9.5774129632E-02, 1.0000000000E+00)]),("D", [(1.0570000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(9.8078321616E+01, 7.5803064967E-03), (1.4764404247E+01, 5.4848620937E-02), (3.3185831473E+00, 2.2074382186E-01)]),("S", [(8.7413869551E-01, 1.0000000000E+00)]),("S", [(2.4459897208E-01, 1.0000000000E+00)]),("P", [(3.0440000000E+00, 1.0000000000E+00)]),("P", [(7.5800000000E-01, 1.0000000000E+00)]),("P", [(1.9404104917E-01, 1.0000000000E+00)]),("D", [(1.9650000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.2692628010E+03, 2.0540968826E-04), (9.4031612431E+02, 1.5916554089E-03), (2.1422107528E+02, 8.2869829707E-03), (6.0759840184E+01, 3.3856374249E-02), (1.9915152032E+01, 1.1103225876E-01), (7.3171509797E+00, 2.7449383329E-01)]),("S", [(2.9724674216E+00, 2.3792456411E-01), (1.2639852314E+00, 3.0765411924E-01)]),("S", [(5.1427489953E-01, 1.0000000000E+00)]),("S", [(7.7030885901E-02, 1.0000000000E+00)]),("S", [(2.8938896433E-02, 1.0000000000E+00)]),("P", [(3.3270000000E+00, 1.0000000000E+00)]),("P", [(4.0000000000E-01, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(2.0097234766E-02, 1.0000000000E+00)]),("D", [(1.3190000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(4.7002365626E+03, 2.3584389316E-04), (7.0482845622E+02, 1.8243791019E-03), (1.6043110478E+02, 9.3966148224E-03), (4.5425347336E+01, 3.6908924159E-02), (1.4798334125E+01, 1.0897561281E-01), (5.3512452537E+00, 2.1694284551E-01)]),("S", [(2.1542044819E+00, 4.4695408857E-01), (9.3363744400E-01, 2.0866985771E-01)]),("S", [(1.8791432989E-01, 1.0000000000E+00)]),("S", [(7.4648267947E-02, 1.0000000000E+00)]),("S", [(3.2650484598E-02, 1.0000000000E+00)]),("P", [(3.6316917145E+00, -2.9033998305E-02)]),("P", [(7.1695694366E-01, -1.6877854032E-01), (1.9541932860E-01, -5.1403419628E-01)]),("P", [(6.0515465890E-02, 1.0000000000E+00)]),("P", [(2.7726324608E-02, 1.0000000000E+00)]),("D", [(1.8000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(8.5648660687E+03, 2.2837198155E-04), (1.2841516263E+03, 1.7682576447E-03), (2.9227871604E+02, 9.1407080516E-03), (8.2775469176E+01, 3.6342638989E-02), (2.7017939269E+01, 1.1063458441E-01), (9.8149619660E+00, 2.3367344321E-01)]),("S", [(3.9318559059E+00, 4.1818777978E-01), (1.6595599712E+00, 2.2325473798E-01)]),("S", [(3.5762965239E-01, 1.0000000000E+00)]),("S", [(1.4246277496E-01, 1.0000000000E+00)]),("S", [(6.0560594768E-02, 1.0000000000E+00)]),("S", [(3.1334737069E-02, 1.0000000000E+00)]),("P", [(2.2453875803E+01, 5.0265575179E-03), (5.1045058330E+00, 3.2801738965E-02), (1.4986081344E+00, 1.3151230768E-01), (5.0927831315E-01, 3.3197167769E-01)]),("P", [(1.8147077798E-01, 4.7314319570E-01)]),("P", [(6.4621893904E-02, 2.5802783943E-01)]),("D", [(6.6100000000E-01, 1.0000000000E+00)]),("D", [(1.9900000000E-01, 1.0000000000E+00)]),("D", [(5.9549085820E-02, 1.0000000000E+00)]),("F", [(4.9000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(1.3575349682E+04, 2.2245814352E-04), (2.0352333680E+03, 1.7232738252E-03), (4.6322562359E+02, 8.9255715314E-03), (1.3120019598E+02, 3.5727984502E-02), (4.2853015891E+01, 1.1076259931E-01), (1.5584185766E+01, 2.4295627626E-01)]),("S", [(6.2067138508E+00, 4.1440263448E-01), (2.5764896527E+00, 2.3744968655E-01)]),("S", [(5.7696339419E-01, 1.0000000000E+00)]),("S", [(2.2972831358E-01, 1.0000000000E+00)]),("S", [(9.5164440028E-02, 1.0000000000E+00)]),("S", [(4.8475401370E-02, 1.0000000000E+00)]),("P", [(3.4697232244E+01, 5.3333657805E-03), (7.9582622826E+00, 3.5864109092E-02), (2.3780826883E+00, 1.4215873329E-01), (8.1433208183E-01, 3.4270471845E-01)]),("P", [(2.8887547253E-01, 4.6445822433E-01)]),("P", [(1.0056823671E-01, 2.4955789874E-01)]),("D", [(1.0970000000E+00, 1.0000000000E+00)]),("D", [(3.1800000000E-01, 1.0000000000E+00)]),("D", [(9.0985336424E-02, 1.0000000000E+00)]),("F", [(7.6100000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(1.9730800647E+04, 2.1887984991E-04), (2.9578958745E+03, 1.6960708803E-03), (6.7322133595E+02, 8.7954603538E-03), (1.9068249494E+02, 3.5359382605E-02), (6.2295441898E+01, 1.1095789217E-01), (2.2654161182E+01, 2.4982972552E-01)]),("S", [(8.9791477428E+00, 4.0623896148E-01), (3.6863002370E+00, 2.4338217176E-01)]),("S", [(8.4660076805E-01, 1.0000000000E+00)]),("S", [(3.3647133771E-01, 1.0000000000E+00)]),("S", [(1.3647653675E-01, 1.0000000000E+00)]),("S", [(6.8441605847E-02, 1.0000000000E+00)]),("P", [(4.9200380510E+01, 5.5552416751E-03), (1.1346790537E+01, 3.8052379723E-02), (3.4273972411E+00, 1.4953671029E-01), (1.1785525134E+00, 3.4949305230E-01)]),("P", [(4.1642204972E-01, 4.5843153697E-01)]),("P", [(1.4260826011E-01, 2.4428771672E-01)]),("D", [(1.6540000000E+00, 1.0000000000E+00)]),("D", [(4.6900000000E-01, 1.0000000000E+00)]),("D", [(1.2829642058E-01, 1.0000000000E+00)]),("F", [(1.0930000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(2.7032382631E+04, 2.1726302465E-04), (4.0523871392E+03, 1.6838662199E-03), (9.2232722710E+02, 8.7395616265E-03), (2.6124070989E+02, 3.5239968808E-02), (8.5354641351E+01, 1.1153519115E-01), (3.1035035245E+01, 2.5588953961E-01)]),("S", [(1.2260860728E+01, 3.9768730901E-01), (4.9987076005E+00, 2.4627849430E-01)]),("S", [(1.1703108158E+00, 1.0000000000E+00)]),("S", [(4.6474740994E-01, 1.0000000000E+00)]),("S", [(1.8504536357E-01, 1.0000000000E+00)]),("S", [(7.0288026270E-02, 1.0000000000E+00)]),("P", [(6.3274954801E+01, 6.0685103418E-03), (1.4627049379E+01, 4.1912575824E-02), (4.4501223456E+00, 1.6153841088E-01), (1.5275799647E+00, 3.5706951311E-01)]),("P", [(5.2935117943E-01, 4.4794207502E-01)]),("P", [(1.7478421270E-01, 2.4446069663E-01)]),("P", [(5.1112745706E-02, 1.0000000000E+00)]),("D", [(2.3140000000E+00, 1.0000000000E+00)]),("D", [(6.4500000000E-01, 1.0000000000E+00)]),("D", [(1.4696477366E-01, 1.0000000000E+00)]),("F", [(1.4280000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(3.5479100441E+04, 2.1545014888E-04), (5.3184728983E+03, 1.6700686527E-03), (1.2104810975E+03, 8.6733211476E-03), (3.4285518140E+02, 3.5049933175E-02), (1.1201943181E+02, 1.1165320133E-01), (4.0714740248E+01, 2.5988506647E-01)]),("S", [(1.6039678111E+01, 3.9422966880E-01), (6.5038186740E+00, 2.4998238551E-01)]),("S", [(1.5440477509E+00, 1.0000000000E+00)]),("S", [(6.1223452862E-01, 1.0000000000E+00)]),("S", [(2.4027979698E-01, 1.0000000000E+00)]),("S", [(9.0918446478E-02, 1.0000000000E+00)]),("P", [(8.0233900483E+01, 6.3685999134E-03), (1.8594010743E+01, 4.4303143530E-02), (5.6867902653E+00, 1.6867248708E-01), (1.9511006294E+00, 3.6166346255E-01)]),("P", [(6.6970211298E-01, 4.4202901491E-01)]),("P", [(2.1651300410E-01, 2.4319875730E-01)]),("P", [(5.9613282472E-02, 1.0000000000E+00)]),("D", [(3.1070000000E+00, 1.0000000000E+00)]),("D", [(8.5500000000E-01, 1.0000000000E+00)]),("D", [(1.8608388111E-01, 1.0000000000E+00)]),("F", [(1.9170000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(4.5069464022E+04, 2.1687155182E-04), (6.7559768656E+03, 1.6812736757E-03), (1.5376502864E+03, 8.7356062782E-03), (4.3551697667E+02, 3.5361266922E-02), (1.4228655638E+02, 1.1321521454E-01), (5.1692153804E+01, 2.6654653104E-01)]),("S", [(2.0315870490E+01, 3.9631959951E-01), (8.2021942646E+00, 2.5582811251E-01)]),("S", [(1.9681276278E+00, 1.0000000000E+00)]),("S", [(7.7904756001E-01, 1.0000000000E+00)]),("S", [(3.0229502043E-01, 1.0000000000E+00)]),("S", [(1.1246533684E-01, 1.0000000000E+00)]),("P", [(9.9782996032E+01, 6.5569234163E-03), (2.3176124101E+01, 4.5888009138E-02), (7.1163945872E+00, 1.7331287812E-01), (2.4418711435E+00, 3.6475267512E-01)]),("P", [(8.3389605766E-01, 4.3831075171E-01)]),("P", [(2.6607311301E-01, 2.4160029835E-01)]),("P", [(7.1067552401E-02, 1.0000000000E+00)]),("D", [(4.0140000000E+00, 1.0000000000E+00)]),("D", [(1.0960000000E+00, 1.0000000000E+00)]),("D", [(2.3044808390E-01, 1.0000000000E+00)]),("F", [(2.5440000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(2.6041109927E+04, 6.1806342811E-04), (3.9061268548E+03, 4.7748604414E-03), (8.8897454993E+02, 2.4471684829E-02), (2.5145497961E+02, 9.4755394977E-02), (8.1650143512E+01, 2.6867496920E-01), (2.8904158401E+01, 4.7925475440E-01), (1.0625782932E+01, 3.3248591469E-01)]),("S", [(5.3769410179E+01, 1.9527731872E-02), (1.6308243025E+01, 9.2648010794E-02), (2.3730384125E+00, -3.9938670172E-01)]),("S", [(9.5730772603E-01, 1.6428595391E+00), (4.0806460959E-01, 5.5692596966E-01)]),("S", [(4.9967582329E-02, 1.0000000000E+00)]),("S", [(1.9268616250E-02, 1.0000000000E+00)]),("P", [(1.3807979989E+02, 5.7951891929E-03), (3.2232700393E+01, 4.1620846251E-02), (9.9816075360E+00, 1.6281916885E-01), (3.4822033928E+00, 3.6011784647E-01), (1.2299134620E+00, 4.4858979889E-01)]),("P", [(4.1743959423E-01, 2.3040759559E-01)]),("P", [(9.1000000000E-02, 1.0000000000E+00)]),("P", [(3.0000000000E-02, 1.0000000000E+00)]),("P", [(1.1082810218E-02, 1.0000000000E+00)]),("D", [(2.6090000000E+00, 1.0000000000E+00)]),("D", [(4.3000000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(3.1438349555E+04, 6.0912311326E-04), (4.7155153354E+03, 4.7066196465E-03), (1.0731629247E+03, 2.4135820657E-02), (3.0357238768E+02, 9.3628959834E-02), (9.8626251042E+01, 2.6646742093E-01), (3.4943808417E+01, 4.7890929917E-01), (1.2859785199E+01, 3.3698490286E-01)]),("S", [(6.4876913004E+01, 1.9180889307E-02), (1.9725520777E+01, 9.0913704392E-02), (2.8951804339E+00, -3.9563756125E-01)]),("S", [(1.1960454710E+00, 1.6827603373E+00), (5.4329451156E-01, 5.2141091954E-01)]),("S", [(1.0099104092E-01, 1.0000000000E+00)]),("S", [(3.6865728085E-02, 1.0000000000E+00)]),("P", [(1.7987189612E+02, 5.3799549018E-03), (4.2120069376E+01, 3.9318014098E-02), (1.3120503032E+01, 1.5740129476E-01), (4.6257503609E+00, 3.5919094128E-01), (1.6695211016E+00, 4.5533379310E-01)]),("P", [(5.8551012105E-01, 2.1986432910E-01)]),("P", [(1.8914796195E-01, 1.0000000000E+00)]),("P", [(5.3768755187E-02, 1.0000000000E+00)]),("P", [(2.4576992970E-02, 1.0000000000E+00)]),("D", [(3.4440000000E+00, 1.0000000000E+00)]),("D", [(2.9000000000E-01, 1.0000000000E+00)]),("D", [(7.0000000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(3.7792550772E+04, 5.7047888709E-04), (5.6680682165E+03, 4.4093016538E-03), (1.2898582841E+03, 2.2630967411E-02), (3.6486596028E+02, 8.8025644295E-02), (1.1857631515E+02, 2.5223701612E-01), (4.2024867605E+01, 4.5960547169E-01), (1.5499501629E+01, 3.3277886014E-01)]),("S", [(7.5208026598E+01, 1.9250560190E-02), (2.3031408972E+01, 8.7906743952E-02), (3.6348797649E+00, -3.4246704535E-01)]),("S", [(1.6065049957E+00, 1.5106266058E+00), (7.6103394581E-01, 5.8071016470E-01)]),("S", [(1.6556708849E-01, 1.0000000000E+00)]),("S", [(6.0041577113E-02, 1.0000000000E+00)]),("S", [(2.8536651434E-02, 1.0000000000E+00)]),("P", [(4.5252303192E+02, 2.3110812466E-03), (1.0708195049E+02, 1.8568641823E-02), (3.4131021255E+01, 8.7216237035E-02), (1.2587037428E+01, 2.6902101523E-01), (4.9811919704E+00, 5.2128324272E-01)]),("P", [(2.0070350900E+00, 6.0271687494E-01)]),("P", [(8.0083714374E-01, 1.0000000000E+00)]),("P", [(2.0178927472E-01, 1.0000000000E+00)]),("P", [(5.7895550392E-02, 1.0000000000E+00)]),("D", [(1.5700000000E+00, 1.0000000000E+00)]),("D", [(3.3300000000E-01, 1.0000000000E+00)]),("D", [(1.1100000000E-01, 1.0000000000E+00)]),("D", [(4.1068707243E-02, 1.0000000000E+00)]),("F", [(2.4400000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(4.4773358078E+04, 5.5914765868E-04), (6.7171992104E+03, 4.3206040189E-03), (1.5288960325E+03, 2.2187096460E-02), (4.3254746585E+02, 8.6489249116E-02), (1.4061505226E+02, 2.4939889716E-01), (4.9857636724E+01, 4.6017197366E-01), (1.8434974885E+01, 3.4250236575E-01)]),("S", [(8.6533886111E+01, 2.1300063007E-02), (2.6624606846E+01, 9.4676139318E-02), (4.4953057159E+00, -3.2616264859E-01)]),("S", [(2.1035045710E+00, 1.3980803850E+00), (1.0106094922E+00, 6.3865786699E-01)]),("S", [(2.3701751489E-01, 1.0000000000E+00)]),("S", [(8.5703405362E-02, 1.0000000000E+00)]),("S", [(3.7000771280E-02, 1.0000000000E+00)]),("P", [(3.9447503628E+02, 2.6285693959E-03), (9.3137683104E+01, 2.0556257749E-02), (2.9519608742E+01, 9.2070262801E-02), (1.0781663791E+01, 2.5565889739E-01), (4.1626574778E+00, 4.2111707185E-01)]),("P", [(1.6247972989E+00, 3.4401746318E-01)]),("P", [(5.4306660493E-01, 1.0000000000E+00)]),("P", [(2.0582073956E-01, 1.0000000000E+00)]),("P", [(7.0053487306E-02, 1.0000000000E+00)]),("D", [(2.3030000000E+00, 1.0000000000E+00)]),("D", [(4.7600000000E-01, 1.0000000000E+00)]),("D", [(1.6000000000E-01, 1.0000000000E+00)]),("D", [(5.5036090333E-02, 1.0000000000E+00)]),("F", [(3.3600000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(5.2426999233E+04, 5.5207164100E-04), (7.8632660552E+03, 4.2678595308E-03), (1.7895227333E+03, 2.1931529186E-02), (5.0627300165E+02, 8.5667168373E-02), (1.6460698546E+02, 2.4840686605E-01), (5.8391918722E+01, 4.6336753971E-01), (2.1643663201E+01, 3.5350558156E-01)]),("S", [(9.9013837620E+01, 2.1895679958E-02), (3.0550439817E+01, 9.5650470295E-02), (5.4537087661E+00, -2.9454270186E-01)]),("S", [(2.6503362563E+00, 1.3294381200E+00), (1.2726688867E+00, 6.6109396473E-01)]),("S", [(3.1645005203E-01, 1.0000000000E+00)]),("S", [(1.1417466938E-01, 1.0000000000E+00)]),("S", [(4.8831315225E-02, 1.0000000000E+00)]),("P", [(4.7227219248E+02, 2.5710623052E-03), (1.1158882756E+02, 2.0250297999E-02), (3.5445936418E+01, 9.1580716787E-02), (1.2990776875E+01, 2.5749454014E-01), (5.0486221658E+00, 4.2862899758E-01)]),("P", [(1.9934049566E+00, 3.4359817849E-01)]),("P", [(6.6527284430E-01, 1.0000000000E+00)]),("P", [(2.5516832128E-01, 1.0000000000E+00)]),("P", [(9.0357762251E-02, 1.0000000000E+00)]),("D", [(3.1200000000E+00, 1.0000000000E+00)]),("D", [(6.4800000000E-01, 1.0000000000E+00)]),("D", [(2.1800000000E-01, 1.0000000000E+00)]),("D", [(7.3569545334E-02, 1.0000000000E+00)]),("F", [(4.5200000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(6.0700928104E+04, 5.4695944225E-04), (9.1026106854E+03, 4.2297224557E-03), (2.0714166009E+03, 2.1747824159E-02), (5.8602476821E+02, 8.5100053589E-02), (1.9055395021E+02, 2.4799128459E-01), (6.7630384260E+01, 4.6703640406E-01), (2.5127306905E+01, 3.6434587550E-01)]),("S", [(1.1257463010E+02, 2.1670040240E-02), (3.4795554217E+01, 9.3602301760E-02), (6.5115556215E+00, -2.6068001422E-01)]),("S", [(3.2399032261E+00, 1.2842089435E+00), (1.5477160881E+00, 6.6036416584E-01)]),("S", [(4.0541030112E-01, 1.0000000000E+00)]),("S", [(1.4550651059E-01, 1.0000000000E+00)]),("S", [(5.0178839540E-02, 1.0000000000E+00)]),("P", [(5.6436716027E+02, 2.4796796317E-03), (1.3342624379E+02, 1.9677930250E-02), (4.2468271189E+01, 8.9980008258E-02), (1.5616527580E+01, 2.5705880575E-01), (6.1093988469E+00, 4.3515167292E-01)]),("P", [(2.4404160198E+00, 3.4883240595E-01)]),("P", [(8.3882201296E-01, 1.0000000000E+00)]),("P", [(3.1288746900E-01, 1.0000000000E+00)]),("P", [(1.0770109004E-01, 1.0000000000E+00)]),("P", [(3.0359833649E-02, 1.0000000000E+00)]),("D", [(3.7560000000E+00, 1.0000000000E+00)]),("D", [(8.1200000000E-01, 1.0000000000E+00)]),("D", [(2.7300000000E-01, 1.0000000000E+00)]),("D", [(8.1547024970E-02, 1.0000000000E+00)]),("F", [(5.5700000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(6.9507990945E+04, 5.4314897497E-04), (1.0426156880E+04, 4.1990463961E-03), (2.3732334061E+03, 2.1592141679E-02), (6.7156420071E+02, 8.4598850094E-02), (2.1841999790E+02, 2.4757249724E-01), (7.7572249714E+01, 4.7016930228E-01), (2.8888815277E+01, 3.7436370716E-01)]),("S", [(1.2710527185E+02, 2.5182166603E-02), (3.9339582961E+01, 1.0786112456E-01), (7.6740679989E+00, -2.7408821574E-01)]),("S", [(3.8745627630E+00, 1.3213875014E+00), (1.8385832573E+00, 6.8636955368E-01)]),("S", [(5.0229057542E-01, 1.0000000000E+00)]),("S", [(1.7962723420E-01, 1.0000000000E+00)]),("S", [(6.1133095854E-02, 1.0000000000E+00)]),("P", [(6.6650423284E+02, 2.3632663836E-03), (1.5764241690E+02, 1.8879300374E-02), (5.0262520978E+01, 8.7206341273E-02), (1.8536078105E+01, 2.5285612970E-01), (7.2940532777E+00, 4.3507154820E-01)]),("P", [(2.9433248995E+00, 3.5026513165E-01)]),("P", [(1.0404970818E+00, 1.0000000000E+00)]),("P", [(3.8456415080E-01, 1.0000000000E+00)]),("P", [(1.3069642732E-01, 1.0000000000E+00)]),("P", [(3.5381580722E-02, 1.0000000000E+00)]),("D", [(4.6100000000E+00, 1.0000000000E+00)]),("D", [(1.0110000000E+00, 1.0000000000E+00)]),("D", [(3.3900000000E-01, 1.0000000000E+00)]),("D", [(9.9601358790E-02, 1.0000000000E+00)]),("F", [(7.0600000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(7.9111422998E+04, 5.1029325002E-04), (1.1864746009E+04, 3.9463036981E-03), (2.7001642973E+03, 2.0307073910E-02), (7.6395943485E+02, 7.9691825214E-02), (2.4845150561E+02, 2.3420623836E-01), (8.8283581000E+01, 4.4833849481E-01), (3.2948607069E+01, 3.6408167400E-01)]),("S", [(1.4255358000E+02, 2.6387407001E-02), (4.4163688009E+01, 1.1226433999E-01), (8.9524995000E+00, -2.6178922001E-01)]),("S", [(4.5546920941E+00, 1.3002484998E+00), (2.1444079001E+00, 6.7197237009E-01)]),("S", [(6.0708777004E-01, 1.0000000000E+00)]),("S", [(2.1651431999E-01, 1.0000000000E+00)]),("S", [(7.3732186754E-02, 1.0000000000E+00)]),("P", [(7.7677541998E+02, 2.2028005003E-03), (1.8380107018E+02, 1.7694180008E-02), (5.8694003175E+01, 8.2431293717E-02), (2.1701591695E+01, 2.4207278863E-01), (8.5821489635E+00, 4.2263558251E-01)]),("P", [(3.4922679161E+00, 3.4151806086E-01)]),("P", [(1.2637426998E+00, 1.0000000000E+00)]),("P", [(4.6607870005E-01, 1.0000000000E+00)]),("P", [(1.5766003000E-01, 1.0000000000E+00)]),("P", [(4.1779128551E-02, 1.0000000000E+00)]),("D", [(5.5510000000E+00, 1.0000000000E+00)]),("D", [(1.2350000000E+00, 1.0000000000E+00)]),("D", [(4.1200000000E-01, 1.0000000000E+00)]),("D", [(1.2039311859E-01, 1.0000000000E+00)]),("F", [(8.9000000000E-01, 1.0000000000E+00)])],
}






DEF2_QZVP = {

    1 : [("S", [(1.9069169000E+02, 7.0815167000E-04), (2.8605532000E+01, 5.4678827000E-03), (6.5095943000E+00, 2.7966605000E-02), (1.8412455000E+00, 1.0764538000E-01)]),("S", [(5.9853725000E-01, 1.0000000000E+00)]),("S", [(2.1397624000E-01, 1.0000000000E+00)]),("S", [(8.0316286000E-02, 1.0000000000E+00)]),("P", [(2.2920000000E+00, 1.0000000000E+00)]),("P", [(8.3800000000E-01, 1.0000000000E+00)]),("P", [(2.9200000000E-01, 1.0000000000E+00)]),("D", [(2.0620000000E+00, 1.0000000000E+00)]),("D", [(6.6200000000E-01, 1.0000000000E+00)]),("F", [(1.3970000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(1.1446470809E+03, 3.5861578618E-04), (1.7164596667E+02, 2.7725434466E-03), (3.9066056254E+01, 1.4241892216E-02), (1.1051401989E+01, 5.5457352277E-02), (3.5725574473E+00, 1.6170511810E-01)]),("S", [(1.2429415962E+00, 3.3177883833E-01)]),("S", [(4.4807668730E-01, 4.2014033068E-01)]),("S", [(1.6411579128E-01, 1.8652273014E-01)]),("P", [(5.9940000000E+00, 1.0000000000E+00)]),("P", [(1.7450000000E+00, 1.0000000000E+00)]),("P", [(5.6000000000E-01, 1.0000000000E+00)]),("D", [(4.2990000000E+00, 1.0000000000E+00)]),("D", [(1.2230000000E+00, 1.0000000000E+00)]),("F", [(2.6800000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(1.4853977085E+04, 4.2711388815E-05), (2.2252236477E+03, 3.3235310800E-04), (5.0488739008E+02, 1.7518436649E-03), (1.4245847548E+02, 7.3477995850E-03), (4.6315599580E+01, 2.5899837683E-02), (1.6655335474E+01, 7.6670682431E-02), (6.4331186199E+00, 1.8276075765E-01), (2.6027043858E+00, 3.2655434038E-01), (1.0897245405E+00, 3.7000429828E-01)]),("S", [(4.4236595971E+00, 1.1120987921E-01), (1.2356394990E+00, 7.9987335862E-01)]),("S", [(4.6067470602E-01, 1.0000000000E+00)]),("S", [(9.6617166955E-02, 1.0000000000E+00)]),("S", [(4.5915452296E-02, 1.0000000000E+00)]),("S", [(2.1140048730E-02, 1.0000000000E+00)]),("P", [(3.3270000000E+00, 1.0000000000E+00)]),("P", [(3.2605109206E+00, 8.6504749023E-03), (6.5003043115E-01, 4.7614123736E-02), (1.6941671073E-01, 2.1001138000E-01)]),("P", [(5.5732343767E-02, 1.0000000000E+00)]),("P", [(2.0489959241E-02, 1.0000000000E+00)]),("D", [(2.3000000000E-01, 1.0000000000E+00)]),("D", [(7.5700000000E-02, 1.0000000000E+00)]),("F", [(1.3500000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(2.9646704407E+04, 3.7943320869E-05), (4.4287614354E+03, 2.9605352634E-04), (1.0054701332E+03, 1.5572477356E-03), (2.8415339578E+02, 6.5323195769E-03), (9.2504356461E+01, 2.3116861005E-02), (3.3311016058E+01, 6.9458515249E-02), (1.2911553314E+01, 1.7014371331E-01), (5.2665497361E+00, 3.1682711374E-01)]),("S", [(2.2995386380E+01, 8.1791122747E-02), (6.6236354176E+00, 8.0000038023E-01)]),("S", [(2.2289595003E+00, 1.0000000000E+00)]),("S", [(9.5302433451E-01, 1.0000000000E+00)]),("S", [(2.4651602436E-01, 1.0000000000E+00)]),("S", [(1.0153895315E-01, 1.0000000000E+00)]),("S", [(4.1551374604E-02, 1.0000000000E+00)]),("P", [(6.4000000000E+00, 1.0000000000E+00)]),("P", [(1.4099789445E+01, 3.8508871145E-03), (3.1803188482E+00, 2.4152380771E-02), (9.0489220496E-01, 9.7926925156E-02), (3.0411585508E-01, 2.9470380030E-01)]),("P", [(1.1302606657E-01, 1.0000000000E+00)]),("P", [(4.2831733691E-02, 1.0000000000E+00)]),("D", [(3.6000000000E-01, 1.0000000000E+00)]),("D", [(9.0000000000E-02, 1.0000000000E+00)]),("F", [(2.9000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(4.6447667056E+04, 3.8388681014E-05), (6.9576889042E+03, 2.9836770197E-04), (1.5834428403E+03, 1.5645478806E-03), (4.4846601009E+02, 6.5476770742E-03), (1.4628639262E+02, 2.3139008388E-02), (5.2784386084E+01, 6.9615799357E-02), (2.0519396170E+01, 1.7119636876E-01), (8.4185659258E+00, 3.1913191837E-01)]),("S", [(3.6510018312E+01, 7.8990617244E-02), (1.0541854005E+01, 7.8926384265E-01)]),("S", [(3.6004091387E+00, 1.0000000000E+00)]),("S", [(1.5617023749E+00, 1.0000000000E+00)]),("S", [(4.4997370775E-01, 1.0000000000E+00)]),("S", [(1.8075230237E-01, 1.0000000000E+00)]),("S", [(7.1596696319E-02, 1.0000000000E+00)]),("P", [(7.2240462760E+01, 8.6539170411E-04), (1.6807707208E+01, 6.8762067365E-03), (5.2259410747E+00, 3.0976687148E-02), (1.8508350671E+00, 1.0432357913E-01), (7.2206782663E-01, 2.6164137577E-01)]),("P", [(2.9491018056E-01, 1.0000000000E+00)]),("P", [(1.2201141839E-01, 1.0000000000E+00)]),("P", [(4.9865393625E-02, 1.0000000000E+00)]),("D", [(1.1100000000E+00, 1.0000000000E+00)]),("D", [(4.0200000000E-01, 1.0000000000E+00)]),("D", [(1.4500000000E-01, 1.0000000000E+00)]),("F", [(8.8200000000E-01, 1.0000000000E+00)]),("F", [(3.1100000000E-01, 1.0000000000E+00)]),("G", [(6.7300000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(6.7025071029E+04, 3.8736308501E-05), (1.0039986538E+04, 3.0107917575E-04), (2.2849316911E+03, 1.5787918095E-03), (6.4714122130E+02, 6.6087087195E-03), (2.1109472335E+02, 2.3367123250E-02), (7.6177643862E+01, 7.0420716898E-02), (2.9633839163E+01, 1.7360344953E-01), (1.2187785081E+01, 3.2292305648E-01)]),("S", [(5.3026006299E+01, 7.4897404492E-02), (1.5258502776E+01, 7.6136220983E-01)]),("S", [(5.2403957464E+00, 1.0000000000E+00)]),("S", [(2.2905022379E+00, 1.0000000000E+00)]),("S", [(6.9673283006E-01, 1.0000000000E+00)]),("S", [(2.7599337363E-01, 1.0000000000E+00)]),("S", [(1.0739884389E-01, 1.0000000000E+00)]),("P", [(1.0512555082E+02, 8.4647553844E-04), (2.4884461066E+01, 6.6274038534E-03), (7.8637230826E+00, 3.0120390419E-02), (2.8407001835E+00, 9.9951435476E-02), (1.1227137335E+00, 2.3826299282E-01)]),("P", [(4.6050725555E-01, 1.0000000000E+00)]),("P", [(1.8937530913E-01, 1.0000000000E+00)]),("P", [(7.5983791611E-02, 1.0000000000E+00)]),("D", [(1.8480000000E+00, 1.0000000000E+00)]),("D", [(6.4900000000E-01, 1.0000000000E+00)]),("D", [(2.2800000000E-01, 1.0000000000E+00)]),("F", [(1.4190000000E+00, 1.0000000000E+00)]),("F", [(4.8500000000E-01, 1.0000000000E+00)]),("G", [(1.0110000000E+00, 1.0000000000E+00)])],
    7 : [("S", [(9.0726889210E+04, 3.9257887368E-05), (1.3590528801E+04, 3.0513316455E-04), (3.0929883781E+03, 1.6000560446E-03), (8.7599876362E+02, 6.6982937306E-03), (2.8574469982E+02, 2.3690078765E-02), (1.0311913417E+02, 7.1455405268E-02), (4.0128556777E+01, 1.7632774876E-01), (1.6528095704E+01, 3.2677592815E-01)]),("S", [(6.9390960983E+01, 8.0052094386E-02), (2.0428200596E+01, 7.8268063538E-01)]),("S", [(7.1292587972E+00, 1.0000000000E+00)]),("S", [(3.1324304893E+00, 1.0000000000E+00)]),("S", [(9.8755778723E-01, 1.0000000000E+00)]),("S", [(3.8765721307E-01, 1.0000000000E+00)]),("S", [(1.4909883075E-01, 1.0000000000E+00)]),("P", [(1.5005742670E+02, -8.6216165986E-04), (3.5491599483E+01, -6.8571273236E-03), (1.1247864223E+01, -3.1795688855E-02), (4.0900305195E+00, -1.0537396822E-01), (1.6220573146E+00, -2.4519708041E-01)]),("P", [(6.6442261530E-01, 1.0000000000E+00)]),("P", [(2.7099770070E-01, 1.0000000000E+00)]),("P", [(1.0688749984E-01, 1.0000000000E+00)]),("D", [(2.8370000000E+00, 1.0000000000E+00)]),("D", [(9.6800000000E-01, 1.0000000000E+00)]),("D", [(3.3500000000E-01, 1.0000000000E+00)]),("F", [(2.0270000000E+00, 1.0000000000E+00)]),("F", [(6.8500000000E-01, 1.0000000000E+00)]),("G", [(1.4270000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(1.1650646908E+05, 4.0383857939E-05), (1.7504349724E+04, 3.1255139004E-04), (3.9934513230E+03, 1.6341473495E-03), (1.1330063186E+03, 6.8283224757E-03), (3.6999569594E+02, 2.4124410221E-02), (1.3362074349E+02, 7.2730206154E-02), (5.2035643649E+01, 1.7934429892E-01), (2.1461939313E+01, 3.3059588895E-01)]),("S", [(8.9835051252E+01, 9.6468652996E-02), (2.6428010844E+01, 9.4117481120E-01)]),("S", [(9.2822824649E+00, 1.0000000000E+00)]),("S", [(4.0947728533E+00, 1.0000000000E+00)]),("S", [(1.3255349078E+00, 1.0000000000E+00)]),("S", [(5.1877230787E-01, 1.0000000000E+00)]),("S", [(1.9772676454E-01, 1.0000000000E+00)]),("P", [(1.9115255810E+02, 2.5115697705E-03), (4.5233356739E+01, 2.0039240864E-02), (1.4353465922E+01, 9.3609064762E-02), (5.2422371832E+00, 3.0618127124E-01), (2.0792418599E+00, 6.7810501439E-01)]),("P", [(8.4282371424E-01, 1.0000000000E+00)]),("P", [(3.3617694891E-01, 1.0000000000E+00)]),("P", [(1.2863997974E-01, 1.0000000000E+00)]),("D", [(3.7750000000E+00, 1.0000000000E+00)]),("D", [(1.3000000000E+00, 1.0000000000E+00)]),("D", [(4.4400000000E-01, 1.0000000000E+00)]),("F", [(2.6660000000E+00, 1.0000000000E+00)]),("F", [(8.5900000000E-01, 1.0000000000E+00)]),("G", [(1.8460000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.3253597345E+05, 4.7387482743E-05), (1.9758112588E+04, 3.7070120897E-04), (4.4851996947E+03, 1.9450784713E-03), (1.2738151020E+03, 8.0573291994E-03), (4.1893831236E+02, 2.7992880781E-02), (1.5255721985E+02, 8.2735120175E-02), (5.9821524823E+01, 1.9854169012E-01), (2.4819076932E+01, 3.4860632233E-01)]),("S", [(1.0074446673E+02, 1.0505068816E-01), (3.0103728290E+01, 9.4068472434E-01)]),("S", [(1.0814283272E+01, 1.0000000000E+00)]),("S", [(4.8172886770E+00, 1.0000000000E+00)]),("S", [(1.6559334213E+00, 1.0000000000E+00)]),("S", [(6.4893519582E-01, 1.0000000000E+00)]),("S", [(2.4778104545E-01, 1.0000000000E+00)]),("P", [(2.4096654114E+02, 3.0389933451E-03), (5.7020699781E+01, 2.4357738582E-02), (1.8126952120E+01, 1.1442925768E-01), (6.6457404621E+00, 3.7064659853E-01), (2.6375722892E+00, 7.9791551766E-01)]),("P", [(1.0638217200E+00, 1.0000000000E+00)]),("P", [(4.1932562750E-01, 1.0000000000E+00)]),("P", [(1.5747588299E-01, 1.0000000000E+00)]),("D", [(5.0140000000E+00, 1.0000000000E+00)]),("D", [(1.7250000000E+00, 1.0000000000E+00)]),("D", [(5.8600000000E-01, 1.0000000000E+00)]),("F", [(3.5620000000E+00, 1.0000000000E+00)]),("F", [(1.1480000000E+00, 1.0000000000E+00)]),("G", [(2.3760000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.6067627955E+05, 4.7387482743E-05), (2.3953195039E+04, 3.7070120897E-04), (5.4375063711E+03, 1.9450784713E-03), (1.5442741024E+03, 8.0573291994E-03), (5.0788814269E+02, 2.7992880781E-02), (1.8494847750E+02, 8.2735120175E-02), (7.2522952039E+01, 1.9854169012E-01), (3.0088713575E+01, 3.4860632233E-01)]),("S", [(1.2068434940E+02, 1.0400597014E-01), (3.6074294857E+01, 9.1999988986E-01)]),("S", [(1.3052091210E+01, 1.0000000000E+00)]),("S", [(5.7974783353E+00, 1.0000000000E+00)]),("S", [(2.0478626895E+00, 1.0000000000E+00)]),("S", [(8.0214571994E-01, 1.0000000000E+00)]),("S", [(3.0548672205E-01, 1.0000000000E+00)]),("P", [(4.9843397466E+02, 4.3118261050E-04), (1.1814109217E+02, 3.6206397703E-03), (3.8032529735E+01, 1.8245344081E-02), (1.4183731950E+01, 6.5133035572E-02), (5.7864190721E+00, 1.6708915089E-01), (2.4599622160E+00, 2.9268019613E-01)]),("P", [(1.0421999498E+00, 1.0000000000E+00)]),("P", [(4.2987748916E-01, 1.0000000000E+00)]),("P", [(1.6889708453E-01, 1.0000000000E+00)]),("D", [(6.4710000000E+00, 1.0000000000E+00)]),("D", [(2.2130000000E+00, 1.0000000000E+00)]),("D", [(7.4700000000E-01, 1.0000000000E+00)]),("F", [(4.6570000000E+00, 1.0000000000E+00)]),("F", [(1.5240000000E+00, 1.0000000000E+00)]),("G", [(2.9830000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.7985220081E+05, 2.0671384468E-05), (5.6886006378E+04, 1.6070466617E-04), (1.2942701838E+04, 8.4462905848E-04), (3.6643017904E+03, 3.5519026029E-03), (1.1947417499E+03, 1.2754034468E-02), (4.3098192917E+02, 3.9895462742E-02), (1.6783169424E+02, 1.0720154498E-01), (6.9306669040E+01, 2.3339516913E-01), (2.9951170886E+01, 3.6333077287E-01), (1.3380791097E+01, 3.0544770974E-01)]),("S", [(1.2174011283E+02, 3.6142427284E-02), (3.7044143387E+01, 2.8820961687E-01), (1.3995422624E+01, 7.9337384869E-01)]),("S", [(5.9827797428E+00, 1.0000000000E+00)]),("S", [(2.4830455118E+00, 1.0000000000E+00)]),("S", [(1.0452506187E+00, 1.0000000000E+00)]),("S", [(4.3875640383E-01, 1.0000000000E+00)]),("S", [(6.5595633185E-02, 1.0000000000E+00)]),("S", [(3.0561925072E-02, 1.0000000000E+00)]),("S", [(1.5509064018E-02, 1.0000000000E+00)]),("P", [(6.9077627017E+02, 3.7478518415E-04), (1.6382806121E+02, 3.1775441030E-03), (5.2876460769E+01, 1.6333581338E-02), (1.9812270493E+01, 5.9754902585E-02), (8.1320378784E+00, 1.5879328812E-01), (3.4969068377E+00, 2.9049363260E-01), (1.5117244146E+00, 3.6368131139E-01), (6.4479294912E-01, 2.8195867334E-01)]),("P", [(2.6145823312E-01, 1.0000000000E+00)]),("P", [(1.1704726116E-01, 1.0000000000E+00)]),("P", [(4.0494747666E-02, 1.0000000000E+00)]),("P", [(1.5666707355E-02, 1.0000000000E+00)]),("D", [(2.9000000000E+00, 1.0000000000E+00)]),("D", [(2.3000000000E-01, 1.0000000000E+00)]),("D", [(7.5700000000E-02, 1.0000000000E+00)]),("F", [(1.3500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(6.0596778753E+05, 1.4430176617E-05), (9.0569094692E+04, 1.1249871807E-04), (2.0574252844E+04, 5.9264972450E-04), (5.8186284865E+03, 2.4988770424E-03), (1.8956296075E+03, 9.0230775141E-03), (6.8345941021E+02, 2.8579919806E-02), (2.6618219762E+02, 7.9064453923E-02), (1.1011220010E+02, 1.8269337814E-01), (4.7777041234E+01, 3.2157193967E-01), (2.1542166149E+01, 3.5028259389E-01)]),("S", [(1.7412136378E+02, 2.2931111278E-02), (5.3484972498E+01, 1.9151777552E-01), (2.0500213307E+01, 6.1155711055E-01)]),("S", [(9.8056826922E+00, 1.0000000000E+00)]),("S", [(3.5402989348E+00, 1.0000000000E+00)]),("S", [(1.5450850046E+00, 1.0000000000E+00)]),("S", [(6.6540195432E-01, 1.0000000000E+00)]),("S", [(1.4355439071E-01, 1.0000000000E+00)]),("S", [(6.6624673468E-02, 1.0000000000E+00)]),("S", [(2.9772648266E-02, 1.0000000000E+00)]),("P", [(8.9320460829E+02, 3.4958267428E-04), (2.1178258286E+02, 2.9811890111E-03), (6.8443200537E+01, 1.5517845758E-02), (2.5727265349E+01, 5.7578659557E-02), (1.0606634281E+01, 1.5610307853E-01), (4.5934126484E+00, 2.9230912580E-01), (2.0100469807E+00, 3.7219024072E-01), (8.7384841536E-01, 2.7578088891E-01)]),("P", [(3.5615076666E-01, 1.0000000000E+00)]),("P", [(1.8995954356E-01, 1.0000000000E+00)]),("P", [(7.4580135669E-02, 1.0000000000E+00)]),("P", [(2.9221641309E-02, 1.0000000000E+00)]),("D", [(3.8100000000E+00, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)]),("D", [(5.2000000000E-02, 1.0000000000E+00)]),("F", [(1.6000000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(7.5455078265E+05, 1.3421335886E-05), (1.1299938922E+05, 1.0433210786E-04), (2.5715831759E+04, 5.4841772080E-04), (7.2836030283E+03, 2.3089081391E-03), (2.3760008796E+03, 8.3309974593E-03), (8.5765468087E+02, 2.6417963653E-02), (3.3438922598E+02, 7.3443646426E-02), (1.3848504731E+02, 1.7184039139E-01), (6.0150368808E+01, 3.1041980030E-01), (2.7127610860E+01, 3.5669190596E-01)]),("S", [(2.2536500065E+02, 2.1522039783E-02), (6.9341968124E+01, 1.8531780493E-01), (2.6619335712E+01, 6.3533181245E-01)]),("S", [(1.2349420671E+01, 1.0000000000E+00)]),("S", [(4.5878785994E+00, 1.0000000000E+00)]),("S", [(2.0571338103E+00, 1.0000000000E+00)]),("S", [(9.0864549395E-01, 1.0000000000E+00)]),("S", [(2.4226988038E-01, 1.0000000000E+00)]),("S", [(1.1248136869E-01, 1.0000000000E+00)]),("S", [(4.8156492817E-02, 1.0000000000E+00)]),("P", [(1.4896119522E+03, 2.0177122443E-04), (3.5301399267E+02, 1.7508109203E-03), (1.1440764069E+02, 9.4424704664E-03), (4.3312186111E+01, 3.6868004006E-02), (1.8027322216E+01, 1.0892874007E-01), (7.9675432403E+00, 2.3265901004E-01), (3.6090399541E+00, 3.4643587076E-01), (1.6456081630E+00, 3.3440809866E-01)]),("P", [(3.4731187489E+01, 1.7823073020E-02), (1.2553083630E+00, -5.9991263926E-01)]),("P", [(7.3635534282E-01, 1.0000000000E+00)]),("P", [(2.4686465559E-01, 1.0000000000E+00)]),("P", [(9.4821971828E-02, 1.0000000000E+00)]),("P", [(3.6214165654E-02, 1.0000000000E+00)]),("D", [(1.9700000000E+00, 1.0000000000E+00)]),("D", [(4.3700000000E-01, 1.0000000000E+00)]),("D", [(1.9500000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-02, 1.0000000000E+00)]),("F", [(4.0100000000E-01, 1.0000000000E+00)]),("F", [(1.5400000000E-01, 1.0000000000E+00)]),("G", [(3.5700000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(9.1807069565E+05, 1.2658787192E-05), (1.3748525386E+05, 9.8408387461E-05), (3.1287772714E+04, 5.1731398244E-04), (8.8616105697E+03, 2.1785441933E-03), (2.8906943156E+03, 7.8656853716E-03), (1.0434063979E+03, 2.4987553674E-02), (4.0680160276E+02, 6.9761818844E-02), (1.6848360207E+02, 1.6473362733E-01), (7.3185628823E+01, 3.0285602647E-01), (3.2998485420E+01, 3.6007260439E-01)]),("S", [(2.7878751325E+02, 2.0185463814E-02), (8.5910228722E+01, 1.7720406747E-01), (3.2992604031E+01, 6.3587860906E-01)]),("S", [(1.5033693254E+01, 1.0000000000E+00)]),("S", [(5.7257514773E+00, 1.0000000000E+00)]),("S", [(2.6146521029E+00, 1.0000000000E+00)]),("S", [(1.1757833152E+00, 1.0000000000E+00)]),("S", [(3.5432330868E-01, 1.0000000000E+00)]),("S", [(1.6248335806E-01, 1.0000000000E+00)]),("S", [(6.8332457181E-02, 1.0000000000E+00)]),("P", [(1.7758850516E+03, 2.0186971126E-04), (4.2083786849E+02, 1.7544314655E-03), (1.3642290243E+02, 9.5039532290E-03), (5.1700991737E+01, 3.7325741590E-02), (2.1559456002E+01, 1.1085396329E-01), (9.5559200095E+00, 2.3756407046E-01), (4.3529819473E+00, 3.5295783750E-01), (2.0096964381E+00, 3.2885931947E-01)]),("P", [(4.6418153780E+01, 2.5858819154E-02), (1.9076417796E+00, -5.9874865899E-01)]),("P", [(9.2337221587E-01, 1.0000000000E+00)]),("P", [(3.4515720370E-01, 1.0000000000E+00)]),("P", [(1.3656834761E-01, 1.0000000000E+00)]),("P", [(5.2987060586E-02, 1.0000000000E+00)]),("D", [(2.6450000000E+00, 1.0000000000E+00)]),("D", [(6.0800000000E-01, 1.0000000000E+00)]),("D", [(2.7200000000E-01, 1.0000000000E+00)]),("D", [(1.1300000000E-01, 1.0000000000E+00)]),("F", [(5.4100000000E-01, 1.0000000000E+00)]),("F", [(2.1200000000E-01, 1.0000000000E+00)]),("G", [(4.6100000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.0905617138E+06, 1.2142449664E-05), (1.6331639461E+05, 9.4395292614E-05), (3.7166607451E+04, 4.9622262177E-04), (1.0526880945E+04, 2.0900040747E-03), (3.4339976028E+03, 7.5489230025E-03), (1.2395360480E+03, 2.4010423937E-02), (4.8327456199E+02, 6.7231473697E-02), (2.0016911586E+02, 1.5978669881E-01), (8.6960394829E+01, 2.9735906782E-01), (3.9211283369E+01, 3.6187171850E-01)]),("S", [(3.3675883662E+02, 1.9154721050E-02), (1.0372179793E+02, 1.7134079100E-01), (3.9771861240E+01, 6.3689655985E-01)]),("S", [(1.7888612952E+01, 1.0000000000E+00)]),("S", [(6.9644556879E+00, 1.0000000000E+00)]),("S", [(3.2198092087E+00, 1.0000000000E+00)]),("S", [(1.4669943979E+00, 1.0000000000E+00)]),("S", [(4.7765437532E-01, 1.0000000000E+00)]),("S", [(2.1637789241E-01, 1.0000000000E+00)]),("S", [(9.0235894336E-02, 1.0000000000E+00)]),("P", [(2.0196711374E+03, 2.1359172406E-04), (4.7860125090E+02, 1.8568771399E-03), (1.5514942504E+02, 1.0070690115E-02), (5.8816356575E+01, 3.9605153679E-02), (2.4544512785E+01, 1.1736067844E-01), (1.0883571061E+01, 2.4950540642E-01), (4.9624791285E+00, 3.6421287984E-01), (2.3002912343E+00, 3.1764127123E-01)]),("P", [(5.9371345016E+01, 3.9432918004E-01), (3.0694590986E+00, -6.3522960431E+00)]),("P", [(1.0634401739E+00, 1.0000000000E+00)]),("P", [(4.5022152161E-01, 1.0000000000E+00)]),("P", [(1.8267271344E-01, 1.0000000000E+00)]),("P", [(7.1610333771E-02, 1.0000000000E+00)]),("D", [(3.3430000000E+00, 1.0000000000E+00)]),("D", [(8.0700000000E-01, 1.0000000000E+00)]),("D", [(3.6500000000E-01, 1.0000000000E+00)]),("D", [(1.5400000000E-01, 1.0000000000E+00)]),("F", [(7.0300000000E-01, 1.0000000000E+00)]),("F", [(2.8000000000E-01, 1.0000000000E+00)]),("G", [(5.9700000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(1.2734109023E+06, 1.1767088246E-05), (1.9069783007E+05, 9.1478610166E-05), (4.3397885330E+04, 4.8090078640E-04), (1.2291809677E+04, 2.0257193592E-03), (4.0097420824E+03, 7.3190096406E-03), (1.4473531030E+03, 2.3300499900E-02), (5.6430102913E+02, 6.5386213610E-02), (2.3374506243E+02, 1.5614449910E-01), (1.0156402814E+02, 2.9318563787E-01), (4.5805907187E+01, 3.6287914289E-01)]),("S", [(3.9427281503E+02, 1.8753305081E-02), (1.2172249591E+02, 1.6870726663E-01), (4.6754125963E+01, 6.3806830653E-01)]),("S", [(2.0923008254E+01, 1.0000000000E+00)]),("S", [(8.2685567800E+00, 1.0000000000E+00)]),("S", [(3.8629345671E+00, 1.0000000000E+00)]),("S", [(1.7794684781E+00, 1.0000000000E+00)]),("S", [(6.1064260103E-01, 1.0000000000E+00)]),("S", [(2.7412269445E-01, 1.0000000000E+00)]),("S", [(1.1325939107E-01, 1.0000000000E+00)]),("P", [(2.1898930459E+03, 2.3912552864E-04), (5.1894596592E+02, 2.0772032158E-03), (1.6819560151E+02, 1.1242420571E-02), (6.3745282788E+01, 4.4069933941E-02), (2.6597033077E+01, 1.2918778608E-01), (1.1774251449E+01, 2.6910820167E-01), (5.3534379024E+00, 3.7855928620E-01), (2.4701911802E+00, 2.9692134655E-01)]),("P", [(8.2120288349E+01, -3.9420318847E-02), (4.9523532869E+00, 6.4048403090E-01)]),("P", [(1.0828262029E+00, 1.0000000000E+00)]),("P", [(4.9271277356E-01, 1.0000000000E+00)]),("P", [(2.0483450942E-01, 1.0000000000E+00)]),("P", [(8.0743615716E-02, 1.0000000000E+00)]),("D", [(4.1590000000E+00, 1.0000000000E+00)]),("D", [(1.0190000000E+00, 1.0000000000E+00)]),("D", [(4.6400000000E-01, 1.0000000000E+00)]),("D", [(1.9400000000E-01, 1.0000000000E+00)]),("F", [(8.6900000000E-01, 1.0000000000E+00)]),("F", [(3.3500000000E-01, 1.0000000000E+00)]),("G", [(6.8300000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.4674590095E+06, 1.1478257194E-05), (2.1975616433E+05, 8.9234299775E-05), (5.0010770301E+04, 4.6911086186E-04), (1.4164823918E+04, 1.9762446133E-03), (4.6207465525E+03, 7.1419937783E-03), (1.6678991635E+03, 2.2753219445E-02), (6.5029199265E+02, 6.3959782953E-02), (2.6938037376E+02, 1.5331059238E-01), (1.1706752106E+02, 2.8986952417E-01), (5.2811766843E+01, 3.6348071452E-01)]),("S", [(4.6142769988E+02, 1.8019457578E-02), (1.4212665355E+02, 1.6489442314E-01), (5.4437838768E+01, 6.3891587584E-01)]),("S", [(2.4160770219E+01, 1.0000000000E+00)]),("S", [(9.7083540306E+00, 1.0000000000E+00)]),("S", [(4.5640696733E+00, 1.0000000000E+00)]),("S", [(2.1194744832E+00, 1.0000000000E+00)]),("S", [(7.5722365394E-01, 1.0000000000E+00)]),("S", [(3.3747224597E-01, 1.0000000000E+00)]),("S", [(1.3860775149E-01, 1.0000000000E+00)]),("P", [(2.5019457890E+03, 2.4242618410E-04), (5.9288059285E+02, 2.1079961749E-03), (1.9218089186E+02, 1.1432693869E-02), (7.2875710488E+01, 4.4956698060E-02), (3.0436358370E+01, 1.3197476145E-01), (1.3490178902E+01, 2.7493639225E-01), (6.1478071413E+00, 3.8347236372E-01), (2.8450944820E+00, 2.8871943885E-01)]),("P", [(1.0539397936E+02, -3.4311760144E-02), (6.7369738513E+00, 6.4060818902E-01)]),("P", [(1.2421095772E+00, 1.0000000000E+00)]),("P", [(5.5669714254E-01, 1.0000000000E+00)]),("P", [(2.3387801464E-01, 1.0000000000E+00)]),("P", [(9.3164490890E-02, 1.0000000000E+00)]),("D", [(5.1910000000E+00, 1.0000000000E+00)]),("D", [(1.2760000000E+00, 1.0000000000E+00)]),("D", [(5.8300000000E-01, 1.0000000000E+00)]),("D", [(2.4300000000E-01, 1.0000000000E+00)]),("F", [(1.0890000000E+00, 1.0000000000E+00)]),("F", [(4.2300000000E-01, 1.0000000000E+00)]),("G", [(8.2700000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.6734219494E+06, 1.1245990413E-05), (2.5060175373E+05, 8.7428115798E-05), (5.7030912120E+04, 4.5961641275E-04), (1.6153303915E+04, 1.9363890742E-03), (5.2694109288E+03, 6.9994046497E-03), (1.9020315541E+03, 2.2312068196E-02), (7.4157677159E+02, 6.2808077697E-02), (3.0720901906E+02, 1.5101140997E-01), (1.3352786203E+02, 2.8713661177E-01), (6.0253381291E+01, 3.6385490107E-01)]),("S", [(5.2202426206E+02, 1.8071323116E-02), (1.6151290469E+02, 1.6449219884E-01), (6.2126369433E+01, 6.3897334899E-01)]),("S", [(2.7590930012E+01, 1.0000000000E+00)]),("S", [(1.1175528881E+01, 1.0000000000E+00)]),("S", [(5.2959065137E+00, 1.0000000000E+00)]),("S", [(2.4782798895E+00, 1.0000000000E+00)]),("S", [(9.1262095059E-01, 1.0000000000E+00)]),("S", [(4.0426067549E-01, 1.0000000000E+00)]),("S", [(1.6562886306E-01, 1.0000000000E+00)]),("P", [(1.2878765667E+02, -3.0509457751E-02), (8.4357867977E+00, 6.4101621070E-01)]),("P", [(2.8684504581E+03, 2.4010435121E-04), (6.7971869590E+02, 2.0904788665E-03), (2.2036758824E+02, 1.1369316839E-02), (8.3620439734E+01, 4.4901977398E-02), (3.4964322657E+01, 1.3237156770E-01), (1.5525131784E+01, 2.7709277600E-01), (7.0992979806E+00, 3.8613979646E-01), (3.3021336602E+00, 2.8492925372E-01)]),("P", [(1.4618661855E+00, 1.0000000000E+00)]),("P", [(6.4658010096E-01, 1.0000000000E+00)]),("P", [(2.7229295755E-01, 1.0000000000E+00)]),("P", [(1.0905182449E-01, 1.0000000000E+00)]),("D", [(6.3150000000E+00, 1.0000000000E+00)]),("D", [(1.5620000000E+00, 1.0000000000E+00)]),("D", [(7.1500000000E-01, 1.0000000000E+00)]),("D", [(2.9700000000E-01, 1.0000000000E+00)]),("F", [(1.3250000000E+00, 1.0000000000E+00)]),("F", [(5.4300000000E-01, 1.0000000000E+00)]),("G", [(1.0070000000E+00, 1.0000000000E+00)])],
}


DEF2_QZVPD = {

    1 : [("S", [(1.9069169000E+02, 7.0815167000E-04), (2.8605532000E+01, 5.4678827000E-03), (6.5095943000E+00, 2.7966605000E-02), (1.8412455000E+00, 1.0764538000E-01)]),("S", [(5.9853725000E-01, 1.0000000000E+00)]),("S", [(2.1397624000E-01, 1.0000000000E+00)]),("S", [(8.0316286000E-02, 1.0000000000E+00)]),("P", [(2.2920000000E+00, 1.0000000000E+00)]),("P", [(8.3800000000E-01, 1.0000000000E+00)]),("P", [(2.9200000000E-01, 1.0000000000E+00)]),("P", [(8.4063199228E-02, 1.0000000000E+00)]),("D", [(2.0620000000E+00, 1.0000000000E+00)]),("D", [(6.6200000000E-01, 1.0000000000E+00)]),("F", [(1.3970000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(1.1446470809E+03, 3.5861578618E-04), (1.7164596667E+02, 2.7725434466E-03), (3.9066056254E+01, 1.4241892216E-02), (1.1051401989E+01, 5.5457352277E-02), (3.5725574473E+00, 1.6170511810E-01)]),("S", [(1.2429415962E+00, 3.3177883833E-01)]),("S", [(4.4807668730E-01, 4.2014033068E-01)]),("S", [(1.6411579128E-01, 1.8652273014E-01)]),("P", [(5.9940000000E+00, 1.0000000000E+00)]),("P", [(1.7450000000E+00, 1.0000000000E+00)]),("P", [(5.6000000000E-01, 1.0000000000E+00)]),("P", [(1.6298923122E-01, 1.0000000000E+00)]),("D", [(4.2990000000E+00, 1.0000000000E+00)]),("D", [(1.2230000000E+00, 1.0000000000E+00)]),("F", [(2.6800000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(1.4853977085E+04, 4.2711388815E-05), (2.2252236477E+03, 3.3235310800E-04), (5.0488739008E+02, 1.7518436649E-03), (1.4245847548E+02, 7.3477995850E-03), (4.6315599580E+01, 2.5899837683E-02), (1.6655335474E+01, 7.6670682431E-02), (6.4331186199E+00, 1.8276075765E-01), (2.6027043858E+00, 3.2655434038E-01), (1.0897245405E+00, 3.7000429828E-01)]),("S", [(4.4236595971E+00, 1.1120987921E-01), (1.2356394990E+00, 7.9987335862E-01)]),("S", [(4.6067470602E-01, 1.0000000000E+00)]),("S", [(9.6617166955E-02, 1.0000000000E+00)]),("S", [(4.5915452296E-02, 1.0000000000E+00)]),("S", [(2.1140048730E-02, 1.0000000000E+00)]),("P", [(3.3270000000E+00, 1.0000000000E+00)]),("P", [(3.2605109206E+00, 8.6504749023E-03), (6.5003043115E-01, 4.7614123736E-02), (1.6941671073E-01, 2.1001138000E-01)]),("P", [(5.5732343767E-02, 1.0000000000E+00)]),("P", [(2.0489959241E-02, 1.0000000000E+00)]),("D", [(2.3000000000E-01, 1.0000000000E+00)]),("D", [(7.5700000000E-02, 1.0000000000E+00)]),("F", [(1.3500000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(2.9646704407E+04, 3.7943320869E-05), (4.4287614354E+03, 2.9605352634E-04), (1.0054701332E+03, 1.5572477356E-03), (2.8415339578E+02, 6.5323195769E-03), (9.2504356461E+01, 2.3116861005E-02), (3.3311016058E+01, 6.9458515249E-02), (1.2911553314E+01, 1.7014371331E-01), (5.2665497361E+00, 3.1682711374E-01)]),("S", [(2.2995386380E+01, 8.1791122747E-02), (6.6236354176E+00, 8.0000038023E-01)]),("S", [(2.2289595003E+00, 1.0000000000E+00)]),("S", [(9.5302433451E-01, 1.0000000000E+00)]),("S", [(2.4651602436E-01, 1.0000000000E+00)]),("S", [(1.0153895315E-01, 1.0000000000E+00)]),("S", [(4.1551374604E-02, 1.0000000000E+00)]),("P", [(6.4000000000E+00, 1.0000000000E+00)]),("P", [(1.4099789445E+01, 3.8508871145E-03), (3.1803188482E+00, 2.4152380771E-02), (9.0489220496E-01, 9.7926925156E-02), (3.0411585508E-01, 2.9470380030E-01)]),("P", [(1.1302606657E-01, 1.0000000000E+00)]),("P", [(4.2831733691E-02, 1.0000000000E+00)]),("D", [(3.6000000000E-01, 1.0000000000E+00)]),("D", [(9.0000000000E-02, 1.0000000000E+00)]),("F", [(2.9000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(4.6447667056E+04, 3.8388681014E-05), (6.9576889042E+03, 2.9836770197E-04), (1.5834428403E+03, 1.5645478806E-03), (4.4846601009E+02, 6.5476770742E-03), (1.4628639262E+02, 2.3139008388E-02), (5.2784386084E+01, 6.9615799357E-02), (2.0519396170E+01, 1.7119636876E-01), (8.4185659258E+00, 3.1913191837E-01)]),("S", [(3.6510018312E+01, 7.8990617244E-02), (1.0541854005E+01, 7.8926384265E-01)]),("S", [(3.6004091387E+00, 1.0000000000E+00)]),("S", [(1.5617023749E+00, 1.0000000000E+00)]),("S", [(4.4997370775E-01, 1.0000000000E+00)]),("S", [(1.8075230237E-01, 1.0000000000E+00)]),("S", [(7.1596696319E-02, 1.0000000000E+00)]),("S", [(2.9941402533E-02, 1.0000000000E+00)]),("P", [(7.2240462760E+01, 8.6539170411E-04), (1.6807707208E+01, 6.8762067365E-03), (5.2259410747E+00, 3.0976687148E-02), (1.8508350671E+00, 1.0432357913E-01), (7.2206782663E-01, 2.6164137577E-01)]),("P", [(2.9491018056E-01, 1.0000000000E+00)]),("P", [(1.2201141839E-01, 1.0000000000E+00)]),("P", [(4.9865393625E-02, 1.0000000000E+00)]),("D", [(1.1100000000E+00, 1.0000000000E+00)]),("D", [(4.0200000000E-01, 1.0000000000E+00)]),("D", [(1.4500000000E-01, 1.0000000000E+00)]),("D", [(5.0896593879E-02, 1.0000000000E+00)]),("F", [(8.8200000000E-01, 1.0000000000E+00)]),("F", [(3.1100000000E-01, 1.0000000000E+00)]),("G", [(6.7300000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(6.7025071029E+04, 3.8736308501E-05), (1.0039986538E+04, 3.0107917575E-04), (2.2849316911E+03, 1.5787918095E-03), (6.4714122130E+02, 6.6087087195E-03), (2.1109472335E+02, 2.3367123250E-02), (7.6177643862E+01, 7.0420716898E-02), (2.9633839163E+01, 1.7360344953E-01), (1.2187785081E+01, 3.2292305648E-01)]),("S", [(5.3026006299E+01, 7.4897404492E-02), (1.5258502776E+01, 7.6136220983E-01)]),("S", [(5.2403957464E+00, 1.0000000000E+00)]),("S", [(2.2905022379E+00, 1.0000000000E+00)]),("S", [(6.9673283006E-01, 1.0000000000E+00)]),("S", [(2.7599337363E-01, 1.0000000000E+00)]),("S", [(1.0739884389E-01, 1.0000000000E+00)]),("S", [(4.4981404899E-02, 1.0000000000E+00)]),("P", [(1.0512555082E+02, 8.4647553844E-04), (2.4884461066E+01, 6.6274038534E-03), (7.8637230826E+00, 3.0120390419E-02), (2.8407001835E+00, 9.9951435476E-02), (1.1227137335E+00, 2.3826299282E-01)]),("P", [(4.6050725555E-01, 1.0000000000E+00)]),("P", [(1.8937530913E-01, 1.0000000000E+00)]),("P", [(7.5983791611E-02, 1.0000000000E+00)]),("D", [(1.8480000000E+00, 1.0000000000E+00)]),("D", [(6.4900000000E-01, 1.0000000000E+00)]),("D", [(2.2800000000E-01, 1.0000000000E+00)]),("D", [(7.6889830417E-02, 1.0000000000E+00)]),("F", [(1.4190000000E+00, 1.0000000000E+00)]),("F", [(4.8500000000E-01, 1.0000000000E+00)]),("G", [(1.0110000000E+00, 1.0000000000E+00)])],
    7 : [("S", [(9.0726889210E+04, 3.9257887368E-05), (1.3590528801E+04, 3.0513316455E-04), (3.0929883781E+03, 1.6000560446E-03), (8.7599876362E+02, 6.6982937306E-03), (2.8574469982E+02, 2.3690078765E-02), (1.0311913417E+02, 7.1455405268E-02), (4.0128556777E+01, 1.7632774876E-01), (1.6528095704E+01, 3.2677592815E-01)]),("S", [(6.9390960983E+01, 8.0052094386E-02), (2.0428200596E+01, 7.8268063538E-01)]),("S", [(7.1292587972E+00, 1.0000000000E+00)]),("S", [(3.1324304893E+00, 1.0000000000E+00)]),("S", [(9.8755778723E-01, 1.0000000000E+00)]),("S", [(3.8765721307E-01, 1.0000000000E+00)]),("S", [(1.4909883075E-01, 1.0000000000E+00)]),("S", [(6.2151630318E-02, 1.0000000000E+00)]),("P", [(1.5005742670E+02, -8.6216165986E-04), (3.5491599483E+01, -6.8571273236E-03), (1.1247864223E+01, -3.1795688855E-02), (4.0900305195E+00, -1.0537396822E-01), (1.6220573146E+00, -2.4519708041E-01)]),("P", [(6.6442261530E-01, 1.0000000000E+00)]),("P", [(2.7099770070E-01, 1.0000000000E+00)]),("P", [(1.0688749984E-01, 1.0000000000E+00)]),("D", [(2.8370000000E+00, 1.0000000000E+00)]),("D", [(9.6800000000E-01, 1.0000000000E+00)]),("D", [(3.3500000000E-01, 1.0000000000E+00)]),("D", [(1.0825280010E-01, 1.0000000000E+00)]),("F", [(2.0270000000E+00, 1.0000000000E+00)]),("F", [(6.8500000000E-01, 1.0000000000E+00)]),("G", [(1.4270000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(1.1650646908E+05, 4.0383857939E-05), (1.7504349724E+04, 3.1255139004E-04), (3.9934513230E+03, 1.6341473495E-03), (1.1330063186E+03, 6.8283224757E-03), (3.6999569594E+02, 2.4124410221E-02), (1.3362074349E+02, 7.2730206154E-02), (5.2035643649E+01, 1.7934429892E-01), (2.1461939313E+01, 3.3059588895E-01)]),("S", [(8.9835051252E+01, 9.6468652996E-02), (2.6428010844E+01, 9.4117481120E-01)]),("S", [(9.2822824649E+00, 1.0000000000E+00)]),("S", [(4.0947728533E+00, 1.0000000000E+00)]),("S", [(1.3255349078E+00, 1.0000000000E+00)]),("S", [(5.1877230787E-01, 1.0000000000E+00)]),("S", [(1.9772676454E-01, 1.0000000000E+00)]),("S", [(6.9638535104E-02, 1.0000000000E+00)]),("P", [(1.9115255810E+02, 2.5115697705E-03), (4.5233356739E+01, 2.0039240864E-02), (1.4353465922E+01, 9.3609064762E-02), (5.2422371832E+00, 3.0618127124E-01), (2.0792418599E+00, 6.7810501439E-01)]),("P", [(8.4282371424E-01, 1.0000000000E+00)]),("P", [(3.3617694891E-01, 1.0000000000E+00)]),("P", [(1.2863997974E-01, 1.0000000000E+00)]),("P", [(4.3598162776E-02, 1.0000000000E+00)]),("D", [(3.7750000000E+00, 1.0000000000E+00)]),("D", [(1.3000000000E+00, 1.0000000000E+00)]),("D", [(4.4400000000E-01, 1.0000000000E+00)]),("D", [(1.2546378695E-01, 1.0000000000E+00)]),("F", [(2.6660000000E+00, 1.0000000000E+00)]),("F", [(8.5900000000E-01, 1.0000000000E+00)]),("G", [(1.8460000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.3253597345E+05, 4.7387482743E-05), (1.9758112588E+04, 3.7070120897E-04), (4.4851996947E+03, 1.9450784713E-03), (1.2738151020E+03, 8.0573291994E-03), (4.1893831236E+02, 2.7992880781E-02), (1.5255721985E+02, 8.2735120175E-02), (5.9821524823E+01, 1.9854169012E-01), (2.4819076932E+01, 3.4860632233E-01)]),("S", [(1.0074446673E+02, 1.0505068816E-01), (3.0103728290E+01, 9.4068472434E-01)]),("S", [(1.0814283272E+01, 1.0000000000E+00)]),("S", [(4.8172886770E+00, 1.0000000000E+00)]),("S", [(1.6559334213E+00, 1.0000000000E+00)]),("S", [(6.4893519582E-01, 1.0000000000E+00)]),("S", [(2.4778104545E-01, 1.0000000000E+00)]),("S", [(8.7626236800E-02, 1.0000000000E+00)]),("P", [(2.4096654114E+02, 3.0389933451E-03), (5.7020699781E+01, 2.4357738582E-02), (1.8126952120E+01, 1.1442925768E-01), (6.6457404621E+00, 3.7064659853E-01), (2.6375722892E+00, 7.9791551766E-01)]),("P", [(1.0638217200E+00, 1.0000000000E+00)]),("P", [(4.1932562750E-01, 1.0000000000E+00)]),("P", [(1.5747588299E-01, 1.0000000000E+00)]),("P", [(4.6772400332E-02, 1.0000000000E+00)]),("D", [(5.0140000000E+00, 1.0000000000E+00)]),("D", [(1.7250000000E+00, 1.0000000000E+00)]),("D", [(5.8600000000E-01, 1.0000000000E+00)]),("D", [(1.5967986245E-01, 1.0000000000E+00)]),("F", [(3.5620000000E+00, 1.0000000000E+00)]),("F", [(1.1480000000E+00, 1.0000000000E+00)]),("G", [(2.3760000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.6067627955E+05, 4.7387482743E-05), (2.3953195039E+04, 3.7070120897E-04), (5.4375063711E+03, 1.9450784713E-03), (1.5442741024E+03, 8.0573291994E-03), (5.0788814269E+02, 2.7992880781E-02), (1.8494847750E+02, 8.2735120175E-02), (7.2522952039E+01, 1.9854169012E-01), (3.0088713575E+01, 3.4860632233E-01)]),("S", [(1.2068434940E+02, 1.0400597014E-01), (3.6074294857E+01, 9.1999988986E-01)]),("S", [(1.3052091210E+01, 1.0000000000E+00)]),("S", [(5.7974783353E+00, 1.0000000000E+00)]),("S", [(2.0478626895E+00, 1.0000000000E+00)]),("S", [(8.0214571994E-01, 1.0000000000E+00)]),("S", [(3.0548672205E-01, 1.0000000000E+00)]),("S", [(1.0621243030E-01, 1.0000000000E+00)]),("P", [(4.9843397466E+02, 4.3118261050E-04), (1.1814109217E+02, 3.6206397703E-03), (3.8032529735E+01, 1.8245344081E-02), (1.4183731950E+01, 6.5133035572E-02), (5.7864190721E+00, 1.6708915089E-01), (2.4599622160E+00, 2.9268019613E-01)]),("P", [(1.0421999498E+00, 1.0000000000E+00)]),("P", [(4.2987748916E-01, 1.0000000000E+00)]),("P", [(1.6889708453E-01, 1.0000000000E+00)]),("P", [(4.7169461232E-02, 1.0000000000E+00)]),("D", [(6.4710000000E+00, 1.0000000000E+00)]),("D", [(2.2130000000E+00, 1.0000000000E+00)]),("D", [(7.4700000000E-01, 1.0000000000E+00)]),("D", [(1.9790459442E-01, 1.0000000000E+00)]),("F", [(4.6570000000E+00, 1.0000000000E+00)]),("F", [(1.5240000000E+00, 1.0000000000E+00)]),("G", [(2.9830000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.7985220081E+05, 2.0671384468E-05), (5.6886006378E+04, 1.6070466617E-04), (1.2942701838E+04, 8.4462905848E-04), (3.6643017904E+03, 3.5519026029E-03), (1.1947417499E+03, 1.2754034468E-02), (4.3098192917E+02, 3.9895462742E-02), (1.6783169424E+02, 1.0720154498E-01), (6.9306669040E+01, 2.3339516913E-01), (2.9951170886E+01, 3.6333077287E-01), (1.3380791097E+01, 3.0544770974E-01)]),("S", [(1.2174011283E+02, 3.6142427284E-02), (3.7044143387E+01, 2.8820961687E-01), (1.3995422624E+01, 7.9337384869E-01)]),("S", [(5.9827797428E+00, 1.0000000000E+00)]),("S", [(2.4830455118E+00, 1.0000000000E+00)]),("S", [(1.0452506187E+00, 1.0000000000E+00)]),("S", [(4.3875640383E-01, 1.0000000000E+00)]),("S", [(6.5595633185E-02, 1.0000000000E+00)]),("S", [(3.0561925072E-02, 1.0000000000E+00)]),("S", [(1.5509064018E-02, 1.0000000000E+00)]),("P", [(6.9077627017E+02, 3.7478518415E-04), (1.6382806121E+02, 3.1775441030E-03), (5.2876460769E+01, 1.6333581338E-02), (1.9812270493E+01, 5.9754902585E-02), (8.1320378784E+00, 1.5879328812E-01), (3.4969068377E+00, 2.9049363260E-01), (1.5117244146E+00, 3.6368131139E-01), (6.4479294912E-01, 2.8195867334E-01)]),("P", [(2.6145823312E-01, 1.0000000000E+00)]),("P", [(1.1704726116E-01, 1.0000000000E+00)]),("P", [(4.0494747666E-02, 1.0000000000E+00)]),("P", [(1.5666707355E-02, 1.0000000000E+00)]),("D", [(2.9000000000E+00, 1.0000000000E+00)]),("D", [(2.3000000000E-01, 1.0000000000E+00)]),("D", [(7.5700000000E-02, 1.0000000000E+00)]),("F", [(1.3500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(6.0596778753E+05, 1.4430176617E-05), (9.0569094692E+04, 1.1249871807E-04), (2.0574252844E+04, 5.9264972450E-04), (5.8186284865E+03, 2.4988770424E-03), (1.8956296075E+03, 9.0230775141E-03), (6.8345941021E+02, 2.8579919806E-02), (2.6618219762E+02, 7.9064453923E-02), (1.1011220010E+02, 1.8269337814E-01), (4.7777041234E+01, 3.2157193967E-01), (2.1542166149E+01, 3.5028259389E-01)]),("S", [(1.7412136378E+02, 2.2931111278E-02), (5.3484972498E+01, 1.9151777552E-01), (2.0500213307E+01, 6.1155711055E-01)]),("S", [(9.8056826922E+00, 1.0000000000E+00)]),("S", [(3.5402989348E+00, 1.0000000000E+00)]),("S", [(1.5450850046E+00, 1.0000000000E+00)]),("S", [(6.6540195432E-01, 1.0000000000E+00)]),("S", [(1.4355439071E-01, 1.0000000000E+00)]),("S", [(6.6624673468E-02, 1.0000000000E+00)]),("S", [(2.9772648266E-02, 1.0000000000E+00)]),("P", [(8.9320460829E+02, 3.4958267428E-04), (2.1178258286E+02, 2.9811890111E-03), (6.8443200537E+01, 1.5517845758E-02), (2.5727265349E+01, 5.7578659557E-02), (1.0606634281E+01, 1.5610307853E-01), (4.5934126484E+00, 2.9230912580E-01), (2.0100469807E+00, 3.7219024072E-01), (8.7384841536E-01, 2.7578088891E-01)]),("P", [(3.5615076666E-01, 1.0000000000E+00)]),("P", [(1.8995954356E-01, 1.0000000000E+00)]),("P", [(7.4580135669E-02, 1.0000000000E+00)]),("P", [(2.9221641309E-02, 1.0000000000E+00)]),("D", [(3.8100000000E+00, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)]),("D", [(5.2000000000E-02, 1.0000000000E+00)]),("F", [(1.6000000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(7.5455078265E+05, 1.3421335886E-05), (1.1299938922E+05, 1.0433210786E-04), (2.5715831759E+04, 5.4841772080E-04), (7.2836030283E+03, 2.3089081391E-03), (2.3760008796E+03, 8.3309974593E-03), (8.5765468087E+02, 2.6417963653E-02), (3.3438922598E+02, 7.3443646426E-02), (1.3848504731E+02, 1.7184039139E-01), (6.0150368808E+01, 3.1041980030E-01), (2.7127610860E+01, 3.5669190596E-01)]),("S", [(2.2536500065E+02, 2.1522039783E-02), (6.9341968124E+01, 1.8531780493E-01), (2.6619335712E+01, 6.3533181245E-01)]),("S", [(1.2349420671E+01, 1.0000000000E+00)]),("S", [(4.5878785994E+00, 1.0000000000E+00)]),("S", [(2.0571338103E+00, 1.0000000000E+00)]),("S", [(9.0864549395E-01, 1.0000000000E+00)]),("S", [(2.4226988038E-01, 1.0000000000E+00)]),("S", [(1.1248136869E-01, 1.0000000000E+00)]),("S", [(4.8156492817E-02, 1.0000000000E+00)]),("S", [(2.1134849794E-02, 1.0000000000E+00)]),("P", [(1.4896119522E+03, 2.0177122443E-04), (3.5301399267E+02, 1.7508109203E-03), (1.1440764069E+02, 9.4424704664E-03), (4.3312186111E+01, 3.6868004006E-02), (1.8027322216E+01, 1.0892874007E-01), (7.9675432403E+00, 2.3265901004E-01), (3.6090399541E+00, 3.4643587076E-01), (1.6456081630E+00, 3.3440809866E-01)]),("P", [(3.4731187489E+01, 1.7823073020E-02), (1.2553083630E+00, -5.9991263926E-01)]),("P", [(7.3635534282E-01, 1.0000000000E+00)]),("P", [(2.4686465559E-01, 1.0000000000E+00)]),("P", [(9.4821971828E-02, 1.0000000000E+00)]),("P", [(3.6214165654E-02, 1.0000000000E+00)]),("D", [(1.9700000000E+00, 1.0000000000E+00)]),("D", [(4.3700000000E-01, 1.0000000000E+00)]),("D", [(1.9500000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-02, 1.0000000000E+00)]),("D", [(3.0664004967E-02, 1.0000000000E+00)]),("F", [(4.0100000000E-01, 1.0000000000E+00)]),("F", [(1.5400000000E-01, 1.0000000000E+00)]),("G", [(3.5700000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(9.1807069565E+05, 1.2658787192E-05), (1.3748525386E+05, 9.8408387461E-05), (3.1287772714E+04, 5.1731398244E-04), (8.8616105697E+03, 2.1785441933E-03), (2.8906943156E+03, 7.8656853716E-03), (1.0434063979E+03, 2.4987553674E-02), (4.0680160276E+02, 6.9761818844E-02), (1.6848360207E+02, 1.6473362733E-01), (7.3185628823E+01, 3.0285602647E-01), (3.2998485420E+01, 3.6007260439E-01)]),("S", [(2.7878751325E+02, 2.0185463814E-02), (8.5910228722E+01, 1.7720406747E-01), (3.2992604031E+01, 6.3587860906E-01)]),("S", [(1.5033693254E+01, 1.0000000000E+00)]),("S", [(5.7257514773E+00, 1.0000000000E+00)]),("S", [(2.6146521029E+00, 1.0000000000E+00)]),("S", [(1.1757833152E+00, 1.0000000000E+00)]),("S", [(3.5432330868E-01, 1.0000000000E+00)]),("S", [(1.6248335806E-01, 1.0000000000E+00)]),("S", [(6.8332457181E-02, 1.0000000000E+00)]),("S", [(3.0521078871E-02, 1.0000000000E+00)]),("P", [(1.7758850516E+03, 2.0186971126E-04), (4.2083786849E+02, 1.7544314655E-03), (1.3642290243E+02, 9.5039532290E-03), (5.1700991737E+01, 3.7325741590E-02), (2.1559456002E+01, 1.1085396329E-01), (9.5559200095E+00, 2.3756407046E-01), (4.3529819473E+00, 3.5295783750E-01), (2.0096964381E+00, 3.2885931947E-01)]),("P", [(4.6418153780E+01, 2.5858819154E-02), (1.9076417796E+00, -5.9874865899E-01)]),("P", [(9.2337221587E-01, 1.0000000000E+00)]),("P", [(3.4515720370E-01, 1.0000000000E+00)]),("P", [(1.3656834761E-01, 1.0000000000E+00)]),("P", [(5.2987060586E-02, 1.0000000000E+00)]),("D", [(2.6450000000E+00, 1.0000000000E+00)]),("D", [(6.0800000000E-01, 1.0000000000E+00)]),("D", [(2.7200000000E-01, 1.0000000000E+00)]),("D", [(1.1300000000E-01, 1.0000000000E+00)]),("D", [(4.4018904228E-02, 1.0000000000E+00)]),("F", [(5.4100000000E-01, 1.0000000000E+00)]),("F", [(2.1200000000E-01, 1.0000000000E+00)]),("G", [(4.6100000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.0905617138E+06, 1.2142449664E-05), (1.6331639461E+05, 9.4395292614E-05), (3.7166607451E+04, 4.9622262177E-04), (1.0526880945E+04, 2.0900040747E-03), (3.4339976028E+03, 7.5489230025E-03), (1.2395360480E+03, 2.4010423937E-02), (4.8327456199E+02, 6.7231473697E-02), (2.0016911586E+02, 1.5978669881E-01), (8.6960394829E+01, 2.9735906782E-01), (3.9211283369E+01, 3.6187171850E-01)]),("S", [(3.3675883662E+02, 1.9154721050E-02), (1.0372179793E+02, 1.7134079100E-01), (3.9771861240E+01, 6.3689655985E-01)]),("S", [(1.7888612952E+01, 1.0000000000E+00)]),("S", [(6.9644556879E+00, 1.0000000000E+00)]),("S", [(3.2198092087E+00, 1.0000000000E+00)]),("S", [(1.4669943979E+00, 1.0000000000E+00)]),("S", [(4.7765437532E-01, 1.0000000000E+00)]),("S", [(2.1637789241E-01, 1.0000000000E+00)]),("S", [(9.0235894336E-02, 1.0000000000E+00)]),("S", [(4.0836372978E-02, 1.0000000000E+00)]),("P", [(2.0196711374E+03, 2.1359172406E-04), (4.7860125090E+02, 1.8568771399E-03), (1.5514942504E+02, 1.0070690115E-02), (5.8816356575E+01, 3.9605153679E-02), (2.4544512785E+01, 1.1736067844E-01), (1.0883571061E+01, 2.4950540642E-01), (4.9624791285E+00, 3.6421287984E-01), (2.3002912343E+00, 3.1764127123E-01)]),("P", [(5.9371345016E+01, 3.9432918004E-01), (3.0694590986E+00, -6.3522960431E+00)]),("P", [(1.0634401739E+00, 1.0000000000E+00)]),("P", [(4.5022152161E-01, 1.0000000000E+00)]),("P", [(1.8267271344E-01, 1.0000000000E+00)]),("P", [(7.1610333771E-02, 1.0000000000E+00)]),("D", [(3.3430000000E+00, 1.0000000000E+00)]),("D", [(8.0700000000E-01, 1.0000000000E+00)]),("D", [(3.6500000000E-01, 1.0000000000E+00)]),("D", [(1.5400000000E-01, 1.0000000000E+00)]),("D", [(5.9687731846E-02, 1.0000000000E+00)]),("F", [(7.0300000000E-01, 1.0000000000E+00)]),("F", [(2.8000000000E-01, 1.0000000000E+00)]),("G", [(5.9700000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(1.2734109023E+06, 1.1767088246E-05), (1.9069783007E+05, 9.1478610166E-05), (4.3397885330E+04, 4.8090078640E-04), (1.2291809677E+04, 2.0257193592E-03), (4.0097420824E+03, 7.3190096406E-03), (1.4473531030E+03, 2.3300499900E-02), (5.6430102913E+02, 6.5386213610E-02), (2.3374506243E+02, 1.5614449910E-01), (1.0156402814E+02, 2.9318563787E-01), (4.5805907187E+01, 3.6287914289E-01)]),("S", [(3.9427281503E+02, 1.8753305081E-02), (1.2172249591E+02, 1.6870726663E-01), (4.6754125963E+01, 6.3806830653E-01)]),("S", [(2.0923008254E+01, 1.0000000000E+00)]),("S", [(8.2685567800E+00, 1.0000000000E+00)]),("S", [(3.8629345671E+00, 1.0000000000E+00)]),("S", [(1.7794684781E+00, 1.0000000000E+00)]),("S", [(6.1064260103E-01, 1.0000000000E+00)]),("S", [(2.7412269445E-01, 1.0000000000E+00)]),("S", [(1.1325939107E-01, 1.0000000000E+00)]),("S", [(4.4392624881E-02, 1.0000000000E+00)]),("P", [(2.1898930459E+03, 2.3912552864E-04), (5.1894596592E+02, 2.0772032158E-03), (1.6819560151E+02, 1.1242420571E-02), (6.3745282788E+01, 4.4069933941E-02), (2.6597033077E+01, 1.2918778608E-01), (1.1774251449E+01, 2.6910820167E-01), (5.3534379024E+00, 3.7855928620E-01), (2.4701911802E+00, 2.9692134655E-01)]),("P", [(8.2120288349E+01, -3.9420318847E-02), (4.9523532869E+00, 6.4048403090E-01)]),("P", [(1.0828262029E+00, 1.0000000000E+00)]),("P", [(4.9271277356E-01, 1.0000000000E+00)]),("P", [(2.0483450942E-01, 1.0000000000E+00)]),("P", [(8.0743615716E-02, 1.0000000000E+00)]),("P", [(2.5661157833E-02, 1.0000000000E+00)]),("D", [(4.1590000000E+00, 1.0000000000E+00)]),("D", [(1.0190000000E+00, 1.0000000000E+00)]),("D", [(4.6400000000E-01, 1.0000000000E+00)]),("D", [(1.9400000000E-01, 1.0000000000E+00)]),("D", [(6.7889829366E-02, 1.0000000000E+00)]),("F", [(8.6900000000E-01, 1.0000000000E+00)]),("F", [(3.3500000000E-01, 1.0000000000E+00)]),("G", [(6.8300000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.4674590095E+06, 1.1478257194E-05), (2.1975616433E+05, 8.9234299775E-05), (5.0010770301E+04, 4.6911086186E-04), (1.4164823918E+04, 1.9762446133E-03), (4.6207465525E+03, 7.1419937783E-03), (1.6678991635E+03, 2.2753219445E-02), (6.5029199265E+02, 6.3959782953E-02), (2.6938037376E+02, 1.5331059238E-01), (1.1706752106E+02, 2.8986952417E-01), (5.2811766843E+01, 3.6348071452E-01)]),("S", [(4.6142769988E+02, 1.8019457578E-02), (1.4212665355E+02, 1.6489442314E-01), (5.4437838768E+01, 6.3891587584E-01)]),("S", [(2.4160770219E+01, 1.0000000000E+00)]),("S", [(9.7083540306E+00, 1.0000000000E+00)]),("S", [(4.5640696733E+00, 1.0000000000E+00)]),("S", [(2.1194744832E+00, 1.0000000000E+00)]),("S", [(7.5722365394E-01, 1.0000000000E+00)]),("S", [(3.3747224597E-01, 1.0000000000E+00)]),("S", [(1.3860775149E-01, 1.0000000000E+00)]),("S", [(5.4234565513E-02, 1.0000000000E+00)]),("P", [(2.5019457890E+03, 2.4242618410E-04), (5.9288059285E+02, 2.1079961749E-03), (1.9218089186E+02, 1.1432693869E-02), (7.2875710488E+01, 4.4956698060E-02), (3.0436358370E+01, 1.3197476145E-01), (1.3490178902E+01, 2.7493639225E-01), (6.1478071413E+00, 3.8347236372E-01), (2.8450944820E+00, 2.8871943885E-01)]),("P", [(1.0539397936E+02, -3.4311760144E-02), (6.7369738513E+00, 6.4060818902E-01)]),("P", [(1.2421095772E+00, 1.0000000000E+00)]),("P", [(5.5669714254E-01, 1.0000000000E+00)]),("P", [(2.3387801464E-01, 1.0000000000E+00)]),("P", [(9.3164490890E-02, 1.0000000000E+00)]),("P", [(2.7256146018E-02, 1.0000000000E+00)]),("D", [(5.1910000000E+00, 1.0000000000E+00)]),("D", [(1.2760000000E+00, 1.0000000000E+00)]),("D", [(5.8300000000E-01, 1.0000000000E+00)]),("D", [(2.4300000000E-01, 1.0000000000E+00)]),("D", [(8.3456538402E-02, 1.0000000000E+00)]),("F", [(1.0890000000E+00, 1.0000000000E+00)]),("F", [(4.2300000000E-01, 1.0000000000E+00)]),("G", [(8.2700000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.6734219494E+06, 1.1245990413E-05), (2.5060175373E+05, 8.7428115798E-05), (5.7030912120E+04, 4.5961641275E-04), (1.6153303915E+04, 1.9363890742E-03), (5.2694109288E+03, 6.9994046497E-03), (1.9020315541E+03, 2.2312068196E-02), (7.4157677159E+02, 6.2808077697E-02), (3.0720901906E+02, 1.5101140997E-01), (1.3352786203E+02, 2.8713661177E-01), (6.0253381291E+01, 3.6385490107E-01)]),("S", [(5.2202426206E+02, 1.8071323116E-02), (1.6151290469E+02, 1.6449219884E-01), (6.2126369433E+01, 6.3897334899E-01)]),("S", [(2.7590930012E+01, 1.0000000000E+00)]),("S", [(1.1175528881E+01, 1.0000000000E+00)]),("S", [(5.2959065137E+00, 1.0000000000E+00)]),("S", [(2.4782798895E+00, 1.0000000000E+00)]),("S", [(9.1262095059E-01, 1.0000000000E+00)]),("S", [(4.0426067549E-01, 1.0000000000E+00)]),("S", [(1.6562886306E-01, 1.0000000000E+00)]),("S", [(6.5129364729E-02, 1.0000000000E+00)]),("P", [(1.2878765667E+02, -3.0509457751E-02), (8.4357867977E+00, 6.4101621070E-01)]),("P", [(2.8684504581E+03, 2.4010435121E-04), (6.7971869590E+02, 2.0904788665E-03), (2.2036758824E+02, 1.1369316839E-02), (8.3620439734E+01, 4.4901977398E-02), (3.4964322657E+01, 1.3237156770E-01), (1.5525131784E+01, 2.7709277600E-01), (7.0992979806E+00, 3.8613979646E-01), (3.3021336602E+00, 2.8492925372E-01)]),("P", [(1.4618661855E+00, 1.0000000000E+00)]),("P", [(6.4658010096E-01, 1.0000000000E+00)]),("P", [(2.7229295755E-01, 1.0000000000E+00)]),("P", [(1.0905182449E-01, 1.0000000000E+00)]),("P", [(3.0101149684E-02, 1.0000000000E+00)]),("D", [(6.3150000000E+00, 1.0000000000E+00)]),("D", [(1.5620000000E+00, 1.0000000000E+00)]),("D", [(7.1500000000E-01, 1.0000000000E+00)]),("D", [(2.9700000000E-01, 1.0000000000E+00)]),("D", [(1.0105028218E-01, 1.0000000000E+00)]),("F", [(1.3250000000E+00, 1.0000000000E+00)]),("F", [(5.4300000000E-01, 1.0000000000E+00)]),("G", [(1.0070000000E+00, 1.0000000000E+00)])],
}





DEF2_QZVPP = {

    1 : [("S", [(1.9069169000E+02, 7.0815167000E-04), (2.8605532000E+01, 5.4678827000E-03), (6.5095943000E+00, 2.7966605000E-02), (1.8412455000E+00, 1.0764538000E-01)]),("S", [(5.9853725000E-01, 1.0000000000E+00)]),("S", [(2.1397624000E-01, 1.0000000000E+00)]),("S", [(8.0316286000E-02, 1.0000000000E+00)]),("P", [(2.2920000000E+00, 1.0000000000E+00)]),("P", [(8.3800000000E-01, 1.0000000000E+00)]),("P", [(2.9200000000E-01, 1.0000000000E+00)]),("D", [(2.0620000000E+00, 1.0000000000E+00)]),("D", [(6.6200000000E-01, 1.0000000000E+00)]),("F", [(1.3970000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(1.1446470809E+03, 3.5861578618E-04), (1.7164596667E+02, 2.7725434466E-03), (3.9066056254E+01, 1.4241892216E-02), (1.1051401989E+01, 5.5457352277E-02), (3.5725574473E+00, 1.6170511810E-01)]),("S", [(1.2429415962E+00, 3.3177883833E-01)]),("S", [(4.4807668730E-01, 4.2014033068E-01)]),("S", [(1.6411579128E-01, 1.8652273014E-01)]),("P", [(5.9940000000E+00, 1.0000000000E+00)]),("P", [(1.7450000000E+00, 1.0000000000E+00)]),("P", [(5.6000000000E-01, 1.0000000000E+00)]),("D", [(4.2990000000E+00, 1.0000000000E+00)]),("D", [(1.2230000000E+00, 1.0000000000E+00)]),("F", [(2.6800000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(1.4853977085E+04, 4.2711388815E-05), (2.2252236477E+03, 3.3235310800E-04), (5.0488739008E+02, 1.7518436649E-03), (1.4245847548E+02, 7.3477995850E-03), (4.6315599580E+01, 2.5899837683E-02), (1.6655335474E+01, 7.6670682431E-02), (6.4331186199E+00, 1.8276075765E-01), (2.6027043858E+00, 3.2655434038E-01), (1.0897245405E+00, 3.7000429828E-01)]),("S", [(4.4236595971E+00, 1.1120987921E-01), (1.2356394990E+00, 7.9987335862E-01)]),("S", [(4.6067470602E-01, 1.0000000000E+00)]),("S", [(9.6617166955E-02, 1.0000000000E+00)]),("S", [(4.5915452296E-02, 1.0000000000E+00)]),("S", [(2.1140048730E-02, 1.0000000000E+00)]),("P", [(3.3270000000E+00, 1.0000000000E+00)]),("P", [(3.2605109206E+00, 8.6504749023E-03), (6.5003043115E-01, 4.7614123736E-02), (1.6941671073E-01, 2.1001138000E-01)]),("P", [(5.5732343767E-02, 1.0000000000E+00)]),("P", [(2.0489959241E-02, 1.0000000000E+00)]),("D", [(2.3000000000E-01, 1.0000000000E+00)]),("D", [(7.5700000000E-02, 1.0000000000E+00)]),("F", [(1.3500000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(2.9646704407E+04, 3.7943320869E-05), (4.4287614354E+03, 2.9605352634E-04), (1.0054701332E+03, 1.5572477356E-03), (2.8415339578E+02, 6.5323195769E-03), (9.2504356461E+01, 2.3116861005E-02), (3.3311016058E+01, 6.9458515249E-02), (1.2911553314E+01, 1.7014371331E-01), (5.2665497361E+00, 3.1682711374E-01)]),("S", [(2.2995386380E+01, 8.1791122747E-02), (6.6236354176E+00, 8.0000038023E-01)]),("S", [(2.2289595003E+00, 1.0000000000E+00)]),("S", [(9.5302433451E-01, 1.0000000000E+00)]),("S", [(2.4651602436E-01, 1.0000000000E+00)]),("S", [(1.0153895315E-01, 1.0000000000E+00)]),("S", [(4.1551374604E-02, 1.0000000000E+00)]),("P", [(6.4000000000E+00, 1.0000000000E+00)]),("P", [(1.4099789445E+01, 3.8508871145E-03), (3.1803188482E+00, 2.4152380771E-02), (9.0489220496E-01, 9.7926925156E-02), (3.0411585508E-01, 2.9470380030E-01)]),("P", [(1.1302606657E-01, 1.0000000000E+00)]),("P", [(4.2831733691E-02, 1.0000000000E+00)]),("D", [(3.6000000000E-01, 1.0000000000E+00)]),("D", [(9.0000000000E-02, 1.0000000000E+00)]),("F", [(2.9000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(4.6447667056E+04, 3.8388681014E-05), (6.9576889042E+03, 2.9836770197E-04), (1.5834428403E+03, 1.5645478806E-03), (4.4846601009E+02, 6.5476770742E-03), (1.4628639262E+02, 2.3139008388E-02), (5.2784386084E+01, 6.9615799357E-02), (2.0519396170E+01, 1.7119636876E-01), (8.4185659258E+00, 3.1913191837E-01)]),("S", [(3.6510018312E+01, 7.8990617244E-02), (1.0541854005E+01, 7.8926384265E-01)]),("S", [(3.6004091387E+00, 1.0000000000E+00)]),("S", [(1.5617023749E+00, 1.0000000000E+00)]),("S", [(4.4997370775E-01, 1.0000000000E+00)]),("S", [(1.8075230237E-01, 1.0000000000E+00)]),("S", [(7.1596696319E-02, 1.0000000000E+00)]),("P", [(7.2240462760E+01, 8.6539170411E-04), (1.6807707208E+01, 6.8762067365E-03), (5.2259410747E+00, 3.0976687148E-02), (1.8508350671E+00, 1.0432357913E-01), (7.2206782663E-01, 2.6164137577E-01)]),("P", [(2.9491018056E-01, 1.0000000000E+00)]),("P", [(1.2201141839E-01, 1.0000000000E+00)]),("P", [(4.9865393625E-02, 1.0000000000E+00)]),("D", [(1.1100000000E+00, 1.0000000000E+00)]),("D", [(4.0200000000E-01, 1.0000000000E+00)]),("D", [(1.4500000000E-01, 1.0000000000E+00)]),("F", [(8.8200000000E-01, 1.0000000000E+00)]),("F", [(3.1100000000E-01, 1.0000000000E+00)]),("G", [(6.7300000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(6.7025071029E+04, 3.8736308501E-05), (1.0039986538E+04, 3.0107917575E-04), (2.2849316911E+03, 1.5787918095E-03), (6.4714122130E+02, 6.6087087195E-03), (2.1109472335E+02, 2.3367123250E-02), (7.6177643862E+01, 7.0420716898E-02), (2.9633839163E+01, 1.7360344953E-01), (1.2187785081E+01, 3.2292305648E-01)]),("S", [(5.3026006299E+01, 7.4897404492E-02), (1.5258502776E+01, 7.6136220983E-01)]),("S", [(5.2403957464E+00, 1.0000000000E+00)]),("S", [(2.2905022379E+00, 1.0000000000E+00)]),("S", [(6.9673283006E-01, 1.0000000000E+00)]),("S", [(2.7599337363E-01, 1.0000000000E+00)]),("S", [(1.0739884389E-01, 1.0000000000E+00)]),("P", [(1.0512555082E+02, 8.4647553844E-04), (2.4884461066E+01, 6.6274038534E-03), (7.8637230826E+00, 3.0120390419E-02), (2.8407001835E+00, 9.9951435476E-02), (1.1227137335E+00, 2.3826299282E-01)]),("P", [(4.6050725555E-01, 1.0000000000E+00)]),("P", [(1.8937530913E-01, 1.0000000000E+00)]),("P", [(7.5983791611E-02, 1.0000000000E+00)]),("D", [(1.8480000000E+00, 1.0000000000E+00)]),("D", [(6.4900000000E-01, 1.0000000000E+00)]),("D", [(2.2800000000E-01, 1.0000000000E+00)]),("F", [(1.4190000000E+00, 1.0000000000E+00)]),("F", [(4.8500000000E-01, 1.0000000000E+00)]),("G", [(1.0110000000E+00, 1.0000000000E+00)])],
    7 : [("S", [(9.0726889210E+04, 3.9257887368E-05), (1.3590528801E+04, 3.0513316455E-04), (3.0929883781E+03, 1.6000560446E-03), (8.7599876362E+02, 6.6982937306E-03), (2.8574469982E+02, 2.3690078765E-02), (1.0311913417E+02, 7.1455405268E-02), (4.0128556777E+01, 1.7632774876E-01), (1.6528095704E+01, 3.2677592815E-01)]),("S", [(6.9390960983E+01, 8.0052094386E-02), (2.0428200596E+01, 7.8268063538E-01)]),("S", [(7.1292587972E+00, 1.0000000000E+00)]),("S", [(3.1324304893E+00, 1.0000000000E+00)]),("S", [(9.8755778723E-01, 1.0000000000E+00)]),("S", [(3.8765721307E-01, 1.0000000000E+00)]),("S", [(1.4909883075E-01, 1.0000000000E+00)]),("P", [(1.5005742670E+02, -8.6216165986E-04), (3.5491599483E+01, -6.8571273236E-03), (1.1247864223E+01, -3.1795688855E-02), (4.0900305195E+00, -1.0537396822E-01), (1.6220573146E+00, -2.4519708041E-01)]),("P", [(6.6442261530E-01, 1.0000000000E+00)]),("P", [(2.7099770070E-01, 1.0000000000E+00)]),("P", [(1.0688749984E-01, 1.0000000000E+00)]),("D", [(2.8370000000E+00, 1.0000000000E+00)]),("D", [(9.6800000000E-01, 1.0000000000E+00)]),("D", [(3.3500000000E-01, 1.0000000000E+00)]),("F", [(2.0270000000E+00, 1.0000000000E+00)]),("F", [(6.8500000000E-01, 1.0000000000E+00)]),("G", [(1.4270000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(1.1650646908E+05, 4.0383857939E-05), (1.7504349724E+04, 3.1255139004E-04), (3.9934513230E+03, 1.6341473495E-03), (1.1330063186E+03, 6.8283224757E-03), (3.6999569594E+02, 2.4124410221E-02), (1.3362074349E+02, 7.2730206154E-02), (5.2035643649E+01, 1.7934429892E-01), (2.1461939313E+01, 3.3059588895E-01)]),("S", [(8.9835051252E+01, 9.6468652996E-02), (2.6428010844E+01, 9.4117481120E-01)]),("S", [(9.2822824649E+00, 1.0000000000E+00)]),("S", [(4.0947728533E+00, 1.0000000000E+00)]),("S", [(1.3255349078E+00, 1.0000000000E+00)]),("S", [(5.1877230787E-01, 1.0000000000E+00)]),("S", [(1.9772676454E-01, 1.0000000000E+00)]),("P", [(1.9115255810E+02, 2.5115697705E-03), (4.5233356739E+01, 2.0039240864E-02), (1.4353465922E+01, 9.3609064762E-02), (5.2422371832E+00, 3.0618127124E-01), (2.0792418599E+00, 6.7810501439E-01)]),("P", [(8.4282371424E-01, 1.0000000000E+00)]),("P", [(3.3617694891E-01, 1.0000000000E+00)]),("P", [(1.2863997974E-01, 1.0000000000E+00)]),("D", [(3.7750000000E+00, 1.0000000000E+00)]),("D", [(1.3000000000E+00, 1.0000000000E+00)]),("D", [(4.4400000000E-01, 1.0000000000E+00)]),("F", [(2.6660000000E+00, 1.0000000000E+00)]),("F", [(8.5900000000E-01, 1.0000000000E+00)]),("G", [(1.8460000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.3253597345E+05, 4.7387482743E-05), (1.9758112588E+04, 3.7070120897E-04), (4.4851996947E+03, 1.9450784713E-03), (1.2738151020E+03, 8.0573291994E-03), (4.1893831236E+02, 2.7992880781E-02), (1.5255721985E+02, 8.2735120175E-02), (5.9821524823E+01, 1.9854169012E-01), (2.4819076932E+01, 3.4860632233E-01)]),("S", [(1.0074446673E+02, 1.0505068816E-01), (3.0103728290E+01, 9.4068472434E-01)]),("S", [(1.0814283272E+01, 1.0000000000E+00)]),("S", [(4.8172886770E+00, 1.0000000000E+00)]),("S", [(1.6559334213E+00, 1.0000000000E+00)]),("S", [(6.4893519582E-01, 1.0000000000E+00)]),("S", [(2.4778104545E-01, 1.0000000000E+00)]),("P", [(2.4096654114E+02, 3.0389933451E-03), (5.7020699781E+01, 2.4357738582E-02), (1.8126952120E+01, 1.1442925768E-01), (6.6457404621E+00, 3.7064659853E-01), (2.6375722892E+00, 7.9791551766E-01)]),("P", [(1.0638217200E+00, 1.0000000000E+00)]),("P", [(4.1932562750E-01, 1.0000000000E+00)]),("P", [(1.5747588299E-01, 1.0000000000E+00)]),("D", [(5.0140000000E+00, 1.0000000000E+00)]),("D", [(1.7250000000E+00, 1.0000000000E+00)]),("D", [(5.8600000000E-01, 1.0000000000E+00)]),("F", [(3.5620000000E+00, 1.0000000000E+00)]),("F", [(1.1480000000E+00, 1.0000000000E+00)]),("G", [(2.3760000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.6067627955E+05, 4.7387482743E-05), (2.3953195039E+04, 3.7070120897E-04), (5.4375063711E+03, 1.9450784713E-03), (1.5442741024E+03, 8.0573291994E-03), (5.0788814269E+02, 2.7992880781E-02), (1.8494847750E+02, 8.2735120175E-02), (7.2522952039E+01, 1.9854169012E-01), (3.0088713575E+01, 3.4860632233E-01)]),("S", [(1.2068434940E+02, 1.0400597014E-01), (3.6074294857E+01, 9.1999988986E-01)]),("S", [(1.3052091210E+01, 1.0000000000E+00)]),("S", [(5.7974783353E+00, 1.0000000000E+00)]),("S", [(2.0478626895E+00, 1.0000000000E+00)]),("S", [(8.0214571994E-01, 1.0000000000E+00)]),("S", [(3.0548672205E-01, 1.0000000000E+00)]),("P", [(4.9843397466E+02, 4.3118261050E-04), (1.1814109217E+02, 3.6206397703E-03), (3.8032529735E+01, 1.8245344081E-02), (1.4183731950E+01, 6.5133035572E-02), (5.7864190721E+00, 1.6708915089E-01), (2.4599622160E+00, 2.9268019613E-01)]),("P", [(1.0421999498E+00, 1.0000000000E+00)]),("P", [(4.2987748916E-01, 1.0000000000E+00)]),("P", [(1.6889708453E-01, 1.0000000000E+00)]),("D", [(6.4710000000E+00, 1.0000000000E+00)]),("D", [(2.2130000000E+00, 1.0000000000E+00)]),("D", [(7.4700000000E-01, 1.0000000000E+00)]),("F", [(4.6570000000E+00, 1.0000000000E+00)]),("F", [(1.5240000000E+00, 1.0000000000E+00)]),("G", [(2.9830000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.7985220081E+05, 2.0671384468E-05), (5.6886006378E+04, 1.6070466617E-04), (1.2942701838E+04, 8.4462905848E-04), (3.6643017904E+03, 3.5519026029E-03), (1.1947417499E+03, 1.2754034468E-02), (4.3098192917E+02, 3.9895462742E-02), (1.6783169424E+02, 1.0720154498E-01), (6.9306669040E+01, 2.3339516913E-01), (2.9951170886E+01, 3.6333077287E-01), (1.3380791097E+01, 3.0544770974E-01)]),("S", [(1.2174011283E+02, 3.6142427284E-02), (3.7044143387E+01, 2.8820961687E-01), (1.3995422624E+01, 7.9337384869E-01)]),("S", [(5.9827797428E+00, 1.0000000000E+00)]),("S", [(2.4830455118E+00, 1.0000000000E+00)]),("S", [(1.0452506187E+00, 1.0000000000E+00)]),("S", [(4.3875640383E-01, 1.0000000000E+00)]),("S", [(6.5595633185E-02, 1.0000000000E+00)]),("S", [(3.0561925072E-02, 1.0000000000E+00)]),("S", [(1.5509064018E-02, 1.0000000000E+00)]),("P", [(6.9077627017E+02, 3.7478518415E-04), (1.6382806121E+02, 3.1775441030E-03), (5.2876460769E+01, 1.6333581338E-02), (1.9812270493E+01, 5.9754902585E-02), (8.1320378784E+00, 1.5879328812E-01), (3.4969068377E+00, 2.9049363260E-01), (1.5117244146E+00, 3.6368131139E-01), (6.4479294912E-01, 2.8195867334E-01)]),("P", [(2.6145823312E-01, 1.0000000000E+00)]),("P", [(1.1704726116E-01, 1.0000000000E+00)]),("P", [(4.0494747666E-02, 1.0000000000E+00)]),("P", [(1.5666707355E-02, 1.0000000000E+00)]),("D", [(5.3000000000E+00, 1.0000000000E+00)]),("D", [(1.5900000000E+00, 1.0000000000E+00)]),("D", [(2.3000000000E-01, 1.0000000000E+00)]),("D", [(7.5700000000E-02, 1.0000000000E+00)]),("F", [(4.2700000000E+00, 1.0000000000E+00)]),("F", [(1.3500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(6.0596778753E+05, 1.4430176617E-05), (9.0569094692E+04, 1.1249871807E-04), (2.0574252844E+04, 5.9264972450E-04), (5.8186284865E+03, 2.4988770424E-03), (1.8956296075E+03, 9.0230775141E-03), (6.8345941021E+02, 2.8579919806E-02), (2.6618219762E+02, 7.9064453923E-02), (1.1011220010E+02, 1.8269337814E-01), (4.7777041234E+01, 3.2157193967E-01), (2.1542166149E+01, 3.5028259389E-01)]),("S", [(1.7412136378E+02, 2.2931111278E-02), (5.3484972498E+01, 1.9151777552E-01), (2.0500213307E+01, 6.1155711055E-01)]),("S", [(9.8056826922E+00, 1.0000000000E+00)]),("S", [(3.5402989348E+00, 1.0000000000E+00)]),("S", [(1.5450850046E+00, 1.0000000000E+00)]),("S", [(6.6540195432E-01, 1.0000000000E+00)]),("S", [(1.4355439071E-01, 1.0000000000E+00)]),("S", [(6.6624673468E-02, 1.0000000000E+00)]),("S", [(2.9772648266E-02, 1.0000000000E+00)]),("P", [(8.9320460829E+02, 3.4958267428E-04), (2.1178258286E+02, 2.9811890111E-03), (6.8443200537E+01, 1.5517845758E-02), (2.5727265349E+01, 5.7578659557E-02), (1.0606634281E+01, 1.5610307853E-01), (4.5934126484E+00, 2.9230912580E-01), (2.0100469807E+00, 3.7219024072E-01), (8.7384841536E-01, 2.7578088891E-01)]),("P", [(3.5615076666E-01, 1.0000000000E+00)]),("P", [(1.8995954356E-01, 1.0000000000E+00)]),("P", [(7.4580135669E-02, 1.0000000000E+00)]),("P", [(2.9221641309E-02, 1.0000000000E+00)]),("D", [(6.9300000000E+00, 1.0000000000E+00)]),("D", [(2.1000000000E+00, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)]),("D", [(5.2000000000E-02, 1.0000000000E+00)]),("F", [(5.6000000000E+00, 1.0000000000E+00)]),("F", [(1.6000000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(7.5455078265E+05, 1.3421335886E-05), (1.1299938922E+05, 1.0433210786E-04), (2.5715831759E+04, 5.4841772080E-04), (7.2836030283E+03, 2.3089081391E-03), (2.3760008796E+03, 8.3309974593E-03), (8.5765468087E+02, 2.6417963653E-02), (3.3438922598E+02, 7.3443646426E-02), (1.3848504731E+02, 1.7184039139E-01), (6.0150368808E+01, 3.1041980030E-01), (2.7127610860E+01, 3.5669190596E-01)]),("S", [(2.2536500065E+02, 2.1522039783E-02), (6.9341968124E+01, 1.8531780493E-01), (2.6619335712E+01, 6.3533181245E-01)]),("S", [(1.2349420671E+01, 1.0000000000E+00)]),("S", [(4.5878785994E+00, 1.0000000000E+00)]),("S", [(2.0571338103E+00, 1.0000000000E+00)]),("S", [(9.0864549395E-01, 1.0000000000E+00)]),("S", [(2.4226988038E-01, 1.0000000000E+00)]),("S", [(1.1248136869E-01, 1.0000000000E+00)]),("S", [(4.8156492817E-02, 1.0000000000E+00)]),("P", [(1.4896119522E+03, 2.0177122443E-04), (3.5301399267E+02, 1.7508109203E-03), (1.1440764069E+02, 9.4424704664E-03), (4.3312186111E+01, 3.6868004006E-02), (1.8027322216E+01, 1.0892874007E-01), (7.9675432403E+00, 2.3265901004E-01), (3.6090399541E+00, 3.4643587076E-01), (1.6456081630E+00, 3.3440809866E-01)]),("P", [(3.4731187489E+01, 1.7823073020E-02), (1.2553083630E+00, -5.9991263926E-01)]),("P", [(7.3635534282E-01, 1.0000000000E+00)]),("P", [(2.4686465559E-01, 1.0000000000E+00)]),("P", [(9.4821971828E-02, 1.0000000000E+00)]),("P", [(3.6214165654E-02, 1.0000000000E+00)]),("D", [(1.9700000000E+00, 1.0000000000E+00)]),("D", [(4.3700000000E-01, 1.0000000000E+00)]),("D", [(1.9500000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-02, 1.0000000000E+00)]),("F", [(4.0100000000E-01, 1.0000000000E+00)]),("F", [(1.5400000000E-01, 1.0000000000E+00)]),("G", [(3.5700000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(9.1807069565E+05, 1.2658787192E-05), (1.3748525386E+05, 9.8408387461E-05), (3.1287772714E+04, 5.1731398244E-04), (8.8616105697E+03, 2.1785441933E-03), (2.8906943156E+03, 7.8656853716E-03), (1.0434063979E+03, 2.4987553674E-02), (4.0680160276E+02, 6.9761818844E-02), (1.6848360207E+02, 1.6473362733E-01), (7.3185628823E+01, 3.0285602647E-01), (3.2998485420E+01, 3.6007260439E-01)]),("S", [(2.7878751325E+02, 2.0185463814E-02), (8.5910228722E+01, 1.7720406747E-01), (3.2992604031E+01, 6.3587860906E-01)]),("S", [(1.5033693254E+01, 1.0000000000E+00)]),("S", [(5.7257514773E+00, 1.0000000000E+00)]),("S", [(2.6146521029E+00, 1.0000000000E+00)]),("S", [(1.1757833152E+00, 1.0000000000E+00)]),("S", [(3.5432330868E-01, 1.0000000000E+00)]),("S", [(1.6248335806E-01, 1.0000000000E+00)]),("S", [(6.8332457181E-02, 1.0000000000E+00)]),("P", [(1.7758850516E+03, 2.0186971126E-04), (4.2083786849E+02, 1.7544314655E-03), (1.3642290243E+02, 9.5039532290E-03), (5.1700991737E+01, 3.7325741590E-02), (2.1559456002E+01, 1.1085396329E-01), (9.5559200095E+00, 2.3756407046E-01), (4.3529819473E+00, 3.5295783750E-01), (2.0096964381E+00, 3.2885931947E-01)]),("P", [(4.6418153780E+01, 2.5858819154E-02), (1.9076417796E+00, -5.9874865899E-01)]),("P", [(9.2337221587E-01, 1.0000000000E+00)]),("P", [(3.4515720370E-01, 1.0000000000E+00)]),("P", [(1.3656834761E-01, 1.0000000000E+00)]),("P", [(5.2987060586E-02, 1.0000000000E+00)]),("D", [(2.6450000000E+00, 1.0000000000E+00)]),("D", [(6.0800000000E-01, 1.0000000000E+00)]),("D", [(2.7200000000E-01, 1.0000000000E+00)]),("D", [(1.1300000000E-01, 1.0000000000E+00)]),("F", [(5.4100000000E-01, 1.0000000000E+00)]),("F", [(2.1200000000E-01, 1.0000000000E+00)]),("G", [(4.6100000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.0905617138E+06, 1.2142449664E-05), (1.6331639461E+05, 9.4395292614E-05), (3.7166607451E+04, 4.9622262177E-04), (1.0526880945E+04, 2.0900040747E-03), (3.4339976028E+03, 7.5489230025E-03), (1.2395360480E+03, 2.4010423937E-02), (4.8327456199E+02, 6.7231473697E-02), (2.0016911586E+02, 1.5978669881E-01), (8.6960394829E+01, 2.9735906782E-01), (3.9211283369E+01, 3.6187171850E-01)]),("S", [(3.3675883662E+02, 1.9154721050E-02), (1.0372179793E+02, 1.7134079100E-01), (3.9771861240E+01, 6.3689655985E-01)]),("S", [(1.7888612952E+01, 1.0000000000E+00)]),("S", [(6.9644556879E+00, 1.0000000000E+00)]),("S", [(3.2198092087E+00, 1.0000000000E+00)]),("S", [(1.4669943979E+00, 1.0000000000E+00)]),("S", [(4.7765437532E-01, 1.0000000000E+00)]),("S", [(2.1637789241E-01, 1.0000000000E+00)]),("S", [(9.0235894336E-02, 1.0000000000E+00)]),("P", [(2.0196711374E+03, 2.1359172406E-04), (4.7860125090E+02, 1.8568771399E-03), (1.5514942504E+02, 1.0070690115E-02), (5.8816356575E+01, 3.9605153679E-02), (2.4544512785E+01, 1.1736067844E-01), (1.0883571061E+01, 2.4950540642E-01), (4.9624791285E+00, 3.6421287984E-01), (2.3002912343E+00, 3.1764127123E-01)]),("P", [(5.9371345016E+01, 3.9432918004E-01), (3.0694590986E+00, -6.3522960431E+00)]),("P", [(1.0634401739E+00, 1.0000000000E+00)]),("P", [(4.5022152161E-01, 1.0000000000E+00)]),("P", [(1.8267271344E-01, 1.0000000000E+00)]),("P", [(7.1610333771E-02, 1.0000000000E+00)]),("D", [(3.3430000000E+00, 1.0000000000E+00)]),("D", [(8.0700000000E-01, 1.0000000000E+00)]),("D", [(3.6500000000E-01, 1.0000000000E+00)]),("D", [(1.5400000000E-01, 1.0000000000E+00)]),("F", [(7.0300000000E-01, 1.0000000000E+00)]),("F", [(2.8000000000E-01, 1.0000000000E+00)]),("G", [(5.9700000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(1.2734109023E+06, 1.1767088246E-05), (1.9069783007E+05, 9.1478610166E-05), (4.3397885330E+04, 4.8090078640E-04), (1.2291809677E+04, 2.0257193592E-03), (4.0097420824E+03, 7.3190096406E-03), (1.4473531030E+03, 2.3300499900E-02), (5.6430102913E+02, 6.5386213610E-02), (2.3374506243E+02, 1.5614449910E-01), (1.0156402814E+02, 2.9318563787E-01), (4.5805907187E+01, 3.6287914289E-01)]),("S", [(3.9427281503E+02, 1.8753305081E-02), (1.2172249591E+02, 1.6870726663E-01), (4.6754125963E+01, 6.3806830653E-01)]),("S", [(2.0923008254E+01, 1.0000000000E+00)]),("S", [(8.2685567800E+00, 1.0000000000E+00)]),("S", [(3.8629345671E+00, 1.0000000000E+00)]),("S", [(1.7794684781E+00, 1.0000000000E+00)]),("S", [(6.1064260103E-01, 1.0000000000E+00)]),("S", [(2.7412269445E-01, 1.0000000000E+00)]),("S", [(1.1325939107E-01, 1.0000000000E+00)]),("P", [(2.1898930459E+03, 2.3912552864E-04), (5.1894596592E+02, 2.0772032158E-03), (1.6819560151E+02, 1.1242420571E-02), (6.3745282788E+01, 4.4069933941E-02), (2.6597033077E+01, 1.2918778608E-01), (1.1774251449E+01, 2.6910820167E-01), (5.3534379024E+00, 3.7855928620E-01), (2.4701911802E+00, 2.9692134655E-01)]),("P", [(8.2120288349E+01, -3.9420318847E-02), (4.9523532869E+00, 6.4048403090E-01)]),("P", [(1.0828262029E+00, 1.0000000000E+00)]),("P", [(4.9271277356E-01, 1.0000000000E+00)]),("P", [(2.0483450942E-01, 1.0000000000E+00)]),("P", [(8.0743615716E-02, 1.0000000000E+00)]),("D", [(4.1590000000E+00, 1.0000000000E+00)]),("D", [(1.0190000000E+00, 1.0000000000E+00)]),("D", [(4.6400000000E-01, 1.0000000000E+00)]),("D", [(1.9400000000E-01, 1.0000000000E+00)]),("F", [(8.6900000000E-01, 1.0000000000E+00)]),("F", [(3.3500000000E-01, 1.0000000000E+00)]),("G", [(6.8300000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.4674590095E+06, 1.1478257194E-05), (2.1975616433E+05, 8.9234299775E-05), (5.0010770301E+04, 4.6911086186E-04), (1.4164823918E+04, 1.9762446133E-03), (4.6207465525E+03, 7.1419937783E-03), (1.6678991635E+03, 2.2753219445E-02), (6.5029199265E+02, 6.3959782953E-02), (2.6938037376E+02, 1.5331059238E-01), (1.1706752106E+02, 2.8986952417E-01), (5.2811766843E+01, 3.6348071452E-01)]),("S", [(4.6142769988E+02, 1.8019457578E-02), (1.4212665355E+02, 1.6489442314E-01), (5.4437838768E+01, 6.3891587584E-01)]),("S", [(2.4160770219E+01, 1.0000000000E+00)]),("S", [(9.7083540306E+00, 1.0000000000E+00)]),("S", [(4.5640696733E+00, 1.0000000000E+00)]),("S", [(2.1194744832E+00, 1.0000000000E+00)]),("S", [(7.5722365394E-01, 1.0000000000E+00)]),("S", [(3.3747224597E-01, 1.0000000000E+00)]),("S", [(1.3860775149E-01, 1.0000000000E+00)]),("P", [(2.5019457890E+03, 2.4242618410E-04), (5.9288059285E+02, 2.1079961749E-03), (1.9218089186E+02, 1.1432693869E-02), (7.2875710488E+01, 4.4956698060E-02), (3.0436358370E+01, 1.3197476145E-01), (1.3490178902E+01, 2.7493639225E-01), (6.1478071413E+00, 3.8347236372E-01), (2.8450944820E+00, 2.8871943885E-01)]),("P", [(1.0539397936E+02, -3.4311760144E-02), (6.7369738513E+00, 6.4060818902E-01)]),("P", [(1.2421095772E+00, 1.0000000000E+00)]),("P", [(5.5669714254E-01, 1.0000000000E+00)]),("P", [(2.3387801464E-01, 1.0000000000E+00)]),("P", [(9.3164490890E-02, 1.0000000000E+00)]),("D", [(5.1910000000E+00, 1.0000000000E+00)]),("D", [(1.2760000000E+00, 1.0000000000E+00)]),("D", [(5.8300000000E-01, 1.0000000000E+00)]),("D", [(2.4300000000E-01, 1.0000000000E+00)]),("F", [(1.0890000000E+00, 1.0000000000E+00)]),("F", [(4.2300000000E-01, 1.0000000000E+00)]),("G", [(8.2700000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.6734219494E+06, 1.1245990413E-05), (2.5060175373E+05, 8.7428115798E-05), (5.7030912120E+04, 4.5961641275E-04), (1.6153303915E+04, 1.9363890742E-03), (5.2694109288E+03, 6.9994046497E-03), (1.9020315541E+03, 2.2312068196E-02), (7.4157677159E+02, 6.2808077697E-02), (3.0720901906E+02, 1.5101140997E-01), (1.3352786203E+02, 2.8713661177E-01), (6.0253381291E+01, 3.6385490107E-01)]),("S", [(5.2202426206E+02, 1.8071323116E-02), (1.6151290469E+02, 1.6449219884E-01), (6.2126369433E+01, 6.3897334899E-01)]),("S", [(2.7590930012E+01, 1.0000000000E+00)]),("S", [(1.1175528881E+01, 1.0000000000E+00)]),("S", [(5.2959065137E+00, 1.0000000000E+00)]),("S", [(2.4782798895E+00, 1.0000000000E+00)]),("S", [(9.1262095059E-01, 1.0000000000E+00)]),("S", [(4.0426067549E-01, 1.0000000000E+00)]),("S", [(1.6562886306E-01, 1.0000000000E+00)]),("P", [(1.2878765667E+02, -3.0509457751E-02), (8.4357867977E+00, 6.4101621070E-01)]),("P", [(2.8684504581E+03, 2.4010435121E-04), (6.7971869590E+02, 2.0904788665E-03), (2.2036758824E+02, 1.1369316839E-02), (8.3620439734E+01, 4.4901977398E-02), (3.4964322657E+01, 1.3237156770E-01), (1.5525131784E+01, 2.7709277600E-01), (7.0992979806E+00, 3.8613979646E-01), (3.3021336602E+00, 2.8492925372E-01)]),("P", [(1.4618661855E+00, 1.0000000000E+00)]),("P", [(6.4658010096E-01, 1.0000000000E+00)]),("P", [(2.7229295755E-01, 1.0000000000E+00)]),("P", [(1.0905182449E-01, 1.0000000000E+00)]),("D", [(6.3150000000E+00, 1.0000000000E+00)]),("D", [(1.5620000000E+00, 1.0000000000E+00)]),("D", [(7.1500000000E-01, 1.0000000000E+00)]),("D", [(2.9700000000E-01, 1.0000000000E+00)]),("F", [(1.3250000000E+00, 1.0000000000E+00)]),("F", [(5.4300000000E-01, 1.0000000000E+00)]),("G", [(1.0070000000E+00, 1.0000000000E+00)])],
}





DEF2_QZVPPD = {

    1 : [("S", [(1.9069169000E+02, 7.0815167000E-04), (2.8605532000E+01, 5.4678827000E-03), (6.5095943000E+00, 2.7966605000E-02), (1.8412455000E+00, 1.0764538000E-01)]),("S", [(5.9853725000E-01, 1.0000000000E+00)]),("S", [(2.1397624000E-01, 1.0000000000E+00)]),("S", [(8.0316286000E-02, 1.0000000000E+00)]),("P", [(2.2920000000E+00, 1.0000000000E+00)]),("P", [(8.3800000000E-01, 1.0000000000E+00)]),("P", [(2.9200000000E-01, 1.0000000000E+00)]),("P", [(8.4063199228E-02, 1.0000000000E+00)]),("D", [(2.0620000000E+00, 1.0000000000E+00)]),("D", [(6.6200000000E-01, 1.0000000000E+00)]),("F", [(1.3970000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(1.1446470809E+03, 3.5861578618E-04), (1.7164596667E+02, 2.7725434466E-03), (3.9066056254E+01, 1.4241892216E-02), (1.1051401989E+01, 5.5457352277E-02), (3.5725574473E+00, 1.6170511810E-01)]),("S", [(1.2429415962E+00, 3.3177883833E-01)]),("S", [(4.4807668730E-01, 4.2014033068E-01)]),("S", [(1.6411579128E-01, 1.8652273014E-01)]),("P", [(5.9940000000E+00, 1.0000000000E+00)]),("P", [(1.7450000000E+00, 1.0000000000E+00)]),("P", [(5.6000000000E-01, 1.0000000000E+00)]),("P", [(1.6298923122E-01, 1.0000000000E+00)]),("D", [(4.2990000000E+00, 1.0000000000E+00)]),("D", [(1.2230000000E+00, 1.0000000000E+00)]),("F", [(2.6800000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(1.4853977085E+04, 4.2711388815E-05), (2.2252236477E+03, 3.3235310800E-04), (5.0488739008E+02, 1.7518436649E-03), (1.4245847548E+02, 7.3477995850E-03), (4.6315599580E+01, 2.5899837683E-02), (1.6655335474E+01, 7.6670682431E-02), (6.4331186199E+00, 1.8276075765E-01), (2.6027043858E+00, 3.2655434038E-01), (1.0897245405E+00, 3.7000429828E-01)]),("S", [(4.4236595971E+00, 1.1120987921E-01), (1.2356394990E+00, 7.9987335862E-01)]),("S", [(4.6067470602E-01, 1.0000000000E+00)]),("S", [(9.6617166955E-02, 1.0000000000E+00)]),("S", [(4.5915452296E-02, 1.0000000000E+00)]),("S", [(2.1140048730E-02, 1.0000000000E+00)]),("P", [(3.3270000000E+00, 1.0000000000E+00)]),("P", [(3.2605109206E+00, 8.6504749023E-03), (6.5003043115E-01, 4.7614123736E-02), (1.6941671073E-01, 2.1001138000E-01)]),("P", [(5.5732343767E-02, 1.0000000000E+00)]),("P", [(2.0489959241E-02, 1.0000000000E+00)]),("D", [(2.3000000000E-01, 1.0000000000E+00)]),("D", [(7.5700000000E-02, 1.0000000000E+00)]),("F", [(1.3500000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(2.9646704407E+04, 3.7943320869E-05), (4.4287614354E+03, 2.9605352634E-04), (1.0054701332E+03, 1.5572477356E-03), (2.8415339578E+02, 6.5323195769E-03), (9.2504356461E+01, 2.3116861005E-02), (3.3311016058E+01, 6.9458515249E-02), (1.2911553314E+01, 1.7014371331E-01), (5.2665497361E+00, 3.1682711374E-01)]),("S", [(2.2995386380E+01, 8.1791122747E-02), (6.6236354176E+00, 8.0000038023E-01)]),("S", [(2.2289595003E+00, 1.0000000000E+00)]),("S", [(9.5302433451E-01, 1.0000000000E+00)]),("S", [(2.4651602436E-01, 1.0000000000E+00)]),("S", [(1.0153895315E-01, 1.0000000000E+00)]),("S", [(4.1551374604E-02, 1.0000000000E+00)]),("P", [(6.4000000000E+00, 1.0000000000E+00)]),("P", [(1.4099789445E+01, 3.8508871145E-03), (3.1803188482E+00, 2.4152380771E-02), (9.0489220496E-01, 9.7926925156E-02), (3.0411585508E-01, 2.9470380030E-01)]),("P", [(1.1302606657E-01, 1.0000000000E+00)]),("P", [(4.2831733691E-02, 1.0000000000E+00)]),("D", [(3.6000000000E-01, 1.0000000000E+00)]),("D", [(9.0000000000E-02, 1.0000000000E+00)]),("F", [(2.9000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(4.6447667056E+04, 3.8388681014E-05), (6.9576889042E+03, 2.9836770197E-04), (1.5834428403E+03, 1.5645478806E-03), (4.4846601009E+02, 6.5476770742E-03), (1.4628639262E+02, 2.3139008388E-02), (5.2784386084E+01, 6.9615799357E-02), (2.0519396170E+01, 1.7119636876E-01), (8.4185659258E+00, 3.1913191837E-01)]),("S", [(3.6510018312E+01, 7.8990617244E-02), (1.0541854005E+01, 7.8926384265E-01)]),("S", [(3.6004091387E+00, 1.0000000000E+00)]),("S", [(1.5617023749E+00, 1.0000000000E+00)]),("S", [(4.4997370775E-01, 1.0000000000E+00)]),("S", [(1.8075230237E-01, 1.0000000000E+00)]),("S", [(7.1596696319E-02, 1.0000000000E+00)]),("S", [(2.9941402533E-02, 1.0000000000E+00)]),("P", [(7.2240462760E+01, 8.6539170411E-04), (1.6807707208E+01, 6.8762067365E-03), (5.2259410747E+00, 3.0976687148E-02), (1.8508350671E+00, 1.0432357913E-01), (7.2206782663E-01, 2.6164137577E-01)]),("P", [(2.9491018056E-01, 1.0000000000E+00)]),("P", [(1.2201141839E-01, 1.0000000000E+00)]),("P", [(4.9865393625E-02, 1.0000000000E+00)]),("D", [(1.1100000000E+00, 1.0000000000E+00)]),("D", [(4.0200000000E-01, 1.0000000000E+00)]),("D", [(1.4500000000E-01, 1.0000000000E+00)]),("D", [(5.0896593879E-02, 1.0000000000E+00)]),("F", [(8.8200000000E-01, 1.0000000000E+00)]),("F", [(3.1100000000E-01, 1.0000000000E+00)]),("G", [(6.7300000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(6.7025071029E+04, 3.8736308501E-05), (1.0039986538E+04, 3.0107917575E-04), (2.2849316911E+03, 1.5787918095E-03), (6.4714122130E+02, 6.6087087195E-03), (2.1109472335E+02, 2.3367123250E-02), (7.6177643862E+01, 7.0420716898E-02), (2.9633839163E+01, 1.7360344953E-01), (1.2187785081E+01, 3.2292305648E-01)]),("S", [(5.3026006299E+01, 7.4897404492E-02), (1.5258502776E+01, 7.6136220983E-01)]),("S", [(5.2403957464E+00, 1.0000000000E+00)]),("S", [(2.2905022379E+00, 1.0000000000E+00)]),("S", [(6.9673283006E-01, 1.0000000000E+00)]),("S", [(2.7599337363E-01, 1.0000000000E+00)]),("S", [(1.0739884389E-01, 1.0000000000E+00)]),("S", [(4.4981404899E-02, 1.0000000000E+00)]),("P", [(1.0512555082E+02, 8.4647553844E-04), (2.4884461066E+01, 6.6274038534E-03), (7.8637230826E+00, 3.0120390419E-02), (2.8407001835E+00, 9.9951435476E-02), (1.1227137335E+00, 2.3826299282E-01)]),("P", [(4.6050725555E-01, 1.0000000000E+00)]),("P", [(1.8937530913E-01, 1.0000000000E+00)]),("P", [(7.5983791611E-02, 1.0000000000E+00)]),("D", [(1.8480000000E+00, 1.0000000000E+00)]),("D", [(6.4900000000E-01, 1.0000000000E+00)]),("D", [(2.2800000000E-01, 1.0000000000E+00)]),("D", [(7.6889830417E-02, 1.0000000000E+00)]),("F", [(1.4190000000E+00, 1.0000000000E+00)]),("F", [(4.8500000000E-01, 1.0000000000E+00)]),("G", [(1.0110000000E+00, 1.0000000000E+00)])],
    7 : [("S", [(9.0726889210E+04, 3.9257887368E-05), (1.3590528801E+04, 3.0513316455E-04), (3.0929883781E+03, 1.6000560446E-03), (8.7599876362E+02, 6.6982937306E-03), (2.8574469982E+02, 2.3690078765E-02), (1.0311913417E+02, 7.1455405268E-02), (4.0128556777E+01, 1.7632774876E-01), (1.6528095704E+01, 3.2677592815E-01)]),("S", [(6.9390960983E+01, 8.0052094386E-02), (2.0428200596E+01, 7.8268063538E-01)]),("S", [(7.1292587972E+00, 1.0000000000E+00)]),("S", [(3.1324304893E+00, 1.0000000000E+00)]),("S", [(9.8755778723E-01, 1.0000000000E+00)]),("S", [(3.8765721307E-01, 1.0000000000E+00)]),("S", [(1.4909883075E-01, 1.0000000000E+00)]),("S", [(6.2151630318E-02, 1.0000000000E+00)]),("P", [(1.5005742670E+02, -8.6216165986E-04), (3.5491599483E+01, -6.8571273236E-03), (1.1247864223E+01, -3.1795688855E-02), (4.0900305195E+00, -1.0537396822E-01), (1.6220573146E+00, -2.4519708041E-01)]),("P", [(6.6442261530E-01, 1.0000000000E+00)]),("P", [(2.7099770070E-01, 1.0000000000E+00)]),("P", [(1.0688749984E-01, 1.0000000000E+00)]),("D", [(2.8370000000E+00, 1.0000000000E+00)]),("D", [(9.6800000000E-01, 1.0000000000E+00)]),("D", [(3.3500000000E-01, 1.0000000000E+00)]),("D", [(1.0825280010E-01, 1.0000000000E+00)]),("F", [(2.0270000000E+00, 1.0000000000E+00)]),("F", [(6.8500000000E-01, 1.0000000000E+00)]),("G", [(1.4270000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(1.1650646908E+05, 4.0383857939E-05), (1.7504349724E+04, 3.1255139004E-04), (3.9934513230E+03, 1.6341473495E-03), (1.1330063186E+03, 6.8283224757E-03), (3.6999569594E+02, 2.4124410221E-02), (1.3362074349E+02, 7.2730206154E-02), (5.2035643649E+01, 1.7934429892E-01), (2.1461939313E+01, 3.3059588895E-01)]),("S", [(8.9835051252E+01, 9.6468652996E-02), (2.6428010844E+01, 9.4117481120E-01)]),("S", [(9.2822824649E+00, 1.0000000000E+00)]),("S", [(4.0947728533E+00, 1.0000000000E+00)]),("S", [(1.3255349078E+00, 1.0000000000E+00)]),("S", [(5.1877230787E-01, 1.0000000000E+00)]),("S", [(1.9772676454E-01, 1.0000000000E+00)]),("S", [(6.9638535104E-02, 1.0000000000E+00)]),("P", [(1.9115255810E+02, 2.5115697705E-03), (4.5233356739E+01, 2.0039240864E-02), (1.4353465922E+01, 9.3609064762E-02), (5.2422371832E+00, 3.0618127124E-01), (2.0792418599E+00, 6.7810501439E-01)]),("P", [(8.4282371424E-01, 1.0000000000E+00)]),("P", [(3.3617694891E-01, 1.0000000000E+00)]),("P", [(1.2863997974E-01, 1.0000000000E+00)]),("P", [(4.3598162776E-02, 1.0000000000E+00)]),("D", [(3.7750000000E+00, 1.0000000000E+00)]),("D", [(1.3000000000E+00, 1.0000000000E+00)]),("D", [(4.4400000000E-01, 1.0000000000E+00)]),("D", [(1.2546378695E-01, 1.0000000000E+00)]),("F", [(2.6660000000E+00, 1.0000000000E+00)]),("F", [(8.5900000000E-01, 1.0000000000E+00)]),("G", [(1.8460000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.3253597345E+05, 4.7387482743E-05), (1.9758112588E+04, 3.7070120897E-04), (4.4851996947E+03, 1.9450784713E-03), (1.2738151020E+03, 8.0573291994E-03), (4.1893831236E+02, 2.7992880781E-02), (1.5255721985E+02, 8.2735120175E-02), (5.9821524823E+01, 1.9854169012E-01), (2.4819076932E+01, 3.4860632233E-01)]),("S", [(1.0074446673E+02, 1.0505068816E-01), (3.0103728290E+01, 9.4068472434E-01)]),("S", [(1.0814283272E+01, 1.0000000000E+00)]),("S", [(4.8172886770E+00, 1.0000000000E+00)]),("S", [(1.6559334213E+00, 1.0000000000E+00)]),("S", [(6.4893519582E-01, 1.0000000000E+00)]),("S", [(2.4778104545E-01, 1.0000000000E+00)]),("S", [(8.7626236800E-02, 1.0000000000E+00)]),("P", [(2.4096654114E+02, 3.0389933451E-03), (5.7020699781E+01, 2.4357738582E-02), (1.8126952120E+01, 1.1442925768E-01), (6.6457404621E+00, 3.7064659853E-01), (2.6375722892E+00, 7.9791551766E-01)]),("P", [(1.0638217200E+00, 1.0000000000E+00)]),("P", [(4.1932562750E-01, 1.0000000000E+00)]),("P", [(1.5747588299E-01, 1.0000000000E+00)]),("P", [(4.6772400332E-02, 1.0000000000E+00)]),("D", [(5.0140000000E+00, 1.0000000000E+00)]),("D", [(1.7250000000E+00, 1.0000000000E+00)]),("D", [(5.8600000000E-01, 1.0000000000E+00)]),("D", [(1.5967986245E-01, 1.0000000000E+00)]),("F", [(3.5620000000E+00, 1.0000000000E+00)]),("F", [(1.1480000000E+00, 1.0000000000E+00)]),("G", [(2.3760000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.6067627955E+05, 4.7387482743E-05), (2.3953195039E+04, 3.7070120897E-04), (5.4375063711E+03, 1.9450784713E-03), (1.5442741024E+03, 8.0573291994E-03), (5.0788814269E+02, 2.7992880781E-02), (1.8494847750E+02, 8.2735120175E-02), (7.2522952039E+01, 1.9854169012E-01), (3.0088713575E+01, 3.4860632233E-01)]),("S", [(1.2068434940E+02, 1.0400597014E-01), (3.6074294857E+01, 9.1999988986E-01)]),("S", [(1.3052091210E+01, 1.0000000000E+00)]),("S", [(5.7974783353E+00, 1.0000000000E+00)]),("S", [(2.0478626895E+00, 1.0000000000E+00)]),("S", [(8.0214571994E-01, 1.0000000000E+00)]),("S", [(3.0548672205E-01, 1.0000000000E+00)]),("S", [(1.0621243030E-01, 1.0000000000E+00)]),("P", [(4.9843397466E+02, 4.3118261050E-04), (1.1814109217E+02, 3.6206397703E-03), (3.8032529735E+01, 1.8245344081E-02), (1.4183731950E+01, 6.5133035572E-02), (5.7864190721E+00, 1.6708915089E-01), (2.4599622160E+00, 2.9268019613E-01)]),("P", [(1.0421999498E+00, 1.0000000000E+00)]),("P", [(4.2987748916E-01, 1.0000000000E+00)]),("P", [(1.6889708453E-01, 1.0000000000E+00)]),("P", [(4.7169461232E-02, 1.0000000000E+00)]),("D", [(6.4710000000E+00, 1.0000000000E+00)]),("D", [(2.2130000000E+00, 1.0000000000E+00)]),("D", [(7.4700000000E-01, 1.0000000000E+00)]),("D", [(1.9790459442E-01, 1.0000000000E+00)]),("F", [(4.6570000000E+00, 1.0000000000E+00)]),("F", [(1.5240000000E+00, 1.0000000000E+00)]),("G", [(2.9830000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.7985220081E+05, 2.0671384468E-05), (5.6886006378E+04, 1.6070466617E-04), (1.2942701838E+04, 8.4462905848E-04), (3.6643017904E+03, 3.5519026029E-03), (1.1947417499E+03, 1.2754034468E-02), (4.3098192917E+02, 3.9895462742E-02), (1.6783169424E+02, 1.0720154498E-01), (6.9306669040E+01, 2.3339516913E-01), (2.9951170886E+01, 3.6333077287E-01), (1.3380791097E+01, 3.0544770974E-01)]),("S", [(1.2174011283E+02, 3.6142427284E-02), (3.7044143387E+01, 2.8820961687E-01), (1.3995422624E+01, 7.9337384869E-01)]),("S", [(5.9827797428E+00, 1.0000000000E+00)]),("S", [(2.4830455118E+00, 1.0000000000E+00)]),("S", [(1.0452506187E+00, 1.0000000000E+00)]),("S", [(4.3875640383E-01, 1.0000000000E+00)]),("S", [(6.5595633185E-02, 1.0000000000E+00)]),("S", [(3.0561925072E-02, 1.0000000000E+00)]),("S", [(1.5509064018E-02, 1.0000000000E+00)]),("P", [(6.9077627017E+02, 3.7478518415E-04), (1.6382806121E+02, 3.1775441030E-03), (5.2876460769E+01, 1.6333581338E-02), (1.9812270493E+01, 5.9754902585E-02), (8.1320378784E+00, 1.5879328812E-01), (3.4969068377E+00, 2.9049363260E-01), (1.5117244146E+00, 3.6368131139E-01), (6.4479294912E-01, 2.8195867334E-01)]),("P", [(2.6145823312E-01, 1.0000000000E+00)]),("P", [(1.1704726116E-01, 1.0000000000E+00)]),("P", [(4.0494747666E-02, 1.0000000000E+00)]),("P", [(1.5666707355E-02, 1.0000000000E+00)]),("D", [(5.3000000000E+00, 1.0000000000E+00)]),("D", [(1.5900000000E+00, 1.0000000000E+00)]),("D", [(2.3000000000E-01, 1.0000000000E+00)]),("D", [(7.5700000000E-02, 1.0000000000E+00)]),("F", [(4.2700000000E+00, 1.0000000000E+00)]),("F", [(1.3500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(6.0596778753E+05, 1.4430176617E-05), (9.0569094692E+04, 1.1249871807E-04), (2.0574252844E+04, 5.9264972450E-04), (5.8186284865E+03, 2.4988770424E-03), (1.8956296075E+03, 9.0230775141E-03), (6.8345941021E+02, 2.8579919806E-02), (2.6618219762E+02, 7.9064453923E-02), (1.1011220010E+02, 1.8269337814E-01), (4.7777041234E+01, 3.2157193967E-01), (2.1542166149E+01, 3.5028259389E-01)]),("S", [(1.7412136378E+02, 2.2931111278E-02), (5.3484972498E+01, 1.9151777552E-01), (2.0500213307E+01, 6.1155711055E-01)]),("S", [(9.8056826922E+00, 1.0000000000E+00)]),("S", [(3.5402989348E+00, 1.0000000000E+00)]),("S", [(1.5450850046E+00, 1.0000000000E+00)]),("S", [(6.6540195432E-01, 1.0000000000E+00)]),("S", [(1.4355439071E-01, 1.0000000000E+00)]),("S", [(6.6624673468E-02, 1.0000000000E+00)]),("S", [(2.9772648266E-02, 1.0000000000E+00)]),("P", [(8.9320460829E+02, 3.4958267428E-04), (2.1178258286E+02, 2.9811890111E-03), (6.8443200537E+01, 1.5517845758E-02), (2.5727265349E+01, 5.7578659557E-02), (1.0606634281E+01, 1.5610307853E-01), (4.5934126484E+00, 2.9230912580E-01), (2.0100469807E+00, 3.7219024072E-01), (8.7384841536E-01, 2.7578088891E-01)]),("P", [(3.5615076666E-01, 1.0000000000E+00)]),("P", [(1.8995954356E-01, 1.0000000000E+00)]),("P", [(7.4580135669E-02, 1.0000000000E+00)]),("P", [(2.9221641309E-02, 1.0000000000E+00)]),("D", [(6.9300000000E+00, 1.0000000000E+00)]),("D", [(2.1000000000E+00, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)]),("D", [(5.2000000000E-02, 1.0000000000E+00)]),("F", [(5.6000000000E+00, 1.0000000000E+00)]),("F", [(1.6000000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(7.5455078265E+05, 1.3421335886E-05), (1.1299938922E+05, 1.0433210786E-04), (2.5715831759E+04, 5.4841772080E-04), (7.2836030283E+03, 2.3089081391E-03), (2.3760008796E+03, 8.3309974593E-03), (8.5765468087E+02, 2.6417963653E-02), (3.3438922598E+02, 7.3443646426E-02), (1.3848504731E+02, 1.7184039139E-01), (6.0150368808E+01, 3.1041980030E-01), (2.7127610860E+01, 3.5669190596E-01)]),("S", [(2.2536500065E+02, 2.1522039783E-02), (6.9341968124E+01, 1.8531780493E-01), (2.6619335712E+01, 6.3533181245E-01)]),("S", [(1.2349420671E+01, 1.0000000000E+00)]),("S", [(4.5878785994E+00, 1.0000000000E+00)]),("S", [(2.0571338103E+00, 1.0000000000E+00)]),("S", [(9.0864549395E-01, 1.0000000000E+00)]),("S", [(2.4226988038E-01, 1.0000000000E+00)]),("S", [(1.1248136869E-01, 1.0000000000E+00)]),("S", [(4.8156492817E-02, 1.0000000000E+00)]),("S", [(2.1134849794E-02, 1.0000000000E+00)]),("P", [(1.4896119522E+03, 2.0177122443E-04), (3.5301399267E+02, 1.7508109203E-03), (1.1440764069E+02, 9.4424704664E-03), (4.3312186111E+01, 3.6868004006E-02), (1.8027322216E+01, 1.0892874007E-01), (7.9675432403E+00, 2.3265901004E-01), (3.6090399541E+00, 3.4643587076E-01), (1.6456081630E+00, 3.3440809866E-01)]),("P", [(3.4731187489E+01, 1.7823073020E-02), (1.2553083630E+00, -5.9991263926E-01)]),("P", [(7.3635534282E-01, 1.0000000000E+00)]),("P", [(2.4686465559E-01, 1.0000000000E+00)]),("P", [(9.4821971828E-02, 1.0000000000E+00)]),("P", [(3.6214165654E-02, 1.0000000000E+00)]),("D", [(1.9700000000E+00, 1.0000000000E+00)]),("D", [(4.3700000000E-01, 1.0000000000E+00)]),("D", [(1.9500000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-02, 1.0000000000E+00)]),("D", [(3.0664004967E-02, 1.0000000000E+00)]),("F", [(4.0100000000E-01, 1.0000000000E+00)]),("F", [(1.5400000000E-01, 1.0000000000E+00)]),("G", [(3.5700000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(9.1807069565E+05, 1.2658787192E-05), (1.3748525386E+05, 9.8408387461E-05), (3.1287772714E+04, 5.1731398244E-04), (8.8616105697E+03, 2.1785441933E-03), (2.8906943156E+03, 7.8656853716E-03), (1.0434063979E+03, 2.4987553674E-02), (4.0680160276E+02, 6.9761818844E-02), (1.6848360207E+02, 1.6473362733E-01), (7.3185628823E+01, 3.0285602647E-01), (3.2998485420E+01, 3.6007260439E-01)]),("S", [(2.7878751325E+02, 2.0185463814E-02), (8.5910228722E+01, 1.7720406747E-01), (3.2992604031E+01, 6.3587860906E-01)]),("S", [(1.5033693254E+01, 1.0000000000E+00)]),("S", [(5.7257514773E+00, 1.0000000000E+00)]),("S", [(2.6146521029E+00, 1.0000000000E+00)]),("S", [(1.1757833152E+00, 1.0000000000E+00)]),("S", [(3.5432330868E-01, 1.0000000000E+00)]),("S", [(1.6248335806E-01, 1.0000000000E+00)]),("S", [(6.8332457181E-02, 1.0000000000E+00)]),("S", [(3.0521078871E-02, 1.0000000000E+00)]),("P", [(1.7758850516E+03, 2.0186971126E-04), (4.2083786849E+02, 1.7544314655E-03), (1.3642290243E+02, 9.5039532290E-03), (5.1700991737E+01, 3.7325741590E-02), (2.1559456002E+01, 1.1085396329E-01), (9.5559200095E+00, 2.3756407046E-01), (4.3529819473E+00, 3.5295783750E-01), (2.0096964381E+00, 3.2885931947E-01)]),("P", [(4.6418153780E+01, 2.5858819154E-02), (1.9076417796E+00, -5.9874865899E-01)]),("P", [(9.2337221587E-01, 1.0000000000E+00)]),("P", [(3.4515720370E-01, 1.0000000000E+00)]),("P", [(1.3656834761E-01, 1.0000000000E+00)]),("P", [(5.2987060586E-02, 1.0000000000E+00)]),("D", [(2.6450000000E+00, 1.0000000000E+00)]),("D", [(6.0800000000E-01, 1.0000000000E+00)]),("D", [(2.7200000000E-01, 1.0000000000E+00)]),("D", [(1.1300000000E-01, 1.0000000000E+00)]),("D", [(4.4018904228E-02, 1.0000000000E+00)]),("F", [(5.4100000000E-01, 1.0000000000E+00)]),("F", [(2.1200000000E-01, 1.0000000000E+00)]),("G", [(4.6100000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.0905617138E+06, 1.2142449664E-05), (1.6331639461E+05, 9.4395292614E-05), (3.7166607451E+04, 4.9622262177E-04), (1.0526880945E+04, 2.0900040747E-03), (3.4339976028E+03, 7.5489230025E-03), (1.2395360480E+03, 2.4010423937E-02), (4.8327456199E+02, 6.7231473697E-02), (2.0016911586E+02, 1.5978669881E-01), (8.6960394829E+01, 2.9735906782E-01), (3.9211283369E+01, 3.6187171850E-01)]),("S", [(3.3675883662E+02, 1.9154721050E-02), (1.0372179793E+02, 1.7134079100E-01), (3.9771861240E+01, 6.3689655985E-01)]),("S", [(1.7888612952E+01, 1.0000000000E+00)]),("S", [(6.9644556879E+00, 1.0000000000E+00)]),("S", [(3.2198092087E+00, 1.0000000000E+00)]),("S", [(1.4669943979E+00, 1.0000000000E+00)]),("S", [(4.7765437532E-01, 1.0000000000E+00)]),("S", [(2.1637789241E-01, 1.0000000000E+00)]),("S", [(9.0235894336E-02, 1.0000000000E+00)]),("S", [(4.0836372978E-02, 1.0000000000E+00)]),("P", [(2.0196711374E+03, 2.1359172406E-04), (4.7860125090E+02, 1.8568771399E-03), (1.5514942504E+02, 1.0070690115E-02), (5.8816356575E+01, 3.9605153679E-02), (2.4544512785E+01, 1.1736067844E-01), (1.0883571061E+01, 2.4950540642E-01), (4.9624791285E+00, 3.6421287984E-01), (2.3002912343E+00, 3.1764127123E-01)]),("P", [(5.9371345016E+01, 3.9432918004E-01), (3.0694590986E+00, -6.3522960431E+00)]),("P", [(1.0634401739E+00, 1.0000000000E+00)]),("P", [(4.5022152161E-01, 1.0000000000E+00)]),("P", [(1.8267271344E-01, 1.0000000000E+00)]),("P", [(7.1610333771E-02, 1.0000000000E+00)]),("D", [(3.3430000000E+00, 1.0000000000E+00)]),("D", [(8.0700000000E-01, 1.0000000000E+00)]),("D", [(3.6500000000E-01, 1.0000000000E+00)]),("D", [(1.5400000000E-01, 1.0000000000E+00)]),("D", [(5.9687731846E-02, 1.0000000000E+00)]),("F", [(7.0300000000E-01, 1.0000000000E+00)]),("F", [(2.8000000000E-01, 1.0000000000E+00)]),("G", [(5.9700000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(1.2734109023E+06, 1.1767088246E-05), (1.9069783007E+05, 9.1478610166E-05), (4.3397885330E+04, 4.8090078640E-04), (1.2291809677E+04, 2.0257193592E-03), (4.0097420824E+03, 7.3190096406E-03), (1.4473531030E+03, 2.3300499900E-02), (5.6430102913E+02, 6.5386213610E-02), (2.3374506243E+02, 1.5614449910E-01), (1.0156402814E+02, 2.9318563787E-01), (4.5805907187E+01, 3.6287914289E-01)]),("S", [(3.9427281503E+02, 1.8753305081E-02), (1.2172249591E+02, 1.6870726663E-01), (4.6754125963E+01, 6.3806830653E-01)]),("S", [(2.0923008254E+01, 1.0000000000E+00)]),("S", [(8.2685567800E+00, 1.0000000000E+00)]),("S", [(3.8629345671E+00, 1.0000000000E+00)]),("S", [(1.7794684781E+00, 1.0000000000E+00)]),("S", [(6.1064260103E-01, 1.0000000000E+00)]),("S", [(2.7412269445E-01, 1.0000000000E+00)]),("S", [(1.1325939107E-01, 1.0000000000E+00)]),("S", [(4.4392624881E-02, 1.0000000000E+00)]),("P", [(2.1898930459E+03, 2.3912552864E-04), (5.1894596592E+02, 2.0772032158E-03), (1.6819560151E+02, 1.1242420571E-02), (6.3745282788E+01, 4.4069933941E-02), (2.6597033077E+01, 1.2918778608E-01), (1.1774251449E+01, 2.6910820167E-01), (5.3534379024E+00, 3.7855928620E-01), (2.4701911802E+00, 2.9692134655E-01)]),("P", [(8.2120288349E+01, -3.9420318847E-02), (4.9523532869E+00, 6.4048403090E-01)]),("P", [(1.0828262029E+00, 1.0000000000E+00)]),("P", [(4.9271277356E-01, 1.0000000000E+00)]),("P", [(2.0483450942E-01, 1.0000000000E+00)]),("P", [(8.0743615716E-02, 1.0000000000E+00)]),("P", [(2.5661157833E-02, 1.0000000000E+00)]),("D", [(4.1590000000E+00, 1.0000000000E+00)]),("D", [(1.0190000000E+00, 1.0000000000E+00)]),("D", [(4.6400000000E-01, 1.0000000000E+00)]),("D", [(1.9400000000E-01, 1.0000000000E+00)]),("D", [(6.7889829366E-02, 1.0000000000E+00)]),("F", [(8.6900000000E-01, 1.0000000000E+00)]),("F", [(3.3500000000E-01, 1.0000000000E+00)]),("G", [(6.8300000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.4674590095E+06, 1.1478257194E-05), (2.1975616433E+05, 8.9234299775E-05), (5.0010770301E+04, 4.6911086186E-04), (1.4164823918E+04, 1.9762446133E-03), (4.6207465525E+03, 7.1419937783E-03), (1.6678991635E+03, 2.2753219445E-02), (6.5029199265E+02, 6.3959782953E-02), (2.6938037376E+02, 1.5331059238E-01), (1.1706752106E+02, 2.8986952417E-01), (5.2811766843E+01, 3.6348071452E-01)]),("S", [(4.6142769988E+02, 1.8019457578E-02), (1.4212665355E+02, 1.6489442314E-01), (5.4437838768E+01, 6.3891587584E-01)]),("S", [(2.4160770219E+01, 1.0000000000E+00)]),("S", [(9.7083540306E+00, 1.0000000000E+00)]),("S", [(4.5640696733E+00, 1.0000000000E+00)]),("S", [(2.1194744832E+00, 1.0000000000E+00)]),("S", [(7.5722365394E-01, 1.0000000000E+00)]),("S", [(3.3747224597E-01, 1.0000000000E+00)]),("S", [(1.3860775149E-01, 1.0000000000E+00)]),("S", [(5.4234565513E-02, 1.0000000000E+00)]),("P", [(2.5019457890E+03, 2.4242618410E-04), (5.9288059285E+02, 2.1079961749E-03), (1.9218089186E+02, 1.1432693869E-02), (7.2875710488E+01, 4.4956698060E-02), (3.0436358370E+01, 1.3197476145E-01), (1.3490178902E+01, 2.7493639225E-01), (6.1478071413E+00, 3.8347236372E-01), (2.8450944820E+00, 2.8871943885E-01)]),("P", [(1.0539397936E+02, -3.4311760144E-02), (6.7369738513E+00, 6.4060818902E-01)]),("P", [(1.2421095772E+00, 1.0000000000E+00)]),("P", [(5.5669714254E-01, 1.0000000000E+00)]),("P", [(2.3387801464E-01, 1.0000000000E+00)]),("P", [(9.3164490890E-02, 1.0000000000E+00)]),("P", [(2.7256146018E-02, 1.0000000000E+00)]),("D", [(5.1910000000E+00, 1.0000000000E+00)]),("D", [(1.2760000000E+00, 1.0000000000E+00)]),("D", [(5.8300000000E-01, 1.0000000000E+00)]),("D", [(2.4300000000E-01, 1.0000000000E+00)]),("D", [(8.3456538402E-02, 1.0000000000E+00)]),("F", [(1.0890000000E+00, 1.0000000000E+00)]),("F", [(4.2300000000E-01, 1.0000000000E+00)]),("G", [(8.2700000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.6734219494E+06, 1.1245990413E-05), (2.5060175373E+05, 8.7428115798E-05), (5.7030912120E+04, 4.5961641275E-04), (1.6153303915E+04, 1.9363890742E-03), (5.2694109288E+03, 6.9994046497E-03), (1.9020315541E+03, 2.2312068196E-02), (7.4157677159E+02, 6.2808077697E-02), (3.0720901906E+02, 1.5101140997E-01), (1.3352786203E+02, 2.8713661177E-01), (6.0253381291E+01, 3.6385490107E-01)]),("S", [(5.2202426206E+02, 1.8071323116E-02), (1.6151290469E+02, 1.6449219884E-01), (6.2126369433E+01, 6.3897334899E-01)]),("S", [(2.7590930012E+01, 1.0000000000E+00)]),("S", [(1.1175528881E+01, 1.0000000000E+00)]),("S", [(5.2959065137E+00, 1.0000000000E+00)]),("S", [(2.4782798895E+00, 1.0000000000E+00)]),("S", [(9.1262095059E-01, 1.0000000000E+00)]),("S", [(4.0426067549E-01, 1.0000000000E+00)]),("S", [(1.6562886306E-01, 1.0000000000E+00)]),("S", [(6.5129364729E-02, 1.0000000000E+00)]),("P", [(1.2878765667E+02, -3.0509457751E-02), (8.4357867977E+00, 6.4101621070E-01)]),("P", [(2.8684504581E+03, 2.4010435121E-04), (6.7971869590E+02, 2.0904788665E-03), (2.2036758824E+02, 1.1369316839E-02), (8.3620439734E+01, 4.4901977398E-02), (3.4964322657E+01, 1.3237156770E-01), (1.5525131784E+01, 2.7709277600E-01), (7.0992979806E+00, 3.8613979646E-01), (3.3021336602E+00, 2.8492925372E-01)]),("P", [(1.4618661855E+00, 1.0000000000E+00)]),("P", [(6.4658010096E-01, 1.0000000000E+00)]),("P", [(2.7229295755E-01, 1.0000000000E+00)]),("P", [(1.0905182449E-01, 1.0000000000E+00)]),("P", [(3.0101149684E-02, 1.0000000000E+00)]),("D", [(6.3150000000E+00, 1.0000000000E+00)]),("D", [(1.5620000000E+00, 1.0000000000E+00)]),("D", [(7.1500000000E-01, 1.0000000000E+00)]),("D", [(2.9700000000E-01, 1.0000000000E+00)]),("D", [(1.0105028218E-01, 1.0000000000E+00)]),("F", [(1.3250000000E+00, 1.0000000000E+00)]),("F", [(5.4300000000E-01, 1.0000000000E+00)]),("G", [(1.0070000000E+00, 1.0000000000E+00)])],
}



_6_31GBRADCOMMAPKET =  {

    1 : [("S", [(1.8731136960E+01, 3.3494604340E-02), (2.8253943650E+00, 2.3472695350E-01), (6.4012169230E-01, 8.1375732610E-01)]),("S", [(1.6127775880E-01, 1.0000000000E+00)]),("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 4.0139739350E-02), (5.7780300000E+00, 2.6124609700E-01), (1.2417740000E+00, 7.9318462460E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)]),("P", [(1.1000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.4241891500E+02, 2.1426078100E-03), (9.6798515300E+01, 1.6208871500E-02), (2.2091121200E+01, 7.7315572500E-02), (6.2010702500E+00, 2.4578605200E-01), (1.9351176800E+00, 4.7018900400E-01), (6.3673578900E-01, 3.4547084500E-01)]),("S", [(2.3249184080E+00, -3.5091745740E-02), (6.3243035560E-01, -1.9123284310E-01), (7.9053434750E-02, 1.0839877950E+00)]),("P", [(2.3249184080E+00, 8.9415080430E-03), (6.3243035560E-01, 1.4100946400E-01), (7.9053434750E-02, 9.4536369530E-01)]),("S", [(3.5961971750E-02, 1.0000000000E+00)]),("P", [(3.5961971750E-02, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645856900E+03, 1.9447575900E-03), (1.8993680600E+02, 1.4835052000E-02), (4.3159089000E+01, 7.2090546290E-02), (1.2098662700E+01, 2.3715415000E-01), (3.8063232200E+00, 4.6919865190E-01), (1.2728903000E+00, 3.5652022790E-01)]),("S", [(3.1964630980E+00, -1.1264872850E-01), (7.4781330380E-01, -2.2950640790E-01), (2.1996633020E-01, 1.1869167640E+00)]),("P", [(3.1964630980E+00, 5.5980199800E-02), (7.4781330380E-01, 2.6155061100E-01), (2.1996633020E-01, 7.9397233890E-01)]),("S", [(8.2309900700E-02, 1.0000000000E+00)]),("P", [(8.2309900700E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688822500E+03, 1.8662745900E-03), (3.1064957000E+02, 1.4251481700E-02), (7.0683033000E+01, 6.9551618500E-02), (1.9861080300E+01, 2.3257293300E-01), (6.2993048400E+00, 4.6707871200E-01), (2.1270269700E+00, 3.6343144000E-01)]),("S", [(4.7279710710E+00, -1.3039379740E-01), (1.1903377360E+00, -1.3078895140E-01), (3.5941168290E-01, 1.1309444840E+00)]),("P", [(4.7279710710E+00, 7.4597579920E-02), (1.1903377360E+00, 3.0784667710E-01), (3.5941168290E-01, 7.4345683420E-01)]),("S", [(1.2675124690E-01, 1.0000000000E+00)]),("P", [(1.2675124690E-01, 1.0000000000E+00)]),("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475248800E+03, 1.8347371320E-03), (4.5736951800E+02, 1.4037322810E-02), (1.0394868500E+02, 6.8842622260E-02), (2.9210155300E+01, 2.3218444320E-01), (9.2866629600E+00, 4.6794134840E-01), (3.1639269600E+00, 3.6231198530E-01)]),("S", [(7.8682723500E+00, -1.1933241980E-01), (1.8812885400E+00, -1.6085415170E-01), (5.4424925800E-01, 1.1434564380E+00)]),("P", [(7.8682723500E+00, 6.8999066590E-02), (1.8812885400E+00, 3.1642396100E-01), (5.4424925800E-01, 7.4430829090E-01)]),("S", [(1.6871447820E-01, 1.0000000000E+00)]),("P", [(1.6871447820E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735114600E+03, 1.8347721600E-03), (6.2745791100E+02, 1.3994627000E-02), (1.4290209300E+02, 6.8586551810E-02), (4.0234329300E+01, 2.3224087300E-01), (1.2820212900E+01, 4.6906994810E-01), (4.3904370100E+00, 3.6045519910E-01)]),("S", [(1.1626361860E+01, -1.1496118170E-01), (2.7162798070E+00, -1.6911747860E-01), (7.7221839660E-01, 1.1458519470E+00)]),("P", [(1.1626361860E+01, 6.7579743880E-02), (2.7162798070E+00, 3.2390729590E-01), (7.7221839660E-01, 7.4089513980E-01)]),("S", [(2.1203149750E-01, 1.0000000000E+00)]),("P", [(2.1203149750E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(5.4846716600E+03, 1.8310744300E-03), (8.2523494600E+02, 1.3950172200E-02), (1.8804695800E+02, 6.8445078100E-02), (5.2964500000E+01, 2.3271433600E-01), (1.6897570400E+01, 4.7019289800E-01), (5.7996353400E+00, 3.5852085300E-01)]),("S", [(1.5539616250E+01, -1.1077754950E-01), (3.5999335860E+00, -1.4802626270E-01), (1.0137617500E+00, 1.1307670150E+00)]),("P", [(1.5539616250E+01, 7.0874268230E-02), (3.5999335860E+00, 3.3975283910E-01), (1.0137617500E+00, 7.2715857730E-01)]),("S", [(2.7000582260E-01, 1.0000000000E+00)]),("P", [(2.7000582260E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169010E-03), (1.0513660900E+03, 1.3916079610E-02), (2.3928569000E+02, 6.8405324530E-02), (6.7397445300E+01, 2.3318576010E-01), (2.1519957300E+01, 4.7126743920E-01), (7.4031013000E+00, 3.5661854620E-01)]),("S", [(2.0847952800E+01, -1.0850697510E-01), (4.8083083400E+00, -1.4645165810E-01), (1.3440698600E+00, 1.1286885810E+00)]),("P", [(2.0847952800E+01, 7.1628724240E-02), (4.8083083400E+00, 3.4591210270E-01), (1.3440698600E+00, 7.2246995640E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843480500E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623310E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712610E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828720E-01), (6.1017550100E+00, -1.4616382130E-01), (1.6962715300E+00, 1.1277735030E+00)]),("P", [(2.6532131000E+01, 7.1909588510E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051210E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9376592770E-03), (1.4998900000E+03, 1.4806994480E-02), (3.4195100000E+02, 7.2705472880E-02), (9.4679600000E+01, 2.5262890580E-01), (2.9734500000E+01, 4.9324181600E-01), (1.0006300000E+01, 3.1316888320E-01)]),("S", [(1.5096300000E+02, -3.5420835040E-03), (3.5587800000E+01, -4.3958843480E-02), (1.1168300000E+01, -1.0975210860E-01), (3.9020100000E+00, 1.8739818540E-01), (1.3817700000E+00, 6.4669963970E-01), (4.6638200000E-01, 3.0605830270E-01)]),("P", [(1.5096300000E+02, 5.0016597100E-03), (3.5587800000E+01, 3.5510897940E-02), (1.1168300000E+01, 1.4282499170E-01), (3.9020100000E+00, 3.3861998030E-01), (1.3817700000E+00, 4.5157897380E-01), (4.6638200000E-01, 2.7327098410E-01)]),("S", [(4.9796600000E-01, -2.4850315930E-01), (8.4352900000E-02, -1.3170408440E-01), (6.6635000000E-02, 1.2335207910E+00)]),("P", [(4.9796600000E-01, -2.3022500430E-02), (8.4352900000E-02, 9.5035901760E-01), (6.6635000000E-02, 5.9857901110E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778293170E-03), (1.7599300000E+03, 1.5113994780E-02), (4.0084600000E+02, 7.3910774480E-02), (1.1280700000E+02, 2.4919091400E-01), (3.5999700000E+01, 4.8792783160E-01), (1.2182800000E+01, 3.1966188960E-01)]),("S", [(1.8918000000E+02, -3.2371704710E-03), (4.5211900000E+01, -4.1007905970E-02), (1.4356300000E+01, -1.1260001640E-01), (5.1388600000E+00, 1.4863302160E-01), (1.9065200000E+00, 6.1649708980E-01), (7.0588700000E-01, 3.6482905310E-01)]),("P", [(1.8918000000E+02, 4.9281299210E-03), (4.5211900000E+01, 3.4988799440E-02), (1.4356300000E+01, 1.4072499770E-01), (5.1388600000E+00, 3.3364199470E-01), (1.9065200000E+00, 4.4493999290E-01), (7.0588700000E-01, 2.6925399570E-01)]),("S", [(9.2934000000E-01, -2.1229089850E-01), (2.6903500000E-01, -1.0798545700E-01), (1.1737900000E-01, 1.1758449770E+00)]),("P", [(9.2934000000E-01, -2.2419181230E-02), (2.6903500000E-01, 1.9227083900E-01), (1.1737900000E-01, 8.4618029160E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426699470E-03), (2.0987500000E+03, 1.4859899590E-02), (4.7770500000E+02, 7.2849398000E-02), (1.3436000000E+02, 2.4682999320E-01), (4.2870900000E+01, 4.8725798660E-01), (1.4518900000E+01, 3.2349599110E-01)]),("S", [(2.3966800000E+02, -2.9261900280E-03), (5.7441900000E+01, -3.7408300360E-02), (1.8285900000E+01, -1.1448700110E-01), (6.5991400000E+00, 1.1563500110E-01), (2.4904900000E+00, 6.1259500580E-01), (9.4454500000E-01, 3.9379900370E-01)]),("P", [(2.3966800000E+02, 4.6028455820E-03), (5.7441900000E+01, 3.3198968130E-02), (1.8285900000E+01, 1.3628186920E-01), (6.5991400000E+00, 3.3047568280E-01), (2.4904900000E+00, 4.4914556890E-01), (9.4454500000E-01, 2.6570374500E-01)]),("S", [(1.2779000000E+00, -2.2760692450E-01), (3.9759000000E-01, 1.4458358730E-03), (1.6009500000E-01, 1.0927944390E+00)]),("P", [(1.2779000000E+00, -1.7512601890E-02), (3.9759000000E-01, 2.4453302640E-01), (1.6009500000E-01, 8.0493408670E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594802160E-03), (2.4255800000E+03, 1.4928801640E-02), (5.5386700000E+02, 7.2847808010E-02), (1.5634000000E+02, 2.4613002710E-01), (5.0068300000E+01, 4.8591405350E-01), (1.7017800000E+01, 3.2500203580E-01)]),("S", [(2.9271800000E+02, -2.7809414150E-03), (6.9873100000E+01, -3.5714618170E-02), (2.2336300000E+01, -1.1498505850E-01), (8.1503900000E+00, 9.3563447600E-02), (3.1345800000E+00, 6.0301730680E-01), (1.2254300000E+00, 4.1895921310E-01)]),("P", [(2.9271800000E+02, 4.4382645210E-03), (6.9873100000E+01, 3.2667933280E-02), (2.2336300000E+01, 1.3472113720E-01), (8.1503900000E+00, 3.2867833480E-01), (3.1345800000E+00, 4.4964045800E-01), (1.2254300000E+00, 2.6137226620E-01)]),("S", [(1.7273800000E+00, -2.4463100420E-01), (5.7292200000E-01, 4.3157377170E-03), (2.2219200000E-01, 1.0981845080E+00)]),("P", [(1.7273800000E+00, -1.7795106050E-02), (5.7292200000E-01, 2.5353908630E-01), (2.2219200000E-01, 8.0066927240E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8515989230E-03), (2.9094200000E+03, 1.4206191740E-02), (6.6136400000E+02, 6.9999459280E-02), (1.8575900000E+02, 2.4007886030E-01), (5.9194300000E+01, 4.8476171800E-01), (2.0031000000E+01, 3.3519980500E-01)]),("S", [(3.3947800000E+02, -2.7821701050E-03), (8.1010100000E+01, -3.6049901350E-02), (2.5878000000E+01, -1.1663100440E-01), (9.4522100000E+00, 9.6832803640E-02), (3.6656600000E+00, 6.1441802310E-01), (1.4674600000E+00, 4.0379801520E-01)]),("P", [(3.3947800000E+02, 4.5646161910E-03), (8.1010100000E+01, 3.3693571880E-02), (2.5878000000E+01, 1.3975488340E-01), (9.4522100000E+00, 3.3936171680E-01), (3.6656600000E+00, 4.5092062370E-01), (1.4674600000E+00, 2.3858580090E-01)]),("S", [(2.1562300000E+00, -2.5292411390E-01), (7.4899700000E-01, 3.2851844680E-02), (2.8314500000E-01, 1.0812547620E+00)]),("P", [(2.1562300000E+00, -1.7765312730E-02), (7.4899700000E-01, 2.7405819640E-01), (2.8314500000E-01, 7.8542156300E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8692408490E-03), (3.3014900000E+03, 1.4230306460E-02), (7.5414600000E+02, 6.9696231660E-02), (2.1271100000E+02, 2.3848710830E-01), (6.7989600000E+01, 4.8330721950E-01), (2.3051500000E+01, 3.3807415360E-01)]),("S", [(4.2373500000E+02, -2.3767704990E-03), (1.0071000000E+02, -3.1693006650E-02), (3.2159900000E+01, -1.1331702380E-01), (1.1807900000E+01, 5.6090011770E-02), (4.6311000000E+00, 5.9225512430E-01), (1.8702500000E+00, 4.5500609550E-01)]),("P", [(4.2373500000E+02, 4.0610099820E-03), (1.0071000000E+02, 3.0681299860E-02), (3.2159900000E+01, 1.3045199940E-01), (1.1807900000E+01, 3.2720499850E-01), (4.6311000000E+00, 4.5285099800E-01), (1.8702500000E+00, 2.5604199890E-01)]),("S", [(2.6158400000E+00, -2.5037311420E-01), (9.2216700000E-01, 6.6956763100E-02), (3.4128700000E-01, 1.0545062690E+00)]),("P", [(2.6158400000E+00, -1.4510489550E-02), (9.2216700000E-01, 3.1026277650E-01), (3.4128700000E-01, 7.5448245650E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8329598480E-03), (3.7803500000E+03, 1.4034198830E-02), (8.6047400000E+02, 6.9097394260E-02), (2.4214500000E+02, 2.3745198030E-01), (7.7334900000E+01, 4.8303395990E-01), (2.6247000000E+01, 3.3985597180E-01)]),("S", [(4.9176500000E+02, -2.2973914170E-03), (1.1698400000E+02, -3.0713718940E-02), (3.7415300000E+01, -1.1252806940E-01), (1.3783400000E+01, 4.5016327760E-02), (5.4521500000E+00, 5.8935336340E-01), (2.2258800000E+00, 4.6520628680E-01)]),("P", [(4.9176500000E+02, 3.9894008790E-03), (1.1698400000E+02, 3.0317706680E-02), (3.7415300000E+01, 1.2988002860E-01), (1.3783400000E+01, 3.2795107230E-01), (5.4521500000E+00, 4.5352710000E-01), (2.2258800000E+00, 2.5215405560E-01)]),("S", [(3.1864900000E+00, -2.5182802800E-01), (1.1442700000E+00, 6.1589251410E-02), (4.2037700000E-01, 1.0601843280E+00)]),("P", [(3.1864900000E+00, -1.4299314720E-02), (1.1442700000E+00, 3.2357233310E-01), (4.2037700000E-01, 7.4350776530E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252601920E-03), (4.2576200000E+03, 1.3968601470E-02), (9.6985700000E+02, 6.8707307230E-02), (2.7326300000E+02, 2.3620402490E-01), (8.7369500000E+01, 4.8221405080E-01), (2.9686700000E+01, 3.4204303600E-01)]),("S", [(5.7589100000E+02, -2.1597208950E-03), (1.3681600000E+02, -2.9077512060E-02), (4.3809800000E+01, -1.1082704600E-01), (1.6209400000E+01, 2.7699911480E-02), (6.4608400000E+00, 5.7761323950E-01), (2.6511400000E+00, 4.8868820260E-01)]),("P", [(5.7589100000E+02, 3.8066498420E-03), (1.3681600000E+02, 2.9230498790E-02), (4.3809800000E+01, 1.2646699480E-01), (1.6209400000E+01, 3.2350998660E-01), (6.4608400000E+00, 4.5489598110E-01), (2.6511400000E+00, 2.5662998940E-01)]),("S", [(3.8602800000E+00, -2.5559296040E-01), (1.4137300000E+00, 3.7806742060E-02), (5.1664600000E-01, 1.0805640600E+00)]),("P", [(3.8602800000E+00, -1.5919690400E-02), (1.4137300000E+00, 3.2464580420E-01), (5.1664600000E-01, 7.4398955120E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}


_6_31GBRA2DFCOMMAPKET = {

    1 : [("S", [(1.8731137000E+01, 3.3494600000E-02), (2.8253937000E+00, 2.3472695000E-01), (6.4012170000E-01, 8.1375733000E-01)]),("S", [(1.6127780000E-01, 1.0000000000E+00)]),("P", [(1.5000000000E+00, 1.0000000000E+00)]),("P", [(3.7500000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 2.3766000000E-02), (5.7780300000E+00, 1.5467900000E-01), (1.2417740000E+00, 4.6963000000E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)]),("P", [(1.5000000000E+00, 1.0000000000E+00)]),("P", [(3.7500000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(6.4241892000E+02, 2.1426000000E-03), (9.6798515000E+01, 1.6208900000E-02), (2.2091121000E+01, 7.7315600000E-02), (6.2010703000E+00, 2.4578600000E-01), (1.9351177000E+00, 4.7018900000E-01), (6.3673580000E-01, 3.4547080000E-01)]),("S", [(2.3249184000E+00, -3.5091700000E-02), (6.3243060000E-01, -1.9123280000E-01), (7.9053400000E-02, 1.0839878000E+00)]),("P", [(2.3249184000E+00, 8.9415000000E-03), (6.3243060000E-01, 1.4100950000E-01), (7.9053400000E-02, 9.4536370000E-01)]),("S", [(3.5962000000E-02, 1.0000000000E+00)]),("P", [(3.5962000000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E-01, 1.0000000000E+00)]),("F", [(1.5000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645857000E+03, 1.9448000000E-03), (1.8993681000E+02, 1.4835100000E-02), (4.3159089000E+01, 7.2090600000E-02), (1.2098663000E+01, 2.3715420000E-01), (3.8063232000E+00, 4.6919870000E-01), (1.2728903000E+00, 3.5652020000E-01)]),("S", [(3.1964631000E+00, -1.1264870000E-01), (7.4781330000E-01, -2.2950640000E-01), (2.1996630000E-01, 1.1869167000E+00)]),("P", [(3.1964631000E+00, 5.5980200000E-02), (7.4781330000E-01, 2.6155060000E-01), (2.1996630000E-01, 7.9397230000E-01)]),("S", [(8.2309900000E-02, 1.0000000000E+00)]),("P", [(8.2309900000E-02, 1.0000000000E+00)]),("D", [(5.1000000000E-01, 1.0000000000E+00)]),("D", [(1.2750000000E-01, 1.0000000000E+00)]),("F", [(2.6000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688823000E+03, 1.8663000000E-03), (3.1064957000E+02, 1.4251500000E-02), (7.0683033000E+01, 6.9551600000E-02), (1.9861080000E+01, 2.3257290000E-01), (6.2993048000E+00, 4.6707870000E-01), (2.1270270000E+00, 3.6343140000E-01)]),("S", [(4.7279710000E+00, -1.3039380000E-01), (1.1903377000E+00, -1.3078890000E-01), (3.5941170000E-01, 1.1309444000E+00)]),("P", [(4.7279710000E+00, 7.4597600000E-02), (1.1903377000E+00, 3.0784670000E-01), (3.5941170000E-01, 7.4345680000E-01)]),("S", [(1.2675120000E-01, 1.0000000000E+00)]),("P", [(1.2675120000E-01, 1.0000000000E+00)]),("D", [(8.0200000000E-01, 1.0000000000E+00)]),("D", [(2.0050000000E-01, 1.0000000000E+00)]),("F", [(5.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475249000E+03, 1.8347000000E-03), (4.5736951000E+02, 1.4037300000E-02), (1.0394869000E+02, 6.8842600000E-02), (2.9210155000E+01, 2.3218440000E-01), (9.2866630000E+00, 4.6794130000E-01), (3.1639270000E+00, 3.6231200000E-01)]),("S", [(7.8682724000E+00, -1.1933240000E-01), (1.8812885000E+00, -1.6085420000E-01), (5.4424930000E-01, 1.1434564000E+00)]),("P", [(7.8682724000E+00, 6.8999100000E-02), (1.8812885000E+00, 3.1642400000E-01), (5.4424930000E-01, 7.4430830000E-01)]),("S", [(1.6871440000E-01, 1.0000000000E+00)]),("P", [(1.6871440000E-01, 1.0000000000E+00)]),("D", [(1.2520000000E+00, 1.0000000000E+00)]),("D", [(3.1300000000E-01, 1.0000000000E+00)]),("F", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735110000E+03, 1.8348000000E-03), (6.2745790000E+02, 1.3995000000E-02), (1.4290210000E+02, 6.8587000000E-02), (4.0234330000E+01, 2.3224100000E-01), (1.2820210000E+01, 4.6907000000E-01), (4.3904370000E+00, 3.6045500000E-01)]),("S", [(1.1626358000E+01, -1.1496100000E-01), (2.7162800000E+00, -1.6911800000E-01), (7.7221800000E-01, 1.1458520000E+00)]),("P", [(1.1626358000E+01, 6.7580000000E-02), (2.7162800000E+00, 3.2390700000E-01), (7.7221800000E-01, 7.4089500000E-01)]),("S", [(2.1203130000E-01, 1.0000000000E+00)]),("P", [(2.1203130000E-01, 1.0000000000E+00)]),("D", [(1.8260000000E+00, 1.0000000000E+00)]),("D", [(4.5650000000E-01, 1.0000000000E+00)]),("F", [(1.0000000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(5.4846717000E+03, 1.8311000000E-03), (8.2523495000E+02, 1.3950100000E-02), (1.8804696000E+02, 6.8445100000E-02), (5.2964500000E+01, 2.3271430000E-01), (1.6897570000E+01, 4.7019300000E-01), (5.7996353000E+00, 3.5852090000E-01)]),("S", [(1.5539616000E+01, -1.1077750000E-01), (3.5999336000E+00, -1.4802630000E-01), (1.0137618000E+00, 1.1307670000E+00)]),("P", [(1.5539616000E+01, 7.0874300000E-02), (3.5999336000E+00, 3.3975280000E-01), (1.0137618000E+00, 7.2715860000E-01)]),("S", [(2.7000580000E-01, 1.0000000000E+00)]),("P", [(2.7000580000E-01, 1.0000000000E+00)]),("D", [(2.5840000000E+00, 1.0000000000E+00)]),("D", [(6.4600000000E-01, 1.0000000000E+00)]),("F", [(1.4000000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169000E-03), (1.0513660900E+03, 1.3916079600E-02), (2.3928569000E+02, 6.8405324500E-02), (6.7397445300E+01, 2.3318576000E-01), (2.1519957300E+01, 4.7126743900E-01), (7.4031013000E+00, 3.5661854600E-01)]),("S", [(2.0847952800E+01, -1.0850697500E-01), (4.8083083400E+00, -1.4645165800E-01), (1.3440698600E+00, 1.1286885800E+00)]),("P", [(2.0847952800E+01, 7.1628724300E-02), (4.8083083400E+00, 3.4591210300E-01), (1.3440698600E+00, 7.2246995700E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("D", [(3.5000000000E+00, 1.0000000000E+00)]),("D", [(8.7500000000E-01, 1.0000000000E+00)]),("F", [(1.8500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843481000E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623300E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712600E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828700E-01), (6.1017550100E+00, -1.4616382100E-01), (1.6962715300E+00, 1.1277735000E+00)]),("P", [(2.6532131000E+01, 7.1909588500E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051200E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("D", [(4.6080000000E+00, 1.0000000000E+00)]),("D", [(1.1520000000E+00, 1.0000000000E+00)]),("F", [(2.5000000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9377000000E-03), (1.4998900000E+03, 1.4807000000E-02), (3.4195100000E+02, 7.2706000000E-02), (9.4679700000E+01, 2.5262900000E-01), (2.9734500000E+01, 4.9324200000E-01), (1.0006300000E+01, 3.1316900000E-01)]),("S", [(1.5096300000E+02, -3.5421000000E-03), (3.5587800000E+01, -4.3959000000E-02), (1.1168300000E+01, -1.0975210000E-01), (3.9020100000E+00, 1.8739800000E-01), (1.3817700000E+00, 6.4669900000E-01), (4.6638200000E-01, 3.0605800000E-01)]),("P", [(1.5096300000E+02, 5.0017000000E-03), (3.5587800000E+01, 3.5511000000E-02), (1.1168300000E+01, 1.4282500000E-01), (3.9020100000E+00, 3.3862000000E-01), (1.3817700000E+00, 4.5157900000E-01), (4.6638200000E-01, 2.7327100000E-01)]),("S", [(4.9796600000E-01, -2.4850300000E-01), (8.4353000000E-02, -1.3170400000E-01), (6.6635000000E-02, 1.2335200000E+00)]),("P", [(4.9796600000E-01, -2.3023000000E-02), (8.4353000000E-02, 9.5035900000E-01), (6.6635000000E-02, 5.9858000000E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("D", [(3.5000000000E-01, 1.0000000000E+00)]),("D", [(8.7500000000E-02, 1.0000000000E+00)]),("F", [(1.5000000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778000000E-03), (1.7599300000E+03, 1.5114000000E-02), (4.0084600000E+02, 7.3911000000E-02), (1.1280700000E+02, 2.4919100000E-01), (3.5999700000E+01, 4.8792800000E-01), (1.2182800000E+01, 3.1966200000E-01)]),("S", [(1.8918000000E+02, -3.2372000000E-03), (4.5211900000E+01, -4.1008000000E-02), (1.4356300000E+01, -1.1260000000E-01), (5.1388600000E+00, 1.4863300000E-01), (1.9065200000E+00, 6.1649700000E-01), (7.0588700000E-01, 3.6482900000E-01)]),("P", [(1.8918000000E+02, 4.9281000000E-03), (4.5211900000E+01, 3.4989000000E-02), (1.4356300000E+01, 1.4072500000E-01), (5.1388600000E+00, 3.3364200000E-01), (1.9065200000E+00, 4.4494000000E-01), (7.0588700000E-01, 2.6925400000E-01)]),("S", [(9.2934000000E-01, -2.1229000000E-01), (2.6903500000E-01, -1.0798500000E-01), (1.1737900000E-01, 1.1758400000E+00)]),("P", [(9.2934000000E-01, -2.2419000000E-02), (2.6903500000E-01, 1.9227000000E-01), (1.1737900000E-01, 8.4618100000E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("D", [(3.5000000000E-01, 1.0000000000E+00)]),("D", [(8.7500000000E-02, 1.0000000000E+00)]),("F", [(2.0000000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426700000E-03), (2.0987500000E+03, 1.4859900000E-02), (4.7770500000E+02, 7.2849400000E-02), (1.3436000000E+02, 2.4683000000E-01), (4.2870900000E+01, 4.8725800000E-01), (1.4518900000E+01, 3.2349600000E-01)]),("S", [(2.3966800000E+02, -2.9261900000E-03), (5.7441900000E+01, -3.7408000000E-02), (1.8285900000E+01, -1.1448700000E-01), (6.5991400000E+00, 1.1563500000E-01), (2.4904900000E+00, 6.1259500000E-01), (9.4454000000E-01, 3.9379900000E-01)]),("P", [(2.3966800000E+02, 4.6028500000E-03), (5.7441900000E+01, 3.3199000000E-02), (1.8285900000E+01, 1.3628200000E-01), (6.5991400000E+00, 3.3047600000E-01), (2.4904900000E+00, 4.4914600000E-01), (9.4454000000E-01, 2.6570400000E-01)]),("S", [(1.2779000000E+00, -2.2760600000E-01), (3.9759000000E-01, 1.4458300000E-03), (1.6009500000E-01, 1.0927900000E+00)]),("P", [(1.2779000000E+00, -1.7513000000E-02), (3.9759000000E-01, 2.4453300000E-01), (1.6009500000E-01, 8.0493400000E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("D", [(1.6250000000E-01, 1.0000000000E+00)]),("F", [(2.5000000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594800000E-03), (2.4255800000E+03, 1.4928800000E-02), (5.5386700000E+02, 7.2847800000E-02), (1.5634000000E+02, 2.4613000000E-01), (5.0068300000E+01, 4.8591400000E-01), (1.7017800000E+01, 3.2500200000E-01)]),("S", [(2.9271800000E+02, -2.7809400000E-03), (6.9873100000E+01, -3.5714600000E-02), (2.2336300000E+01, -1.1498500000E-01), (8.1503900000E+00, 9.3563400000E-02), (3.1345800000E+00, 6.0301700000E-01), (1.2254300000E+00, 4.1895900000E-01)]),("P", [(2.9271800000E+02, 4.4382600000E-03), (6.9873100000E+01, 3.2667900000E-02), (2.2336300000E+01, 1.3472100000E-01), (8.1503900000E+00, 3.2867800000E-01), (3.1345800000E+00, 4.4964000000E-01), (1.2254300000E+00, 2.6137200000E-01)]),("S", [(1.7273800000E+00, -2.4463000000E-01), (5.7292200000E-01, 4.3157200000E-03), (2.2219200000E-01, 1.0981800000E+00)]),("P", [(1.7273800000E+00, -1.7795100000E-02), (5.7292200000E-01, 2.5353900000E-01), (2.2219200000E-01, 8.0066900000E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("D", [(9.0000000000E-01, 1.0000000000E+00)]),("D", [(2.2500000000E-01, 1.0000000000E+00)]),("F", [(3.2000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8516000000E-03), (2.9094200000E+03, 1.4206200000E-02), (6.6136400000E+02, 6.9999500000E-02), (1.8575900000E+02, 2.4007900000E-01), (5.9194300000E+01, 4.8476200000E-01), (2.0031000000E+01, 3.3520000000E-01)]),("S", [(3.3947800000E+02, -2.7821700000E-03), (8.1010100000E+01, -3.6049900000E-02), (2.5878000000E+01, -1.1663100000E-01), (9.4522100000E+00, 9.6832800000E-02), (3.6656600000E+00, 6.1441800000E-01), (1.4674600000E+00, 4.0379800000E-01)]),("P", [(3.3947800000E+02, 4.5646200000E-03), (8.1010100000E+01, 3.3693600000E-02), (2.5878000000E+01, 1.3975500000E-01), (9.4522100000E+00, 3.3936200000E-01), (3.6656600000E+00, 4.5092100000E-01), (1.4674600000E+00, 2.3858600000E-01)]),("S", [(2.1562300000E+00, -2.5292300000E-01), (7.4899700000E-01, 3.2851700000E-02), (2.8314500000E-01, 1.0812500000E+00)]),("P", [(2.1562300000E+00, -1.7765300000E-02), (7.4899700000E-01, 2.7405800000E-01), (2.8314500000E-01, 7.8542100000E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("D", [(1.1000000000E+00, 1.0000000000E+00)]),("D", [(2.7500000000E-01, 1.0000000000E+00)]),("F", [(4.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8690000000E-03), (3.3014900000E+03, 1.4230000000E-02), (7.5414600000E+02, 6.9696000000E-02), (2.1271100000E+02, 2.3848700000E-01), (6.7989600000E+01, 4.8330700000E-01), (2.3051500000E+01, 3.3807400000E-01)]),("S", [(4.2373500000E+02, -2.3767000000E-03), (1.0071000000E+02, -3.1693000000E-02), (3.2159900000E+01, -1.1331700000E-01), (1.1807900000E+01, 5.6090000000E-02), (4.6311000000E+00, 5.9225500000E-01), (1.8702500000E+00, 4.5500600000E-01)]),("P", [(4.2373500000E+02, 4.0610000000E-03), (1.0071000000E+02, 3.0681000000E-02), (3.2159900000E+01, 1.3045200000E-01), (1.1807900000E+01, 3.2720500000E-01), (4.6311000000E+00, 4.5285100000E-01), (1.8702500000E+00, 2.5604200000E-01)]),("S", [(2.6158400000E+00, -2.5037400000E-01), (9.2216700000E-01, 6.6957000000E-02), (3.4128700000E-01, 1.0545100000E+00)]),("P", [(2.6158400000E+00, -1.4511000000E-02), (9.2216700000E-01, 3.1026300000E-01), (3.4128700000E-01, 7.5448300000E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("D", [(1.3000000000E+00, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)]),("F", [(5.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8330000000E-03), (3.7803500000E+03, 1.4034000000E-02), (8.6047400000E+02, 6.9097000000E-02), (2.4214500000E+02, 2.3745200000E-01), (7.7334900000E+01, 4.8303400000E-01), (2.6247000000E+01, 3.3985600000E-01)]),("S", [(4.9176500000E+02, -2.2974000000E-03), (1.1698400000E+02, -3.0714000000E-02), (3.7415300000E+01, -1.1252800000E-01), (1.3783400000E+01, 4.5016000000E-02), (5.4521500000E+00, 5.8935300000E-01), (2.2258800000E+00, 4.6520600000E-01)]),("P", [(4.9176500000E+02, 3.9894000000E-03), (1.1698400000E+02, 3.0318000000E-02), (3.7415300000E+01, 1.2988000000E-01), (1.3783400000E+01, 3.2795100000E-01), (5.4521500000E+00, 4.5352700000E-01), (2.2258800000E+00, 2.5215400000E-01)]),("S", [(3.1864900000E+00, -2.5183000000E-01), (1.1442700000E+00, 6.1589000000E-02), (4.2037700000E-01, 1.0601800000E+00)]),("P", [(3.1864900000E+00, -1.4299000000E-02), (1.1442700000E+00, 3.2357200000E-01), (4.2037700000E-01, 7.4350700000E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("D", [(1.5000000000E+00, 1.0000000000E+00)]),("D", [(3.7500000000E-01, 1.0000000000E+00)]),("F", [(7.0000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252600000E-03), (4.2576200000E+03, 1.3968600000E-02), (9.6985700000E+02, 6.8707300000E-02), (2.7326300000E+02, 2.3620400000E-01), (8.7369500000E+01, 4.8221400000E-01), (2.9686700000E+01, 3.4204300000E-01)]),("S", [(5.7589100000E+02, -2.1597200000E-03), (1.3681600000E+02, -2.9077500000E-02), (4.3809800000E+01, -1.1082700000E-01), (1.6209400000E+01, 2.7699900000E-02), (6.4608400000E+00, 5.7761300000E-01), (2.6511400000E+00, 4.8868800000E-01)]),("P", [(5.7589100000E+02, 3.8066500000E-03), (1.3681600000E+02, 2.9230500000E-02), (4.3809800000E+01, 1.2646700000E-01), (1.6209400000E+01, 3.2351000000E-01), (6.4608400000E+00, 4.5489600000E-01), (2.6511400000E+00, 2.5663000000E-01)]),("S", [(3.8602800000E+00, -2.5559200000E-01), (1.4137300000E+00, 3.7806600000E-02), (5.1664600000E-01, 1.0805600000E+00)]),("P", [(3.8602800000E+00, -1.5919700000E-02), (1.4137300000E+00, 3.2464600000E-01), (5.1664600000E-01, 7.4399000000E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("D", [(1.7000000000E+00, 1.0000000000E+00)]),("D", [(4.2500000000E-01, 1.0000000000E+00)]),("F", [(8.5000000000E-01, 1.0000000000E+00)])],
}




_6_31GBRA3DFCOMMA3PDKET = {

    1 : [("S", [(1.8731137000E+01, 3.3494600000E-02), (2.8253937000E+00, 2.3472695000E-01), (6.4012170000E-01, 8.1375733000E-01)]),("S", [(1.6127780000E-01, 1.0000000000E+00)]),("P", [(3.0000000000E+00, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)]),("P", [(1.8750000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(3.8421634000E+01, 2.3766000000E-02), (5.7780300000E+00, 1.5467900000E-01), (1.2417740000E+00, 4.6963000000E-01)]),("S", [(2.9796400000E-01, 1.0000000000E+00)]),("P", [(3.0000000000E+00, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)]),("P", [(1.8750000000E-01, 1.0000000000E+00)]),("D", [(2.0000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(6.4241892000E+02, 2.1426000000E-03), (9.6798515000E+01, 1.6208900000E-02), (2.2091121000E+01, 7.7315600000E-02), (6.2010703000E+00, 2.4578600000E-01), (1.9351177000E+00, 4.7018900000E-01), (6.3673580000E-01, 3.4547080000E-01)]),("S", [(2.3249184000E+00, -3.5091700000E-02), (6.3243060000E-01, -1.9123280000E-01), (7.9053400000E-02, 1.0839878000E+00)]),("P", [(2.3249184000E+00, 8.9415000000E-03), (6.3243060000E-01, 1.4100950000E-01), (7.9053400000E-02, 9.4536370000E-01)]),("S", [(3.5962000000E-02, 1.0000000000E+00)]),("P", [(3.5962000000E-02, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)]),("D", [(5.0000000000E-02, 1.0000000000E+00)]),("F", [(1.5000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.2645857000E+03, 1.9448000000E-03), (1.8993681000E+02, 1.4835100000E-02), (4.3159089000E+01, 7.2090600000E-02), (1.2098663000E+01, 2.3715420000E-01), (3.8063232000E+00, 4.6919870000E-01), (1.2728903000E+00, 3.5652020000E-01)]),("S", [(3.1964631000E+00, -1.1264870000E-01), (7.4781330000E-01, -2.2950640000E-01), (2.1996630000E-01, 1.1869167000E+00)]),("P", [(3.1964631000E+00, 5.5980200000E-02), (7.4781330000E-01, 2.6155060000E-01), (2.1996630000E-01, 7.9397230000E-01)]),("S", [(8.2309900000E-02, 1.0000000000E+00)]),("P", [(8.2309900000E-02, 1.0000000000E+00)]),("D", [(1.0200000000E+00, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)]),("D", [(6.3750000000E-02, 1.0000000000E+00)]),("F", [(2.6000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.0688823000E+03, 1.8663000000E-03), (3.1064957000E+02, 1.4251500000E-02), (7.0683033000E+01, 6.9551600000E-02), (1.9861080000E+01, 2.3257290000E-01), (6.2993048000E+00, 4.6707870000E-01), (2.1270270000E+00, 3.6343140000E-01)]),("S", [(4.7279710000E+00, -1.3039380000E-01), (1.1903377000E+00, -1.3078890000E-01), (3.5941170000E-01, 1.1309444000E+00)]),("P", [(4.7279710000E+00, 7.4597600000E-02), (1.1903377000E+00, 3.0784670000E-01), (3.5941170000E-01, 7.4345680000E-01)]),("S", [(1.2675120000E-01, 1.0000000000E+00)]),("P", [(1.2675120000E-01, 1.0000000000E+00)]),("D", [(1.6040000000E+00, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)]),("D", [(1.0025000000E-01, 1.0000000000E+00)]),("F", [(5.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(3.0475249000E+03, 1.8347000000E-03), (4.5736951000E+02, 1.4037300000E-02), (1.0394869000E+02, 6.8842600000E-02), (2.9210155000E+01, 2.3218440000E-01), (9.2866630000E+00, 4.6794130000E-01), (3.1639270000E+00, 3.6231200000E-01)]),("S", [(7.8682724000E+00, -1.1933240000E-01), (1.8812885000E+00, -1.6085420000E-01), (5.4424930000E-01, 1.1434564000E+00)]),("P", [(7.8682724000E+00, 6.8999100000E-02), (1.8812885000E+00, 3.1642400000E-01), (5.4424930000E-01, 7.4430830000E-01)]),("S", [(1.6871440000E-01, 1.0000000000E+00)]),("P", [(1.6871440000E-01, 1.0000000000E+00)]),("D", [(2.5040000000E+00, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)]),("D", [(1.5650000000E-01, 1.0000000000E+00)]),("F", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(4.1735110000E+03, 1.8348000000E-03), (6.2745790000E+02, 1.3995000000E-02), (1.4290210000E+02, 6.8587000000E-02), (4.0234330000E+01, 2.3224100000E-01), (1.2820210000E+01, 4.6907000000E-01), (4.3904370000E+00, 3.6045500000E-01)]),("S", [(1.1626358000E+01, -1.1496100000E-01), (2.7162800000E+00, -1.6911800000E-01), (7.7221800000E-01, 1.1458520000E+00)]),("P", [(1.1626358000E+01, 6.7580000000E-02), (2.7162800000E+00, 3.2390700000E-01), (7.7221800000E-01, 7.4089500000E-01)]),("S", [(2.1203130000E-01, 1.0000000000E+00)]),("P", [(2.1203130000E-01, 1.0000000000E+00)]),("D", [(3.6520000000E+00, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)]),("D", [(2.2825000000E-01, 1.0000000000E+00)]),("F", [(1.0000000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(5.4846717000E+03, 1.8311000000E-03), (8.2523495000E+02, 1.3950100000E-02), (1.8804696000E+02, 6.8445100000E-02), (5.2964500000E+01, 2.3271430000E-01), (1.6897570000E+01, 4.7019300000E-01), (5.7996353000E+00, 3.5852090000E-01)]),("S", [(1.5539616000E+01, -1.1077750000E-01), (3.5999336000E+00, -1.4802630000E-01), (1.0137618000E+00, 1.1307670000E+00)]),("P", [(1.5539616000E+01, 7.0874300000E-02), (3.5999336000E+00, 3.3975280000E-01), (1.0137618000E+00, 7.2715860000E-01)]),("S", [(2.7000580000E-01, 1.0000000000E+00)]),("P", [(2.7000580000E-01, 1.0000000000E+00)]),("D", [(5.1600000000E+00, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)]),("D", [(3.2250000000E-01, 1.0000000000E+00)]),("F", [(1.4000000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(7.0017130900E+03, 1.8196169000E-03), (1.0513660900E+03, 1.3916079600E-02), (2.3928569000E+02, 6.8405324500E-02), (6.7397445300E+01, 2.3318576000E-01), (2.1519957300E+01, 4.7126743900E-01), (7.4031013000E+00, 3.5661854600E-01)]),("S", [(2.0847952800E+01, -1.0850697500E-01), (4.8083083400E+00, -1.4645165800E-01), (1.3440698600E+00, 1.1286885800E+00)]),("P", [(2.0847952800E+01, 7.1628724300E-02), (4.8083083400E+00, 3.4591210300E-01), (1.3440698600E+00, 7.2246995700E-01)]),("S", [(3.5815139300E-01, 1.0000000000E+00)]),("P", [(3.5815139300E-01, 1.0000000000E+00)]),("D", [(7.0000000000E+00, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)]),("D", [(4.3750000000E-01, 1.0000000000E+00)]),("F", [(1.8500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(8.4258515300E+03, 1.8843481000E-03), (1.2685194000E+03, 1.4336899400E-02), (2.8962141400E+02, 7.0109623300E-02), (8.1859004000E+01, 2.3737326600E-01), (2.6251507900E+01, 4.7300712600E-01), (9.0947205100E+00, 3.4840124100E-01)]),("S", [(2.6532131000E+01, -1.0711828700E-01), (6.1017550100E+00, -1.4616382100E-01), (1.6962715300E+00, 1.1277735000E+00)]),("P", [(2.6532131000E+01, 7.1909588500E-02), (6.1017550100E+00, 3.4951337200E-01), (1.6962715300E+00, 7.1994051200E-01)]),("S", [(4.4581870000E-01, 1.0000000000E+00)]),("P", [(4.4581870000E-01, 1.0000000000E+00)]),("D", [(9.2160000000E+00, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)]),("D", [(5.7600000000E-01, 1.0000000000E+00)]),("F", [(2.5000000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(9.9932000000E+03, 1.9377000000E-03), (1.4998900000E+03, 1.4807000000E-02), (3.4195100000E+02, 7.2706000000E-02), (9.4679700000E+01, 2.5262900000E-01), (2.9734500000E+01, 4.9324200000E-01), (1.0006300000E+01, 3.1316900000E-01)]),("S", [(1.5096300000E+02, -3.5421000000E-03), (3.5587800000E+01, -4.3959000000E-02), (1.1168300000E+01, -1.0975210000E-01), (3.9020100000E+00, 1.8739800000E-01), (1.3817700000E+00, 6.4669900000E-01), (4.6638200000E-01, 3.0605800000E-01)]),("P", [(1.5096300000E+02, 5.0017000000E-03), (3.5587800000E+01, 3.5511000000E-02), (1.1168300000E+01, 1.4282500000E-01), (3.9020100000E+00, 3.3862000000E-01), (1.3817700000E+00, 4.5157900000E-01), (4.6638200000E-01, 2.7327100000E-01)]),("S", [(4.9796600000E-01, -2.4850300000E-01), (8.4353000000E-02, -1.3170400000E-01), (6.6635000000E-02, 1.2335200000E+00)]),("P", [(4.9796600000E-01, -2.3023000000E-02), (8.4353000000E-02, 9.5035900000E-01), (6.6635000000E-02, 5.9858000000E-02)]),("S", [(2.5954400000E-02, 1.0000000000E+00)]),("P", [(2.5954400000E-02, 1.0000000000E+00)]),("D", [(7.0000000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)]),("D", [(4.3750000000E-02, 1.0000000000E+00)]),("F", [(1.5000000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(1.1722800000E+04, 1.9778000000E-03), (1.7599300000E+03, 1.5114000000E-02), (4.0084600000E+02, 7.3911000000E-02), (1.1280700000E+02, 2.4919100000E-01), (3.5999700000E+01, 4.8792800000E-01), (1.2182800000E+01, 3.1966200000E-01)]),("S", [(1.8918000000E+02, -3.2372000000E-03), (4.5211900000E+01, -4.1008000000E-02), (1.4356300000E+01, -1.1260000000E-01), (5.1388600000E+00, 1.4863300000E-01), (1.9065200000E+00, 6.1649700000E-01), (7.0588700000E-01, 3.6482900000E-01)]),("P", [(1.8918000000E+02, 4.9281000000E-03), (4.5211900000E+01, 3.4989000000E-02), (1.4356300000E+01, 1.4072500000E-01), (5.1388600000E+00, 3.3364200000E-01), (1.9065200000E+00, 4.4494000000E-01), (7.0588700000E-01, 2.6925400000E-01)]),("S", [(9.2934000000E-01, -2.1229000000E-01), (2.6903500000E-01, -1.0798500000E-01), (1.1737900000E-01, 1.1758400000E+00)]),("P", [(9.2934000000E-01, -2.2419000000E-02), (2.6903500000E-01, 1.9227000000E-01), (1.1737900000E-01, 8.4618100000E-01)]),("S", [(4.2106100000E-02, 1.0000000000E+00)]),("P", [(4.2106100000E-02, 1.0000000000E+00)]),("D", [(7.0000000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)]),("D", [(4.3750000000E-02, 1.0000000000E+00)]),("F", [(2.0000000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(1.3983100000E+04, 1.9426700000E-03), (2.0987500000E+03, 1.4859900000E-02), (4.7770500000E+02, 7.2849400000E-02), (1.3436000000E+02, 2.4683000000E-01), (4.2870900000E+01, 4.8725800000E-01), (1.4518900000E+01, 3.2349600000E-01)]),("S", [(2.3966800000E+02, -2.9261900000E-03), (5.7441900000E+01, -3.7408000000E-02), (1.8285900000E+01, -1.1448700000E-01), (6.5991400000E+00, 1.1563500000E-01), (2.4904900000E+00, 6.1259500000E-01), (9.4454000000E-01, 3.9379900000E-01)]),("P", [(2.3966800000E+02, 4.6028500000E-03), (5.7441900000E+01, 3.3199000000E-02), (1.8285900000E+01, 1.3628200000E-01), (6.5991400000E+00, 3.3047600000E-01), (2.4904900000E+00, 4.4914600000E-01), (9.4454000000E-01, 2.6570400000E-01)]),("S", [(1.2779000000E+00, -2.2760600000E-01), (3.9759000000E-01, 1.4458300000E-03), (1.6009500000E-01, 1.0927900000E+00)]),("P", [(1.2779000000E+00, -1.7513000000E-02), (3.9759000000E-01, 2.4453300000E-01), (1.6009500000E-01, 8.0493400000E-01)]),("S", [(5.5657700000E-02, 1.0000000000E+00)]),("P", [(5.5657700000E-02, 1.0000000000E+00)]),("D", [(1.3000000000E+00, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)]),("D", [(8.1250000000E-02, 1.0000000000E+00)]),("F", [(2.5000000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.6115900000E+04, 1.9594800000E-03), (2.4255800000E+03, 1.4928800000E-02), (5.5386700000E+02, 7.2847800000E-02), (1.5634000000E+02, 2.4613000000E-01), (5.0068300000E+01, 4.8591400000E-01), (1.7017800000E+01, 3.2500200000E-01)]),("S", [(2.9271800000E+02, -2.7809400000E-03), (6.9873100000E+01, -3.5714600000E-02), (2.2336300000E+01, -1.1498500000E-01), (8.1503900000E+00, 9.3563400000E-02), (3.1345800000E+00, 6.0301700000E-01), (1.2254300000E+00, 4.1895900000E-01)]),("P", [(2.9271800000E+02, 4.4382600000E-03), (6.9873100000E+01, 3.2667900000E-02), (2.2336300000E+01, 1.3472100000E-01), (8.1503900000E+00, 3.2867800000E-01), (3.1345800000E+00, 4.4964000000E-01), (1.2254300000E+00, 2.6137200000E-01)]),("S", [(1.7273800000E+00, -2.4463000000E-01), (5.7292200000E-01, 4.3157200000E-03), (2.2219200000E-01, 1.0981800000E+00)]),("P", [(1.7273800000E+00, -1.7795100000E-02), (5.7292200000E-01, 2.5353900000E-01), (2.2219200000E-01, 8.0066900000E-01)]),("S", [(7.7836900000E-02, 1.0000000000E+00)]),("P", [(7.7836900000E-02, 1.0000000000E+00)]),("D", [(1.8000000000E+00, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)]),("D", [(1.1250000000E-01, 1.0000000000E+00)]),("F", [(3.2000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.9413300000E+04, 1.8516000000E-03), (2.9094200000E+03, 1.4206200000E-02), (6.6136400000E+02, 6.9999500000E-02), (1.8575900000E+02, 2.4007900000E-01), (5.9194300000E+01, 4.8476200000E-01), (2.0031000000E+01, 3.3520000000E-01)]),("S", [(3.3947800000E+02, -2.7821700000E-03), (8.1010100000E+01, -3.6049900000E-02), (2.5878000000E+01, -1.1663100000E-01), (9.4522100000E+00, 9.6832800000E-02), (3.6656600000E+00, 6.1441800000E-01), (1.4674600000E+00, 4.0379800000E-01)]),("P", [(3.3947800000E+02, 4.5646200000E-03), (8.1010100000E+01, 3.3693600000E-02), (2.5878000000E+01, 1.3975500000E-01), (9.4522100000E+00, 3.3936200000E-01), (3.6656600000E+00, 4.5092100000E-01), (1.4674600000E+00, 2.3858600000E-01)]),("S", [(2.1562300000E+00, -2.5292300000E-01), (7.4899700000E-01, 3.2851700000E-02), (2.8314500000E-01, 1.0812500000E+00)]),("P", [(2.1562300000E+00, -1.7765300000E-02), (7.4899700000E-01, 2.7405800000E-01), (2.8314500000E-01, 7.8542100000E-01)]),("S", [(9.9831700000E-02, 1.0000000000E+00)]),("P", [(9.9831700000E-02, 1.0000000000E+00)]),("D", [(2.2000000000E+00, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)]),("D", [(1.3750000000E-01, 1.0000000000E+00)]),("F", [(4.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(2.1917100000E+04, 1.8690000000E-03), (3.3014900000E+03, 1.4230000000E-02), (7.5414600000E+02, 6.9696000000E-02), (2.1271100000E+02, 2.3848700000E-01), (6.7989600000E+01, 4.8330700000E-01), (2.3051500000E+01, 3.3807400000E-01)]),("S", [(4.2373500000E+02, -2.3767000000E-03), (1.0071000000E+02, -3.1693000000E-02), (3.2159900000E+01, -1.1331700000E-01), (1.1807900000E+01, 5.6090000000E-02), (4.6311000000E+00, 5.9225500000E-01), (1.8702500000E+00, 4.5500600000E-01)]),("P", [(4.2373500000E+02, 4.0610000000E-03), (1.0071000000E+02, 3.0681000000E-02), (3.2159900000E+01, 1.3045200000E-01), (1.1807900000E+01, 3.2720500000E-01), (4.6311000000E+00, 4.5285100000E-01), (1.8702500000E+00, 2.5604200000E-01)]),("S", [(2.6158400000E+00, -2.5037400000E-01), (9.2216700000E-01, 6.6957000000E-02), (3.4128700000E-01, 1.0545100000E+00)]),("P", [(2.6158400000E+00, -1.4511000000E-02), (9.2216700000E-01, 3.1026300000E-01), (3.4128700000E-01, 7.5448300000E-01)]),("S", [(1.1716700000E-01, 1.0000000000E+00)]),("P", [(1.1716700000E-01, 1.0000000000E+00)]),("D", [(2.6000000000E+00, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("D", [(1.6250000000E-01, 1.0000000000E+00)]),("F", [(5.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.5180100000E+04, 1.8330000000E-03), (3.7803500000E+03, 1.4034000000E-02), (8.6047400000E+02, 6.9097000000E-02), (2.4214500000E+02, 2.3745200000E-01), (7.7334900000E+01, 4.8303400000E-01), (2.6247000000E+01, 3.3985600000E-01)]),("S", [(4.9176500000E+02, -2.2974000000E-03), (1.1698400000E+02, -3.0714000000E-02), (3.7415300000E+01, -1.1252800000E-01), (1.3783400000E+01, 4.5016000000E-02), (5.4521500000E+00, 5.8935300000E-01), (2.2258800000E+00, 4.6520600000E-01)]),("P", [(4.9176500000E+02, 3.9894000000E-03), (1.1698400000E+02, 3.0318000000E-02), (3.7415300000E+01, 1.2988000000E-01), (1.3783400000E+01, 3.2795100000E-01), (5.4521500000E+00, 4.5352700000E-01), (2.2258800000E+00, 2.5215400000E-01)]),("S", [(3.1864900000E+00, -2.5183000000E-01), (1.1442700000E+00, 6.1589000000E-02), (4.2037700000E-01, 1.0601800000E+00)]),("P", [(3.1864900000E+00, -1.4299000000E-02), (1.1442700000E+00, 3.2357200000E-01), (4.2037700000E-01, 7.4350700000E-01)]),("S", [(1.4265700000E-01, 1.0000000000E+00)]),("P", [(1.4265700000E-01, 1.0000000000E+00)]),("D", [(3.0000000000E+00, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)]),("D", [(1.8750000000E-01, 1.0000000000E+00)]),("F", [(7.0000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(2.8348300000E+04, 1.8252600000E-03), (4.2576200000E+03, 1.3968600000E-02), (9.6985700000E+02, 6.8707300000E-02), (2.7326300000E+02, 2.3620400000E-01), (8.7369500000E+01, 4.8221400000E-01), (2.9686700000E+01, 3.4204300000E-01)]),("S", [(5.7589100000E+02, -2.1597200000E-03), (1.3681600000E+02, -2.9077500000E-02), (4.3809800000E+01, -1.1082700000E-01), (1.6209400000E+01, 2.7699900000E-02), (6.4608400000E+00, 5.7761300000E-01), (2.6511400000E+00, 4.8868800000E-01)]),("P", [(5.7589100000E+02, 3.8066500000E-03), (1.3681600000E+02, 2.9230500000E-02), (4.3809800000E+01, 1.2646700000E-01), (1.6209400000E+01, 3.2351000000E-01), (6.4608400000E+00, 4.5489600000E-01), (2.6511400000E+00, 2.5663000000E-01)]),("S", [(3.8602800000E+00, -2.5559200000E-01), (1.4137300000E+00, 3.7806600000E-02), (5.1664600000E-01, 1.0805600000E+00)]),("P", [(3.8602800000E+00, -1.5919700000E-02), (1.4137300000E+00, 3.2464600000E-01), (5.1664600000E-01, 7.4399000000E-01)]),("S", [(1.7388800000E-01, 1.0000000000E+00)]),("P", [(1.7388800000E-01, 1.0000000000E+00)]),("D", [(3.4000000000E+00, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)]),("D", [(2.1250000000E-01, 1.0000000000E+00)]),("F", [(8.5000000000E-01, 1.0000000000E+00)])],
}




_6_31GBRADCOMMAPKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]),("S", [(8.7404700000E-01, 1.0000000000E+00)]),("S", [(2.4456400000E-01, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)])],
}



_6_311GBRA2DFCOMMA2PDKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("P", [(1.5000000000E+00, 1.0000000000E+00)]),("P", [(3.7500000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]),("S", [(8.7404700000E-01, 1.0000000000E+00)]),("S", [(2.4456400000E-01, 1.0000000000E+00)]),("P", [(1.5000000000E+00, 1.0000000000E+00)]),("P", [(3.7500000000E-01, 1.0000000000E+00)]),("D", [(2.0000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E-01, 1.0000000000E+00)]),("F", [(1.5000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("D", [(5.1000000000E-01, 1.0000000000E+00)]),("D", [(1.2750000000E-01, 1.0000000000E+00)]),("F", [(2.6000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("D", [(8.0200000000E-01, 1.0000000000E+00)]),("D", [(2.0050000000E-01, 1.0000000000E+00)]),("F", [(5.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("D", [(1.2520000000E+00, 1.0000000000E+00)]),("D", [(3.1300000000E-01, 1.0000000000E+00)]),("F", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("D", [(1.8260000000E+00, 1.0000000000E+00)]),("D", [(4.5650000000E-01, 1.0000000000E+00)]),("F", [(1.0000000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("D", [(2.5840000000E+00, 1.0000000000E+00)]),("D", [(6.4600000000E-01, 1.0000000000E+00)]),("F", [(1.4000000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("D", [(3.5000000000E+00, 1.0000000000E+00)]),("D", [(8.7500000000E-01, 1.0000000000E+00)]),("F", [(1.8500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("D", [(4.6080000000E+00, 1.0000000000E+00)]),("D", [(1.1520000000E+00, 1.0000000000E+00)]),("F", [(2.5000000000E+00, 1.0000000000E+00)])],
}






_6_311PLUSGBRA2DCOMMAPKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(9.8124300000E+01, 2.8745200000E-02), (1.4768900000E+01, 2.0806100000E-01), (3.3188300000E+00, 8.3763500000E-01)]),("S", [(8.7404700000E-01, 1.0000000000E+00)]),("S", [(2.4456400000E-01, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(5.1000000000E-01, 1.0000000000E+00)]),("D", [(1.2750000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(8.0200000000E-01, 1.0000000000E+00)]),("D", [(2.0050000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(1.2520000000E+00, 1.0000000000E+00)]),("D", [(3.1300000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(1.8260000000E+00, 1.0000000000E+00)]),("D", [(4.5650000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(2.5840000000E+00, 1.0000000000E+00)]),("D", [(6.4600000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(3.5000000000E+00, 1.0000000000E+00)]),("D", [(8.7500000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(4.6080000000E+00, 1.0000000000E+00)]),("D", [(1.1520000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(3.5000000000E-01, 1.0000000000E+00)]),("D", [(8.7500000000E-02, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(3.5000000000E-01, 1.0000000000E+00)]),("D", [(8.7500000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("D", [(1.6250000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(9.0000000000E-01, 1.0000000000E+00)]),("D", [(2.2500000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(1.1000000000E+00, 1.0000000000E+00)]),("D", [(2.7500000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(1.3000000000E+00, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(1.5000000000E+00, 1.0000000000E+00)]),("D", [(3.7500000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(1.7000000000E+00, 1.0000000000E+00)]),("D", [(4.2500000000E-01, 1.0000000000E+00)])],
}



_6_311PLUSPLUSGBRA2DCOMMA2PKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("S", [(3.6000000000E-02, 1.0000000000E+00)]),("P", [(1.5000000000E+00, 1.0000000000E+00)]),("P", [(3.7500000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(5.1000000000E-01, 1.0000000000E+00)]),("D", [(1.2750000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(8.0200000000E-01, 1.0000000000E+00)]),("D", [(2.0050000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(1.2520000000E+00, 1.0000000000E+00)]),("D", [(3.1300000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(1.8260000000E+00, 1.0000000000E+00)]),("D", [(4.5650000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(2.5840000000E+00, 1.0000000000E+00)]),("D", [(6.4600000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(3.5000000000E+00, 1.0000000000E+00)]),("D", [(8.7500000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(4.6080000000E+00, 1.0000000000E+00)]),("D", [(1.1520000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(3.5000000000E-01, 1.0000000000E+00)]),("D", [(8.7500000000E-02, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(3.5000000000E-01, 1.0000000000E+00)]),("D", [(8.7500000000E-02, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("D", [(1.6250000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(9.0000000000E-01, 1.0000000000E+00)]),("D", [(2.2500000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(1.1000000000E+00, 1.0000000000E+00)]),("D", [(2.7500000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(1.3000000000E+00, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(1.5000000000E+00, 1.0000000000E+00)]),("D", [(3.7500000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(1.7000000000E+00, 1.0000000000E+00)]),("D", [(4.2500000000E-01, 1.0000000000E+00)])],
}


_6_311PLUSPLUSGBRA3DFCOMMA3PDKET = {

    1 : [("S", [(3.3865000000E+01, 2.5493800000E-02), (5.0947900000E+00, 1.9037300000E-01), (1.1587900000E+00, 8.5216100000E-01)]),("S", [(3.2584000000E-01, 1.0000000000E+00)]),("S", [(1.0274100000E-01, 1.0000000000E+00)]),("S", [(3.6000000000E-02, 1.0000000000E+00)]),("P", [(3.0000000000E+00, 1.0000000000E+00)]),("P", [(7.5000000000E-01, 1.0000000000E+00)]),("P", [(1.8750000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(9.0046000000E+02, 2.2870400000E-03), (1.3443300000E+02, 1.7635000000E-02), (3.0436500000E+01, 8.7343400000E-02), (8.6263900000E+00, 2.8097700000E-01), (2.4833200000E+00, 6.5874100000E-01), (3.0317900000E-01, 1.1871200000E-01)]),("S", [(4.8689000000E+00, 9.3329300000E-02), (8.5692400000E-01, 9.4304500000E-01), (2.4322700000E-01, -2.7982700000E-03)]),("P", [(4.8689000000E+00, 3.2766100000E-02), (8.5692400000E-01, 1.5979200000E-01), (2.4322700000E-01, 8.8566700000E-01)]),("S", [(6.3507000000E-02, 1.0000000000E+00)]),("P", [(6.3507000000E-02, 1.0000000000E+00)]),("S", [(2.4368300000E-02, 1.0000000000E+00)]),("P", [(2.4368300000E-02, 1.0000000000E+00)]),("S", [(7.4000000000E-03, 1.0000000000E+00)]),("P", [(7.4000000000E-03, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)]),("D", [(2.0000000000E-01, 1.0000000000E+00)]),("D", [(5.0000000000E-02, 1.0000000000E+00)]),("F", [(1.5000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.6828000000E+03, 2.2857400000E-03), (2.5171500000E+02, 1.7593800000E-02), (5.7411600000E+01, 8.6331500000E-02), (1.6517100000E+01, 2.8183500000E-01), (4.8536400000E+00, 6.4059400000E-01), (6.2686300000E-01, 1.4446700000E-01)]),("S", [(8.3093800000E+00, 1.0862100000E-01), (1.7407500000E+00, 9.2730100000E-01), (4.8581600000E-01, -2.9716900000E-03)]),("P", [(8.3093800000E+00, 3.6134400000E-02), (1.7407500000E+00, 2.1695800000E-01), (4.8581600000E-01, 8.4183900000E-01)]),("S", [(1.6361300000E-01, 1.0000000000E+00)]),("P", [(1.6361300000E-01, 1.0000000000E+00)]),("S", [(5.6728500000E-02, 1.0000000000E+00)]),("P", [(5.6728500000E-02, 1.0000000000E+00)]),("S", [(2.0700000000E-02, 1.0000000000E+00)]),("P", [(2.0700000000E-02, 1.0000000000E+00)]),("D", [(1.0200000000E+00, 1.0000000000E+00)]),("D", [(2.5500000000E-01, 1.0000000000E+00)]),("D", [(6.3750000000E-02, 1.0000000000E+00)]),("F", [(2.6000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(2.8588900000E+03, 2.1537500000E-03), (4.2814000000E+02, 1.6582300000E-02), (9.7528200000E+01, 8.2187000000E-02), (2.7969300000E+01, 2.7661800000E-01), (8.2157700000E+00, 6.2931600000E-01), (1.1127800000E+00, 1.7377000000E-01)]),("S", [(1.3241500000E+01, 1.1744300000E-01), (3.0016600000E+00, 9.1800200000E-01), (9.1285600000E-01, -2.6510500000E-03)]),("P", [(1.3241500000E+01, 4.1810000000E-02), (3.0016600000E+00, 2.3657500000E-01), (9.1285600000E-01, 8.1621400000E-01)]),("S", [(3.1545400000E-01, 1.0000000000E+00)]),("P", [(3.1545400000E-01, 1.0000000000E+00)]),("S", [(9.8856300000E-02, 1.0000000000E+00)]),("P", [(9.8856300000E-02, 1.0000000000E+00)]),("S", [(3.1500000000E-02, 1.0000000000E+00)]),("P", [(3.1500000000E-02, 1.0000000000E+00)]),("D", [(1.6040000000E+00, 1.0000000000E+00)]),("D", [(4.0100000000E-01, 1.0000000000E+00)]),("D", [(1.0025000000E-01, 1.0000000000E+00)]),("F", [(5.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.5632400000E+03, 1.9666500000E-03), (6.8202400000E+02, 1.5230600000E-02), (1.5497300000E+02, 7.6126900000E-02), (4.4455300000E+01, 2.6080100000E-01), (1.3029000000E+01, 6.1646200000E-01), (1.8277300000E+00, 2.2100600000E-01)]),("S", [(2.0964200000E+01, 1.1466000000E-01), (4.8033100000E+00, 9.1999900000E-01), (1.4593300000E+00, -3.0306800000E-03)]),("P", [(2.0964200000E+01, 4.0248700000E-02), (4.8033100000E+00, 2.3759400000E-01), (1.4593300000E+00, 8.1585400000E-01)]),("S", [(4.8345600000E-01, 1.0000000000E+00)]),("P", [(4.8345600000E-01, 1.0000000000E+00)]),("S", [(1.4558500000E-01, 1.0000000000E+00)]),("P", [(1.4558500000E-01, 1.0000000000E+00)]),("S", [(4.3800000000E-02, 1.0000000000E+00)]),("P", [(4.3800000000E-02, 1.0000000000E+00)]),("D", [(2.5040000000E+00, 1.0000000000E+00)]),("D", [(6.2600000000E-01, 1.0000000000E+00)]),("D", [(1.5650000000E-01, 1.0000000000E+00)]),("F", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.2934800000E+03, 1.9697900000E-03), (9.4904400000E+02, 1.4961300000E-02), (2.1877600000E+02, 7.3500600000E-02), (6.3691600000E+01, 2.4893700000E-01), (1.8828200000E+01, 6.0246000000E-01), (2.7202300000E+00, 2.5620200000E-01)]),("S", [(3.0633100000E+01, 1.1190600000E-01), (7.0261400000E+00, 9.2166600000E-01), (2.1120500000E+00, -2.5691900000E-03)]),("P", [(3.0633100000E+01, 3.8311900000E-02), (7.0261400000E+00, 2.3740300000E-01), (2.1120500000E+00, 8.1759200000E-01)]),("S", [(6.8400900000E-01, 1.0000000000E+00)]),("P", [(6.8400900000E-01, 1.0000000000E+00)]),("S", [(2.0087800000E-01, 1.0000000000E+00)]),("P", [(2.0087800000E-01, 1.0000000000E+00)]),("S", [(6.3900000000E-02, 1.0000000000E+00)]),("P", [(6.3900000000E-02, 1.0000000000E+00)]),("D", [(3.6520000000E+00, 1.0000000000E+00)]),("D", [(9.1300000000E-01, 1.0000000000E+00)]),("D", [(2.2825000000E-01, 1.0000000000E+00)]),("F", [(1.0000000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(8.5885000000E+03, 1.8951500000E-03), (1.2972300000E+03, 1.4385900000E-02), (2.9929600000E+02, 7.0732000000E-02), (8.7377100000E+01, 2.4000100000E-01), (2.5678900000E+01, 5.9479700000E-01), (3.7400400000E+00, 2.8080200000E-01)]),("S", [(4.2117500000E+01, 1.1388900000E-01), (9.6283700000E+00, 9.2081100000E-01), (2.8533200000E+00, -3.2744700000E-03)]),("P", [(4.2117500000E+01, 3.6511400000E-02), (9.6283700000E+00, 2.3715300000E-01), (2.8533200000E+00, 8.1970200000E-01)]),("S", [(9.0566100000E-01, 1.0000000000E+00)]),("P", [(9.0566100000E-01, 1.0000000000E+00)]),("S", [(2.5561100000E-01, 1.0000000000E+00)]),("P", [(2.5561100000E-01, 1.0000000000E+00)]),("S", [(8.4500000000E-02, 1.0000000000E+00)]),("P", [(8.4500000000E-02, 1.0000000000E+00)]),("D", [(5.1600000000E+00, 1.0000000000E+00)]),("D", [(1.2920000000E+00, 1.0000000000E+00)]),("D", [(3.2250000000E-01, 1.0000000000E+00)]),("F", [(1.4000000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.1427100000E+04, 1.8009300000E-03), (1.7223500000E+03, 1.3741900000E-02), (3.9574600000E+02, 6.8133400000E-02), (1.1513900000E+02, 2.3332500000E-01), (3.3602600000E+01, 5.8908600000E-01), (4.9190100000E+00, 2.9950500000E-01)]),("S", [(5.5444100000E+01, 1.1453600000E-01), (1.2632300000E+01, 9.2051200000E-01), (3.7175600000E+00, -3.3780400000E-03)]),("P", [(5.5444100000E+01, 3.5460900000E-02), (1.2632300000E+01, 2.3745100000E-01), (3.7175600000E+00, 8.2045800000E-01)]),("S", [(1.1654500000E+00, 1.0000000000E+00)]),("P", [(1.1654500000E+00, 1.0000000000E+00)]),("S", [(3.2189200000E-01, 1.0000000000E+00)]),("P", [(3.2189200000E-01, 1.0000000000E+00)]),("S", [(1.0760000000E-01, 1.0000000000E+00)]),("P", [(1.0760000000E-01, 1.0000000000E+00)]),("D", [(7.0000000000E+00, 1.0000000000E+00)]),("D", [(1.7500000000E+00, 1.0000000000E+00)]),("D", [(4.3750000000E-01, 1.0000000000E+00)]),("F", [(1.8500000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(1.3995700000E+04, 1.8327600000E-03), (2.1171000000E+03, 1.3882700000E-02), (4.9042500000E+02, 6.8068700000E-02), (1.4383300000E+02, 2.3132800000E-01), (4.1926500000E+01, 5.8589000000E-01), (6.1568400000E+00, 3.0588300000E-01)]),("S", [(6.9121100000E+01, 1.1914900000E-01), (1.5835000000E+01, 9.1737500000E-01), (4.6732600000E+00, -4.0583900000E-03)]),("P", [(6.9121100000E+01, 3.5657400000E-02), (1.5835000000E+01, 2.3947700000E-01), (4.6732600000E+00, 8.1846100000E-01)]),("S", [(1.4575600000E+00, 1.0000000000E+00)]),("P", [(1.4575600000E+00, 1.0000000000E+00)]),("S", [(3.9705700000E-01, 1.0000000000E+00)]),("P", [(3.9705700000E-01, 1.0000000000E+00)]),("S", [(1.3000000000E-01, 1.0000000000E+00)]),("P", [(1.3000000000E-01, 1.0000000000E+00)]),("D", [(9.2160000000E+00, 1.0000000000E+00)]),("D", [(2.3040000000E+00, 1.0000000000E+00)]),("D", [(5.7600000000E-01, 1.0000000000E+00)]),("F", [(2.5000000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(3.6166400000E+04, 1.0320000000E-03), (5.3725800000E+03, 8.0710000000E-03), (1.2132100000E+03, 4.2129000000E-02), (3.3962300000E+02, 1.6978900000E-01), (1.0955300000E+02, 5.1462100000E-01), (3.8777300000E+01, 3.7981700000E-01)]),("S", [(3.8777300000E+01, 3.7476200000E-01), (1.4575900000E+01, 5.7576900000E-01), (5.2699300000E+00, 1.1293300000E-01)]),("S", [(1.8277700000E+00, 1.0000000000E+00)]),("S", [(6.1994800000E-01, 1.0000000000E+00)]),("S", [(5.7240000000E-02, 1.0000000000E+00)]),("S", [(2.4048000000E-02, 1.0000000000E+00)]),("P", [(1.4464500000E+02, 1.1485000000E-02), (3.3907400000E+01, 8.2383000000E-02), (1.0628500000E+01, 3.1965800000E-01), (3.8238900000E+00, 7.0129500000E-01)]),("P", [(1.4442900000E+00, 6.3850600000E-01), (5.5262100000E-01, 4.2536500000E-01)]),("P", [(1.8872000000E-01, 1.0000000000E+00)]),("P", [(4.6501000000E-02, 1.0000000000E+00)]),("P", [(1.6285000000E-02, 1.0000000000E+00)]),("S", [(7.6000000000E-03, 1.0000000000E+00)]),("P", [(7.6000000000E-03, 1.0000000000E+00)]),("D", [(7.0000000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)]),("D", [(4.3750000000E-02, 1.0000000000E+00)]),("F", [(1.5000000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(4.3866500000E+04, 9.1800000000E-04), (6.6053700000E+03, 7.0470000000E-03), (1.5132600000E+03, 3.5941000000E-02), (4.3231700000E+02, 1.4146100000E-01), (1.4214900000E+02, 4.2676400000E-01), (5.1398300000E+01, 4.9797500000E-01)]),("S", [(5.1398300000E+01, 2.5135500000E-01), (1.9919600000E+01, 6.1867100000E-01), (8.0247400000E+00, 1.8841700000E-01)]),("S", [(2.5081700000E+00, 1.0000000000E+00)]),("S", [(8.7153100000E-01, 1.0000000000E+00)]),("S", [(1.0818800000E-01, 1.0000000000E+00)]),("S", [(4.0130000000E-02, 1.0000000000E+00)]),("P", [(1.9385400000E+02, 1.0188000000E-02), (4.5442000000E+01, 7.5360000000E-02), (1.4186400000E+01, 3.0741900000E-01), (5.0575100000E+00, 7.1757500000E-01)]),("P", [(1.8886100000E+00, 6.6733900000E-01), (7.2265200000E-01, 3.9464900000E-01)]),("P", [(2.3641700000E-01, 1.0000000000E+00)]),("P", [(9.3358000000E-02, 1.0000000000E+00)]),("P", [(3.4809000000E-02, 1.0000000000E+00)]),("S", [(1.4600000000E-02, 1.0000000000E+00)]),("P", [(1.4600000000E-02, 1.0000000000E+00)]),("D", [(7.0000000000E-01, 1.0000000000E+00)]),("D", [(1.7500000000E-01, 1.0000000000E+00)]),("D", [(4.3750000000E-02, 1.0000000000E+00)]),("F", [(2.0000000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(5.4866489000E+04, 8.3900000000E-04), (8.2117665000E+03, 6.5270000000E-03), (1.8661761000E+03, 3.3666000000E-02), (5.3112934000E+02, 1.3290200000E-01), (1.7511797000E+02, 4.0126600000E-01), (6.4005500000E+01, 5.3133800000E-01)]),("S", [(6.4005500000E+01, 2.0230500000E-01), (2.5292507000E+01, 6.2479000000E-01), (1.0534910000E+01, 2.2743900000E-01)]),("S", [(3.2067110000E+00, 1.0000000000E+00)]),("S", [(1.1525550000E+00, 1.0000000000E+00)]),("S", [(1.7667800000E-01, 1.0000000000E+00)]),("S", [(6.5237000000E-02, 1.0000000000E+00)]),("P", [(2.5928362000E+02, 9.4480000000E-03), (6.1076870000E+01, 7.0974000000E-02), (1.9303237000E+01, 2.9563600000E-01), (7.0108820000E+00, 7.2821900000E-01)]),("P", [(2.6738650000E+00, 6.4446700000E-01), (1.0365960000E+00, 4.1741300000E-01)]),("P", [(3.1681900000E-01, 1.0000000000E+00)]),("P", [(1.1425700000E-01, 1.0000000000E+00)]),("S", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(3.1800000000E-02, 1.0000000000E+00)]),("P", [(4.1397000000E-02, 1.0000000000E+00)]),("D", [(1.3000000000E+00, 1.0000000000E+00)]),("D", [(3.2500000000E-01, 1.0000000000E+00)]),("D", [(8.1250000000E-02, 1.0000000000E+00)]),("F", [(2.5000000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(6.9379230000E+04, 7.5700000000E-04), (1.0354940000E+04, 5.9320000000E-03), (2.3338796000E+03, 3.1088000000E-02), (6.5714295000E+02, 1.2496700000E-01), (2.1430113000E+02, 3.8689700000E-01), (7.7629168000E+01, 5.5488800000E-01)]),("S", [(7.7629168000E+01, 1.7788100000E-01), (3.0630807000E+01, 6.2776500000E-01), (1.2801295000E+01, 2.4762300000E-01)]),("S", [(3.9268660000E+00, 1.0000000000E+00)]),("S", [(1.4523430000E+00, 1.0000000000E+00)]),("S", [(2.5623400000E-01, 1.0000000000E+00)]),("S", [(9.4279000000E-02, 1.0000000000E+00)]),("P", [(3.3548319000E+02, 8.8660000000E-03), (7.8900366000E+01, 6.8299000000E-02), (2.4988150000E+01, 2.9095800000E-01), (9.2197110000E+00, 7.3211700000E-01)]),("P", [(3.6211400000E+00, 6.1987900000E-01), (1.4513100000E+00, 4.3914800000E-01)]),("P", [(5.0497700000E-01, 1.0000000000E+00)]),("P", [(1.8631700000E-01, 1.0000000000E+00)]),("P", [(6.5432000000E-02, 1.0000000000E+00)]),("S", [(3.3100000000E-02, 1.0000000000E+00)]),("P", [(3.3100000000E-02, 1.0000000000E+00)]),("D", [(1.8000000000E+00, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)]),("D", [(1.1250000000E-01, 1.0000000000E+00)]),("F", [(3.2000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(7.7492400000E+04, 7.8100000000E-04), (1.1605800000E+04, 6.0680000000E-03), (2.6459600000E+03, 3.1160000000E-02), (7.5497600000E+02, 1.2343100000E-01), (2.4875500000E+02, 3.7820900000E-01), (9.1156500000E+01, 5.6326200000E-01)]),("S", [(9.1156500000E+01, 1.6025500000E-01), (3.6225700000E+01, 6.2764700000E-01), (1.5211300000E+01, 2.6384900000E-01)]),("S", [(4.7941700000E+00, 1.0000000000E+00)]),("S", [(1.8079300000E+00, 1.0000000000E+00)]),("S", [(3.5681600000E-01, 1.0000000000E+00)]),("S", [(1.1478300000E-01, 1.0000000000E+00)]),("P", [(3.8484300000E+02, 9.2060000000E-03), (9.0552100000E+01, 6.9874000000E-02), (2.9133900000E+01, 2.9247000000E-01), (1.0886200000E+01, 7.2810300000E-01)]),("P", [(4.3525900000E+00, 6.2834900000E-01), (1.7770600000E+00, 4.2804400000E-01)]),("P", [(6.9700500000E-01, 1.0000000000E+00)]),("P", [(2.5353200000E-01, 1.0000000000E+00)]),("P", [(6.8493000000E-02, 1.0000000000E+00)]),("S", [(3.4800000000E-02, 1.0000000000E+00)]),("P", [(3.4800000000E-02, 1.0000000000E+00)]),("D", [(2.2000000000E+00, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)]),("D", [(1.3750000000E-01, 1.0000000000E+00)]),("F", [(4.5000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(9.3413400000E+04, 7.4300000000E-04), (1.3961700000E+04, 5.7930000000E-03), (3.1699100000E+03, 2.9954000000E-02), (9.0245600000E+02, 1.1902800000E-01), (2.9715800000E+02, 3.6843200000E-01), (1.0870200000E+02, 5.7729900000E-01)]),("S", [(1.0870200000E+02, 1.4318600000E-01), (4.3155300000E+01, 6.2446500000E-01), (1.8107900000E+01, 2.8336600000E-01)]),("S", [(5.5600900000E+00, 1.0000000000E+00)]),("S", [(2.1318300000E+00, 1.0000000000E+00)]),("S", [(4.2040300000E-01, 1.0000000000E+00)]),("S", [(1.3604500000E-01, 1.0000000000E+00)]),("P", [(4.9504000000E+02, 8.3090000000E-03), (1.1722100000E+02, 6.4024000000E-02), (3.7774900000E+01, 2.7761400000E-01), (1.4058400000E+01, 7.4507600000E-01)]),("P", [(5.5657400000E+00, 6.1371200000E-01), (2.2629700000E+00, 4.4381800000E-01)]),("P", [(8.0799400000E-01, 1.0000000000E+00)]),("P", [(2.7746000000E-01, 1.0000000000E+00)]),("P", [(7.7141000000E-02, 1.0000000000E+00)]),("S", [(4.0500000000E-02, 1.0000000000E+00)]),("P", [(4.0500000000E-02, 1.0000000000E+00)]),("D", [(2.6000000000E+00, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("D", [(1.6250000000E-01, 1.0000000000E+00)]),("F", [(5.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.0581900000E+05, 7.3800000000E-04), (1.5872000000E+04, 5.7180000000E-03), (3.6196500000E+03, 2.9495000000E-02), (1.0308000000E+03, 1.1728600000E-01), (3.3990800000E+02, 3.6294900000E-01), (1.2453800000E+02, 5.8414900000E-01)]),("S", [(1.2453800000E+02, 1.3417700000E-01), (4.9513500000E+01, 6.2425000000E-01), (2.0805600000E+01, 2.9175600000E-01)]),("S", [(6.5834600000E+00, 1.0000000000E+00)]),("S", [(2.5646800000E+00, 1.0000000000E+00)]),("S", [(5.5976300000E-01, 1.0000000000E+00)]),("S", [(1.8327300000E-01, 1.0000000000E+00)]),("P", [(5.8977600000E+02, 2.3910000000E-03), (1.3984900000E+02, 1.8504000000E-02), (4.5141300000E+01, 8.1377000000E-02), (1.6873300000E+01, 2.2155200000E-01), (6.7411000000E+00, 7.7256900000E-01)]),("P", [(6.7411000000E+00, -1.5722440000E+00), (2.7715200000E+00, 9.9238900000E-01)]),("P", [(1.0238700000E+00, 1.0000000000E+00)]),("P", [(3.8136800000E-01, 1.0000000000E+00)]),("P", [(1.0943700000E-01, 1.0000000000E+00)]),("S", [(4.8300000000E-02, 1.0000000000E+00)]),("P", [(4.8300000000E-02, 1.0000000000E+00)]),("D", [(3.0000000000E+00, 1.0000000000E+00)]),("D", [(7.5000000000E-01, 1.0000000000E+00)]),("D", [(1.8750000000E-01, 1.0000000000E+00)]),("F", [(7.0000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.1802238000E+05, 7.4700000000E-04), (1.7683541000E+04, 5.7900000000E-03), (4.0277657000E+03, 2.9919000000E-02), (1.1453977000E+03, 1.1920600000E-01), (3.7716375000E+02, 3.6902800000E-01), (1.3815969000E+02, 5.7645900000E-01)]),("S", [(1.3815969000E+02, 1.4392700000E-01), (5.4989117000E+01, 6.2293800000E-01), (2.3170667000E+01, 2.8396400000E-01)]),("S", [(7.3778600000E+00, 1.0000000000E+00)]),("S", [(2.9236880000E+00, 1.0000000000E+00)]),("S", [(6.5040500000E-01, 1.0000000000E+00)]),("S", [(2.3282500000E-01, 1.0000000000E+00)]),("P", [(6.6306201000E+02, 3.0820000000E-03), (1.5709281000E+02, 2.4165000000E-02), (5.0231100000E+01, 1.0822300000E-01), (1.8635345000E+01, 2.9419200000E-01), (7.4465370000E+00, 6.8786200000E-01)]),("P", [(7.4465370000E+00, -1.2144820000E-01), (3.0956980000E+00, 1.6323700000E-01)]),("P", [(1.1064630000E+00, 1.0000000000E+00)]),("P", [(4.1560100000E-01, 1.0000000000E+00)]),("P", [(1.4544900000E-01, 1.0000000000E+00)]),("S", [(6.0000000000E-02, 1.0000000000E+00)]),("P", [(6.0000000000E-02, 1.0000000000E+00)]),("D", [(3.4000000000E+00, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)]),("D", [(2.1250000000E-01, 1.0000000000E+00)]),("F", [(8.5000000000E-01, 1.0000000000E+00)])],
}


PC_0 = {

    1 : [("S", [(4.3448000000E+00, 7.9299000000E-02), (6.6049000000E-01, 4.2422000000E-01)]),("S", [(1.3669000000E-01, 1.0000000000E+00)])],
    2 : [("S", [(1.3210000000E+01, 4.3506000000E-02), (1.9434000000E+00, 2.1166000000E-01)]),("S", [(3.5443000000E-01, 1.0000000000E+00)])],
    3 : [("S", [(9.7585000000E+01, 2.2358000000E-02), (1.4755000000E+01, 1.4499000000E-01), (3.2233000000E+00, 3.4594000000E-01)]),("S", [(1.4755000000E+01, 5.1875000000E-03), (3.2233000000E+00, 1.6698000000E-01), (7.8789000000E-01, 7.2566000000E-01)]),("S", [(5.1009000000E-02, 1.0000000000E+00)]),("P", [(3.3000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(1.8754000000E+02, 1.0373000000E-02), (2.8345000000E+01, 7.2007000000E-02), (6.2614000000E+00, 1.3085000000E-01)]),("S", [(2.8345000000E+01, -2.0507000000E-03), (6.2614000000E+00, 1.1312000000E-01), (1.5837000000E+00, 3.0800000000E-01)]),("S", [(1.0858000000E-01, 1.0000000000E+00)]),("P", [(4.3000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(3.0373000000E+02, 2.0121000000E-02), (4.5894000000E+01, 1.4410000000E-01), (1.0205000000E+01, 2.1961000000E-01)]),("S", [(4.5894000000E+01, -7.3818000000E-03), (1.0205000000E+01, 2.6153000000E-01), (2.6348000000E+00, 5.9363000000E-01)]),("S", [(1.9380000000E-01, 1.0000000000E+00)]),("P", [(2.7904000000E+00, 1.0986000000E-01), (5.4195000000E-01, 4.6846000000E-01)]),("P", [(1.1876000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(4.4651000000E+02, 1.9788000000E-02), (6.7462000000E+01, 1.4534000000E-01), (1.5067000000E+01, 1.8529000000E-01)]),("S", [(6.7462000000E+01, -9.9910000000E-03), (1.5067000000E+01, 2.8519000000E-01), (3.9388000000E+00, 5.6687000000E-01)]),("S", [(2.9991000000E-01, 1.0000000000E+00)]),("P", [(4.3677000000E+00, 1.1671000000E-01), (8.6982000000E-01, 4.7558000000E-01)]),("P", [(1.8893000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(6.1585000000E+02, 1.9588000000E-02), (9.3044000000E+01, 1.4695000000E-01), (2.0848000000E+01, 1.5560000000E-01)]),("S", [(9.3044000000E+01, -1.2059000000E-02), (2.0848000000E+01, 3.0199000000E-01), (5.4959000000E+00, 5.4090000000E-01)]),("S", [(4.2624000000E-01, 1.0000000000E+00)]),("P", [(6.2680000000E+00, 1.2134000000E-01), (1.2690000000E+00, 4.8013000000E-01)]),("P", [(2.7390000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(8.1212000000E+02, 1.9450000000E-02), (1.2269000000E+02, 1.4774000000E-01), (2.7561000000E+01, 1.3593000000E-01)]),("S", [(1.2269000000E+02, -1.3189000000E-02), (2.7561000000E+01, 3.1510000000E-01), (7.3173000000E+00, 5.2964000000E-01)]),("S", [(5.7924000000E-01, 1.0000000000E+00)]),("P", [(8.2178000000E+00, 1.2902000000E-01), (1.6754000000E+00, 4.8706000000E-01)]),("P", [(3.5067000000E-01, 1.0000000000E+00)])],
    9 : [("S", [(1.0351000000E+03, 1.9351000000E-02), (1.5638000000E+02, 1.4884000000E-01), (3.5196000000E+01, 1.1735000000E-01)]),("S", [(1.5638000000E+02, -1.4337000000E-02), (3.5196000000E+01, 3.2551000000E-01), (9.3921000000E+00, 5.1681000000E-01)]),("S", [(7.5342000000E-01, 1.0000000000E+00)]),("P", [(1.0599000000E+01, 1.3273000000E-01), (2.1733000000E+00, 4.8938000000E-01)]),("P", [(4.4871000000E-01, 1.0000000000E+00)])],
    10 : [("S", [(1.2845000000E+03, 9.6419000000E-02), (1.9405000000E+02, 7.5019000000E-01), (4.3746000000E+01, 4.9765000000E-01)]),("S", [(1.9405000000E+02, -7.6708000000E-03), (4.3746000000E+01, 1.6693000000E-01), (1.1720000000E+01, 2.5151000000E-01)]),("S", [(9.4746000000E-01, 1.0000000000E+00)]),("P", [(1.3348000000E+01, 6.7467000000E-02), (2.7509000000E+00, 2.4531000000E-01)]),("P", [(5.6434000000E-01, 1.0000000000E+00)])],
    11 : [("S", [(4.5535000000E+03, 5.1583000000E-03), (6.8509000000E+02, 3.8600000000E-02), (1.5573000000E+02, 1.7385000000E-01), (4.3371000000E+01, 4.5646000000E-01), (1.3339000000E+01, 4.6291000000E-01)]),("S", [(6.8509000000E+02, -3.1854000000E-04), (1.5573000000E+02, -2.2648000000E-03), (4.3371000000E+01, -3.3923000000E-02), (1.3339000000E+01, -7.6309000000E-02), (2.3106000000E+00, 3.3727000000E-01)]),("S", [(1.5573000000E+02, -7.9131000000E-05), (4.3371000000E+01, -6.9906000000E-04), (1.3339000000E+01, -3.2083000000E-03), (2.3106000000E+00, 3.0056000000E-02), (7.0669000000E-01, 1.1224000000E-01)]),("S", [(4.5942000000E-02, 1.0000000000E+00)]),("P", [(3.7793000000E+01, 4.6495000000E-02), (8.4134000000E+00, 2.4311000000E-01), (2.3210000000E+00, 5.1738000000E-01), (6.0658000000E-01, 4.2598000000E-01)]),("P", [(9.5000000000E-02, 1.0000000000E+00)])],
    12 : [("S", [(5.5208000000E+03, 2.5254000000E-03), (8.3056000000E+02, 1.8932000000E-02), (1.8881000000E+02, 8.5565000000E-02), (5.2638000000E+01, 2.2776000000E-01), (1.6233000000E+01, 2.3565000000E-01)]),("S", [(8.3056000000E+02, -1.5190000000E-04), (1.8881000000E+02, -8.7484000000E-04), (5.2638000000E+01, -1.4607000000E-02), (1.6233000000E+01, -3.0349000000E-02), (2.9275000000E+00, 1.0360000000E-01)]),("S", [(1.8881000000E+02, -1.1604000000E-03), (5.2638000000E+01, -9.8845000000E-03), (1.6233000000E+01, -4.3416000000E-02), (2.9275000000E+00, 3.1196000000E-01), (9.4786000000E-01, 7.2645000000E-01)]),("S", [(7.6361000000E-02, 1.0000000000E+00)]),("P", [(4.9453000000E+01, 2.1739000000E-02), (1.1116000000E+01, 1.1690000000E-01), (3.1273000000E+00, 2.5993000000E-01), (8.5940000000E-01, 2.1108000000E-01)]),("P", [(1.6000000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(6.5871000000E+03, 4.9570000000E-03), (9.9092000000E+02, 3.7223000000E-02), (2.2529000000E+02, 1.6875000000E-01), (6.2866000000E+01, 4.5505000000E-01), (1.9432000000E+01, 4.8010000000E-01)]),("S", [(9.9092000000E+02, -2.9525000000E-04), (2.2529000000E+02, -1.5674000000E-03), (6.2866000000E+01, -2.7434000000E-02), (1.9432000000E+01, -5.5600000000E-02), (3.6207000000E+00, 1.5188000000E-01)]),("S", [(2.2529000000E+02, -3.4383000000E-04), (6.2866000000E+01, -2.9381000000E-03), (1.9432000000E+01, -1.3047000000E-02), (3.6207000000E+00, 8.5679000000E-02), (1.2288000000E+00, 1.8025000000E-01)]),("S", [(1.2068000000E-01, 1.0000000000E+00)]),("P", [(7.2365000000E+01, 3.2953000000E-02), (1.6543000000E+01, 1.8989000000E-01), (4.8239000000E+00, 4.7489000000E-01), (1.4336000000E+00, 4.8033000000E-01)]),("P", [(1.6543000000E+01, -3.0607000000E-04), (4.8239000000E+00, -8.1254000000E-03), (1.4336000000E+00, 5.8216000000E-03), (2.7689000000E-01, 3.7831000000E-01)]),("P", [(6.9856000000E-02, 1.0000000000E+00)])],
    14 : [("S", [(7.7430000000E+03, 4.8825000000E-03), (1.1647000000E+03, 3.6741000000E-02), (2.6482000000E+02, 1.6695000000E-01), (7.3968000000E+01, 4.5657000000E-01), (2.2912000000E+01, 4.9043000000E-01)]),("S", [(1.1647000000E+03, -2.9111000000E-04), (2.6482000000E+02, -1.4122000000E-03), (7.3968000000E+01, -2.6046000000E-02), (2.2912000000E+01, -5.0933000000E-02), (4.3741000000E+00, 1.0810000000E-01)]),("S", [(2.6482000000E+02, -4.4437000000E-04), (7.3968000000E+01, -3.8067000000E-03), (2.2912000000E+01, -1.7078000000E-02), (4.3741000000E+00, 1.0569000000E-01), (1.5391000000E+00, 2.0429000000E-01)]),("S", [(1.6775000000E-01, 1.0000000000E+00)]),("P", [(9.6080000000E+01, 2.8035000000E-02), (2.2150000000E+01, 1.6819000000E-01), (6.5630000000E+00, 4.5197000000E-01), (2.0250000000E+00, 5.0132000000E-01)]),("P", [(2.2150000000E+01, -7.2797000000E-04), (6.5630000000E+00, -8.6644000000E-03), (2.0250000000E+00, -3.5344000000E-03), (4.7780000000E-01, 3.1085000000E-01)]),("P", [(1.2824000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(8.9908000000E+03, 4.8206000000E-03), (1.3524000000E+03, 3.6392000000E-02), (3.0750000000E+02, 1.6563000000E-01), (8.5941000000E+01, 4.6217000000E-01), (2.6671000000E+01, 5.0582000000E-01)]),("S", [(1.3524000000E+03, 2.8900000000E-03), (3.0750000000E+02, 1.2638000000E-02), (8.5941000000E+01, 2.4790000000E-01), (2.6671000000E+01, 4.6165000000E-01), (5.1881000000E+00, -7.0180000000E-01)]),("S", [(3.0750000000E+02, -5.4512000000E-04), (8.5941000000E+01, -4.6838000000E-03), (2.6671000000E+01, -2.1160000000E-02), (5.1881000000E+00, 1.2534000000E-01), (1.8778000000E+00, 2.2427000000E-01)]),("S", [(2.2010000000E-01, 1.0000000000E+00)]),("P", [(1.1164000000E+02, 2.8663000000E-02), (2.5784000000E+01, 1.7265000000E-01), (7.6725000000E+00, 4.6181000000E-01), (2.3996000000E+00, 4.9174000000E-01)]),("P", [(2.5784000000E+01, -6.8545000000E-04), (7.6725000000E+00, -1.1296000000E-02), (2.3996000000E+00, 3.2962000000E-03), (5.4943000000E-01, 4.1613000000E-01)]),("P", [(1.4748000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(1.0330000000E+04, 4.7685000000E-03), (1.5537000000E+03, 3.6209000000E-02), (3.5330000000E+02, 1.6491000000E-01), (9.8805000000E+01, 4.7548000000E-01), (3.0716000000E+01, 5.3319000000E-01)]),("S", [(1.5537000000E+03, 2.8903000000E-03), (3.5330000000E+02, 1.1575000000E-02), (9.8805000000E+01, 2.3993000000E-01), (3.0716000000E+01, 4.2830000000E-01), (6.0633000000E+00, -4.2063000000E-01)]),("S", [(3.5330000000E+02, -6.2501000000E-04), (9.8805000000E+01, -5.3881000000E-03), (3.0716000000E+01, -2.4521000000E-02), (6.0633000000E+00, 1.4132000000E-01), (2.2475000000E+00, 2.3948000000E-01)]),("S", [(2.7853000000E-01, 1.0000000000E+00)]),("P", [(1.2976000000E+02, 2.8706000000E-02), (3.0025000000E+01, 1.7420000000E-01), (8.9733000000E+00, 4.6709000000E-01), (2.8440000000E+00, 4.8571000000E-01)]),("P", [(3.0025000000E+01, -1.1785000000E-04), (8.9733000000E+00, -1.4483000000E-02), (2.8440000000E+00, 1.5939000000E-02), (6.3361000000E-01, 4.8257000000E-01)]),("P", [(1.6955000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(1.1759000000E+04, 4.7248000000E-03), (1.7686000000E+03, 3.6546000000E-02), (4.0219000000E+02, 1.6597000000E-01), (1.1254000000E+02, 5.2530000000E-01), (3.5039000000E+01, 6.1955000000E-01)]),("S", [(1.7686000000E+03, 2.9053000000E-03), (4.0219000000E+02, 1.0664000000E-02), (1.1254000000E+02, 2.3406000000E-01), (3.5039000000E+01, 3.9915000000E-01), (6.9956000000E+00, -1.7937000000E-01)]),("S", [(4.0219000000E+02, -6.9197000000E-04), (1.1254000000E+02, -5.9760000000E-03), (3.5039000000E+01, -2.7394000000E-02), (6.9956000000E+00, 1.5516000000E-01), (2.6455000000E+00, 2.5028000000E-01)]),("S", [(3.4157000000E-01, 1.0000000000E+00)]),("P", [(1.5066000000E+02, 2.8299000000E-02), (3.4933000000E+01, 1.7336000000E-01), (1.0492000000E+01, 4.6812000000E-01), (3.3726000000E+00, 4.8321000000E-01)]),("P", [(3.4933000000E+01, 1.3598000000E-04), (1.0492000000E+01, -1.6611000000E-02), (3.3726000000E+00, 2.2351000000E-02), (7.5989000000E-01, 5.1192000000E-01)]),("P", [(2.0289000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(1.3277000000E+04, 2.3440000000E-02), (1.9969000000E+03, 1.5225000000E-01), (4.5412000000E+02, 7.2536000000E-01), (1.2714000000E+02, 3.3180000000E-01), (3.9637000000E+01, -6.3751000000E-01)]),("S", [(1.9969000000E+03, 1.4689000000E-03), (4.5412000000E+02, 4.9282000000E-03), (1.2714000000E+02, 1.1458000000E-01), (3.9637000000E+01, 1.8548000000E-01), (7.9817000000E+00, 2.1894000000E-02)]),("S", [(4.5412000000E+02, -3.7469000000E-04), (1.2714000000E+02, -3.2462000000E-03), (3.9637000000E+01, -1.4968000000E-02), (7.9817000000E+00, 8.3936000000E-02), (3.0706000000E+00, 1.2891000000E-01)]),("S", [(4.0908000000E-01, 1.0000000000E+00)]),("P", [(1.7323000000E+02, 1.3955000000E-02), (4.0239000000E+01, 8.6226000000E-02), (1.2139000000E+01, 2.3423000000E-01), (3.9527000000E+00, 2.4062000000E-01)]),("P", [(4.0239000000E+01, 1.3131000000E-04), (1.2139000000E+01, -9.1482000000E-03), (3.9527000000E+00, 1.3280000000E-02), (9.0626000000E-01, 2.6503000000E-01)]),("P", [(2.4252000000E-01, 1.0000000000E+00)])],
}



PC_1 = {

    1 : [("S", [(1.2252000000E+01, 2.2822000000E-02), (1.8687000000E+00, 1.5564000000E-01), (4.1821000000E-01, 4.8898000000E-01)]),("S", [(1.0610000000E-01, 1.0000000000E+00)]),("P", [(1.0000000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(3.6865000000E+01, 1.3062000000E-02), (5.5802000000E+00, 8.3491000000E-02), (1.1917000000E+00, 2.3874000000E-01)]),("S", [(2.6892000000E-01, 1.0000000000E+00)]),("P", [(1.4500000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(2.6643000000E+02, 6.5268000000E-03), (4.0160000000E+01, 4.8065000000E-02), (9.1082000000E+00, 2.0095000000E-01), (2.4543000000E+00, 4.7007000000E-01), (6.9626000000E-01, 4.3130000000E-01)]),("S", [(4.0160000000E+01, 1.1022000000E-05), (9.1082000000E+00, -2.6849000000E-03), (2.4543000000E+00, -1.8711000000E-02), (6.9626000000E-01, -8.8948000000E-02), (8.3726000000E-02, 5.1836000000E-01)]),("S", [(3.0827000000E-02, 1.0000000000E+00)]),("P", [(1.6200000000E+00, 4.1655100000E-02), (3.2000000000E-01, 1.1044500000E-01)]),("P", [(9.0000000000E-02, 1.0000000000E+00)])],
    4 : [("S", [(5.1490000000E+02, 3.0031000000E-03), (7.7558000000E+01, 2.2243000000E-02), (1.7609000000E+01, 9.5434000000E-02), (4.8080000000E+00, 2.3228000000E-01), (1.4085000000E+00, 2.2104000000E-01)]),("S", [(7.7558000000E+01, -9.6000000000E-05), (1.7609000000E+01, -1.2179000000E-03), (4.8080000000E+00, -1.3699000000E-02), (1.4085000000E+00, -4.2159000000E-02), (1.9069000000E-01, 2.6185000000E-01)]),("S", [(6.1125000000E-02, 1.0000000000E+00)]),("P", [(3.4200000000E+00, 4.9351900000E-02), (6.8000000000E-01, 1.8982100000E-01)]),("P", [(1.9000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(8.4398000000E+02, 5.7361000000E-03), (1.2708000000E+02, 4.2625000000E-02), (2.8862000000E+01, 1.8576000000E-01), (7.9297000000E+00, 4.6258000000E-01), (2.3637000000E+00, 4.4646000000E-01)]),("S", [(1.2708000000E+02, -2.3303000000E-04), (2.8862000000E+01, -2.5479000000E-03), (7.9297000000E+00, -3.0928000000E-02), (2.3637000000E+00, -8.6956000000E-02), (3.4670000000E-01, 5.3062000000E-01)]),("S", [(1.0595000000E-01, 1.0000000000E+00)]),("P", [(5.8128000000E+00, 4.3084000000E-02), (1.2002000000E+00, 2.2295000000E-01), (3.2686000000E-01, 5.0977000000E-01)]),("P", [(8.6264000000E-02, 1.0000000000E+00)]),("D", [(7.0000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(1.2526000000E+03, 5.5736000000E-03), (1.8857000000E+02, 4.1496000000E-02), (4.2839000000E+01, 1.8263000000E-01), (1.1818000000E+01, 4.6129000000E-01), (3.5567000000E+00, 4.4931000000E-01)]),("S", [(1.8857000000E+02, -2.7440000000E-04), (4.2839000000E+01, -2.5583000000E-03), (1.1818000000E+01, -3.3375000000E-02), (3.5567000000E+00, -8.7305000000E-02), (5.4258000000E-01, 5.3415000000E-01)]),("S", [(1.6058000000E-01, 1.0000000000E+00)]),("P", [(9.1426000000E+00, 4.4499000000E-02), (1.9298000000E+00, 2.3108000000E-01), (5.2522000000E-01, 5.1227000000E-01)]),("P", [(1.3608000000E-01, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(1.7422000000E+03, 5.4603000000E-03), (2.6221000000E+02, 4.0706000000E-02), (5.9585000000E+01, 1.8038000000E-01), (1.6489000000E+01, 4.6009000000E-01), (4.9945000000E+00, 4.5173000000E-01)]),("S", [(2.6221000000E+02, -3.0104000000E-04), (5.9585000000E+01, -2.5511000000E-03), (1.6489000000E+01, -3.4900000000E-02), (4.9945000000E+00, -8.7429000000E-02), (7.8673000000E-01, 5.2660000000E-01)]),("S", [(2.2837000000E-01, 1.0000000000E+00)]),("P", [(1.3132000000E+01, 4.5515000000E-02), (2.8108000000E+00, 2.3740000000E-01), (7.6351000000E-01, 5.1392000000E-01)]),("P", [(1.9560000000E-01, 1.0000000000E+00)]),("D", [(9.0000000000E-01, 1.0000000000E+00)])],
    8 : [("S", [(2.3067000000E+03, 5.3940000000E-03), (3.4715000000E+02, 4.0248000000E-02), (7.8890000000E+01, 1.7921000000E-01), (2.1876000000E+01, 4.5978000000E-01), (6.6646000000E+00, 4.5234000000E-01)]),("S", [(3.4715000000E+02, -3.1692000000E-04), (7.8890000000E+01, -2.5944000000E-03), (2.1876000000E+01, -3.6241000000E-02), (6.6646000000E+00, -8.7790000000E-02), (1.0669000000E+00, 5.3320000000E-01)]),("S", [(3.0700000000E-01, 1.0000000000E+00)]),("P", [(1.7022000000E+01, 4.8919000000E-02), (3.6838000000E+00, 2.4962000000E-01), (9.9234000000E-01, 5.1347000000E-01)]),("P", [(2.4487000000E-01, 1.0000000000E+00)]),("D", [(1.0000000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(2.9546000000E+03, 5.3337000000E-03), (4.4462000000E+02, 3.9827000000E-02), (1.0105000000E+02, 1.7802000000E-01), (2.8068000000E+01, 4.5917000000E-01), (8.5837000000E+00, 4.5352000000E-01)]),("S", [(4.4462000000E+02, -3.2962000000E-04), (1.0105000000E+02, -2.6083000000E-03), (2.8068000000E+01, -3.7150000000E-02), (8.5837000000E+00, -8.8178000000E-02), (1.3986000000E+00, 5.3135000000E-01)]),("S", [(3.9969000000E-01, 1.0000000000E+00)]),("P", [(2.1883000000E+01, 5.0244000000E-02), (4.7769000000E+00, 2.5424000000E-01), (1.2823000000E+00, 5.1220000000E-01)]),("P", [(3.1127000000E-01, 1.0000000000E+00)]),("D", [(1.1000000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(3.6801000000E+03, 2.6454000000E-03), (5.5375000000E+02, 1.9763000000E-02), (1.2587000000E+02, 8.8590000000E-02), (3.5013000000E+01, 2.2936000000E-01), (1.0739000000E+01, 2.2717000000E-01)]),("S", [(5.5375000000E+02, -1.7209000000E-04), (1.2587000000E+02, -1.3016000000E-03), (3.5013000000E+01, -1.8980000000E-02), (1.0739000000E+01, -4.4008000000E-02), (1.7693000000E+00, 2.6525000000E-01)]),("S", [(5.0149000000E-01, 1.0000000000E+00)]),("P", [(2.7502000000E+01, 2.5477000000E-02), (6.0397000000E+00, 1.2855000000E-01), (1.6190000000E+00, 2.5586000000E-01)]),("P", [(3.8970000000E-01, 1.0000000000E+00)]),("D", [(1.2000000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(2.3207000000E+04, 6.8100000000E-04), (3.4816000000E+03, 5.2512000000E-03), (7.9272000000E+02, 2.6845000000E-02), (2.2459000000E+02, 1.0188000000E-01), (7.3088000000E+01, 2.7784000000E-01), (2.6190000000E+01, 4.4489000000E-01), (9.8661000000E+00, 2.8206000000E-01), (2.2155000000E+00, 2.0685000000E-02)]),("S", [(3.4816000000E+03, -2.9323000000E-05), (7.9272000000E+02, -3.2202000000E-05), (2.2459000000E+02, -2.1485000000E-03), (7.3088000000E+01, -9.3157000000E-03), (2.6190000000E+01, -6.7071000000E-02), (9.8661000000E+00, -5.6789000000E-02), (2.2155000000E+00, 5.3608000000E-01), (6.8590000000E-01, 5.9788000000E-01)]),("S", [(7.9272000000E+02, -1.3245000000E-06), (2.2459000000E+02, -1.4384000000E-05), (7.3088000000E+01, -1.3687000000E-04), (2.6190000000E+01, -4.8799000000E-04), (9.8661000000E+00, -2.8773000000E-04), (2.2155000000E+00, -1.0178000000E-02), (6.8590000000E-01, -1.1118000000E-01), (7.0337000000E-02, 5.9314000000E-01)]),("S", [(2.6075000000E-02, 1.0000000000E+00)]),("P", [(1.3555000000E+02, 6.1759000000E-03), (3.1562000000E+01, 4.5315000000E-02), (9.6774000000E+00, 1.7332000000E-01), (3.3691000000E+00, 3.6867000000E-01), (1.1703000000E+00, 4.3998000000E-01), (3.7651000000E-01, 2.2613000000E-01)]),("P", [(7.5000000000E-02, 1.0000000000E+00)])],
    12 : [("S", [(2.8706000000E+04, 3.2513000000E-04), (4.3066000000E+03, 2.5076000000E-03), (9.8054000000E+02, 1.2837000000E-02), (2.7776000000E+02, 4.8964000000E-02), (9.0386000000E+01, 1.3517000000E-01), (3.2374000000E+01, 2.2193000000E-01), (1.2177000000E+01, 1.4657000000E-01), (2.8166000000E+00, 1.1980000000E-02)]),("S", [(4.3066000000E+03, -1.3056000000E-05), (9.8054000000E+02, -1.7980000000E-05), (2.7776000000E+02, -1.0036000000E-03), (9.0386000000E+01, -4.6819000000E-03), (3.2374000000E+01, -3.3051000000E-02), (1.2177000000E+01, -3.2718000000E-02), (2.8166000000E+00, 2.6039000000E-01), (9.2173000000E-01, 3.0808000000E-01)]),("S", [(9.8054000000E+02, 1.1627000000E-06), (2.7776000000E+02, 1.3509000000E-05), (9.0386000000E+01, 1.3542000000E-04), (3.2374000000E+01, 7.5360000000E-04), (1.2177000000E+01, 1.5098000000E-03), (2.8166000000E+00, -1.9068000000E-02), (9.2173000000E-01, -7.5241000000E-02), (1.2709000000E-01, 2.7161000000E-01)]),("S", [(4.6080000000E-02, 1.0000000000E+00)]),("P", [(1.7760000000E+02, 2.8259000000E-03), (4.1442000000E+01, 2.1103000000E-02), (1.2818000000E+01, 8.2666000000E-02), (4.5174000000E+00, 1.8285000000E-01), (1.6128000000E+00, 2.2357000000E-01), (5.4531000000E-01, 1.0869000000E-01)]),("P", [(1.3500000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(3.4702000000E+04, 6.2778000000E-04), (5.2059000000E+03, 4.8432000000E-03), (1.1852000000E+03, 2.4817000000E-02), (3.3570000000E+02, 9.5028000000E-02), (1.0923000000E+02, 2.6474000000E-01), (3.9120000000E+01, 4.4274000000E-01), (1.4709000000E+01, 3.0134000000E-01), (3.4896000000E+00, 2.6817000000E-02)]),("S", [(5.2059000000E+03, -2.3879000000E-05), (1.1852000000E+03, -3.9366000000E-05), (3.3570000000E+02, -1.9168000000E-03), (1.0923000000E+02, -9.4362000000E-03), (3.9120000000E+01, -6.5739000000E-02), (1.4709000000E+01, -7.2232000000E-02), (3.4896000000E+00, 5.0924000000E-01), (1.1936000000E+00, 6.2972000000E-01)]),("S", [(1.1852000000E+03, 4.5128000000E-06), (3.3570000000E+02, 5.3226000000E-05), (1.0923000000E+02, 5.3333000000E-04), (3.9120000000E+01, 2.8754000000E-03), (1.4709000000E+01, 5.6734000000E-03), (3.4896000000E+00, -5.8133000000E-02), (1.1936000000E+00, -1.8753000000E-01), (1.9585000000E-01, 5.8570000000E-01)]),("S", [(7.1439000000E-02, 1.0000000000E+00)]),("P", [(2.5436000000E+02, 4.2903000000E-03), (5.9514000000E+01, 3.3055000000E-02), (1.8708000000E+01, 1.3486000000E-01), (6.7362000000E+00, 3.2836000000E-01), (2.5045000000E+00, 4.4798000000E-01), (9.0820000000E-01, 2.7040000000E-01)]),("P", [(5.9514000000E+01, 1.0756000000E-04), (1.8708000000E+01, -8.1981000000E-04), (6.7362000000E+00, -1.4216000000E-03), (2.5045000000E+00, -1.4930000000E-02), (9.0820000000E-01, 4.3779000000E-02), (2.0909000000E-01, 4.9039000000E-01)]),("P", [(5.6990000000E-02, 1.0000000000E+00)]),("D", [(4.0000000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(4.1263000000E+04, 6.0948000000E-04), (6.1896000000E+03, 4.7040000000E-03), (1.4087000000E+03, 2.4136000000E-02), (3.9877000000E+02, 9.2762000000E-02), (1.2969000000E+02, 2.6047000000E-01), (4.6425000000E+01, 4.4181000000E-01), (1.7455000000E+01, 3.0755000000E-01), (4.2222000000E+00, 2.9197000000E-02)]),("S", [(6.1896000000E+03, -2.2459000000E-05), (1.4087000000E+03, -4.1760000000E-05), (3.9877000000E+02, -1.8570000000E-03), (1.2969000000E+02, -9.5113000000E-03), (4.6425000000E+01, -6.5800000000E-02), (1.7455000000E+01, -7.7546000000E-02), (4.2222000000E+00, 5.0303000000E-01), (1.4933000000E+00, 6.3824000000E-01)]),("S", [(1.4087000000E+03, 6.0314000000E-06), (3.9877000000E+02, 7.4278000000E-05), (1.2969000000E+02, 7.4023000000E-04), (4.6425000000E+01, 4.0107000000E-03), (1.7455000000E+01, 8.0137000000E-03), (4.2222000000E+00, -7.4091000000E-02), (1.4933000000E+00, -2.1382000000E-01), (2.7174000000E-01, 6.1263000000E-01)]),("S", [(9.8502000000E-02, 1.0000000000E+00)]),("P", [(3.1073000000E+02, 4.1198000000E-03), (7.2823000000E+01, 3.1981000000E-02), (2.2985000000E+01, 1.3224000000E-01), (8.3213000000E+00, 3.2779000000E-01), (3.1353000000E+00, 4.5107000000E-01), (1.1693000000E+00, 2.6345000000E-01)]),("P", [(7.2823000000E+01, 1.2637000000E-04), (2.2985000000E+01, -1.0252000000E-03), (8.3213000000E+00, -2.1337000000E-03), (3.1353000000E+00, -1.8774000000E-02), (1.1693000000E+00, 5.7528000000E-02), (2.9281000000E-01, 5.3571000000E-01)]),("P", [(8.2704000000E-02, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(4.8073000000E+04, 5.9919000000E-04), (7.2114000000E+03, 4.6241000000E-03), (1.6417000000E+03, 2.3724000000E-02), (4.6492000000E+02, 9.1303000000E-02), (1.5128000000E+02, 2.5750000000E-01), (5.4189000000E+01, 4.4080000000E-01), (2.0397000000E+01, 3.1175000000E-01), (5.0241000000E+00, 3.1118000000E-02)]),("S", [(7.2114000000E+03, -2.1309000000E-05), (1.6417000000E+03, -4.4215000000E-05), (4.6492000000E+02, -1.8221000000E-03), (1.5128000000E+02, -9.6366000000E-03), (5.4189000000E+01, -6.6072000000E-02), (2.0397000000E+01, -8.1824000000E-02), (5.0241000000E+00, 4.9850000000E-01), (1.8246000000E+00, 6.4492000000E-01)]),("S", [(1.6417000000E+03, 7.1340000000E-06), (4.6492000000E+02, 8.4500000000E-05), (1.5128000000E+02, 8.5919000000E-04), (5.4189000000E+01, 4.6478000000E-03), (2.0397000000E+01, 9.5811000000E-03), (5.0241000000E+00, -8.4066000000E-02), (1.8246000000E+00, -2.3124000000E-01), (3.5871000000E-01, 6.2350000000E-01)]),("S", [(1.2887000000E-01, 1.0000000000E+00)]),("P", [(3.6624000000E+02, 4.0977000000E-03), (8.5931000000E+01, 3.1956000000E-02), (2.7172000000E+01, 1.3326000000E-01), (9.8685000000E+00, 3.3228000000E-01), (3.7533000000E+00, 4.5335000000E-01), (1.4302000000E+00, 2.5058000000E-01)]),("P", [(8.5931000000E+01, 1.2506000000E-04), (2.7172000000E+01, -1.2166000000E-03), (9.8685000000E+00, -3.1813000000E-03), (3.7533000000E+00, -2.1475000000E-02), (1.4302000000E+00, 6.9991000000E-02), (3.8653000000E-01, 5.6081000000E-01)]),("P", [(1.1146000000E-01, 1.0000000000E+00)]),("D", [(5.0000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(5.5510000000E+04, 5.8899000000E-04), (8.3267000000E+03, 4.5460000000E-03), (1.8956000000E+03, 2.3335000000E-02), (5.3677000000E+02, 8.9965000000E-02), (1.7467000000E+02, 2.5486000000E-01), (6.2578000000E+01, 4.3998000000E-01), (2.3573000000E+01, 3.1549000000E-01), (5.8908000000E+00, 3.2826000000E-02)]),("S", [(8.3267000000E+03, -2.0480000000E-05), (1.8956000000E+03, -4.5429000000E-05), (5.3677000000E+02, -1.7937000000E-03), (1.7467000000E+02, -9.7252000000E-03), (6.2578000000E+01, -6.6399000000E-02), (2.3573000000E+01, -8.5502000000E-02), (5.8908000000E+00, 4.9588000000E-01), (2.1852000000E+00, 6.4988000000E-01)]),("S", [(1.8956000000E+03, 8.3461000000E-06), (5.3677000000E+02, 9.8306000000E-05), (1.7467000000E+02, 1.0040000000E-03), (6.2578000000E+01, 5.4409000000E-03), (2.3573000000E+01, 1.1400000000E-02), (5.8908000000E+00, -9.5134000000E-02), (2.1852000000E+00, -2.4735000000E-01), (4.5282000000E-01, 6.3973000000E-01)]),("S", [(1.6202000000E-01, 1.0000000000E+00)]),("P", [(3.9545000000E+02, 4.6266000000E-03), (9.2834000000E+01, 3.5898000000E-02), (2.9266000000E+01, 1.4854000000E-01), (1.0595000000E+01, 3.5878000000E-01), (4.0272000000E+00, 4.5417000000E-01), (1.5484000000E+00, 2.0545000000E-01)]),("P", [(9.2834000000E+01, 1.0934000000E-04), (2.9266000000E+01, -1.6358000000E-03), (1.0595000000E+01, -5.0632000000E-03), (4.0272000000E+00, -2.3960000000E-02), (1.5484000000E+00, 1.0388000000E-01), (4.6134000000E-01, 5.8063000000E-01)]),("P", [(1.3131000000E-01, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(6.3474000000E+04, 5.8028000000E-04), (9.5213000000E+03, 4.4792000000E-03), (2.1674000000E+03, 2.3000000000E-02), (6.1374000000E+02, 8.8815000000E-02), (1.9972000000E+02, 2.5258000000E-01), (7.1564000000E+01, 4.3921000000E-01), (2.6979000000E+01, 3.1868000000E-01), (6.8260000000E+00, 3.4348000000E-02)]),("S", [(9.5213000000E+03, -1.9824000000E-05), (2.1674000000E+03, -4.7199000000E-05), (6.1374000000E+02, -1.7713000000E-03), (1.9972000000E+02, -9.8068000000E-03), (7.1564000000E+01, -6.6719000000E-02), (2.6979000000E+01, -8.8711000000E-02), (6.8260000000E+00, 4.9396000000E-01), (2.5767000000E+00, 6.5392000000E-01)]),("S", [(2.1674000000E+03, 9.0584000000E-06), (6.1374000000E+02, 1.0695000000E-04), (1.9972000000E+02, 1.0981000000E-03), (7.1564000000E+01, 5.9898000000E-03), (2.6979000000E+01, 1.2762000000E-02), (6.8260000000E+00, -1.0307000000E-01), (2.5767000000E+00, -2.5941000000E-01), (5.5752000000E-01, 6.4899000000E-01)]),("S", [(1.9825000000E-01, 1.0000000000E+00)]),("P", [(4.1312000000E+02, 5.4251000000E-03), (9.7054000000E+01, 4.1644000000E-02), (3.0482000000E+01, 1.6990000000E-01), (1.0962000000E+01, 3.9275000000E-01), (4.1276000000E+00, 4.5045000000E-01), (1.5338000000E+00, 1.5308000000E-01)]),("P", [(9.7054000000E+01, -1.7356000000E-06), (3.0482000000E+01, -2.1441000000E-03), (1.0962000000E+01, -9.1298000000E-03), (4.1276000000E+00, -1.9906000000E-02), (1.5338000000E+00, 1.5582000000E-01), (5.1439000000E-01, 5.8772000000E-01)]),("P", [(1.5026000000E-01, 1.0000000000E+00)]),("D", [(6.0000000000E-01, 1.0000000000E+00)])],
    18 : [("S", [(7.1956000000E+04, 2.8643000000E-04), (1.0793000000E+04, 2.2112000000E-03), (2.4570000000E+03, 1.1358000000E-02), (6.9572000000E+02, 4.3916000000E-02), (2.2640000000E+02, 1.2531000000E-01), (8.1138000000E+01, 2.1927000000E-01), (3.0612000000E+01, 1.6071000000E-01), (7.8291000000E+00, 1.7849000000E-02)]),("S", [(1.0793000000E+04, -9.6136000000E-06), (2.4570000000E+03, -2.4144000000E-05), (6.9572000000E+02, -8.7570000000E-04), (2.2640000000E+02, -4.9376000000E-03), (8.1138000000E+01, -3.3512000000E-02), (3.0612000000E+01, -4.5764000000E-02), (7.8291000000E+00, 2.4633000000E-01), (2.9987000000E+00, 3.2863000000E-01)]),("S", [(2.4570000000E+03, 4.7854000000E-06), (6.9572000000E+02, 5.7582000000E-05), (2.2640000000E+02, 5.8972000000E-04), (8.1138000000E+01, 3.2399000000E-03), (3.0612000000E+01, 6.9814000000E-03), (7.8291000000E+00, -5.4918000000E-02), (2.9987000000E+00, -1.3472000000E-01), (6.7242000000E-01, 3.2734000000E-01)]),("S", [(2.3719000000E-01, 1.0000000000E+00)]),("P", [(4.4627000000E+02, 2.9613000000E-03), (1.0494000000E+02, 2.2591000000E-02), (3.2931000000E+01, 9.1524000000E-02), (1.1809000000E+01, 2.0642000000E-01), (4.4304000000E+00, 2.2195000000E-01), (1.5751000000E+00, 6.2765000000E-02)]),("P", [(1.0494000000E+02, -6.0823000000E-05), (3.2931000000E+01, -1.2661000000E-03), (1.1809000000E+01, -6.4589000000E-03), (4.4304000000E+00, -6.5105000000E-03), (1.5751000000E+00, 1.0436000000E-01), (5.6412000000E-01, 2.9184000000E-01)]),("P", [(1.7015000000E-01, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)])],
}


PC_2 = {

    1 : [("S", [(7.5423000000E+01, 2.4065000000E-03), (1.1350000000E+01, 1.8487000000E-02), (2.5993000000E+00, 8.9742000000E-02), (7.3513000000E-01, 2.8111000000E-01)]),("S", [(2.3167000000E-01, 1.0000000000E+00)]),("S", [(7.4147000000E-02, 1.0000000000E+00)]),("P", [(1.6000000000E+00, 1.0000000000E+00)]),("P", [(4.5000000000E-01, 1.0000000000E+00)]),("D", [(1.2500000000E+00, 1.0000000000E+00)])],
    2 : [("S", [(2.2177000000E+02, 1.4409000000E-03), (3.3375000000E+01, 1.0894000000E-02), (7.6114000000E+00, 5.0189000000E-02), (2.1089000000E+00, 1.4290000000E-01)]),("S", [(6.2878000000E-01, 1.0000000000E+00)]),("S", [(1.8469000000E-01, 1.0000000000E+00)]),("P", [(2.4000000000E+00, 1.0000000000E+00)]),("P", [(6.8000000000E-01, 1.0000000000E+00)]),("D", [(1.8000000000E+00, 1.0000000000E+00)])],
    3 : [("S", [(1.5025000000E+03, 7.6137000000E-04), (2.2551000000E+02, 5.8712000000E-03), (5.1375000000E+01, 2.9743000000E-02), (1.4547000000E+01, 1.0959000000E-01), (4.6645000000E+00, 2.8000000000E-01), (1.6008000000E+00, 4.4163000000E-01), (5.5262000000E-01, 2.9922000000E-01)]),("S", [(2.2551000000E+02, 2.1045000000E-05), (5.1375000000E+01, -1.2177000000E-04), (1.4547000000E+01, -1.0088000000E-04), (4.6645000000E+00, -8.4426000000E-03), (1.6008000000E+00, -2.4394000000E-02), (5.5262000000E-01, -9.0718000000E-02), (1.0366000000E-01, 3.0659000000E-01)]),("S", [(4.7442000000E-02, 1.0000000000E+00)]),("S", [(1.9790000000E-02, 1.0000000000E+00)]),("P", [(5.3000000000E+00, 5.5450300000E-03), (1.1500000000E+00, 5.5091200000E-02), (2.9000000000E-01, 8.7561500000E-02)]),("P", [(9.0000000000E-02, 1.0000000000E+00)]),("D", [(2.6000000000E-01, 1.0000000000E+00)])],
    4 : [("S", [(3.0673000000E+03, 3.2683000000E-04), (4.6024000000E+02, 2.5234000000E-03), (1.0480000000E+02, 1.2867000000E-02), (2.9657000000E+01, 4.8607000000E-02), (9.5365000000E+00, 1.3111000000E-01), (3.3106000000E+00, 2.2124000000E-01), (1.1804000000E+00, 1.6345000000E-01)]),("S", [(4.6024000000E+02, 1.3894000000E-06), (1.0480000000E+02, -3.6431000000E-05), (2.9657000000E+01, -3.3196000000E-04), (9.5365000000E+00, -3.6618000000E-03), (3.3106000000E+00, -1.7540000000E-02), (1.1804000000E+00, -4.0905000000E-02), (2.6871000000E-01, 1.1357000000E-01)]),("S", [(1.1774000000E-01, 1.0000000000E+00)]),("S", [(4.4685000000E-02, 1.0000000000E+00)]),("P", [(8.2400000000E+00, 1.1147100000E-02), (1.7900000000E+00, 8.0549400000E-02), (4.5000000000E-01, 2.5574400000E-01)]),("P", [(1.4000000000E-01, 1.0000000000E+00)]),("D", [(3.9000000000E-01, 1.0000000000E+00)])],
    5 : [("S", [(5.2204000000E+03, 5.9530000000E-04), (7.8319000000E+02, 4.5991000000E-03), (1.7831000000E+02, 2.3530000000E-02), (5.0469000000E+01, 9.0015000000E-02), (1.6285000000E+01, 2.4988000000E-01), (5.6885000000E+00, 4.4237000000E-01), (2.0495000000E+00, 3.4536000000E-01)]),("S", [(7.8319000000E+02, 1.6588000000E-06), (1.7831000000E+02, -6.8521000000E-05), (5.0469000000E+01, -6.8751000000E-04), (1.6285000000E+01, -7.3690000000E-03), (5.6885000000E+00, -3.7098000000E-02), (2.0495000000E+00, -8.6986000000E-02), (5.1473000000E-01, 2.1361000000E-01)]),("S", [(2.1556000000E-01, 1.0000000000E+00)]),("S", [(7.7457000000E-02, 1.0000000000E+00)]),("P", [(2.1810000000E+01, 5.8841000000E-03), (4.8930000000E+00, 4.1798000000E-02), (1.3976000000E+00, 1.5567000000E-01), (4.7747000000E-01, 3.5997000000E-01)]),("P", [(1.6419000000E-01, 1.0000000000E+00)]),("P", [(5.4004000000E-02, 1.0000000000E+00)]),("D", [(1.0500000000E+00, 1.0000000000E+00)]),("D", [(3.5000000000E-01, 1.0000000000E+00)]),("F", [(8.7000000000E-01, 1.0000000000E+00)])],
    6 : [("S", [(7.8571000000E+03, 5.6825000000E-04), (1.1787000000E+03, 4.3915000000E-03), (2.6832000000E+02, 2.2504000000E-02), (7.5948000000E+01, 8.6653000000E-02), (2.4559000000E+01, 2.4405000000E-01), (8.6212000000E+00, 4.4148000000E-01), (3.1278000000E+00, 3.5332000000E-01)]),("S", [(1.1787000000E+03, -5.9492000000E-07), (2.6832000000E+02, -6.2748000000E-05), (7.5948000000E+01, -7.5773000000E-04), (2.4559000000E+01, -7.3308000000E-03), (8.6212000000E+00, -3.8932000000E-02), (3.1278000000E+00, -8.8908000000E-02), (8.2202000000E-01, 2.1689000000E-01)]),("S", [(3.3017000000E-01, 1.0000000000E+00)]),("S", [(1.1463000000E-01, 1.0000000000E+00)]),("P", [(3.3775000000E+01, 6.0294000000E-03), (7.6766000000E+00, 4.3228000000E-02), (2.2357000000E+00, 1.6301000000E-01), (7.6447000000E-01, 3.6504000000E-01)]),("P", [(2.6232000000E-01, 1.0000000000E+00)]),("P", [(8.4638000000E-02, 1.0000000000E+00)]),("D", [(1.4000000000E+00, 1.0000000000E+00)]),("D", [(4.5000000000E-01, 1.0000000000E+00)]),("F", [(9.5000000000E-01, 1.0000000000E+00)])],
    7 : [("S", [(1.1101000000E+04, 5.4570000000E-04), (1.6653000000E+03, 4.2181000000E-03), (3.7908000000E+02, 2.1645000000E-02), (1.0730000000E+02, 8.3776000000E-02), (3.4747000000E+01, 2.3881000000E-01), (1.2249000000E+01, 4.3994000000E-01), (4.4743000000E+00, 3.6364000000E-01)]),("S", [(1.6653000000E+03, -1.5006000000E-06), (3.7908000000E+02, -5.8753000000E-05), (1.0730000000E+02, -7.7453000000E-04), (3.4747000000E+01, -7.2161000000E-03), (1.2249000000E+01, -3.9338000000E-02), (4.4743000000E+00, -9.0611000000E-02), (1.2404000000E+00, 1.9882000000E-01)]),("S", [(4.8743000000E-01, 1.0000000000E+00)]),("S", [(1.6484000000E-01, 1.0000000000E+00)]),("P", [(4.7934000000E+01, 6.1706000000E-03), (1.0999000000E+01, 4.4260000000E-02), (3.2472000000E+00, 1.6825000000E-01), (1.1114000000E+00, 3.6848000000E-01)]),("P", [(3.7988000000E-01, 1.0000000000E+00)]),("P", [(1.2068000000E-01, 1.0000000000E+00)]),("D", [(1.8000000000E+00, 1.0000000000E+00)]),("D", [(5.5000000000E-01, 1.0000000000E+00)]),("F", [(1.0200000000E+00, 1.0000000000E+00)])],
    8 : [("S", [(1.4782000000E+04, 5.3519000000E-04), (2.2173000000E+03, 4.1375000000E-03), (5.0474000000E+02, 2.1245000000E-02), (1.4287000000E+02, 8.2453000000E-02), (4.6300000000E+01, 2.3671000000E-01), (1.6337000000E+01, 4.4039000000E-01), (5.9828000000E+00, 3.6465000000E-01)]),("S", [(2.2173000000E+03, -1.9275000000E-06), (5.0474000000E+02, -5.7964000000E-05), (1.4287000000E+02, -7.9494000000E-04), (4.6300000000E+01, -7.3125000000E-03), (1.6337000000E+01, -4.0574000000E-02), (5.9828000000E+00, -9.1594000000E-02), (1.6718000000E+00, 2.0940000000E-01)]),("S", [(6.4662000000E-01, 1.0000000000E+00)]),("S", [(2.1669000000E-01, 1.0000000000E+00)]),("P", [(6.0424000000E+01, 6.8949000000E-03), (1.3935000000E+01, 4.9005000000E-02), (4.1531000000E+00, 1.8255000000E-01), (1.4158000000E+00, 3.7633000000E-01)]),("P", [(4.7549000000E-01, 1.0000000000E+00)]),("P", [(1.4529000000E-01, 1.0000000000E+00)]),("D", [(2.2000000000E+00, 1.0000000000E+00)]),("D", [(6.5000000000E-01, 1.0000000000E+00)]),("F", [(1.1000000000E+00, 1.0000000000E+00)])],
    9 : [("S", [(1.9127000000E+04, 5.2250000000E-04), (2.8689000000E+03, 4.0400000000E-03), (6.5304000000E+02, 2.0759000000E-02), (1.8485000000E+02, 8.0800000000E-02), (5.9949000000E+01, 2.3362000000E-01), (2.1190000000E+01, 4.3965000000E-01), (7.7846000000E+00, 3.7041000000E-01)]),("S", [(2.8689000000E+03, -2.2576000000E-06), (6.5304000000E+02, -5.6640000000E-05), (1.8485000000E+02, -7.9782000000E-04), (5.9949000000E+01, -7.2713000000E-03), (2.1190000000E+01, -4.0831000000E-02), (7.7846000000E+00, -9.2931000000E-02), (2.2305000000E+00, 2.0228000000E-01)]),("S", [(8.5409000000E-01, 1.0000000000E+00)]),("S", [(2.8336000000E-01, 1.0000000000E+00)]),("P", [(7.7639000000E+01, 7.0313000000E-03), (1.7965000000E+01, 5.0137000000E-02), (5.4005000000E+00, 1.8553000000E-01), (1.8423000000E+00, 3.7546000000E-01)]),("P", [(6.1532000000E-01, 1.0000000000E+00)]),("P", [(1.8489000000E-01, 1.0000000000E+00)]),("D", [(2.8000000000E+00, 1.0000000000E+00)]),("D", [(8.0000000000E-01, 1.0000000000E+00)]),("F", [(1.2000000000E+00, 1.0000000000E+00)])],
    10 : [("S", [(2.3880000000E+04, 2.5835000000E-04), (3.5817000000E+03, 1.9977000000E-03), (8.1527000000E+02, 1.0269000000E-02), (2.3077000000E+02, 4.0036000000E-02), (7.4884000000E+01, 1.1619000000E-01), (2.6518000000E+01, 2.1959000000E-01), (9.7692000000E+00, 1.8608000000E-01)]),("S", [(3.5817000000E+03, -7.6535000000E-06), (8.1527000000E+02, -1.3638000000E-04), (2.3077000000E+02, -2.0669000000E-03), (7.4884000000E+01, -1.8140000000E-02), (2.6518000000E+01, -1.0379000000E-01), (9.7692000000E+00, -2.3283000000E-01), (2.8305000000E+00, 5.0798000000E-01)]),("S", [(1.0740000000E+00, 1.0000000000E+00)]),("S", [(3.5400000000E-01, 1.0000000000E+00)]),("P", [(9.7106000000E+01, 3.5668000000E-03), (2.2543000000E+01, 2.5467000000E-02), (6.8173000000E+00, 9.4016000000E-02), (2.3263000000E+00, 1.8796000000E-01)]),("P", [(7.7425000000E-01, 1.0000000000E+00)]),("P", [(2.3026000000E-01, 1.0000000000E+00)]),("D", [(3.0000000000E+00, 1.0000000000E+00)]),("D", [(8.5000000000E-01, 1.0000000000E+00)]),("F", [(1.3000000000E+00, 1.0000000000E+00)])],
    11 : [("S", [(6.4474000000E+04, 1.9020000000E-04), (9.6675000000E+03, 1.4749000000E-03), (2.2008000000E+03, 7.6778000000E-03), (6.2326000000E+02, 3.1333000000E-02), (2.0312000000E+02, 1.0269000000E-01), (7.2944000000E+01, 2.5598000000E-01), (2.8026000000E+01, 4.1517000000E-01), (1.1055000000E+01, 3.0473000000E-01), (3.5490000000E+00, 4.1146000000E-02)]),("S", [(2.2008000000E+03, -1.3881000000E-05), (6.2326000000E+02, -1.4242000000E-04), (2.0312000000E+02, -1.8159000000E-03), (7.2944000000E+01, -1.2093000000E-02), (2.8026000000E+01, -5.6823000000E-02), (1.1055000000E+01, -1.0110000000E-01), (3.5490000000E+00, 2.2498000000E-01), (1.4082000000E+00, 5.1711000000E-01)]),("S", [(2.2008000000E+03, 4.5001000000E-06), (6.2326000000E+02, 5.4469000000E-05), (2.0312000000E+02, 6.1058000000E-04), (7.2944000000E+01, 4.1724000000E-03), (2.8026000000E+01, 1.8231000000E-02), (1.1055000000E+01, 2.9017000000E-02), (3.5490000000E+00, -5.2932000000E-02), (1.4082000000E+00, 1.6965000000E-01), (5.1148000000E-01, 8.8478000000E-01)]),("S", [(6.9534000000E-02, 1.0000000000E+00)]),("S", [(2.5120000000E-02, 1.0000000000E+00)]),("P", [(2.3053000000E+02, 2.5420000000E-03), (5.4201000000E+01, 1.9768000000E-02), (1.7402000000E+01, 8.3480000000E-02), (6.3152000000E+00, 2.3282000000E-01), (2.4084000000E+00, 3.8216000000E-01), (9.0249000000E-01, 3.7985000000E-01), (3.1450000000E-01, 1.6287000000E-01)]),("P", [(1.2100000000E-01, 1.0000000000E+00)]),("P", [(3.9000000000E-02, 1.0000000000E+00)]),("D", [(2.4000000000E-01, 1.0000000000E+00)])],
    12 : [("S", [(8.1017000000E+04, 8.9053000000E-05), (1.2143000000E+04, 6.9088000000E-04), (2.7638000000E+03, 3.5999000000E-03), (7.8258000000E+02, 1.4721000000E-02), (2.5503000000E+02, 4.8572000000E-02), (9.1623000000E+01, 1.2279000000E-01), (3.5245000000E+01, 2.0506000000E-01), (1.3960000000E+01, 1.5913000000E-01), (4.7422000000E+00, 2.5099000000E-02)]),("S", [(1.2143000000E+04, -5.2134000000E-07), (2.7638000000E+03, -4.0753000000E-06), (7.8258000000E+02, -8.2483000000E-05), (2.5503000000E+02, -7.0727000000E-04), (9.1623000000E+01, -5.4913000000E-03), (3.5245000000E+01, -2.3859000000E-02), (1.3960000000E+01, -4.9827000000E-02), (4.7422000000E+00, 8.3497000000E-02), (1.9284000000E+00, 1.9563000000E-01)]),("S", [(2.7638000000E+03, -1.1910000000E-05), (7.8258000000E+02, -1.4781000000E-04), (2.5503000000E+02, -1.6872000000E-03), (9.1623000000E+01, -1.1864000000E-02), (3.5245000000E+01, -5.5983000000E-02), (1.3960000000E+01, -1.1311000000E-01), (4.7422000000E+00, 1.8789000000E-01), (1.9284000000E+00, 7.1752000000E-01), (7.2424000000E-01, 5.3956000000E-01)]),("S", [(1.2353000000E-01, 1.0000000000E+00)]),("S", [(4.3032000000E-02, 1.0000000000E+00)]),("P", [(3.0905000000E+02, 1.1130000000E-03), (7.2894000000E+01, 8.7569000000E-03), (2.3494000000E+01, 3.8254000000E-02), (8.5619000000E+00, 1.1095000000E-01), (3.3068000000E+00, 1.9174000000E-01), (1.2694000000E+00, 1.9457000000E-01), (4.6106000000E-01, 7.8204000000E-02)]),("P", [(1.7700000000E-01, 1.0000000000E+00)]),("P", [(5.7000000000E-02, 1.0000000000E+00)]),("D", [(3.0000000000E-01, 1.0000000000E+00)])],
    13 : [("S", [(9.9741000000E+04, 1.6809000000E-04), (1.4949000000E+04, 1.3041000000E-03), (3.4025000000E+03, 6.7979000000E-03), (9.6353000000E+02, 2.7840000000E-02), (3.1405000000E+02, 9.2358000000E-02), (1.1288000000E+02, 2.3633000000E-01), (4.3474000000E+01, 4.0484000000E-01), (1.7292000000E+01, 3.2992000000E-01), (6.1622000000E+00, 5.8521000000E-02)]),("S", [(1.4949000000E+04, -4.8000000000E-07), (3.4025000000E+03, -2.0474000000E-06), (9.6353000000E+02, -5.5733000000E-05), (3.1405000000E+02, -4.1853000000E-04), (1.1288000000E+02, -3.4813000000E-03), (4.3474000000E+01, -1.4864000000E-02), (1.7292000000E+01, -3.4012000000E-02), (6.1622000000E+00, 4.3896000000E-02), (2.5226000000E+00, 1.1958000000E-01)]),("S", [(3.4025000000E+03, -5.5580000000E-06), (9.6353000000E+02, -6.9047000000E-05), (3.1405000000E+02, -7.9085000000E-04), (1.1288000000E+02, -5.6414000000E-03), (4.3474000000E+01, -2.7288000000E-02), (1.7292000000E+01, -5.9538000000E-02), (6.1622000000E+00, 7.2393000000E-02), (2.5226000000E+00, 4.1226000000E-01), (9.6874000000E-01, 3.9165000000E-01)]),("S", [(1.9251000000E-01, 1.0000000000E+00)]),("S", [(6.7813000000E-02, 1.0000000000E+00)]),("P", [(5.7971000000E+02, 1.0561000000E-03), (1.3754000000E+02, 8.6335000000E-03), (4.4067000000E+01, 4.3227000000E-02), (1.6148000000E+01, 1.4168000000E-01), (6.4888000000E+00, 3.0352000000E-01), (2.6632000000E+00, 4.0389000000E-01), (1.0792000000E+00, 2.8642000000E-01)]),("P", [(1.3754000000E+02, 2.9666000000E-05), (4.4067000000E+01, -1.0593000000E-04), (1.6148000000E+01, -1.7370000000E-04), (6.4888000000E+00, -4.4315000000E-03), (2.6632000000E+00, -4.4134000000E-03), (1.0792000000E+00, -1.0723000000E-03), (3.9771000000E-01, 1.7650000000E-01)]),("P", [(1.3475000000E-01, 1.0000000000E+00)]),("P", [(4.2185000000E-02, 1.0000000000E+00)]),("D", [(1.4700000000E+00, 1.0000000000E+00)]),("D", [(3.3000000000E-01, 1.0000000000E+00)]),("F", [(4.8000000000E-01, 1.0000000000E+00)])],
    14 : [("S", [(1.2004000000E+05, 1.6075000000E-04), (1.7991000000E+04, 1.2472000000E-03), (4.0948000000E+03, 6.5040000000E-03), (1.1596000000E+03, 2.6665000000E-02), (3.7800000000E+02, 8.8816000000E-02), (1.3593000000E+02, 2.2932000000E-01), (5.2411000000E+01, 4.0025000000E-01), (2.0927000000E+01, 3.3828000000E-01), (7.7130000000E+00, 6.5512000000E-02)]),("S", [(1.7991000000E+04, -6.8403000000E-07), (4.0948000000E+03, -2.1124000000E-06), (1.1596000000E+03, -6.9157000000E-05), (3.7800000000E+02, -4.8080000000E-04), (1.3593000000E+02, -4.1694000000E-03), (5.2411000000E+01, -1.7674000000E-02), (2.0927000000E+01, -4.2891000000E-02), (7.7130000000E+00, 4.5368000000E-02), (3.1604000000E+00, 1.3904000000E-01)]),("S", [(4.0948000000E+03, -3.9897000000E-06), (1.1596000000E+03, -5.0203000000E-05), (3.7800000000E+02, -5.7553000000E-04), (1.3593000000E+02, -4.1524000000E-03), (5.2411000000E+01, -2.0437000000E-02), (2.0927000000E+01, -4.6998000000E-02), (7.7130000000E+00, 4.3986000000E-02), (3.1604000000E+00, 3.3957000000E-01), (1.2348000000E+00, 3.5038000000E-01)]),("S", [(2.6775000000E-01, 1.0000000000E+00)]),("S", [(9.4067000000E-02, 1.0000000000E+00)]),("P", [(6.7713000000E+02, 1.0925000000E-03), (1.6067000000E+02, 8.9601000000E-03), (5.1585000000E+01, 4.4744000000E-02), (1.8948000000E+01, 1.4748000000E-01), (7.6163000000E+00, 3.1480000000E-01), (3.1317000000E+00, 4.1339000000E-01), (1.2703000000E+00, 2.6440000000E-01)]),("P", [(1.6067000000E+02, 7.9014000000E-06), (5.1585000000E+01, -1.3613000000E-04), (1.8948000000E+01, -9.0378000000E-04), (7.6163000000E+00, -5.5863000000E-03), (3.1317000000E+00, -1.0141000000E-02), (1.2703000000E+00, 1.8661000000E-02), (4.3332000000E-01, 2.8890000000E-01)]),("P", [(1.6088000000E-01, 1.0000000000E+00)]),("P", [(5.4883000000E-02, 1.0000000000E+00)]),("D", [(1.6800000000E+00, 1.0000000000E+00)]),("D", [(3.8000000000E-01, 1.0000000000E+00)]),("F", [(5.4000000000E-01, 1.0000000000E+00)])],
    15 : [("S", [(1.4568000000E+05, 1.5016000000E-04), (2.1836000000E+04, 1.1650000000E-03), (4.9699000000E+03, 6.0798000000E-03), (1.4073000000E+03, 2.4968000000E-02), (4.5878000000E+02, 8.3637000000E-02), (1.6508000000E+02, 2.1863000000E-01), (6.3744000000E+01, 3.9234000000E-01), (2.5598000000E+01, 3.5038000000E-01), (9.9307000000E+00, 7.7287000000E-02)]),("S", [(2.1836000000E+04, -1.0509000000E-06), (4.9699000000E+03, -1.6801000000E-06), (1.4073000000E+03, -8.7601000000E-05), (4.5878000000E+02, -5.2745000000E-04), (1.6508000000E+02, -4.9575000000E-03), (6.3744000000E+01, -2.0535000000E-02), (2.5598000000E+01, -5.5030000000E-02), (9.9307000000E+00, 4.1583000000E-02), (3.9686000000E+00, 1.7029000000E-01)]),("S", [(4.9699000000E+03, -2.4863000000E-06), (1.4073000000E+03, -3.0955000000E-05), (4.5878000000E+02, -3.5776000000E-04), (1.6508000000E+02, -2.6123000000E-03), (6.3744000000E+01, -1.3204000000E-02), (2.5598000000E+01, -3.2848000000E-02), (9.9307000000E+00, 1.8182000000E-02), (3.9686000000E+00, 2.5722000000E-01), (1.5565000000E+00, 3.0682000000E-01)]),("S", [(3.5404000000E-01, 1.0000000000E+00)]),("S", [(1.2394000000E-01, 1.0000000000E+00)]),("P", [(8.2623000000E+02, 1.0225000000E-03), (1.9602000000E+02, 8.4472000000E-03), (6.2951000000E+01, 4.2664000000E-02), (2.3222000000E+01, 1.4239000000E-01), (9.3885000000E+00, 3.1059000000E-01), (3.8997000000E+00, 4.1672000000E-01), (1.6048000000E+00, 2.6618000000E-01)]),("P", [(1.9602000000E+02, 4.1082000000E-06), (6.2951000000E+01, -1.5057000000E-04), (2.3222000000E+01, -1.1141000000E-03), (9.3885000000E+00, -6.4942000000E-03), (3.8997000000E+00, -1.2469000000E-02), (1.6048000000E+00, 2.3458000000E-02), (5.6053000000E-01, 3.2270000000E-01)]),("P", [(2.1175000000E-01, 1.0000000000E+00)]),("P", [(7.3486000000E-02, 1.0000000000E+00)]),("D", [(1.8900000000E+00, 1.0000000000E+00)]),("D", [(4.3000000000E-01, 1.0000000000E+00)]),("F", [(6.0000000000E-01, 1.0000000000E+00)])],
    16 : [("S", [(1.7203000000E+05, 1.4353000000E-04), (2.5783000000E+04, 1.1137000000E-03), (5.8682000000E+03, 5.8143000000E-03), (1.6617000000E+03, 2.3903000000E-02), (5.4175000000E+02, 8.0354000000E-02), (1.9501000000E+02, 2.1175000000E-01), (7.5396000000E+01, 3.8664000000E-01), (3.0407000000E+01, 3.5747000000E-01), (1.2121000000E+01, 8.6131000000E-02)]),("S", [(2.5783000000E+04, -1.1862000000E-06), (5.8682000000E+03, -1.2321000000E-06), (1.6617000000E+03, -9.0480000000E-05), (5.4175000000E+02, -5.0105000000E-04), (1.9501000000E+02, -4.9510000000E-03), (7.5396000000E+01, -2.0252000000E-02), (3.0407000000E+01, -5.7388000000E-02), (1.2121000000E+01, 3.4847000000E-02), (4.7777000000E+00, 1.6833000000E-01)]),("S", [(5.8682000000E+03, -2.1972000000E-06), (1.6617000000E+03, -2.8367000000E-05), (5.4175000000E+02, -3.2591000000E-04), (1.9501000000E+02, -2.4131000000E-03), (7.5396000000E+01, -1.2352000000E-02), (3.0407000000E+01, -3.2119000000E-02), (1.2121000000E+01, 1.1611000000E-02), (4.7777000000E+00, 2.5542000000E-01), (1.8906000000E+00, 3.1384000000E-01)]),("S", [(4.4686000000E-01, 1.0000000000E+00)]),("S", [(1.5621000000E-01, 1.0000000000E+00)]),("P", [(9.6812000000E+02, 1.0048000000E-03), (2.2970000000E+02, 8.3313000000E-03), (7.3833000000E+01, 4.2260000000E-02), (2.7312000000E+01, 1.4206000000E-01), (1.1074000000E+01, 3.1256000000E-01), (4.6307000000E+00, 4.1933000000E-01), (1.9299000000E+00, 2.5924000000E-01)]),("P", [(2.2970000000E+02, -8.0693000000E-06), (7.3833000000E+01, -1.6943000000E-04), (2.7312000000E+01, -1.5243000000E-03), (1.1074000000E+01, -7.1993000000E-03), (4.6307000000E+00, -1.6113000000E-02), (1.9299000000E+00, 3.4234000000E-02), (6.8783000000E-01, 3.5518000000E-01)]),("P", [(2.5663000000E-01, 1.0000000000E+00)]),("P", [(8.7251000000E-02, 1.0000000000E+00)]),("D", [(2.1000000000E+00, 1.0000000000E+00)]),("D", [(4.8000000000E-01, 1.0000000000E+00)]),("F", [(6.6000000000E-01, 1.0000000000E+00)])],
    17 : [("S", [(2.0115000000E+05, 1.3753000000E-04), (3.0146000000E+04, 1.0671000000E-03), (6.8610000000E+03, 5.5737000000E-03), (1.9428000000E+03, 2.2935000000E-02), (6.3342000