"use strict";
var _a;
Object.defineProperty(exports, "__esModule", { value: true });
exports.ProwlerAudit = void 0;
const JSII_RTTI_SYMBOL_1 = Symbol.for("jsii.rtti");
// import { Stack, Duration, RemovalPolicy, CustomResource } from 'aws-cdk-lib';
// eslint-disable-next-line no-duplicate-imports
// import { aws_iam as iam, aws_logs as logs, aws_s3 as s3, aws_codebuild as codebuild, aws_lambda as lambda, custom_resources as cr } from 'aws-cdk-lib';
const aws_cdk_lib_1 = require("aws-cdk-lib");
const codebuild = require("aws-cdk-lib/aws-codebuild");
const events = require("aws-cdk-lib/aws-events");
const targets = require("aws-cdk-lib/aws-events-targets");
const iam = require("aws-cdk-lib/aws-iam");
const lambda = require("aws-cdk-lib/aws-lambda");
const logs = require("aws-cdk-lib/aws-logs");
// eslint-disable-next-line import/no-extraneous-dependencies
const s3 = require("aws-cdk-lib/aws-s3");
const cr = require("aws-cdk-lib/custom-resources");
const statement = require("cdk-iam-floyd");
const constructs_1 = require("constructs");
/**
 * Creates a CodeBuild project to audit an AWS account with Prowler and stores the html report in a S3 bucket. This will run onece at the beginning and on a schedule afterwards. Partial contribution from https://github.com/stevecjones
 */
class ProwlerAudit extends constructs_1.Construct {
    constructor(parent, id, props) {
        var _b, _c, _d, _e, _f;
        super(parent, id);
        // defaults
        this.serviceName = (props === null || props === void 0 ? void 0 : props.serviceName) ? props.serviceName : 'prowler';
        this.logsRetentionInDays = (props === null || props === void 0 ? void 0 : props.logsRetentionInDays) ? props.logsRetentionInDays : logs.RetentionDays.THREE_DAYS;
        this.enableScheduler = (props === null || props === void 0 ? void 0 : props.enableScheduler) ? props.enableScheduler : false;
        this.prowlerScheduler = (props === null || props === void 0 ? void 0 : props.prowlerScheduler) ? props.prowlerScheduler : 'cron(0 22 * * ? *)';
        this.prowlerOptions = (props === null || props === void 0 ? void 0 : props.prowlerOptions) ? props.prowlerOptions : '-M text,junit-xml,html,csv,json';
        this.prowlerVersion = (props === null || props === void 0 ? void 0 : props.prowlerVersion) ? props.prowlerVersion : '2.7.0';
        const reportBucket = (_b = props === null || props === void 0 ? void 0 : props.reportBucket) !== null && _b !== void 0 ? _b : new s3.Bucket(this, 'ReportBucket', {
            //bucketName: `${'123456'}-prowler-reports`,
            autoDeleteObjects: true,
            removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        });
        const reportGroup = new codebuild.ReportGroup(this, 'reportGroup', { /**reportGroupName: 'testReportGroup', */ removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY });
        const preBuildCommands = [];
        if (!!(props === null || props === void 0 ? void 0 : props.allowlist)) {
            const prowlerFilename = 'allowlist.txt';
            if (props.allowlist.isZipArchive) {
                preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} .`);
                preBuildCommands.push(`unzip ${props.allowlist.s3ObjectKey} -d prowler`);
            }
            else {
                preBuildCommands.push(`aws s3 cp ${props.allowlist.s3ObjectUrl} prowler/${prowlerFilename}`);
            }
            this.prowlerOptions = this.prowlerOptions + ` -w ${prowlerFilename}`;
        }
        const prowlerBuild = this.codebuildProject = new codebuild.Project(this, 'prowlerBuild', {
            description: 'Run Prowler assessment',
            timeout: aws_cdk_lib_1.Duration.hours(5),
            environment: {
                environmentVariables: {
                    BUCKET_REPORT: { value: reportBucket.bucketName || '' },
                    BUCKET_PREFIX: { value: (_c = props === null || props === void 0 ? void 0 : props.reportBucketPrefix) !== null && _c !== void 0 ? _c : '' },
                    ADDITIONAL_S3_ARGS: { value: (_d = props === null || props === void 0 ? void 0 : props.additionalS3CopyArgs) !== null && _d !== void 0 ? _d : '' },
                    PROWLER_OPTIONS: { value: this.prowlerOptions || '' },
                },
                buildImage: codebuild.LinuxBuildImage.fromCodeBuildImageId('aws/codebuild/amazonlinux2-x86_64-standard:3.0'),
            },
            buildSpec: codebuild.BuildSpec.fromObject({
                version: '0.2',
                phases: {
                    install: {
                        'runtime-versions': {
                            python: 3.8,
                        },
                        'commands': [
                            'echo "Installing Prowler and dependencies..."',
                            'pip3 install detect-secrets',
                            'yum -y install jq',
                            'curl "https://awscli.amazonaws.com/awscli-exe-linux-x86_64.zip" -o "awscliv2.zip"',
                            'unzip awscliv2.zip',
                            './aws/install',
                            `git clone -b ${this.prowlerVersion} https://github.com/prowler-cloud/prowler`,
                        ],
                    },
                    pre_build: {
                        commands: preBuildCommands,
                    },
                    build: {
                        commands: [
                            `echo "Running Prowler as ./prowler ${this.prowlerOptions} && echo OK || echo FAILED"`,
                            'cd prowler',
                            `./prowler ${this.prowlerOptions} && echo OK || echo FAILED`,
                        ],
                    },
                    post_build: {
                        commands: [
                            'echo "Uploading reports to S3..." ',
                            'aws s3 cp --sse AES256 output/ s3://$BUCKET_REPORT/$BUCKET_PREFIX --recursive $ADDITIONAL_S3_ARGS',
                            'echo "Done!"',
                        ],
                    },
                },
                reports: {
                    [reportGroup.reportGroupName]: {
                        'files': ['**/*'],
                        'base-directory': 'prowler/junit-reports',
                        'file-format': 'JunitXml',
                    },
                },
            }),
        });
        if (!!(props === null || props === void 0 ? void 0 : props.allowlist)) {
            props.allowlist.bucket.grantRead(prowlerBuild);
        }
        (_e = prowlerBuild.role) === null || _e === void 0 ? void 0 : _e.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('SecurityAudit'));
        (_f = prowlerBuild.role) === null || _f === void 0 ? void 0 : _f.addManagedPolicy(iam.ManagedPolicy.fromAwsManagedPolicyName('job-function/ViewOnlyAccess'));
        // prowlerBuild.addToRolePolicy(new statement.Dax().allow().to());
        prowlerBuild.addToRolePolicy(new statement.Ds().allow().toListAuthorizedApplications());
        prowlerBuild.addToRolePolicy(new statement.Ec2().allow().toGetEbsEncryptionByDefault());
        prowlerBuild.addToRolePolicy(new statement.Ecr().allow().toDescribeImageScanFindings().toDescribeImages().toDescribeRegistry());
        prowlerBuild.addToRolePolicy(new statement.Tag().allow().toGetTagKeys());
        prowlerBuild.addToRolePolicy(new statement.Lambda().allow().toGetFunction());
        prowlerBuild.addToRolePolicy(new statement.Glue().allow().toSearchTables().toGetConnections());
        prowlerBuild.addToRolePolicy(new statement.Apigateway().allow().toGET());
        prowlerBuild.addToRolePolicy(new iam.PolicyStatement({ actions: ['support:Describe*'], resources: ['*'] }));
        reportBucket.grantPut(prowlerBuild);
        const myRole = new iam.Role(this, 'MyRole', { assumedBy: new iam.ServicePrincipal('lambda.amazonaws.com') });
        const prowlerStartBuildLambda = new lambda.Function(this, 'prowlerStartBuildLambda', {
            runtime: lambda.Runtime.PYTHON_3_6,
            timeout: aws_cdk_lib_1.Duration.seconds(120),
            handler: 'index.lambda_handler',
            code: lambda.Code.fromInline(`import boto3
import cfnresponse
from botocore.exceptions import ClientError
def lambda_handler(event,context):
  props = event['ResourceProperties']
  codebuild_client = boto3.client('codebuild')
  if (event['RequestType'] == 'Create' or event['RequestType'] == 'Update'):
    try:
        response = codebuild_client.start_build(projectName=props['Build'])
        print(response)
        print("Respond: SUCCESS")
        cfnresponse.send(event, context, cfnresponse.SUCCESS, {})
    except Exception as ex:
        print(ex.response['Error']['Message'])
        cfnresponse.send(event, context, cfnresponse.FAILED, ex.response)
      `),
        });
        prowlerStartBuildLambda.addToRolePolicy(new statement.Codebuild().allow().toStartBuild()); // onResource project ...
        const myProvider = new cr.Provider(this, 'MyProvider', {
            onEventHandler: prowlerStartBuildLambda,
            logRetention: this.logsRetentionInDays,
            role: myRole,
        });
        new aws_cdk_lib_1.CustomResource(this, 'Resource1', {
            serviceToken: myProvider.serviceToken,
            properties: {
                Build: prowlerBuild.projectName,
                RERUN_PROWLER: Boolean(this.node.tryGetContext('reRunProwler')) ? Date.now().toString() : '',
            },
        });
        if (this.enableScheduler) {
            const prowlerSchedulerLambda = new lambda.Function(this, 'ScheduleLambda', {
                runtime: lambda.Runtime.PYTHON_3_6,
                timeout: aws_cdk_lib_1.Duration.seconds(120),
                handler: 'index.lambda_handler',
                environment: {
                    buildName: prowlerBuild.projectName,
                },
                code: lambda.Code.fromInline(`import boto3
        import os
        def lambda_handler(event,context):
          codebuild_client = boto3.client('codebuild')
          print("Running Prowler scheduled!: " + os.environ['buildName'])
          project_name = os.environ['buildName']
          response = codebuild_client.start_build(projectName=project_name)
          print(response)
          print("Respond: SUCCESS")
        `),
            });
            new events.Rule(this, 'Scheduler', {
                description: 'A schedule for the Lambda function that triggers Prowler in CodeBuild.',
                targets: [new targets.LambdaFunction(prowlerSchedulerLambda)],
                schedule: events.Schedule.expression(this.prowlerScheduler || ''),
            });
        }
    }
}
exports.ProwlerAudit = ProwlerAudit;
_a = JSII_RTTI_SYMBOL_1;
ProwlerAudit[_a] = { fqn: "cdk-prowler.ProwlerAudit", version: "2.20.0" };
//# sourceMappingURL=data:application/json;base64,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