"use strict";
/**
 *  Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 *
 *  Licensed under the Apache License, Version 2.0 (the "License"). You may not use this file except in compliance
 *  with the License. A copy of the License is located at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  or in the 'license' file accompanying this file. This file is distributed on an 'AS IS' BASIS, WITHOUT WARRANTIES
 *  OR CONDITIONS OF ANY KIND, express or implied. See the License for the specific language governing permissions
 *  and limitations under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
/// !cdk-integ *
const aws_cdk_lib_1 = require("aws-cdk-lib");
const lib_1 = require("../lib");
const lambda = require("aws-cdk-lib/aws-lambda");
const cloudfront = require("aws-cdk-lib/aws-cloudfront");
const apigateway = require("aws-cdk-lib/aws-apigateway");
const cdk = require("aws-cdk-lib");
const aws_cdk_lib_2 = require("aws-cdk-lib");
const origins = require("aws-cdk-lib/aws-cloudfront-origins");
const core_1 = require("@aws-solutions-constructs/core");
const integ_tests_alpha_1 = require("@aws-cdk/integ-tests-alpha");
// Setup
const app = new aws_cdk_lib_1.App();
const stack = new aws_cdk_lib_1.Stack(app, core_1.generateIntegStackName(__filename));
stack.templateOptions.description = 'Integration Test for aws-cloudfront-apigateway-lambda';
const lambdaProps = {
    code: lambda.Code.fromAsset(`${__dirname}/lambda`),
    runtime: lambda.Runtime.NODEJS_16_X,
    handler: 'index.handler'
};
// Some Caching for static content
const someCachePolicy = new cloudfront.CachePolicy(stack, 'SomeCachePolicy', {
    cachePolicyName: 'SomeCachePolicy',
    defaultTtl: aws_cdk_lib_2.Duration.hours(8),
    minTtl: aws_cdk_lib_2.Duration.hours(5),
    maxTtl: aws_cdk_lib_2.Duration.hours(10),
});
// Disable Caching for dynamic content
const noCachePolicy = new cloudfront.CachePolicy(stack, 'NoCachePolicy', {
    cachePolicyName: 'NoCachePolicy',
    defaultTtl: aws_cdk_lib_2.Duration.minutes(0),
    minTtl: aws_cdk_lib_2.Duration.minutes(0),
    maxTtl: aws_cdk_lib_2.Duration.minutes(0),
});
const construct = new lib_1.CloudFrontToApiGatewayToLambda(stack, 'cf-api-lambda-override', {
    lambdaFunctionProps: lambdaProps,
    apiGatewayProps: {
        proxy: false,
        defaultMethodOptions: {
            authorizationType: apigateway.AuthorizationType.CUSTOM,
            authorizer: core_1.CreateApiAuthorizer(stack, `${core_1.generateIntegStackName(__filename)}-authorizer`)
        },
    },
    cloudFrontDistributionProps: {
        defaultBehavior: {
            cachePolicy: someCachePolicy
        }
    },
    cloudFrontLoggingBucketProps: {
        removalPolicy: aws_cdk_lib_1.RemovalPolicy.DESTROY,
        autoDeleteObjects: true
    },
});
const apiEndPoint = construct.apiGateway;
const apiEndPointUrlWithoutProtocol = cdk.Fn.select(1, cdk.Fn.split("://", apiEndPoint.url));
const apiEndPointDomainName = cdk.Fn.select(0, cdk.Fn.split("/", apiEndPointUrlWithoutProtocol));
const staticResource = construct.apiGateway.root.addResource('static');
const staticMethod = staticResource.addMethod('GET', new apigateway.HttpIntegration('http://amazon.com'));
const dynamicResource = construct.apiGateway.root.addResource('dynamic');
const dynamicMethod = dynamicResource.addMethod('GET');
// Add behavior
construct.cloudFrontWebDistribution.addBehavior('/dynamic', new origins.HttpOrigin(apiEndPointDomainName, {
    originPath: `/${apiEndPoint.deploymentStage.stageName}/dynamic`
}), {
    cachePolicy: noCachePolicy
});
// Suppress CFN_NAG warnings
suppressWarnings(staticMethod);
suppressWarnings(dynamicMethod);
function suppressWarnings(method) {
    const child = method.node.findChild('Resource');
    child.cfnOptions.metadata = {
        cfn_nag: {
            rules_to_suppress: [{
                    id: 'W59',
                    reason: `AWS::ApiGateway::Method AuthorizationType is set to 'NONE' because API Gateway behind CloudFront does not support AWS_IAM authentication`
                }]
        }
    };
}
core_1.suppressCustomHandlerCfnNagWarnings(stack, 'Custom::S3AutoDeleteObjectsCustomResourceProvider');
// Synth
new integ_tests_alpha_1.IntegTest(stack, 'Integ', { testCases: [
        stack
    ] });
//# sourceMappingURL=data:application/json;base64,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