# pylint: disable=line-too-long, unused-wildcard-import
"""
.. _`German Federal Institute of Hydrology (BfG)`: https://www.bafg.de/EN

|hland_96c| combines `HBV96's` :cite:p:`ref-Lindstrom1997HBV96` and `COSERO's`
:cite:p:`ref-Kling2005,ref-Kling2006` process equations.  We implemented it on behalf
of the `German Federal Institute of Hydrology (BfG)`_ as an alternative to |hland_96|
for modelling large river basins in central Europe.  All processes "above the soil"
(input data correction, interception, snowmelt) and "inside the soil" (evaporation,
generation of effective precipitation), as well as the handling of water areas, are
identical with |hland_96| (and so with HBV96). Most processes "below the soil" agree
with COSERO (runoff generation and runoff concentration).

The following figure shows the general structure of |hland_96c|:

.. image:: HydPy-H-HBV96-COSERO.png

Comparing the above figure with the analogue figure of |hland_96| reveals that
|hland_96c| models three instead of two runoff components, requiring a larger number of
vertically arranged storages.  The two upper storages (|BW1| and |BW2|) are
zone-specific.  In comparison, the upper zone layer (|UZ|) of |hland_96| is
subbasin-specific.  For the subbasin-wide lower zone storage (|LZ|), there is no
difference to |hland_96|.  Like all |hland| application models, |hland_96c| allows an
optional submodel for runoff concentration.  Using |rconc_nash| and setting the number
of storages to one agrees with COSERO's approach to model runoff concentration
with a single bucket. In agreement with COSERO but in contrast to |hland_96p|,
|hland_96c| does not take base flow into account when calculating runoff concentration.


Integration tests
=================

.. how_to_understand_integration_tests::

We define the settings of the following test calculations as similar to the ones of
application model |hland_96| as possible.  Hence, comparing the test results gives a
good impression of the functional differences of both models.

>>> from hydpy import pub
>>> pub.timegrids = "01.01.2000", "05.01.2000", "1h"

>>> from hydpy.models.hland_96c import *
>>> parameterstep("1h")
>>> from hydpy import Node, Element
>>> outlet = Node("outlet")
>>> land = Element("land", outlets=outlet)
>>> land.model = model

>>> nmbzones(1)
>>> sclass(1)
>>> area(1.0)
>>> zonearea(1.0)
>>> zonez(1.0)
>>> psi(1.0)

We assign identical values to all parameters, besides those that are specific to
|hland_96c| (|H1|, |TAb1|, |TVs1|, |H1|, |TAb1|, |TVs1|):

>>> zonetype(FIELD)

>>> pcorr(1.1)
>>> pcalt(0.1)
>>> rfcf(1.1)
>>> sfcf(1.3)
>>> tcorr(0.6)
>>> tcalt(0.6)
>>> icmax(2.0)
>>> sfdist(1.0)
>>> smax(inf)
>>> sred(0.0)
>>> tt(0.0)
>>> ttint(2.0)
>>> dttm(1.0)
>>> cfmax(0.5)
>>> cfvar(0.1)
>>> gmelt(1.0)
>>> gvar(0.2)
>>> cfr(0.1)
>>> whc(0.4)
>>> fc(200.0)
>>> beta(2.0)
>>> h1(10.0)
>>> tab1(2.0)
>>> tvs1(2.0)
>>> h2(10.0)
>>> tab2(10.0)
>>> tvs2(10.0)
>>> k4(0.005)
>>> gamma(0.0)

>>> with model.add_aetmodel_v1("evap_aet_hbv96"):
...     temperaturethresholdice(0.0)
...     soilmoisturelimit(0.8)
...     excessreduction(0.5)
...     with model.add_petmodel_v1("evap_pet_hbv96"):
...         evapotranspirationfactor(0.7)
...         airtemperaturefactor(0.1)
...         altitudefactor(-0.1)
...         precipitationfactor(0.1)

>>> with model.add_rconcmodel_v1("rconc_nash"):
...     retentiontime(3.0)
...     nmbstorages(5)
...     nmbsteps(100)

>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = (
...     inputs.p, fluxes.qab1, fluxes.qab2, fluxes.q1, fluxes.rt)
>>> IntegrationTest.plotting_options.axis2 = inputs.t
>>> test = IntegrationTest(land)
>>> test.dateformat = "%d/%m %H:00"

>>> test.inits = ((states.ic, 0.0),
...               (states.sp, 0.0),
...               (states.wc, 0.0),
...               (states.sm, 100.0),
...               (states.bw1, 0.0),
...               (states.bw2, 0.0),
...               (states.lz, 10.0),
...               (model.rconcmodel.sequences.states.sc, 0.05))

>>> inputs.p.series = (
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.2, 0.0, 0.0, 1.3, 5.6, 2.9, 4.9, 10.6, 0.1, 0.7, 3.0, 2.1, 10.4, 3.5, 3.4,
...     1.2, 0.1, 0.0, 0.0, 0.4, 0.1, 3.6, 5.9, 1.1, 20.7, 37.9, 8.2, 3.6, 7.5, 18.5,
...     15.4, 6.3, 1.9, 4.9, 2.7, 0.5, 0.2, 0.5, 2.4, 0.4, 0.2, 0.0, 0.0, 0.3, 2.6,
...     0.7, 0.3, 0.3, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     1.3, 0.0, 0.0, 0.0, 0.7, 0.4, 0.1, 0.4, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 0.0, 0.0, 0.0, 0.0)
>>> inputs.t.series = (
...     21.2, 19.4, 18.9, 18.3, 18.9, 22.5, 25.1, 28.3, 27.8, 31.4, 32.2, 35.2, 37.1,
...     31.2, 24.3, 25.4, 25.9, 23.7, 21.6, 21.2, 20.4, 19.8, 19.6, 19.2, 19.2, 19.2,
...     18.9, 18.7, 18.5, 18.3, 18.5, 18.8, 18.8, 19.0, 19.2, 19.3, 19.0, 18.8, 18.7,
...     17.8, 17.4, 17.3, 16.8, 16.5, 16.3, 16.2, 15.5, 14.6, 14.7, 14.6, 14.1, 14.3,
...     14.9, 15.7, 16.0, 16.7, 17.1, 16.2, 15.9, 16.3, 16.3, 16.4, 16.5, 18.4, 18.3,
...     18.1, 16.7, 15.2, 13.4, 12.4, 11.6, 11.0, 10.5, 11.7, 11.9, 11.2, 11.1, 11.9,
...     12.2, 11.8, 11.4, 11.6, 13.0, 17.1, 18.2, 22.4, 21.4, 21.8, 22.2, 20.1, 17.8,
...     15.2, 14.5, 12.4, 11.7, 11.9)
>>> petinputs = model.aetmodel.petmodel.sequences.inputs
>>> petinputs.normalairtemperature.series = inputs.t.series - 1.0
>>> petinputs.normalevapotranspiration.series = (
...     0.100707, 0.097801, 0.096981, 0.09599, 0.096981, 0.102761, 0.291908, 1.932875,
...     4.369536, 7.317556, 8.264362, 9.369867, 5.126178, 6.62503, 7.397619, 2.39151,
...     1.829834, 1.136569, 0.750986, 0.223895, 0.099425, 0.098454, 0.098128, 0.097474,
...     0.097474, 0.097474, 0.096981, 0.096652, 0.096321, 0.09599, 0.187298, 1.264612,
...     3.045538, 1.930758, 2.461001, 6.215945, 3.374783, 8.821555, 4.046025, 2.110757,
...     2.239257, 2.877848, 1.591452, 0.291604, 0.092622, 0.092451, 0.091248, 0.089683,
...     0.089858, 0.089683, 0.088805, 0.089157, 0.090207, 0.091593, 0.154861, 0.470369,
...     1.173726, 4.202296, 4.359715, 5.305753, 5.376027, 4.658915, 7.789594, 4.851567,
...     5.30692, 3.286036, 1.506216, 0.274762, 0.087565, 0.085771, 0.084317, 0.083215,
...     0.082289, 0.0845, 0.084864, 0.083584, 0.0834, 0.084864, 0.310229, 1.391958,
...     3.195876, 5.191651, 7.155036, 8.391432, 8.391286, 10.715238, 9.383394, 7.861915,
...     6.298329, 2.948416, 1.309232, 0.32955, 0.089508, 0.085771, 0.0845, 0.084864)

>>> test.reset_inits()
>>> conditions = model.conditions

.. _hland_96c_field:

field
_____

|hland_96c| neither implements an approach analogue to HBV96's contributing area
concept nor a substep mechanism controlling the numerical accuracy of the runoff
generation module.  Hence, we provide only a single "field" example, that is comparable
both with the :ref:`first <hland_96_field>` and the :ref:`second <hland_96_resparea>`
example of |hland_96|:

.. integration-test::

    >>> test("hland_96c_field")
    |        date |    p |    t |   tc | fracrain | rfc | sfc |    cfact | swe | gact |     pc |       ei |        tf | spl | wcl | spg | wcg | glmelt | melt | refr |       in_ |         r |  sr |       ea |     qvs1 |     qab1 |     qvs2 |     qab2 |  el |       q1 |     inrc |    outrc |       rt |       qt |       ic |  sp |  wc |         sm |       bw1 |       bw2 |        lz |   outlet |
    -------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.048465 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 |     0.05 |      0.0 | 0.160107 | 0.210107 | 0.058363 |      0.0 | 0.0 | 0.0 |  99.951535 |       0.0 |       0.0 |      9.95 | 0.058363 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.047044 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 |  0.04975 |      0.0 | 0.073779 | 0.123529 | 0.034314 |      0.0 | 0.0 | 0.0 |  99.904491 |       0.0 |       0.0 |   9.90025 | 0.034314 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046628 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.049501 |      0.0 | 0.014282 | 0.063783 | 0.017717 |      0.0 | 0.0 | 0.0 |  99.857863 |       0.0 |       0.0 |  9.850749 | 0.017717 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.04613 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.049254 |      0.0 | 0.001674 | 0.050927 | 0.014147 |      0.0 | 0.0 | 0.0 |  99.811734 |       0.0 |       0.0 |  9.801495 | 0.014147 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.046584 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.049007 |      0.0 | 0.000147 | 0.049155 | 0.013654 |      0.0 | 0.0 | 0.0 |   99.76515 |       0.0 |       0.0 |  9.752488 | 0.013654 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.049338 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048762 |      0.0 | 0.000011 | 0.048773 | 0.013548 |      0.0 | 0.0 | 0.0 |  99.715812 |       0.0 |       0.0 |  9.703725 | 0.013548 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.140081 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048519 |      0.0 | 0.000001 | 0.048519 | 0.013478 |      0.0 | 0.0 | 0.0 |   99.57573 |       0.0 |       0.0 |  9.655206 | 0.013478 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  0.92625 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048276 |      0.0 |      0.0 | 0.048276 |  0.01341 |      0.0 | 0.0 | 0.0 |  98.649481 |       0.0 |       0.0 |   9.60693 |  0.01341 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  2.07444 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048035 |      0.0 |      0.0 | 0.048035 | 0.013343 |      0.0 | 0.0 | 0.0 |  96.575041 |       0.0 |       0.0 |  9.558896 | 0.013343 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.400961 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047794 |      0.0 |      0.0 | 0.047794 | 0.013276 |      0.0 | 0.0 | 0.0 |   93.17408 |       0.0 |       0.0 |  9.511101 | 0.013276 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.705742 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047556 |      0.0 |      0.0 | 0.047556 |  0.01321 |      0.0 | 0.0 | 0.0 |  89.468338 |       0.0 |       0.0 |  9.463546 |  0.01321 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.03435 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047318 |      0.0 |      0.0 | 0.047318 | 0.013144 |      0.0 | 0.0 | 0.0 |  85.433988 |       0.0 |       0.0 |  9.416228 | 0.013144 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.107634 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047081 |      0.0 |      0.0 | 0.047081 | 0.013078 |      0.0 | 0.0 | 0.0 |  83.326354 |       0.0 |       0.0 |  9.369147 | 0.013078 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.656691 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046846 |      0.0 |      0.0 | 0.046846 | 0.013013 |      0.0 | 0.0 | 0.0 |  80.669664 |       0.0 |       0.0 |  9.322301 | 0.013013 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.871924 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046612 |      0.0 |      0.0 | 0.046612 | 0.012948 |      0.0 | 0.0 | 0.0 |   77.79774 |       0.0 |       0.0 |   9.27569 | 0.012948 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  0.242 |    0.242 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.873977 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046378 |      0.0 |      0.0 | 0.046378 | 0.012883 |      0.0 | 0.0 | 0.0 |  76.923763 |       0.0 |       0.0 |  9.229311 | 0.012883 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.677397 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046147 |      0.0 |      0.0 | 0.046147 | 0.012818 |      0.0 | 0.0 | 0.0 |  76.246366 |       0.0 |       0.0 |  9.183165 | 0.012818 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.417048 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045916 |      0.0 |      0.0 | 0.045916 | 0.012754 |      0.0 | 0.0 | 0.0 |  75.829319 |       0.0 |       0.0 |  9.137249 | 0.012754 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  1.573 | 0.494092 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.117083 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045686 |      0.0 |      0.0 | 0.045686 | 0.012691 | 1.078908 | 0.0 | 0.0 |  75.712235 |       0.0 |       0.0 |  9.091563 | 0.012691 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  6.776 |  0.08755 |  5.854908 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.854908 |  0.839058 | 0.0 | 0.022087 | 0.178771 |      0.0 | 0.008648 |      0.0 | 0.0 | 0.045501 |      0.0 |      0.0 | 0.045501 | 0.012639 |  1.91245 | 0.0 | 0.0 |  80.705998 |  0.660288 |  0.170123 |  9.054709 | 0.012639 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  3.509 |   0.0539 |   3.42145 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.42145 |  0.557137 | 0.0 | 0.014076 | 0.378507 |      0.0 | 0.034499 |      0.0 | 0.0 | 0.045446 |      0.0 |      0.0 | 0.045446 | 0.012624 |   1.9461 | 0.0 | 0.0 |  83.556234 |  0.838917 |  0.514131 |  9.043763 | 0.012624 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  5.929 | 0.041902 |    5.8751 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    5.8751 |  1.025446 | 0.0 | 0.011576 | 0.548571 |      0.0 | 0.075463 |      0.0 | 0.0 | 0.045596 |      0.0 |      0.0 | 0.045596 | 0.012666 | 1.958098 | 0.0 | 0.0 |  88.394311 |  1.315792 |  0.987239 |  9.073629 | 0.012666 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 | 12.826 | 0.020954 | 12.784098 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.784098 |  2.497231 | 0.0 | 0.006462 | 1.049787 |      0.0 | 0.144731 |      0.0 | 0.0 | 0.046092 |      0.0 |      0.0 | 0.046092 | 0.012803 | 1.979046 | 0.0 | 0.0 |  98.674717 |  2.763236 |  1.892296 |  9.172268 | 0.012803 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.450977 | 0.0 |  0.0 |  0.121 | 0.074152 |  0.100046 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.100046 |  0.024353 | 0.0 | 0.022883 | 1.092437 |      0.0 | 0.232922 |      0.0 | 0.0 | 0.047026 |      0.0 |      0.0 | 0.047026 | 0.013063 | 1.925848 | 0.0 | 0.0 |  98.727527 |  1.695152 |  2.751812 |  9.358164 | 0.013063 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.847 |  0.06896 |  0.772848 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.772848 |  0.188326 | 0.0 | 0.021402 | 0.707115 |      0.0 | 0.296076 |      0.0 | 0.0 | 0.048271 |      0.0 |      0.0 | 0.048271 | 0.013409 |  1.93104 | 0.0 | 0.0 |  99.290647 |  1.176363 |  3.162851 |  9.605968 | 0.013409 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |   3.63 | 0.052207 |   3.56104 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.56104 |  0.877675 | 0.0 | 0.016637 | 0.649861 |      0.0 | 0.332422 |      0.0 | 0.0 | 0.049692 |      0.0 |      0.0 | 0.049692 | 0.013803 | 1.947793 | 0.0 | 0.0 | 101.957376 |  1.404176 |  3.480291 |  9.888698 | 0.013803 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  2.541 | 0.057919 |  2.488793 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.488793 |  0.646794 | 0.0 | 0.018787 | 0.690307 |      0.0 | 0.364587 |      0.0 | 0.0 | 0.051266 |      0.0 |      0.0 | 0.051266 | 0.014241 | 1.942081 | 0.0 | 0.0 | 103.780587 |  1.360663 |  3.806011 | 10.202018 | 0.014241 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 12.584 | 0.021144 | 12.526081 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.526081 |  3.372776 | 0.0 | 0.007462 | 1.253987 |      0.0 |  0.42285 |      0.0 | 0.0 | 0.053124 |      0.0 |      0.0 | 0.053124 | 0.014757 | 1.978856 | 0.0 | 0.0 |  112.92643 |  3.479452 |  4.637148 | 10.571744 | 0.014757 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.235 | 0.048561 |  4.213856 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.213856 |  1.343417 | 0.0 | 0.017573 | 1.655288 |      0.0 | 0.521356 |      0.0 | 0.0 | 0.055466 |      0.0 |      0.0 | 0.055466 | 0.015407 | 1.951439 | 0.0 | 0.0 | 115.779297 |  3.167581 |   5.77108 | 11.037635 | 0.015407 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.114 | 0.048983 |  4.065439 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.065439 |  1.362415 | 0.0 | 0.018136 | 1.536624 |      0.0 | 0.623524 |      0.0 | 0.0 | 0.058306 |      0.0 |      0.0 | 0.058306 | 0.016196 | 1.951017 | 0.0 | 0.0 | 118.464185 |  2.993372 |   6.68418 | 11.602853 | 0.016196 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  1.452 | 0.124728 |  1.403017 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.403017 |   0.49224 | 0.0 | 0.046529 | 1.282677 |      0.0 | 0.698132 |      0.0 | 0.0 | 0.061505 |      0.0 |      0.0 | 0.061505 | 0.017085 | 1.875272 | 0.0 | 0.0 | 119.328432 |  2.202935 |  7.268725 |  12.23948 | 0.017085 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.121 |  0.96204 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.358746 | 0.866787 |      0.0 | 0.733641 |      0.0 | 0.0 | 0.064866 |      0.0 |      0.0 | 0.064866 | 0.018018 | 1.034232 | 0.0 | 0.0 | 118.969686 |  1.336147 |  7.401871 | 12.908255 | 0.018018 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.0 | 1.034232 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.527265 | 0.525733 |      0.0 | 0.729813 |      0.0 | 0.0 |  0.06819 |      0.0 |      0.0 |  0.06819 | 0.018942 |      0.0 | 0.0 | 0.0 | 117.442421 |  0.810414 |  7.197791 | 13.569878 | 0.018942 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.091248 | 0.318873 |      0.0 | 0.700386 |      0.0 | 0.0 | 0.071351 |      0.0 |      0.0 | 0.071351 |  0.01982 |      0.0 | 0.0 | 0.0 | 116.351173 |  0.491541 |  6.816279 | 14.198912 |  0.01982 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.312905 | 0.193406 |      0.0 | 0.658011 |      0.0 | 0.0 | 0.074285 |      0.0 |      0.0 | 0.074285 | 0.020635 |      0.0 | 0.0 | 0.0 | 115.038267 |  0.298135 |  6.351675 | 14.782638 | 0.020635 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.399893 | 0.117307 |      0.0 | 0.610116 |      0.0 | 0.0 | 0.076964 |      0.0 |      0.0 | 0.076964 | 0.021379 |      0.0 | 0.0 | 0.0 | 111.638374 |  0.180828 |  5.858865 | 15.315791 | 0.021379 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.356 | 1.680962 |     2.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     2.356 |  0.734078 | 0.0 | 0.594957 | 0.227554 |      0.0 | 0.568552 |      0.0 | 0.0 | 0.079422 |      0.0 |      0.0 | 0.079422 | 0.022062 | 0.319038 | 0.0 | 0.0 | 112.665339 |  0.687352 |  5.517867 | 15.804922 | 0.022062 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  7.139 |      2.0 |  5.458038 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.458038 |  1.732037 | 0.0 | 1.873211 | 0.639482 |      0.0 | 0.556029 |      0.0 | 0.0 | 0.081805 |      0.0 |      0.0 | 0.081805 | 0.022724 |      0.0 | 0.0 | 0.0 | 114.518129 |  1.779907 |   5.60132 | 16.279146 | 0.022724 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  1.331 |    1.331 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.674068 | 0.700339 |      0.0 | 0.566914 |      0.0 | 0.0 |  0.08423 |      0.0 |      0.0 |  0.08423 | 0.023397 |      0.0 | 0.0 | 0.0 | 112.844061 |  1.079568 |  5.734744 |  16.76183 | 0.023397 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 25.047 | 0.132786 |    23.047 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    23.047 |  7.336887 | 0.0 | 0.053344 | 1.987984 |      0.0 |   0.6419 |      0.0 | 0.0 | 0.087019 |      0.0 |      0.0 | 0.087019 | 0.024172 | 1.867214 | 0.0 | 0.0 | 128.500829 |  6.428471 |  7.080828 | 17.316712 | 0.024172 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 45.859 | 0.017578 | 45.726214 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.726214 | 18.876311 | 0.0 | 0.008533 | 6.359947 | 1.571769 | 0.980593 | 0.057313 | 0.0 | 0.091487 | 1.629082 | 0.116284 |  0.20777 | 0.057714 | 1.982422 | 0.0 | 0.0 | 155.342199 | 17.373066 | 12.402869 | 18.205818 | 0.057714 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  9.922 | 0.821583 |  9.904422 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.904422 |   5.97514 | 0.0 | 0.408921 |   7.5097 |   2.5097 | 1.522612 | 0.522612 | 0.0 | 0.098642 | 3.032312 | 1.032129 | 1.130771 | 0.314103 | 1.178417 | 0.0 | 0.0 | 158.862561 | 13.328805 | 17.867345 | 19.629788 | 0.314103 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  4.356 | 0.792694 |  3.534417 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.534417 |   2.22998 | 0.0 | 0.396347 | 5.541587 | 0.571295 | 1.925722 | 0.925722 | 0.0 | 0.107778 | 1.497017 | 2.187312 |  2.29509 | 0.637525 | 1.207306 | 0.0 | 0.0 | 159.770651 |  9.445903 | 20.557487 | 21.447733 | 0.637525 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  9.075 | 0.090607 |  8.282306 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  8.282306 |   5.28549 | 0.0 | 0.045304 | 4.842807 |      0.0 | 2.136821 | 1.136821 | 0.0 | 0.117923 | 1.136821 |  1.99624 | 2.114163 | 0.587267 | 1.909393 | 0.0 | 0.0 | 162.722163 |  9.888586 | 22.126653 | 23.466631 | 0.587267 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 22.385 | 0.007604 | 22.294393 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 22.294393 | 14.758053 | 0.0 | 0.003802 |  6.75958 | 1.759897 | 2.368798 | 1.368798 | 0.0 | 0.129177 | 3.128695 | 1.545296 | 1.674474 | 0.465132 | 1.992396 | 0.0 | 0.0 | 170.254701 | 16.127162 | 25.148638 | 25.706251 | 0.465132 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 | 18.634 | 0.011044 | 18.626396 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.626396 | 13.497927 | 0.0 | 0.005522 | 8.499659 | 3.499659 | 2.724177 | 1.724177 | 0.0 | 0.142152 | 5.223836 | 2.469326 | 2.611478 | 0.725411 | 1.988956 | 0.0 | 0.0 | 175.377648 | 17.625771 | 29.199943 | 28.288276 | 0.725411 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  7.623 | 0.032783 |  7.611956 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.611956 |  5.853084 | 0.0 | 0.016392 | 7.567119 | 2.567119 | 3.047697 | 2.047697 | 0.0 |  0.15668 | 4.614817 | 4.123532 | 4.280212 | 1.188948 | 1.967217 | 0.0 | 0.0 | 177.120128 | 13.344616 | 31.671668 | 31.179293 | 1.188948 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451153 | 0.0 |  0.0 |  2.299 | 0.054873 |  2.266217 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.266217 |  1.777368 | 0.0 | 0.027436 | 5.461444 | 0.525218 | 3.173119 | 2.173119 | 0.0 | 0.171762 | 2.698337 | 4.546143 | 4.717906 | 1.310529 | 1.945127 | 0.0 | 0.0 | 177.581541 |  9.135322 | 31.786874 |  34.18065 | 1.310529 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  5.929 | 0.038243 |  5.874127 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.874127 |  4.631045 | 0.0 | 0.019122 | 4.581166 |      0.0 |  3.14234 |  2.14234 | 0.0 | 0.186615 |  2.14234 | 3.507185 |   3.6938 | 1.026056 | 1.961757 | 0.0 | 0.0 | 178.805502 |  9.185201 |  31.08336 | 37.136375 | 1.026056 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  3.267 |  0.04981 |  3.228757 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.228757 |  2.580697 | 0.0 | 0.024905 | 4.163942 |      0.0 |  3.05904 |  2.05904 | 0.0 | 0.200977 |  2.05904 | 2.537066 | 2.738043 | 0.760567 |  1.95019 | 0.0 | 0.0 | 179.428656 |  7.601957 | 30.129222 | 39.994438 | 0.760567 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.605 | 0.064366 |   0.55519 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   0.55519 |  0.446854 | 0.0 | 0.032183 | 3.086344 |      0.0 | 2.922101 | 1.922101 | 0.0 | 0.214583 | 1.922101 | 2.149951 | 2.364534 | 0.656815 | 1.935634 | 0.0 | 0.0 |  179.50481 |  4.962466 | 28.371363 | 42.701957 | 0.656815 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.242 | 0.067009 |  0.177634 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.177634 |  0.143093 | 0.0 | 0.033505 | 1.983066 |      0.0 | 2.711115 | 1.711115 | 0.0 | 0.227065 | 1.711115 | 1.979601 | 2.206667 | 0.612963 | 1.932991 | 0.0 | 0.0 | 179.505846 |  3.122494 | 25.932198 | 45.186007 | 0.612963 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.605 | 0.065382 |  0.537991 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.537991 |  0.433383 | 0.0 | 0.032691 | 1.320943 |      0.0 | 2.459038 | 1.459038 | 0.0 | 0.238225 | 1.459038 | 1.796155 |  2.03438 | 0.565106 | 1.934618 | 0.0 | 0.0 | 179.577763 |  2.234934 | 23.335066 | 47.406819 | 0.565106 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  2.904 | 0.052751 |  2.838618 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.838618 |  2.288506 | 0.0 | 0.026376 |  1.36697 |      0.0 | 2.225803 | 1.225803 | 0.0 | 0.248163 | 1.225803 | 1.569178 | 1.817341 | 0.504817 | 1.947249 | 0.0 | 0.0 | 180.101499 |   3.15647 |  21.25043 | 49.384459 | 0.504817 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.484 | 0.113609 |  0.431249 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.431249 |  0.349705 | 0.0 | 0.056805 | 1.316483 |      0.0 | 2.034498 | 1.034498 | 0.0 | 0.257095 | 1.034498 | 1.335006 | 1.592101 |  0.44225 | 1.886391 | 0.0 | 0.0 | 180.126238 |  2.189693 | 19.497917 | 51.161863 |  0.44225 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.242 | 0.353524 |  0.128391 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.128391 |  0.104143 | 0.0 | 0.176762 | 0.883766 |      0.0 | 1.855397 | 0.855397 | 0.0 | 0.265086 | 0.855397 | 1.125533 | 1.390619 | 0.386283 | 1.646476 | 0.0 | 0.0 | 179.973724 |   1.41007 | 17.670888 | 52.752173 | 0.386283 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 | 0.903769 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.451885 | 0.554819 |      0.0 | 1.674402 | 0.674402 | 0.0 | 0.272133 | 0.674402 | 0.937458 | 1.209591 | 0.335997 | 0.742707 | 0.0 | 0.0 | 179.521839 |   0.85525 | 15.876904 | 54.154442 | 0.335997 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 | 0.742707 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.864415 | 0.336515 |      0.0 | 1.501582 | 0.501582 | 0.0 |  0.27828 | 0.501582 | 0.756425 | 1.034706 | 0.287418 |      0.0 | 0.0 | 0.0 | 176.657424 |  0.518736 | 14.210255 | 55.377744 | 0.287418 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.055807 | 0.204107 |      0.0 | 1.344326 | 0.344326 | 0.0 |  0.28361 | 0.344326 |  0.58122 |  0.86483 | 0.240231 |      0.0 | 0.0 | 0.0 | 173.601617 |  0.314629 |  12.72571 | 56.438459 | 0.240231 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  3.146 |      2.0 |     1.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |     1.146 |   0.86344 | 0.0 | 1.982694 | 0.307763 |      0.0 | 1.214628 | 0.214628 | 0.0 | 0.288265 | 0.214628 | 0.418599 | 0.706864 | 0.196351 |      0.0 | 0.0 | 0.0 | 171.901483 |  0.870306 | 11.604216 | 57.364822 | 0.196351 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.847 |    0.847 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  3.37986 | 0.342439 |      0.0 | 1.114606 | 0.114606 | 0.0 | 0.292397 | 0.114606 | 0.277859 | 0.570256 | 0.158404 |      0.0 | 0.0 | 0.0 | 168.521623 |  0.527868 | 10.717443 | 58.187031 | 0.158404 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.277978 |   0.2077 |      0.0 | 1.027918 | 0.029027 | 0.0 | 0.296075 | 0.029027 | 0.163728 | 0.459803 | 0.127723 |      0.0 | 0.0 | 0.0 | 165.243645 |  0.320168 |  9.868198 | 58.918874 | 0.127723 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.602665 | 0.125976 |      0.0 | 0.945177 |      0.0 | 0.0 |  0.29932 |      0.0 | 0.073266 | 0.372586 | 0.103496 |      0.0 | 0.0 | 0.0 |  159.64098 |  0.194192 |  9.048997 | 59.564731 | 0.103496 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.727324 | 0.076408 |      0.0 | 0.864822 |      0.0 | 0.0 | 0.302148 |      0.0 | 0.020602 |  0.32275 | 0.089653 |      0.0 | 0.0 | 0.0 | 155.913656 |  0.117783 |  8.260583 | 60.127406 | 0.089653 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.981965 | 0.046344 |      0.0 |  0.78834 |      0.0 | 0.0 | 0.304579 |      0.0 | 0.003329 | 0.307908 |  0.08553 |      0.0 | 0.0 | 0.0 | 151.931691 |  0.071439 |  7.518587 | 60.611167 |  0.08553 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 2.402655 | 0.028109 |      0.0 | 0.716848 |      0.0 | 0.0 |  0.30664 |      0.0 | 0.000362 | 0.307002 | 0.085278 |      0.0 | 0.0 | 0.0 | 149.529036 |   0.04333 |  6.829848 | 61.021375 | 0.085278 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.083886 | 0.017049 |      0.0 | 0.650771 |      0.0 | 0.0 | 0.308361 |      0.0 | 0.000031 | 0.308391 | 0.085664 |      0.0 | 0.0 | 0.0 |  148.44515 |  0.026281 |  6.196126 | 61.363785 | 0.085664 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.196288 | 0.010341 |      0.0 |  0.59014 |      0.0 | 0.0 |  0.30977 |      0.0 | 0.000002 | 0.309772 | 0.086048 |      0.0 | 0.0 | 0.0 | 148.248862 |   0.01594 |  5.616328 | 61.644155 | 0.086048 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.062473 | 0.006272 |      0.0 | 0.534768 |      0.0 | 0.0 | 0.310895 |      0.0 |      0.0 | 0.310895 |  0.08636 |      0.0 | 0.0 | 0.0 | 148.186389 |  0.009668 |  5.087832 | 61.868028 |  0.08636 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.061167 | 0.003804 |      0.0 | 0.484355 |      0.0 | 0.0 | 0.311762 |      0.0 |      0.0 | 0.311762 | 0.086601 |      0.0 | 0.0 | 0.0 | 148.125222 |  0.005864 |  4.607281 | 62.040621 | 0.086601 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060106 | 0.002307 |      0.0 | 0.438552 |      0.0 | 0.0 | 0.312396 |      0.0 |      0.0 | 0.312396 | 0.086777 |      0.0 | 0.0 | 0.0 | 148.065116 |  0.003557 |  4.171036 | 62.166778 | 0.086777 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.059296 | 0.001399 |      0.0 | 0.396994 |      0.0 | 0.0 | 0.312819 |      0.0 |      0.0 | 0.312819 | 0.086894 |      0.0 | 0.0 | 0.0 |  148.00582 |  0.002157 |  3.775441 | 62.250953 | 0.086894 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.058613 | 0.000849 |      0.0 | 0.359322 |      0.0 | 0.0 | 0.313051 |      0.0 |      0.0 | 0.313051 | 0.086959 |      0.0 | 0.0 | 0.0 | 147.947208 |  0.001308 |  3.416968 | 62.297224 | 0.086959 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060164 | 0.000515 |      0.0 | 0.325192 |      0.0 | 0.0 | 0.313112 |      0.0 |      0.0 | 0.313112 | 0.086976 |      0.0 | 0.0 | 0.0 | 147.887044 |  0.000794 |  3.092291 | 62.309304 | 0.086976 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.060398 | 0.000312 |      0.0 | 0.294285 |      0.0 | 0.0 | 0.313018 |      0.0 |      0.0 | 0.313018 | 0.086949 |      0.0 | 0.0 | 0.0 | 147.826646 |  0.000481 |  2.798317 | 62.290571 | 0.086949 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  1.573 | 0.054992 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.025404 | 0.000189 |      0.0 | 0.266304 |      0.0 | 0.0 | 0.312784 |      0.0 |      0.0 | 0.312784 | 0.086885 | 1.518008 | 0.0 | 0.0 | 147.801242 |  0.000292 |  2.532202 | 62.244091 | 0.086885 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 | 0.064218 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.029661 | 0.000115 |      0.0 | 0.240976 |      0.0 | 0.0 | 0.312425 |      0.0 |      0.0 | 0.312425 | 0.086785 |  1.45379 | 0.0 | 0.0 | 147.771581 |  0.000177 |  2.291341 | 62.172642 | 0.086785 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 | 0.065345 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.030176 |  0.00007 |      0.0 | 0.218053 |      0.0 | 0.0 | 0.311953 |      0.0 |      0.0 | 0.311953 | 0.086654 | 1.388445 | 0.0 | 0.0 | 147.741405 |  0.000107 |  2.073357 | 62.078742 | 0.086654 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 | 0.238876 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.110287 | 0.000042 |      0.0 | 0.197308 |      0.0 | 0.0 |  0.31138 |      0.0 |      0.0 |  0.31138 | 0.086495 | 1.149568 | 0.0 | 0.0 | 147.631118 |  0.000065 |  1.876091 |  61.96467 | 0.086495 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.847 | 0.984764 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.454318 | 0.000026 |      0.0 | 0.178535 |      0.0 | 0.0 | 0.310716 |      0.0 |      0.0 | 0.310716 |  0.08631 | 1.011805 | 0.0 | 0.0 |   147.1768 |   0.00004 |  1.697582 | 61.832489 |  0.08631 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.484 | 1.495805 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.502702 | 0.000016 |      0.0 | 0.161547 |      0.0 | 0.0 |  0.30997 |      0.0 |      0.0 |  0.30997 | 0.086103 |      0.0 | 0.0 | 0.0 | 145.674097 |  0.000024 |  1.536051 | 61.684066 | 0.086103 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.595867 | 0.000009 |      0.0 | 0.146175 |      0.0 | 0.0 | 0.309151 |      0.0 |      0.0 | 0.309151 | 0.085875 |      0.0 | 0.0 | 0.0 |  142.07823 |  0.000015 |  1.389885 |  61.52109 | 0.085875 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  4.66112 | 0.000006 |      0.0 | 0.132265 |      0.0 | 0.0 | 0.308267 |      0.0 |      0.0 | 0.308267 |  0.08563 |      0.0 | 0.0 | 0.0 | 137.417111 |  0.000009 |  1.257625 | 61.345088 |  0.08563 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.549421 | 0.000003 |      0.0 | 0.119679 |      0.0 | 0.0 | 0.307324 |      0.0 |      0.0 | 0.307324 | 0.085368 |      0.0 | 0.0 | 0.0 |  131.86769 |  0.000005 |   1.13795 | 61.157443 | 0.085368 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 5.325221 | 0.000002 |      0.0 |  0.10829 |      0.0 | 0.0 | 0.306329 |      0.0 |      0.0 | 0.306329 | 0.085091 |      0.0 | 0.0 | 0.0 | 126.542469 |  0.000003 |  1.029662 | 60.959405 | 0.085091 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 6.525426 | 0.000001 |      0.0 | 0.097985 |      0.0 | 0.0 | 0.305287 |      0.0 |      0.0 | 0.305287 | 0.084802 |      0.0 | 0.0 | 0.0 | 120.017043 |  0.000002 |  0.931678 | 60.752103 | 0.084802 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 |  5.41968 | 0.000001 |      0.0 | 0.088661 |      0.0 | 0.0 | 0.304204 |      0.0 |      0.0 | 0.304204 | 0.084501 |      0.0 | 0.0 | 0.0 | 114.597363 |  0.000001 |  0.843017 | 60.536561 | 0.084501 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 4.335845 |      0.0 |      0.0 | 0.080224 |      0.0 | 0.0 | 0.303084 |      0.0 |      0.0 | 0.303084 |  0.08419 |      0.0 | 0.0 | 0.0 | 110.261519 |  0.000001 |  0.762794 |   60.3137 |  0.08419 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 3.342105 |      0.0 |      0.0 | 0.072589 |      0.0 | 0.0 | 0.301931 |      0.0 |      0.0 | 0.301931 |  0.08387 |      0.0 | 0.0 | 0.0 | 106.919414 |       0.0 |  0.690205 | 60.084358 |  0.08387 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 1.517107 |      0.0 |      0.0 | 0.065682 |      0.0 | 0.0 |  0.30075 |      0.0 |      0.0 |  0.30075 | 0.083542 |      0.0 | 0.0 | 0.0 | 105.402307 |       0.0 |  0.624523 |  59.84929 | 0.083542 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.664106 |      0.0 |      0.0 | 0.059431 |      0.0 | 0.0 | 0.299544 |      0.0 |      0.0 | 0.299544 | 0.083207 |      0.0 | 0.0 | 0.0 | 104.738201 |       0.0 |  0.565092 | 59.609178 | 0.083207 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.166111 |      0.0 |      0.0 | 0.053776 |      0.0 | 0.0 | 0.298315 |      0.0 |      0.0 | 0.298315 | 0.082865 |      0.0 | 0.0 | 0.0 | 104.572091 |       0.0 |  0.511317 | 59.364638 | 0.082865 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.045045 |      0.0 |      0.0 | 0.048658 |      0.0 | 0.0 | 0.297066 |      0.0 |      0.0 | 0.297066 | 0.082518 |      0.0 | 0.0 | 0.0 | 104.527046 |       0.0 |  0.462659 |  59.11623 | 0.082518 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.043146 |      0.0 |      0.0 | 0.044028 |      0.0 | 0.0 | 0.295801 |      0.0 |      0.0 | 0.295801 | 0.082167 |      0.0 | 0.0 | 0.0 |   104.4839 |       0.0 |  0.418631 | 58.864457 | 0.082167 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.042489 |      0.0 |      0.0 | 0.039838 |      0.0 | 0.0 | 0.294521 |      0.0 |      0.0 | 0.294521 | 0.081812 |      0.0 | 0.0 | 0.0 | 104.441411 |       0.0 |  0.378793 | 58.609773 | 0.081812 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 | 0.451548 | 0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |       0.0 |       0.0 | 0.0 | 0.042655 |      0.0 |      0.0 | 0.036047 |      0.0 | 0.0 | 0.293229 |      0.0 |      0.0 | 0.293229 | 0.081453 |      0.0 | 0.0 | 0.0 | 104.398756 |       0.0 |  0.342746 | 58.352591 | 0.081453 |

There is no indication of an error in the water balance:

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96c_ilake:

internal lake
_____________

Regarding the simulation of internal lakes, the functionalities of |hland_96c| are
identical to |hland_96|.  Hence, the simulated baseflow shown in the following table
agrees with the :ref:`hland_96_ilake` example of |hland_96|.  Nevertheless, the
resulting subbasin's outflow appears less smoothed due to |hland_96c| not sending its
baseflow through the linear storage cascade:

.. integration-test::

    >>> zonetype(ILAKE)
    >>> model.aetmodel.parameters.control.temperaturethresholdice(13.0)
    >>> test.inits.sc = 0.0
    >>> test.reset_inits()
    >>> conditions = model.conditions
    >>> test("hland_96c_ilake")
    |        date |    p |    t |   tc | fracrain | rfc | sfc | cfact | swe | gact |     pc |  ei |     tf | spl | wcl | spg | wcg | glmelt | melt | refr |    in_ |      r |  sr |  ea | qvs1 | qab1 | qvs2 | qab2 |       el |       q1 | inrc | outrc |       rt |       qt |  ic |  sp |  wc |  sm | bw1 | bw2 |         lz |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.077544 | 0.049612 |  0.0 |   0.0 | 0.049612 | 0.013781 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.872843 | 0.013781 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.075307 | 0.048988 |  0.0 |   0.0 | 0.048988 | 0.013608 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.748549 | 0.013608 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.074675 | 0.048369 |  0.0 |   0.0 | 0.048369 | 0.013436 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.625504 | 0.013436 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.073912 | 0.047758 |  0.0 |   0.0 | 0.047758 | 0.013266 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.503834 | 0.013266 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.074675 | 0.047146 |  0.0 |   0.0 | 0.047146 | 0.013096 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.382013 | 0.013096 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.079126 | 0.046514 |  0.0 |   0.0 | 0.046514 | 0.012921 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   9.256372 | 0.012921 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.224769 | 0.045158 |  0.0 |   0.0 | 0.045158 | 0.012544 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   8.986445 | 0.012544 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 1.488314 | 0.037491 |  0.0 |   0.0 | 0.037491 | 0.010414 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   7.460641 | 0.010414 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 3.364543 |  0.02048 |  0.0 |   0.0 |  0.02048 | 0.005689 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   4.075618 | 0.005689 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 5.634518 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -1.558901 |      0.0 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 6.363559 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -7.922459 |      0.0 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 7.214798 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -15.137257 |      0.0 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 3.947157 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -19.084414 |      0.0 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 5.101273 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -24.185687 |      0.0 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 5.696167 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -29.881854 |      0.0 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.242 |  0.242 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 1.797434 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -31.437288 |      0.0 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 1.408972 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -32.84626 |      0.0 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.875158 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -33.721418 |      0.0 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.573 |  1.573 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.494092 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -32.64251 |      0.0 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  6.776 | 0.0 |  6.776 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  6.776 |  6.776 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.08755 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -25.954061 |      0.0 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  3.509 | 0.0 |  3.509 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.509 |  3.509 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |   0.0539 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -22.498961 |      0.0 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  5.929 | 0.0 |  5.929 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.929 |  5.929 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.041902 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | -16.611863 |      0.0 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 12.826 | 0.0 | 12.826 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.826 | 12.826 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.020954 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -3.806816 |      0.0 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.074152 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -3.759969 |      0.0 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.847 |  0.847 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.06896 |      0.0 |  0.0 |   0.0 |      0.0 |      0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  -2.981928 |      0.0 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |   3.63 | 0.0 |   3.63 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |   3.63 |   3.63 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.052207 | 0.002979 |  0.0 |   0.0 | 0.002979 | 0.000828 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   0.592885 | 0.000828 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  2.541 | 0.0 |  2.541 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.541 |  2.541 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.057919 |  0.01538 |  0.0 |   0.0 |  0.01538 | 0.004272 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   3.060586 | 0.004272 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 12.584 | 0.0 | 12.584 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 12.584 | 12.584 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.021144 | 0.078117 |  0.0 |   0.0 | 0.078117 | 0.021699 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  15.545325 | 0.021699 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.235 | 0.0 |  4.235 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.235 |  4.235 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.048561 | 0.098659 |  0.0 |   0.0 | 0.098659 | 0.027405 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  19.633105 | 0.027405 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.114 | 0.0 |  4.114 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.114 |  4.114 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.048983 | 0.118491 |  0.0 |   0.0 | 0.118491 | 0.032914 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  23.579631 | 0.032914 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.452 | 0.0 |  1.452 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.452 |  1.452 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.124728 | 0.124535 |  0.0 |   0.0 | 0.124535 | 0.034593 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  24.782369 | 0.034593 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.96204 | 0.119707 |  0.0 |   0.0 | 0.119707 | 0.033252 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  23.821622 | 0.033252 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 2.345064 | 0.107383 |  0.0 |   0.0 | 0.107383 | 0.029829 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  21.369175 | 0.029829 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 1.486684 | 0.099412 |  0.0 |   0.0 | 0.099412 | 0.027615 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  19.783079 | 0.027615 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 1.805438 | 0.092308 |  0.0 |   0.0 | 0.092308 | 0.025641 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.369332 | 0.025641 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 4.728713 | 0.068808 |  0.0 |   0.0 | 0.068808 | 0.019113 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  13.692812 | 0.019113 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.356 | 0.0 |  4.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.356 |  4.356 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 1.680962 | 0.081839 |  0.0 |   0.0 | 0.081839 | 0.022733 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   16.28601 | 0.022733 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  7.139 | 0.0 |  7.139 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.139 |  7.139 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 3.326542 | 0.100492 |  0.0 |   0.0 | 0.100492 | 0.027915 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  19.997976 | 0.027915 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.331 | 0.0 |  1.331 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.331 |  1.331 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 2.727186 | 0.093009 |  0.0 |   0.0 | 0.093009 | 0.025836 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  18.508781 | 0.025836 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 25.047 | 0.0 | 25.047 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 25.047 | 25.047 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.132786 | 0.217115 |  0.0 |   0.0 | 0.217115 |  0.06031 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  43.205881 |  0.06031 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 45.859 | 0.0 | 45.859 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 45.859 | 45.859 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.017578 | 0.445237 |  0.0 |   0.0 | 0.445237 | 0.123677 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  88.602066 | 0.123677 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  9.922 | 0.0 |  9.922 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.922 |  9.922 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.821583 | 0.488512 |  0.0 |   0.0 | 0.488512 | 0.135698 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  97.213971 | 0.135698 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  4.356 | 0.0 |  4.356 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  4.356 |  4.356 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.792694 | 0.503886 |  0.0 |   0.0 | 0.503886 | 0.139968 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  100.27339 | 0.139968 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  9.075 | 0.0 |  9.075 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  9.075 |  9.075 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.090607 | 0.546289 |  0.0 |   0.0 | 0.546289 | 0.151747 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 108.711494 | 0.151747 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 22.385 | 0.0 | 22.385 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 22.385 | 22.385 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.007604 | 0.655444 |  0.0 |   0.0 | 0.655444 | 0.182068 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 130.433446 | 0.182068 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 | 18.634 | 0.0 | 18.634 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 | 18.634 | 18.634 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.011044 | 0.745282 |  0.0 |   0.0 | 0.745282 | 0.207023 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  148.31112 | 0.207023 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  7.623 | 0.0 |  7.623 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  7.623 |  7.623 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.032783 | 0.779507 |  0.0 |   0.0 | 0.779507 |  0.21653 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  155.12183 |  0.21653 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  2.299 | 0.0 |  2.299 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.299 |  2.299 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.054873 |  0.78683 |  0.0 |   0.0 |  0.78683 | 0.218564 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 156.579127 | 0.218564 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  5.929 | 0.0 |  5.929 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  5.929 |  5.929 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.038243 | 0.812349 |  0.0 |   0.0 | 0.812349 | 0.225653 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 161.657535 | 0.225653 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  3.267 | 0.0 |  3.267 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.267 |  3.267 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  0.04981 | 0.824374 |  0.0 |   0.0 | 0.824374 | 0.228993 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 164.050351 | 0.228993 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.605 |  0.605 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.064366 | 0.822955 |  0.0 |   0.0 | 0.822955 | 0.228599 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  163.76803 | 0.228599 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.242 |  0.242 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.067009 | 0.819715 |  0.0 |   0.0 | 0.819715 | 0.227699 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 163.123306 | 0.227699 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.605 |  0.605 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.065382 | 0.818315 |  0.0 |   0.0 | 0.818315 |  0.22731 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  162.84461 |  0.22731 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  2.904 | 0.0 |  2.904 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  2.904 |  2.904 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.052751 | 0.828479 |  0.0 |   0.0 | 0.828479 | 0.230133 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 164.867379 | 0.230133 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.113609 | 0.826189 |  0.0 |   0.0 | 0.826189 | 0.229497 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 164.411581 | 0.229497 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.242 |  0.242 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.353524 |   0.8215 |  0.0 |   0.0 |   0.8215 | 0.228195 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 163.478556 | 0.228195 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.903769 | 0.812874 |  0.0 |   0.0 | 0.812874 | 0.225798 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 161.761913 | 0.225798 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 3.235768 | 0.792631 |  0.0 |   0.0 | 0.792631 | 0.220175 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 157.733515 | 0.220175 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.363 |  0.363 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 3.237307 | 0.774296 |  0.0 |   0.0 | 0.774296 | 0.215082 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 154.084911 | 0.215082 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  3.146 | 0.0 |  3.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  3.146 |  3.146 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 2.982694 | 0.771241 |  0.0 |   0.0 | 0.771241 | 0.214234 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 153.476976 | 0.214234 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.847 |  0.847 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  3.80336 | 0.752603 |  0.0 |   0.0 | 0.752603 | 0.209056 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 149.768013 | 0.209056 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.363 |  0.363 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 3.459478 | 0.733358 |  0.0 |   0.0 | 0.733358 |  0.20371 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 145.938177 |  0.20371 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.363 |  0.363 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 5.784165 | 0.702585 |  0.0 |   0.0 | 0.702585 | 0.195163 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 139.814427 | 0.195163 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 3.735707 | 0.680394 |  0.0 |   0.0 | 0.680394 | 0.188998 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 135.398327 | 0.188998 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 4.086328 |  0.65656 |  0.0 |   0.0 |  0.65656 | 0.182378 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 130.655439 | 0.182378 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 2.530248 | 0.640626 |  0.0 |   0.0 | 0.640626 | 0.177952 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 127.484565 | 0.177952 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 1.159786 | 0.631624 |  0.0 |   0.0 | 0.631624 | 0.175451 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 125.693155 | 0.175451 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.211567 | 0.627408 |  0.0 |   0.0 | 0.627408 |  0.17428 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  124.85418 |  0.17428 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.067425 | 0.623934 |  0.0 |   0.0 | 0.623934 | 0.173315 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 124.162821 | 0.173315 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.620814 |  0.0 |   0.0 | 0.620814 | 0.172448 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 123.542007 | 0.172448 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 |  0.61771 |  0.0 |   0.0 |  0.61771 | 0.171586 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 122.924297 | 0.171586 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.614621 |  0.0 |   0.0 | 0.614621 | 0.170728 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 122.309676 | 0.170728 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.611548 |  0.0 |   0.0 | 0.611548 | 0.169875 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 121.698127 | 0.169875 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.608491 |  0.0 |   0.0 | 0.608491 | 0.169025 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 121.089636 | 0.169025 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.605448 |  0.0 |   0.0 | 0.605448 |  0.16818 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 120.484188 |  0.16818 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  1.573 |  1.573 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.610286 |  0.0 |   0.0 | 0.610286 | 0.169524 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 121.446902 | 0.169524 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.607235 |  0.0 |   0.0 | 0.607235 | 0.168676 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 120.839668 | 0.168676 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.604198 |  0.0 |   0.0 | 0.604198 | 0.167833 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  120.23547 | 0.167833 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.601177 |  0.0 |   0.0 | 0.601177 | 0.166994 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.634292 | 0.166994 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.847 |  0.847 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.602406 |  0.0 |   0.0 | 0.602406 | 0.167335 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.878886 | 0.167335 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.601814 |  0.0 |   0.0 | 0.601814 | 0.167171 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.761071 | 0.167171 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.121 |  0.121 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 |  0.59941 |  0.0 |   0.0 |  0.59941 | 0.166503 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 119.282661 | 0.166503 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |  0.484 |  0.484 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 5.249074 | 0.572588 |  0.0 |   0.0 | 0.572588 | 0.159052 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 113.944999 | 0.159052 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 6.461403 | 0.537418 |  0.0 |   0.0 | 0.537418 | 0.149283 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 106.946178 | 0.149283 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  6.46129 | 0.502424 |  0.0 |   0.0 | 0.502424 | 0.139562 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  99.982464 | 0.139562 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 8.250733 | 0.458659 |  0.0 |   0.0 | 0.458659 | 0.127405 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  91.273072 | 0.127405 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 7.225213 | 0.420239 |  0.0 |   0.0 | 0.420239 | 0.116733 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  83.627619 | 0.116733 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 6.053675 |  0.38787 |  0.0 |   0.0 |  0.38787 | 0.107742 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  77.186075 | 0.107742 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 4.849713 | 0.361682 |  0.0 |   0.0 | 0.361682 | 0.100467 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   71.97468 | 0.100467 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |  2.27028 | 0.348522 |  0.0 |   0.0 | 0.348522 | 0.096812 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  69.355877 | 0.096812 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 1.008109 | 0.341739 |  0.0 |   0.0 | 0.341739 | 0.094927 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   68.00603 | 0.094927 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.253754 | 0.338761 |  0.0 |   0.0 | 0.338761 |   0.0941 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  67.413515 |   0.0941 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 | 0.068921 | 0.336723 |  0.0 |   0.0 | 0.336723 | 0.093534 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  67.007871 | 0.093534 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.335039 |  0.0 |   0.0 | 0.335039 | 0.093066 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  66.672832 | 0.093066 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.333364 |  0.0 |   0.0 | 0.333364 | 0.092601 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  66.339467 | 0.092601 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5 |      1.0 | 1.1 | 0.0 |   0.0 | 0.0 |  0.0 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |  0.0 |  0.0 |    0.0 |    0.0 | 0.0 | 0.0 |  0.0 |  0.0 |  0.0 |  0.0 |      0.0 | 0.331697 |  0.0 |   0.0 | 0.331697 | 0.092138 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 0.0 |   66.00777 | 0.092138 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96c_snow_classes:

snow classes
____________

Regarding snow simulation, all functionalities of |hland_96c| are identical with the
ones of |hland_96|.  Hence, all snow data in the following table agrees with the
:ref:`hland_96_snow_classes` example of |hland_96|:

.. integration-test::

    >>> sclass(2)
    >>> sfdist(linear=0.2)
    >>> zonetype(FIELD)
    >>> t_series = inputs.t.series.copy()
    >>> tn_series = petinputs.normalairtemperature.series.copy()
    >>> inputs.t.series[:48] = -20.0
    >>> inputs.t.series[48:] = 20.0
    >>> petinputs.normalairtemperature.series = inputs.t.series
    >>> test("hland_96c_snow_classes")
    |        date |    p |     t |    tc | fracrain | rfc | sfc |    cfact |                    swe | gact |     pc |       ei |        tf | spl | wcl | spg | wcg | glmelt |               melt |      refr |       in_ |        r |  sr |       ea |     qvs1 |     qab1 |     qvs2 |     qab2 |  el |       q1 |     inrc |    outrc |       rt |       qt |       ic |                     sp |                   wc |         sm |       bw1 |       bw2 |        lz |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.044059 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 |     0.05 |      0.0 |      0.0 |     0.05 | 0.013889 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.955941 |       0.0 |       0.0 |      9.95 | 0.013889 |
    | 01/01 01:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.042769 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 |  0.04975 |      0.0 |      0.0 |  0.04975 | 0.013819 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.913172 |       0.0 |       0.0 |   9.90025 | 0.013819 |
    | 01/01 02:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.042392 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.049501 |      0.0 |      0.0 | 0.049501 |  0.01375 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.870779 |       0.0 |       0.0 |  9.850749 |  0.01375 |
    | 01/01 03:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.041941 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.049254 |      0.0 |      0.0 | 0.049254 | 0.013682 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.828838 |       0.0 |       0.0 |  9.801495 | 0.013682 |
    | 01/01 04:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.042357 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.049007 |      0.0 |      0.0 | 0.049007 | 0.013613 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.786481 |       0.0 |       0.0 |  9.752488 | 0.013613 |
    | 01/01 05:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.044862 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048762 |      0.0 |      0.0 | 0.048762 | 0.013545 |      0.0 |        0.0         0.0 |       0.0        0.0 |  99.741619 |       0.0 |       0.0 |  9.703725 | 0.013545 |
    | 01/01 06:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |  0.12738 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048519 |      0.0 |      0.0 | 0.048519 | 0.013477 |      0.0 |        0.0         0.0 |       0.0        0.0 |   99.61424 |       0.0 |       0.0 |  9.655206 | 0.013477 |
    | 01/01 07:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.842371 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048276 |      0.0 |      0.0 | 0.048276 |  0.01341 |      0.0 |        0.0         0.0 |       0.0        0.0 |  98.771869 |       0.0 |       0.0 |   9.60693 |  0.01341 |
    | 01/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.888194 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048035 |      0.0 |      0.0 | 0.048035 | 0.013343 |      0.0 |        0.0         0.0 |       0.0        0.0 |  96.883675 |       0.0 |       0.0 |  9.558896 | 0.013343 |
    | 01/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.101664 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047794 |      0.0 |      0.0 | 0.047794 | 0.013276 |      0.0 |        0.0         0.0 |       0.0        0.0 |  93.782011 |       0.0 |       0.0 |  9.511101 | 0.013276 |
    | 01/01 10:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.390837 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047556 |      0.0 |      0.0 | 0.047556 |  0.01321 |      0.0 |        0.0         0.0 |       0.0        0.0 |  90.391174 |       0.0 |       0.0 |  9.463546 |  0.01321 |
    | 01/01 11:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 3.705421 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047318 |      0.0 |      0.0 | 0.047318 | 0.013144 |      0.0 |        0.0         0.0 |       0.0        0.0 |  86.685753 |       0.0 |       0.0 |  9.416228 | 0.013144 |
    | 01/01 12:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.944104 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047081 |      0.0 |      0.0 | 0.047081 | 0.013078 |      0.0 |        0.0         0.0 |       0.0        0.0 |  84.741649 |       0.0 |       0.0 |  9.369147 | 0.013078 |
    | 01/01 13:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.456195 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046846 |      0.0 |      0.0 | 0.046846 | 0.013013 |      0.0 |        0.0         0.0 |       0.0        0.0 |  82.285455 |       0.0 |       0.0 |  9.322301 | 0.013013 |
    | 01/01 14:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 2.663134 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046612 |      0.0 |      0.0 | 0.046612 | 0.012948 |      0.0 |        0.0         0.0 |       0.0        0.0 |   79.62232 |       0.0 |       0.0 |   9.27569 | 0.012948 |
    | 01/01 15:00 |  0.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |  0.286 |    0.286 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.809588 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046378 |      0.0 |      0.0 | 0.046378 | 0.012883 |      0.0 |        0.0         0.0 |       0.0        0.0 |  78.812732 |       0.0 |       0.0 |  9.229311 | 0.012883 |
    | 01/01 16:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.630937 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046147 |      0.0 |      0.0 | 0.046147 | 0.012818 |      0.0 |        0.0         0.0 |       0.0        0.0 |  78.181795 |       0.0 |       0.0 |  9.183165 | 0.012818 |
    | 01/01 17:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.388758 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045916 |      0.0 |      0.0 | 0.045916 | 0.012754 |      0.0 |        0.0         0.0 |       0.0        0.0 |  77.793036 |       0.0 |       0.0 |  9.137249 | 0.012754 |
    | 01/01 18:00 |  1.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 |  0.0 |  1.859 |  0.43651 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.106117 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045686 |      0.0 |      0.0 | 0.045686 | 0.012691 |  1.42249 |        0.0         0.0 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  9.091563 | 0.012691 |
    | 01/01 19:00 |  5.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   5.944392    8.916588 |  0.0 |  8.008 | 0.070365 |   7.43049 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045458 |      0.0 |      0.0 | 0.045458 | 0.012627 | 1.929635 |   5.944392    8.916588 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  9.046105 | 0.012627 |
    | 01/01 20:00 |  2.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |   9.205699   13.808549 |  0.0 |  4.147 | 0.045972 |  4.076635 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045231 |      0.0 |      0.0 | 0.045231 | 0.012564 | 1.954028 |   9.205699   13.808549 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  9.000874 | 0.012564 |
    | 01/01 21:00 |  4.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  14.774522   22.161783 |  0.0 |  7.007 |   0.0342 |  6.961028 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045004 |      0.0 |      0.0 | 0.045004 | 0.012501 |   1.9658 |  14.774522   22.161783 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |   8.95587 | 0.012501 |
    | 01/01 22:00 | 10.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.873562   40.310343 |  0.0 | 15.158 | 0.015086 |   15.1238 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.044779 |      0.0 |      0.0 | 0.044779 | 0.012439 | 1.984914 |  26.873562   40.310343 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.911091 | 0.012439 |
    | 01/01 23:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |  26.975893    40.46384 |  0.0 |  0.143 | 0.067263 |  0.127914 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.044555 |      0.0 |      0.0 | 0.044555 | 0.012377 | 1.932737 |  26.975893    40.46384 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.866535 | 0.012377 |
    | 02/01 00:00 |  0.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  27.722883   41.584324 |  0.0 |  1.001 | 0.061733 |  0.933737 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.044333 |      0.0 |      0.0 | 0.044333 | 0.012315 | 1.938267 |  27.722883   41.584324 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.822202 | 0.012315 |
    | 02/01 01:00 |  3.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  31.105497   46.658245 |  0.0 |   4.29 |  0.04443 |  4.228267 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.044111 |      0.0 |      0.0 | 0.044111 | 0.012253 |  1.95557 |  31.105497   46.658245 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.778091 | 0.012253 |
    | 02/01 02:00 |  2.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  33.472353   50.208529 |  0.0 |  3.003 | 0.050277 |   2.95857 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 |  0.04389 |      0.0 |      0.0 |  0.04389 | 0.012192 | 1.949723 |  33.472353   50.208529 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.734201 | 0.012192 |
    | 02/01 03:00 | 10.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  45.329731   67.994597 |  0.0 | 14.872 | 0.015291 | 14.821723 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.043671 |      0.0 |      0.0 | 0.043671 | 0.012131 | 1.984709 |  45.329731   67.994597 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |   8.69053 | 0.012131 |
    | 02/01 04:00 |  3.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  49.321499   73.982248 |  0.0 |  5.005 | 0.040875 |  4.989709 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.043453 |      0.0 |      0.0 | 0.043453 |  0.01207 | 1.959125 |  49.321499   73.982248 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.647077 |  0.01207 |
    | 02/01 05:00 |  3.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  53.178399   79.767599 |  0.0 |  4.862 | 0.041321 |  4.821125 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.043235 |      0.0 |      0.0 | 0.043235 |  0.01201 | 1.958679 |  53.178399   79.767599 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.603842 |  0.01201 |
    | 02/01 06:00 |  1.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.518142   81.777214 |  0.0 |  1.716 | 0.110435 |  1.674679 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.043019 |      0.0 |      0.0 | 0.043019 |  0.01195 | 1.889565 |  54.518142   81.777214 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.560823 |  0.01195 |
    | 02/01 07:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |  0.143 |  0.87266 |  0.032565 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.042804 |      0.0 |      0.0 | 0.042804 |  0.01189 |  1.12734 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.518019 |  0.01189 |
    | 02/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |    0.0 |  1.12734 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 |  0.04259 |      0.0 |      0.0 |  0.04259 | 0.011831 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.475429 | 0.011831 |
    | 02/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.042377 |      0.0 |      0.0 | 0.042377 | 0.011771 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.433051 | 0.011771 |
    | 02/01 10:00 |  0.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |  0.572 |    0.572 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.042165 |      0.0 |      0.0 | 0.042165 | 0.011713 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.390886 | 0.011713 |
    | 02/01 11:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  54.544195   81.816292 |  0.0 |  0.143 |    0.143 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.041954 |      0.0 |      0.0 | 0.041954 | 0.011654 |      0.0 |  54.544195   81.816292 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.348932 | 0.011654 |
    | 02/01 12:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  57.062595   85.593892 |  0.0 |  5.148 | 1.411787 |     3.148 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.041745 |      0.0 |      0.0 | 0.041745 | 0.011596 | 0.588213 |  57.062595   85.593892 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.307187 | 0.011596 |
    | 02/01 13:00 |  5.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  62.682765   94.024148 |  0.0 |  8.437 |      2.0 |  7.025213 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.041536 |      0.0 |      0.0 | 0.041536 | 0.011538 |      0.0 |  62.682765   94.024148 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.265651 | 0.011538 |
    | 02/01 14:00 |  1.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  62.682765   94.024148 |  0.0 |  1.573 |    1.573 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.041328 |      0.0 |      0.0 | 0.041328 |  0.01148 |      0.0 |  62.682765   94.024148 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.224323 |  0.01148 |
    | 02/01 15:00 | 20.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  84.763565  127.145348 |  0.0 | 29.601 | 0.076556 |    27.601 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.041122 |      0.0 |      0.0 | 0.041122 | 0.011423 | 1.923444 |  84.763565  127.145348 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.183201 | 0.011423 |
    | 02/01 16:00 | 37.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  128.05992   192.08988 |  0.0 | 54.197 | 0.006942 | 54.120444 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.040916 |      0.0 |      0.0 | 0.040916 | 0.011366 | 1.993058 |  128.05992   192.08988 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.142285 | 0.011366 |
    | 02/01 17:00 |  8.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 137.435167   206.15275 |  0.0 | 11.726 | 0.623609 | 11.719058 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.040711 |      0.0 |      0.0 | 0.040711 | 0.011309 | 1.376391 | 137.435167   206.15275 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.101574 | 0.011309 |
    | 02/01 18:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |  141.05468   211.58202 |  0.0 |  5.148 | 0.665759 |  4.524391 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.040508 |      0.0 |      0.0 | 0.040508 | 0.011252 | 1.334241 |  141.05468   211.58202 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.061066 | 0.011252 |
    | 02/01 19:00 |  7.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 149.102073  223.653109 |  0.0 | 10.725 | 0.069841 | 10.059241 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.040305 |      0.0 |      0.0 | 0.040305 | 0.011196 | 1.930159 | 149.102073  223.653109 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  8.020761 | 0.011196 |
    | 02/01 20:00 | 18.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   170.2102    255.3153 |  0.0 | 26.455 | 0.004601 | 26.385159 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.040104 |      0.0 |      0.0 | 0.040104 |  0.01114 | 1.995399 |   170.2102    255.3153 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  7.980657 |  0.01114 |
    | 02/01 21:00 | 15.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 187.824119  281.736179 |  0.0 | 22.022 | 0.007155 | 22.017399 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.039903 |      0.0 |      0.0 | 0.039903 | 0.011084 | 1.992845 | 187.824119  281.736179 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  7.940753 | 0.011084 |
    | 02/01 22:00 |  6.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 195.025595  292.538393 |  0.0 |  9.009 | 0.025946 |  9.001845 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.039704 |      0.0 |      0.0 | 0.039704 | 0.011029 | 1.974054 | 195.025595  292.538393 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |   7.90105 | 0.011029 |
    | 02/01 23:00 |  1.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 | 197.178439  295.767658 |  0.0 |  2.717 | 0.047842 |  2.691054 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.039505 |      0.0 |      0.0 | 0.039505 | 0.010974 | 1.952158 | 197.178439  295.767658 |       0.0        0.0 |  77.686919 |       0.0 |       0.0 |  7.861544 | 0.010974 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 201.883365  302.825047 |  0.0 |  5.929 | 0.034767 |  5.881158 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.039308 |      0.0 |      0.0 | 0.039308 | 0.010919 | 1.965233 | 188.332116  286.921335 | 13.551249  15.903712 |  77.686919 |       0.0 |       0.0 |  7.822237 | 0.010919 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 204.469152  306.703727 |  0.0 |  3.267 | 0.045282 |  3.232233 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.039111 |      0.0 |      0.0 | 0.039111 | 0.010864 | 1.954718 | 179.485793  278.075012 | 24.983359  28.628715 |  77.686919 |       0.0 |       0.0 |  7.783126 | 0.010864 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 204.916926  307.375389 |  0.0 |  0.605 | 0.058514 |  0.559718 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.038916 |      0.0 |      0.0 | 0.038916 |  0.01081 | 1.941486 |  170.63947  269.228689 | 34.277456    38.1467 |  77.686919 |       0.0 |       0.0 |   7.74421 |  0.01081 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 205.063715  307.595572 |  0.0 |  0.242 | 0.060918 |  0.183486 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.038721 |      0.0 |      0.0 | 0.038721 | 0.010756 | 1.939082 | 161.793147  260.382367 | 43.270568  47.213206 |  77.686919 |       0.0 |       0.0 |  7.705489 | 0.010756 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 205.498981  308.248471 |  0.0 |  0.605 | 0.059438 |  0.544082 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |      0.0 |      0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.038527 |      0.0 |      0.0 | 0.038527 | 0.010702 | 1.940562 | 152.946824  251.536044 | 52.552156  56.712427 |  77.686919 |       0.0 |       0.0 |  7.666961 | 0.010702 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 201.740702  311.661946 |  0.0 |  2.904 | 0.047956 |  2.844562 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  3.016964 | 0.455204 | 0.0 |      0.0 | 0.096986 |      0.0 | 0.004692 |      0.0 | 0.0 | 0.038358 |      0.0 |      0.0 | 0.038358 | 0.010655 | 1.952044 | 144.100502  242.689721 | 57.640201  68.972225 |   80.24868 |  0.358218 |  0.092295 |  7.633295 | 0.010655 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  189.35585  312.185199 |  0.0 |  0.484 | 0.103281 |  0.436044 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.366844 | 1.025038 | 0.0 |      0.0 | 0.359344 |      0.0 | 0.026166 |      0.0 | 0.0 | 0.038297 |      0.0 |      0.0 | 0.038297 | 0.010638 | 1.896719 | 135.254179  233.843398 | 54.101671  78.341801 |  85.590485 |  1.023912 |  0.425472 |  7.621163 | 0.010638 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 176.970998  312.351661 |  0.0 |  0.242 | 0.321386 |  0.138719 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.247914 | 1.144263 | 0.0 |      0.0 | 0.646676 |      0.0 | 0.071771 |      0.0 | 0.0 | 0.038465 |      0.0 |      0.0 | 0.038465 | 0.010685 | 1.678614 | 126.407856  224.997075 | 50.563142  87.354586 |  90.694135 |  1.521499 |  1.000377 |   7.65447 | 0.010685 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 164.586146  302.611053 |  0.0 |    0.0 | 0.821608 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 |  11.06273 | 2.274892 | 0.0 |      0.0 | 1.083355 |      0.0 | 0.147605 |      0.0 | 0.0 |  0.03901 |      0.0 |      0.0 |  0.03901 | 0.010836 | 0.857006 | 117.561533  216.150752 | 47.024613  86.460301 |  99.481974 |  2.713036 |  1.936127 |  7.763065 | 0.010836 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 152.201294  290.226201 |  0.0 |    0.0 | 0.857006 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.064218 | 0.0 |      0.0 | 1.720363 |      0.0 | 0.267468 |      0.0 | 0.0 | 0.040153 |      0.0 |      0.0 | 0.040153 | 0.011154 |      0.0 |  108.71521   207.30443 | 43.486084  82.921772 | 108.802608 |  4.056891 |  3.389022 |   7.99038 | 0.011154 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 139.816442  277.841349 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 3.665299 | 0.0 |      0.0 | 2.377196 |      0.0 | 0.437503 |      0.0 | 0.0 | 0.042139 |      0.0 |      0.0 | 0.042139 | 0.011705 |      0.0 |  99.868887  198.458107 | 39.947555  79.383243 | 117.522161 |  5.344995 |  5.328715 |  8.385744 | 0.011705 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  127.43159  265.456497 |  0.0 |  3.146 |      2.0 |     1.146 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.530852 |  4.67202 | 0.0 |      0.0 | 3.098518 |      0.0 | 0.656983 |      0.0 | 0.0 | 0.045214 |      0.0 |      0.0 | 0.045214 | 0.012559 |      0.0 |  91.022565  189.611784 | 36.409026  75.844714 | 126.380993 |  6.918496 |   7.77025 |  8.997512 | 0.012559 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 115.046738  253.071645 |  0.0 |  0.847 |    0.847 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 4.945319 | 0.0 |      0.0 | 3.775872 |      0.0 | 0.922038 | 0.007068 | 0.0 | 0.049598 | 0.007068 | 0.000504 | 0.050102 | 0.013917 |      0.0 |  82.176242  180.765461 | 32.870497  72.306184 | 133.820525 |  8.087943 | 10.617017 |  9.869953 | 0.013917 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 102.661886  240.686793 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 5.544677 | 0.0 |      0.0 | 4.363714 |      0.0 | 1.213435 | 0.213435 | 0.0 | 0.055417 | 0.213435 | 0.018774 | 0.074191 | 0.020609 |      0.0 |  73.329919  171.919138 | 29.331968  68.767655 |   140.6607 |  9.268907 |  13.55386 | 11.027971 | 0.020609 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  90.277035  228.301941 |  0.0 |  0.363 |    0.363 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 6.125991 | 0.0 |      0.0 | 4.948893 | 0.046855 | 1.507017 | 0.507017 | 0.0 | 0.062675 | 0.553872 | 0.148842 | 0.211516 | 0.058755 |      0.0 |  64.483596  163.072815 | 25.793438  65.229126 |  146.91956 | 10.399151 | 16.488718 | 12.472314 | 0.058755 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  77.892183   215.91709 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 6.683286 | 0.0 |      0.0 | 5.435779 | 0.435779 | 1.795816 | 0.795816 | 0.0 | 0.071341 | 1.231595 | 0.439406 | 0.510746 | 0.141874 |      0.0 |  55.637273  154.226493 | 22.254909  61.690597 | 152.621126 | 11.210879 | 19.332865 | 14.196789 | 0.141874 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  65.507331  203.532238 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 7.212073 | 0.0 |      0.0 | 5.789599 | 0.789599 | 2.070163 | 1.070163 | 0.0 | 0.081335 | 1.859762 | 0.956214 | 1.037549 | 0.288208 |      0.0 |   46.79095   145.38017 |  18.71638  58.152068 | 157.793905 | 11.843755 | 21.982139 | 16.185617 | 0.288208 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  53.122479  191.147386 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 7.709235 | 0.0 |      0.0 | 6.081074 | 1.081074 | 2.323927 | 1.323927 | 0.0 | 0.092548 | 2.405001 | 1.570257 | 1.662805 |  0.46189 |      0.0 |  37.944628  136.533847 | 15.177851  54.613539 | 162.469522 | 12.390842 | 24.415357 | 18.416997 |  0.46189 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  40.737627  178.762534 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.172871 | 0.0 |      0.0 | 6.339267 | 1.339267 | 2.556552 | 1.556552 | 0.0 | 0.104868 | 2.895819 | 2.147802 | 2.252669 | 0.625742 |      0.0 |  29.098305  127.687524 | 11.639322   51.07501 | 166.681503 | 12.885179 | 26.641521 | 20.868681 | 0.625742 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  28.352775  166.377682 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.602123 | 0.0 |      0.0 | 6.574464 | 1.574464 | 2.769333 | 1.769333 | 0.0 |  0.11819 | 3.343797 | 2.665903 | 2.784093 | 0.773359 |      0.0 |  20.251982  118.841201 |  8.100793   47.53648 | 170.464232 | 13.338374 | 28.677319 | 23.519824 | 0.773359 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  15.967923   153.99283 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 8.996993 | 0.0 |      0.0 | 6.790333 | 1.790333 | 2.963955 | 1.963955 | 0.0 | 0.132419 | 3.754288 | 3.134871 | 3.267289 |  0.90758 |      0.0 |  11.405659  109.994878 |  4.562264  43.997951 | 173.852091 |   13.7547 | 30.539742 |  26.35136 |  0.90758 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   3.583071  141.607978 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 9.358164 | 0.0 |      0.0 | 6.988351 | 1.988351 | 3.142028 | 2.142028 | 0.0 | 0.147467 | 4.130379 | 3.563097 | 3.710564 | 1.030712 |      0.0 |   2.559336  101.148556 |  1.023735  40.459422 | 176.878779 | 14.136162 | 32.244038 |  29.34592 | 1.030712 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  129.223126 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 | 2.559336  8.846323 | 0.0   0.0 |  7.983961 | 6.244676 | 0.0 | 0.029511 | 6.536222 | 1.536222 | 3.275324 | 2.275324 | 0.0 | 0.163106 | 3.811546 | 3.907878 | 4.070984 | 1.130829 |      0.0 |        0.0   92.302233 |       0.0  36.920893 | 178.588554 | 12.308394 | 33.229612 | 32.458138 | 1.130829 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  116.838274 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.846323 | 0.0   0.0 |  6.192426 | 4.937511 | 0.0 | 0.029125 | 5.718098 | 0.718098 | 3.326341 | 2.326341 | 0.0 | 0.178922 | 3.044438 | 3.857672 | 4.036594 | 1.121276 |      0.0 |        0.0    83.45591 |       0.0  33.382364 | 179.814343 |  10.80971 | 33.295028 | 35.605557 | 1.121276 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  104.447803 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.007794 | 0.0 | 0.028801 | 5.257351 | 0.257351 | 3.310695 | 2.310695 | 0.0 | 0.194581 | 2.568045 | 3.361072 | 3.555653 | 0.987682 |      0.0 |        0.0   74.605574 |       0.0  29.842229 | 180.972983 | 10.302803 |  32.93099 |  38.72167 | 0.987682 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   92.057332 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.072538 | 0.0 | 0.029575 | 5.109047 | 0.109047 | 3.270755 | 2.270755 | 0.0 | 0.209962 | 2.379802 | 2.824755 | 3.034717 | 0.842977 |      0.0 |        0.0   65.755237 |       0.0  26.302095 | 182.066106 | 10.157248 | 32.498525 | 41.782463 | 0.842977 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   79.666862 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.134002 | 0.0 | 0.029702 | 5.074348 | 0.074348 | 3.229934 | 2.229934 | 0.0 | 0.225062 | 2.304282 | 2.502119 | 2.727181 |  0.75755 |      0.0 |        0.0   56.904901 |       0.0   22.76196 | 183.097637 | 10.142554 | 32.113004 | 44.787336 |  0.75755 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   67.276391 |  0.0 |  1.573 | 0.049993 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.192342 | 0.0 | 0.012498 | 5.080435 | 0.080435 | 3.195278 | 2.195278 | 0.0 | 0.239913 | 2.275713 |  2.35545 | 2.595363 | 0.720934 | 1.523007 |        0.0   48.054565 |       0.0  19.221826 | 184.088032 | 10.174026 | 31.802884 | 47.742701 | 0.720934 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    54.88592 |  0.0 |    0.0 |  0.05838 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.248666 | 0.0 | 0.014595 | 5.100742 | 0.100742 | 3.168121 | 2.168121 | 0.0 | 0.254554 | 2.268863 | 2.295988 | 2.550542 | 0.708484 | 1.464627 |        0.0   39.204229 |       0.0  15.681691 | 185.020006 | 10.221207 | 31.567384 | 50.656268 | 0.708484 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   42.495449 |  0.0 |    0.0 | 0.059405 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.301945 | 0.0 | 0.014851 | 5.125455 | 0.125455 | 3.147934 | 2.147934 | 0.0 | 0.269021 | 2.273388 |  2.27541 | 2.544431 | 0.706787 | 1.405222 |        0.0   30.353892 |       0.0  12.141557 | 185.898446 | 10.272243 |  31.39697 |  53.53518 | 0.706787 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   30.104979 |  0.0 |    0.0 |  0.21716 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 |  5.35241 | 0.0 |  0.05429 | 5.150867 | 0.150867 | 3.133679 | 2.133679 | 0.0 | 0.283344 | 2.284546 | 2.273171 | 2.556516 | 0.710143 | 1.188062 |        0.0   21.503556 |       0.0   8.601422 | 186.686981 | 10.322918 | 31.280481 | 56.385515 | 0.710143 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   17.714508 |  0.0 |  0.847 |  0.89524 |  0.035062 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.230298 | 5.428463 | 0.0 |  0.22381 | 5.180873 | 0.180873 | 3.124526 | 2.124526 | 0.0 |  0.29755 | 2.305398 | 2.280981 | 2.578532 | 0.716259 |  1.10476 |        0.0    12.65322 |       0.0   5.061288 | 187.265006 | 10.389635 | 31.212302 |  59.21249 | 0.716259 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    5.324037 |  0.0 |  0.484 |  1.58876 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  8.850336 | 0.0   0.0 |  6.195235 | 5.431391 | 0.0 | 0.668518 | 5.202498 | 0.202498 | 3.119359 | 2.119359 | 0.0 | 0.311659 | 2.321857 |  2.29669 | 2.608349 | 0.724541 |      0.0 |        0.0    3.802884 |       0.0   1.521153 | 187.360333 |  10.41603 | 31.176083 |  62.02019 | 0.724541 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |  0.121 |    0.121 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0  3.802884 | 0.0   0.0 |  2.662019 |  2.33618 | 0.0 | 3.529947 |  4.59056 | 0.014729 | 3.087421 | 2.087421 | 0.0 | 0.325538 |  2.10215 | 2.297293 | 2.622831 | 0.728564 |      0.0 |        0.0         0.0 |       0.0        0.0 | 184.156223 |  8.146922 | 30.591801 | 64.782073 | 0.728564 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |  0.484 |    0.484 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.529885 | 3.205564 |      0.0 |  2.96961 |  1.96961 | 0.0 | 0.338758 |  1.96961 |  2.18513 | 2.523889 |  0.70108 |      0.0 |        0.0         0.0 |       0.0        0.0 | 179.626338 |  4.941358 | 28.858145 | 67.412924 |  0.70108 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 5.874002 | 1.944273 |      0.0 | 2.753418 | 1.753418 | 0.0 | 0.350832 | 1.753418 | 2.028915 | 2.379747 | 0.661041 |      0.0 |        0.0         0.0 |       0.0        0.0 | 173.752336 |  2.997085 | 26.295581 | 69.815511 | 0.661041 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 |   5.8739 | 1.179261 |      0.0 | 2.485338 | 1.485338 | 0.0 | 0.361504 | 1.485338 | 1.839891 | 2.201396 | 0.611499 |      0.0 |        0.0         0.0 |       0.0        0.0 | 167.878435 |  1.817824 | 23.504166 | 71.939344 | 0.611499 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 7.500667 | 0.715258 |      0.0 | 2.210611 | 1.210611 | 0.0 |  0.37075 | 1.210611 | 1.599826 | 1.970575 | 0.547382 |      0.0 |        0.0         0.0 |       0.0        0.0 | 160.377769 |  1.102566 | 20.798201 | 73.779206 | 0.547382 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 6.568376 | 0.433826 |      0.0 | 1.952179 | 0.952179 | 0.0 | 0.378657 | 0.952179 | 1.334157 | 1.712814 | 0.475782 |      0.0 |        0.0         0.0 |       0.0        0.0 | 153.809393 |   0.66874 |  18.32767 | 75.352728 | 0.475782 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 5.290409 | 0.263129 |      0.0 |  1.72027 |  0.72027 | 0.0 | 0.385365 |  0.72027 | 1.071766 | 1.457131 | 0.404759 |      0.0 |        0.0         0.0 |       0.0        0.0 | 148.518984 |  0.405611 | 16.150259 | 76.687633 | 0.404759 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 4.092469 | 0.159596 |      0.0 | 1.518073 | 0.518073 | 0.0 | 0.391029 | 0.518073 | 0.829392 |  1.22042 | 0.339006 |      0.0 |        0.0         0.0 |       0.0        0.0 | 144.426515 |  0.246016 | 14.273709 | 77.814677 | 0.339006 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 1.863004 |   0.0968 |      0.0 | 1.345052 | 0.345052 | 0.0 | 0.395799 | 0.345052 | 0.614217 | 1.010016 |  0.28056 |      0.0 |        0.0         0.0 |       0.0        0.0 | 142.563511 |  0.149216 | 12.680404 |  78.76393 |  0.28056 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.816588 | 0.058712 |      0.0 |  1.19886 |  0.19886 | 0.0 | 0.399814 |  0.19886 | 0.427894 | 0.827708 | 0.229919 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.746923 |  0.090504 | 11.341397 | 79.562976 | 0.229919 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.204368 | 0.035611 |      0.0 | 1.076418 | 0.076418 | 0.0 | 0.403197 | 0.076418 | 0.269188 | 0.672385 | 0.186774 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.542555 |  0.054894 | 10.224172 | 80.236197 | 0.186774 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.055428 | 0.021599 |      0.0 | 0.973783 | 0.002492 | 0.0 |  0.40605 | 0.002492 | 0.137514 | 0.543563 |  0.15099 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.487127 |  0.033295 |  9.269496 |  80.80393 |  0.15099 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.053093 |   0.0131 |      0.0 | 0.882743 |      0.0 | 0.0 | 0.408433 |      0.0 | 0.045671 | 0.454105 |  0.12614 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.434035 |  0.020194 |  8.399853 |  81.27824 |  0.12614 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.052286 | 0.007946 |      0.0 | 0.799736 |      0.0 | 0.0 |  0.41039 |      0.0 | 0.008515 | 0.418905 | 0.116362 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.381748 |  0.012248 |  7.608063 | 81.667586 | 0.116362 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 |  0.0 |    0.0 |      0.0 |       0.0 | 0.0 | 0.0 | 0.0 | 0.0 |    0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |      0.0 | 0.0 | 0.052492 | 0.004819 |      0.0 | 0.724236 |      0.0 | 0.0 | 0.411959 |      0.0 | 0.001038 | 0.412998 | 0.114722 |      0.0 |        0.0         0.0 |       0.0        0.0 | 141.329256 |  0.007429 |  6.888646 | 81.979863 | 0.114722 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96c_glacier:

glacier
_______

Regarding glacier simulation, all functionalities of |hland_96c| are identical with the
ones of |hland_96|.  Hence, all snow data in the following table agrees with the
:ref:`hland_96_glacier` example of |hland_96|:

.. integration-test::

    >>> zonetype(GLACIER)
    >>> test.inits.sc = 0.05
    >>> test.reset_inits()
    >>> conditions = model.conditions
    >>> test("hland_96c_glacier")
    |        date |    p |     t |    tc | fracrain | rfc | sfc |    cfact |                    swe |     gact |     pc |  ei |     tf | spl | wcl | spg | wcg |    glmelt |               melt |      refr |       in_ |         r |  sr |  ea |      qvs1 |     qab1 |     qvs2 |     qab2 |  el |       q1 |      inrc |     outrc |        rt |       qt |  ic |                     sp |                   wc |  sm |       bw1 |       bw2 |         lz |   outlet |
    ----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 |     0.05 |       0.0 |  0.160107 |  0.210107 | 0.058363 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |       9.95 | 0.058363 |
    | 01/01 01:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 |  0.04975 |       0.0 |  0.073779 |  0.123529 | 0.034314 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |    9.90025 | 0.034314 |
    | 01/01 02:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.049501 |       0.0 |  0.014282 |  0.063783 | 0.017717 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.850749 | 0.017717 |
    | 01/01 03:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.049254 |       0.0 |  0.001674 |  0.050927 | 0.014147 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.801495 | 0.014147 |
    | 01/01 04:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.049007 |       0.0 |  0.000147 |  0.049155 | 0.013654 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.752488 | 0.013654 |
    | 01/01 05:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048762 |       0.0 |  0.000011 |  0.048773 | 0.013548 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.703725 | 0.013548 |
    | 01/01 06:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048519 |       0.0 |  0.000001 |  0.048519 | 0.013478 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.655206 | 0.013478 |
    | 01/01 07:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048276 |       0.0 |       0.0 |  0.048276 |  0.01341 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |    9.60693 |  0.01341 |
    | 01/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.048035 |       0.0 |       0.0 |  0.048035 | 0.013343 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.558896 | 0.013343 |
    | 01/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047794 |       0.0 |       0.0 |  0.047794 | 0.013276 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.511101 | 0.013276 |
    | 01/01 10:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047556 |       0.0 |       0.0 |  0.047556 |  0.01321 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.463546 |  0.01321 |
    | 01/01 11:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047318 |       0.0 |       0.0 |  0.047318 | 0.013144 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.416228 | 0.013144 |
    | 01/01 12:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.047081 |       0.0 |       0.0 |  0.047081 | 0.013078 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.369147 | 0.013078 |
    | 01/01 13:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046846 |       0.0 |       0.0 |  0.046846 | 0.013013 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.322301 | 0.013013 |
    | 01/01 14:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |        0.0         0.0 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046612 |       0.0 |       0.0 |  0.046612 | 0.012948 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |    9.27569 | 0.012948 |
    | 01/01 15:00 |  0.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     0.2288      0.3432 | 0.901953 |  0.286 | 0.0 |  0.286 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046378 |       0.0 |       0.0 |  0.046378 | 0.012883 | 0.0 |     0.2288      0.3432 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.229311 | 0.012883 |
    | 01/01 16:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     0.2288      0.3432 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.046147 |       0.0 |       0.0 |  0.046147 | 0.012818 | 0.0 |     0.2288      0.3432 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.183165 | 0.012818 |
    | 01/01 17:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     0.2288      0.3432 | 0.901953 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045916 |       0.0 |       0.0 |  0.045916 | 0.012754 | 0.0 |     0.2288      0.3432 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.137249 | 0.012754 |
    | 01/01 18:00 |  1.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |      1.716       2.574 | 0.901953 |  1.859 | 0.0 |  1.859 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045686 |       0.0 |       0.0 |  0.045686 | 0.012691 | 0.0 |      1.716       2.574 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.091563 | 0.012691 |
    | 01/01 19:00 |  5.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     8.1224     12.1836 | 0.901953 |  8.008 | 0.0 |  8.008 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045458 |       0.0 |       0.0 |  0.045458 | 0.012627 | 0.0 |     8.1224     12.1836 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.046105 | 0.012627 |
    | 01/01 20:00 |  2.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |      11.44       17.16 | 0.901953 |  4.147 | 0.0 |  4.147 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045231 |       0.0 |       0.0 |  0.045231 | 0.012564 | 0.0 |      11.44       17.16 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   9.000874 | 0.012564 |
    | 01/01 21:00 |  4.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    17.0456     25.5684 | 0.901953 |  7.007 | 0.0 |  7.007 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.045004 |       0.0 |       0.0 |  0.045004 | 0.012501 | 0.0 |    17.0456     25.5684 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |    8.95587 | 0.012501 |
    | 01/01 22:00 | 10.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |     29.172      43.758 | 0.901953 | 15.158 | 0.0 | 15.158 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.044779 |       0.0 |       0.0 |  0.044779 | 0.012439 | 0.0 |     29.172      43.758 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.911091 | 0.012439 |
    | 01/01 23:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.450977 |    29.2864     43.9296 | 0.901953 |  0.143 | 0.0 |  0.143 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.044555 |       0.0 |       0.0 |  0.044555 | 0.012377 | 0.0 |    29.2864     43.9296 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.866535 | 0.012377 |
    | 02/01 00:00 |  0.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    30.0872     45.1308 | 0.902305 |  1.001 | 0.0 |  1.001 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.044333 |       0.0 |       0.0 |  0.044333 | 0.012315 | 0.0 |    30.0872     45.1308 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.822202 | 0.012315 |
    | 02/01 01:00 |  3.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    33.5192     50.2788 | 0.902305 |   4.29 | 0.0 |   4.29 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.044111 |       0.0 |       0.0 |  0.044111 | 0.012253 | 0.0 |    33.5192     50.2788 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.778091 | 0.012253 |
    | 02/01 02:00 |  2.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    35.9216     53.8824 | 0.902305 |  3.003 | 0.0 |  3.003 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 |  0.04389 |       0.0 |       0.0 |   0.04389 | 0.012192 | 0.0 |    35.9216     53.8824 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.734201 | 0.012192 |
    | 02/01 03:00 | 10.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    47.8192     71.7288 | 0.902305 | 14.872 | 0.0 | 14.872 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.043671 |       0.0 |       0.0 |  0.043671 | 0.012131 | 0.0 |    47.8192     71.7288 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |    8.69053 | 0.012131 |
    | 02/01 04:00 |  3.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    51.8232     77.7348 | 0.902305 |  5.005 | 0.0 |  5.005 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.043453 |       0.0 |       0.0 |  0.043453 |  0.01207 | 0.0 |    51.8232     77.7348 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.647077 |  0.01207 |
    | 02/01 05:00 |  3.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    55.7128     83.5692 | 0.902305 |  4.862 | 0.0 |  4.862 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.043235 |       0.0 |       0.0 |  0.043235 |  0.01201 | 0.0 |    55.7128     83.5692 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.603842 |  0.01201 |
    | 02/01 06:00 |  1.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    57.0856     85.6284 | 0.902305 |  1.716 | 0.0 |  1.716 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.043019 |       0.0 |       0.0 |  0.043019 |  0.01195 | 0.0 |    57.0856     85.6284 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.560823 |  0.01195 |
    | 02/01 07:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |       57.2        85.8 | 0.902305 |  0.143 | 0.0 |  0.143 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.042804 |       0.0 |       0.0 |  0.042804 |  0.01189 | 0.0 |       57.2        85.8 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.518019 |  0.01189 |
    | 02/01 08:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |       57.2        85.8 | 0.902305 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 |  0.04259 |       0.0 |       0.0 |   0.04259 | 0.011831 | 0.0 |       57.2        85.8 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.475429 | 0.011831 |
    | 02/01 09:00 |  0.0 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |       57.2        85.8 | 0.902305 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.042377 |       0.0 |       0.0 |  0.042377 | 0.011771 | 0.0 |       57.2        85.8 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.433051 | 0.011771 |
    | 02/01 10:00 |  0.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    57.6576     86.4864 | 0.902305 |  0.572 | 0.0 |  0.572 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.042165 |       0.0 |       0.0 |  0.042165 | 0.011713 | 0.0 |    57.6576     86.4864 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.390886 | 0.011713 |
    | 02/01 11:00 |  0.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |     57.772      86.658 | 0.902305 |  0.143 | 0.0 |  0.143 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.041954 |       0.0 |       0.0 |  0.041954 | 0.011654 | 0.0 |     57.772      86.658 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.348932 | 0.011654 |
    | 02/01 12:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    61.8904     92.8356 | 0.902305 |  5.148 | 0.0 |  5.148 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.041745 |       0.0 |       0.0 |  0.041745 | 0.011596 | 0.0 |    61.8904     92.8356 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.307187 | 0.011596 |
    | 02/01 13:00 |  5.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |      68.64      102.96 | 0.902305 |  8.437 | 0.0 |  8.437 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.041536 |       0.0 |       0.0 |  0.041536 | 0.011538 | 0.0 |      68.64      102.96 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.265651 | 0.011538 |
    | 02/01 14:00 |  1.1 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    69.8984    104.8476 | 0.902305 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.041328 |       0.0 |       0.0 |  0.041328 |  0.01148 | 0.0 |    69.8984    104.8476 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.224323 |  0.01148 |
    | 02/01 15:00 | 20.7 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    93.5792    140.3688 | 0.902305 | 29.601 | 0.0 | 29.601 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.041122 |       0.0 |       0.0 |  0.041122 | 0.011423 | 0.0 |    93.5792    140.3688 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.183201 | 0.011423 |
    | 02/01 16:00 | 37.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   136.9368    205.4052 | 0.902305 | 54.197 | 0.0 | 54.197 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.040916 |       0.0 |       0.0 |  0.040916 | 0.011366 | 0.0 |   136.9368    205.4052 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.142285 | 0.011366 |
    | 02/01 17:00 |  8.2 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   146.3176    219.4764 | 0.902305 | 11.726 | 0.0 | 11.726 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.040711 |       0.0 |       0.0 |  0.040711 | 0.011309 | 0.0 |   146.3176    219.4764 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.101574 | 0.011309 |
    | 02/01 18:00 |  3.6 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    150.436     225.654 | 0.902305 |  5.148 | 0.0 |  5.148 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.040508 |       0.0 |       0.0 |  0.040508 | 0.011252 | 0.0 |    150.436     225.654 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.061066 | 0.011252 |
    | 02/01 19:00 |  7.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |    159.016     238.524 | 0.902305 | 10.725 | 0.0 | 10.725 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.040305 |       0.0 |       0.0 |  0.040305 | 0.011196 | 0.0 |    159.016     238.524 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   8.020761 | 0.011196 |
    | 02/01 20:00 | 18.5 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |     180.18      270.27 | 0.902305 | 26.455 | 0.0 | 26.455 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.040104 |       0.0 |       0.0 |  0.040104 |  0.01114 | 0.0 |     180.18      270.27 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   7.980657 |  0.01114 |
    | 02/01 21:00 | 15.4 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   197.7976    296.6964 | 0.902305 | 22.022 | 0.0 | 22.022 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.039903 |       0.0 |       0.0 |  0.039903 | 0.011084 | 0.0 |   197.7976    296.6964 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   7.940753 | 0.011084 |
    | 02/01 22:00 |  6.3 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   205.0048    307.5072 | 0.902305 |  9.009 | 0.0 |  9.009 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.039704 |       0.0 |       0.0 |  0.039704 | 0.011029 | 0.0 |   205.0048    307.5072 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |    7.90105 | 0.011029 |
    | 02/01 23:00 |  1.9 | -20.0 | -19.4 |      0.0 | 0.0 | 1.3 | 0.451153 |   207.1784    310.7676 | 0.902305 |  2.717 | 0.0 |  2.717 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 |      0.0       0.0 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.039505 |       0.0 |       0.0 |  0.039505 | 0.010974 | 0.0 |   207.1784    310.7676 |       0.0        0.0 | 0.0 |       0.0 |       0.0 |   7.861544 | 0.010974 |
    | 03/01 00:00 |  4.9 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   211.9216    317.8824 | 0.902686 |  5.929 | 0.0 |  5.929 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.039308 |       0.0 |       0.0 |  0.039308 | 0.010919 | 0.0 | 198.332077  301.921277 | 13.589523  15.961123 | 0.0 |       0.0 |       0.0 |   7.822237 | 0.010919 |
    | 03/01 01:00 |  2.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   214.5352    321.8028 | 0.902686 |  3.267 | 0.0 |  3.267 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.039111 |       0.0 |       0.0 |  0.039111 | 0.010864 | 0.0 | 189.485754  293.074954 | 25.049446  28.727846 | 0.0 |       0.0 |       0.0 |   7.783126 | 0.010864 |
    | 03/01 02:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   215.0192    322.5288 | 0.902686 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.038916 |       0.0 |       0.0 |  0.038916 |  0.01081 | 0.0 | 180.639432  284.228632 | 34.379768  38.300168 | 0.0 |       0.0 |       0.0 |    7.74421 |  0.01081 |
    | 03/01 03:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   215.2128    322.8192 | 0.902686 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.038721 |       0.0 |       0.0 |  0.038721 | 0.010756 | 0.0 | 171.793109  275.382309 | 43.419691  47.436891 | 0.0 |       0.0 |       0.0 |   7.705489 | 0.010756 |
    | 03/01 04:00 |  0.5 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   215.6968    323.5452 | 0.902686 |  0.605 | 0.0 |  0.605 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |       0.0 |       0.0 | 0.0 | 0.0 |       0.0 |      0.0 |      0.0 |      0.0 | 0.0 | 0.038527 |       0.0 |       0.0 |  0.038527 | 0.010702 | 0.0 | 162.946786  266.535986 | 52.750014  57.009214 | 0.0 |       0.0 |       0.0 |   7.666961 | 0.010702 |
    | 03/01 05:00 |  2.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 215.740648      327.03 | 0.902686 |  2.904 | 0.0 |  2.904 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  1.139676 |  1.139676 | 0.0 | 0.0 |  0.242821 |      0.0 | 0.011746 |      0.0 | 0.0 | 0.038394 |       0.0 |       0.0 |  0.038394 | 0.010665 | 0.0 | 154.100463  257.689663 | 61.640185  69.340337 | 0.0 |  0.896855 |  0.231075 |   7.640314 | 0.010665 |
    | 03/01 06:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 203.355796    327.6108 | 0.902686 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.386026 |  6.386026 | 0.0 | 0.0 |    1.7135 |      0.0 | 0.104879 |      0.0 | 0.0 | 0.038726 |       0.0 |       0.0 |  0.038726 | 0.010757 | 0.0 |  145.25414   248.84334 | 58.101656   78.76746 | 0.0 |  5.569381 |  1.839696 |   7.706467 | 0.010757 |
    | 03/01 07:00 |  0.2 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 190.970944    327.9012 | 0.902686 |  0.242 | 0.0 |  0.242 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |  6.289226 |  6.289226 | 0.0 | 0.0 |  3.531371 |      0.0 | 0.345897 |      0.0 | 0.0 | 0.040262 |       0.0 |       0.0 |  0.040262 | 0.011184 | 0.0 | 136.407817  239.997017 | 54.563127  87.904183 | 0.0 |  8.327235 |   5.02517 |   8.012103 | 0.011184 |
    | 03/01 08:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 178.586092  323.610972 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 |   8.33754 |   8.33754 | 0.0 | 0.0 |   5.03373 | 0.213799 | 0.721711 |      0.0 | 0.0 | 0.043669 |  0.213799 |  0.015261 |   0.05893 | 0.016369 | 0.0 | 127.561495  231.150695 | 51.024598  92.460278 | 0.0 | 11.417246 |   9.33719 |   8.690144 | 0.016369 |
    | 03/01 09:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  166.20124   311.22612 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |  6.806303 | 1.806303 | 1.211489 | 0.215243 | 0.0 | 0.049508 |  2.021546 |  0.251347 |  0.300855 | 0.083571 | 0.0 | 118.715172  222.304372 | 47.486069  88.921749 | 0.0 | 15.189492 |  14.71676 |   9.852125 | 0.083571 |
    | 03/01 10:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 153.816388  298.841268 | 0.902686 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.747852 | 12.747852 | 0.0 | 0.0 |   8.06533 |  3.06533 | 1.758349 | 0.758349 | 0.0 | 0.058052 |  3.823679 |  1.358903 |  1.416955 | 0.393599 | 0.0 | 109.868849  213.458049 |  43.94754   85.38322 | 0.0 | 16.806684 | 20.265392 |  11.552422 | 0.393599 |
    | 03/01 11:00 |  2.6 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 141.431536  286.456416 | 0.902686 |  3.146 | 0.0 |  3.146 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 15.530852 | 15.530852 | 0.0 | 0.0 |  9.088364 | 4.088364 | 2.309153 | 1.309153 | 0.0 | 0.069308 |  5.397517 |  3.013036 |  3.082343 | 0.856207 | 0.0 | 101.022526  204.611726 |  40.40901   81.84469 | 0.0 | 19.160807 | 25.735451 |  13.792267 | 0.856207 |
    | 03/01 12:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 129.046684  274.071564 | 0.902686 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 13.231852 | 13.231852 | 0.0 | 0.0 |  9.409532 | 4.409532 | 2.819969 | 1.819969 | 0.0 | 0.083061 |  6.229501 |  4.616421 |  4.699482 | 1.305412 | 0.0 |  92.176203  195.765403 | 36.870481  78.306161 | 0.0 | 18.573595 | 30.505045 |  16.529174 | 1.305412 |
    | 03/01 13:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 116.661833  261.686713 | 0.902686 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.747852 | 12.747852 | 0.0 | 0.0 |  9.134911 | 4.134911 |   3.2394 |   2.2394 | 0.0 | 0.098843 |   6.37431 |    5.7344 |  5.833242 | 1.620345 | 0.0 |   83.32988   186.91908 | 33.331952  74.767632 | 0.0 | 18.051626 | 34.161156 |  19.669731 | 1.620345 |
    | 03/01 14:00 |  0.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 | 104.276981  249.301861 | 0.902686 |  0.363 | 0.0 |  0.363 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.747852 | 12.747852 | 0.0 | 0.0 |  8.969937 | 3.969937 | 3.563045 | 2.563045 | 0.0 | 0.116164 |  6.532982 |  6.237824 |  6.353988 | 1.764997 | 0.0 |  74.483558  178.072758 | 29.793423  71.229103 | 0.0 | 17.859604 | 37.005004 |  23.116612 | 1.764997 |
    | 03/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  91.892129  236.917009 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |  8.842476 | 3.842476 | 3.814827 | 2.814827 | 0.0 | 0.134657 |  6.657303 |  6.452423 |   6.58708 | 1.829744 | 0.0 |  65.637235  169.226435 | 26.254894  67.690574 | 0.0 | 17.559504 | 39.217826 |  26.796782 | 1.829744 |
    | 03/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  79.507277  224.532157 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |  8.747626 | 3.747626 | 4.010944 | 3.010944 | 0.0 | 0.154039 |   6.75857 |  6.595967 |  6.750006 | 1.875002 | 0.0 |  56.790912  160.380112 | 22.716365  64.152045 | 0.0 | 17.449103 | 40.943565 |  30.653687 | 1.875002 |
    | 03/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  67.122425  212.147305 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |  8.712733 | 3.712733 | 4.165722 | 3.165722 | 0.0 | 0.174097 |  6.878455 |  6.711773 |   6.88587 | 1.912742 | 0.0 |  47.944589  151.533789 | 19.177836  60.613516 | 0.0 | 17.408488 | 42.324855 |  34.645311 | 1.912742 |
    | 03/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  54.737573  199.762453 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |  8.699897 | 3.699897 | 4.290313 | 3.290313 | 0.0 | 0.194678 |   6.99021 |  6.825388 |  7.020066 | 1.950018 | 0.0 |  39.098266  142.687466 | 15.639306  57.074986 | 0.0 | 17.393547 | 43.444125 |  38.740947 | 1.950018 |
    | 03/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  42.352721  187.377601 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |  8.695174 | 3.695174 | 4.391537 | 3.391537 | 0.0 | 0.215662 |  7.086711 |  6.938246 |  7.153908 | 1.987197 | 0.0 |  30.251943  133.841143 | 12.100777  53.536457 | 0.0 | 17.388051 | 44.356226 |  42.916821 | 1.987197 |
    | 03/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  29.967869  174.992749 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |  8.693437 | 3.693437 | 4.474123 | 3.474123 | 0.0 | 0.236955 |   7.16756 |  7.040672 |  7.277626 | 2.021563 | 0.0 |  21.405621  124.994821 |  8.562248  49.997928 | 0.0 | 17.386029 | 45.101416 |  47.153989 | 2.021563 |
    | 03/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |  17.583017  162.607897 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |  8.692798 | 3.692798 | 4.541633 | 3.541633 | 0.0 | 0.258478 |  7.234431 |  7.128549 |  7.387027 | 2.051952 | 0.0 |  12.559298  116.148498 |  5.023719  46.459399 | 0.0 | 17.385285 | 45.710947 |  51.437145 | 2.051952 |
    | 03/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |   5.198165  150.223045 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |       0.0 | 8.846323  8.846323 | 0.0   0.0 | 12.384852 | 12.384852 | 0.0 | 0.0 |  8.692563 | 3.692563 | 4.596867 | 3.596867 | 0.0 |  0.28017 |   7.28943 |  7.202005 |  7.482175 | 2.078382 | 0.0 |   3.712975  107.302175 |   1.48519   42.92087 | 0.0 | 17.385011 | 46.209776 |  55.753842 | 2.078382 |
    | 03/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451343 |        0.0  137.838193 | 0.902686 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.846323 | 3.712975  8.846323 | 0.0   0.0 | 17.637831 | 17.637831 | 0.0 | 0.0 |  9.658708 | 4.658708 |  4.68732 |  3.68732 | 0.0 | 0.302206 |  8.346028 |  7.334901 |  7.637107 | 2.121419 | 0.0 |        0.0   98.455852 |       0.0  39.382341 | 0.0 | 20.705427 | 47.493845 |  60.138955 | 2.121419 |
    | 04/01 00:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  125.447722 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 |  10.23134 |  5.23134 | 4.830515 | 3.830515 | 0.0 | 0.324847 |  9.061855 |  7.939704 |  8.264552 | 2.295709 | 0.0 |        0.0   89.605516 |       0.0  35.842206 | 0.0 | 20.288319 | 49.064154 |  64.644623 | 2.295709 |
    | 04/01 01:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  113.057251 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 10.099509 | 5.099509 | 4.966666 | 3.966666 | 0.0 | 0.348056 |  9.066175 |   8.66773 |  9.015786 | 2.504385 | 0.0 |        0.0    80.75518 |       0.0  32.302072 | 0.0 | 20.134873 | 50.230329 |  69.263234 | 2.504385 |
    | 04/01 02:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0  100.666781 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 10.051011 | 5.051011 | 5.070091 | 4.070091 | 0.0 | 0.371667 |  9.121102 |  8.998644 |  9.370311 | 2.602864 | 0.0 |        0.0   71.904843 |       0.0  28.761937 | 0.0 | 20.078424 | 51.141157 |  73.961658 | 2.602864 |
    | 04/01 03:00 |  1.3 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    88.27631 | 0.903096 |  1.573 | 0.0 |  1.573 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 16.618572 | 16.618572 | 0.0 | 0.0 | 10.322506 | 5.322506 | 5.165357 | 4.165357 | 0.0 | 0.395635 |  9.487863 |  9.115565 |    9.5112 |    2.642 | 0.0 |        0.0   63.054507 |       0.0  25.221803 | 0.0 | 21.051983 | 52.132949 |   78.73138 |    2.642 |
    | 04/01 04:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   75.885839 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 10.340872 | 5.340872 | 5.256108 | 4.256108 | 0.0 | 0.419937 |   9.59698 |  9.333439 |  9.753377 | 2.709271 | 0.0 |        0.0   54.204171 |       0.0  21.681668 | 0.0 |  20.41581 | 52.961606 |  83.567551 | 2.709271 |
    | 04/01 05:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   63.495368 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 10.139803 | 5.139803 | 5.321797 | 4.321797 | 0.0 | 0.444447 |  9.461601 |  9.509658 |  9.954105 | 2.765029 | 0.0 |        0.0   45.353835 |       0.0  18.141534 | 0.0 | 20.181775 | 53.457814 |  88.444902 | 2.765029 |
    | 04/01 06:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   51.104898 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.045572 | 15.045572 | 0.0 | 0.0 | 10.065834 | 5.065834 | 5.363307 | 4.363307 | 0.0 | 0.469041 |  9.429142 |  9.504566 |  9.973607 | 2.770446 | 0.0 |        0.0   36.503498 |       0.0  14.601399 | 0.0 | 20.095678 | 53.797034 |  93.339168 | 2.770446 |
    | 04/01 07:00 |  0.7 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   38.714427 | 0.903096 |  0.847 | 0.0 |  0.847 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.892572 | 15.892572 | 0.0 | 0.0 | 10.194419 | 5.194419 | 5.400074 | 4.400074 | 0.0 | 0.493696 |  9.594493 |  9.464672 |  9.958369 | 2.766213 | 0.0 |        0.0   27.653162 |       0.0  11.061265 | 0.0 | 20.599411 | 54.191306 |  98.245545 | 2.766213 |
    | 04/01 08:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   26.323956 | 0.903096 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.529572 | 15.529572 | 0.0 | 0.0 | 10.286859 | 5.286859 | 5.440137 | 4.440137 | 0.0 | 0.518428 |  9.726996 |  9.537081 | 10.055509 | 2.793197 | 0.0 |        0.0   18.802826 |       0.0    7.52113 | 0.0 | 20.555264 | 54.597891 | 103.167254 | 2.793197 |
    | 04/01 09:00 |  0.1 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0   13.933485 | 0.903096 |  0.121 | 0.0 |  0.121 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.166572 | 15.166572 | 0.0 | 0.0 | 10.206136 | 5.206136 | 5.473208 | 4.473208 | 0.0 | 0.543202 |  9.679344 |   9.65362 | 10.196822 | 2.832451 | 0.0 |        0.0     9.95249 |       0.0   3.980996 | 0.0 | 20.309564 | 54.857611 | 108.097259 | 2.832451 |
    | 04/01 10:00 |  0.4 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0    1.543015 | 0.903096 |  0.484 | 0.0 |  0.484 | 0.0 | 0.0 | 0.0 | 0.0 |  8.850336 |      0.0  8.850336 | 0.0   0.0 | 15.529572 | 15.529572 | 0.0 | 0.0 |  10.19525 |  5.19525 | 5.496238 | 4.496238 | 0.0 | 0.567967 |  9.691488 |  9.688057 | 10.256024 | 2.848896 | 0.0 |        0.0    1.102153 |       0.0   0.440861 | 0.0 | 20.448635 | 55.060386 | 113.025529 | 2.848896 |
    | 04/01 11:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0  1.102153 | 0.0   0.0 |  18.47218 |  18.47218 | 0.0 | 0.0 | 10.780468 | 5.780468 |  5.54202 |  4.54202 | 0.0 | 0.592838 | 10.322487 |  9.733854 | 10.326692 | 2.868526 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |  22.35988 | 55.756814 | 117.974712 | 2.868526 |
    | 04/01 12:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.242625 | 6.242625 | 5.626782 | 4.626782 | 0.0 | 0.618007 | 10.869407 | 10.090843 |  10.70885 | 2.974681 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 22.575302 | 56.745876 | 122.983486 | 2.974681 |
    | 04/01 13:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.310712 | 6.310712 | 5.719613 | 4.719613 | 0.0 | 0.643515 | 11.030325 | 10.594502 | 11.238018 | 3.121672 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 22.654551 |  57.61736 | 128.059584 | 3.121672 |
    | 04/01 14:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.335759 | 6.335759 | 5.799773 | 4.799773 | 0.0 | 0.669297 | 11.135532 | 10.917538 | 11.586835 | 3.218565 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 22.683705 | 58.353574 |  133.19006 | 3.218565 |
    | 04/01 15:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.344974 | 6.344974 | 5.866931 | 4.866931 | 0.0 | 0.695285 | 11.211904 | 11.076786 | 11.772071 |  3.27002 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 22.694431 | 58.964686 | 138.361706 |  3.27002 |
    | 04/01 16:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.348363 | 6.348363 | 5.922477 | 4.922477 | 0.0 | 0.721421 | 11.270841 | 11.172647 | 11.894068 | 3.303908 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 22.698376 | 59.468094 | 143.562762 | 3.303908 |
    | 04/01 17:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.349611 | 6.349611 | 5.968162 | 4.968162 | 0.0 | 0.747655 | 11.317773 | 11.241449 | 11.989104 | 3.330307 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 22.699828 | 59.881381 |  148.78327 | 3.330307 |
    | 04/01 18:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.350069 | 6.350069 | 6.005642 | 5.005642 | 0.0 | 0.773945 | 11.355711 | 11.294672 | 12.068617 | 3.352394 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 22.700362 | 60.220167 | 154.014967 | 3.352394 |
    | 04/01 19:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.350238 | 6.350238 | 6.036355 | 5.036355 | 0.0 | 0.800257 | 11.386593 | 11.337149 | 12.137405 | 3.371501 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 22.700558 | 60.497694 | 159.251066 | 3.371501 |
    | 04/01 20:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 |   11.3503 |   6.3503 | 6.061512 | 5.061512 | 0.0 | 0.826563 | 11.411812 | 11.371525 | 12.198087 | 3.388358 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |  22.70063 | 60.724971 | 164.486014 | 3.388358 |
    | 04/01 21:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.350323 | 6.350323 | 6.082112 | 5.082112 | 0.0 | 0.852841 | 11.432435 | 11.399522 | 12.252363 | 3.403434 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 22.700657 |  60.91107 | 169.715286 | 3.403434 |
    | 04/01 22:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.350331 | 6.350331 | 6.098979 | 5.098979 | 0.0 | 0.879071 | 11.449311 |  11.42239 | 12.301462 | 3.417073 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 | 22.700667 | 61.063442 | 174.935194 | 3.417073 |
    | 04/01 23:00 |  0.0 |  20.0 |  20.6 |      1.0 | 1.1 | 0.0 | 0.451548 |        0.0         0.0 | 0.903096 |    0.0 | 0.0 |    0.0 | 0.0 | 0.0 | 0.0 | 0.0 | 17.700673 |      0.0       0.0 | 0.0   0.0 | 17.700673 | 17.700673 | 0.0 | 0.0 | 11.350334 | 6.350334 |  6.11279 |  5.11279 | 0.0 |  0.90524 | 11.463124 | 11.441093 | 12.346333 | 3.429537 | 0.0 |        0.0         0.0 |       0.0        0.0 | 0.0 |  22.70067 | 61.188197 | 180.142744 | 3.429537 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96c_multiple_zones:

multiple zones
______________

This example focuses on the spatial configuration of |hland_96c|.  The most marked
difference to |hland_96| is the separate modelling of the faster runoff components
(|QAb1|, |QAb2|) for each zone.  Hence, there are five distinct values for each
runoff component in the following table.  In contrast, the table of the corresponding
:ref:`hland_96_multiple_zones` example of |hland_96| contains only a single value for
the only fast runoff component |hland_fluxes.Q0| of |hland_96|:

.. integration-test::

    >>> name2value = {par.name: par.value for par in control}
    >>> nmbzones(5)
    >>> sclass(1)
    >>> area(15.0)
    >>> zonearea(5.0, 4.0, 3.0, 2.0, 1.0)
    >>> zonetype(FIELD, FOREST, GLACIER, ILAKE, SEALED)
    >>> sfdist(1.0)
    >>> for name, value in name2value.items():
    ...     if name not in ("nmbzones", "sclass", "area", "zonearea", "zonetype", "sfdist"):
    ...         control[name].value = value
    >>> model.add_aetmodel_v1.update(model, model.aetmodel, refresh=True)
    >>> model.aetmodel.add_petmodel_v1.update(model.aetmodel, model.aetmodel.petmodel, refresh=True)
    >>> aetcontrol = model.aetmodel.parameters.control
    >>> aetcontrol.temperaturethresholdice(0.0)
    >>> aetcontrol.soilmoisturelimit(0.8)
    >>> aetcontrol.excessreduction(0.5)
    >>> petcontrol = model.aetmodel.petmodel.parameters.control
    >>> petcontrol.evapotranspirationfactor(0.7)
    >>> petcontrol.airtemperaturefactor(0.1)
    >>> petcontrol.altitudefactor(-0.1)
    >>> petcontrol.precipitationfactor(0.1)
    >>> inputs.t.series = t_series
    >>> petinputs.normalairtemperature.series = tn_series
    >>> test("hland_96c_multiple_zones",
    ...      axis1=(inputs.p, fluxes.rt, fluxes.qt),
    ...      axis2=inputs.t)
    |        date |    p |    t |                           tc |                     fracrain |                     rfc |                     sfc |                                       cfact |                     swe |                          gact |                                     pc |                                     ei |                                              tf |                     spl |                     wcl |                     spg |                     wcg |                           glmelt |                     melt |                     refr |                                                in_ |                                                  r |                            sr |                                ea |                                     qvs1 |                                     qab1 |                                    qvs2 |                                    qab2 |                           el |       q1 |      inrc |    outrc |       rt |        qt |                                     ic |                      sp |                      wc |                                    sm |                                       bw1 |                                       bw2 |         lz |    outlet |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | 21.2 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.760624     0.0        0.0 |       0.0        0.0  18.760624     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.048465  0.048465  0.0  0.0  0.0 |      0.0       0.0   3.969792  0.0   0.0 |      0.0       0.0   0.439608  0.0   0.0 |      0.0       0.0  0.192035  0.0   0.0 |      0.0       0.0       0.0  0.0   0.0 | 0.0  0.0  0.0  0.077544  0.0 |  0.05015 |  0.101448 | 0.167348 | 0.191842 |  0.799342 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.951535   99.951535  0.0  0.0  0.0 |       0.0        0.0  14.351224  0.0  0.0 |       0.0        0.0   3.777757  0.0  0.0 |   9.979922 |  0.799342 |
    | 01/01 01:00 |  0.0 | 19.4 | 20.0  20.0  20.0  20.0  20.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.137109  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.137109     0.0        0.0 |       0.0        0.0  17.137109     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.047044  0.047044  0.0  0.0  0.0 |      0.0       0.0   8.607745  0.0   0.0 |      0.0       0.0   3.607745  0.0   0.0 |      0.0       0.0  0.775772  0.0   0.0 |      0.0       0.0   0.01715  0.0   0.0 | 0.0  0.0  0.0  0.075307  0.0 | 0.050677 |  0.836514 | 0.184285 | 0.207012 |  0.862549 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.904491   99.904491  0.0  0.0  0.0 |       0.0        0.0  19.272842  0.0  0.0 |       0.0        0.0  11.592581  0.0  0.0 |  10.084724 |  0.862549 |
    | 01/01 02:00 |  0.0 | 18.9 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132     0.0        0.0 |       0.0        0.0  16.686132     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.046628  0.046628  0.0  0.0  0.0 |      0.0       0.0  10.080321  0.0   0.0 |      0.0       0.0   5.080321  0.0   0.0 |      0.0       0.0  1.569546  0.0   0.0 |      0.0       0.0  0.569546  0.0   0.0 | 0.0  0.0  0.0  0.074675  0.0 | 0.052052 |  1.303815 | 0.561199 | 0.534955 |  2.228978 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.857863   99.857863  0.0  0.0  0.0 |       0.0        0.0  20.798332  0.0  0.0 |       0.0        0.0  19.533809  0.0  0.0 |  10.358335 |  2.228978 |
    | 01/01 03:00 |  0.0 | 18.3 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   16.14496  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   16.14496     0.0        0.0 |       0.0        0.0   16.14496     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.04613   0.04613  0.0  0.0  0.0 |      0.0       0.0  10.462925  0.0   0.0 |      0.0       0.0   5.462925  0.0   0.0 |      0.0       0.0  2.307212  0.0   0.0 |      0.0       0.0  1.307212  0.0   0.0 | 0.0  0.0  0.0  0.073912  0.0 | 0.054211 |  1.562339 | 1.062103 | 0.971086 |  4.046191 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.811734   99.811734  0.0  0.0  0.0 |       0.0        0.0  21.017443  0.0  0.0 |       0.0        0.0  26.382309  0.0  0.0 |  10.787968 |  4.046191 |
    | 01/01 04:00 |  0.0 | 18.9 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.686132  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.686132     0.0        0.0 |       0.0        0.0  16.686132     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.046584  0.046584  0.0  0.0  0.0 |      0.0       0.0   10.63172  0.0   0.0 |      0.0       0.0    5.63172  0.0   0.0 |      0.0       0.0  2.935828  0.0   0.0 |      0.0       0.0  1.935828  0.0   0.0 | 0.0  0.0  0.0  0.074675  0.0 | 0.057032 |  1.746357 | 1.419064 | 1.283085 |  5.346189 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   99.76515    99.76515  0.0  0.0  0.0 |       0.0        0.0  21.440135  0.0  0.0 |       0.0        0.0  32.142373  0.0  0.0 |  11.349374 |  5.346189 |
    | 01/01 05:00 |  0.0 | 22.5 | 23.1  23.1  23.1  23.1  23.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.933163  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.933163     0.0        0.0 |       0.0        0.0  19.933163     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.049338  0.049338  0.0  0.0  0.0 |      0.0       0.0  11.362574  0.0   0.0 |      0.0       0.0   6.362574  0.0   0.0 |      0.0       0.0  3.492113  0.0   0.0 |      0.0       0.0  2.492113  0.0   0.0 | 0.0  0.0  0.0  0.079126  0.0 | 0.060432 |  2.043389 | 1.662241 | 1.497012 |   6.23755 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.715812   99.715812  0.0  0.0  0.0 |       0.0        0.0   23.64815  0.0  0.0 |       0.0        0.0  37.520722  0.0  0.0 |  12.025949 |   6.23755 |
    | 01/01 06:00 |  0.0 | 25.1 | 25.7  25.7  25.7  25.7  25.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.278241  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.278241     0.0        0.0 |       0.0        0.0  22.278241     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.140081  0.140081  0.0  0.0  0.0 |      0.0       0.0  12.491793  0.0   0.0 |      0.0       0.0   7.491793  0.0   0.0 |      0.0       0.0  4.032455  0.0   0.0 |      0.0       0.0  3.032455  0.0   0.0 | 0.0  0.0  0.0  0.224769  0.0 |  0.06429 |  2.428673 | 1.925588 | 1.728847 |  7.203529 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   99.57573    99.57573  0.0  0.0  0.0 |       0.0        0.0  25.942805  0.0  0.0 |       0.0        0.0  42.947605  0.0  0.0 |  12.793647 |  7.203529 |
    | 01/01 07:00 |  0.0 | 28.3 | 28.9  28.9  28.9  28.9  28.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  25.164491  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  25.164491     0.0        0.0 |       0.0        0.0  25.164491     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  0.92625   0.92625  0.0  0.0  0.0 |      0.0       0.0  13.747938  0.0   0.0 |      0.0       0.0   8.747938  0.0   0.0 |      0.0       0.0   4.58314  0.0   0.0 |      0.0       0.0   3.58314  0.0   0.0 | 0.0  0.0  0.0  1.488314  0.0 | 0.067816 |  2.845633 | 2.266922 |  2.02796 |  8.449834 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.649481   98.649481  0.0  0.0  0.0 |       0.0        0.0   28.61142  0.0  0.0 |       0.0        0.0  48.529263  0.0  0.0 |  13.495316 |  8.449834 |
    | 01/01 08:00 |  0.0 | 27.8 | 28.4  28.4  28.4  28.4  28.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  24.713515  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  24.713515     0.0        0.0 |       0.0        0.0  24.713515     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  2.07444   2.07444  0.0  0.0  0.0 |      0.0       0.0  14.508429  0.0   0.0 |      0.0       0.0   9.508429  0.0   0.0 |      0.0       0.0  5.124643  0.0   0.0 |      0.0       0.0  4.124643  0.0   0.0 | 0.0  0.0  0.0  3.364543  0.0 | 0.070564 |  3.146093 | 2.653682 | 2.365717 |  9.857156 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  96.575041   96.575041  0.0  0.0  0.0 |       0.0        0.0  29.308077  0.0  0.0 |       0.0        0.0  53.788406  0.0  0.0 |  14.042241 |  9.857156 |
    | 01/01 09:00 |  0.0 | 31.4 | 32.0  32.0  32.0  32.0  32.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.960546  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.960546     0.0        0.0 |       0.0        0.0  27.960546     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.400961  3.400961  0.0  0.0  0.0 |      0.0       0.0  15.325872  0.0   0.0 |      0.0       0.0  10.325872  0.0   0.0 |      0.0       0.0  5.639582  0.0   0.0 |      0.0       0.0  4.639582  0.0   0.0 | 0.0  0.0  0.0  5.634518  0.0 | 0.072229 |  3.453566 | 3.000924 | 2.668214 |  11.11756 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   93.17408    93.17408  0.0  0.0  0.0 |       0.0        0.0  31.616877  0.0  0.0 |       0.0        0.0  58.835115  0.0  0.0 |  14.373563 |  11.11756 |
    | 01/01 10:00 |  0.0 | 32.2 | 32.8  32.8  32.8  32.8  32.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  28.682108  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  28.682108     0.0        0.0 |       0.0        0.0  28.682108     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.705742  3.705742  0.0  0.0  0.0 |      0.0       0.0  16.188317  0.0   0.0 |      0.0       0.0  11.188317  0.0   0.0 |      0.0       0.0  6.137374  0.0   0.0 |      0.0       0.0  5.137374  0.0   0.0 | 0.0  0.0  0.0  6.363559  0.0 | 0.073898 |  3.767467 | 3.315115 | 2.942071 | 12.258631 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  89.468338   89.468338  0.0  0.0  0.0 |       0.0        0.0  32.922352  0.0  0.0 |       0.0        0.0  63.748684  0.0  0.0 |  14.705736 | 12.258631 |
    | 01/01 11:00 |  0.0 | 35.2 | 35.8  35.8  35.8  35.8  35.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  31.387967  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  31.387967     0.0        0.0 |       0.0        0.0  31.387967     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  4.03435   4.03435  0.0  0.0  0.0 |      0.0       0.0   17.09864  0.0   0.0 |      0.0       0.0   12.09864  0.0   0.0 |      0.0       0.0  6.625341  0.0   0.0 |      0.0       0.0  5.625341  0.0   0.0 | 0.0  0.0  0.0  7.214798  0.0 | 0.075474 |   4.09015 | 3.627409 | 3.214197 | 13.392487 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  85.433988   85.433988  0.0  0.0  0.0 |       0.0        0.0  35.113039  0.0  0.0 |       0.0        0.0  68.596643  0.0  0.0 |  15.019293 | 13.392487 |
    | 01/01 12:00 |  0.0 | 37.1 | 37.7  37.7  37.7  37.7  37.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  33.101678  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  33.101678     0.0        0.0 |       0.0        0.0  33.101678     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.107634  2.107634  0.0  0.0  0.0 |      0.0       0.0  18.106249  0.0   0.0 |      0.0       0.0  13.106249  0.0   0.0 |      0.0       0.0  7.111917  0.0   0.0 |      0.0       0.0  6.111917  0.0   0.0 | 0.0  0.0  0.0  3.947157  0.0 | 0.079897 |  4.434961 |  3.94741 | 3.495659 | 14.565246 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  83.326354   83.326354  0.0  0.0  0.0 |       0.0        0.0  37.002219  0.0  0.0 |       0.0        0.0  73.479058  0.0  0.0 |  15.899498 | 14.565246 |
    | 01/01 13:00 |  0.0 | 31.2 | 31.8  31.8  31.8  31.8  31.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  27.780155  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  27.780155     0.0        0.0 |       0.0        0.0  27.780155     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.656691  2.656691  0.0  0.0  0.0 |      0.0       0.0  17.724504  0.0   0.0 |      0.0       0.0  12.724504  0.0   0.0 |      0.0       0.0  7.536557  0.0   0.0 |      0.0       0.0  6.536557  0.0   0.0 | 0.0  0.0  0.0  5.101273  0.0 | 0.083929 |   4.44486 | 4.258057 |  3.76865 | 15.702708 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  80.669664   80.669664  0.0  0.0  0.0 |       0.0        0.0  34.333366  0.0  0.0 |       0.0        0.0  77.130448  0.0  0.0 |  16.701793 | 15.702708 |
    | 01/01 14:00 |  0.0 | 24.3 | 24.9  24.9  24.9  24.9  24.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.556679  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.556679     0.0        0.0 |       0.0        0.0  21.556679     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.871924  2.871924  0.0  0.0  0.0 |      0.0       0.0  15.736241  0.0   0.0 |      0.0       0.0  10.736241  0.0   0.0 |      0.0       0.0  7.774395  0.0   0.0 |      0.0       0.0  6.774395  0.0   0.0 | 0.0  0.0  0.0  5.696167  0.0 |  0.08777 |  4.040916 | 4.379808 | 3.877752 | 16.157302 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   77.79774    77.79774  0.0  0.0  0.0 |       0.0        0.0  29.417562  0.0  0.0 |       0.0        0.0  78.317899  0.0  0.0 |  17.466227 | 16.157302 |
    | 01/01 15:00 |  0.2 | 25.4 | 26.0  26.0  26.0  26.0  26.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.242   0.242   0.242   0.242   0.242 |    0.242     0.242  0.0  0.0     0.242 |       0.0        0.0   0.242   0.242        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.548827  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.790827   0.242        0.0 |       0.0        0.0  22.790827   0.242        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.873977  0.873977  0.0  0.0  0.0 |      0.0       0.0   14.40956  0.0   0.0 |      0.0       0.0    9.40956  0.0   0.0 |      0.0       0.0  7.819895  0.0   0.0 |      0.0       0.0  6.819895  0.0   0.0 | 0.0  0.0  0.0  1.797434  0.0 | 0.094599 |  3.745259 | 4.188204 | 3.718069 | 15.491954 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  76.923763   76.923763  0.0  0.0  0.0 |       0.0        0.0  28.389269  0.0  0.0 |       0.0        0.0  78.087669  0.0  0.0 |  18.825115 | 15.491954 |
    | 01/01 16:00 |  0.0 | 25.9 | 26.5  26.5  26.5  26.5  26.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  22.999804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  22.999804     0.0        0.0 |       0.0        0.0  22.999804     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.677397  0.677397  0.0  0.0  0.0 |      0.0       0.0  14.122997  0.0   0.0 |      0.0       0.0   9.122997  0.0   0.0 |      0.0       0.0  7.785609  0.0   0.0 |      0.0       0.0  6.785609  0.0   0.0 | 0.0  0.0  0.0  1.408972  0.0 | 0.101461 |  3.671217 | 3.899539 | 3.474298 |  14.47624 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  76.246366   76.246366  0.0  0.0  0.0 |       0.0        0.0   28.14308  0.0  0.0 |       0.0        0.0  77.639447  0.0  0.0 |  20.190717 |  14.47624 |
    | 01/01 17:00 |  0.0 | 23.7 | 24.3  24.3  24.3  24.3  24.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  21.015507  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  21.015507     0.0        0.0 |       0.0        0.0  21.015507     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.417048  0.417048  0.0  0.0  0.0 |      0.0       0.0  13.680195  0.0   0.0 |      0.0       0.0   8.680195  0.0   0.0 |      0.0       0.0   7.72425  0.0   0.0 |      0.0       0.0   6.72425  0.0   0.0 | 0.0  0.0  0.0  0.875158  0.0 | 0.108604 |  3.554872 | 3.722976 | 3.327944 | 13.866432 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  75.829319   75.829319  0.0  0.0  0.0 |       0.0        0.0  26.798197  0.0  0.0 |       0.0        0.0  76.871142  0.0  0.0 |  21.612287 | 13.866432 |
    | 01/01 18:00 |  1.3 | 21.6 | 22.2  22.2  22.2  22.2  22.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  1.573   1.573   1.573   1.573   1.573 | 0.494092  0.494092  0.0  0.0  0.494092 |       0.0        0.0   1.573   1.573        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.121405  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  20.694405   1.573        0.0 |       0.0        0.0  20.694405   1.573        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.117083  0.117083  0.0  0.0  0.0 |      0.0       0.0  13.196068  0.0   0.0 |      0.0       0.0   8.196068  0.0   0.0 |      0.0       0.0  7.631945  0.0   0.0 |      0.0       0.0  6.631945  0.0   0.0 | 0.0  0.0  0.0  0.494092  0.0 | 0.117009 |  3.421849 | 3.604509 | 3.233116 | 13.471317 | 1.078908  1.078908  0.0  0.0  1.078908 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  75.712235   75.712235  0.0  0.0  0.0 |       0.0        0.0  26.100468  0.0  0.0 |       0.0        0.0  75.803321  0.0  0.0 |  23.284824 | 13.471317 |
    | 01/01 19:00 |  5.6 | 21.2 | 21.8  21.8  21.8  21.8  21.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  6.776   6.776   6.776   6.776   6.776 |  0.08755   0.08755  0.0  0.0   0.08755 |  5.854908   5.854908   6.776   6.776   5.854908 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.760624  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.854908   5.854908  25.536624   6.776   5.854908 |  0.839058   0.839058  25.536624   6.776   5.854908 | 0.0  0.0  0.0  0.0   5.854908 | 0.022087  0.022087  0.0  0.0  0.0 | 0.178771  0.178771  13.866219  0.0   0.0 |      0.0       0.0   8.866219  0.0   0.0 | 0.008648  0.008648  7.566544  0.0   0.0 |      0.0       0.0  6.566544  0.0   0.0 | 0.0  0.0  0.0   0.08755  0.0 | 0.129336 |  4.011784 | 3.531515 |  3.18136 | 13.255667 |  1.91245   1.91245  0.0  0.0   1.91245 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  80.705998   80.705998  0.0  0.0  0.0 |  0.660288   0.660288  28.904653  0.0  0.0 |  0.170123   0.170123  75.536452  0.0  0.0 |  25.737942 | 13.255667 |
    | 01/01 20:00 |  2.9 | 20.4 | 21.0  21.0  21.0  21.0  21.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  3.509   3.509   3.509   3.509   3.509 |   0.0539    0.0539  0.0  0.0    0.0539 |   3.42145    3.42145   3.509   3.509    3.42145 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.039062  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   3.42145    3.42145  21.548062   3.509    3.42145 |  0.557137   0.557137  21.548062   3.509    3.42145 | 0.0  0.0  0.0  0.0    3.42145 | 0.014076  0.014076  0.0  0.0  0.0 | 0.378507  0.378507  14.018856  0.0   0.0 |      0.0       0.0   9.018856  0.0   0.0 | 0.034499  0.034499  7.549504  0.0   0.0 |      0.0       0.0  6.549504  0.0   0.0 | 0.0  0.0  0.0    0.0539  0.0 | 0.139357 |  3.855887 | 3.760443 | 3.389118 | 14.121323 |   1.9461    1.9461  0.0  0.0    1.9461 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  83.556234   83.556234  0.0  0.0  0.0 |  0.838917   0.838917  27.415003  0.0  0.0 |  0.514131   0.514131  75.456299  0.0  0.0 |  27.732099 | 14.121323 |
    | 01/01 21:00 |  4.9 | 19.8 | 20.4  20.4  20.4  20.4  20.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  5.929   5.929   5.929   5.929   5.929 | 0.041902  0.041902  0.0  0.0  0.041902 |    5.8751     5.8751   5.929   5.929     5.8751 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   17.49789  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    5.8751     5.8751   23.42689   5.929     5.8751 |  1.025446   1.025446   23.42689   5.929     5.8751 | 0.0  0.0  0.0  0.0     5.8751 | 0.011576  0.011576  0.0  0.0  0.0 | 0.548571  0.548571  13.893628  0.0   0.0 |      0.0       0.0   8.893628  0.0   0.0 | 0.075463  0.075463  7.536375  0.0   0.0 |      0.0       0.0  6.536375  0.0   0.0 | 0.0  0.0  0.0  0.041902  0.0 | 0.151183 |  4.012701 | 3.886322 |  3.50925 | 14.621874 | 1.958098  1.958098  0.0  0.0  1.958098 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  88.394311   88.394311  0.0  0.0  0.0 |  1.315792   1.315792  28.054638  0.0  0.0 |  0.987239   0.987239  75.277176  0.0  0.0 |   30.08538 | 14.621874 |
    | 01/01 22:00 | 10.6 | 19.6 | 20.2  20.2  20.2  20.2  20.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 | 12.826  12.826  12.826  12.826  12.826 | 0.020954  0.020954  0.0  0.0  0.020954 | 12.784098  12.784098  12.826  12.826  12.784098 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.317499  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.784098  12.784098  30.143499  12.826  12.784098 |  2.497231   2.497231  30.143499  12.826  12.784098 | 0.0  0.0  0.0  0.0  12.784098 | 0.006462  0.006462  0.0  0.0  0.0 | 1.049787  1.049787  15.331242  0.0   0.0 |      0.0       0.0  10.331242  0.0   0.0 | 0.144731  0.144731   7.58746  0.0   0.0 |      0.0       0.0   6.58746  0.0   0.0 | 0.0  0.0  0.0  0.020954  0.0 | 0.168168 |  4.887708 | 4.015363 | 3.636938 | 15.153907 | 1.979046  1.979046  0.0  0.0  1.979046 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.674717   98.674717  0.0  0.0  0.0 |  2.763236   2.763236  32.535653  0.0  0.0 |  1.892296   1.892296  76.433499  0.0  0.0 |   33.46543 | 15.153907 |
    | 01/01 23:00 |  0.1 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.450977  0.450977  0.450977  0.0  0.450977 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.121   0.121   0.121   0.121   0.121 | 0.074152  0.074152  0.0  0.0  0.074152 |  0.100046   0.100046   0.121   0.121   0.100046 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.956718  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.100046   0.100046  17.077718   0.121   0.100046 |  0.024353   0.024353  17.077718   0.121   0.100046 | 0.0  0.0  0.0  0.0   0.100046 | 0.022883  0.022883  0.0  0.0  0.0 | 1.092437  1.092437  14.344197  0.0   0.0 |      0.0       0.0   9.344197  0.0   0.0 | 0.232922  0.232922  7.646042  0.0   0.0 |      0.0       0.0  6.646042  0.0   0.0 | 0.0  0.0  0.0  0.074152  0.0 | 0.176301 |  3.697751 | 4.416366 | 3.992065 | 16.633606 | 1.925848  1.925848  0.0  0.0  1.925848 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  98.727527   98.727527  0.0  0.0  0.0 |  1.695152   1.695152  25.924977  0.0  0.0 |  2.751812   2.751812  76.485611  0.0  0.0 |  35.083994 | 16.633606 |
    | 02/01 00:00 |  0.7 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.847   0.847   0.847   0.847   0.847 |  0.06896   0.06896  0.0  0.0   0.06896 |  0.772848   0.772848   0.847   0.847   0.772848 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.772848   0.772848  17.810337   0.847   0.772848 |  0.188326   0.188326  17.810337   0.847   0.772848 | 0.0  0.0  0.0  0.0   0.772848 | 0.021402  0.021402  0.0  0.0  0.0 | 0.707115  0.707115  12.389583  0.0   0.0 |      0.0       0.0   7.389583  0.0   0.0 | 0.296076  0.296076  7.559237  0.0   0.0 |      0.0       0.0  6.559237  0.0   0.0 | 0.0  0.0  0.0   0.06896  0.0 | 0.185027 |  3.278408 | 4.103119 | 3.728728 | 15.536365 |  1.93104   1.93104  0.0  0.0   1.93104 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  99.290647   99.290647  0.0  0.0  0.0 |  1.176363   1.176363  23.956149  0.0  0.0 |  3.162851   3.162851   74.75672  0.0  0.0 |  36.820287 | 15.536365 |
    | 02/01 01:00 |  3.0 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |   3.63    3.63    3.63    3.63    3.63 | 0.052207  0.052207  0.0  0.0  0.052207 |   3.56104    3.56104    3.63    3.63    3.56104 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   3.56104    3.56104  20.593337    3.63    3.56104 |  0.877675   0.877675  20.593337    3.63    3.56104 | 0.0  0.0  0.0  0.0    3.56104 | 0.016637  0.016637  0.0  0.0  0.0 | 0.649861  0.649861  12.279218  0.0   0.0 |      0.0       0.0   7.279218  0.0   0.0 | 0.332422  0.332422  7.397372  0.0   0.0 |      0.0       0.0  6.397372  0.0   0.0 | 0.0  0.0  0.0  0.052207  0.0 | 0.195651 |  3.430062 | 3.558625 |  3.26675 | 13.611457 | 1.947793  1.947793  0.0  0.0  1.947793 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 101.957376  101.957376  0.0  0.0  0.0 |  1.404176   1.404176  24.991049  0.0  0.0 |  3.480291   3.480291  73.241195  0.0  0.0 |  38.934599 | 13.611457 |
    | 02/01 02:00 |  2.1 | 18.9 | 19.5  19.5  19.5  19.5  19.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  2.541   2.541   2.541   2.541   2.541 | 0.057919  0.057919  0.0  0.0  0.057919 |  2.488793   2.488793   2.541   2.541   2.488793 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.692645  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.488793   2.488793  19.233645   2.541   2.488793 |  0.646794   0.646794  19.233645   2.541   2.488793 | 0.0  0.0  0.0  0.0   2.488793 | 0.018787  0.018787  0.0  0.0  0.0 | 0.690307  0.690307  12.356208  0.0   0.0 |      0.0       0.0   7.356208  0.0   0.0 | 0.364587  0.364587  7.263618  0.0   0.0 |      0.0       0.0  6.263618  0.0   0.0 | 0.0  0.0  0.0  0.057919  0.0 | 0.205401 |  3.334482 | 3.403875 | 3.141733 | 13.090554 | 1.942081  1.942081  0.0  0.0  1.942081 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 103.780587  103.780587  0.0  0.0  0.0 |  1.360663   1.360663  24.512279  0.0  0.0 |  3.806011   3.806011  72.070167  0.0  0.0 |  40.874791 | 13.090554 |
    | 02/01 03:00 | 10.4 | 18.7 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 12.584  12.584  12.584  12.584  12.584 | 0.021144  0.021144  0.0  0.0  0.021144 | 12.526081  12.526081  12.584  12.584  12.526081 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 12.526081  12.526081  29.096184  12.584  12.526081 |  3.372776   3.372776  29.096184  12.584  12.526081 | 0.0  0.0  0.0  0.0  12.526081 | 0.007462  0.007462  0.0  0.0  0.0 | 1.253987  1.253987     14.019  0.0   0.0 |      0.0       0.0      9.019  0.0   0.0 |  0.42285   0.42285  7.235345  0.0   0.0 |      0.0       0.0  6.235345  0.0   0.0 | 0.0  0.0  0.0  0.021144  0.0 | 0.222459 |  4.483778 | 3.449888 | 3.197531 | 13.323047 | 1.978856  1.978856  0.0  0.0  1.978856 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  112.92643   112.92643  0.0  0.0  0.0 |  3.479452   3.479452  30.570462  0.0  0.0 |  4.637148   4.637148  72.618477  0.0  0.0 |  44.269289 | 13.323047 |
    | 02/01 04:00 |  3.5 | 18.5 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.235   4.235   4.235   4.235   4.235 | 0.048561  0.048561  0.0  0.0  0.048561 |  4.213856   4.213856   4.235   4.235   4.213856 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  4.213856   4.213856  20.566723   4.235   4.213856 |  1.343417   1.343417  20.566723   4.235   4.213856 | 0.0  0.0  0.0  0.0   4.213856 | 0.017573  0.017573  0.0  0.0  0.0 | 1.655288  1.655288  14.364845  0.0   0.0 |      0.0       0.0   9.364845  0.0   0.0 | 0.521356  0.521356  7.301236  0.0   0.0 |      0.0       0.0  6.301236  0.0   0.0 | 0.0  0.0  0.0  0.048561  0.0 | 0.233835 |  3.939392 | 3.959988 | 3.650236 | 15.209315 | 1.951439  1.951439  0.0  0.0  1.951439 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.779297  115.779297  0.0  0.0  0.0 |  3.167581   3.167581  27.407496  0.0  0.0 |   5.77108    5.77108  73.380849  0.0  0.0 |  46.533225 | 15.209315 |
    | 02/01 05:00 |  3.4 | 18.3 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.114   4.114   4.114   4.114   4.114 | 0.048983  0.048983  0.0  0.0  0.048983 |  4.065439   4.065439   4.114   4.114   4.065439 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.151262  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  4.065439   4.065439  20.265262   4.114   4.065439 |  1.362415   1.362415  20.265262   4.114   4.065439 | 0.0  0.0  0.0  0.0   4.065439 | 0.018136  0.018136  0.0  0.0  0.0 | 1.536624  1.536624  13.309706  0.0   0.0 |      0.0       0.0   8.309706  0.0   0.0 | 0.623524  0.623524  7.320925  0.0   0.0 |      0.0       0.0  6.320925  0.0   0.0 | 0.0  0.0  0.0  0.048983  0.0 | 0.245418 |  3.689025 | 4.060104 | 3.747813 | 15.615889 | 1.951017  1.951017  0.0  0.0  1.951017 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 118.464185  118.464185  0.0  0.0  0.0 |  2.993372   2.993372  26.053346  0.0  0.0 |   6.68418    6.68418  73.048706  0.0  0.0 |   48.83813 | 15.615889 |
    | 02/01 06:00 |  1.2 | 18.5 | 19.1  19.1  19.1  19.1  19.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.452   1.452   1.452   1.452   1.452 | 0.124728  0.124728  0.0  0.0  0.124728 |  1.403017   1.403017   1.452   1.452   1.403017 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.331723  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.403017   1.403017  17.783723   1.452   1.403017 |   0.49224    0.49224  17.783723   1.452   1.403017 | 0.0  0.0  0.0  0.0   1.403017 | 0.046529  0.046529  0.0  0.0  0.0 | 1.282677  1.282677  12.425259  0.0   0.0 |      0.0       0.0   7.425259  0.0   0.0 | 0.698132  0.698132  7.249405  0.0   0.0 |      0.0       0.0  6.249405  0.0   0.0 | 0.0  0.0  0.0  0.124728  0.0 |  0.25515 |  3.263616 | 3.799807 | 3.531306 | 14.713774 | 1.875272  1.875272  0.0  0.0  1.875272 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 119.328432  119.328432  0.0  0.0  0.0 |  2.202935   2.202935   23.98655  0.0  0.0 |  7.268725   7.268725  71.975156  0.0  0.0 |  50.774834 | 14.713774 |
    | 02/01 07:00 |  0.1 | 18.8 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.121   0.121   0.121   0.121 |  0.96204   0.96204  0.0  0.0   0.96204 |       0.0        0.0   0.121   0.121        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.723415   0.121        0.0 |       0.0        0.0  16.723415   0.121        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.358746  0.358746  0.0  0.0  0.0 | 0.866787  0.866787  11.576995  0.0   0.0 |      0.0       0.0   6.576995  0.0   0.0 | 0.733641  0.733641  7.112383  0.0   0.0 |      0.0       0.0  6.112383  0.0   0.0 | 0.0  0.0  0.0   0.96204  0.0 | 0.263252 |  2.928318 | 3.446772 | 3.232904 | 13.470433 | 1.034232  1.034232  0.0  0.0  1.034232 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 118.969686  118.969686  0.0  0.0  0.0 |  1.336147   1.336147  22.555976  0.0  0.0 |  7.401871   7.401871  70.327385  0.0  0.0 |  52.387141 | 13.470433 |
    | 02/01 08:00 |  0.0 | 18.8 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 1.034232  1.034232  0.0  0.0  1.034232 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.602415     0.0        0.0 |       0.0        0.0  16.602415     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.527265  1.527265  0.0  0.0  0.0 | 0.525733  0.525733   11.10259  0.0   0.0 |      0.0       0.0    6.10259  0.0   0.0 | 0.729813  0.729813  6.940823  0.0   0.0 |      0.0       0.0  5.940823  0.0   0.0 | 0.0  0.0  0.0  2.345064  0.0 | 0.270043 |  2.779249 | 3.099219 |  2.93803 | 12.241793 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 117.442421  117.442421  0.0  0.0  0.0 |  0.810414   0.810414   21.95321  0.0  0.0 |  7.197791   7.197791   68.54833  0.0  0.0 |  53.738574 | 12.241793 |
    | 02/01 09:00 |  0.0 | 19.0 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.782876     0.0        0.0 |       0.0        0.0  16.782876     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.091248  1.091248  0.0  0.0  0.0 | 0.318873  0.318873  10.945274  0.0   0.0 |      0.0       0.0   5.945274  0.0   0.0 | 0.700386  0.700386  6.772212  0.0   0.0 |      0.0       0.0  5.772212  0.0   0.0 | 0.0  0.0  0.0  1.486684  0.0 | 0.277138 |  2.704035 | 2.869816 | 2.745836 | 11.440985 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 116.351173  116.351173  0.0  0.0  0.0 |  0.491541   0.491541  21.845538  0.0  0.0 |  6.816279   6.816279   66.94918  0.0  0.0 |  55.150488 | 11.440985 |
    | 02/01 10:00 |  0.4 | 19.2 | 19.8  19.8  19.8  19.8  19.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.484   0.484   0.484   0.484   0.484 |    0.484     0.484  0.0  0.0     0.484 |       0.0        0.0   0.484   0.484        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.963337  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.447337   0.484        0.0 |       0.0        0.0  17.447337   0.484        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.312905  1.312905  0.0  0.0  0.0 | 0.193406  0.193406  11.033464  0.0   0.0 |      0.0       0.0   6.033464  0.0   0.0 | 0.658011  0.658011  6.631403  0.0   0.0 |      0.0       0.0  5.631403  0.0   0.0 | 0.0  0.0  0.0  1.805438  0.0 | 0.284029 |  2.691892 | 2.751337 | 2.649586 |  11.03994 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 115.038267  115.038267  0.0  0.0  0.0 |  0.298135   0.298135  22.225948  0.0  0.0 |  6.351675   6.351675  65.719837  0.0  0.0 |  56.521705 |  11.03994 |
    | 02/01 11:00 |  0.1 | 19.3 | 19.9  19.9  19.9  19.9  19.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.121   0.121   0.121   0.121 |    0.121     0.121  0.0  0.0     0.121 |       0.0        0.0   0.121   0.121        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.053567  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.174567   0.121        0.0 |       0.0        0.0  17.174567   0.121        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.399893  3.399893  0.0  0.0  0.0 | 0.117307  0.117307  11.103523  0.0   0.0 |      0.0       0.0   6.103523  0.0   0.0 | 0.610116  0.610116  6.523263  0.0   0.0 |      0.0       0.0  5.523263  0.0   0.0 | 0.0  0.0  0.0  4.728713  0.0 | 0.288268 |  2.683104 | 2.704452 | 2.612908 | 10.887119 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 111.638374  111.638374  0.0  0.0  0.0 |  0.180828   0.180828  22.193469  0.0  0.0 |  5.858865   5.858865  64.776834  0.0  0.0 |  57.365252 | 10.887119 |
    | 02/01 12:00 |  3.6 | 19.0 | 19.6  19.6  19.6  19.6  19.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   4.356   4.356   4.356   4.356 | 1.680962  1.680962  0.0  0.0  1.680962 |     2.356      2.356   4.356   4.356      2.356 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.782876  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |     2.356      2.356  21.138876   4.356      2.356 |  0.734078   0.734078  21.138876   4.356      2.356 | 0.0  0.0  0.0  0.0      2.356 | 0.594957  0.594957  0.0  0.0  0.0 | 0.227554  0.227554  11.822452  0.0   0.0 |      0.0       0.0   6.822452  0.0   0.0 | 0.568552  0.568552  6.471459  0.0   0.0 |      0.0       0.0  5.471459  0.0   0.0 | 0.0  0.0  0.0  1.680962  0.0 | 0.297498 |  3.018287 | 2.712676 | 2.628651 | 10.952712 | 0.319038  0.319038  0.0  0.0  0.319038 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 112.665339  112.665339  0.0  0.0  0.0 |  0.687352   0.687352  24.687442  0.0  0.0 |  5.517867   5.517867  64.656367  0.0  0.0 |  59.202142 | 10.952712 |
    | 02/01 13:00 |  5.9 | 18.8 | 19.4  19.4  19.4  19.4  19.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  7.139   7.139   7.139   7.139   7.139 |      2.0       2.0  0.0  0.0       2.0 |  5.458038   5.458038   7.139   7.139   5.458038 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.602415  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.458038   5.458038  23.741415   7.139   5.458038 |  1.732037   1.732037  23.741415   7.139   5.458038 | 0.0  0.0  0.0  0.0   5.458038 | 1.873211  1.873211  0.0  0.0  0.0 | 0.639482  0.639482  13.089408  0.0   0.0 |      0.0       0.0   8.089408  0.0   0.0 | 0.556029  0.556029  6.519868  0.0   0.0 |      0.0       0.0  5.519868  0.0   0.0 | 0.0  0.0  0.0  3.326542  0.0 | 0.307507 |  3.560451 | 2.914476 | 2.812886 | 11.720357 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 114.518129  114.518129  0.0  0.0  0.0 |  1.779907   1.779907  27.250041  0.0  0.0 |   5.60132    5.60132  65.706037  0.0  0.0 |  61.193833 | 11.720357 |
    | 02/01 14:00 |  1.1 | 18.7 | 19.3  19.3  19.3  19.3  19.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.331   1.331   1.331   1.331   1.331 |    1.331     1.331  0.0  0.0     1.331 |       0.0        0.0   1.331   1.331        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.512184  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.843184   1.331        0.0 |       0.0        0.0  17.843184   1.331        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.674068  1.674068  0.0  0.0  0.0 | 0.700339  0.700339  12.814425  0.0   0.0 |      0.0       0.0   7.814425  0.0   0.0 | 0.566914  0.566914  6.602128  0.0   0.0 |      0.0       0.0  5.602128  0.0   0.0 | 0.0  0.0  0.0  2.727186  0.0 | 0.313868 |  3.096128 | 3.267642 |   3.1249 | 13.020417 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 112.844061  112.844061  0.0  0.0  0.0 |  1.079568   1.079568  24.464376  0.0  0.0 |  5.734744   5.734744  66.316205  0.0  0.0 |  62.459697 | 13.020417 |
    | 02/01 15:00 | 20.7 | 17.8 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 25.047  25.047  25.047  25.047  25.047 | 0.132786  0.132786  0.0  0.0  0.132786 |    23.047     23.047  25.047  25.047     23.047 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   15.70011  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |    23.047     23.047   40.74711  25.047     23.047 |  7.336887   7.336887   40.74711  25.047     23.047 | 0.0  0.0  0.0  0.0     23.047 | 0.053344  0.053344  0.0  0.0  0.0 | 1.987984  1.987984  16.146928  0.0   0.0 |      0.0       0.0  11.146928  0.0   0.0 |   0.6419    0.6419  6.813481  0.0   0.0 |      0.0       0.0  5.813481  0.0   0.0 | 0.0  0.0  0.0  0.132786  0.0 | 0.339458 |  5.686787 | 3.431587 | 3.290869 | 13.711956 | 1.867214  1.867214  0.0  0.0  1.867214 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 128.500829  128.500829  0.0  0.0  0.0 |  6.428471   6.428471   37.91763  0.0  0.0 |  7.080828   7.080828   69.83617  0.0  0.0 |  67.552095 | 13.711956 |
    | 02/01 16:00 | 37.9 | 17.4 | 18.0  18.0  18.0  18.0  18.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 45.859  45.859  45.859  45.859  45.859 | 0.017578  0.017578  0.0  0.0  0.017578 | 45.726214  45.726214  45.859  45.859  45.726214 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.339188  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 45.726214  45.726214  61.198188  45.859  45.726214 | 18.876311  18.876311  61.198188  45.859  45.726214 | 0.0  0.0  0.0  0.0  45.726214 | 0.008533  0.008533  0.0  0.0  0.0 | 6.359947  6.359947  24.160733  0.0   0.0 | 1.571769  1.571769  19.160733  0.0   0.0 | 0.980593  0.980593  7.507774  0.0   0.0 | 0.057313  0.057313  6.507774  0.0   0.0 | 0.0  0.0  0.0  0.017578  0.0 |   0.3817 | 10.568729 | 4.959579 | 4.654555 | 19.393981 | 1.982422  1.982422  0.0  0.0  1.982422 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 155.342199  155.342199  0.0  0.0  0.0 | 17.373066  17.373066  55.794352  0.0  0.0 | 12.402869  12.402869  79.981356  0.0  0.0 |  75.958359 | 19.393981 |
    | 02/01 17:00 |  8.2 | 17.3 | 17.9  17.9  17.9  17.9  17.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  9.922   9.922   9.922   9.922   9.922 | 0.821583  0.821583  0.0  0.0  0.821583 |  9.904422   9.904422   9.922   9.922   9.904422 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.248957  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  9.904422   9.904422  25.170957   9.922   9.904422 |   5.97514    5.97514  25.170957   9.922   9.904422 | 0.0  0.0  0.0  0.0   9.904422 | 0.408921  0.408921  0.0  0.0  0.0 |   7.5097    7.5097  23.184016  0.0   0.0 |   2.5097    2.5097  18.184016  0.0   0.0 | 1.522612  1.522612  8.381542  0.0   0.0 | 0.522612  0.522612  7.381542  0.0   0.0 | 0.0  0.0  0.0  0.821583  0.0 | 0.400166 |  8.760916 | 8.140043 | 7.428192 | 30.950801 | 1.178417  1.178417  0.0  0.0  1.178417 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 158.862561  158.862561  0.0  0.0  0.0 | 13.328805  13.328805  39.597277  0.0  0.0 | 17.867345  17.867345  87.402287  0.0  0.0 |  79.633119 | 30.950801 |
    | 02/01 18:00 |  3.6 | 16.8 | 17.4  17.4  17.4  17.4  17.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   4.356   4.356   4.356   4.356 | 0.792694  0.792694  0.0  0.0  0.792694 |  3.534417   3.534417   4.356   4.356   3.534417 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.797804  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.534417   3.534417  19.153804   4.356   3.534417 |   2.22998    2.22998  19.153804   4.356   3.534417 | 0.0  0.0  0.0  0.0   3.534417 | 0.396347  0.396347  0.0  0.0  0.0 | 5.541587  5.541587  16.957971  0.0   0.0 | 0.571295  0.571295  11.957971  0.0   0.0 | 1.925722  1.925722  8.762589  0.0   0.0 | 0.925722  0.925722  7.762589  0.0   0.0 | 0.0  0.0  0.0  0.792694  0.0 | 0.416289 |  5.859173 | 8.898213 | 8.100321 | 33.751339 | 1.207306  1.207306  0.0  0.0  1.207306 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 159.770651  159.770651  0.0  0.0  0.0 |  9.445903   9.445903  29.835141  0.0  0.0 | 20.557487  20.557487  87.835079  0.0  0.0 |  82.841536 | 33.751339 |
    | 02/01 19:00 |  7.5 | 16.5 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  9.075   9.075   9.075   9.075   9.075 | 0.090607  0.090607  0.0  0.0  0.090607 |  8.282306   8.282306   9.075   9.075   8.282306 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.527113  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  8.282306   8.282306  23.602113   9.075   8.282306 |   5.28549    5.28549  23.602113   9.075   8.282306 | 0.0  0.0  0.0  0.0   8.282306 | 0.045304  0.045304  0.0  0.0  0.0 | 4.842807  4.842807  14.690768  0.0   0.0 |      0.0       0.0   9.690768  0.0   0.0 | 2.136821  2.136821  8.695649  0.0   0.0 | 1.136821  1.136821  7.695649  0.0   0.0 | 0.0  0.0  0.0  0.090607  0.0 |  0.43681 |   5.43638 | 7.170877 | 6.622449 | 27.593538 | 1.909393  1.909393  0.0  0.0  1.909393 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 162.722163  162.722163  0.0  0.0  0.0 |  9.888586   9.888586  29.055718  0.0  0.0 | 22.126653  22.126653  86.134549  0.0  0.0 |  86.925234 | 27.593538 |
    | 02/01 20:00 | 18.5 | 16.3 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 22.385  22.385  22.385  22.385  22.385 | 0.007604  0.007604  0.0  0.0  0.007604 | 22.294393  22.294393  22.385  22.385  22.294393 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.346652  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 22.294393  22.294393  36.731652  22.385  22.294393 | 14.758053  14.758053  36.731652  22.385  22.294393 | 0.0  0.0  0.0  0.0  22.294393 | 0.003802  0.003802  0.0  0.0  0.0 |  6.75958   6.75958  16.859467  0.0   0.0 | 1.759897  1.759897  11.859467  0.0   0.0 | 2.368798  2.368798  8.643076  0.0   0.0 | 1.368798  1.368798  7.643076  0.0   0.0 | 0.0  0.0  0.0  0.007604  0.0 | 0.467484 |   8.38156 | 6.081112 | 5.706616 | 23.777568 | 1.992396  1.992396  0.0  0.0  1.992396 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 170.254701  170.254701  0.0  0.0  0.0 | 16.127162  16.127162  37.068437  0.0  0.0 | 25.148638  25.148638  86.707865  0.0  0.0 |  93.029407 | 23.777568 |
    | 02/01 21:00 | 15.4 | 16.2 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 | 18.634  18.634  18.634  18.634  18.634 | 0.011044  0.011044  0.0  0.0  0.011044 | 18.626396  18.626396  18.634  18.634  18.626396 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.256421  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 18.626396  18.626396  32.890421  18.634  18.626396 | 13.497927  13.497927  32.890421  18.634  18.626396 | 0.0  0.0  0.0  0.0  18.626396 | 0.005522  0.005522  0.0  0.0  0.0 | 8.499659  8.499659  18.685414  0.0   0.0 | 3.499659  3.499659  13.685414  0.0   0.0 | 2.724177  2.724177  8.780541  0.0   0.0 | 1.724177  1.724177  7.780541  0.0   0.0 | 0.0  0.0  0.0  0.011044  0.0 | 0.496613 | 10.002983 | 7.301415 | 6.791399 | 28.297494 | 1.988956  1.988956  0.0  0.0  1.988956 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 175.377648  175.377648  0.0  0.0  0.0 | 17.625771  17.625771   37.58803  0.0  0.0 | 29.199943  29.199943  88.832196  0.0  0.0 |  98.826017 | 28.297494 |
    | 02/01 22:00 |  6.3 | 15.5 | 16.1  16.1  16.1  16.1  16.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  7.623   7.623   7.623   7.623   7.623 | 0.032783  0.032783  0.0  0.0  0.032783 |  7.611956   7.611956   7.623   7.623   7.611956 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.624808  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  7.611956   7.611956  21.247808   7.623   7.611956 |  5.853084   5.853084  21.247808   7.623   7.611956 | 0.0  0.0  0.0  0.0   7.611956 | 0.016392  0.016392  0.0  0.0  0.0 | 7.567119  7.567119  16.708098  0.0   0.0 | 2.567119  2.567119  11.708098  0.0   0.0 | 3.047697  3.047697  8.880488  0.0   0.0 | 2.047697  2.047697  7.880488  0.0   0.0 | 0.0  0.0  0.0  0.032783  0.0 | 0.518863 |  8.300851 | 8.950361 | 8.241252 | 34.338549 | 1.967217  1.967217  0.0  0.0  1.967217 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.120128  177.120128  0.0  0.0  0.0 | 13.344616  13.344616  30.419642  0.0  0.0 | 31.671668  31.671668  88.779318  0.0  0.0 | 103.253666 | 34.338549 |
    | 02/01 23:00 |  1.9 | 14.6 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451153  0.451153  0.451153  0.0  0.451153 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902305  0.0   0.0 |  2.299   2.299   2.299   2.299   2.299 | 0.054873  0.054873  0.0  0.0  0.054873 |  2.266217   2.266217   2.299   2.299   2.266217 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.812733  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.266217   2.266217  15.111733   2.299   2.266217 |  1.777368   1.777368  15.111733   2.299   2.266217 | 0.0  0.0  0.0  0.0   2.266217 | 0.027436  0.027436  0.0  0.0  0.0 | 5.461444  5.461444  13.313787  0.0   0.0 | 0.525218  0.525218   8.313787  0.0   0.0 | 3.173119  3.173119   8.71675  0.0   0.0 | 2.173119  2.173119   7.71675  0.0   0.0 | 0.0  0.0  0.0  0.054873  0.0 |  0.53741 |  5.741759 | 8.684208 | 8.027897 | 33.449569 | 1.945127  1.945127  0.0  0.0  1.945127 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 177.581541  177.581541  0.0  0.0  0.0 |  9.135322   9.135322  23.903801  0.0  0.0 | 31.786874  31.786874  85.659605  0.0  0.0 | 106.944583 | 33.449569 |
    | 03/01 00:00 |  4.9 | 14.7 | 15.3  15.3  15.3  15.3  15.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  5.929   5.929   5.929   5.929   5.929 | 0.038243  0.038243  0.0  0.0  0.038243 |  5.874127   5.874127   5.929   5.929   5.874127 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   12.90841  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  5.874127   5.874127   18.83741   5.929   5.874127 |  4.631045   4.631045   18.83741   5.929   5.874127 | 0.0  0.0  0.0  0.0   5.874127 | 0.019122  0.019122  0.0  0.0  0.0 | 4.581166  4.581166  11.939689  0.0   0.0 |      0.0       0.0   6.939689  0.0   0.0 |  3.14234   3.14234  8.369651  0.0   0.0 |  2.14234   2.14234  7.369651  0.0   0.0 | 0.0  0.0  0.0  0.038243  0.0 | 0.557998 |   5.23717 | 6.923604 | 6.521256 | 27.171898 | 1.961757  1.961757  0.0  0.0  1.961757 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 178.805502  178.805502  0.0  0.0  0.0 |  9.185201   9.185201  23.861834  0.0  0.0 |  31.08336   31.08336  81.859991  0.0  0.0 | 111.041694 | 27.171898 |
    | 03/01 01:00 |  2.7 | 14.6 | 15.2  15.2  15.2  15.2  15.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  3.267   3.267   3.267   3.267   3.267 |  0.04981   0.04981  0.0  0.0   0.04981 |  3.228757   3.228757   3.267   3.267   3.228757 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.818141  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  3.228757   3.228757  16.085141   3.267   3.228757 |  2.580697   2.580697  16.085141   3.267   3.228757 | 0.0  0.0  0.0  0.0   3.228757 | 0.024905  0.024905  0.0  0.0  0.0 | 4.163942  4.163942  11.420173  0.0   0.0 |      0.0       0.0   6.420173  0.0   0.0 |  3.05904   3.05904  8.000947  0.0   0.0 |  2.05904   2.05904  7.000947  0.0   0.0 | 0.0  0.0  0.0   0.04981  0.0 | 0.575912 |  4.771037 | 5.646757 | 5.431373 | 22.630722 |  1.95019   1.95019  0.0  0.0   1.95019 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.428656  179.428656  0.0  0.0  0.0 |  7.601957   7.601957  22.106629  0.0  0.0 | 30.129222  30.129222  78.278269  0.0  0.0 | 114.606395 | 22.630722 |
    | 03/01 02:00 |  0.5 | 14.1 | 14.7  14.7  14.7  14.7  14.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.605   0.605   0.605   0.605   0.605 | 0.064366  0.064366  0.0  0.0  0.064366 |   0.55519    0.55519   0.605   0.605    0.55519 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.366798  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   0.55519    0.55519  12.971798   0.605    0.55519 |  0.446854   0.446854  12.971798   0.605    0.55519 | 0.0  0.0  0.0  0.0    0.55519 | 0.032183  0.032183  0.0  0.0  0.0 | 3.086344  3.086344  10.292755  0.0   0.0 |      0.0       0.0   5.292755  0.0   0.0 | 2.922101  2.922101  7.623526  0.0   0.0 | 1.922101  1.922101  6.623526  0.0   0.0 | 0.0  0.0  0.0  0.064366  0.0 | 0.590979 |  4.123304 | 4.992247 | 4.878195 |  20.32581 | 1.935634  1.935634  0.0  0.0  1.935634 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  179.50481   179.50481  0.0  0.0  0.0 |  4.962466   4.962466  19.492918  0.0  0.0 | 28.371363  28.371363  74.323972  0.0  0.0 | 117.604757 |  20.32581 |
    | 03/01 03:00 |  0.2 | 14.3 | 14.9  14.9  14.9  14.9  14.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.242   0.242   0.242   0.242 | 0.067009  0.067009  0.0  0.0  0.067009 |  0.177634   0.177634   0.242   0.242   0.177634 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.547335  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.177634   0.177634  12.789335   0.242   0.177634 |  0.143093   0.143093  12.789335   0.242   0.177634 | 0.0  0.0  0.0  0.0   0.177634 | 0.033505  0.033505  0.0  0.0  0.0 | 1.983066  1.983066   9.433102  0.0   0.0 |      0.0       0.0   4.433102  0.0   0.0 | 2.711115  2.711115  7.224875  0.0   0.0 | 1.711115  1.711115  6.224875  0.0   0.0 | 0.0  0.0  0.0  0.067009  0.0 | 0.604604 |  3.657816 | 4.413047 | 4.388937 | 18.287239 | 1.932991  1.932991  0.0  0.0  1.932991 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.505846  179.505846  0.0  0.0  0.0 |  3.122494   3.122494  18.416048  0.0  0.0 | 25.932198  25.932198  70.307325  0.0  0.0 | 120.316199 | 18.287239 |
    | 03/01 04:00 |  0.5 | 14.9 | 15.5  15.5  15.5  15.5  15.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.605   0.605   0.605   0.605   0.605 | 0.065382  0.065382  0.0  0.0  0.065382 |  0.537991   0.537991   0.605   0.605   0.537991 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.088947  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.537991   0.537991  13.693947   0.605   0.537991 |  0.433383   0.433383  13.693947   0.605   0.537991 | 0.0  0.0  0.0  0.0   0.537991 | 0.032691  0.032691  0.0  0.0  0.0 | 1.320943  1.320943   9.259141  0.0   0.0 |      0.0       0.0   4.259141  0.0   0.0 | 2.459038  2.459038  6.852681  0.0   0.0 | 1.459038  1.459038  5.852681  0.0   0.0 | 0.0  0.0  0.0  0.065382  0.0 | 0.617213 |  3.384984 | 3.896308 | 3.952865 | 16.470272 | 1.934618  1.934618  0.0  0.0  1.934618 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.577763  179.577763  0.0  0.0  0.0 |  2.234934   2.234934  18.591714  0.0  0.0 | 23.335066  23.335066  66.861103  0.0  0.0 | 122.825316 | 16.470272 |
    | 03/01 05:00 |  2.4 | 15.7 | 16.3  16.3  16.3  16.3  16.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  2.904   2.904   2.904   2.904   2.904 | 0.052751  0.052751  0.0  0.0  0.052751 |  2.838618   2.838618   2.904   2.904   2.838618 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.811096  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  2.838618   2.838618  16.715096   2.904   2.838618 |  2.288506   2.288506  16.715096   2.904   2.838618 | 0.0  0.0  0.0  0.0   2.838618 | 0.026376  0.026376  0.0  0.0  0.0 |  1.36697   1.36697   9.870371  0.0   0.0 |      0.0       0.0   4.870371  0.0   0.0 | 2.225803  2.225803  6.568956  0.0   0.0 | 1.225803  1.225803  5.568956  0.0   0.0 | 0.0  0.0  0.0  0.052751  0.0 | 0.630356 |  3.476063 | 3.553521 |  3.66805 | 15.283543 | 1.947249  1.947249  0.0  0.0  1.947249 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 180.101499  180.101499  0.0  0.0  0.0 |   3.15647    3.15647  20.566068  0.0  0.0 |  21.25043   21.25043  64.593561  0.0  0.0 | 125.440788 | 15.283543 |
    | 03/01 06:00 |  0.4 | 16.0 | 16.6  16.6  16.6  16.6  16.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.484   0.484   0.484   0.484   0.484 | 0.113609  0.113609  0.0  0.0  0.113609 |  0.431249   0.431249   0.484   0.484   0.431249 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.081902  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.431249   0.431249  14.565902   0.484   0.431249 |  0.349705   0.349705  14.565902   0.484   0.431249 | 0.0  0.0  0.0  0.0   0.431249 | 0.056805  0.056805  0.0  0.0  0.0 | 1.316483  1.316483  10.099064  0.0   0.0 |      0.0       0.0   5.099064  0.0   0.0 | 2.034498  2.034498  6.374148  0.0   0.0 | 1.034498  1.034498  5.374148  0.0   0.0 | 0.0  0.0  0.0  0.113609  0.0 | 0.640837 |  3.166259 | 3.442678 | 3.581769 | 14.924038 | 1.886391  1.886391  0.0  0.0  1.886391 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 180.126238  180.126238  0.0  0.0  0.0 |  2.189693   2.189693  19.933842  0.0  0.0 | 19.497917  19.497917  62.944329  0.0  0.0 | 127.526644 | 14.924038 |
    | 03/01 07:00 |  0.2 | 16.7 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.242   0.242   0.242   0.242 | 0.353524  0.353524  0.0  0.0  0.353524 |  0.128391   0.128391   0.242   0.242   0.128391 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.128391   0.128391  14.955782   0.242   0.128391 |  0.104143   0.104143  14.955782   0.242   0.128391 | 0.0  0.0  0.0  0.0   0.128391 | 0.176762  0.176762  0.0  0.0  0.0 | 0.883766  0.883766   9.970957  0.0   0.0 |      0.0       0.0   4.970957  0.0   0.0 | 1.855397  1.855397  6.218671  0.0   0.0 | 0.855397  0.855397  5.218671  0.0   0.0 | 0.0  0.0  0.0  0.353524  0.0 |  0.65018 |  2.953527 | 3.279189 | 3.448799 | 14.369996 | 1.646476  1.646476  0.0  0.0  1.646476 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.973724  179.973724  0.0  0.0  0.0 |   1.41007    1.41007  19.947711  0.0  0.0 | 17.670888  17.670888  61.477943  0.0  0.0 |  129.38586 | 14.369996 |
    | 03/01 08:00 |  0.0 | 17.1 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 0.903769  0.903769  0.0  0.0  0.903769 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.074856  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.074856     0.0        0.0 |       0.0        0.0  15.074856     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.451885  0.451885  0.0  0.0  0.0 | 0.554819  0.554819   9.997242  0.0   0.0 |      0.0       0.0   4.997242  0.0   0.0 | 1.674402  1.674402  6.086997  0.0   0.0 | 0.674402  0.674402  5.086997  0.0   0.0 | 0.0  0.0  0.0  0.903769  0.0 | 0.658188 |  2.794026 | 3.058216 | 3.264763 | 13.603177 | 0.742707  0.742707  0.0  0.0  0.742707 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 179.521839  179.521839  0.0  0.0  0.0 |   0.85525    0.85525  20.028082  0.0  0.0 | 15.876904  15.876904  60.301192  0.0  0.0 |  130.97932 | 13.603177 |
    | 03/01 09:00 |  0.0 | 16.2 | 16.8  16.8  16.8  16.8  16.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 0.742707  0.742707  0.0  0.0  0.742707 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.262439  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.262439     0.0        0.0 |       0.0        0.0  14.262439     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.864415  2.864415  0.0  0.0  0.0 | 0.336515  0.336515   9.873209  0.0   0.0 |      0.0       0.0   4.873209  0.0   0.0 | 1.501582  1.501582  5.974534  0.0   0.0 | 0.501582  0.501582  4.974534  0.0   0.0 | 0.0  0.0  0.0  3.235768  0.0 | 0.663813 |  2.619805 | 2.870414 | 3.107251 | 12.946878 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 176.657424  176.657424  0.0  0.0  0.0 |  0.518736   0.518736  19.544103  0.0  0.0 | 14.210255  14.210255  59.225332  0.0  0.0 | 132.098814 | 12.946878 |
    | 03/01 10:00 |  0.3 | 15.9 | 16.5  16.5  16.5  16.5  16.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.363   0.363   0.363   0.363 |    0.363     0.363  0.0  0.0     0.363 |       0.0        0.0   0.363   0.363        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.991633  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.354633   0.363        0.0 |       0.0        0.0  14.354633   0.363        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.055807  3.055807  0.0  0.0  0.0 | 0.204107  0.204107   9.737201  0.0   0.0 |      0.0       0.0   4.737201  0.0   0.0 | 1.344326  1.344326  5.870655  0.0   0.0 | 0.344326  0.344326  4.870655  0.0   0.0 | 0.0  0.0  0.0  3.237307  0.0 | 0.669052 |  2.455577 | 2.698172 | 2.962864 | 12.345267 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 173.601617  173.601617  0.0  0.0  0.0 |  0.314629   0.314629  19.424335  0.0  0.0 |  12.72571   12.72571  58.221222  0.0  0.0 | 133.141354 | 12.345267 |
    | 03/01 11:00 |  2.6 | 16.3 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  3.146   3.146   3.146   3.146   3.146 |      2.0       2.0  0.0  0.0       2.0 |     1.146      1.146   3.146   3.146      1.146 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |     1.146      1.146  17.498707   3.146      1.146 |   0.86344    0.86344  17.498707   3.146      1.146 | 0.0  0.0  0.0  0.0      1.146 | 1.982694  1.982694  0.0  0.0  0.0 | 0.307763  0.307763  10.277667  0.0   0.0 |      0.0       0.0   5.277667  0.0   0.0 | 1.214628  1.214628  5.804957  0.0   0.0 | 0.214628  0.214628  4.804957  0.0   0.0 | 0.0  0.0  0.0  2.982694  0.0 | 0.675947 |  2.563502 | 2.549737 | 2.840656 | 11.836067 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 171.901483  171.901483  0.0  0.0  0.0 |  0.870306   0.870306  21.367709  0.0  0.0 | 11.604216  11.604216  57.888976  0.0  0.0 | 134.513488 | 11.836067 |
    | 03/01 12:00 |  0.7 | 16.3 | 16.9  16.9  16.9  16.9  16.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.847   0.847   0.847   0.847   0.847 |    0.847     0.847  0.0  0.0     0.847 |       0.0        0.0   0.847   0.847        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.352707  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.199707   0.847        0.0 |       0.0        0.0  15.199707   0.847        0.0 | 0.0  0.0  0.0  0.0        0.0 |  3.37986   3.37986  0.0  0.0  0.0 | 0.342439  0.342439  10.469013  0.0   0.0 |      0.0       0.0   5.469013  0.0   0.0 | 1.114606  1.114606  5.783804  0.0   0.0 | 0.114606  0.114606  4.783804  0.0   0.0 | 0.0  0.0  0.0   3.80336  0.0 | 0.680235 |  2.445377 | 2.524332 | 2.822641 | 11.761004 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 168.521623  168.521623  0.0  0.0  0.0 |  0.527868   0.527868  20.629391  0.0  0.0 | 10.717443  10.717443  57.790381  0.0  0.0 | 135.366834 | 11.761004 |
    | 03/01 13:00 |  0.3 | 16.4 | 17.0  17.0  17.0  17.0  17.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.363   0.363   0.363   0.363 |    0.363     0.363  0.0  0.0     0.363 |       0.0        0.0   0.363   0.363        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.442976  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.805976   0.363        0.0 |       0.0        0.0  14.805976   0.363        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.277978  3.277978  0.0  0.0  0.0 |   0.2077    0.2077  10.163237  0.0   0.0 |      0.0       0.0   5.163237  0.0   0.0 | 1.027918  1.027918  5.760549  0.0   0.0 | 0.029027  0.029027  4.760549  0.0   0.0 | 0.0  0.0  0.0  3.459478  0.0 | 0.684098 |    2.3102 |  2.47909 | 2.787037 | 11.612653 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 165.243645  165.243645  0.0  0.0  0.0 |  0.320168   0.320168  20.108894  0.0  0.0 |  9.868198   9.868198   57.43252  0.0  0.0 |  136.13559 | 11.612653 |
    | 03/01 14:00 |  0.3 | 16.5 | 17.1  17.1  17.1  17.1  17.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.363   0.363   0.363   0.363 |    0.363     0.363  0.0  0.0     0.363 |       0.0        0.0   0.363   0.363        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.533245  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.896245   0.363        0.0 |       0.0        0.0  14.896245   0.363        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.602665  5.602665  0.0  0.0  0.0 | 0.125976  0.125976  10.015332  0.0   0.0 |      0.0       0.0   5.015332  0.0   0.0 | 0.945177  0.945177  5.721188  0.0   0.0 |      0.0       0.0  4.721188  0.0   0.0 | 0.0  0.0  0.0  5.784165  0.0 | 0.685974 |  2.246889 | 2.372414 | 2.696334 | 11.234727 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  159.64098   159.64098  0.0  0.0  0.0 |  0.194192   0.194192  19.974474  0.0  0.0 |  9.048997   9.048997  57.005475  0.0  0.0 | 136.508747 | 11.234727 |
    | 03/01 15:00 |  0.0 | 18.4 | 19.0  19.0  19.0  19.0  19.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.248348  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.248348     0.0        0.0 |       0.0        0.0  16.248348     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.727324  3.727324  0.0  0.0  0.0 | 0.076408  0.076408  10.221553  0.0   0.0 |      0.0       0.0   5.221553  0.0   0.0 | 0.864822  0.864822   5.69214  0.0   0.0 |      0.0       0.0   4.69214  0.0   0.0 | 0.0  0.0  0.0  3.735707  0.0 | 0.688754 |  2.287775 | 2.289162 | 2.626778 | 10.944907 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 155.913656  155.913656  0.0  0.0  0.0 |  0.117783   0.117783  20.779716  0.0  0.0 |  8.260583   8.260583  56.842748  0.0  0.0 | 137.062022 | 10.944907 |
    | 03/01 16:00 |  0.0 | 18.3 | 18.9  18.9  18.9  18.9  18.9 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.158079  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.158079     0.0        0.0 |       0.0        0.0  16.158079     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.981965  3.981965  0.0  0.0  0.0 | 0.046344  0.046344  10.459454  0.0   0.0 |      0.0       0.0   5.459454  0.0   0.0 |  0.78834   0.78834  5.688532  0.0   0.0 |      0.0       0.0  4.688532  0.0   0.0 | 0.0  0.0  0.0  4.086328  0.0 |  0.69102 |  2.341843 |  2.28079 | 2.621637 | 10.923486 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 151.931691  151.931691  0.0  0.0  0.0 |  0.071439   0.071439  21.018887  0.0  0.0 |  7.518587   7.518587  56.925139  0.0  0.0 | 137.513002 | 10.923486 |
    | 03/01 17:00 |  0.0 | 18.1 | 18.7  18.7  18.7  18.7  18.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.977542  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.977542     0.0        0.0 |       0.0        0.0  15.977542     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 2.402655  2.402655  0.0  0.0  0.0 | 0.028109  0.028109  10.501839  0.0   0.0 |      0.0       0.0   5.501839  0.0   0.0 | 0.716848  0.716848  5.697984  0.0   0.0 |      0.0       0.0  4.697984  0.0   0.0 | 0.0  0.0  0.0  2.530248  0.0 | 0.694167 |  2.353805 |  2.31691 | 2.655878 | 11.066158 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 149.529036  149.529036  0.0  0.0  0.0 |   0.04333    0.04333  20.992752  0.0  0.0 |  6.829848   6.829848   57.03101  0.0  0.0 | 138.139198 | 11.066158 |
    | 03/01 18:00 |  0.0 | 16.7 | 17.3  17.3  17.3  17.3  17.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  14.713782  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  14.713782     0.0        0.0 |       0.0        0.0  14.713782     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.083886  1.083886  0.0  0.0  0.0 | 0.017049  0.017049  10.261123  0.0   0.0 |      0.0       0.0   5.261123  0.0   0.0 | 0.650771  0.650771  5.696307  0.0   0.0 |      0.0       0.0  4.696307  0.0   0.0 | 0.0  0.0  0.0  1.159786  0.0 | 0.698063 |  2.297868 | 2.339881 | 2.679422 | 11.164258 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  148.44515   148.44515  0.0  0.0  0.0 |  0.026281   0.026281  20.184289  0.0  0.0 |  6.196126   6.196126  56.899518  0.0  0.0 | 138.914442 | 11.164258 |
    | 03/01 19:00 |  0.0 | 15.2 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.359753  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.359753     0.0        0.0 |       0.0        0.0  13.359753     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.196288  0.196288  0.0  0.0  0.0 | 0.010341  0.010341    9.75654  0.0   0.0 |      0.0       0.0    4.75654  0.0   0.0 |  0.59014   0.59014  5.660762  0.0   0.0 |      0.0       0.0  4.660762  0.0   0.0 | 0.0  0.0  0.0  0.211567  0.0 | 0.702383 |  2.173223 | 2.311362 | 2.658738 | 11.078073 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.248862  148.248862  0.0  0.0  0.0 |   0.01594    0.01594  19.030962  0.0  0.0 |  5.616328   5.616328  56.334534  0.0  0.0 | 139.774231 | 11.078073 |
    | 03/01 20:00 |  0.0 | 13.4 | 14.0  14.0  14.0  14.0  14.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.734918  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.734918     0.0        0.0 |       0.0        0.0  11.734918     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.062473  0.062473  0.0  0.0  0.0 | 0.006272  0.006272   9.093147  0.0   0.0 |      0.0       0.0   4.093147  0.0   0.0 | 0.534768  0.534768   5.57849  0.0   0.0 |      0.0       0.0   4.57849  0.0   0.0 | 0.0  0.0  0.0  0.067425  0.0 | 0.706519 |  2.001147 | 2.218748 | 2.582332 | 10.759717 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.186389  148.186389  0.0  0.0  0.0 |  0.009668   0.009668  17.579585  0.0  0.0 |  5.087832   5.087832  55.270702  0.0  0.0 |  140.59725 | 10.759717 |
    | 03/01 21:00 |  0.0 | 12.4 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.832232  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.832232     0.0        0.0 |       0.0        0.0  10.832232     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.061167  0.061167  0.0  0.0  0.0 | 0.003804  0.003804   8.468385  0.0   0.0 |      0.0       0.0   3.468385  0.0   0.0 | 0.484355  0.484355  5.452814  0.0   0.0 |      0.0       0.0  4.452814  0.0   0.0 | 0.0  0.0  0.0  0.066044  0.0 | 0.710338 |  1.827969 | 2.073274 |  2.45982 | 10.249251 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.125222  148.125222  0.0  0.0  0.0 |  0.005864   0.005864  16.475047  0.0  0.0 |  4.607281   4.607281  53.833458  0.0  0.0 | 141.357308 | 10.249251 |
    | 03/01 22:00 |  0.0 | 11.6 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.110083  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.110083     0.0        0.0 |       0.0        0.0  10.110083     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.060106  0.060106  0.0  0.0  0.0 | 0.002307  0.002307   7.986453  0.0   0.0 |      0.0       0.0   2.986453  0.0   0.0 | 0.438552  0.438552  5.299983  0.0   0.0 |      0.0       0.0  4.299983  0.0   0.0 | 0.0  0.0  0.0  0.064924  0.0 | 0.713828 |  1.681485 |  1.90713 | 2.319085 |  9.662856 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 148.065116  148.065116  0.0  0.0  0.0 |  0.003557   0.003557  15.612226  0.0  0.0 |  4.171036   4.171036  52.219945  0.0  0.0 | 142.051842 |  9.662856 |
    | 03/01 23:00 |  0.0 | 11.0 | 11.6  11.6  11.6  11.6  11.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451343  0.451343  0.451343  0.0  0.451343 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.902686  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.568472  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.568472     0.0        0.0 |       0.0        0.0   9.568472     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.059296  0.059296  0.0  0.0  0.0 | 0.001399  0.001399   7.614125  0.0   0.0 |      0.0       0.0   2.614125  0.0   0.0 | 0.396994  0.396994  5.136308  0.0   0.0 |      0.0       0.0  4.136308  0.0   0.0 | 0.0  0.0  0.0  0.064076  0.0 | 0.716993 |  1.557792 | 1.751464 | 2.187129 |  9.113037 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  148.00582   148.00582  0.0  0.0  0.0 |  0.002157   0.002157  14.952447  0.0  0.0 |  3.775441   3.775441  50.561455  0.0  0.0 | 142.681544 |  9.113037 |
    | 04/01 00:00 |  0.0 | 10.5 | 11.1  11.1  11.1  11.1  11.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.121265  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.121265     0.0        0.0 |       0.0        0.0   9.121265     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.058613  0.058613  0.0  0.0  0.0 | 0.000849  0.000849   7.323336  0.0   0.0 |      0.0       0.0   2.323336  0.0   0.0 | 0.359322  0.359322  4.972374  0.0   0.0 |      0.0       0.0  3.972374  0.0   0.0 | 0.0  0.0  0.0  0.063363  0.0 | 0.719845 |  1.452856 | 1.617202 |  2.07343 |  8.639294 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.947208  147.947208  0.0  0.0  0.0 |  0.001308   0.001308   14.42704  0.0  0.0 |  3.416968   3.416968  48.940043  0.0  0.0 | 143.249148 |  8.639294 |
    | 04/01 01:00 |  0.0 | 11.7 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498     0.0        0.0 |       0.0        0.0   10.20498     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.060164  0.060164  0.0  0.0  0.0 | 0.000515  0.000515   7.356614  0.0   0.0 |      0.0       0.0   2.356614  0.0   0.0 | 0.325192  0.325192  4.826977  0.0   0.0 |      0.0       0.0  3.826977  0.0   0.0 | 0.0  0.0  0.0  0.065065  0.0 | 0.722416 |  1.426982 |  1.50774 | 1.980964 |  8.254015 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.887044  147.887044  0.0  0.0  0.0 |  0.000794   0.000794  14.918792  0.0  0.0 |  3.092291   3.092291  47.642703  0.0  0.0 | 143.760841 |  8.254015 |
    | 04/01 02:00 |  0.0 | 11.9 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599     0.0        0.0 |       0.0        0.0  10.385599     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.060398  0.060398  0.0  0.0  0.0 | 0.000312  0.000312    7.54526  0.0   0.0 |      0.0       0.0    2.54526  0.0   0.0 | 0.294285  0.294285  4.718226  0.0   0.0 |      0.0       0.0  3.718226  0.0   0.0 | 0.0  0.0  0.0  0.065345  0.0 | 0.724759 |   1.44542 |   1.4494 | 1.932588 |   8.05245 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.826646  147.826646  0.0  0.0  0.0 |  0.000481   0.000481   15.21387  0.0  0.0 |  2.798317   2.798317  46.751511  0.0  0.0 |  144.22698 |   8.05245 |
    | 04/01 03:00 |  1.3 | 11.2 | 11.8  11.8  11.8  11.8  11.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  1.573   1.573   1.573   1.573   1.573 | 0.054992  0.054992  0.0  0.0  0.054992 |       0.0        0.0   1.573   1.573        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.753432  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.326432   1.573        0.0 |       0.0        0.0  11.326432   1.573        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.025404  0.025404  0.0  0.0  0.0 | 0.000189  0.000189   7.811579  0.0   0.0 |      0.0       0.0   2.811579  0.0   0.0 | 0.266304  0.266304  4.649924  0.0   0.0 |      0.0       0.0  3.649924  0.0   0.0 | 0.0  0.0  0.0  0.054992  0.0 | 0.728057 |  1.491116 | 1.444069 | 1.931047 |  8.046028 | 1.518008  1.518008  0.0  0.0  1.518008 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.801242  147.801242  0.0  0.0  0.0 |  0.000292   0.000292  15.917143  0.0  0.0 |  2.532202   2.532202  46.263241  0.0  0.0 | 144.883389 |  8.046028 |
    | 04/01 04:00 |  0.0 | 11.1 | 11.7  11.7  11.7  11.7  11.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 0.064218  0.064218  0.0  0.0  0.064218 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.663122  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   9.663122     0.0        0.0 |       0.0        0.0   9.663122     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.029661  0.029661  0.0  0.0  0.0 | 0.000115  0.000115   7.727907  0.0   0.0 |      0.0       0.0   2.727907  0.0   0.0 | 0.240976  0.240976  4.601752  0.0   0.0 |      0.0       0.0  3.601752  0.0   0.0 | 0.0  0.0  0.0  0.064218  0.0 | 0.730076 |  1.460691 | 1.468212 | 1.953855 |  8.141061 |  1.45379   1.45379  0.0  0.0   1.45379 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.771581  147.771581  0.0  0.0  0.0 |  0.000177   0.000177  15.124451  0.0  0.0 |  2.291341   2.291341  45.787644  0.0  0.0 | 145.285142 |  8.141061 |
    | 04/01 05:00 |  0.0 | 11.9 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 0.065345  0.065345  0.0  0.0  0.065345 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599     0.0        0.0 |       0.0        0.0  10.385599     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.030176  0.030176  0.0  0.0  0.0 |  0.00007   0.00007   7.610261  0.0   0.0 |      0.0       0.0   2.610261  0.0   0.0 | 0.218053  0.218053  4.553138  0.0   0.0 |      0.0       0.0  3.553138  0.0   0.0 | 0.0  0.0  0.0  0.065345  0.0 | 0.731958 |  1.422323 | 1.467037 | 1.954593 |  8.144137 | 1.388445  1.388445  0.0  0.0  1.388445 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.741405  147.741405  0.0  0.0  0.0 |  0.000107   0.000107  15.289528  0.0  0.0 |  2.073357   2.073357   45.29163  0.0  0.0 | 145.659698 |  8.144137 |
    | 04/01 06:00 |  0.0 | 12.2 | 12.8  12.8  12.8  12.8  12.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 | 0.238876  0.238876  0.0  0.0  0.238876 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.656527  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.656527     0.0        0.0 |       0.0        0.0  10.656527     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.110287  0.110287  0.0  0.0  0.0 | 0.000042  0.000042    7.71227  0.0   0.0 |      0.0       0.0    2.71227  0.0   0.0 | 0.197308  0.197308  4.512958  0.0   0.0 |      0.0       0.0  3.512958  0.0   0.0 | 0.0  0.0  0.0  0.238876  0.0 | 0.733597 |  1.436591 | 1.441823 |  1.93427 |   8.05946 | 1.149568  1.149568  0.0  0.0  1.149568 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 147.631118  147.631118  0.0  0.0  0.0 |  0.000065   0.000065  15.521516  0.0  0.0 |  1.876091   1.876091  44.977983  0.0  0.0 | 145.985879 |   8.05946 |
    | 04/01 07:00 |  0.7 | 11.8 | 12.4  12.4  12.4  12.4  12.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.847   0.847   0.847   0.847   0.847 | 0.984764  0.984764  0.0  0.0  0.984764 |       0.0        0.0   0.847   0.847        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.295289  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.142289   0.847        0.0 |       0.0        0.0  11.142289   0.847        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.454318  0.454318  0.0  0.0  0.0 | 0.000026  0.000026   7.874943  0.0   0.0 |      0.0       0.0   2.874943  0.0   0.0 | 0.178535  0.178535  4.492148  0.0   0.0 |      0.0       0.0  3.492148  0.0   0.0 | 0.0  0.0  0.0  0.984764  0.0 | 0.735218 |  1.469329 | 1.436256 | 1.930958 |  8.045659 | 1.011805  1.011805  0.0  0.0  1.011805 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   147.1768    147.1768  0.0  0.0  0.0 |   0.00004    0.00004  15.913919  0.0  0.0 |  1.697582   1.697582  44.868629  0.0  0.0 | 146.308356 |  8.045659 |
    | 04/01 08:00 |  0.4 | 11.4 | 12.0  12.0  12.0  12.0  12.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   0.484   0.484   0.484   0.484 | 1.495805  1.495805  0.0  0.0  1.495805 |       0.0        0.0   0.484   0.484        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   9.934051  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.418051   0.484        0.0 |       0.0        0.0  10.418051   0.484        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.502702  1.502702  0.0  0.0  0.0 | 0.000016  0.000016    7.86575  0.0   0.0 |      0.0       0.0    2.86575  0.0   0.0 | 0.161547  0.161547  4.481807  0.0   0.0 |      0.0       0.0  3.481807  0.0   0.0 | 0.0  0.0  0.0  2.344557  0.0 | 0.735534 |  1.464821 | 1.454211 | 1.946815 |  8.111729 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 145.674097  145.674097  0.0  0.0  0.0 |  0.000024   0.000024  15.600471  0.0  0.0 |  1.536051   1.536051  44.770765  0.0  0.0 | 146.371267 |  8.111729 |
    | 04/01 09:00 |  0.1 | 11.6 | 12.2  12.2  12.2  12.2  12.2 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.121   0.121   0.121   0.121   0.121 |    0.121     0.121  0.0  0.0     0.121 |       0.0        0.0   0.121   0.121        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.11467  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.23567   0.121        0.0 |       0.0        0.0   10.23567   0.121        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.595867  3.595867  0.0  0.0  0.0 | 0.000009  0.000009   7.733134  0.0   0.0 |      0.0       0.0   2.733134  0.0   0.0 | 0.146175  0.146175  4.466727  0.0   0.0 |      0.0       0.0  3.466727  0.0   0.0 | 0.0  0.0  0.0  3.949492  0.0 | 0.734377 |  1.430737 | 1.461501 | 1.952053 |  8.133555 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  142.07823   142.07823  0.0  0.0  0.0 |  0.000015   0.000015  15.369873  0.0  0.0 |  1.389885   1.389885  44.570446  0.0  0.0 | 146.141088 |  8.133555 |
    | 04/01 10:00 |  0.4 | 13.0 | 13.6  13.6  13.6  13.6  13.6 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   0.484   0.484   0.484   0.484 |    0.484     0.484  0.0  0.0     0.484 |       0.0        0.0   0.484   0.484        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  11.379004  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  11.863004   0.484        0.0 |       0.0        0.0  11.863004   0.484        0.0 | 0.0  0.0  0.0  0.0        0.0 |  4.66112   4.66112  0.0  0.0  0.0 | 0.000006  0.000006   7.959582  0.0   0.0 |      0.0       0.0   2.959582  0.0   0.0 | 0.132265  0.132265  4.459175  0.0   0.0 |      0.0       0.0  3.459175  0.0   0.0 | 0.0  0.0  0.0  5.249074  0.0 | 0.732505 |  1.481252 | 1.448974 | 1.939448 |  8.081035 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 137.417111  137.417111  0.0  0.0  0.0 |  0.000009   0.000009  16.313712  0.0  0.0 |  1.257625   1.257625  44.611678  0.0  0.0 | 145.768423 |  8.081035 |
    | 04/01 11:00 |  0.0 | 17.1 | 17.7  17.7  17.7  17.7  17.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.081695  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.081695     0.0        0.0 |       0.0        0.0  15.081695     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.549421  5.549421  0.0  0.0  0.0 | 0.000003  0.000003   8.849938  0.0   0.0 |      0.0       0.0   3.849938  0.0   0.0 | 0.119679  0.119679  4.504605  0.0   0.0 |      0.0       0.0  3.504605  0.0   0.0 | 0.0  0.0  0.0  6.461403  0.0 | 0.729438 |  1.697202 | 1.477888 | 1.961645 |  8.173522 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  131.86769   131.86769  0.0  0.0  0.0 |  0.000005   0.000005  18.695532  0.0  0.0 |   1.13795    1.13795  45.452407  0.0  0.0 | 145.158137 |  8.173522 |
    | 04/01 12:00 |  0.0 | 18.2 | 18.8  18.8  18.8  18.8  18.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  16.075101  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  16.075101     0.0        0.0 |       0.0        0.0  16.075101     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 5.325221  5.325221  0.0  0.0  0.0 | 0.000002  0.000002   9.785463  0.0   0.0 |      0.0       0.0   4.785463  0.0   0.0 |  0.10829   0.10829  4.624611  0.0   0.0 |      0.0       0.0  3.624611  0.0   0.0 | 0.0  0.0  0.0   6.46129  0.0 | 0.726478 |  1.940786 | 1.618328 | 2.080598 |  8.669157 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 126.542469  126.542469  0.0  0.0  0.0 |  0.000003   0.000003  20.199706  0.0  0.0 |  1.029662   1.029662  46.988647  0.0  0.0 |  144.56922 |  8.669157 |
    | 04/01 13:00 |  0.0 | 22.4 | 23.0  23.0  23.0  23.0  23.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.868102  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.868102     0.0        0.0 |       0.0        0.0  19.868102     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 6.525426  6.525426  0.0  0.0  0.0 | 0.000001  0.000001  10.958556  0.0   0.0 |      0.0       0.0   5.958556  0.0   0.0 | 0.097985  0.097985   4.81878  0.0   0.0 |      0.0       0.0   3.81878  0.0   0.0 | 0.0  0.0  0.0  8.250733  0.0 | 0.722431 |  2.256308 | 1.840725 | 2.269564 |  9.456515 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 120.017043  120.017043  0.0  0.0  0.0 |  0.000002   0.000002  23.150695  0.0  0.0 |  0.931678   0.931678  49.309643  0.0  0.0 |   143.7637 |  9.456515 |
    | 04/01 14:00 |  0.0 | 21.4 | 22.0  22.0  22.0  22.0  22.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  18.965006  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  18.965006     0.0        0.0 |       0.0        0.0  18.965006     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 |  5.41968   5.41968  0.0  0.0  0.0 | 0.000001  0.000001  11.725132  0.0   0.0 |      0.0       0.0   6.725132  0.0   0.0 | 0.088661  0.088661  5.065039  0.0   0.0 |      0.0       0.0  4.065039  0.0   0.0 | 0.0  0.0  0.0  7.225213  0.0 | 0.719369 |  2.490039 |  2.11566 | 2.504984 | 10.437431 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 114.597363  114.597363  0.0  0.0  0.0 |  0.000001   0.000001  23.665438  0.0  0.0 |  0.843017   0.843017  51.904696  0.0  0.0 | 143.154519 | 10.437431 |
    | 04/01 15:00 |  0.0 | 21.8 | 22.4  22.4  22.4  22.4  22.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.326244  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.326244     0.0        0.0 |       0.0        0.0  19.326244     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 4.335845  4.335845  0.0  0.0  0.0 |      0.0       0.0  11.954268  0.0   0.0 |      0.0       0.0   6.954268  0.0   0.0 | 0.080224  0.080224  5.310971  0.0   0.0 |      0.0       0.0  4.310971  0.0   0.0 | 0.0  0.0  0.0  6.053675  0.0 | 0.717397 |   2.59967 | 2.369159 | 2.722842 | 11.345174 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 110.261519  110.261519  0.0  0.0  0.0 |  0.000001   0.000001  24.083147  0.0  0.0 |  0.762794   0.762794  54.237023  0.0  0.0 | 142.761949 | 11.345174 |
    | 04/01 16:00 |  0.0 | 22.2 | 22.8  22.8  22.8  22.8  22.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  19.687483  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  19.687483     0.0        0.0 |       0.0        0.0  19.687483     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 3.342105  3.342105  0.0  0.0  0.0 |      0.0       0.0  12.152735  0.0   0.0 |      0.0       0.0   7.152735  0.0   0.0 | 0.072589  0.072589  5.531654  0.0   0.0 |      0.0       0.0  4.531654  0.0   0.0 | 0.0  0.0  0.0  4.849713  0.0 | 0.716506 |  2.696397 | 2.537016 | 2.867486 | 11.947859 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 106.919414  106.919414  0.0  0.0  0.0 |       0.0        0.0   24.46516  0.0  0.0 |  0.690205   0.690205   56.32645  0.0  0.0 | 142.584646 | 11.947859 |
    | 04/01 17:00 |  0.0 | 20.1 | 20.7  20.7  20.7  20.7  20.7 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  17.790982  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  17.790982     0.0        0.0 |       0.0        0.0  17.790982     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 1.517107  1.517107  0.0  0.0  0.0 |      0.0       0.0  11.924632  0.0   0.0 |      0.0       0.0   6.924632  0.0   0.0 | 0.065682  0.065682  5.710347  0.0   0.0 |      0.0       0.0  4.710347  0.0   0.0 | 0.0  0.0  0.0   2.27028  0.0 | 0.717631 |  2.684995 | 2.642557 | 2.960005 | 12.333353 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 105.402307  105.402307  0.0  0.0  0.0 |       0.0        0.0  23.406878  0.0  0.0 |  0.624523   0.624523  57.830389  0.0  0.0 | 142.808559 | 12.333353 |
    | 04/01 18:00 |  0.0 | 17.8 | 18.4  18.4  18.4  18.4  18.4 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  15.713862  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  15.713862     0.0        0.0 |       0.0        0.0  15.713862     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.664106  0.664106  0.0  0.0  0.0 |      0.0       0.0  11.208086  0.0   0.0 |      0.0       0.0   6.208086  0.0   0.0 | 0.059431  0.059431  5.813102  0.0   0.0 |      0.0       0.0  4.813102  0.0   0.0 | 0.0  0.0  0.0  1.008109  0.0 | 0.719742 |  2.543351 | 2.670467 | 2.986164 | 12.442351 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.738201  104.738201  0.0  0.0  0.0 |       0.0        0.0  21.704567  0.0  0.0 |  0.565092   0.565092  58.412271  0.0  0.0 | 143.228672 | 12.442351 |
    | 04/01 19:00 |  0.0 | 15.2 | 15.8  15.8  15.8  15.8  15.8 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  13.365814  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  13.365814     0.0        0.0 |       0.0        0.0  13.365814     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.166111  0.166111  0.0  0.0  0.0 |      0.0       0.0  10.238154  0.0   0.0 |      0.0       0.0   5.238154  0.0   0.0 | 0.053776  0.053776  5.820422  0.0   0.0 |      0.0       0.0  4.820422  0.0   0.0 | 0.0  0.0  0.0  0.253754  0.0 | 0.722371 |   2.32121 | 2.588003 | 2.917149 | 12.154787 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.572091  104.572091  0.0  0.0  0.0 |       0.0        0.0  19.594073  0.0  0.0 |  0.511317   0.511317  58.009581  0.0  0.0 | 143.751854 | 12.154787 |
    | 04/01 20:00 |  0.0 | 14.5 | 15.1  15.1  15.1  15.1  15.1 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  12.733647  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  12.733647     0.0        0.0 |       0.0        0.0  12.733647     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.045045  0.045045  0.0  0.0  0.0 |      0.0       0.0    9.45483  0.0   0.0 |      0.0       0.0    4.45483  0.0   0.0 | 0.048658  0.048658  5.747244  0.0   0.0 |      0.0       0.0  4.747244  0.0   0.0 | 0.0  0.0  0.0  0.068921  0.0 | 0.725024 |  2.123556 | 2.413913 | 2.768747 | 11.536447 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.527046  104.527046  0.0  0.0  0.0 |       0.0        0.0  18.418059  0.0  0.0 |  0.462659   0.462659  56.969924  0.0  0.0 | 144.279816 | 11.536447 |
    | 04/01 21:00 |  0.0 | 12.4 | 13.0  13.0  13.0  13.0  13.0 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.837146  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.837146     0.0        0.0 |       0.0        0.0  10.837146     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.043146  0.043146  0.0  0.0  0.0 |      0.0       0.0   8.734297  0.0   0.0 |      0.0       0.0   3.734297  0.0   0.0 | 0.044028  0.044028  5.619274  0.0   0.0 |      0.0       0.0  4.619274  0.0   0.0 | 0.0  0.0  0.0  0.066044  0.0 | 0.727514 |  1.927747 | 2.213938 | 2.597759 | 10.823996 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   104.4839    104.4839  0.0  0.0  0.0 |       0.0        0.0  16.786611  0.0  0.0 |  0.418631   0.418631  55.465673  0.0  0.0 | 144.775301 | 10.823996 |
    | 04/01 22:00 |  0.0 | 11.7 | 12.3  12.3  12.3  12.3  12.3 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0   10.20498  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0   10.20498     0.0        0.0 |       0.0        0.0   10.20498     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.042489  0.042489  0.0  0.0  0.0 |      0.0       0.0   8.102381  0.0   0.0 |      0.0       0.0   3.102381  0.0   0.0 | 0.039838  0.039838  5.453346  0.0   0.0 |      0.0       0.0  4.453346  0.0   0.0 | 0.0  0.0  0.0  0.065065  0.0 | 0.729801 |  1.743629 | 2.017045 | 2.429253 | 10.121888 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.441411  104.441411  0.0  0.0  0.0 |       0.0        0.0  15.786829  0.0  0.0 |  0.378793   0.378793   53.66136  0.0  0.0 | 145.230389 | 10.121888 |
    | 04/01 23:00 |  0.0 | 11.9 | 12.5  12.5  12.5  12.5  12.5 | 1.0  1.0  1.0  1.0       1.0 | 1.1  1.1  1.1  1.1  1.1 | 0.0  0.0  0.0  0.0  0.0 | 0.451548  0.451548  0.451548  0.0  0.451548 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.903096  0.0   0.0 |    0.0     0.0     0.0     0.0     0.0 |      0.0       0.0  0.0  0.0       0.0 |       0.0        0.0     0.0     0.0        0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  10.385599  0.0     0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |       0.0        0.0  10.385599     0.0        0.0 |       0.0        0.0  10.385599     0.0        0.0 | 0.0  0.0  0.0  0.0        0.0 | 0.042655  0.042655  0.0  0.0  0.0 |      0.0       0.0   7.819612  0.0   0.0 |      0.0       0.0   2.819612  0.0   0.0 | 0.036047  0.036047  5.276572  0.0   0.0 |      0.0       0.0  4.276572  0.0   0.0 | 0.0  0.0  0.0  0.065345  0.0 | 0.731875 |  1.637581 | 1.833184 | 2.271842 |  9.466009 |      0.0       0.0  0.0  0.0       0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 104.398756  104.398756  0.0  0.0  0.0 |       0.0        0.0  15.533203  0.0  0.0 |  0.342746   0.342746  51.927829  0.0  0.0 | 145.643047 |  9.466009 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

.. _hland_96c_snow_redistribution:

snow redistribution
___________________

Regarding snow redistribution, all functionalities of |hland_96c| are identical with the
ones of |hland_96|.  Hence, all snow data in the following table agrees with the
:ref:`hland_96_snow_redistribution_1` example of |hland_96|:

.. integration-test::

    >>> zonez(0.0, 10.0, 30.0, 0.0, 20.0)
    >>> smax(200.0)
    >>> sred(n_zones=1)
    >>> inputs.t.series = -0.4
    >>> pcorr *= 2.0
    >>> pcalt(0.05)
    >>> test("hland_96c_snow_redistribution",
    ...      axis1=(states.sp, states.wc), axis2=(factors.tc, fluxes.pc))
    |        date |    p |    t |                         tc |                     fracrain |                      rfc |                      sfc |                                       cfact |                                                 swe |                          gact |                                        pc |                                ei |                                          tf |                                                 spl |                                             wcl |                                                 spg |                                             wcg |                     glmelt |                          melt |                               refr |                                    in_ |                                    r |                      sr |                      ea |                               qvs1 |                     qab1 |                               qvs2 |                          qab2 |                      el |       q1 |     inrc |    outrc |       rt |       qt |                                  ic |                                                  sp |                                                  wc |                                    sm |                               bw1 |                                bw2 |        lz |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |     0.05 |      0.0 | 0.160107 | 0.185426 | 0.772608 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |      9.95 | 0.772608 |
    | 01/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.04975 |      0.0 | 0.073779 | 0.110375 | 0.459897 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |   9.90025 | 0.459897 |
    | 01/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049501 |      0.0 | 0.014282 | 0.058579 | 0.244077 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.850749 | 0.244077 |
    | 01/01 03:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049254 |      0.0 | 0.001674 | 0.047421 | 0.197586 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.801495 | 0.197586 |
    | 01/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049007 |      0.0 | 0.000147 | 0.045868 | 0.191116 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.752488 | 0.191116 |
    | 01/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048762 |      0.0 | 0.000011 | 0.045521 | 0.189671 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.703725 | 0.189671 |
    | 01/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048519 |      0.0 | 0.000001 | 0.045285 | 0.188686 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.655206 | 0.188686 |
    | 01/01 07:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048276 |      0.0 |      0.0 | 0.045058 |  0.18774 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |   9.60693 |  0.18774 |
    | 01/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048035 |      0.0 |      0.0 | 0.044832 | 0.186801 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.558896 | 0.186801 |
    | 01/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047794 |      0.0 |      0.0 | 0.044608 | 0.185867 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.511101 | 0.185867 |
    | 01/01 10:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047556 |      0.0 |      0.0 | 0.044385 | 0.184938 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.463546 | 0.184938 |
    | 01/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047318 |      0.0 |      0.0 | 0.044163 | 0.184013 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.416228 | 0.184013 |
    | 01/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047081 |      0.0 |      0.0 | 0.043942 | 0.183093 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.369147 | 0.183093 |
    | 01/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046846 |      0.0 |      0.0 | 0.043723 | 0.182178 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.322301 | 0.182178 |
    | 01/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046612 |      0.0 |      0.0 | 0.043504 | 0.181267 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |   9.27569 | 0.181267 |
    | 01/01 15:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0    1.144   0.242       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046551 |      0.0 |      0.0 | 0.043448 | 0.181033 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |   9.26371 | 0.181033 |
    | 01/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046319 |      0.0 |      0.0 | 0.043231 | 0.180128 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.217391 | 0.180128 |
    | 01/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046087 |      0.0 |      0.0 | 0.043014 | 0.179227 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.171304 | 0.179227 |
    | 01/01 18:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    1.224502        8.58  0.0       4.435 | 0.0  0.0  0.901953  0.0   0.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |    0.0     1.894    7.436   1.573     4.435 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0   0.669498  0.0   1.573  0.0 |      0.0  0.167375  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.035661  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0  0.001725  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046983 |      0.0 |      0.0 |  0.04385 |  0.18271 | 1.815       2.0  0.0  0.0       2.0 |        0.0    0.874644        8.58  0.0       4.435 |        0.0    0.349858         0.0  0.0         0.0 |      100.0  100.502124  0.0  0.0  0.0 |      0.0  0.131714  0.0  0.0  0.0 |       0.0  0.033936  0.0  0.0  0.0 |  9.349529 |  0.18271 |
    | 01/01 19:00 |  5.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   10.249462      40.612  0.0      28.459 | 0.0  0.0  0.901953  0.0   0.0 |  6.776  14.5376   32.032   6.776   24.024 | 0.0       0.0  0.0  0.0       0.0 |  6.591   14.5376   32.032   6.776    24.024 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     6.591    5.51264  0.0   6.776  0.0 |  1.64775  1.392035  0.0   6.776  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.351072  0.348414  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.016983  0.020084  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.051647 |      0.0 |      0.0 | 0.048204 | 0.200848 |   2.0       2.0  0.0  0.0       2.0 |        0.0    7.321044      40.612  0.0      28.459 |        0.0    2.928418         0.0  0.0         0.0 |  104.94325  104.622729  0.0  0.0  0.0 | 1.296678  1.175334  0.0  0.0  0.0 |  0.334089  0.362266  0.0  0.0  0.0 | 10.277686 | 0.200848 |
    | 01/01 20:00 |  2.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   14.950102        57.2  0.0        40.9 | 0.0  0.0  0.901953  0.0   0.0 |  3.509   7.5284   16.588   3.509   12.441 | 0.0       0.0  0.0  0.0       0.0 |  3.509    7.5284   16.588   3.509    12.441 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     3.509    2.82776  0.0   3.509  0.0 | 0.966123  0.773811  0.0   3.509  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.716047  0.627327  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.066431  0.064821  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.054106 |      0.0 |      0.0 | 0.050499 | 0.210413 |   2.0       2.0  0.0  0.0       2.0 |        0.0   10.678644        57.2  0.0        40.9 |        0.0    4.271458         0.0  0.0         0.0 | 107.486127  106.676678  0.0  0.0  0.0 | 1.546755  1.321818  0.0  0.0  0.0 |  0.983705  0.924773  0.0  0.0  0.0 | 10.767111 | 0.210413 |
    | 01/01 21:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   22.853942      85.228  0.0      61.921 | 0.0  0.0  0.901953  0.0   0.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     5.929    4.81656  0.0   5.929  0.0 | 1.712483  1.370301  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.973464  0.812053  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.140702  0.127286  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.058504 |      0.0 |      0.0 | 0.054603 | 0.227514 |   2.0       2.0  0.0  0.0       2.0 |        0.0   16.324244      85.228  0.0      61.921 |        0.0    6.529698         0.0  0.0         0.0 | 111.702644  110.122937  0.0  0.0  0.0 | 2.285773  1.880066  0.0  0.0  0.0 |  1.816467   1.60954  0.0  0.0  0.0 | 11.642226 | 0.227514 |
    | 01/01 22:00 | 10.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   39.886902      145.86  0.0     107.395 | 0.0  0.0  0.901953  0.0   0.0 | 12.826  27.5176   60.632  12.826   45.474 | 0.0       0.0  0.0  0.0       0.0 | 12.826   27.5176   60.632  12.826    45.474 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |    12.826   10.48464  0.0  12.826  0.0 | 4.000904  3.178697  0.0  12.826  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.75182  1.417006  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.257602  0.221714  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.068149 |      0.0 |      0.0 | 0.063606 | 0.265025 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.490644      145.86  0.0     107.395 |        0.0   11.396258         0.0  0.0         0.0 |  120.52774  117.428881  0.0  0.0  0.0 | 4.534858  3.641757  0.0  0.0  0.0 |  3.310684  2.804831  0.0  0.0  0.0 |  13.56171 | 0.265025 |
    | 01/01 23:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   40.103062     146.432  0.0     107.824 | 0.0  0.0  0.901953  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121    0.04344  0.0   0.121  0.0 | 0.043944  0.014975  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.79369   1.43611  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.401822  0.336386  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.069093 |      0.0 |      0.0 | 0.064487 | 0.268695 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.645044     146.432  0.0     107.824 |        0.0   11.458018         0.0  0.0         0.0 | 120.604796  117.457345  0.0  0.0  0.0 | 2.785112  2.220622  0.0  0.0  0.0 |  4.702552  3.904556  0.0  0.0  0.0 |  13.74952 | 0.268695 |
    | 02/01 00:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   41.280182     150.436  0.0     110.827 | 0.0  0.0  0.902305  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.847    0.64008  0.0   0.847  0.0 | 0.308001  0.220767  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.161479  0.920784  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.503693   0.41611  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.070846 |      0.0 |      0.0 | 0.066123 | 0.275514 |   2.0       2.0  0.0  0.0       2.0 |        0.0   29.485844     150.436  0.0     110.827 |        0.0   11.794338         0.0  0.0         0.0 | 121.143795  117.876658  0.0  0.0  0.0 | 1.931634  1.520606  0.0  0.0  0.0 |  5.360339  4.409229  0.0  0.0  0.0 | 14.098453 | 0.275514 |
    | 02/01 01:00 |  3.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   46.140982     167.596  0.0     123.697 | 0.0  0.0  0.902305  0.0   0.0 |   3.63    7.788    17.16    3.63    12.87 | 0.0       0.0  0.0  0.0       0.0 |   3.63     7.788    17.16    3.63     12.87 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |      3.63     2.9272  0.0    3.63  0.0 | 1.331831  1.016829  0.0    3.63  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   1.0438  0.814959  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.560597  0.459017  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.074742 |      0.0 |      0.0 | 0.069759 | 0.290663 |   2.0       2.0  0.0  0.0       2.0 |        0.0   32.957844     167.596  0.0     123.697 |        0.0   13.183138         0.0  0.0         0.0 | 123.441964  119.787029  0.0  0.0  0.0 | 2.219664  1.722476  0.0  0.0  0.0 |  5.843543  4.765172  0.0  0.0  0.0 | 14.873643 | 0.290663 |
    | 02/01 02:00 |  2.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   49.560342     179.608  0.0     132.706 | 0.0  0.0  0.902305  0.0   0.0 |  2.541   5.4516   12.012   2.541    9.009 | 0.0       0.0  0.0  0.0       0.0 |  2.541    5.4516   12.012   2.541     9.009 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     2.541    2.03224  0.0   2.541  0.0 | 0.967989  0.729012  0.0   2.541  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.079611  0.833066  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.608312  0.493765  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.077975 |      0.0 |      0.0 | 0.072777 | 0.303236 |   2.0       2.0  0.0  0.0       2.0 |        0.0   35.400244     179.608  0.0     132.706 |        0.0   14.160098         0.0  0.0         0.0 | 125.014975  121.090257  0.0  0.0  0.0 | 2.108042  1.618422  0.0  0.0  0.0 |  6.314841  5.104472  0.0  0.0  0.0 | 15.516998 | 0.303236 |
    | 02/01 03:00 | 10.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   99.386482       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 | 12.584  26.9984   59.488  12.584   44.616 | 0.0       0.0  0.0  0.0       0.0 | 12.584   26.9984   59.488  12.584    44.616 |        0.0         0.0      39.096  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0     23.6525         0.0  0.0      22.678 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |    12.584    0.82476  0.0  12.584  0.0 | 4.916803  0.302333  0.0  12.584  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.87703  0.701215  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.691736  0.519675  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.088551 |      0.0 |      0.0 | 0.082648 | 0.344366 |   2.0       2.0  0.0  0.0       2.0 |        0.0   70.990344       200.0  0.0       200.0 |        0.0   28.396138         0.0  0.0         0.0 | 132.682172  121.612683  0.0  0.0  0.0 | 5.147815   1.21954  0.0  0.0  0.0 |  7.500135  5.286012  0.0  0.0  0.0 | 17.621688 | 0.344366 |
    | 02/01 04:00 |  3.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  127.241232       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.235    9.086    20.02   4.235   15.015 | 0.0       0.0  0.0  0.0       0.0 |  4.235     9.086    20.02   4.235    15.015 |        0.0         0.0       20.02  0.0      15.015 |       0.0        0.0        0.0  0.0        0.0 |        0.0    18.76875         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.235        0.0  0.0   4.235  0.0 | 1.863883       0.0  0.0   4.235  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.422628  0.479852  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.830925  0.526243  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.093369 |      0.0 |      0.0 | 0.087144 | 0.363102 |   2.0       2.0  0.0  0.0       2.0 |        0.0   93.803094       200.0  0.0       200.0 |        0.0   33.438138         0.0  0.0         0.0 |  135.05329  121.612683  0.0  0.0  0.0 | 4.589069  0.739689  0.0  0.0  0.0 |  9.091839  5.239621  0.0  0.0  0.0 | 18.580433 | 0.363102 |
    | 02/01 05:00 |  3.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  154.300132       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.114   8.8264   19.448   4.114   14.586 | 0.0       0.0  0.0  0.0       0.0 |  4.114    8.8264   19.448   4.114    14.586 |        0.0         0.0      19.448  0.0      14.586 |       0.0        0.0        0.0  0.0        0.0 |        0.0     18.2325         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.114        0.0  0.0   4.114  0.0 | 1.875921       0.0  0.0   4.114  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.205344  0.291045  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.971844  0.512695  0.0  0.0   0.0 |  0.00443  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.098309 | 0.001704 | 0.000122 |  0.09186 |  0.38275 |   2.0       2.0  0.0  0.0       2.0 |        0.0  115.965194       200.0  0.0       200.0 |        0.0   38.334938         0.0  0.0         0.0 | 137.291368  121.612683  0.0  0.0  0.0 | 4.259646  0.448644  0.0  0.0  0.0 | 10.320909  5.017971  0.0  0.0  0.0 | 19.563411 |  0.38275 |
    | 02/01 06:00 |  1.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  163.850332       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.452   3.1152    6.864   1.452    5.148 | 0.0       0.0  0.0  0.0       0.0 |  1.452    3.1152    6.864   1.452     5.148 |        0.0         0.0       6.864  0.0       5.148 |       0.0        0.0        0.0  0.0        0.0 |        0.0       6.435         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     1.452        0.0  0.0   1.452  0.0 | 0.684216       0.0  0.0   1.452  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.82182  0.176528  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.067569  0.486062  0.0  0.0   0.0 | 0.067569  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.101455 | 0.025988 | 0.002708 | 0.097038 | 0.404326 |   2.0       2.0  0.0  0.0       2.0 |        0.0  123.812994       200.0  0.0       200.0 |        0.0   40.037338         0.0  0.0         0.0 | 138.059152  121.612683  0.0  0.0  0.0 | 3.122042  0.272116  0.0  0.0  0.0 | 11.007591  4.708436  0.0  0.0  0.0 | 20.189534 | 0.404326 |
    | 02/01 07:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121        0.0  0.0   0.121  0.0 | 0.057657       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.240712  0.107069  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.102594  0.453246  0.0  0.0   0.0 | 0.102594  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.103651 | 0.039459 | 0.016417 | 0.110968 | 0.462368 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.503644       200.0  0.0       200.0 |        0.0   40.142538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 | 1.938987  0.165047  0.0  0.0  0.0 | 11.043115  4.362259  0.0  0.0  0.0 | 20.626452 | 0.462368 |
    | 02/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.762932  0.064941  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.083441  0.418265  0.0  0.0   0.0 | 0.083441  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.105664 | 0.032093 | 0.031141 | 0.125609 |  0.52337 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.543644       200.0  0.0       200.0 |        0.0   40.102538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 | 1.176055  0.100106  0.0  0.0  0.0 | 10.639164  4.008935  0.0  0.0  0.0 | 21.027235 |  0.52337 |
    | 02/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.462742  0.039389  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.032772  0.383406  0.0  0.0   0.0 | 0.032772  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.107528 | 0.012605 | 0.032501 | 0.128527 |  0.53553 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.583644       200.0  0.0       200.0 |        0.0   40.062538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 | 0.713313  0.060717  0.0  0.0  0.0 | 10.036361  3.664917  0.0  0.0  0.0 | 21.398098 |  0.53553 |
    | 02/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  167.829582       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 |        0.0         0.0       2.288  0.0       1.716 |       0.0        0.0        0.0  0.0        0.0 |        0.0       2.145         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.484        0.0  0.0   0.484  0.0 | 0.230842       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.32985   0.02389  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.971033  0.349919  0.0  0.0   0.0 | 0.000098  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.10957 | 0.000038 | 0.020506 | 0.120037 | 0.500154 |   2.0       2.0  0.0  0.0       2.0 |        0.0  127.226244       200.0  0.0       200.0 |        0.0   40.603338         0.0  0.0         0.0 | 138.375653  121.612683  0.0  0.0  0.0 | 0.614305  0.036827  0.0  0.0  0.0 |  9.395081  3.338889  0.0  0.0  0.0 | 21.804446 | 0.500154 |
    | 02/01 11:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  168.625432       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121        0.0  0.0   0.121  0.0 | 0.057922       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254051   0.01449  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.90635  0.318438  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.111182 |      0.0 | 0.007085 | 0.109911 | 0.457961 |   2.0       2.0  0.0  0.0       2.0 |        0.0  127.916894       200.0  0.0       200.0 |        0.0   40.708538         0.0  0.0         0.0 | 138.438731  121.612683  0.0  0.0  0.0 | 0.418176  0.022337  0.0  0.0  0.0 |  8.742782  3.034941  0.0  0.0  0.0 | 22.125228 | 0.457961 |
    | 02/01 12:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  197.276032       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |        0.0         0.0      20.592  0.0      15.444 |       0.0        0.0        0.0  0.0        0.0 |        0.0      19.305         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.356        0.0  0.0   4.356  0.0 | 2.087099       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.609219  0.008789  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.861456  0.289238  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.115689 |      0.0 | 0.001252 | 0.109061 | 0.454422 |   2.0       2.0  0.0  0.0       2.0 |        0.0  151.380294       200.0  0.0       200.0 |        0.0   45.895738         0.0  0.0         0.0 | 140.707632  121.612683  0.0  0.0  0.0 | 1.896056  0.013548  0.0  0.0  0.0 |  8.490545  2.754492  0.0  0.0  0.0 | 23.022127 | 0.454422 |
    | 02/01 13:00 |  5.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  42.523945       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  7.139  15.3164   33.748   7.139   25.311 | 0.0       0.0  0.0  0.0       0.0 |  7.139   15.3164   33.748   7.139    25.311 |        0.0    9.366467      33.748  0.0      25.311 |       0.0   3.225965        0.0  0.0        0.0 |  32.804173         0.0         0.0  0.0         0.0 |  2.580772        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0   7.139  0.0 |      0.0       0.0  0.0   7.139  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.74604  0.005331  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.844071  0.262382  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.122092 |      0.0 | 0.000142 | 0.114076 | 0.475316 |   2.0       2.0  0.0  0.0       2.0 |   30.45818  148.803428       200.0  0.0       200.0 |  12.065766   51.196572         0.0  0.0         0.0 | 140.707632  121.612683  0.0  0.0  0.0 | 1.150016  0.008217  0.0  0.0  0.0 |  8.392514   2.49744  0.0  0.0  0.0 | 24.296312 | 0.475316 |
    | 02/01 14:00 |  1.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  48.329573       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.331   2.8556    6.292   1.331    4.719 | 0.0       0.0  0.0  0.0       0.0 |  1.331    2.8556    6.292   1.331     4.719 |        0.0    2.112422       6.292  0.0       4.719 |       0.0   0.743178        0.0  0.0        0.0 |   6.408937         0.0         0.0  0.0         0.0 |  0.594543        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |  2.528852        0.0  0.0   1.331  0.0 | 1.251696       0.0  0.0   1.331  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.719184  0.003233  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.833443  0.237819  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.12426 |      0.0 | 0.000012 | 0.115987 | 0.483279 |   2.0       2.0  0.0  0.0       2.0 |  34.521124  147.989406       200.0  0.0       200.0 |  13.808449   52.010594         0.0  0.0         0.0 | 141.984788  121.612683  0.0  0.0  0.0 | 1.682528  0.004984  0.0  0.0  0.0 |  8.278255  2.262854  0.0  0.0  0.0 | 24.727802 | 0.483279 |
    | 02/01 15:00 | 20.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 201.988205  191.984574  201.988205  0.0  201.988205 | 0.0  0.0  0.902305  0.0   0.0 | 25.047  53.7372  118.404  25.047   88.803 | 0.0       0.0  0.0  0.0       0.0 | 25.047   53.7372  118.404  25.047    88.803 |        0.0   36.356814     118.404  0.0      88.803 |       0.0  17.380386        0.0  0.0        0.0 | 115.042936    1.778447    1.778447  0.0    1.778447 | 13.568696   0.209758   0.209758  0.0   0.209758 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.209758  0.0  0.209758 |       0.0  10.003631  0.0  25.047  0.0 |      0.0  3.698754  0.0  25.047  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.662023  0.790022  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.819805  0.253556  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.143356 |      0.0 | 0.000001 |   0.1338 | 0.557498 |   2.0       2.0  0.0  0.0       2.0 | 147.218067  137.131839  201.988205  0.0  201.988205 |  54.770138   54.852736         0.0  0.0         0.0 | 141.984788  127.917561  0.0  0.0  0.0 | 1.020505  2.913715  0.0  0.0  0.0 |  8.120473  2.799321  0.0  0.0  0.0 | 28.527821 | 0.557498 |
    | 02/01 16:00 | 37.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 290.938862  302.004664  309.356596  0.0  309.356596 | 0.0  0.0  0.902305  0.0   0.0 | 45.859  98.3884  216.788  45.859  162.591 | 0.0       0.0  0.0  0.0       0.0 | 45.859   98.3884  216.788  45.859   162.591 |  28.420627   56.173848  218.776205  0.0  164.579205 | 19.426578  34.199126        0.0  0.0        0.0 |  91.362027   91.362027   91.362027  0.0   91.362027 | 17.994569  17.994569  17.994569  0.0  17.994569 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 | 18.417734   7.351931  0.0  45.859  0.0 |  9.28239  3.007473  0.0  45.859  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.379256  1.787234  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.88786  0.352847  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.177485 |      0.0 |      0.0 | 0.165653 |  0.69022 |   2.0       2.0  0.0  0.0       2.0 | 207.813473  215.717617  292.002027  0.0  291.702027 |  83.125389   86.287047   17.354569  0.0   17.654569 | 151.120131  132.262019  0.0  0.0  0.0 | 7.923639  4.133955  0.0  0.0  0.0 |  9.611869  4.233708  0.0  0.0  0.0 | 35.319527 |  0.69022 |
    | 02/01 17:00 |  8.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 323.906781  323.906781  323.906781  0.0  323.906781 | 0.0  0.0  0.902305  0.0   0.0 |  9.922  21.2872   46.904   9.922   35.178 | 0.0       0.0  0.0  0.0       0.0 |  9.922   21.2872   46.904   9.922    35.178 |  69.667573   85.844423  148.648653  0.0  137.128384 | 31.193289  37.447441   7.611943  0.0   7.406211 |  98.060761   98.060761   98.060761  0.0   98.060761 | 25.846019  25.846019  25.846019  0.0  25.846019 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   9.922  0.0 |      0.0       0.0  0.0   9.922  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 3.117709  1.626584  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.063687  0.481575  0.0  0.0   0.0 | 0.067201  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.186272 | 0.025846 | 0.001845 | 0.175453 | 0.731054 |   2.0       2.0  0.0  0.0       2.0 | 233.860668  237.354756  288.958135  0.0  288.152404 |  90.046113   86.552025   34.948645  0.0   35.754377 | 151.120131  132.262019  0.0  0.0  0.0 |  4.80593   2.50737  0.0  0.0  0.0 | 11.598691  5.378717  0.0  0.0  0.0 | 37.068165 | 0.731054 |
    | 02/01 18:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 334.397735  334.397735  334.397735  0.0  334.397735 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |  91.376852   96.554068  129.839697  0.0  124.668627 | 36.885929  36.698313  14.659084  0.0  14.682154 | 104.406788  104.406788  104.406788  0.0  104.406788 | 29.990946  29.990946  29.990946  0.0  29.990946 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   4.356  0.0 |      0.0       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.890986  0.986573  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.186619  0.559577  0.0  0.0   0.0 | 0.186619  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.191371 | 0.071776 | 0.018065 | 0.194269 | 0.809453 |   2.0       2.0  0.0  0.0       2.0 | 244.544611  249.365877  284.757227  0.0  283.674565 |  89.853124   85.031858   49.640507  0.0   50.723169 | 151.120131  132.262019  0.0  0.0  0.0 | 2.914944  1.520797  0.0  0.0  0.0 | 12.116439  5.805713  0.0  0.0  0.0 | 38.082751 | 0.809453 |
    | 02/01 19:00 |  7.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 356.253888  356.253888  356.253888  0.0  356.253888 | 0.0  0.0  0.902305  0.0   0.0 |  9.075    19.47     42.9   9.075   32.175 | 0.0       0.0  0.0  0.0       0.0 |  9.075     19.47     42.9   9.075    32.175 | 102.149255  112.159273  153.970933  0.0  143.523838 | 41.323479  41.708462  23.326802  0.0  23.048897 | 120.370769  120.370769  120.370769  0.0  120.370769 | 35.883119  35.883119  35.883119  0.0  35.883119 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   9.075  0.0 |      0.0       0.0  0.0   9.075  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.146941  0.598387  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.198703  0.581433  0.0  0.0   0.0 | 0.198703  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.199867 | 0.076424 | 0.050313 | 0.230147 | 0.958946 |   2.0       2.0  0.0  0.0       2.0 | 260.420132  266.197374  294.697064  0.0  293.036497 |  95.833757   90.056515   61.556824  0.0   63.217392 | 151.120131  132.262019  0.0  0.0  0.0 | 1.768003   0.92241  0.0  0.0  0.0 | 11.865973  5.822667  0.0  0.0  0.0 | 39.773545 | 0.958946 |
    | 02/01 20:00 | 18.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 410.165735  410.165735  410.165735  0.0  410.165735 | 0.0  0.0  0.902305  0.0   0.0 | 22.385   48.026   105.82  22.385   79.365 | 0.0       0.0  0.0  0.0       0.0 | 22.385    48.026   105.82  22.385    79.365 | 122.864197  145.201755  227.160779  0.0   201.42567 | 55.774692  59.078133   34.91311  0.0  34.193218 | 159.848924  159.848924  159.848924  0.0  159.848924 | 50.316811  50.316811  50.316811  0.0  50.316811 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  22.385  0.0 |      0.0       0.0  0.0  22.385  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.695655   0.36294  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.15487  0.571657  0.0  0.0   0.0 |  0.15487  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.217736 | 0.059565 | 0.069177 | 0.263173 | 1.096555 |   2.0       2.0  0.0  0.0       2.0 | 295.058865  302.048542  333.845209  0.0   331.16475 | 115.106869  108.117192   76.320526  0.0   79.000985 | 151.120131  132.262019  0.0  0.0  0.0 | 1.072348   0.55947  0.0  0.0  0.0 | 11.251888   5.61395  0.0  0.0  0.0 | 43.329451 | 1.096555 |
    | 02/01 21:00 | 15.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 455.043704  455.043704  455.043704  0.0  455.043704 | 0.0  0.0  0.902305  0.0   0.0 | 18.634  39.9784   88.088  18.634   66.066 | 0.0       0.0  0.0  0.0       0.0 | 18.634   39.9784   88.088  18.634    66.066 | 157.438041  177.637793  252.568414  0.0  230.408289 | 71.361694  72.506342  45.685321  0.0  45.823446 | 191.219608  191.219608  191.219608  0.0  191.219608 | 63.824096  63.824096  63.824096  0.0  63.824096 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  18.634  0.0 |      0.0       0.0  0.0  18.634  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.421936  0.220134  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.086395  0.544887  0.0  0.0   0.0 | 0.086395  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.232676 | 0.033229 | 0.064047 | 0.272671 |  1.13613 |   2.0       2.0  0.0  0.0       2.0 | 326.494439  333.287958  361.224403  0.0  358.382069 | 128.549264  121.755746     93.8193  0.0   96.661635 | 151.120131  132.262019  0.0  0.0  0.0 | 0.650412  0.339336  0.0  0.0  0.0 | 10.501033  5.289198  0.0  0.0  0.0 | 46.302455 |  1.13613 |
    | 02/01 22:00 |  6.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 473.402873  473.402873  473.402873  0.0  473.402873 | 0.0  0.0  0.902305  0.0   0.0 |  7.623  16.3548   36.036   7.623   27.027 | 0.0       0.0  0.0  0.0       0.0 |  7.623   16.3548   36.036   7.623    27.027 | 185.358526  196.033453  235.469802  0.0  225.511749 | 77.308178  75.365051  55.609902  0.0  56.558955 | 203.295969  203.295969  203.295969  0.0  203.295969 | 70.106904  70.106904  70.106904  0.0  70.106904 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   7.623  0.0 |      0.0       0.0  0.0   7.623  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255917  0.133518  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.010508  0.509793  0.0  0.0   0.0 | 0.015492  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.23949 | 0.005959 | 0.043863 | 0.261539 | 1.089744 |   2.0       2.0  0.0  0.0       2.0 | 342.085889  347.797674   365.72657  0.0  363.533289 | 131.316984  125.605199  107.676303  0.0  109.869584 | 151.120131  132.262019  0.0  0.0  0.0 | 0.394495  0.205817  0.0  0.0  0.0 |   9.73095  4.912923  0.0  0.0  0.0 | 47.658516 | 1.089744 |
    | 02/01 23:00 |  1.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 478.939765  478.939765  478.939765  0.0  478.939765 | 0.0  0.0  0.902305  0.0   0.0 |  2.299   4.9324   10.868   2.299    8.151 | 0.0       0.0  0.0  0.0       0.0 |  2.299    4.9324   10.868   2.299     8.151 | 198.262243  203.642415  221.064023  0.0  217.315563 |  77.43963  74.692858   63.20685  0.0   64.23831 |  206.64527   206.64527   206.64527  0.0   206.64527 | 72.294496  72.294496  72.294496  0.0  72.294496 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.299  0.0 |      0.0       0.0  0.0   2.299  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.155222  0.080983  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.933531  0.471444  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.242275 |      0.0 | 0.019541 | 0.243059 | 1.012746 |   2.0       2.0  0.0  0.0       2.0 | 348.122922  353.014129  362.815817  0.0  361.353996 | 130.816843  125.925637  116.123948  0.0   117.58577 | 151.120131  132.262019  0.0  0.0  0.0 | 0.239273  0.124835  0.0  0.0  0.0 |  8.952641  4.522462  0.0  0.0  0.0 | 48.212771 | 1.012746 |
    | 03/01 00:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 493.219119  493.219119  493.219119  0.0  493.219119 | 0.0  0.0  0.902686  0.0   0.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 | 204.528223  212.738588  236.654993  0.0  229.412995 | 80.340542  78.921577  70.312772  0.0  70.547771 | 216.382573  216.382573  216.382573  0.0  216.382573 | 76.836547  76.836547  76.836547  0.0  76.836547 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   5.929  0.0 |      0.0       0.0  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.094147  0.049119  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.856511  0.432745  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.247447 |      0.0 | 0.005078 | 0.235351 |  0.98063 |   2.0       2.0  0.0  0.0       2.0 | 357.630288  362.303713  371.211396  0.0  369.684574 | 135.588831  130.915406  122.007723  0.0  123.534546 | 151.120131  132.262019  0.0  0.0  0.0 | 0.145127  0.075716  0.0  0.0  0.0 |  8.190277  4.138836  0.0  0.0  0.0 | 49.241863 |  0.98063 |
    | 03/01 01:00 |  2.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 501.087335  501.087335  501.087335  0.0  501.087335 | 0.0  0.0  0.902686  0.0   0.0 |  3.267   7.0092   15.444   3.267   11.583 | 0.0       0.0  0.0  0.0       0.0 |  3.267    7.0092   15.444   3.267    11.583 | 213.565721  219.302218  234.627313  0.0  230.211324 | 82.920398  80.926101  74.035807  0.0  74.590795 | 221.471596  221.471596  221.471596  0.0  221.471596 | 79.615739  79.615739  79.615739  0.0  79.615739 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   3.267  0.0 |      0.0       0.0  0.0   3.267  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.057103  0.029792  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.78217  0.395303  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.250504 |      0.0 | 0.000782 | 0.234482 | 0.977007 |   2.0       2.0  0.0  0.0       2.0 | 363.189179   367.60189  374.139679  0.0  372.867845 | 137.898156  133.485444  126.947655  0.0   128.21949 | 151.120131  132.262019  0.0  0.0  0.0 | 0.088024  0.045924  0.0  0.0  0.0 |  7.465209  3.773324  0.0  0.0  0.0 | 49.850363 | 0.977007 |
    | 03/01 02:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 502.544412  502.544412  502.544412  0.0  502.544412 | 0.0  0.0  0.902686  0.0   0.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 218.403559  221.603573  227.380998  0.0  225.971212 | 83.288776  80.781762  76.566337  0.0  77.261123 | 222.042022  222.042022  222.042022  0.0  222.042022 | 80.502389  80.502389  80.502389  0.0  80.502389 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.034635   0.01807  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.712084  0.359953  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.25147 |      0.0 | 0.000083 | 0.234777 | 0.978237 |   2.0       2.0  0.0  0.0       2.0 | 364.480659   368.65234  372.300704  0.0  371.423655 | 138.063753  133.892072  130.243708  0.0  131.120756 | 151.120131  132.262019  0.0  0.0  0.0 | 0.053389  0.027854  0.0  0.0  0.0 |   6.78776   3.43144  0.0  0.0  0.0 | 50.042482 | 0.978237 |
    | 03/01 03:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 503.127242  503.127242  503.127242  0.0  503.127242 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 219.496367  222.227265  225.160687  0.0  224.375469 | 83.290045  80.836347  78.527724  0.0  79.026943 |  222.01911   222.01911   222.01911  0.0   222.01911 | 81.108133  81.108133  81.108133  0.0  81.108133 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021007   0.01096  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.646957  0.327075  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.252008 |      0.0 | 0.000007 | 0.235213 | 0.980055 |   2.0       2.0  0.0  0.0       2.0 | 364.656418  368.712985  370.943126  0.0  370.265296 | 138.470825  134.414258  132.184116  0.0  132.861946 | 151.120131  132.262019  0.0  0.0  0.0 | 0.032382  0.016895  0.0  0.0  0.0 |   6.16181  3.115325  0.0  0.0  0.0 | 50.149551 | 0.980055 |
    | 03/01 04:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 504.584319  504.584319  504.584319  0.0  504.584319 | 0.0  0.0  0.902686  0.0   0.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 219.874572  222.866862   226.05113  0.0  225.000538 |  83.85767   81.55838  79.936113  0.0  80.271705 | 222.614941  222.614941  222.614941  0.0  222.614941 | 81.969378  81.969378  81.969378  0.0  81.969378 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.012741  0.006647  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.58699  0.296784  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.252652 |      0.0 |      0.0 | 0.235809 | 0.982538 |   2.0       2.0  0.0  0.0       2.0 | 365.049803  369.073064  371.006938  0.0    370.3647 | 139.534516  135.511255  133.577381  0.0  134.219619 | 151.120131  132.262019  0.0  0.0  0.0 | 0.019641  0.010247  0.0  0.0  0.0 |  5.587561  2.825189  0.0  0.0  0.0 | 50.277762 | 0.982538 |
    | 03/01 05:00 |  2.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 511.578288  511.578288  511.578288  0.0  511.578288 | 0.0  0.0  0.902686  0.0   0.0 |  2.904   6.2304   13.728   2.904   10.296 | 0.0       0.0  0.0  0.0       0.0 |  2.904    6.2304   13.728   2.904    10.296 |   221.1845  226.239983  236.278139  0.0   232.79694 | 86.303819  84.574736   82.03418  0.0  82.083379 | 227.116446  227.116446  227.116446  0.0  227.116446 | 84.461843  84.461843  84.461843  0.0  84.461843 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.904  0.0 |      0.0       0.0  0.0   2.904  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.007728  0.004032  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.532101  0.269047  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254798 |      0.0 |      0.0 | 0.237811 |  0.99088 |   2.0       2.0  0.0  0.0       2.0 | 368.634765  372.735126  376.213244  0.0  375.320206 | 142.943524  138.843162  135.365044  0.0  136.258083 | 151.120131  132.262019  0.0  0.0  0.0 | 0.011913  0.006215  0.0  0.0  0.0 |  5.063189  2.560173  0.0  0.0  0.0 | 50.704729 |  0.99088 |
    | 03/01 06:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  512.74395   512.74395   512.74395  0.0   512.74395 | 0.0  0.0  0.902686  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 224.654326  227.589692  231.186173  0.0  230.127809 | 87.407963  85.026997  82.680115  0.0   83.16648 | 227.485902  227.485902  227.485902  0.0  227.485902 | 85.258048  85.258048  85.258048  0.0  85.258048 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.484  0.0 |      0.0       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.004687  0.002445  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.482053  0.243751  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255078 |      0.0 |      0.0 | 0.238073 | 0.991971 |   2.0       2.0  0.0  0.0       2.0 | 369.119357  373.128936  375.440973  0.0  374.734299 | 143.624593  139.615014  137.302977  0.0  138.009651 | 151.120131  132.262019  0.0  0.0  0.0 | 0.007225   0.00377  0.0  0.0  0.0 |  4.585823  2.318868  0.0  0.0  0.0 | 50.760598 | 0.991971 |
    | 03/01 07:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 225.209233  227.873792  230.021905  0.0  229.334171 | 87.776717  85.389358  83.866045  0.0  84.267779 | 227.457017  227.457017  227.457017  0.0  227.457017 | 85.869764  85.869764  85.869764  0.0  85.869764 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.002843  0.001483  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.436536  0.220741  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255071 |      0.0 |      0.0 | 0.238066 | 0.991942 |   2.0       2.0  0.0  0.0       2.0 | 369.020157  372.980961  374.660085  0.0  374.055145 | 144.306624   140.34582  138.666696  0.0  139.271636 | 151.120131  132.262019  0.0  0.0  0.0 | 0.004382  0.002286  0.0  0.0  0.0 |   4.15213   2.09961  0.0  0.0  0.0 | 50.759073 | 0.991942 |
    | 03/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.244235  227.661848   228.68676  0.0  228.317514 | 88.082546  85.664932  84.640021  0.0  85.009267 | 227.018951  227.018951  227.018951  0.0  227.018951 |  86.30783   86.30783   86.30783  0.0   86.30783 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.001724    0.0009  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.395211  0.199848  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254787 |      0.0 |      0.0 | 0.237801 | 0.990837 |   2.0       2.0  0.0  0.0       2.0 |  368.44789  372.378063  373.632276  0.0  373.096582 | 144.878891  140.948718  139.694505  0.0  140.230199 | 151.120131  132.262019  0.0  0.0  0.0 | 0.002658  0.001387  0.0  0.0  0.0 |  3.758643  1.900662  0.0  0.0  0.0 | 50.702533 | 0.990837 |
    | 03/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.894931  227.293849  228.059401  0.0  227.732422 | 88.431849  86.032932   85.26738  0.0  85.594359 | 226.581591  226.581591  226.581591  0.0  226.581591 |  86.74519   86.74519   86.74519  0.0   86.74519 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.001046  0.000546  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.357733  0.180898  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.25441 |      0.0 |      0.0 | 0.237449 | 0.989372 |   2.0       2.0  0.0  0.0       2.0 | 367.787565  371.705805  372.794465  0.0  372.285751 | 145.539216  141.620976  140.532315  0.0   141.04103 | 151.120131  132.262019  0.0  0.0  0.0 | 0.001612  0.000841  0.0  0.0  0.0 |  3.401956  1.720309  0.0  0.0  0.0 |  50.62757 | 0.989372 |
    | 03/01 10:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.201027  514.201027  514.201027  0.0  514.201027 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 |  224.59314  227.312585  229.081589  0.0  228.387074 | 89.096641  86.792995  85.961192  0.0  86.226707 | 226.757529  226.757529  226.757529  0.0  226.757529 | 87.443498  87.443498  87.443498  0.0  87.443498 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.363  0.0 |      0.0       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.000634  0.000331  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.32377  0.163725  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254209 |      0.0 |      0.0 | 0.237262 | 0.988591 |   2.0       2.0  0.0  0.0       2.0 | 367.604971  371.533949  372.826406  0.0  372.283206 | 146.596056  142.667078  141.374621  0.0  141.917821 | 151.120131  132.262019  0.0  0.0  0.0 | 0.000978   0.00051  0.0  0.0  0.0 |  3.078821  1.556915  0.0  0.0  0.0 | 50.587628 | 0.988591 |
    | 03/01 11:00 |  2.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 519.961252  521.777827  521.777827  0.0  521.777827 | 0.0  0.0  0.902686  0.0   0.0 |  3.146   6.7496   14.872   3.146   11.154 | 0.0       0.0  0.0  0.0       0.0 |  3.146    6.7496   14.872   3.146    11.154 | 225.493409  230.729522  241.140791  0.0  237.464602 | 91.853618  90.221105  87.932236  0.0  87.890425 | 231.636316  231.636316  231.636316  0.0  231.636316 | 90.141511  90.141511  90.141511  0.0  90.141511 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.816574        0.0  0.0   3.146  0.0 | 1.037141       0.0  0.0   3.146  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.221359  0.000201  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.303697   0.14817  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255939 |      0.0 |      0.0 | 0.238877 | 0.995319 |   2.0       2.0  0.0  0.0       2.0 | 371.400895  375.455142  378.833931  0.0   377.94892 | 148.560358  146.322685  142.943896  0.0  143.828907 | 151.899565  132.262019  0.0  0.0  0.0 |  0.81676  0.000309  0.0  0.0  0.0 |  2.996484  1.408946  0.0  0.0  0.0 | 50.931915 | 0.995319 |
    | 03/01 12:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.773236  523.119052  523.119052  0.0  523.119052 | 0.0  0.0  0.902686  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 228.776083   232.53573  237.211776  0.0  235.766768 |  92.03217  91.059297  88.570051  0.0  89.014059 | 232.417341  232.417341  232.417341  0.0  232.417341 | 90.701711  90.701711  90.701711  0.0  90.701711 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.345816        0.0  0.0   0.847  0.0 | 0.776316       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.486773  0.000122  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   0.3087  0.134085  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.256007 |      0.0 |      0.0 |  0.23894 | 0.995584 |   2.0       2.0  0.0  0.0       2.0 | 372.695169  376.177553  378.683497  0.0  377.942492 | 149.078068  146.941499  144.435556  0.0   145.17656 | 152.469064  132.262019  0.0  0.0  0.0 | 1.106303  0.000188  0.0  0.0  0.0 |  3.174556  1.274983  0.0  0.0  0.0 | 50.945468 | 0.995584 |
    | 03/01 13:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.992672  523.475677  523.475677  0.0  523.475677 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 229.937343  232.782522  235.439858  0.0  234.597488 | 92.198893   91.11533  89.395194  0.0  89.808564 | 232.441067  232.441067  232.441067  0.0  232.441067 |  91.03461   91.03461   91.03461  0.0   91.03461 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.483005        0.0  0.0   0.363  0.0 | 0.861878       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.618929  0.000074  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.332039  0.121334  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255753 |      0.0 |      0.0 | 0.238703 | 0.994595 |   2.0       2.0  0.0  0.0       2.0 | 372.851909  376.219298  378.040705  0.0  377.413071 | 149.140763  147.256379  145.434972  0.0  146.062606 | 153.090191  132.262019  0.0  0.0  0.0 | 1.349252  0.000114  0.0  0.0  0.0 |  3.461446  1.153723  0.0  0.0  0.0 | 50.894824 | 0.994595 |
    | 03/01 14:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 522.008863  523.779537  523.779537  0.0  523.779537 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 230.094041  232.906119  235.140283  0.0  234.368133 | 92.261631  91.348357  90.051394  0.0  90.394544 |  232.45259   232.45259   232.45259  0.0   232.45259 | 91.326947  91.326947  91.326947  0.0  91.326947 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.770674        0.0  0.0   0.363  0.0 | 1.037465       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.751933  0.000045  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.365774  0.109793  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255543 |      0.0 |      0.0 | 0.238507 |  0.99378 |   2.0       2.0  0.0  0.0       2.0 | 372.863473  376.148968  377.709011  0.0  377.124528 | 149.145389  147.630568  146.070525  0.0  146.655009 |   153.8234  132.262019  0.0  0.0  0.0 | 1.634784  0.000069  0.0  0.0  0.0 |  3.847605  1.043974  0.0  0.0  0.0 | 50.853141 |  0.99378 |
    | 03/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.265413  523.098508  523.098508  0.0  523.098508 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 230.006331  232.520231  233.484587  0.0  233.123282 | 92.002532  91.259306   90.29495  0.0  90.656254 | 231.822278  231.822278  231.822278  0.0  231.822278 |  91.27623   91.27623   91.27623  0.0   91.27623 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.833096        0.0  0.0     0.0  0.0 | 1.084351       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.874271  0.000027  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.40844  0.099349  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255137 |      0.0 |      0.0 | 0.238128 | 0.992199 |   2.0       2.0  0.0  0.0       2.0 | 372.332438  375.491016  376.686703  0.0  376.163524 | 148.932975  147.607492  146.411805  0.0  146.934984 | 154.572144  132.262019  0.0  0.0  0.0 | 1.844864  0.000042  0.0  0.0  0.0 |  4.313436  0.944653  0.0  0.0  0.0 | 50.772261 | 0.992199 |
    | 03/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 520.375077  522.393471  522.393471  0.0  522.393471 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 229.475295  231.926846  232.665377  0.0  232.342229 | 91.790118  91.171662  90.433131  0.0  90.756279 | 231.186324  231.186324  231.186324  0.0  231.186324 | 91.207147  91.207147  91.207147  0.0  91.207147 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.018394        0.0  0.0     0.0  0.0 | 1.205615       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.982767  0.000016  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.458018  0.089896  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254808 |      0.0 |      0.0 |  0.23782 | 0.990919 |   2.0       2.0  0.0  0.0       2.0 | 371.696484  374.790495   375.84765  0.0  375.347619 | 148.678593  147.602977  146.545821  0.0  147.045852 | 155.384924  132.262019  0.0  0.0  0.0 | 2.067711  0.000025  0.0  0.0  0.0 |  4.838185  0.854773  0.0  0.0  0.0 | 50.706716 | 0.990919 |
    | 03/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  519.45772  521.617166  521.617166  0.0  521.617166 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.839341  231.300763  231.953183  0.0  231.644591 | 91.535736  91.092708  90.440288  0.0  90.748881 | 230.531069  230.531069  230.531069  0.0  230.531069 | 91.086097  91.086097  91.086097  0.0  91.086097 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.159446        0.0  0.0     0.0  0.0 | 1.303467       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   1.0913   0.00001  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.513205  0.081343  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254566 |      0.0 |      0.0 | 0.237595 |  0.98998 |   2.0       2.0  0.0  0.0       2.0 | 371.041228    374.0608  375.065536  0.0  374.574097 | 148.416491  147.556365   146.55163  0.0  147.043069 | 156.240903  132.262019  0.0  0.0  0.0 | 2.279879  0.000015  0.0  0.0  0.0 |  5.416279   0.77344  0.0  0.0  0.0 | 50.658678 |  0.98998 |
    | 03/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.519709   520.78661   520.78661  0.0   520.78661 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.184085    230.6373  231.256796  0.0  230.953787 | 91.273634  90.979866   90.36037  0.0  90.663379 | 229.861062  229.861062  229.861062  0.0  229.861062 | 90.925548  90.925548  90.925548  0.0  90.925548 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |    2.2669        0.0  0.0     0.0  0.0 | 1.383445       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.191821  0.000006  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.573081  0.073603  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254422 |      0.0 |      0.0 |  0.23746 | 0.989418 |   2.0       2.0  0.0  0.0       2.0 | 370.371221  373.324563  374.309801  0.0  373.821372 | 148.148488  147.462047  146.476809  0.0  146.965237 | 157.124358  132.262019  0.0  0.0  0.0 | 2.471503  0.000009  0.0  0.0  0.0 |   6.03502  0.699843  0.0  0.0  0.0 | 50.629957 | 0.989418 |
    | 03/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 517.565962  519.914725  519.914725  0.0  519.914725 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 227.514078  229.955069  230.561942  0.0  230.261087 | 91.005631  90.831541  90.224668  0.0  90.525523 | 229.179814  229.179814  229.179814  0.0  229.179814 | 90.734911  90.734911  90.734911  0.0  90.734911 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.348763        0.0  0.0     0.0  0.0 |  1.44966       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.281327  0.000004  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.636291  0.066599  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254381 |      0.0 |      0.0 | 0.237422 |  0.98926 |   2.0       2.0  0.0  0.0       2.0 | 369.689973  372.589307  373.567673  0.0  373.080099 | 147.875989  147.325418  146.347052  0.0  146.834625 | 158.023461  132.262019  0.0  0.0  0.0 | 2.639836  0.000006  0.0  0.0  0.0 |  6.680056  0.633248  0.0  0.0  0.0 | 50.621851 |  0.98926 |
    | 03/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 516.600219  519.011355  519.011355  0.0  519.011355 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  226.83283  229.262223  229.864233  0.0  229.564218 | 90.733132  90.652502  90.050492  0.0  90.350507 | 228.489997  228.489997  228.489997  0.0  228.489997 | 90.521358  90.521358  90.521358  0.0  90.521358 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.411136        0.0  0.0     0.0  0.0 | 1.505237       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.359402  0.000002  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.701451  0.060262  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254448 |      0.0 |      0.0 | 0.237485 |  0.98952 |   2.0       2.0  0.0  0.0       2.0 | 369.000156  371.857081  372.833437  0.0  372.345878 | 147.600062  147.154274  146.177917  0.0  146.665476 |  158.92936  132.262019  0.0  0.0  0.0 | 2.785671  0.000003  0.0  0.0  0.0 |  7.338007  0.572988  0.0  0.0  0.0 | 50.635139 |  0.98952 |
    | 03/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 515.625325  518.083995  518.083995  0.0  518.083995 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 226.143013   228.56269  229.162809  0.0   228.86313 | 90.457205  90.448665  89.848546  0.0  90.148225 | 227.793645  227.793645  227.793645  0.0  227.793645 |  90.29035   90.29035   90.29035  0.0   90.29035 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.458669        0.0  0.0     0.0  0.0 |  1.55256       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.426866  0.000001  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.767327  0.054527  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254624 |      0.0 |      0.0 | 0.237649 | 0.990204 |   2.0       2.0  0.0  0.0       2.0 | 368.303804  371.128036  372.104273  0.0  371.616393 | 147.321522  146.955959  145.979722  0.0  146.467601 | 159.835469  132.262019  0.0  0.0  0.0 | 2.911364  0.000002  0.0  0.0  0.0 |  7.997546  0.518463  0.0  0.0  0.0 |  50.67014 | 0.990204 |
    | 03/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.643451  517.138353  517.138353  0.0  517.138353 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.446661  227.858589  228.457962  0.0  228.158422 | 90.178664  90.225405  89.626033  0.0  89.925572 | 227.092305  227.092305  227.092305  0.0  227.092305 | 90.046047  90.046047  90.046047  0.0  90.046047 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.494902        0.0  0.0     0.0  0.0 | 1.593455       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.485036  0.000001  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.832905  0.049338  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254909 |      0.0 |      0.0 | 0.237915 | 0.991311 |   2.0       2.0  0.0  0.0       2.0 | 367.602465  370.401752  371.378616  0.0  370.890276 | 147.040986  146.736601  145.759736  0.0  146.248076 | 160.736916  132.262019  0.0  0.0  0.0 | 3.019783  0.000001  0.0  0.0  0.0 |  8.649678  0.469125  0.0  0.0  0.0 | 50.726794 | 0.991311 |
    | 03/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.656246  516.178775  516.178775  0.0  516.178775 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.745322  227.151208  227.750276  0.0  227.450799 | 89.898129  89.987145  89.388076  0.0  89.687554 | 226.387159  226.387159  226.387159  0.0  226.387159 | 89.791616  89.791616  89.791616  0.0  89.791616 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.522529        0.0  0.0     0.0  0.0 | 1.629324       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.535338       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.897396  0.044643  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |   0.2553 |      0.0 |      0.0 |  0.23828 | 0.992834 |   2.0       2.0  0.0  0.0       2.0 | 366.897318  369.677703  370.655499  0.0  370.166637 | 146.758927  146.501072  145.523276  0.0  146.012138 | 161.630121  132.262019  0.0  0.0  0.0 | 3.113769  0.000001  0.0  0.0  0.0 |  9.287619  0.424482  0.0  0.0  0.0 | 50.804748 | 0.992834 |
    | 04/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.663479  515.208571  515.208571  0.0  515.208571 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0  0.000576  0.0  0.0  0.001152 |    0.0       0.0      0.0     0.0       0.0 | 224.040175  226.441398  227.040334  0.0  226.740888 |  89.61607  89.737377   89.13844  0.0  89.437887 | 225.679105  225.679105  225.679105  0.0  225.679105 | 89.529467  89.529467  89.529467  0.0  89.529467 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.545093        0.0  0.0     0.0  0.0 | 1.662219       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.579327       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.960233  0.040395  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255796 |      0.0 |      0.0 | 0.238743 | 0.994763 |   2.0  1.999424  0.0  0.0  1.998848 | 366.188199   368.95541   369.93427  0.0  369.444854 |  146.47528  146.253161  145.274302  0.0  145.763718 | 162.512995  132.262019  0.0  0.0  0.0 | 3.196661       0.0  0.0  0.0  0.0 |  9.906714  0.384088  0.0  0.0  0.0 | 50.903433 | 0.994763 |
    | 04/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.668518   514.22969   514.22969  0.0   514.22969 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.331056  225.729761  226.328635  0.0  226.029206 | 89.332422   89.47881  88.879937  0.0  89.179365 | 224.968418  224.968418  224.968418  0.0  224.968418 | 89.261271  89.261271  89.261271  0.0  89.261271 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.561172        0.0  0.0     0.0  0.0 | 1.691044       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.618084       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.020428  0.036551  0.0  0.0   0.0 | 0.021125  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.256392 | 0.008125 |  0.00058 | 0.239801 | 0.999173 |   2.0  1.999424  0.0  0.0  1.998848 | 365.477513  368.234067  369.214053  0.0  368.724065 | 146.191005  145.995622  145.015637  0.0  145.505624 | 163.383123  132.262019  0.0  0.0  0.0 | 3.269621       0.0  0.0  0.0  0.0 | 10.483246  0.347537  0.0  0.0  0.0 | 51.021923 | 0.999173 |
    | 04/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.671177  513.244623  513.244623  0.0  513.244623 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  222.62037  225.016328  225.615167  0.0  225.315751 | 89.048148  89.213361  88.614523  0.0  88.913939 | 224.256032  224.256032  224.256032  0.0  224.256032 | 88.988592  88.988592  88.988592  0.0  88.988592 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.573447        0.0  0.0     0.0  0.0 | 1.717393       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.652405       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.074349  0.033073  0.0  0.0   0.0 | 0.074349  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.257075 | 0.028596 | 0.006109 | 0.245232 | 1.021799 |   2.0  1.999424  0.0  0.0  1.998848 | 364.765126  367.513771  368.494918  0.0  368.004346 | 145.906051  145.730853  144.749706  0.0  145.240277 | 164.239177  132.262019  0.0  0.0  0.0 | 3.334608       0.0  0.0  0.0  0.0 | 10.986953  0.314465  0.0  0.0  0.0 | 51.157993 | 1.021799 |
    | 04/01 03:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.766676   516.04297   516.04297  0.0   516.04297 | 0.0  0.0  0.903096  0.0   0.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |  1.573  3.374224    7.436   1.573  5.575848 | 222.346669  226.148508  231.531107  0.0   229.56884 | 89.897507  90.470339  89.149517  0.0  89.251632 | 226.191503  226.191503  226.191503  0.0  226.191503 | 89.851467  89.851467  89.851467  0.0  89.851467 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  3.276294        0.0  0.0   1.573  0.0 |  2.20941       0.0  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.782806       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.126109  0.029925  0.0  0.0   0.0 | 0.126109  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.258967 | 0.048503 | 0.020584 | 0.259542 | 1.081425 |   2.0       2.0  0.0  0.0       2.0 | 366.261912  369.083712  371.231314  0.0  370.542858 | 146.504765  146.959259  144.811656  0.0  145.500113 | 165.306061  132.262019  0.0  0.0  0.0 | 3.761212       0.0  0.0  0.0  0.0 | 11.517542   0.28454  0.0  0.0  0.0 | 51.534472 | 1.081425 |
    | 04/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.270864  514.782857  514.782857  0.0  514.782857 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.404769  226.039922  227.355189  0.0  226.933554 | 89.361908  90.003049  88.687781  0.0  89.109416 | 225.398666  225.398666  225.398666  0.0  225.398666 | 89.384192  89.384192  89.384192  0.0  89.384192 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.511993        0.0  0.0     0.0  0.0 | 1.716074       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.845551       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.177136  0.027078  0.0  0.0   0.0 | 0.177136  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.259813 | 0.068129 | 0.039611 | 0.276822 | 1.153423 |   2.0       2.0  0.0  0.0       2.0 |  365.90776  368.482456   369.91479  0.0  369.347969 | 146.363104  146.300402  144.868067  0.0  145.434888 |  166.10198  132.262019  0.0  0.0  0.0 | 3.631736       0.0  0.0  0.0  0.0 |  12.00882  0.257462  0.0  0.0  0.0 | 51.702801 | 1.153423 |
    | 04/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  511.28054  513.816706  513.816706  0.0  513.816706 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.050617  225.322111  226.197965  0.0  225.851361 | 89.220247  89.460746  88.584892  0.0  88.931496 | 224.691291  224.691291  224.691291  0.0  224.691291 | 89.125415  89.125415  89.125415  0.0  89.125415 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.536166        0.0  0.0     0.0  0.0 | 1.749312       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.801687       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.219609  0.024501  0.0  0.0   0.0 | 0.219609  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.260727 | 0.084465 | 0.059042 | 0.294515 | 1.227146 |   2.0       2.0  0.0  0.0       2.0 | 365.200386  367.891636  369.048117  0.0  368.527899 | 146.080154   145.92507  144.768589  0.0  145.288807 | 166.888834  132.262019  0.0  0.0  0.0 |  3.57936       0.0  0.0  0.0  0.0 | 12.371289  0.232961  0.0  0.0  0.0 | 51.884649 | 1.227146 |
    | 04/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.286928  512.841258  512.841258  0.0  512.841258 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 222.343243  224.692074  225.398402  0.0  225.080676 | 88.937297  89.124632  88.418304  0.0   88.73603 | 223.981569  223.981569  223.981569  0.0  223.981569 | 88.859689  88.859689  88.859689  0.0  88.859689 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.554329        0.0  0.0     0.0  0.0 | 1.778572       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.787314       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.251788  0.022169  0.0  0.0   0.0 | 0.251788  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.26169 | 0.096842 | 0.076521 | 0.310563 | 1.294011 |   2.0       2.0  0.0  0.0       2.0 | 364.490663   367.22113  368.271283  0.0  367.768792 | 145.796265  145.620127  144.569974  0.0  145.072466 | 167.664591  132.262019  0.0  0.0  0.0 | 3.570619       0.0  0.0  0.0  0.0 | 12.655026  0.210792  0.0  0.0  0.0 |  52.07636 | 1.294011 |
    | 04/01 07:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.955818  513.898731  513.898731  0.0  513.898731 | 0.0  0.0  0.903096  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 221.870249  225.005946  228.218517  0.0  227.018407 | 89.263679  89.652512  88.626741  0.0  88.825851 | 224.696076  224.696076  224.696076  0.0  224.696076 | 89.202655  89.202655  89.202655  0.0  89.202655 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.942913        0.0  0.0   0.847  0.0 | 2.068236       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.84559       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.280234   0.02006  0.0  0.0   0.0 | 0.280234  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.263302 | 0.107782 | 0.090764 | 0.324411 | 1.351711 |   2.0       2.0  0.0  0.0       2.0 | 364.968442   367.75206  369.392842  0.0  368.789461 | 145.987377  146.146671  144.505889  0.0   145.10927 | 168.539268  132.262019  0.0  0.0  0.0 | 3.793265       0.0  0.0  0.0  0.0 | 12.940149  0.190733  0.0  0.0  0.0 | 52.397016 | 1.351711 |
    | 04/01 08:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.181898  513.932503  513.932503  0.0  513.932503 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 222.246491  225.198159  227.670615  0.0   226.79139 | 89.193327  89.738972  88.516116  0.0  88.823341 | 224.756025  224.756025  224.756025  0.0  224.756025 | 89.176478  89.176478  89.176478  0.0  89.176478 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.750605        0.0  0.0   0.484  0.0 | 1.953307       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.908708       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.309031  0.018151  0.0  0.0   0.0 | 0.309031  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.264694 | 0.118858 | 0.102689 | 0.336045 | 1.400189 |   2.0       2.0  0.0  0.0       2.0 | 365.129927  367.807526  369.406253  0.0  368.810097 | 146.051971  146.124977   144.52625  0.0  145.122406 | 169.336566  132.262019  0.0  0.0  0.0 | 3.837864       0.0  0.0  0.0  0.0 | 13.230794  0.172582  0.0  0.0  0.0 | 52.674162 | 1.400189 |
    | 04/01 09:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.581776  513.165993  513.165993  0.0  513.165993 | 0.0  0.0  0.903096  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 | 222.306591  224.815129  226.159005  0.0  225.667272 | 88.996307  89.376974  88.345498  0.0  88.694231 | 224.225981  224.225981  224.225981  0.0  224.225981 | 88.940012  88.940012  88.940012  0.0  88.940012 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.584217        0.0  0.0   0.121  0.0 | 1.852552       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.904789       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.33519  0.016423  0.0  0.0   0.0 |  0.33519  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.265865 | 0.128919 | 0.113795 | 0.346763 | 1.444846 |   2.0       2.0  0.0  0.0       2.0 | 364.701269  367.372779  368.685229  0.0  368.137806 | 145.880507  145.793214  144.480764  0.0  145.028188 | 170.068231  132.262019  0.0  0.0  0.0 | 3.785627       0.0  0.0  0.0  0.0 | 13.465203  0.156159  0.0  0.0  0.0 | 52.907129 | 1.444846 |
    | 04/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.579336  513.337725  513.337725  0.0  513.337725 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 221.979417  224.762609  227.032845  0.0   226.18834 | 89.086359  89.441784  88.421148  0.0  88.693653 | 224.325723  224.325723  224.325723  0.0  224.325723 | 89.012002  89.012002  89.012002  0.0  89.012002 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.758389        0.0  0.0   0.484  0.0 | 1.994536       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.914487       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.35689   0.01486  0.0  0.0   0.0 |  0.35689  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.267326 | 0.137265 | 0.124174 | 0.357122 | 1.488007 |   2.0       2.0  0.0  0.0       2.0 | 364.699526  367.433493  368.906107  0.0  368.331188 |  145.87981  145.904232  144.431618  0.0  145.006537 | 170.832084  132.262019  0.0  0.0  0.0 | 3.865677       0.0  0.0  0.0  0.0 |  13.66591  0.141298  0.0  0.0  0.0 | 53.197795 | 1.488007 |
    | 04/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  509.74056  512.276806  512.276806  0.0  512.276806 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.842383  224.278811  225.177685  0.0  224.826758 | 88.736953  89.058914  88.160041  0.0  88.510967 | 223.591306  223.591306  223.591306  0.0  223.591306 | 88.685501  88.685501  88.685501  0.0  88.685501 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.536246        0.0  0.0     0.0  0.0 |  1.85042       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.915278       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.375118  0.013446  0.0  0.0   0.0 | 0.375118  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.268464 | 0.144276 | 0.133243 | 0.366044 | 1.525182 |   2.0       2.0  0.0  0.0       2.0 |   364.1004  366.785987  367.959728  0.0  367.435736 |  145.64016  145.490819  144.317078  0.0  144.841071 |  171.51791  132.262019  0.0  0.0  0.0 | 3.800818       0.0  0.0  0.0  0.0 | 13.830952  0.127852  0.0  0.0  0.0 | 53.424287 | 1.525182 |
    | 04/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 508.746874  511.301327  511.301327  0.0  511.301327 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.243257  223.587629  224.303125  0.0  223.983707 | 88.497303  88.689177  87.973681  0.0  88.293099 |  222.88153   222.88153   222.88153  0.0   222.88153 | 88.419797  88.419797  88.419797  0.0  88.419797 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.554453        0.0  0.0     0.0  0.0 | 1.878698       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.895783       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.389164  0.012167  0.0  0.0   0.0 | 0.389164  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.269619 | 0.149678 | 0.140863 | 0.373726 |  1.55719 |   2.0       2.0  0.0  0.0       2.0 | 363.390624  366.119888  367.178132  0.0  366.673558 |  145.35625  145.181439  144.123195  0.0  144.627769 | 172.193665  132.262019  0.0  0.0  0.0 | 3.783733       0.0  0.0  0.0  0.0 | 13.948408  0.115685  0.0  0.0  0.0 | 53.654274 |  1.55719 |
    | 04/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 507.750567  510.318845  510.318845  0.0  510.318845 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 220.533481  222.908881  223.553184  0.0  223.245979 | 88.213393  88.392446  87.748143  0.0  88.055348 | 222.169882  222.169882  222.169882  0.0  222.169882 | 88.148963  88.148963  88.148963  0.0  88.148963 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.568278        0.0  0.0     0.0  0.0 | 1.903778       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.894404       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.399745  0.011009  0.0  0.0   0.0 | 0.399745  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.270787 | 0.153748 | 0.146986 | 0.380122 | 1.583842 |   2.0       2.0  0.0  0.0       2.0 | 362.678976  365.420888   366.43483  0.0  365.937461 | 145.071591  144.897957  143.884015  0.0  144.381384 | 172.858165  132.262019  0.0  0.0  0.0 | 3.793107       0.0  0.0  0.0  0.0 | 14.043323  0.104676  0.0  0.0  0.0 | 53.886541 | 1.583842 |
    | 04/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 506.752198  509.331046  509.331046  0.0  509.331046 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.821833  222.208114   222.82468  0.0  222.522236 | 87.928733  88.110731  87.494165  0.0  87.796609 | 221.456761  221.456761  221.456761  0.0  221.456761 | 87.874284  87.874284  87.874284  0.0  87.874284 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.578848        0.0  0.0     0.0  0.0 | 1.926396       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.902912       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.408746  0.009961  0.0  0.0   0.0 | 0.408746  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.271963 |  0.15721 |  0.15173 | 0.385331 | 1.605547 |   2.0       2.0  0.0  0.0       2.0 | 361.965856  364.709536  365.706911  0.0  365.211987 | 144.786342   144.62151  143.624134  0.0  144.119059 | 173.510617  132.262019  0.0  0.0  0.0 | 3.816591       0.0  0.0  0.0  0.0 | 14.128744  0.094715  0.0  0.0  0.0 | 54.120548 | 1.605547 |
    | 04/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 505.752249  508.339181  508.339181  0.0  508.339181 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.108713  221.498342  222.104076  0.0  221.803495 | 87.643485  87.832704   87.22697  0.0  87.527551 | 220.742512  220.742512  220.742512  0.0  220.742512 | 87.596669  87.596669  87.596669  0.0  87.596669 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.586932        0.0  0.0     0.0  0.0 |  1.94705       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.916553       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.417126  0.009013  0.0  0.0   0.0 | 0.417126  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.273146 | 0.160433 | 0.155562 | 0.389757 | 1.623988 |   2.0       2.0  0.0  0.0       2.0 | 361.251606  363.993706  364.985347  0.0  364.491004 | 144.500643  144.345475  143.353834  0.0  143.848177 | 174.150499  132.262019  0.0  0.0  0.0 | 3.847089       0.0  0.0  0.0  0.0 | 14.211043  0.085702  0.0  0.0  0.0 | 54.356094 | 1.623988 |
    | 04/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 504.751087  507.344207  507.344207  0.0  507.344207 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 218.394464  220.784715  221.386207  0.0  221.086357 | 87.357785  87.554466  86.952974  0.0  87.252824 | 220.027397  220.027397  220.027397  0.0  220.027397 | 87.316811  87.316811  87.316811  0.0  87.316811 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |   2.59312        0.0  0.0     0.0  0.0 | 1.966129       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.932618       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.425338  0.008156  0.0  0.0   0.0 | 0.425338  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.274337 | 0.163591 | 0.158948 | 0.393803 | 1.640846 |   2.0       2.0  0.0  0.0       2.0 | 360.536491  363.276387  364.266537  0.0  363.772044 | 144.214596   144.06782   143.07767  0.0  143.572164 |  174.77749  132.262019  0.0  0.0  0.0 |   3.8806       0.0  0.0  0.0  0.0 | 14.292985  0.077546  0.0  0.0  0.0 | 54.593136 | 1.640846 |
    | 04/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 503.748995  506.346853  506.346853  0.0  506.346853 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 217.679348  220.069317   220.66914  0.0  220.369581 | 87.071739   87.27489  86.675067  0.0  86.974626 | 219.311616  219.311616  219.311616  0.0  219.311616 | 87.035237  87.035237  87.035237  0.0  87.035237 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.597858        0.0  0.0     0.0  0.0 | 1.983931       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.949596       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.43356  0.007379  0.0  0.0   0.0 |  0.43356  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.275536 | 0.166754 | 0.162161 | 0.397707 | 1.657112 |   2.0       2.0  0.0  0.0       2.0 | 359.820711  362.558686  363.549013  0.0  363.054079 | 143.928284  143.788167  142.797841  0.0  143.292775 | 175.391417  132.262019  0.0  0.0  0.0 | 3.914935       0.0  0.0  0.0  0.0 | 14.375462  0.070167  0.0  0.0  0.0 | 54.831694 | 1.657112 |
    | 04/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 502.746187  505.347677  505.347677  0.0  505.347677 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.963568  219.353023  219.952184  0.0  219.652742 | 86.785427  86.993831   86.39467  0.0  86.694112 | 218.595325  218.595325  218.595325  0.0  218.595325 | 86.752352  86.752352  86.752352  0.0  86.752352 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |   2.60149        0.0  0.0     0.0  0.0 | 2.000686       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.966676       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.441835  0.006677  0.0  0.0   0.0 | 0.441835  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.276743 | 0.169936 | 0.165332 | 0.401581 | 1.673252 |   2.0       2.0  0.0  0.0       2.0 | 359.104419  361.840989  362.832154  0.0  362.336662 | 143.641768  143.506688  142.515523  0.0  143.011015 | 175.992222  132.262019  0.0  0.0  0.0 | 3.948945       0.0  0.0  0.0  0.0 | 14.458468  0.063489  0.0  0.0  0.0 |   55.0718 | 1.673252 |
    | 04/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 501.742828  504.347104  504.347104  0.0  504.347104 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.247276  218.636219  219.235114  0.0  218.935721 | 86.498911  86.711458  86.112564  0.0  86.411956 |  217.87864   217.87864   217.87864  0.0   217.87864 | 86.468464  86.468464  86.468464  0.0  86.468464 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.604276        0.0  0.0     0.0  0.0 | 2.016573       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.983442       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.450143  0.006042  0.0  0.0   0.0 | 0.450143  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.277957 | 0.173132 | 0.168507 | 0.405466 | 1.689442 |   2.0       2.0  0.0  0.0       2.0 | 358.387734   361.12341   362.11568  0.0  361.619581 | 143.355094  143.223694  142.231424  0.0  142.727523 | 176.579925  132.262019  0.0  0.0  0.0 | 3.982075       0.0  0.0  0.0  0.0 | 14.541624  0.057448  0.0  0.0  0.0 | 55.313477 | 1.689442 |
    | 04/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 500.739045   503.34546   503.34546  0.0   503.34546 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 215.530592  217.919094  218.517877  0.0  218.218507 | 86.212237   86.42801  85.829227  0.0  86.128597 | 217.161652  217.161652  217.161652  0.0  217.161652 | 86.183808  86.183808  86.183808  0.0  86.183808 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.606415        0.0  0.0     0.0  0.0 | 2.031731       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.999708       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.458442  0.005467  0.0  0.0   0.0 | 0.458442  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.27918 | 0.176324 | 0.171695 |  0.40937 | 1.705708 |   2.0       2.0  0.0  0.0       2.0 | 357.670746  360.405968  361.399455  0.0  360.902725 | 143.068299  142.939492  141.946005  0.0  142.442734 | 177.154609  132.262019  0.0  0.0  0.0 | 4.014098       0.0  0.0  0.0  0.0 | 14.624449  0.051981  0.0  0.0  0.0 | 55.556731 | 1.705708 |
    | 04/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 499.734933  502.342992  502.342992  0.0  502.342992 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.813603  217.201748  217.800482  0.0  217.501124 | 85.925441  86.143711  85.544978  0.0  85.844336 | 216.444429  216.444429  216.444429  0.0  216.444429 | 85.898563  85.898563  85.898563  0.0  85.898563 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.608059        0.0  0.0     0.0  0.0 | 2.046267       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.015405       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.466683  0.004947  0.0  0.0   0.0 | 0.466683  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.28041 | 0.179494 | 0.174886 | 0.413283 | 1.722014 |   2.0       2.0  0.0  0.0       2.0 | 356.953524  359.688648  360.683402  0.0  360.186031 | 142.781409  142.654344   141.65959  0.0  142.156961 | 177.716401  132.262019  0.0  0.0  0.0 | 4.044961       0.0  0.0  0.0  0.0 | 14.706487  0.047034  0.0  0.0  0.0 |  55.80155 | 1.722014 |
    | 04/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 498.730567  501.339893  501.339893  0.0  501.339893 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.096381  216.484243  217.082951  0.0    216.7836 | 85.638552   85.85875  85.260042  0.0  85.559392 | 215.727025  215.727025  215.727025  0.0  215.727025 | 85.612867  85.612867  85.612867  0.0  85.612867 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.609325        0.0  0.0     0.0  0.0 | 2.060266       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.030531       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.474827  0.004476  0.0  0.0   0.0 | 0.474827  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.281648 | 0.182626 | 0.178062 | 0.417192 |   1.7383 |   2.0       2.0  0.0  0.0       2.0 |  356.23612  358.971431  359.967476  0.0  359.469456 | 142.494448  142.368462  141.372416  0.0  141.870437 |  178.26546  132.262019  0.0  0.0  0.0 | 4.074695       0.0  0.0  0.0  0.0 | 14.787363  0.042558  0.0  0.0  0.0 | 56.047905 |   1.7383 |
    | 04/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 497.726004  500.336306  500.336306  0.0  500.336306 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 213.378977  215.766617  216.365309  0.0  216.065964 | 85.351591  85.573276  84.974584  0.0  85.273929 |  215.00948   215.00948   215.00948  0.0   215.00948 | 85.326826  85.326826  85.326826  0.0  85.326826 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.610302        0.0  0.0     0.0  0.0 | 2.073792       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.045112       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.48284   0.00405  0.0  0.0   0.0 |  0.48284  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.282893 | 0.185708 | 0.181209 | 0.421081 | 1.754506 |   2.0       2.0  0.0  0.0       2.0 | 355.518575  358.254294  359.251648  0.0  358.752972 |  142.20743  142.082012  141.084658  0.0  141.583334 |  178.80197  132.262019  0.0  0.0  0.0 | 4.103375       0.0  0.0  0.0  0.0 | 14.866795  0.038508  0.0  0.0  0.0 | 56.295755 | 1.754506 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0

The submodel rconcmodel is optional. We repeat the last integration test, but without
a submodel for runoff concentration.

>>> model.rconcmodel = None

.. integration-test::

    >>> test("hland_96c_snow_redistribution",
    ...      axis1=(states.sp, states.wc), axis2=(factors.tc, fluxes.pc))
    |        date |    p |    t |                         tc |                     fracrain |                      rfc |                      sfc |                                       cfact |                                                 swe |                          gact |                                        pc |                                ei |                                          tf |                                                 spl |                                             wcl |                                                 spg |                                             wcg |                     glmelt |                          melt |                               refr |                                    in_ |                                    r |                      sr |                      ea |                               qvs1 |                     qab1 |                               qvs2 |                          qab2 |                      el |       q1 |     inrc |    outrc |       rt |       qt |                                  ic |                                                  sp |                                                  wc |                                    sm |                               bw1 |                                bw2 |        lz |   outlet |
    ---------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 01/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |     0.05 |      0.0 |      0.0 | 0.046667 | 0.194444 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |      9.95 | 0.194444 |
    | 01/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.04975 |      0.0 |      0.0 | 0.046433 | 0.193472 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |   9.90025 | 0.193472 |
    | 01/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049501 |      0.0 |      0.0 | 0.046201 | 0.192505 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.850749 | 0.192505 |
    | 01/01 03:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049254 |      0.0 |      0.0 |  0.04597 | 0.191542 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.801495 | 0.191542 |
    | 01/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.049007 |      0.0 |      0.0 |  0.04574 | 0.190585 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.752488 | 0.190585 |
    | 01/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048762 |      0.0 |      0.0 | 0.045512 | 0.189632 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.703725 | 0.189632 |
    | 01/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048519 |      0.0 |      0.0 | 0.045284 | 0.188684 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.655206 | 0.188684 |
    | 01/01 07:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048276 |      0.0 |      0.0 | 0.045058 |  0.18774 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |   9.60693 |  0.18774 |
    | 01/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.048035 |      0.0 |      0.0 | 0.044832 | 0.186801 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.558896 | 0.186801 |
    | 01/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047794 |      0.0 |      0.0 | 0.044608 | 0.185867 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.511101 | 0.185867 |
    | 01/01 10:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047556 |      0.0 |      0.0 | 0.044385 | 0.184938 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.463546 | 0.184938 |
    | 01/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047318 |      0.0 |      0.0 | 0.044163 | 0.184013 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.416228 | 0.184013 |
    | 01/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.047081 |      0.0 |      0.0 | 0.043942 | 0.183093 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.369147 | 0.183093 |
    | 01/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046846 |      0.0 |      0.0 | 0.043723 | 0.182178 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.322301 | 0.182178 |
    | 01/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0         0.0  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046612 |      0.0 |      0.0 | 0.043504 | 0.181267 |   0.0       0.0  0.0  0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |   9.27569 | 0.181267 |
    | 01/01 15:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0    1.144   0.242       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046551 |      0.0 |      0.0 | 0.043448 | 0.181033 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |   9.26371 | 0.181033 |
    | 01/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046319 |      0.0 |      0.0 | 0.043231 | 0.180128 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.217391 | 0.180128 |
    | 01/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0         0.0       1.144  0.0         0.0 | 0.0  0.0  0.901953  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0   0.0       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0       0.0  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046087 |      0.0 |      0.0 | 0.043014 | 0.179227 | 0.242    0.5192  0.0  0.0     0.858 |        0.0         0.0       1.144  0.0         0.0 |        0.0         0.0         0.0  0.0         0.0 |      100.0       100.0  0.0  0.0  0.0 |      0.0       0.0  0.0  0.0  0.0 |       0.0       0.0  0.0  0.0  0.0 |  9.171304 | 0.179227 |
    | 01/01 18:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0    1.224502        8.58  0.0       4.435 | 0.0  0.0  0.901953  0.0   0.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |    0.0     1.894    7.436   1.573     4.435 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0   0.669498  0.0   1.573  0.0 |      0.0  0.167375  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |      0.0  0.035661  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |      0.0  0.001725  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.046983 |      0.0 |      0.0 |  0.04385 |  0.18271 | 1.815       2.0  0.0  0.0       2.0 |        0.0    0.874644        8.58  0.0       4.435 |        0.0    0.349858         0.0  0.0         0.0 |      100.0  100.502124  0.0  0.0  0.0 |      0.0  0.131714  0.0  0.0  0.0 |       0.0  0.033936  0.0  0.0  0.0 |  9.349529 |  0.18271 |
    | 01/01 19:00 |  5.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   10.249462      40.612  0.0      28.459 | 0.0  0.0  0.901953  0.0   0.0 |  6.776  14.5376   32.032   6.776   24.024 | 0.0       0.0  0.0  0.0       0.0 |  6.591   14.5376   32.032   6.776    24.024 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     6.591    5.51264  0.0   6.776  0.0 |  1.64775  1.392035  0.0   6.776  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.351072  0.348414  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.016983  0.020084  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.051647 |      0.0 |      0.0 | 0.048204 | 0.200848 |   2.0       2.0  0.0  0.0       2.0 |        0.0    7.321044      40.612  0.0      28.459 |        0.0    2.928418         0.0  0.0         0.0 |  104.94325  104.622729  0.0  0.0  0.0 | 1.296678  1.175334  0.0  0.0  0.0 |  0.334089  0.362266  0.0  0.0  0.0 | 10.277686 | 0.200848 |
    | 01/01 20:00 |  2.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   14.950102        57.2  0.0        40.9 | 0.0  0.0  0.901953  0.0   0.0 |  3.509   7.5284   16.588   3.509   12.441 | 0.0       0.0  0.0  0.0       0.0 |  3.509    7.5284   16.588   3.509    12.441 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     3.509    2.82776  0.0   3.509  0.0 | 0.966123  0.773811  0.0   3.509  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.716047  0.627327  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.066431  0.064821  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.054106 |      0.0 |      0.0 | 0.050499 | 0.210413 |   2.0       2.0  0.0  0.0       2.0 |        0.0   10.678644        57.2  0.0        40.9 |        0.0    4.271458         0.0  0.0         0.0 | 107.486127  106.676678  0.0  0.0  0.0 | 1.546755  1.321818  0.0  0.0  0.0 |  0.983705  0.924773  0.0  0.0  0.0 | 10.767111 | 0.210413 |
    | 01/01 21:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   22.853942      85.228  0.0      61.921 | 0.0  0.0  0.901953  0.0   0.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     5.929    4.81656  0.0   5.929  0.0 | 1.712483  1.370301  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.973464  0.812053  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.140702  0.127286  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.058504 |      0.0 |      0.0 | 0.054603 | 0.227514 |   2.0       2.0  0.0  0.0       2.0 |        0.0   16.324244      85.228  0.0      61.921 |        0.0    6.529698         0.0  0.0         0.0 | 111.702644  110.122937  0.0  0.0  0.0 | 2.285773  1.880066  0.0  0.0  0.0 |  1.816467   1.60954  0.0  0.0  0.0 | 11.642226 | 0.227514 |
    | 01/01 22:00 | 10.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   39.886902      145.86  0.0     107.395 | 0.0  0.0  0.901953  0.0   0.0 | 12.826  27.5176   60.632  12.826   45.474 | 0.0       0.0  0.0  0.0       0.0 | 12.826   27.5176   60.632  12.826    45.474 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |    12.826   10.48464  0.0  12.826  0.0 | 4.000904  3.178697  0.0  12.826  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.75182  1.417006  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.257602  0.221714  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.068149 |      0.0 |      0.0 | 0.063606 | 0.265025 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.490644      145.86  0.0     107.395 |        0.0   11.396258         0.0  0.0         0.0 |  120.52774  117.428881  0.0  0.0  0.0 | 4.534858  3.641757  0.0  0.0  0.0 |  3.310684  2.804831  0.0  0.0  0.0 |  13.56171 | 0.265025 |
    | 01/01 23:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.450977  0.450977  0.450977  0.0  0.450977 |        0.0   40.103062     146.432  0.0     107.824 | 0.0  0.0  0.901953  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121    0.04344  0.0   0.121  0.0 | 0.043944  0.014975  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.79369   1.43611  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.401822  0.336386  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.069093 |      0.0 |      0.0 | 0.064487 | 0.268695 |   2.0       2.0  0.0  0.0       2.0 |        0.0   28.645044     146.432  0.0     107.824 |        0.0   11.458018         0.0  0.0         0.0 | 120.604796  117.457345  0.0  0.0  0.0 | 2.785112  2.220622  0.0  0.0  0.0 |  4.702552  3.904556  0.0  0.0  0.0 |  13.74952 | 0.268695 |
    | 02/01 00:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   41.280182     150.436  0.0     110.827 | 0.0  0.0  0.902305  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.847    0.64008  0.0   0.847  0.0 | 0.308001  0.220767  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.161479  0.920784  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.503693   0.41611  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.070846 |      0.0 |      0.0 | 0.066123 | 0.275514 |   2.0       2.0  0.0  0.0       2.0 |        0.0   29.485844     150.436  0.0     110.827 |        0.0   11.794338         0.0  0.0         0.0 | 121.143795  117.876658  0.0  0.0  0.0 | 1.931634  1.520606  0.0  0.0  0.0 |  5.360339  4.409229  0.0  0.0  0.0 | 14.098453 | 0.275514 |
    | 02/01 01:00 |  3.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   46.140982     167.596  0.0     123.697 | 0.0  0.0  0.902305  0.0   0.0 |   3.63    7.788    17.16    3.63    12.87 | 0.0       0.0  0.0  0.0       0.0 |   3.63     7.788    17.16    3.63     12.87 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |      3.63     2.9272  0.0    3.63  0.0 | 1.331831  1.016829  0.0    3.63  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   1.0438  0.814959  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.560597  0.459017  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.074742 |      0.0 |      0.0 | 0.069759 | 0.290663 |   2.0       2.0  0.0  0.0       2.0 |        0.0   32.957844     167.596  0.0     123.697 |        0.0   13.183138         0.0  0.0         0.0 | 123.441964  119.787029  0.0  0.0  0.0 | 2.219664  1.722476  0.0  0.0  0.0 |  5.843543  4.765172  0.0  0.0  0.0 | 14.873643 | 0.290663 |
    | 02/01 02:00 |  2.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   49.560342     179.608  0.0     132.706 | 0.0  0.0  0.902305  0.0   0.0 |  2.541   5.4516   12.012   2.541    9.009 | 0.0       0.0  0.0  0.0       0.0 |  2.541    5.4516   12.012   2.541     9.009 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     2.541    2.03224  0.0   2.541  0.0 | 0.967989  0.729012  0.0   2.541  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.079611  0.833066  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.608312  0.493765  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.077975 |      0.0 |      0.0 | 0.072777 | 0.303236 |   2.0       2.0  0.0  0.0       2.0 |        0.0   35.400244     179.608  0.0     132.706 |        0.0   14.160098         0.0  0.0         0.0 | 125.014975  121.090257  0.0  0.0  0.0 | 2.108042  1.618422  0.0  0.0  0.0 |  6.314841  5.104472  0.0  0.0  0.0 | 15.516998 | 0.303236 |
    | 02/01 03:00 | 10.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0   99.386482       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 | 12.584  26.9984   59.488  12.584   44.616 | 0.0       0.0  0.0  0.0       0.0 | 12.584   26.9984   59.488  12.584    44.616 |        0.0         0.0      39.096  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0     23.6525         0.0  0.0      22.678 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |    12.584    0.82476  0.0  12.584  0.0 | 4.916803  0.302333  0.0  12.584  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.87703  0.701215  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.691736  0.519675  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.088551 |      0.0 |      0.0 | 0.082648 | 0.344366 |   2.0       2.0  0.0  0.0       2.0 |        0.0   70.990344       200.0  0.0       200.0 |        0.0   28.396138         0.0  0.0         0.0 | 132.682172  121.612683  0.0  0.0  0.0 | 5.147815   1.21954  0.0  0.0  0.0 |  7.500135  5.286012  0.0  0.0  0.0 | 17.621688 | 0.344366 |
    | 02/01 04:00 |  3.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  127.241232       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.235    9.086    20.02   4.235   15.015 | 0.0       0.0  0.0  0.0       0.0 |  4.235     9.086    20.02   4.235    15.015 |        0.0         0.0       20.02  0.0      15.015 |       0.0        0.0        0.0  0.0        0.0 |        0.0    18.76875         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.235        0.0  0.0   4.235  0.0 | 1.863883       0.0  0.0   4.235  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.422628  0.479852  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.830925  0.526243  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.093369 |      0.0 |      0.0 | 0.087144 | 0.363102 |   2.0       2.0  0.0  0.0       2.0 |        0.0   93.803094       200.0  0.0       200.0 |        0.0   33.438138         0.0  0.0         0.0 |  135.05329  121.612683  0.0  0.0  0.0 | 4.589069  0.739689  0.0  0.0  0.0 |  9.091839  5.239621  0.0  0.0  0.0 | 18.580433 | 0.363102 |
    | 02/01 05:00 |  3.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  154.300132       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.114   8.8264   19.448   4.114   14.586 | 0.0       0.0  0.0  0.0       0.0 |  4.114    8.8264   19.448   4.114    14.586 |        0.0         0.0      19.448  0.0      14.586 |       0.0        0.0        0.0  0.0        0.0 |        0.0     18.2325         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.114        0.0  0.0   4.114  0.0 | 1.875921       0.0  0.0   4.114  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.205344  0.291045  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.971844  0.512695  0.0  0.0   0.0 |  0.00443  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.098309 | 0.001704 | 0.001704 | 0.093231 | 0.388464 |   2.0       2.0  0.0  0.0       2.0 |        0.0  115.965194       200.0  0.0       200.0 |        0.0   38.334938         0.0  0.0         0.0 | 137.291368  121.612683  0.0  0.0  0.0 | 4.259646  0.448644  0.0  0.0  0.0 | 10.320909  5.017971  0.0  0.0  0.0 | 19.563411 | 0.388464 |
    | 02/01 06:00 |  1.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  163.850332       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.452   3.1152    6.864   1.452    5.148 | 0.0       0.0  0.0  0.0       0.0 |  1.452    3.1152    6.864   1.452     5.148 |        0.0         0.0       6.864  0.0       5.148 |       0.0        0.0        0.0  0.0        0.0 |        0.0       6.435         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     1.452        0.0  0.0   1.452  0.0 | 0.684216       0.0  0.0   1.452  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.82182  0.176528  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.067569  0.486062  0.0  0.0   0.0 | 0.067569  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.101455 | 0.025988 | 0.025988 | 0.117214 | 0.488392 |   2.0       2.0  0.0  0.0       2.0 |        0.0  123.812994       200.0  0.0       200.0 |        0.0   40.037338         0.0  0.0         0.0 | 138.059152  121.612683  0.0  0.0  0.0 | 3.122042  0.272116  0.0  0.0  0.0 | 11.007591  4.708436  0.0  0.0  0.0 | 20.189534 | 0.488392 |
    | 02/01 07:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121        0.0  0.0   0.121  0.0 | 0.057657       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.240712  0.107069  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.102594  0.453246  0.0  0.0   0.0 | 0.102594  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.103651 | 0.039459 | 0.039459 | 0.130939 | 0.545578 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.503644       200.0  0.0       200.0 |        0.0   40.142538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 | 1.938987  0.165047  0.0  0.0  0.0 | 11.043115  4.362259  0.0  0.0  0.0 | 20.626452 | 0.545578 |
    | 02/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.762932  0.064941  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.083441  0.418265  0.0  0.0   0.0 | 0.083441  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.105664 | 0.032093 | 0.032093 | 0.126434 | 0.526808 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.543644       200.0  0.0       200.0 |        0.0   40.102538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 | 1.176055  0.100106  0.0  0.0  0.0 | 10.639164  4.008935  0.0  0.0  0.0 | 21.027235 | 0.526808 |
    | 02/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  164.646182       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 |        0.0         0.0         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.462742  0.039389  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.032772  0.383406  0.0  0.0   0.0 | 0.032772  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.107528 | 0.012605 | 0.012605 | 0.111284 | 0.463682 |   2.0       2.0  0.0  0.0       2.0 |        0.0  124.583644       200.0  0.0       200.0 |        0.0   40.062538         0.0  0.0         0.0 | 138.122495  121.612683  0.0  0.0  0.0 | 0.713313  0.060717  0.0  0.0  0.0 | 10.036361  3.664917  0.0  0.0  0.0 | 21.398098 | 0.463682 |
    | 02/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  167.829582       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 |        0.0         0.0       2.288  0.0       1.716 |       0.0        0.0        0.0  0.0        0.0 |        0.0       2.145         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.484        0.0  0.0   0.484  0.0 | 0.230842       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.32985   0.02389  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.971033  0.349919  0.0  0.0   0.0 | 0.000098  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.10957 | 0.000038 | 0.000038 | 0.102298 | 0.426242 |   2.0       2.0  0.0  0.0       2.0 |        0.0  127.226244       200.0  0.0       200.0 |        0.0   40.603338         0.0  0.0         0.0 | 138.375653  121.612683  0.0  0.0  0.0 | 0.614305  0.036827  0.0  0.0  0.0 |  9.395081  3.338889  0.0  0.0  0.0 | 21.804446 | 0.426242 |
    | 02/01 11:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  168.625432       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 |        0.0         0.0       0.572  0.0       0.429 |       0.0        0.0        0.0  0.0        0.0 |        0.0     0.53625         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     0.121        0.0  0.0   0.121  0.0 | 0.057922       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254051   0.01449  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.90635  0.318438  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.111182 |      0.0 |      0.0 |  0.10377 | 0.432375 |   2.0       2.0  0.0  0.0       2.0 |        0.0  127.916894       200.0  0.0       200.0 |        0.0   40.708538         0.0  0.0         0.0 | 138.438731  121.612683  0.0  0.0  0.0 | 0.418176  0.022337  0.0  0.0  0.0 |  8.742782  3.034941  0.0  0.0  0.0 | 22.125228 | 0.432375 |
    | 02/01 12:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |        0.0  197.276032       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |        0.0         0.0      20.592  0.0      15.444 |       0.0        0.0        0.0  0.0        0.0 |        0.0      19.305         0.0  0.0         0.0 |       0.0        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |     4.356        0.0  0.0   4.356  0.0 | 2.087099       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.609219  0.008789  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.861456  0.289238  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.115689 |      0.0 |      0.0 | 0.107976 | 0.449902 |   2.0       2.0  0.0  0.0       2.0 |        0.0  151.380294       200.0  0.0       200.0 |        0.0   45.895738         0.0  0.0         0.0 | 140.707632  121.612683  0.0  0.0  0.0 | 1.896056  0.013548  0.0  0.0  0.0 |  8.490545  2.754492  0.0  0.0  0.0 | 23.022127 | 0.449902 |
    | 02/01 13:00 |  5.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  42.523945       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  7.139  15.3164   33.748   7.139   25.311 | 0.0       0.0  0.0  0.0       0.0 |  7.139   15.3164   33.748   7.139    25.311 |        0.0    9.366467      33.748  0.0      25.311 |       0.0   3.225965        0.0  0.0        0.0 |  32.804173         0.0         0.0  0.0         0.0 |  2.580772        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |       0.0        0.0  0.0   7.139  0.0 |      0.0       0.0  0.0   7.139  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.74604  0.005331  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.844071  0.262382  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.122092 |      0.0 |      0.0 | 0.113953 | 0.474802 |   2.0       2.0  0.0  0.0       2.0 |   30.45818  148.803428       200.0  0.0       200.0 |  12.065766   51.196572         0.0  0.0         0.0 | 140.707632  121.612683  0.0  0.0  0.0 | 1.150016  0.008217  0.0  0.0  0.0 |  8.392514   2.49744  0.0  0.0  0.0 | 24.296312 | 0.474802 |
    | 02/01 14:00 |  1.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 |  48.329573       200.0       200.0  0.0       200.0 | 0.0  0.0  0.902305  0.0   0.0 |  1.331   2.8556    6.292   1.331    4.719 | 0.0       0.0  0.0  0.0       0.0 |  1.331    2.8556    6.292   1.331     4.719 |        0.0    2.112422       6.292  0.0       4.719 |       0.0   0.743178        0.0  0.0        0.0 |   6.408937         0.0         0.0  0.0         0.0 |  0.594543        0.0        0.0  0.0        0.0 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04       0.0  0.0       0.0 |  2.528852        0.0  0.0   1.331  0.0 | 1.251696       0.0  0.0   1.331  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.719184  0.003233  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.833443  0.237819  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.12426 |      0.0 |      0.0 | 0.115976 | 0.483235 |   2.0       2.0  0.0  0.0       2.0 |  34.521124  147.989406       200.0  0.0       200.0 |  13.808449   52.010594         0.0  0.0         0.0 | 141.984788  121.612683  0.0  0.0  0.0 | 1.682528  0.004984  0.0  0.0  0.0 |  8.278255  2.262854  0.0  0.0  0.0 | 24.727802 | 0.483235 |
    | 02/01 15:00 | 20.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 201.988205  191.984574  201.988205  0.0  201.988205 | 0.0  0.0  0.902305  0.0   0.0 | 25.047  53.7372  118.404  25.047   88.803 | 0.0       0.0  0.0  0.0       0.0 | 25.047   53.7372  118.404  25.047    88.803 |        0.0   36.356814     118.404  0.0      88.803 |       0.0  17.380386        0.0  0.0        0.0 | 115.042936    1.778447    1.778447  0.0    1.778447 | 13.568696   0.209758   0.209758  0.0   0.209758 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04  0.209758  0.0  0.209758 |       0.0  10.003631  0.0  25.047  0.0 |      0.0  3.698754  0.0  25.047  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.662023  0.790022  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.819805  0.253556  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.143356 |      0.0 |      0.0 | 0.133799 | 0.557495 |   2.0       2.0  0.0  0.0       2.0 | 147.218067  137.131839  201.988205  0.0  201.988205 |  54.770138   54.852736         0.0  0.0         0.0 | 141.984788  127.917561  0.0  0.0  0.0 | 1.020505  2.913715  0.0  0.0  0.0 |  8.120473  2.799321  0.0  0.0  0.0 | 28.527821 | 0.557495 |
    | 02/01 16:00 | 37.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 290.938862  302.004664  309.356596  0.0  309.356596 | 0.0  0.0  0.902305  0.0   0.0 | 45.859  98.3884  216.788  45.859  162.591 | 0.0       0.0  0.0  0.0       0.0 | 45.859   98.3884  216.788  45.859   162.591 |  28.420627   56.173848  218.776205  0.0  164.579205 | 19.426578  34.199126        0.0  0.0        0.0 |  91.362027   91.362027   91.362027  0.0   91.362027 | 17.994569  17.994569  17.994569  0.0  17.994569 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 | 18.417734   7.351931  0.0  45.859  0.0 |  9.28239  3.007473  0.0  45.859  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.379256  1.787234  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.88786  0.352847  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.177485 |      0.0 |      0.0 | 0.165653 |  0.69022 |   2.0       2.0  0.0  0.0       2.0 | 207.813473  215.717617  292.002027  0.0  291.702027 |  83.125389   86.287047   17.354569  0.0   17.654569 | 151.120131  132.262019  0.0  0.0  0.0 | 7.923639  4.133955  0.0  0.0  0.0 |  9.611869  4.233708  0.0  0.0  0.0 | 35.319527 |  0.69022 |
    | 02/01 17:00 |  8.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 323.906781  323.906781  323.906781  0.0  323.906781 | 0.0  0.0  0.902305  0.0   0.0 |  9.922  21.2872   46.904   9.922   35.178 | 0.0       0.0  0.0  0.0       0.0 |  9.922   21.2872   46.904   9.922    35.178 |  69.667573   85.844423  148.648653  0.0  137.128384 | 31.193289  37.447441   7.611943  0.0   7.406211 |  98.060761   98.060761   98.060761  0.0   98.060761 | 25.846019  25.846019  25.846019  0.0  25.846019 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   9.922  0.0 |      0.0       0.0  0.0   9.922  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 3.117709  1.626584  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.063687  0.481575  0.0  0.0   0.0 | 0.067201  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.186272 | 0.025846 | 0.025846 | 0.196254 | 0.817726 |   2.0       2.0  0.0  0.0       2.0 | 233.860668  237.354756  288.958135  0.0  288.152404 |  90.046113   86.552025   34.948645  0.0   35.754377 | 151.120131  132.262019  0.0  0.0  0.0 |  4.80593   2.50737  0.0  0.0  0.0 | 11.598691  5.378717  0.0  0.0  0.0 | 37.068165 | 0.817726 |
    | 02/01 18:00 |  3.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 334.397735  334.397735  334.397735  0.0  334.397735 | 0.0  0.0  0.902305  0.0   0.0 |  4.356   9.3456   20.592   4.356   15.444 | 0.0       0.0  0.0  0.0       0.0 |  4.356    9.3456   20.592   4.356    15.444 |  91.376852   96.554068  129.839697  0.0  124.668627 | 36.885929  36.698313  14.659084  0.0  14.682154 | 104.406788  104.406788  104.406788  0.0  104.406788 | 29.990946  29.990946  29.990946  0.0  29.990946 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   4.356  0.0 |      0.0       0.0  0.0   4.356  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.890986  0.986573  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.186619  0.559577  0.0  0.0   0.0 | 0.186619  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.191371 | 0.071776 | 0.071776 | 0.240819 | 1.003412 |   2.0       2.0  0.0  0.0       2.0 | 244.544611  249.365877  284.757227  0.0  283.674565 |  89.853124   85.031858   49.640507  0.0   50.723169 | 151.120131  132.262019  0.0  0.0  0.0 | 2.914944  1.520797  0.0  0.0  0.0 | 12.116439  5.805713  0.0  0.0  0.0 | 38.082751 | 1.003412 |
    | 02/01 19:00 |  7.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 356.253888  356.253888  356.253888  0.0  356.253888 | 0.0  0.0  0.902305  0.0   0.0 |  9.075    19.47     42.9   9.075   32.175 | 0.0       0.0  0.0  0.0       0.0 |  9.075     19.47     42.9   9.075    32.175 | 102.149255  112.159273  153.970933  0.0  143.523838 | 41.323479  41.708462  23.326802  0.0  23.048897 | 120.370769  120.370769  120.370769  0.0  120.370769 | 35.883119  35.883119  35.883119  0.0  35.883119 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   9.075  0.0 |      0.0       0.0  0.0   9.075  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.146941  0.598387  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.198703  0.581433  0.0  0.0   0.0 | 0.198703  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.199867 | 0.076424 | 0.076424 | 0.252777 | 1.053238 |   2.0       2.0  0.0  0.0       2.0 | 260.420132  266.197374  294.697064  0.0  293.036497 |  95.833757   90.056515   61.556824  0.0   63.217392 | 151.120131  132.262019  0.0  0.0  0.0 | 1.768003   0.92241  0.0  0.0  0.0 | 11.865973  5.822667  0.0  0.0  0.0 | 39.773545 | 1.053238 |
    | 02/01 20:00 | 18.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 410.165735  410.165735  410.165735  0.0  410.165735 | 0.0  0.0  0.902305  0.0   0.0 | 22.385   48.026   105.82  22.385   79.365 | 0.0       0.0  0.0  0.0       0.0 | 22.385    48.026   105.82  22.385    79.365 | 122.864197  145.201755  227.160779  0.0   201.42567 | 55.774692  59.078133   34.91311  0.0  34.193218 | 159.848924  159.848924  159.848924  0.0  159.848924 | 50.316811  50.316811  50.316811  0.0  50.316811 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  22.385  0.0 |      0.0       0.0  0.0  22.385  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.695655   0.36294  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.15487  0.571657  0.0  0.0   0.0 |  0.15487  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.217736 | 0.059565 | 0.059565 | 0.254844 | 1.061848 |   2.0       2.0  0.0  0.0       2.0 | 295.058865  302.048542  333.845209  0.0   331.16475 | 115.106869  108.117192   76.320526  0.0   79.000985 | 151.120131  132.262019  0.0  0.0  0.0 | 1.072348   0.55947  0.0  0.0  0.0 | 11.251888   5.61395  0.0  0.0  0.0 | 43.329451 | 1.061848 |
    | 02/01 21:00 | 15.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 455.043704  455.043704  455.043704  0.0  455.043704 | 0.0  0.0  0.902305  0.0   0.0 | 18.634  39.9784   88.088  18.634   66.066 | 0.0       0.0  0.0  0.0       0.0 | 18.634   39.9784   88.088  18.634    66.066 | 157.438041  177.637793  252.568414  0.0  230.408289 | 71.361694  72.506342  45.685321  0.0  45.823446 | 191.219608  191.219608  191.219608  0.0  191.219608 | 63.824096  63.824096  63.824096  0.0  63.824096 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0  18.634  0.0 |      0.0       0.0  0.0  18.634  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.421936  0.220134  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.086395  0.544887  0.0  0.0   0.0 | 0.086395  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.232676 | 0.033229 | 0.033229 | 0.245962 | 1.024843 |   2.0       2.0  0.0  0.0       2.0 | 326.494439  333.287958  361.224403  0.0  358.382069 | 128.549264  121.755746     93.8193  0.0   96.661635 | 151.120131  132.262019  0.0  0.0  0.0 | 0.650412  0.339336  0.0  0.0  0.0 | 10.501033  5.289198  0.0  0.0  0.0 | 46.302455 | 1.024843 |
    | 02/01 22:00 |  6.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 473.402873  473.402873  473.402873  0.0  473.402873 | 0.0  0.0  0.902305  0.0   0.0 |  7.623  16.3548   36.036   7.623   27.027 | 0.0       0.0  0.0  0.0       0.0 |  7.623   16.3548   36.036   7.623    27.027 | 185.358526  196.033453  235.469802  0.0  225.511749 | 77.308178  75.365051  55.609902  0.0  56.558955 | 203.295969  203.295969  203.295969  0.0  203.295969 | 70.106904  70.106904  70.106904  0.0  70.106904 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   7.623  0.0 |      0.0       0.0  0.0   7.623  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255917  0.133518  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.010508  0.509793  0.0  0.0   0.0 | 0.015492  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.23949 | 0.005959 | 0.005959 | 0.228688 | 0.952867 |   2.0       2.0  0.0  0.0       2.0 | 342.085889  347.797674   365.72657  0.0  363.533289 | 131.316984  125.605199  107.676303  0.0  109.869584 | 151.120131  132.262019  0.0  0.0  0.0 | 0.394495  0.205817  0.0  0.0  0.0 |   9.73095  4.912923  0.0  0.0  0.0 | 47.658516 | 0.952867 |
    | 02/01 23:00 |  1.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451153  0.451153  0.451153  0.0  0.451153 | 478.939765  478.939765  478.939765  0.0  478.939765 | 0.0  0.0  0.902305  0.0   0.0 |  2.299   4.9324   10.868   2.299    8.151 | 0.0       0.0  0.0  0.0       0.0 |  2.299    4.9324   10.868   2.299     8.151 | 198.262243  203.642415  221.064023  0.0  217.315563 |  77.43963  74.692858   63.20685  0.0   64.23831 |  206.64527   206.64527   206.64527  0.0   206.64527 | 72.294496  72.294496  72.294496  0.0  72.294496 | 0.0  0.0  0.0  0.0     0.0 | 2.345993  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.299  0.0 |      0.0       0.0  0.0   2.299  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.155222  0.080983  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.933531  0.471444  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.242275 |      0.0 |      0.0 | 0.226124 | 0.942181 |   2.0       2.0  0.0  0.0       2.0 | 348.122922  353.014129  362.815817  0.0  361.353996 | 130.816843  125.925637  116.123948  0.0   117.58577 | 151.120131  132.262019  0.0  0.0  0.0 | 0.239273  0.124835  0.0  0.0  0.0 |  8.952641  4.522462  0.0  0.0  0.0 | 48.212771 | 0.942181 |
    | 03/01 00:00 |  4.9 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 493.219119  493.219119  493.219119  0.0  493.219119 | 0.0  0.0  0.902686  0.0   0.0 |  5.929  12.7204   28.028   5.929   21.021 | 0.0       0.0  0.0  0.0       0.0 |  5.929   12.7204   28.028   5.929    21.021 | 204.528223  212.738588  236.654993  0.0  229.412995 | 80.340542  78.921577  70.312772  0.0  70.547771 | 216.382573  216.382573  216.382573  0.0  216.382573 | 76.836547  76.836547  76.836547  0.0  76.836547 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   5.929  0.0 |      0.0       0.0  0.0   5.929  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.094147  0.049119  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.856511  0.432745  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.247447 |      0.0 |      0.0 |  0.23095 | 0.962292 |   2.0       2.0  0.0  0.0       2.0 | 357.630288  362.303713  371.211396  0.0  369.684574 | 135.588831  130.915406  122.007723  0.0  123.534546 | 151.120131  132.262019  0.0  0.0  0.0 | 0.145127  0.075716  0.0  0.0  0.0 |  8.190277  4.138836  0.0  0.0  0.0 | 49.241863 | 0.962292 |
    | 03/01 01:00 |  2.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 501.087335  501.087335  501.087335  0.0  501.087335 | 0.0  0.0  0.902686  0.0   0.0 |  3.267   7.0092   15.444   3.267   11.583 | 0.0       0.0  0.0  0.0       0.0 |  3.267    7.0092   15.444   3.267    11.583 | 213.565721  219.302218  234.627313  0.0  230.211324 | 82.920398  80.926101  74.035807  0.0  74.590795 | 221.471596  221.471596  221.471596  0.0  221.471596 | 79.615739  79.615739  79.615739  0.0  79.615739 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   3.267  0.0 |      0.0       0.0  0.0   3.267  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.057103  0.029792  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.78217  0.395303  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.250504 |      0.0 |      0.0 | 0.233804 | 0.974184 |   2.0       2.0  0.0  0.0       2.0 | 363.189179   367.60189  374.139679  0.0  372.867845 | 137.898156  133.485444  126.947655  0.0   128.21949 | 151.120131  132.262019  0.0  0.0  0.0 | 0.088024  0.045924  0.0  0.0  0.0 |  7.465209  3.773324  0.0  0.0  0.0 | 49.850363 | 0.974184 |
    | 03/01 02:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 502.544412  502.544412  502.544412  0.0  502.544412 | 0.0  0.0  0.902686  0.0   0.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 218.403559  221.603573  227.380998  0.0  225.971212 | 83.288776  80.781762  76.566337  0.0  77.261123 | 222.042022  222.042022  222.042022  0.0  222.042022 | 80.502389  80.502389  80.502389  0.0  80.502389 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.034635   0.01807  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.712084  0.359953  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.25147 |      0.0 |      0.0 | 0.234705 | 0.977938 |   2.0       2.0  0.0  0.0       2.0 | 364.480659   368.65234  372.300704  0.0  371.423655 | 138.063753  133.892072  130.243708  0.0  131.120756 | 151.120131  132.262019  0.0  0.0  0.0 | 0.053389  0.027854  0.0  0.0  0.0 |   6.78776   3.43144  0.0  0.0  0.0 | 50.042482 | 0.977938 |
    | 03/01 03:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 503.127242  503.127242  503.127242  0.0  503.127242 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 219.496367  222.227265  225.160687  0.0  224.375469 | 83.290045  80.836347  78.527724  0.0  79.026943 |  222.01911   222.01911   222.01911  0.0   222.01911 | 81.108133  81.108133  81.108133  0.0  81.108133 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.021007   0.01096  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.646957  0.327075  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.252008 |      0.0 |      0.0 | 0.235207 |  0.98003 |   2.0       2.0  0.0  0.0       2.0 | 364.656418  368.712985  370.943126  0.0  370.265296 | 138.470825  134.414258  132.184116  0.0  132.861946 | 151.120131  132.262019  0.0  0.0  0.0 | 0.032382  0.016895  0.0  0.0  0.0 |   6.16181  3.115325  0.0  0.0  0.0 | 50.149551 |  0.98003 |
    | 03/01 04:00 |  0.5 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 504.584319  504.584319  504.584319  0.0  504.584319 | 0.0  0.0  0.902686  0.0   0.0 |  0.605    1.298     2.86   0.605    2.145 | 0.0       0.0  0.0  0.0       0.0 |  0.605     1.298     2.86   0.605     2.145 | 219.874572  222.866862   226.05113  0.0  225.000538 |  83.85767   81.55838  79.936113  0.0  80.271705 | 222.614941  222.614941  222.614941  0.0  222.614941 | 81.969378  81.969378  81.969378  0.0  81.969378 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.605  0.0 |      0.0       0.0  0.0   0.605  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.012741  0.006647  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.58699  0.296784  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.252652 |      0.0 |      0.0 | 0.235809 | 0.982536 |   2.0       2.0  0.0  0.0       2.0 | 365.049803  369.073064  371.006938  0.0    370.3647 | 139.534516  135.511255  133.577381  0.0  134.219619 | 151.120131  132.262019  0.0  0.0  0.0 | 0.019641  0.010247  0.0  0.0  0.0 |  5.587561  2.825189  0.0  0.0  0.0 | 50.277762 | 0.982536 |
    | 03/01 05:00 |  2.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 511.578288  511.578288  511.578288  0.0  511.578288 | 0.0  0.0  0.902686  0.0   0.0 |  2.904   6.2304   13.728   2.904   10.296 | 0.0       0.0  0.0  0.0       0.0 |  2.904    6.2304   13.728   2.904    10.296 |   221.1845  226.239983  236.278139  0.0   232.79694 | 86.303819  84.574736   82.03418  0.0  82.083379 | 227.116446  227.116446  227.116446  0.0  227.116446 | 84.461843  84.461843  84.461843  0.0  84.461843 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   2.904  0.0 |      0.0       0.0  0.0   2.904  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.007728  0.004032  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.532101  0.269047  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254798 |      0.0 |      0.0 | 0.237811 |  0.99088 |   2.0       2.0  0.0  0.0       2.0 | 368.634765  372.735126  376.213244  0.0  375.320206 | 142.943524  138.843162  135.365044  0.0  136.258083 | 151.120131  132.262019  0.0  0.0  0.0 | 0.011913  0.006215  0.0  0.0  0.0 |  5.063189  2.560173  0.0  0.0  0.0 | 50.704729 |  0.99088 |
    | 03/01 06:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  512.74395   512.74395   512.74395  0.0   512.74395 | 0.0  0.0  0.902686  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 224.654326  227.589692  231.186173  0.0  230.127809 | 87.407963  85.026997  82.680115  0.0   83.16648 | 227.485902  227.485902  227.485902  0.0  227.485902 | 85.258048  85.258048  85.258048  0.0  85.258048 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.484  0.0 |      0.0       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.004687  0.002445  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.482053  0.243751  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255078 |      0.0 |      0.0 | 0.238073 | 0.991971 |   2.0       2.0  0.0  0.0       2.0 | 369.119357  373.128936  375.440973  0.0  374.734299 | 143.624593  139.615014  137.302977  0.0  138.009651 | 151.120131  132.262019  0.0  0.0  0.0 | 0.007225   0.00377  0.0  0.0  0.0 |  4.585823  2.318868  0.0  0.0  0.0 | 50.760598 | 0.991971 |
    | 03/01 07:00 |  0.2 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |  0.242   0.5192    1.144   0.242    0.858 | 0.0       0.0  0.0  0.0       0.0 |  0.242    0.5192    1.144   0.242     0.858 | 225.209233  227.873792  230.021905  0.0  229.334171 | 87.776717  85.389358  83.866045  0.0  84.267779 | 227.457017  227.457017  227.457017  0.0  227.457017 | 85.869764  85.869764  85.869764  0.0  85.869764 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.242  0.0 |      0.0       0.0  0.0   0.242  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.002843  0.001483  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.436536  0.220741  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255071 |      0.0 |      0.0 | 0.238066 | 0.991942 |   2.0       2.0  0.0  0.0       2.0 | 369.020157  372.980961  374.660085  0.0  374.055145 | 144.306624   140.34582  138.666696  0.0  139.271636 | 151.120131  132.262019  0.0  0.0  0.0 | 0.004382  0.002286  0.0  0.0  0.0 |   4.15213   2.09961  0.0  0.0  0.0 | 50.759073 | 0.991942 |
    | 03/01 08:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.244235  227.661848   228.68676  0.0  228.317514 | 88.082546  85.664932  84.640021  0.0  85.009267 | 227.018951  227.018951  227.018951  0.0  227.018951 |  86.30783   86.30783   86.30783  0.0   86.30783 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.001724    0.0009  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.395211  0.199848  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254787 |      0.0 |      0.0 | 0.237801 | 0.990837 |   2.0       2.0  0.0  0.0       2.0 |  368.44789  372.378063  373.632276  0.0  373.096582 | 144.878891  140.948718  139.694505  0.0  140.230199 | 151.120131  132.262019  0.0  0.0  0.0 | 0.002658  0.001387  0.0  0.0  0.0 |  3.758643  1.900662  0.0  0.0  0.0 | 50.702533 | 0.990837 |
    | 03/01 09:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.326781  513.326781  513.326781  0.0  513.326781 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.894931  227.293849  228.059401  0.0  227.732422 | 88.431849  86.032932   85.26738  0.0  85.594359 | 226.581591  226.581591  226.581591  0.0  226.581591 |  86.74519   86.74519   86.74519  0.0   86.74519 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0     0.0  0.0 |      0.0       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.001046  0.000546  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.357733  0.180898  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.25441 |      0.0 |      0.0 | 0.237449 | 0.989372 |   2.0       2.0  0.0  0.0       2.0 | 367.787565  371.705805  372.794465  0.0  372.285751 | 145.539216  141.620976  140.532315  0.0   141.04103 | 151.120131  132.262019  0.0  0.0  0.0 | 0.001612  0.000841  0.0  0.0  0.0 |  3.401956  1.720309  0.0  0.0  0.0 |  50.62757 | 0.989372 |
    | 03/01 10:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.201027  514.201027  514.201027  0.0  514.201027 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 |  224.59314  227.312585  229.081589  0.0  228.387074 | 89.096641  86.792995  85.961192  0.0  86.226707 | 226.757529  226.757529  226.757529  0.0  226.757529 | 87.443498  87.443498  87.443498  0.0  87.443498 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |       0.0        0.0  0.0   0.363  0.0 |      0.0       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.000634  0.000331  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.32377  0.163725  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254209 |      0.0 |      0.0 | 0.237262 | 0.988591 |   2.0       2.0  0.0  0.0       2.0 | 367.604971  371.533949  372.826406  0.0  372.283206 | 146.596056  142.667078  141.374621  0.0  141.917821 | 151.120131  132.262019  0.0  0.0  0.0 | 0.000978   0.00051  0.0  0.0  0.0 |  3.078821  1.556915  0.0  0.0  0.0 | 50.587628 | 0.988591 |
    | 03/01 11:00 |  2.6 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 519.961252  521.777827  521.777827  0.0  521.777827 | 0.0  0.0  0.902686  0.0   0.0 |  3.146   6.7496   14.872   3.146   11.154 | 0.0       0.0  0.0  0.0       0.0 |  3.146    6.7496   14.872   3.146    11.154 | 225.493409  230.729522  241.140791  0.0  237.464602 | 91.853618  90.221105  87.932236  0.0  87.890425 | 231.636316  231.636316  231.636316  0.0  231.636316 | 90.141511  90.141511  90.141511  0.0  90.141511 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.816574        0.0  0.0   3.146  0.0 | 1.037141       0.0  0.0   3.146  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.221359  0.000201  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.303697   0.14817  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255939 |      0.0 |      0.0 | 0.238877 | 0.995319 |   2.0       2.0  0.0  0.0       2.0 | 371.400895  375.455142  378.833931  0.0   377.94892 | 148.560358  146.322685  142.943896  0.0  143.828907 | 151.899565  132.262019  0.0  0.0  0.0 |  0.81676  0.000309  0.0  0.0  0.0 |  2.996484  1.408946  0.0  0.0  0.0 | 50.931915 | 0.995319 |
    | 03/01 12:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.773236  523.119052  523.119052  0.0  523.119052 | 0.0  0.0  0.902686  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 228.776083   232.53573  237.211776  0.0  235.766768 |  92.03217  91.059297  88.570051  0.0  89.014059 | 232.417341  232.417341  232.417341  0.0  232.417341 | 90.701711  90.701711  90.701711  0.0  90.701711 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.345816        0.0  0.0   0.847  0.0 | 0.776316       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.486773  0.000122  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |   0.3087  0.134085  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.256007 |      0.0 |      0.0 |  0.23894 | 0.995584 |   2.0       2.0  0.0  0.0       2.0 | 372.695169  376.177553  378.683497  0.0  377.942492 | 149.078068  146.941499  144.435556  0.0   145.17656 | 152.469064  132.262019  0.0  0.0  0.0 | 1.106303  0.000188  0.0  0.0  0.0 |  3.174556  1.274983  0.0  0.0  0.0 | 50.945468 | 0.995584 |
    | 03/01 13:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.992672  523.475677  523.475677  0.0  523.475677 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 229.937343  232.782522  235.439858  0.0  234.597488 | 92.198893   91.11533  89.395194  0.0  89.808564 | 232.441067  232.441067  232.441067  0.0  232.441067 |  91.03461   91.03461   91.03461  0.0   91.03461 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.483005        0.0  0.0   0.363  0.0 | 0.861878       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.618929  0.000074  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.332039  0.121334  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255753 |      0.0 |      0.0 | 0.238703 | 0.994595 |   2.0       2.0  0.0  0.0       2.0 | 372.851909  376.219298  378.040705  0.0  377.413071 | 149.140763  147.256379  145.434972  0.0  146.062606 | 153.090191  132.262019  0.0  0.0  0.0 | 1.349252  0.000114  0.0  0.0  0.0 |  3.461446  1.153723  0.0  0.0  0.0 | 50.894824 | 0.994595 |
    | 03/01 14:00 |  0.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 522.008863  523.779537  523.779537  0.0  523.779537 | 0.0  0.0  0.902686  0.0   0.0 |  0.363   0.7788    1.716   0.363    1.287 | 0.0       0.0  0.0  0.0       0.0 |  0.363    0.7788    1.716   0.363     1.287 | 230.094041  232.906119  235.140283  0.0  234.368133 | 92.261631  91.348357  90.051394  0.0  90.394544 |  232.45259   232.45259   232.45259  0.0   232.45259 | 91.326947  91.326947  91.326947  0.0  91.326947 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.770674        0.0  0.0   0.363  0.0 | 1.037465       0.0  0.0   0.363  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.751933  0.000045  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.365774  0.109793  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255543 |      0.0 |      0.0 | 0.238507 |  0.99378 |   2.0       2.0  0.0  0.0       2.0 | 372.863473  376.148968  377.709011  0.0  377.124528 | 149.145389  147.630568  146.070525  0.0  146.655009 |   153.8234  132.262019  0.0  0.0  0.0 | 1.634784  0.000069  0.0  0.0  0.0 |  3.847605  1.043974  0.0  0.0  0.0 | 50.853141 |  0.99378 |
    | 03/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 521.265413  523.098508  523.098508  0.0  523.098508 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 230.006331  232.520231  233.484587  0.0  233.123282 | 92.002532  91.259306   90.29495  0.0  90.656254 | 231.822278  231.822278  231.822278  0.0  231.822278 |  91.27623   91.27623   91.27623  0.0   91.27623 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  1.833096        0.0  0.0     0.0  0.0 | 1.084351       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.874271  0.000027  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  0.40844  0.099349  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255137 |      0.0 |      0.0 | 0.238128 | 0.992199 |   2.0       2.0  0.0  0.0       2.0 | 372.332438  375.491016  376.686703  0.0  376.163524 | 148.932975  147.607492  146.411805  0.0  146.934984 | 154.572144  132.262019  0.0  0.0  0.0 | 1.844864  0.000042  0.0  0.0  0.0 |  4.313436  0.944653  0.0  0.0  0.0 | 50.772261 | 0.992199 |
    | 03/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 520.375077  522.393471  522.393471  0.0  522.393471 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 229.475295  231.926846  232.665377  0.0  232.342229 | 91.790118  91.171662  90.433131  0.0  90.756279 | 231.186324  231.186324  231.186324  0.0  231.186324 | 91.207147  91.207147  91.207147  0.0  91.207147 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.018394        0.0  0.0     0.0  0.0 | 1.205615       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.982767  0.000016  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.458018  0.089896  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254808 |      0.0 |      0.0 |  0.23782 | 0.990919 |   2.0       2.0  0.0  0.0       2.0 | 371.696484  374.790495   375.84765  0.0  375.347619 | 148.678593  147.602977  146.545821  0.0  147.045852 | 155.384924  132.262019  0.0  0.0  0.0 | 2.067711  0.000025  0.0  0.0  0.0 |  4.838185  0.854773  0.0  0.0  0.0 | 50.706716 | 0.990919 |
    | 03/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 |  519.45772  521.617166  521.617166  0.0  521.617166 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.839341  231.300763  231.953183  0.0  231.644591 | 91.535736  91.092708  90.440288  0.0  90.748881 | 230.531069  230.531069  230.531069  0.0  230.531069 | 91.086097  91.086097  91.086097  0.0  91.086097 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.159446        0.0  0.0     0.0  0.0 | 1.303467       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |   1.0913   0.00001  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.513205  0.081343  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254566 |      0.0 |      0.0 | 0.237595 |  0.98998 |   2.0       2.0  0.0  0.0       2.0 | 371.041228    374.0608  375.065536  0.0  374.574097 | 148.416491  147.556365   146.55163  0.0  147.043069 | 156.240903  132.262019  0.0  0.0  0.0 | 2.279879  0.000015  0.0  0.0  0.0 |  5.416279   0.77344  0.0  0.0  0.0 | 50.658678 |  0.98998 |
    | 03/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 518.519709   520.78661   520.78661  0.0   520.78661 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 228.184085    230.6373  231.256796  0.0  230.953787 | 91.273634  90.979866   90.36037  0.0  90.663379 | 229.861062  229.861062  229.861062  0.0  229.861062 | 90.925548  90.925548  90.925548  0.0  90.925548 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |    2.2669        0.0  0.0     0.0  0.0 | 1.383445       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.191821  0.000006  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.573081  0.073603  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254422 |      0.0 |      0.0 |  0.23746 | 0.989418 |   2.0       2.0  0.0  0.0       2.0 | 370.371221  373.324563  374.309801  0.0  373.821372 | 148.148488  147.462047  146.476809  0.0  146.965237 | 157.124358  132.262019  0.0  0.0  0.0 | 2.471503  0.000009  0.0  0.0  0.0 |   6.03502  0.699843  0.0  0.0  0.0 | 50.629957 | 0.989418 |
    | 03/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 517.565962  519.914725  519.914725  0.0  519.914725 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 227.514078  229.955069  230.561942  0.0  230.261087 | 91.005631  90.831541  90.224668  0.0  90.525523 | 229.179814  229.179814  229.179814  0.0  229.179814 | 90.734911  90.734911  90.734911  0.0  90.734911 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.348763        0.0  0.0     0.0  0.0 |  1.44966       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.281327  0.000004  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.636291  0.066599  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254381 |      0.0 |      0.0 | 0.237422 |  0.98926 |   2.0       2.0  0.0  0.0       2.0 | 369.689973  372.589307  373.567673  0.0  373.080099 | 147.875989  147.325418  146.347052  0.0  146.834625 | 158.023461  132.262019  0.0  0.0  0.0 | 2.639836  0.000006  0.0  0.0  0.0 |  6.680056  0.633248  0.0  0.0  0.0 | 50.621851 |  0.98926 |
    | 03/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 516.600219  519.011355  519.011355  0.0  519.011355 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  226.83283  229.262223  229.864233  0.0  229.564218 | 90.733132  90.652502  90.050492  0.0  90.350507 | 228.489997  228.489997  228.489997  0.0  228.489997 | 90.521358  90.521358  90.521358  0.0  90.521358 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.411136        0.0  0.0     0.0  0.0 | 1.505237       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.359402  0.000002  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.701451  0.060262  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254448 |      0.0 |      0.0 | 0.237485 |  0.98952 |   2.0       2.0  0.0  0.0       2.0 | 369.000156  371.857081  372.833437  0.0  372.345878 | 147.600062  147.154274  146.177917  0.0  146.665476 |  158.92936  132.262019  0.0  0.0  0.0 | 2.785671  0.000003  0.0  0.0  0.0 |  7.338007  0.572988  0.0  0.0  0.0 | 50.635139 |  0.98952 |
    | 03/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 515.625325  518.083995  518.083995  0.0  518.083995 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 226.143013   228.56269  229.162809  0.0   228.86313 | 90.457205  90.448665  89.848546  0.0  90.148225 | 227.793645  227.793645  227.793645  0.0  227.793645 |  90.29035   90.29035   90.29035  0.0   90.29035 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.458669        0.0  0.0     0.0  0.0 |  1.55256       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.426866  0.000001  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.767327  0.054527  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254624 |      0.0 |      0.0 | 0.237649 | 0.990204 |   2.0       2.0  0.0  0.0       2.0 | 368.303804  371.128036  372.104273  0.0  371.616393 | 147.321522  146.955959  145.979722  0.0  146.467601 | 159.835469  132.262019  0.0  0.0  0.0 | 2.911364  0.000002  0.0  0.0  0.0 |  7.997546  0.518463  0.0  0.0  0.0 |  50.67014 | 0.990204 |
    | 03/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 514.643451  517.138353  517.138353  0.0  517.138353 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 225.446661  227.858589  228.457962  0.0  228.158422 | 90.178664  90.225405  89.626033  0.0  89.925572 | 227.092305  227.092305  227.092305  0.0  227.092305 | 90.046047  90.046047  90.046047  0.0  90.046047 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.494902        0.0  0.0     0.0  0.0 | 1.593455       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.485036  0.000001  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.832905  0.049338  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.254909 |      0.0 |      0.0 | 0.237915 | 0.991311 |   2.0       2.0  0.0  0.0       2.0 | 367.602465  370.401752  371.378616  0.0  370.890276 | 147.040986  146.736601  145.759736  0.0  146.248076 | 160.736916  132.262019  0.0  0.0  0.0 | 3.019783  0.000001  0.0  0.0  0.0 |  8.649678  0.469125  0.0  0.0  0.0 | 50.726794 | 0.991311 |
    | 03/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451343  0.451343  0.451343  0.0  0.451343 | 513.656246  516.178775  516.178775  0.0  516.178775 | 0.0  0.0  0.902686  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 224.745322  227.151208  227.750276  0.0  227.450799 | 89.898129  89.987145  89.388076  0.0  89.687554 | 226.387159  226.387159  226.387159  0.0  226.387159 | 89.791616  89.791616  89.791616  0.0  89.791616 | 0.0  0.0  0.0  0.0     0.0 | 2.346984  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.522529        0.0  0.0     0.0  0.0 | 1.629324       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.535338       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.897396  0.044643  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |   0.2553 |      0.0 |      0.0 |  0.23828 | 0.992834 |   2.0       2.0  0.0  0.0       2.0 | 366.897318  369.677703  370.655499  0.0  370.166637 | 146.758927  146.501072  145.523276  0.0  146.012138 | 161.630121  132.262019  0.0  0.0  0.0 | 3.113769  0.000001  0.0  0.0  0.0 |  9.287619  0.424482  0.0  0.0  0.0 | 50.804748 | 0.992834 |
    | 04/01 00:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.663479  515.208571  515.208571  0.0  515.208571 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0  0.000576  0.0  0.0  0.001152 |    0.0       0.0      0.0     0.0       0.0 | 224.040175  226.441398  227.040334  0.0  226.740888 |  89.61607  89.737377   89.13844  0.0  89.437887 | 225.679105  225.679105  225.679105  0.0  225.679105 | 89.529467  89.529467  89.529467  0.0  89.529467 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.545093        0.0  0.0     0.0  0.0 | 1.662219       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.579327       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 0.960233  0.040395  0.0  0.0   0.0 |      0.0  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.255796 |      0.0 |      0.0 | 0.238743 | 0.994763 |   2.0  1.999424  0.0  0.0  1.998848 | 366.188199   368.95541   369.93427  0.0  369.444854 |  146.47528  146.253161  145.274302  0.0  145.763718 | 162.512995  132.262019  0.0  0.0  0.0 | 3.196661       0.0  0.0  0.0  0.0 |  9.906714  0.384088  0.0  0.0  0.0 | 50.903433 | 0.994763 |
    | 04/01 01:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.668518   514.22969   514.22969  0.0   514.22969 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.331056  225.729761  226.328635  0.0  226.029206 | 89.332422   89.47881  88.879937  0.0  89.179365 | 224.968418  224.968418  224.968418  0.0  224.968418 | 89.261271  89.261271  89.261271  0.0  89.261271 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.561172        0.0  0.0     0.0  0.0 | 1.691044       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.618084       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.020428  0.036551  0.0  0.0   0.0 | 0.021125  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.256392 | 0.008125 | 0.008125 |  0.24634 | 1.026418 |   2.0  1.999424  0.0  0.0  1.998848 | 365.477513  368.234067  369.214053  0.0  368.724065 | 146.191005  145.995622  145.015637  0.0  145.505624 | 163.383123  132.262019  0.0  0.0  0.0 | 3.269621       0.0  0.0  0.0  0.0 | 10.483246  0.347537  0.0  0.0  0.0 | 51.021923 | 1.026418 |
    | 04/01 02:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.671177  513.244623  513.244623  0.0  513.244623 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 |  222.62037  225.016328  225.615167  0.0  225.315751 | 89.048148  89.213361  88.614523  0.0  88.913939 | 224.256032  224.256032  224.256032  0.0  224.256032 | 88.988592  88.988592  88.988592  0.0  88.988592 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.573447        0.0  0.0     0.0  0.0 | 1.717393       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.652405       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.074349  0.033073  0.0  0.0   0.0 | 0.074349  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.257075 | 0.028596 | 0.028596 |  0.26472 |    1.103 |   2.0  1.999424  0.0  0.0  1.998848 | 364.765126  367.513771  368.494918  0.0  368.004346 | 145.906051  145.730853  144.749706  0.0  145.240277 | 164.239177  132.262019  0.0  0.0  0.0 | 3.334608       0.0  0.0  0.0  0.0 | 10.986953  0.314465  0.0  0.0  0.0 | 51.157993 |    1.103 |
    | 04/01 03:00 |  1.3 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.766676   516.04297   516.04297  0.0   516.04297 | 0.0  0.0  0.903096  0.0   0.0 |  1.573   3.3748    7.436   1.573    5.577 | 0.0       0.0  0.0  0.0       0.0 |  1.573  3.374224    7.436   1.573  5.575848 | 222.346669  226.148508  231.531107  0.0   229.56884 | 89.897507  90.470339  89.149517  0.0  89.251632 | 226.191503  226.191503  226.191503  0.0  226.191503 | 89.851467  89.851467  89.851467  0.0  89.851467 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  3.276294        0.0  0.0   1.573  0.0 |  2.20941       0.0  0.0   1.573  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.782806       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.126109  0.029925  0.0  0.0   0.0 | 0.126109  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.258967 | 0.048503 | 0.048503 | 0.283739 | 1.182245 |   2.0       2.0  0.0  0.0       2.0 | 366.261912  369.083712  371.231314  0.0  370.542858 | 146.504765  146.959259  144.811656  0.0  145.500113 | 165.306061  132.262019  0.0  0.0  0.0 | 3.761212       0.0  0.0  0.0  0.0 | 11.517542   0.28454  0.0  0.0  0.0 | 51.534472 | 1.182245 |
    | 04/01 04:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 512.270864  514.782857  514.782857  0.0  514.782857 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.404769  226.039922  227.355189  0.0  226.933554 | 89.361908  90.003049  88.687781  0.0  89.109416 | 225.398666  225.398666  225.398666  0.0  225.398666 | 89.384192  89.384192  89.384192  0.0  89.384192 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.511993        0.0  0.0     0.0  0.0 | 1.716074       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.845551       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.177136  0.027078  0.0  0.0   0.0 | 0.177136  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.259813 | 0.068129 | 0.068129 | 0.301538 | 1.256407 |   2.0       2.0  0.0  0.0       2.0 |  365.90776  368.482456   369.91479  0.0  369.347969 | 146.363104  146.300402  144.868067  0.0  145.434888 |  166.10198  132.262019  0.0  0.0  0.0 | 3.631736       0.0  0.0  0.0  0.0 |  12.00882  0.257462  0.0  0.0  0.0 | 51.702801 | 1.256407 |
    | 04/01 05:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  511.28054  513.816706  513.816706  0.0  513.816706 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 223.050617  225.322111  226.197965  0.0  225.851361 | 89.220247  89.460746  88.584892  0.0  88.931496 | 224.691291  224.691291  224.691291  0.0  224.691291 | 89.125415  89.125415  89.125415  0.0  89.125415 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.536166        0.0  0.0     0.0  0.0 | 1.749312       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.801687       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.219609  0.024501  0.0  0.0   0.0 | 0.219609  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.260727 | 0.084465 | 0.084465 | 0.316548 | 1.318951 |   2.0       2.0  0.0  0.0       2.0 | 365.200386  367.891636  369.048117  0.0  368.527899 | 146.080154   145.92507  144.768589  0.0  145.288807 | 166.888834  132.262019  0.0  0.0  0.0 |  3.57936       0.0  0.0  0.0  0.0 | 12.371289  0.232961  0.0  0.0  0.0 | 51.884649 | 1.318951 |
    | 04/01 06:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.286928  512.841258  512.841258  0.0  512.841258 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 222.343243  224.692074  225.398402  0.0  225.080676 | 88.937297  89.124632  88.418304  0.0   88.73603 | 223.981569  223.981569  223.981569  0.0  223.981569 | 88.859689  88.859689  88.859689  0.0  88.859689 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.554329        0.0  0.0     0.0  0.0 | 1.778572       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.787314       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.251788  0.022169  0.0  0.0   0.0 | 0.251788  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.26169 | 0.096842 | 0.096842 | 0.328174 |  1.36739 |   2.0       2.0  0.0  0.0       2.0 | 364.490663   367.22113  368.271283  0.0  367.768792 | 145.796265  145.620127  144.569974  0.0  145.072466 | 167.664591  132.262019  0.0  0.0  0.0 | 3.570619       0.0  0.0  0.0  0.0 | 12.655026  0.210792  0.0  0.0  0.0 |  52.07636 |  1.36739 |
    | 04/01 07:00 |  0.7 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.955818  513.898731  513.898731  0.0  513.898731 | 0.0  0.0  0.903096  0.0   0.0 |  0.847   1.8172    4.004   0.847    3.003 | 0.0       0.0  0.0  0.0       0.0 |  0.847    1.8172    4.004   0.847     3.003 | 221.870249  225.005946  228.218517  0.0  227.018407 | 89.263679  89.652512  88.626741  0.0  88.825851 | 224.696076  224.696076  224.696076  0.0  224.696076 | 89.202655  89.202655  89.202655  0.0  89.202655 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.942913        0.0  0.0   0.847  0.0 | 2.068236       0.0  0.0   0.847  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 |  1.84559       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.280234   0.02006  0.0  0.0   0.0 | 0.280234  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.263302 | 0.107782 | 0.107782 | 0.339159 | 1.413164 |   2.0       2.0  0.0  0.0       2.0 | 364.968442   367.75206  369.392842  0.0  368.789461 | 145.987377  146.146671  144.505889  0.0   145.10927 | 168.539268  132.262019  0.0  0.0  0.0 | 3.793265       0.0  0.0  0.0  0.0 | 12.940149  0.190733  0.0  0.0  0.0 | 52.397016 | 1.413164 |
    | 04/01 08:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 511.181898  513.932503  513.932503  0.0  513.932503 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 222.246491  225.198159  227.670615  0.0   226.79139 | 89.193327  89.738972  88.516116  0.0  88.823341 | 224.756025  224.756025  224.756025  0.0  224.756025 | 89.176478  89.176478  89.176478  0.0  89.176478 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.750605        0.0  0.0   0.484  0.0 | 1.953307       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.908708       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.309031  0.018151  0.0  0.0   0.0 | 0.309031  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.264694 | 0.118858 | 0.118858 | 0.350058 | 1.458577 |   2.0       2.0  0.0  0.0       2.0 | 365.129927  367.807526  369.406253  0.0  368.810097 | 146.051971  146.124977   144.52625  0.0  145.122406 | 169.336566  132.262019  0.0  0.0  0.0 | 3.837864       0.0  0.0  0.0  0.0 | 13.230794  0.172582  0.0  0.0  0.0 | 52.674162 | 1.458577 |
    | 04/01 09:00 |  0.1 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.581776  513.165993  513.165993  0.0  513.165993 | 0.0  0.0  0.903096  0.0   0.0 |  0.121   0.2596    0.572   0.121    0.429 | 0.0       0.0  0.0  0.0       0.0 |  0.121    0.2596    0.572   0.121     0.429 | 222.306591  224.815129  226.159005  0.0  225.667272 | 88.996307  89.376974  88.345498  0.0  88.694231 | 224.225981  224.225981  224.225981  0.0  224.225981 | 88.940012  88.940012  88.940012  0.0  88.940012 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.584217        0.0  0.0   0.121  0.0 | 1.852552       0.0  0.0   0.121  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.904789       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.33519  0.016423  0.0  0.0   0.0 |  0.33519  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.265865 | 0.128919 | 0.128919 | 0.359871 | 1.499461 |   2.0       2.0  0.0  0.0       2.0 | 364.701269  367.372779  368.685229  0.0  368.137806 | 145.880507  145.793214  144.480764  0.0  145.028188 | 170.068231  132.262019  0.0  0.0  0.0 | 3.785627       0.0  0.0  0.0  0.0 | 13.465203  0.156159  0.0  0.0  0.0 | 52.907129 | 1.499461 |
    | 04/01 10:00 |  0.4 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 510.579336  513.337725  513.337725  0.0  513.337725 | 0.0  0.0  0.903096  0.0   0.0 |  0.484   1.0384    2.288   0.484    1.716 | 0.0       0.0  0.0  0.0       0.0 |  0.484    1.0384    2.288   0.484     1.716 | 221.979417  224.762609  227.032845  0.0   226.18834 | 89.086359  89.441784  88.421148  0.0  88.693653 | 224.325723  224.325723  224.325723  0.0  224.325723 | 89.012002  89.012002  89.012002  0.0  89.012002 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.758389        0.0  0.0   0.484  0.0 | 1.994536       0.0  0.0   0.484  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.914487       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.35689   0.01486  0.0  0.0   0.0 |  0.35689  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.267326 | 0.137265 | 0.137265 | 0.368467 |  1.53528 |   2.0       2.0  0.0  0.0       2.0 | 364.699526  367.433493  368.906107  0.0  368.331188 |  145.87981  145.904232  144.431618  0.0  145.006537 | 170.832084  132.262019  0.0  0.0  0.0 | 3.865677       0.0  0.0  0.0  0.0 |  13.66591  0.141298  0.0  0.0  0.0 | 53.197795 |  1.53528 |
    | 04/01 11:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 |  509.74056  512.276806  512.276806  0.0  512.276806 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.842383  224.278811  225.177685  0.0  224.826758 | 88.736953  89.058914  88.160041  0.0  88.510967 | 223.591306  223.591306  223.591306  0.0  223.591306 | 88.685501  88.685501  88.685501  0.0  88.685501 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.536246        0.0  0.0     0.0  0.0 |  1.85042       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.915278       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.375118  0.013446  0.0  0.0   0.0 | 0.375118  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.268464 | 0.144276 | 0.144276 | 0.375605 | 1.565023 |   2.0       2.0  0.0  0.0       2.0 |   364.1004  366.785987  367.959728  0.0  367.435736 |  145.64016  145.490819  144.317078  0.0  144.841071 |  171.51791  132.262019  0.0  0.0  0.0 | 3.800818       0.0  0.0  0.0  0.0 | 13.830952  0.127852  0.0  0.0  0.0 | 53.424287 | 1.565023 |
    | 04/01 12:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 508.746874  511.301327  511.301327  0.0  511.301327 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 221.243257  223.587629  224.303125  0.0  223.983707 | 88.497303  88.689177  87.973681  0.0  88.293099 |  222.88153   222.88153   222.88153  0.0   222.88153 | 88.419797  88.419797  88.419797  0.0  88.419797 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.554453        0.0  0.0     0.0  0.0 | 1.878698       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.895783       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.389164  0.012167  0.0  0.0   0.0 | 0.389164  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.269619 | 0.149678 | 0.149678 | 0.381366 | 1.589025 |   2.0       2.0  0.0  0.0       2.0 | 363.390624  366.119888  367.178132  0.0  366.673558 |  145.35625  145.181439  144.123195  0.0  144.627769 | 172.193665  132.262019  0.0  0.0  0.0 | 3.783733       0.0  0.0  0.0  0.0 | 13.948408  0.115685  0.0  0.0  0.0 | 53.654274 | 1.589025 |
    | 04/01 13:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 507.750567  510.318845  510.318845  0.0  510.318845 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 220.533481  222.908881  223.553184  0.0  223.245979 | 88.213393  88.392446  87.748143  0.0  88.055348 | 222.169882  222.169882  222.169882  0.0  222.169882 | 88.148963  88.148963  88.148963  0.0  88.148963 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.568278        0.0  0.0     0.0  0.0 | 1.903778       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.894404       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.399745  0.011009  0.0  0.0   0.0 | 0.399745  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.270787 | 0.153748 | 0.153748 | 0.385982 |  1.60826 |   2.0       2.0  0.0  0.0       2.0 | 362.678976  365.420888   366.43483  0.0  365.937461 | 145.071591  144.897957  143.884015  0.0  144.381384 | 172.858165  132.262019  0.0  0.0  0.0 | 3.793107       0.0  0.0  0.0  0.0 | 14.043323  0.104676  0.0  0.0  0.0 | 53.886541 |  1.60826 |
    | 04/01 14:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 506.752198  509.331046  509.331046  0.0  509.331046 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.821833  222.208114   222.82468  0.0  222.522236 | 87.928733  88.110731  87.494165  0.0  87.796609 | 221.456761  221.456761  221.456761  0.0  221.456761 | 87.874284  87.874284  87.874284  0.0  87.874284 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.578848        0.0  0.0     0.0  0.0 | 1.926396       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.902912       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.408746  0.009961  0.0  0.0   0.0 | 0.408746  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.271963 |  0.15721 |  0.15721 |  0.39008 | 1.625334 |   2.0       2.0  0.0  0.0       2.0 | 361.965856  364.709536  365.706911  0.0  365.211987 | 144.786342   144.62151  143.624134  0.0  144.119059 | 173.510617  132.262019  0.0  0.0  0.0 | 3.816591       0.0  0.0  0.0  0.0 | 14.128744  0.094715  0.0  0.0  0.0 | 54.120548 | 1.625334 |
    | 04/01 15:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 505.752249  508.339181  508.339181  0.0  508.339181 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 219.108713  221.498342  222.104076  0.0  221.803495 | 87.643485  87.832704   87.22697  0.0  87.527551 | 220.742512  220.742512  220.742512  0.0  220.742512 | 87.596669  87.596669  87.596669  0.0  87.596669 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.586932        0.0  0.0     0.0  0.0 |  1.94705       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.916553       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.417126  0.009013  0.0  0.0   0.0 | 0.417126  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.273146 | 0.160433 | 0.160433 | 0.393979 | 1.641577 |   2.0       2.0  0.0  0.0       2.0 | 361.251606  363.993706  364.985347  0.0  364.491004 | 144.500643  144.345475  143.353834  0.0  143.848177 | 174.150499  132.262019  0.0  0.0  0.0 | 3.847089       0.0  0.0  0.0  0.0 | 14.211043  0.085702  0.0  0.0  0.0 | 54.356094 | 1.641577 |
    | 04/01 16:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 504.751087  507.344207  507.344207  0.0  507.344207 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 218.394464  220.784715  221.386207  0.0  221.086357 | 87.357785  87.554466  86.952974  0.0  87.252824 | 220.027397  220.027397  220.027397  0.0  220.027397 | 87.316811  87.316811  87.316811  0.0  87.316811 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |   2.59312        0.0  0.0     0.0  0.0 | 1.966129       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.932618       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.425338  0.008156  0.0  0.0   0.0 | 0.425338  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.274337 | 0.163591 | 0.163591 | 0.397828 | 1.657615 |   2.0       2.0  0.0  0.0       2.0 | 360.536491  363.276387  364.266537  0.0  363.772044 | 144.214596   144.06782   143.07767  0.0  143.572164 |  174.77749  132.262019  0.0  0.0  0.0 |   3.8806       0.0  0.0  0.0  0.0 | 14.292985  0.077546  0.0  0.0  0.0 | 54.593136 | 1.657615 |
    | 04/01 17:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 503.748995  506.346853  506.346853  0.0  506.346853 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 217.679348  220.069317   220.66914  0.0  220.369581 | 87.071739   87.27489  86.675067  0.0  86.974626 | 219.311616  219.311616  219.311616  0.0  219.311616 | 87.035237  87.035237  87.035237  0.0  87.035237 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.597858        0.0  0.0     0.0  0.0 | 1.983931       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.949596       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.43356  0.007379  0.0  0.0   0.0 |  0.43356  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.275536 | 0.166754 | 0.166754 | 0.401687 | 1.673696 |   2.0       2.0  0.0  0.0       2.0 | 359.820711  362.558686  363.549013  0.0  363.054079 | 143.928284  143.788167  142.797841  0.0  143.292775 | 175.391417  132.262019  0.0  0.0  0.0 | 3.914935       0.0  0.0  0.0  0.0 | 14.375462  0.070167  0.0  0.0  0.0 | 54.831694 | 1.673696 |
    | 04/01 18:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 502.746187  505.347677  505.347677  0.0  505.347677 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.963568  219.353023  219.952184  0.0  219.652742 | 86.785427  86.993831   86.39467  0.0  86.694112 | 218.595325  218.595325  218.595325  0.0  218.595325 | 86.752352  86.752352  86.752352  0.0  86.752352 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |   2.60149        0.0  0.0     0.0  0.0 | 2.000686       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.966676       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.441835  0.006677  0.0  0.0   0.0 | 0.441835  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.276743 | 0.169936 | 0.169936 | 0.405571 | 1.689881 |   2.0       2.0  0.0  0.0       2.0 | 359.104419  361.840989  362.832154  0.0  362.336662 | 143.641768  143.506688  142.515523  0.0  143.011015 | 175.992222  132.262019  0.0  0.0  0.0 | 3.948945       0.0  0.0  0.0  0.0 | 14.458468  0.063489  0.0  0.0  0.0 |   55.0718 | 1.689881 |
    | 04/01 19:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 501.742828  504.347104  504.347104  0.0  504.347104 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 216.247276  218.636219  219.235114  0.0  218.935721 | 86.498911  86.711458  86.112564  0.0  86.411956 |  217.87864   217.87864   217.87864  0.0   217.87864 | 86.468464  86.468464  86.468464  0.0  86.468464 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.604276        0.0  0.0     0.0  0.0 | 2.016573       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.983442       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.450143  0.006042  0.0  0.0   0.0 | 0.450143  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.277957 | 0.173132 | 0.173132 | 0.409474 | 1.706143 |   2.0       2.0  0.0  0.0       2.0 | 358.387734   361.12341   362.11568  0.0  361.619581 | 143.355094  143.223694  142.231424  0.0  142.727523 | 176.579925  132.262019  0.0  0.0  0.0 | 3.982075       0.0  0.0  0.0  0.0 | 14.541624  0.057448  0.0  0.0  0.0 | 55.313477 | 1.706143 |
    | 04/01 20:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 500.739045   503.34546   503.34546  0.0   503.34546 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 215.530592  217.919094  218.517877  0.0  218.218507 | 86.212237   86.42801  85.829227  0.0  86.128597 | 217.161652  217.161652  217.161652  0.0  217.161652 | 86.183808  86.183808  86.183808  0.0  86.183808 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.606415        0.0  0.0     0.0  0.0 | 2.031731       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 1.999708       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.458442  0.005467  0.0  0.0   0.0 | 0.458442  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.27918 | 0.176324 | 0.176324 | 0.413381 | 1.722423 |   2.0       2.0  0.0  0.0       2.0 | 357.670746  360.405968  361.399455  0.0  360.902725 | 143.068299  142.939492  141.946005  0.0  142.442734 | 177.154609  132.262019  0.0  0.0  0.0 | 4.014098       0.0  0.0  0.0  0.0 | 14.624449  0.051981  0.0  0.0  0.0 | 55.556731 | 1.722423 |
    | 04/01 21:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 499.734933  502.342992  502.342992  0.0  502.342992 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.813603  217.201748  217.800482  0.0  217.501124 | 85.925441  86.143711  85.544978  0.0  85.844336 | 216.444429  216.444429  216.444429  0.0  216.444429 | 85.898563  85.898563  85.898563  0.0  85.898563 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.608059        0.0  0.0     0.0  0.0 | 2.046267       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.015405       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.466683  0.004947  0.0  0.0   0.0 | 0.466683  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 |  0.28041 | 0.179494 | 0.179494 | 0.417277 | 1.738654 |   2.0       2.0  0.0  0.0       2.0 | 356.953524  359.688648  360.683402  0.0  360.186031 | 142.781409  142.654344   141.65959  0.0  142.156961 | 177.716401  132.262019  0.0  0.0  0.0 | 4.044961       0.0  0.0  0.0  0.0 | 14.706487  0.047034  0.0  0.0  0.0 |  55.80155 | 1.738654 |
    | 04/01 22:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 498.730567  501.339893  501.339893  0.0  501.339893 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 214.096381  216.484243  217.082951  0.0    216.7836 | 85.638552   85.85875  85.260042  0.0  85.559392 | 215.727025  215.727025  215.727025  0.0  215.727025 | 85.612867  85.612867  85.612867  0.0  85.612867 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.609325        0.0  0.0     0.0  0.0 | 2.060266       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.030531       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 | 1.474827  0.004476  0.0  0.0   0.0 | 0.474827  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.281648 | 0.182626 | 0.182626 | 0.421147 | 1.754779 |   2.0       2.0  0.0  0.0       2.0 |  356.23612  358.971431  359.967476  0.0  359.469456 | 142.494448  142.368462  141.372416  0.0  141.870437 |  178.26546  132.262019  0.0  0.0  0.0 | 4.074695       0.0  0.0  0.0  0.0 | 14.787363  0.042558  0.0  0.0  0.0 | 56.047905 | 1.754779 |
    | 04/01 23:00 |  0.0 | -0.4 | 6.2  0.2  -11.8  6.2  -5.8 | 1.0  0.6  0.0  1.0       0.0 | 1.1  0.66  0.0  1.1  0.0 | 0.0  0.52  1.3  0.0  1.3 | 0.451548  0.451548  0.451548  0.0  0.451548 | 497.726004  500.336306  500.336306  0.0  500.336306 | 0.0  0.0  0.903096  0.0   0.0 |    0.0      0.0      0.0     0.0      0.0 | 0.0       0.0  0.0  0.0       0.0 |    0.0       0.0      0.0     0.0       0.0 | 213.378977  215.766617  216.365309  0.0  216.065964 | 85.351591  85.573276  84.974584  0.0  85.273929 |  215.00948   215.00948   215.00948  0.0   215.00948 | 85.326826  85.326826  85.326826  0.0  85.326826 | 0.0  0.0  0.0  0.0     0.0 | 2.348048  0.0  0.0  0.0   0.0 | 0.0  0.04      0.64  0.0      0.34 |  2.610302        0.0  0.0     0.0  0.0 | 2.073792       0.0  0.0     0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.0  0.0  0.0  0.0  0.0 | 2.045112       0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0   0.0 |  1.48284   0.00405  0.0  0.0   0.0 |  0.48284  0.0  0.0  0.0   0.0 | 0.0  0.0  0.0  0.0  0.0 | 0.282893 | 0.185708 | 0.185708 |  0.42498 | 1.770752 |   2.0       2.0  0.0  0.0       2.0 | 355.518575  358.254294  359.251648  0.0  358.752972 |  142.20743  142.082012  141.084658  0.0  141.583334 |  178.80197  132.262019  0.0  0.0  0.0 | 4.103375       0.0  0.0  0.0  0.0 | 14.866795  0.038508  0.0  0.0  0.0 | 56.295755 | 1.770752 |

There is no indication of an error in the water balance:

>>> round_(model.check_waterbalance(conditions))
0.0
"""
# import...
# ...from HydPy
from hydpy.exe.modelimports import *
from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.interfaces import aetinterfaces
from hydpy.interfaces import rconcinterfaces

# ...from hland
from hydpy.models.hland import hland_model
from hydpy.models.hland import hland_masks
from hydpy.models.hland.hland_constants import *


class Model(
    hland_model.Main_AETModel_V1,
    hland_model.Main_RConcModel_V1,
    hland_model.Sub_TempModel_V1,
    hland_model.Sub_PrecipModel_V1,
    hland_model.Sub_IntercModel_V1,
    hland_model.Sub_SoilWaterModel_V1,
    hland_model.Sub_SnowCoverModel_V1,
):
    """|hland_96c.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="H-HBV96-COSERO", description="fusion of SMHI-IHMS-HBV96 and COSERO"
    )
    __HYDPY_ROOTMODEL__ = True

    INLET_METHODS = ()
    OBSERVER_METHODS = ()
    RECEIVER_METHODS = ()
    RUN_METHODS = (
        hland_model.Calc_TC_V1,
        hland_model.Calc_FracRain_V1,
        hland_model.Calc_RFC_SFC_V1,
        hland_model.Calc_PC_V1,
        hland_model.Calc_TF_Ic_V1,
        hland_model.Calc_SP_WC_V1,
        hland_model.Calc_SPL_WCL_SP_WC_V1,
        hland_model.Calc_SPG_WCG_SP_WC_V1,
        hland_model.Calc_CFAct_V1,
        hland_model.Calc_Melt_SP_WC_V1,
        hland_model.Calc_Refr_SP_WC_V1,
        hland_model.Calc_In_WC_V1,
        hland_model.Calc_SWE_V1,
        hland_model.Calc_SR_V1,
        hland_model.Calc_GAct_V1,
        hland_model.Calc_GlMelt_In_V1,
        hland_model.Calc_EI_Ic_V1,
        hland_model.Calc_R_SM_V1,
        hland_model.Calc_EA_SM_V1,
        hland_model.Calc_QAb1_QVs1_BW1_V1,
        hland_model.Calc_QAb2_QVs2_BW2_V1,
        hland_model.Calc_InRC_V3,
        hland_model.Calc_OutRC_V1,
        hland_model.Calc_LZ_V2,
        hland_model.Calc_EL_LZ_V1,
        hland_model.Calc_Q1_LZ_V1,
        hland_model.Calc_RT_V2,
        hland_model.Calc_QT_V1,
    )
    INTERFACE_METHODS = (
        hland_model.Get_Temperature_V1,
        hland_model.Get_MeanTemperature_V1,
        hland_model.Get_Precipitation_V1,
        hland_model.Get_InterceptedWater_V1,
        hland_model.Get_SoilWater_V1,
        hland_model.Get_SnowCover_V1,
    )
    ADD_METHODS = (
        hland_model.Calc_EI_Ic_AETModel_V1,
        hland_model.Calc_EA_SM_AETModel_V1,
        hland_model.Calc_EL_LZ_AETModel_V1,
        hland_model.Calc_QAb_QVs_BW_V1,
        hland_model.Calc_OutRC_RConcModel_V1,
    )
    OUTLET_METHODS = (hland_model.Pass_Q_V1,)
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (aetinterfaces.AETModel_V1, rconcinterfaces.RConcModel_V1)
    SUBMODELS = ()

    aetmodel = modeltools.SubmodelProperty(aetinterfaces.AETModel_V1)
    rconcmodel = modeltools.SubmodelProperty(
        rconcinterfaces.RConcModel_V1, optional=True
    )

    def check_waterbalance(self, initial_conditions: ConditionsModel) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            \Sigma In_{hru} - \Sigma Out_{hru} - \Sigma Out_{basin}
            + \Delta Vol_{hru} + \Delta Vol_{snow} + \Delta Vol_{basin}
            - \Delta Vol_{rconc}
            \\ \\
            \Sigma In_{hru} =
            \sum_{k=1}^{N_{hru}} A_Z^k \cdot \sum_{t=t0}^{t1} PC_t^k + GLMelt_t^k
            \\
            \Sigma Out_{hru} =
            \sum_{k=1}^{N_{hru}} A_Z^k \cdot \sum_{t=t0}^{t1} EI_t^k + EA_t^k + EL_t^k
            \\
            \Sigma Out_{basin} = \sum_{t=t0}^{t1} RT_t
            \\
            \Delta Vol_{snow} = \sum_{k=1}^{N_{hru}} A_Z^k \cdot \frac{1}{N_{snow}}
            \cdot \sum_{c=1}^{N_{snow}} \left(SP_{t0}^{k,s} - SP_{t1}^{k,s}\right) +
            \left(WC_{t0}^{k,s} - WC_{t1}^{k,s}\right)
            \\
            \Delta Vol_{hru} = \sum_{k=1}^{N_{hru}} A_Z^k \cdot \Big(
            \big(IC_{t0}^k - IC_{t1}^k\big) + \big(SM_{t0}^k - SM_{t1}^k\big) +
            \big( BW1_{t0}^k - BW2_{t1}^k \big)
            + \big( BW2_{t0}^k - BW2_{t1}^k \big)
            \Big)
            \\
            \Delta Vol_{basin} = A_L \cdot \big( LZ_{t0} - LZ_{t1} \big)
            \\
            \Delta Vol_{rconc} = \begin{cases}
            rconcmodel.get\_waterbalance &|\ rconcmodel \\
            0 &|\ \overline{rconcmodel} \end{cases}
            \\ \\
            N_{hru} = NmbZones \\
            N_{snow} = SClass \\
            A_Z = RelZoneAreas \\
            A_L = RelLowerZoneArea

        The returned error should always be in scale with numerical precision so that
        it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation run via
        property |Sequences.conditions|.  See the integration tests of the application
        model |hland_96c| for some examples.
        """
        fluxes = self.sequences.fluxes
        last = self.sequences.states
        first = initial_conditions["model"]["states"]
        derived = self.parameters.derived
        areas = derived.relzoneareas.values
        zonetypes = self.parameters.control.zonetype.values
        idxs_lake = zonetypes == ILAKE
        idxs_glac = zonetypes == GLACIER
        idxs_seal = zonetypes == SEALED
        idxs_land = ~idxs_lake
        idxs_upper = idxs_land * ~idxs_seal
        idxs_soil = idxs_upper * ~idxs_glac
        return (
            numpy.sum(fluxes.pc.series * areas)
            + numpy.sum((fluxes.glmelt.series * areas)[:, idxs_glac])
            - numpy.sum((fluxes.ei.series * areas)[:, idxs_land])
            - numpy.sum((fluxes.ea.series * areas)[:, idxs_soil])
            - numpy.sum((fluxes.el.series * areas)[:, idxs_lake])
            - numpy.sum(fluxes.rt.series)
            - numpy.sum(((last.ic - first["ic"]) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.wc - first["wc"], axis=0) * areas)[idxs_land])
            - numpy.sum((numpy.mean(last.sp - first["sp"], axis=0) * areas)[idxs_land])
            - numpy.sum(((last.sm - first["sm"]) * areas)[idxs_soil])
            - numpy.sum(((last.bw1 - first["bw1"]) * areas)[idxs_upper])
            - numpy.sum(((last.bw2 - first["bw2"]) * areas)[idxs_upper])
            - (last.lz - first["lz"]) * derived.rellowerzonearea
            - self._get_rconcmodel_waterbalance(initial_conditions)
            * derived.rellandarea
        )


class Masks(masktools.Masks):
    """Masks applicable to |hland_96c|."""

    CLASSES = hland_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
