# pylint: disable=line-too-long, unused-wildcard-import
"""
|whmod_rural| is the official version of the water balance model WHMod (German
abbreviation for "Wasserhaushaltsmodell").  It is an SVAT-like model frequently used
in Germany to estimate groundwater recharge and related properties.  It was initially
developed by :cite:t:`ref-Probst2002`.

The following list summarises the main components of |whmod_rural|:

 * Simulate interception via simple "bucket" storages.
 * Reflect snow processes with the degree day method.
 * Calculate evapotranspiration via selectable submodels that may follow, for example,
   the FAO56 guidelines on determining grass reference evapotranspiration
   :cite:p:`ref-Allen1998`.
 * Estimate percolation based on :cite:t:`ref-Armbruster2002`.
 * Optionally include capillary rise.
 * Optionally include irrigation.
 * Use the baseflow index approach to adjust the actual groundwater recharge.
 * Consider the travelling time of water percolating through the vadose zone.

Integration tests
=================

.. how_to_understand_integration_tests::

We perform all simulation runs over the same period of two months with a daily
simulation step size:

>>> from hydpy import Element, pub, Timegrid
>>> pub.timegrids = "2017-02-10", "2017-04-10", "1d"

As a 1-dimensional SVAT model, |whmod_rural| requires no connections to adjacent
model instances, so a simple |Element| object without any |Node| connections suffices
for holding a |whmod_rural| instance:

>>> element = Element("element")
>>> from hydpy.models.whmod_rural import *
>>> parameterstep("1d")
>>> element.model = model

Our test site has a size of 10 m²:

>>> area(10.0)

Initially, do not differentiate it into multiple zones:

>>> nmbzones(1)
>>> zonearea(10.0)

The single zone is of sandy soil grown with grass:

>>> landtype(GRASS)
>>> soiltype(SAND)

Land type- and month-specific interception capacities can be defined to reflect
different vegetation types and states:

>>> interceptioncapacity.grass_feb = 0.4
>>> interceptioncapacity.grass_mar = 0.6
>>> interceptioncapacity.grass_apr = 0.8

The values of some parameters can be set zone- or land type-specific:

>>> degreedayfactor(grass=4.5)

The values of others can be set zone- or soil type-specific:

>>> availablefieldcapacity(sand=200.0)

The rooting and groundwater depths complement the previously set relative available
field capacity to allow for determining the maximum soil water available for plants:

>>> rootingdepth(0.5)
>>> groundwaterdepth(1.0)

The groundwater depth also controls the capillary rise, which can be turned on or off
by a flag and further configured by a threshold and a limit parameter:

>>> withcapillaryrise(True)
>>> capillarythreshold(0.8)
>>> capillarylimit(0.4)

Another optional feature is the land type- and month-specific irrigation of too-dry
soils:

>>> withexternalirrigation(False)
>>> irrigationtrigger(0.6)
>>> irrigationtarget(0.7)

The following two parameters allow for the modification of the amount of and time when
percolating water actually reaches the groundwater table:

>>> baseflowindex(0.8)
>>> rechargedelay(5.0)

|whmod_rural| requires a submodel for calculating actual evapotranspiration.  Here, we
combine |evap_ret_io| to provide externally available reference evapotranspiration
estimates, |evap_pet_mlc| to adjust these to the respective land type, and
|evap_aet_minhas| to convert potential to actual evapotranspiration in agreement with
the current moisture conditions:

>>> with model.add_aetmodel_v1("evap_aet_minhas") as aetmodel:
...     dissefactor(grass=4.0, deciduous=6.0, corn=3.0, conifer=6.0, springwheat=6.0,
...                 winterwheat=6.0)
...     with model.add_petmodel_v1("evap_pet_mlc") as petmodel:
...         landmonthfactor.grass = 1.0
...         dampingfactor(1.0)
...         with model.add_retmodel_v1("evap_ret_io") as retmodel:
...             evapotranspirationfactor(1.0)

We initialise a test function object, which prepares and runs the tests and prints
their results for the given sequences:

>>> from hydpy import IntegrationTest
>>> IntegrationTest.plotting_options.axis1 = (
...     inputs.precipitation, fluxes.actualrecharge, fluxes.delayedrecharge
... )
>>> test = IntegrationTest(element)
>>> test.dateformat = "%Y-%m-%d"

We set all initial conditions to zero, meaning all simulation runs start with empty
storages:

>>> test.inits = (
...     (states.interceptedwater, 0.0),
...     (states.snowpack, 0.0),
...     (states.soilmoisture, 0.0),
...     (states.deepwater, 0.0),
...     (petmodel.sequences.logs.loggedpotentialevapotranspiration, 0.0),
... )

The following meteorological data shows a shift from winter to spring in the form of a
rise in temperature and potential evapotranspiration and includes two heavy rainfall
events:

>>> inputs.temperature.series = (
...     -2.8, -1.5, -0.9, -1.6, -1.3, 1.7, 4.4, 4.5, 3.4, 4.8, 6.7, 5.8, 6.5, 5.0, 3.0,
...     3.1, 7.1, 9.4, 4.6, 3.7, 4.7, 5.9, 7.7, 6.3, 3.7, 1.6, 4.0, 5.6, 5.8, 5.7, 4.6,
...     4.2, 7.4, 6.3, 8.7, 6.4, 5.2, 5.1, 8.7, 6.2, 5.9, 5.2, 5.2, 5.9, 6.7, 7.0, 8.3,
...     9.0, 12.4, 15.0, 11.8, 9.4, 8.1, 7.9, 7.5, 7.2, 8.1, 8.6, 10.5)
>>> inputs.precipitation.series = (
...     0.0, 0.4, 0.0, 0.0, 0.0, 0.0, 0.2, 4.5, 0.0, 3.2, 4.6, 2.3, 18.0, 19.2, 0.4,
...     8.3, 5.3, 0.7, 2.7, 1.6, 2.5, 0.6, 0.2, 1.7, 0.3, 0.0, 1.8, 8.9, 0.0, 0.0,
...     0.0, 0.9, 0.1, 0.0, 0.0, 3.9, 8.7, 26.4, 11.5, 0.9, 0.0, 0.0, 0.0, 0.0, 0.0,
...     0.0, 0.0, 1.5, 0.3, 0.2, 4.5, 0.0, 0.0, 0.0, 0.4, 0.0, 0.0, 0.0, 0.0)
>>> retmodel.sequences.inputs.referenceevapotranspiration.series = (
...     0.6, 0.8, 0.7, 0.4, 0.4, 0.4, 0.4, 0.3, 0.3, 0.4, 0.3, 0.6, 0.8, 0.5, 0.8,
...     0.5, 0.4, 1.3, 0.9, 0.7, 0.7, 1.1, 1.0, 0.8, 0.6, 0.7, 0.7, 0.5, 0.8, 1.0,
...     1.2, 0.9, 0.9, 1.2, 1.4, 1.1, 1.1, 0.5, 0.6, 1.5, 2.0, 1.6, 1.6, 1.2, 1.3,
...     1.6, 1.9, 0.8, 1.5, 2.7, 1.5, 1.6, 2.0, 2.1, 1.7, 1.7, 0.8, 1.3, 2.5)

.. _whmod_rural_grassland:

grassland
_________

Due to the dry starting conditions, initially substantial capillary rise, and the two
heavy precipitation events, soil moisture increases during the simulation.  So,
percolation and subsequent groundwater recharge are more pronounced  for the second
event:

.. integration-test::

    >>> conditions = test("whmod_rural_grassland", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | deepwater |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |               0.0375 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |          3.75 |              66.25 |                0.0 |             -3.75 |      0.0 |          -3.75 |        -0.67976 |              0.0 |      0.0 |         3.75 |  -3.07024 |
    | 2017-02-11 |           0.4 |        -1.5 |             0.070632 |                     0.4 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.030545 |             0.0 |                0.430545 |      3.343748 |          62.936798 |                0.0 |         -3.343748 |      0.0 |      -3.343748 |       -1.162659 |              0.0 |      0.0 |     7.063202 | -5.251329 |
    | 2017-02-12 |           0.0 |        -0.9 |             0.099731 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.100328 |             0.0 |                0.100328 |      3.010193 |          60.026932 |                0.0 |         -3.010193 |      0.0 |      -3.010193 |        -1.49756 |              0.0 |      0.0 |     9.973068 | -6.763962 |
    | 2017-02-13 |           0.0 |        -1.6 |             0.126288 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.080508 |             0.0 |                0.080508 |      2.736205 |          57.371235 |                0.0 |         -2.736205 |      0.0 |      -2.736205 |       -1.722088 |              0.0 |      0.0 |    12.628765 | -7.778079 |
    | 2017-02-14 |           0.0 |        -1.3 |             0.150286 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.101248 |             0.0 |                0.101248 |      2.501132 |          54.971351 |                0.0 |         -2.501132 |      0.0 |      -2.501132 |       -1.863305 |              0.0 |      0.0 |    15.028649 | -8.415907 |
    | 2017-02-15 |           0.0 |         1.7 |             0.172097 |                     0.0 |         0.0 |              7.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.119564 |             0.0 |                0.119564 |      2.300641 |          52.790274 |                0.0 |         -2.300641 |      0.0 |      -2.300641 |       -1.942581 |              0.0 |      0.0 |    17.209726 | -8.773967 |
    | 2017-02-16 |           0.2 |         4.4 |             0.192698 |                     0.2 |         0.0 |              19.8 |      0.0 |     0.0 |           0.0 |         0.0 |               0.067899 |             0.0 |                0.267899 |      2.127988 |          50.730184 |                0.0 |         -2.127988 |      0.0 |      -2.127988 |       -1.976189 |              0.0 |      0.0 |    19.269816 | -8.925766 |
    | 2017-02-17 |           4.5 |         4.5 |             0.252771 |                     0.3 |         4.1 |             20.25 |      0.0 |     4.1 |           0.0 |    0.065722 |                    0.0 |             0.0 |                     0.3 |      1.973055 |          44.722852 |                0.0 |         -1.907333 |      0.0 |      -1.907333 |       -1.963708 |              0.1 |      0.0 |    25.277148 | -8.869391 |
    | 2017-02-18 |           0.0 |         3.4 |             0.267458 |                     0.1 |         0.0 |              15.3 |      0.0 |     0.0 |           0.0 |         0.0 |               0.095868 |             0.0 |                0.195868 |      1.564558 |          43.254162 |                0.0 |         -1.564558 |      0.0 |      -1.564558 |       -1.891354 |              0.0 |      0.0 |    26.745838 | -8.542595 |
    | 2017-02-19 |           3.2 |         4.8 |             0.309177 |                     0.4 |         2.8 |              21.6 |      0.0 |     2.8 |           0.0 |    0.102207 |                    0.0 |             0.0 |                     0.4 |      1.474104 |          39.082265 |                0.0 |         -1.371898 |      0.0 |      -1.371898 |       -1.797193 |              0.0 |      0.0 |    30.917735 |   -8.1173 |
    | 2017-02-20 |           4.6 |         6.7 |             0.361335 |                     0.3 |         4.2 |             30.15 |      0.0 |     4.2 |           0.0 |    0.220591 |                    0.0 |             0.0 |                     0.3 |       1.23632 |          33.866536 |                0.0 |         -1.015729 |      0.0 |      -1.015729 |       -1.655537 |              0.1 |      0.0 |    36.133464 | -7.477491 |
    | 2017-02-21 |           2.3 |         5.8 |             0.388275 |                     0.4 |         2.0 |              26.1 |      0.0 |     2.0 |           0.0 |     0.15535 |               0.127491 |             0.0 |                0.527491 |      0.976903 |          31.172475 |                0.0 |         -0.821553 |      0.0 |      -0.821553 |       -1.504361 |              0.0 |      0.0 |    38.827525 | -6.794683 |
    | 2017-02-22 |          18.0 |         6.5 |               0.5538 |                     0.4 |        17.6 |             29.25 |      0.0 |    17.6 |           0.0 |    1.637522 |               0.268404 |             0.0 |                0.668404 |      0.858419 |          14.619981 |                0.0 |          0.779103 | 0.155821 |       0.623282 |       -1.118685 |              0.0 |      0.0 |    55.380019 | -5.052715 |
    | 2017-02-23 |          19.2 |         5.0 |              0.70165 |                     0.4 |        18.8 |              22.5 |      0.0 |    18.8 |           0.0 |    4.265135 |               0.083065 |             0.0 |                0.483065 |      0.333134 |                0.0 |                0.0 |          3.932001 |   0.7864 |         3.1456 |       -0.345701 |              0.0 |      0.0 |    70.164954 | -1.561413 |
    | 2017-02-24 |           0.4 |         3.0 |             0.698974 |                     0.4 |         0.0 |              13.5 |      0.0 |     0.0 |           0.0 |         0.0 |               0.367107 |             0.0 |                0.767107 |      0.099589 |                0.0 |                0.0 |         -0.099589 |      0.0 |      -0.099589 |       -0.301089 |              0.0 |      0.0 |    69.897436 | -1.359914 |
    | 2017-02-25 |           8.3 |         3.1 |             0.745929 |                     0.4 |         7.9 |             13.95 |      0.0 |     7.9 |           0.0 |    3.215139 |               0.091655 |             0.0 |                0.491655 |      0.102292 |                0.0 |                0.0 |          3.112847 | 0.622569 |       2.490278 |          0.2049 |              0.0 |      0.0 |    74.592934 |  0.925464 |
    | 2017-02-26 |           5.3 |         7.1 |             0.772067 |                     0.4 |         4.9 |             31.95 |      0.0 |     4.9 |           0.0 |    2.347723 |                    0.0 |             0.0 |                     0.4 |      0.061503 |                0.0 |                0.0 |           2.28622 | 0.457244 |       1.828976 |        0.499295 |              0.0 |      0.0 |    77.206714 |  2.255144 |
    | 2017-02-27 |           0.7 |         9.4 |              0.76543 |                     0.4 |         0.3 |              42.3 |      0.0 |     0.3 |           0.0 |    0.156717 |               0.851358 |             0.0 |                1.251358 |      0.044407 |                0.0 |                0.0 |           0.11231 | 0.022462 |       0.089848 |        0.425075 |              0.0 |      0.0 |    76.543046 |  1.919918 |
    | 2017-02-28 |           2.7 |         4.6 |             0.772439 |                     0.4 |         2.3 |              20.7 |      0.0 |     2.3 |           0.0 |    1.175743 |               0.471796 |             0.0 |                0.871796 |        0.0484 |                0.0 |                0.0 |          1.127343 | 0.225469 |       0.901875 |        0.511504 |              0.0 |      0.0 |    77.243907 |  2.310288 |
    | 2017-03-01 |           1.6 |         3.7 |             0.776705 |                     0.6 |         1.0 |             16.65 |      0.0 |     1.0 |           0.0 |    0.523023 |               0.094608 |             0.0 |                0.694608 |       0.04419 |                0.0 |                0.0 |          0.478833 | 0.095767 |       0.383067 |        0.488222 |              0.0 |      0.0 |    77.670465 |  2.205132 |
    | 2017-03-02 |           2.5 |         4.7 |             0.785099 |                     0.6 |         1.9 |             21.15 |      0.0 |     1.9 |           0.0 |    1.007577 |               0.094757 |             0.0 |                0.694757 |      0.041751 |                0.0 |                0.0 |          0.965826 | 0.193165 |       0.772661 |        0.539782 |              0.0 |      0.0 |    78.509882 |  2.438011 |
    | 2017-03-03 |           0.6 |         5.9 |             0.780719 |                     0.6 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               0.475214 |             0.0 |                1.075214 |      0.037218 |                0.0 |                0.0 |         -0.037218 |      0.0 |      -0.037218 |         0.43519 |              0.0 |      0.0 |    78.071886 |  1.965603 |
    | 2017-03-04 |           0.2 |         7.7 |             0.773523 |                     0.2 |         0.0 |             34.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.759159 |             0.0 |                0.959159 |       0.03954 |                0.0 |                0.0 |          -0.03954 |      0.0 |       -0.03954 |        0.349136 |              0.0 |      0.0 |    77.352267 |  1.576927 |
    | 2017-03-05 |           1.7 |         6.3 |             0.777292 |                     0.6 |         1.1 |             28.35 |      0.0 |     1.1 |           0.0 |    0.577354 |               0.189293 |             0.0 |                0.789293 |      0.043562 |                0.0 |                0.0 |          0.533792 | 0.106758 |       0.427034 |        0.363256 |              0.0 |      0.0 |    77.729182 |  1.640704 |
    | 2017-03-06 |           0.3 |         3.7 |             0.774863 |                     0.3 |         0.0 |             16.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.284332 |             0.0 |                0.584332 |      0.041423 |                0.0 |                0.0 |         -0.041423 |      0.0 |      -0.041423 |        0.289901 |              0.0 |      0.0 |    77.486273 |  1.309381 |
    | 2017-03-07 |           0.0 |         1.6 |             0.768662 |                     0.0 |         0.0 |               7.2 |      0.0 |     0.0 |           0.0 |         0.0 |               0.662852 |             0.0 |                0.662852 |      0.042793 |                0.0 |                0.0 |         -0.042793 |      0.0 |      -0.042793 |        0.229593 |              0.0 |      0.0 |    76.866214 |  1.036994 |
    | 2017-03-08 |           1.8 |         4.0 |             0.773982 |                     0.6 |         1.2 |              18.0 |      0.0 |     1.2 |           0.0 |    0.619953 |               0.094475 |             0.0 |                0.694475 |      0.046427 |                0.0 |                0.0 |          0.573526 | 0.114705 |       0.458821 |        0.271145 |              0.0 |      0.0 |    77.398213 |   1.22467 |
    | 2017-03-09 |           8.9 |         5.6 |             0.813786 |                     0.5 |         8.3 |              25.2 |      0.0 |     8.3 |           0.0 |    4.362895 |                    0.0 |             0.0 |                     0.5 |      0.043297 |                0.0 |                0.0 |          4.319598 |  0.86392 |       3.455679 |        0.848403 |              0.1 |      0.0 |    81.378615 |  3.831945 |
    | 2017-03-10 |           0.0 |         5.8 |             0.807311 |                     0.1 |         0.0 |              26.1 |      0.0 |     0.0 |           0.0 |         0.0 |               0.671692 |             0.0 |                0.771692 |      0.024214 |                0.0 |                0.0 |         -0.024214 |      0.0 |      -0.024214 |        0.690225 |              0.0 |      0.0 |    80.731137 |  3.117507 |
    | 2017-03-11 |           0.0 |         5.7 |             0.798004 |                     0.0 |         0.0 |             25.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.957582 |             0.0 |                0.957582 |      0.026829 |                0.0 |                0.0 |         -0.026829 |      0.0 |      -0.026829 |        0.560245 |              0.0 |      0.0 |    79.800384 |  2.530433 |
    | 2017-03-12 |           0.0 |         4.6 |             0.786857 |                     0.0 |         0.0 |              20.7 |      0.0 |     0.0 |           0.0 |         0.0 |               1.145587 |             0.0 |                1.145587 |      0.030907 |                0.0 |                0.0 |         -0.030907 |      0.0 |      -0.030907 |        0.453087 |              0.0 |      0.0 |    78.685705 |  2.046439 |
    | 2017-03-13 |           0.9 |         4.2 |             0.785723 |                     0.6 |         0.3 |              18.9 |      0.0 |     0.3 |           0.0 |    0.164363 |               0.285305 |             0.0 |                0.885305 |      0.036312 |                0.0 |                0.0 |          0.128051 |  0.02561 |       0.102441 |        0.389526 |              0.0 |      0.0 |    78.572349 |  1.759354 |
    | 2017-03-14 |           0.1 |         7.4 |             0.778487 |                     0.1 |         0.0 |              33.3 |      0.0 |     0.0 |           0.0 |         0.0 |                0.76051 |             0.0 |                 0.86051 |      0.036894 |                0.0 |                0.0 |         -0.036894 |      0.0 |      -0.036894 |        0.312229 |              0.0 |      0.0 |    77.848732 |  1.410231 |
    | 2017-03-15 |           0.0 |         6.3 |             0.767517 |                     0.0 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |               1.137822 |             0.0 |                1.137822 |      0.040759 |                0.0 |                0.0 |         -0.040759 |      0.0 |      -0.040759 |        0.248243 |              0.0 |      0.0 |     76.75167 |  1.121228 |
    | 2017-03-16 |           0.0 |         8.7 |             0.754767 |                     0.0 |         0.0 |             39.15 |      0.0 |     0.0 |           0.0 |         0.0 |               1.322076 |             0.0 |                1.322076 |       0.04712 |                0.0 |                0.0 |          -0.04712 |      0.0 |       -0.04712 |        0.194703 |              0.0 |      0.0 |    75.476714 |  0.879406 |
    | 2017-03-17 |           3.9 |         6.4 |             0.767336 |                     0.6 |         3.3 |              28.8 |      0.0 |     3.3 |           0.0 |    1.628564 |               0.469837 |             0.0 |                1.069837 |      0.055305 |                0.0 |                0.0 |          1.573259 | 0.314652 |       1.258607 |        0.387556 |              0.0 |      0.0 |    76.733618 |  1.750457 |
    | 2017-03-18 |           8.7 |         5.2 |             0.802421 |                     0.6 |         8.1 |              23.4 |      0.0 |     8.1 |           0.0 |    4.166588 |               0.472138 |             0.0 |                1.072138 |       0.04723 |                0.0 |                0.0 |          4.119358 | 0.823872 |       3.295486 |        0.914674 |              0.0 |      0.0 |    80.242122 |  4.131269 |
    | 2017-03-19 |          26.4 |         5.1 |             0.912235 |                     0.5 |        25.8 |             22.95 |      0.0 |    25.8 |           0.0 |   14.847535 |                    0.0 |             0.0 |                     0.5 |      0.028924 |                0.0 |                0.0 |         14.818611 | 2.963722 |      11.854889 |        2.897799 |              0.1 |      0.0 |    91.223511 | 13.088359 |
    | 2017-03-20 |          11.5 |         8.7 |             0.934912 |                     0.6 |        11.0 |             39.15 |      0.0 |    11.0 |           0.0 |    8.734827 |                    0.0 |             0.0 |                     0.6 |      0.002535 |                0.0 |                0.0 |          8.732292 | 1.746458 |       6.985833 |        3.638834 |              0.0 |      0.0 |     93.49122 | 16.435358 |
    | 2017-03-21 |           0.9 |         6.2 |             0.926488 |                     0.6 |         0.3 |              27.9 |      0.0 |     0.3 |           0.0 |    0.253368 |               0.890068 |             0.0 |                1.490068 |      0.001034 |                0.0 |                0.0 |          0.252334 | 0.050467 |       0.201867 |        3.015817 |              0.0 |      0.0 |    92.648818 | 13.621408 |
    | 2017-03-22 |           0.0 |         5.9 |             0.906757 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                1.97465 |             0.0 |                 1.97465 |       0.00149 |                0.0 |                0.0 |          -0.00149 |      0.0 |       -0.00149 |        2.468872 |              0.0 |      0.0 |    90.675658 | 11.151046 |
    | 2017-03-23 |           0.0 |         5.2 |             0.891055 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.573237 |             0.0 |                1.573237 |       0.00304 |                0.0 |                0.0 |          -0.00304 |      0.0 |       -0.00304 |        2.020791 |              0.0 |      0.0 |    89.105461 |  9.127215 |
    | 2017-03-24 |           0.0 |         5.2 |             0.875426 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.567721 |             0.0 |                1.567721 |      0.004849 |                0.0 |                0.0 |         -0.004849 |      0.0 |      -0.004849 |        1.653604 |              0.0 |      0.0 |     87.54259 |  7.468762 |
    | 2017-03-25 |           0.0 |         5.9 |             0.863784 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.171422 |             0.0 |                1.171422 |       0.00725 |                0.0 |                0.0 |          -0.00725 |      0.0 |       -0.00725 |        1.352543 |              0.0 |      0.0 |    86.378418 |  6.108969 |
    | 2017-03-26 |           0.0 |         6.7 |             0.851226 |                     0.0 |         0.0 |             30.15 |      0.0 |     0.0 |           0.0 |         0.0 |                1.26533 |             0.0 |                 1.26533 |      0.009478 |                0.0 |                0.0 |         -0.009478 |      0.0 |      -0.009478 |         1.10565 |              0.0 |      0.0 |    85.122566 |  4.993841 |
    | 2017-03-27 |           0.0 |         7.0 |             0.835827 |                     0.0 |         0.0 |              31.5 |      0.0 |     0.0 |           0.0 |         0.0 |               1.552174 |             0.0 |                1.552174 |      0.012349 |                0.0 |                0.0 |         -0.012349 |      0.0 |      -0.012349 |        0.902991 |              0.0 |      0.0 |     83.58274 |  4.078501 |
    | 2017-03-28 |           0.0 |         8.3 |              0.81764 |                     0.0 |         0.0 |             37.35 |      0.0 |     0.0 |           0.0 |         0.0 |               1.835299 |             0.0 |                1.835299 |      0.016593 |                0.0 |                0.0 |         -0.016593 |      0.0 |      -0.016593 |        0.736299 |              0.0 |      0.0 |    81.764034 |  3.325609 |
    | 2017-03-29 |           1.5 |         9.0 |             0.819517 |                     0.6 |         0.9 |              40.5 |      0.0 |     0.9 |           0.0 |     0.54295 |               0.192143 |             0.0 |                0.792143 |      0.022741 |                0.0 |                0.0 |          0.520209 | 0.104042 |       0.416167 |        0.678269 |              0.0 |      0.0 |    81.951683 |  3.063507 |
    | 2017-03-30 |           0.3 |        12.4 |             0.808202 |                     0.3 |         0.0 |              55.8 |      0.0 |     0.0 |           0.0 |         0.0 |               1.153525 |             0.0 |                1.453525 |      0.022047 |                0.0 |                0.0 |         -0.022047 |      0.0 |      -0.022047 |        0.551323 |              0.0 |      0.0 |    80.820205 |  2.490137 |
    | 2017-03-31 |           0.2 |        15.0 |              0.78452 |                     0.2 |         0.0 |              67.5 |      0.0 |     0.0 |           0.0 |         0.0 |               2.394642 |             0.0 |                2.594642 |      0.026458 |                0.0 |                0.0 |         -0.026458 |      0.0 |      -0.026458 |        0.446589 |              0.0 |      0.0 |    78.452021 |   2.01709 |
    | 2017-04-01 |           4.5 |        11.8 |             0.795123 |                     0.8 |         3.7 |              53.1 |      0.0 |     3.7 |           0.0 |    2.012061 |               0.665164 |             0.0 |                1.465164 |      0.037519 |                0.0 |                0.0 |          1.974542 | 0.394908 |       1.579634 |        0.651975 |              0.0 |      0.0 |    79.512315 |  2.944748 |
    | 2017-04-02 |           0.0 |         9.4 |             0.780186 |                     0.0 |         0.0 |              42.3 |      0.0 |     0.0 |           0.0 |         0.0 |               1.525967 |             0.0 |                1.525967 |      0.032249 |                0.0 |                0.0 |         -0.032249 |      0.0 |      -0.032249 |        0.527947 |              0.0 |      0.0 |    78.018597 |  2.384553 |
    | 2017-04-03 |           0.0 |         8.1 |             0.761609 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               1.897533 |             0.0 |                1.897533 |      0.039829 |                0.0 |                0.0 |         -0.039829 |      0.0 |      -0.039829 |        0.425026 |              0.0 |      0.0 |    76.160892 |  1.919698 |
    | 2017-04-04 |           0.0 |         7.9 |             0.742331 |                     0.0 |         0.0 |             35.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.978631 |             0.0 |                1.978631 |      0.050804 |                0.0 |                0.0 |         -0.050804 |      0.0 |      -0.050804 |        0.338773 |              0.0 |      0.0 |    74.233066 |   1.53012 |
    | 2017-04-05 |           0.4 |         7.5 |             0.730818 |                     0.4 |         0.0 |             33.75 |      0.0 |     0.0 |           0.0 |         0.0 |               1.215386 |             0.0 |                1.615386 |      0.064153 |                0.0 |                0.0 |         -0.064153 |      0.0 |      -0.064153 |        0.265735 |              0.0 |      0.0 |    73.081834 |  1.200232 |
    | 2017-04-06 |           0.0 |         7.2 |             0.715734 |                     0.0 |         0.0 |              32.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.581527 |             0.0 |                1.581527 |      0.073142 |                0.0 |                0.0 |         -0.073142 |      0.0 |      -0.073142 |        0.204307 |              0.0 |      0.0 |    71.573449 |  0.922783 |
    | 2017-04-07 |           0.0 |         8.1 |             0.709204 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               0.739189 |             0.0 |                0.739189 |       0.08614 |                0.0 |                0.0 |          -0.08614 |      0.0 |       -0.08614 |        0.151658 |              0.0 |      0.0 |      70.9204 |  0.684986 |
    | 2017-04-08 |           0.0 |         8.6 |             0.698151 |                     0.0 |         0.0 |              38.7 |      0.0 |     0.0 |           0.0 |         0.0 |               1.197483 |             0.0 |                1.197483 |      0.092214 |                0.0 |                0.0 |         -0.092214 |      0.0 |      -0.092214 |        0.107451 |              0.0 |      0.0 |    69.815131 |  0.485321 |
    | 2017-04-09 |           0.0 |        10.5 |             0.676278 |                     0.0 |         0.0 |             47.25 |      0.0 |     0.0 |           0.0 |         0.0 |               2.290433 |             0.0 |                2.290433 |      0.103133 |                0.0 |                0.0 |         -0.103133 |      0.0 |      -0.103133 |        0.069279 |              0.0 |      0.0 |    67.627832 |  0.312908 |

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_rural_irrigation:

irrigation
__________

With irrigation enabled, |whmod_rural| increases the soil moisture to the target value
of 70 % at the end of the first simulated day, which increases percolation and
reduces capillary rise.  After the last simulated day, soil moisture remains slightly
below the target value because the related trigger, set to 60 %, is still not reached:

.. integration-test::

    >>> withexternalirrigation(True)
    >>> conditions = test("whmod_rural_irrigation", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | deepwater |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |                  0.7 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |          3.75 |              66.25 |              66.25 |             -3.75 |      0.0 |          -3.75 |        -0.67976 |              0.0 |      0.0 |         70.0 |  -3.07024 |
    | 2017-02-11 |           0.4 |        -1.5 |             0.697344 |                     0.4 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.366807 |             0.0 |                0.766807 |       0.10125 |                0.0 |                0.0 |          -0.10125 |      0.0 |       -0.10125 |       -0.574894 |              0.0 |      0.0 |    69.734443 | -2.596597 |
    | 2017-02-12 |           0.0 |        -0.9 |             0.691973 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.641062 |             0.0 |                0.641062 |      0.103963 |                0.0 |                0.0 |         -0.103963 |      0.0 |      -0.103963 |       -0.489528 |              0.0 |      0.0 |    69.197344 | -2.211031 |
    | 2017-02-13 |           0.0 |        -1.6 |             0.689416 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.365324 |             0.0 |                0.365324 |      0.109596 |                0.0 |                0.0 |         -0.109596 |      0.0 |      -0.109596 |       -0.420658 |              0.0 |      0.0 |    68.941615 | -1.899969 |
    | 2017-02-14 |           0.0 |        -1.3 |             0.686891 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |               0.364843 |             0.0 |                0.364843 |      0.112349 |                0.0 |                0.0 |         -0.112349 |      0.0 |      -0.112349 |       -0.364771 |              0.0 |      0.0 |    68.689121 | -1.647546 |
    | 2017-02-15 |           0.0 |         1.7 |             0.684399 |                     0.0 |         0.0 |              7.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.364364 |             0.0 |                0.364364 |      0.115111 |                0.0 |                0.0 |         -0.115111 |      0.0 |      -0.115111 |       -0.319516 |              0.0 |      0.0 |    68.439868 | -1.443142 |
    | 2017-02-16 |           0.2 |         4.4 |             0.683758 |                     0.2 |         0.0 |              19.8 |      0.0 |     0.0 |           0.0 |         0.0 |               0.181943 |             0.0 |                0.381943 |      0.117882 |                0.0 |                0.0 |         -0.117882 |      0.0 |      -0.117882 |       -0.282966 |              0.0 |      0.0 |    68.375807 | -1.278058 |
    | 2017-02-17 |           4.5 |         4.5 |             0.710155 |                     0.3 |         4.1 |             20.25 |      0.0 |     4.1 |           0.0 |    1.578928 |                    0.0 |             0.0 |                     0.3 |      0.118601 |                0.0 |                0.0 |          1.460327 | 0.292065 |       1.168262 |       -0.019903 |              0.1 |      0.0 |     71.01548 | -0.089894 |
    | 2017-02-18 |           0.0 |         3.4 |             0.709225 |                     0.1 |         0.0 |              15.3 |      0.0 |     0.0 |           0.0 |         0.0 |               0.184312 |             0.0 |                0.284312 |      0.091312 |                0.0 |                0.0 |         -0.091312 |      0.0 |      -0.091312 |       -0.032847 |              0.0 |      0.0 |     70.92248 | -0.148359 |
    | 2017-02-19 |           3.2 |         4.8 |             0.726327 |                     0.4 |         2.8 |              21.6 |      0.0 |     2.8 |           0.0 |    1.181957 |                    0.0 |             0.0 |                     0.4 |      0.092194 |                0.0 |                0.0 |          1.089763 | 0.217953 |        0.87181 |        0.131139 |              0.0 |      0.0 |    72.632718 |  0.592312 |
    | 2017-02-20 |           4.6 |         6.7 |             0.750274 |                     0.3 |         4.2 |             30.15 |      0.0 |     4.2 |           0.0 |    1.882214 |                    0.0 |             0.0 |                     0.3 |      0.076865 |                0.0 |                0.0 |          1.805349 |  0.36107 |       1.444279 |        0.369171 |              0.1 |      0.0 |    75.027368 |   1.66742 |
    | 2017-02-21 |           2.3 |         5.8 |             0.759258 |                     0.4 |         2.0 |              26.1 |      0.0 |     2.0 |           0.0 |    0.972325 |               0.187596 |             0.0 |                0.587596 |      0.058402 |                0.0 |                0.0 |          0.913923 | 0.182785 |       0.731139 |        0.434785 |              0.0 |      0.0 |    75.925849 |  1.963774 |
    | 2017-02-22 |          18.0 |         6.5 |             0.843856 |                     0.4 |        17.6 |             29.25 |      0.0 |    17.6 |           0.0 |    8.815997 |               0.376537 |             0.0 |                0.776537 |      0.052322 |                0.0 |                0.0 |          8.763675 | 1.752735 |        7.01094 |         1.62684 |              0.0 |      0.0 |    84.385637 |  7.347874 |
    | 2017-02-23 |          19.2 |         5.0 |             0.908264 |                     0.4 |        18.8 |              22.5 |      0.0 |    18.8 |           0.0 |   12.276653 |               0.096815 |             0.0 |                0.496815 |      0.014276 |                0.0 |                0.0 |         12.262377 | 2.452475 |       9.809902 |        3.110177 |              0.0 |      0.0 |    90.826445 | 14.047599 |
    | 2017-02-24 |           0.4 |         3.0 |             0.904359 |                     0.4 |         0.0 |              13.5 |      0.0 |     0.0 |           0.0 |         0.0 |               0.393437 |             0.0 |                0.793437 |      0.002895 |                0.0 |                0.0 |         -0.002895 |      0.0 |      -0.002895 |        2.545873 |              0.0 |      0.0 |    90.435903 | 11.498831 |
    | 2017-02-25 |           8.3 |         3.1 |             0.921028 |                     0.4 |         7.9 |             13.95 |      0.0 |     7.9 |           0.0 |    6.138124 |               0.098276 |             0.0 |                0.498276 |      0.003281 |                0.0 |                0.0 |          6.134843 | 1.226969 |       4.907875 |        2.974031 |              0.0 |      0.0 |    92.102783 | 13.432674 |
    | 2017-02-26 |           5.3 |         7.1 |             0.930188 |                     0.4 |         4.9 |             31.95 |      0.0 |     4.9 |           0.0 |    3.985795 |                    0.0 |             0.0 |                     0.4 |      0.001847 |                0.0 |                0.0 |          3.983948 |  0.79679 |       3.187158 |        3.012665 |              0.0 |      0.0 |    93.018835 | 13.607168 |
    | 2017-02-27 |           0.7 |         9.4 |             0.921807 |                     0.4 |         0.3 |              42.3 |      0.0 |     0.3 |           0.0 |    0.250166 |               0.889246 |             0.0 |                1.289246 |      0.001276 |                0.0 |                0.0 |          0.248891 | 0.049778 |       0.199112 |        2.502654 |              0.0 |      0.0 |    92.180699 | 11.303627 |
    | 2017-02-28 |           2.7 |         4.6 |             0.921144 |                     0.4 |         2.3 |              20.7 |      0.0 |     2.3 |           0.0 |    1.874859 |               0.493195 |             0.0 |                0.893195 |      0.001793 |                0.0 |                0.0 |          1.873066 | 0.374613 |       1.498453 |        2.320623 |              0.0 |      0.0 |    92.114437 | 10.481456 |
    | 2017-03-01 |           1.6 |         3.7 |              0.92204 |                     0.6 |         1.0 |             16.65 |      0.0 |     1.0 |           0.0 |    0.813686 |               0.098626 |             0.0 |                0.698626 |      0.001839 |                0.0 |                0.0 |          0.811847 | 0.162369 |       0.649478 |        2.017696 |              0.0 |      0.0 |    92.203964 |  9.113238 |
    | 2017-03-02 |           2.5 |         4.7 |             0.924573 |                     0.6 |         1.9 |             21.15 |      0.0 |     1.9 |           0.0 |    1.549778 |               0.098644 |             0.0 |                0.698644 |      0.001777 |                0.0 |                0.0 |          1.548001 |   0.3096 |       1.238401 |        1.876434 |              0.0 |      0.0 |     92.45732 |  8.475205 |
    | 2017-03-03 |           0.6 |         5.9 |             0.919655 |                     0.6 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               0.493472 |             0.0 |                1.093472 |      0.001609 |                0.0 |                0.0 |         -0.001609 |      0.0 |      -0.001609 |        1.536002 |              0.0 |      0.0 |    91.965457 |  6.937593 |
    | 2017-03-04 |           0.2 |         7.7 |             0.911786 |                     0.2 |         0.0 |             34.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.788765 |             0.0 |                0.988765 |      0.001945 |                0.0 |                0.0 |         -0.001945 |      0.0 |      -0.001945 |         1.25722 |              0.0 |      0.0 |    91.178637 |  5.678428 |
    | 2017-03-05 |           1.7 |         6.3 |             0.912119 |                     0.6 |         1.1 |             28.35 |      0.0 |     1.1 |           0.0 |    0.872404 |               0.196867 |             0.0 |                0.796867 |      0.002574 |                0.0 |                0.0 |           0.86983 | 0.173966 |       0.695864 |        1.155463 |              0.0 |      0.0 |     91.21194 |   5.21883 |
    | 2017-03-06 |           0.3 |         3.7 |             0.909192 |                     0.3 |         0.0 |             16.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.295321 |             0.0 |                0.595321 |      0.002545 |                0.0 |                0.0 |         -0.002545 |      0.0 |      -0.002545 |        0.945552 |              0.0 |      0.0 |    90.919163 |  4.270732 |
    | 2017-03-07 |           0.0 |         1.6 |             0.902333 |                     0.0 |         0.0 |               7.2 |      0.0 |     0.0 |           0.0 |         0.0 |               0.688653 |             0.0 |                0.688653 |      0.002808 |                0.0 |                0.0 |         -0.002808 |      0.0 |      -0.002808 |        0.773643 |              0.0 |      0.0 |    90.233319 |  3.494281 |
    | 2017-03-08 |           1.8 |         4.0 |             0.904114 |                     0.6 |         1.2 |              18.0 |      0.0 |     1.2 |           0.0 |    0.927139 |               0.098232 |             0.0 |                0.698232 |      0.003494 |                0.0 |                0.0 |          0.923646 | 0.184729 |       0.738917 |        0.767349 |              0.0 |      0.0 |    90.411441 |  3.465849 |
    | 2017-03-09 |           8.9 |         5.6 |             0.922702 |                     0.5 |         8.3 |              25.2 |      0.0 |     8.3 |           0.0 |    6.444537 |                    0.0 |             0.0 |                     0.5 |      0.003306 |                0.0 |                0.0 |          6.441231 | 1.288246 |       5.152985 |         1.56233 |              0.1 |      0.0 |    92.270209 |  7.056505 |
    | 2017-03-10 |           0.0 |         5.8 |             0.915813 |                     0.1 |         0.0 |              26.1 |      0.0 |     0.0 |           0.0 |         0.0 |                 0.6906 |             0.0 |                  0.7906 |      0.001732 |                0.0 |                0.0 |         -0.001732 |      0.0 |      -0.001732 |        1.278813 |              0.0 |      0.0 |    91.581342 |  5.775959 |
    | 2017-03-11 |           0.0 |         5.7 |             0.905984 |                     0.0 |         0.0 |             25.65 |      0.0 |     0.0 |           0.0 |         0.0 |               0.985171 |             0.0 |                0.985171 |      0.002237 |                0.0 |                0.0 |         -0.002237 |      0.0 |      -0.002237 |        1.046598 |              0.0 |      0.0 |    90.598408 |  4.727124 |
    | 2017-03-12 |           0.0 |         4.6 |             0.894218 |                     0.0 |         0.0 |              20.7 |      0.0 |     0.0 |           0.0 |         0.0 |                1.17973 |             0.0 |                 1.17973 |      0.003116 |                0.0 |                0.0 |         -0.003116 |      0.0 |      -0.003116 |        0.856317 |              0.0 |      0.0 |    89.421795 |   3.86769 |
    | 2017-03-13 |           0.9 |         4.2 |             0.892055 |                     0.6 |         0.3 |              18.9 |      0.0 |     0.3 |           0.0 |    0.226588 |               0.294161 |             0.0 |                0.894161 |      0.004439 |                0.0 |                0.0 |          0.222149 |  0.04443 |       0.177719 |        0.733308 |              0.0 |      0.0 |    89.205485 |  3.312101 |
    | 2017-03-14 |           0.1 |         7.4 |             0.884262 |                     0.1 |         0.0 |              33.3 |      0.0 |     0.0 |           0.0 |         0.0 |               0.784041 |             0.0 |                0.884041 |      0.004717 |                0.0 |                0.0 |         -0.004717 |      0.0 |      -0.004717 |        0.599527 |              0.0 |      0.0 |    88.426161 |  2.707857 |
    | 2017-03-15 |           0.0 |         6.3 |             0.872581 |                     0.0 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |               1.173923 |             0.0 |                1.173923 |      0.005814 |                0.0 |                0.0 |         -0.005814 |      0.0 |      -0.005814 |        0.489797 |              0.0 |      0.0 |    87.258052 |  2.212246 |
    | 2017-03-16 |           0.0 |         8.7 |             0.859001 |                     0.0 |         0.0 |             39.15 |      0.0 |     0.0 |           0.0 |         0.0 |               1.365698 |             0.0 |                1.365698 |      0.007758 |                0.0 |                0.0 |         -0.007758 |      0.0 |      -0.007758 |        0.399606 |              0.0 |      0.0 |    85.900111 |  1.804882 |
    | 2017-03-17 |           3.9 |         6.4 |             0.864712 |                     0.6 |         3.3 |              28.8 |      0.0 |     3.3 |           0.0 |    2.253344 |               0.486061 |             0.0 |                1.086061 |      0.010512 |                0.0 |                0.0 |          2.242832 | 0.448566 |       1.794266 |        0.652415 |              0.0 |      0.0 |    86.471218 |  2.946733 |
    | 2017-03-18 |           8.7 |         5.2 |               0.8847 |                     0.6 |         8.1 |              23.4 |      0.0 |     8.1 |           0.0 |    5.623705 |               0.486781 |             0.0 |                1.086781 |      0.009286 |                0.0 |                0.0 |          5.614419 | 1.122884 |       4.491535 |        1.348329 |              0.0 |      0.0 |    88.470017 |  6.089939 |
    | 2017-03-19 |          26.4 |         5.1 |             0.953057 |                     0.5 |        25.8 |             22.95 |      0.0 |    25.8 |           0.0 |   18.970089 |                    0.0 |             0.0 |                     0.5 |      0.005748 |                0.0 |                0.0 |         18.964341 | 3.792868 |      15.171473 |         3.85404 |              0.1 |      0.0 |    95.305676 | 17.407372 |
    | 2017-03-20 |          11.5 |         8.7 |             0.965567 |                     0.6 |        11.0 |             39.15 |      0.0 |    11.0 |           0.0 |     9.74939 |                    0.0 |             0.0 |                     0.6 |      0.000388 |                0.0 |                0.0 |          9.749002 |   1.9498 |       7.799201 |        4.569177 |              0.0 |      0.0 |    96.556674 | 20.637397 |
    | 2017-03-21 |           0.9 |         6.2 |              0.95687 |                     0.6 |         0.3 |              27.9 |      0.0 |     0.3 |           0.0 |     0.27474 |               0.895054 |             0.0 |                1.495054 |      0.000153 |                0.0 |                0.0 |          0.274587 | 0.054917 |        0.21967 |        3.780745 |              0.0 |      0.0 |    95.687033 | 17.076322 |
    | 2017-03-22 |           0.0 |         5.9 |             0.937013 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.985997 |             0.0 |                1.985997 |      0.000301 |                0.0 |                0.0 |         -0.000301 |      0.0 |      -0.000301 |        3.095357 |              0.0 |      0.0 |    93.701337 | 13.980664 |
    | 2017-03-23 |           0.0 |         5.2 |             0.921193 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.582984 |             0.0 |                1.582984 |      0.000937 |                0.0 |                0.0 |         -0.000937 |      0.0 |      -0.000937 |        2.534094 |              0.0 |      0.0 |     92.11929 | 11.445632 |
    | 2017-03-24 |           0.0 |         5.2 |             0.905431 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.578027 |             0.0 |                1.578027 |      0.001835 |                0.0 |                0.0 |         -0.001835 |      0.0 |      -0.001835 |        2.074408 |              0.0 |      0.0 |    90.543098 |  9.369388 |
    | 2017-03-25 |           0.0 |         5.9 |             0.893667 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.179588 |             0.0 |                1.179588 |      0.003172 |                0.0 |                0.0 |         -0.003172 |      0.0 |      -0.003172 |        1.697807 |              0.0 |      0.0 |    89.366681 |   7.66841 |
    | 2017-03-26 |           0.0 |         6.7 |             0.880967 |                     0.0 |         0.0 |             30.15 |      0.0 |     0.0 |           0.0 |         0.0 |               1.274537 |             0.0 |                1.274537 |      0.004509 |                0.0 |                0.0 |         -0.004509 |      0.0 |      -0.004509 |         1.38923 |              0.0 |      0.0 |    88.096653 |  6.274671 |
    | 2017-03-27 |           0.0 |         7.0 |              0.86539 |                     0.0 |         0.0 |              31.5 |      0.0 |     0.0 |           0.0 |         0.0 |                  1.564 |             0.0 |                   1.564 |      0.006325 |                0.0 |                0.0 |         -0.006325 |      0.0 |      -0.006325 |        1.136258 |              0.0 |      0.0 |    86.538977 |  5.132088 |
    | 2017-03-28 |           0.0 |         8.3 |              0.84698 |                     0.0 |         0.0 |             37.35 |      0.0 |     0.0 |           0.0 |         0.0 |                1.85009 |             0.0 |                 1.85009 |      0.009147 |                0.0 |                0.0 |         -0.009147 |      0.0 |      -0.009147 |        0.928632 |              0.0 |      0.0 |    84.698034 |   4.19431 |
    | 2017-03-29 |           1.5 |         9.0 |             0.848245 |                     0.6 |         0.9 |              40.5 |      0.0 |     0.9 |           0.0 |    0.593189 |               0.193797 |             0.0 |                0.793797 |      0.013436 |                0.0 |                0.0 |          0.579753 | 0.115951 |       0.463802 |        0.844372 |              0.0 |      0.0 |    84.824484 |   3.81374 |
    | 2017-03-30 |           0.3 |        12.4 |             0.836744 |                     0.3 |         0.0 |              55.8 |      0.0 |     0.0 |           0.0 |         0.0 |               1.163186 |             0.0 |                1.463186 |      0.013106 |                0.0 |                0.0 |         -0.013106 |      0.0 |      -0.013106 |        0.688938 |              0.0 |      0.0 |    83.674404 |  3.111696 |
    | 2017-03-31 |           0.2 |        15.0 |             0.812752 |                     0.2 |         0.0 |              67.5 |      0.0 |     0.0 |           0.0 |         0.0 |               2.415503 |             0.0 |                2.615503 |      0.016317 |                0.0 |                0.0 |         -0.016317 |      0.0 |      -0.016317 |        0.561097 |              0.0 |      0.0 |    81.275217 |  2.534282 |
    | 2017-04-01 |           4.5 |        11.8 |             0.821296 |                     0.8 |         3.7 |              53.1 |      0.0 |     3.7 |           0.0 |    2.198782 |               0.671473 |             0.0 |                1.471473 |       0.02462 |                0.0 |                0.0 |          2.174162 | 0.434832 |        1.73933 |        0.774674 |              0.0 |      0.0 |    82.129582 |  3.498937 |
    | 2017-04-02 |           0.0 |         9.4 |             0.806121 |                     0.0 |         0.0 |              42.3 |      0.0 |     0.0 |           0.0 |         0.0 |               1.538871 |             0.0 |                1.538871 |      0.021401 |                0.0 |                0.0 |         -0.021401 |      0.0 |      -0.021401 |         0.63037 |              0.0 |      0.0 |    80.612112 |  2.847166 |
    | 2017-04-03 |           0.0 |         8.1 |              0.78725 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               1.914427 |             0.0 |                1.914427 |      0.027329 |                0.0 |                0.0 |         -0.027329 |      0.0 |      -0.027329 |         0.51115 |              0.0 |      0.0 |    78.725014 |  2.308687 |
    | 2017-04-04 |           0.0 |         7.9 |             0.767637 |                     0.0 |         0.0 |             35.55 |      0.0 |     0.0 |           0.0 |         0.0 |               1.997409 |             0.0 |                1.997409 |      0.036111 |                0.0 |                0.0 |         -0.036111 |      0.0 |      -0.036111 |        0.411948 |              0.0 |      0.0 |    76.763716 |  1.860628 |
    | 2017-04-05 |           0.4 |         7.5 |             0.755831 |                     0.4 |         0.0 |             33.75 |      0.0 |     0.0 |           0.0 |         0.0 |               1.227698 |             0.0 |                1.627698 |      0.047047 |                0.0 |                0.0 |         -0.047047 |      0.0 |      -0.047047 |        0.328746 |              0.0 |      0.0 |    75.583065 |  1.484835 |
    | 2017-04-06 |           0.0 |         7.2 |             0.740395 |                     0.0 |         0.0 |              32.4 |      0.0 |     0.0 |           0.0 |         0.0 |               1.598123 |             0.0 |                1.598123 |      0.054589 |                0.0 |                0.0 |         -0.054589 |      0.0 |      -0.054589 |         0.25926 |              0.0 |      0.0 |    74.039531 |  1.170986 |
    | 2017-04-07 |           0.0 |         8.1 |             0.733578 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |               0.747322 |             0.0 |                0.747322 |       0.06561 |                0.0 |                0.0 |          -0.06561 |      0.0 |       -0.06561 |        0.200371 |              0.0 |      0.0 |     73.35782 |  0.905006 |
    | 2017-04-08 |           0.0 |         8.6 |             0.722179 |                     0.0 |         0.0 |              38.7 |      0.0 |     0.0 |           0.0 |         0.0 |                1.21086 |             0.0 |                 1.21086 |      0.070915 |                0.0 |                0.0 |         -0.070915 |      0.0 |      -0.070915 |        0.151195 |              0.0 |      0.0 |    72.217875 |  0.682895 |
    | 2017-04-09 |           0.0 |        10.5 |             0.699815 |                     0.0 |         0.0 |             47.25 |      0.0 |     0.0 |           0.0 |         0.0 |               2.316823 |             0.0 |                2.316823 |      0.080413 |                0.0 |                0.0 |         -0.080413 |      0.0 |      -0.080413 |        0.109211 |              0.0 |      0.0 |    69.981465 |  0.493271 |

>>> from hydpy import round_
>>> round_(model.check_waterbalance(conditions))
0.0

>>> withexternalirrigation(False)

.. _whmod_rural_water:

water
_____

|whmod_rural| supports simulating water areas.  To do so, we must change the land type
to |WATER| and the soil type to |NONE|:

>>> landtype(WATER)
>>> soiltype(NONE)
>>> landmonthfactor = petmodel.parameters.control.landmonthfactor
>>> landmonthfactor.water_feb = 1.217
>>> landmonthfactor.water_mar = 1.256
>>> landmonthfactor.water_apr = 1.283

Water areas possess no storage volume.  Hence, precipitation and evaporation translate
directly into percolation and capillary rise:

.. integration-test::

    >>> conditions = test("whmod_rural_water", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | deepwater |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.7302 |                  0.7302 |           0.0 |                0.0 |                0.0 |           -0.7302 |      0.0 |        -0.7302 |       -0.132363 |              0.0 |      0.0 |          0.0 | -0.597837 |
    | 2017-02-11 |           0.4 |        -1.5 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.9736 |                  0.9736 |           0.0 |                0.0 |                0.0 |           -0.5736 |      0.0 |        -0.5736 |       -0.212346 |              0.0 |      0.0 |          0.0 | -0.959092 |
    | 2017-02-12 |           0.0 |        -0.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.8519 |                  0.8519 |           0.0 |                0.0 |                0.0 |           -0.8519 |      0.0 |        -0.8519 |       -0.328277 |              0.0 |      0.0 |          0.0 | -1.482715 |
    | 2017-02-13 |           0.0 |        -1.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |                0.0 |           -0.4868 |      0.0 |        -0.4868 |       -0.357012 |              0.0 |      0.0 |          0.0 | -1.612502 |
    | 2017-02-14 |           0.0 |        -1.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |                0.0 |           -0.4868 |      0.0 |        -0.4868 |       -0.380539 |              0.0 |      0.0 |          0.0 | -1.718763 |
    | 2017-02-15 |           0.0 |         1.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |                0.0 |           -0.4868 |      0.0 |        -0.4868 |       -0.399801 |              0.0 |      0.0 |          0.0 | -1.805762 |
    | 2017-02-16 |           0.2 |         4.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |                0.0 |           -0.2868 |      0.0 |        -0.2868 |       -0.379317 |              0.0 |      0.0 |          0.0 | -1.713245 |
    | 2017-02-17 |           4.5 |         4.5 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.3651 |                  0.3651 |           0.0 |                0.0 |                0.0 |            4.1349 |  0.82698 |        3.30792 |        0.289065 |              0.0 |      0.0 |          0.0 |  1.305609 |
    | 2017-02-18 |           0.0 |         3.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.3651 |                  0.3651 |           0.0 |                0.0 |                0.0 |           -0.3651 |      0.0 |        -0.3651 |        0.170485 |              0.0 |      0.0 |          0.0 |  0.770024 |
    | 2017-02-19 |           3.2 |         4.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |                0.0 |            2.7132 |  0.54264 |        2.17056 |        0.533037 |              0.0 |      0.0 |          0.0 |  2.407546 |
    | 2017-02-20 |           4.6 |         6.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.3651 |                  0.3651 |           0.0 |                0.0 |                0.0 |            4.2349 |  0.84698 |        3.38792 |         1.05054 |              0.0 |      0.0 |          0.0 |  4.744927 |
    | 2017-02-21 |           2.3 |         5.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.7302 |                  0.7302 |           0.0 |                0.0 |                0.0 |            1.5698 |  0.31396 |        1.25584 |        1.087754 |              0.0 |      0.0 |          0.0 |  4.913012 |
    | 2017-02-22 |          18.0 |         6.5 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.9736 |                  0.9736 |           0.0 |                0.0 |                0.0 |           17.0264 |  3.40528 |       13.62112 |        3.359668 |              0.0 |      0.0 |          0.0 | 15.174464 |
    | 2017-02-23 |          19.2 |         5.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.6085 |                  0.6085 |           0.0 |                0.0 |                0.0 |           18.5915 |   3.7183 |        14.8732 |        5.446717 |              0.0 |      0.0 |          0.0 | 24.600947 |
    | 2017-02-24 |           0.4 |         3.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.9736 |                  0.9736 |           0.0 |                0.0 |                0.0 |           -0.5736 |      0.0 |        -0.5736 |        4.355419 |              0.0 |      0.0 |          0.0 | 19.671928 |
    | 2017-02-25 |           8.3 |         3.1 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.6085 |                  0.6085 |           0.0 |                0.0 |                0.0 |            7.6915 |   1.5383 |         6.1532 |        4.681301 |              0.0 |      0.0 |          0.0 | 21.143826 |
    | 2017-02-26 |           5.3 |         7.1 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.4868 |                  0.4868 |           0.0 |                0.0 |                0.0 |            4.8132 |  0.96264 |        3.85056 |        4.530714 |              0.0 |      0.0 |          0.0 | 20.463673 |
    | 2017-02-27 |           0.7 |         9.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.5821 |                  1.5821 |           0.0 |                0.0 |                0.0 |           -0.8821 |      0.0 |        -0.8821 |        3.549537 |              0.0 |      0.0 |          0.0 | 16.032036 |
    | 2017-02-28 |           2.7 |         4.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.0953 |                  1.0953 |           0.0 |                0.0 |                0.0 |            1.6047 |  0.32094 |        1.28376 |        3.138821 |              0.0 |      0.0 |          0.0 | 14.176974 |
    | 2017-03-01 |           1.6 |         3.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.8792 |                  0.8792 |           0.0 |                0.0 |                0.0 |            0.7208 |  0.14416 |        0.57664 |        2.674377 |              0.0 |      0.0 |          0.0 | 12.079238 |
    | 2017-03-02 |           2.5 |         4.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.8792 |                  0.8792 |           0.0 |                0.0 |                0.0 |            1.6208 |  0.32416 |        1.29664 |        2.424635 |              0.0 |      0.0 |          0.0 | 10.951243 |
    | 2017-03-03 |           0.6 |         5.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.3816 |                  1.3816 |           0.0 |                0.0 |                0.0 |           -0.7816 |      0.0 |        -0.7816 |        1.843443 |              0.0 |      0.0 |          0.0 |  8.326199 |
    | 2017-03-04 |           0.2 |         7.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           1.256 |                   1.256 |           0.0 |                0.0 |                0.0 |            -1.056 |      0.0 |         -1.056 |        1.317864 |              0.0 |      0.0 |          0.0 |  5.952336 |
    | 2017-03-05 |           1.7 |         6.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.0048 |                  1.0048 |           0.0 |                0.0 |                0.0 |            0.6952 |  0.13904 |        0.55616 |         1.17979 |              0.0 |      0.0 |          0.0 |  5.328705 |
    | 2017-03-06 |           0.3 |         3.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.7536 |                  0.7536 |           0.0 |                0.0 |                0.0 |           -0.4536 |      0.0 |        -0.4536 |        0.883707 |              0.0 |      0.0 |          0.0 |  3.991399 |
    | 2017-03-07 |           0.0 |         1.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.8792 |                  0.8792 |           0.0 |                0.0 |                0.0 |           -0.8792 |      0.0 |        -0.8792 |        0.564146 |              0.0 |      0.0 |          0.0 |  2.548053 |
    | 2017-03-08 |           1.8 |         4.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.8792 |                  0.8792 |           0.0 |                0.0 |                0.0 |            0.9208 |  0.18416 |        0.73664 |        0.595414 |              0.0 |      0.0 |          0.0 |  2.689279 |
    | 2017-03-09 |           8.9 |         5.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           0.628 |                   0.628 |           0.0 |                0.0 |                0.0 |             8.272 |   1.6544 |         6.6176 |        1.687051 |              0.0 |      0.0 |          0.0 |  7.619828 |
    | 2017-03-10 |           0.0 |         5.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.0048 |                  1.0048 |           0.0 |                0.0 |                0.0 |           -1.0048 |      0.0 |        -1.0048 |        1.199101 |              0.0 |      0.0 |          0.0 |  5.415927 |
    | 2017-03-11 |           0.0 |         5.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           1.256 |                   1.256 |           0.0 |                0.0 |                0.0 |            -1.256 |      0.0 |         -1.256 |        0.754067 |              0.0 |      0.0 |          0.0 |   3.40586 |
    | 2017-03-12 |           0.0 |         4.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.5072 |                  1.5072 |           0.0 |                0.0 |                0.0 |           -1.5072 |      0.0 |        -1.5072 |        0.344169 |              0.0 |      0.0 |          0.0 |  1.554491 |
    | 2017-03-13 |           0.9 |         4.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.1304 |                  1.1304 |           0.0 |                0.0 |                0.0 |           -0.2304 |      0.0 |        -0.2304 |        0.240017 |              0.0 |      0.0 |          0.0 |  1.084074 |
    | 2017-03-14 |           0.1 |         7.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.1304 |                  1.1304 |           0.0 |                0.0 |                0.0 |           -1.0304 |      0.0 |        -1.0304 |         0.00973 |              0.0 |      0.0 |          0.0 |  0.043945 |
    | 2017-03-15 |           0.0 |         6.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.5072 |                  1.5072 |           0.0 |                0.0 |                0.0 |           -1.5072 |      0.0 |        -1.5072 |       -0.265243 |              0.0 |      0.0 |          0.0 | -1.198012 |
    | 2017-03-16 |           0.0 |         8.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.7584 |                  1.7584 |           0.0 |                0.0 |                0.0 |           -1.7584 |      0.0 |        -1.7584 |       -0.535907 |              0.0 |      0.0 |          0.0 | -2.420505 |
    | 2017-03-17 |           3.9 |         6.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.3816 |                  1.3816 |           0.0 |                0.0 |                0.0 |            2.5184 |  0.50368 |        2.01472 |       -0.073556 |              0.0 |      0.0 |          0.0 | -0.332229 |
    | 2017-03-18 |           8.7 |         5.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.3816 |                  1.3816 |           0.0 |                0.0 |                0.0 |            7.3184 |  1.46368 |        5.85472 |        1.001058 |              0.0 |      0.0 |          0.0 |  4.521433 |
    | 2017-03-19 |          26.4 |         5.1 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           0.628 |                   0.628 |           0.0 |                0.0 |                0.0 |            25.772 |   5.1544 |        20.6176 |        4.556934 |              0.0 |      0.0 |          0.0 |   20.5821 |
    | 2017-03-20 |          11.5 |         8.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          0.7536 |                  0.7536 |           0.0 |                0.0 |                0.0 |           10.7464 |  2.14928 |        8.59712 |        5.289295 |              0.0 |      0.0 |          0.0 | 23.889924 |
    | 2017-03-21 |           0.9 |         6.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           1.884 |                   1.884 |           0.0 |                0.0 |                0.0 |            -0.984 |      0.0 |         -0.984 |         4.15214 |              0.0 |      0.0 |          0.0 | 18.753785 |
    | 2017-03-22 |           0.0 |         5.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           2.512 |                   2.512 |           0.0 |                0.0 |                0.0 |            -2.512 |      0.0 |         -2.512 |        2.944136 |              0.0 |      0.0 |          0.0 | 13.297649 |
    | 2017-03-23 |           0.0 |         5.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.0096 |                  2.0096 |           0.0 |                0.0 |                0.0 |           -2.0096 |      0.0 |        -2.0096 |        2.046176 |              0.0 |      0.0 |          0.0 |  9.241873 |
    | 2017-03-24 |           0.0 |         5.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.0096 |                  2.0096 |           0.0 |                0.0 |                0.0 |           -2.0096 |      0.0 |        -2.0096 |        1.310989 |              0.0 |      0.0 |          0.0 |  5.921284 |
    | 2017-03-25 |           0.0 |         5.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.5072 |                  1.5072 |           0.0 |                0.0 |                0.0 |           -1.5072 |      0.0 |        -1.5072 |        0.800138 |              0.0 |      0.0 |          0.0 |  3.613946 |
    | 2017-03-26 |           0.0 |         6.7 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.6328 |                  1.6328 |           0.0 |                0.0 |                0.0 |           -1.6328 |      0.0 |        -1.6328 |        0.359121 |              0.0 |      0.0 |          0.0 |  1.622025 |
    | 2017-03-27 |           0.0 |         7.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.0096 |                  2.0096 |           0.0 |                0.0 |                0.0 |           -2.0096 |      0.0 |        -2.0096 |       -0.070255 |              0.0 |      0.0 |          0.0 | -0.317319 |
    | 2017-03-28 |           0.0 |         8.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.3864 |                  2.3864 |           0.0 |                0.0 |                0.0 |           -2.3864 |      0.0 |        -2.3864 |       -0.490101 |              0.0 |      0.0 |          0.0 | -2.213618 |
    | 2017-03-29 |           1.5 |         9.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.0048 |                  1.0048 |           0.0 |                0.0 |                0.0 |            0.4952 |  0.09904 |        0.39616 |       -0.329449 |              0.0 |      0.0 |          0.0 | -1.488009 |
    | 2017-03-30 |           0.3 |        12.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           1.884 |                   1.884 |           0.0 |                0.0 |                0.0 |            -1.584 |      0.0 |         -1.584 |       -0.556861 |              0.0 |      0.0 |          0.0 | -2.515148 |
    | 2017-03-31 |           0.2 |        15.0 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          3.3912 |                  3.3912 |           0.0 |                0.0 |                0.0 |           -3.1912 |      0.0 |        -3.1912 |       -1.034385 |              0.0 |      0.0 |          0.0 | -4.671963 |
    | 2017-04-01 |           4.5 |        11.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.9245 |                  1.9245 |           0.0 |                0.0 |                0.0 |            2.5755 |   0.5151 |         2.0604 |       -0.473396 |              0.0 |      0.0 |          0.0 | -2.138167 |
    | 2017-04-02 |           0.0 |         9.4 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.0528 |                  2.0528 |           0.0 |                0.0 |                0.0 |           -2.0528 |      0.0 |        -2.0528 |       -0.759693 |              0.0 |      0.0 |          0.0 | -3.431273 |
    | 2017-04-03 |           0.0 |         8.1 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |           2.566 |                   2.566 |           0.0 |                0.0 |                0.0 |            -2.566 |      0.0 |         -2.566 |       -1.087121 |              0.0 |      0.0 |          0.0 | -4.910152 |
    | 2017-04-04 |           0.0 |         7.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.6943 |                  2.6943 |           0.0 |                0.0 |                0.0 |           -2.6943 |      0.0 |        -2.6943 |       -1.378453 |              0.0 |      0.0 |          0.0 | -6.225999 |
    | 2017-04-05 |           0.4 |         7.5 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.1811 |                  2.1811 |           0.0 |                0.0 |                0.0 |           -1.7811 |      0.0 |        -1.7811 |       -1.451441 |              0.0 |      0.0 |          0.0 | -6.555658 |
    | 2017-04-06 |           0.0 |         7.2 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          2.1811 |                  2.1811 |           0.0 |                0.0 |                0.0 |           -2.1811 |      0.0 |        -2.1811 |       -1.583706 |              0.0 |      0.0 |          0.0 | -7.153052 |
    | 2017-04-07 |           0.0 |         8.1 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.0264 |                  1.0264 |           0.0 |                0.0 |                0.0 |           -1.0264 |      0.0 |        -1.0264 |       -1.482683 |              0.0 |      0.0 |          0.0 | -6.696769 |
    | 2017-04-08 |           0.0 |         8.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          1.6679 |                  1.6679 |           0.0 |                0.0 |                0.0 |           -1.6679 |      0.0 |        -1.6679 |       -1.516257 |              0.0 |      0.0 |          0.0 | -6.848412 |
    | 2017-04-09 |           0.0 |        10.5 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |          3.2075 |                  3.2075 |           0.0 |                0.0 |                0.0 |           -3.2075 |      0.0 |        -3.2075 |       -1.822828 |              0.0 |      0.0 |          0.0 | -8.233084 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_rural_sealed:

sealed
______

Sealed areas have no soil storage.  All rainfall or meltwater reaching the sealed
surface turns directly into surface runoff.  The only other loss is interception
evaporation:

.. integration-test::

    >>> landtype(SEALED)
    >>> interceptioncapacity.sealed = 2.0
    >>> landmonthfactor.sealed = 1.0
    >>> conditions = test("whmod_rural_sealed", get_conditions="2017-02-10")
    |       date | precipitation | temperature | relativesoilmoisture | interceptionevaporation | throughfall | potentialsnowmelt | snowmelt | ponding | surfacerunoff | percolation | soilevapotranspiration | lakeevaporation | totalevapotranspiration | capillaryrise | requiredirrigation | externalirrigation | potentialrecharge | baseflow | actualrecharge | delayedrecharge | interceptedwater | snowpack | soilmoisture | deepwater |
    --------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-02-11 |           0.4 |        -1.5 |                  0.0 |                     0.4 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-02-12 |           0.0 |        -0.9 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-02-13 |           0.0 |        -1.6 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-02-14 |           0.0 |        -1.3 |                  0.0 |                     0.0 |         0.0 |               0.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-02-15 |           0.0 |         1.7 |                  0.0 |                     0.0 |         0.0 |              7.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-02-16 |           0.2 |         4.4 |                  0.0 |                     0.2 |         0.0 |              19.8 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.2 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-02-17 |           4.5 |         4.5 |                  0.0 |                     0.3 |         2.5 |             20.25 |      0.0 |     2.5 |           2.5 |         0.0 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.7 |      0.0 |          0.0 |       0.0 |
    | 2017-02-18 |           0.0 |         3.4 |                  0.0 |                     0.3 |         0.0 |              15.3 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.4 |      0.0 |          0.0 |       0.0 |
    | 2017-02-19 |           3.2 |         4.8 |                  0.0 |                     0.4 |         2.6 |              21.6 |      0.0 |     2.6 |           2.6 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.6 |      0.0 |          0.0 |       0.0 |
    | 2017-02-20 |           4.6 |         6.7 |                  0.0 |                     0.3 |         4.2 |             30.15 |      0.0 |     4.2 |           4.2 |         0.0 |                    0.0 |             0.0 |                     0.3 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.7 |      0.0 |          0.0 |       0.0 |
    | 2017-02-21 |           2.3 |         5.8 |                  0.0 |                     0.6 |         2.0 |              26.1 |      0.0 |     2.0 |           2.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.4 |      0.0 |          0.0 |       0.0 |
    | 2017-02-22 |          18.0 |         6.5 |                  0.0 |                     0.8 |        17.4 |             29.25 |      0.0 |    17.4 |          17.4 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.2 |      0.0 |          0.0 |       0.0 |
    | 2017-02-23 |          19.2 |         5.0 |                  0.0 |                     0.5 |        18.4 |              22.5 |      0.0 |    18.4 |          18.4 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |       0.0 |
    | 2017-02-24 |           0.4 |         3.0 |                  0.0 |                     0.8 |         0.0 |              13.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.1 |      0.0 |          0.0 |       0.0 |
    | 2017-02-25 |           8.3 |         3.1 |                  0.0 |                     0.5 |         7.4 |             13.95 |      0.0 |     7.4 |           7.4 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |       0.0 |
    | 2017-02-26 |           5.3 |         7.1 |                  0.0 |                     0.4 |         4.8 |             31.95 |      0.0 |     4.8 |           4.8 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.6 |      0.0 |          0.0 |       0.0 |
    | 2017-02-27 |           0.7 |         9.4 |                  0.0 |                     1.3 |         0.3 |              42.3 |      0.0 |     0.3 |           0.3 |         0.0 |                    0.0 |             0.0 |                     1.3 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.7 |      0.0 |          0.0 |       0.0 |
    | 2017-02-28 |           2.7 |         4.6 |                  0.0 |                     0.9 |         1.4 |              20.7 |      0.0 |     1.4 |           1.4 |         0.0 |                    0.0 |             0.0 |                     0.9 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.1 |      0.0 |          0.0 |       0.0 |
    | 2017-03-01 |           1.6 |         3.7 |                  0.0 |                     0.7 |         0.7 |             16.65 |      0.0 |     0.7 |           0.7 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.3 |      0.0 |          0.0 |       0.0 |
    | 2017-03-02 |           2.5 |         4.7 |                  0.0 |                     0.7 |         1.8 |             21.15 |      0.0 |     1.8 |           1.8 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.3 |      0.0 |          0.0 |       0.0 |
    | 2017-03-03 |           0.6 |         5.9 |                  0.0 |                     1.1 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     1.1 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.8 |      0.0 |          0.0 |       0.0 |
    | 2017-03-04 |           0.2 |         7.7 |                  0.0 |                     1.0 |         0.0 |             34.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     1.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-05 |           1.7 |         6.3 |                  0.0 |                     0.8 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.9 |      0.0 |          0.0 |       0.0 |
    | 2017-03-06 |           0.3 |         3.7 |                  0.0 |                     0.6 |         0.0 |             16.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.6 |      0.0 |          0.0 |       0.0 |
    | 2017-03-07 |           0.0 |         1.6 |                  0.0 |                     0.6 |         0.0 |               7.2 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-08 |           1.8 |         4.0 |                  0.0 |                     0.7 |         0.0 |              18.0 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.1 |      0.0 |          0.0 |       0.0 |
    | 2017-03-09 |           8.9 |         5.6 |                  0.0 |                     0.5 |         8.0 |              25.2 |      0.0 |     8.0 |           8.0 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |       0.0 |
    | 2017-03-10 |           0.0 |         5.8 |                  0.0 |                     0.8 |         0.0 |              26.1 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.7 |      0.0 |          0.0 |       0.0 |
    | 2017-03-11 |           0.0 |         5.7 |                  0.0 |                     0.7 |         0.0 |             25.65 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.7 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-12 |           0.0 |         4.6 |                  0.0 |                     0.0 |         0.0 |              20.7 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-13 |           0.9 |         4.2 |                  0.0 |                     0.9 |         0.0 |              18.9 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.9 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-14 |           0.1 |         7.4 |                  0.0 |                     0.1 |         0.0 |              33.3 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.1 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-15 |           0.0 |         6.3 |                  0.0 |                     0.0 |         0.0 |             28.35 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-16 |           0.0 |         8.7 |                  0.0 |                     0.0 |         0.0 |             39.15 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-17 |           3.9 |         6.4 |                  0.0 |                     1.1 |         1.9 |              28.8 |      0.0 |     1.9 |           1.9 |         0.0 |                    0.0 |             0.0 |                     1.1 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.9 |      0.0 |          0.0 |       0.0 |
    | 2017-03-18 |           8.7 |         5.2 |                  0.0 |                     1.1 |         7.6 |              23.4 |      0.0 |     7.6 |           7.6 |         0.0 |                    0.0 |             0.0 |                     1.1 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.9 |      0.0 |          0.0 |       0.0 |
    | 2017-03-19 |          26.4 |         5.1 |                  0.0 |                     0.5 |        25.3 |             22.95 |      0.0 |    25.3 |          25.3 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.5 |      0.0 |          0.0 |       0.0 |
    | 2017-03-20 |          11.5 |         8.7 |                  0.0 |                     0.6 |        11.0 |             39.15 |      0.0 |    11.0 |          11.0 |         0.0 |                    0.0 |             0.0 |                     0.6 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              1.4 |      0.0 |          0.0 |       0.0 |
    | 2017-03-21 |           0.9 |         6.2 |                  0.0 |                     1.5 |         0.3 |              27.9 |      0.0 |     0.3 |           0.3 |         0.0 |                    0.0 |             0.0 |                     1.5 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.5 |      0.0 |          0.0 |       0.0 |
    | 2017-03-22 |           0.0 |         5.9 |                  0.0 |                     0.5 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-23 |           0.0 |         5.2 |                  0.0 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-24 |           0.0 |         5.2 |                  0.0 |                     0.0 |         0.0 |              23.4 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-25 |           0.0 |         5.9 |                  0.0 |                     0.0 |         0.0 |             26.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-26 |           0.0 |         6.7 |                  0.0 |                     0.0 |         0.0 |             30.15 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-27 |           0.0 |         7.0 |                  0.0 |                     0.0 |         0.0 |              31.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-28 |           0.0 |         8.3 |                  0.0 |                     0.0 |         0.0 |             37.35 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-29 |           1.5 |         9.0 |                  0.0 |                     0.8 |         0.0 |              40.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.8 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.7 |      0.0 |          0.0 |       0.0 |
    | 2017-03-30 |           0.3 |        12.4 |                  0.0 |                     1.0 |         0.0 |              55.8 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     1.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-03-31 |           0.2 |        15.0 |                  0.0 |                     0.2 |         0.0 |              67.5 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.2 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-04-01 |           4.5 |        11.8 |                  0.0 |                     1.5 |         2.5 |              53.1 |      0.0 |     2.5 |           2.5 |         0.0 |                    0.0 |             0.0 |                     1.5 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.5 |      0.0 |          0.0 |       0.0 |
    | 2017-04-02 |           0.0 |         9.4 |                  0.0 |                     0.5 |         0.0 |              42.3 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.5 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-04-03 |           0.0 |         8.1 |                  0.0 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-04-04 |           0.0 |         7.9 |                  0.0 |                     0.0 |         0.0 |             35.55 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-04-05 |           0.4 |         7.5 |                  0.0 |                     0.4 |         0.0 |             33.75 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.4 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-04-06 |           0.0 |         7.2 |                  0.0 |                     0.0 |         0.0 |              32.4 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-04-07 |           0.0 |         8.1 |                  0.0 |                     0.0 |         0.0 |             36.45 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-04-08 |           0.0 |         8.6 |                  0.0 |                     0.0 |         0.0 |              38.7 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |
    | 2017-04-09 |           0.0 |        10.5 |                  0.0 |                     0.0 |         0.0 |             47.25 |      0.0 |     0.0 |           0.0 |         0.0 |                    0.0 |             0.0 |                     0.0 |           0.0 |                0.0 |                0.0 |               0.0 |      0.0 |            0.0 |             0.0 |              0.0 |      0.0 |          0.0 |       0.0 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_rural_multiple_zones:

multiple zones
______________

For the following example, we divide the considered site into four zones of different
land types.  The new land type |CONIFER| relies, like all other land types except
|WATER| and |SEALED|, on the same equations as |GRASS| (while parametrisation is
usually different, of course):

>>> name2value = {par.name: par.value for par in control}
>>> nmbzones(4)
>>> landtype(GRASS, CONIFER, WATER, SEALED)
>>> soiltype(SAND, SAND, NONE, NONE)
>>> zonearea(2.5)
>>> interceptioncapacity.conifer = 2.0
>>> landmonthfactor.conifer = 1.335

The `name2value` dictionary helps us to restore all other settings of the previous
examples with little effort:

>>> for name, value in name2value.items():
...     if name not in ("nmbzones", "landtype", "soiltype", "zonearea", "interceptioncapacity"):
...         control[name].value = value
>>> model.update_parameters()
>>> aetmodel.parameters.control.dissefactor(4.0)
>>> petmodel.parameters.control.dampingfactor(1.0)
>>> retmodel.parameters.control.evapotranspirationfactor(1.0)

The simulation results for the individual zones are very similar to those of the
corresponding previous results because |whmod_rural| does not consider lateral
exchanges:

.. integration-test::

    >>> conditions = test("whmod_rural_multiple_zones", get_conditions="2017-02-10")
    |       date | precipitation | temperature |                          relativesoilmoisture |                   interceptionevaporation |                     throughfall |                    potentialsnowmelt |                snowmelt |                     ponding |                surfacerunoff |                            percolation |                          soilevapotranspiration |                   lakeevaporation |                             totalevapotranspiration |                          capillaryrise |                            requiredirrigation |                externalirrigation |                                potentialrecharge |                              baseflow | actualrecharge | delayedrecharge |                   interceptedwater |                snowpack |                            soilmoisture | deepwater |
    -----------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |        -2.8 |   0.0375    0.0375  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.7302              0.0 |      0.0       0.0  0.7302                      0.0 |     3.75      3.75  0.0            0.0 |     66.25      66.25  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |     -3.75      -3.75  -0.7302                0.0 |      0.0       0.0      0.0       0.0 |       -2.05755 |       -0.372971 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |      3.75       3.75  0.0           0.0 | -1.684579 |
    | 2017-02-11 |           0.4 |        -1.5 | 0.070632  0.070427  0.0                   0.0 | 0.4     0.4  0.0                      0.4 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.030545  0.051011  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.430545  0.451011  0.9736                      0.4 | 3.343748  3.343748  0.0            0.0 | 62.936798  62.957263  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -3.343748  -3.343748  -0.5736                0.0 |      0.0       0.0      0.0       0.0 |      -1.815274 |       -0.634416 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  7.063202   7.042737  0.0           0.0 | -2.865437 |
    | 2017-02-12 |           0.0 |        -0.9 | 0.099731  0.099214  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.100328  0.133554  0.0                     0.0 | 0.0  0.0  0.8519              0.0 | 0.100328  0.133554  0.8519                      0.0 | 3.010193  3.012182  0.0            0.0 | 60.026932  60.078634  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -3.010193  -3.012182  -0.8519                0.0 |      0.0       0.0      0.0       0.0 |      -1.718569 |       -0.830939 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  9.973068   9.921366  0.0           0.0 | -3.753067 |
    | 2017-02-13 |           0.0 |        -1.6 | 0.126288  0.125554  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.080508  0.106933  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.080508  0.106933  0.4868                      0.0 | 2.736205  2.740922  0.0            0.0 | 57.371235  57.444646  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.736205  -2.740922  -0.4868                0.0 |      0.0       0.0      0.0       0.0 |      -1.490982 |       -0.950585 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 12.628765  12.555354  0.0           0.0 | -4.293464 |
    | 2017-02-14 |           0.0 |        -1.3 | 0.150286  0.149284  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.101248  0.134409  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.101248  0.134409  0.4868                      0.0 | 2.501132  2.507442  0.0            0.0 | 54.971351  55.071613  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.501132  -2.507442  -0.4868                0.0 |      0.0       0.0      0.0       0.0 |      -1.373844 |       -1.027309 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 15.028649  14.928387  0.0           0.0 | -4.639999 |
    | 2017-02-15 |           0.0 |         1.7 | 0.172097  0.170786  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  7.65   7.65  0.0               7.65 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.119564  0.158608  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.119564  0.158608  0.4868                      0.0 | 2.300641  2.308794  0.0            0.0 | 52.790274  52.921427  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.300641  -2.308794  -0.4868                0.0 |      0.0       0.0      0.0       0.0 |      -1.274059 |       -1.072037 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 17.209726  17.078573  0.0           0.0 | -4.842021 |
    | 2017-02-16 |           0.2 |         4.4 | 0.192698  0.191041  0.0                   0.0 | 0.2     0.2  0.0                      0.2 |  0.0      0.0  0.0          0.0 |  19.8   19.8  0.0               19.8 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.067899  0.112586  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.267899  0.312586  0.4868                      0.2 | 2.127988  2.138117  0.0            0.0 | 50.730184  50.895895  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.127988  -2.138117  -0.2868                0.0 |      0.0       0.0      0.0       0.0 |      -1.138226 |       -1.084035 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 19.269816  19.104105  0.0           0.0 | -4.896213 |
    | 2017-02-17 |           4.5 |         4.5 | 0.252771  0.235501  0.0                   0.0 | 0.3  0.4005  0.0                      0.3 |  4.1      2.5  0.0          2.5 | 20.25  20.25  0.0              20.25 | 0.0  0.0  0.0       0.0 |  4.1      2.5  0.0      2.5 | 0.0  0.0  0.0            2.5 |  0.065722   0.039215  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 |      0.3    0.4005  0.3651                      0.3 | 1.973055   1.98523  0.0            0.0 | 44.722852  46.449881  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.907333  -1.946015   4.1349                0.0 |      0.0       0.0  0.82698       0.0 |      -0.136357 |        -0.91225 | 0.1  1.5995  0.0               1.7 | 0.0  0.0  0.0       0.0 | 25.277148  23.550119  0.0           0.0 | -4.120319 |
    | 2017-02-18 |           0.0 |         3.4 | 0.267458  0.252257  0.0                   0.0 | 0.1  0.4005  0.0                      0.3 |  0.0      0.0  0.0          0.0 |  15.3   15.3  0.0               15.3 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.095868       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 | 0.195868    0.4005  0.3651                      0.3 | 1.564558  1.675567  0.0            0.0 | 43.254162  44.774314  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.564558  -1.675567  -0.3651                0.0 |      0.0       0.0      0.0       0.0 |      -0.901306 |       -0.910266 | 0.0   1.199  0.0               1.4 | 0.0  0.0  0.0       0.0 | 26.745838  25.225686  0.0           0.0 | -4.111359 |
    | 2017-02-19 |           3.2 |         4.8 | 0.309177  0.291169  0.0                   0.0 | 0.4   0.534  0.0                      0.4 |  2.8    2.399  0.0          2.6 |  21.6   21.6  0.0               21.6 | 0.0  0.0  0.0       0.0 |  2.8    2.399  0.0      2.6 | 0.0  0.0  0.0            2.6 |  0.102207   0.075607  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.4868              0.0 |      0.4     0.534  0.4868                      0.4 | 1.474104  1.567792  0.0            0.0 | 39.082265  40.883128  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.371898  -1.492186   2.7132                0.0 |      0.0       0.0  0.54264       0.0 |      -0.173381 |       -0.776692 | 0.0   1.466  0.0               1.6 | 0.0  0.0  0.0       0.0 | 30.917735  29.116872  0.0           0.0 | -3.508048 |
    | 2017-02-20 |           4.6 |         6.7 | 0.361335  0.343347  0.0                   0.0 | 0.3  0.4005  0.0                      0.3 |  4.2    4.066  0.0          4.2 | 30.15  30.15  0.0              30.15 | 0.0  0.0  0.0       0.0 |  4.2    4.066  0.0      4.2 | 0.0  0.0  0.0            4.2 |  0.220591   0.183689  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 |      0.3    0.4005  0.3651                      0.3 |  1.23632  1.335549  0.0            0.0 | 33.866536  35.665268  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.015729   -1.15186   4.2349                0.0 |      0.0       0.0  0.84698       0.0 |       0.305083 |       -0.580599 | 0.1  1.5995  0.0               1.7 | 0.0  0.0  0.0       0.0 | 36.133464  34.334732  0.0           0.0 | -2.622366 |
    | 2017-02-21 |           2.3 |         5.8 | 0.388275  0.371662  0.0                   0.0 | 0.4   0.801  0.0                      0.6 |  2.0   1.8995  0.0          2.0 |  26.1   26.1  0.0               26.1 | 0.0  0.0  0.0       0.0 |  2.0   1.8995  0.0      2.0 | 0.0  0.0  0.0            2.0 |   0.15535   0.129794  0.0          0.0 | 0.127491       0.0  0.0                     0.0 | 0.0  0.0  0.7302              0.0 | 0.527491     0.801  0.7302                      0.6 | 0.976903   1.06179  0.0            0.0 | 31.172475  32.833773  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.821553  -0.931995   1.5698                0.0 |      0.0       0.0  0.31396       0.0 |      -0.124427 |       -0.497909 | 0.0   1.199  0.0               1.4 | 0.0  0.0  0.0       0.0 | 38.827525  37.166227  0.0           0.0 | -2.248884 |
    | 2017-02-22 |          18.0 |         6.5 |   0.5538  0.538616  0.0                   0.0 | 0.4   1.068  0.0                      0.8 | 17.6   17.199  0.0         17.4 | 29.25  29.25  0.0              29.25 | 0.0  0.0  0.0       0.0 | 17.6   17.199  0.0     17.4 | 0.0  0.0  0.0           17.4 |  1.637522    1.43386  0.0          0.0 | 0.268404       0.0  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.668404     1.068  0.9736                      0.8 | 0.858419  0.930274  0.0            0.0 | 14.619981  16.138359  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.779103   0.503586  17.0264                0.0 | 0.155821  0.100717  3.40528       0.0 |       3.661818 |        0.256121 | 0.0   0.932  0.0               1.2 | 0.0  0.0  0.0       0.0 | 55.380019  53.861641  0.0           0.0 |  1.156812 |
    | 2017-02-23 |          19.2 |         5.0 |  0.70165  0.685256  0.0                   0.0 | 0.4  0.6675  0.0                      0.5 | 18.8   18.132  0.0         18.4 |  22.5   22.5  0.0               22.5 | 0.0  0.0  0.0       0.0 | 18.8   18.132  0.0     18.4 | 0.0  0.0  0.0           18.4 |  4.265135   3.836314  0.0          0.0 | 0.083065       0.0  0.0                     0.0 | 0.0  0.0  0.6085              0.0 | 0.483065    0.6675  0.6085                      0.5 | 0.333134  0.368314  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  3.932001   3.468001  18.5915                0.0 |   0.7864    0.6936   3.7183       0.0 |         5.1983 |        1.151986 | 0.0  1.3325  0.0               1.5 | 0.0  0.0  0.0       0.0 | 70.164954   68.52564  0.0           0.0 |  5.203126 |
    | 2017-02-24 |           0.4 |         3.0 | 0.698974  0.686426  0.0                   0.0 | 0.4   1.068  0.0                      0.8 |  0.0      0.0  0.0          0.0 |  13.5   13.5  0.0               13.5 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.367107       0.0  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.767107     1.068  0.9736                      0.8 | 0.099589  0.116924  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.099589  -0.116924  -0.5736                0.0 |      0.0       0.0      0.0       0.0 |      -0.197528 |        0.907361 | 0.0  0.6645  0.0               1.1 | 0.0  0.0  0.0       0.0 | 69.897436  68.642564  0.0           0.0 |  4.098237 |
    | 2017-02-25 |           8.3 |         3.1 | 0.745929  0.730143  0.0                   0.0 | 0.4  0.6675  0.0                      0.5 |  7.9   6.9645  0.0          7.4 | 13.95  13.95  0.0              13.95 | 0.0  0.0  0.0       0.0 |  7.9   6.9645  0.0      7.4 | 0.0  0.0  0.0            7.4 |  3.215139   2.708403  0.0          0.0 | 0.091655       0.0  0.0                     0.0 | 0.0  0.0  0.6085              0.0 | 0.491655    0.6675  0.6085                      0.5 | 0.102292  0.115625  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  3.112847   2.592778   7.6915                0.0 | 0.622569  0.518556   1.5383       0.0 |       2.679425 |        1.228582 | 0.0  1.3325  0.0               1.5 | 0.0  0.0  0.0       0.0 | 74.592934  73.014286  0.0           0.0 |   5.54908 |
    | 2017-02-26 |           5.3 |         7.1 | 0.772067   0.75617  0.0                   0.0 | 0.4   0.534  0.0                      0.4 |  4.9   4.6325  0.0          4.8 | 31.95  31.95  0.0              31.95 | 0.0  0.0  0.0       0.0 |  4.9   4.6325  0.0      4.8 | 0.0  0.0  0.0            4.8 |  2.347723   2.103522  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.4868              0.0 |      0.4     0.534  0.4868                      0.4 | 0.061503  0.073694  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |   2.28622   2.029828   4.8132                0.0 | 0.457244  0.405966  0.96264       0.0 |        1.82585 |        1.336848 | 0.0   1.466  0.0               1.6 | 0.0  0.0  0.0       0.0 | 77.206714  75.616958  0.0           0.0 |  6.038082 |
    | 2017-02-27 |           0.7 |         9.4 |  0.76543   0.75755  0.0                   0.0 | 0.4  1.7355  0.0                      1.3 |  0.3    0.166  0.0          0.3 |  42.3   42.3  0.0               42.3 | 0.0  0.0  0.0       0.0 |  0.3    0.166  0.0      0.3 | 0.0  0.0  0.0            0.3 |  0.156717   0.082304  0.0          0.0 | 0.851358       0.0  0.0                     0.0 | 0.0  0.0  1.5821              0.0 | 1.251358    1.7355  1.5821                      1.3 | 0.044407  0.054362  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |   0.11231   0.027942  -0.8821                0.0 | 0.022462  0.005588      0.0       0.0 |      -0.192474 |        1.059629 | 0.0  0.2645  0.0               0.7 | 0.0  0.0  0.0       0.0 | 76.543046  75.755016  0.0           0.0 |  4.785979 |
    | 2017-02-28 |           2.7 |         4.6 | 0.772439  0.762926  0.0                   0.0 | 0.4  1.2015  0.0                      0.9 |  2.3   0.9645  0.0          1.4 |  20.7   20.7  0.0               20.7 | 0.0  0.0  0.0       0.0 |  2.3   0.9645  0.0      1.4 | 0.0  0.0  0.0            1.4 |  1.175743   0.480403  0.0          0.0 | 0.471796       0.0  0.0                     0.0 | 0.0  0.0  1.0953              0.0 | 0.871796    1.2015  1.0953                      0.9 |   0.0484  0.053444  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  1.127343   0.426959   1.6047                0.0 | 0.225469  0.085392  0.32094       0.0 |         0.6318 |        0.982077 | 0.0  0.7985  0.0               1.1 | 0.0  0.0  0.0       0.0 | 77.243907  76.292557  0.0           0.0 |  4.435702 |
    | 2017-03-01 |           1.6 |         3.7 | 0.776705   0.76539  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.0   0.3985  0.0          0.7 | 16.65  16.65  0.0              16.65 | 0.0  0.0  0.0       0.0 |  1.0   0.3985  0.0      0.7 | 0.0  0.0  0.0            0.7 |  0.523023   0.202041  0.0          0.0 | 0.094608       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.694608    0.9345  0.8792                      0.7 |  0.04419  0.049967  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.478833   0.152074   0.7208                0.0 | 0.095767  0.030415  0.14416       0.0 |       0.270341 |        0.853061 | 0.0  1.0655  0.0               1.3 | 0.0  0.0  0.0       0.0 | 77.670465  76.538983  0.0           0.0 |  3.852983 |
    | 2017-03-02 |           2.5 |         4.7 | 0.785099  0.773527  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.9   1.5655  0.0          1.8 | 21.15  21.15  0.0              21.15 | 0.0  0.0  0.0       0.0 |  1.9   1.5655  0.0      1.8 | 0.0  0.0  0.0            1.8 |  1.007577   0.800166  0.0          0.0 | 0.094757       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.694757    0.9345  0.8792                      0.7 | 0.041751  0.048425  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.965826    0.75174   1.6208                0.0 | 0.193165  0.150348  0.32416       0.0 |       0.667673 |        0.819456 | 0.0  1.0655  0.0               1.3 | 0.0  0.0  0.0       0.0 | 78.509882  77.352742  0.0           0.0 |    3.7012 |
    | 2017-03-03 |           0.6 |         5.9 | 0.780719  0.773963  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  0.0      0.0  0.0          0.0 | 26.55  26.55  0.0              26.55 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.475214       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 1.075214    1.4685  1.3816                      1.1 | 0.037218  0.043559  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.037218  -0.043559  -0.7816                0.0 |      0.0       0.0      0.0       0.0 |      -0.215594 |        0.631833 | 0.0   0.197  0.0               0.8 | 0.0  0.0  0.0       0.0 | 78.071886  77.396302  0.0           0.0 |  2.853773 |
    | 2017-03-04 |           0.2 |         7.7 | 0.773523  0.765517  0.0                   0.0 | 0.2   0.397  0.0                      1.0 |  0.0      0.0  0.0          0.0 | 34.65  34.65  0.0              34.65 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.759159  0.887927  0.0                     0.0 | 0.0  0.0   1.256              0.0 | 0.959159  1.284927   1.256                      1.0 |  0.03954  0.043308  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.03954  -0.043308   -1.056                0.0 |      0.0       0.0      0.0       0.0 |      -0.284712 |        0.465692 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 77.352267  76.551682  0.0           0.0 |  2.103369 |
    | 2017-03-05 |           1.7 |         6.3 | 0.777292     0.766  0.0                   0.0 | 0.6   1.068  0.0                      0.8 |  1.1      0.0  0.0          0.0 | 28.35  28.35  0.0              28.35 | 0.0  0.0  0.0       0.0 |  1.1      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |  0.577354        0.0  0.0          0.0 | 0.189293       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.789293     1.068  1.0048                      0.8 | 0.043562  0.048347  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.533792  -0.048347   0.6952                0.0 | 0.106758       0.0  0.13904       0.0 |       0.233712 |        0.423641 | 0.0   0.632  0.0               0.9 | 0.0  0.0  0.0       0.0 | 77.729182  76.600029  0.0           0.0 |   1.91344 |
    | 2017-03-06 |           0.3 |         3.7 | 0.774863  0.766481  0.0                   0.0 | 0.3   0.801  0.0                      0.6 |  0.0      0.0  0.0          0.0 | 16.65  16.65  0.0              16.65 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.284332       0.0  0.0                     0.0 | 0.0  0.0  0.7536              0.0 | 0.584332     0.801  0.7536                      0.6 | 0.041423  0.048048  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.041423  -0.048048  -0.4536                0.0 |      0.0       0.0      0.0       0.0 |      -0.135768 |        0.322237 | 0.0   0.131  0.0               0.6 | 0.0  0.0  0.0       0.0 | 77.486273  76.648077  0.0           0.0 |  1.455435 |
    | 2017-03-07 |           0.0 |         1.6 | 0.768662  0.759374  0.0                   0.0 | 0.0   0.131  0.0                      0.6 |  0.0      0.0  0.0          0.0 |   7.2    7.2  0.0                7.2 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.662852  0.758479  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.662852  0.889479  0.8792                      0.6 | 0.042793  0.047753  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.042793  -0.047753  -0.8792                0.0 |      0.0       0.0      0.0       0.0 |      -0.242436 |        0.219879 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 76.866214  75.937351  0.0           0.0 |  0.993119 |
    | 2017-03-08 |           1.8 |         4.0 | 0.773982  0.759896  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.2      0.0  0.0          0.0 |  18.0   18.0  0.0               18.0 | 0.0  0.0  0.0       0.0 |  1.2      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |  0.619953        0.0  0.0          0.0 | 0.094475       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.694475    0.9345  0.8792                      0.7 | 0.046427  0.052247  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.573526  -0.052247   0.9208                0.0 | 0.114705       0.0  0.18416       0.0 |       0.285803 |        0.231829 | 0.0  0.8655  0.0               1.1 | 0.0  0.0  0.0       0.0 | 77.398213  75.989598  0.0           0.0 |  1.047093 |
    | 2017-03-09 |           8.9 |         5.6 | 0.813786   0.79909  0.0                   0.0 | 0.5  0.6675  0.0                      0.5 |  8.3   7.7655  0.0          8.0 |  25.2   25.2  0.0               25.2 | 0.0  0.0  0.0       0.0 |  8.3   7.7655  0.0      8.0 | 0.0  0.0  0.0            8.0 |  4.362895   3.898012  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0   0.628              0.0 |      0.5    0.6675   0.628                      0.5 | 0.043297  0.051907  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  4.319598   3.846104    8.272                0.0 |  0.86392  0.769221   1.6544       0.0 |        3.28754 |        0.785736 | 0.1  1.3325  0.0               1.5 | 0.0  0.0  0.0       0.0 | 81.378615  79.908994  0.0           0.0 |  3.548898 |
    | 2017-03-10 |           0.0 |         5.8 | 0.807311  0.799394  0.0                   0.0 | 0.1   1.068  0.0                      0.8 |  0.0      0.0  0.0          0.0 |  26.1   26.1  0.0               26.1 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.671692       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.771692     1.068  1.0048                      0.8 | 0.024214  0.030411  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.024214  -0.030411  -1.0048                0.0 |      0.0       0.0      0.0       0.0 |      -0.264856 |        0.595296 | 0.0  0.2645  0.0               0.7 | 0.0  0.0  0.0       0.0 | 80.731137  79.939405  0.0           0.0 |  2.688746 |
    | 2017-03-11 |           0.0 |         5.7 | 0.798004  0.789472  0.0                   0.0 | 0.0  0.2645  0.0                      0.7 |  0.0      0.0  0.0          0.0 | 25.65  25.65  0.0              25.65 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.957582  1.022433  0.0                     0.0 | 0.0  0.0   1.256              0.0 | 0.957582  1.286933   1.256                      0.7 | 0.026829  0.030274  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.026829  -0.030274   -1.256                0.0 |      0.0       0.0      0.0       0.0 |      -0.328276 |        0.427881 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 79.800384  78.947245  0.0           0.0 |   1.93259 |
    | 2017-03-12 |           0.0 |         4.6 | 0.786857  0.774573  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  20.7   20.7  0.0               20.7 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.145587  1.524918  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 1.145587  1.524918  1.5072                      0.0 | 0.030907  0.034991  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.030907  -0.034991  -1.5072                0.0 |      0.0       0.0      0.0       0.0 |      -0.393275 |         0.27903 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 78.685705  77.457319  0.0           0.0 |  1.260285 |
    | 2017-03-13 |           0.9 |         4.2 | 0.785723  0.772148  0.0                   0.0 | 0.6     0.9  0.0                      0.9 |  0.3      0.0  0.0          0.0 |  18.9   18.9  0.0               18.9 | 0.0  0.0  0.0       0.0 |  0.3      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |  0.164363        0.0  0.0          0.0 | 0.285305  0.285469  0.0                     0.0 | 0.0  0.0  1.1304              0.0 | 0.885305  1.185469  1.1304                      0.9 | 0.036312  0.042958  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.128051  -0.042958  -0.2304                0.0 |  0.02561       0.0      0.0       0.0 |      -0.042729 |        0.220705 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 78.572349  77.214808  0.0           0.0 |   0.99685 |
    | 2017-03-14 |           0.1 |         7.4 | 0.778487  0.762172  0.0                   0.0 | 0.1     0.1  0.0                      0.1 |  0.0      0.0  0.0          0.0 |  33.3   33.3  0.0               33.3 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |  0.76051  1.041999  0.0                     0.0 | 0.0  0.0  1.1304              0.0 |  0.86051  1.141999  1.1304                      0.1 | 0.036894   0.04436  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.036894   -0.04436  -1.0304                0.0 |      0.0       0.0      0.0       0.0 |      -0.277913 |        0.130321 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 77.848732  76.217169  0.0           0.0 |  0.588615 |
    | 2017-03-15 |           0.0 |         6.3 | 0.767517  0.747579  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 28.35  28.35  0.0              28.35 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.137822  1.509741  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 1.137822  1.509741  1.5072                      0.0 | 0.040759  0.050445  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.040759  -0.050445  -1.5072                0.0 |      0.0       0.0      0.0       0.0 |      -0.399601 |        0.034262 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  76.75167  74.757873  0.0           0.0 |  0.154752 |
    | 2017-03-16 |           0.0 |         8.7 | 0.754767   0.73067  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 39.15  39.15  0.0              39.15 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.322076  1.751156  0.0                     0.0 | 0.0  0.0  1.7584              0.0 | 1.322076  1.751156  1.7584                      0.0 |  0.04712  0.060313  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.04712  -0.060313  -1.7584                0.0 |      0.0       0.0      0.0       0.0 |      -0.466458 |       -0.056503 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 75.476714  73.067029  0.0           0.0 | -0.255204 |
    | 2017-03-17 |           3.9 |         6.4 | 0.767336   0.74176  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  3.3      1.9  0.0          1.9 |  28.8   28.8  0.0               28.8 | 0.0  0.0  0.0       0.0 |  3.3      1.9  0.0      1.9 | 0.0  0.0  0.0            1.9 |  1.628564   0.864316  0.0          0.0 | 0.469837       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 1.069837    1.4685  1.3816                      1.1 | 0.055305  0.073263  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  1.573259   0.791053   2.5184                0.0 | 0.314652  0.158211  0.50368       0.0 |       0.976542 |        0.130757 | 0.0  0.5315  0.0               0.9 | 0.0  0.0  0.0       0.0 | 76.733618  74.175976  0.0           0.0 |  0.590582 |
    | 2017-03-18 |           8.7 |         5.2 | 0.802421  0.780557  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  8.1   7.2315  0.0          7.6 |  23.4   23.4  0.0               23.4 | 0.0  0.0  0.0       0.0 |  8.1   7.2315  0.0      7.6 | 0.0  0.0  0.0            7.6 |  4.166588   3.416397  0.0          0.0 | 0.472138       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 1.072138    1.4685  1.3816                      1.1 |  0.04723  0.064581  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  4.119358   3.351816   7.3184                0.0 | 0.823872  0.670363  1.46368       0.0 |       2.957915 |        0.643233 | 0.0  0.5315  0.0               0.9 | 0.0  0.0  0.0       0.0 | 80.242122   78.05566  0.0           0.0 |  2.905264 |
    | 2017-03-19 |          26.4 |         5.1 | 0.912235  0.896403  0.0                   0.0 | 0.5  0.6675  0.0                      0.5 | 25.8  24.9315  0.0         25.3 | 22.95  22.95  0.0              22.95 | 0.0  0.0  0.0       0.0 | 25.8  24.9315  0.0     25.3 | 0.0  0.0  0.0           25.3 | 14.847535  13.386472  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0   0.628              0.0 |      0.5    0.6675   0.628                      0.5 | 0.028924  0.039628  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 14.818611  13.346844   25.772                0.0 | 2.963722  2.669369   5.1544       0.0 |      10.787491 |        2.482075 | 0.1  1.3325  0.0               1.5 | 0.0  0.0  0.0       0.0 | 91.223511  89.640316  0.0           0.0 | 11.210679 |
    | 2017-03-20 |          11.5 |         8.7 | 0.934912   0.92245  0.0                   0.0 | 0.6   0.801  0.0                      0.6 | 11.0  10.8325  0.0         11.0 | 39.15  39.15  0.0              39.15 | 0.0  0.0  0.0       0.0 | 11.0  10.8325  0.0     11.0 | 0.0  0.0  0.0           11.0 |  8.734827   8.231984  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.7536              0.0 |      0.6     0.801  0.7536                      0.6 | 0.002535  0.004169  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  8.732292   8.227815  10.7464                0.0 | 1.746458  1.645563  2.14928       0.0 |       5.541301 |        3.036619 | 0.0   1.199  0.0               1.4 | 0.0  0.0  0.0       0.0 |  93.49122  92.245002  0.0           0.0 | 13.715362 |
    | 2017-03-21 |           0.9 |         6.2 | 0.926488  0.922624  0.0                   0.0 | 0.6     2.0  0.0                      1.5 |  0.3    0.099  0.0          0.3 |  27.9   27.9  0.0               27.9 | 0.0  0.0  0.0       0.0 |  0.3    0.099  0.0      0.3 | 0.0  0.0  0.0            0.3 |  0.253368   0.080842  0.0          0.0 | 0.890068  0.002466  0.0                     0.0 | 0.0  0.0   1.884              0.0 | 1.490068  2.002466   1.884                      1.5 | 0.001034  0.001749  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.252334   0.079093   -0.984                0.0 | 0.050467  0.015819      0.0       0.0 |      -0.179715 |        2.453597 | 0.0     0.0  0.0               0.5 | 0.0  0.0  0.0       0.0 | 92.648818  92.262442  0.0           0.0 |  11.08205 |
    | 2017-03-22 |           0.0 |         5.9 | 0.906757  0.896301  0.0                   0.0 | 0.0     0.0  0.0                      0.5 |  0.0      0.0  0.0          0.0 | 26.55  26.55  0.0              26.55 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |  1.97465  2.634102  0.0                     0.0 | 0.0  0.0   2.512              0.0 |  1.97465  2.634102   2.512                      0.5 |  0.00149  0.001737  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.00149  -0.001737   -2.512                0.0 |      0.0       0.0      0.0       0.0 |      -0.628807 |        1.894852 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 90.675658  89.630077  0.0           0.0 |  8.558392 |
    | 2017-03-23 |           0.0 |         5.2 | 0.891055  0.875388  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  23.4   23.4  0.0               23.4 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.573237  2.095416  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.573237  2.095416  2.0096                      0.0 |  0.00304  0.004182  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.00304  -0.004182  -2.0096                0.0 |      0.0       0.0      0.0       0.0 |      -0.504205 |        1.459976 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 89.105461  87.538843  0.0           0.0 |   6.59421 |
    | 2017-03-24 |           0.0 |         5.2 | 0.875426   0.85461  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  23.4   23.4  0.0               23.4 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.567721  2.085112  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.567721  2.085112  2.0096                      0.0 | 0.004849  0.007256  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.004849  -0.007256  -2.0096                0.0 |      0.0       0.0      0.0       0.0 |      -0.505426 |        1.103709 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  87.54259  85.460987  0.0           0.0 |  4.985075 |
    | 2017-03-25 |           0.0 |         5.9 | 0.863784   0.83917  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 26.55  26.55  0.0              26.55 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.171422  1.555529  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 1.171422  1.555529  1.5072                      0.0 |  0.00725  0.011525  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.00725  -0.011525  -1.5072                0.0 |      0.0       0.0      0.0       0.0 |      -0.381494 |        0.834488 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 86.378418  83.916983  0.0           0.0 |  3.769093 |
    | 2017-03-26 |           0.0 |         6.7 | 0.851226  0.822546  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 30.15  30.15  0.0              30.15 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |  1.26533  1.678004  0.0                     0.0 | 0.0  0.0  1.6328              0.0 |  1.26533  1.678004  1.6328                      0.0 | 0.009478    0.0156  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.009478    -0.0156  -1.6328                0.0 |      0.0       0.0      0.0       0.0 |       -0.41447 |         0.60809 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 85.122566  82.254579  0.0           0.0 |  2.746534 |
    | 2017-03-27 |           0.0 |         7.0 | 0.835827  0.802204  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  31.5   31.5  0.0               31.5 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.552174  2.055175  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.552174  2.055175  2.0096                      0.0 | 0.012349  0.020955  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.012349  -0.020955  -2.0096                0.0 |      0.0       0.0      0.0       0.0 |      -0.510726 |        0.405283 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |  83.58274  80.220359  0.0           0.0 |  1.830525 |
    | 2017-03-28 |           0.0 |         8.3 |  0.81764  0.778245  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 37.35  37.35  0.0              37.35 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.835299  2.424864  0.0                     0.0 | 0.0  0.0  2.3864              0.0 | 1.835299  2.424864  2.3864                      0.0 | 0.016593  0.029019  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.016593  -0.029019  -2.3864                0.0 |      0.0       0.0      0.0       0.0 |      -0.608003 |        0.221606 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 81.764034  77.824514  0.0           0.0 |  1.000916 |
    | 2017-03-29 |           1.5 |         9.0 | 0.819517  0.778654  0.0                   0.0 | 0.6   1.068  0.0                      0.8 |  0.9      0.0  0.0          0.0 |  40.5   40.5  0.0               40.5 | 0.0  0.0  0.0       0.0 |  0.9      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |   0.54295        0.0  0.0          0.0 | 0.192143       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.792143     1.068  1.0048                      0.8 | 0.022741  0.040893  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.520209  -0.040893   0.4952                0.0 | 0.104042       0.0  0.09904       0.0 |       0.192858 |        0.216395 | 0.0   0.432  0.0               0.7 | 0.0  0.0  0.0       0.0 | 81.951683  77.865407  0.0           0.0 |   0.97738 |
    | 2017-03-30 |           0.3 |        12.4 | 0.808202  0.767013  0.0                   0.0 | 0.3   0.732  0.0                      1.0 |  0.0      0.0  0.0          0.0 |  55.8   55.8  0.0               55.8 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.153525  1.204742  0.0                     0.0 | 0.0  0.0   1.884              0.0 | 1.453525  1.936742   1.884                      1.0 | 0.022047  0.040667  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.022047  -0.040667   -1.584                0.0 |      0.0       0.0      0.0       0.0 |      -0.411678 |        0.102544 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 80.820205  76.701333  0.0           0.0 |  0.463157 |
    | 2017-03-31 |           0.2 |        15.0 |  0.78452  0.735344  0.0                   0.0 | 0.2     0.2  0.0                      0.2 |  0.0      0.0  0.0          0.0 |  67.5   67.5  0.0               67.5 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 2.394642  3.214392  0.0                     0.0 | 0.0  0.0  3.3912              0.0 | 2.594642  3.414392  3.3912                      0.2 | 0.026458  0.047427  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.026458  -0.047427  -3.1912                0.0 |      0.0       0.0      0.0       0.0 |      -0.816271 |       -0.064009 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 78.452021  73.534368  0.0           0.0 | -0.289106 |
    | 2017-04-01 |           4.5 |        11.8 | 0.795123  0.749459  0.0                   0.0 | 0.8     2.0  0.0                      1.5 |  3.7      2.5  0.0          2.5 |  53.1   53.1  0.0               53.1 | 0.0  0.0  0.0       0.0 |  3.7      2.5  0.0      2.5 | 0.0  0.0  0.0            2.5 |  2.012061   1.155605  0.0          0.0 | 0.665164   0.00233  0.0                     0.0 | 0.0  0.0  1.9245              0.0 | 1.465164   2.00233  1.9245                      1.5 | 0.037519  0.069515  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  1.974542    1.08609   2.5755                0.0 | 0.394908  0.217218   0.5151       0.0 |       1.127226 |        0.151926 | 0.0     0.0  0.0               0.5 | 0.0  0.0  0.0       0.0 | 79.512315  74.945948  0.0           0.0 |  0.686195 |
    | 2017-04-02 |           0.0 |         9.4 | 0.780186  0.730021  0.0                   0.0 | 0.0     0.0  0.0                      0.5 |  0.0      0.0  0.0          0.0 |  42.3   42.3  0.0               42.3 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.525967   2.00286  0.0                     0.0 | 0.0  0.0  2.0528              0.0 | 1.525967   2.00286  2.0528                      0.5 | 0.032249  0.058975  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.032249  -0.058975  -2.0528                0.0 |      0.0       0.0      0.0       0.0 |      -0.536006 |        0.027225 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 78.018597  73.002063  0.0           0.0 |  0.122965 |
    | 2017-04-03 |           0.0 |         8.1 | 0.761609  0.705928  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 36.45  36.45  0.0              36.45 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.897533  2.483057  0.0                     0.0 | 0.0  0.0   2.566              0.0 | 1.897533  2.483057   2.566                      0.0 | 0.039829  0.073794  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.039829  -0.073794   -2.566                0.0 |      0.0       0.0      0.0       0.0 |      -0.669906 |       -0.099144 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 76.160892    70.5928  0.0           0.0 | -0.447797 |
    | 2017-04-04 |           0.0 |         7.9 | 0.742331  0.681098  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 35.55  35.55  0.0              35.55 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.978631  2.578347  0.0                     0.0 | 0.0  0.0  2.6943              0.0 | 1.978631  2.578347  2.6943                      0.0 | 0.050804  0.095366  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.050804  -0.095366  -2.6943                0.0 |      0.0       0.0      0.0       0.0 |      -0.710118 |       -0.209894 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 74.233066  68.109819  0.0           0.0 | -0.948021 |
    | 2017-04-05 |           0.4 |         7.5 | 0.730818  0.665337  0.0                   0.0 | 0.4     0.4  0.0                      0.4 |  0.0      0.0  0.0          0.0 | 33.75  33.75  0.0              33.75 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.215386   1.69773  0.0                     0.0 | 0.0  0.0  2.1811              0.0 | 1.615386   2.09773  2.1811                      0.4 | 0.064153  0.121619  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.064153  -0.121619  -1.7811                0.0 |      0.0       0.0      0.0       0.0 |      -0.491718 |        -0.26098 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 73.081834  66.533709  0.0           0.0 | -1.178758 |
    | 2017-04-06 |           0.0 |         7.2 | 0.715734  0.646312  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  32.4   32.4  0.0               32.4 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.581527  2.043089  0.0                     0.0 | 0.0  0.0  2.1811              0.0 | 1.581527  2.043089  2.1811                      0.0 | 0.073142  0.140558  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.073142  -0.140558  -2.1811                0.0 |      0.0       0.0      0.0       0.0 |        -0.5987 |       -0.322199 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 71.573449  64.631177  0.0           0.0 |  -1.45526 |
    | 2017-04-07 |           0.0 |         8.1 | 0.709204  0.638464  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 36.45  36.45  0.0              36.45 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.739189  0.950662  0.0                     0.0 | 0.0  0.0  1.0264              0.0 | 0.739189  0.950662  1.0264                      0.0 |  0.08614  0.165918  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.08614  -0.165918  -1.0264                0.0 |      0.0       0.0      0.0       0.0 |      -0.319614 |        -0.32173 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 |   70.9204  63.846432  0.0           0.0 | -1.453144 |
    | 2017-04-08 |           0.0 |         8.6 | 0.698151  0.624864  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  38.7   38.7  0.0               38.7 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.197483  1.537253  0.0                     0.0 | 0.0  0.0  1.6679              0.0 | 1.197483  1.537253  1.6679                      0.0 | 0.092214  0.177209  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.092214  -0.177209  -1.6679                0.0 |      0.0       0.0      0.0       0.0 |      -0.484331 |       -0.351205 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 69.815131  62.486387  0.0           0.0 |  -1.58627 |
    | 2017-04-09 |           0.0 |        10.5 | 0.676278  0.597543  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 47.25  47.25  0.0              47.25 | 0.0  0.0  0.0       0.0 |  0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 2.290433  2.930078  0.0                     0.0 | 0.0  0.0  3.2075              0.0 | 2.290433  2.930078  3.2075                      0.0 | 0.103133  0.197969  0.0            0.0 |       0.0  10.245721  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.103133  -0.197969  -3.2075                0.0 |      0.0       0.0      0.0       0.0 |      -0.877151 |       -0.446542 | 0.0     0.0  0.0               0.0 | 0.0  0.0  0.0       0.0 | 67.627832  59.754279  0.0           0.0 | -2.016878 |

>>> round_(model.check_waterbalance(conditions))
0.0

.. _whmod_rural_snow:

snow
____

We reduce the complete air temperature time series by 8 °C to demonstrate the
functioning of the snow routine:

>>> inputs.temperature.series -= 8.0

As the emergence of iced layers is not considered, no snowpack builds up on the water
zone.  For all other zones, snow accumulation and melting follow a simple degree day
method that does not differentiate between liquid and frozen water stored in the snow
layer:

.. integration-test::

    >>> conditions = test("whmod_rural_snow", get_conditions="2017-02-10")
    |       date | precipitation | temperature |                          relativesoilmoisture |                   interceptionevaporation |                     throughfall |                    potentialsnowmelt |                     snowmelt |                      ponding |                surfacerunoff |                            percolation |                          soilevapotranspiration |                   lakeevaporation |                             totalevapotranspiration |                          capillaryrise |                            requiredirrigation |                externalirrigation |                                potentialrecharge |                              baseflow | actualrecharge | delayedrecharge |                   interceptedwater |                       snowpack |                            soilmoisture | deepwater |
    ------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------------
    | 2017-02-10 |           0.0 |       -10.8 |   0.0375    0.0375  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.7302              0.0 |      0.0       0.0  0.7302                      0.0 |     3.75      3.75  0.0            0.0 |     66.25      66.25  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |     -3.75      -3.75  -0.7302                0.0 |      0.0       0.0      0.0       0.0 |       -2.05755 |       -0.372971 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 |      3.75       3.75  0.0           0.0 | -1.684579 |
    | 2017-02-11 |           0.4 |        -9.5 | 0.070632  0.070427  0.0                   0.0 | 0.4     0.4  0.0                      0.4 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.030545  0.051011  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.430545  0.451011  0.9736                      0.4 | 3.343748  3.343748  0.0            0.0 | 62.936798  62.957263  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -3.343748  -3.343748  -0.5736                0.0 |      0.0       0.0      0.0       0.0 |      -1.815274 |       -0.634416 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 |  7.063202   7.042737  0.0           0.0 | -2.865437 |
    | 2017-02-12 |           0.0 |        -8.9 | 0.099731  0.099214  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.100328  0.133554  0.0                     0.0 | 0.0  0.0  0.8519              0.0 | 0.100328  0.133554  0.8519                      0.0 | 3.010193  3.012182  0.0            0.0 | 60.026932  60.078634  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -3.010193  -3.012182  -0.8519                0.0 |      0.0       0.0      0.0       0.0 |      -1.718569 |       -0.830939 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 |  9.973068   9.921366  0.0           0.0 | -3.753067 |
    | 2017-02-13 |           0.0 |        -9.6 | 0.126288  0.125554  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.080508  0.106933  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.080508  0.106933  0.4868                      0.0 | 2.736205  2.740922  0.0            0.0 | 57.371235  57.444646  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.736205  -2.740922  -0.4868                0.0 |      0.0       0.0      0.0       0.0 |      -1.490982 |       -0.950585 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 | 12.628765  12.555354  0.0           0.0 | -4.293464 |
    | 2017-02-14 |           0.0 |        -9.3 | 0.150286  0.149284  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.101248  0.134409  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.101248  0.134409  0.4868                      0.0 | 2.501132  2.507442  0.0            0.0 | 54.971351  55.071613  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.501132  -2.507442  -0.4868                0.0 |      0.0       0.0      0.0       0.0 |      -1.373844 |       -1.027309 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 | 15.028649  14.928387  0.0           0.0 | -4.639999 |
    | 2017-02-15 |           0.0 |        -6.3 | 0.172097  0.170786  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.119564  0.158608  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.119564  0.158608  0.4868                      0.0 | 2.300641  2.308794  0.0            0.0 | 52.790274  52.921427  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.300641  -2.308794  -0.4868                0.0 |      0.0       0.0      0.0       0.0 |      -1.274059 |       -1.072037 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 | 17.209726  17.078573  0.0           0.0 | -4.842021 |
    | 2017-02-16 |           0.2 |        -3.6 | 0.192698  0.191041  0.0                   0.0 | 0.2     0.2  0.0                      0.2 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.067899  0.112586  0.0                     0.0 | 0.0  0.0  0.4868              0.0 | 0.267899  0.312586  0.4868                      0.2 | 2.127988  2.138117  0.0            0.0 | 50.730184  50.895895  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -2.127988  -2.138117  -0.2868                0.0 |      0.0       0.0      0.0       0.0 |      -1.138226 |       -1.084035 | 0.0     0.0  0.0               0.0 |    0.0      0.0  0.0       0.0 | 19.269816  19.104105  0.0           0.0 | -4.896213 |
    | 2017-02-17 |           4.5 |        -3.5 | 0.212429  0.210893  0.0                   0.0 | 0.3  0.4005  0.0                      0.3 |  4.1      2.5  0.0          2.5 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 |      0.3    0.4005  0.3651                      0.3 | 1.973055   1.98523  0.0            0.0 |  48.75713  48.910666  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.973055   -1.98523   4.1349                0.0 |      0.0       0.0  0.82698       0.0 |      -0.162591 |       -0.917006 | 0.1  1.5995  0.0               1.7 |    4.1      2.5  0.0       2.5 |  21.24287  21.089334  0.0           0.0 | -4.141798 |
    | 2017-02-18 |           0.0 |        -4.6 | 0.229925   0.22932  0.0                   0.0 | 0.1  0.4005  0.0                      0.3 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.082315       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 | 0.182315    0.4005  0.3651                      0.3 | 1.831896  1.842631  0.0            0.0 | 47.007549  47.068035  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.831896  -1.842631  -0.3651                0.0 |      0.0       0.0      0.0       0.0 |      -1.009907 |       -0.933846 | 0.0   1.199  0.0               1.4 |    4.1      2.5  0.0       2.5 | 22.992451  22.931965  0.0           0.0 | -4.217859 |
    | 2017-02-19 |           3.2 |        -3.2 |  0.24705  0.246485  0.0                   0.0 | 0.4   0.534  0.0                      0.4 |  2.8    2.399  0.0          2.6 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.4868              0.0 |      0.4     0.534  0.4868                      0.4 | 1.712502  1.716541  0.0            0.0 | 45.295046  45.351494  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.712502  -1.716541   2.7132                0.0 |      0.0       0.0  0.54264       0.0 |      -0.314621 |       -0.821599 | 0.0   1.466  0.0               1.6 |    6.9    4.899  0.0       5.1 | 24.704954  24.648506  0.0           0.0 | -3.710881 |
    | 2017-02-20 |           4.6 |        -1.3 | 0.263057  0.262529  0.0                   0.0 | 0.3  0.4005  0.0                      0.3 |  4.2    4.066  0.0          4.2 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.3651              0.0 |      0.3    0.4005  0.3651                      0.3 | 1.600776  1.604379  0.0            0.0 | 43.694271  43.747115  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.600776  -1.604379   4.2349                0.0 |      0.0       0.0  0.84698       0.0 |       0.045691 |       -0.664386 | 0.1  1.5995  0.0               1.7 |   11.1    8.965  0.0       9.3 | 26.305729  26.252885  0.0           0.0 | -3.000803 |
    | 2017-02-21 |           2.3 |        -2.2 | 0.277074  0.277569  0.0                   0.0 | 0.4   0.801  0.0                      0.6 |  2.0   1.8995  0.0          2.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.099174       0.0  0.0                     0.0 | 0.0  0.0  0.7302              0.0 | 0.499174     0.801  0.7302                      0.6 | 1.500833  1.504064  0.0            0.0 | 42.292611  42.243051  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.500833  -1.504064   1.5698                0.0 |      0.0       0.0  0.31396       0.0 |      -0.437264 |       -0.623216 | 0.0   1.199  0.0               1.4 |   13.1  10.8645  0.0      11.3 | 27.707389  27.756949  0.0           0.0 | -2.814852 |
    | 2017-02-22 |          18.0 |        -1.5 |  0.28917  0.291709  0.0                   0.0 | 0.4   1.068  0.0                      0.8 | 17.6   17.199  0.0         17.4 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.207154       0.0  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.607154     1.068  0.9736                      0.8 | 1.416815  1.413903  0.0            0.0 | 41.082951  40.829148  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.416815  -1.413903  17.0264                0.0 |      0.0       0.0  3.40528       0.0 |       2.697601 |       -0.021254 | 0.0   0.932  0.0               1.2 |   30.7  28.0635  0.0      28.7 | 28.917049  29.170852  0.0           0.0 | -0.095997 |
    | 2017-02-23 |          19.2 |        -3.0 | 0.302103  0.305034  0.0                   0.0 | 0.4  0.6675  0.0                      0.5 | 18.8   18.132  0.0         18.4 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.053641       0.0  0.0                     0.0 | 0.0  0.0  0.6085              0.0 | 0.453641    0.6675  0.6085                      0.5 | 1.346876    1.3325  0.0            0.0 | 39.789715  39.496648  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.346876    -1.3325  18.5915                0.0 |      0.0       0.0   3.7183       0.0 |       3.048456 |         0.53519 | 0.0  1.3325  0.0               1.5 |   49.5  46.1955  0.0      47.1 | 30.210285  30.503352  0.0           0.0 |  2.417269 |
    | 2017-02-24 |           0.4 |        -5.0 | 0.312627  0.317621  0.0                   0.0 | 0.4   1.068  0.0                      0.8 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.222284       0.0  0.0                     0.0 | 0.0  0.0  0.9736              0.0 | 0.622284     1.068  0.9736                      0.8 | 1.274693  1.258702  0.0            0.0 | 38.737306  38.237946  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.274693  -1.258702  -0.5736                0.0 |      0.0       0.0      0.0       0.0 |      -0.776749 |        0.297376 | 0.0  0.6645  0.0               1.1 |   49.5  46.1955  0.0      47.1 | 31.262694  31.762054  0.0           0.0 |  1.343144 |
    | 2017-02-25 |           8.3 |        -4.9 | 0.324235  0.329536  0.0                   0.0 | 0.4  0.6675  0.0                      0.5 |  7.9   6.9645  0.0          7.4 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.057104       0.0  0.0                     0.0 | 0.0  0.0  0.6085              0.0 | 0.457104    0.6675  0.6085                      0.5 | 1.217892  1.191541  0.0            0.0 | 37.576518  37.046405  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.217892  -1.191541   7.6915                0.0 |      0.0       0.0   1.5383       0.0 |       0.935942 |        0.413128 | 0.0  1.3325  0.0               1.5 |   57.4    53.16  0.0      54.5 | 32.423482  32.953595  0.0           0.0 |  1.865958 |
    | 2017-02-26 |           5.3 |        -0.9 | 0.335807  0.340838  0.0                   0.0 | 0.4   0.534  0.0                      0.4 |  4.9   4.6325  0.0          4.8 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.4868              0.0 |      0.4     0.534  0.4868                      0.4 | 1.157227  1.130206  0.0            0.0 | 36.419291  35.916199  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -1.157227  -1.130206   4.8132                0.0 |      0.0       0.0  0.96264       0.0 |       0.390782 |        0.409077 | 0.0   1.466  0.0               1.6 |   62.3  57.7925  0.0      59.3 | 33.580709  34.083801  0.0           0.0 |  1.847662 |
    | 2017-02-27 |           0.7 |         1.4 | 0.403097    0.4119  0.0                   0.0 | 0.4  1.7355  0.0                      1.3 |  0.3    0.166  0.0          0.3 |   6.3    6.3  0.0                6.3 |   6.3     6.3  0.0       6.3 |   6.6    6.466  0.0      6.6 | 0.0  0.0  0.0            6.6 |  0.426533   0.433766  0.0          0.0 | 0.543217       0.0  0.0                     0.0 | 0.0  0.0  1.5821              0.0 | 0.943217    1.7355  1.5821                      1.3 | 1.098788  1.074009  0.0            0.0 | 29.690253  28.809956  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.672255  -0.640242  -0.8821                0.0 |      0.0       0.0      0.0       0.0 |      -0.548649 |        0.235471 | 0.0  0.2645  0.0               0.7 |   56.0  51.4925  0.0      53.0 | 40.309747  41.190044  0.0           0.0 |  1.063542 |
    | 2017-02-28 |           2.7 |        -3.4 |  0.40763  0.419528  0.0                   0.0 | 0.4  1.2015  0.0                      0.9 |  2.3   0.9645  0.0          1.4 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.344257       0.0  0.0                     0.0 | 0.0  0.0  1.0953              0.0 | 0.744257    1.2015  1.0953                      0.9 |  0.79752  0.762753  0.0            0.0 |  29.23699  28.047204  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.79752  -0.762753   1.6047                0.0 |      0.0       0.0  0.32094       0.0 |      -0.069128 |        0.180257 | 0.0  0.7985  0.0               1.1 |   58.3   52.457  0.0      54.4 |  40.76301  41.952796  0.0           0.0 |  0.814157 |
    | 2017-03-01 |           1.6 |        -4.3 | 0.414731  0.426863  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.0   0.3985  0.0          0.7 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.069373       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.669373    0.9345  0.8792                      0.7 | 0.779489  0.733458  0.0            0.0 | 28.526874  27.313746  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.779489  -0.733458   0.7208                0.0 |      0.0       0.0  0.14416       0.0 |      -0.234077 |        0.105151 | 0.0  1.0655  0.0               1.3 |   59.3  52.8555  0.0      55.1 | 41.473126  42.686254  0.0           0.0 |  0.474929 |
    | 2017-03-02 |           2.5 |        -3.3 | 0.421547  0.433923  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.9   1.5655  0.0          1.8 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.070178       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.670178    0.9345  0.8792                      0.7 | 0.751791  0.706005  0.0            0.0 | 27.845261  26.607741  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.751791  -0.706005   1.6208                0.0 |      0.0       0.0  0.32416       0.0 |      -0.040289 |        0.078787 | 0.0  1.0655  0.0               1.3 |   61.2   54.421  0.0      56.9 | 42.154739  43.392259  0.0           0.0 |  0.355853 |
    | 2017-03-03 |           0.6 |        -2.1 | 0.425259  0.440725  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.354684       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 0.954684    1.4685  1.3816                      1.1 |  0.72583  0.680235  0.0            0.0 | 27.474116  25.927506  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.72583  -0.680235  -0.7816                0.0 |      0.0       0.0      0.0       0.0 |      -0.546916 |       -0.034634 | 0.0   0.197  0.0               0.8 |   61.2   54.421  0.0      56.9 | 42.525884  44.072494  0.0           0.0 | -0.156429 |
    | 2017-03-04 |           0.2 |        -0.3 | 0.426671  0.440438  0.0                   0.0 | 0.2   0.397  0.0                      1.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.570753  0.684711  0.0                     0.0 | 0.0  0.0   1.256              0.0 | 0.770753  1.081711   1.256                      1.0 | 0.711948  0.656006  0.0            0.0 | 27.332921  25.956212  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.711948  -0.656006   -1.056                0.0 |      0.0       0.0      0.0       0.0 |      -0.605988 |       -0.138203 | 0.0     0.0  0.0               0.0 |   61.2   54.421  0.0      56.9 | 42.667079  44.043788  0.0           0.0 | -0.624214 |
    | 2017-03-05 |           1.7 |        -1.7 | 0.432308  0.447008  0.0                   0.0 | 0.6   1.068  0.0                      0.8 |  1.1      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.142996       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.742996     1.068  1.0048                      0.8 | 0.706714  0.657016  0.0            0.0 | 26.769203  25.299195  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.706714  -0.657016   0.6952                0.0 |      0.0       0.0  0.13904       0.0 |      -0.201892 |       -0.149748 | 0.0   0.632  0.0               0.9 |   62.3   54.421  0.0      56.9 | 43.230797  44.700805  0.0           0.0 | -0.676359 |
    | 2017-03-06 |           0.3 |        -4.3 | 0.437006  0.453349  0.0                   0.0 | 0.3   0.801  0.0                      0.6 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.216318       0.0  0.0                     0.0 | 0.0  0.0  0.7536              0.0 | 0.516318     0.801  0.7536                      0.6 | 0.686072  0.634144  0.0            0.0 | 26.299449  24.665052  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.686072  -0.634144  -0.4536                0.0 |      0.0       0.0      0.0       0.0 |      -0.443454 |       -0.202988 | 0.0   0.131  0.0               0.6 |   62.3   54.421  0.0      56.9 | 43.700551  45.334948  0.0           0.0 | -0.916825 |
    | 2017-03-07 |           0.0 |        -6.4 | 0.438615  0.453506  0.0                   0.0 | 0.0   0.131  0.0                      0.6 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.508241  0.596954  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.508241  0.727954  0.8792                      0.6 | 0.669181  0.612577  0.0            0.0 | 26.138509  24.649429  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.669181  -0.612577  -0.8792                0.0 |      0.0       0.0      0.0       0.0 |      -0.540239 |       -0.264121 | 0.0     0.0  0.0               0.0 |   62.3   54.421  0.0      56.9 | 43.861491  45.350571  0.0           0.0 | -1.192944 |
    | 2017-03-08 |           1.8 |        -4.0 | 0.444522  0.459626  0.0                   0.0 | 0.6  0.9345  0.0                      0.7 |  1.2      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.072776       0.0  0.0                     0.0 | 0.0  0.0  0.8792              0.0 | 0.672776    0.9345  0.8792                      0.7 | 0.663459  0.612052  0.0            0.0 | 25.547826  24.037377  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.663459  -0.612052   0.9208                0.0 |      0.0       0.0  0.18416       0.0 |      -0.134718 |       -0.240664 | 0.0  0.8655  0.0               1.1 |   63.5   54.421  0.0      56.9 | 44.452174  45.962623  0.0           0.0 | -1.086997 |
    | 2017-03-09 |           8.9 |        -2.4 | 0.450949  0.465543  0.0                   0.0 | 0.5  0.6675  0.0                      0.5 |  8.3   7.7655  0.0          8.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0   0.628              0.0 |      0.5    0.6675   0.628                      0.5 | 0.642736  0.591717  0.0            0.0 |  24.90509   23.44566  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.642736  -0.591717    8.272                0.0 |      0.0       0.0   1.6544       0.0 |       1.345787 |        0.046911 | 0.1  1.3325  0.0               1.5 |   71.8  62.1865  0.0      64.9 |  45.09491   46.55434  0.0           0.0 |  0.211879 |
    | 2017-03-10 |           0.0 |        -2.2 | 0.451972  0.471268  0.0                   0.0 | 0.1   1.068  0.0                      0.8 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.518358       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.618358     1.068  1.0048                      0.8 | 0.620682  0.572491  0.0            0.0 | 24.802766   22.87317  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.620682  -0.572491  -1.0048                0.0 |      0.0       0.0      0.0       0.0 |      -0.549493 |       -0.061199 | 0.0  0.2645  0.0               0.7 |   71.8  62.1865  0.0      64.9 | 45.197234   47.12683  0.0           0.0 | -0.276415 |
    | 2017-03-11 |           0.0 |        -2.3 | 0.450729  0.468669  0.0                   0.0 | 0.0  0.2645  0.0                      0.7 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |  0.74155  0.814195  0.0                     0.0 | 0.0  0.0   1.256              0.0 |  0.74155  1.078695   1.256                      0.7 | 0.617218   0.55429  0.0            0.0 | 24.927098  23.133074  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.617218   -0.55429   -1.256                0.0 |      0.0       0.0      0.0       0.0 |      -0.606877 |       -0.160114 | 0.0     0.0  0.0               0.0 |   71.8  62.1865  0.0      64.9 | 45.072902  46.866926  0.0           0.0 | -0.723179 |
    | 2017-03-12 |           0.0 |        -3.4 |  0.44806   0.46215  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.888345  1.214433  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 0.888345  1.214433  1.5072                      0.0 | 0.621429  0.562505  0.0            0.0 | 25.194015  23.785002  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.621429  -0.562505  -1.5072                0.0 |      0.0       0.0      0.0       0.0 |      -0.672783 |       -0.253045 | 0.0     0.0  0.0               0.0 |   71.8  62.1865  0.0      64.9 | 44.805985  46.214998  0.0           0.0 | -1.142917 |
    | 2017-03-13 |           0.9 |        -3.8 | 0.452152  0.465718  0.0                   0.0 | 0.6     0.9  0.0                      0.9 |  0.3      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.221269  0.226642  0.0                     0.0 | 0.0  0.0  1.1304              0.0 | 0.821269  1.126642  1.1304                      0.9 | 0.630532  0.583465  0.0            0.0 | 24.784751  23.428179  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.630532  -0.583465  -0.2304                0.0 |      0.0       0.0      0.0       0.0 |      -0.361099 |       -0.272632 | 0.0     0.0  0.0               0.0 |   72.1  62.1865  0.0      64.9 | 45.215249  46.571821  0.0           0.0 | -1.231384 |
    | 2017-03-14 |           0.1 |        -0.6 | 0.452385  0.463119  0.0                   0.0 | 0.1     0.1  0.0                      0.1 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.593386  0.831864  0.0                     0.0 | 0.0  0.0  1.1304              0.0 | 0.693386  0.931864  1.1304                      0.1 |  0.61661  0.571929  0.0            0.0 | 24.761528  23.688114  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.61661  -0.571929  -1.0304                0.0 |      0.0       0.0      0.0       0.0 |      -0.554735 |       -0.323768 | 0.0     0.0  0.0               0.0 |   72.1  62.1865  0.0      64.9 | 45.238472  46.311886  0.0           0.0 | -1.462351 |
    | 2017-03-15 |           0.0 |        -1.7 | 0.449639  0.456864  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.890361  1.205774  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 0.890361  1.205774  1.5072                      0.0 | 0.615826  0.580318  0.0            0.0 | 25.036063   24.31357  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.615826  -0.580318  -1.5072                0.0 |      0.0       0.0      0.0       0.0 |      -0.675836 |       -0.387587 | 0.0     0.0  0.0               0.0 |   72.1  62.1865  0.0      64.9 | 44.963937   45.68643  0.0           0.0 | -1.750599 |
    | 2017-03-16 |           0.0 |         0.7 | 0.472806  0.476012  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  3.15   3.15  0.0               3.15 |  3.15    3.15  0.0      3.15 |  3.15     3.15  0.0     3.15 | 0.0  0.0  0.0           3.15 |  0.423588   0.440881  0.0          0.0 | 1.034849  1.395158  0.0                     0.0 | 0.0  0.0  1.7584              0.0 | 1.034849  1.395158  1.7584                      0.0 | 0.625134  0.600837  0.0            0.0 | 22.719366  22.398773  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.201546  -0.159955  -1.7584                0.0 |      0.0       0.0      0.0       0.0 |      -0.529975 |       -0.413398 | 0.0     0.0  0.0               0.0 |  68.95  59.0365  0.0     61.75 | 47.280634  47.601227  0.0           0.0 | -1.867176 |
    | 2017-03-17 |           3.9 |        -1.6 | 0.474491  0.481407  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  3.3      1.9  0.0          1.9 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.381023       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 0.981023    1.4685  1.3816                      1.1 | 0.549467  0.539504  0.0            0.0 | 22.550921  21.859269  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.549467  -0.539504   2.5184                0.0 |      0.0       0.0  0.50368       0.0 |       0.231437 |       -0.296509 | 0.0  0.5315  0.0               0.9 |  72.25  60.9365  0.0     63.65 | 47.449079  48.140731  0.0           0.0 |  -1.33923 |
    | 2017-03-18 |           8.7 |        -2.8 | 0.476115  0.486637  0.0                   0.0 | 0.6  1.4685  0.0                      1.1 |  8.1   7.2315  0.0          7.6 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.381825       0.0  0.0                     0.0 | 0.0  0.0  1.3816              0.0 | 0.981825    1.4685  1.3816                      1.1 | 0.544217  0.523011  0.0            0.0 | 22.388529  21.336258  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.544217  -0.523011   7.3184                0.0 |      0.0       0.0  1.46368       0.0 |       1.196873 |       -0.025805 | 0.0  0.5315  0.0               0.9 |  80.35   68.168  0.0     71.25 | 47.611471  48.663742  0.0           0.0 | -0.116552 |
    | 2017-03-19 |          26.4 |        -2.9 | 0.481507  0.491711  0.0                   0.0 | 0.5  0.6675  0.0                      0.5 | 25.8  24.9315  0.0         25.3 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0   0.628              0.0 |      0.5    0.6675   0.628                      0.5 | 0.539188  0.507346  0.0            0.0 | 21.849341  20.828913  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.539188  -0.507346   25.772                0.0 |      0.0       0.0   5.1544       0.0 |       4.892767 |        0.865781 | 0.1  1.3325  0.0               1.5 | 106.15  93.0995  0.0     96.55 | 48.150659  49.171087  0.0           0.0 |  3.910434 |
    | 2017-03-20 |          11.5 |         0.7 | 0.605637  0.612925  0.0                   0.0 | 0.6   0.801  0.0                      0.6 | 11.0  10.8325  0.0         11.0 |  3.15   3.15  0.0               3.15 |  3.15    3.15  0.0      3.15 | 14.15  13.9825  0.0    14.15 | 0.0  0.0  0.0          14.15 |  2.259628   2.353568  0.0          0.0 |      0.0       0.0  0.0                     0.0 | 0.0  0.0  0.7536              0.0 |      0.6     0.801  0.7536                      0.6 |  0.52271  0.492452  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  1.736918   1.861116  10.7464                0.0 | 0.347384  0.372223  2.14928       0.0 |       2.868887 |        1.228882 | 0.0   1.199  0.0               1.4 |  103.0  89.9495  0.0      93.4 | 60.563741  61.292471  0.0           0.0 |  5.550438 |
    | 2017-03-21 |           0.9 |        -1.8 | 0.600142  0.615078  0.0                   0.0 | 0.6     2.0  0.0                      1.5 |  0.3    0.099  0.0          0.3 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 0.779586  0.002177  0.0                     0.0 | 0.0  0.0   1.884              0.0 | 1.379586  2.002177   1.884                      1.5 | 0.229995  0.217479  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.229995  -0.217479   -0.984                0.0 |      0.0       0.0      0.0       0.0 |      -0.357869 |        0.941253 | 0.0     0.0  0.0               0.5 |  103.3  90.0485  0.0      93.7 |  60.01415  61.507773  0.0           0.0 |  4.251317 |
    | 2017-03-22 |           0.0 |        -2.1 | 0.585285  0.593933  0.0                   0.0 | 0.0     0.0  0.0                      0.5 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.725432  2.328388  0.0                     0.0 | 0.0  0.0   2.512              0.0 | 1.725432  2.328388   2.512                      0.5 | 0.239745   0.21387  0.0            0.0 | 11.471537  10.606744  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.239745   -0.21387   -2.512                0.0 |      0.0       0.0      0.0       0.0 |      -0.741404 |        0.636239 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 58.528463  59.393256  0.0           0.0 |  2.873673 |
    | 2017-03-23 |           0.0 |        -2.8 | 0.574312  0.578102  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.364731  1.834174  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.364731  1.834174  2.0096                      0.0 | 0.267474  0.251088  0.0            0.0 | 12.568794   12.18983  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.267474  -0.251088  -2.0096                0.0 |      0.0       0.0      0.0       0.0 |       -0.63204 |        0.406339 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 57.431206   57.81017  0.0           0.0 |  1.835294 |
    | 2017-03-24 |           0.0 |        -2.8 | 0.563678  0.562803  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.352701  1.811468  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.352701  1.811468  2.0096                      0.0 | 0.289271  0.281614  0.0            0.0 | 13.632224  13.719684  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.289271  -0.281614  -2.0096                0.0 |      0.0       0.0      0.0       0.0 |      -0.645121 |        0.215742 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 56.367776  56.280316  0.0           0.0 |  0.974431 |
    | 2017-03-25 |           0.0 |        -2.1 | 0.556738  0.552524  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.005469  1.341289  0.0                     0.0 | 0.0  0.0  1.5072              0.0 | 1.005469  1.341289  1.5072                      0.0 | 0.311497  0.313374  0.0            0.0 | 14.326197  14.747599  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.311497  -0.313374  -1.5072                0.0 |      0.0       0.0      0.0       0.0 |      -0.533018 |        0.080015 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 55.673803  55.252401  0.0           0.0 |  0.361399 |
    | 2017-03-26 |           0.0 |        -1.3 | 0.549177  0.541485  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.082673  1.439933  0.0                     0.0 | 0.0  0.0  1.6328              0.0 | 1.082673  1.439933  1.6328                      0.0 | 0.326597  0.336001  0.0            0.0 | 15.082272  15.851531  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.326597  -0.336001  -1.6328                0.0 |      0.0       0.0      0.0       0.0 |       -0.57385 |       -0.038511 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 54.917728  54.148469  0.0           0.0 |  -0.17394 |
    | 2017-03-27 |           0.0 |        -1.0 | 0.539378  0.527557  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.323482  1.754272  0.0                     0.0 | 0.0  0.0  2.0096              0.0 | 1.323482  1.754272  2.0096                      0.0 | 0.343596  0.361487  0.0            0.0 | 16.062157  17.244316  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.343596  -0.361487  -2.0096                0.0 |      0.0       0.0      0.0       0.0 |      -0.678671 |       -0.154552 | 0.0     0.0  0.0               0.0 |  103.3  90.0485  0.0      93.7 | 53.937843  52.755684  0.0           0.0 | -0.698059 |
    | 2017-03-28 |           0.0 |         0.3 | 0.538104  0.521748  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  1.35   1.35  0.0               1.35 |  1.35    1.35  0.0      1.35 |  1.35     1.35  0.0     1.35 | 0.0  0.0  0.0           1.35 |  0.286644   0.271135  0.0          0.0 | 1.557337  2.055221  0.0                     0.0 | 0.0  0.0  2.3864              0.0 | 1.557337  2.055221  2.3864                      0.0 | 0.366492  0.395439  0.0            0.0 | 16.189647  17.825232  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.079847  -0.124305  -2.3864                0.0 |      0.0       0.0      0.0       0.0 |      -0.647638 |       -0.243933 | 0.0     0.0  0.0               0.0 | 101.95  88.6985  0.0     92.35 | 53.810353  52.174768  0.0           0.0 | -1.101763 |
    | 2017-03-29 |           1.5 |         1.0 | 0.582764   0.56206  0.0                   0.0 | 0.6   1.068  0.0                      0.8 |  0.9      0.0  0.0          0.0 |   4.5    4.5  0.0                4.5 |   4.5     4.5  0.0       4.5 |   5.4      4.5  0.0      4.5 | 0.0  0.0  0.0            4.5 |  1.139787   0.879008  0.0          0.0 | 0.163731       0.0  0.0                     0.0 | 0.0  0.0  1.0048              0.0 | 0.763731     1.068  1.0048                      0.8 | 0.369543  0.410206  0.0            0.0 | 11.723622  13.794033  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.770244   0.468802   0.4952                0.0 | 0.154049   0.09376  0.09904       0.0 |       0.346849 |       -0.136843 | 0.0   0.432  0.0               0.7 |  97.45  84.1985  0.0     87.85 | 58.276378  56.205967  0.0           0.0 | -0.618072 |
    | 2017-03-30 |           0.3 |         4.4 |  0.72222  0.705958  0.0                   0.0 | 0.3   0.732  0.0                      1.0 |  0.0      0.0  0.0          0.0 |  19.8   19.8  0.0               19.8 |  19.8    19.8  0.0      19.8 |  19.8     19.8  0.0     19.8 | 0.0  0.0  0.0           19.8 |  5.105211    4.66207  0.0          0.0 | 1.021504  1.063053  0.0                     0.0 | 0.0  0.0   1.884              0.0 | 1.321504  1.795053   1.884                      1.0 | 0.272381  0.314975  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |   4.83283   4.347095   -1.584                0.0 | 0.966566  0.869419      0.0       0.0 |       1.439985 |        0.148988 | 0.0     0.0  0.0               0.0 |  77.65  64.3985  0.0     68.05 | 72.222044  70.595819  0.0           0.0 |  0.672926 |
    | 2017-03-31 |           0.2 |         7.0 | 0.875684  0.859162  0.0                   0.0 | 0.2     0.2  0.0                      0.2 |  0.0      0.0  0.0          0.0 |  31.5   31.5  0.0               31.5 |  31.5    31.5  0.0      31.5 |  31.5     31.5  0.0     31.5 | 0.0  0.0  0.0           31.5 | 13.917106  13.143815  0.0          0.0 | 2.316867  3.131125  0.0                     0.0 | 0.0  0.0  3.3912              0.0 | 2.516867  3.331125  3.3912                      0.2 | 0.080377  0.095336  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 13.836729  13.048479  -3.1912                0.0 | 2.767346  2.609696      0.0       0.0 |       4.579241 |        0.952056 | 0.0     0.0  0.0               0.0 |  46.15  32.8985  0.0     36.55 | 87.568449  85.916215  0.0           0.0 |  4.300111 |
    | 2017-04-01 |           4.5 |         3.8 | 0.927868  0.921345  0.0                   0.0 | 0.8     2.0  0.0                      1.5 |  3.7      2.5  0.0          2.5 |  17.1   17.1  0.0               17.1 |  17.1    17.1  0.0      17.1 |  20.8     19.6  0.0     19.6 | 0.0  0.0  0.0           19.6 | 14.905504  13.389799  0.0          0.0 | 0.683373   0.00243  0.0                     0.0 | 0.0  0.0  1.9245              0.0 | 1.483373   2.00243  1.9245                      1.5 | 0.007205  0.010476  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | 14.898299  13.379323   2.5755                0.0 |  2.97966  2.675865   0.5151       0.0 |       6.170624 |        1.898022 | 0.0     0.0  0.0               0.5 |  29.05  15.7985  0.0     19.45 | 92.786777  92.134462  0.0           0.0 |  8.572713 |
    | 2017-04-02 |           0.0 |         1.4 | 0.922874  0.912005  0.0                   0.0 | 0.0     0.0  0.0                      0.5 |  0.0      0.0  0.0          0.0 |   6.3    6.3  0.0                6.3 |   6.3     6.3  0.0       6.3 |   6.3      6.3  0.0      6.3 | 0.0  0.0  0.0            6.3 |   5.22066   5.129019  0.0          0.0 | 1.580155  2.106732  0.0                     0.0 | 0.0  0.0  2.0528              0.0 | 1.580155  2.106732  2.0528                      0.5 | 0.001407  0.001825  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  5.219252   5.127194  -2.0528                0.0 |  1.04385  1.025439      0.0       0.0 |       1.556089 |         1.83604 | 0.0     0.0  0.0               0.0 |  22.75   9.4985  0.0     13.15 | 92.287369  91.200536  0.0           0.0 |  8.292762 |
    | 2017-04-03 |           0.0 |         0.1 |  0.90398  0.886677  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  0.45   0.45  0.0               0.45 |  0.45    0.45  0.0      0.45 |  0.45     0.45  0.0     0.45 | 0.0  0.0  0.0           0.45 |  0.367887   0.357108  0.0          0.0 |  1.97321  2.628296  0.0                     0.0 | 0.0  0.0   2.566              0.0 |  1.97321  2.628296   2.566                      0.0 |  0.00172  0.002555  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.366166   0.354553   -2.566                0.0 | 0.073233  0.070911      0.0       0.0 |      -0.497356 |        1.413067 | 0.0     0.0  0.0               0.0 |   22.3   9.0485  0.0      12.7 | 90.397992  88.667687  0.0           0.0 |  6.382338 |
    | 2017-04-04 |           0.0 |        -0.1 | 0.883377   0.85929  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 2.063624  2.744142  0.0                     0.0 | 0.0  0.0  2.6943              0.0 | 2.063624  2.744142  2.6943                      0.0 |  0.00332  0.005457  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  -0.00332  -0.005457  -2.6943                0.0 |      0.0       0.0      0.0       0.0 |      -0.675769 |        1.034425 | 0.0     0.0  0.0               0.0 |   22.3   9.0485  0.0      12.7 | 88.337688  85.929002  0.0           0.0 |  4.672144 |
    | 2017-04-05 |           0.4 |        -0.5 | 0.870722  0.841219  0.0                   0.0 | 0.4     0.4  0.0                      0.4 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.271483  1.817519  0.0                     0.0 | 0.0  0.0  2.1811              0.0 | 1.671483  2.217519  2.1811                      0.4 | 0.005948  0.010447  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.005948  -0.010447  -1.7811                0.0 |      0.0       0.0      0.0       0.0 |      -0.449374 |        0.765458 | 0.0     0.0  0.0               0.0 |   22.3   9.0485  0.0      12.7 | 87.072153  84.121931  0.0           0.0 |  3.457311 |
    | 2017-04-06 |           0.0 |        -0.8 | 0.854227  0.819414  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   0.0    0.0  0.0                0.0 |   0.0     0.0  0.0       0.0 |   0.0      0.0  0.0      0.0 | 0.0  0.0  0.0            0.0 |       0.0        0.0  0.0          0.0 | 1.657579  2.195586  0.0                     0.0 | 0.0  0.0  2.1811              0.0 | 1.657579  2.195586  2.1811                      0.0 | 0.008102  0.015012  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 | -0.008102  -0.015012  -2.1811                0.0 |      0.0       0.0      0.0       0.0 |      -0.551053 |        0.526815 | 0.0     0.0  0.0               0.0 |   22.3   9.0485  0.0      12.7 | 85.422677  81.941357  0.0           0.0 |  2.379443 |
    | 2017-04-07 |           0.0 |         0.1 | 0.848046  0.811139  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |  0.45   0.45  0.0               0.45 |  0.45    0.45  0.0      0.45 |  0.45     0.45  0.0     0.45 | 0.0  0.0  0.0           0.45 |  0.303005   0.272955  0.0          0.0 | 0.776714  1.026604  0.0                     0.0 | 0.0  0.0  1.0264              0.0 | 0.776714  1.026604  1.0264                      0.0 | 0.011616  0.022084  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  0.291389   0.250871  -1.0264                0.0 | 0.058278  0.050174      0.0       0.0 |      -0.148148 |        0.404465 | 0.0     0.0  0.0               0.0 |  21.85   8.5985  0.0     12.25 | 84.804574  81.113881  0.0           0.0 |   1.82683 |
    | 2017-04-08 |           0.0 |         0.6 | 0.844725  0.805787  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 |   2.7    2.7  0.0                2.7 |   2.7     2.7  0.0       2.7 |   2.7      2.7  0.0      2.7 | 0.0  0.0  0.0            2.7 |  1.785191   1.596533  0.0          0.0 |  1.26005  1.663922  0.0                     0.0 | 0.0  0.0  1.6679              0.0 |  1.26005  1.663922  1.6679                      0.0 | 0.013157  0.025262  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  1.772033   1.571272  -1.6679                0.0 | 0.354407  0.314254      0.0       0.0 |       0.251686 |        0.376771 | 0.0     0.0  0.0               0.0 |  19.15   5.8985  0.0      9.55 | 84.472491  80.578687  0.0           0.0 |  1.701745 |
    | 2017-04-09 |           0.0 |         2.5 | 0.859502  0.798799  0.0                   0.0 | 0.0     0.0  0.0                      0.0 |  0.0      0.0  0.0          0.0 | 11.25  11.25  0.0              11.25 | 11.25  5.8985  0.0      9.55 | 11.25   5.8985  0.0     9.55 | 0.0  0.0  0.0           9.55 |  7.365396   3.430356  0.0          0.0 | 2.420952  3.194353  0.0                     0.0 | 0.0  0.0  3.2075              0.0 | 2.420952  3.194353  3.2075                      0.0 | 0.014039  0.027471  0.0            0.0 |       0.0        0.0  0.0                 0.0 | 0.0  0.0  0.0                 0.0 |  7.351357   3.402885  -3.2075                0.0 | 1.470271  0.680577      0.0       0.0 |       1.348973 |        0.553001 | 0.0     0.0  0.0               0.0 |    7.9      0.0  0.0       0.0 | 85.950181  79.879949  0.0           0.0 |  2.497717 |

>>> round_(model.check_waterbalance(conditions))
0.0
"""

import numpy

from hydpy.core import masktools
from hydpy.core import modeltools
from hydpy.core.typingtools import *
from hydpy.exe.modelimports import *
from hydpy.interfaces import aetinterfaces

from hydpy.models.whmod import whmod_masks
from hydpy.models.whmod import whmod_model
from hydpy.models.whmod.whmod_constants import *


class Model(
    whmod_model.Main_AETModel_V1,
    whmod_model.Sub_TempModel_V1,
    whmod_model.Sub_PrecipModel_V1,
    whmod_model.Sub_IntercModel_V1,
    whmod_model.Sub_SoilWaterModel_V1,
    whmod_model.Sub_SnowCoverModel_V1,
):
    """|whmod_rural.DOCNAME.complete|."""

    DOCNAME = modeltools.DocName(
        short="WHMod-Rural", description="WHMod for rural regions"
    )
    __HYDPY_ROOTMODEL__ = True

    INLET_METHODS = ()
    OBSERVER_METHODS = ()
    RECEIVER_METHODS = ()
    ADD_METHODS = (
        whmod_model.Calc_InterceptionEvaporation_InterceptedWater_AETModel_V1,
        whmod_model.Calc_LakeEvaporation_AETModel_V1,
        whmod_model.Calc_SoilEvapotranspiration_AETModel_V1,
    )
    INTERFACE_METHODS = (
        whmod_model.Get_Temperature_V1,
        whmod_model.Get_MeanTemperature_V1,
        whmod_model.Get_Precipitation_V1,
        whmod_model.Get_InterceptedWater_V1,
        whmod_model.Get_SoilWater_V1,
        whmod_model.Get_SnowCover_V1,
    )
    RUN_METHODS = (
        whmod_model.Calc_Throughfall_InterceptedWater_V1,
        whmod_model.Calc_InterceptionEvaporation_InterceptedWater_V1,
        whmod_model.Calc_LakeEvaporation_V1,
        whmod_model.Calc_PotentialSnowmelt_V1,
        whmod_model.Calc_Snowmelt_Snowpack_V1,
        whmod_model.Calc_Ponding_V1,
        whmod_model.Calc_SurfaceRunoff_V1,
        whmod_model.Calc_RelativeSoilMoisture_V1,
        whmod_model.Calc_Percolation_V1,
        whmod_model.Calc_SoilEvapotranspiration_V1,
        whmod_model.Calc_TotalEvapotranspiration_V1,
        whmod_model.Calc_CapillaryRise_V1,
        whmod_model.Calc_SoilMoisture_V1,
        whmod_model.Calc_RelativeSoilMoisture_V1,
        whmod_model.Calc_RequiredIrrigation_V1,
        whmod_model.Calc_ExternalIrrigation_SoilMoisture_V1,
        whmod_model.Calc_RelativeSoilMoisture_V1,
        whmod_model.Calc_PotentialRecharge_V1,
        whmod_model.Calc_Baseflow_V1,
        whmod_model.Calc_ActualRecharge_V1,
        whmod_model.Calc_DelayedRecharge_DeepWater_V1,
    )
    OUTLET_METHODS = ()
    SENDER_METHODS = ()
    SUBMODELINTERFACES = (aetinterfaces.AETModel_V1,)
    SUBMODELS = ()

    aetmodel = modeltools.SubmodelProperty(aetinterfaces.AETModel_V1)

    def check_waterbalance(self, initial_conditions: ConditionsModel) -> float:
        r"""Determine the water balance error of the previous simulation run in mm.

        Method |Model.check_waterbalance| calculates the balance error as follows:

          .. math::
            Error = \Sigma In - \Sigma Out - \Delta Vol
            \\ \\
            \Sigma In = \sum_{t=t_0}^{t_1} \left(
            P_t + \sum_{k=1}^{N} Z^k \cdot E_t^k \right)
            \\
            \Sigma Out = \sum_{t=t_0}^{t_1} \left(
            T_t + \sum_{k=1}^{N} Z^k \cdot \Big( S_t^k + B_t^k + R_t^k \Big) \right)
            \\
            \Delta Vol = f_{\Delta}\big(D\big) + \sum_{k=1}^{N} Z^k \cdot \Big(
            f_{\Delta} \big( I^k \big) + f_{\Delta} \big( P^k \big) +
            f_{\Delta} \big( M^k \big) \Big)
            \\ \\
            f_{\Delta}(x) = x_{t1} - x_{t0}
            \\ \\
            N = NmbZones \\
            Z = ZoneRatio \\
            P = Precipitation \\
            T = TotalEvapotranspiration \\
            S = SurfaceRunoff \\
            B = Baseflow \\
            R = DelayedRecharge \\
            I = InterceptedWater \\
            P = Snowpack \\
            M = SoilMoisture \\
            D = DeepWater

        The returned error should always be in scale with numerical precision so that
        it does not affect the simulation results in any relevant manner.

        Pick the required initial conditions before starting the simulation run via
        property |Sequences.conditions|.  See the integration tests of the application
        model |whmod_rural| for some examples.
        """

        fluxes = self.sequences.fluxes
        last = self.sequences.states
        first = initial_conditions["model"]["states"]
        r = self.parameters.derived.zoneratio.values
        return (
            sum(self.sequences.inputs.precipitation.series)
            - numpy.sum(r * fluxes.totalevapotranspiration.series)
            - numpy.sum(r * fluxes.surfacerunoff.series)
            + numpy.sum(r * fluxes.externalirrigation.series)
            - numpy.sum(r * fluxes.baseflow.series)
            - sum(fluxes.delayedrecharge.series)
            - sum(r * (last.interceptedwater - first["interceptedwater"]))
            - sum(r * (last.snowpack - first["snowpack"]))
            - sum(r * (last.soilmoisture - first["soilmoisture"]))
            - (last.deepwater - first["deepwater"])
        )


class Masks(masktools.Masks):
    """Masks applicable to |whmod_rural|."""

    CLASSES = whmod_masks.Masks.CLASSES


tester = Tester()
cythonizer = Cythonizer()
